/* vim:set et sts=4 sw=4:
 *
 * ibus-xkb - IBus XKB
 *
 * Copyright(c) 2012 Takao Fujiwara <takao.fujiwara1@gmail.com>
 * Copyright(c) 2011 Peng Huang <shawn.p.huang@gmail.com>
 * Copyright (c) 2012 Red Hat, Inc.
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or(at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this program; if not, write to the
 * Free Software Foundation, Inc., 59 Temple Place, Suite 330,
 * Boston, MA  02111-1307  USA
 */

using IBus;
using IBusXKB;
using GLib;

private void print_xml(string  layout,
                       string  layout_desc,
                       string? variant,
                       string? variant_desc,
                       string  lang) {
    string name = "xkb:%s:%s:%s".printf(layout, variant ?? "", lang);
    string keymap = layout;
    string desc = layout_desc;
    string symbol = lang;

    if (variant != null) {
        keymap = "%s(%s)".printf(layout, variant);
    }

    if (variant_desc != null) {
        desc = variant_desc;
    }

    desc = desc.replace("<", "&lt;").replace(">", "&gt;");

    if (lang.length > 2) {
        symbol = lang[0:2];
    }

    string engine = "
              <engine>
                        <name>%s</name>
                        <language>%s</language>
                        <license>GPL</license>
                        <author>Peng Huang &lt;shawn.p.huang@gmail.com&gt;</author>
                        <layout>%s</layout>
                        <longname>%s</longname>
                        <description>%s</description>
                        <icon>ibus-keyboard</icon>
                        <symbol>%s</symbol>
                        <rank>%d</rank>
              </engine>".printf(name, lang, keymap, desc, desc, symbol, 99);
    print (engine);
}

private void print_component() {
    IBusXKB.ConfigRegistry registry = new IBusXKB.ConfigRegistry();
    GLib.List layouts = registry.layout_list_get_layouts();
    GLib.List variants;
    GLib.List langs;
    string layout_desc;
    const string header = "<engines>";
    const string footer = "
</engines>";

    print (header);
    for (unowned GLib.List<string> l = layouts; l != null; l = l.next) {
        variants = registry.layout_list_get_variants(l.data);
        langs = registry.layout_lang_get_langs(l.data);
        layout_desc = registry.layout_desc_get_desc(l.data);
        for (unowned GLib.List<string> lg = langs; lg != null; lg = lg.next) {
            print_xml(l.data, layout_desc, null, null, lg.data);
        }
        for (unowned GLib.List<string> v = variants; v != null; v = v.next) {
            var l_v = "%s(%s)".printf(l.data, v.data);
            unowned GLib.List<string> l_v_langs = null;
            GLib.List<string> _l_v_langs = registry.layout_lang_get_langs(l_v);
            l_v_langs = _l_v_langs;
            if (l_v_langs == null) {
                l_v_langs = langs;
            }
            for (unowned GLib.List<string> lg = l_v_langs; lg != null; lg = lg.next) {
                print_xml(l.data,
                          layout_desc,
                          v.data,
                          registry.variant_desc_get_desc(l_v),
                          lg.data);
            }
        }
    }
    print (footer);
}

class DummyEngine : IBus.EngineSimple {
}

public int main(string[] args) {
    IBus.init();

    if (args.length >= 2) {
        if (args[1] == "--xml" || args[1] == "-x") {
            print_component();
            return 0;
        }
    }

    IBus.Bus bus = new IBus.Bus();
    if (!bus.is_connected()) {
        warning("ibus-daemon does not exist.");
        return 1;
    }

    uint flags = 
        IBus.BusNameFlag.REPLACE_EXISTING |
        IBus.BusNameFlag.ALLOW_REPLACEMENT;
    uint retval = bus.request_name("org.freedesktop.IBus.XKB", flags);

    if (retval == 0) {
        warning("Registry bus name org.freedesktop.IBus.XKB failed!");
        return 1;
    }

    bus.disconnected.connect((bus) => {
        debug("bus disconnected");
        IBus.quit();
    });

    IBus.Factory factory = new IBus.Factory(bus.get_connection());
    
    int id = 0;

    factory.create_engine.connect((factory, name) => {
        const string path = "/org/freedesktop/IBus/engine/xkb/%d";
        IBus.Engine engine = new IBus.Engine.with_type(
            typeof(IBus.EngineSimple), name,
            path.printf(++id), bus.get_connection());
        return engine;
    });

    IBus.main();

    return 0;
}
