#include <stdio.h>
#include "uninameslist.h"
#include "nameslist-dll.h"

/* This file was generated using the program 'buildnameslist.c' */


/*
The data contained in these arrays were derived from data contained in
NamesList.txt which came from www.unicode.org. Below is the copyright
notice for the information given:

UNICODE LICENSE V3

COPYRIGHT AND PERMISSION NOTICE

Copyright © 1991-2024 Unicode, Inc.

NOTICE TO USER: Carefully read the following legal agreement. BY
DOWNLOADING, INSTALLING, COPYING OR OTHERWISE USING DATA FILES, AND/OR
SOFTWARE, YOU UNEQUIVOCALLY ACCEPT, AND AGREE TO BE BOUND BY, ALL OF THE
TERMS AND CONDITIONS OF THIS AGREEMENT. IF YOU DO NOT AGREE, DO NOT
DOWNLOAD, INSTALL, COPY, DISTRIBUTE OR USE THE DATA FILES OR SOFTWARE.
Permission is hereby granted, free of charge, to any person obtaining a
copy of data files and any associated documentation (the "Data Files") or
software and any associated documentation (the "Software") to deal in the
Data Files or Software without restriction, including without limitation
the rights to use, copy, modify, merge, publish, distribute, and/or sell
copies of the Data Files or Software, and to permit persons to whom the
Data Files or Software are furnished to do so, provided that either (a)
this copyright and permission notice appear with all copies of the Data
Files or Software, or (b) this copyright and permission notice appear in
associated Documentation.

THE DATA FILES AND SOFTWARE ARE PROVIDED "AS IS", WITHOUT WARRANTY OF ANY
KIND, EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF
MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT OF
THIRD PARTY RIGHTS.

IN NO EVENT SHALL THE COPYRIGHT HOLDER OR HOLDERS INCLUDED IN THIS NOTICE
BE LIABLE FOR ANY CLAIM, OR ANY SPECIAL INDIRECT OR CONSEQUENTIAL DAMAGES,
OR ANY DAMAGES WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS,
WHETHER IN AN ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION,
ARISING OUT OF OR IN CONNECTION WITH THE USE OR PERFORMANCE OF THE DATA
FILES OR SOFTWARE.

Except as contained in this notice, the name of a copyright holder shall
not be used in advertising or otherwise to promote the sale, use or other
dealings in these Data Files or Software without prior written
authorization of the copyright holder.

*/

/* Retrieve a pointer to the name of a Unicode codepoint. */
UN_DLL_EXPORT
const char *uniNamesList_name(unsigned long uni) {
	const char *pt=NULL;

	if ( uni<0x110000 )
		pt=UnicodeNameAnnot[uni>>16][(uni>>8)&0xff][uni&0xff].name;
	return( pt );
}

/* Retrieve a pointer to annotation details of a Unicode codepoint. */
UN_DLL_EXPORT
const char *uniNamesList_annot(unsigned long uni) {
	const char *pt=NULL;

	if ( uni<0x110000 )
		pt=UnicodeNameAnnot[uni>>16][(uni>>8)&0xff][uni&0xff].annot;
	return( pt );
}

/* Retrieve Nameslist.txt version number. */
UN_DLL_EXPORT
const char *uniNamesList_NamesListVersion(void) {
	return( "Nameslist-Version: 16.0" );
}


/* These functions are available in libuninameslist-0.4.20140731 and higher */

/* Return number of blocks in this NamesList. */
UN_DLL_EXPORT
int uniNamesList_blockCount(void) {
	return( UNICODE_EN_BLOCK_MAX );
}

/* Return block number for this unicode value, -1 if unlisted unicode value */
UN_DLL_EXPORT
int uniNamesList_blockNumber(unsigned long uni) {
	int i;

	if ( uni<0x110000 ) {
		for (i=0; i<UNICODE_EN_BLOCK_MAX; i++) {
			if ( uni<(unsigned long)(UnicodeBlock[i].start) ) break;
			if ( uni<=(unsigned long)(UnicodeBlock[i].end) ) return( i );
		}
	}
	return( -1 );
}

/* Return unicode value starting this Unicode block (-1 if bad uniBlock). */
UN_DLL_EXPORT
long uniNamesList_blockStart(int uniBlock) {
	if ( uniBlock<0 || uniBlock>=UNICODE_EN_BLOCK_MAX )
		return( -1 );
	return( (long)(UnicodeBlock[uniBlock].start) );
}

/* Return unicode value ending this Unicode block (-1 if bad uniBlock). */
UN_DLL_EXPORT
long uniNamesList_blockEnd(int uniBlock) {
	if ( uniBlock<0 || uniBlock>=UNICODE_EN_BLOCK_MAX )
		return( -1 );
	return( (long)(UnicodeBlock[uniBlock].end) );
}

/* Return a pointer to the blockname for this unicode block. */
UN_DLL_EXPORT
const char * uniNamesList_blockName(int uniBlock) {
	if ( uniBlock<0 || uniBlock>=UNICODE_EN_BLOCK_MAX )
		return( NULL );
	return( UnicodeBlock[uniBlock].name );
}


/* These functions are available in libuninameslist-20171118 and higher */

/* Return count of how many names2 are found in this version of library */
UN_DLL_EXPORT
int uniNamesList_names2cnt(void) {
	return( 36 );
}

static const unsigned long unicode_name2code[] = {
	0x01A2, 0x01A3, 0x0616, 0x0709, 0x0CDE, 0x0E9D, 0x0E9F, 0x0EA3,
	0x0EA5, 0x0FD0, 0x11EC, 0x11ED, 0x11EE, 0x11EF, 0x1BBD, 0x2118,
	0x2448, 0x2449, 0x2B7A, 0x2B7C, 0xA015, 0xAA6E, 0xFE18, 0xFEFF,
	74452, 74453, 74535, 92171, 93782, 93783, 93814, 93815,
	110593, 118981, 125081, 125082
};

static const char unicode_name2vals[] = {
	3,24, 3,22, 3,47, 3,34, 3,19, 3,17, 3,17, 3,13,
	3,13, 3,35, 3,32, 3,37, 3,30, 3,33, 3,26, 3,29,
	3,17, 3,16, 3,59, 3,60, 3,26, 3,25, 3,61, 3,15,
	3,24, 3,42, 3,20, 3,30, 3,28, 3,29, 3,26, 3,27,
	3,21, 3,52, 3,31, 3,32
};

/* Return unicode value with names2 (0<=count<uniNamesList_names2cnt(). */
UN_DLL_EXPORT
long uniNamesList_names2val(int count) {
	if ( count<0 || count>=36 ) return( -1 );
	return( (long)(unicode_name2code[count]) );
}

/* Return list location for this unicode value. Return -1 if not found. */
UN_DLL_EXPORT
int uniNamesList_names2getU(unsigned long uni) {
	int i;

	if ( uni<0x110000 ) for ( i=0; i<36; ++i ) {
		if ( uni==unicode_name2code[i] ) return( i );
		if ( uni<unicode_name2code[i] ) break;
	}
	return( -1 );
}

/* Stringlength of names2. Use this if you want to truncate annotations */
UN_DLL_EXPORT
int uniNamesList_names2lnC(int count) {
	if ( count<0 || count>=36 ) return( -1 );
	return( (int)(unicode_name2vals[(count<<1)+1]) );
}

UN_DLL_EXPORT
int uniNamesList_names2lnU(unsigned long uni) {
	return( uniNamesList_names2lnC(uniNamesList_names2getU(uni)) );
}

/* Return pointer to start of normalized alias names2 within annotation */
UN_DLL_EXPORT
const char *uniNamesList_names2anC(int count) {
	int c;
	const char *pt;

	if ( count<0 || count>=36 ) return( NULL );
	c = unicode_name2vals[count<<1];
	pt = uniNamesList_annot((unsigned long)(uniNamesList_names2val(count)));
	while ( --c>=0 ) ++pt;
	return( pt );
}

UN_DLL_EXPORT
const char *uniNamesList_names2anU(unsigned long uni) {
	return( uniNamesList_names2anC(uniNamesList_names2getU(uni)) );
}


/* These functions are available in libuninameslist-20200413 and higher */

UN_DLL_LOCAL
int uniNamesList_haveFR(unsigned int lang) {
#ifdef WANTLIBOFR
	if ( lang==1 ) return( 1 );
#endif
	return( 0 );
}

#ifndef WANTLIBOFR
/* make these internal stubs since there's no French lib */
UN_DLL_LOCAL const char *uniNamesList_NamesListVersionFR(void) {return( NULL );}
UN_DLL_LOCAL const char *uniNamesList_nameFR(unsigned long uni) {return( NULL );}
UN_DLL_LOCAL const char *uniNamesList_annotFR(unsigned long uni) {return( NULL );}
UN_DLL_LOCAL int uniNamesList_blockCountFR(void) {return( -1 );}
UN_DLL_LOCAL int uniNamesList_blockNumberFR(unsigned long uni) {return( -1 );}
UN_DLL_LOCAL long uniNamesList_blockStartFR(int uniBlock) {return( -1 );}
UN_DLL_LOCAL long uniNamesList_blockEndFR(int uniBlock) {return( -1 );}
UN_DLL_LOCAL const char *uniNamesList_blockNameFR(int uniBlock) {return( NULL );}
#endif

/* Return language codes available from libraries. 0=English, 1=French. */
UN_DLL_EXPORT
const char *uniNamesList_Languages(unsigned int lang) {
	if ( uniNamesList_haveFR(lang) )
		return( "FR" );
	else if ( lang==0 )
		return( "EN" );
	return( NULL );
}

UN_DLL_EXPORT
const char *uniNamesList_NamesListVersionAlt(unsigned int lang) {
	if ( uniNamesList_haveFR(lang) )
		return( (const char *)(uniNamesList_NamesListVersionFR()) );
	else if ( lang==0 )
		return( uniNamesList_NamesListVersion() );
	return( NULL );
}

/* Return pointer to name/annotation for this unicode value using lang. */
/* Return English if language does not have information for this Ucode. */
UN_DLL_EXPORT
const char *uniNamesList_nameAlt(unsigned long uni, unsigned int lang) {
	const char *pt=NULL;

	if ( uni<0x110000 ) {
		if ( uniNamesList_haveFR(lang) )
			pt=(const char *)(uniNamesList_nameFR(uni));
		if ( pt==NULL )
			pt=uniNamesList_name(uni);
	}
	return( pt );
}

UN_DLL_EXPORT
const char *uniNamesList_annotAlt(unsigned long uni, unsigned int lang) {
	const char *pt=NULL;

	if ( uni<0x110000 ) {
		if ( uniNamesList_haveFR(lang) )
			pt=(const char *)(uniNamesList_annotFR(uni));
		if ( pt==NULL )
			pt=uniNamesList_annot(uni);
	}
	return( pt );
}

/* Returns 2 lang pointers to names/annotations for this unicode value, */
/* Return str0=English, and str1=language_version (or NULL if no info). */
UN_DLL_EXPORT
int uniNamesList_nameBoth(unsigned long uni, unsigned int lang, const char **str0, const char **str1) {
	int error=-1;
	*str0=*str1=NULL;

	if ( uni<0x110000 ) {
		error=0;
		*str0=uniNamesList_name(uni);
		if ( uniNamesList_haveFR(lang) )
			*str1=(const char *)(uniNamesList_nameFR(uni));
		else if ( lang==0 )
			*str1=*str0;
	}
	return( error );
}

UN_DLL_EXPORT
int uniNamesList_annotBoth(unsigned long uni, unsigned int lang, const char **str0, const char **str1) {
	int error=-1;
	*str0=*str1=NULL;

	if ( uni<0x110000 ) {
		error=0;
		*str0=uniNamesList_annot(uni);
		if ( uniNamesList_haveFR(lang) )
			*str1=(const char *)(uniNamesList_annotFR(uni));
		else if ( lang==0 )
			*str1=*str0;
	}
	return( error );
}

/* Common access. Blocklists won't sync if they are different versions. */
UN_DLL_EXPORT
int uniNamesList_blockCountAlt(unsigned int lang) {
	int c=-1;

	if ( uniNamesList_haveFR(lang) )
		c=(int)(uniNamesList_blockCountFR());
	if ( c<0 )
		c=UNICODE_EN_BLOCK_MAX;
	return( c );
}

UN_DLL_EXPORT
long uniNamesList_blockStartAlt(int uniBlock, unsigned int lang) {
	long c=-1;

	if ( uniNamesList_haveFR(lang) )
		c=(long)(uniNamesList_blockStartFR(uniBlock));
	if ( c<0 )
		c=uniNamesList_blockStart(uniBlock);
	return( c );
}

UN_DLL_EXPORT
long uniNamesList_blockEndAlt(int uniBlock, unsigned int lang) {
	long c=-1;

	if ( uniNamesList_haveFR(lang) )
		c=(long)(uniNamesList_blockEndFR(uniBlock));
	if ( c<0 )
		c=uniNamesList_blockEnd(uniBlock);
	return( c );
}

UN_DLL_EXPORT
const char *uniNamesList_blockNameAlt(int uniBlock, unsigned int lang) {
	const char *pt=NULL;

	if ( uniNamesList_haveFR(lang) )
		pt=(const char *)(uniNamesList_blockNameFR(uniBlock));
	if ( pt==NULL )
		pt=uniNamesList_blockName(uniBlock);
	return( pt );
}

UN_DLL_EXPORT
int uniNamesList_blockNumberBoth(unsigned long uni, unsigned int lang, int *bn0, int *bn1) {
	int error=-1;

	*bn0=*bn1=-1;
	if ( uni<0x110000 ) {
		error=0;
		*bn0=uniNamesList_blockNumber(uni);
		if ( uniNamesList_haveFR(lang) )
			*bn1=(int)(uniNamesList_blockNumberFR(uni));
		else if ( lang==0 )
			*bn1=*bn0;
	}
	return( error );
}


UN_DLL_LOCAL
static const struct unicode_nameannot nullarray[] = {
	{ NULL, NULL }, { NULL, NULL }, { NULL, NULL }, { NULL, NULL },
	{ NULL, NULL }, { NULL, NULL }, { NULL, NULL }, { NULL, NULL },
	{ NULL, NULL }, { NULL, NULL }, { NULL, NULL }, { NULL, NULL },
	{ NULL, NULL }, { NULL, NULL }, { NULL, NULL }, { NULL, NULL },
	{ NULL, NULL }, { NULL, NULL }, { NULL, NULL }, { NULL, NULL },
	{ NULL, NULL }, { NULL, NULL }, { NULL, NULL }, { NULL, NULL },
	{ NULL, NULL }, { NULL, NULL }, { NULL, NULL }, { NULL, NULL },
	{ NULL, NULL }, { NULL, NULL }, { NULL, NULL }, { NULL, NULL },
	{ NULL, NULL }, { NULL, NULL }, { NULL, NULL }, { NULL, NULL },
	{ NULL, NULL }, { NULL, NULL }, { NULL, NULL }, { NULL, NULL },
	{ NULL, NULL }, { NULL, NULL }, { NULL, NULL }, { NULL, NULL },
	{ NULL, NULL }, { NULL, NULL }, { NULL, NULL }, { NULL, NULL },
	{ NULL, NULL }, { NULL, NULL }, { NULL, NULL }, { NULL, NULL },
	{ NULL, NULL }, { NULL, NULL }, { NULL, NULL }, { NULL, NULL },
	{ NULL, NULL }, { NULL, NULL }, { NULL, NULL }, { NULL, NULL },
	{ NULL, NULL }, { NULL, NULL }, { NULL, NULL }, { NULL, NULL },
	{ NULL, NULL }, { NULL, NULL }, { NULL, NULL }, { NULL, NULL },
	{ NULL, NULL }, { NULL, NULL }, { NULL, NULL }, { NULL, NULL },
	{ NULL, NULL }, { NULL, NULL }, { NULL, NULL }, { NULL, NULL },
	{ NULL, NULL }, { NULL, NULL }, { NULL, NULL }, { NULL, NULL },
	{ NULL, NULL }, { NULL, NULL }, { NULL, NULL }, { NULL, NULL },
	{ NULL, NULL }, { NULL, NULL }, { NULL, NULL }, { NULL, NULL },
	{ NULL, NULL }, { NULL, NULL }, { NULL, NULL }, { NULL, NULL },
	{ NULL, NULL }, { NULL, NULL }, { NULL, NULL }, { NULL, NULL },
	{ NULL, NULL }, { NULL, NULL }, { NULL, NULL }, { NULL, NULL },
	{ NULL, NULL }, { NULL, NULL }, { NULL, NULL }, { NULL, NULL },
	{ NULL, NULL }, { NULL, NULL }, { NULL, NULL }, { NULL, NULL },
	{ NULL, NULL }, { NULL, NULL }, { NULL, NULL }, { NULL, NULL },
	{ NULL, NULL }, { NULL, NULL }, { NULL, NULL }, { NULL, NULL },
	{ NULL, NULL }, { NULL, NULL }, { NULL, NULL }, { NULL, NULL },
	{ NULL, NULL }, { NULL, NULL }, { NULL, NULL }, { NULL, NULL },
	{ NULL, NULL }, { NULL, NULL }, { NULL, NULL }, { NULL, NULL },
	{ NULL, NULL }, { NULL, NULL }, { NULL, NULL }, { NULL, NULL },
	{ NULL, NULL }, { NULL, NULL }, { NULL, NULL }, { NULL, NULL },
	{ NULL, NULL }, { NULL, NULL }, { NULL, NULL }, { NULL, NULL },
	{ NULL, NULL }, { NULL, NULL }, { NULL, NULL }, { NULL, NULL },
	{ NULL, NULL }, { NULL, NULL }, { NULL, NULL }, { NULL, NULL },
	{ NULL, NULL }, { NULL, NULL }, { NULL, NULL }, { NULL, NULL },
	{ NULL, NULL }, { NULL, NULL }, { NULL, NULL }, { NULL, NULL },
	{ NULL, NULL }, { NULL, NULL }, { NULL, NULL }, { NULL, NULL },
	{ NULL, NULL }, { NULL, NULL }, { NULL, NULL }, { NULL, NULL },
	{ NULL, NULL }, { NULL, NULL }, { NULL, NULL }, { NULL, NULL },
	{ NULL, NULL }, { NULL, NULL }, { NULL, NULL }, { NULL, NULL },
	{ NULL, NULL }, { NULL, NULL }, { NULL, NULL }, { NULL, NULL },
	{ NULL, NULL }, { NULL, NULL }, { NULL, NULL }, { NULL, NULL },
	{ NULL, NULL }, { NULL, NULL }, { NULL, NULL }, { NULL, NULL },
	{ NULL, NULL }, { NULL, NULL }, { NULL, NULL }, { NULL, NULL },
	{ NULL, NULL }, { NULL, NULL }, { NULL, NULL }, { NULL, NULL },
	{ NULL, NULL }, { NULL, NULL }, { NULL, NULL }, { NULL, NULL },
	{ NULL, NULL }, { NULL, NULL }, { NULL, NULL }, { NULL, NULL },
	{ NULL, NULL }, { NULL, NULL }, { NULL, NULL }, { NULL, NULL },
	{ NULL, NULL }, { NULL, NULL }, { NULL, NULL }, { NULL, NULL },
	{ NULL, NULL }, { NULL, NULL }, { NULL, NULL }, { NULL, NULL },
	{ NULL, NULL }, { NULL, NULL }, { NULL, NULL }, { NULL, NULL },
	{ NULL, NULL }, { NULL, NULL }, { NULL, NULL }, { NULL, NULL },
	{ NULL, NULL }, { NULL, NULL }, { NULL, NULL }, { NULL, NULL },
	{ NULL, NULL }, { NULL, NULL }, { NULL, NULL }, { NULL, NULL },
	{ NULL, NULL }, { NULL, NULL }, { NULL, NULL }, { NULL, NULL },
	{ NULL, NULL }, { NULL, NULL }, { NULL, NULL }, { NULL, NULL },
	{ NULL, NULL }, { NULL, NULL }, { NULL, NULL }, { NULL, NULL },
	{ NULL, NULL }, { NULL, NULL }, { NULL, NULL }, { NULL, NULL },
	{ NULL, NULL }, { NULL, NULL }, { NULL, NULL }, { NULL, NULL },
	{ NULL, NULL }, { NULL, NULL }, { NULL, NULL }, { NULL, NULL },
	{ NULL, NULL }, { NULL, NULL }, { NULL, NULL }, { NULL, NULL },
	{ NULL, NULL }, { NULL, NULL }, { NULL, NULL }, { NULL, NULL }
};

UN_DLL_LOCAL
static const struct unicode_nameannot nullarray2[] = {
	{ NULL, NULL }, { NULL, NULL }, { NULL, NULL }, { NULL, NULL },
	{ NULL, NULL }, { NULL, NULL }, { NULL, NULL }, { NULL, NULL },
	{ NULL, NULL }, { NULL, NULL }, { NULL, NULL }, { NULL, NULL },
	{ NULL, NULL }, { NULL, NULL }, { NULL, NULL }, { NULL, NULL },
	{ NULL, NULL }, { NULL, NULL }, { NULL, NULL }, { NULL, NULL },
	{ NULL, NULL }, { NULL, NULL }, { NULL, NULL }, { NULL, NULL },
	{ NULL, NULL }, { NULL, NULL }, { NULL, NULL }, { NULL, NULL },
	{ NULL, NULL }, { NULL, NULL }, { NULL, NULL }, { NULL, NULL },
	{ NULL, NULL }, { NULL, NULL }, { NULL, NULL }, { NULL, NULL },
	{ NULL, NULL }, { NULL, NULL }, { NULL, NULL }, { NULL, NULL },
	{ NULL, NULL }, { NULL, NULL }, { NULL, NULL }, { NULL, NULL },
	{ NULL, NULL }, { NULL, NULL }, { NULL, NULL }, { NULL, NULL },
	{ NULL, NULL }, { NULL, NULL }, { NULL, NULL }, { NULL, NULL },
	{ NULL, NULL }, { NULL, NULL }, { NULL, NULL }, { NULL, NULL },
	{ NULL, NULL }, { NULL, NULL }, { NULL, NULL }, { NULL, NULL },
	{ NULL, NULL }, { NULL, NULL }, { NULL, NULL }, { NULL, NULL },
	{ NULL, NULL }, { NULL, NULL }, { NULL, NULL }, { NULL, NULL },
	{ NULL, NULL }, { NULL, NULL }, { NULL, NULL }, { NULL, NULL },
	{ NULL, NULL }, { NULL, NULL }, { NULL, NULL }, { NULL, NULL },
	{ NULL, NULL }, { NULL, NULL }, { NULL, NULL }, { NULL, NULL },
	{ NULL, NULL }, { NULL, NULL }, { NULL, NULL }, { NULL, NULL },
	{ NULL, NULL }, { NULL, NULL }, { NULL, NULL }, { NULL, NULL },
	{ NULL, NULL }, { NULL, NULL }, { NULL, NULL }, { NULL, NULL },
	{ NULL, NULL }, { NULL, NULL }, { NULL, NULL }, { NULL, NULL },
	{ NULL, NULL }, { NULL, NULL }, { NULL, NULL }, { NULL, NULL },
	{ NULL, NULL }, { NULL, NULL }, { NULL, NULL }, { NULL, NULL },
	{ NULL, NULL }, { NULL, NULL }, { NULL, NULL }, { NULL, NULL },
	{ NULL, NULL }, { NULL, NULL }, { NULL, NULL }, { NULL, NULL },
	{ NULL, NULL }, { NULL, NULL }, { NULL, NULL }, { NULL, NULL },
	{ NULL, NULL }, { NULL, NULL }, { NULL, NULL }, { NULL, NULL },
	{ NULL, NULL }, { NULL, NULL }, { NULL, NULL }, { NULL, NULL },
	{ NULL, NULL }, { NULL, NULL }, { NULL, NULL }, { NULL, NULL },
	{ NULL, NULL }, { NULL, NULL }, { NULL, NULL }, { NULL, NULL },
	{ NULL, NULL }, { NULL, NULL }, { NULL, NULL }, { NULL, NULL },
	{ NULL, NULL }, { NULL, NULL }, { NULL, NULL }, { NULL, NULL },
	{ NULL, NULL }, { NULL, NULL }, { NULL, NULL }, { NULL, NULL },
	{ NULL, NULL }, { NULL, NULL }, { NULL, NULL }, { NULL, NULL },
	{ NULL, NULL }, { NULL, NULL }, { NULL, NULL }, { NULL, NULL },
	{ NULL, NULL }, { NULL, NULL }, { NULL, NULL }, { NULL, NULL },
	{ NULL, NULL }, { NULL, NULL }, { NULL, NULL }, { NULL, NULL },
	{ NULL, NULL }, { NULL, NULL }, { NULL, NULL }, { NULL, NULL },
	{ NULL, NULL }, { NULL, NULL }, { NULL, NULL }, { NULL, NULL },
	{ NULL, NULL }, { NULL, NULL }, { NULL, NULL }, { NULL, NULL },
	{ NULL, NULL }, { NULL, NULL }, { NULL, NULL }, { NULL, NULL },
	{ NULL, NULL }, { NULL, NULL }, { NULL, NULL }, { NULL, NULL },
	{ NULL, NULL }, { NULL, NULL }, { NULL, NULL }, { NULL, NULL },
	{ NULL, NULL }, { NULL, NULL }, { NULL, NULL }, { NULL, NULL },
	{ NULL, NULL }, { NULL, NULL }, { NULL, NULL }, { NULL, NULL },
	{ NULL, NULL }, { NULL, NULL }, { NULL, NULL }, { NULL, NULL },
	{ NULL, NULL }, { NULL, NULL }, { NULL, NULL }, { NULL, NULL },
	{ NULL, NULL }, { NULL, NULL }, { NULL, NULL }, { NULL, NULL },
	{ NULL, NULL }, { NULL, NULL }, { NULL, NULL }, { NULL, NULL },
	{ NULL, NULL }, { NULL, NULL }, { NULL, NULL }, { NULL, NULL },
	{ NULL, NULL }, { NULL, NULL }, { NULL, NULL }, { NULL, NULL },
	{ NULL, NULL }, { NULL, NULL }, { NULL, NULL }, { NULL, NULL },
	{ NULL, NULL }, { NULL, NULL }, { NULL, NULL }, { NULL, NULL },
	{ NULL, NULL }, { NULL, NULL }, { NULL, NULL }, { NULL, NULL },
	{ NULL, NULL }, { NULL, NULL }, { NULL, NULL }, { NULL, NULL },
	{ NULL, NULL }, { NULL, NULL }, { NULL, NULL }, { NULL, NULL },
	{ NULL, NULL }, { NULL, NULL }, { NULL, NULL }, { NULL, NULL },
	{ NULL, NULL }, { NULL, NULL }, { NULL, NULL }, { NULL, NULL },
	{ NULL, NULL }, { NULL, NULL }, { NULL, NULL }, { NULL, NULL },
	{ NULL, NULL }, { NULL, NULL }, { NULL, NULL }, { NULL, NULL },
	{ NULL, NULL }, { NULL, NULL }, { NULL, NULL }, { NULL, NULL },
	{ NULL, NULL }, { NULL, NULL },
	{ NULL, "	* the value ?FFFE is guaranteed not to be a Unicode character at all" },
	{ NULL, "	* the value ?FFFF is guaranteed not to be a Unicode character at all" },
};

UN_DLL_LOCAL
static const struct unicode_nameannot * const nullnullarray[] = {
	nullarray, nullarray, nullarray, nullarray, nullarray, nullarray, nullarray, nullarray,
	nullarray, nullarray, nullarray, nullarray, nullarray, nullarray, nullarray, nullarray,
	nullarray, nullarray, nullarray, nullarray, nullarray, nullarray, nullarray, nullarray,
	nullarray, nullarray, nullarray, nullarray, nullarray, nullarray, nullarray, nullarray,
	nullarray, nullarray, nullarray, nullarray, nullarray, nullarray, nullarray, nullarray,
	nullarray, nullarray, nullarray, nullarray, nullarray, nullarray, nullarray, nullarray,
	nullarray, nullarray, nullarray, nullarray, nullarray, nullarray, nullarray, nullarray,
	nullarray, nullarray, nullarray, nullarray, nullarray, nullarray, nullarray, nullarray,
	nullarray, nullarray, nullarray, nullarray, nullarray, nullarray, nullarray, nullarray,
	nullarray, nullarray, nullarray, nullarray, nullarray, nullarray, nullarray, nullarray,
	nullarray, nullarray, nullarray, nullarray, nullarray, nullarray, nullarray, nullarray,
	nullarray, nullarray, nullarray, nullarray, nullarray, nullarray, nullarray, nullarray,
	nullarray, nullarray, nullarray, nullarray, nullarray, nullarray, nullarray, nullarray,
	nullarray, nullarray, nullarray, nullarray, nullarray, nullarray, nullarray, nullarray,
	nullarray, nullarray, nullarray, nullarray, nullarray, nullarray, nullarray, nullarray,
	nullarray, nullarray, nullarray, nullarray, nullarray, nullarray, nullarray, nullarray,
	nullarray, nullarray, nullarray, nullarray, nullarray, nullarray, nullarray, nullarray,
	nullarray, nullarray, nullarray, nullarray, nullarray, nullarray, nullarray, nullarray,
	nullarray, nullarray, nullarray, nullarray, nullarray, nullarray, nullarray, nullarray,
	nullarray, nullarray, nullarray, nullarray, nullarray, nullarray, nullarray, nullarray,
	nullarray, nullarray, nullarray, nullarray, nullarray, nullarray, nullarray, nullarray,
	nullarray, nullarray, nullarray, nullarray, nullarray, nullarray, nullarray, nullarray,
	nullarray, nullarray, nullarray, nullarray, nullarray, nullarray, nullarray, nullarray,
	nullarray, nullarray, nullarray, nullarray, nullarray, nullarray, nullarray, nullarray,
	nullarray, nullarray, nullarray, nullarray, nullarray, nullarray, nullarray, nullarray,
	nullarray, nullarray, nullarray, nullarray, nullarray, nullarray, nullarray, nullarray,
	nullarray, nullarray, nullarray, nullarray, nullarray, nullarray, nullarray, nullarray,
	nullarray, nullarray, nullarray, nullarray, nullarray, nullarray, nullarray, nullarray,
	nullarray, nullarray, nullarray, nullarray, nullarray, nullarray, nullarray, nullarray,
	nullarray, nullarray, nullarray, nullarray, nullarray, nullarray, nullarray, nullarray,
	nullarray, nullarray, nullarray, nullarray, nullarray, nullarray, nullarray, nullarray,
	nullarray, nullarray, nullarray, nullarray, nullarray, nullarray, nullarray, nullarray,
	nullarray, nullarray, nullarray, nullarray, nullarray, nullarray, nullarray, nullarray2
};

UN_DLL_EXPORT
const struct unicode_block UnicodeBlock[] = {
	{ 0x0, 0x7f, "C0 Controls and Basic Latin (Basic Latin)" },
	{ 0x80, 0xff, "C1 Controls and Latin-1 Supplement (Latin-1 Supplement)" },
	{ 0x100, 0x17f, "Latin Extended-A" },
	{ 0x180, 0x24f, "Latin Extended-B" },
	{ 0x250, 0x2af, "IPA Extensions" },
	{ 0x2b0, 0x2ff, "Spacing Modifier Letters" },
	{ 0x300, 0x36f, "Combining Diacritical Marks" },
	{ 0x370, 0x3ff, "Greek and Coptic" },
	{ 0x400, 0x4ff, "Cyrillic" },
	{ 0x500, 0x52f, "Cyrillic Supplement" },
	{ 0x530, 0x58f, "Armenian" },
	{ 0x590, 0x5ff, "Hebrew" },
	{ 0x600, 0x6ff, "Arabic" },
	{ 0x700, 0x74f, "Syriac" },
	{ 0x750, 0x77f, "Arabic Supplement" },
	{ 0x780, 0x7bf, "Thaana" },
	{ 0x7c0, 0x7ff, "NKo" },
	{ 0x800, 0x83f, "Samaritan" },
	{ 0x840, 0x85f, "Mandaic" },
	{ 0x860, 0x86f, "Syriac Supplement" },
	{ 0x870, 0x89f, "Arabic Extended-B" },
	{ 0x8a0, 0x8ff, "Arabic Extended-A" },
	{ 0x900, 0x97f, "Devanagari" },
	{ 0x980, 0x9ff, "Bengali" },
	{ 0xa00, 0xa7f, "Gurmukhi" },
	{ 0xa80, 0xaff, "Gujarati" },
	{ 0xb00, 0xb7f, "Oriya" },
	{ 0xb80, 0xbff, "Tamil" },
	{ 0xc00, 0xc7f, "Telugu" },
	{ 0xc80, 0xcff, "Kannada" },
	{ 0xd00, 0xd7f, "Malayalam" },
	{ 0xd80, 0xdff, "Sinhala" },
	{ 0xe00, 0xe7f, "Thai" },
	{ 0xe80, 0xeff, "Lao" },
	{ 0xf00, 0xfff, "Tibetan" },
	{ 0x1000, 0x109f, "Myanmar" },
	{ 0x10a0, 0x10ff, "Georgian" },
	{ 0x1100, 0x11ff, "Hangul Jamo" },
	{ 0x1200, 0x137f, "Ethiopic" },
	{ 0x1380, 0x139f, "Ethiopic Supplement" },
	{ 0x13a0, 0x13ff, "Cherokee" },
	{ 0x1400, 0x167f, "Unified Canadian Aboriginal Syllabics" },
	{ 0x1680, 0x169f, "Ogham" },
	{ 0x16a0, 0x16ff, "Runic" },
	{ 0x1700, 0x171f, "Tagalog" },
	{ 0x1720, 0x173f, "Hanunoo" },
	{ 0x1740, 0x175f, "Buhid" },
	{ 0x1760, 0x177f, "Tagbanwa" },
	{ 0x1780, 0x17ff, "Khmer" },
	{ 0x1800, 0x18af, "Mongolian" },
	{ 0x18b0, 0x18ff, "Unified Canadian Aboriginal Syllabics Extended" },
	{ 0x1900, 0x194f, "Limbu" },
	{ 0x1950, 0x197f, "Tai Le" },
	{ 0x1980, 0x19df, "New Tai Lue" },
	{ 0x19e0, 0x19ff, "Khmer Symbols" },
	{ 0x1a00, 0x1a1f, "Buginese" },
	{ 0x1a20, 0x1aaf, "Tai Tham" },
	{ 0x1ab0, 0x1aff, "Combining Diacritical Marks Extended" },
	{ 0x1b00, 0x1b7f, "Balinese" },
	{ 0x1b80, 0x1bbf, "Sundanese" },
	{ 0x1bc0, 0x1bff, "Batak" },
	{ 0x1c00, 0x1c4f, "Lepcha" },
	{ 0x1c50, 0x1c7f, "Ol Chiki" },
	{ 0x1c80, 0x1c8f, "Cyrillic Extended-C" },
	{ 0x1c90, 0x1cbf, "Georgian Extended" },
	{ 0x1cc0, 0x1ccf, "Sundanese Supplement" },
	{ 0x1cd0, 0x1cff, "Vedic Extensions" },
	{ 0x1d00, 0x1d7f, "Phonetic Extensions" },
	{ 0x1d80, 0x1dbf, "Phonetic Extensions Supplement" },
	{ 0x1dc0, 0x1dff, "Combining Diacritical Marks Supplement" },
	{ 0x1e00, 0x1eff, "Latin Extended Additional" },
	{ 0x1f00, 0x1fff, "Greek Extended" },
	{ 0x2000, 0x206f, "General Punctuation" },
	{ 0x2070, 0x209f, "Superscripts and Subscripts" },
	{ 0x20a0, 0x20cf, "Currency Symbols" },
	{ 0x20d0, 0x20ff, "Combining Diacritical Marks for Symbols" },
	{ 0x2100, 0x214f, "Letterlike Symbols" },
	{ 0x2150, 0x218f, "Number Forms" },
	{ 0x2190, 0x21ff, "Arrows" },
	{ 0x2200, 0x22ff, "Mathematical Operators" },
	{ 0x2300, 0x23ff, "Miscellaneous Technical" },
	{ 0x2400, 0x243f, "Control Pictures" },
	{ 0x2440, 0x245f, "Optical Character Recognition" },
	{ 0x2460, 0x24ff, "Enclosed Alphanumerics" },
	{ 0x2500, 0x257f, "Box Drawing" },
	{ 0x2580, 0x259f, "Block Elements" },
	{ 0x25a0, 0x25ff, "Geometric Shapes" },
	{ 0x2600, 0x26ff, "Miscellaneous Symbols" },
	{ 0x2700, 0x27bf, "Dingbats" },
	{ 0x27c0, 0x27ef, "Miscellaneous Mathematical Symbols-A" },
	{ 0x27f0, 0x27ff, "Supplemental Arrows-A" },
	{ 0x2800, 0x28ff, "Braille Patterns" },
	{ 0x2900, 0x297f, "Supplemental Arrows-B" },
	{ 0x2980, 0x29ff, "Miscellaneous Mathematical Symbols-B" },
	{ 0x2a00, 0x2aff, "Supplemental Mathematical Operators" },
	{ 0x2b00, 0x2bff, "Miscellaneous Symbols and Arrows" },
	{ 0x2c00, 0x2c5f, "Glagolitic" },
	{ 0x2c60, 0x2c7f, "Latin Extended-C" },
	{ 0x2c80, 0x2cff, "Coptic" },
	{ 0x2d00, 0x2d2f, "Georgian Supplement" },
	{ 0x2d30, 0x2d7f, "Tifinagh" },
	{ 0x2d80, 0x2ddf, "Ethiopic Extended" },
	{ 0x2de0, 0x2dff, "Cyrillic Extended-A" },
	{ 0x2e00, 0x2e7f, "Supplemental Punctuation" },
	{ 0x2e80, 0x2eff, "CJK Radicals Supplement" },
	{ 0x2f00, 0x2fdf, "Kangxi Radicals" },
	{ 0x2ff0, 0x2fff, "Ideographic Description Characters" },
	{ 0x3000, 0x303f, "CJK Symbols and Punctuation" },
	{ 0x3040, 0x309f, "Hiragana" },
	{ 0x30a0, 0x30ff, "Katakana" },
	{ 0x3100, 0x312f, "Bopomofo" },
	{ 0x3130, 0x318f, "Hangul Compatibility Jamo" },
	{ 0x3190, 0x319f, "Kanbun" },
	{ 0x31a0, 0x31bf, "Bopomofo Extended" },
	{ 0x31c0, 0x31ef, "CJK Strokes" },
	{ 0x31f0, 0x31ff, "Katakana Phonetic Extensions" },
	{ 0x3200, 0x32ff, "Enclosed CJK Letters and Months" },
	{ 0x3300, 0x33ff, "CJK Compatibility" },
	{ 0x3400, 0x4dbf, "CJK Unified Ideographs Extension A" },
	{ 0x4dc0, 0x4dff, "Yijing Hexagram Symbols" },
	{ 0x4e00, 0x9fff, "CJK Unified Ideographs" },
	{ 0xa000, 0xa48f, "Yi Syllables" },
	{ 0xa490, 0xa4cf, "Yi Radicals" },
	{ 0xa4d0, 0xa4ff, "Lisu" },
	{ 0xa500, 0xa63f, "Vai" },
	{ 0xa640, 0xa69f, "Cyrillic Extended-B" },
	{ 0xa6a0, 0xa6ff, "Bamum" },
	{ 0xa700, 0xa71f, "Modifier Tone Letters" },
	{ 0xa720, 0xa7ff, "Latin Extended-D" },
	{ 0xa800, 0xa82f, "Syloti Nagri" },
	{ 0xa830, 0xa83f, "Common Indic Number Forms" },
	{ 0xa840, 0xa87f, "Phags-pa" },
	{ 0xa880, 0xa8df, "Saurashtra" },
	{ 0xa8e0, 0xa8ff, "Devanagari Extended" },
	{ 0xa900, 0xa92f, "Kayah Li" },
	{ 0xa930, 0xa95f, "Rejang" },
	{ 0xa960, 0xa97f, "Hangul Jamo Extended-A" },
	{ 0xa980, 0xa9df, "Javanese" },
	{ 0xa9e0, 0xa9ff, "Myanmar Extended-B" },
	{ 0xaa00, 0xaa5f, "Cham" },
	{ 0xaa60, 0xaa7f, "Myanmar Extended-A" },
	{ 0xaa80, 0xaadf, "Tai Viet" },
	{ 0xaae0, 0xaaff, "Meetei Mayek Extensions" },
	{ 0xab00, 0xab2f, "Ethiopic Extended-A" },
	{ 0xab30, 0xab6f, "Latin Extended-E" },
	{ 0xab70, 0xabbf, "Cherokee Supplement" },
	{ 0xabc0, 0xabff, "Meetei Mayek" },
	{ 0xac00, 0xd7a3, "Hangul Syllables" },
	{ 0xd7b0, 0xd7ff, "Hangul Jamo Extended-B" },
	{ 0xd800, 0xdb7f, "High Surrogates" },
	{ 0xdb80, 0xdbff, "High Private Use Surrogates" },
	{ 0xdc00, 0xdfff, "Low Surrogates" },
	{ 0xe000, 0xf8ff, "Private Use Area" },
	{ 0xf900, 0xfaff, "CJK Compatibility Ideographs" },
	{ 0xfb00, 0xfb4f, "Alphabetic Presentation Forms" },
	{ 0xfb50, 0xfdff, "Arabic Presentation Forms-A" },
	{ 0xfe00, 0xfe0f, "Variation Selectors" },
	{ 0xfe10, 0xfe1f, "Vertical Forms" },
	{ 0xfe20, 0xfe2f, "Combining Half Marks" },
	{ 0xfe30, 0xfe4f, "CJK Compatibility Forms" },
	{ 0xfe50, 0xfe6f, "Small Form Variants" },
	{ 0xfe70, 0xfeff, "Arabic Presentation Forms-B" },
	{ 0xff00, 0xffef, "Halfwidth and Fullwidth Forms" },
	{ 0xfff0, 0xffff, "Specials" },
	{ 0x10000, 0x1007f, "Linear B Syllabary" },
	{ 0x10080, 0x100ff, "Linear B Ideograms" },
	{ 0x10100, 0x1013f, "Aegean Numbers" },
	{ 0x10140, 0x1018f, "Ancient Greek Numbers" },
	{ 0x10190, 0x101cf, "Ancient Symbols" },
	{ 0x101d0, 0x101ff, "Phaistos Disc" },
	{ 0x10280, 0x1029f, "Lycian" },
	{ 0x102a0, 0x102df, "Carian" },
	{ 0x102e0, 0x102ff, "Coptic Epact Numbers" },
	{ 0x10300, 0x1032f, "Old Italic" },
	{ 0x10330, 0x1034f, "Gothic" },
	{ 0x10350, 0x1037f, "Old Permic" },
	{ 0x10380, 0x1039f, "Ugaritic" },
	{ 0x103a0, 0x103df, "Old Persian" },
	{ 0x10400, 0x1044f, "Deseret" },
	{ 0x10450, 0x1047f, "Shavian" },
	{ 0x10480, 0x104af, "Osmanya" },
	{ 0x104b0, 0x104ff, "Osage" },
	{ 0x10500, 0x1052f, "Elbasan" },
	{ 0x10530, 0x1056f, "Caucasian Albanian" },
	{ 0x10570, 0x105bf, "Vithkuqi" },
	{ 0x105c0, 0x105ff, "Todhri" },
	{ 0x10600, 0x1077f, "Linear A" },
	{ 0x10780, 0x107bf, "Latin Extended-F" },
	{ 0x10800, 0x1083f, "Cypriot Syllabary" },
	{ 0x10840, 0x1085f, "Imperial Aramaic" },
	{ 0x10860, 0x1087f, "Palmyrene" },
	{ 0x10880, 0x108af, "Nabataean" },
	{ 0x108e0, 0x108ff, "Hatran" },
	{ 0x10900, 0x1091f, "Phoenician" },
	{ 0x10920, 0x1093f, "Lydian" },
	{ 0x10980, 0x1099f, "Meroitic Hieroglyphs" },
	{ 0x109a0, 0x109ff, "Meroitic Cursive" },
	{ 0x10a00, 0x10a5f, "Kharoshthi" },
	{ 0x10a60, 0x10a7f, "Old South Arabian" },
	{ 0x10a80, 0x10a9f, "Old North Arabian" },
	{ 0x10ac0, 0x10aff, "Manichaean" },
	{ 0x10b00, 0x10b3f, "Avestan" },
	{ 0x10b40, 0x10b5f, "Inscriptional Parthian" },
	{ 0x10b60, 0x10b7f, "Inscriptional Pahlavi" },
	{ 0x10b80, 0x10baf, "Psalter Pahlavi" },
	{ 0x10c00, 0x10c4f, "Old Turkic" },
	{ 0x10c80, 0x10cff, "Old Hungarian" },
	{ 0x10d00, 0x10d3f, "Hanifi Rohingya" },
	{ 0x10d40, 0x10d8f, "Garay" },
	{ 0x10e60, 0x10e7f, "Rumi Numeral Symbols" },
	{ 0x10e80, 0x10ebf, "Yezidi" },
	{ 0x10ec0, 0x10eff, "Arabic Extended-C" },
	{ 0x10f00, 0x10f2f, "Old Sogdian" },
	{ 0x10f30, 0x10f6f, "Sogdian" },
	{ 0x10f70, 0x10faf, "Old Uyghur" },
	{ 0x10fb0, 0x10fdf, "Chorasmian" },
	{ 0x10fe0, 0x10fff, "Elymaic" },
	{ 0x11000, 0x1107f, "Brahmi" },
	{ 0x11080, 0x110cf, "Kaithi" },
	{ 0x110d0, 0x110ff, "Sora Sompeng" },
	{ 0x11100, 0x1114f, "Chakma" },
	{ 0x11150, 0x1117f, "Mahajani" },
	{ 0x11180, 0x111df, "Sharada" },
	{ 0x111e0, 0x111ff, "Sinhala Archaic Numbers" },
	{ 0x11200, 0x1124f, "Khojki" },
	{ 0x11280, 0x112af, "Multani" },
	{ 0x112b0, 0x112ff, "Khudawadi" },
	{ 0x11300, 0x1137f, "Grantha" },
	{ 0x11380, 0x113ff, "Tulu-Tigalari" },
	{ 0x11400, 0x1147f, "Newa" },
	{ 0x11480, 0x114df, "Tirhuta" },
	{ 0x11580, 0x115ff, "Siddham" },
	{ 0x11600, 0x1165f, "Modi" },
	{ 0x11660, 0x1167f, "Mongolian Supplement" },
	{ 0x11680, 0x116cf, "Takri" },
	{ 0x116d0, 0x116ff, "Myanmar Extended-C" },
	{ 0x11700, 0x1174f, "Ahom" },
	{ 0x11800, 0x1184f, "Dogra" },
	{ 0x118a0, 0x118ff, "Warang Citi" },
	{ 0x11900, 0x1195f, "Dives Akuru" },
	{ 0x119a0, 0x119ff, "Nandinagari" },
	{ 0x11a00, 0x11a4f, "Zanabazar Square" },
	{ 0x11a50, 0x11aaf, "Soyombo" },
	{ 0x11ab0, 0x11abf, "Unified Canadian Aboriginal Syllabics Extended-A" },
	{ 0x11ac0, 0x11aff, "Pau Cin Hau" },
	{ 0x11b00, 0x11b5f, "Devanagari Extended-A" },
	{ 0x11bc0, 0x11bff, "Sunuwar" },
	{ 0x11c00, 0x11c6f, "Bhaiksuki" },
	{ 0x11c70, 0x11cbf, "Marchen" },
	{ 0x11d00, 0x11d5f, "Masaram Gondi" },
	{ 0x11d60, 0x11daf, "Gunjala Gondi" },
	{ 0x11ee0, 0x11eff, "Makasar" },
	{ 0x11f00, 0x11f5f, "Kawi" },
	{ 0x11fb0, 0x11fbf, "Lisu Supplement" },
	{ 0x11fc0, 0x11fff, "Tamil Supplement" },
	{ 0x12000, 0x123ff, "Cuneiform" },
	{ 0x12400, 0x1247f, "Cuneiform Numbers and Punctuation" },
	{ 0x12480, 0x1254f, "Early Dynastic Cuneiform" },
	{ 0x12f90, 0x12fff, "Cypro-Minoan" },
	{ 0x13000, 0x1342f, "Egyptian Hieroglyphs" },
	{ 0x13430, 0x1345f, "Egyptian Hieroglyph Format Controls" },
	{ 0x13460, 0x143ff, "Egyptian Hieroglyphs Extended-A" },
	{ 0x14400, 0x1467f, "Anatolian Hieroglyphs" },
	{ 0x16100, 0x1613f, "Gurung Khema" },
	{ 0x16800, 0x16a3f, "Bamum Supplement" },
	{ 0x16a40, 0x16a6f, "Mro" },
	{ 0x16a70, 0x16acf, "Tangsa" },
	{ 0x16ad0, 0x16aff, "Bassa Vah" },
	{ 0x16b00, 0x16b8f, "Pahawh Hmong" },
	{ 0x16d40, 0x16d7f, "Kirat Rai" },
	{ 0x16e40, 0x16e9f, "Medefaidrin" },
	{ 0x16f00, 0x16f9f, "Miao" },
	{ 0x16fe0, 0x16fff, "Ideographic Symbols and Punctuation" },
	{ 0x17000, 0x187f7, "Tangut" },
	{ 0x18800, 0x18aff, "Tangut Components" },
	{ 0x18b00, 0x18cff, "Khitan Small Script" },
	{ 0x18d00, 0x18d08, "Tangut Supplement" },
	{ 0x1aff0, 0x1afff, "Kana Extended-B" },
	{ 0x1b000, 0x1b0ff, "Kana Supplement" },
	{ 0x1b100, 0x1b12f, "Kana Extended-A" },
	{ 0x1b130, 0x1b16f, "Small Kana Extension" },
	{ 0x1b170, 0x1b2ff, "Nushu" },
	{ 0x1bc00, 0x1bc9f, "Duployan" },
	{ 0x1bca0, 0x1bcaf, "Shorthand Format Controls" },
	{ 0x1cc00, 0x1cebf, "Symbols for Legacy Computing Supplement" },
	{ 0x1cf00, 0x1cfcf, "Znamenny Musical Notation" },
	{ 0x1d000, 0x1d0ff, "Byzantine Musical Symbols" },
	{ 0x1d100, 0x1d1ff, "Musical Symbols" },
	{ 0x1d200, 0x1d24f, "Ancient Greek Musical Notation" },
	{ 0x1d2c0, 0x1d2df, "Kaktovik Numerals" },
	{ 0x1d2e0, 0x1d2ff, "Mayan Numerals" },
	{ 0x1d300, 0x1d35f, "Tai Xuan Jing Symbols" },
	{ 0x1d360, 0x1d37f, "Counting Rod Numerals" },
	{ 0x1d400, 0x1d7ff, "Mathematical Alphanumeric Symbols" },
	{ 0x1d800, 0x1daaf, "Sutton SignWriting" },
	{ 0x1df00, 0x1dfff, "Latin Extended-G" },
	{ 0x1e000, 0x1e02f, "Glagolitic Supplement" },
	{ 0x1e030, 0x1e08f, "Cyrillic Extended-D" },
	{ 0x1e100, 0x1e14f, "Nyiakeng Puachue Hmong" },
	{ 0x1e290, 0x1e2bf, "Toto" },
	{ 0x1e2c0, 0x1e2ff, "Wancho" },
	{ 0x1e4d0, 0x1e4ff, "Nag Mundari" },
	{ 0x1e5d0, 0x1e5ff, "Ol Onal" },
	{ 0x1e7e0, 0x1e7ff, "Ethiopic Extended-B" },
	{ 0x1e800, 0x1e8df, "Mende Kikakui" },
	{ 0x1e900, 0x1e95f, "Adlam" },
	{ 0x1ec70, 0x1ecbf, "Indic Siyaq Numbers" },
	{ 0x1ed00, 0x1ed4f, "Ottoman Siyaq Numbers" },
	{ 0x1ee00, 0x1eeff, "Arabic Mathematical Alphabetic Symbols" },
	{ 0x1f000, 0x1f02f, "Mahjong Tiles" },
	{ 0x1f030, 0x1f09f, "Domino Tiles" },
	{ 0x1f0a0, 0x1f0ff, "Playing Cards" },
	{ 0x1f100, 0x1f1ff, "Enclosed Alphanumeric Supplement" },
	{ 0x1f200, 0x1f2ff, "Enclosed Ideographic Supplement" },
	{ 0x1f300, 0x1f5ff, "Miscellaneous Symbols and Pictographs" },
	{ 0x1f600, 0x1f64f, "Emoticons" },
	{ 0x1f650, 0x1f67f, "Ornamental Dingbats" },
	{ 0x1f680, 0x1f6ff, "Transport and Map Symbols" },
	{ 0x1f700, 0x1f77f, "Alchemical Symbols" },
	{ 0x1f780, 0x1f7ff, "Geometric Shapes Extended" },
	{ 0x1f800, 0x1f8ff, "Supplemental Arrows-C" },
	{ 0x1f900, 0x1f9ff, "Supplemental Symbols and Pictographs" },
	{ 0x1fa00, 0x1fa6f, "Chess Symbols" },
	{ 0x1fa70, 0x1faff, "Symbols and Pictographs Extended-A" },
	{ 0x1fb00, 0x1fbff, "Symbols for Legacy Computing" },
	{ 0x1ff80, 0x1ffff, "Unassigned" },
	{ 0x20000, 0x2a6df, "CJK Unified Ideographs Extension B" },
	{ 0x2a700, 0x2b739, "CJK Unified Ideographs Extension C" },
	{ 0x2b740, 0x2b81d, "CJK Unified Ideographs Extension D" },
	{ 0x2b820, 0x2cea1, "CJK Unified Ideographs Extension E" },
	{ 0x2ceb0, 0x2ebe0, "CJK Unified Ideographs Extension F" },
	{ 0x2ebf0, 0x2ee5d, "CJK Unified Ideographs Extension I" },
	{ 0x2f800, 0x2fa1f, "CJK Compatibility Ideographs Supplement" },
	{ 0x2ff80, 0x2ffff, "Unassigned" },
	{ 0x30000, 0x3134a, "CJK Unified Ideographs Extension G" },
	{ 0x31350, 0x323af, "CJK Unified Ideographs Extension H" },
	{ 0x3ff80, 0x3ffff, "Unassigned" },
	{ 0x4ff80, 0x4ffff, "Unassigned" },
	{ 0x5ff80, 0x5ffff, "Unassigned" },
	{ 0x6ff80, 0x6ffff, "Unassigned" },
	{ 0x7ff80, 0x7ffff, "Unassigned" },
	{ 0x8ff80, 0x8ffff, "Unassigned" },
	{ 0x9ff80, 0x9ffff, "Unassigned" },
	{ 0xaff80, 0xaffff, "Unassigned" },
	{ 0xbff80, 0xbffff, "Unassigned" },
	{ 0xcff80, 0xcffff, "Unassigned" },
	{ 0xdff80, 0xdffff, "Unassigned" },
	{ 0xe0000, 0xe007f, "Tags" },
	{ 0xe0100, 0xe01ef, "Variation Selectors Supplement" },
	{ 0xeff80, 0xeffff, "Unassigned" },
	{ 0xfff80, 0xfffff, "Supplementary Private Use Area-A" },
	{ 0x10ff80, 0x10ffff, "Supplementary Private Use Area-B" }
};

UN_DLL_LOCAL
static const struct unicode_nameannot una_00_00[] = {
/* 0000 */ { NULL,"	= NULL"},
/* 0001 */ { NULL,"	= START OF HEADING"},
/* 0002 */ { NULL,"	= START OF TEXT"},
/* 0003 */ { NULL,"	= END OF TEXT"},
/* 0004 */ { NULL,"	= END OF TRANSMISSION"},
/* 0005 */ { NULL,"	= ENQUIRY"},
/* 0006 */ { NULL,"	= ACKNOWLEDGE"},
/* 0007 */ { NULL,"	= BELL"},
/* 0008 */ { NULL,"	= BACKSPACE"},
/* 0009 */ { NULL,"	= CHARACTER TABULATION\n"
	"	= horizontal tabulation (HT)\n"
	"	= tab"},
/* 000A */ { NULL,"	= LINE FEED (LF)\n"
	"	= new line (NL)\n"
	"	= end of line (EOL)"},
/* 000B */ { NULL,"	= LINE TABULATION\n"
	"	= vertical tabulation (VT)"},
/* 000C */ { NULL,"	= FORM FEED (FF)"},
/* 000D */ { NULL,"	= CARRIAGE RETURN (CR)"},
/* 000E */ { NULL,"	= SHIFT OUT\n"
	"	* known as LOCKING-SHIFT ONE in 8-bit environments"},
/* 000F */ { NULL,"	= SHIFT IN\n"
	"	* known as LOCKING-SHIFT ZERO in 8-bit environments"},
/* 0010 */ { NULL,"	= DATA LINK ESCAPE"},
/* 0011 */ { NULL,"	= DEVICE CONTROL ONE"},
/* 0012 */ { NULL,"	= DEVICE CONTROL TWO"},
/* 0013 */ { NULL,"	= DEVICE CONTROL THREE"},
/* 0014 */ { NULL,"	= DEVICE CONTROL FOUR"},
/* 0015 */ { NULL,"	= NEGATIVE ACKNOWLEDGE"},
/* 0016 */ { NULL,"	= SYNCHRONOUS IDLE"},
/* 0017 */ { NULL,"	= END OF TRANSMISSION BLOCK"},
/* 0018 */ { NULL,"	= CANCEL"},
/* 0019 */ { NULL,"	= END OF MEDIUM"},
/* 001A */ { NULL,"	= SUBSTITUTE\n"
	"	x (replacement character - FFFD)"},
/* 001B */ { NULL,"	= ESCAPE"},
/* 001C */ { NULL,"	= INFORMATION SEPARATOR FOUR\n"
	"	= file separator (FS)"},
/* 001D */ { NULL,"	= INFORMATION SEPARATOR THREE\n"
	"	= group separator (GS)"},
/* 001E */ { NULL,"	= INFORMATION SEPARATOR TWO\n"
	"	= record separator (RS)"},
/* 001F */ { NULL,"	= INFORMATION SEPARATOR ONE\n"
	"	= unit separator (US)"},
/* 0020 */ { "SPACE","	* sometimes considered a control code\n"
	"	* other space characters: 2000-200A\n"
	"	x (no-break space - 00A0)\n"
	"	x (zero width space - 200B)\n"
	"	x (narrow no-break space - 202F)\n"
	"	x (word joiner - 2060)\n"
	"	x (symbol for space - 2420)\n"
	"	x (blank symbol - 2422)\n"
	"	x (open box - 2423)\n"
	"	x (ideographic space - 3000)\n"
	"	x (zero width no-break space - FEFF)"},
/* 0021 */ { "EXCLAMATION MARK","	= factorial\n"
	"	= bang\n"
	"	x (inverted exclamation mark - 00A1)\n"
	"	x (latin letter retroflex click - 01C3)\n"
	"	x (double exclamation mark - 203C)\n"
	"	x (interrobang - 203D)\n"
	"	x (warning sign - 26A0)\n"
	"	x (heavy exclamation mark symbol - 2757)\n"
	"	x (heavy exclamation mark ornament - 2762)\n"
	"	x (medieval exclamation mark - 2E53)\n"
	"	x (modifier letter raised exclamation mark - A71D)"},
/* 0022 */ { "QUOTATION MARK","	= double quote\n"
	"	* neutral (vertical), used as opening or closing quotation mark\n"
	"	* preferred characters in English for paired quotation marks are 201C & 201D\n"
	"	* 05F4 is preferred for gershayim when writing Hebrew\n"
	"	x (modifier letter double prime - 02BA)\n"
	"	x (double acute accent - 02DD)\n"
	"	x (modifier letter double apostrophe - 02EE)\n"
	"	x (combining double acute accent - 030B)\n"
	"	x (combining double vertical line above - 030E)\n"
	"	x (hebrew punctuation gershayim - 05F4)\n"
	"	x (left double quotation mark - 201C)\n"
	"	x (right double quotation mark - 201D)\n"
	"	x (double prime - 2033)\n"
	"	x (ditto mark - 3003)"},
/* 0023 */ { "NUMBER SIGN","	= pound sign (weight)\n"
	"	= hashtag, hash\n"
	"	= crosshatch, octothorpe\n"
	"	* for denoting musical sharp 266F is preferred\n"
	"	x (l b bar symbol - 2114)\n"
	"	x (numero sign - 2116)\n"
	"	x (viewdata square - 2317)\n"
	"	x (music sharp sign - 266F)\n"
	"	x (equals sign and slanted parallel - 29E3)"},
/* 0024 */ { "DOLLAR SIGN","	= milréis, escudo\n"
	"	* used for many peso currencies in Latin America and elsewhere\n"
	"	* glyph may have one or two vertical bars\n"
	"	* other currency symbol characters start at 20A0\n"
	"	x (cent sign - 00A2)\n"
	"	x (currency sign - 00A4)\n"
	"	x (peso sign - 20B1)\n"
	"	x (heavy dollar sign - 1F4B2)"},
/* 0025 */ { "PERCENT SIGN","	x (arabic percent sign - 066A)\n"
	"	x (per mille sign - 2030)\n"
	"	x (per ten thousand sign - 2031)\n"
	"	x (commercial minus sign - 2052)"},
/* 0026 */ { "AMPERSAND","	= and\n"
	"	* originally derived from a ligature of 'e' and 't'\n"
	"	x (tironian sign et - 204A)\n"
	"	x (turned ampersand - 214B)\n"
	"	x (heavy ampersand ornament - 1F674)"},
/* 0027 */ { "APOSTROPHE","	= apostrophe-quote (1.0)\n"
	"	= single quote\n"
	"	= APL quote\n"
	"	* neutral (vertical) glyph with mixed usage\n"
	"	* 2019 is preferred for apostrophe\n"
	"	* preferred characters in English for paired quotation marks are 2018 & 2019\n"
	"	* 05F3 is preferred for geresh when writing Hebrew\n"
	"	x (modifier letter prime - 02B9)\n"
	"	x (modifier letter apostrophe - 02BC)\n"
	"	x (modifier letter vertical line - 02C8)\n"
	"	x (combining acute accent - 0301)\n"
	"	x (combining vertical line above - 030D)\n"
	"	x (hebrew punctuation geresh - 05F3)\n"
	"	x (left single quotation mark - 2018)\n"
	"	x (right single quotation mark - 2019)\n"
	"	x (prime - 2032)\n"
	"	x (latin small letter saltillo - A78C)"},
/* 0028 */ { "LEFT PARENTHESIS","	= opening parenthesis (1.0)"},
/* 0029 */ { "RIGHT PARENTHESIS","	= closing parenthesis (1.0)\n"
	"	* see discussion on semantics of paired bracketing characters"},
/* 002A */ { "ASTERISK","	= star\n"
	"	* can have five or six spokes\n"
	"	x (arabic five pointed star - 066D)\n"
	"	x (reference mark - 203B)\n"
	"	x (asterism - 2042)\n"
	"	x (low asterisk - 204E)\n"
	"	x (two asterisks aligned vertically - 2051)\n"
	"	x (combining asterisk above - 20F0)\n"
	"	x (asterisk operator - 2217)\n"
	"	x (sextile - 26B9)\n"
	"	x (heavy asterisk - 2731)\n"
	"	x (slavonic asterisk - A673)\n"
	"	x (medium six spoked asterisk - 1F7B6)"},
/* 002B */ { "PLUS SIGN","	x (modifier letter plus sign - 02D6)\n"
	"	x (minus sign - 2212)\n"
	"	x (heavy plus sign - 2795)\n"
	"	x (hebrew letter alternative plus sign - FB29)\n"
	"	x (light greek cross - 1F7A2)"},
/* 002C */ { "COMMA","	= the use as decimal or thousands separator is locale dependent\n"
	"	x (arabic comma - 060C)\n"
	"	x (arabic decimal separator - 066B)\n"
	"	x (single low-9 quotation mark - 201A)\n"
	"	x (hypodiastole - 2E12)\n"
	"	x (reversed comma - 2E41)\n"
	"	x (medieval comma - 2E4C)\n"
	"	x (ideographic comma - 3001)"},
/* 002D */ { "HYPHEN-MINUS","	= hyphen, dash\n"
	"	= minus sign\n"
	"	* used generically for hyphen, minus sign or en dash, all of which have dedicated alternatives\n"
	"	x (soft hyphen - 00AD)\n"
	"	x (modifier letter minus sign - 02D7)\n"
	"	x (hyphen - 2010)\n"
	"	x (non-breaking hyphen - 2011)\n"
	"	x (figure dash - 2012)\n"
	"	x (en dash - 2013)\n"
	"	x (hyphenation point - 2027)\n"
	"	x (hyphen bullet - 2043)\n"
	"	x (minus sign - 2212)\n"
	"	x (roman uncia sign - 10191)"},
/* 002E */ { "FULL STOP","	= period, dot, decimal point\n"
	"	* the use as decimal point is locale dependent\n"
	"	* may be rendered as a raised decimal point in old style numbers\n"
	"	x (middle dot - 00B7)\n"
	"	x (arabic full stop - 06D4)\n"
	"	x (one dot leader - 2024)\n"
	"	x (horizontal ellipsis - 2026)\n"
	"	x (raised dot - 2E33)\n"
	"	x (stenographic full stop - 2E3C)\n"
	"	x (ideographic full stop - 3002)"},
/* 002F */ { "SOLIDUS","	= slash, forward slash, virgule\n"
	"	x (combining long solidus overlay - 0338)\n"
	"	x (fraction slash - 2044)\n"
	"	x (division slash - 2215)\n"
	"	x (mathematical rising diagonal - 27CB)\n"
	"	x (big solidus - 29F8)\n"
	"	x (dotted solidus - 2E4A)"},
/* 0030 */ { "DIGIT ZERO","	~ 0030 FE00 short diagonal stroke form"},
/* 0031 */ { "DIGIT ONE",NULL},
/* 0032 */ { "DIGIT TWO","	x (latin letter two with stroke - 01BB)\n"
	"	x (turned digit two - 218A)"},
/* 0033 */ { "DIGIT THREE","	x (turned digit three - 218B)\n"
	"	x (latin capital letter reversed open e - A7AB)"},
/* 0034 */ { "DIGIT FOUR","	x (latin capital letter cuatrillo - A72C)"},
/* 0035 */ { "DIGIT FIVE","	x (latin capital letter tone five - 01BC)"},
/* 0036 */ { "DIGIT SIX",NULL},
/* 0037 */ { "DIGIT SEVEN",NULL},
/* 0038 */ { "DIGIT EIGHT",NULL},
/* 0039 */ { "DIGIT NINE",NULL},
/* 003A */ { "COLON","	* also used to denote division or scale; for that mathematical use 2236 is preferred\n"
	"	* in Finnish and Swedish, also used as intra-word punctation (abbreviation mark)\n"
	"	x (modifier letter triangular colon - 02D0)\n"
	"	x (modifier letter raised colon - 02F8)\n"
	"	x (armenian full stop - 0589)\n"
	"	x (hebrew punctuation sof pasuq - 05C3)\n"
	"	x (ethiopic wordspace - 1361)\n"
	"	x (ethiopic colon - 1365)\n"
	"	x (two dot punctuation - 205A)\n"
	"	x (tricolon - 205D)\n"
	"	x (ratio - 2236)\n"
	"	x (modifier letter colon - A789)\n"
	"	x (presentation form for vertical two dot leader - FE30)"},
/* 003B */ { "SEMICOLON","	* this, and not 037E, is the preferred character for 'Greek question mark'\n"
	"	x (greek question mark - 037E)\n"
	"	x (arabic semicolon - 061B)\n"
	"	x (reversed semicolon - 204F)\n"
	"	x (turned semicolon - 2E35)"},
/* 003C */ { "LESS-THAN SIGN","	* paired with 003E for ASCII-based angle bracket markup conventions\n"
	"	x (modifier letter left arrowhead - 02C2)\n"
	"	x (single left-pointing angle quotation mark - 2039)\n"
	"	x (left-pointing angle bracket - 2329)\n"
	"	x (mathematical left angle bracket - 27E8)\n"
	"	x (left angle bracket - 3008)"},
/* 003D */ { "EQUALS SIGN","	* other related characters: 2241-2263\n"
	"	x (canadian syllabics hyphen - 1400)\n"
	"	x (almost equal to - 2248)\n"
	"	x (not equal to - 2260)\n"
	"	x (identical to - 2261)\n"
	"	x (double hyphen - 2E40)\n"
	"	x (katakana-hiragana double hyphen - 30A0)\n"
	"	x (modifier letter short equals sign - A78A)\n"
	"	x (small equals sign - FE66)\n"
	"	x (roman sextans sign - 10190)\n"
	"	x (heavy equals sign - 1F7F0)"},
/* 003E */ { "GREATER-THAN SIGN","	x (modifier letter right arrowhead - 02C3)\n"
	"	x (single right-pointing angle quotation mark - 203A)\n"
	"	x (right-pointing angle bracket - 232A)\n"
	"	x (mathematical right angle bracket - 27E9)\n"
	"	x (right angle bracket - 3009)"},
/* 003F */ { "QUESTION MARK","	x (inverted question mark - 00BF)\n"
	"	x (greek question mark - 037E)\n"
	"	x (arabic question mark - 061F)\n"
	"	x (interrobang - 203D)\n"
	"	x (double question mark - 2047)\n"
	"	x (black question mark ornament - 2753)\n"
	"	x (uncertainty sign - 2BD1)\n"
	"	x (reversed question mark - 2E2E)\n"
	"	x (medieval question mark - 2E54)\n"
	"	x (replacement character - FFFD)"},
/* 0040 */ { "COMMERCIAL AT","	= at sign\n"
	"	= arroba (old Spanish unit of weight)\n"
	"	x (circled latin small letter a - 24D0)"},
/* 0041 */ { "LATIN CAPITAL LETTER A",NULL},
/* 0042 */ { "LATIN CAPITAL LETTER B","	x (script capital b - 212C)"},
/* 0043 */ { "LATIN CAPITAL LETTER C","	x (greek capital lunate sigma symbol - 03F9)\n"
	"	x (double-struck capital c - 2102)\n"
	"	x (degree celsius - 2103)\n"
	"	x (black-letter capital c - 212D)\n"
	"	x (roman numeral one hundred - 216D)"},
/* 0044 */ { "LATIN CAPITAL LETTER D","	x (roman numeral five hundred - 216E)"},
/* 0045 */ { "LATIN CAPITAL LETTER E","	x (latin capital letter open e - 0190)\n"
	"	x (euler constant - 2107)\n"
	"	x (script capital e - 2130)"},
/* 0046 */ { "LATIN CAPITAL LETTER F","	x (degree fahrenheit - 2109)\n"
	"	x (script capital f - 2131)\n"
	"	x (turned capital f - 2132)"},
/* 0047 */ { "LATIN CAPITAL LETTER G",NULL},
/* 0048 */ { "LATIN CAPITAL LETTER H","	x (script capital h - 210B)\n"
	"	x (black-letter capital h - 210C)\n"
	"	x (double-struck capital h - 210D)"},
/* 0049 */ { "LATIN CAPITAL LETTER I","	* Turkish and Azerbaijani use 0131 for lowercase\n"
	"	x (latin capital letter i with dot above - 0130)\n"
	"	x (script capital i - 2110)\n"
	"	x (black-letter capital i - 2111)\n"
	"	x (roman numeral one - 2160)"},
/* 004A */ { "LATIN CAPITAL LETTER J",NULL},
/* 004B */ { "LATIN CAPITAL LETTER K","	x (kelvin sign - 212A)"},
/* 004C */ { "LATIN CAPITAL LETTER L","	x (script capital l - 2112)"},
/* 004D */ { "LATIN CAPITAL LETTER M","	x (script capital m - 2133)\n"
	"	x (roman numeral one thousand - 216F)"},
/* 004E */ { "LATIN CAPITAL LETTER N","	x (double-struck capital n - 2115)"},
/* 004F */ { "LATIN CAPITAL LETTER O",NULL},
/* 0050 */ { "LATIN CAPITAL LETTER P","	x (script capital p - 2118)\n"
	"	x (double-struck capital p - 2119)"},
/* 0051 */ { "LATIN CAPITAL LETTER Q","	x (cyrillic capital letter qa - 051A)\n"
	"	x (double-struck capital q - 211A)\n"
	"	x (rotated capital q - 213A)"},
/* 0052 */ { "LATIN CAPITAL LETTER R","	x (script capital r - 211B)\n"
	"	x (black-letter capital r - 211C)\n"
	"	x (double-struck capital r - 211D)"},
/* 0053 */ { "LATIN CAPITAL LETTER S",NULL},
/* 0054 */ { "LATIN CAPITAL LETTER T",NULL},
/* 0055 */ { "LATIN CAPITAL LETTER U",NULL},
/* 0056 */ { "LATIN CAPITAL LETTER V","	x (roman numeral five - 2164)"},
/* 0057 */ { "LATIN CAPITAL LETTER W","	x (cyrillic capital letter we - 051C)"},
/* 0058 */ { "LATIN CAPITAL LETTER X","	x (roman numeral ten - 2169)"},
/* 0059 */ { "LATIN CAPITAL LETTER Y",NULL},
/* 005A */ { "LATIN CAPITAL LETTER Z","	x (double-struck capital z - 2124)\n"
	"	x (black-letter capital z - 2128)"},
/* 005B */ { "LEFT SQUARE BRACKET","	= opening square bracket (1.0)\n"
	"	* other bracket characters: 27E6-27EB, 2983-2998, 3008-301B"},
/* 005C */ { "REVERSE SOLIDUS","	= backslash\n"
	"	x (combining reverse solidus overlay - 20E5)\n"
	"	x (set minus - 2216)\n"
	"	x (mathematical falling diagonal - 27CD)\n"
	"	x (big reverse solidus - 29F9)"},
/* 005D */ { "RIGHT SQUARE BRACKET","	= closing square bracket (1.0)"},
/* 005E */ { "CIRCUMFLEX ACCENT","	* this is a spacing character\n"
	"	* used in ASCII-based representation of exponentiation\n"
	"	x (modifier letter up arrowhead - 02C4)\n"
	"	x (modifier letter circumflex accent - 02C6)\n"
	"	x (combining circumflex accent - 0302)\n"
	"	x (caret - 2038)\n"
	"	x (logical and - 2227)\n"
	"	x (up arrowhead - 2303)"},
/* 005F */ { "LOW LINE","	= spacing underscore (1.0)\n"
	"	* this is a spacing character\n"
	"	* joins with adjacent instances of itself\n"
	"	x (modifier letter low macron - 02CD)\n"
	"	x (combining macron below - 0331)\n"
	"	x (combining low line - 0332)\n"
	"	x (double low line - 2017)\n"
	"	x (combining conjoining macron below - FE2D)"},
/* 0060 */ { "GRAVE ACCENT","	= backtick, backquote\n"
	"	* this is a spacing character\n"
	"	* in some technical contexts functions as opening quote, paired with apostrophe\n"
	"	x (acute accent - 00B4)\n"
	"	x (modifier letter grave accent - 02CB)\n"
	"	x (combining grave accent - 0300)\n"
	"	x (greek varia - 1FEF)\n"
	"	x (left single quotation mark - 2018)\n"
	"	x (reversed prime - 2035)"},
/* 0061 */ { "LATIN SMALL LETTER A","	x (latin small letter alpha - 0251)"},
/* 0062 */ { "LATIN SMALL LETTER B",NULL},
/* 0063 */ { "LATIN SMALL LETTER C","	x (greek lunate sigma symbol - 03F2)\n"
	"	x (latin letter small capital c - 1D04)"},
/* 0064 */ { "LATIN SMALL LETTER D","	x (cyrillic small letter komi de - 0501)"},
/* 0065 */ { "LATIN SMALL LETTER E","	x (estimated symbol - 212E)\n"
	"	x (script small e - 212F)\n"
	"	x (latin small letter blackletter e - AB32)"},
/* 0066 */ { "LATIN SMALL LETTER F","	x (latin small letter f with hook - 0192)\n"
	"	x (latin small letter lenis f - AB35)"},
/* 0067 */ { "LATIN SMALL LETTER G","	x (latin small letter script g - 0261)\n"
	"	x (script small g - 210A)"},
/* 0068 */ { "LATIN SMALL LETTER H","	x (cyrillic small letter shha - 04BB)\n"
	"	x (planck constant - 210E)"},
/* 0069 */ { "LATIN SMALL LETTER I","	* Turkish and Azerbaijani use 0130 for uppercase\n"
	"	x (latin small letter dotless i - 0131)\n"
	"	x (superscript latin small letter i - 2071)\n"
	"	x (mathematical italic small dotless i - 1D6A4)"},
/* 006A */ { "LATIN SMALL LETTER J","	x (latin small letter dotless j - 0237)\n"
	"	x (greek letter yot - 03F3)\n"
	"	x (mathematical italic small dotless j - 1D6A5)"},
/* 006B */ { "LATIN SMALL LETTER K",NULL},
/* 006C */ { "LATIN SMALL LETTER L","	x (script small l - 2113)\n"
	"	x (mathematical script small l - 1D4C1)"},
/* 006D */ { "LATIN SMALL LETTER M",NULL},
/* 006E */ { "LATIN SMALL LETTER N","	x (superscript latin small letter n - 207F)"},
/* 006F */ { "LATIN SMALL LETTER O","	x (latin letter small capital o - 1D0F)\n"
	"	x (script small o - 2134)\n"
	"	x (latin small letter blackletter o - AB3D)"},
/* 0070 */ { "LATIN SMALL LETTER P",NULL},
/* 0071 */ { "LATIN SMALL LETTER Q","	x (cyrillic small letter qa - 051B)"},
/* 0072 */ { "LATIN SMALL LETTER R",NULL},
/* 0073 */ { "LATIN SMALL LETTER S","	x (latin small letter long s - 017F)\n"
	"	x (latin letter small capital s - A731)"},
/* 0074 */ { "LATIN SMALL LETTER T",NULL},
/* 0075 */ { "LATIN SMALL LETTER U",NULL},
/* 0076 */ { "LATIN SMALL LETTER V",NULL},
/* 0077 */ { "LATIN SMALL LETTER W","	x (cyrillic small letter we - 051D)\n"
	"	x (latin letter small capital w - 1D21)"},
/* 0078 */ { "LATIN SMALL LETTER X","	x (multiplication sign - 00D7)"},
/* 0079 */ { "LATIN SMALL LETTER Y",NULL},
/* 007A */ { "LATIN SMALL LETTER Z","	x (latin small letter z with stroke - 01B6)\n"
	"	x (latin letter small capital z - 1D22)"},
/* 007B */ { "LEFT CURLY BRACKET","	= opening curly bracket (1.0)\n"
	"	= left brace"},
/* 007C */ { "VERTICAL LINE","	= vertical bar, pipe\n"
	"	* used in pairs to indicate absolute value\n"
	"	* also used as an unpaired separator or as a fence\n"
	"	x (broken bar - 00A6)\n"
	"	x (latin letter dental click - 01C0)\n"
	"	x (hebrew punctuation paseq - 05C0)\n"
	"	x (devanagari danda - 0964)\n"
	"	x (combining long vertical line overlay - 20D2)\n"
	"	x (divides - 2223)\n"
	"	x (light vertical bar - 2758)"},
/* 007D */ { "RIGHT CURLY BRACKET","	= closing curly bracket (1.0)\n"
	"	= right brace"},
/* 007E */ { "TILDE","	* this is a spacing character\n"
	"	x (small tilde - 02DC)\n"
	"	x (combining tilde - 0303)\n"
	"	x (combining tilde overlay - 0334)\n"
	"	x (swung dash - 2053)\n"
	"	x (tilde operator - 223C)\n"
	"	x (tilde with ring above - 2E1B)\n"
	"	x (vertical tilde - 2E2F)\n"
	"	x (wave dash - 301C)\n"
	"	x (fullwidth tilde - FF5E)"},
/* 007F */ { NULL,"	= DELETE"},
/* 0080 */ { NULL,NULL},
/* 0081 */ { NULL,NULL},
/* 0082 */ { NULL,"	= BREAK PERMITTED HERE\n"
	"	x (zero width space - 200B)"},
/* 0083 */ { NULL,"	= NO BREAK HERE\n"
	"	x (word joiner - 2060)"},
/* 0084 */ { NULL,"	* formerly known as INDEX"},
/* 0085 */ { NULL,"	= NEXT LINE (NEL)"},
/* 0086 */ { NULL,"	= START OF SELECTED AREA"},
/* 0087 */ { NULL,"	= END OF SELECTED AREA"},
/* 0088 */ { NULL,"	= CHARACTER TABULATION SET"},
/* 0089 */ { NULL,"	= CHARACTER TABULATION WITH JUSTIFICATION"},
/* 008A */ { NULL,"	= LINE TABULATION SET"},
/* 008B */ { NULL,"	= PARTIAL LINE FORWARD"},
/* 008C */ { NULL,"	= PARTIAL LINE BACKWARD"},
/* 008D */ { NULL,"	= REVERSE LINE FEED"},
/* 008E */ { NULL,"	= SINGLE SHIFT TWO"},
/* 008F */ { NULL,"	= SINGLE SHIFT THREE"},
/* 0090 */ { NULL,"	= DEVICE CONTROL STRING"},
/* 0091 */ { NULL,"	= PRIVATE USE ONE"},
/* 0092 */ { NULL,"	= PRIVATE USE TWO"},
/* 0093 */ { NULL,"	= SET TRANSMIT STATE"},
/* 0094 */ { NULL,"	= CANCEL CHARACTER"},
/* 0095 */ { NULL,"	= MESSAGE WAITING"},
/* 0096 */ { NULL,"	= START OF GUARDED AREA"},
/* 0097 */ { NULL,"	= END OF GUARDED AREA"},
/* 0098 */ { NULL,"	= START OF STRING"},
/* 0099 */ { NULL,NULL},
/* 009A */ { NULL,"	= SINGLE CHARACTER INTRODUCER"},
/* 009B */ { NULL,"	= CONTROL SEQUENCE INTRODUCER"},
/* 009C */ { NULL,"	= STRING TERMINATOR"},
/* 009D */ { NULL,"	= OPERATING SYSTEM COMMAND"},
/* 009E */ { NULL,"	= PRIVACY MESSAGE"},
/* 009F */ { NULL,"	= APPLICATION PROGRAM COMMAND"},
/* 00A0 */ { "NO-BREAK SPACE","	* commonly abbreviated as NBSP\n"
	"	x (space - 0020)\n"
	"	x (figure space - 2007)\n"
	"	x (narrow no-break space - 202F)\n"
	"	x (word joiner - 2060)\n"
	"	x (shouldered open box - 237D)\n"
	"	x (zero width no-break space - FEFF)\n"
	"	# <noBreak> 0020"},
/* 00A1 */ { "INVERTED EXCLAMATION MARK","	* Spanish, Asturian, Galician\n"
	"	* used in extended IPA to represent sublaminal lower alveolar percussive\n"
	"	x (exclamation mark - 0021)\n"
	"	x (modifier letter raised inverted exclamation mark - A71E)"},
/* 00A2 */ { "CENT SIGN","	x (latin small letter c with stroke - 023C)\n"
	"	x (mill sign - 20A5)\n"
	"	x (cedi sign - 20B5)\n"
	"	x (musical symbol cut time - 1D135)"},
/* 00A3 */ { "POUND SIGN","	= pound sterling, Irish punt, lira, etc.\n"
	"	* not to be confused with the unit of weight\n"
	"	* glyph may have one or two crossbars, in distinction from 20A4\n"
	"	x (lira sign - 20A4)\n"
	"	x (turkish lira sign - 20BA)\n"
	"	x (roman semuncia sign - 10192)"},
/* 00A4 */ { "CURRENCY SIGN","	* generic or unknown currency\n"
	"	* other currency symbol characters start at 20A0\n"
	"	x (dollar sign - 0024)"},
/* 00A5 */ { "YEN SIGN","	= yuan sign (renminbi)\n"
	"	* glyph may have one or two crossbars, but the official symbol in Japan and China has two\n"
	"	* in native context, specific ideographs may be used for units of these currencies\n"
	"	x (latin capital letter y with stroke - 024E)\n"
	"	x (cyrillic capital letter straight u with stroke - 04B0)\n"
	"	x 5143\n"
	"	x 5186"},
/* 00A6 */ { "BROKEN BAR","	= broken vertical bar (1.0)\n"
	"	= parted rule (in typography)\n"
	"	x (vertical line - 007C)"},
/* 00A7 */ { "SECTION SIGN","	* used in referencing numbered sections or paragraphs\n"
	"	x (top half section sign - 2E39)"},
/* 00A8 */ { "DIAERESIS","	* this is a spacing character\n"
	"	x (combining diaeresis - 0308)\n"
	"	# 0020 0308"},
/* 00A9 */ { "COPYRIGHT SIGN","	x (sound recording copyright - 2117)\n"
	"	x (circled latin capital letter c - 24B8)\n"
	"	x (copyleft symbol - 1F12F)\n"
	"	x (mask work symbol - 1F1AD)"},
/* 00AA */ { "FEMININE ORDINAL INDICATOR","	* Spanish\n"
	"	x (modifier letter small a - 1D43)\n"
	"	# <super> 0061"},
/* 00AB */ { "LEFT-POINTING DOUBLE ANGLE QUOTATION MARK","	= left guillemet\n"
	"	= chevrons (in typography)\n"
	"	* usually opening, sometimes closing\n"
	"	x (much less-than - 226A)\n"
	"	x (left double angle bracket - 300A)"},
/* 00AC */ { "NOT SIGN","	= angled dash (in typography)\n"
	"	x (reversed not sign - 2310)\n"
	"	x (turned not sign - 2319)"},
/* 00AD */ { "SOFT HYPHEN","	= discretionary hyphen\n"
	"	* commonly abbreviated as SHY\n"
	"	* an invisible format character indicating a possible hyphenation location"},
/* 00AE */ { "REGISTERED SIGN","	= registered trade mark sign (1.0)\n"
	"	x (trade mark sign - 2122)\n"
	"	x (circled latin capital letter r - 24C7)\n"
	"	x (raised md sign - 1F16B)\n"
	"	x (raised mr sign - 1F16C)"},
/* 00AF */ { "MACRON","	= overline, APL overbar\n"
	"	* this is a spacing character\n"
	"	* unlike 203E, adjacent instances of this character do not join\n"
	"	x (modifier letter macron - 02C9)\n"
	"	x (combining macron - 0304)\n"
	"	x (combining overline - 0305)\n"
	"	x (overline - 203E)\n"
	"	x (combining conjoining macron - FE26)\n"
	"	# 0020 0304"},
/* 00B0 */ { "DEGREE SIGN","	* this is a spacing character\n"
	"	x (ring above - 02DA)\n"
	"	x (combining ring above - 030A)\n"
	"	x (superscript zero - 2070)\n"
	"	x (degree celsius - 2103)\n"
	"	x (degree fahrenheit - 2109)\n"
	"	x (ring operator - 2218)"},
/* 00B1 */ { "PLUS-MINUS SIGN","	= biquintile (astrological aspect: 144°)\n"
	"	x (minus-or-plus sign - 2213)"},
/* 00B2 */ { "SUPERSCRIPT TWO","	= squared\n"
	"	* other superscript digit characters: 2070-2079\n"
	"	x (superscript one - 00B9)\n"
	"	# <super> 0032"},
/* 00B3 */ { "SUPERSCRIPT THREE","	= cubed\n"
	"	x (superscript one - 00B9)\n"
	"	# <super> 0033"},
/* 00B4 */ { "ACUTE ACCENT","	* this is a spacing character\n"
	"	x (grave accent - 0060)\n"
	"	x (modifier letter prime - 02B9)\n"
	"	x (modifier letter acute accent - 02CA)\n"
	"	x (double acute accent - 02DD)\n"
	"	x (combining acute accent - 0301)\n"
	"	x (greek tonos - 0384)\n"
	"	x (greek oxia - 1FFD)\n"
	"	x (prime - 2032)\n"
	"	# 0020 0301"},
/* 00B5 */ { "MICRO SIGN","	# 03BC greek small letter mu"},
/* 00B6 */ { "PILCROW SIGN","	= paragraph sign\n"
	"	* section sign in some European usage\n"
	"	x (reversed pilcrow sign - 204B)\n"
	"	x (curved stem paragraph sign ornament - 2761)\n"
	"	x (capitulum - 2E3F)"},
/* 00B7 */ { "MIDDLE DOT","	= midpoint (in typography)\n"
	"	= Georgian comma\n"
	"	= Greek middle dot (ano teleia)\n"
	"	* also used as a raised decimal point or to denote multiplication; for multiplication 22C5 is preferred\n"
	"	* used as a vowel length mark (part of words) in many Amerindian orthographies\n"
	"	x (full stop - 002E)\n"
	"	x (dot above - 02D9)\n"
	"	x (greek ano teleia - 0387)\n"
	"	x (bullet - 2022)\n"
	"	x (one dot leader - 2024)\n"
	"	x (hyphenation point - 2027)\n"
	"	x (bullet operator - 2219)\n"
	"	x (dot operator - 22C5)\n"
	"	x (word separator middle dot - 2E31)\n"
	"	x (raised dot - 2E33)\n"
	"	x (katakana middle dot - 30FB)\n"
	"	x (latin letter sinological dot - A78F)"},
/* 00B8 */ { "CEDILLA","	* this is a spacing character\n"
	"	* other spacing accent characters: 02D8-02DB\n"
	"	x (combining cedilla - 0327)\n"
	"	# 0020 0327"},
/* 00B9 */ { "SUPERSCRIPT ONE","	x (superscript two - 00B2)\n"
	"	x (superscript three - 00B3)\n"
	"	x (superscript zero - 2070)\n"
	"	# <super> 0031"},
/* 00BA */ { "MASCULINE ORDINAL INDICATOR","	* Spanish\n"
	"	x (modifier letter small o - 1D52)\n"
	"	x (superscript zero - 2070)\n"
	"	x (numero sign - 2116)\n"
	"	# <super> 006F"},
/* 00BB */ { "RIGHT-POINTING DOUBLE ANGLE QUOTATION MARK","	= right guillemet\n"
	"	* usually closing, sometimes opening\n"
	"	x (much greater-than - 226B)\n"
	"	x (right double angle bracket - 300B)"},
/* 00BC */ { "VULGAR FRACTION ONE QUARTER","	# <fraction> 0031 2044 0034"},
/* 00BD */ { "VULGAR FRACTION ONE HALF","	# <fraction> 0031 2044 0032"},
/* 00BE */ { "VULGAR FRACTION THREE QUARTERS","	# <fraction> 0033 2044 0034"},
/* 00BF */ { "INVERTED QUESTION MARK","	= turned question mark\n"
	"	* Spanish\n"
	"	x (question mark - 003F)\n"
	"	x (inverted interrobang - 2E18)\n"
	"	x (reversed question mark - 2E2E)"},
/* 00C0 */ { "LATIN CAPITAL LETTER A WITH GRAVE","	: 0041 0300"},
/* 00C1 */ { "LATIN CAPITAL LETTER A WITH ACUTE","	: 0041 0301"},
/* 00C2 */ { "LATIN CAPITAL LETTER A WITH CIRCUMFLEX","	: 0041 0302"},
/* 00C3 */ { "LATIN CAPITAL LETTER A WITH TILDE","	: 0041 0303"},
/* 00C4 */ { "LATIN CAPITAL LETTER A WITH DIAERESIS","	: 0041 0308"},
/* 00C5 */ { "LATIN CAPITAL LETTER A WITH RING ABOVE","	x (angstrom sign - 212B)\n"
	"	: 0041 030A"},
/* 00C6 */ { "LATIN CAPITAL LETTER AE","	= latin capital ligature ae (1.1)\n"
	"	x (cyrillic capital ligature a ie - 04D4)"},
/* 00C7 */ { "LATIN CAPITAL LETTER C WITH CEDILLA","	: 0043 0327"},
/* 00C8 */ { "LATIN CAPITAL LETTER E WITH GRAVE","	: 0045 0300"},
/* 00C9 */ { "LATIN CAPITAL LETTER E WITH ACUTE","	: 0045 0301"},
/* 00CA */ { "LATIN CAPITAL LETTER E WITH CIRCUMFLEX","	: 0045 0302"},
/* 00CB */ { "LATIN CAPITAL LETTER E WITH DIAERESIS","	: 0045 0308"},
/* 00CC */ { "LATIN CAPITAL LETTER I WITH GRAVE","	: 0049 0300"},
/* 00CD */ { "LATIN CAPITAL LETTER I WITH ACUTE","	: 0049 0301"},
/* 00CE */ { "LATIN CAPITAL LETTER I WITH CIRCUMFLEX","	: 0049 0302"},
/* 00CF */ { "LATIN CAPITAL LETTER I WITH DIAERESIS","	: 0049 0308"},
/* 00D0 */ { "LATIN CAPITAL LETTER ETH","	* lowercase is 00F0\n"
	"	x (latin capital letter d with stroke - 0110)\n"
	"	x (latin capital letter african d - 0189)\n"
	"	x (latin capital letter d with short stroke overlay - A7C7)"},
/* 00D1 */ { "LATIN CAPITAL LETTER N WITH TILDE","	: 004E 0303"},
/* 00D2 */ { "LATIN CAPITAL LETTER O WITH GRAVE","	: 004F 0300"},
/* 00D3 */ { "LATIN CAPITAL LETTER O WITH ACUTE","	: 004F 0301"},
/* 00D4 */ { "LATIN CAPITAL LETTER O WITH CIRCUMFLEX","	: 004F 0302"},
/* 00D5 */ { "LATIN CAPITAL LETTER O WITH TILDE","	: 004F 0303"},
/* 00D6 */ { "LATIN CAPITAL LETTER O WITH DIAERESIS","	: 004F 0308"},
/* 00D7 */ { "MULTIPLICATION SIGN","	= z notation Cartesian product\n"
	"	x (latin small letter x - 0078)\n"
	"	x (saltire - 2613)\n"
	"	x (multiplication x - 2715)\n"
	"	x (heavy ballot x - 2718)\n"
	"	x (cross mark - 274C)\n"
	"	x (vector or cross product - 2A2F)\n"
	"	x (cancellation x - 1F5D9)\n"
	"	x (light saltire - 1F7A9)"},
/* 00D8 */ { "LATIN CAPITAL LETTER O WITH STROKE","	= o slash\n"
	"	x (latin capital letter o with middle tilde - 019F)\n"
	"	x (empty set - 2205)\n"
	"	x (latin capital letter o with long stroke overlay - A74A)"},
/* 00D9 */ { "LATIN CAPITAL LETTER U WITH GRAVE","	: 0055 0300"},
/* 00DA */ { "LATIN CAPITAL LETTER U WITH ACUTE","	: 0055 0301"},
/* 00DB */ { "LATIN CAPITAL LETTER U WITH CIRCUMFLEX","	: 0055 0302"},
/* 00DC */ { "LATIN CAPITAL LETTER U WITH DIAERESIS","	: 0055 0308"},
/* 00DD */ { "LATIN CAPITAL LETTER Y WITH ACUTE","	: 0059 0301"},
/* 00DE */ { "LATIN CAPITAL LETTER THORN",NULL},
/* 00DF */ { "LATIN SMALL LETTER SHARP S","	= Eszett\n"
	"	* German\n"
	"	* not used in Swiss High German\n"
	"	* uppercase is \"SS\" (standard case mapping), alternatively 1E9E\n"
	"	* typographically the glyph for this character can be based on a ligature of 017F with either 0073 or with an old-style glyph for 007A (the latter similar in appearance to 0292). Both forms exist interchangeably today.\n"
	"	x (latin small letter long s - 017F)\n"
	"	x (latin small letter ezh - 0292)\n"
	"	x (greek small letter beta - 03B2)\n"
	"	x (latin capital letter sharp s - 1E9E)\n"
	"	x (latin small letter beta - A7B5)\n"
	"	x (latin small letter middle scots s - A7D7)"},
/* 00E0 */ { "LATIN SMALL LETTER A WITH GRAVE","	: 0061 0300"},
/* 00E1 */ { "LATIN SMALL LETTER A WITH ACUTE","	: 0061 0301"},
/* 00E2 */ { "LATIN SMALL LETTER A WITH CIRCUMFLEX","	: 0061 0302"},
/* 00E3 */ { "LATIN SMALL LETTER A WITH TILDE","	* Portuguese\n"
	"	: 0061 0303"},
/* 00E4 */ { "LATIN SMALL LETTER A WITH DIAERESIS","	: 0061 0308"},
/* 00E5 */ { "LATIN SMALL LETTER A WITH RING ABOVE","	* Danish, Norwegian, Swedish, Walloon\n"
	"	: 0061 030A"},
/* 00E6 */ { "LATIN SMALL LETTER AE","	= latin small ligature ae (1.1)\n"
	"	= ash (from Old English æsc)\n"
	"	* Danish, Norwegian, Icelandic, Faroese, Old English, French, IPA\n"
	"	x (latin small ligature oe - 0153)\n"
	"	x (cyrillic small ligature a ie - 04D5)\n"
	"	x (latin small letter turned ae - 1D02)\n"
	"	x (latin small letter a reversed-schwa - AB31)"},
/* 00E7 */ { "LATIN SMALL LETTER C WITH CEDILLA","	: 0063 0327"},
/* 00E8 */ { "LATIN SMALL LETTER E WITH GRAVE","	: 0065 0300"},
/* 00E9 */ { "LATIN SMALL LETTER E WITH ACUTE","	: 0065 0301"},
/* 00EA */ { "LATIN SMALL LETTER E WITH CIRCUMFLEX","	: 0065 0302"},
/* 00EB */ { "LATIN SMALL LETTER E WITH DIAERESIS","	: 0065 0308"},
/* 00EC */ { "LATIN SMALL LETTER I WITH GRAVE","	* Italian, Malagasy\n"
	"	: 0069 0300"},
/* 00ED */ { "LATIN SMALL LETTER I WITH ACUTE","	: 0069 0301"},
/* 00EE */ { "LATIN SMALL LETTER I WITH CIRCUMFLEX","	: 0069 0302"},
/* 00EF */ { "LATIN SMALL LETTER I WITH DIAERESIS","	: 0069 0308"},
/* 00F0 */ { "LATIN SMALL LETTER ETH","	* Icelandic, Faroese, Old English, IPA\n"
	"	* uppercase is 00D0\n"
	"	x (latin small letter d with stroke - 0111)\n"
	"	x (greek small letter delta - 03B4)\n"
	"	x (latin small letter delta - 1E9F)\n"
	"	x (partial differential - 2202)\n"
	"	x (latin small letter insular d - A77A)"},
/* 00F1 */ { "LATIN SMALL LETTER N WITH TILDE","	: 006E 0303"},
/* 00F2 */ { "LATIN SMALL LETTER O WITH GRAVE","	: 006F 0300"},
/* 00F3 */ { "LATIN SMALL LETTER O WITH ACUTE","	: 006F 0301"},
/* 00F4 */ { "LATIN SMALL LETTER O WITH CIRCUMFLEX","	: 006F 0302"},
/* 00F5 */ { "LATIN SMALL LETTER O WITH TILDE","	* Portuguese, Estonian, Livonian\n"
	"	: 006F 0303"},
/* 00F6 */ { "LATIN SMALL LETTER O WITH DIAERESIS","	: 006F 0308"},
/* 00F7 */ { "DIVISION SIGN","	= obelus\n"
	"	* occasionally used as an alternate, more visually distinct shape for a hyphen in some contexts\n"
	"	* historically used as a punctuation mark to denote questionable passages in manuscripts\n"
	"	x (syriac harklean obelus - 070B)\n"
	"	x (commercial minus sign - 2052)\n"
	"	x (division slash - 2215)\n"
	"	x (divides - 2223)\n"
	"	x (ratio - 2236)\n"
	"	x (heavy division sign - 2797)\n"
	"	x (long division - 27CC)\n"
	"	x (dotted obelos - 2E13)"},
/* 00F8 */ { "LATIN SMALL LETTER O WITH STROKE","	= o slash\n"
	"	* Danish, Norwegian, Faroese, IPA\n"
	"	x (latin small letter barred o - 0275)\n"
	"	x (diameter sign - 2300)\n"
	"	x (latin small letter o with long stroke overlay - A74B)\n"
	"	x (latin small letter old polish o - A7C1)\n"
	"	x (latin small letter blackletter o with stroke - AB3E)"},
/* 00F9 */ { "LATIN SMALL LETTER U WITH GRAVE","	* French, Italian\n"
	"	: 0075 0300"},
/* 00FA */ { "LATIN SMALL LETTER U WITH ACUTE","	: 0075 0301"},
/* 00FB */ { "LATIN SMALL LETTER U WITH CIRCUMFLEX","	: 0075 0302"},
/* 00FC */ { "LATIN SMALL LETTER U WITH DIAERESIS","	: 0075 0308"},
/* 00FD */ { "LATIN SMALL LETTER Y WITH ACUTE","	* Czech, Slovak, Icelandic, Faroese, Welsh, Malagasy\n"
	"	: 0079 0301"},
/* 00FE */ { "LATIN SMALL LETTER THORN","	* Icelandic, Old English, phonetics\n"
	"	* Runic letter borrowed into Latin script\n"
	"	* replaced by \"th\" in modern English\n"
	"	x (greek small letter sho - 03F8)\n"
	"	x (runic letter thurisaz thurs thorn - 16A6)\n"
	"	x (latin small letter double thorn - A7D3)"},
/* 00FF */ { "LATIN SMALL LETTER Y WITH DIAERESIS","	* French, medieval Hungarian orthography\n"
	"	* uppercase is 0178\n"
	"	: 0079 0308"}
};

UN_DLL_LOCAL
static const struct unicode_nameannot una_00_01[] = {
/* 0100 */ { "LATIN CAPITAL LETTER A WITH MACRON","	: 0041 0304"},
/* 0101 */ { "LATIN SMALL LETTER A WITH MACRON","	* Latvian, Latin, ...\n"
	"	: 0061 0304"},
/* 0102 */ { "LATIN CAPITAL LETTER A WITH BREVE","	: 0041 0306"},
/* 0103 */ { "LATIN SMALL LETTER A WITH BREVE","	* Romanian, Vietnamese, Latin, ...\n"
	"	: 0061 0306"},
/* 0104 */ { "LATIN CAPITAL LETTER A WITH OGONEK","	: 0041 0328"},
/* 0105 */ { "LATIN SMALL LETTER A WITH OGONEK","	* Polish, Lithuanian, ...\n"
	"	: 0061 0328"},
/* 0106 */ { "LATIN CAPITAL LETTER C WITH ACUTE","	: 0043 0301"},
/* 0107 */ { "LATIN SMALL LETTER C WITH ACUTE","	* Polish, Croatian, Serbian, ...\n"
	"	x (cyrillic small letter tshe - 045B)\n"
	"	: 0063 0301"},
/* 0108 */ { "LATIN CAPITAL LETTER C WITH CIRCUMFLEX","	: 0043 0302"},
/* 0109 */ { "LATIN SMALL LETTER C WITH CIRCUMFLEX","	* Esperanto\n"
	"	: 0063 0302"},
/* 010A */ { "LATIN CAPITAL LETTER C WITH DOT ABOVE","	: 0043 0307"},
/* 010B */ { "LATIN SMALL LETTER C WITH DOT ABOVE","	* Maltese, Irish Gaelic (old orthography)\n"
	"	: 0063 0307"},
/* 010C */ { "LATIN CAPITAL LETTER C WITH CARON","	: 0043 030C"},
/* 010D */ { "LATIN SMALL LETTER C WITH CARON","	* Czech, Slovak, Slovenian, and many other languages\n"
	"	: 0063 030C"},
/* 010E */ { "LATIN CAPITAL LETTER D WITH CARON","	* the form using caron/hacek is preferred in all contexts\n"
	"	: 0044 030C"},
/* 010F */ { "LATIN SMALL LETTER D WITH CARON","	* Czech, Slovak\n"
	"	* the form using apostrophe is preferred in typesetting\n"
	"	: 0064 030C"},
/* 0110 */ { "LATIN CAPITAL LETTER D WITH STROKE","	x (latin capital letter eth - 00D0)\n"
	"	x (latin capital letter african d - 0189)\n"
	"	x (latin capital letter d with short stroke overlay - A7C7)"},
/* 0111 */ { "LATIN SMALL LETTER D WITH STROKE","	* Croatian, Serbian, Vietnamese, Sami, Moro\n"
	"	* an alternate glyph with the stroke through the bowl is used in Americanist orthographies\n"
	"	x (latin small letter eth - 00F0)\n"
	"	x (cyrillic small letter dje - 0452)\n"
	"	x (latin small letter d with middle tilde - 1D6D)\n"
	"	x (dong sign - 20AB)\n"
	"	x (latin small letter d with short stroke overlay - A7C8)"},
/* 0112 */ { "LATIN CAPITAL LETTER E WITH MACRON","	: 0045 0304"},
/* 0113 */ { "LATIN SMALL LETTER E WITH MACRON","	* Latvian, Latin, ...\n"
	"	: 0065 0304"},
/* 0114 */ { "LATIN CAPITAL LETTER E WITH BREVE","	: 0045 0306"},
/* 0115 */ { "LATIN SMALL LETTER E WITH BREVE","	* Malay, Latin, ...\n"
	"	: 0065 0306"},
/* 0116 */ { "LATIN CAPITAL LETTER E WITH DOT ABOVE","	: 0045 0307"},
/* 0117 */ { "LATIN SMALL LETTER E WITH DOT ABOVE","	* Lithuanian\n"
	"	: 0065 0307"},
/* 0118 */ { "LATIN CAPITAL LETTER E WITH OGONEK","	: 0045 0328"},
/* 0119 */ { "LATIN SMALL LETTER E WITH OGONEK","	* Polish, Lithuanian, ...\n"
	"	: 0065 0328"},
/* 011A */ { "LATIN CAPITAL LETTER E WITH CARON","	: 0045 030C"},
/* 011B */ { "LATIN SMALL LETTER E WITH CARON","	* Czech, ...\n"
	"	: 0065 030C"},
/* 011C */ { "LATIN CAPITAL LETTER G WITH CIRCUMFLEX","	: 0047 0302"},
/* 011D */ { "LATIN SMALL LETTER G WITH CIRCUMFLEX","	* Esperanto\n"
	"	: 0067 0302"},
/* 011E */ { "LATIN CAPITAL LETTER G WITH BREVE","	: 0047 0306"},
/* 011F */ { "LATIN SMALL LETTER G WITH BREVE","	* Turkish, Azerbaijani\n"
	"	x (latin small letter g with caron - 01E7)\n"
	"	: 0067 0306"},
/* 0120 */ { "LATIN CAPITAL LETTER G WITH DOT ABOVE","	: 0047 0307"},
/* 0121 */ { "LATIN SMALL LETTER G WITH DOT ABOVE","	* Maltese, Irish Gaelic (old orthography)\n"
	"	: 0067 0307"},
/* 0122 */ { "LATIN CAPITAL LETTER G WITH CEDILLA","	: 0047 0327"},
/* 0123 */ { "LATIN SMALL LETTER G WITH CEDILLA","	* Latvian\n"
	"	* there are three major glyph variants\n"
	"	: 0067 0327"},
/* 0124 */ { "LATIN CAPITAL LETTER H WITH CIRCUMFLEX","	* lowercase in Nawdm is 0266\n"
	"	: 0048 0302"},
/* 0125 */ { "LATIN SMALL LETTER H WITH CIRCUMFLEX","	* Esperanto\n"
	"	: 0068 0302"},
/* 0126 */ { "LATIN CAPITAL LETTER H WITH STROKE","	x (modifier letter capital h with stroke - A7F8)"},
/* 0127 */ { "LATIN SMALL LETTER H WITH STROKE","	* Maltese, IPA, ...\n"
	"	x (cyrillic small letter tshe - 045B)\n"
	"	x (planck constant over two pi - 210F)"},
/* 0128 */ { "LATIN CAPITAL LETTER I WITH TILDE","	: 0049 0303"},
/* 0129 */ { "LATIN SMALL LETTER I WITH TILDE","	* Greenlandic (old orthography), Kikuyu\n"
	"	: 0069 0303"},
/* 012A */ { "LATIN CAPITAL LETTER I WITH MACRON","	: 0049 0304"},
/* 012B */ { "LATIN SMALL LETTER I WITH MACRON","	* Latvian, Latin, ...\n"
	"	: 0069 0304"},
/* 012C */ { "LATIN CAPITAL LETTER I WITH BREVE","	: 0049 0306"},
/* 012D */ { "LATIN SMALL LETTER I WITH BREVE","	* Latin, ...\n"
	"	: 0069 0306"},
/* 012E */ { "LATIN CAPITAL LETTER I WITH OGONEK","	: 0049 0328"},
/* 012F */ { "LATIN SMALL LETTER I WITH OGONEK","	* Lithuanian, ...\n"
	"	: 0069 0328"},
/* 0130 */ { "LATIN CAPITAL LETTER I WITH DOT ABOVE","	= i dot\n"
	"	* Turkish, Azerbaijani\n"
	"	* lowercase is 0069\n"
	"	x (latin capital letter i - 0049)\n"
	"	: 0049 0307"},
/* 0131 */ { "LATIN SMALL LETTER DOTLESS I","	* Turkish, Azerbaijani\n"
	"	* uppercase is 0049\n"
	"	x (latin small letter i - 0069)\n"
	"	x (latin small letter iota - 0269)\n"
	"	x (latin letter small capital i - 026A)\n"
	"	x (cyrillic small letter byelorussian-ukrainian i - 0456)\n"
	"	x (mathematical italic small dotless i - 1D6A4)"},
/* 0132 */ { "LATIN CAPITAL LIGATURE IJ","	# 0049 004A"},
/* 0133 */ { "LATIN SMALL LIGATURE IJ","	* Dutch\n"
	"	# 0069 006A"},
/* 0134 */ { "LATIN CAPITAL LETTER J WITH CIRCUMFLEX","	: 004A 0302"},
/* 0135 */ { "LATIN SMALL LETTER J WITH CIRCUMFLEX","	* Esperanto\n"
	"	: 006A 0302"},
/* 0136 */ { "LATIN CAPITAL LETTER K WITH CEDILLA","	: 004B 0327"},
/* 0137 */ { "LATIN SMALL LETTER K WITH CEDILLA","	* Latvian\n"
	"	: 006B 0327"},
/* 0138 */ { "LATIN SMALL LETTER KRA","	* Greenlandic (old orthography), Labrador Inuttut\n"
	"	x (cyrillic small letter ka - 043A)\n"
	"	x (latin letter small capital k - 1D0B)"},
/* 0139 */ { "LATIN CAPITAL LETTER L WITH ACUTE","	: 004C 0301"},
/* 013A */ { "LATIN SMALL LETTER L WITH ACUTE","	* Slovak\n"
	"	: 006C 0301"},
/* 013B */ { "LATIN CAPITAL LETTER L WITH CEDILLA","	: 004C 0327"},
/* 013C */ { "LATIN SMALL LETTER L WITH CEDILLA","	* Latvian\n"
	"	: 006C 0327"},
/* 013D */ { "LATIN CAPITAL LETTER L WITH CARON","	* the form using apostrophe is preferred in typesetting\n"
	"	: 004C 030C"},
/* 013E */ { "LATIN SMALL LETTER L WITH CARON","	* Slovak\n"
	"	* the form using apostrophe is preferred in typesetting\n"
	"	: 006C 030C"},
/* 013F */ { "LATIN CAPITAL LETTER L WITH MIDDLE DOT","	* some fonts show the middle dot inside the L, but the preferred form has the dot following the L\n"
	"	# 004C 00B7"},
/* 0140 */ { "LATIN SMALL LETTER L WITH MIDDLE DOT","	* Catalan legacy compatibility character for ISO/IEC 6937\n"
	"	* preferred representation for Catalan: 006C 00B7\n"
	"	# 006C 00B7"},
/* 0141 */ { "LATIN CAPITAL LETTER L WITH STROKE","	x (latin capital letter l with bar - 023D)\n"
	"	x (latin letter small capital l with stroke - 1D0C)\n"
	"	x (latin capital letter l with middle tilde - 2C62)\n"
	"	x (latin capital letter l with high stroke - A748)"},
/* 0142 */ { "LATIN SMALL LETTER L WITH STROKE","	* Polish, Sorbian, Iñupiaq, ...\n"
	"	* also used for Latin-based orthographies for Belarusian and Ukrainian\n"
	"	x (latin small letter l with bar - 019A)\n"
	"	x (latin small letter l with middle tilde - 026B)"},
/* 0143 */ { "LATIN CAPITAL LETTER N WITH ACUTE","	: 004E 0301"},
/* 0144 */ { "LATIN SMALL LETTER N WITH ACUTE","	* Polish, ...\n"
	"	: 006E 0301"},
/* 0145 */ { "LATIN CAPITAL LETTER N WITH CEDILLA","	: 004E 0327"},
/* 0146 */ { "LATIN SMALL LETTER N WITH CEDILLA","	* Latvian\n"
	"	: 006E 0327"},
/* 0147 */ { "LATIN CAPITAL LETTER N WITH CARON","	: 004E 030C"},
/* 0148 */ { "LATIN SMALL LETTER N WITH CARON","	* Czech, Slovak\n"
	"	: 006E 030C"},
/* 0149 */ { "LATIN SMALL LETTER N PRECEDED BY APOSTROPHE","	= latin small letter apostrophe n (1.0)\n"
	"	* this character is deprecated and its use is strongly discouraged\n"
	"	* Afrikaans\n"
	"	* legacy compatibility character for ISO/IEC 6937\n"
	"	* uppercase is 02BC 004E\n"
	"	# 02BC 006E"},
/* 014A */ { "LATIN CAPITAL LETTER ENG","	* glyph may also have appearance of large form of the small letter\n"
	"	x (latin capital letter n with long right leg - 0220)\n"
	"	x (latin capital letter n with descender - A790)"},
/* 014B */ { "LATIN SMALL LETTER ENG","	= engma, angma\n"
	"	* Sami, Mende, IPA, ...\n"
	"	* velar nasal, nasalization\n"
	"	x (latin small letter n with long right leg - 019E)\n"
	"	x (latin small letter n with palatal hook - 1D87)\n"
	"	x (latin small letter n with descender - A791)"},
/* 014C */ { "LATIN CAPITAL LETTER O WITH MACRON","	: 004F 0304"},
/* 014D */ { "LATIN SMALL LETTER O WITH MACRON","	* Latvian, Latin, ...\n"
	"	: 006F 0304"},
/* 014E */ { "LATIN CAPITAL LETTER O WITH BREVE","	: 004F 0306"},
/* 014F */ { "LATIN SMALL LETTER O WITH BREVE","	* Latin\n"
	"	: 006F 0306"},
/* 0150 */ { "LATIN CAPITAL LETTER O WITH DOUBLE ACUTE","	: 004F 030B"},
/* 0151 */ { "LATIN SMALL LETTER O WITH DOUBLE ACUTE","	* Hungarian\n"
	"	: 006F 030B"},
/* 0152 */ { "LATIN CAPITAL LIGATURE OE",NULL},
/* 0153 */ { "LATIN SMALL LIGATURE OE","	= ethel (from Old English eðel)\n"
	"	* French, IPA, Old Icelandic, Old English, ...\n"
	"	x (latin small letter ae - 00E6)\n"
	"	x (latin letter small capital oe - 0276)\n"
	"	x (latin small letter turned oe - 1D14)\n"
	"	x (modifier letter small ligature oe - A7F9)\n"
	"	x (latin small letter inverted oe - AB40)\n"
	"	x (latin small letter open oe - AB62)"},
/* 0154 */ { "LATIN CAPITAL LETTER R WITH ACUTE","	: 0052 0301"},
/* 0155 */ { "LATIN SMALL LETTER R WITH ACUTE","	* Slovak, ...\n"
	"	: 0072 0301"},
/* 0156 */ { "LATIN CAPITAL LETTER R WITH CEDILLA","	: 0052 0327"},
/* 0157 */ { "LATIN SMALL LETTER R WITH CEDILLA","	* Livonian\n"
	"	: 0072 0327"},
/* 0158 */ { "LATIN CAPITAL LETTER R WITH CARON","	: 0052 030C"},
/* 0159 */ { "LATIN SMALL LETTER R WITH CARON","	* Czech, ...\n"
	"	: 0072 030C"},
/* 015A */ { "LATIN CAPITAL LETTER S WITH ACUTE","	: 0053 0301"},
/* 015B */ { "LATIN SMALL LETTER S WITH ACUTE","	* Polish, Indic transliteration, ...\n"
	"	: 0073 0301"},
/* 015C */ { "LATIN CAPITAL LETTER S WITH CIRCUMFLEX","	: 0053 0302"},
/* 015D */ { "LATIN SMALL LETTER S WITH CIRCUMFLEX","	* Esperanto\n"
	"	: 0073 0302"},
/* 015E */ { "LATIN CAPITAL LETTER S WITH CEDILLA","	: 0053 0327"},
/* 015F */ { "LATIN SMALL LETTER S WITH CEDILLA","	* Turkish, Azerbaijani, ...\n"
	"	* the character 0219 should be used instead for Romanian\n"
	"	x (latin small letter s with comma below - 0219)\n"
	"	: 0073 0327"},
/* 0160 */ { "LATIN CAPITAL LETTER S WITH CARON","	: 0053 030C"},
/* 0161 */ { "LATIN SMALL LETTER S WITH CARON","	* Czech, Estonian, Finnish, Slovak, and many other languages\n"
	"	: 0073 030C"},
/* 0162 */ { "LATIN CAPITAL LETTER T WITH CEDILLA","	: 0054 0327"},
/* 0163 */ { "LATIN SMALL LETTER T WITH CEDILLA","	* Semitic transliteration, ...\n"
	"	* the character 021B should be used instead for Romanian\n"
	"	x (latin small letter t with comma below - 021B)\n"
	"	: 0074 0327"},
/* 0164 */ { "LATIN CAPITAL LETTER T WITH CARON","	* the form using caron/hacek is preferred in all contexts\n"
	"	: 0054 030C"},
/* 0165 */ { "LATIN SMALL LETTER T WITH CARON","	* Czech, Slovak\n"
	"	* the form using apostrophe is preferred in typesetting\n"
	"	: 0074 030C"},
/* 0166 */ { "LATIN CAPITAL LETTER T WITH STROKE",NULL},
/* 0167 */ { "LATIN SMALL LETTER T WITH STROKE","	* Sami\n"
	"	x (latin small letter t with middle tilde - 1D75)"},
/* 0168 */ { "LATIN CAPITAL LETTER U WITH TILDE","	: 0055 0303"},
/* 0169 */ { "LATIN SMALL LETTER U WITH TILDE","	* Greenlandic (old orthography), Kikuyu\n"
	"	: 0075 0303"},
/* 016A */ { "LATIN CAPITAL LETTER U WITH MACRON","	: 0055 0304"},
/* 016B */ { "LATIN SMALL LETTER U WITH MACRON","	* Latvian, Lithuanian, Latin, ...\n"
	"	: 0075 0304"},
/* 016C */ { "LATIN CAPITAL LETTER U WITH BREVE","	: 0055 0306"},
/* 016D */ { "LATIN SMALL LETTER U WITH BREVE","	* Latin, Esperanto, ...\n"
	"	: 0075 0306"},
/* 016E */ { "LATIN CAPITAL LETTER U WITH RING ABOVE","	: 0055 030A"},
/* 016F */ { "LATIN SMALL LETTER U WITH RING ABOVE","	* Czech, ...\n"
	"	: 0075 030A"},
/* 0170 */ { "LATIN CAPITAL LETTER U WITH DOUBLE ACUTE","	: 0055 030B"},
/* 0171 */ { "LATIN SMALL LETTER U WITH DOUBLE ACUTE","	* Hungarian\n"
	"	: 0075 030B"},
/* 0172 */ { "LATIN CAPITAL LETTER U WITH OGONEK","	: 0055 0328"},
/* 0173 */ { "LATIN SMALL LETTER U WITH OGONEK","	* Lithuanian\n"
	"	: 0075 0328"},
/* 0174 */ { "LATIN CAPITAL LETTER W WITH CIRCUMFLEX","	: 0057 0302"},
/* 0175 */ { "LATIN SMALL LETTER W WITH CIRCUMFLEX","	* Welsh\n"
	"	: 0077 0302"},
/* 0176 */ { "LATIN CAPITAL LETTER Y WITH CIRCUMFLEX","	: 0059 0302"},
/* 0177 */ { "LATIN SMALL LETTER Y WITH CIRCUMFLEX","	* Welsh\n"
	"	: 0079 0302"},
/* 0178 */ { "LATIN CAPITAL LETTER Y WITH DIAERESIS","	* French, Igbo\n"
	"	* lowercase is 00FF\n"
	"	: 0059 0308"},
/* 0179 */ { "LATIN CAPITAL LETTER Z WITH ACUTE","	: 005A 0301"},
/* 017A */ { "LATIN SMALL LETTER Z WITH ACUTE","	* Polish, ...\n"
	"	: 007A 0301"},
/* 017B */ { "LATIN CAPITAL LETTER Z WITH DOT ABOVE","	: 005A 0307"},
/* 017C */ { "LATIN SMALL LETTER Z WITH DOT ABOVE","	* Polish, ...\n"
	"	: 007A 0307"},
/* 017D */ { "LATIN CAPITAL LETTER Z WITH CARON","	: 005A 030C"},
/* 017E */ { "LATIN SMALL LETTER Z WITH CARON","	* Czech, Estonian, Finnish, Slovak, Slovenian, and many other languages\n"
	"	: 007A 030C"},
/* 017F */ { "LATIN SMALL LETTER LONG S","	* in common use in Roman types until the 18th century\n"
	"	* in current use in Fraktur and Gaelic types\n"
	"	x (latin small letter sharp s - 00DF)\n"
	"	x (latin small ligature long s t - FB05)\n"
	"	# 0073 latin small letter s"},
/* 0180 */ { "LATIN SMALL LETTER B WITH STROKE","	* Americanist and Indo-Europeanist usage for phonetic beta\n"
	"	* Americanist orthographies use an alternate glyph with the stroke through the bowl\n"
	"	* also used in Old Saxon manuscripts\n"
	"	* uppercase is 0243\n"
	"	x (greek small letter beta - 03B2)\n"
	"	x (latin small letter b with middle tilde - 1D6C)\n"
	"	x (blank symbol - 2422)"},
/* 0181 */ { "LATIN CAPITAL LETTER B WITH HOOK","	* African languages\n"
	"	* lowercase is 0253"},
/* 0182 */ { "LATIN CAPITAL LETTER B WITH TOPBAR","	x (cyrillic capital letter be - 0411)"},
/* 0183 */ { "LATIN SMALL LETTER B WITH TOPBAR","	* Zhuang (old orthography)\n"
	"	* former Soviet minority language scripts\n"
	"	x (latin small letter b with hook - 0253)\n"
	"	x (cyrillic small letter be - 0431)"},
/* 0184 */ { "LATIN CAPITAL LETTER TONE SIX",NULL},
/* 0185 */ { "LATIN SMALL LETTER TONE SIX","	* Zhuang (old orthography)\n"
	"	* Zhuang tone three is Cyrillic ze\n"
	"	* Zhuang tone four is Cyrillic che\n"
	"	x (latin small letter tone two - 01A8)\n"
	"	x (latin small letter tone five - 01BD)\n"
	"	x (cyrillic small letter ze - 0437)\n"
	"	x (cyrillic small letter che - 0447)\n"
	"	x (cyrillic small letter soft sign - 044C)"},
/* 0186 */ { "LATIN CAPITAL LETTER OPEN O","	* typographically a turned C\n"
	"	* African languages\n"
	"	* lowercase is 0254\n"
	"	x (latin letter small capital open o - 1D10)\n"
	"	x (roman numeral reversed one hundred - 2183)"},
/* 0187 */ { "LATIN CAPITAL LETTER C WITH HOOK",NULL},
/* 0188 */ { "LATIN SMALL LETTER C WITH HOOK","	* African"},
/* 0189 */ { "LATIN CAPITAL LETTER AFRICAN D","	* Ewe\n"
	"	* lowercase is 0256\n"
	"	x (latin capital letter eth - 00D0)\n"
	"	x (latin capital letter d with stroke - 0110)"},
/* 018A */ { "LATIN CAPITAL LETTER D WITH HOOK","	* Pan-Nigerian alphabet\n"
	"	* lowercase is 0257"},
/* 018B */ { "LATIN CAPITAL LETTER D WITH TOPBAR",NULL},
/* 018C */ { "LATIN SMALL LETTER D WITH TOPBAR","	* former-Soviet minority language scripts\n"
	"	* Zhuang (old orthography)"},
/* 018D */ { "LATIN SMALL LETTER TURNED DELTA","	= reversed Polish-hook o\n"
	"	* archaic phonetic for labialized alveolar fricative\n"
	"	* recommended spellings 007A 02B7 or 007A 032B"},
/* 018E */ { "LATIN CAPITAL LETTER REVERSED E","	= turned capital e\n"
	"	* Pan-Nigerian alphabet\n"
	"	* lowercase is 01DD\n"
	"	x (there exists - 2203)\n"
	"	x (latin letter small capital turned e - 2C7B)"},
/* 018F */ { "LATIN CAPITAL LETTER SCHWA","	* Azerbaijani, ...\n"
	"	* lowercase is 0259\n"
	"	x (cyrillic capital letter schwa - 04D8)"},
/* 0190 */ { "LATIN CAPITAL LETTER OPEN E","	= capital epsilon\n"
	"	* African languages\n"
	"	* lowercase is 025B\n"
	"	* not to be confused with glyph variants of capital E\n"
	"	x (euler constant - 2107)\n"
	"	x (turned digit three - 218B)\n"
	"	x (latin capital letter tresillo - A72A)"},
/* 0191 */ { "LATIN CAPITAL LETTER F WITH HOOK","	* African\n"
	"	x (latin capital letter f with stroke - A798)"},
/* 0192 */ { "LATIN SMALL LETTER F WITH HOOK","	= script f\n"
	"	= Florin currency symbol (Netherlands)\n"
	"	= function symbol\n"
	"	* used as abbreviation convention for folder"},
/* 0193 */ { "LATIN CAPITAL LETTER G WITH HOOK","	* African\n"
	"	* lowercase is 0260"},
/* 0194 */ { "LATIN CAPITAL LETTER GAMMA","	* African\n"
	"	* lowercase is 0263"},
/* 0195 */ { "LATIN SMALL LETTER HV","	= hwair\n"
	"	* Gothic transliteration\n"
	"	* uppercase is 01F6\n"
	"	x (gothic letter hwair - 10348)"},
/* 0196 */ { "LATIN CAPITAL LETTER IOTA","	* African languages\n"
	"	* lowercase is 0269\n"
	"	x (cyrillic capital letter iota - A646)"},
/* 0197 */ { "LATIN CAPITAL LETTER I WITH STROKE","	= barred i, i bar\n"
	"	* African languages\n"
	"	* lowercase is 0268\n"
	"	* ISO 6438 gives lowercase as 026A, not 0268\n"
	"	x (latin letter small capital i - 026A)\n"
	"	x (latin small capital letter i with stroke - 1D7B)"},
/* 0198 */ { "LATIN CAPITAL LETTER K WITH HOOK",NULL},
/* 0199 */ { "LATIN SMALL LETTER K WITH HOOK","	* Hausa, Pan-Nigerian alphabet"},
/* 019A */ { "LATIN SMALL LETTER L WITH BAR","	= barred l\n"
	"	* Americanist phonetic usage for IPA ɬ\n"
	"	* uppercase is 023D\n"
	"	x (latin small letter l with stroke - 0142)\n"
	"	x (latin small letter l with belt - 026C)\n"
	"	x (latin small letter l with double bar - 2C61)\n"
	"	x (latin small letter l with high stroke - A749)"},
/* 019B */ { "LATIN SMALL LETTER LAMBDA WITH STROKE","	= barred lambda, lambda bar\n"
	"	* Americanist phonetic usage for IPA tɬ\n"
	"	* uppercase is A7DC\n"
	"	x (greek small letter lamda - 03BB)"},
/* 019C */ { "LATIN CAPITAL LETTER TURNED M","	* Zhuang (old orthography)\n"
	"	* lowercase is 026F\n"
	"	x (latin letter small capital turned m - A7FA)\n"
	"	x (latin epigraphic letter inverted m - A7FD)"},
/* 019D */ { "LATIN CAPITAL LETTER N WITH LEFT HOOK","	* African\n"
	"	* lowercase is 0272"},
/* 019E */ { "LATIN SMALL LETTER N WITH LONG RIGHT LEG","	* archaic phonetic for Japanese 3093\n"
	"	* recommended spelling for syllabic n is 006E 0329\n"
	"	* Lakota (indicates nasalization of vowel), superseded by 014B\n"
	"	* uppercase is 0220\n"
	"	x (latin small letter eng - 014B)\n"
	"	x (hiragana letter n - 3093)"},
/* 019F */ { "LATIN CAPITAL LETTER O WITH MIDDLE TILDE","	= barred o, o bar\n"
	"	* African languages\n"
	"	* the formal name does not describe the glyph\n"
	"	* lowercase is 0275\n"
	"	x (cyrillic capital letter barred o - 04E8)"},
/* 01A0 */ { "LATIN CAPITAL LETTER O WITH HORN","	: 004F 031B"},
/* 01A1 */ { "LATIN SMALL LETTER O WITH HORN","	* Vietnamese\n"
	"	: 006F 031B"},
/* 01A2 */ { "LATIN CAPITAL LETTER OI","	% LATIN CAPITAL LETTER GHA"},
/* 01A3 */ { "LATIN SMALL LETTER OI","	% LATIN SMALL LETTER GHA\n"
	"	* Pan-Turkic Latin alphabets"},
/* 01A4 */ { "LATIN CAPITAL LETTER P WITH HOOK",NULL},
/* 01A5 */ { "LATIN SMALL LETTER P WITH HOOK","	* African"},
/* 01A6 */ { "LATIN LETTER YR","	* Old Norse\n"
	"	* from German Standard DIN 31624 and ISO 5426-2\n"
	"	* lowercase is 0280"},
/* 01A7 */ { "LATIN CAPITAL LETTER TONE TWO","	x (latin letter voiced laryngeal spirant - 1D24)\n"
	"	x (cyrillic capital letter reversed dze - A644)\n"
	"	x (roman sextula sign - 10193)"},
/* 01A8 */ { "LATIN SMALL LETTER TONE TWO","	* Zhuang (old orthography)\n"
	"	* typographically a reversed S\n"
	"	x (latin small letter tone six - 0185)\n"
	"	x (cyrillic small letter reversed dze - A645)"},
/* 01A9 */ { "LATIN CAPITAL LETTER ESH","	* African languages\n"
	"	* lowercase is 0283\n"
	"	x (greek capital letter sigma - 03A3)\n"
	"	x (n-ary summation - 2211)"},
/* 01AA */ { "LATIN LETTER REVERSED ESH LOOP","	* archaic phonetic for labialized palatoalveolar or palatal fricative\n"
	"	* Twi\n"
	"	* recommended spellings 0283 02B7, 00E7 02B7, 0068 0265, etc."},
/* 01AB */ { "LATIN SMALL LETTER T WITH PALATAL HOOK","	* archaic phonetic for palatalized alveolar or dental stop\n"
	"	* recommended spelling 0074 02B2"},
/* 01AC */ { "LATIN CAPITAL LETTER T WITH HOOK","	* a glyph variant with hook at the right also occurs"},
/* 01AD */ { "LATIN SMALL LETTER T WITH HOOK","	* African languages"},
/* 01AE */ { "LATIN CAPITAL LETTER T WITH RETROFLEX HOOK","	* African languages\n"
	"	* lowercase is 0288"},
/* 01AF */ { "LATIN CAPITAL LETTER U WITH HORN","	: 0055 031B"},
/* 01B0 */ { "LATIN SMALL LETTER U WITH HORN","	* Vietnamese\n"
	"	: 0075 031B"},
/* 01B1 */ { "LATIN CAPITAL LETTER UPSILON","	* African languages\n"
	"	* typographically, turned capital Greek omega 03A9\n"
	"	* lowercase is 028A\n"
	"	x (inverted ohm sign - 2127)"},
/* 01B2 */ { "LATIN CAPITAL LETTER V WITH HOOK","	= script v\n"
	"	* African\n"
	"	* lowercase is 028B"},
/* 01B3 */ { "LATIN CAPITAL LETTER Y WITH HOOK","	* a glyph variant with hook at the left also occurs"},
/* 01B4 */ { "LATIN SMALL LETTER Y WITH HOOK","	* Bini, Esoko, and other Edo languages in West Africa"},
/* 01B5 */ { "LATIN CAPITAL LETTER Z WITH STROKE",NULL},
/* 01B6 */ { "LATIN SMALL LETTER Z WITH STROKE","	= barred z, z bar\n"
	"	* Pan-Turkic Latin orthography\n"
	"	* also a handwritten variant of Latin \"z\"\n"
	"	x (latin small letter z - 007A)\n"
	"	x (latin small letter z with middle tilde - 1D76)"},
/* 01B7 */ { "LATIN CAPITAL LETTER EZH","	* African, Skolt Sami\n"
	"	* lowercase is 0292\n"
	"	x (latin capital letter yogh - 021C)\n"
	"	x (cyrillic capital letter abkhasian dze - 04E0)\n"
	"	x (latin letter small capital ezh - 1D23)\n"
	"	x (latin capital letter reversed open e - A7AB)"},
/* 01B8 */ { "LATIN CAPITAL LETTER EZH REVERSED","	x (latin capital letter open e - 0190)"},
/* 01B9 */ { "LATIN SMALL LETTER EZH REVERSED","	* archaic phonetic for voiced pharyngeal fricative\n"
	"	* sometimes typographically rendered as a turned digit 3\n"
	"	* recommended spelling is 0295\n"
	"	x (latin letter pharyngeal voiced fricative - 0295)\n"
	"	x (arabic letter ain - 0639)"},
/* 01BA */ { "LATIN SMALL LETTER EZH WITH TAIL","	* archaic phonetic for labialized voiced palatoalveolar or palatal fricative\n"
	"	* Twi\n"
	"	* recommended spellings 0292 02B7 or 006A 02B7"},
/* 01BB */ { "LATIN LETTER TWO WITH STROKE","	* archaic phonetic for [dz] affricate\n"
	"	* recommended spellings 0292 or 0064 007A\n"
	"	x (latin capital letter tone two - 01A7)"},
/* 01BC */ { "LATIN CAPITAL LETTER TONE FIVE",NULL},
/* 01BD */ { "LATIN SMALL LETTER TONE FIVE","	* Zhuang (old orthography)\n"
	"	x (latin small letter tone six - 0185)"},
/* 01BE */ { "LATIN LETTER INVERTED GLOTTAL STOP WITH STROKE","	* archaic phonetic for [ts] affricate\n"
	"	* recommended spelling 0074 0073\n"
	"	* letter form is actually derived from ligation of ts, rather than inverted glottal stop\n"
	"	x (latin letter inverted glottal stop - 0296)\n"
	"	x (latin small letter ts digraph - 02A6)"},
/* 01BF */ { "LATIN LETTER WYNN","	= wen\n"
	"	* Runic letter borrowed into Latin script\n"
	"	* replaced by \"w\" in modern transcriptions of Old English\n"
	"	* uppercase is 01F7\n"
	"	x (runic letter wunjo wynn w - 16B9)"},
/* 01C0 */ { "LATIN LETTER DENTAL CLICK","	= pipe\n"
	"	* \"c\" in Zulu orthography\n"
	"	x (solidus - 002F)\n"
	"	x (vertical line - 007C)\n"
	"	x (latin small letter turned t - 0287)\n"
	"	x (divides - 2223)"},
/* 01C1 */ { "LATIN LETTER LATERAL CLICK","	= double pipe\n"
	"	* \"x\" in Zulu orthography\n"
	"	x (latin letter inverted glottal stop - 0296)\n"
	"	x (double vertical line - 2016)\n"
	"	x (parallel to - 2225)"},
/* 01C2 */ { "LATIN LETTER ALVEOLAR CLICK","	= double-barred pipe\n"
	"	* palatoalveolar click (IPA)\n"
	"	x (not equal to - 2260)\n"
	"	x (thermodynamic - 29E7)"},
/* 01C3 */ { "LATIN LETTER RETROFLEX CLICK","	= latin letter exclamation mark (1.0)\n"
	"	* (post)alveolar click (IPA)\n"
	"	* \"q\" in Zulu orthography\n"
	"	x (exclamation mark - 0021)\n"
	"	x (latin letter stretched c - 0297)\n"
	"	x (modifier letter raised exclamation mark - A71D)\n"
	"	x (latin letter retroflex click with retroflex hook - 1DF0A)"},
/* 01C4 */ { "LATIN CAPITAL LETTER DZ WITH CARON","	# 0044 017D"},
/* 01C5 */ { "LATIN CAPITAL LETTER D WITH SMALL LETTER Z WITH CARON","	# 0044 017E"},
/* 01C6 */ { "LATIN SMALL LETTER DZ WITH CARON","	x (cyrillic small letter dzhe - 045F)\n"
	"	# 0064 017E"},
/* 01C7 */ { "LATIN CAPITAL LETTER LJ","	# 004C 004A"},
/* 01C8 */ { "LATIN CAPITAL LETTER L WITH SMALL LETTER J","	# 004C 006A"},
/* 01C9 */ { "LATIN SMALL LETTER LJ","	x (cyrillic small letter lje - 0459)\n"
	"	# 006C 006A"},
/* 01CA */ { "LATIN CAPITAL LETTER NJ","	# 004E 004A"},
/* 01CB */ { "LATIN CAPITAL LETTER N WITH SMALL LETTER J","	# 004E 006A"},
/* 01CC */ { "LATIN SMALL LETTER NJ","	x (cyrillic small letter nje - 045A)\n"
	"	# 006E 006A"},
/* 01CD */ { "LATIN CAPITAL LETTER A WITH CARON","	: 0041 030C"},
/* 01CE */ { "LATIN SMALL LETTER A WITH CARON","	* Pinyin third tone\n"
	"	: 0061 030C"},
/* 01CF */ { "LATIN CAPITAL LETTER I WITH CARON","	: 0049 030C"},
/* 01D0 */ { "LATIN SMALL LETTER I WITH CARON","	* Pinyin third tone\n"
	"	: 0069 030C"},
/* 01D1 */ { "LATIN CAPITAL LETTER O WITH CARON","	: 004F 030C"},
/* 01D2 */ { "LATIN SMALL LETTER O WITH CARON","	* Pinyin third tone\n"
	"	: 006F 030C"},
/* 01D3 */ { "LATIN CAPITAL LETTER U WITH CARON","	: 0055 030C"},
/* 01D4 */ { "LATIN SMALL LETTER U WITH CARON","	* Pinyin third tone\n"
	"	: 0075 030C"},
/* 01D5 */ { "LATIN CAPITAL LETTER U WITH DIAERESIS AND MACRON","	: 00DC 0304"},
/* 01D6 */ { "LATIN SMALL LETTER U WITH DIAERESIS AND MACRON","	* Pinyin first tone\n"
	"	: 00FC 0304"},
/* 01D7 */ { "LATIN CAPITAL LETTER U WITH DIAERESIS AND ACUTE","	: 00DC 0301"},
/* 01D8 */ { "LATIN SMALL LETTER U WITH DIAERESIS AND ACUTE","	* Pinyin second tone\n"
	"	: 00FC 0301"},
/* 01D9 */ { "LATIN CAPITAL LETTER U WITH DIAERESIS AND CARON","	: 00DC 030C"},
/* 01DA */ { "LATIN SMALL LETTER U WITH DIAERESIS AND CARON","	* Pinyin third tone\n"
	"	: 00FC 030C"},
/* 01DB */ { "LATIN CAPITAL LETTER U WITH DIAERESIS AND GRAVE","	: 00DC 0300"},
/* 01DC */ { "LATIN SMALL LETTER U WITH DIAERESIS AND GRAVE","	* Pinyin fourth tone\n"
	"	: 00FC 0300"},
/* 01DD */ { "LATIN SMALL LETTER TURNED E","	* Pan-Nigerian alphabet\n"
	"	* all other usages of schwa are 0259\n"
	"	* uppercase is 018E\n"
	"	x (latin small letter schwa - 0259)"},
/* 01DE */ { "LATIN CAPITAL LETTER A WITH DIAERESIS AND MACRON","	: 00C4 0304"},
/* 01DF */ { "LATIN SMALL LETTER A WITH DIAERESIS AND MACRON","	* Livonian, Uralicist usage\n"
	"	: 00E4 0304"},
/* 01E0 */ { "LATIN CAPITAL LETTER A WITH DOT ABOVE AND MACRON","	: 0226 0304"},
/* 01E1 */ { "LATIN SMALL LETTER A WITH DOT ABOVE AND MACRON","	* Uralicist usage\n"
	"	: 0227 0304"},
/* 01E2 */ { "LATIN CAPITAL LETTER AE WITH MACRON","	: 00C6 0304"},
/* 01E3 */ { "LATIN SMALL LETTER AE WITH MACRON","	* Old Norse, Old English\n"
	"	: 00E6 0304"},
/* 01E4 */ { "LATIN CAPITAL LETTER G WITH STROKE",NULL},
/* 01E5 */ { "LATIN SMALL LETTER G WITH STROKE","	* Skolt Sami"},
/* 01E6 */ { "LATIN CAPITAL LETTER G WITH CARON","	: 0047 030C"},
/* 01E7 */ { "LATIN SMALL LETTER G WITH CARON","	* Skolt Sami\n"
	"	x (latin small letter g with breve - 011F)\n"
	"	: 0067 030C"},
/* 01E8 */ { "LATIN CAPITAL LETTER K WITH CARON","	: 004B 030C"},
/* 01E9 */ { "LATIN SMALL LETTER K WITH CARON","	* Skolt Sami\n"
	"	: 006B 030C"},
/* 01EA */ { "LATIN CAPITAL LETTER O WITH OGONEK","	: 004F 0328"},
/* 01EB */ { "LATIN SMALL LETTER O WITH OGONEK","	* Sami, Iroquoian, Old Icelandic\n"
	"	: 006F 0328"},
/* 01EC */ { "LATIN CAPITAL LETTER O WITH OGONEK AND MACRON","	: 01EA 0304"},
/* 01ED */ { "LATIN SMALL LETTER O WITH OGONEK AND MACRON","	* Old Icelandic, Livonian (in recent linguistic scholarship)\n"
	"	: 01EB 0304"},
/* 01EE */ { "LATIN CAPITAL LETTER EZH WITH CARON","	: 01B7 030C"},
/* 01EF */ { "LATIN SMALL LETTER EZH WITH CARON","	* Skolt Sami\n"
	"	: 0292 030C"},
/* 01F0 */ { "LATIN SMALL LETTER J WITH CARON","	* IPA and many languages\n"
	"	: 006A 030C"},
/* 01F1 */ { "LATIN CAPITAL LETTER DZ","	# 0044 005A"},
/* 01F2 */ { "LATIN CAPITAL LETTER D WITH SMALL LETTER Z","	# 0044 007A"},
/* 01F3 */ { "LATIN SMALL LETTER DZ","	# 0064 007A"},
/* 01F4 */ { "LATIN CAPITAL LETTER G WITH ACUTE","	: 0047 0301"},
/* 01F5 */ { "LATIN SMALL LETTER G WITH ACUTE","	* Macedonian and Serbian transliteration\n"
	"	: 0067 0301"},
/* 01F6 */ { "LATIN CAPITAL LETTER HWAIR","	* lowercase is 0195\n"
	"	x (cyrillic capital letter komi nje - 050A)\n"
	"	x (gothic letter hwair - 10348)"},
/* 01F7 */ { "LATIN CAPITAL LETTER WYNN","	= wen\n"
	"	* lowercase is 01BF"},
/* 01F8 */ { "LATIN CAPITAL LETTER N WITH GRAVE","	: 004E 0300"},
/* 01F9 */ { "LATIN SMALL LETTER N WITH GRAVE","	* Pinyin\n"
	"	: 006E 0300"},
/* 01FA */ { "LATIN CAPITAL LETTER A WITH RING ABOVE AND ACUTE","	: 00C5 0301"},
/* 01FB */ { "LATIN SMALL LETTER A WITH RING ABOVE AND ACUTE","	: 00E5 0301"},
/* 01FC */ { "LATIN CAPITAL LETTER AE WITH ACUTE","	: 00C6 0301"},
/* 01FD */ { "LATIN SMALL LETTER AE WITH ACUTE","	: 00E6 0301"},
/* 01FE */ { "LATIN CAPITAL LETTER O WITH STROKE AND ACUTE","	: 00D8 0301"},
/* 01FF */ { "LATIN SMALL LETTER O WITH STROKE AND ACUTE","	: 00F8 0301"}
};

UN_DLL_LOCAL
static const struct unicode_nameannot una_00_02[] = {
/* 0200 */ { "LATIN CAPITAL LETTER A WITH DOUBLE GRAVE","	: 0041 030F"},
/* 0201 */ { "LATIN SMALL LETTER A WITH DOUBLE GRAVE","	: 0061 030F"},
/* 0202 */ { "LATIN CAPITAL LETTER A WITH INVERTED BREVE","	: 0041 0311"},
/* 0203 */ { "LATIN SMALL LETTER A WITH INVERTED BREVE","	: 0061 0311"},
/* 0204 */ { "LATIN CAPITAL LETTER E WITH DOUBLE GRAVE","	: 0045 030F"},
/* 0205 */ { "LATIN SMALL LETTER E WITH DOUBLE GRAVE","	: 0065 030F"},
/* 0206 */ { "LATIN CAPITAL LETTER E WITH INVERTED BREVE","	: 0045 0311"},
/* 0207 */ { "LATIN SMALL LETTER E WITH INVERTED BREVE","	: 0065 0311"},
/* 0208 */ { "LATIN CAPITAL LETTER I WITH DOUBLE GRAVE","	: 0049 030F"},
/* 0209 */ { "LATIN SMALL LETTER I WITH DOUBLE GRAVE","	: 0069 030F"},
/* 020A */ { "LATIN CAPITAL LETTER I WITH INVERTED BREVE","	: 0049 0311"},
/* 020B */ { "LATIN SMALL LETTER I WITH INVERTED BREVE","	: 0069 0311"},
/* 020C */ { "LATIN CAPITAL LETTER O WITH DOUBLE GRAVE","	: 004F 030F"},
/* 020D */ { "LATIN SMALL LETTER O WITH DOUBLE GRAVE","	: 006F 030F"},
/* 020E */ { "LATIN CAPITAL LETTER O WITH INVERTED BREVE","	: 004F 0311"},
/* 020F */ { "LATIN SMALL LETTER O WITH INVERTED BREVE","	: 006F 0311"},
/* 0210 */ { "LATIN CAPITAL LETTER R WITH DOUBLE GRAVE","	: 0052 030F"},
/* 0211 */ { "LATIN SMALL LETTER R WITH DOUBLE GRAVE","	: 0072 030F"},
/* 0212 */ { "LATIN CAPITAL LETTER R WITH INVERTED BREVE","	: 0052 0311"},
/* 0213 */ { "LATIN SMALL LETTER R WITH INVERTED BREVE","	: 0072 0311"},
/* 0214 */ { "LATIN CAPITAL LETTER U WITH DOUBLE GRAVE","	: 0055 030F"},
/* 0215 */ { "LATIN SMALL LETTER U WITH DOUBLE GRAVE","	: 0075 030F"},
/* 0216 */ { "LATIN CAPITAL LETTER U WITH INVERTED BREVE","	: 0055 0311"},
/* 0217 */ { "LATIN SMALL LETTER U WITH INVERTED BREVE","	: 0075 0311"},
/* 0218 */ { "LATIN CAPITAL LETTER S WITH COMMA BELOW","	: 0053 0326"},
/* 0219 */ { "LATIN SMALL LETTER S WITH COMMA BELOW","	x (latin small letter s with cedilla - 015F)\n"
	"	: 0073 0326"},
/* 021A */ { "LATIN CAPITAL LETTER T WITH COMMA BELOW","	: 0054 0326"},
/* 021B */ { "LATIN SMALL LETTER T WITH COMMA BELOW","	x (latin small letter t with cedilla - 0163)\n"
	"	: 0074 0326"},
/* 021C */ { "LATIN CAPITAL LETTER YOGH","	x (latin capital letter ezh - 01B7)\n"
	"	x (cyrillic capital letter ze - 0417)\n"
	"	x (latin capital letter reversed open e - A7AB)"},
/* 021D */ { "LATIN SMALL LETTER YOGH","	* Middle English, Scots\n"
	"	x (latin small letter ezh - 0292)\n"
	"	x (latin small letter insular g - 1D79)\n"
	"	x (latin small letter et - A76B)"},
/* 021E */ { "LATIN CAPITAL LETTER H WITH CARON","	: 0048 030C"},
/* 021F */ { "LATIN SMALL LETTER H WITH CARON","	* Finnish Romany\n"
	"	: 0068 030C"},
/* 0220 */ { "LATIN CAPITAL LETTER N WITH LONG RIGHT LEG","	* Lakota (nasalization), superseded by 014A\n"
	"	* lowercase is 019E\n"
	"	x (latin capital letter eng - 014A)"},
/* 0221 */ { "LATIN SMALL LETTER D WITH CURL","	* phonetic use in Sinology"},
/* 0222 */ { "LATIN CAPITAL LETTER OU","	x (latin letter small capital ou - 1D15)"},
/* 0223 */ { "LATIN SMALL LETTER OU","	* Algonquin, Huron\n"
	"	x (digit eight - 0038)"},
/* 0224 */ { "LATIN CAPITAL LETTER Z WITH HOOK","	x (latin capital letter z with descender - 2C6B)"},
/* 0225 */ { "LATIN SMALL LETTER Z WITH HOOK","	* Middle High German"},
/* 0226 */ { "LATIN CAPITAL LETTER A WITH DOT ABOVE","	: 0041 0307"},
/* 0227 */ { "LATIN SMALL LETTER A WITH DOT ABOVE","	* Uralicist usage\n"
	"	: 0061 0307"},
/* 0228 */ { "LATIN CAPITAL LETTER E WITH CEDILLA","	: 0045 0327"},
/* 0229 */ { "LATIN SMALL LETTER E WITH CEDILLA","	: 0065 0327"},
/* 022A */ { "LATIN CAPITAL LETTER O WITH DIAERESIS AND MACRON","	: 00D6 0304"},
/* 022B */ { "LATIN SMALL LETTER O WITH DIAERESIS AND MACRON","	: 00F6 0304"},
/* 022C */ { "LATIN CAPITAL LETTER O WITH TILDE AND MACRON","	: 00D5 0304"},
/* 022D */ { "LATIN SMALL LETTER O WITH TILDE AND MACRON","	: 00F5 0304"},
/* 022E */ { "LATIN CAPITAL LETTER O WITH DOT ABOVE","	: 004F 0307"},
/* 022F */ { "LATIN SMALL LETTER O WITH DOT ABOVE","	: 006F 0307"},
/* 0230 */ { "LATIN CAPITAL LETTER O WITH DOT ABOVE AND MACRON","	: 022E 0304"},
/* 0231 */ { "LATIN SMALL LETTER O WITH DOT ABOVE AND MACRON","	: 022F 0304"},
/* 0232 */ { "LATIN CAPITAL LETTER Y WITH MACRON","	: 0059 0304"},
/* 0233 */ { "LATIN SMALL LETTER Y WITH MACRON","	* also Cornish\n"
	"	: 0079 0304"},
/* 0234 */ { "LATIN SMALL LETTER L WITH CURL",NULL},
/* 0235 */ { "LATIN SMALL LETTER N WITH CURL",NULL},
/* 0236 */ { "LATIN SMALL LETTER T WITH CURL",NULL},
/* 0237 */ { "LATIN SMALL LETTER DOTLESS J","	* this is a non-casing letter\n"
	"	x (greek letter yot - 03F3)\n"
	"	x (cyrillic small letter je - 0458)\n"
	"	x (mathematical italic small dotless j - 1D6A5)"},
/* 0238 */ { "LATIN SMALL LETTER DB DIGRAPH",NULL},
/* 0239 */ { "LATIN SMALL LETTER QP DIGRAPH",NULL},
/* 023A */ { "LATIN CAPITAL LETTER A WITH STROKE","	* lowercase is 2C65"},
/* 023B */ { "LATIN CAPITAL LETTER C WITH STROKE","	x (cedi sign - 20B5)"},
/* 023C */ { "LATIN SMALL LETTER C WITH STROKE","	* also used in Americanist linguistics\n"
	"	x (cent sign - 00A2)"},
/* 023D */ { "LATIN CAPITAL LETTER L WITH BAR","	* lowercase is 019A\n"
	"	x (latin capital letter l with stroke - 0141)\n"
	"	x (latin capital letter l with double bar - 2C60)\n"
	"	x (latin capital letter l with middle tilde - 2C62)\n"
	"	x (latin capital letter l with high stroke - A748)"},
/* 023E */ { "LATIN CAPITAL LETTER T WITH DIAGONAL STROKE","	* lowercase is 2C66"},
/* 023F */ { "LATIN SMALL LETTER S WITH SWASH TAIL","	* voiceless labio-alveolar fricative\n"
	"	* uppercase is 2C7E\n"
	"	x (latin small letter s with hook - 0282)"},
/* 0240 */ { "LATIN SMALL LETTER Z WITH SWASH TAIL","	* voiced labio-alveolar fricative\n"
	"	* uppercase is 2C7F\n"
	"	x (latin small letter z with retroflex hook - 0290)"},
/* 0241 */ { "LATIN CAPITAL LETTER GLOTTAL STOP",NULL},
/* 0242 */ { "LATIN SMALL LETTER GLOTTAL STOP","	* casing use in Chipewyan, Dogrib, Slavey (Canadian aboriginal orthographies)\n"
	"	x (latin letter glottal stop - 0294)\n"
	"	x (modifier letter glottal stop - 02C0)"},
/* 0243 */ { "LATIN CAPITAL LETTER B WITH STROKE","	* lowercase is 0180\n"
	"	x (latin letter small capital barred b - 1D03)"},
/* 0244 */ { "LATIN CAPITAL LETTER U BAR","	* lowercase is 0289"},
/* 0245 */ { "LATIN CAPITAL LETTER TURNED V","	* lowercase is 028C\n"
	"	x (greek capital letter lamda - 039B)"},
/* 0246 */ { "LATIN CAPITAL LETTER E WITH STROKE",NULL},
/* 0247 */ { "LATIN SMALL LETTER E WITH STROKE",NULL},
/* 0248 */ { "LATIN CAPITAL LETTER J WITH STROKE",NULL},
/* 0249 */ { "LATIN SMALL LETTER J WITH STROKE",NULL},
/* 024A */ { "LATIN CAPITAL LETTER SMALL Q WITH HOOK TAIL",NULL},
/* 024B */ { "LATIN SMALL LETTER Q WITH HOOK TAIL",NULL},
/* 024C */ { "LATIN CAPITAL LETTER R WITH STROKE",NULL},
/* 024D */ { "LATIN SMALL LETTER R WITH STROKE","	x (latin small letter r with middle tilde - 1D72)"},
/* 024E */ { "LATIN CAPITAL LETTER Y WITH STROKE",NULL},
/* 024F */ { "LATIN SMALL LETTER Y WITH STROKE",NULL},
/* 0250 */ { "LATIN SMALL LETTER TURNED A","	* low central unrounded vowel\n"
	"	* uppercase is 2C6F"},
/* 0251 */ { "LATIN SMALL LETTER ALPHA","	= latin small letter script a (1.0)\n"
	"	* low back unrounded vowel\n"
	"	* the representative glyph appears like an allograph of Latin \"a\", but sometimes the character is rendered more like a Greek \"α\"\n"
	"	* uppercase is 2C6D\n"
	"	x (greek small letter alpha - 03B1)"},
/* 0252 */ { "LATIN SMALL LETTER TURNED ALPHA","	* low back rounded vowel\n"
	"	* uppercase is 2C70\n"
	"	x (latin small letter inverted alpha - AB64)"},
/* 0253 */ { "LATIN SMALL LETTER B WITH HOOK","	* implosive bilabial stop\n"
	"	* uppercase is 0181\n"
	"	x (cyrillic small letter be - 0431)"},
/* 0254 */ { "LATIN SMALL LETTER OPEN O","	* typographically a turned c\n"
	"	* lower-mid back rounded vowel\n"
	"	* in older Danish usage \"ɔ:\" stands for \"that is\"\n"
	"	* uppercase is 0186\n"
	"	x (latin letter small capital open o - 1D10)\n"
	"	x (latin small letter reversed c - 2184)"},
/* 0255 */ { "LATIN SMALL LETTER C WITH CURL","	* voiceless alveolo-palatal laminal fricative\n"
	"	* used in transcription of Mandarin Chinese\n"
	"	* sound spelled with 015B in Polish"},
/* 0256 */ { "LATIN SMALL LETTER D WITH TAIL","	= d retroflex hook\n"
	"	* voiced retroflex stop\n"
	"	* uppercase is 0189"},
/* 0257 */ { "LATIN SMALL LETTER D WITH HOOK","	* implosive dental or alveolar stop\n"
	"	* Ewe, Pan-Nigerian alphabet\n"
	"	* uppercase is 018A"},
/* 0258 */ { "LATIN SMALL LETTER REVERSED E","	* upper-mid central unrounded vowel"},
/* 0259 */ { "LATIN SMALL LETTER SCHWA","	* mid-central unrounded vowel\n"
	"	* uppercase is 018F\n"
	"	* variant uppercase form 018E is associated with 01DD\n"
	"	x (latin small letter turned e - 01DD)\n"
	"	x (cyrillic small letter schwa - 04D9)"},
/* 025A */ { "LATIN SMALL LETTER SCHWA WITH HOOK","	* rhotacized schwa"},
/* 025B */ { "LATIN SMALL LETTER OPEN E","	= epsilon\n"
	"	* lower-mid front unrounded vowel\n"
	"	* uppercase is 0190\n"
	"	x (greek small letter epsilon - 03B5)\n"
	"	x (cyrillic small letter reversed ze - 0511)"},
/* 025C */ { "LATIN SMALL LETTER REVERSED OPEN E","	= reversed epsilon\n"
	"	* lower-mid central unrounded vowel\n"
	"	* uppercase is A7AB\n"
	"	x (cyrillic small letter ze - 0437)\n"
	"	x (latin small letter turned open e - 1D08)"},
/* 025D */ { "LATIN SMALL LETTER REVERSED OPEN E WITH HOOK","	* rhotacized lower-mid central vowel"},
/* 025E */ { "LATIN SMALL LETTER CLOSED REVERSED OPEN E","	= closed reversed epsilon\n"
	"	* lower-mid central rounded vowel\n"
	"	x (latin small letter closed open e - 029A)"},
/* 025F */ { "LATIN SMALL LETTER DOTLESS J WITH STROKE","	* voiced palatal stop\n"
	"	* typographically a turned f, but better thought of as a form of j\n"
	"	* \"gy\" in Hungarian orthography\n"
	"	* also archaic phonetic for palatoalveolar affricate 02A4"},
/* 0260 */ { "LATIN SMALL LETTER G WITH HOOK","	* implosive velar stop\n"
	"	* uppercase is 0193"},
/* 0261 */ { "LATIN SMALL LETTER SCRIPT G","	* voiced velar stop\n"
	"	* uppercase is A7AC\n"
	"	x (latin small letter g - 0067)\n"
	"	x (script small g - 210A)"},
/* 0262 */ { "LATIN LETTER SMALL CAPITAL G","	* voiced uvular stop"},
/* 0263 */ { "LATIN SMALL LETTER GAMMA","	* voiced velar fricative\n"
	"	* uppercase is 0194\n"
	"	x (greek small letter gamma - 03B3)"},
/* 0264 */ { "LATIN SMALL LETTER RAMS HORN","	= latin small letter baby gamma (1.0)\n"
	"	* upper-mid back unrounded vowel\n"
	"	* uppercase is A7CB"},
/* 0265 */ { "LATIN SMALL LETTER TURNED H","	* voiced rounded palatal approximant\n"
	"	* uppercase is A78D"},
/* 0266 */ { "LATIN SMALL LETTER H WITH HOOK","	* breathy-voiced glottal fricative\n"
	"	* uppercase is A7AA\n"
	"	* uppercase in Nawdm is 0124\n"
	"	x (modifier letter small h with hook - 02B1)"},
/* 0267 */ { "LATIN SMALL LETTER HENG WITH HOOK","	* voiceless coarticulated velar and palatoalveolar fricative\n"
	"	* \"sj\" in some Swedish dialects\n"
	"	x (latin small letter heng - A727)"},
/* 0268 */ { "LATIN SMALL LETTER I WITH STROKE","	= barred i, i bar\n"
	"	* high central unrounded vowel\n"
	"	* uppercase is 0197\n"
	"	* ISO 6438 gives lowercase of 0197 as 026A, not 0268\n"
	"	x (latin small capital letter i with stroke - 1D7B)"},
/* 0269 */ { "LATIN SMALL LETTER IOTA","	* semi-high front unrounded vowel\n"
	"	* obsoleted by IPA in 1989\n"
	"	* preferred use is 026A latin letter small capital i\n"
	"	* uppercase is 0196\n"
	"	x (latin small letter dotless i - 0131)\n"
	"	x (greek small letter iota - 03B9)\n"
	"	x (cyrillic small letter iota - A647)\n"
	"	x (mathematical italic small dotless i - 1D6A4)"},
/* 026A */ { "LATIN LETTER SMALL CAPITAL I","	* semi-high front unrounded vowel\n"
	"	* preferred IPA alternate for 0269\n"
	"	* uppercase is A7AE\n"
	"	* ISO 6438 gives this as lowercase of 0197 instead of 0268\n"
	"	* glyph should have top and bottom serifs even in sans serif IPA font designs, to avoid confusion with lowercase l\n"
	"	x (latin small letter dotless i - 0131)"},
/* 026B */ { "LATIN SMALL LETTER L WITH MIDDLE TILDE","	* velarized voiced alveolar lateral approximant\n"
	"	* uppercase is 2C62\n"
	"	x (latin small letter l with stroke - 0142)\n"
	"	x (latin small letter l with inverted lazy s - AB37)\n"
	"	x (latin small letter l with double middle tilde - AB38)"},
/* 026C */ { "LATIN SMALL LETTER L WITH BELT","	* voiceless alveolar lateral fricative\n"
	"	* uppercase is A7AD\n"
	"	x (latin small letter l with bar - 019A)"},
/* 026D */ { "LATIN SMALL LETTER L WITH RETROFLEX HOOK","	* voiced retroflex lateral\n"
	"	x (latin small letter l with retroflex hook and belt - A78E)"},
/* 026E */ { "LATIN SMALL LETTER LEZH","	* voiced lateral fricative\n"
	"	* \"dhl\" in Zulu orthography"},
/* 026F */ { "LATIN SMALL LETTER TURNED M","	* high back unrounded vowel\n"
	"	* uppercase is 019C\n"
	"	x (latin letter small capital turned m - A7FA)"},
/* 0270 */ { "LATIN SMALL LETTER TURNED M WITH LONG LEG","	* voiced velar approximant"},
/* 0271 */ { "LATIN SMALL LETTER M WITH HOOK","	* voiced labiodental nasal\n"
	"	* uppercase is 2C6E\n"
	"	x (latin small letter m with palatal hook - 1D86)"},
/* 0272 */ { "LATIN SMALL LETTER N WITH LEFT HOOK","	* voiced palatal nasal\n"
	"	* uppercase is 019D"},
/* 0273 */ { "LATIN SMALL LETTER N WITH RETROFLEX HOOK","	* voiced retroflex nasal"},
/* 0274 */ { "LATIN LETTER SMALL CAPITAL N","	* voiced uvular nasal"},
/* 0275 */ { "LATIN SMALL LETTER BARRED O","	= o bar\n"
	"	* rounded mid-central vowel, i.e. rounded schwa\n"
	"	* uppercase is 019F\n"
	"	x (greek small letter theta - 03B8)\n"
	"	x (cyrillic small letter fita - 0473)\n"
	"	x (cyrillic small letter barred o - 04E9)"},
/* 0276 */ { "LATIN LETTER SMALL CAPITAL OE","	* low front rounded vowel\n"
	"	x (latin small ligature oe - 0153)"},
/* 0277 */ { "LATIN SMALL LETTER CLOSED OMEGA","	* near-close near-back rounded vowel\n"
	"	* obsoleted by IPA in 1989\n"
	"	* preferred use is 028A latin small letter upsilon\n"
	"	x (latin small letter omega - A7B7)\n"
	"	x (greek letter small capital omega - AB65)"},
/* 0278 */ { "LATIN SMALL LETTER PHI","	* voiceless bilabial fricative\n"
	"	* this is a non-casing letter\n"
	"	x (greek small letter phi - 03C6)\n"
	"	x (greek phi symbol - 03D5)\n"
	"	x (latin small letter tailless phi - 2C77)"},
/* 0279 */ { "LATIN SMALL LETTER TURNED R","	* voiced alveolar approximant\n"
	"	x (modifier letter small turned r - 02B4)"},
/* 027A */ { "LATIN SMALL LETTER TURNED R WITH LONG LEG","	* voiced lateral flap"},
/* 027B */ { "LATIN SMALL LETTER TURNED R WITH HOOK","	* voiced retroflex approximant\n"
	"	x (modifier letter small turned r with hook - 02B5)"},
/* 027C */ { "LATIN SMALL LETTER R WITH LONG LEG","	* voiced strident apico-alveolar trill\n"
	"	* obsoleted by IPA in 1989\n"
	"	* this sound is spelled with 0159 in Czech\n"
	"	* preferred phonetic representation for Czech is 0072 031D\n"
	"	* in current use in Gaelic types (as glyph variant of 0072)\n"
	"	x (latin small letter insular r - A783)\n"
	"	x (latin small letter insular s - A785)"},
/* 027D */ { "LATIN SMALL LETTER R WITH TAIL","	* voiced retroflex flap\n"
	"	* uppercase is 2C64\n"
	"	x (latin small letter turned r with tail - 2C79)"},
/* 027E */ { "LATIN SMALL LETTER R WITH FISHHOOK","	* voiced alveolar flap or tap\n"
	"	x (latin small letter long s - 017F)"},
/* 027F */ { "LATIN SMALL LETTER REVERSED R WITH FISHHOOK","	= long leg turned iota (a misnomer)\n"
	"	* apical dental vowel\n"
	"	* used by linguists working on Chinese and other Sino-Tibetan languages\n"
	"	* IPA spelling - 007A 0329\n"
	"	* preferred presentation is with a descender"},
/* 0280 */ { "LATIN LETTER SMALL CAPITAL R","	* voiced uvular trill\n"
	"	* Germanic, Old Norse\n"
	"	* uppercase is 01A6"},
/* 0281 */ { "LATIN LETTER SMALL CAPITAL INVERTED R","	* voiced uvular fricative or approximant\n"
	"	x (modifier letter small capital inverted r - 02B6)"},
/* 0282 */ { "LATIN SMALL LETTER S WITH HOOK","	* voiceless retroflex fricative\n"
	"	* uppercase is A7C5"},
/* 0283 */ { "LATIN SMALL LETTER ESH","	* voiceless postalveolar fricative\n"
	"	* uppercase is 01A9\n"
	"	x (integral - 222B)\n"
	"	x (latin small letter baseline esh - AB4D)"},
/* 0284 */ { "LATIN SMALL LETTER DOTLESS J WITH STROKE AND HOOK","	* implosive palatal stop\n"
	"	* typographically based on 025F, not on 0283"},
/* 0285 */ { "LATIN SMALL LETTER SQUAT REVERSED ESH","	* apical retroflex vowel\n"
	"	* used by linguists working on Chinese and other Sino-Tibetan languages\n"
	"	* IPA spelling - 0290 0329\n"
	"	* in origin 027F plus the retroflex hook 0322, despite its name\n"
	"	* preferred presentation is with a descender"},
/* 0286 */ { "LATIN SMALL LETTER ESH WITH CURL","	* palatalized voiceless postalveolar fricative\n"
	"	* suggested spelling - 0283 02B2"},
/* 0287 */ { "LATIN SMALL LETTER TURNED T","	* dental click (sound of \"tsk tsk\")\n"
	"	* uppercase is A7B1\n"
	"	x (latin letter dental click - 01C0)"},
/* 0288 */ { "LATIN SMALL LETTER T WITH RETROFLEX HOOK","	* voiceless retroflex stop\n"
	"	* uppercase is 01AE\n"
	"	x (latin small letter t with hook - 01AD)\n"
	"	x (latin small letter t with hook and retroflex hook - 1DF09)"},
/* 0289 */ { "LATIN SMALL LETTER U BAR","	* high central rounded vowel\n"
	"	* uppercase is 0244\n"
	"	x (latin small capital letter u with stroke - 1D7E)"},
/* 028A */ { "LATIN SMALL LETTER UPSILON","	* near-close near-back rounded vowel\n"
	"	* typographically, a turned Greek small capital omega AB65\n"
	"	* preferred IPA alternate to 0277\n"
	"	* uppercase is 01B1\n"
	"	x (greek small letter upsilon - 03C5)"},
/* 028B */ { "LATIN SMALL LETTER V WITH HOOK","	= latin small letter script v (1.0)\n"
	"	* voiced labiodental approximant\n"
	"	* uppercase is 01B2\n"
	"	x (greek small letter upsilon - 03C5)"},
/* 028C */ { "LATIN SMALL LETTER TURNED V","	= caret, wedge\n"
	"	* lower-mid back unrounded vowel\n"
	"	* uppercase is 0245\n"
	"	x (greek capital letter lamda - 039B)\n"
	"	x (greek letter small capital lamda - 1D27)\n"
	"	x (logical and - 2227)"},
/* 028D */ { "LATIN SMALL LETTER TURNED W","	* voiceless rounded labiovelar approximant\n"
	"	x (modifier letter small turned w - AB69)"},
/* 028E */ { "LATIN SMALL LETTER TURNED Y","	* voiced lateral approximant"},
/* 028F */ { "LATIN LETTER SMALL CAPITAL Y","	* semi-high front rounded vowel"},
/* 0290 */ { "LATIN SMALL LETTER Z WITH RETROFLEX HOOK","	* voiced retroflex fricative"},
/* 0291 */ { "LATIN SMALL LETTER Z WITH CURL","	* voiced alveolo-palatal laminal fricative\n"
	"	* sound spelled with 017A in Polish"},
/* 0292 */ { "LATIN SMALL LETTER EZH","	= dram sign\n"
	"	* voiced postalveolar fricative\n"
	"	* mistakenly named yogh in Unicode 1.0\n"
	"	* uppercase is 01B7\n"
	"	* Skolt Sami\n"
	"	x (latin small letter yogh - 021D)\n"
	"	x (cyrillic small letter abkhasian dze - 04E1)\n"
	"	x (ounce sign - 2125)\n"
	"	x (alchemical symbol for half dram - 1F772)\n"
	"	x 2CF01"},
/* 0293 */ { "LATIN SMALL LETTER EZH WITH CURL","	* palatalized voiced postalveolar fricative"},
/* 0294 */ { "LATIN LETTER GLOTTAL STOP","	* this is a caseless letter\n"
	"	* used in IPA, other phonetic notations, and those orthographies which use a caseless glottal stop\n"
	"	x (latin capital letter glottal stop - 0241)\n"
	"	x (modifier letter glottal stop - 02C0)"},
/* 0295 */ { "LATIN LETTER PHARYNGEAL VOICED FRICATIVE","	= reversed glottal stop\n"
	"	* voiced pharyngeal fricative\n"
	"	* ain\n"
	"	x (latin small letter ezh reversed - 01B9)\n"
	"	x (modifier letter reversed glottal stop - 02C1)"},
/* 0296 */ { "LATIN LETTER INVERTED GLOTTAL STOP","	* lateral click\n"
	"	x (latin letter lateral click - 01C1)"},
/* 0297 */ { "LATIN LETTER STRETCHED C","	* palatal (or alveolar) click\n"
	"	x (latin letter retroflex click - 01C3)\n"
	"	x (complement - 2201)"},
/* 0298 */ { "LATIN LETTER BILABIAL CLICK","	= bullseye\n"
	"	* this is a non-casing letter\n"
	"	x (circled dot operator - 2299)\n"
	"	x (cyrillic capital letter monocular o - A668)"},
/* 0299 */ { "LATIN LETTER SMALL CAPITAL B","	* bilabial trill\n"
	"	x (cyrillic small letter ve - 0432)"},
/* 029A */ { "LATIN SMALL LETTER CLOSED OPEN E","	= closed epsilon\n"
	"	* lower-mid front rounded vowel\n"
	"	* non-IPA alternate for the preferred 0153\n"
	"	x (latin small letter closed reversed open e - 025E)"},
/* 029B */ { "LATIN LETTER SMALL CAPITAL G WITH HOOK","	* voiced uvular implosive"},
/* 029C */ { "LATIN LETTER SMALL CAPITAL H","	* voiceless epiglottal fricative\n"
	"	x (cyrillic small letter en - 043D)\n"
	"	x (modifier letter capital h - 1D34)\n"
	"	x (modifier letter capital h with stroke - A7F8)"},
/* 029D */ { "LATIN SMALL LETTER J WITH CROSSED-TAIL","	* voiced palatal fricative\n"
	"	* uppercase is A7B2"},
/* 029E */ { "LATIN SMALL LETTER TURNED K","	* proposed for velar click\n"
	"	* uppercase is A7B0\n"
	"	* withdrawn by IPA in 1970\n"
	"	x (latin letter small capital turned k - 1DF10)"},
/* 029F */ { "LATIN LETTER SMALL CAPITAL L","	* velar lateral approximant"},
/* 02A0 */ { "LATIN SMALL LETTER Q WITH HOOK","	* voiceless uvular implosive"},
/* 02A1 */ { "LATIN LETTER GLOTTAL STOP WITH STROKE","	* voiced epiglottal stop"},
/* 02A2 */ { "LATIN LETTER REVERSED GLOTTAL STOP WITH STROKE","	* voiced epiglottal fricative"},
/* 02A3 */ { "LATIN SMALL LETTER DZ DIGRAPH","	* voiced dental affricate\n"
	"	x (latin small letter dz digraph with retroflex hook - AB66)"},
/* 02A4 */ { "LATIN SMALL LETTER DEZH DIGRAPH","	* voiced postalveolar affricate"},
/* 02A5 */ { "LATIN SMALL LETTER DZ DIGRAPH WITH CURL","	* voiced alveolo-palatal affricate"},
/* 02A6 */ { "LATIN SMALL LETTER TS DIGRAPH","	* voiceless dental affricate\n"
	"	x (latin letter inverted glottal stop with stroke - 01BE)\n"
	"	x (latin small letter ts digraph with retroflex hook - AB67)"},
/* 02A7 */ { "LATIN SMALL LETTER TESH DIGRAPH","	* voiceless postalveolar affricate\n"
	"	x (latin small letter tz - A729)"},
/* 02A8 */ { "LATIN SMALL LETTER TC DIGRAPH WITH CURL","	* voiceless alveolo-palatal affricate"},
/* 02A9 */ { "LATIN SMALL LETTER FENG DIGRAPH","	* velopharyngeal fricative"},
/* 02AA */ { "LATIN SMALL LETTER LS DIGRAPH","	* lateral alveolar fricative (lisp)"},
/* 02AB */ { "LATIN SMALL LETTER LZ DIGRAPH","	* voiced lateral alveolar fricative"},
/* 02AC */ { "LATIN LETTER BILABIAL PERCUSSIVE","	* audible lip smack"},
/* 02AD */ { "LATIN LETTER BIDENTAL PERCUSSIVE","	* audible teeth gnashing"},
/* 02AE */ { "LATIN SMALL LETTER TURNED H WITH FISHHOOK",NULL},
/* 02AF */ { "LATIN SMALL LETTER TURNED H WITH FISHHOOK AND TAIL",NULL},
/* 02B0 */ { "MODIFIER LETTER SMALL H","	* aspiration\n"
	"	# <super> 0068"},
/* 02B1 */ { "MODIFIER LETTER SMALL H WITH HOOK","	* breathy voiced, murmured\n"
	"	x (latin small letter h with hook - 0266)\n"
	"	x (combining diaeresis below - 0324)\n"
	"	# <super> 0266"},
/* 02B2 */ { "MODIFIER LETTER SMALL J","	* palatalization\n"
	"	x (combining palatalized hook below - 0321)\n"
	"	# <super> 006A"},
/* 02B3 */ { "MODIFIER LETTER SMALL R","	# <super> 0072"},
/* 02B4 */ { "MODIFIER LETTER SMALL TURNED R","	x (latin small letter turned r - 0279)\n"
	"	# <super> 0279"},
/* 02B5 */ { "MODIFIER LETTER SMALL TURNED R WITH HOOK","	x (latin small letter turned r with hook - 027B)\n"
	"	# <super> 027B"},
/* 02B6 */ { "MODIFIER LETTER SMALL CAPITAL INVERTED R","	* preceding four used for r-coloring or r-offglides\n"
	"	x (latin letter small capital inverted r - 0281)\n"
	"	# <super> 0281"},
/* 02B7 */ { "MODIFIER LETTER SMALL W","	* labialization\n"
	"	x (combining inverted double arch below - 032B)\n"
	"	# <super> 0077"},
/* 02B8 */ { "MODIFIER LETTER SMALL Y","	* palatalization\n"
	"	* common Americanist usage for 02B2\n"
	"	# <super> 0079"},
/* 02B9 */ { "MODIFIER LETTER PRIME","	* stress, emphasis\n"
	"	* transliteration of mjagkij znak (Cyrillic soft sign: palatalization)\n"
	"	* transliteration of Hebrew geresh\n"
	"	x (apostrophe - 0027)\n"
	"	x (acute accent - 00B4)\n"
	"	x (modifier letter acute accent - 02CA)\n"
	"	x (combining acute accent - 0301)\n"
	"	x (greek numeral sign - 0374)\n"
	"	x (hebrew punctuation geresh - 05F3)\n"
	"	x (prime - 2032)"},
/* 02BA */ { "MODIFIER LETTER DOUBLE PRIME","	* exaggerated stress, contrastive stress\n"
	"	* transliteration of tverdyj znak (Cyrillic hard sign: no palatalization)\n"
	"	x (quotation mark - 0022)\n"
	"	x (combining double acute accent - 030B)\n"
	"	x (double prime - 2033)"},
/* 02BB */ { "MODIFIER LETTER TURNED COMMA","	* typographical alternate for 02BD or 02BF\n"
	"	* used in some Polynesian orthographies for glottal stop\n"
	"	* Hawaiʻian: ʻokina\n"
	"	* Tongan: fakauʻa\n"
	"	x (combining turned comma above - 0312)\n"
	"	x (nko low tone apostrophe - 07F5)\n"
	"	x (left single quotation mark - 2018)"},
/* 02BC */ { "MODIFIER LETTER APOSTROPHE","	= apostrophe\n"
	"	* glottal stop, glottalization, ejective\n"
	"	* many languages use this as a letter of their alphabets\n"
	"	* used as a tone marker in Bodo and Dogri\n"
	"	* indicates vowel elongation, or various truncations and ellipsis in Maithili\n"
	"	* used as a modifier letter in the Lisu script\n"
	"	* 2019 is the preferred character for a punctuation apostrophe\n"
	"	x (apostrophe - 0027)\n"
	"	x (combining comma above - 0313)\n"
	"	x (combining comma above right - 0315)\n"
	"	x (armenian apostrophe - 055A)\n"
	"	x (nko high tone apostrophe - 07F4)\n"
	"	x (greek psili - 1FBF)\n"
	"	x (right single quotation mark - 2019)"},
/* 02BD */ { "MODIFIER LETTER REVERSED COMMA","	* weak aspiration\n"
	"	x (combining reversed comma above - 0314)\n"
	"	x (armenian modifier letter left half ring - 0559)\n"
	"	x (greek dasia - 1FFE)\n"
	"	x (single high-reversed-9 quotation mark - 201B)"},
/* 02BE */ { "MODIFIER LETTER RIGHT HALF RING","	* transliteration of Arabic hamza (glottal stop)\n"
	"	* transliteration of Hebrew alef\n"
	"	x (armenian apostrophe - 055A)\n"
	"	x (hebrew letter alef - 05D0)\n"
	"	x (arabic letter hamza - 0621)"},
/* 02BF */ { "MODIFIER LETTER LEFT HALF RING","	* transliteration of Arabic ain (voiced pharyngeal fricative)\n"
	"	* transliteration of Hebrew ayin\n"
	"	x (armenian modifier letter left half ring - 0559)\n"
	"	x (hebrew letter ayin - 05E2)\n"
	"	x (arabic letter ain - 0639)"},
/* 02C0 */ { "MODIFIER LETTER GLOTTAL STOP","	* ejective or glottalized\n"
	"	* typographical alternate for 02BC or 02BE\n"
	"	x (latin letter glottal stop - 0294)\n"
	"	x (combining hook above - 0309)"},
/* 02C1 */ { "MODIFIER LETTER REVERSED GLOTTAL STOP","	* typographical alternate for 02BF\n"
	"	x (latin letter pharyngeal voiced fricative - 0295)"},
/* 02C2 */ { "MODIFIER LETTER LEFT ARROWHEAD","	* fronted articulation"},
/* 02C3 */ { "MODIFIER LETTER RIGHT ARROWHEAD","	* backed articulation"},
/* 02C4 */ { "MODIFIER LETTER UP ARROWHEAD","	* raised articulation\n"
	"	x (circumflex accent - 005E)\n"
	"	x (up arrowhead - 2303)"},
/* 02C5 */ { "MODIFIER LETTER DOWN ARROWHEAD","	* lowered articulation"},
/* 02C6 */ { "MODIFIER LETTER CIRCUMFLEX ACCENT","	* rising-falling tone, falling tone, secondary stress, etc.\n"
	"	x (circumflex accent - 005E)\n"
	"	x (combining circumflex accent - 0302)\n"
	"	x (modifier letter low circumflex accent - A788)"},
/* 02C7 */ { "CARON","	= hacek\n"
	"	* falling-rising tone\n"
	"	* Mandarin Chinese third tone\n"
	"	x (combining caron - 030C)"},
/* 02C8 */ { "MODIFIER LETTER VERTICAL LINE","	* primary stress, downstep\n"
	"	* precedes letter or syllable modified\n"
	"	x (apostrophe - 0027)\n"
	"	x (combining vertical line above - 030D)"},
/* 02C9 */ { "MODIFIER LETTER MACRON","	* high level tone\n"
	"	* precedes or follows letter or syllable modified\n"
	"	* Mandarin Chinese first tone\n"
	"	x (macron - 00AF)\n"
	"	x (combining macron - 0304)"},
/* 02CA */ { "MODIFIER LETTER ACUTE ACCENT","	* high-rising tone (IPA), high tone, primary stress\n"
	"	* Mandarin Chinese second tone\n"
	"	x (acute accent - 00B4)\n"
	"	x (modifier letter prime - 02B9)\n"
	"	x (combining acute accent - 0301)\n"
	"	x (greek numeral sign - 0374)\n"
	"	x (armenian emphasis mark - 055B)"},
/* 02CB */ { "MODIFIER LETTER GRAVE ACCENT","	* high-falling tone (IPA), low tone, secondary or tertiary stress\n"
	"	* Mandarin Chinese fourth tone\n"
	"	x (grave accent - 0060)\n"
	"	x (combining grave accent - 0300)\n"
	"	x (armenian comma - 055D)"},
/* 02CC */ { "MODIFIER LETTER LOW VERTICAL LINE","	* secondary stress\n"
	"	* precedes letter or syllable modified\n"
	"	x (combining vertical line below - 0329)"},
/* 02CD */ { "MODIFIER LETTER LOW MACRON","	* low level tone\n"
	"	* used as a modifier letter in the Lisu script\n"
	"	x (low line - 005F)\n"
	"	x (combining macron below - 0331)"},
/* 02CE */ { "MODIFIER LETTER LOW GRAVE ACCENT","	* low-falling tone"},
/* 02CF */ { "MODIFIER LETTER LOW ACUTE ACCENT","	* low-rising tone\n"
	"	x (greek lower numeral sign - 0375)"},
/* 02D0 */ { "MODIFIER LETTER TRIANGULAR COLON","	* length mark\n"
	"	x (colon - 003A)"},
/* 02D1 */ { "MODIFIER LETTER HALF TRIANGULAR COLON","	* half-length mark\n"
	"	x (middle dot - 00B7)"},
/* 02D2 */ { "MODIFIER LETTER CENTRED RIGHT HALF RING","	* more rounded articulation"},
/* 02D3 */ { "MODIFIER LETTER CENTRED LEFT HALF RING","	* less rounded articulation"},
/* 02D4 */ { "MODIFIER LETTER UP TACK","	* vowel raising or closing\n"
	"	x (combining up tack below - 031D)\n"
	"	x (combining dot below - 0323)"},
/* 02D5 */ { "MODIFIER LETTER DOWN TACK","	* vowel lowering or opening\n"
	"	x (combining left half ring below - 031C)\n"
	"	x (combining down tack below - 031E)"},
/* 02D6 */ { "MODIFIER LETTER PLUS SIGN","	* advanced or fronted articulation\n"
	"	x (combining plus sign below - 031F)"},
/* 02D7 */ { "MODIFIER LETTER MINUS SIGN","	* retracted or backed articulation\n"
	"	* glyph may have small end-serifs\n"
	"	x (combining minus sign below - 0320)\n"
	"	x (minus sign - 2212)"},
/* 02D8 */ { "BREVE","	x (combining breve - 0306)\n"
	"	# 0020 0306"},
/* 02D9 */ { "DOT ABOVE","	* Mandarin Chinese fifth tone (light or neutral)\n"
	"	x (combining dot above - 0307)\n"
	"	# 0020 0307"},
/* 02DA */ { "RING ABOVE","	x (degree sign - 00B0)\n"
	"	x (combining ring above - 030A)\n"
	"	# 0020 030A"},
/* 02DB */ { "OGONEK","	x (combining ogonek - 0328)\n"
	"	# 0020 0328"},
/* 02DC */ { "SMALL TILDE","	x (tilde - 007E)\n"
	"	x (combining tilde - 0303)\n"
	"	x (tilde operator - 223C)\n"
	"	# 0020 0303"},
/* 02DD */ { "DOUBLE ACUTE ACCENT","	x (combining double acute accent - 030B)\n"
	"	# 0020 030B"},
/* 02DE */ { "MODIFIER LETTER RHOTIC HOOK","	* rhotacization in vowel\n"
	"	* often ligated: 025A = 0259 + 02DE; 025D = 025C + 02DE"},
/* 02DF */ { "MODIFIER LETTER CROSS ACCENT","	* Swedish grave accent"},
/* 02E0 */ { "MODIFIER LETTER SMALL GAMMA","	* these modifier letters are occasionally used in transcription of affricates\n"
	"	# <super> 0263"},
/* 02E1 */ { "MODIFIER LETTER SMALL L","	# <super> 006C"},
/* 02E2 */ { "MODIFIER LETTER SMALL S","	# <super> 0073"},
/* 02E3 */ { "MODIFIER LETTER SMALL X","	# <super> 0078"},
/* 02E4 */ { "MODIFIER LETTER SMALL REVERSED GLOTTAL STOP","	# <super> 0295"},
/* 02E5 */ { "MODIFIER LETTER EXTRA-HIGH TONE BAR",NULL},
/* 02E6 */ { "MODIFIER LETTER HIGH TONE BAR",NULL},
/* 02E7 */ { "MODIFIER LETTER MID TONE BAR",NULL},
/* 02E8 */ { "MODIFIER LETTER LOW TONE BAR",NULL},
/* 02E9 */ { "MODIFIER LETTER EXTRA-LOW TONE BAR",NULL},
/* 02EA */ { "MODIFIER LETTER YIN DEPARTING TONE MARK",NULL},
/* 02EB */ { "MODIFIER LETTER YANG DEPARTING TONE MARK",NULL},
/* 02EC */ { "MODIFIER LETTER VOICING",NULL},
/* 02ED */ { "MODIFIER LETTER UNASPIRATED","	x (modifier letter short equals sign - A78A)"},
/* 02EE */ { "MODIFIER LETTER DOUBLE APOSTROPHE","	* Nenets"},
/* 02EF */ { "MODIFIER LETTER LOW DOWN ARROWHEAD",NULL},
/* 02F0 */ { "MODIFIER LETTER LOW UP ARROWHEAD",NULL},
/* 02F1 */ { "MODIFIER LETTER LOW LEFT ARROWHEAD",NULL},
/* 02F2 */ { "MODIFIER LETTER LOW RIGHT ARROWHEAD",NULL},
/* 02F3 */ { "MODIFIER LETTER LOW RING","	x (combining ring below - 0325)"},
/* 02F4 */ { "MODIFIER LETTER MIDDLE GRAVE ACCENT",NULL},
/* 02F5 */ { "MODIFIER LETTER MIDDLE DOUBLE GRAVE ACCENT",NULL},
/* 02F6 */ { "MODIFIER LETTER MIDDLE DOUBLE ACUTE ACCENT",NULL},
/* 02F7 */ { "MODIFIER LETTER LOW TILDE",NULL},
/* 02F8 */ { "MODIFIER LETTER RAISED COLON",NULL},
/* 02F9 */ { "MODIFIER LETTER BEGIN HIGH TONE",NULL},
/* 02FA */ { "MODIFIER LETTER END HIGH TONE",NULL},
/* 02FB */ { "MODIFIER LETTER BEGIN LOW TONE",NULL},
/* 02FC */ { "MODIFIER LETTER END LOW TONE",NULL},
/* 02FD */ { "MODIFIER LETTER SHELF",NULL},
/* 02FE */ { "MODIFIER LETTER OPEN SHELF",NULL},
/* 02FF */ { "MODIFIER LETTER LOW LEFT ARROW",NULL}
};

UN_DLL_LOCAL
static const struct unicode_nameannot una_00_03[] = {
/* 0300 */ { "COMBINING GRAVE ACCENT","	= Greek varia\n"
	"	* Pinyin: marks Mandarin Chinese fourth tone\n"
	"	x (grave accent - 0060)\n"
	"	x (modifier letter grave accent - 02CB)"},
/* 0301 */ { "COMBINING ACUTE ACCENT","	= stress mark\n"
	"	= Greek oxia, tonos\n"
	"	* Pinyin: marks Mandarin Chinese second tone\n"
	"	x (apostrophe - 0027)\n"
	"	x (acute accent - 00B4)\n"
	"	x (modifier letter prime - 02B9)\n"
	"	x (modifier letter acute accent - 02CA)\n"
	"	x (greek tonos - 0384)"},
/* 0302 */ { "COMBINING CIRCUMFLEX ACCENT","	= hat\n"
	"	x (circumflex accent - 005E)\n"
	"	x (modifier letter circumflex accent - 02C6)"},
/* 0303 */ { "COMBINING TILDE","	* IPA: nasalization\n"
	"	* Vietnamese tone mark\n"
	"	x (tilde - 007E)\n"
	"	x (small tilde - 02DC)"},
/* 0304 */ { "COMBINING MACRON","	= long\n"
	"	* Pinyin: marks Mandarin Chinese first tone\n"
	"	* distinguish from the following\n"
	"	x (macron - 00AF)\n"
	"	x (modifier letter macron - 02C9)"},
/* 0305 */ { "COMBINING OVERLINE","	= overscore, vinculum\n"
	"	* connects on left and right\n"
	"	x (macron - 00AF)"},
/* 0306 */ { "COMBINING BREVE","	= short\n"
	"	= Greek vrachy\n"
	"	x (breve - 02D8)\n"
	"	x (combining cyrillic kavyka - A67C)"},
/* 0307 */ { "COMBINING DOT ABOVE","	= derivative (Newtonian notation)\n"
	"	* IPA (withdrawn in 1976): palatalization\n"
	"	x (dot above - 02D9)"},
/* 0308 */ { "COMBINING DIAERESIS","	= double dot above, umlaut\n"
	"	= Greek dialytika\n"
	"	= double derivative\n"
	"	x (diaeresis - 00A8)"},
/* 0309 */ { "COMBINING HOOK ABOVE","	= hoi\n"
	"	* kerns left or right of circumflex over vowels\n"
	"	* Vietnamese tone mark\n"
	"	x (modifier letter glottal stop - 02C0)"},
/* 030A */ { "COMBINING RING ABOVE","	x (degree sign - 00B0)\n"
	"	x (ring above - 02DA)"},
/* 030B */ { "COMBINING DOUBLE ACUTE ACCENT","	* Hungarian, Chuvash\n"
	"	x (quotation mark - 0022)\n"
	"	x (modifier letter double prime - 02BA)\n"
	"	x (double acute accent - 02DD)"},
/* 030C */ { "COMBINING CARON","	= hacek, V above\n"
	"	* Pinyin: marks Mandarin Chinese third tone\n"
	"	x (caron - 02C7)"},
/* 030D */ { "COMBINING VERTICAL LINE ABOVE","	* this is not the Greek tonos\n"
	"	* Marshallese\n"
	"	x (modifier letter vertical line - 02C8)"},
/* 030E */ { "COMBINING DOUBLE VERTICAL LINE ABOVE","	* Marshallese\n"
	"	x (quotation mark - 0022)"},
/* 030F */ { "COMBINING DOUBLE GRAVE ACCENT","	* Serbian and Croatian phonetics"},
/* 0310 */ { "COMBINING CANDRABINDU","	x (devanagari sign candrabindu - 0901)"},
/* 0311 */ { "COMBINING INVERTED BREVE","	x (combining cyrillic palatalization - 0484)\n"
	"	x (combining cyrillic pokrytie - 0487)"},
/* 0312 */ { "COMBINING TURNED COMMA ABOVE","	= cedilla above\n"
	"	* Latvian (but not used in decomposition)\n"
	"	x (modifier letter turned comma - 02BB)"},
/* 0313 */ { "COMBINING COMMA ABOVE","	= Greek psili, smooth breathing mark\n"
	"	* Americanist: ejective or glottalization\n"
	"	x (modifier letter apostrophe - 02BC)\n"
	"	x (combining cyrillic psili pneumata - 0486)\n"
	"	x (armenian apostrophe - 055A)"},
/* 0314 */ { "COMBINING REVERSED COMMA ABOVE","	= Greek dasia, rough breathing mark\n"
	"	x (modifier letter reversed comma - 02BD)\n"
	"	x (combining cyrillic dasia pneumata - 0485)\n"
	"	x (armenian modifier letter left half ring - 0559)"},
/* 0315 */ { "COMBINING COMMA ABOVE RIGHT","	x (modifier letter apostrophe - 02BC)"},
/* 0316 */ { "COMBINING GRAVE ACCENT BELOW",NULL},
/* 0317 */ { "COMBINING ACUTE ACCENT BELOW",NULL},
/* 0318 */ { "COMBINING LEFT TACK BELOW","	x (modifier letter left tack - AB6A)"},
/* 0319 */ { "COMBINING RIGHT TACK BELOW","	x (modifier letter right tack - AB6B)"},
/* 031A */ { "COMBINING LEFT ANGLE ABOVE","	* IPA: unreleased stop"},
/* 031B */ { "COMBINING HORN","	* Vietnamese"},
/* 031C */ { "COMBINING LEFT HALF RING BELOW","	* IPA: open variety of vowel\n"
	"	x (modifier letter down tack - 02D5)"},
/* 031D */ { "COMBINING UP TACK BELOW","	* IPA: vowel raising or closing\n"
	"	x (modifier letter up tack - 02D4)"},
/* 031E */ { "COMBINING DOWN TACK BELOW","	* IPA: vowel lowering or opening\n"
	"	x (modifier letter down tack - 02D5)"},
/* 031F */ { "COMBINING PLUS SIGN BELOW","	* IPA: advanced or fronted articulation\n"
	"	x (modifier letter plus sign - 02D6)\n"
	"	x (combining plus sign above - 1AC8)"},
/* 0320 */ { "COMBINING MINUS SIGN BELOW","	* IPA: retracted or backed articulation\n"
	"	* glyph may have small end-serifs\n"
	"	x (modifier letter minus sign - 02D7)"},
/* 0321 */ { "COMBINING PALATALIZED HOOK BELOW","	* IPA: palatalization\n"
	"	x (modifier letter small j - 02B2)"},
/* 0322 */ { "COMBINING RETROFLEX HOOK BELOW","	* IPA: retroflexion"},
/* 0323 */ { "COMBINING DOT BELOW","	= nang\n"
	"	* IPA: closer variety of vowel\n"
	"	* Americanist and Indo-Europeanist: retraction or retroflexion\n"
	"	* Semiticist: velarization or pharyngealization\n"
	"	* Vietnamese tone mark\n"
	"	x (modifier letter up tack - 02D4)"},
/* 0324 */ { "COMBINING DIAERESIS BELOW","	* IPA: breathy-voice or murmur\n"
	"	x (modifier letter small h with hook - 02B1)"},
/* 0325 */ { "COMBINING RING BELOW","	* IPA: voiceless\n"
	"	* vocalic (in Latin transliteration of Indic sonorants)\n"
	"	* Madurese\n"
	"	x (modifier letter low ring - 02F3)"},
/* 0326 */ { "COMBINING COMMA BELOW","	* Romanian, Latvian, Livonian"},
/* 0327 */ { "COMBINING CEDILLA","	* French, Turkish, Azerbaijani\n"
	"	x (cedilla - 00B8)"},
/* 0328 */ { "COMBINING OGONEK","	= nasal hook\n"
	"	* Americanist: nasalization\n"
	"	* Polish, Lithuanian\n"
	"	x (ogonek - 02DB)\n"
	"	x (combining open mark below - 1AB7)"},
/* 0329 */ { "COMBINING VERTICAL LINE BELOW","	* IPA: syllabic\n"
	"	* Yoruba\n"
	"	* German dialectology\n"
	"	* may attach or not; shape may vary by local preferences\n"
	"	x (modifier letter low vertical line - 02CC)"},
/* 032A */ { "COMBINING BRIDGE BELOW","	* IPA: dental"},
/* 032B */ { "COMBINING INVERTED DOUBLE ARCH BELOW","	* IPA: labialization\n"
	"	x (modifier letter small w - 02B7)\n"
	"	x (combining inverted double arch above - 1AC7)"},
/* 032C */ { "COMBINING CARON BELOW","	* IPA: voiced"},
/* 032D */ { "COMBINING CIRCUMFLEX ACCENT BELOW","	* Americanist: fronted articulation"},
/* 032E */ { "COMBINING BREVE BELOW","	* Hittite transcription"},
/* 032F */ { "COMBINING INVERTED BREVE BELOW","	* Americanist: fronted articulation (variant of 032D)\n"
	"	* Indo-Europeanist: semivowel"},
/* 0330 */ { "COMBINING TILDE BELOW","	* IPA: creaky voice"},
/* 0331 */ { "COMBINING MACRON BELOW","	x (low line - 005F)\n"
	"	x (modifier letter low macron - 02CD)"},
/* 0332 */ { "COMBINING LOW LINE","	= underline, underscore\n"
	"	* connects on left and right\n"
	"	x (low line - 005F)"},
/* 0333 */ { "COMBINING DOUBLE LOW LINE","	= double underline, double underscore\n"
	"	* connects on left and right\n"
	"	x (combining equals sign below - 0347)\n"
	"	x (double low line - 2017)"},
/* 0334 */ { "COMBINING TILDE OVERLAY","	* IPA: velarization or pharyngealization"},
/* 0335 */ { "COMBINING SHORT STROKE OVERLAY",NULL},
/* 0336 */ { "COMBINING LONG STROKE OVERLAY","	= strikethrough\n"
	"	* connects on left and right"},
/* 0337 */ { "COMBINING SHORT SOLIDUS OVERLAY","	= short slash overlay"},
/* 0338 */ { "COMBINING LONG SOLIDUS OVERLAY","	= long slash overlay"},
/* 0339 */ { "COMBINING RIGHT HALF RING BELOW",NULL},
/* 033A */ { "COMBINING INVERTED BRIDGE BELOW","	* IPA: apical\n"
	"	x (combining wide inverted bridge below - 1DF9)"},
/* 033B */ { "COMBINING SQUARE BELOW","	* IPA: laminal\n"
	"	* preferred glyph shape is a horizontal rectangle for IPA usage"},
/* 033C */ { "COMBINING SEAGULL BELOW","	* IPA: linguolabial"},
/* 033D */ { "COMBINING X ABOVE",NULL},
/* 033E */ { "COMBINING VERTICAL TILDE","	= yerik\n"
	"	* used for Cyrillic yerik, indicates an omitted jer\n"
	"	* this is not a palatalization mark\n"
	"	x (combining cyrillic palatalization - 0484)\n"
	"	x (vertical tilde - 2E2F)\n"
	"	x (combining cyrillic payerok - A67D)"},
/* 033F */ { "COMBINING DOUBLE OVERLINE","	* connects on left and right"},
/* 0340 */ { "COMBINING GRAVE TONE MARK","	: 0300 combining grave accent"},
/* 0341 */ { "COMBINING ACUTE TONE MARK","	: 0301 combining acute accent"},
/* 0342 */ { "COMBINING GREEK PERISPOMENI","	* Greek-specific form of circumflex for rising-falling accent\n"
	"	* alternative glyph is similar to an inverted breve\n"
	"	x (combining tilde - 0303)\n"
	"	x (combining inverted breve - 0311)"},
/* 0343 */ { "COMBINING GREEK KORONIS","	: 0313 combining comma above"},
/* 0344 */ { "COMBINING GREEK DIALYTIKA TONOS","	* use of this character is discouraged\n"
	"	: 0308 0301"},
/* 0345 */ { "COMBINING GREEK YPOGEGRAMMENI","	= greek non-spacing iota below (1.0)\n"
	"	= iota subscript\n"
	"	* note special casing issues\n"
	"	x (greek ypogegrammeni - 037A)\n"
	"	x (greek capital letter iota - 0399)"},
/* 0346 */ { "COMBINING BRIDGE ABOVE","	* IPA: dentolabial\n"
	"	x (combining wide bridge above - 20E9)"},
/* 0347 */ { "COMBINING EQUALS SIGN BELOW","	* IPA: alveolar"},
/* 0348 */ { "COMBINING DOUBLE VERTICAL LINE BELOW","	* IPA: strong articulation"},
/* 0349 */ { "COMBINING LEFT ANGLE BELOW","	* IPA: weak articulation"},
/* 034A */ { "COMBINING NOT TILDE ABOVE","	* IPA: denasal"},
/* 034B */ { "COMBINING HOMOTHETIC ABOVE","	* IPA: nasal escape"},
/* 034C */ { "COMBINING ALMOST EQUAL TO ABOVE","	* IPA: velopharyngeal friction"},
/* 034D */ { "COMBINING LEFT RIGHT ARROW BELOW","	* IPA: labial spreading"},
/* 034E */ { "COMBINING UPWARDS ARROW BELOW","	* IPA: whistled articulation"},
/* 034F */ { "COMBINING GRAPHEME JOINER","	* commonly abbreviated as CGJ\n"
	"	* has no visible glyph\n"
	"	* the name of this character is misleading; it does not actually join graphemes"},
/* 0350 */ { "COMBINING RIGHT ARROWHEAD ABOVE",NULL},
/* 0351 */ { "COMBINING LEFT HALF RING ABOVE",NULL},
/* 0352 */ { "COMBINING FERMATA",NULL},
/* 0353 */ { "COMBINING X BELOW",NULL},
/* 0354 */ { "COMBINING LEFT ARROWHEAD BELOW",NULL},
/* 0355 */ { "COMBINING RIGHT ARROWHEAD BELOW",NULL},
/* 0356 */ { "COMBINING RIGHT ARROWHEAD AND UP ARROWHEAD BELOW",NULL},
/* 0357 */ { "COMBINING RIGHT HALF RING ABOVE",NULL},
/* 0358 */ { "COMBINING DOT ABOVE RIGHT","	* Latin transliterations of the Southern Min dialects of Chinese\n"
	"	x (combining dot above left - 1DF8)"},
/* 0359 */ { "COMBINING ASTERISK BELOW","	x (low asterisk - 204E)"},
/* 035A */ { "COMBINING DOUBLE RING BELOW","	* Kharoshthi transliteration"},
/* 035B */ { "COMBINING ZIGZAG ABOVE","	* Latin abbreviation, Lithuanian phonetics and medievalist transcriptions"},
/* 035C */ { "COMBINING DOUBLE BREVE BELOW","	= ligature tie below, papyrological hyphen\n"
	"	* a common glyph alternate connects the horizontal midpoints of the characters\n"
	"	x (combining double breve - 035D)\n"
	"	x (combining double inverted breve - 0361)\n"
	"	x (combining ligature left half below - FE27)"},
/* 035D */ { "COMBINING DOUBLE BREVE",NULL},
/* 035E */ { "COMBINING DOUBLE MACRON",NULL},
/* 035F */ { "COMBINING DOUBLE MACRON BELOW",NULL},
/* 0360 */ { "COMBINING DOUBLE TILDE","	x (combining double tilde left half - FE22)"},
/* 0361 */ { "COMBINING DOUBLE INVERTED BREVE","	= ligature tie\n"
	"	x (combining ligature left half - FE20)"},
/* 0362 */ { "COMBINING DOUBLE RIGHTWARDS ARROW BELOW","	* IPA: sliding articulation"},
/* 0363 */ { "COMBINING LATIN SMALL LETTER A",NULL},
/* 0364 */ { "COMBINING LATIN SMALL LETTER E",NULL},
/* 0365 */ { "COMBINING LATIN SMALL LETTER I",NULL},
/* 0366 */ { "COMBINING LATIN SMALL LETTER O",NULL},
/* 0367 */ { "COMBINING LATIN SMALL LETTER U",NULL},
/* 0368 */ { "COMBINING LATIN SMALL LETTER C",NULL},
/* 0369 */ { "COMBINING LATIN SMALL LETTER D",NULL},
/* 036A */ { "COMBINING LATIN SMALL LETTER H",NULL},
/* 036B */ { "COMBINING LATIN SMALL LETTER M",NULL},
/* 036C */ { "COMBINING LATIN SMALL LETTER R",NULL},
/* 036D */ { "COMBINING LATIN SMALL LETTER T",NULL},
/* 036E */ { "COMBINING LATIN SMALL LETTER V",NULL},
/* 036F */ { "COMBINING LATIN SMALL LETTER X",NULL},
/* 0370 */ { "GREEK CAPITAL LETTER HETA","	x (latin capital letter half h - 2C75)"},
/* 0371 */ { "GREEK SMALL LETTER HETA","	x (latin small letter half h - 2C76)"},
/* 0372 */ { "GREEK CAPITAL LETTER ARCHAIC SAMPI",NULL},
/* 0373 */ { "GREEK SMALL LETTER ARCHAIC SAMPI",NULL},
/* 0374 */ { "GREEK NUMERAL SIGN","	= dexia keraia\n"
	"	* indicates numeric use of letters\n"
	"	x (modifier letter acute accent - 02CA)\n"
	"	: 02B9 modifier letter prime"},
/* 0375 */ { "GREEK LOWER NUMERAL SIGN","	= aristeri keraia\n"
	"	* indicates numeric use of letters\n"
	"	x (modifier letter low acute accent - 02CF)"},
/* 0376 */ { "GREEK CAPITAL LETTER PAMPHYLIAN DIGAMMA",NULL},
/* 0377 */ { "GREEK SMALL LETTER PAMPHYLIAN DIGAMMA",NULL},
/* 0378 */ { NULL,NULL},
/* 0379 */ { NULL,NULL},
/* 037A */ { "GREEK YPOGEGRAMMENI","	= iota subscript\n"
	"	x (combining greek ypogegrammeni - 0345)\n"
	"	# 0020 0345"},
/* 037B */ { "GREEK SMALL REVERSED LUNATE SIGMA SYMBOL",NULL},
/* 037C */ { "GREEK SMALL DOTTED LUNATE SIGMA SYMBOL",NULL},
/* 037D */ { "GREEK SMALL REVERSED DOTTED LUNATE SIGMA SYMBOL",NULL},
/* 037E */ { "GREEK QUESTION MARK","	= erotimatiko\n"
	"	* sentence-final punctuation\n"
	"	* 003B is the preferred character\n"
	"	x (question mark - 003F)\n"
	"	: 003B semicolon"},
/* 037F */ { "GREEK CAPITAL LETTER YOT","	* lowercase is 03F3"},
/* 0380 */ { NULL,NULL},
/* 0381 */ { NULL,NULL},
/* 0382 */ { NULL,NULL},
/* 0383 */ { NULL,NULL},
/* 0384 */ { "GREEK TONOS","	x (acute accent - 00B4)\n"
	"	x (combining vertical line above - 030D)\n"
	"	# 0020 0301"},
/* 0385 */ { "GREEK DIALYTIKA TONOS","	* compatibility decomposition (NFKD) is 0020 0308 0301\n"
	"	: 00A8 0301"},
/* 0386 */ { "GREEK CAPITAL LETTER ALPHA WITH TONOS","	: 0391 0301"},
/* 0387 */ { "GREEK ANO TELEIA","	* functions in Greek like a semicolon\n"
	"	* 00B7 is the preferred character\n"
	"	: 00B7 middle dot"},
/* 0388 */ { "GREEK CAPITAL LETTER EPSILON WITH TONOS","	: 0395 0301"},
/* 0389 */ { "GREEK CAPITAL LETTER ETA WITH TONOS","	: 0397 0301"},
/* 038A */ { "GREEK CAPITAL LETTER IOTA WITH TONOS","	: 0399 0301"},
/* 038B */ { NULL,NULL},
/* 038C */ { "GREEK CAPITAL LETTER OMICRON WITH TONOS","	: 039F 0301"},
/* 038D */ { NULL,NULL},
/* 038E */ { "GREEK CAPITAL LETTER UPSILON WITH TONOS","	: 03A5 0301"},
/* 038F */ { "GREEK CAPITAL LETTER OMEGA WITH TONOS","	: 03A9 0301"},
/* 0390 */ { "GREEK SMALL LETTER IOTA WITH DIALYTIKA AND TONOS","	: 03CA 0301"},
/* 0391 */ { "GREEK CAPITAL LETTER ALPHA",NULL},
/* 0392 */ { "GREEK CAPITAL LETTER BETA",NULL},
/* 0393 */ { "GREEK CAPITAL LETTER GAMMA","	= gamma function\n"
	"	x (double-struck capital gamma - 213E)"},
/* 0394 */ { "GREEK CAPITAL LETTER DELTA","	x (increment - 2206)"},
/* 0395 */ { "GREEK CAPITAL LETTER EPSILON",NULL},
/* 0396 */ { "GREEK CAPITAL LETTER ZETA",NULL},
/* 0397 */ { "GREEK CAPITAL LETTER ETA",NULL},
/* 0398 */ { "GREEK CAPITAL LETTER THETA",NULL},
/* 0399 */ { "GREEK CAPITAL LETTER IOTA","	= iota adscript"},
/* 039A */ { "GREEK CAPITAL LETTER KAPPA",NULL},
/* 039B */ { "GREEK CAPITAL LETTER LAMDA",NULL},
/* 039C */ { "GREEK CAPITAL LETTER MU",NULL},
/* 039D */ { "GREEK CAPITAL LETTER NU",NULL},
/* 039E */ { "GREEK CAPITAL LETTER XI",NULL},
/* 039F */ { "GREEK CAPITAL LETTER OMICRON",NULL},
/* 03A0 */ { "GREEK CAPITAL LETTER PI","	x (double-struck capital pi - 213F)\n"
	"	x (n-ary product - 220F)"},
/* 03A1 */ { "GREEK CAPITAL LETTER RHO",NULL},
/* 03A2 */ { NULL,NULL},
/* 03A3 */ { "GREEK CAPITAL LETTER SIGMA","	x (latin capital letter esh - 01A9)\n"
	"	x (n-ary summation - 2211)"},
/* 03A4 */ { "GREEK CAPITAL LETTER TAU",NULL},
/* 03A5 */ { "GREEK CAPITAL LETTER UPSILON",NULL},
/* 03A6 */ { "GREEK CAPITAL LETTER PHI",NULL},
/* 03A7 */ { "GREEK CAPITAL LETTER CHI",NULL},
/* 03A8 */ { "GREEK CAPITAL LETTER PSI",NULL},
/* 03A9 */ { "GREEK CAPITAL LETTER OMEGA","	x (ohm sign - 2126)\n"
	"	x (inverted ohm sign - 2127)"},
/* 03AA */ { "GREEK CAPITAL LETTER IOTA WITH DIALYTIKA","	: 0399 0308"},
/* 03AB */ { "GREEK CAPITAL LETTER UPSILON WITH DIALYTIKA","	: 03A5 0308"},
/* 03AC */ { "GREEK SMALL LETTER ALPHA WITH TONOS","	: 03B1 0301"},
/* 03AD */ { "GREEK SMALL LETTER EPSILON WITH TONOS","	: 03B5 0301"},
/* 03AE */ { "GREEK SMALL LETTER ETA WITH TONOS","	: 03B7 0301"},
/* 03AF */ { "GREEK SMALL LETTER IOTA WITH TONOS","	: 03B9 0301"},
/* 03B0 */ { "GREEK SMALL LETTER UPSILON WITH DIALYTIKA AND TONOS","	: 03CB 0301"},
/* 03B1 */ { "GREEK SMALL LETTER ALPHA","	x (latin small letter alpha - 0251)\n"
	"	x (proportional to - 221D)"},
/* 03B2 */ { "GREEK SMALL LETTER BETA","	x (latin small letter sharp s - 00DF)\n"
	"	x (latin small letter b with stroke - 0180)\n"
	"	x (latin small letter beta - A7B5)"},
/* 03B3 */ { "GREEK SMALL LETTER GAMMA","	x (latin small letter gamma - 0263)\n"
	"	x (double-struck small gamma - 213D)"},
/* 03B4 */ { "GREEK SMALL LETTER DELTA",NULL},
/* 03B5 */ { "GREEK SMALL LETTER EPSILON","	x (latin small letter open e - 025B)"},
/* 03B6 */ { "GREEK SMALL LETTER ZETA",NULL},
/* 03B7 */ { "GREEK SMALL LETTER ETA",NULL},
/* 03B8 */ { "GREEK SMALL LETTER THETA","	x (latin small letter barred o - 0275)\n"
	"	x (cyrillic small letter fita - 0473)"},
/* 03B9 */ { "GREEK SMALL LETTER IOTA","	x (latin small letter iota - 0269)\n"
	"	x (turned greek small letter iota - 2129)"},
/* 03BA */ { "GREEK SMALL LETTER KAPPA",NULL},
/* 03BB */ { "GREEK SMALL LETTER LAMDA","	= lambda\n"
	"	x (latin small letter lambda - A7DB)"},
/* 03BC */ { "GREEK SMALL LETTER MU","	x (micro sign - 00B5)"},
/* 03BD */ { "GREEK SMALL LETTER NU",NULL},
/* 03BE */ { "GREEK SMALL LETTER XI",NULL},
/* 03BF */ { "GREEK SMALL LETTER OMICRON",NULL},
/* 03C0 */ { "GREEK SMALL LETTER PI","	* math constant 3.141592..."},
/* 03C1 */ { "GREEK SMALL LETTER RHO",NULL},
/* 03C2 */ { "GREEK SMALL LETTER FINAL SIGMA","	= stigma (the Modern Greek name for this letterform)\n"
	"	* not to be confused with the actual stigma letter\n"
	"	x (greek small letter stigma - 03DB)"},
/* 03C3 */ { "GREEK SMALL LETTER SIGMA","	* used symbolically with a numeric value 200"},
/* 03C4 */ { "GREEK SMALL LETTER TAU",NULL},
/* 03C5 */ { "GREEK SMALL LETTER UPSILON","	x (latin small letter upsilon - 028A)\n"
	"	x (latin small letter v with hook - 028B)"},
/* 03C6 */ { "GREEK SMALL LETTER PHI","	* the ordinary Greek letter, showing considerable glyph variation\n"
	"	* in mathematical contexts, the loopy glyph is preferred, to contrast with 03D5\n"
	"	x (latin small letter phi - 0278)\n"
	"	x (greek phi symbol - 03D5)"},
/* 03C7 */ { "GREEK SMALL LETTER CHI","	x (latin small letter chi - AB53)"},
/* 03C8 */ { "GREEK SMALL LETTER PSI",NULL},
/* 03C9 */ { "GREEK SMALL LETTER OMEGA","	x (latin small letter omega - A7B7)"},
/* 03CA */ { "GREEK SMALL LETTER IOTA WITH DIALYTIKA","	: 03B9 0308"},
/* 03CB */ { "GREEK SMALL LETTER UPSILON WITH DIALYTIKA","	: 03C5 0308"},
/* 03CC */ { "GREEK SMALL LETTER OMICRON WITH TONOS","	: 03BF 0301"},
/* 03CD */ { "GREEK SMALL LETTER UPSILON WITH TONOS","	: 03C5 0301"},
/* 03CE */ { "GREEK SMALL LETTER OMEGA WITH TONOS","	: 03C9 0301"},
/* 03CF */ { "GREEK CAPITAL KAI SYMBOL","	x (greek kai symbol - 03D7)"},
/* 03D0 */ { "GREEK BETA SYMBOL","	= curled beta\n"
	"	# 03B2 greek small letter beta"},
/* 03D1 */ { "GREEK THETA SYMBOL","	= script theta\n"
	"	* used as a technical symbol\n"
	"	# 03B8 greek small letter theta"},
/* 03D2 */ { "GREEK UPSILON WITH HOOK SYMBOL","	# 03A5 greek capital letter upsilon"},
/* 03D3 */ { "GREEK UPSILON WITH ACUTE AND HOOK SYMBOL","	: 03D2 0301"},
/* 03D4 */ { "GREEK UPSILON WITH DIAERESIS AND HOOK SYMBOL","	: 03D2 0308"},
/* 03D5 */ { "GREEK PHI SYMBOL","	* used as a technical symbol, with a stroked glyph\n"
	"	* maps to \"phi1\" symbol entities\n"
	"	# 03C6 greek small letter phi"},
/* 03D6 */ { "GREEK PI SYMBOL","	= omega pi\n"
	"	* used as a technical symbol\n"
	"	* a variant of pi, looking like omega\n"
	"	# 03C0 greek small letter pi"},
/* 03D7 */ { "GREEK KAI SYMBOL","	* used as an ampersand\n"
	"	x (coptic symbol kai - 2CE4)"},
/* 03D8 */ { "GREEK LETTER ARCHAIC KOPPA",NULL},
/* 03D9 */ { "GREEK SMALL LETTER ARCHAIC KOPPA","	* the Q-shaped archaic koppas are the ordinary alphabetic letters and can also be used as symbols with a numeric value of 90 in classical and pre-classical texts"},
/* 03DA */ { "GREEK LETTER STIGMA","	* apparently in origin a cursive form of digamma\n"
	"	* the name \"stigma\" originally applied to a medieval sigma-tau ligature, whose shape was confusably similar to the cursive digamma\n"
	"	* used as a symbol with a numeric value of 6\n"
	"	x (roman numeral six late form - 2185)"},
/* 03DB */ { "GREEK SMALL LETTER STIGMA","	x (greek small letter final sigma - 03C2)"},
/* 03DC */ { "GREEK LETTER DIGAMMA",NULL},
/* 03DD */ { "GREEK SMALL LETTER DIGAMMA","	* used as a symbol with a numeric value of 6"},
/* 03DE */ { "GREEK LETTER KOPPA",NULL},
/* 03DF */ { "GREEK SMALL LETTER KOPPA","	* used in modern Greek as a symbol with a numeric value of 90, as in the dating of legal documentation"},
/* 03E0 */ { "GREEK LETTER SAMPI",NULL},
/* 03E1 */ { "GREEK SMALL LETTER SAMPI","	* used as a symbol with a numeric value of 900"},
/* 03E2 */ { "COPTIC CAPITAL LETTER SHEI",NULL},
/* 03E3 */ { "COPTIC SMALL LETTER SHEI",NULL},
/* 03E4 */ { "COPTIC CAPITAL LETTER FEI",NULL},
/* 03E5 */ { "COPTIC SMALL LETTER FEI",NULL},
/* 03E6 */ { "COPTIC CAPITAL LETTER KHEI",NULL},
/* 03E7 */ { "COPTIC SMALL LETTER KHEI",NULL},
/* 03E8 */ { "COPTIC CAPITAL LETTER HORI",NULL},
/* 03E9 */ { "COPTIC SMALL LETTER HORI",NULL},
/* 03EA */ { "COPTIC CAPITAL LETTER GANGIA",NULL},
/* 03EB */ { "COPTIC SMALL LETTER GANGIA",NULL},
/* 03EC */ { "COPTIC CAPITAL LETTER SHIMA",NULL},
/* 03ED */ { "COPTIC SMALL LETTER SHIMA",NULL},
/* 03EE */ { "COPTIC CAPITAL LETTER DEI",NULL},
/* 03EF */ { "COPTIC SMALL LETTER DEI",NULL},
/* 03F0 */ { "GREEK KAPPA SYMBOL","	= script kappa\n"
	"	* used as technical symbol\n"
	"	# 03BA greek small letter kappa"},
/* 03F1 */ { "GREEK RHO SYMBOL","	= tailed rho\n"
	"	* used as technical symbol\n"
	"	# 03C1 greek small letter rho"},
/* 03F2 */ { "GREEK LUNATE SIGMA SYMBOL","	= greek small letter lunate sigma (1.0)\n"
	"	# 03C2 greek small letter final sigma"},
/* 03F3 */ { "GREEK LETTER YOT","	* uppercase is 037F"},
/* 03F4 */ { "GREEK CAPITAL THETA SYMBOL","	x (cyrillic capital letter fita - 0472)\n"
	"	# 0398 greek capital letter theta"},
/* 03F5 */ { "GREEK LUNATE EPSILON SYMBOL","	= straight epsilon\n"
	"	x (small element of - 220A)\n"
	"	# 03B5 greek small letter epsilon"},
/* 03F6 */ { "GREEK REVERSED LUNATE EPSILON SYMBOL","	= reversed straight epsilon\n"
	"	x (small contains as member - 220D)"},
/* 03F7 */ { "GREEK CAPITAL LETTER SHO",NULL},
/* 03F8 */ { "GREEK SMALL LETTER SHO",NULL},
/* 03F9 */ { "GREEK CAPITAL LUNATE SIGMA SYMBOL","	# 03A3 greek capital letter sigma"},
/* 03FA */ { "GREEK CAPITAL LETTER SAN",NULL},
/* 03FB */ { "GREEK SMALL LETTER SAN",NULL},
/* 03FC */ { "GREEK RHO WITH STROKE SYMBOL","	* used with abbreviations containing 03C1"},
/* 03FD */ { "GREEK CAPITAL REVERSED LUNATE SIGMA SYMBOL","	= antisigma"},
/* 03FE */ { "GREEK CAPITAL DOTTED LUNATE SIGMA SYMBOL","	= sigma periestigmenon"},
/* 03FF */ { "GREEK CAPITAL REVERSED DOTTED LUNATE SIGMA SYMBOL","	= antisigma periestigmenon"}
};

UN_DLL_LOCAL
static const struct unicode_nameannot una_00_04[] = {
/* 0400 */ { "CYRILLIC CAPITAL LETTER IE WITH GRAVE","	: 0415 0300"},
/* 0401 */ { "CYRILLIC CAPITAL LETTER IO","	: 0415 0308"},
/* 0402 */ { "CYRILLIC CAPITAL LETTER DJE",NULL},
/* 0403 */ { "CYRILLIC CAPITAL LETTER GJE","	: 0413 0301"},
/* 0404 */ { "CYRILLIC CAPITAL LETTER UKRAINIAN IE",NULL},
/* 0405 */ { "CYRILLIC CAPITAL LETTER DZE",NULL},
/* 0406 */ { "CYRILLIC CAPITAL LETTER BYELORUSSIAN-UKRAINIAN I","	x (latin capital letter i - 0049)\n"
	"	x (cyrillic small letter byelorussian-ukrainian i - 0456)\n"
	"	x (cyrillic letter palochka - 04C0)"},
/* 0407 */ { "CYRILLIC CAPITAL LETTER YI","	: 0406 0308"},
/* 0408 */ { "CYRILLIC CAPITAL LETTER JE",NULL},
/* 0409 */ { "CYRILLIC CAPITAL LETTER LJE",NULL},
/* 040A */ { "CYRILLIC CAPITAL LETTER NJE",NULL},
/* 040B */ { "CYRILLIC CAPITAL LETTER TSHE",NULL},
/* 040C */ { "CYRILLIC CAPITAL LETTER KJE","	: 041A 0301"},
/* 040D */ { "CYRILLIC CAPITAL LETTER I WITH GRAVE","	: 0418 0300"},
/* 040E */ { "CYRILLIC CAPITAL LETTER SHORT U","	: 0423 0306"},
/* 040F */ { "CYRILLIC CAPITAL LETTER DZHE",NULL},
/* 0410 */ { "CYRILLIC CAPITAL LETTER A",NULL},
/* 0411 */ { "CYRILLIC CAPITAL LETTER BE","	x (latin small letter b with topbar - 0183)"},
/* 0412 */ { "CYRILLIC CAPITAL LETTER VE",NULL},
/* 0413 */ { "CYRILLIC CAPITAL LETTER GHE",NULL},
/* 0414 */ { "CYRILLIC CAPITAL LETTER DE",NULL},
/* 0415 */ { "CYRILLIC CAPITAL LETTER IE",NULL},
/* 0416 */ { "CYRILLIC CAPITAL LETTER ZHE",NULL},
/* 0417 */ { "CYRILLIC CAPITAL LETTER ZE",NULL},
/* 0418 */ { "CYRILLIC CAPITAL LETTER I","		This letter is also used to represent electrolarynx phonation in the Voice Quality Symbols for extended IPA."},
/* 0419 */ { "CYRILLIC CAPITAL LETTER SHORT I","	: 0418 0306"},
/* 041A */ { "CYRILLIC CAPITAL LETTER KA",NULL},
/* 041B */ { "CYRILLIC CAPITAL LETTER EL",NULL},
/* 041C */ { "CYRILLIC CAPITAL LETTER EM",NULL},
/* 041D */ { "CYRILLIC CAPITAL LETTER EN",NULL},
/* 041E */ { "CYRILLIC CAPITAL LETTER O",NULL},
/* 041F */ { "CYRILLIC CAPITAL LETTER PE",NULL},
/* 0420 */ { "CYRILLIC CAPITAL LETTER ER",NULL},
/* 0421 */ { "CYRILLIC CAPITAL LETTER ES",NULL},
/* 0422 */ { "CYRILLIC CAPITAL LETTER TE",NULL},
/* 0423 */ { "CYRILLIC CAPITAL LETTER U","	x (cyrillic capital letter uk - 0478)\n"
	"	x (cyrillic small letter straight u - 04AF)\n"
	"	x (cyrillic capital letter monograph uk - A64A)"},
/* 0424 */ { "CYRILLIC CAPITAL LETTER EF",NULL},
/* 0425 */ { "CYRILLIC CAPITAL LETTER HA",NULL},
/* 0426 */ { "CYRILLIC CAPITAL LETTER TSE",NULL},
/* 0427 */ { "CYRILLIC CAPITAL LETTER CHE",NULL},
/* 0428 */ { "CYRILLIC CAPITAL LETTER SHA",NULL},
/* 0429 */ { "CYRILLIC CAPITAL LETTER SHCHA",NULL},
/* 042A */ { "CYRILLIC CAPITAL LETTER HARD SIGN",NULL},
/* 042B */ { "CYRILLIC CAPITAL LETTER YERU",NULL},
/* 042C */ { "CYRILLIC CAPITAL LETTER SOFT SIGN",NULL},
/* 042D */ { "CYRILLIC CAPITAL LETTER E",NULL},
/* 042E */ { "CYRILLIC CAPITAL LETTER YU",NULL},
/* 042F */ { "CYRILLIC CAPITAL LETTER YA",NULL},
/* 0430 */ { "CYRILLIC SMALL LETTER A",NULL},
/* 0431 */ { "CYRILLIC SMALL LETTER BE",NULL},
/* 0432 */ { "CYRILLIC SMALL LETTER VE",NULL},
/* 0433 */ { "CYRILLIC SMALL LETTER GHE",NULL},
/* 0434 */ { "CYRILLIC SMALL LETTER DE",NULL},
/* 0435 */ { "CYRILLIC SMALL LETTER IE",NULL},
/* 0436 */ { "CYRILLIC SMALL LETTER ZHE",NULL},
/* 0437 */ { "CYRILLIC SMALL LETTER ZE",NULL},
/* 0438 */ { "CYRILLIC SMALL LETTER I",NULL},
/* 0439 */ { "CYRILLIC SMALL LETTER SHORT I","	: 0438 0306"},
/* 043A */ { "CYRILLIC SMALL LETTER KA",NULL},
/* 043B */ { "CYRILLIC SMALL LETTER EL",NULL},
/* 043C */ { "CYRILLIC SMALL LETTER EM",NULL},
/* 043D */ { "CYRILLIC SMALL LETTER EN",NULL},
/* 043E */ { "CYRILLIC SMALL LETTER O",NULL},
/* 043F */ { "CYRILLIC SMALL LETTER PE",NULL},
/* 0440 */ { "CYRILLIC SMALL LETTER ER",NULL},
/* 0441 */ { "CYRILLIC SMALL LETTER ES",NULL},
/* 0442 */ { "CYRILLIC SMALL LETTER TE",NULL},
/* 0443 */ { "CYRILLIC SMALL LETTER U",NULL},
/* 0444 */ { "CYRILLIC SMALL LETTER EF",NULL},
/* 0445 */ { "CYRILLIC SMALL LETTER HA",NULL},
/* 0446 */ { "CYRILLIC SMALL LETTER TSE",NULL},
/* 0447 */ { "CYRILLIC SMALL LETTER CHE",NULL},
/* 0448 */ { "CYRILLIC SMALL LETTER SHA",NULL},
/* 0449 */ { "CYRILLIC SMALL LETTER SHCHA",NULL},
/* 044A */ { "CYRILLIC SMALL LETTER HARD SIGN",NULL},
/* 044B */ { "CYRILLIC SMALL LETTER YERU","	x (cyrillic small letter yeru with back yer - A651)"},
/* 044C */ { "CYRILLIC SMALL LETTER SOFT SIGN","	x (latin small letter tone six - 0185)\n"
	"	x (cyrillic small letter neutral yer - A64F)"},
/* 044D */ { "CYRILLIC SMALL LETTER E",NULL},
/* 044E */ { "CYRILLIC SMALL LETTER YU","	x (cyrillic small letter reversed yu - A655)"},
/* 044F */ { "CYRILLIC SMALL LETTER YA",NULL},
/* 0450 */ { "CYRILLIC SMALL LETTER IE WITH GRAVE","	* Macedonian\n"
	"	: 0435 0300"},
/* 0451 */ { "CYRILLIC SMALL LETTER IO","	* Russian, ...\n"
	"	: 0435 0308"},
/* 0452 */ { "CYRILLIC SMALL LETTER DJE","	* Serbian\n"
	"	x (latin small letter d with stroke - 0111)"},
/* 0453 */ { "CYRILLIC SMALL LETTER GJE","	* Macedonian\n"
	"	: 0433 0301"},
/* 0454 */ { "CYRILLIC SMALL LETTER UKRAINIAN IE","	= Old Cyrillic yest"},
/* 0455 */ { "CYRILLIC SMALL LETTER DZE","	* Macedonian\n"
	"	x (cyrillic small letter dzelo - A643)"},
/* 0456 */ { "CYRILLIC SMALL LETTER BYELORUSSIAN-UKRAINIAN I","	= Old Cyrillic i"},
/* 0457 */ { "CYRILLIC SMALL LETTER YI","	* Ukrainian\n"
	"	: 0456 0308"},
/* 0458 */ { "CYRILLIC SMALL LETTER JE","	* Serbian, Azerbaijani, Altay"},
/* 0459 */ { "CYRILLIC SMALL LETTER LJE","	* Serbian, Macedonian\n"
	"	x (latin small letter lj - 01C9)"},
/* 045A */ { "CYRILLIC SMALL LETTER NJE","	* Serbian, Macedonian\n"
	"	x (latin small letter nj - 01CC)"},
/* 045B */ { "CYRILLIC SMALL LETTER TSHE","	* Serbian\n"
	"	x (latin small letter c with acute - 0107)\n"
	"	x (latin small letter h with stroke - 0127)\n"
	"	x (cyrillic capital letter tshe - 040B)\n"
	"	x (planck constant over two pi - 210F)\n"
	"	x (cyrillic small letter djerv - A649)"},
/* 045C */ { "CYRILLIC SMALL LETTER KJE","	* Macedonian\n"
	"	: 043A 0301"},
/* 045D */ { "CYRILLIC SMALL LETTER I WITH GRAVE","	* Macedonian, Bulgarian\n"
	"	: 0438 0300"},
/* 045E */ { "CYRILLIC SMALL LETTER SHORT U","	* Byelorussian, Uzbek\n"
	"	: 0443 0306"},
/* 045F */ { "CYRILLIC SMALL LETTER DZHE","	* Serbian, Macedonian, Abkhasian\n"
	"	x (latin small letter dz with caron - 01C6)"},
/* 0460 */ { "CYRILLIC CAPITAL LETTER OMEGA",NULL},
/* 0461 */ { "CYRILLIC SMALL LETTER OMEGA","	x (cyrillic small letter broad omega - A64D)"},
/* 0462 */ { "CYRILLIC CAPITAL LETTER YAT",NULL},
/* 0463 */ { "CYRILLIC SMALL LETTER YAT","	x (cyrillic small letter iotified yat - A653)\n"
	"	x (latin small letter sakha yat - AB60)"},
/* 0464 */ { "CYRILLIC CAPITAL LETTER IOTIFIED E",NULL},
/* 0465 */ { "CYRILLIC SMALL LETTER IOTIFIED E","	x (latin small letter iotified e - AB61)"},
/* 0466 */ { "CYRILLIC CAPITAL LETTER LITTLE YUS",NULL},
/* 0467 */ { "CYRILLIC SMALL LETTER LITTLE YUS","	x (cyrillic small letter closed little yus - A659)\n"
	"	x (cyrillic small letter blended yus - A65B)"},
/* 0468 */ { "CYRILLIC CAPITAL LETTER IOTIFIED LITTLE YUS",NULL},
/* 0469 */ { "CYRILLIC SMALL LETTER IOTIFIED LITTLE YUS","	x (cyrillic small letter ya - 044F)\n"
	"	x (cyrillic small letter iotified closed little yus - A65D)"},
/* 046A */ { "CYRILLIC CAPITAL LETTER BIG YUS",NULL},
/* 046B */ { "CYRILLIC SMALL LETTER BIG YUS","	x (cyrillic small letter blended yus - A65B)"},
/* 046C */ { "CYRILLIC CAPITAL LETTER IOTIFIED BIG YUS",NULL},
/* 046D */ { "CYRILLIC SMALL LETTER IOTIFIED BIG YUS",NULL},
/* 046E */ { "CYRILLIC CAPITAL LETTER KSI",NULL},
/* 046F */ { "CYRILLIC SMALL LETTER KSI",NULL},
/* 0470 */ { "CYRILLIC CAPITAL LETTER PSI",NULL},
/* 0471 */ { "CYRILLIC SMALL LETTER PSI",NULL},
/* 0472 */ { "CYRILLIC CAPITAL LETTER FITA",NULL},
/* 0473 */ { "CYRILLIC SMALL LETTER FITA","	x (latin small letter barred o - 0275)\n"
	"	x (greek small letter theta - 03B8)"},
/* 0474 */ { "CYRILLIC CAPITAL LETTER IZHITSA",NULL},
/* 0475 */ { "CYRILLIC SMALL LETTER IZHITSA",NULL},
/* 0476 */ { "CYRILLIC CAPITAL LETTER IZHITSA WITH DOUBLE GRAVE ACCENT","	: 0474 030F"},
/* 0477 */ { "CYRILLIC SMALL LETTER IZHITSA WITH DOUBLE GRAVE ACCENT","	: 0475 030F"},
/* 0478 */ { "CYRILLIC CAPITAL LETTER UK","	* for \"digraph onik\" the preferred spelling is 041E 0443\n"
	"	* for \"monograph uk\" the preferred character is A64A\n"
	"	x (cyrillic capital letter u - 0423)"},
/* 0479 */ { "CYRILLIC SMALL LETTER UK","	* for \"digraph onik\" the preferred spelling is 043E 0443\n"
	"	* for \"monograph uk\" the preferred character is A64B"},
/* 047A */ { "CYRILLIC CAPITAL LETTER ROUND OMEGA",NULL},
/* 047B */ { "CYRILLIC SMALL LETTER ROUND OMEGA",NULL},
/* 047C */ { "CYRILLIC CAPITAL LETTER OMEGA WITH TITLO","	= Cyrillic \"beautiful omega\"\n"
	"	* despite its name, this character does not have a titlo, nor is it composed of an omega plus a diacritic\n"
	"	x (cyrillic capital letter broad omega - A64C)"},
/* 047D */ { "CYRILLIC SMALL LETTER OMEGA WITH TITLO","	x (greek small letter omega with psili and perispomeni - 1F66)"},
/* 047E */ { "CYRILLIC CAPITAL LETTER OT",NULL},
/* 047F */ { "CYRILLIC SMALL LETTER OT",NULL},
/* 0480 */ { "CYRILLIC CAPITAL LETTER KOPPA",NULL},
/* 0481 */ { "CYRILLIC SMALL LETTER KOPPA",NULL},
/* 0482 */ { "CYRILLIC THOUSANDS SIGN",NULL},
/* 0483 */ { "COMBINING CYRILLIC TITLO","	* not used with titlo letters\n"
	"	x (combining cyrillic vzmet - A66F)"},
/* 0484 */ { "COMBINING CYRILLIC PALATALIZATION","	* not used for kamora\n"
	"	x (modifier letter apostrophe - 02BC)\n"
	"	x (combining inverted breve - 0311)"},
/* 0485 */ { "COMBINING CYRILLIC DASIA PNEUMATA","	x (combining reversed comma above - 0314)"},
/* 0486 */ { "COMBINING CYRILLIC PSILI PNEUMATA","	x (combining comma above - 0313)"},
/* 0487 */ { "COMBINING CYRILLIC POKRYTIE","	* used only with titlo letters\n"
	"	* also attested in Glagolitic\n"
	"	x (combining inverted breve - 0311)\n"
	"	x (combining cyrillic vzmet - A66F)"},
/* 0488 */ { "COMBINING CYRILLIC HUNDRED THOUSANDS SIGN","	* use 20DD for ten thousands sign"},
/* 0489 */ { "COMBINING CYRILLIC MILLIONS SIGN","	x (combining cyrillic ten millions sign - A670)"},
/* 048A */ { "CYRILLIC CAPITAL LETTER SHORT I WITH TAIL",NULL},
/* 048B */ { "CYRILLIC SMALL LETTER SHORT I WITH TAIL","	* Kildin Sami"},
/* 048C */ { "CYRILLIC CAPITAL LETTER SEMISOFT SIGN",NULL},
/* 048D */ { "CYRILLIC SMALL LETTER SEMISOFT SIGN","	* Kildin Sami"},
/* 048E */ { "CYRILLIC CAPITAL LETTER ER WITH TICK",NULL},
/* 048F */ { "CYRILLIC SMALL LETTER ER WITH TICK","	* Kildin Sami"},
/* 0490 */ { "CYRILLIC CAPITAL LETTER GHE WITH UPTURN",NULL},
/* 0491 */ { "CYRILLIC SMALL LETTER GHE WITH UPTURN","	* Ukrainian"},
/* 0492 */ { "CYRILLIC CAPITAL LETTER GHE WITH STROKE",NULL},
/* 0493 */ { "CYRILLIC SMALL LETTER GHE WITH STROKE","	* Azerbaijani, Bashkir, ...\n"
	"	* full bar form preferred over half-barred \"F\"-type"},
/* 0494 */ { "CYRILLIC CAPITAL LETTER GHE WITH MIDDLE HOOK",NULL},
/* 0495 */ { "CYRILLIC SMALL LETTER GHE WITH MIDDLE HOOK","	* Yakut, older orthographies for Abkhasian"},
/* 0496 */ { "CYRILLIC CAPITAL LETTER ZHE WITH DESCENDER",NULL},
/* 0497 */ { "CYRILLIC SMALL LETTER ZHE WITH DESCENDER","	* Tatar, ..."},
/* 0498 */ { "CYRILLIC CAPITAL LETTER ZE WITH DESCENDER",NULL},
/* 0499 */ { "CYRILLIC SMALL LETTER ZE WITH DESCENDER","	* Bashkir\n"
	"	* letterforms with right hooks are preferred, although occasional variants with left hooks occur"},
/* 049A */ { "CYRILLIC CAPITAL LETTER KA WITH DESCENDER",NULL},
/* 049B */ { "CYRILLIC SMALL LETTER KA WITH DESCENDER","	* Abkhasian, Tajik, ..."},
/* 049C */ { "CYRILLIC CAPITAL LETTER KA WITH VERTICAL STROKE",NULL},
/* 049D */ { "CYRILLIC SMALL LETTER KA WITH VERTICAL STROKE","	* Azerbaijani"},
/* 049E */ { "CYRILLIC CAPITAL LETTER KA WITH STROKE",NULL},
/* 049F */ { "CYRILLIC SMALL LETTER KA WITH STROKE","	* Abkhasian"},
/* 04A0 */ { "CYRILLIC CAPITAL LETTER BASHKIR KA",NULL},
/* 04A1 */ { "CYRILLIC SMALL LETTER BASHKIR KA",NULL},
/* 04A2 */ { "CYRILLIC CAPITAL LETTER EN WITH DESCENDER",NULL},
/* 04A3 */ { "CYRILLIC SMALL LETTER EN WITH DESCENDER","	* Bashkir, ..."},
/* 04A4 */ { "CYRILLIC CAPITAL LIGATURE EN GHE",NULL},
/* 04A5 */ { "CYRILLIC SMALL LIGATURE EN GHE","	* Altay, Mari, Yakut\n"
	"	* also used for soft en in Old Church Slavonic\n"
	"	* this is not a decomposable ligature"},
/* 04A6 */ { "CYRILLIC CAPITAL LETTER PE WITH MIDDLE HOOK",NULL},
/* 04A7 */ { "CYRILLIC SMALL LETTER PE WITH MIDDLE HOOK","	* older orthographies for Abkhasian"},
/* 04A8 */ { "CYRILLIC CAPITAL LETTER ABKHASIAN HA",NULL},
/* 04A9 */ { "CYRILLIC SMALL LETTER ABKHASIAN HA",NULL},
/* 04AA */ { "CYRILLIC CAPITAL LETTER ES WITH DESCENDER",NULL},
/* 04AB */ { "CYRILLIC SMALL LETTER ES WITH DESCENDER","	* Bashkir, Chuvash\n"
	"	* letterforms with right hooks are preferred, although occasional variants with left hooks occur\n"
	"	* in Chuvashia, letterforms identical to or similar in form to 00E7 regularly occur"},
/* 04AC */ { "CYRILLIC CAPITAL LETTER TE WITH DESCENDER",NULL},
/* 04AD */ { "CYRILLIC SMALL LETTER TE WITH DESCENDER","	* Abkhasian"},
/* 04AE */ { "CYRILLIC CAPITAL LETTER STRAIGHT U",NULL},
/* 04AF */ { "CYRILLIC SMALL LETTER STRAIGHT U","	* stem is straight, unlike LETTER U\n"
	"	* Azerbaijani, Bashkir, ...\n"
	"	x (cyrillic capital letter u - 0423)"},
/* 04B0 */ { "CYRILLIC CAPITAL LETTER STRAIGHT U WITH STROKE",NULL},
/* 04B1 */ { "CYRILLIC SMALL LETTER STRAIGHT U WITH STROKE","	* Kazakh"},
/* 04B2 */ { "CYRILLIC CAPITAL LETTER HA WITH DESCENDER",NULL},
/* 04B3 */ { "CYRILLIC SMALL LETTER HA WITH DESCENDER","	* Abkhasian, Tajik, Uzbek"},
/* 04B4 */ { "CYRILLIC CAPITAL LIGATURE TE TSE",NULL},
/* 04B5 */ { "CYRILLIC SMALL LIGATURE TE TSE","	* Abkhasian\n"
	"	* this is not a decomposable ligature"},
/* 04B6 */ { "CYRILLIC CAPITAL LETTER CHE WITH DESCENDER",NULL},
/* 04B7 */ { "CYRILLIC SMALL LETTER CHE WITH DESCENDER","	* Abkhasian, Tajik"},
/* 04B8 */ { "CYRILLIC CAPITAL LETTER CHE WITH VERTICAL STROKE",NULL},
/* 04B9 */ { "CYRILLIC SMALL LETTER CHE WITH VERTICAL STROKE","	* Azerbaijani"},
/* 04BA */ { "CYRILLIC CAPITAL LETTER SHHA",NULL},
/* 04BB */ { "CYRILLIC SMALL LETTER SHHA","	* originally derived from Latin \"h\", but uppercase form 04BA is closer to an inverted che (0427)\n"
	"	* Azerbaijani, Bashkir, ...\n"
	"	x (latin small letter h - 0068)"},
/* 04BC */ { "CYRILLIC CAPITAL LETTER ABKHASIAN CHE",NULL},
/* 04BD */ { "CYRILLIC SMALL LETTER ABKHASIAN CHE",NULL},
/* 04BE */ { "CYRILLIC CAPITAL LETTER ABKHASIAN CHE WITH DESCENDER",NULL},
/* 04BF */ { "CYRILLIC SMALL LETTER ABKHASIAN CHE WITH DESCENDER","	* some older Abkhaz fonts show a descender shaped like a right hook (ogonek or reversed comma shape)"},
/* 04C0 */ { "CYRILLIC LETTER PALOCHKA","	* aspiration sign in many Caucasian languages\n"
	"	* is usually not cased, but the formal lowercase is 04CF\n"
	"	x (latin capital letter i - 0049)\n"
	"	x (cyrillic capital letter byelorussian-ukrainian i - 0406)"},
/* 04C1 */ { "CYRILLIC CAPITAL LETTER ZHE WITH BREVE","	: 0416 0306"},
/* 04C2 */ { "CYRILLIC SMALL LETTER ZHE WITH BREVE","	* Moldavian\n"
	"	: 0436 0306"},
/* 04C3 */ { "CYRILLIC CAPITAL LETTER KA WITH HOOK",NULL},
/* 04C4 */ { "CYRILLIC SMALL LETTER KA WITH HOOK","	* Chukchi"},
/* 04C5 */ { "CYRILLIC CAPITAL LETTER EL WITH TAIL",NULL},
/* 04C6 */ { "CYRILLIC SMALL LETTER EL WITH TAIL","	* Kildin Sami"},
/* 04C7 */ { "CYRILLIC CAPITAL LETTER EN WITH HOOK",NULL},
/* 04C8 */ { "CYRILLIC SMALL LETTER EN WITH HOOK","	* Chukchi"},
/* 04C9 */ { "CYRILLIC CAPITAL LETTER EN WITH TAIL",NULL},
/* 04CA */ { "CYRILLIC SMALL LETTER EN WITH TAIL","	* Kildin Sami"},
/* 04CB */ { "CYRILLIC CAPITAL LETTER KHAKASSIAN CHE",NULL},
/* 04CC */ { "CYRILLIC SMALL LETTER KHAKASSIAN CHE",NULL},
/* 04CD */ { "CYRILLIC CAPITAL LETTER EM WITH TAIL",NULL},
/* 04CE */ { "CYRILLIC SMALL LETTER EM WITH TAIL","	* Kildin Sami"},
/* 04CF */ { "CYRILLIC SMALL LETTER PALOCHKA",NULL},
/* 04D0 */ { "CYRILLIC CAPITAL LETTER A WITH BREVE","	: 0410 0306"},
/* 04D1 */ { "CYRILLIC SMALL LETTER A WITH BREVE","	: 0430 0306"},
/* 04D2 */ { "CYRILLIC CAPITAL LETTER A WITH DIAERESIS","	: 0410 0308"},
/* 04D3 */ { "CYRILLIC SMALL LETTER A WITH DIAERESIS","	: 0430 0308"},
/* 04D4 */ { "CYRILLIC CAPITAL LIGATURE A IE",NULL},
/* 04D5 */ { "CYRILLIC SMALL LIGATURE A IE","	* this is not a decomposable ligature\n"
	"	x (latin small letter ae - 00E6)"},
/* 04D6 */ { "CYRILLIC CAPITAL LETTER IE WITH BREVE","	: 0415 0306"},
/* 04D7 */ { "CYRILLIC SMALL LETTER IE WITH BREVE","	: 0435 0306"},
/* 04D8 */ { "CYRILLIC CAPITAL LETTER SCHWA",NULL},
/* 04D9 */ { "CYRILLIC SMALL LETTER SCHWA","	x (latin small letter schwa - 0259)"},
/* 04DA */ { "CYRILLIC CAPITAL LETTER SCHWA WITH DIAERESIS","	: 04D8 0308"},
/* 04DB */ { "CYRILLIC SMALL LETTER SCHWA WITH DIAERESIS","	: 04D9 0308"},
/* 04DC */ { "CYRILLIC CAPITAL LETTER ZHE WITH DIAERESIS","	: 0416 0308"},
/* 04DD */ { "CYRILLIC SMALL LETTER ZHE WITH DIAERESIS","	: 0436 0308"},
/* 04DE */ { "CYRILLIC CAPITAL LETTER ZE WITH DIAERESIS","	: 0417 0308"},
/* 04DF */ { "CYRILLIC SMALL LETTER ZE WITH DIAERESIS","	: 0437 0308"},
/* 04E0 */ { "CYRILLIC CAPITAL LETTER ABKHASIAN DZE",NULL},
/* 04E1 */ { "CYRILLIC SMALL LETTER ABKHASIAN DZE","	x (latin small letter ezh - 0292)"},
/* 04E2 */ { "CYRILLIC CAPITAL LETTER I WITH MACRON","	: 0418 0304"},
/* 04E3 */ { "CYRILLIC SMALL LETTER I WITH MACRON","	: 0438 0304"},
/* 04E4 */ { "CYRILLIC CAPITAL LETTER I WITH DIAERESIS","	: 0418 0308"},
/* 04E5 */ { "CYRILLIC SMALL LETTER I WITH DIAERESIS","	: 0438 0308"},
/* 04E6 */ { "CYRILLIC CAPITAL LETTER O WITH DIAERESIS","	: 041E 0308"},
/* 04E7 */ { "CYRILLIC SMALL LETTER O WITH DIAERESIS","	: 043E 0308"},
/* 04E8 */ { "CYRILLIC CAPITAL LETTER BARRED O",NULL},
/* 04E9 */ { "CYRILLIC SMALL LETTER BARRED O","	x (latin small letter barred o - 0275)"},
/* 04EA */ { "CYRILLIC CAPITAL LETTER BARRED O WITH DIAERESIS","	: 04E8 0308"},
/* 04EB */ { "CYRILLIC SMALL LETTER BARRED O WITH DIAERESIS","	: 04E9 0308"},
/* 04EC */ { "CYRILLIC CAPITAL LETTER E WITH DIAERESIS","	: 042D 0308"},
/* 04ED */ { "CYRILLIC SMALL LETTER E WITH DIAERESIS","	* Kildin Sami\n"
	"	: 044D 0308"},
/* 04EE */ { "CYRILLIC CAPITAL LETTER U WITH MACRON","	: 0423 0304"},
/* 04EF */ { "CYRILLIC SMALL LETTER U WITH MACRON","	: 0443 0304"},
/* 04F0 */ { "CYRILLIC CAPITAL LETTER U WITH DIAERESIS","	: 0423 0308"},
/* 04F1 */ { "CYRILLIC SMALL LETTER U WITH DIAERESIS","	: 0443 0308"},
/* 04F2 */ { "CYRILLIC CAPITAL LETTER U WITH DOUBLE ACUTE","	: 0423 030B"},
/* 04F3 */ { "CYRILLIC SMALL LETTER U WITH DOUBLE ACUTE","	: 0443 030B"},
/* 04F4 */ { "CYRILLIC CAPITAL LETTER CHE WITH DIAERESIS","	: 0427 0308"},
/* 04F5 */ { "CYRILLIC SMALL LETTER CHE WITH DIAERESIS","	: 0447 0308"},
/* 04F6 */ { "CYRILLIC CAPITAL LETTER GHE WITH DESCENDER",NULL},
/* 04F7 */ { "CYRILLIC SMALL LETTER GHE WITH DESCENDER","	* Yupik"},
/* 04F8 */ { "CYRILLIC CAPITAL LETTER YERU WITH DIAERESIS","	: 042B 0308"},
/* 04F9 */ { "CYRILLIC SMALL LETTER YERU WITH DIAERESIS","	: 044B 0308"},
/* 04FA */ { "CYRILLIC CAPITAL LETTER GHE WITH STROKE AND HOOK",NULL},
/* 04FB */ { "CYRILLIC SMALL LETTER GHE WITH STROKE AND HOOK",NULL},
/* 04FC */ { "CYRILLIC CAPITAL LETTER HA WITH HOOK",NULL},
/* 04FD */ { "CYRILLIC SMALL LETTER HA WITH HOOK",NULL},
/* 04FE */ { "CYRILLIC CAPITAL LETTER HA WITH STROKE",NULL},
/* 04FF */ { "CYRILLIC SMALL LETTER HA WITH STROKE",NULL}
};

UN_DLL_LOCAL
static const struct unicode_nameannot una_00_05[] = {
/* 0500 */ { "CYRILLIC CAPITAL LETTER KOMI DE",NULL},
/* 0501 */ { "CYRILLIC SMALL LETTER KOMI DE",NULL},
/* 0502 */ { "CYRILLIC CAPITAL LETTER KOMI DJE",NULL},
/* 0503 */ { "CYRILLIC SMALL LETTER KOMI DJE",NULL},
/* 0504 */ { "CYRILLIC CAPITAL LETTER KOMI ZJE",NULL},
/* 0505 */ { "CYRILLIC SMALL LETTER KOMI ZJE",NULL},
/* 0506 */ { "CYRILLIC CAPITAL LETTER KOMI DZJE",NULL},
/* 0507 */ { "CYRILLIC SMALL LETTER KOMI DZJE",NULL},
/* 0508 */ { "CYRILLIC CAPITAL LETTER KOMI LJE",NULL},
/* 0509 */ { "CYRILLIC SMALL LETTER KOMI LJE",NULL},
/* 050A */ { "CYRILLIC CAPITAL LETTER KOMI NJE",NULL},
/* 050B */ { "CYRILLIC SMALL LETTER KOMI NJE",NULL},
/* 050C */ { "CYRILLIC CAPITAL LETTER KOMI SJE",NULL},
/* 050D */ { "CYRILLIC SMALL LETTER KOMI SJE",NULL},
/* 050E */ { "CYRILLIC CAPITAL LETTER KOMI TJE",NULL},
/* 050F */ { "CYRILLIC SMALL LETTER KOMI TJE","	x (cyrillic small letter tje - 1C8A)\n"
	"		These letters are for the obsolete Molodtsov alphabet, used in the 1920s."},
/* 0510 */ { "CYRILLIC CAPITAL LETTER REVERSED ZE",NULL},
/* 0511 */ { "CYRILLIC SMALL LETTER REVERSED ZE","	* also used for Enets"},
/* 0512 */ { "CYRILLIC CAPITAL LETTER EL WITH HOOK",NULL},
/* 0513 */ { "CYRILLIC SMALL LETTER EL WITH HOOK","	* also used for Itelmen"},
/* 0514 */ { "CYRILLIC CAPITAL LETTER LHA",NULL},
/* 0515 */ { "CYRILLIC SMALL LETTER LHA","	= voiceless l"},
/* 0516 */ { "CYRILLIC CAPITAL LETTER RHA",NULL},
/* 0517 */ { "CYRILLIC SMALL LETTER RHA","	= voiceless r"},
/* 0518 */ { "CYRILLIC CAPITAL LETTER YAE",NULL},
/* 0519 */ { "CYRILLIC SMALL LETTER YAE",NULL},
/* 051A */ { "CYRILLIC CAPITAL LETTER QA",NULL},
/* 051B */ { "CYRILLIC SMALL LETTER QA",NULL},
/* 051C */ { "CYRILLIC CAPITAL LETTER WE",NULL},
/* 051D */ { "CYRILLIC SMALL LETTER WE",NULL},
/* 051E */ { "CYRILLIC CAPITAL LETTER ALEUT KA",NULL},
/* 051F */ { "CYRILLIC SMALL LETTER ALEUT KA","	* used for [q] in Aleut"},
/* 0520 */ { "CYRILLIC CAPITAL LETTER EL WITH MIDDLE HOOK",NULL},
/* 0521 */ { "CYRILLIC SMALL LETTER EL WITH MIDDLE HOOK","	= palatalized l"},
/* 0522 */ { "CYRILLIC CAPITAL LETTER EN WITH MIDDLE HOOK",NULL},
/* 0523 */ { "CYRILLIC SMALL LETTER EN WITH MIDDLE HOOK","	= palatalized n"},
/* 0524 */ { "CYRILLIC CAPITAL LETTER PE WITH DESCENDER",NULL},
/* 0525 */ { "CYRILLIC SMALL LETTER PE WITH DESCENDER","	* used in modern Abkhaz orthography\n"
	"	x (cyrillic small letter pe with middle hook - 04A7)"},
/* 0526 */ { "CYRILLIC CAPITAL LETTER SHHA WITH DESCENDER",NULL},
/* 0527 */ { "CYRILLIC SMALL LETTER SHHA WITH DESCENDER",NULL},
/* 0528 */ { "CYRILLIC CAPITAL LETTER EN WITH LEFT HOOK",NULL},
/* 0529 */ { "CYRILLIC SMALL LETTER EN WITH LEFT HOOK",NULL},
/* 052A */ { "CYRILLIC CAPITAL LETTER DZZHE",NULL},
/* 052B */ { "CYRILLIC SMALL LETTER DZZHE","	* also used for Ossetian until 1924"},
/* 052C */ { "CYRILLIC CAPITAL LETTER DCHE",NULL},
/* 052D */ { "CYRILLIC SMALL LETTER DCHE",NULL},
/* 052E */ { "CYRILLIC CAPITAL LETTER EL WITH DESCENDER",NULL},
/* 052F */ { "CYRILLIC SMALL LETTER EL WITH DESCENDER","	* also used for Nenets"},
/* 0530 */ { NULL,NULL},
/* 0531 */ { "ARMENIAN CAPITAL LETTER AYB",NULL},
/* 0532 */ { "ARMENIAN CAPITAL LETTER BEN",NULL},
/* 0533 */ { "ARMENIAN CAPITAL LETTER GIM",NULL},
/* 0534 */ { "ARMENIAN CAPITAL LETTER DA",NULL},
/* 0535 */ { "ARMENIAN CAPITAL LETTER ECH",NULL},
/* 0536 */ { "ARMENIAN CAPITAL LETTER ZA",NULL},
/* 0537 */ { "ARMENIAN CAPITAL LETTER EH",NULL},
/* 0538 */ { "ARMENIAN CAPITAL LETTER ET",NULL},
/* 0539 */ { "ARMENIAN CAPITAL LETTER TO",NULL},
/* 053A */ { "ARMENIAN CAPITAL LETTER ZHE",NULL},
/* 053B */ { "ARMENIAN CAPITAL LETTER INI",NULL},
/* 053C */ { "ARMENIAN CAPITAL LETTER LIWN",NULL},
/* 053D */ { "ARMENIAN CAPITAL LETTER XEH",NULL},
/* 053E */ { "ARMENIAN CAPITAL LETTER CA",NULL},
/* 053F */ { "ARMENIAN CAPITAL LETTER KEN",NULL},
/* 0540 */ { "ARMENIAN CAPITAL LETTER HO",NULL},
/* 0541 */ { "ARMENIAN CAPITAL LETTER JA",NULL},
/* 0542 */ { "ARMENIAN CAPITAL LETTER GHAD",NULL},
/* 0543 */ { "ARMENIAN CAPITAL LETTER CHEH",NULL},
/* 0544 */ { "ARMENIAN CAPITAL LETTER MEN",NULL},
/* 0545 */ { "ARMENIAN CAPITAL LETTER YI",NULL},
/* 0546 */ { "ARMENIAN CAPITAL LETTER NOW",NULL},
/* 0547 */ { "ARMENIAN CAPITAL LETTER SHA",NULL},
/* 0548 */ { "ARMENIAN CAPITAL LETTER VO",NULL},
/* 0549 */ { "ARMENIAN CAPITAL LETTER CHA",NULL},
/* 054A */ { "ARMENIAN CAPITAL LETTER PEH",NULL},
/* 054B */ { "ARMENIAN CAPITAL LETTER JHEH",NULL},
/* 054C */ { "ARMENIAN CAPITAL LETTER RA",NULL},
/* 054D */ { "ARMENIAN CAPITAL LETTER SEH",NULL},
/* 054E */ { "ARMENIAN CAPITAL LETTER VEW",NULL},
/* 054F */ { "ARMENIAN CAPITAL LETTER TIWN",NULL},
/* 0550 */ { "ARMENIAN CAPITAL LETTER REH",NULL},
/* 0551 */ { "ARMENIAN CAPITAL LETTER CO",NULL},
/* 0552 */ { "ARMENIAN CAPITAL LETTER YIWN",NULL},
/* 0553 */ { "ARMENIAN CAPITAL LETTER PIWR",NULL},
/* 0554 */ { "ARMENIAN CAPITAL LETTER KEH",NULL},
/* 0555 */ { "ARMENIAN CAPITAL LETTER OH",NULL},
/* 0556 */ { "ARMENIAN CAPITAL LETTER FEH",NULL},
/* 0557 */ { NULL,NULL},
/* 0558 */ { NULL,NULL},
/* 0559 */ { "ARMENIAN MODIFIER LETTER LEFT HALF RING","	x (modifier letter reversed comma - 02BD)\n"
	"	x (modifier letter left half ring - 02BF)\n"
	"	x (combining reversed comma above - 0314)"},
/* 055A */ { "ARMENIAN APOSTROPHE","	= armenian modifier letter right half ring (1.0)\n"
	"	x (modifier letter apostrophe - 02BC)\n"
	"	x (modifier letter right half ring - 02BE)\n"
	"	x (combining comma above - 0313)"},
/* 055B */ { "ARMENIAN EMPHASIS MARK","	= shesht\n"
	"	x (modifier letter acute accent - 02CA)"},
/* 055C */ { "ARMENIAN EXCLAMATION MARK","	= batsaganchakan nshan\n"
	"	* marks intonation on word, rather than occurring finally"},
/* 055D */ { "ARMENIAN COMMA","	= bowt\n"
	"	x (modifier letter grave accent - 02CB)"},
/* 055E */ { "ARMENIAN QUESTION MARK","	= hartsakan nshan\n"
	"	* marks intonation on word, rather than occurring finally"},
/* 055F */ { "ARMENIAN ABBREVIATION MARK","	= patiw"},
/* 0560 */ { "ARMENIAN SMALL LETTER TURNED AYB","	* used for phonetic notation"},
/* 0561 */ { "ARMENIAN SMALL LETTER AYB",NULL},
/* 0562 */ { "ARMENIAN SMALL LETTER BEN",NULL},
/* 0563 */ { "ARMENIAN SMALL LETTER GIM",NULL},
/* 0564 */ { "ARMENIAN SMALL LETTER DA",NULL},
/* 0565 */ { "ARMENIAN SMALL LETTER ECH",NULL},
/* 0566 */ { "ARMENIAN SMALL LETTER ZA",NULL},
/* 0567 */ { "ARMENIAN SMALL LETTER EH",NULL},
/* 0568 */ { "ARMENIAN SMALL LETTER ET",NULL},
/* 0569 */ { "ARMENIAN SMALL LETTER TO",NULL},
/* 056A */ { "ARMENIAN SMALL LETTER ZHE",NULL},
/* 056B */ { "ARMENIAN SMALL LETTER INI",NULL},
/* 056C */ { "ARMENIAN SMALL LETTER LIWN",NULL},
/* 056D */ { "ARMENIAN SMALL LETTER XEH",NULL},
/* 056E */ { "ARMENIAN SMALL LETTER CA",NULL},
/* 056F */ { "ARMENIAN SMALL LETTER KEN",NULL},
/* 0570 */ { "ARMENIAN SMALL LETTER HO",NULL},
/* 0571 */ { "ARMENIAN SMALL LETTER JA",NULL},
/* 0572 */ { "ARMENIAN SMALL LETTER GHAD",NULL},
/* 0573 */ { "ARMENIAN SMALL LETTER CHEH",NULL},
/* 0574 */ { "ARMENIAN SMALL LETTER MEN",NULL},
/* 0575 */ { "ARMENIAN SMALL LETTER YI",NULL},
/* 0576 */ { "ARMENIAN SMALL LETTER NOW",NULL},
/* 0577 */ { "ARMENIAN SMALL LETTER SHA",NULL},
/* 0578 */ { "ARMENIAN SMALL LETTER VO",NULL},
/* 0579 */ { "ARMENIAN SMALL LETTER CHA",NULL},
/* 057A */ { "ARMENIAN SMALL LETTER PEH",NULL},
/* 057B */ { "ARMENIAN SMALL LETTER JHEH",NULL},
/* 057C */ { "ARMENIAN SMALL LETTER RA",NULL},
/* 057D */ { "ARMENIAN SMALL LETTER SEH",NULL},
/* 057E */ { "ARMENIAN SMALL LETTER VEW",NULL},
/* 057F */ { "ARMENIAN SMALL LETTER TIWN",NULL},
/* 0580 */ { "ARMENIAN SMALL LETTER REH",NULL},
/* 0581 */ { "ARMENIAN SMALL LETTER CO",NULL},
/* 0582 */ { "ARMENIAN SMALL LETTER YIWN",NULL},
/* 0583 */ { "ARMENIAN SMALL LETTER PIWR",NULL},
/* 0584 */ { "ARMENIAN SMALL LETTER KEH",NULL},
/* 0585 */ { "ARMENIAN SMALL LETTER OH",NULL},
/* 0586 */ { "ARMENIAN SMALL LETTER FEH",NULL},
/* 0587 */ { "ARMENIAN SMALL LIGATURE ECH YIWN","	# 0565 0582"},
/* 0588 */ { "ARMENIAN SMALL LETTER YI WITH STROKE","	* used for phonetic notation"},
/* 0589 */ { "ARMENIAN FULL STOP","	= vertsaket\n"
	"	* may also be used for historic Georgian texts\n"
	"	x (colon - 003A)"},
/* 058A */ { "ARMENIAN HYPHEN","	= yentamna"},
/* 058B */ { NULL,NULL},
/* 058C */ { NULL,NULL},
/* 058D */ { "RIGHT-FACING ARMENIAN ETERNITY SIGN",NULL},
/* 058E */ { "LEFT-FACING ARMENIAN ETERNITY SIGN","	* maps to AST 34.005:1997"},
/* 058F */ { "ARMENIAN DRAM SIGN",NULL},
/* 0590 */ { NULL,NULL},
/* 0591 */ { "HEBREW ACCENT ETNAHTA","	= atnah"},
/* 0592 */ { "HEBREW ACCENT SEGOL","	= segolta"},
/* 0593 */ { "HEBREW ACCENT SHALSHELET",NULL},
/* 0594 */ { "HEBREW ACCENT ZAQEF QATAN",NULL},
/* 0595 */ { "HEBREW ACCENT ZAQEF GADOL",NULL},
/* 0596 */ { "HEBREW ACCENT TIPEHA","	= tarha, me'ayla ~ mayla"},
/* 0597 */ { "HEBREW ACCENT REVIA",NULL},
/* 0598 */ { "HEBREW ACCENT ZARQA","	= tsinorit, zinorit; tsinor, zinor\n"
	"	* This character is to be used when Zarqa or Tsinor are placed above, and also for Tsinorit.\n"
	"	x (hebrew accent zinor - 05AE)"},
/* 0599 */ { "HEBREW ACCENT PASHTA",NULL},
/* 059A */ { "HEBREW ACCENT YETIV",NULL},
/* 059B */ { "HEBREW ACCENT TEVIR",NULL},
/* 059C */ { "HEBREW ACCENT GERESH","	= teres"},
/* 059D */ { "HEBREW ACCENT GERESH MUQDAM",NULL},
/* 059E */ { "HEBREW ACCENT GERSHAYIM",NULL},
/* 059F */ { "HEBREW ACCENT QARNEY PARA","	= pazer gadol"},
/* 05A0 */ { "HEBREW ACCENT TELISHA GEDOLA",NULL},
/* 05A1 */ { "HEBREW ACCENT PAZER","	= pazer qatan"},
/* 05A2 */ { "HEBREW ACCENT ATNAH HAFUKH","	x (hebrew accent yerah ben yomo - 05AA)"},
/* 05A3 */ { "HEBREW ACCENT MUNAH",NULL},
/* 05A4 */ { "HEBREW ACCENT MAHAPAKH",NULL},
/* 05A5 */ { "HEBREW ACCENT MERKHA","	= yored"},
/* 05A6 */ { "HEBREW ACCENT MERKHA KEFULA",NULL},
/* 05A7 */ { "HEBREW ACCENT DARGA",NULL},
/* 05A8 */ { "HEBREW ACCENT QADMA","	= azla"},
/* 05A9 */ { "HEBREW ACCENT TELISHA QETANA",NULL},
/* 05AA */ { "HEBREW ACCENT YERAH BEN YOMO","	= galgal\n"
	"	x (hebrew accent atnah hafukh - 05A2)"},
/* 05AB */ { "HEBREW ACCENT OLE",NULL},
/* 05AC */ { "HEBREW ACCENT ILUY",NULL},
/* 05AD */ { "HEBREW ACCENT DEHI",NULL},
/* 05AE */ { "HEBREW ACCENT ZINOR","	= tsinor; zarqa\n"
	"	* This character is to be used when Zarqa or Tsinor are placed above left.\n"
	"	x (hebrew accent zarqa - 0598)"},
/* 05AF */ { "HEBREW MARK MASORA CIRCLE",NULL},
/* 05B0 */ { "HEBREW POINT SHEVA",NULL},
/* 05B1 */ { "HEBREW POINT HATAF SEGOL",NULL},
/* 05B2 */ { "HEBREW POINT HATAF PATAH",NULL},
/* 05B3 */ { "HEBREW POINT HATAF QAMATS",NULL},
/* 05B4 */ { "HEBREW POINT HIRIQ",NULL},
/* 05B5 */ { "HEBREW POINT TSERE",NULL},
/* 05B6 */ { "HEBREW POINT SEGOL",NULL},
/* 05B7 */ { "HEBREW POINT PATAH","	* furtive patah is not a distinct character"},
/* 05B8 */ { "HEBREW POINT QAMATS","	* used generically or as qamats gadol in orthography which distinguishes that from qamats qatan\n"
	"	x (hebrew point qamats qatan - 05C7)"},
/* 05B9 */ { "HEBREW POINT HOLAM",NULL},
/* 05BA */ { "HEBREW POINT HOLAM HASER FOR VAV",NULL},
/* 05BB */ { "HEBREW POINT QUBUTS",NULL},
/* 05BC */ { "HEBREW POINT DAGESH OR MAPIQ","	= shuruq\n"
	"	* falls within the base letter"},
/* 05BD */ { "HEBREW POINT METEG","	= siluq\n"
	"	* may be used as a Hebrew accent sof pasuq"},
/* 05BE */ { "HEBREW PUNCTUATION MAQAF",NULL},
/* 05BF */ { "HEBREW POINT RAFE","	x (hebrew point judeo-spanish varika - FB1E)"},
/* 05C0 */ { "HEBREW PUNCTUATION PASEQ","	= legarmeh\n"
	"	* may be treated as spacing punctuation, not as a point\n"
	"	x (vertical line - 007C)"},
/* 05C1 */ { "HEBREW POINT SHIN DOT",NULL},
/* 05C2 */ { "HEBREW POINT SIN DOT",NULL},
/* 05C3 */ { "HEBREW PUNCTUATION SOF PASUQ","	* may be used as a Hebrew punctuation colon\n"
	"	x (colon - 003A)"},
/* 05C4 */ { "HEBREW MARK UPPER DOT",NULL},
/* 05C5 */ { "HEBREW MARK LOWER DOT","	* punctum extraordinarium (Psalms 27:13)\n"
	"	x (hebrew point hiriq - 05B4)"},
/* 05C6 */ { "HEBREW PUNCTUATION NUN HAFUKHA","	* does not historically derive from the letter nun\n"
	"	x (hebrew letter nun - 05E0)"},
/* 05C7 */ { "HEBREW POINT QAMATS QATAN","	x (hebrew point qamats - 05B8)"},
/* 05C8 */ { NULL,NULL},
/* 05C9 */ { NULL,NULL},
/* 05CA */ { NULL,NULL},
/* 05CB */ { NULL,NULL},
/* 05CC */ { NULL,NULL},
/* 05CD */ { NULL,NULL},
/* 05CE */ { NULL,NULL},
/* 05CF */ { NULL,NULL},
/* 05D0 */ { "HEBREW LETTER ALEF","	= aleph\n"
	"	x (alef symbol - 2135)"},
/* 05D1 */ { "HEBREW LETTER BET","	x (bet symbol - 2136)"},
/* 05D2 */ { "HEBREW LETTER GIMEL","	x (gimel symbol - 2137)"},
/* 05D3 */ { "HEBREW LETTER DALET","	x (dalet symbol - 2138)"},
/* 05D4 */ { "HEBREW LETTER HE",NULL},
/* 05D5 */ { "HEBREW LETTER VAV",NULL},
/* 05D6 */ { "HEBREW LETTER ZAYIN",NULL},
/* 05D7 */ { "HEBREW LETTER HET",NULL},
/* 05D8 */ { "HEBREW LETTER TET",NULL},
/* 05D9 */ { "HEBREW LETTER YOD",NULL},
/* 05DA */ { "HEBREW LETTER FINAL KAF",NULL},
/* 05DB */ { "HEBREW LETTER KAF",NULL},
/* 05DC */ { "HEBREW LETTER LAMED",NULL},
/* 05DD */ { "HEBREW LETTER FINAL MEM",NULL},
/* 05DE */ { "HEBREW LETTER MEM",NULL},
/* 05DF */ { "HEBREW LETTER FINAL NUN",NULL},
/* 05E0 */ { "HEBREW LETTER NUN",NULL},
/* 05E1 */ { "HEBREW LETTER SAMEKH",NULL},
/* 05E2 */ { "HEBREW LETTER AYIN",NULL},
/* 05E3 */ { "HEBREW LETTER FINAL PE",NULL},
/* 05E4 */ { "HEBREW LETTER PE",NULL},
/* 05E5 */ { "HEBREW LETTER FINAL TSADI",NULL},
/* 05E6 */ { "HEBREW LETTER TSADI","	= zade"},
/* 05E7 */ { "HEBREW LETTER QOF",NULL},
/* 05E8 */ { "HEBREW LETTER RESH",NULL},
/* 05E9 */ { "HEBREW LETTER SHIN",NULL},
/* 05EA */ { "HEBREW LETTER TAV",NULL},
/* 05EB */ { NULL,NULL},
/* 05EC */ { NULL,NULL},
/* 05ED */ { NULL,NULL},
/* 05EE */ { NULL,NULL},
/* 05EF */ { "HEBREW YOD TRIANGLE",NULL},
/* 05F0 */ { "HEBREW LIGATURE YIDDISH DOUBLE VAV","	= tsvey vovn"},
/* 05F1 */ { "HEBREW LIGATURE YIDDISH VAV YOD",NULL},
/* 05F2 */ { "HEBREW LIGATURE YIDDISH DOUBLE YOD","	= tsvey yudn"},
/* 05F3 */ { "HEBREW PUNCTUATION GERESH","	x (apostrophe - 0027)"},
/* 05F4 */ { "HEBREW PUNCTUATION GERSHAYIM","	x (quotation mark - 0022)"},
/* 05F5 */ { NULL,NULL},
/* 05F6 */ { NULL,NULL},
/* 05F7 */ { NULL,NULL},
/* 05F8 */ { NULL,NULL},
/* 05F9 */ { NULL,NULL},
/* 05FA */ { NULL,NULL},
/* 05FB */ { NULL,NULL},
/* 05FC */ { NULL,NULL},
/* 05FD */ { NULL,NULL},
/* 05FE */ { NULL,NULL},
/* 05FF */ { NULL,NULL}
};

UN_DLL_LOCAL
static const struct unicode_nameannot una_00_06[] = {
/* 0600 */ { "ARABIC NUMBER SIGN",NULL},
/* 0601 */ { "ARABIC SIGN SANAH",NULL},
/* 0602 */ { "ARABIC FOOTNOTE MARKER",NULL},
/* 0603 */ { "ARABIC SIGN SAFHA",NULL},
/* 0604 */ { "ARABIC SIGN SAMVAT","	* used for writing Samvat era dates in Urdu"},
/* 0605 */ { "ARABIC NUMBER MARK ABOVE","	* may be used with Coptic Epact numbers"},
/* 0606 */ { "ARABIC-INDIC CUBE ROOT","	x (cube root - 221B)"},
/* 0607 */ { "ARABIC-INDIC FOURTH ROOT","	x (fourth root - 221C)"},
/* 0608 */ { "ARABIC RAY",NULL},
/* 0609 */ { "ARABIC-INDIC PER MILLE SIGN","	x (per mille sign - 2030)"},
/* 060A */ { "ARABIC-INDIC PER TEN THOUSAND SIGN","	x (per ten thousand sign - 2031)"},
/* 060B */ { "AFGHANI SIGN",NULL},
/* 060C */ { "ARABIC COMMA","	* also used with Thaana and Syriac in modern text\n"
	"	x (comma - 002C)\n"
	"	x (turned comma - 2E32)\n"
	"	x (reversed comma - 2E41)"},
/* 060D */ { "ARABIC DATE SEPARATOR",NULL},
/* 060E */ { "ARABIC POETIC VERSE SIGN",NULL},
/* 060F */ { "ARABIC SIGN MISRA",NULL},
/* 0610 */ { "ARABIC SIGN SALLALLAHOU ALAYHE WASSALLAM","	* represents sallallahu alayhe wasallam \"may God's peace and blessings be upon him\""},
/* 0611 */ { "ARABIC SIGN ALAYHE ASSALLAM","	* represents alayhe assalam \"upon him be peace\"\n"
	"	x (arabic ligature alayhi as-salaam - FD47)"},
/* 0612 */ { "ARABIC SIGN RAHMATULLAH ALAYHE","	* represents rahmatullah alayhe \"may God have mercy upon him\"\n"
	"	x (arabic ligature rahimahu allaah - FD40)"},
/* 0613 */ { "ARABIC SIGN RADI ALLAHOU ANHU","	* represents radi allahu 'anhu \"may God be pleased with him\"\n"
	"	x (arabic ligature radi allaahu anh - FD41)"},
/* 0614 */ { "ARABIC SIGN TAKHALLUS","	* sign placed over the name or nom-de-plume of a poet, or in some writings used to mark all proper names"},
/* 0615 */ { "ARABIC SMALL HIGH TAH","	* marks a recommended pause position in some Qurans published in Iran and Pakistan\n"
	"	* should not be confused with the small TAH sign used as a diacritic for some letters such as 0679"},
/* 0616 */ { "ARABIC SMALL HIGH LIGATURE ALEF WITH LAM WITH YEH","	% ARABIC SMALL HIGH LIGATURE ALEF WITH YEH BARREE\n"
	"	* early Persian"},
/* 0617 */ { "ARABIC SMALL HIGH ZAIN",NULL},
/* 0618 */ { "ARABIC SMALL FATHA","	* should not be confused with 064E FATHA"},
/* 0619 */ { "ARABIC SMALL DAMMA","	* should not be confused with 064F DAMMA"},
/* 061A */ { "ARABIC SMALL KASRA","	* should not be confused with 0650 KASRA"},
/* 061B */ { "ARABIC SEMICOLON","	* also used with Thaana and Syriac in modern text\n"
	"	x (semicolon - 003B)\n"
	"	x (reversed semicolon - 204F)\n"
	"	x (turned semicolon - 2E35)"},
/* 061C */ { "ARABIC LETTER MARK","	* commonly abbreviated ALM\n"
	"	x (right-to-left mark - 200F)"},
/* 061D */ { "ARABIC END OF TEXT MARK",NULL},
/* 061E */ { "ARABIC TRIPLE DOT PUNCTUATION MARK",NULL},
/* 061F */ { "ARABIC QUESTION MARK","	* also used with Thaana and Syriac in modern text\n"
	"	x (question mark - 003F)\n"
	"	x (reversed question mark - 2E2E)"},
/* 0620 */ { "ARABIC LETTER KASHMIRI YEH","	* used in Kashmiri to represent half ye / taler ye"},
/* 0621 */ { "ARABIC LETTER HAMZA","	x (modifier letter right half ring - 02BE)"},
/* 0622 */ { "ARABIC LETTER ALEF WITH MADDA ABOVE","	: 0627 0653"},
/* 0623 */ { "ARABIC LETTER ALEF WITH HAMZA ABOVE","	: 0627 0654"},
/* 0624 */ { "ARABIC LETTER WAW WITH HAMZA ABOVE","	: 0648 0654"},
/* 0625 */ { "ARABIC LETTER ALEF WITH HAMZA BELOW","	: 0627 0655"},
/* 0626 */ { "ARABIC LETTER YEH WITH HAMZA ABOVE","	* in Kyrgyz the hamza is consistently positioned to the top right in isolate and final forms\n"
	"	: 064A 0654"},
/* 0627 */ { "ARABIC LETTER ALEF",NULL},
/* 0628 */ { "ARABIC LETTER BEH",NULL},
/* 0629 */ { "ARABIC LETTER TEH MARBUTA",NULL},
/* 062A */ { "ARABIC LETTER TEH",NULL},
/* 062B */ { "ARABIC LETTER THEH",NULL},
/* 062C */ { "ARABIC LETTER JEEM",NULL},
/* 062D */ { "ARABIC LETTER HAH",NULL},
/* 062E */ { "ARABIC LETTER KHAH",NULL},
/* 062F */ { "ARABIC LETTER DAL",NULL},
/* 0630 */ { "ARABIC LETTER THAL",NULL},
/* 0631 */ { "ARABIC LETTER REH",NULL},
/* 0632 */ { "ARABIC LETTER ZAIN",NULL},
/* 0633 */ { "ARABIC LETTER SEEN",NULL},
/* 0634 */ { "ARABIC LETTER SHEEN",NULL},
/* 0635 */ { "ARABIC LETTER SAD",NULL},
/* 0636 */ { "ARABIC LETTER DAD",NULL},
/* 0637 */ { "ARABIC LETTER TAH",NULL},
/* 0638 */ { "ARABIC LETTER ZAH",NULL},
/* 0639 */ { "ARABIC LETTER AIN","	x (latin small letter ezh reversed - 01B9)\n"
	"	x (modifier letter left half ring - 02BF)"},
/* 063A */ { "ARABIC LETTER GHAIN",NULL},
/* 063B */ { "ARABIC LETTER KEHEH WITH TWO DOTS ABOVE",NULL},
/* 063C */ { "ARABIC LETTER KEHEH WITH THREE DOTS BELOW",NULL},
/* 063D */ { "ARABIC LETTER FARSI YEH WITH INVERTED V","	* Azerbaijani"},
/* 063E */ { "ARABIC LETTER FARSI YEH WITH TWO DOTS ABOVE",NULL},
/* 063F */ { "ARABIC LETTER FARSI YEH WITH THREE DOTS ABOVE",NULL},
/* 0640 */ { "ARABIC TATWEEL","	= kashida\n"
	"	* inserted to stretch characters or to carry tashkil with no base letter\n"
	"	* also used with Adlam, Hanifi Rohingya, Mandaic, Manichaean, Psalter Pahlavi, Sogdian, and Syriac"},
/* 0641 */ { "ARABIC LETTER FEH",NULL},
/* 0642 */ { "ARABIC LETTER QAF",NULL},
/* 0643 */ { "ARABIC LETTER KAF",NULL},
/* 0644 */ { "ARABIC LETTER LAM",NULL},
/* 0645 */ { "ARABIC LETTER MEEM","	* Sindhi uses a shape with a short tail"},
/* 0646 */ { "ARABIC LETTER NOON",NULL},
/* 0647 */ { "ARABIC LETTER HEH",NULL},
/* 0648 */ { "ARABIC LETTER WAW",NULL},
/* 0649 */ { "ARABIC LETTER ALEF MAKSURA","	* represents YEH-shaped dual-joining letter with no dots in any positional form\n"
	"	* not intended for use in combination with 0654\n"
	"	x (arabic letter yeh with hamza above - 0626)"},
/* 064A */ { "ARABIC LETTER YEH","	* loses its dots when used in combination with 0654\n"
	"	* retains its dots when used in combination with other combining marks\n"
	"	x (arabic letter yeh with two dots below and hamza above - 08A8)"},
/* 064B */ { "ARABIC FATHATAN",NULL},
/* 064C */ { "ARABIC DAMMATAN","	* a common alternative form is written as two intertwined dammas, one of which is turned 180 degrees"},
/* 064D */ { "ARABIC KASRATAN",NULL},
/* 064E */ { "ARABIC FATHA",NULL},
/* 064F */ { "ARABIC DAMMA",NULL},
/* 0650 */ { "ARABIC KASRA",NULL},
/* 0651 */ { "ARABIC SHADDA",NULL},
/* 0652 */ { "ARABIC SUKUN","	* marks absence of a vowel after the base consonant\n"
	"	* used in some Qurans to mark a long vowel as ignored\n"
	"	* can have a variety of shapes, including a circular one and a shape that looks like '06E1'\n"
	"	x (arabic small high dotless head of khah - 06E1)"},
/* 0653 */ { "ARABIC MADDAH ABOVE","	* used for madd jaa'iz in South Asian and Indonesian orthographies\n"
	"	x (arabic pepet - 0897)\n"
	"	x (arabic madda waajib - 089C)\n"
	"	x (arabic doubled madda - 089E)\n"
	"	x (arabic half madda over madda - 089F)"},
/* 0654 */ { "ARABIC HAMZA ABOVE","	* restricted to hamza and ezafe semantics\n"
	"	* is not used as a diacritic to form new letters"},
/* 0655 */ { "ARABIC HAMZA BELOW",NULL},
/* 0656 */ { "ARABIC SUBSCRIPT ALEF",NULL},
/* 0657 */ { "ARABIC INVERTED DAMMA","	= ulta pesh\n"
	"	* Kashmiri, Urdu"},
/* 0658 */ { "ARABIC MARK NOON GHUNNA","	* Baluchi\n"
	"	* indicates nasalization in Urdu"},
/* 0659 */ { "ARABIC ZWARAKAY","	* Pashto"},
/* 065A */ { "ARABIC VOWEL SIGN SMALL V ABOVE","	* African languages"},
/* 065B */ { "ARABIC VOWEL SIGN INVERTED SMALL V ABOVE","	* African languages"},
/* 065C */ { "ARABIC VOWEL SIGN DOT BELOW","	* African languages\n"
	"	* also used in Quranic text in African and other orthographies"},
/* 065D */ { "ARABIC REVERSED DAMMA","	* African languages"},
/* 065E */ { "ARABIC FATHA WITH TWO DOTS","	* Kalami"},
/* 065F */ { "ARABIC WAVY HAMZA BELOW","	* Kashmiri"},
/* 0660 */ { "ARABIC-INDIC DIGIT ZERO",NULL},
/* 0661 */ { "ARABIC-INDIC DIGIT ONE",NULL},
/* 0662 */ { "ARABIC-INDIC DIGIT TWO",NULL},
/* 0663 */ { "ARABIC-INDIC DIGIT THREE",NULL},
/* 0664 */ { "ARABIC-INDIC DIGIT FOUR",NULL},
/* 0665 */ { "ARABIC-INDIC DIGIT FIVE",NULL},
/* 0666 */ { "ARABIC-INDIC DIGIT SIX",NULL},
/* 0667 */ { "ARABIC-INDIC DIGIT SEVEN",NULL},
/* 0668 */ { "ARABIC-INDIC DIGIT EIGHT",NULL},
/* 0669 */ { "ARABIC-INDIC DIGIT NINE",NULL},
/* 066A */ { "ARABIC PERCENT SIGN","	x (percent sign - 0025)"},
/* 066B */ { "ARABIC DECIMAL SEPARATOR",NULL},
/* 066C */ { "ARABIC THOUSANDS SEPARATOR","	x (apostrophe - 0027)\n"
	"	x (right single quotation mark - 2019)"},
/* 066D */ { "ARABIC FIVE POINTED STAR","	* appearance rather variable\n"
	"	x (asterisk - 002A)"},
/* 066E */ { "ARABIC LETTER DOTLESS BEH",NULL},
/* 066F */ { "ARABIC LETTER DOTLESS QAF",NULL},
/* 0670 */ { "ARABIC LETTER SUPERSCRIPT ALEF","	* actually a vowel sign, despite the name"},
/* 0671 */ { "ARABIC LETTER ALEF WASLA","	* Quranic Arabic"},
/* 0672 */ { "ARABIC LETTER ALEF WITH WAVY HAMZA ABOVE","	* Baluchi, Kashmiri"},
/* 0673 */ { "ARABIC LETTER ALEF WITH WAVY HAMZA BELOW","	* Kashmiri\n"
	"	* this character is deprecated and its use is strongly discouraged\n"
	"	* use the sequence 0627 065F instead"},
/* 0674 */ { "ARABIC LETTER HIGH HAMZA","	* Kazakh, Jawi\n"
	"	* forms digraphs"},
/* 0675 */ { "ARABIC LETTER HIGH HAMZA ALEF","	* preferred spelling is 0674 0627\n"
	"	# 0627 0674"},
/* 0676 */ { "ARABIC LETTER HIGH HAMZA WAW","	* preferred spelling is 0674 0648\n"
	"	# 0648 0674"},
/* 0677 */ { "ARABIC LETTER U WITH HAMZA ABOVE","	* preferred spelling is 0674 06C7\n"
	"	# 06C7 0674"},
/* 0678 */ { "ARABIC LETTER HIGH HAMZA YEH","	* preferred spelling is 0674 0649\n"
	"	# 064A 0674"},
/* 0679 */ { "ARABIC LETTER TTEH","	* Urdu"},
/* 067A */ { "ARABIC LETTER TTEHEH","	* Sindhi"},
/* 067B */ { "ARABIC LETTER BEEH","	* Sindhi"},
/* 067C */ { "ARABIC LETTER TEH WITH RING","	* Pashto"},
/* 067D */ { "ARABIC LETTER TEH WITH THREE DOTS ABOVE DOWNWARDS","	* Sindhi"},
/* 067E */ { "ARABIC LETTER PEH","	* Persian, Urdu, ..."},
/* 067F */ { "ARABIC LETTER TEHEH","	* Sindhi"},
/* 0680 */ { "ARABIC LETTER BEHEH","	* Sindhi"},
/* 0681 */ { "ARABIC LETTER HAH WITH HAMZA ABOVE","	* Pashto\n"
	"	* represents the phoneme /dz/"},
/* 0682 */ { "ARABIC LETTER HAH WITH TWO DOTS VERTICAL ABOVE","	* not used in modern Pashto"},
/* 0683 */ { "ARABIC LETTER NYEH","	* Sindhi"},
/* 0684 */ { "ARABIC LETTER DYEH","	* Sindhi, Bosnian"},
/* 0685 */ { "ARABIC LETTER HAH WITH THREE DOTS ABOVE","	* Pashto, Khwarazmian\n"
	"	* represents the phoneme /ts/ in Pashto"},
/* 0686 */ { "ARABIC LETTER TCHEH","	* Persian, Urdu, ..."},
/* 0687 */ { "ARABIC LETTER TCHEHEH","	* Sindhi"},
/* 0688 */ { "ARABIC LETTER DDAL","	* Urdu"},
/* 0689 */ { "ARABIC LETTER DAL WITH RING","	* Pashto"},
/* 068A */ { "ARABIC LETTER DAL WITH DOT BELOW","	* Sindhi, early Persian, Pegon, Malagasy"},
/* 068B */ { "ARABIC LETTER DAL WITH DOT BELOW AND SMALL TAH","	* Lahnda"},
/* 068C */ { "ARABIC LETTER DAHAL","	* Sindhi"},
/* 068D */ { "ARABIC LETTER DDAHAL","	* Sindhi"},
/* 068E */ { "ARABIC LETTER DUL","	* older shape for DUL, now obsolete in Sindhi\n"
	"	* Burushaski"},
/* 068F */ { "ARABIC LETTER DAL WITH THREE DOTS ABOVE DOWNWARDS","	* Sindhi\n"
	"	* current shape used for DUL"},
/* 0690 */ { "ARABIC LETTER DAL WITH FOUR DOTS ABOVE","	* Old Urdu, not in current use"},
/* 0691 */ { "ARABIC LETTER RREH","	* Urdu"},
/* 0692 */ { "ARABIC LETTER REH WITH SMALL V","	* Kurdish"},
/* 0693 */ { "ARABIC LETTER REH WITH RING","	* Pashto"},
/* 0694 */ { "ARABIC LETTER REH WITH DOT BELOW","	* Kurdish, early Persian"},
/* 0695 */ { "ARABIC LETTER REH WITH SMALL V BELOW","	* Kurdish"},
/* 0696 */ { "ARABIC LETTER REH WITH DOT BELOW AND DOT ABOVE","	* Pashto"},
/* 0697 */ { "ARABIC LETTER REH WITH TWO DOTS ABOVE","	* Dargwa"},
/* 0698 */ { "ARABIC LETTER JEH","	* Persian, Urdu, ..."},
/* 0699 */ { "ARABIC LETTER REH WITH FOUR DOTS ABOVE","	* Sindhi"},
/* 069A */ { "ARABIC LETTER SEEN WITH DOT BELOW AND DOT ABOVE","	* Pashto"},
/* 069B */ { "ARABIC LETTER SEEN WITH THREE DOTS BELOW","	* early Persian"},
/* 069C */ { "ARABIC LETTER SEEN WITH THREE DOTS BELOW AND THREE DOTS ABOVE","	* Moroccan Arabic"},
/* 069D */ { "ARABIC LETTER SAD WITH TWO DOTS BELOW","	* Turkic"},
/* 069E */ { "ARABIC LETTER SAD WITH THREE DOTS ABOVE","	* Berber, Burushaski"},
/* 069F */ { "ARABIC LETTER TAH WITH THREE DOTS ABOVE","	* Old Hausa"},
/* 06A0 */ { "ARABIC LETTER AIN WITH THREE DOTS ABOVE","	* Jawi"},
/* 06A1 */ { "ARABIC LETTER DOTLESS FEH","	* Adighe"},
/* 06A2 */ { "ARABIC LETTER FEH WITH DOT MOVED BELOW","	* Maghrib Arabic"},
/* 06A3 */ { "ARABIC LETTER FEH WITH DOT BELOW","	* Ingush"},
/* 06A4 */ { "ARABIC LETTER VEH","	* Middle Eastern Arabic for foreign words\n"
	"	* Kurdish, Khwarazmian, early Persian, Jawi"},
/* 06A5 */ { "ARABIC LETTER FEH WITH THREE DOTS BELOW","	* North African Arabic for foreign words"},
/* 06A6 */ { "ARABIC LETTER PEHEH","	* Sindhi"},
/* 06A7 */ { "ARABIC LETTER QAF WITH DOT ABOVE","	* Maghrib Arabic, Uyghur"},
/* 06A8 */ { "ARABIC LETTER QAF WITH THREE DOTS ABOVE","	* Tunisian Arabic"},
/* 06A9 */ { "ARABIC LETTER KEHEH","	= kaf mashkula\n"
	"	* Persian, Urdu, Sindhi, ..."},
/* 06AA */ { "ARABIC LETTER SWASH KAF","	* represents a letter distinct from Arabic KAF (0643) in Sindhi"},
/* 06AB */ { "ARABIC LETTER KAF WITH RING","	* Pashto\n"
	"	* may appear like an Arabic KAF (0643) with a ring below the base"},
/* 06AC */ { "ARABIC LETTER KAF WITH DOT ABOVE","	* use for the Jawi gaf is not recommended, although it may be found in some existing text data; recommended character for Jawi gaf is 0762\n"
	"	x (arabic letter keheh with dot above - 0762)"},
/* 06AD */ { "ARABIC LETTER NG","	* Uyghur, Kazakh, Moroccan Arabic, early Jawi, early Persian, ..."},
/* 06AE */ { "ARABIC LETTER KAF WITH THREE DOTS BELOW","	* Berber, early Persian\n"
	"	* Pegon alternative for 08B4"},
/* 06AF */ { "ARABIC LETTER GAF","	* Persian, Urdu, ..."},
/* 06B0 */ { "ARABIC LETTER GAF WITH RING","	* Lahnda"},
/* 06B1 */ { "ARABIC LETTER NGOEH","	* Sindhi"},
/* 06B2 */ { "ARABIC LETTER GAF WITH TWO DOTS BELOW","	* not used in Sindhi"},
/* 06B3 */ { "ARABIC LETTER GUEH","	* Sindhi"},
/* 06B4 */ { "ARABIC LETTER GAF WITH THREE DOTS ABOVE","	* not used in Sindhi"},
/* 06B5 */ { "ARABIC LETTER LAM WITH SMALL V","	* Kurdish, Bosnian"},
/* 06B6 */ { "ARABIC LETTER LAM WITH DOT ABOVE","	* Kurdish"},
/* 06B7 */ { "ARABIC LETTER LAM WITH THREE DOTS ABOVE","	* Kurdish"},
/* 06B8 */ { "ARABIC LETTER LAM WITH THREE DOTS BELOW",NULL},
/* 06B9 */ { "ARABIC LETTER NOON WITH DOT BELOW",NULL},
/* 06BA */ { "ARABIC LETTER NOON GHUNNA","	* Urdu, archaic Arabic\n"
	"	* dotless in all four contextual forms"},
/* 06BB */ { "ARABIC LETTER RNOON","	* Sindhi"},
/* 06BC */ { "ARABIC LETTER NOON WITH RING","	* Pashto"},
/* 06BD */ { "ARABIC LETTER NOON WITH THREE DOTS ABOVE","	* Jawi"},
/* 06BE */ { "ARABIC LETTER HEH DOACHASHMEE","	* forms aspirate digraphs in Urdu and other languages of South Asia\n"
	"	* represents the glottal fricative /h/ in Uyghur"},
/* 06BF */ { "ARABIC LETTER TCHEH WITH DOT ABOVE",NULL},
/* 06C0 */ { "ARABIC LETTER HEH WITH YEH ABOVE","	= arabic letter hamzah on ha (1.0)\n"
	"	* for ezafe, use 0654 over the language-appropriate base letter\n"
	"	* actually a ligature, not an independent letter\n"
	"	: 06D5 0654"},
/* 06C1 */ { "ARABIC LETTER HEH GOAL","	* Urdu"},
/* 06C2 */ { "ARABIC LETTER HEH GOAL WITH HAMZA ABOVE","	* Urdu\n"
	"	* actually a ligature, not an independent letter\n"
	"	: 06C1 0654"},
/* 06C3 */ { "ARABIC LETTER TEH MARBUTA GOAL","	* Urdu"},
/* 06C4 */ { "ARABIC LETTER WAW WITH RING","	* Kashmiri"},
/* 06C5 */ { "ARABIC LETTER KIRGHIZ OE","	* Kyrgyz\n"
	"	* a glyph variant occurs which replaces the looped tail with a horizontal bar through the tail"},
/* 06C6 */ { "ARABIC LETTER OE","	* Uyghur, Kurdish, Kazakh, Azerbaijani, Bosnian"},
/* 06C7 */ { "ARABIC LETTER U","	* Azerbaijani, Kazakh, Kyrgyz, Uyghur"},
/* 06C8 */ { "ARABIC LETTER YU","	* Uyghur"},
/* 06C9 */ { "ARABIC LETTER KIRGHIZ YU","	* Kazakh, Kyrgyz, Bosnian"},
/* 06CA */ { "ARABIC LETTER WAW WITH TWO DOTS ABOVE","	* Kurdish"},
/* 06CB */ { "ARABIC LETTER VE","	* Uyghur, Kazakh"},
/* 06CC */ { "ARABIC LETTER FARSI YEH","	* Arabic, Persian, Urdu, Kashmiri, ...\n"
	"	* initial and medial forms of this letter have two horizontal dots below\n"
	"	* retains its dots in initial and medial forms when used in combination with 0654\n"
	"	x (arabic letter alef maksura - 0649)\n"
	"	x (arabic letter yeh - 064A)"},
/* 06CD */ { "ARABIC LETTER YEH WITH TAIL","	* Pashto, Sindhi\n"
	"	* should not be used in Kashmiri to represent half ye /taler ye\n"
	"	x (arabic letter kashmiri yeh - 0620)"},
/* 06CE */ { "ARABIC LETTER YEH WITH SMALL V","	* Kurdish"},
/* 06CF */ { "ARABIC LETTER WAW WITH DOT ABOVE",NULL},
/* 06D0 */ { "ARABIC LETTER E","	* Pashto, Uyghur\n"
	"	* used as the letter bbeh in Sindhi"},
/* 06D1 */ { "ARABIC LETTER YEH WITH THREE DOTS BELOW","	* Mende languages, Hausa"},
/* 06D2 */ { "ARABIC LETTER YEH BARREE","	* Urdu"},
/* 06D3 */ { "ARABIC LETTER YEH BARREE WITH HAMZA ABOVE","	* Urdu\n"
	"	* actually a ligature, not an independent letter\n"
	"	: 06D2 0654"},
/* 06D4 */ { "ARABIC FULL STOP","	* Urdu"},
/* 06D5 */ { "ARABIC LETTER AE","	* Uyghur, Kazakh, Kyrgyz"},
/* 06D6 */ { "ARABIC SMALL HIGH LIGATURE SAD WITH LAM WITH ALEF MAKSURA",NULL},
/* 06D7 */ { "ARABIC SMALL HIGH LIGATURE QAF WITH LAM WITH ALEF MAKSURA",NULL},
/* 06D8 */ { "ARABIC SMALL HIGH MEEM INITIAL FORM",NULL},
/* 06D9 */ { "ARABIC SMALL HIGH LAM ALEF",NULL},
/* 06DA */ { "ARABIC SMALL HIGH JEEM",NULL},
/* 06DB */ { "ARABIC SMALL HIGH THREE DOTS",NULL},
/* 06DC */ { "ARABIC SMALL HIGH SEEN",NULL},
/* 06DD */ { "ARABIC END OF AYAH",NULL},
/* 06DE */ { "ARABIC START OF RUB EL HIZB","	* indicates boundaries of parts of sections\n"
	"	* typically depicted as an eight-sided symbol, which may or may not appear starlike"},
/* 06DF */ { "ARABIC SMALL HIGH ROUNDED ZERO","	* smaller than the typical circular shape used for 0652"},
/* 06E0 */ { "ARABIC SMALL HIGH UPRIGHT RECTANGULAR ZERO","	* the term \"rectangular zero\" is a translation of the Arabic name of this sign"},
/* 06E1 */ { "ARABIC SMALL HIGH DOTLESS HEAD OF KHAH","	= Arabic jazm\n"
	"	* presentation form of 0652, using font technology to select the variant is preferred\n"
	"	* used in some Qurans to mark absence of a vowel\n"
	"	x (arabic sukun - 0652)"},
/* 06E2 */ { "ARABIC SMALL HIGH MEEM ISOLATED FORM",NULL},
/* 06E3 */ { "ARABIC SMALL LOW SEEN",NULL},
/* 06E4 */ { "ARABIC SMALL HIGH MADDA","	* typically used with 06E5, 06E6, 06E7, and 08F3"},
/* 06E5 */ { "ARABIC SMALL WAW","	x (arabic small low waw - 08D3)\n"
	"	x (arabic small high waw - 08F3)"},
/* 06E6 */ { "ARABIC SMALL YEH",NULL},
/* 06E7 */ { "ARABIC SMALL HIGH YEH",NULL},
/* 06E8 */ { "ARABIC SMALL HIGH NOON",NULL},
/* 06E9 */ { "ARABIC PLACE OF SAJDAH","	* there is a range of acceptable glyphs for this character"},
/* 06EA */ { "ARABIC EMPTY CENTRE LOW STOP",NULL},
/* 06EB */ { "ARABIC EMPTY CENTRE HIGH STOP",NULL},
/* 06EC */ { "ARABIC ROUNDED HIGH STOP WITH FILLED CENTRE","	* also used in Quranic text in African and other orthographies to represent wasla, ikhtilas, etc."},
/* 06ED */ { "ARABIC SMALL LOW MEEM",NULL},
/* 06EE */ { "ARABIC LETTER DAL WITH INVERTED V",NULL},
/* 06EF */ { "ARABIC LETTER REH WITH INVERTED V","	* also used in early Persian"},
/* 06F0 */ { "EXTENDED ARABIC-INDIC DIGIT ZERO",NULL},
/* 06F1 */ { "EXTENDED ARABIC-INDIC DIGIT ONE",NULL},
/* 06F2 */ { "EXTENDED ARABIC-INDIC DIGIT TWO",NULL},
/* 06F3 */ { "EXTENDED ARABIC-INDIC DIGIT THREE",NULL},
/* 06F4 */ { "EXTENDED ARABIC-INDIC DIGIT FOUR","	* Urdu and Kashmiri have a different glyph than Persian"},
/* 06F5 */ { "EXTENDED ARABIC-INDIC DIGIT FIVE",NULL},
/* 06F6 */ { "EXTENDED ARABIC-INDIC DIGIT SIX","	* Sindhi, Urdu, and Kashmiri have a different glyph than Persian"},
/* 06F7 */ { "EXTENDED ARABIC-INDIC DIGIT SEVEN","	* Sindhi, Urdu, and Kashmiri have a different glyph than Persian"},
/* 06F8 */ { "EXTENDED ARABIC-INDIC DIGIT EIGHT",NULL},
/* 06F9 */ { "EXTENDED ARABIC-INDIC DIGIT NINE",NULL},
/* 06FA */ { "ARABIC LETTER SHEEN WITH DOT BELOW",NULL},
/* 06FB */ { "ARABIC LETTER DAD WITH DOT BELOW",NULL},
/* 06FC */ { "ARABIC LETTER GHAIN WITH DOT BELOW",NULL},
/* 06FD */ { "ARABIC SIGN SINDHI AMPERSAND",NULL},
/* 06FE */ { "ARABIC SIGN SINDHI POSTPOSITION MEN",NULL},
/* 06FF */ { "ARABIC LETTER HEH WITH INVERTED V",NULL}
};

UN_DLL_LOCAL
static const struct unicode_nameannot una_00_07[] = {
/* 0700 */ { "SYRIAC END OF PARAGRAPH","	* marks the end of a paragraph"},
/* 0701 */ { "SYRIAC SUPRALINEAR FULL STOP","	* marks interrogations, imperatives, and pauses, especially in Biblical texts"},
/* 0702 */ { "SYRIAC SUBLINEAR FULL STOP","	* marks subordinate clauses and minor pauses, especially in Biblical texts"},
/* 0703 */ { "SYRIAC SUPRALINEAR COLON","	* marks expressions of wonder and has a distinct pausal value in Biblical texts"},
/* 0704 */ { "SYRIAC SUBLINEAR COLON","	* used at the end of verses of supplications"},
/* 0705 */ { "SYRIAC HORIZONTAL COLON","	* joins two words closely together in a context to which a rising tone is suitable"},
/* 0706 */ { "SYRIAC COLON SKEWED LEFT","	* marks a dependent clause"},
/* 0707 */ { "SYRIAC COLON SKEWED RIGHT","	* marks the end of a subdivision of the apodosis, or latter part of a Biblical verse"},
/* 0708 */ { "SYRIAC SUPRALINEAR COLON SKEWED LEFT","	* marks a minor phrase division"},
/* 0709 */ { "SYRIAC SUBLINEAR COLON SKEWED RIGHT","	% SYRIAC SUBLINEAR COLON SKEWED LEFT\n"
	"	* marks the end of a real or rhetorical question\n"
	"	* character name is a misnomer"},
/* 070A */ { "SYRIAC CONTRACTION","	* a contraction mark, mostly used in East Syriac\n"
	"	* placed at the end of an incomplete word"},
/* 070B */ { "SYRIAC HARKLEAN OBELUS","	* marks the beginning of a phrase, word, or morpheme that has a marginal note\n"
	"	x (division sign - 00F7)"},
/* 070C */ { "SYRIAC HARKLEAN METOBELUS","	* marks the end of a section with a marginal note"},
/* 070D */ { "SYRIAC HARKLEAN ASTERISCUS","	* marks the beginning of a phrase, word, or morpheme that has a marginal note"},
/* 070E */ { NULL,NULL},
/* 070F */ { "SYRIAC ABBREVIATION MARK","	= SAM\n"
	"	* marks the beginning of a Syriac abbreviation"},
/* 0710 */ { "SYRIAC LETTER ALAPH",NULL},
/* 0711 */ { "SYRIAC LETTER SUPERSCRIPT ALAPH","	* used in East Syriac texts to indicate an etymological Alaph"},
/* 0712 */ { "SYRIAC LETTER BETH",NULL},
/* 0713 */ { "SYRIAC LETTER GAMAL",NULL},
/* 0714 */ { "SYRIAC LETTER GAMAL GARSHUNI","	* used in Garshuni documents"},
/* 0715 */ { "SYRIAC LETTER DALATH",NULL},
/* 0716 */ { "SYRIAC LETTER DOTLESS DALATH RISH","	* ambiguous form for undifferentiated early dalath/rish"},
/* 0717 */ { "SYRIAC LETTER HE",NULL},
/* 0718 */ { "SYRIAC LETTER WAW",NULL},
/* 0719 */ { "SYRIAC LETTER ZAIN",NULL},
/* 071A */ { "SYRIAC LETTER HETH",NULL},
/* 071B */ { "SYRIAC LETTER TETH",NULL},
/* 071C */ { "SYRIAC LETTER TETH GARSHUNI","	* used in Garshuni documents"},
/* 071D */ { "SYRIAC LETTER YUDH",NULL},
/* 071E */ { "SYRIAC LETTER YUDH HE","	* mostly used in East Syriac texts"},
/* 071F */ { "SYRIAC LETTER KAPH",NULL},
/* 0720 */ { "SYRIAC LETTER LAMADH",NULL},
/* 0721 */ { "SYRIAC LETTER MIM",NULL},
/* 0722 */ { "SYRIAC LETTER NUN",NULL},
/* 0723 */ { "SYRIAC LETTER SEMKATH",NULL},
/* 0724 */ { "SYRIAC LETTER FINAL SEMKATH",NULL},
/* 0725 */ { "SYRIAC LETTER E",NULL},
/* 0726 */ { "SYRIAC LETTER PE",NULL},
/* 0727 */ { "SYRIAC LETTER REVERSED PE","	* used in Christian Palestinian Aramaic"},
/* 0728 */ { "SYRIAC LETTER SADHE",NULL},
/* 0729 */ { "SYRIAC LETTER QAPH",NULL},
/* 072A */ { "SYRIAC LETTER RISH",NULL},
/* 072B */ { "SYRIAC LETTER SHIN",NULL},
/* 072C */ { "SYRIAC LETTER TAW",NULL},
/* 072D */ { "SYRIAC LETTER PERSIAN BHETH",NULL},
/* 072E */ { "SYRIAC LETTER PERSIAN GHAMAL",NULL},
/* 072F */ { "SYRIAC LETTER PERSIAN DHALATH",NULL},
/* 0730 */ { "SYRIAC PTHAHA ABOVE",NULL},
/* 0731 */ { "SYRIAC PTHAHA BELOW",NULL},
/* 0732 */ { "SYRIAC PTHAHA DOTTED",NULL},
/* 0733 */ { "SYRIAC ZQAPHA ABOVE",NULL},
/* 0734 */ { "SYRIAC ZQAPHA BELOW",NULL},
/* 0735 */ { "SYRIAC ZQAPHA DOTTED",NULL},
/* 0736 */ { "SYRIAC RBASA ABOVE",NULL},
/* 0737 */ { "SYRIAC RBASA BELOW",NULL},
/* 0738 */ { "SYRIAC DOTTED ZLAMA HORIZONTAL",NULL},
/* 0739 */ { "SYRIAC DOTTED ZLAMA ANGULAR",NULL},
/* 073A */ { "SYRIAC HBASA ABOVE",NULL},
/* 073B */ { "SYRIAC HBASA BELOW",NULL},
/* 073C */ { "SYRIAC HBASA-ESASA DOTTED",NULL},
/* 073D */ { "SYRIAC ESASA ABOVE",NULL},
/* 073E */ { "SYRIAC ESASA BELOW",NULL},
/* 073F */ { "SYRIAC RWAHA",NULL},
/* 0740 */ { "SYRIAC FEMININE DOT","	* feminine marker used with the Taw feminine suffix\n"
	"	x (combining dot above left - 1DF8)"},
/* 0741 */ { "SYRIAC QUSHSHAYA","	* indicates a plosive pronunciation"},
/* 0742 */ { "SYRIAC RUKKAKHA","	* indicates an aspirated (spirantized) pronunciation"},
/* 0743 */ { "SYRIAC TWO VERTICAL DOTS ABOVE","	* accent mark used in ancient manuscripts"},
/* 0744 */ { "SYRIAC TWO VERTICAL DOTS BELOW","	* accent mark used in ancient manuscripts"},
/* 0745 */ { "SYRIAC THREE DOTS ABOVE","	* diacritic used in Turoyo for letters not found in Syriac"},
/* 0746 */ { "SYRIAC THREE DOTS BELOW","	* diacritic used in Turoyo for letters not found in Syriac"},
/* 0747 */ { "SYRIAC OBLIQUE LINE ABOVE","	* indication of a silent letter"},
/* 0748 */ { "SYRIAC OBLIQUE LINE BELOW","	* indication of a silent letter\n"
	"	* also used to indicate numbers multiplied by a certain constant"},
/* 0749 */ { "SYRIAC MUSIC","	* a music mark\n"
	"	* also used in the Syrian Orthodox Anaphora book to mark the breaking of the Eucharist bread"},
/* 074A */ { "SYRIAC BARREKH","	* a diacritic cross used in liturgical texts"},
/* 074B */ { NULL,NULL},
/* 074C */ { NULL,NULL},
/* 074D */ { "SYRIAC LETTER SOGDIAN ZHAIN",NULL},
/* 074E */ { "SYRIAC LETTER SOGDIAN KHAPH",NULL},
/* 074F */ { "SYRIAC LETTER SOGDIAN FE",NULL},
/* 0750 */ { "ARABIC LETTER BEH WITH THREE DOTS HORIZONTALLY BELOW",NULL},
/* 0751 */ { "ARABIC LETTER BEH WITH DOT BELOW AND THREE DOTS ABOVE",NULL},
/* 0752 */ { "ARABIC LETTER BEH WITH THREE DOTS POINTING UPWARDS BELOW",NULL},
/* 0753 */ { "ARABIC LETTER BEH WITH THREE DOTS POINTING UPWARDS BELOW AND TWO DOTS ABOVE",NULL},
/* 0754 */ { "ARABIC LETTER BEH WITH TWO DOTS BELOW AND DOT ABOVE",NULL},
/* 0755 */ { "ARABIC LETTER BEH WITH INVERTED SMALL V BELOW",NULL},
/* 0756 */ { "ARABIC LETTER BEH WITH SMALL V",NULL},
/* 0757 */ { "ARABIC LETTER HAH WITH TWO DOTS ABOVE",NULL},
/* 0758 */ { "ARABIC LETTER HAH WITH THREE DOTS POINTING UPWARDS BELOW",NULL},
/* 0759 */ { "ARABIC LETTER DAL WITH TWO DOTS VERTICALLY BELOW AND SMALL TAH","	* Saraiki"},
/* 075A */ { "ARABIC LETTER DAL WITH INVERTED SMALL V BELOW",NULL},
/* 075B */ { "ARABIC LETTER REH WITH STROKE",NULL},
/* 075C */ { "ARABIC LETTER SEEN WITH FOUR DOTS ABOVE","	* Shina"},
/* 075D */ { "ARABIC LETTER AIN WITH TWO DOTS ABOVE",NULL},
/* 075E */ { "ARABIC LETTER AIN WITH THREE DOTS POINTING DOWNWARDS ABOVE",NULL},
/* 075F */ { "ARABIC LETTER AIN WITH TWO DOTS VERTICALLY ABOVE",NULL},
/* 0760 */ { "ARABIC LETTER FEH WITH TWO DOTS BELOW",NULL},
/* 0761 */ { "ARABIC LETTER FEH WITH THREE DOTS POINTING UPWARDS BELOW",NULL},
/* 0762 */ { "ARABIC LETTER KEHEH WITH DOT ABOVE","	* Jawi, preferred to 06AC\n"
	"	x (arabic letter kaf with dot above - 06AC)"},
/* 0763 */ { "ARABIC LETTER KEHEH WITH THREE DOTS ABOVE","	* Moroccan Arabic, Amazigh, Burushaski\n"
	"	x (arabic letter ng - 06AD)"},
/* 0764 */ { "ARABIC LETTER KEHEH WITH THREE DOTS POINTING UPWARDS BELOW",NULL},
/* 0765 */ { "ARABIC LETTER MEEM WITH DOT ABOVE",NULL},
/* 0766 */ { "ARABIC LETTER MEEM WITH DOT BELOW","	* Maba"},
/* 0767 */ { "ARABIC LETTER NOON WITH TWO DOTS BELOW","	* Arwi"},
/* 0768 */ { "ARABIC LETTER NOON WITH SMALL TAH","	* Saraiki, Pathwari"},
/* 0769 */ { "ARABIC LETTER NOON WITH SMALL V","	* Gojri"},
/* 076A */ { "ARABIC LETTER LAM WITH BAR",NULL},
/* 076B */ { "ARABIC LETTER REH WITH TWO DOTS VERTICALLY ABOVE","	* Torwali, Ormuri"},
/* 076C */ { "ARABIC LETTER REH WITH HAMZA ABOVE","	* Ormuri\n"
	"	* represents a voiced alveolo-palatal laminal fricative\n"
	"	x (latin small letter z with curl - 0291)"},
/* 076D */ { "ARABIC LETTER SEEN WITH TWO DOTS VERTICALLY ABOVE","	* Kalami, Ormuri"},
/* 076E */ { "ARABIC LETTER HAH WITH SMALL ARABIC LETTER TAH BELOW",NULL},
/* 076F */ { "ARABIC LETTER HAH WITH SMALL ARABIC LETTER TAH AND TWO DOTS",NULL},
/* 0770 */ { "ARABIC LETTER SEEN WITH SMALL ARABIC LETTER TAH AND TWO DOTS",NULL},
/* 0771 */ { "ARABIC LETTER REH WITH SMALL ARABIC LETTER TAH AND TWO DOTS",NULL},
/* 0772 */ { "ARABIC LETTER HAH WITH SMALL ARABIC LETTER TAH ABOVE",NULL},
/* 0773 */ { "ARABIC LETTER ALEF WITH EXTENDED ARABIC-INDIC DIGIT TWO ABOVE",NULL},
/* 0774 */ { "ARABIC LETTER ALEF WITH EXTENDED ARABIC-INDIC DIGIT THREE ABOVE",NULL},
/* 0775 */ { "ARABIC LETTER FARSI YEH WITH EXTENDED ARABIC-INDIC DIGIT TWO ABOVE",NULL},
/* 0776 */ { "ARABIC LETTER FARSI YEH WITH EXTENDED ARABIC-INDIC DIGIT THREE ABOVE",NULL},
/* 0777 */ { "ARABIC LETTER FARSI YEH WITH EXTENDED ARABIC-INDIC DIGIT FOUR BELOW",NULL},
/* 0778 */ { "ARABIC LETTER WAW WITH EXTENDED ARABIC-INDIC DIGIT TWO ABOVE",NULL},
/* 0779 */ { "ARABIC LETTER WAW WITH EXTENDED ARABIC-INDIC DIGIT THREE ABOVE",NULL},
/* 077A */ { "ARABIC LETTER YEH BARREE WITH EXTENDED ARABIC-INDIC DIGIT TWO ABOVE",NULL},
/* 077B */ { "ARABIC LETTER YEH BARREE WITH EXTENDED ARABIC-INDIC DIGIT THREE ABOVE",NULL},
/* 077C */ { "ARABIC LETTER HAH WITH EXTENDED ARABIC-INDIC DIGIT FOUR BELOW",NULL},
/* 077D */ { "ARABIC LETTER SEEN WITH EXTENDED ARABIC-INDIC DIGIT FOUR ABOVE",NULL},
/* 077E */ { "ARABIC LETTER SEEN WITH INVERTED V",NULL},
/* 077F */ { "ARABIC LETTER KAF WITH TWO DOTS ABOVE",NULL},
/* 0780 */ { "THAANA LETTER HAA",NULL},
/* 0781 */ { "THAANA LETTER SHAVIYANI",NULL},
/* 0782 */ { "THAANA LETTER NOONU",NULL},
/* 0783 */ { "THAANA LETTER RAA",NULL},
/* 0784 */ { "THAANA LETTER BAA",NULL},
/* 0785 */ { "THAANA LETTER LHAVIYANI",NULL},
/* 0786 */ { "THAANA LETTER KAAFU",NULL},
/* 0787 */ { "THAANA LETTER ALIFU",NULL},
/* 0788 */ { "THAANA LETTER VAAVU",NULL},
/* 0789 */ { "THAANA LETTER MEEMU",NULL},
/* 078A */ { "THAANA LETTER FAAFU",NULL},
/* 078B */ { "THAANA LETTER DHAALU",NULL},
/* 078C */ { "THAANA LETTER THAA",NULL},
/* 078D */ { "THAANA LETTER LAAMU",NULL},
/* 078E */ { "THAANA LETTER GAAFU",NULL},
/* 078F */ { "THAANA LETTER GNAVIYANI",NULL},
/* 0790 */ { "THAANA LETTER SEENU",NULL},
/* 0791 */ { "THAANA LETTER DAVIYANI",NULL},
/* 0792 */ { "THAANA LETTER ZAVIYANI",NULL},
/* 0793 */ { "THAANA LETTER TAVIYANI",NULL},
/* 0794 */ { "THAANA LETTER YAA",NULL},
/* 0795 */ { "THAANA LETTER PAVIYANI",NULL},
/* 0796 */ { "THAANA LETTER JAVIYANI",NULL},
/* 0797 */ { "THAANA LETTER CHAVIYANI",NULL},
/* 0798 */ { "THAANA LETTER TTAA",NULL},
/* 0799 */ { "THAANA LETTER HHAA",NULL},
/* 079A */ { "THAANA LETTER KHAA",NULL},
/* 079B */ { "THAANA LETTER THAALU",NULL},
/* 079C */ { "THAANA LETTER ZAA",NULL},
/* 079D */ { "THAANA LETTER SHEENU",NULL},
/* 079E */ { "THAANA LETTER SAADHU",NULL},
/* 079F */ { "THAANA LETTER DAADHU",NULL},
/* 07A0 */ { "THAANA LETTER TO",NULL},
/* 07A1 */ { "THAANA LETTER ZO",NULL},
/* 07A2 */ { "THAANA LETTER AINU",NULL},
/* 07A3 */ { "THAANA LETTER GHAINU",NULL},
/* 07A4 */ { "THAANA LETTER QAAFU",NULL},
/* 07A5 */ { "THAANA LETTER WAAVU",NULL},
/* 07A6 */ { "THAANA ABAFILI",NULL},
/* 07A7 */ { "THAANA AABAAFILI",NULL},
/* 07A8 */ { "THAANA IBIFILI",NULL},
/* 07A9 */ { "THAANA EEBEEFILI",NULL},
/* 07AA */ { "THAANA UBUFILI",NULL},
/* 07AB */ { "THAANA OOBOOFILI",NULL},
/* 07AC */ { "THAANA EBEFILI",NULL},
/* 07AD */ { "THAANA EYBEYFILI",NULL},
/* 07AE */ { "THAANA OBOFILI",NULL},
/* 07AF */ { "THAANA OABOAFILI",NULL},
/* 07B0 */ { "THAANA SUKUN",NULL},
/* 07B1 */ { "THAANA LETTER NAA",NULL},
/* 07B2 */ { NULL,NULL},
/* 07B3 */ { NULL,NULL},
/* 07B4 */ { NULL,NULL},
/* 07B5 */ { NULL,NULL},
/* 07B6 */ { NULL,NULL},
/* 07B7 */ { NULL,NULL},
/* 07B8 */ { NULL,NULL},
/* 07B9 */ { NULL,NULL},
/* 07BA */ { NULL,NULL},
/* 07BB */ { NULL,NULL},
/* 07BC */ { NULL,NULL},
/* 07BD */ { NULL,NULL},
/* 07BE */ { NULL,NULL},
/* 07BF */ { NULL,NULL},
/* 07C0 */ { "NKO DIGIT ZERO",NULL},
/* 07C1 */ { "NKO DIGIT ONE",NULL},
/* 07C2 */ { "NKO DIGIT TWO",NULL},
/* 07C3 */ { "NKO DIGIT THREE",NULL},
/* 07C4 */ { "NKO DIGIT FOUR",NULL},
/* 07C5 */ { "NKO DIGIT FIVE",NULL},
/* 07C6 */ { "NKO DIGIT SIX",NULL},
/* 07C7 */ { "NKO DIGIT SEVEN",NULL},
/* 07C8 */ { "NKO DIGIT EIGHT",NULL},
/* 07C9 */ { "NKO DIGIT NINE",NULL},
/* 07CA */ { "NKO LETTER A",NULL},
/* 07CB */ { "NKO LETTER EE",NULL},
/* 07CC */ { "NKO LETTER I",NULL},
/* 07CD */ { "NKO LETTER E",NULL},
/* 07CE */ { "NKO LETTER U",NULL},
/* 07CF */ { "NKO LETTER OO",NULL},
/* 07D0 */ { "NKO LETTER O",NULL},
/* 07D1 */ { "NKO LETTER DAGBASINNA",NULL},
/* 07D2 */ { "NKO LETTER N",NULL},
/* 07D3 */ { "NKO LETTER BA",NULL},
/* 07D4 */ { "NKO LETTER PA",NULL},
/* 07D5 */ { "NKO LETTER TA",NULL},
/* 07D6 */ { "NKO LETTER JA",NULL},
/* 07D7 */ { "NKO LETTER CHA",NULL},
/* 07D8 */ { "NKO LETTER DA",NULL},
/* 07D9 */ { "NKO LETTER RA",NULL},
/* 07DA */ { "NKO LETTER RRA",NULL},
/* 07DB */ { "NKO LETTER SA",NULL},
/* 07DC */ { "NKO LETTER GBA",NULL},
/* 07DD */ { "NKO LETTER FA",NULL},
/* 07DE */ { "NKO LETTER KA",NULL},
/* 07DF */ { "NKO LETTER LA",NULL},
/* 07E0 */ { "NKO LETTER NA WOLOSO",NULL},
/* 07E1 */ { "NKO LETTER MA",NULL},
/* 07E2 */ { "NKO LETTER NYA",NULL},
/* 07E3 */ { "NKO LETTER NA",NULL},
/* 07E4 */ { "NKO LETTER HA",NULL},
/* 07E5 */ { "NKO LETTER WA",NULL},
/* 07E6 */ { "NKO LETTER YA",NULL},
/* 07E7 */ { "NKO LETTER NYA WOLOSO",NULL},
/* 07E8 */ { "NKO LETTER JONA JA",NULL},
/* 07E9 */ { "NKO LETTER JONA CHA",NULL},
/* 07EA */ { "NKO LETTER JONA RA","	x (nko letter ra - 07D9)"},
/* 07EB */ { "NKO COMBINING SHORT HIGH TONE","	x (combining macron - 0304)"},
/* 07EC */ { "NKO COMBINING SHORT LOW TONE","	x (combining tilde - 0303)"},
/* 07ED */ { "NKO COMBINING SHORT RISING TONE","	x (combining dot above - 0307)"},
/* 07EE */ { "NKO COMBINING LONG DESCENDING TONE","	x (combining circumflex accent - 0302)"},
/* 07EF */ { "NKO COMBINING LONG HIGH TONE",NULL},
/* 07F0 */ { "NKO COMBINING LONG LOW TONE",NULL},
/* 07F1 */ { "NKO COMBINING LONG RISING TONE",NULL},
/* 07F2 */ { "NKO COMBINING NASALIZATION MARK","	x (combining dot below - 0323)"},
/* 07F3 */ { "NKO COMBINING DOUBLE DOT ABOVE","	x (combining diaeresis - 0308)"},
/* 07F4 */ { "NKO HIGH TONE APOSTROPHE","	x (modifier letter apostrophe - 02BC)"},
/* 07F5 */ { "NKO LOW TONE APOSTROPHE","	x (modifier letter turned comma - 02BB)"},
/* 07F6 */ { "NKO SYMBOL OO DENNEN","	* indicates remote future placement of the topic under discussion"},
/* 07F7 */ { "NKO SYMBOL GBAKURUNEN","	* marks end of a major section of text\n"
	"	* symbol represents the three stones holding a cooking pot over a fire"},
/* 07F8 */ { "NKO COMMA",NULL},
/* 07F9 */ { "NKO EXCLAMATION MARK",NULL},
/* 07FA */ { "NKO LAJANYALAN","	x (low line - 005F)\n"
	"	x (arabic tatweel - 0640)"},
/* 07FB */ { NULL,NULL},
/* 07FC */ { NULL,NULL},
/* 07FD */ { "NKO DANTAYALAN","	* used to abbreviate units of measure\n"
	"	x (combining bridge below - 032A)"},
/* 07FE */ { "NKO DOROME SIGN","	x (nko letter da - 07D8)"},
/* 07FF */ { "NKO TAMAN SIGN","	x (nko letter ta - 07D5)"}
};

UN_DLL_LOCAL
static const struct unicode_nameannot una_00_08[] = {
/* 0800 */ { "SAMARITAN LETTER ALAF",NULL},
/* 0801 */ { "SAMARITAN LETTER BIT",NULL},
/* 0802 */ { "SAMARITAN LETTER GAMAN",NULL},
/* 0803 */ { "SAMARITAN LETTER DALAT",NULL},
/* 0804 */ { "SAMARITAN LETTER IY",NULL},
/* 0805 */ { "SAMARITAN LETTER BAA",NULL},
/* 0806 */ { "SAMARITAN LETTER ZEN",NULL},
/* 0807 */ { "SAMARITAN LETTER IT",NULL},
/* 0808 */ { "SAMARITAN LETTER TIT",NULL},
/* 0809 */ { "SAMARITAN LETTER YUT",NULL},
/* 080A */ { "SAMARITAN LETTER KAAF",NULL},
/* 080B */ { "SAMARITAN LETTER LABAT",NULL},
/* 080C */ { "SAMARITAN LETTER MIM",NULL},
/* 080D */ { "SAMARITAN LETTER NUN",NULL},
/* 080E */ { "SAMARITAN LETTER SINGAAT",NULL},
/* 080F */ { "SAMARITAN LETTER IN",NULL},
/* 0810 */ { "SAMARITAN LETTER FI",NULL},
/* 0811 */ { "SAMARITAN LETTER TSAADIY",NULL},
/* 0812 */ { "SAMARITAN LETTER QUF",NULL},
/* 0813 */ { "SAMARITAN LETTER RISH",NULL},
/* 0814 */ { "SAMARITAN LETTER SHAN",NULL},
/* 0815 */ { "SAMARITAN LETTER TAAF",NULL},
/* 0816 */ { "SAMARITAN MARK IN",NULL},
/* 0817 */ { "SAMARITAN MARK IN-ALAF",NULL},
/* 0818 */ { "SAMARITAN MARK OCCLUSION",NULL},
/* 0819 */ { "SAMARITAN MARK DAGESH",NULL},
/* 081A */ { "SAMARITAN MODIFIER LETTER EPENTHETIC YUT",NULL},
/* 081B */ { "SAMARITAN MARK EPENTHETIC YUT",NULL},
/* 081C */ { "SAMARITAN VOWEL SIGN LONG E","	= fatha al-nida"},
/* 081D */ { "SAMARITAN VOWEL SIGN E",NULL},
/* 081E */ { "SAMARITAN VOWEL SIGN OVERLONG AA","	= fatha al-ima"},
/* 081F */ { "SAMARITAN VOWEL SIGN LONG AA",NULL},
/* 0820 */ { "SAMARITAN VOWEL SIGN AA",NULL},
/* 0821 */ { "SAMARITAN VOWEL SIGN OVERLONG A","	= fatha al-iha"},
/* 0822 */ { "SAMARITAN VOWEL SIGN LONG A",NULL},
/* 0823 */ { "SAMARITAN VOWEL SIGN A",NULL},
/* 0824 */ { "SAMARITAN MODIFIER LETTER SHORT A",NULL},
/* 0825 */ { "SAMARITAN VOWEL SIGN SHORT A","	= fatha"},
/* 0826 */ { "SAMARITAN VOWEL SIGN LONG U","	= damma"},
/* 0827 */ { "SAMARITAN VOWEL SIGN U",NULL},
/* 0828 */ { "SAMARITAN MODIFIER LETTER I",NULL},
/* 0829 */ { "SAMARITAN VOWEL SIGN LONG I","	= kasra"},
/* 082A */ { "SAMARITAN VOWEL SIGN I",NULL},
/* 082B */ { "SAMARITAN VOWEL SIGN O",NULL},
/* 082C */ { "SAMARITAN VOWEL SIGN SUKUN",NULL},
/* 082D */ { "SAMARITAN MARK NEQUDAA",NULL},
/* 082E */ { NULL,NULL},
/* 082F */ { NULL,NULL},
/* 0830 */ { "SAMARITAN PUNCTUATION NEQUDAA","	= word separator"},
/* 0831 */ { "SAMARITAN PUNCTUATION AFSAAQ","	= interruption"},
/* 0832 */ { "SAMARITAN PUNCTUATION ANGED","	= restraint"},
/* 0833 */ { "SAMARITAN PUNCTUATION BAU","	= prayer"},
/* 0834 */ { "SAMARITAN PUNCTUATION ATMAAU","	= surprise"},
/* 0835 */ { "SAMARITAN PUNCTUATION SHIYYAALAA","	= question"},
/* 0836 */ { "SAMARITAN ABBREVIATION MARK",NULL},
/* 0837 */ { "SAMARITAN PUNCTUATION MELODIC QITSA",NULL},
/* 0838 */ { "SAMARITAN PUNCTUATION ZIQAA","	= shouting"},
/* 0839 */ { "SAMARITAN PUNCTUATION QITSA","	= end of section"},
/* 083A */ { "SAMARITAN PUNCTUATION ZAEF","	= outburst"},
/* 083B */ { "SAMARITAN PUNCTUATION TURU","	= teaching"},
/* 083C */ { "SAMARITAN PUNCTUATION ARKAANU","	= submissiveness"},
/* 083D */ { "SAMARITAN PUNCTUATION SOF MASHFAAT","	= full stop"},
/* 083E */ { "SAMARITAN PUNCTUATION ANNAAU","	= rest"},
/* 083F */ { NULL,NULL},
/* 0840 */ { "MANDAIC LETTER HALQA","	= a"},
/* 0841 */ { "MANDAIC LETTER AB",NULL},
/* 0842 */ { "MANDAIC LETTER AG",NULL},
/* 0843 */ { "MANDAIC LETTER AD",NULL},
/* 0844 */ { "MANDAIC LETTER AH",NULL},
/* 0845 */ { "MANDAIC LETTER USHENNA","	= u"},
/* 0846 */ { "MANDAIC LETTER AZ",NULL},
/* 0847 */ { "MANDAIC LETTER IT","	= pharyngeal hu"},
/* 0848 */ { "MANDAIC LETTER ATT",NULL},
/* 0849 */ { "MANDAIC LETTER AKSA","	= i"},
/* 084A */ { "MANDAIC LETTER AK",NULL},
/* 084B */ { "MANDAIC LETTER AL",NULL},
/* 084C */ { "MANDAIC LETTER AM",NULL},
/* 084D */ { "MANDAIC LETTER AN",NULL},
/* 084E */ { "MANDAIC LETTER AS",NULL},
/* 084F */ { "MANDAIC LETTER IN",NULL},
/* 0850 */ { "MANDAIC LETTER AP",NULL},
/* 0851 */ { "MANDAIC LETTER ASZ",NULL},
/* 0852 */ { "MANDAIC LETTER AQ",NULL},
/* 0853 */ { "MANDAIC LETTER AR",NULL},
/* 0854 */ { "MANDAIC LETTER ASH",NULL},
/* 0855 */ { "MANDAIC LETTER AT",NULL},
/* 0856 */ { "MANDAIC LETTER DUSHENNA","	= di"},
/* 0857 */ { "MANDAIC LETTER KAD",NULL},
/* 0858 */ { "MANDAIC LETTER AIN",NULL},
/* 0859 */ { "MANDAIC AFFRICATION MARK",NULL},
/* 085A */ { "MANDAIC VOCALIZATION MARK",NULL},
/* 085B */ { "MANDAIC GEMINATION MARK",NULL},
/* 085C */ { NULL,NULL},
/* 085D */ { NULL,NULL},
/* 085E */ { "MANDAIC PUNCTUATION",NULL},
/* 085F */ { NULL,NULL},
/* 0860 */ { "SYRIAC LETTER MALAYALAM NGA","	x (malayalam letter nga - 0D19)"},
/* 0861 */ { "SYRIAC LETTER MALAYALAM JA","	x (malayalam letter ja - 0D1C)"},
/* 0862 */ { "SYRIAC LETTER MALAYALAM NYA","	x (malayalam letter nya - 0D1E)"},
/* 0863 */ { "SYRIAC LETTER MALAYALAM TTA","	x (malayalam letter tta - 0D1F)"},
/* 0864 */ { "SYRIAC LETTER MALAYALAM NNA","	x (malayalam letter nna - 0D23)"},
/* 0865 */ { "SYRIAC LETTER MALAYALAM NNNA","	x (malayalam letter nnna - 0D29)"},
/* 0866 */ { "SYRIAC LETTER MALAYALAM BHA","	x (malayalam letter bha - 0D2D)"},
/* 0867 */ { "SYRIAC LETTER MALAYALAM RA","	x (malayalam letter ra - 0D30)"},
/* 0868 */ { "SYRIAC LETTER MALAYALAM LLA","	x (malayalam letter lla - 0D33)"},
/* 0869 */ { "SYRIAC LETTER MALAYALAM LLLA","	x (malayalam letter llla - 0D34)"},
/* 086A */ { "SYRIAC LETTER MALAYALAM SSA","	x (malayalam letter ssa - 0D37)"},
/* 086B */ { NULL,NULL},
/* 086C */ { NULL,NULL},
/* 086D */ { NULL,NULL},
/* 086E */ { NULL,NULL},
/* 086F */ { NULL,NULL},
/* 0870 */ { "ARABIC LETTER ALEF WITH ATTACHED FATHA",NULL},
/* 0871 */ { "ARABIC LETTER ALEF WITH ATTACHED TOP RIGHT FATHA",NULL},
/* 0872 */ { "ARABIC LETTER ALEF WITH RIGHT MIDDLE STROKE",NULL},
/* 0873 */ { "ARABIC LETTER ALEF WITH LEFT MIDDLE STROKE",NULL},
/* 0874 */ { "ARABIC LETTER ALEF WITH ATTACHED KASRA",NULL},
/* 0875 */ { "ARABIC LETTER ALEF WITH ATTACHED BOTTOM RIGHT KASRA",NULL},
/* 0876 */ { "ARABIC LETTER ALEF WITH ATTACHED ROUND DOT ABOVE",NULL},
/* 0877 */ { "ARABIC LETTER ALEF WITH ATTACHED RIGHT ROUND DOT",NULL},
/* 0878 */ { "ARABIC LETTER ALEF WITH ATTACHED LEFT ROUND DOT",NULL},
/* 0879 */ { "ARABIC LETTER ALEF WITH ATTACHED ROUND DOT BELOW",NULL},
/* 087A */ { "ARABIC LETTER ALEF WITH DOT ABOVE",NULL},
/* 087B */ { "ARABIC LETTER ALEF WITH ATTACHED TOP RIGHT FATHA AND DOT ABOVE",NULL},
/* 087C */ { "ARABIC LETTER ALEF WITH RIGHT MIDDLE STROKE AND DOT ABOVE",NULL},
/* 087D */ { "ARABIC LETTER ALEF WITH ATTACHED BOTTOM RIGHT KASRA AND DOT ABOVE",NULL},
/* 087E */ { "ARABIC LETTER ALEF WITH ATTACHED TOP RIGHT FATHA AND LEFT RING",NULL},
/* 087F */ { "ARABIC LETTER ALEF WITH RIGHT MIDDLE STROKE AND LEFT RING",NULL},
/* 0880 */ { "ARABIC LETTER ALEF WITH ATTACHED BOTTOM RIGHT KASRA AND LEFT RING",NULL},
/* 0881 */ { "ARABIC LETTER ALEF WITH ATTACHED RIGHT HAMZA",NULL},
/* 0882 */ { "ARABIC LETTER ALEF WITH ATTACHED LEFT HAMZA",NULL},
/* 0883 */ { "ARABIC TATWEEL WITH OVERSTRUCK HAMZA",NULL},
/* 0884 */ { "ARABIC TATWEEL WITH OVERSTRUCK WAW",NULL},
/* 0885 */ { "ARABIC TATWEEL WITH TWO DOTS BELOW",NULL},
/* 0886 */ { "ARABIC LETTER THIN YEH","	* no final or isolated forms are attested"},
/* 0887 */ { "ARABIC BASELINE ROUND DOT",NULL},
/* 0888 */ { "ARABIC RAISED ROUND DOT",NULL},
/* 0889 */ { "ARABIC LETTER NOON WITH INVERTED SMALL V",NULL},
/* 088A */ { "ARABIC LETTER HAH WITH INVERTED SMALL V BELOW",NULL},
/* 088B */ { "ARABIC LETTER TAH WITH DOT BELOW","	* Pegon, Malagasy"},
/* 088C */ { "ARABIC LETTER TAH WITH THREE DOTS BELOW","	* Pegon alternative for 088B"},
/* 088D */ { "ARABIC LETTER KEHEH WITH TWO DOTS VERTICALLY BELOW","	* Sundanese Pegon"},
/* 088E */ { "ARABIC VERTICAL TAIL","	* mark used to indicate abbreviations in early movable type texts from Iran \n"
	"	* only attested in final form"},
/* 088F */ { NULL,NULL},
/* 0890 */ { "ARABIC POUND MARK ABOVE","	* Egyptian pound"},
/* 0891 */ { "ARABIC PIASTRE MARK ABOVE","	* Egyptian piastre"},
/* 0892 */ { NULL,NULL},
/* 0893 */ { NULL,NULL},
/* 0894 */ { NULL,NULL},
/* 0895 */ { NULL,NULL},
/* 0896 */ { NULL,NULL},
/* 0897 */ { "ARABIC PEPET","	x (arabic maddah above - 0653)"},
/* 0898 */ { "ARABIC SMALL HIGH WORD AL-JUZ",NULL},
/* 0899 */ { "ARABIC SMALL LOW WORD ISHMAAM",NULL},
/* 089A */ { "ARABIC SMALL LOW WORD IMAALA",NULL},
/* 089B */ { "ARABIC SMALL LOW WORD TASHEEL",NULL},
/* 089C */ { "ARABIC MADDA WAAJIB","	x (arabic maddah above - 0653)"},
/* 089D */ { "ARABIC SUPERSCRIPT ALEF MOKHASSAS",NULL},
/* 089E */ { "ARABIC DOUBLED MADDA",NULL},
/* 089F */ { "ARABIC HALF MADDA OVER MADDA",NULL},
/* 08A0 */ { "ARABIC LETTER BEH WITH SMALL V BELOW",NULL},
/* 08A1 */ { "ARABIC LETTER BEH WITH HAMZA ABOVE","	* Adamawa Fulfulde (Cameroon)\n"
	"	* used for the implosive bilabial stop\n"
	"	x (latin small letter b with hook - 0253)"},
/* 08A2 */ { "ARABIC LETTER JEEM WITH TWO DOTS ABOVE",NULL},
/* 08A3 */ { "ARABIC LETTER TAH WITH TWO DOTS ABOVE",NULL},
/* 08A4 */ { "ARABIC LETTER FEH WITH DOT BELOW AND THREE DOTS ABOVE",NULL},
/* 08A5 */ { "ARABIC LETTER QAF WITH DOT BELOW",NULL},
/* 08A6 */ { "ARABIC LETTER LAM WITH DOUBLE BAR",NULL},
/* 08A7 */ { "ARABIC LETTER MEEM WITH THREE DOTS ABOVE",NULL},
/* 08A8 */ { "ARABIC LETTER YEH WITH TWO DOTS BELOW AND HAMZA ABOVE","	* Adamawa Fulfulde\n"
	"	* used for the implosive palatal approximant, realized as pharyngealization of the approximant\n"
	"	x (latin small letter y with hook - 01B4)"},
/* 08A9 */ { "ARABIC LETTER YEH WITH TWO DOTS BELOW AND DOT ABOVE","	* Adamawa Fulfulde\n"
	"	* used for the voiced palatal nasal\n"
	"	x (latin small letter n with left hook - 0272)"},
/* 08AA */ { "ARABIC LETTER REH WITH LOOP","	= bottya-reh"},
/* 08AB */ { "ARABIC LETTER WAW WITH DOT WITHIN","	= nota-wa"},
/* 08AC */ { "ARABIC LETTER ROHINGYA YEH","	= bottya-yeh"},
/* 08AD */ { "ARABIC LETTER LOW ALEF","	* Bashkir, Tatar"},
/* 08AE */ { "ARABIC LETTER DAL WITH THREE DOTS BELOW","	* Belarusian\n"
	"	* Pegon alternative for 068A"},
/* 08AF */ { "ARABIC LETTER SAD WITH THREE DOTS BELOW","	* Belarusian"},
/* 08B0 */ { "ARABIC LETTER GAF WITH INVERTED STROKE","	* Crimean Tatar, Chechen, Lak"},
/* 08B1 */ { "ARABIC LETTER STRAIGHT WAW","	* Tatar"},
/* 08B2 */ { "ARABIC LETTER ZAIN WITH INVERTED V ABOVE",NULL},
/* 08B3 */ { "ARABIC LETTER AIN WITH THREE DOTS BELOW",NULL},
/* 08B4 */ { "ARABIC LETTER KAF WITH DOT BELOW","	* also used in Pegon\n"
	"	x (arabic letter kaf with three dots below - 06AE)"},
/* 08B5 */ { "ARABIC LETTER QAF WITH DOT BELOW AND NO DOTS ABOVE",NULL},
/* 08B6 */ { "ARABIC LETTER BEH WITH SMALL MEEM ABOVE",NULL},
/* 08B7 */ { "ARABIC LETTER PEH WITH SMALL MEEM ABOVE",NULL},
/* 08B8 */ { "ARABIC LETTER TEH WITH SMALL TEH ABOVE",NULL},
/* 08B9 */ { "ARABIC LETTER REH WITH SMALL NOON ABOVE",NULL},
/* 08BA */ { "ARABIC LETTER YEH WITH TWO DOTS BELOW AND SMALL NOON ABOVE",NULL},
/* 08BB */ { "ARABIC LETTER AFRICAN FEH","	* initial and medial forms have one dot below\n"
	"	x (arabic letter dotless feh - 06A1)\n"
	"	x (arabic letter feh with dot moved below - 06A2)"},
/* 08BC */ { "ARABIC LETTER AFRICAN QAF","	* initial and medial forms have one dot above\n"
	"	x (arabic letter dotless qaf - 066F)\n"
	"	x (arabic letter qaf with dot above - 06A7)\n"
	"	x (arabic letter african qaf with three dots above - 08C4)"},
/* 08BD */ { "ARABIC LETTER AFRICAN NOON","	* initial and medial forms have one dot above\n"
	"	x (arabic letter noon ghunna - 06BA)\n"
	"	x (arabic letter noon - 0646)"},
/* 08BE */ { "ARABIC LETTER PEH WITH SMALL V",NULL},
/* 08BF */ { "ARABIC LETTER TEH WITH SMALL V",NULL},
/* 08C0 */ { "ARABIC LETTER TTEH WITH SMALL V",NULL},
/* 08C1 */ { "ARABIC LETTER TCHEH WITH SMALL V",NULL},
/* 08C2 */ { "ARABIC LETTER KEHEH WITH SMALL V",NULL},
/* 08C3 */ { "ARABIC LETTER GHAIN WITH THREE DOTS ABOVE",NULL},
/* 08C4 */ { "ARABIC LETTER AFRICAN QAF WITH THREE DOTS ABOVE","	* this letter shows an additional single dot above for the qaf only in initial and medial forms\n"
	"	x (arabic letter african qaf - 08BC)"},
/* 08C5 */ { "ARABIC LETTER JEEM WITH THREE DOTS ABOVE",NULL},
/* 08C6 */ { "ARABIC LETTER JEEM WITH THREE DOTS BELOW",NULL},
/* 08C7 */ { "ARABIC LETTER LAM WITH SMALL ARABIC LETTER TAH ABOVE",NULL},
/* 08C8 */ { "ARABIC LETTER GRAF",NULL},
/* 08C9 */ { "ARABIC SMALL FARSI YEH",NULL},
/* 08CA */ { "ARABIC SMALL HIGH FARSI YEH",NULL},
/* 08CB */ { "ARABIC SMALL HIGH YEH BARREE WITH TWO DOTS BELOW",NULL},
/* 08CC */ { "ARABIC SMALL HIGH WORD SAH","	= sign of waqf"},
/* 08CD */ { "ARABIC SMALL HIGH ZAH",NULL},
/* 08CE */ { "ARABIC LARGE ROUND DOT ABOVE",NULL},
/* 08CF */ { "ARABIC LARGE ROUND DOT BELOW",NULL},
/* 08D0 */ { "ARABIC SUKUN BELOW",NULL},
/* 08D1 */ { "ARABIC LARGE CIRCLE BELOW",NULL},
/* 08D2 */ { "ARABIC LARGE ROUND DOT INSIDE CIRCLE BELOW",NULL},
/* 08D3 */ { "ARABIC SMALL LOW WAW",NULL},
/* 08D4 */ { "ARABIC SMALL HIGH WORD AR-RUB",NULL},
/* 08D5 */ { "ARABIC SMALL HIGH SAD",NULL},
/* 08D6 */ { "ARABIC SMALL HIGH AIN",NULL},
/* 08D7 */ { "ARABIC SMALL HIGH QAF",NULL},
/* 08D8 */ { "ARABIC SMALL HIGH NOON WITH KASRA",NULL},
/* 08D9 */ { "ARABIC SMALL LOW NOON WITH KASRA",NULL},
/* 08DA */ { "ARABIC SMALL HIGH WORD ATH-THALATHA",NULL},
/* 08DB */ { "ARABIC SMALL HIGH WORD AS-SAJDA",NULL},
/* 08DC */ { "ARABIC SMALL HIGH WORD AN-NISF",NULL},
/* 08DD */ { "ARABIC SMALL HIGH WORD SAKTA","	x (arabic small low word sakta - 10EFD)"},
/* 08DE */ { "ARABIC SMALL HIGH WORD QIF",NULL},
/* 08DF */ { "ARABIC SMALL HIGH WORD WAQFA",NULL},
/* 08E0 */ { "ARABIC SMALL HIGH FOOTNOTE MARKER",NULL},
/* 08E1 */ { "ARABIC SMALL HIGH SIGN SAFHA",NULL},
/* 08E2 */ { "ARABIC DISPUTED END OF AYAH",NULL},
/* 08E3 */ { "ARABIC TURNED DAMMA BELOW",NULL},
/* 08E4 */ { "ARABIC CURLY FATHA",NULL},
/* 08E5 */ { "ARABIC CURLY DAMMA",NULL},
/* 08E6 */ { "ARABIC CURLY KASRA",NULL},
/* 08E7 */ { "ARABIC CURLY FATHATAN",NULL},
/* 08E8 */ { "ARABIC CURLY DAMMATAN",NULL},
/* 08E9 */ { "ARABIC CURLY KASRATAN",NULL},
/* 08EA */ { "ARABIC TONE ONE DOT ABOVE",NULL},
/* 08EB */ { "ARABIC TONE TWO DOTS ABOVE",NULL},
/* 08EC */ { "ARABIC TONE LOOP ABOVE",NULL},
/* 08ED */ { "ARABIC TONE ONE DOT BELOW",NULL},
/* 08EE */ { "ARABIC TONE TWO DOTS BELOW",NULL},
/* 08EF */ { "ARABIC TONE LOOP BELOW",NULL},
/* 08F0 */ { "ARABIC OPEN FATHATAN","	= successive fathatan"},
/* 08F1 */ { "ARABIC OPEN DAMMATAN","	= successive dammatan"},
/* 08F2 */ { "ARABIC OPEN KASRATAN","	= successive kasratan"},
/* 08F3 */ { "ARABIC SMALL HIGH WAW",NULL},
/* 08F4 */ { "ARABIC FATHA WITH RING",NULL},
/* 08F5 */ { "ARABIC FATHA WITH DOT ABOVE",NULL},
/* 08F6 */ { "ARABIC KASRA WITH DOT BELOW","	* also used in Philippine languages"},
/* 08F7 */ { "ARABIC LEFT ARROWHEAD ABOVE",NULL},
/* 08F8 */ { "ARABIC RIGHT ARROWHEAD ABOVE","	* also used in Quranic text in African and other orthographies to represent damma"},
/* 08F9 */ { "ARABIC LEFT ARROWHEAD BELOW",NULL},
/* 08FA */ { "ARABIC RIGHT ARROWHEAD BELOW",NULL},
/* 08FB */ { "ARABIC DOUBLE RIGHT ARROWHEAD ABOVE","	* also used in Quranic text in African and other orthographies to represent dammatan"},
/* 08FC */ { "ARABIC DOUBLE RIGHT ARROWHEAD ABOVE WITH DOT",NULL},
/* 08FD */ { "ARABIC RIGHT ARROWHEAD ABOVE WITH DOT",NULL},
/* 08FE */ { "ARABIC DAMMA WITH DOT","	* Philippine languages"},
/* 08FF */ { "ARABIC MARK SIDEWAYS NOON GHUNNA",NULL}
};

UN_DLL_LOCAL
static const struct unicode_nameannot una_00_09[] = {
/* 0900 */ { "DEVANAGARI SIGN INVERTED CANDRABINDU","	= vaidika adhomukha candrabindu"},
/* 0901 */ { "DEVANAGARI SIGN CANDRABINDU","	= anunasika\n"
	"	x (combining candrabindu - 0310)"},
/* 0902 */ { "DEVANAGARI SIGN ANUSVARA","	= bindu"},
/* 0903 */ { "DEVANAGARI SIGN VISARGA",NULL},
/* 0904 */ { "DEVANAGARI LETTER SHORT A","	* used for short e in Awadhi\n"
	"	* also used in Devanagari transliterations of some South Indian and Kashmiri languages by a publisher in Lucknow"},
/* 0905 */ { "DEVANAGARI LETTER A",NULL},
/* 0906 */ { "DEVANAGARI LETTER AA",NULL},
/* 0907 */ { "DEVANAGARI LETTER I",NULL},
/* 0908 */ { "DEVANAGARI LETTER II",NULL},
/* 0909 */ { "DEVANAGARI LETTER U",NULL},
/* 090A */ { "DEVANAGARI LETTER UU",NULL},
/* 090B */ { "DEVANAGARI LETTER VOCALIC R",NULL},
/* 090C */ { "DEVANAGARI LETTER VOCALIC L",NULL},
/* 090D */ { "DEVANAGARI LETTER CANDRA E",NULL},
/* 090E */ { "DEVANAGARI LETTER SHORT E","	* Kashmiri, Bihari languages\n"
	"	* also used for transcribing Dravidian short e"},
/* 090F */ { "DEVANAGARI LETTER E",NULL},
/* 0910 */ { "DEVANAGARI LETTER AI",NULL},
/* 0911 */ { "DEVANAGARI LETTER CANDRA O",NULL},
/* 0912 */ { "DEVANAGARI LETTER SHORT O","	* Kashmiri, Bihari languages\n"
	"	* also used for transcribing Dravidian short o"},
/* 0913 */ { "DEVANAGARI LETTER O",NULL},
/* 0914 */ { "DEVANAGARI LETTER AU",NULL},
/* 0915 */ { "DEVANAGARI LETTER KA",NULL},
/* 0916 */ { "DEVANAGARI LETTER KHA",NULL},
/* 0917 */ { "DEVANAGARI LETTER GA",NULL},
/* 0918 */ { "DEVANAGARI LETTER GHA",NULL},
/* 0919 */ { "DEVANAGARI LETTER NGA",NULL},
/* 091A */ { "DEVANAGARI LETTER CA",NULL},
/* 091B */ { "DEVANAGARI LETTER CHA",NULL},
/* 091C */ { "DEVANAGARI LETTER JA",NULL},
/* 091D */ { "DEVANAGARI LETTER JHA",NULL},
/* 091E */ { "DEVANAGARI LETTER NYA",NULL},
/* 091F */ { "DEVANAGARI LETTER TTA",NULL},
/* 0920 */ { "DEVANAGARI LETTER TTHA",NULL},
/* 0921 */ { "DEVANAGARI LETTER DDA",NULL},
/* 0922 */ { "DEVANAGARI LETTER DDHA",NULL},
/* 0923 */ { "DEVANAGARI LETTER NNA",NULL},
/* 0924 */ { "DEVANAGARI LETTER TA",NULL},
/* 0925 */ { "DEVANAGARI LETTER THA",NULL},
/* 0926 */ { "DEVANAGARI LETTER DA",NULL},
/* 0927 */ { "DEVANAGARI LETTER DHA",NULL},
/* 0928 */ { "DEVANAGARI LETTER NA",NULL},
/* 0929 */ { "DEVANAGARI LETTER NNNA","	* for transcribing Dravidian alveolar n\n"
	"	: 0928 093C"},
/* 092A */ { "DEVANAGARI LETTER PA",NULL},
/* 092B */ { "DEVANAGARI LETTER PHA",NULL},
/* 092C */ { "DEVANAGARI LETTER BA",NULL},
/* 092D */ { "DEVANAGARI LETTER BHA",NULL},
/* 092E */ { "DEVANAGARI LETTER MA",NULL},
/* 092F */ { "DEVANAGARI LETTER YA",NULL},
/* 0930 */ { "DEVANAGARI LETTER RA",NULL},
/* 0931 */ { "DEVANAGARI LETTER RRA","	* for transcribing Dravidian alveolar r\n"
	"	* half form is represented as \"Eyelash RA\"\n"
	"	: 0930 093C"},
/* 0932 */ { "DEVANAGARI LETTER LA",NULL},
/* 0933 */ { "DEVANAGARI LETTER LLA",NULL},
/* 0934 */ { "DEVANAGARI LETTER LLLA","	* for transcribing Dravidian l\n"
	"	: 0933 093C"},
/* 0935 */ { "DEVANAGARI LETTER VA",NULL},
/* 0936 */ { "DEVANAGARI LETTER SHA",NULL},
/* 0937 */ { "DEVANAGARI LETTER SSA",NULL},
/* 0938 */ { "DEVANAGARI LETTER SA",NULL},
/* 0939 */ { "DEVANAGARI LETTER HA",NULL},
/* 093A */ { "DEVANAGARI VOWEL SIGN OE",NULL},
/* 093B */ { "DEVANAGARI VOWEL SIGN OOE",NULL},
/* 093C */ { "DEVANAGARI SIGN NUKTA","	* for extending the alphabet to new letters"},
/* 093D */ { "DEVANAGARI SIGN AVAGRAHA",NULL},
/* 093E */ { "DEVANAGARI VOWEL SIGN AA",NULL},
/* 093F */ { "DEVANAGARI VOWEL SIGN I","	* stands to the left of the consonant"},
/* 0940 */ { "DEVANAGARI VOWEL SIGN II",NULL},
/* 0941 */ { "DEVANAGARI VOWEL SIGN U",NULL},
/* 0942 */ { "DEVANAGARI VOWEL SIGN UU",NULL},
/* 0943 */ { "DEVANAGARI VOWEL SIGN VOCALIC R",NULL},
/* 0944 */ { "DEVANAGARI VOWEL SIGN VOCALIC RR",NULL},
/* 0945 */ { "DEVANAGARI VOWEL SIGN CANDRA E","	= candra"},
/* 0946 */ { "DEVANAGARI VOWEL SIGN SHORT E","	* Kashmiri, Bihari languages\n"
	"	* also used for transcribing Dravidian short e"},
/* 0947 */ { "DEVANAGARI VOWEL SIGN E",NULL},
/* 0948 */ { "DEVANAGARI VOWEL SIGN AI",NULL},
/* 0949 */ { "DEVANAGARI VOWEL SIGN CANDRA O",NULL},
/* 094A */ { "DEVANAGARI VOWEL SIGN SHORT O","	* Kashmiri, Bihari languages\n"
	"	* also used for transcribing Dravidian short o"},
/* 094B */ { "DEVANAGARI VOWEL SIGN O",NULL},
/* 094C */ { "DEVANAGARI VOWEL SIGN AU",NULL},
/* 094D */ { "DEVANAGARI SIGN VIRAMA","	= halant (the preferred Hindi name)\n"
	"	* suppresses inherent vowel"},
/* 094E */ { "DEVANAGARI VOWEL SIGN PRISHTHAMATRA E","	* character has historic use only\n"
	"	* combines with E to form AI, with AA to form O, and with O to form AU"},
/* 094F */ { "DEVANAGARI VOWEL SIGN AW","	* Kashmiri, Bihari languages"},
/* 0950 */ { "DEVANAGARI OM","	x (om symbol - 1F549)"},
/* 0951 */ { "DEVANAGARI STRESS SIGN UDATTA","	= Vedic tone svarita\n"
	"	* mostly used for svarita, with rare use for udatta\n"
	"	* used also in Vedic texts written in other scripts\n"
	"	x (vedic tone double svarita - 1CDA)"},
/* 0952 */ { "DEVANAGARI STRESS SIGN ANUDATTA","	= Vedic tone anudatta\n"
	"	* used also in Vedic texts written in other scripts\n"
	"	x (vedic tone kathaka anudatta - 1CDC)"},
/* 0953 */ { "DEVANAGARI GRAVE ACCENT","	x (combining grave accent - 0300)"},
/* 0954 */ { "DEVANAGARI ACUTE ACCENT","	x (combining acute accent - 0301)"},
/* 0955 */ { "DEVANAGARI VOWEL SIGN CANDRA LONG E","	* used in transliteration of Avestan"},
/* 0956 */ { "DEVANAGARI VOWEL SIGN UE",NULL},
/* 0957 */ { "DEVANAGARI VOWEL SIGN UUE",NULL},
/* 0958 */ { "DEVANAGARI LETTER QA","	: 0915 093C"},
/* 0959 */ { "DEVANAGARI LETTER KHHA","	: 0916 093C"},
/* 095A */ { "DEVANAGARI LETTER GHHA","	: 0917 093C"},
/* 095B */ { "DEVANAGARI LETTER ZA","	: 091C 093C"},
/* 095C */ { "DEVANAGARI LETTER DDDHA","	: 0921 093C"},
/* 095D */ { "DEVANAGARI LETTER RHA","	: 0922 093C"},
/* 095E */ { "DEVANAGARI LETTER FA","	: 092B 093C"},
/* 095F */ { "DEVANAGARI LETTER YYA","	: 092F 093C"},
/* 0960 */ { "DEVANAGARI LETTER VOCALIC RR",NULL},
/* 0961 */ { "DEVANAGARI LETTER VOCALIC LL",NULL},
/* 0962 */ { "DEVANAGARI VOWEL SIGN VOCALIC L",NULL},
/* 0963 */ { "DEVANAGARI VOWEL SIGN VOCALIC LL",NULL},
/* 0964 */ { "DEVANAGARI DANDA","	= purna viram\n"
	"	* phrase separator"},
/* 0965 */ { "DEVANAGARI DOUBLE DANDA","	= deergh viram"},
/* 0966 */ { "DEVANAGARI DIGIT ZERO","	* also used to represent an anusvara following digits indicating secondary svara-s in Samavedic texts"},
/* 0967 */ { "DEVANAGARI DIGIT ONE",NULL},
/* 0968 */ { "DEVANAGARI DIGIT TWO",NULL},
/* 0969 */ { "DEVANAGARI DIGIT THREE",NULL},
/* 096A */ { "DEVANAGARI DIGIT FOUR",NULL},
/* 096B */ { "DEVANAGARI DIGIT FIVE",NULL},
/* 096C */ { "DEVANAGARI DIGIT SIX",NULL},
/* 096D */ { "DEVANAGARI DIGIT SEVEN",NULL},
/* 096E */ { "DEVANAGARI DIGIT EIGHT",NULL},
/* 096F */ { "DEVANAGARI DIGIT NINE",NULL},
/* 0970 */ { "DEVANAGARI ABBREVIATION SIGN","	* intended for Devanagari-specific abbreviations"},
/* 0971 */ { "DEVANAGARI SIGN HIGH SPACING DOT",NULL},
/* 0972 */ { "DEVANAGARI LETTER CANDRA A",NULL},
/* 0973 */ { "DEVANAGARI LETTER OE",NULL},
/* 0974 */ { "DEVANAGARI LETTER OOE",NULL},
/* 0975 */ { "DEVANAGARI LETTER AW",NULL},
/* 0976 */ { "DEVANAGARI LETTER UE",NULL},
/* 0977 */ { "DEVANAGARI LETTER UUE",NULL},
/* 0978 */ { "DEVANAGARI LETTER MARWARI DDA",NULL},
/* 0979 */ { "DEVANAGARI LETTER ZHA","	* used in transliteration of Avestan\n"
	"	x (gujarati letter zha - 0AF9)"},
/* 097A */ { "DEVANAGARI LETTER HEAVY YA","	* used for an affricated glide JJYA"},
/* 097B */ { "DEVANAGARI LETTER GGA",NULL},
/* 097C */ { "DEVANAGARI LETTER JJA",NULL},
/* 097D */ { "DEVANAGARI LETTER GLOTTAL STOP","	* used for writing Limbu in Devanagari\n"
	"	* a glyph variant has the connecting top bar"},
/* 097E */ { "DEVANAGARI LETTER DDDA",NULL},
/* 097F */ { "DEVANAGARI LETTER BBA",NULL},
/* 0980 */ { "BENGALI ANJI","	= siddham, siddhirastu\n"
	"	* used at the beginning of texts as an invocation"},
/* 0981 */ { "BENGALI SIGN CANDRABINDU",NULL},
/* 0982 */ { "BENGALI SIGN ANUSVARA",NULL},
/* 0983 */ { "BENGALI SIGN VISARGA",NULL},
/* 0984 */ { NULL,NULL},
/* 0985 */ { "BENGALI LETTER A",NULL},
/* 0986 */ { "BENGALI LETTER AA",NULL},
/* 0987 */ { "BENGALI LETTER I",NULL},
/* 0988 */ { "BENGALI LETTER II",NULL},
/* 0989 */ { "BENGALI LETTER U",NULL},
/* 098A */ { "BENGALI LETTER UU",NULL},
/* 098B */ { "BENGALI LETTER VOCALIC R",NULL},
/* 098C */ { "BENGALI LETTER VOCALIC L",NULL},
/* 098D */ { NULL,NULL},
/* 098E */ { NULL,NULL},
/* 098F */ { "BENGALI LETTER E",NULL},
/* 0990 */ { "BENGALI LETTER AI",NULL},
/* 0991 */ { NULL,NULL},
/* 0992 */ { NULL,NULL},
/* 0993 */ { "BENGALI LETTER O",NULL},
/* 0994 */ { "BENGALI LETTER AU",NULL},
/* 0995 */ { "BENGALI LETTER KA",NULL},
/* 0996 */ { "BENGALI LETTER KHA",NULL},
/* 0997 */ { "BENGALI LETTER GA",NULL},
/* 0998 */ { "BENGALI LETTER GHA",NULL},
/* 0999 */ { "BENGALI LETTER NGA",NULL},
/* 099A */ { "BENGALI LETTER CA",NULL},
/* 099B */ { "BENGALI LETTER CHA",NULL},
/* 099C */ { "BENGALI LETTER JA",NULL},
/* 099D */ { "BENGALI LETTER JHA",NULL},
/* 099E */ { "BENGALI LETTER NYA",NULL},
/* 099F */ { "BENGALI LETTER TTA",NULL},
/* 09A0 */ { "BENGALI LETTER TTHA",NULL},
/* 09A1 */ { "BENGALI LETTER DDA",NULL},
/* 09A2 */ { "BENGALI LETTER DDHA",NULL},
/* 09A3 */ { "BENGALI LETTER NNA",NULL},
/* 09A4 */ { "BENGALI LETTER TA",NULL},
/* 09A5 */ { "BENGALI LETTER THA",NULL},
/* 09A6 */ { "BENGALI LETTER DA",NULL},
/* 09A7 */ { "BENGALI LETTER DHA",NULL},
/* 09A8 */ { "BENGALI LETTER NA",NULL},
/* 09A9 */ { NULL,NULL},
/* 09AA */ { "BENGALI LETTER PA",NULL},
/* 09AB */ { "BENGALI LETTER PHA",NULL},
/* 09AC */ { "BENGALI LETTER BA","	= Bengali va, wa"},
/* 09AD */ { "BENGALI LETTER BHA",NULL},
/* 09AE */ { "BENGALI LETTER MA",NULL},
/* 09AF */ { "BENGALI LETTER YA",NULL},
/* 09B0 */ { "BENGALI LETTER RA",NULL},
/* 09B1 */ { NULL,NULL},
/* 09B2 */ { "BENGALI LETTER LA",NULL},
/* 09B3 */ { NULL,NULL},
/* 09B4 */ { NULL,NULL},
/* 09B5 */ { NULL,NULL},
/* 09B6 */ { "BENGALI LETTER SHA",NULL},
/* 09B7 */ { "BENGALI LETTER SSA",NULL},
/* 09B8 */ { "BENGALI LETTER SA",NULL},
/* 09B9 */ { "BENGALI LETTER HA",NULL},
/* 09BA */ { NULL,NULL},
/* 09BB */ { NULL,NULL},
/* 09BC */ { "BENGALI SIGN NUKTA","	* for extending the alphabet to new letters"},
/* 09BD */ { "BENGALI SIGN AVAGRAHA",NULL},
/* 09BE */ { "BENGALI VOWEL SIGN AA",NULL},
/* 09BF */ { "BENGALI VOWEL SIGN I","	* stands to the left of the consonant"},
/* 09C0 */ { "BENGALI VOWEL SIGN II",NULL},
/* 09C1 */ { "BENGALI VOWEL SIGN U",NULL},
/* 09C2 */ { "BENGALI VOWEL SIGN UU",NULL},
/* 09C3 */ { "BENGALI VOWEL SIGN VOCALIC R",NULL},
/* 09C4 */ { "BENGALI VOWEL SIGN VOCALIC RR",NULL},
/* 09C5 */ { NULL,NULL},
/* 09C6 */ { NULL,NULL},
/* 09C7 */ { "BENGALI VOWEL SIGN E","	* stands to the left of the consonant"},
/* 09C8 */ { "BENGALI VOWEL SIGN AI","	* stands to the left of the consonant"},
/* 09C9 */ { NULL,NULL},
/* 09CA */ { NULL,NULL},
/* 09CB */ { "BENGALI VOWEL SIGN O","	: 09C7 09BE"},
/* 09CC */ { "BENGALI VOWEL SIGN AU","	: 09C7 09D7"},
/* 09CD */ { "BENGALI SIGN VIRAMA","	= hasant (Bengali term for halant)"},
/* 09CE */ { "BENGALI LETTER KHANDA TA","	* a dead consonant form of ta, without implicit vowel, used in some sequences"},
/* 09CF */ { NULL,NULL},
/* 09D0 */ { NULL,NULL},
/* 09D1 */ { NULL,NULL},
/* 09D2 */ { NULL,NULL},
/* 09D3 */ { NULL,NULL},
/* 09D4 */ { NULL,NULL},
/* 09D5 */ { NULL,NULL},
/* 09D6 */ { NULL,NULL},
/* 09D7 */ { "BENGALI AU LENGTH MARK",NULL},
/* 09D8 */ { NULL,NULL},
/* 09D9 */ { NULL,NULL},
/* 09DA */ { NULL,NULL},
/* 09DB */ { NULL,NULL},
/* 09DC */ { "BENGALI LETTER RRA","	: 09A1 09BC"},
/* 09DD */ { "BENGALI LETTER RHA","	: 09A2 09BC"},
/* 09DE */ { NULL,NULL},
/* 09DF */ { "BENGALI LETTER YYA","	: 09AF 09BC"},
/* 09E0 */ { "BENGALI LETTER VOCALIC RR",NULL},
/* 09E1 */ { "BENGALI LETTER VOCALIC LL",NULL},
/* 09E2 */ { "BENGALI VOWEL SIGN VOCALIC L",NULL},
/* 09E3 */ { "BENGALI VOWEL SIGN VOCALIC LL",NULL},
/* 09E4 */ { NULL,"	x (devanagari danda - 0964)"},
/* 09E5 */ { NULL,"	x (devanagari double danda - 0965)"},
/* 09E6 */ { "BENGALI DIGIT ZERO",NULL},
/* 09E7 */ { "BENGALI DIGIT ONE",NULL},
/* 09E8 */ { "BENGALI DIGIT TWO",NULL},
/* 09E9 */ { "BENGALI DIGIT THREE",NULL},
/* 09EA */ { "BENGALI DIGIT FOUR",NULL},
/* 09EB */ { "BENGALI DIGIT FIVE",NULL},
/* 09EC */ { "BENGALI DIGIT SIX",NULL},
/* 09ED */ { "BENGALI DIGIT SEVEN",NULL},
/* 09EE */ { "BENGALI DIGIT EIGHT",NULL},
/* 09EF */ { "BENGALI DIGIT NINE",NULL},
/* 09F0 */ { "BENGALI LETTER RA WITH MIDDLE DIAGONAL","	= Assamese letter ra"},
/* 09F1 */ { "BENGALI LETTER RA WITH LOWER DIAGONAL","	= Assamese letter wa\n"
	"	= bengali letter va with lower diagonal (1.0)"},
/* 09F2 */ { "BENGALI RUPEE MARK","	= taka\n"
	"	* historic currency sign"},
/* 09F3 */ { "BENGALI RUPEE SIGN","	= Bangladeshi taka"},
/* 09F4 */ { "BENGALI CURRENCY NUMERATOR ONE","	* not in current usage"},
/* 09F5 */ { "BENGALI CURRENCY NUMERATOR TWO","	* not in current usage"},
/* 09F6 */ { "BENGALI CURRENCY NUMERATOR THREE","	* not in current usage"},
/* 09F7 */ { "BENGALI CURRENCY NUMERATOR FOUR",NULL},
/* 09F8 */ { "BENGALI CURRENCY NUMERATOR ONE LESS THAN THE DENOMINATOR",NULL},
/* 09F9 */ { "BENGALI CURRENCY DENOMINATOR SIXTEEN",NULL},
/* 09FA */ { "BENGALI ISSHAR","	= ishvar\n"
	"	* represents the name of a deity\n"
	"	= svargiya\n"
	"	* written before the name of a deceased person"},
/* 09FB */ { "BENGALI GANDA MARK",NULL},
/* 09FC */ { "BENGALI LETTER VEDIC ANUSVARA",NULL},
/* 09FD */ { "BENGALI ABBREVIATION SIGN",NULL},
/* 09FE */ { "BENGALI SANDHI MARK","	* indicates external sandhi in Sanskrit documents\n"
	"	x (bengali sign avagraha - 09BD)\n"
	"	x (sharada sandhi mark - 111C9)\n"
	"	x (newa sandhi mark - 1145E)"},
/* 09FF */ { NULL,NULL}
};

UN_DLL_LOCAL
static const struct unicode_nameannot una_00_0A[] = {
/* 0A00 */ { NULL,NULL},
/* 0A01 */ { "GURMUKHI SIGN ADAK BINDI",NULL},
/* 0A02 */ { "GURMUKHI SIGN BINDI",NULL},
/* 0A03 */ { "GURMUKHI SIGN VISARGA",NULL},
/* 0A04 */ { NULL,NULL},
/* 0A05 */ { "GURMUKHI LETTER A","	= aira"},
/* 0A06 */ { "GURMUKHI LETTER AA",NULL},
/* 0A07 */ { "GURMUKHI LETTER I",NULL},
/* 0A08 */ { "GURMUKHI LETTER II",NULL},
/* 0A09 */ { "GURMUKHI LETTER U",NULL},
/* 0A0A */ { "GURMUKHI LETTER UU",NULL},
/* 0A0B */ { NULL,NULL},
/* 0A0C */ { NULL,NULL},
/* 0A0D */ { NULL,NULL},
/* 0A0E */ { NULL,NULL},
/* 0A0F */ { "GURMUKHI LETTER EE",NULL},
/* 0A10 */ { "GURMUKHI LETTER AI",NULL},
/* 0A11 */ { NULL,NULL},
/* 0A12 */ { NULL,NULL},
/* 0A13 */ { "GURMUKHI LETTER OO",NULL},
/* 0A14 */ { "GURMUKHI LETTER AU",NULL},
/* 0A15 */ { "GURMUKHI LETTER KA",NULL},
/* 0A16 */ { "GURMUKHI LETTER KHA",NULL},
/* 0A17 */ { "GURMUKHI LETTER GA",NULL},
/* 0A18 */ { "GURMUKHI LETTER GHA",NULL},
/* 0A19 */ { "GURMUKHI LETTER NGA",NULL},
/* 0A1A */ { "GURMUKHI LETTER CA",NULL},
/* 0A1B */ { "GURMUKHI LETTER CHA",NULL},
/* 0A1C */ { "GURMUKHI LETTER JA",NULL},
/* 0A1D */ { "GURMUKHI LETTER JHA",NULL},
/* 0A1E */ { "GURMUKHI LETTER NYA",NULL},
/* 0A1F */ { "GURMUKHI LETTER TTA",NULL},
/* 0A20 */ { "GURMUKHI LETTER TTHA",NULL},
/* 0A21 */ { "GURMUKHI LETTER DDA",NULL},
/* 0A22 */ { "GURMUKHI LETTER DDHA",NULL},
/* 0A23 */ { "GURMUKHI LETTER NNA",NULL},
/* 0A24 */ { "GURMUKHI LETTER TA",NULL},
/* 0A25 */ { "GURMUKHI LETTER THA",NULL},
/* 0A26 */ { "GURMUKHI LETTER DA",NULL},
/* 0A27 */ { "GURMUKHI LETTER DHA",NULL},
/* 0A28 */ { "GURMUKHI LETTER NA",NULL},
/* 0A29 */ { NULL,NULL},
/* 0A2A */ { "GURMUKHI LETTER PA",NULL},
/* 0A2B */ { "GURMUKHI LETTER PHA",NULL},
/* 0A2C */ { "GURMUKHI LETTER BA",NULL},
/* 0A2D */ { "GURMUKHI LETTER BHA",NULL},
/* 0A2E */ { "GURMUKHI LETTER MA",NULL},
/* 0A2F */ { "GURMUKHI LETTER YA",NULL},
/* 0A30 */ { "GURMUKHI LETTER RA",NULL},
/* 0A31 */ { NULL,NULL},
/* 0A32 */ { "GURMUKHI LETTER LA",NULL},
/* 0A33 */ { "GURMUKHI LETTER LLA","	: 0A32 0A3C\n"
	"		This consonant with nukta is listed in CompositionExclusions.txt. That means that it does not recompose during normalization. The NFC form is the same as the decomposed sequence."},
/* 0A34 */ { NULL,NULL},
/* 0A35 */ { "GURMUKHI LETTER VA",NULL},
/* 0A36 */ { "GURMUKHI LETTER SHA","	: 0A38 0A3C\n"
	"		This consonant with nukta is listed in CompositionExclusions.txt. That means that it does not recompose during normalization. The NFC form is the same as the decomposed sequence."},
/* 0A37 */ { NULL,NULL},
/* 0A38 */ { "GURMUKHI LETTER SA",NULL},
/* 0A39 */ { "GURMUKHI LETTER HA",NULL},
/* 0A3A */ { NULL,NULL},
/* 0A3B */ { NULL,NULL},
/* 0A3C */ { "GURMUKHI SIGN NUKTA","	= pairin bindi\n"
	"	* for extending the alphabet to new letters"},
/* 0A3D */ { NULL,NULL},
/* 0A3E */ { "GURMUKHI VOWEL SIGN AA","	= kanna"},
/* 0A3F */ { "GURMUKHI VOWEL SIGN I","	= sihari\n"
	"	* stands to the left of the consonant"},
/* 0A40 */ { "GURMUKHI VOWEL SIGN II","	= bihari"},
/* 0A41 */ { "GURMUKHI VOWEL SIGN U","	= aunkar"},
/* 0A42 */ { "GURMUKHI VOWEL SIGN UU","	= dulainkar"},
/* 0A43 */ { NULL,NULL},
/* 0A44 */ { NULL,NULL},
/* 0A45 */ { NULL,NULL},
/* 0A46 */ { NULL,NULL},
/* 0A47 */ { "GURMUKHI VOWEL SIGN EE","	= lanvan"},
/* 0A48 */ { "GURMUKHI VOWEL SIGN AI","	= dulanvan"},
/* 0A49 */ { NULL,NULL},
/* 0A4A */ { NULL,NULL},
/* 0A4B */ { "GURMUKHI VOWEL SIGN OO","	= hora"},
/* 0A4C */ { "GURMUKHI VOWEL SIGN AU","	= kanaura"},
/* 0A4D */ { "GURMUKHI SIGN VIRAMA",NULL},
/* 0A4E */ { NULL,NULL},
/* 0A4F */ { NULL,NULL},
/* 0A50 */ { NULL,NULL},
/* 0A51 */ { "GURMUKHI SIGN UDAAT",NULL},
/* 0A52 */ { NULL,NULL},
/* 0A53 */ { NULL,NULL},
/* 0A54 */ { NULL,NULL},
/* 0A55 */ { NULL,NULL},
/* 0A56 */ { NULL,NULL},
/* 0A57 */ { NULL,NULL},
/* 0A58 */ { NULL,NULL},
/* 0A59 */ { "GURMUKHI LETTER KHHA","	: 0A16 0A3C"},
/* 0A5A */ { "GURMUKHI LETTER GHHA","	: 0A17 0A3C"},
/* 0A5B */ { "GURMUKHI LETTER ZA","	: 0A1C 0A3C"},
/* 0A5C */ { "GURMUKHI LETTER RRA",NULL},
/* 0A5D */ { NULL,NULL},
/* 0A5E */ { "GURMUKHI LETTER FA","	: 0A2B 0A3C"},
/* 0A5F */ { NULL,NULL},
/* 0A60 */ { NULL,NULL},
/* 0A61 */ { NULL,NULL},
/* 0A62 */ { NULL,NULL},
/* 0A63 */ { NULL,NULL},
/* 0A64 */ { NULL,"	x (devanagari danda - 0964)"},
/* 0A65 */ { NULL,"	x (devanagari double danda - 0965)"},
/* 0A66 */ { "GURMUKHI DIGIT ZERO",NULL},
/* 0A67 */ { "GURMUKHI DIGIT ONE",NULL},
/* 0A68 */ { "GURMUKHI DIGIT TWO",NULL},
/* 0A69 */ { "GURMUKHI DIGIT THREE",NULL},
/* 0A6A */ { "GURMUKHI DIGIT FOUR",NULL},
/* 0A6B */ { "GURMUKHI DIGIT FIVE",NULL},
/* 0A6C */ { "GURMUKHI DIGIT SIX",NULL},
/* 0A6D */ { "GURMUKHI DIGIT SEVEN",NULL},
/* 0A6E */ { "GURMUKHI DIGIT EIGHT",NULL},
/* 0A6F */ { "GURMUKHI DIGIT NINE",NULL},
/* 0A70 */ { "GURMUKHI TIPPI","	* nasalization"},
/* 0A71 */ { "GURMUKHI ADDAK","	* doubles following consonant"},
/* 0A72 */ { "GURMUKHI IRI",NULL},
/* 0A73 */ { "GURMUKHI URA",NULL},
/* 0A74 */ { "GURMUKHI EK ONKAR","	* God is One"},
/* 0A75 */ { "GURMUKHI SIGN YAKASH","	* some fonts use an alternate glyph shaped more like the lower part of 0A2F"},
/* 0A76 */ { "GURMUKHI ABBREVIATION SIGN","	x (sharada abbreviation sign - 111C7)"},
/* 0A77 */ { NULL,NULL},
/* 0A78 */ { NULL,NULL},
/* 0A79 */ { NULL,NULL},
/* 0A7A */ { NULL,NULL},
/* 0A7B */ { NULL,NULL},
/* 0A7C */ { NULL,NULL},
/* 0A7D */ { NULL,NULL},
/* 0A7E */ { NULL,NULL},
/* 0A7F */ { NULL,NULL},
/* 0A80 */ { NULL,NULL},
/* 0A81 */ { "GUJARATI SIGN CANDRABINDU",NULL},
/* 0A82 */ { "GUJARATI SIGN ANUSVARA",NULL},
/* 0A83 */ { "GUJARATI SIGN VISARGA",NULL},
/* 0A84 */ { NULL,NULL},
/* 0A85 */ { "GUJARATI LETTER A",NULL},
/* 0A86 */ { "GUJARATI LETTER AA",NULL},
/* 0A87 */ { "GUJARATI LETTER I",NULL},
/* 0A88 */ { "GUJARATI LETTER II",NULL},
/* 0A89 */ { "GUJARATI LETTER U",NULL},
/* 0A8A */ { "GUJARATI LETTER UU",NULL},
/* 0A8B */ { "GUJARATI LETTER VOCALIC R",NULL},
/* 0A8C */ { "GUJARATI LETTER VOCALIC L","	* used with Sanskrit text"},
/* 0A8D */ { "GUJARATI VOWEL CANDRA E",NULL},
/* 0A8E */ { NULL,NULL},
/* 0A8F */ { "GUJARATI LETTER E",NULL},
/* 0A90 */ { "GUJARATI LETTER AI",NULL},
/* 0A91 */ { "GUJARATI VOWEL CANDRA O",NULL},
/* 0A92 */ { NULL,NULL},
/* 0A93 */ { "GUJARATI LETTER O",NULL},
/* 0A94 */ { "GUJARATI LETTER AU",NULL},
/* 0A95 */ { "GUJARATI LETTER KA",NULL},
/* 0A96 */ { "GUJARATI LETTER KHA",NULL},
/* 0A97 */ { "GUJARATI LETTER GA",NULL},
/* 0A98 */ { "GUJARATI LETTER GHA",NULL},
/* 0A99 */ { "GUJARATI LETTER NGA",NULL},
/* 0A9A */ { "GUJARATI LETTER CA",NULL},
/* 0A9B */ { "GUJARATI LETTER CHA",NULL},
/* 0A9C */ { "GUJARATI LETTER JA",NULL},
/* 0A9D */ { "GUJARATI LETTER JHA",NULL},
/* 0A9E */ { "GUJARATI LETTER NYA",NULL},
/* 0A9F */ { "GUJARATI LETTER TTA",NULL},
/* 0AA0 */ { "GUJARATI LETTER TTHA",NULL},
/* 0AA1 */ { "GUJARATI LETTER DDA",NULL},
/* 0AA2 */ { "GUJARATI LETTER DDHA",NULL},
/* 0AA3 */ { "GUJARATI LETTER NNA",NULL},
/* 0AA4 */ { "GUJARATI LETTER TA",NULL},
/* 0AA5 */ { "GUJARATI LETTER THA",NULL},
/* 0AA6 */ { "GUJARATI LETTER DA",NULL},
/* 0AA7 */ { "GUJARATI LETTER DHA",NULL},
/* 0AA8 */ { "GUJARATI LETTER NA",NULL},
/* 0AA9 */ { NULL,NULL},
/* 0AAA */ { "GUJARATI LETTER PA",NULL},
/* 0AAB */ { "GUJARATI LETTER PHA",NULL},
/* 0AAC */ { "GUJARATI LETTER BA",NULL},
/* 0AAD */ { "GUJARATI LETTER BHA",NULL},
/* 0AAE */ { "GUJARATI LETTER MA",NULL},
/* 0AAF */ { "GUJARATI LETTER YA",NULL},
/* 0AB0 */ { "GUJARATI LETTER RA",NULL},
/* 0AB1 */ { NULL,NULL},
/* 0AB2 */ { "GUJARATI LETTER LA",NULL},
/* 0AB3 */ { "GUJARATI LETTER LLA",NULL},
/* 0AB4 */ { NULL,NULL},
/* 0AB5 */ { "GUJARATI LETTER VA",NULL},
/* 0AB6 */ { "GUJARATI LETTER SHA",NULL},
/* 0AB7 */ { "GUJARATI LETTER SSA",NULL},
/* 0AB8 */ { "GUJARATI LETTER SA",NULL},
/* 0AB9 */ { "GUJARATI LETTER HA",NULL},
/* 0ABA */ { NULL,NULL},
/* 0ABB */ { NULL,NULL},
/* 0ABC */ { "GUJARATI SIGN NUKTA","	* for extending the alphabet to new letters"},
/* 0ABD */ { "GUJARATI SIGN AVAGRAHA",NULL},
/* 0ABE */ { "GUJARATI VOWEL SIGN AA",NULL},
/* 0ABF */ { "GUJARATI VOWEL SIGN I","	* stands to the left of the consonant"},
/* 0AC0 */ { "GUJARATI VOWEL SIGN II",NULL},
/* 0AC1 */ { "GUJARATI VOWEL SIGN U",NULL},
/* 0AC2 */ { "GUJARATI VOWEL SIGN UU",NULL},
/* 0AC3 */ { "GUJARATI VOWEL SIGN VOCALIC R",NULL},
/* 0AC4 */ { "GUJARATI VOWEL SIGN VOCALIC RR",NULL},
/* 0AC5 */ { "GUJARATI VOWEL SIGN CANDRA E",NULL},
/* 0AC6 */ { NULL,NULL},
/* 0AC7 */ { "GUJARATI VOWEL SIGN E",NULL},
/* 0AC8 */ { "GUJARATI VOWEL SIGN AI",NULL},
/* 0AC9 */ { "GUJARATI VOWEL SIGN CANDRA O",NULL},
/* 0ACA */ { NULL,NULL},
/* 0ACB */ { "GUJARATI VOWEL SIGN O",NULL},
/* 0ACC */ { "GUJARATI VOWEL SIGN AU",NULL},
/* 0ACD */ { "GUJARATI SIGN VIRAMA",NULL},
/* 0ACE */ { NULL,NULL},
/* 0ACF */ { NULL,NULL},
/* 0AD0 */ { "GUJARATI OM",NULL},
/* 0AD1 */ { NULL,NULL},
/* 0AD2 */ { NULL,NULL},
/* 0AD3 */ { NULL,NULL},
/* 0AD4 */ { NULL,NULL},
/* 0AD5 */ { NULL,NULL},
/* 0AD6 */ { NULL,NULL},
/* 0AD7 */ { NULL,NULL},
/* 0AD8 */ { NULL,NULL},
/* 0AD9 */ { NULL,NULL},
/* 0ADA */ { NULL,NULL},
/* 0ADB */ { NULL,NULL},
/* 0ADC */ { NULL,NULL},
/* 0ADD */ { NULL,NULL},
/* 0ADE */ { NULL,NULL},
/* 0ADF */ { NULL,NULL},
/* 0AE0 */ { "GUJARATI LETTER VOCALIC RR",NULL},
/* 0AE1 */ { "GUJARATI LETTER VOCALIC LL",NULL},
/* 0AE2 */ { "GUJARATI VOWEL SIGN VOCALIC L",NULL},
/* 0AE3 */ { "GUJARATI VOWEL SIGN VOCALIC LL",NULL},
/* 0AE4 */ { NULL,"	x (devanagari danda - 0964)"},
/* 0AE5 */ { NULL,"	x (devanagari double danda - 0965)"},
/* 0AE6 */ { "GUJARATI DIGIT ZERO",NULL},
/* 0AE7 */ { "GUJARATI DIGIT ONE",NULL},
/* 0AE8 */ { "GUJARATI DIGIT TWO",NULL},
/* 0AE9 */ { "GUJARATI DIGIT THREE",NULL},
/* 0AEA */ { "GUJARATI DIGIT FOUR",NULL},
/* 0AEB */ { "GUJARATI DIGIT FIVE",NULL},
/* 0AEC */ { "GUJARATI DIGIT SIX",NULL},
/* 0AED */ { "GUJARATI DIGIT SEVEN",NULL},
/* 0AEE */ { "GUJARATI DIGIT EIGHT",NULL},
/* 0AEF */ { "GUJARATI DIGIT NINE",NULL},
/* 0AF0 */ { "GUJARATI ABBREVIATION SIGN",NULL},
/* 0AF1 */ { "GUJARATI RUPEE SIGN","	* preferred spelling is 0AB0 0AC2 0AF0"},
/* 0AF2 */ { NULL,NULL},
/* 0AF3 */ { NULL,NULL},
/* 0AF4 */ { NULL,NULL},
/* 0AF5 */ { NULL,NULL},
/* 0AF6 */ { NULL,NULL},
/* 0AF7 */ { NULL,NULL},
/* 0AF8 */ { NULL,NULL},
/* 0AF9 */ { "GUJARATI LETTER ZHA","	* used in transliteration of Avestan\n"
	"	x (devanagari letter zha - 0979)"},
/* 0AFA */ { "GUJARATI SIGN SUKUN",NULL},
/* 0AFB */ { "GUJARATI SIGN SHADDA",NULL},
/* 0AFC */ { "GUJARATI SIGN MADDAH",NULL},
/* 0AFD */ { "GUJARATI SIGN THREE-DOT NUKTA ABOVE",NULL},
/* 0AFE */ { "GUJARATI SIGN CIRCLE NUKTA ABOVE",NULL},
/* 0AFF */ { "GUJARATI SIGN TWO-CIRCLE NUKTA ABOVE",NULL}
};

UN_DLL_LOCAL
static const struct unicode_nameannot una_00_0B[] = {
/* 0B00 */ { NULL,NULL},
/* 0B01 */ { "ORIYA SIGN CANDRABINDU",NULL},
/* 0B02 */ { "ORIYA SIGN ANUSVARA",NULL},
/* 0B03 */ { "ORIYA SIGN VISARGA",NULL},
/* 0B04 */ { NULL,NULL},
/* 0B05 */ { "ORIYA LETTER A",NULL},
/* 0B06 */ { "ORIYA LETTER AA",NULL},
/* 0B07 */ { "ORIYA LETTER I",NULL},
/* 0B08 */ { "ORIYA LETTER II",NULL},
/* 0B09 */ { "ORIYA LETTER U",NULL},
/* 0B0A */ { "ORIYA LETTER UU",NULL},
/* 0B0B */ { "ORIYA LETTER VOCALIC R",NULL},
/* 0B0C */ { "ORIYA LETTER VOCALIC L",NULL},
/* 0B0D */ { NULL,NULL},
/* 0B0E */ { NULL,NULL},
/* 0B0F */ { "ORIYA LETTER E",NULL},
/* 0B10 */ { "ORIYA LETTER AI",NULL},
/* 0B11 */ { NULL,NULL},
/* 0B12 */ { NULL,NULL},
/* 0B13 */ { "ORIYA LETTER O",NULL},
/* 0B14 */ { "ORIYA LETTER AU",NULL},
/* 0B15 */ { "ORIYA LETTER KA",NULL},
/* 0B16 */ { "ORIYA LETTER KHA",NULL},
/* 0B17 */ { "ORIYA LETTER GA",NULL},
/* 0B18 */ { "ORIYA LETTER GHA",NULL},
/* 0B19 */ { "ORIYA LETTER NGA",NULL},
/* 0B1A */ { "ORIYA LETTER CA",NULL},
/* 0B1B */ { "ORIYA LETTER CHA",NULL},
/* 0B1C */ { "ORIYA LETTER JA",NULL},
/* 0B1D */ { "ORIYA LETTER JHA",NULL},
/* 0B1E */ { "ORIYA LETTER NYA",NULL},
/* 0B1F */ { "ORIYA LETTER TTA",NULL},
/* 0B20 */ { "ORIYA LETTER TTHA",NULL},
/* 0B21 */ { "ORIYA LETTER DDA",NULL},
/* 0B22 */ { "ORIYA LETTER DDHA",NULL},
/* 0B23 */ { "ORIYA LETTER NNA",NULL},
/* 0B24 */ { "ORIYA LETTER TA",NULL},
/* 0B25 */ { "ORIYA LETTER THA",NULL},
/* 0B26 */ { "ORIYA LETTER DA",NULL},
/* 0B27 */ { "ORIYA LETTER DHA",NULL},
/* 0B28 */ { "ORIYA LETTER NA",NULL},
/* 0B29 */ { NULL,NULL},
/* 0B2A */ { "ORIYA LETTER PA",NULL},
/* 0B2B */ { "ORIYA LETTER PHA",NULL},
/* 0B2C */ { "ORIYA LETTER BA","	x (oriya letter va - 0B35)"},
/* 0B2D */ { "ORIYA LETTER BHA",NULL},
/* 0B2E */ { "ORIYA LETTER MA",NULL},
/* 0B2F */ { "ORIYA LETTER YA","	= ja"},
/* 0B30 */ { "ORIYA LETTER RA",NULL},
/* 0B31 */ { NULL,NULL},
/* 0B32 */ { "ORIYA LETTER LA",NULL},
/* 0B33 */ { "ORIYA LETTER LLA",NULL},
/* 0B34 */ { NULL,NULL},
/* 0B35 */ { "ORIYA LETTER VA","	* Kui uses an alternate glyph with the dot above the small circle\n"
	"	x (oriya letter ba - 0B2C)"},
/* 0B36 */ { "ORIYA LETTER SHA",NULL},
/* 0B37 */ { "ORIYA LETTER SSA",NULL},
/* 0B38 */ { "ORIYA LETTER SA",NULL},
/* 0B39 */ { "ORIYA LETTER HA",NULL},
/* 0B3A */ { NULL,NULL},
/* 0B3B */ { NULL,NULL},
/* 0B3C */ { "ORIYA SIGN NUKTA","	* for extending the alphabet to new letters"},
/* 0B3D */ { "ORIYA SIGN AVAGRAHA",NULL},
/* 0B3E */ { "ORIYA VOWEL SIGN AA",NULL},
/* 0B3F */ { "ORIYA VOWEL SIGN I",NULL},
/* 0B40 */ { "ORIYA VOWEL SIGN II",NULL},
/* 0B41 */ { "ORIYA VOWEL SIGN U",NULL},
/* 0B42 */ { "ORIYA VOWEL SIGN UU",NULL},
/* 0B43 */ { "ORIYA VOWEL SIGN VOCALIC R",NULL},
/* 0B44 */ { "ORIYA VOWEL SIGN VOCALIC RR",NULL},
/* 0B45 */ { NULL,NULL},
/* 0B46 */ { NULL,NULL},
/* 0B47 */ { "ORIYA VOWEL SIGN E","	* stands to the left of the consonant"},
/* 0B48 */ { "ORIYA VOWEL SIGN AI","	* pieces left of and above the consonant\n"
	"	: 0B47 0B56"},
/* 0B49 */ { NULL,NULL},
/* 0B4A */ { NULL,NULL},
/* 0B4B */ { "ORIYA VOWEL SIGN O","	: 0B47 0B3E"},
/* 0B4C */ { "ORIYA VOWEL SIGN AU","	: 0B47 0B57"},
/* 0B4D */ { "ORIYA SIGN VIRAMA",NULL},
/* 0B4E */ { NULL,NULL},
/* 0B4F */ { NULL,NULL},
/* 0B50 */ { NULL,NULL},
/* 0B51 */ { NULL,NULL},
/* 0B52 */ { NULL,NULL},
/* 0B53 */ { NULL,NULL},
/* 0B54 */ { NULL,NULL},
/* 0B55 */ { "ORIYA SIGN OVERLINE","	* Kuvi, Kui"},
/* 0B56 */ { "ORIYA AI LENGTH MARK",NULL},
/* 0B57 */ { "ORIYA AU LENGTH MARK",NULL},
/* 0B58 */ { NULL,NULL},
/* 0B59 */ { NULL,NULL},
/* 0B5A */ { NULL,NULL},
/* 0B5B */ { NULL,NULL},
/* 0B5C */ { "ORIYA LETTER RRA","	= dda\n"
	"	: 0B21 0B3C"},
/* 0B5D */ { "ORIYA LETTER RHA","	= ddha\n"
	"	: 0B22 0B3C"},
/* 0B5E */ { NULL,NULL},
/* 0B5F */ { "ORIYA LETTER YYA","	= ya"},
/* 0B60 */ { "ORIYA LETTER VOCALIC RR",NULL},
/* 0B61 */ { "ORIYA LETTER VOCALIC LL",NULL},
/* 0B62 */ { "ORIYA VOWEL SIGN VOCALIC L",NULL},
/* 0B63 */ { "ORIYA VOWEL SIGN VOCALIC LL",NULL},
/* 0B64 */ { NULL,"	x (devanagari danda - 0964)"},
/* 0B65 */ { NULL,"	x (devanagari double danda - 0965)"},
/* 0B66 */ { "ORIYA DIGIT ZERO",NULL},
/* 0B67 */ { "ORIYA DIGIT ONE",NULL},
/* 0B68 */ { "ORIYA DIGIT TWO",NULL},
/* 0B69 */ { "ORIYA DIGIT THREE",NULL},
/* 0B6A */ { "ORIYA DIGIT FOUR",NULL},
/* 0B6B */ { "ORIYA DIGIT FIVE",NULL},
/* 0B6C */ { "ORIYA DIGIT SIX",NULL},
/* 0B6D */ { "ORIYA DIGIT SEVEN",NULL},
/* 0B6E */ { "ORIYA DIGIT EIGHT",NULL},
/* 0B6F */ { "ORIYA DIGIT NINE",NULL},
/* 0B70 */ { "ORIYA ISSHAR",NULL},
/* 0B71 */ { "ORIYA LETTER WA","	x (oriya letter o - 0B13)\n"
	"	x (oriya letter va - 0B35)"},
/* 0B72 */ { "ORIYA FRACTION ONE QUARTER",NULL},
/* 0B73 */ { "ORIYA FRACTION ONE HALF",NULL},
/* 0B74 */ { "ORIYA FRACTION THREE QUARTERS",NULL},
/* 0B75 */ { "ORIYA FRACTION ONE SIXTEENTH",NULL},
/* 0B76 */ { "ORIYA FRACTION ONE EIGHTH",NULL},
/* 0B77 */ { "ORIYA FRACTION THREE SIXTEENTHS",NULL},
/* 0B78 */ { NULL,NULL},
/* 0B79 */ { NULL,NULL},
/* 0B7A */ { NULL,NULL},
/* 0B7B */ { NULL,NULL},
/* 0B7C */ { NULL,NULL},
/* 0B7D */ { NULL,NULL},
/* 0B7E */ { NULL,NULL},
/* 0B7F */ { NULL,NULL},
/* 0B80 */ { NULL,NULL},
/* 0B81 */ { NULL,NULL},
/* 0B82 */ { "TAMIL SIGN ANUSVARA","	* not used in Tamil"},
/* 0B83 */ { "TAMIL SIGN VISARGA","	= aytham\n"
	"	* just as for the Tamil pulli, the glyph for aytham may use either dots or rings"},
/* 0B84 */ { NULL,NULL},
/* 0B85 */ { "TAMIL LETTER A",NULL},
/* 0B86 */ { "TAMIL LETTER AA",NULL},
/* 0B87 */ { "TAMIL LETTER I",NULL},
/* 0B88 */ { "TAMIL LETTER II",NULL},
/* 0B89 */ { "TAMIL LETTER U",NULL},
/* 0B8A */ { "TAMIL LETTER UU",NULL},
/* 0B8B */ { NULL,NULL},
/* 0B8C */ { NULL,NULL},
/* 0B8D */ { NULL,NULL},
/* 0B8E */ { "TAMIL LETTER E",NULL},
/* 0B8F */ { "TAMIL LETTER EE",NULL},
/* 0B90 */ { "TAMIL LETTER AI",NULL},
/* 0B91 */ { NULL,NULL},
/* 0B92 */ { "TAMIL LETTER O",NULL},
/* 0B93 */ { "TAMIL LETTER OO",NULL},
/* 0B94 */ { "TAMIL LETTER AU","	: 0B92 0BD7"},
/* 0B95 */ { "TAMIL LETTER KA",NULL},
/* 0B96 */ { NULL,NULL},
/* 0B97 */ { NULL,NULL},
/* 0B98 */ { NULL,NULL},
/* 0B99 */ { "TAMIL LETTER NGA","	* also denotes the measure kuruni/marakkaal\n"
	"	x (tamil sign kuruni - 11FDA)"},
/* 0B9A */ { "TAMIL LETTER CA",NULL},
/* 0B9B */ { NULL,NULL},
/* 0B9C */ { "TAMIL LETTER JA",NULL},
/* 0B9D */ { NULL,NULL},
/* 0B9E */ { "TAMIL LETTER NYA",NULL},
/* 0B9F */ { "TAMIL LETTER TTA",NULL},
/* 0BA0 */ { NULL,NULL},
/* 0BA1 */ { NULL,NULL},
/* 0BA2 */ { NULL,NULL},
/* 0BA3 */ { "TAMIL LETTER NNA",NULL},
/* 0BA4 */ { "TAMIL LETTER TA","	* also denotes the measure thuuni\n"
	"	x (tamil sign mukkuruni - 11FDC)"},
/* 0BA5 */ { NULL,NULL},
/* 0BA6 */ { NULL,NULL},
/* 0BA7 */ { NULL,NULL},
/* 0BA8 */ { "TAMIL LETTER NA",NULL},
/* 0BA9 */ { "TAMIL LETTER NNNA",NULL},
/* 0BAA */ { "TAMIL LETTER PA","	= maa\n"
	"	x (tamil fraction one twentieth - 11FC8)"},
/* 0BAB */ { NULL,NULL},
/* 0BAC */ { NULL,NULL},
/* 0BAD */ { NULL,NULL},
/* 0BAE */ { "TAMIL LETTER MA",NULL},
/* 0BAF */ { "TAMIL LETTER YA",NULL},
/* 0BB0 */ { "TAMIL LETTER RA",NULL},
/* 0BB1 */ { "TAMIL LETTER RRA",NULL},
/* 0BB2 */ { "TAMIL LETTER LA",NULL},
/* 0BB3 */ { "TAMIL LETTER LLA","	* also denotes the measure kalam\n"
	"	x (tamil sign mukkuruni - 11FDC)"},
/* 0BB4 */ { "TAMIL LETTER LLLA",NULL},
/* 0BB5 */ { "TAMIL LETTER VA","	x (tamil fraction one quarter - 11FD0)"},
/* 0BB6 */ { "TAMIL LETTER SHA",NULL},
/* 0BB7 */ { "TAMIL LETTER SSA",NULL},
/* 0BB8 */ { "TAMIL LETTER SA",NULL},
/* 0BB9 */ { "TAMIL LETTER HA",NULL},
/* 0BBA */ { NULL,NULL},
/* 0BBB */ { NULL,NULL},
/* 0BBC */ { NULL,NULL},
/* 0BBD */ { NULL,NULL},
/* 0BBE */ { "TAMIL VOWEL SIGN AA",NULL},
/* 0BBF */ { "TAMIL VOWEL SIGN I",NULL},
/* 0BC0 */ { "TAMIL VOWEL SIGN II",NULL},
/* 0BC1 */ { "TAMIL VOWEL SIGN U",NULL},
/* 0BC2 */ { "TAMIL VOWEL SIGN UU",NULL},
/* 0BC3 */ { NULL,NULL},
/* 0BC4 */ { NULL,NULL},
/* 0BC5 */ { NULL,NULL},
/* 0BC6 */ { "TAMIL VOWEL SIGN E","	* stands to the left of the consonant"},
/* 0BC7 */ { "TAMIL VOWEL SIGN EE","	* stands to the left of the consonant"},
/* 0BC8 */ { "TAMIL VOWEL SIGN AI","	* stands to the left of the consonant"},
/* 0BC9 */ { NULL,NULL},
/* 0BCA */ { "TAMIL VOWEL SIGN O","	: 0BC6 0BBE"},
/* 0BCB */ { "TAMIL VOWEL SIGN OO","	: 0BC7 0BBE"},
/* 0BCC */ { "TAMIL VOWEL SIGN AU","	: 0BC6 0BD7"},
/* 0BCD */ { "TAMIL SIGN VIRAMA","	= pulli"},
/* 0BCE */ { NULL,NULL},
/* 0BCF */ { NULL,NULL},
/* 0BD0 */ { "TAMIL OM",NULL},
/* 0BD1 */ { NULL,NULL},
/* 0BD2 */ { NULL,NULL},
/* 0BD3 */ { NULL,NULL},
/* 0BD4 */ { NULL,NULL},
/* 0BD5 */ { NULL,NULL},
/* 0BD6 */ { NULL,NULL},
/* 0BD7 */ { "TAMIL AU LENGTH MARK",NULL},
/* 0BD8 */ { NULL,NULL},
/* 0BD9 */ { NULL,NULL},
/* 0BDA */ { NULL,NULL},
/* 0BDB */ { NULL,NULL},
/* 0BDC */ { NULL,NULL},
/* 0BDD */ { NULL,NULL},
/* 0BDE */ { NULL,NULL},
/* 0BDF */ { NULL,NULL},
/* 0BE0 */ { NULL,NULL},
/* 0BE1 */ { NULL,NULL},
/* 0BE2 */ { NULL,NULL},
/* 0BE3 */ { NULL,NULL},
/* 0BE4 */ { NULL,"	x (devanagari danda - 0964)"},
/* 0BE5 */ { NULL,"	x (devanagari double danda - 0965)"},
/* 0BE6 */ { "TAMIL DIGIT ZERO",NULL},
/* 0BE7 */ { "TAMIL DIGIT ONE",NULL},
/* 0BE8 */ { "TAMIL DIGIT TWO",NULL},
/* 0BE9 */ { "TAMIL DIGIT THREE",NULL},
/* 0BEA */ { "TAMIL DIGIT FOUR",NULL},
/* 0BEB */ { "TAMIL DIGIT FIVE",NULL},
/* 0BEC */ { "TAMIL DIGIT SIX",NULL},
/* 0BED */ { "TAMIL DIGIT SEVEN",NULL},
/* 0BEE */ { "TAMIL DIGIT EIGHT",NULL},
/* 0BEF */ { "TAMIL DIGIT NINE",NULL},
/* 0BF0 */ { "TAMIL NUMBER TEN",NULL},
/* 0BF1 */ { "TAMIL NUMBER ONE HUNDRED",NULL},
/* 0BF2 */ { "TAMIL NUMBER ONE THOUSAND",NULL},
/* 0BF3 */ { "TAMIL DAY SIGN","	= naal, naali/padi\n"
	"	* denotes a measure of grain that equals 2 uri or 4 ulakku\n"
	"	x (tamil sign muuvuzhakku - 11FD9)\n"
	"	= pillaiyaar suli\n"
	"	* denotes auspiciousness"},
/* 0BF4 */ { "TAMIL MONTH SIGN","	= maatham"},
/* 0BF5 */ { "TAMIL YEAR SIGN","	= varudam"},
/* 0BF6 */ { "TAMIL DEBIT SIGN","	= patru"},
/* 0BF7 */ { "TAMIL CREDIT SIGN","	= eduppu\n"
	"	* denotes incoming cash which is set aside for unknown expenses\n"
	"	* sometimes used as the credit sign; the traditional credit sign is distinct\n"
	"	x (tamil traditional credit sign - 11FE8)"},
/* 0BF8 */ { "TAMIL AS ABOVE SIGN","	= merpadi"},
/* 0BF9 */ { "TAMIL RUPEE SIGN","	= rupai\n"
	"	x (tamil land sign - 11FE6)"},
/* 0BFA */ { "TAMIL NUMBER SIGN","	= enn, niluvai\n"
	"	* denotes balance\n"
	"	* sometimes used as the number sign; the traditional number sign is distinct\n"
	"	x (tamil traditional number sign - 11FE9)"},
/* 0BFB */ { NULL,NULL},
/* 0BFC */ { NULL,NULL},
/* 0BFD */ { NULL,NULL},
/* 0BFE */ { NULL,NULL},
/* 0BFF */ { NULL,NULL}
};

UN_DLL_LOCAL
static const struct unicode_nameannot una_00_0C[] = {
/* 0C00 */ { "TELUGU SIGN COMBINING CANDRABINDU ABOVE",NULL},
/* 0C01 */ { "TELUGU SIGN CANDRABINDU","	= arasunna\n"
	"	* indicates a contextually elided nasal"},
/* 0C02 */ { "TELUGU SIGN ANUSVARA","	= sunna\n"
	"	* used in Prakrit language texts to indicate gemination of the following consonant"},
/* 0C03 */ { "TELUGU SIGN VISARGA",NULL},
/* 0C04 */ { "TELUGU SIGN COMBINING ANUSVARA ABOVE","	* used in Prakrit language texts for the linguistic anusvara"},
/* 0C05 */ { "TELUGU LETTER A",NULL},
/* 0C06 */ { "TELUGU LETTER AA",NULL},
/* 0C07 */ { "TELUGU LETTER I",NULL},
/* 0C08 */ { "TELUGU LETTER II",NULL},
/* 0C09 */ { "TELUGU LETTER U",NULL},
/* 0C0A */ { "TELUGU LETTER UU",NULL},
/* 0C0B */ { "TELUGU LETTER VOCALIC R",NULL},
/* 0C0C */ { "TELUGU LETTER VOCALIC L",NULL},
/* 0C0D */ { NULL,NULL},
/* 0C0E */ { "TELUGU LETTER E",NULL},
/* 0C0F */ { "TELUGU LETTER EE",NULL},
/* 0C10 */ { "TELUGU LETTER AI",NULL},
/* 0C11 */ { NULL,NULL},
/* 0C12 */ { "TELUGU LETTER O",NULL},
/* 0C13 */ { "TELUGU LETTER OO",NULL},
/* 0C14 */ { "TELUGU LETTER AU",NULL},
/* 0C15 */ { "TELUGU LETTER KA",NULL},
/* 0C16 */ { "TELUGU LETTER KHA",NULL},
/* 0C17 */ { "TELUGU LETTER GA",NULL},
/* 0C18 */ { "TELUGU LETTER GHA",NULL},
/* 0C19 */ { "TELUGU LETTER NGA",NULL},
/* 0C1A */ { "TELUGU LETTER CA",NULL},
/* 0C1B */ { "TELUGU LETTER CHA",NULL},
/* 0C1C */ { "TELUGU LETTER JA",NULL},
/* 0C1D */ { "TELUGU LETTER JHA",NULL},
/* 0C1E */ { "TELUGU LETTER NYA",NULL},
/* 0C1F */ { "TELUGU LETTER TTA",NULL},
/* 0C20 */ { "TELUGU LETTER TTHA",NULL},
/* 0C21 */ { "TELUGU LETTER DDA",NULL},
/* 0C22 */ { "TELUGU LETTER DDHA",NULL},
/* 0C23 */ { "TELUGU LETTER NNA",NULL},
/* 0C24 */ { "TELUGU LETTER TA",NULL},
/* 0C25 */ { "TELUGU LETTER THA",NULL},
/* 0C26 */ { "TELUGU LETTER DA",NULL},
/* 0C27 */ { "TELUGU LETTER DHA",NULL},
/* 0C28 */ { "TELUGU LETTER NA",NULL},
/* 0C29 */ { NULL,NULL},
/* 0C2A */ { "TELUGU LETTER PA",NULL},
/* 0C2B */ { "TELUGU LETTER PHA",NULL},
/* 0C2C */ { "TELUGU LETTER BA",NULL},
/* 0C2D */ { "TELUGU LETTER BHA",NULL},
/* 0C2E */ { "TELUGU LETTER MA",NULL},
/* 0C2F */ { "TELUGU LETTER YA",NULL},
/* 0C30 */ { "TELUGU LETTER RA",NULL},
/* 0C31 */ { "TELUGU LETTER RRA",NULL},
/* 0C32 */ { "TELUGU LETTER LA",NULL},
/* 0C33 */ { "TELUGU LETTER LLA",NULL},
/* 0C34 */ { "TELUGU LETTER LLLA",NULL},
/* 0C35 */ { "TELUGU LETTER VA",NULL},
/* 0C36 */ { "TELUGU LETTER SHA",NULL},
/* 0C37 */ { "TELUGU LETTER SSA",NULL},
/* 0C38 */ { "TELUGU LETTER SA",NULL},
/* 0C39 */ { "TELUGU LETTER HA",NULL},
/* 0C3A */ { NULL,NULL},
/* 0C3B */ { NULL,NULL},
/* 0C3C */ { "TELUGU SIGN NUKTA","	* used for Tamil transliteration and Perso-Arabic phonemes\n"
	"	* can also appear as a large dot, should be placed sufficiently below the baseline of a consonant to avoid confusion/collision with the aspiration marker"},
/* 0C3D */ { "TELUGU SIGN AVAGRAHA",NULL},
/* 0C3E */ { "TELUGU VOWEL SIGN AA",NULL},
/* 0C3F */ { "TELUGU VOWEL SIGN I",NULL},
/* 0C40 */ { "TELUGU VOWEL SIGN II",NULL},
/* 0C41 */ { "TELUGU VOWEL SIGN U",NULL},
/* 0C42 */ { "TELUGU VOWEL SIGN UU",NULL},
/* 0C43 */ { "TELUGU VOWEL SIGN VOCALIC R",NULL},
/* 0C44 */ { "TELUGU VOWEL SIGN VOCALIC RR",NULL},
/* 0C45 */ { NULL,NULL},
/* 0C46 */ { "TELUGU VOWEL SIGN E",NULL},
/* 0C47 */ { "TELUGU VOWEL SIGN EE",NULL},
/* 0C48 */ { "TELUGU VOWEL SIGN AI","	: 0C46 0C56"},
/* 0C49 */ { NULL,NULL},
/* 0C4A */ { "TELUGU VOWEL SIGN O",NULL},
/* 0C4B */ { "TELUGU VOWEL SIGN OO",NULL},
/* 0C4C */ { "TELUGU VOWEL SIGN AU",NULL},
/* 0C4D */ { "TELUGU SIGN VIRAMA","	= halant (the preferred name)"},
/* 0C4E */ { NULL,NULL},
/* 0C4F */ { NULL,NULL},
/* 0C50 */ { NULL,NULL},
/* 0C51 */ { NULL,NULL},
/* 0C52 */ { NULL,NULL},
/* 0C53 */ { NULL,NULL},
/* 0C54 */ { NULL,NULL},
/* 0C55 */ { "TELUGU LENGTH MARK",NULL},
/* 0C56 */ { "TELUGU AI LENGTH MARK",NULL},
/* 0C57 */ { NULL,NULL},
/* 0C58 */ { "TELUGU LETTER TSA",NULL},
/* 0C59 */ { "TELUGU LETTER DZA",NULL},
/* 0C5A */ { "TELUGU LETTER RRRA","	* letter for an alveolar consonant whose exact phonetic value is not known"},
/* 0C5B */ { NULL,NULL},
/* 0C5C */ { NULL,NULL},
/* 0C5D */ { "TELUGU LETTER NAKAARA POLLU","	* vowelless form of na\n"
	"	x (kannada letter nakaara pollu - 0CDD)\n"
	"	x (malayalam letter chillu n - 0D7B)"},
/* 0C5E */ { NULL,NULL},
/* 0C5F */ { NULL,NULL},
/* 0C60 */ { "TELUGU LETTER VOCALIC RR",NULL},
/* 0C61 */ { "TELUGU LETTER VOCALIC LL",NULL},
/* 0C62 */ { "TELUGU VOWEL SIGN VOCALIC L",NULL},
/* 0C63 */ { "TELUGU VOWEL SIGN VOCALIC LL",NULL},
/* 0C64 */ { NULL,"	x (devanagari danda - 0964)"},
/* 0C65 */ { NULL,"	x (devanagari double danda - 0965)"},
/* 0C66 */ { "TELUGU DIGIT ZERO",NULL},
/* 0C67 */ { "TELUGU DIGIT ONE",NULL},
/* 0C68 */ { "TELUGU DIGIT TWO",NULL},
/* 0C69 */ { "TELUGU DIGIT THREE",NULL},
/* 0C6A */ { "TELUGU DIGIT FOUR",NULL},
/* 0C6B */ { "TELUGU DIGIT FIVE",NULL},
/* 0C6C */ { "TELUGU DIGIT SIX",NULL},
/* 0C6D */ { "TELUGU DIGIT SEVEN",NULL},
/* 0C6E */ { "TELUGU DIGIT EIGHT",NULL},
/* 0C6F */ { "TELUGU DIGIT NINE",NULL},
/* 0C70 */ { NULL,NULL},
/* 0C71 */ { NULL,NULL},
/* 0C72 */ { NULL,NULL},
/* 0C73 */ { NULL,NULL},
/* 0C74 */ { NULL,NULL},
/* 0C75 */ { NULL,NULL},
/* 0C76 */ { NULL,NULL},
/* 0C77 */ { "TELUGU SIGN SIDDHAM","	= siddhirastu\n"
	"	* used at the beginning of texts as an invocation\n"
	"	x (devanagari sign siddham - A8FC)\n"
	"	x (sharada sign siddham - 111DB)"},
/* 0C78 */ { "TELUGU FRACTION DIGIT ZERO FOR ODD POWERS OF FOUR",NULL},
/* 0C79 */ { "TELUGU FRACTION DIGIT ONE FOR ODD POWERS OF FOUR",NULL},
/* 0C7A */ { "TELUGU FRACTION DIGIT TWO FOR ODD POWERS OF FOUR",NULL},
/* 0C7B */ { "TELUGU FRACTION DIGIT THREE FOR ODD POWERS OF FOUR",NULL},
/* 0C7C */ { "TELUGU FRACTION DIGIT ONE FOR EVEN POWERS OF FOUR",NULL},
/* 0C7D */ { "TELUGU FRACTION DIGIT TWO FOR EVEN POWERS OF FOUR",NULL},
/* 0C7E */ { "TELUGU FRACTION DIGIT THREE FOR EVEN POWERS OF FOUR",NULL},
/* 0C7F */ { "TELUGU SIGN TUUMU",NULL},
/* 0C80 */ { "KANNADA SIGN SPACING CANDRABINDU",NULL},
/* 0C81 */ { "KANNADA SIGN CANDRABINDU",NULL},
/* 0C82 */ { "KANNADA SIGN ANUSVARA","	* used in Konkani, Havyaka Kannada (Havigannada) and Awadhi texts to indicate regular anusvara"},
/* 0C83 */ { "KANNADA SIGN VISARGA",NULL},
/* 0C84 */ { "KANNADA SIGN SIDDHAM","	= siddhirastu\n"
	"	* used at the beginning of texts as an invocation\n"
	"	x (devanagari sign siddham - A8FC)\n"
	"	x (sharada sign siddham - 111DB)"},
/* 0C85 */ { "KANNADA LETTER A",NULL},
/* 0C86 */ { "KANNADA LETTER AA",NULL},
/* 0C87 */ { "KANNADA LETTER I",NULL},
/* 0C88 */ { "KANNADA LETTER II",NULL},
/* 0C89 */ { "KANNADA LETTER U",NULL},
/* 0C8A */ { "KANNADA LETTER UU",NULL},
/* 0C8B */ { "KANNADA LETTER VOCALIC R",NULL},
/* 0C8C */ { "KANNADA LETTER VOCALIC L",NULL},
/* 0C8D */ { NULL,NULL},
/* 0C8E */ { "KANNADA LETTER E",NULL},
/* 0C8F */ { "KANNADA LETTER EE",NULL},
/* 0C90 */ { "KANNADA LETTER AI",NULL},
/* 0C91 */ { NULL,NULL},
/* 0C92 */ { "KANNADA LETTER O",NULL},
/* 0C93 */ { "KANNADA LETTER OO",NULL},
/* 0C94 */ { "KANNADA LETTER AU",NULL},
/* 0C95 */ { "KANNADA LETTER KA",NULL},
/* 0C96 */ { "KANNADA LETTER KHA",NULL},
/* 0C97 */ { "KANNADA LETTER GA",NULL},
/* 0C98 */ { "KANNADA LETTER GHA",NULL},
/* 0C99 */ { "KANNADA LETTER NGA",NULL},
/* 0C9A */ { "KANNADA LETTER CA",NULL},
/* 0C9B */ { "KANNADA LETTER CHA",NULL},
/* 0C9C */ { "KANNADA LETTER JA",NULL},
/* 0C9D */ { "KANNADA LETTER JHA",NULL},
/* 0C9E */ { "KANNADA LETTER NYA",NULL},
/* 0C9F */ { "KANNADA LETTER TTA",NULL},
/* 0CA0 */ { "KANNADA LETTER TTHA",NULL},
/* 0CA1 */ { "KANNADA LETTER DDA",NULL},
/* 0CA2 */ { "KANNADA LETTER DDHA",NULL},
/* 0CA3 */ { "KANNADA LETTER NNA",NULL},
/* 0CA4 */ { "KANNADA LETTER TA",NULL},
/* 0CA5 */ { "KANNADA LETTER THA",NULL},
/* 0CA6 */ { "KANNADA LETTER DA",NULL},
/* 0CA7 */ { "KANNADA LETTER DHA",NULL},
/* 0CA8 */ { "KANNADA LETTER NA",NULL},
/* 0CA9 */ { NULL,NULL},
/* 0CAA */ { "KANNADA LETTER PA",NULL},
/* 0CAB */ { "KANNADA LETTER PHA",NULL},
/* 0CAC */ { "KANNADA LETTER BA",NULL},
/* 0CAD */ { "KANNADA LETTER BHA",NULL},
/* 0CAE */ { "KANNADA LETTER MA",NULL},
/* 0CAF */ { "KANNADA LETTER YA",NULL},
/* 0CB0 */ { "KANNADA LETTER RA",NULL},
/* 0CB1 */ { "KANNADA LETTER RRA",NULL},
/* 0CB2 */ { "KANNADA LETTER LA",NULL},
/* 0CB3 */ { "KANNADA LETTER LLA",NULL},
/* 0CB4 */ { NULL,NULL},
/* 0CB5 */ { "KANNADA LETTER VA",NULL},
/* 0CB6 */ { "KANNADA LETTER SHA",NULL},
/* 0CB7 */ { "KANNADA LETTER SSA",NULL},
/* 0CB8 */ { "KANNADA LETTER SA",NULL},
/* 0CB9 */ { "KANNADA LETTER HA",NULL},
/* 0CBA */ { NULL,NULL},
/* 0CBB */ { NULL,NULL},
/* 0CBC */ { "KANNADA SIGN NUKTA",NULL},
/* 0CBD */ { "KANNADA SIGN AVAGRAHA",NULL},
/* 0CBE */ { "KANNADA VOWEL SIGN AA",NULL},
/* 0CBF */ { "KANNADA VOWEL SIGN I",NULL},
/* 0CC0 */ { "KANNADA VOWEL SIGN II","	: 0CBF 0CD5"},
/* 0CC1 */ { "KANNADA VOWEL SIGN U",NULL},
/* 0CC2 */ { "KANNADA VOWEL SIGN UU",NULL},
/* 0CC3 */ { "KANNADA VOWEL SIGN VOCALIC R",NULL},
/* 0CC4 */ { "KANNADA VOWEL SIGN VOCALIC RR",NULL},
/* 0CC5 */ { NULL,NULL},
/* 0CC6 */ { "KANNADA VOWEL SIGN E",NULL},
/* 0CC7 */ { "KANNADA VOWEL SIGN EE","	: 0CC6 0CD5"},
/* 0CC8 */ { "KANNADA VOWEL SIGN AI","	: 0CC6 0CD6"},
/* 0CC9 */ { NULL,NULL},
/* 0CCA */ { "KANNADA VOWEL SIGN O","	: 0CC6 0CC2"},
/* 0CCB */ { "KANNADA VOWEL SIGN OO","	: 0CCA 0CD5"},
/* 0CCC */ { "KANNADA VOWEL SIGN AU",NULL},
/* 0CCD */ { "KANNADA SIGN VIRAMA","	* preferred name is halant"},
/* 0CCE */ { NULL,NULL},
/* 0CCF */ { NULL,NULL},
/* 0CD0 */ { NULL,NULL},
/* 0CD1 */ { NULL,NULL},
/* 0CD2 */ { NULL,NULL},
/* 0CD3 */ { NULL,NULL},
/* 0CD4 */ { NULL,NULL},
/* 0CD5 */ { "KANNADA LENGTH MARK",NULL},
/* 0CD6 */ { "KANNADA AI LENGTH MARK",NULL},
/* 0CD7 */ { NULL,NULL},
/* 0CD8 */ { NULL,NULL},
/* 0CD9 */ { NULL,NULL},
/* 0CDA */ { NULL,NULL},
/* 0CDB */ { NULL,NULL},
/* 0CDC */ { NULL,NULL},
/* 0CDD */ { "KANNADA LETTER NAKAARA POLLU","	* vowelless form of na\n"
	"	x (telugu letter nakaara pollu - 0C5D)\n"
	"	x (malayalam letter chillu n - 0D7B)"},
/* 0CDE */ { "KANNADA LETTER FA","	% KANNADA LETTER LLLA\n"
	"	* obsolete historic letter\n"
	"	* name is a mistake for LLLA"},
/* 0CDF */ { NULL,NULL},
/* 0CE0 */ { "KANNADA LETTER VOCALIC RR",NULL},
/* 0CE1 */ { "KANNADA LETTER VOCALIC LL",NULL},
/* 0CE2 */ { "KANNADA VOWEL SIGN VOCALIC L",NULL},
/* 0CE3 */ { "KANNADA VOWEL SIGN VOCALIC LL",NULL},
/* 0CE4 */ { NULL,"	x (devanagari danda - 0964)"},
/* 0CE5 */ { NULL,"	x (devanagari double danda - 0965)"},
/* 0CE6 */ { "KANNADA DIGIT ZERO",NULL},
/* 0CE7 */ { "KANNADA DIGIT ONE",NULL},
/* 0CE8 */ { "KANNADA DIGIT TWO",NULL},
/* 0CE9 */ { "KANNADA DIGIT THREE",NULL},
/* 0CEA */ { "KANNADA DIGIT FOUR",NULL},
/* 0CEB */ { "KANNADA DIGIT FIVE",NULL},
/* 0CEC */ { "KANNADA DIGIT SIX",NULL},
/* 0CED */ { "KANNADA DIGIT SEVEN",NULL},
/* 0CEE */ { "KANNADA DIGIT EIGHT",NULL},
/* 0CEF */ { "KANNADA DIGIT NINE",NULL},
/* 0CF0 */ { NULL,NULL},
/* 0CF1 */ { "KANNADA SIGN JIHVAMULIYA","	x (vedic sign jihvamuliya - 1CF5)"},
/* 0CF2 */ { "KANNADA SIGN UPADHMANIYA","	x (vedic sign upadhmaniya - 1CF6)"},
/* 0CF3 */ { "KANNADA SIGN COMBINING ANUSVARA ABOVE RIGHT","	* used in Konkani, Havyaka Kannada (Havigannada) and Awadhi texts to indicate anunāsikā"},
/* 0CF4 */ { NULL,NULL},
/* 0CF5 */ { NULL,NULL},
/* 0CF6 */ { NULL,NULL},
/* 0CF7 */ { NULL,NULL},
/* 0CF8 */ { NULL,NULL},
/* 0CF9 */ { NULL,NULL},
/* 0CFA */ { NULL,NULL},
/* 0CFB */ { NULL,NULL},
/* 0CFC */ { NULL,NULL},
/* 0CFD */ { NULL,NULL},
/* 0CFE */ { NULL,NULL},
/* 0CFF */ { NULL,NULL}
};

UN_DLL_LOCAL
static const struct unicode_nameannot una_00_0D[] = {
/* 0D00 */ { "MALAYALAM SIGN COMBINING ANUSVARA ABOVE",NULL},
/* 0D01 */ { "MALAYALAM SIGN CANDRABINDU",NULL},
/* 0D02 */ { "MALAYALAM SIGN ANUSVARA","	* used in Prakrit language texts to indicate gemination of the following consonant"},
/* 0D03 */ { "MALAYALAM SIGN VISARGA",NULL},
/* 0D04 */ { "MALAYALAM LETTER VEDIC ANUSVARA",NULL},
/* 0D05 */ { "MALAYALAM LETTER A",NULL},
/* 0D06 */ { "MALAYALAM LETTER AA",NULL},
/* 0D07 */ { "MALAYALAM LETTER I",NULL},
/* 0D08 */ { "MALAYALAM LETTER II",NULL},
/* 0D09 */ { "MALAYALAM LETTER U",NULL},
/* 0D0A */ { "MALAYALAM LETTER UU",NULL},
/* 0D0B */ { "MALAYALAM LETTER VOCALIC R",NULL},
/* 0D0C */ { "MALAYALAM LETTER VOCALIC L",NULL},
/* 0D0D */ { NULL,NULL},
/* 0D0E */ { "MALAYALAM LETTER E",NULL},
/* 0D0F */ { "MALAYALAM LETTER EE",NULL},
/* 0D10 */ { "MALAYALAM LETTER AI",NULL},
/* 0D11 */ { NULL,NULL},
/* 0D12 */ { "MALAYALAM LETTER O",NULL},
/* 0D13 */ { "MALAYALAM LETTER OO",NULL},
/* 0D14 */ { "MALAYALAM LETTER AU",NULL},
/* 0D15 */ { "MALAYALAM LETTER KA",NULL},
/* 0D16 */ { "MALAYALAM LETTER KHA",NULL},
/* 0D17 */ { "MALAYALAM LETTER GA",NULL},
/* 0D18 */ { "MALAYALAM LETTER GHA",NULL},
/* 0D19 */ { "MALAYALAM LETTER NGA",NULL},
/* 0D1A */ { "MALAYALAM LETTER CA","	= cha"},
/* 0D1B */ { "MALAYALAM LETTER CHA","	= chha"},
/* 0D1C */ { "MALAYALAM LETTER JA",NULL},
/* 0D1D */ { "MALAYALAM LETTER JHA",NULL},
/* 0D1E */ { "MALAYALAM LETTER NYA","	= nha"},
/* 0D1F */ { "MALAYALAM LETTER TTA","	= ta"},
/* 0D20 */ { "MALAYALAM LETTER TTHA","	= tta"},
/* 0D21 */ { "MALAYALAM LETTER DDA","	= hard da"},
/* 0D22 */ { "MALAYALAM LETTER DDHA","	= hard dda"},
/* 0D23 */ { "MALAYALAM LETTER NNA","	= hard na"},
/* 0D24 */ { "MALAYALAM LETTER TA","	= tha"},
/* 0D25 */ { "MALAYALAM LETTER THA","	= ttha"},
/* 0D26 */ { "MALAYALAM LETTER DA","	= soft da"},
/* 0D27 */ { "MALAYALAM LETTER DHA","	= soft dda"},
/* 0D28 */ { "MALAYALAM LETTER NA",NULL},
/* 0D29 */ { "MALAYALAM LETTER NNNA","	* scholarly use only"},
/* 0D2A */ { "MALAYALAM LETTER PA",NULL},
/* 0D2B */ { "MALAYALAM LETTER PHA",NULL},
/* 0D2C */ { "MALAYALAM LETTER BA",NULL},
/* 0D2D */ { "MALAYALAM LETTER BHA",NULL},
/* 0D2E */ { "MALAYALAM LETTER MA","	* also used to denote the fraction one eightieth (kaani)"},
/* 0D2F */ { "MALAYALAM LETTER YA",NULL},
/* 0D30 */ { "MALAYALAM LETTER RA",NULL},
/* 0D31 */ { "MALAYALAM LETTER RRA",NULL},
/* 0D32 */ { "MALAYALAM LETTER LA",NULL},
/* 0D33 */ { "MALAYALAM LETTER LLA",NULL},
/* 0D34 */ { "MALAYALAM LETTER LLLA","	= zha"},
/* 0D35 */ { "MALAYALAM LETTER VA",NULL},
/* 0D36 */ { "MALAYALAM LETTER SHA","	= soft sha"},
/* 0D37 */ { "MALAYALAM LETTER SSA","	= sha"},
/* 0D38 */ { "MALAYALAM LETTER SA",NULL},
/* 0D39 */ { "MALAYALAM LETTER HA",NULL},
/* 0D3A */ { "MALAYALAM LETTER TTTA","	* scholarly use only"},
/* 0D3B */ { "MALAYALAM SIGN VERTICAL BAR VIRAMA",NULL},
/* 0D3C */ { "MALAYALAM SIGN CIRCULAR VIRAMA",NULL},
/* 0D3D */ { "MALAYALAM SIGN AVAGRAHA","	= praslesham"},
/* 0D3E */ { "MALAYALAM VOWEL SIGN AA",NULL},
/* 0D3F */ { "MALAYALAM VOWEL SIGN I",NULL},
/* 0D40 */ { "MALAYALAM VOWEL SIGN II",NULL},
/* 0D41 */ { "MALAYALAM VOWEL SIGN U",NULL},
/* 0D42 */ { "MALAYALAM VOWEL SIGN UU",NULL},
/* 0D43 */ { "MALAYALAM VOWEL SIGN VOCALIC R",NULL},
/* 0D44 */ { "MALAYALAM VOWEL SIGN VOCALIC RR",NULL},
/* 0D45 */ { NULL,NULL},
/* 0D46 */ { "MALAYALAM VOWEL SIGN E","	* stands to the left of the consonant"},
/* 0D47 */ { "MALAYALAM VOWEL SIGN EE","	* stands to the left of the consonant"},
/* 0D48 */ { "MALAYALAM VOWEL SIGN AI","	* stands to the left of the consonant"},
/* 0D49 */ { NULL,NULL},
/* 0D4A */ { "MALAYALAM VOWEL SIGN O","	: 0D46 0D3E"},
/* 0D4B */ { "MALAYALAM VOWEL SIGN OO","	: 0D47 0D3E"},
/* 0D4C */ { "MALAYALAM VOWEL SIGN AU","	* archaic form of the /au/ dependent vowel\n"
	"	x (malayalam au length mark - 0D57)\n"
	"	: 0D46 0D57"},
/* 0D4D */ { "MALAYALAM SIGN VIRAMA","	= candrakkala (the preferred name)\n"
	"	= vowel half-u"},
/* 0D4E */ { "MALAYALAM LETTER DOT REPH","	* not used in reformed modern Malayalam orthography"},
/* 0D4F */ { "MALAYALAM SIGN PARA","	* used historically to measure rice"},
/* 0D50 */ { NULL,NULL},
/* 0D51 */ { NULL,NULL},
/* 0D52 */ { NULL,NULL},
/* 0D53 */ { NULL,NULL},
/* 0D54 */ { "MALAYALAM LETTER CHILLU M",NULL},
/* 0D55 */ { "MALAYALAM LETTER CHILLU Y",NULL},
/* 0D56 */ { "MALAYALAM LETTER CHILLU LLL",NULL},
/* 0D57 */ { "MALAYALAM AU LENGTH MARK","	* used alone to write the /au/ dependent vowel in modern texts\n"
	"	x (malayalam vowel sign au - 0D4C)"},
/* 0D58 */ { "MALAYALAM FRACTION ONE ONE-HUNDRED-AND-SIXTIETH","	= arakaani"},
/* 0D59 */ { "MALAYALAM FRACTION ONE FORTIETH","	= aramaa"},
/* 0D5A */ { "MALAYALAM FRACTION THREE EIGHTIETHS","	= muunnukaani"},
/* 0D5B */ { "MALAYALAM FRACTION ONE TWENTIETH","	= orumaa"},
/* 0D5C */ { "MALAYALAM FRACTION ONE TENTH","	= rantumaa"},
/* 0D5D */ { "MALAYALAM FRACTION THREE TWENTIETHS","	= muunnumaa"},
/* 0D5E */ { "MALAYALAM FRACTION ONE FIFTH","	= naalumaa"},
/* 0D5F */ { "MALAYALAM LETTER ARCHAIC II",NULL},
/* 0D60 */ { "MALAYALAM LETTER VOCALIC RR",NULL},
/* 0D61 */ { "MALAYALAM LETTER VOCALIC LL",NULL},
/* 0D62 */ { "MALAYALAM VOWEL SIGN VOCALIC L",NULL},
/* 0D63 */ { "MALAYALAM VOWEL SIGN VOCALIC LL",NULL},
/* 0D64 */ { NULL,"	x (devanagari danda - 0964)"},
/* 0D65 */ { NULL,"	x (devanagari double danda - 0965)"},
/* 0D66 */ { "MALAYALAM DIGIT ZERO",NULL},
/* 0D67 */ { "MALAYALAM DIGIT ONE",NULL},
/* 0D68 */ { "MALAYALAM DIGIT TWO",NULL},
/* 0D69 */ { "MALAYALAM DIGIT THREE",NULL},
/* 0D6A */ { "MALAYALAM DIGIT FOUR",NULL},
/* 0D6B */ { "MALAYALAM DIGIT FIVE",NULL},
/* 0D6C */ { "MALAYALAM DIGIT SIX",NULL},
/* 0D6D */ { "MALAYALAM DIGIT SEVEN",NULL},
/* 0D6E */ { "MALAYALAM DIGIT EIGHT",NULL},
/* 0D6F */ { "MALAYALAM DIGIT NINE",NULL},
/* 0D70 */ { "MALAYALAM NUMBER TEN",NULL},
/* 0D71 */ { "MALAYALAM NUMBER ONE HUNDRED",NULL},
/* 0D72 */ { "MALAYALAM NUMBER ONE THOUSAND",NULL},
/* 0D73 */ { "MALAYALAM FRACTION ONE QUARTER","	= kaal\n"
	"	x (north indic fraction one quarter - A830)"},
/* 0D74 */ { "MALAYALAM FRACTION ONE HALF","	= ara\n"
	"	x (north indic fraction one half - A831)"},
/* 0D75 */ { "MALAYALAM FRACTION THREE QUARTERS","	= mukkaal\n"
	"	x (north indic fraction three quarters - A832)"},
/* 0D76 */ { "MALAYALAM FRACTION ONE SIXTEENTH","	= maakaani"},
/* 0D77 */ { "MALAYALAM FRACTION ONE EIGHTH","	= arakkaal"},
/* 0D78 */ { "MALAYALAM FRACTION THREE SIXTEENTHS","	= muntaani"},
/* 0D79 */ { "MALAYALAM DATE MARK",NULL},
/* 0D7A */ { "MALAYALAM LETTER CHILLU NN",NULL},
/* 0D7B */ { "MALAYALAM LETTER CHILLU N",NULL},
/* 0D7C */ { "MALAYALAM LETTER CHILLU RR","	* historically derived from the full letter ra\n"
	"	* also used for chillu r"},
/* 0D7D */ { "MALAYALAM LETTER CHILLU L","	* historically derived from the full letter ta\n"
	"	* used for chillu t and chillu d"},
/* 0D7E */ { "MALAYALAM LETTER CHILLU LL",NULL},
/* 0D7F */ { "MALAYALAM LETTER CHILLU K",NULL},
/* 0D80 */ { NULL,NULL},
/* 0D81 */ { "SINHALA SIGN CANDRABINDU","	* used in Sanskrit"},
/* 0D82 */ { "SINHALA SIGN ANUSVARAYA","	= anusvara"},
/* 0D83 */ { "SINHALA SIGN VISARGAYA","	= visarga"},
/* 0D84 */ { NULL,NULL},
/* 0D85 */ { "SINHALA LETTER AYANNA","	= sinhala letter a"},
/* 0D86 */ { "SINHALA LETTER AAYANNA","	= sinhala letter aa"},
/* 0D87 */ { "SINHALA LETTER AEYANNA","	= sinhala letter ae"},
/* 0D88 */ { "SINHALA LETTER AEEYANNA","	= sinhala letter aae"},
/* 0D89 */ { "SINHALA LETTER IYANNA","	= sinhala letter i"},
/* 0D8A */ { "SINHALA LETTER IIYANNA","	= sinhala letter ii"},
/* 0D8B */ { "SINHALA LETTER UYANNA","	= sinhala letter u"},
/* 0D8C */ { "SINHALA LETTER UUYANNA","	= sinhala letter uu"},
/* 0D8D */ { "SINHALA LETTER IRUYANNA","	= sinhala letter vocalic r"},
/* 0D8E */ { "SINHALA LETTER IRUUYANNA","	= sinhala letter vocalic rr"},
/* 0D8F */ { "SINHALA LETTER ILUYANNA","	= sinhala letter vocalic l"},
/* 0D90 */ { "SINHALA LETTER ILUUYANNA","	= sinhala letter vocalic ll"},
/* 0D91 */ { "SINHALA LETTER EYANNA","	= sinhala letter e"},
/* 0D92 */ { "SINHALA LETTER EEYANNA","	= sinhala letter ee"},
/* 0D93 */ { "SINHALA LETTER AIYANNA","	= sinhala letter ai"},
/* 0D94 */ { "SINHALA LETTER OYANNA","	= sinhala letter o"},
/* 0D95 */ { "SINHALA LETTER OOYANNA","	= sinhala letter oo"},
/* 0D96 */ { "SINHALA LETTER AUYANNA","	= sinhala letter au"},
/* 0D97 */ { NULL,NULL},
/* 0D98 */ { NULL,NULL},
/* 0D99 */ { NULL,NULL},
/* 0D9A */ { "SINHALA LETTER ALPAPRAANA KAYANNA","	= sinhala letter ka"},
/* 0D9B */ { "SINHALA LETTER MAHAAPRAANA KAYANNA","	= sinhala letter kha"},
/* 0D9C */ { "SINHALA LETTER ALPAPRAANA GAYANNA","	= sinhala letter ga"},
/* 0D9D */ { "SINHALA LETTER MAHAAPRAANA GAYANNA","	= sinhala letter gha"},
/* 0D9E */ { "SINHALA LETTER KANTAJA NAASIKYAYA","	= sinhala letter nga"},
/* 0D9F */ { "SINHALA LETTER SANYAKA GAYANNA","	= sinhala letter nnga"},
/* 0DA0 */ { "SINHALA LETTER ALPAPRAANA CAYANNA","	= sinhala letter ca"},
/* 0DA1 */ { "SINHALA LETTER MAHAAPRAANA CAYANNA","	= sinhala letter cha"},
/* 0DA2 */ { "SINHALA LETTER ALPAPRAANA JAYANNA","	= sinhala letter ja"},
/* 0DA3 */ { "SINHALA LETTER MAHAAPRAANA JAYANNA","	= sinhala letter jha"},
/* 0DA4 */ { "SINHALA LETTER TAALUJA NAASIKYAYA","	= sinhala letter nya"},
/* 0DA5 */ { "SINHALA LETTER TAALUJA SANYOOGA NAAKSIKYAYA","	= sinhala letter jnya"},
/* 0DA6 */ { "SINHALA LETTER SANYAKA JAYANNA","	= sinhala letter nyja"},
/* 0DA7 */ { "SINHALA LETTER ALPAPRAANA TTAYANNA","	= sinhala letter tta"},
/* 0DA8 */ { "SINHALA LETTER MAHAAPRAANA TTAYANNA","	= sinhala letter ttha"},
/* 0DA9 */ { "SINHALA LETTER ALPAPRAANA DDAYANNA","	= sinhala letter dda"},
/* 0DAA */ { "SINHALA LETTER MAHAAPRAANA DDAYANNA","	= sinhala letter ddha"},
/* 0DAB */ { "SINHALA LETTER MUURDHAJA NAYANNA","	= sinhala letter nna"},
/* 0DAC */ { "SINHALA LETTER SANYAKA DDAYANNA","	= sinhala letter nndda"},
/* 0DAD */ { "SINHALA LETTER ALPAPRAANA TAYANNA","	= sinhala letter ta"},
/* 0DAE */ { "SINHALA LETTER MAHAAPRAANA TAYANNA","	= sinhala letter tha"},
/* 0DAF */ { "SINHALA LETTER ALPAPRAANA DAYANNA","	= sinhala letter da"},
/* 0DB0 */ { "SINHALA LETTER MAHAAPRAANA DAYANNA","	= sinhala letter dha"},
/* 0DB1 */ { "SINHALA LETTER DANTAJA NAYANNA","	= sinhala letter na"},
/* 0DB2 */ { NULL,NULL},
/* 0DB3 */ { "SINHALA LETTER SANYAKA DAYANNA","	= sinhala letter nda"},
/* 0DB4 */ { "SINHALA LETTER ALPAPRAANA PAYANNA","	= sinhala letter pa"},
/* 0DB5 */ { "SINHALA LETTER MAHAAPRAANA PAYANNA","	= sinhala letter pha"},
/* 0DB6 */ { "SINHALA LETTER ALPAPRAANA BAYANNA","	= sinhala letter ba"},
/* 0DB7 */ { "SINHALA LETTER MAHAAPRAANA BAYANNA","	= sinhala letter bha"},
/* 0DB8 */ { "SINHALA LETTER MAYANNA","	= sinhala letter ma"},
/* 0DB9 */ { "SINHALA LETTER AMBA BAYANNA","	= sinhala letter mba"},
/* 0DBA */ { "SINHALA LETTER YAYANNA","	= sinhala letter ya"},
/* 0DBB */ { "SINHALA LETTER RAYANNA","	= sinhala letter ra"},
/* 0DBC */ { NULL,NULL},
/* 0DBD */ { "SINHALA LETTER DANTAJA LAYANNA","	= sinhala letter la\n"
	"	* dental"},
/* 0DBE */ { NULL,NULL},
/* 0DBF */ { NULL,NULL},
/* 0DC0 */ { "SINHALA LETTER VAYANNA","	= sinhala letter va"},
/* 0DC1 */ { "SINHALA LETTER TAALUJA SAYANNA","	= sinhala letter sha"},
/* 0DC2 */ { "SINHALA LETTER MUURDHAJA SAYANNA","	= sinhala letter ssa\n"
	"	* retroflex"},
/* 0DC3 */ { "SINHALA LETTER DANTAJA SAYANNA","	= sinhala letter sa\n"
	"	* dental"},
/* 0DC4 */ { "SINHALA LETTER HAYANNA","	= sinhala letter ha"},
/* 0DC5 */ { "SINHALA LETTER MUURDHAJA LAYANNA","	= sinhala letter lla\n"
	"	* retroflex"},
/* 0DC6 */ { "SINHALA LETTER FAYANNA","	= sinhala letter fa"},
/* 0DC7 */ { NULL,NULL},
/* 0DC8 */ { NULL,NULL},
/* 0DC9 */ { NULL,NULL},
/* 0DCA */ { "SINHALA SIGN AL-LAKUNA","	= virama"},
/* 0DCB */ { NULL,NULL},
/* 0DCC */ { NULL,NULL},
/* 0DCD */ { NULL,NULL},
/* 0DCE */ { NULL,NULL},
/* 0DCF */ { "SINHALA VOWEL SIGN AELA-PILLA","	= sinhala vowel sign aa"},
/* 0DD0 */ { "SINHALA VOWEL SIGN KETTI AEDA-PILLA","	= sinhala vowel sign ae"},
/* 0DD1 */ { "SINHALA VOWEL SIGN DIGA AEDA-PILLA","	= sinhala vowel sign aae"},
/* 0DD2 */ { "SINHALA VOWEL SIGN KETTI IS-PILLA","	= sinhala vowel sign i"},
/* 0DD3 */ { "SINHALA VOWEL SIGN DIGA IS-PILLA","	= sinhala vowel sign ii"},
/* 0DD4 */ { "SINHALA VOWEL SIGN KETTI PAA-PILLA","	= sinhala vowel sign u"},
/* 0DD5 */ { NULL,NULL},
/* 0DD6 */ { "SINHALA VOWEL SIGN DIGA PAA-PILLA","	= sinhala vowel sign uu"},
/* 0DD7 */ { NULL,NULL},
/* 0DD8 */ { "SINHALA VOWEL SIGN GAETTA-PILLA","	= sinhala vowel sign vocalic r"},
/* 0DD9 */ { "SINHALA VOWEL SIGN KOMBUVA","	= sinhala vowel sign e"},
/* 0DDA */ { "SINHALA VOWEL SIGN DIGA KOMBUVA","	= sinhala vowel sign ee\n"
	"	: 0DD9 0DCA"},
/* 0DDB */ { "SINHALA VOWEL SIGN KOMBU DEKA","	= sinhala vowel sign ai"},
/* 0DDC */ { "SINHALA VOWEL SIGN KOMBUVA HAA AELA-PILLA","	= sinhala vowel sign o\n"
	"	: 0DD9 0DCF"},
/* 0DDD */ { "SINHALA VOWEL SIGN KOMBUVA HAA DIGA AELA-PILLA","	= sinhala vowel sign oo\n"
	"	: 0DDC 0DCA"},
/* 0DDE */ { "SINHALA VOWEL SIGN KOMBUVA HAA GAYANUKITTA","	= sinhala vowel sign au\n"
	"	: 0DD9 0DDF"},
/* 0DDF */ { "SINHALA VOWEL SIGN GAYANUKITTA","	= sinhala vowel sign vocalic l"},
/* 0DE0 */ { NULL,NULL},
/* 0DE1 */ { NULL,NULL},
/* 0DE2 */ { NULL,NULL},
/* 0DE3 */ { NULL,NULL},
/* 0DE4 */ { NULL,NULL},
/* 0DE5 */ { NULL,NULL},
/* 0DE6 */ { "SINHALA LITH DIGIT ZERO",NULL},
/* 0DE7 */ { "SINHALA LITH DIGIT ONE",NULL},
/* 0DE8 */ { "SINHALA LITH DIGIT TWO",NULL},
/* 0DE9 */ { "SINHALA LITH DIGIT THREE",NULL},
/* 0DEA */ { "SINHALA LITH DIGIT FOUR",NULL},
/* 0DEB */ { "SINHALA LITH DIGIT FIVE",NULL},
/* 0DEC */ { "SINHALA LITH DIGIT SIX",NULL},
/* 0DED */ { "SINHALA LITH DIGIT SEVEN",NULL},
/* 0DEE */ { "SINHALA LITH DIGIT EIGHT",NULL},
/* 0DEF */ { "SINHALA LITH DIGIT NINE",NULL},
/* 0DF0 */ { NULL,NULL},
/* 0DF1 */ { NULL,NULL},
/* 0DF2 */ { "SINHALA VOWEL SIGN DIGA GAETTA-PILLA","	= sinhala vowel sign vocalic rr"},
/* 0DF3 */ { "SINHALA VOWEL SIGN DIGA GAYANUKITTA","	= sinhala vowel sign vocalic ll"},
/* 0DF4 */ { "SINHALA PUNCTUATION KUNDDALIYA","	x (tamil punctuation end of text - 11FFF)"},
/* 0DF5 */ { NULL,NULL},
/* 0DF6 */ { NULL,NULL},
/* 0DF7 */ { NULL,NULL},
/* 0DF8 */ { NULL,NULL},
/* 0DF9 */ { NULL,NULL},
/* 0DFA */ { NULL,NULL},
/* 0DFB */ { NULL,NULL},
/* 0DFC */ { NULL,NULL},
/* 0DFD */ { NULL,NULL},
/* 0DFE */ { NULL,NULL},
/* 0DFF */ { NULL,NULL}
};

UN_DLL_LOCAL
static const struct unicode_nameannot una_00_0E[] = {
/* 0E00 */ { NULL,NULL},
/* 0E01 */ { "THAI CHARACTER KO KAI",NULL},
/* 0E02 */ { "THAI CHARACTER KHO KHAI",NULL},
/* 0E03 */ { "THAI CHARACTER KHO KHUAT",NULL},
/* 0E04 */ { "THAI CHARACTER KHO KHWAI",NULL},
/* 0E05 */ { "THAI CHARACTER KHO KHON",NULL},
/* 0E06 */ { "THAI CHARACTER KHO RAKHANG",NULL},
/* 0E07 */ { "THAI CHARACTER NGO NGU",NULL},
/* 0E08 */ { "THAI CHARACTER CHO CHAN",NULL},
/* 0E09 */ { "THAI CHARACTER CHO CHING",NULL},
/* 0E0A */ { "THAI CHARACTER CHO CHANG",NULL},
/* 0E0B */ { "THAI CHARACTER SO SO",NULL},
/* 0E0C */ { "THAI CHARACTER CHO CHOE",NULL},
/* 0E0D */ { "THAI CHARACTER YO YING",NULL},
/* 0E0E */ { "THAI CHARACTER DO CHADA",NULL},
/* 0E0F */ { "THAI CHARACTER TO PATAK",NULL},
/* 0E10 */ { "THAI CHARACTER THO THAN",NULL},
/* 0E11 */ { "THAI CHARACTER THO NANGMONTHO",NULL},
/* 0E12 */ { "THAI CHARACTER THO PHUTHAO",NULL},
/* 0E13 */ { "THAI CHARACTER NO NEN",NULL},
/* 0E14 */ { "THAI CHARACTER DO DEK",NULL},
/* 0E15 */ { "THAI CHARACTER TO TAO",NULL},
/* 0E16 */ { "THAI CHARACTER THO THUNG",NULL},
/* 0E17 */ { "THAI CHARACTER THO THAHAN",NULL},
/* 0E18 */ { "THAI CHARACTER THO THONG",NULL},
/* 0E19 */ { "THAI CHARACTER NO NU",NULL},
/* 0E1A */ { "THAI CHARACTER BO BAIMAI",NULL},
/* 0E1B */ { "THAI CHARACTER PO PLA",NULL},
/* 0E1C */ { "THAI CHARACTER PHO PHUNG",NULL},
/* 0E1D */ { "THAI CHARACTER FO FA",NULL},
/* 0E1E */ { "THAI CHARACTER PHO PHAN",NULL},
/* 0E1F */ { "THAI CHARACTER FO FAN",NULL},
/* 0E20 */ { "THAI CHARACTER PHO SAMPHAO",NULL},
/* 0E21 */ { "THAI CHARACTER MO MA",NULL},
/* 0E22 */ { "THAI CHARACTER YO YAK",NULL},
/* 0E23 */ { "THAI CHARACTER RO RUA",NULL},
/* 0E24 */ { "THAI CHARACTER RU","	* independent vowel letter used to write Sanskrit"},
/* 0E25 */ { "THAI CHARACTER LO LING",NULL},
/* 0E26 */ { "THAI CHARACTER LU","	* independent vowel letter used to write Sanskrit"},
/* 0E27 */ { "THAI CHARACTER WO WAEN",NULL},
/* 0E28 */ { "THAI CHARACTER SO SALA",NULL},
/* 0E29 */ { "THAI CHARACTER SO RUSI",NULL},
/* 0E2A */ { "THAI CHARACTER SO SUA",NULL},
/* 0E2B */ { "THAI CHARACTER HO HIP",NULL},
/* 0E2C */ { "THAI CHARACTER LO CHULA",NULL},
/* 0E2D */ { "THAI CHARACTER O ANG",NULL},
/* 0E2E */ { "THAI CHARACTER HO NOKHUK","	= ho nok huk"},
/* 0E2F */ { "THAI CHARACTER PAIYANNOI","	= paiyan noi\n"
	"	* ellipsis, abbreviation"},
/* 0E30 */ { "THAI CHARACTER SARA A",NULL},
/* 0E31 */ { "THAI CHARACTER MAI HAN-AKAT",NULL},
/* 0E32 */ { "THAI CHARACTER SARA AA","	x (thai character lakkhangyao - 0E45)"},
/* 0E33 */ { "THAI CHARACTER SARA AM","	# 0E4D 0E32"},
/* 0E34 */ { "THAI CHARACTER SARA I",NULL},
/* 0E35 */ { "THAI CHARACTER SARA II",NULL},
/* 0E36 */ { "THAI CHARACTER SARA UE",NULL},
/* 0E37 */ { "THAI CHARACTER SARA UEE","	= sara uue"},
/* 0E38 */ { "THAI CHARACTER SARA U",NULL},
/* 0E39 */ { "THAI CHARACTER SARA UU",NULL},
/* 0E3A */ { "THAI CHARACTER PHINTHU","	* Pali virama"},
/* 0E3B */ { NULL,NULL},
/* 0E3C */ { NULL,NULL},
/* 0E3D */ { NULL,NULL},
/* 0E3E */ { NULL,NULL},
/* 0E3F */ { "THAI CURRENCY SYMBOL BAHT",NULL},
/* 0E40 */ { "THAI CHARACTER SARA E",NULL},
/* 0E41 */ { "THAI CHARACTER SARA AE",NULL},
/* 0E42 */ { "THAI CHARACTER SARA O",NULL},
/* 0E43 */ { "THAI CHARACTER SARA AI MAIMUAN","	= sara ai mai muan"},
/* 0E44 */ { "THAI CHARACTER SARA AI MAIMALAI","	= sara ai mai malai"},
/* 0E45 */ { "THAI CHARACTER LAKKHANGYAO","	= lakkhang yao\n"
	"	* special vowel length indication used with 0E24 or 0E26\n"
	"	x (thai character sara aa - 0E32)"},
/* 0E46 */ { "THAI CHARACTER MAIYAMOK","	= mai yamok\n"
	"	* repetition"},
/* 0E47 */ { "THAI CHARACTER MAITAIKHU","	= mai taikhu"},
/* 0E48 */ { "THAI CHARACTER MAI EK",NULL},
/* 0E49 */ { "THAI CHARACTER MAI THO",NULL},
/* 0E4A */ { "THAI CHARACTER MAI TRI",NULL},
/* 0E4B */ { "THAI CHARACTER MAI CHATTAWA",NULL},
/* 0E4C */ { "THAI CHARACTER THANTHAKHAT","	* cancellation mark\n"
	"	x (khmer sign toandakhiat - 17CD)"},
/* 0E4D */ { "THAI CHARACTER NIKHAHIT","	= nikkhahit\n"
	"	* final nasal"},
/* 0E4E */ { "THAI CHARACTER YAMAKKAN","	x (lao yamakkan - 0ECE)"},
/* 0E4F */ { "THAI CHARACTER FONGMAN","	* used as a bullet\n"
	"	x (khmer sign phnaek muan - 17D9)"},
/* 0E50 */ { "THAI DIGIT ZERO",NULL},
/* 0E51 */ { "THAI DIGIT ONE",NULL},
/* 0E52 */ { "THAI DIGIT TWO",NULL},
/* 0E53 */ { "THAI DIGIT THREE",NULL},
/* 0E54 */ { "THAI DIGIT FOUR",NULL},
/* 0E55 */ { "THAI DIGIT FIVE",NULL},
/* 0E56 */ { "THAI DIGIT SIX",NULL},
/* 0E57 */ { "THAI DIGIT SEVEN",NULL},
/* 0E58 */ { "THAI DIGIT EIGHT",NULL},
/* 0E59 */ { "THAI DIGIT NINE",NULL},
/* 0E5A */ { "THAI CHARACTER ANGKHANKHU","	* used to mark end of long sections\n"
	"	* used in combination with 0E30 to mark end of a verse"},
/* 0E5B */ { "THAI CHARACTER KHOMUT","	* used to mark end of chapter or document\n"
	"	x (khmer sign koomuut - 17DA)"},
/* 0E5C */ { NULL,NULL},
/* 0E5D */ { NULL,NULL},
/* 0E5E */ { NULL,NULL},
/* 0E5F */ { NULL,NULL},
/* 0E60 */ { NULL,NULL},
/* 0E61 */ { NULL,NULL},
/* 0E62 */ { NULL,NULL},
/* 0E63 */ { NULL,NULL},
/* 0E64 */ { NULL,NULL},
/* 0E65 */ { NULL,NULL},
/* 0E66 */ { NULL,NULL},
/* 0E67 */ { NULL,NULL},
/* 0E68 */ { NULL,NULL},
/* 0E69 */ { NULL,NULL},
/* 0E6A */ { NULL,NULL},
/* 0E6B */ { NULL,NULL},
/* 0E6C */ { NULL,NULL},
/* 0E6D */ { NULL,NULL},
/* 0E6E */ { NULL,NULL},
/* 0E6F */ { NULL,NULL},
/* 0E70 */ { NULL,NULL},
/* 0E71 */ { NULL,NULL},
/* 0E72 */ { NULL,NULL},
/* 0E73 */ { NULL,NULL},
/* 0E74 */ { NULL,NULL},
/* 0E75 */ { NULL,NULL},
/* 0E76 */ { NULL,NULL},
/* 0E77 */ { NULL,NULL},
/* 0E78 */ { NULL,NULL},
/* 0E79 */ { NULL,NULL},
/* 0E7A */ { NULL,NULL},
/* 0E7B */ { NULL,NULL},
/* 0E7C */ { NULL,NULL},
/* 0E7D */ { NULL,NULL},
/* 0E7E */ { NULL,NULL},
/* 0E7F */ { NULL,NULL},
/* 0E80 */ { NULL,NULL},
/* 0E81 */ { "LAO LETTER KO","	= ko kay"},
/* 0E82 */ { "LAO LETTER KHO SUNG","	= kho khay"},
/* 0E83 */ { NULL,NULL},
/* 0E84 */ { "LAO LETTER KHO TAM","	= kho khuay"},
/* 0E85 */ { NULL,NULL},
/* 0E86 */ { "LAO LETTER PALI GHA",NULL},
/* 0E87 */ { "LAO LETTER NGO","	= ngo ngu, ngo ngua"},
/* 0E88 */ { "LAO LETTER CO","	= co cok, co cua"},
/* 0E89 */ { "LAO LETTER PALI CHA",NULL},
/* 0E8A */ { "LAO LETTER SO TAM","	= so sang"},
/* 0E8B */ { NULL,NULL},
/* 0E8C */ { "LAO LETTER PALI JHA",NULL},
/* 0E8D */ { "LAO LETTER NYO","	= nyo nyung"},
/* 0E8E */ { "LAO LETTER PALI NYA",NULL},
/* 0E8F */ { "LAO LETTER PALI TTA",NULL},
/* 0E90 */ { "LAO LETTER PALI TTHA",NULL},
/* 0E91 */ { "LAO LETTER PALI DDA",NULL},
/* 0E92 */ { "LAO LETTER PALI DDHA",NULL},
/* 0E93 */ { "LAO LETTER PALI NNA",NULL},
/* 0E94 */ { "LAO LETTER DO","	= do dek"},
/* 0E95 */ { "LAO LETTER TO","	= to ta"},
/* 0E96 */ { "LAO LETTER THO SUNG","	= tho thong"},
/* 0E97 */ { "LAO LETTER THO TAM","	= tho thung"},
/* 0E98 */ { "LAO LETTER PALI DHA",NULL},
/* 0E99 */ { "LAO LETTER NO","	= no nok"},
/* 0E9A */ { "LAO LETTER BO","	= bo be, bo bet"},
/* 0E9B */ { "LAO LETTER PO","	= po pa"},
/* 0E9C */ { "LAO LETTER PHO SUNG","	= pho pheng"},
/* 0E9D */ { "LAO LETTER FO TAM","	% LAO LETTER FO FON\n"
	"	= fo fa\n"
	"	* name is a mistake for fo sung"},
/* 0E9E */ { "LAO LETTER PHO TAM","	= pho phu"},
/* 0E9F */ { "LAO LETTER FO SUNG","	% LAO LETTER FO FAY\n"
	"	* name is a mistake for fo tam"},
/* 0EA0 */ { "LAO LETTER PALI BHA",NULL},
/* 0EA1 */ { "LAO LETTER MO","	= mo mew, mo ma"},
/* 0EA2 */ { "LAO LETTER YO","	= yo ya"},
/* 0EA3 */ { "LAO LETTER LO LING","	% LAO LETTER RO\n"
	"	= ro rot\n"
	"	* name is a mistake, lo ling is the mnemonic for 0EA5"},
/* 0EA4 */ { NULL,NULL},
/* 0EA5 */ { "LAO LETTER LO LOOT","	% LAO LETTER LO\n"
	"	= lo ling\n"
	"	* name is a mistake, lo loot is the mnemonic for 0EA3"},
/* 0EA6 */ { NULL,NULL},
/* 0EA7 */ { "LAO LETTER WO","	= wo wi"},
/* 0EA8 */ { "LAO LETTER SANSKRIT SHA",NULL},
/* 0EA9 */ { "LAO LETTER SANSKRIT SSA",NULL},
/* 0EAA */ { "LAO LETTER SO SUNG","	= so sya"},
/* 0EAB */ { "LAO LETTER HO SUNG","	= ho hay, ho han"},
/* 0EAC */ { "LAO LETTER PALI LLA",NULL},
/* 0EAD */ { "LAO LETTER O","	= o o"},
/* 0EAE */ { "LAO LETTER HO TAM","	= ho hya, ho hyan"},
/* 0EAF */ { "LAO ELLIPSIS",NULL},
/* 0EB0 */ { "LAO VOWEL SIGN A",NULL},
/* 0EB1 */ { "LAO VOWEL SIGN MAI KAN","	* vowel shortener"},
/* 0EB2 */ { "LAO VOWEL SIGN AA",NULL},
/* 0EB3 */ { "LAO VOWEL SIGN AM","	# 0ECD 0EB2"},
/* 0EB4 */ { "LAO VOWEL SIGN I",NULL},
/* 0EB5 */ { "LAO VOWEL SIGN II",NULL},
/* 0EB6 */ { "LAO VOWEL SIGN Y",NULL},
/* 0EB7 */ { "LAO VOWEL SIGN YY",NULL},
/* 0EB8 */ { "LAO VOWEL SIGN U",NULL},
/* 0EB9 */ { "LAO VOWEL SIGN UU",NULL},
/* 0EBA */ { "LAO SIGN PALI VIRAMA",NULL},
/* 0EBB */ { "LAO VOWEL SIGN MAI KON","	= mai kong"},
/* 0EBC */ { "LAO SEMIVOWEL SIGN LO",NULL},
/* 0EBD */ { "LAO SEMIVOWEL SIGN NYO","	= nyo fyang"},
/* 0EBE */ { NULL,NULL},
/* 0EBF */ { NULL,NULL},
/* 0EC0 */ { "LAO VOWEL SIGN E",NULL},
/* 0EC1 */ { "LAO VOWEL SIGN EI",NULL},
/* 0EC2 */ { "LAO VOWEL SIGN O",NULL},
/* 0EC3 */ { "LAO VOWEL SIGN AY","	= mai muan"},
/* 0EC4 */ { "LAO VOWEL SIGN AI","	= mai may"},
/* 0EC5 */ { NULL,NULL},
/* 0EC6 */ { "LAO KO LA","	* repetition"},
/* 0EC7 */ { NULL,NULL},
/* 0EC8 */ { "LAO TONE MAI EK",NULL},
/* 0EC9 */ { "LAO TONE MAI THO",NULL},
/* 0ECA */ { "LAO TONE MAI TI",NULL},
/* 0ECB */ { "LAO TONE MAI CATAWA",NULL},
/* 0ECC */ { "LAO CANCELLATION MARK",NULL},
/* 0ECD */ { "LAO NIGGAHITA","	* final nasal or long o vowel"},
/* 0ECE */ { "LAO YAMAKKAN","	* marks a consonant that is simultaneously a part of two consonant clusters in Pali text\n"
	"	x (thai character yamakkan - 0E4E)"},
/* 0ECF */ { NULL,NULL},
/* 0ED0 */ { "LAO DIGIT ZERO",NULL},
/* 0ED1 */ { "LAO DIGIT ONE",NULL},
/* 0ED2 */ { "LAO DIGIT TWO",NULL},
/* 0ED3 */ { "LAO DIGIT THREE",NULL},
/* 0ED4 */ { "LAO DIGIT FOUR",NULL},
/* 0ED5 */ { "LAO DIGIT FIVE",NULL},
/* 0ED6 */ { "LAO DIGIT SIX",NULL},
/* 0ED7 */ { "LAO DIGIT SEVEN",NULL},
/* 0ED8 */ { "LAO DIGIT EIGHT",NULL},
/* 0ED9 */ { "LAO DIGIT NINE",NULL},
/* 0EDA */ { NULL,NULL},
/* 0EDB */ { NULL,NULL},
/* 0EDC */ { "LAO HO NO","	# 0EAB 0E99"},
/* 0EDD */ { "LAO HO MO","	# 0EAB 0EA1"},
/* 0EDE */ { "LAO LETTER KHMU GO",NULL},
/* 0EDF */ { "LAO LETTER KHMU NYO",NULL},
/* 0EE0 */ { NULL,NULL},
/* 0EE1 */ { NULL,NULL},
/* 0EE2 */ { NULL,NULL},
/* 0EE3 */ { NULL,NULL},
/* 0EE4 */ { NULL,NULL},
/* 0EE5 */ { NULL,NULL},
/* 0EE6 */ { NULL,NULL},
/* 0EE7 */ { NULL,NULL},
/* 0EE8 */ { NULL,NULL},
/* 0EE9 */ { NULL,NULL},
/* 0EEA */ { NULL,NULL},
/* 0EEB */ { NULL,NULL},
/* 0EEC */ { NULL,NULL},
/* 0EED */ { NULL,NULL},
/* 0EEE */ { NULL,NULL},
/* 0EEF */ { NULL,NULL},
/* 0EF0 */ { NULL,NULL},
/* 0EF1 */ { NULL,NULL},
/* 0EF2 */ { NULL,NULL},
/* 0EF3 */ { NULL,NULL},
/* 0EF4 */ { NULL,NULL},
/* 0EF5 */ { NULL,NULL},
/* 0EF6 */ { NULL,NULL},
/* 0EF7 */ { NULL,NULL},
/* 0EF8 */ { NULL,NULL},
/* 0EF9 */ { NULL,NULL},
/* 0EFA */ { NULL,NULL},
/* 0EFB */ { NULL,NULL},
/* 0EFC */ { NULL,NULL},
/* 0EFD */ { NULL,NULL},
/* 0EFE */ { NULL,NULL},
/* 0EFF */ { NULL,NULL}
};

UN_DLL_LOCAL
static const struct unicode_nameannot una_00_0F[] = {
/* 0F00 */ { "TIBETAN SYLLABLE OM",NULL},
/* 0F01 */ { "TIBETAN MARK GTER YIG MGO TRUNCATED A",NULL},
/* 0F02 */ { "TIBETAN MARK GTER YIG MGO -UM RNAM BCAD MA",NULL},
/* 0F03 */ { "TIBETAN MARK GTER YIG MGO -UM GTER TSHEG MA",NULL},
/* 0F04 */ { "TIBETAN MARK INITIAL YIG MGO MDUN MA","	* honorific; marks beginning of text or start of new folio\n"
	"	x (mongolian birga - 1800)"},
/* 0F05 */ { "TIBETAN MARK CLOSING YIG MGO SGAB MA","	* follows and ligates with initial yig-mgo"},
/* 0F06 */ { "TIBETAN MARK CARET YIG MGO PHUR SHAD MA",NULL},
/* 0F07 */ { "TIBETAN MARK YIG MGO TSHEG SHAD MA",NULL},
/* 0F08 */ { "TIBETAN MARK SBRUL SHAD","	* separates sections of meaning equivalent to topics and sub-topics"},
/* 0F09 */ { "TIBETAN MARK BSKUR YIG MGO","	* list enumerator, used in Bhutan"},
/* 0F0A */ { "TIBETAN MARK BKA- SHOG YIG MGO","	* petition honorific, used in Bhutan"},
/* 0F0B */ { "TIBETAN MARK INTERSYLLABIC TSHEG","	= tsek\n"
	"	* morpheme delimiter (approximate meaning)\n"
	"	* the normal tsheg; provides a break opportunity\n"
	"	* character name is a misnomer"},
/* 0F0C */ { "TIBETAN MARK DELIMITER TSHEG BSTAR","	* a non-breaking tsheg; inhibits line breaking\n"
	"	* character name is a misnomer\n"
	"	# <noBreak> 0F0B"},
/* 0F0D */ { "TIBETAN MARK SHAD","	= shey\n"
	"	* marks end of a section of text (tshig-grub)\n"
	"	x (devanagari danda - 0964)"},
/* 0F0E */ { "TIBETAN MARK NYIS SHAD","	* marks end of a whole topic (don-tshan)\n"
	"	x (devanagari double danda - 0965)"},
/* 0F0F */ { "TIBETAN MARK TSHEG SHAD",NULL},
/* 0F10 */ { "TIBETAN MARK NYIS TSHEG SHAD",NULL},
/* 0F11 */ { "TIBETAN MARK RIN CHEN SPUNGS SHAD","	* shad which follows a tsheg-bar that starts a new line"},
/* 0F12 */ { "TIBETAN MARK RGYA GRAM SHAD",NULL},
/* 0F13 */ { "TIBETAN MARK CARET -DZUD RTAGS ME LONG CAN",NULL},
/* 0F14 */ { "TIBETAN MARK GTER TSHEG","	* used as a comma-like text delimiter\n"
	"	x (khmer sign camnuc pii kuuh - 17D6)"},
/* 0F15 */ { "TIBETAN LOGOTYPE SIGN CHAD RTAGS",NULL},
/* 0F16 */ { "TIBETAN LOGOTYPE SIGN LHAG RTAGS",NULL},
/* 0F17 */ { "TIBETAN ASTROLOGICAL SIGN SGRA GCAN -CHAR RTAGS",NULL},
/* 0F18 */ { "TIBETAN ASTROLOGICAL SIGN -KHYUD PA","	* combines with digits"},
/* 0F19 */ { "TIBETAN ASTROLOGICAL SIGN SDONG TSHUGS","	* combines with digits"},
/* 0F1A */ { "TIBETAN SIGN RDEL DKAR GCIG",NULL},
/* 0F1B */ { "TIBETAN SIGN RDEL DKAR GNYIS",NULL},
/* 0F1C */ { "TIBETAN SIGN RDEL DKAR GSUM","	x (alchemical symbol for oil - 1F746)"},
/* 0F1D */ { "TIBETAN SIGN RDEL NAG GCIG",NULL},
/* 0F1E */ { "TIBETAN SIGN RDEL NAG GNYIS",NULL},
/* 0F1F */ { "TIBETAN SIGN RDEL DKAR RDEL NAG","	= deka dena\n"
	"	* normally indicates a neutral fortune\n"
	"	* may also signify good luck earlier, bad luck later\n"
	"	x (tibetan sign rdel nag rdel dkar - 0FCE)"},
/* 0F20 */ { "TIBETAN DIGIT ZERO",NULL},
/* 0F21 */ { "TIBETAN DIGIT ONE",NULL},
/* 0F22 */ { "TIBETAN DIGIT TWO",NULL},
/* 0F23 */ { "TIBETAN DIGIT THREE",NULL},
/* 0F24 */ { "TIBETAN DIGIT FOUR",NULL},
/* 0F25 */ { "TIBETAN DIGIT FIVE",NULL},
/* 0F26 */ { "TIBETAN DIGIT SIX",NULL},
/* 0F27 */ { "TIBETAN DIGIT SEVEN",NULL},
/* 0F28 */ { "TIBETAN DIGIT EIGHT",NULL},
/* 0F29 */ { "TIBETAN DIGIT NINE",NULL},
/* 0F2A */ { "TIBETAN DIGIT HALF ONE",NULL},
/* 0F2B */ { "TIBETAN DIGIT HALF TWO",NULL},
/* 0F2C */ { "TIBETAN DIGIT HALF THREE",NULL},
/* 0F2D */ { "TIBETAN DIGIT HALF FOUR",NULL},
/* 0F2E */ { "TIBETAN DIGIT HALF FIVE",NULL},
/* 0F2F */ { "TIBETAN DIGIT HALF SIX",NULL},
/* 0F30 */ { "TIBETAN DIGIT HALF SEVEN",NULL},
/* 0F31 */ { "TIBETAN DIGIT HALF EIGHT",NULL},
/* 0F32 */ { "TIBETAN DIGIT HALF NINE",NULL},
/* 0F33 */ { "TIBETAN DIGIT HALF ZERO",NULL},
/* 0F34 */ { "TIBETAN MARK BSDUS RTAGS","	* repetition"},
/* 0F35 */ { "TIBETAN MARK NGAS BZUNG NYI ZLA","	* honorific, emphasis; used like underlining"},
/* 0F36 */ { "TIBETAN MARK CARET -DZUD RTAGS BZHI MIG CAN","	* marks point of text insertion or annotation"},
/* 0F37 */ { "TIBETAN MARK NGAS BZUNG SGOR RTAGS","	* emphasis; used like underlining"},
/* 0F38 */ { "TIBETAN MARK CHE MGO",NULL},
/* 0F39 */ { "TIBETAN MARK TSA -PHRU","	* a lenition mark"},
/* 0F3A */ { "TIBETAN MARK GUG RTAGS GYON",NULL},
/* 0F3B */ { "TIBETAN MARK GUG RTAGS GYAS","	* brackets"},
/* 0F3C */ { "TIBETAN MARK ANG KHANG GYON",NULL},
/* 0F3D */ { "TIBETAN MARK ANG KHANG GYAS","	* used for bracketing with a roof over"},
/* 0F3E */ { "TIBETAN SIGN YAR TSHES",NULL},
/* 0F3F */ { "TIBETAN SIGN MAR TSHES","	* marks which combine with digits"},
/* 0F40 */ { "TIBETAN LETTER KA",NULL},
/* 0F41 */ { "TIBETAN LETTER KHA",NULL},
/* 0F42 */ { "TIBETAN LETTER GA",NULL},
/* 0F43 */ { "TIBETAN LETTER GHA","	: 0F42 0FB7"},
/* 0F44 */ { "TIBETAN LETTER NGA",NULL},
/* 0F45 */ { "TIBETAN LETTER CA",NULL},
/* 0F46 */ { "TIBETAN LETTER CHA",NULL},
/* 0F47 */ { "TIBETAN LETTER JA",NULL},
/* 0F48 */ { NULL,NULL},
/* 0F49 */ { "TIBETAN LETTER NYA",NULL},
/* 0F4A */ { "TIBETAN LETTER TTA",NULL},
/* 0F4B */ { "TIBETAN LETTER TTHA",NULL},
/* 0F4C */ { "TIBETAN LETTER DDA",NULL},
/* 0F4D */ { "TIBETAN LETTER DDHA","	: 0F4C 0FB7"},
/* 0F4E */ { "TIBETAN LETTER NNA",NULL},
/* 0F4F */ { "TIBETAN LETTER TA",NULL},
/* 0F50 */ { "TIBETAN LETTER THA",NULL},
/* 0F51 */ { "TIBETAN LETTER DA",NULL},
/* 0F52 */ { "TIBETAN LETTER DHA","	: 0F51 0FB7"},
/* 0F53 */ { "TIBETAN LETTER NA",NULL},
/* 0F54 */ { "TIBETAN LETTER PA",NULL},
/* 0F55 */ { "TIBETAN LETTER PHA",NULL},
/* 0F56 */ { "TIBETAN LETTER BA",NULL},
/* 0F57 */ { "TIBETAN LETTER BHA","	: 0F56 0FB7"},
/* 0F58 */ { "TIBETAN LETTER MA",NULL},
/* 0F59 */ { "TIBETAN LETTER TSA",NULL},
/* 0F5A */ { "TIBETAN LETTER TSHA",NULL},
/* 0F5B */ { "TIBETAN LETTER DZA",NULL},
/* 0F5C */ { "TIBETAN LETTER DZHA","	: 0F5B 0FB7"},
/* 0F5D */ { "TIBETAN LETTER WA",NULL},
/* 0F5E */ { "TIBETAN LETTER ZHA",NULL},
/* 0F5F */ { "TIBETAN LETTER ZA",NULL},
/* 0F60 */ { "TIBETAN LETTER -A",NULL},
/* 0F61 */ { "TIBETAN LETTER YA",NULL},
/* 0F62 */ { "TIBETAN LETTER RA","	* when followed by a subjoined letter = ra mgo"},
/* 0F63 */ { "TIBETAN LETTER LA",NULL},
/* 0F64 */ { "TIBETAN LETTER SHA",NULL},
/* 0F65 */ { "TIBETAN LETTER SSA","	= reversed sha"},
/* 0F66 */ { "TIBETAN LETTER SA",NULL},
/* 0F67 */ { "TIBETAN LETTER HA",NULL},
/* 0F68 */ { "TIBETAN LETTER A","	* base for dependent vowels"},
/* 0F69 */ { "TIBETAN LETTER KSSA","	: 0F40 0FB5"},
/* 0F6A */ { "TIBETAN LETTER FIXED-FORM RA","	* fixed-form letter not showing the shape variation of the ordinary ra\n"
	"	* used only in transliteration and transcription\n"
	"	x (tibetan letter ra - 0F62)"},
/* 0F6B */ { "TIBETAN LETTER KKA",NULL},
/* 0F6C */ { "TIBETAN LETTER RRA",NULL},
/* 0F6D */ { NULL,NULL},
/* 0F6E */ { NULL,NULL},
/* 0F6F */ { NULL,NULL},
/* 0F70 */ { NULL,NULL},
/* 0F71 */ { "TIBETAN VOWEL SIGN AA","	= a-chung\n"
	"	* common, vowel-lengthening mark"},
/* 0F72 */ { "TIBETAN VOWEL SIGN I",NULL},
/* 0F73 */ { "TIBETAN VOWEL SIGN II","	* use of this character is discouraged\n"
	"	: 0F71 0F72"},
/* 0F74 */ { "TIBETAN VOWEL SIGN U",NULL},
/* 0F75 */ { "TIBETAN VOWEL SIGN UU","	* use of this character is discouraged\n"
	"	: 0F71 0F74"},
/* 0F76 */ { "TIBETAN VOWEL SIGN VOCALIC R","	: 0FB2 0F80"},
/* 0F77 */ { "TIBETAN VOWEL SIGN VOCALIC RR","	* this character is deprecated and its use is strongly discouraged\n"
	"	# 0FB2 0F81"},
/* 0F78 */ { "TIBETAN VOWEL SIGN VOCALIC L","	: 0FB3 0F80"},
/* 0F79 */ { "TIBETAN VOWEL SIGN VOCALIC LL","	* this character is deprecated and its use is strongly discouraged\n"
	"	# 0FB3 0F81"},
/* 0F7A */ { "TIBETAN VOWEL SIGN E",NULL},
/* 0F7B */ { "TIBETAN VOWEL SIGN EE",NULL},
/* 0F7C */ { "TIBETAN VOWEL SIGN O",NULL},
/* 0F7D */ { "TIBETAN VOWEL SIGN OO",NULL},
/* 0F7E */ { "TIBETAN SIGN RJES SU NGA RO","	= anusvara"},
/* 0F7F */ { "TIBETAN SIGN RNAM BCAD","	= visarga"},
/* 0F80 */ { "TIBETAN VOWEL SIGN REVERSED I",NULL},
/* 0F81 */ { "TIBETAN VOWEL SIGN REVERSED II","	* use of this character is discouraged\n"
	"	: 0F71 0F80"},
/* 0F82 */ { "TIBETAN SIGN NYI ZLA NAA DA",NULL},
/* 0F83 */ { "TIBETAN SIGN SNA LDAN","	x (devanagari sign candrabindu - 0901)"},
/* 0F84 */ { "TIBETAN MARK HALANTA","	= srog med\n"
	"	x (devanagari sign virama - 094D)"},
/* 0F85 */ { "TIBETAN MARK PALUTA","	* transliteration of Sanskrit avagraha\n"
	"	x (devanagari sign avagraha - 093D)"},
/* 0F86 */ { "TIBETAN SIGN LCI RTAGS",NULL},
/* 0F87 */ { "TIBETAN SIGN YANG RTAGS",NULL},
/* 0F88 */ { "TIBETAN SIGN LCE TSA CAN","	x (vedic sign jihvamuliya - 1CF5)\n"
	"	x (mongolian letter ali gali damaru - 1882)"},
/* 0F89 */ { "TIBETAN SIGN MCHU CAN","	x (vedic sign upadhmaniya - 1CF6)\n"
	"	x (mongolian letter ali gali inverted ubadama - 1884)"},
/* 0F8A */ { "TIBETAN SIGN GRU CAN RGYINGS","	* always followed by 0F82"},
/* 0F8B */ { "TIBETAN SIGN GRU MED RGYINGS",NULL},
/* 0F8C */ { "TIBETAN SIGN INVERTED MCHU CAN","	x (mongolian letter ali gali ubadama - 1883)"},
/* 0F8D */ { "TIBETAN SUBJOINED SIGN LCE TSA CAN",NULL},
/* 0F8E */ { "TIBETAN SUBJOINED SIGN MCHU CAN",NULL},
/* 0F8F */ { "TIBETAN SUBJOINED SIGN INVERTED MCHU CAN",NULL},
/* 0F90 */ { "TIBETAN SUBJOINED LETTER KA",NULL},
/* 0F91 */ { "TIBETAN SUBJOINED LETTER KHA",NULL},
/* 0F92 */ { "TIBETAN SUBJOINED LETTER GA",NULL},
/* 0F93 */ { "TIBETAN SUBJOINED LETTER GHA","	: 0F92 0FB7"},
/* 0F94 */ { "TIBETAN SUBJOINED LETTER NGA",NULL},
/* 0F95 */ { "TIBETAN SUBJOINED LETTER CA",NULL},
/* 0F96 */ { "TIBETAN SUBJOINED LETTER CHA",NULL},
/* 0F97 */ { "TIBETAN SUBJOINED LETTER JA",NULL},
/* 0F98 */ { NULL,NULL},
/* 0F99 */ { "TIBETAN SUBJOINED LETTER NYA",NULL},
/* 0F9A */ { "TIBETAN SUBJOINED LETTER TTA",NULL},
/* 0F9B */ { "TIBETAN SUBJOINED LETTER TTHA",NULL},
/* 0F9C */ { "TIBETAN SUBJOINED LETTER DDA",NULL},
/* 0F9D */ { "TIBETAN SUBJOINED LETTER DDHA","	: 0F9C 0FB7"},
/* 0F9E */ { "TIBETAN SUBJOINED LETTER NNA",NULL},
/* 0F9F */ { "TIBETAN SUBJOINED LETTER TA",NULL},
/* 0FA0 */ { "TIBETAN SUBJOINED LETTER THA",NULL},
/* 0FA1 */ { "TIBETAN SUBJOINED LETTER DA",NULL},
/* 0FA2 */ { "TIBETAN SUBJOINED LETTER DHA","	: 0FA1 0FB7"},
/* 0FA3 */ { "TIBETAN SUBJOINED LETTER NA",NULL},
/* 0FA4 */ { "TIBETAN SUBJOINED LETTER PA",NULL},
/* 0FA5 */ { "TIBETAN SUBJOINED LETTER PHA",NULL},
/* 0FA6 */ { "TIBETAN SUBJOINED LETTER BA",NULL},
/* 0FA7 */ { "TIBETAN SUBJOINED LETTER BHA","	: 0FA6 0FB7"},
/* 0FA8 */ { "TIBETAN SUBJOINED LETTER MA",NULL},
/* 0FA9 */ { "TIBETAN SUBJOINED LETTER TSA",NULL},
/* 0FAA */ { "TIBETAN SUBJOINED LETTER TSHA",NULL},
/* 0FAB */ { "TIBETAN SUBJOINED LETTER DZA",NULL},
/* 0FAC */ { "TIBETAN SUBJOINED LETTER DZHA","	: 0FAB 0FB7"},
/* 0FAD */ { "TIBETAN SUBJOINED LETTER WA","	= wa-zur, wa-btags (wa ta)"},
/* 0FAE */ { "TIBETAN SUBJOINED LETTER ZHA",NULL},
/* 0FAF */ { "TIBETAN SUBJOINED LETTER ZA",NULL},
/* 0FB0 */ { "TIBETAN SUBJOINED LETTER -A","	= a-chung\n"
	"	* rare, only used for full-sized subjoined letter\n"
	"	x (tibetan vowel sign aa - 0F71)"},
/* 0FB1 */ { "TIBETAN SUBJOINED LETTER YA","	= ya-btags (ya ta)"},
/* 0FB2 */ { "TIBETAN SUBJOINED LETTER RA","	= ra-btags (ra ta)"},
/* 0FB3 */ { "TIBETAN SUBJOINED LETTER LA",NULL},
/* 0FB4 */ { "TIBETAN SUBJOINED LETTER SHA",NULL},
/* 0FB5 */ { "TIBETAN SUBJOINED LETTER SSA","	= reversed subjoined sha"},
/* 0FB6 */ { "TIBETAN SUBJOINED LETTER SA",NULL},
/* 0FB7 */ { "TIBETAN SUBJOINED LETTER HA",NULL},
/* 0FB8 */ { "TIBETAN SUBJOINED LETTER A",NULL},
/* 0FB9 */ { "TIBETAN SUBJOINED LETTER KSSA","	: 0F90 0FB5"},
/* 0FBA */ { "TIBETAN SUBJOINED LETTER FIXED-FORM WA",NULL},
/* 0FBB */ { "TIBETAN SUBJOINED LETTER FIXED-FORM YA",NULL},
/* 0FBC */ { "TIBETAN SUBJOINED LETTER FIXED-FORM RA",NULL},
/* 0FBD */ { NULL,NULL},
/* 0FBE */ { "TIBETAN KU RU KHA","	* often repeated three times; indicates a refrain"},
/* 0FBF */ { "TIBETAN KU RU KHA BZHI MIG CAN","	* marks point of text insertion or annotation\n"
	"	x (reference mark - 203B)"},
/* 0FC0 */ { "TIBETAN CANTILLATION SIGN HEAVY BEAT","	* marks a heavy drum beat"},
/* 0FC1 */ { "TIBETAN CANTILLATION SIGN LIGHT BEAT","	* marks a light drum beat"},
/* 0FC2 */ { "TIBETAN CANTILLATION SIGN CANG TE-U","	* symbol of a small Tibetan hand drum"},
/* 0FC3 */ { "TIBETAN CANTILLATION SIGN SBUB -CHAL","	* symbol of a Tibetan cymbal"},
/* 0FC4 */ { "TIBETAN SYMBOL DRIL BU","	* symbol of a Tibetan hand bell\n"
	"	x (bell - 1F514)"},
/* 0FC5 */ { "TIBETAN SYMBOL RDO RJE",NULL},
/* 0FC6 */ { "TIBETAN SYMBOL PADMA GDAN",NULL},
/* 0FC7 */ { "TIBETAN SYMBOL RDO RJE RGYA GRAM",NULL},
/* 0FC8 */ { "TIBETAN SYMBOL PHUR PA",NULL},
/* 0FC9 */ { "TIBETAN SYMBOL NOR BU",NULL},
/* 0FCA */ { "TIBETAN SYMBOL NOR BU NYIS -KHYIL","	* the double body symbol\n"
	"	x (yin yang - 262F)"},
/* 0FCB */ { "TIBETAN SYMBOL NOR BU GSUM -KHYIL","	* the tri-kaya or triple body symbol"},
/* 0FCC */ { "TIBETAN SYMBOL NOR BU BZHI -KHYIL","	* the quadruple body symbol, a form of the svasti sign\n"
	"	x (left-facing svasti sign - 0FD6)"},
/* 0FCD */ { NULL,NULL},
/* 0FCE */ { "TIBETAN SIGN RDEL NAG RDEL DKAR","	= dena deka\n"
	"	* signifies bad luck earlier, good luck later\n"
	"	x (tibetan sign rdel dkar rdel nag - 0F1F)"},
/* 0FCF */ { "TIBETAN SIGN RDEL NAG GSUM",NULL},
/* 0FD0 */ { "TIBETAN MARK BSKA- SHOG GI MGO RGYAN","	% TIBETAN MARK BKA- SHOG GI MGO RGYAN\n"
	"	* used in Bhutan"},
/* 0FD1 */ { "TIBETAN MARK MNYAM YIG GI MGO RGYAN","	* used in Bhutan"},
/* 0FD2 */ { "TIBETAN MARK NYIS TSHEG","	= nyi tsek"},
/* 0FD3 */ { "TIBETAN MARK INITIAL BRDA RNYING YIG MGO MDUN MA","	= da nying yik go dun ma"},
/* 0FD4 */ { "TIBETAN MARK CLOSING BRDA RNYING YIG MGO SGAB MA","	= da nying yik go kab ma"},
/* 0FD5 */ { "RIGHT-FACING SVASTI SIGN","	= gyung drung nang -khor\n"
	"	* symbol of good luck and well-being in India\n"
	"	x 5350"},
/* 0FD6 */ { "LEFT-FACING SVASTI SIGN","	= gyung drung phyi -khor\n"
	"	x 534D"},
/* 0FD7 */ { "RIGHT-FACING SVASTI SIGN WITH DOTS","	= gyung drung nang -khor bzhi mig can"},
/* 0FD8 */ { "LEFT-FACING SVASTI SIGN WITH DOTS","	= gyung drung phyi -khor bzhi mig can"},
/* 0FD9 */ { "TIBETAN MARK LEADING MCHAN RTAGS",NULL},
/* 0FDA */ { "TIBETAN MARK TRAILING MCHAN RTAGS",NULL},
/* 0FDB */ { NULL,NULL},
/* 0FDC */ { NULL,NULL},
/* 0FDD */ { NULL,NULL},
/* 0FDE */ { NULL,NULL},
/* 0FDF */ { NULL,NULL},
/* 0FE0 */ { NULL,NULL},
/* 0FE1 */ { NULL,NULL},
/* 0FE2 */ { NULL,NULL},
/* 0FE3 */ { NULL,NULL},
/* 0FE4 */ { NULL,NULL},
/* 0FE5 */ { NULL,NULL},
/* 0FE6 */ { NULL,NULL},
/* 0FE7 */ { NULL,NULL},
/* 0FE8 */ { NULL,NULL},
/* 0FE9 */ { NULL,NULL},
/* 0FEA */ { NULL,NULL},
/* 0FEB */ { NULL,NULL},
/* 0FEC */ { NULL,NULL},
/* 0FED */ { NULL,NULL},
/* 0FEE */ { NULL,NULL},
/* 0FEF */ { NULL,NULL},
/* 0FF0 */ { NULL,NULL},
/* 0FF1 */ { NULL,NULL},
/* 0FF2 */ { NULL,NULL},
/* 0FF3 */ { NULL,NULL},
/* 0FF4 */ { NULL,NULL},
/* 0FF5 */ { NULL,NULL},
/* 0FF6 */ { NULL,NULL},
/* 0FF7 */ { NULL,NULL},
/* 0FF8 */ { NULL,NULL},
/* 0FF9 */ { NULL,NULL},
/* 0FFA */ { NULL,NULL},
/* 0FFB */ { NULL,NULL},
/* 0FFC */ { NULL,NULL},
/* 0FFD */ { NULL,NULL},
/* 0FFE */ { NULL,NULL},
/* 0FFF */ { NULL,NULL}
};

UN_DLL_LOCAL
static const struct unicode_nameannot una_00_10[] = {
/* 1000 */ { "MYANMAR LETTER KA","	~ 1000 FE00 dotted form"},
/* 1001 */ { "MYANMAR LETTER KHA",NULL},
/* 1002 */ { "MYANMAR LETTER GA","	~ 1002 FE00 dotted form"},
/* 1003 */ { "MYANMAR LETTER GHA",NULL},
/* 1004 */ { "MYANMAR LETTER NGA","	~ 1004 FE00 dotted form"},
/* 1005 */ { "MYANMAR LETTER CA",NULL},
/* 1006 */ { "MYANMAR LETTER CHA",NULL},
/* 1007 */ { "MYANMAR LETTER JA",NULL},
/* 1008 */ { "MYANMAR LETTER JHA",NULL},
/* 1009 */ { "MYANMAR LETTER NYA",NULL},
/* 100A */ { "MYANMAR LETTER NNYA",NULL},
/* 100B */ { "MYANMAR LETTER TTA",NULL},
/* 100C */ { "MYANMAR LETTER TTHA",NULL},
/* 100D */ { "MYANMAR LETTER DDA",NULL},
/* 100E */ { "MYANMAR LETTER DDHA",NULL},
/* 100F */ { "MYANMAR LETTER NNA",NULL},
/* 1010 */ { "MYANMAR LETTER TA","	~ 1010 FE00 dotted form"},
/* 1011 */ { "MYANMAR LETTER THA","	~ 1011 FE00 dotted form"},
/* 1012 */ { "MYANMAR LETTER DA",NULL},
/* 1013 */ { "MYANMAR LETTER DHA",NULL},
/* 1014 */ { "MYANMAR LETTER NA",NULL},
/* 1015 */ { "MYANMAR LETTER PA","	~ 1015 FE00 dotted form"},
/* 1016 */ { "MYANMAR LETTER PHA",NULL},
/* 1017 */ { "MYANMAR LETTER BA",NULL},
/* 1018 */ { "MYANMAR LETTER BHA",NULL},
/* 1019 */ { "MYANMAR LETTER MA","	~ 1019 FE00 dotted form"},
/* 101A */ { "MYANMAR LETTER YA","	~ 101A FE00 dotted form"},
/* 101B */ { "MYANMAR LETTER RA",NULL},
/* 101C */ { "MYANMAR LETTER LA","	~ 101C FE00 dotted form"},
/* 101D */ { "MYANMAR LETTER WA","	~ 101D FE00 dotted form"},
/* 101E */ { "MYANMAR LETTER SA",NULL},
/* 101F */ { "MYANMAR LETTER HA",NULL},
/* 1020 */ { "MYANMAR LETTER LLA",NULL},
/* 1021 */ { "MYANMAR LETTER A","	* also represents the glottal stop as a consonant"},
/* 1022 */ { "MYANMAR LETTER SHAN A","	~ 1022 FE00 dotted form"},
/* 1023 */ { "MYANMAR LETTER I",NULL},
/* 1024 */ { "MYANMAR LETTER II",NULL},
/* 1025 */ { "MYANMAR LETTER U",NULL},
/* 1026 */ { "MYANMAR LETTER UU","	: 1025 102E"},
/* 1027 */ { "MYANMAR LETTER E",NULL},
/* 1028 */ { "MYANMAR LETTER MON E",NULL},
/* 1029 */ { "MYANMAR LETTER O",NULL},
/* 102A */ { "MYANMAR LETTER AU",NULL},
/* 102B */ { "MYANMAR VOWEL SIGN TALL AA",NULL},
/* 102C */ { "MYANMAR VOWEL SIGN AA",NULL},
/* 102D */ { "MYANMAR VOWEL SIGN I",NULL},
/* 102E */ { "MYANMAR VOWEL SIGN II",NULL},
/* 102F */ { "MYANMAR VOWEL SIGN U",NULL},
/* 1030 */ { "MYANMAR VOWEL SIGN UU",NULL},
/* 1031 */ { "MYANMAR VOWEL SIGN E","	* stands to the left of the consonant\n"
	"	~ 1031 FE00 dotted form"},
/* 1032 */ { "MYANMAR VOWEL SIGN AI",NULL},
/* 1033 */ { "MYANMAR VOWEL SIGN MON II",NULL},
/* 1034 */ { "MYANMAR VOWEL SIGN MON O",NULL},
/* 1035 */ { "MYANMAR VOWEL SIGN E ABOVE",NULL},
/* 1036 */ { "MYANMAR SIGN ANUSVARA",NULL},
/* 1037 */ { "MYANMAR SIGN DOT BELOW","	= aukmyit\n"
	"	* a tone mark"},
/* 1038 */ { "MYANMAR SIGN VISARGA",NULL},
/* 1039 */ { "MYANMAR SIGN VIRAMA",NULL},
/* 103A */ { "MYANMAR SIGN ASAT","	= killer (always rendered visibly)"},
/* 103B */ { "MYANMAR CONSONANT SIGN MEDIAL YA",NULL},
/* 103C */ { "MYANMAR CONSONANT SIGN MEDIAL RA",NULL},
/* 103D */ { "MYANMAR CONSONANT SIGN MEDIAL WA",NULL},
/* 103E */ { "MYANMAR CONSONANT SIGN MEDIAL HA",NULL},
/* 103F */ { "MYANMAR LETTER GREAT SA",NULL},
/* 1040 */ { "MYANMAR DIGIT ZERO",NULL},
/* 1041 */ { "MYANMAR DIGIT ONE",NULL},
/* 1042 */ { "MYANMAR DIGIT TWO",NULL},
/* 1043 */ { "MYANMAR DIGIT THREE",NULL},
/* 1044 */ { "MYANMAR DIGIT FOUR",NULL},
/* 1045 */ { "MYANMAR DIGIT FIVE",NULL},
/* 1046 */ { "MYANMAR DIGIT SIX",NULL},
/* 1047 */ { "MYANMAR DIGIT SEVEN",NULL},
/* 1048 */ { "MYANMAR DIGIT EIGHT",NULL},
/* 1049 */ { "MYANMAR DIGIT NINE",NULL},
/* 104A */ { "MYANMAR SIGN LITTLE SECTION","	x (devanagari danda - 0964)"},
/* 104B */ { "MYANMAR SIGN SECTION","	x (devanagari double danda - 0965)"},
/* 104C */ { "MYANMAR SYMBOL LOCATIVE",NULL},
/* 104D */ { "MYANMAR SYMBOL COMPLETED",NULL},
/* 104E */ { "MYANMAR SYMBOL AFOREMENTIONED",NULL},
/* 104F */ { "MYANMAR SYMBOL GENITIVE",NULL},
/* 1050 */ { "MYANMAR LETTER SHA",NULL},
/* 1051 */ { "MYANMAR LETTER SSA",NULL},
/* 1052 */ { "MYANMAR LETTER VOCALIC R",NULL},
/* 1053 */ { "MYANMAR LETTER VOCALIC RR",NULL},
/* 1054 */ { "MYANMAR LETTER VOCALIC L",NULL},
/* 1055 */ { "MYANMAR LETTER VOCALIC LL",NULL},
/* 1056 */ { "MYANMAR VOWEL SIGN VOCALIC R",NULL},
/* 1057 */ { "MYANMAR VOWEL SIGN VOCALIC RR",NULL},
/* 1058 */ { "MYANMAR VOWEL SIGN VOCALIC L",NULL},
/* 1059 */ { "MYANMAR VOWEL SIGN VOCALIC LL",NULL},
/* 105A */ { "MYANMAR LETTER MON NGA",NULL},
/* 105B */ { "MYANMAR LETTER MON JHA",NULL},
/* 105C */ { "MYANMAR LETTER MON BBA",NULL},
/* 105D */ { "MYANMAR LETTER MON BBE",NULL},
/* 105E */ { "MYANMAR CONSONANT SIGN MON MEDIAL NA",NULL},
/* 105F */ { "MYANMAR CONSONANT SIGN MON MEDIAL MA",NULL},
/* 1060 */ { "MYANMAR CONSONANT SIGN MON MEDIAL LA",NULL},
/* 1061 */ { "MYANMAR LETTER SGAW KAREN SHA",NULL},
/* 1062 */ { "MYANMAR VOWEL SIGN SGAW KAREN EU",NULL},
/* 1063 */ { "MYANMAR TONE MARK SGAW KAREN HATHI",NULL},
/* 1064 */ { "MYANMAR TONE MARK SGAW KAREN KE PHO",NULL},
/* 1065 */ { "MYANMAR LETTER WESTERN PWO KAREN THA",NULL},
/* 1066 */ { "MYANMAR LETTER WESTERN PWO KAREN PWA",NULL},
/* 1067 */ { "MYANMAR VOWEL SIGN WESTERN PWO KAREN EU",NULL},
/* 1068 */ { "MYANMAR VOWEL SIGN WESTERN PWO KAREN UE",NULL},
/* 1069 */ { "MYANMAR SIGN WESTERN PWO KAREN TONE-1",NULL},
/* 106A */ { "MYANMAR SIGN WESTERN PWO KAREN TONE-2",NULL},
/* 106B */ { "MYANMAR SIGN WESTERN PWO KAREN TONE-3",NULL},
/* 106C */ { "MYANMAR SIGN WESTERN PWO KAREN TONE-4",NULL},
/* 106D */ { "MYANMAR SIGN WESTERN PWO KAREN TONE-5",NULL},
/* 106E */ { "MYANMAR LETTER EASTERN PWO KAREN NNA",NULL},
/* 106F */ { "MYANMAR LETTER EASTERN PWO KAREN YWA",NULL},
/* 1070 */ { "MYANMAR LETTER EASTERN PWO KAREN GHWA",NULL},
/* 1071 */ { "MYANMAR VOWEL SIGN GEBA KAREN I",NULL},
/* 1072 */ { "MYANMAR VOWEL SIGN KAYAH OE",NULL},
/* 1073 */ { "MYANMAR VOWEL SIGN KAYAH U",NULL},
/* 1074 */ { "MYANMAR VOWEL SIGN KAYAH EE",NULL},
/* 1075 */ { "MYANMAR LETTER SHAN KA","	~ 1075 FE00 dotted form"},
/* 1076 */ { "MYANMAR LETTER SHAN KHA",NULL},
/* 1077 */ { "MYANMAR LETTER SHAN GA",NULL},
/* 1078 */ { "MYANMAR LETTER SHAN CA","	~ 1078 FE00 dotted form"},
/* 1079 */ { "MYANMAR LETTER SHAN ZA",NULL},
/* 107A */ { "MYANMAR LETTER SHAN NYA","	~ 107A FE00 dotted form"},
/* 107B */ { "MYANMAR LETTER SHAN DA",NULL},
/* 107C */ { "MYANMAR LETTER SHAN NA",NULL},
/* 107D */ { "MYANMAR LETTER SHAN PHA",NULL},
/* 107E */ { "MYANMAR LETTER SHAN FA",NULL},
/* 107F */ { "MYANMAR LETTER SHAN BA",NULL},
/* 1080 */ { "MYANMAR LETTER SHAN THA","	~ 1080 FE00 dotted form"},
/* 1081 */ { "MYANMAR LETTER SHAN HA",NULL},
/* 1082 */ { "MYANMAR CONSONANT SIGN SHAN MEDIAL WA",NULL},
/* 1083 */ { "MYANMAR VOWEL SIGN SHAN AA",NULL},
/* 1084 */ { "MYANMAR VOWEL SIGN SHAN E",NULL},
/* 1085 */ { "MYANMAR VOWEL SIGN SHAN E ABOVE",NULL},
/* 1086 */ { "MYANMAR VOWEL SIGN SHAN FINAL Y",NULL},
/* 1087 */ { "MYANMAR SIGN SHAN TONE-2",NULL},
/* 1088 */ { "MYANMAR SIGN SHAN TONE-3",NULL},
/* 1089 */ { "MYANMAR SIGN SHAN TONE-5",NULL},
/* 108A */ { "MYANMAR SIGN SHAN TONE-6",NULL},
/* 108B */ { "MYANMAR SIGN SHAN COUNCIL TONE-2",NULL},
/* 108C */ { "MYANMAR SIGN SHAN COUNCIL TONE-3",NULL},
/* 108D */ { "MYANMAR SIGN SHAN COUNCIL EMPHATIC TONE",NULL},
/* 108E */ { "MYANMAR LETTER RUMAI PALAUNG FA",NULL},
/* 108F */ { "MYANMAR SIGN RUMAI PALAUNG TONE-5",NULL},
/* 1090 */ { "MYANMAR SHAN DIGIT ZERO",NULL},
/* 1091 */ { "MYANMAR SHAN DIGIT ONE",NULL},
/* 1092 */ { "MYANMAR SHAN DIGIT TWO",NULL},
/* 1093 */ { "MYANMAR SHAN DIGIT THREE",NULL},
/* 1094 */ { "MYANMAR SHAN DIGIT FOUR",NULL},
/* 1095 */ { "MYANMAR SHAN DIGIT FIVE",NULL},
/* 1096 */ { "MYANMAR SHAN DIGIT SIX",NULL},
/* 1097 */ { "MYANMAR SHAN DIGIT SEVEN",NULL},
/* 1098 */ { "MYANMAR SHAN DIGIT EIGHT",NULL},
/* 1099 */ { "MYANMAR SHAN DIGIT NINE",NULL},
/* 109A */ { "MYANMAR SIGN KHAMTI TONE-1",NULL},
/* 109B */ { "MYANMAR SIGN KHAMTI TONE-3",NULL},
/* 109C */ { "MYANMAR VOWEL SIGN AITON A",NULL},
/* 109D */ { "MYANMAR VOWEL SIGN AITON AI",NULL},
/* 109E */ { "MYANMAR SYMBOL SHAN ONE",NULL},
/* 109F */ { "MYANMAR SYMBOL SHAN EXCLAMATION",NULL},
/* 10A0 */ { "GEORGIAN CAPITAL LETTER AN","	x (georgian small letter an - 2D00)"},
/* 10A1 */ { "GEORGIAN CAPITAL LETTER BAN",NULL},
/* 10A2 */ { "GEORGIAN CAPITAL LETTER GAN",NULL},
/* 10A3 */ { "GEORGIAN CAPITAL LETTER DON",NULL},
/* 10A4 */ { "GEORGIAN CAPITAL LETTER EN",NULL},
/* 10A5 */ { "GEORGIAN CAPITAL LETTER VIN",NULL},
/* 10A6 */ { "GEORGIAN CAPITAL LETTER ZEN",NULL},
/* 10A7 */ { "GEORGIAN CAPITAL LETTER TAN",NULL},
/* 10A8 */ { "GEORGIAN CAPITAL LETTER IN",NULL},
/* 10A9 */ { "GEORGIAN CAPITAL LETTER KAN",NULL},
/* 10AA */ { "GEORGIAN CAPITAL LETTER LAS",NULL},
/* 10AB */ { "GEORGIAN CAPITAL LETTER MAN",NULL},
/* 10AC */ { "GEORGIAN CAPITAL LETTER NAR",NULL},
/* 10AD */ { "GEORGIAN CAPITAL LETTER ON",NULL},
/* 10AE */ { "GEORGIAN CAPITAL LETTER PAR",NULL},
/* 10AF */ { "GEORGIAN CAPITAL LETTER ZHAR",NULL},
/* 10B0 */ { "GEORGIAN CAPITAL LETTER RAE",NULL},
/* 10B1 */ { "GEORGIAN CAPITAL LETTER SAN",NULL},
/* 10B2 */ { "GEORGIAN CAPITAL LETTER TAR",NULL},
/* 10B3 */ { "GEORGIAN CAPITAL LETTER UN",NULL},
/* 10B4 */ { "GEORGIAN CAPITAL LETTER PHAR",NULL},
/* 10B5 */ { "GEORGIAN CAPITAL LETTER KHAR",NULL},
/* 10B6 */ { "GEORGIAN CAPITAL LETTER GHAN",NULL},
/* 10B7 */ { "GEORGIAN CAPITAL LETTER QAR",NULL},
/* 10B8 */ { "GEORGIAN CAPITAL LETTER SHIN",NULL},
/* 10B9 */ { "GEORGIAN CAPITAL LETTER CHIN",NULL},
/* 10BA */ { "GEORGIAN CAPITAL LETTER CAN",NULL},
/* 10BB */ { "GEORGIAN CAPITAL LETTER JIL",NULL},
/* 10BC */ { "GEORGIAN CAPITAL LETTER CIL",NULL},
/* 10BD */ { "GEORGIAN CAPITAL LETTER CHAR",NULL},
/* 10BE */ { "GEORGIAN CAPITAL LETTER XAN",NULL},
/* 10BF */ { "GEORGIAN CAPITAL LETTER JHAN",NULL},
/* 10C0 */ { "GEORGIAN CAPITAL LETTER HAE",NULL},
/* 10C1 */ { "GEORGIAN CAPITAL LETTER HE",NULL},
/* 10C2 */ { "GEORGIAN CAPITAL LETTER HIE",NULL},
/* 10C3 */ { "GEORGIAN CAPITAL LETTER WE",NULL},
/* 10C4 */ { "GEORGIAN CAPITAL LETTER HAR",NULL},
/* 10C5 */ { "GEORGIAN CAPITAL LETTER HOE",NULL},
/* 10C6 */ { NULL,NULL},
/* 10C7 */ { "GEORGIAN CAPITAL LETTER YN",NULL},
/* 10C8 */ { NULL,NULL},
/* 10C9 */ { NULL,NULL},
/* 10CA */ { NULL,NULL},
/* 10CB */ { NULL,NULL},
/* 10CC */ { NULL,NULL},
/* 10CD */ { "GEORGIAN CAPITAL LETTER AEN",NULL},
/* 10CE */ { NULL,NULL},
/* 10CF */ { NULL,NULL},
/* 10D0 */ { "GEORGIAN LETTER AN","	x (georgian mtavruli capital letter an - 1C90)"},
/* 10D1 */ { "GEORGIAN LETTER BAN",NULL},
/* 10D2 */ { "GEORGIAN LETTER GAN",NULL},
/* 10D3 */ { "GEORGIAN LETTER DON",NULL},
/* 10D4 */ { "GEORGIAN LETTER EN",NULL},
/* 10D5 */ { "GEORGIAN LETTER VIN",NULL},
/* 10D6 */ { "GEORGIAN LETTER ZEN",NULL},
/* 10D7 */ { "GEORGIAN LETTER TAN",NULL},
/* 10D8 */ { "GEORGIAN LETTER IN",NULL},
/* 10D9 */ { "GEORGIAN LETTER KAN",NULL},
/* 10DA */ { "GEORGIAN LETTER LAS",NULL},
/* 10DB */ { "GEORGIAN LETTER MAN",NULL},
/* 10DC */ { "GEORGIAN LETTER NAR",NULL},
/* 10DD */ { "GEORGIAN LETTER ON",NULL},
/* 10DE */ { "GEORGIAN LETTER PAR",NULL},
/* 10DF */ { "GEORGIAN LETTER ZHAR",NULL},
/* 10E0 */ { "GEORGIAN LETTER RAE",NULL},
/* 10E1 */ { "GEORGIAN LETTER SAN",NULL},
/* 10E2 */ { "GEORGIAN LETTER TAR",NULL},
/* 10E3 */ { "GEORGIAN LETTER UN",NULL},
/* 10E4 */ { "GEORGIAN LETTER PHAR",NULL},
/* 10E5 */ { "GEORGIAN LETTER KHAR",NULL},
/* 10E6 */ { "GEORGIAN LETTER GHAN",NULL},
/* 10E7 */ { "GEORGIAN LETTER QAR",NULL},
/* 10E8 */ { "GEORGIAN LETTER SHIN",NULL},
/* 10E9 */ { "GEORGIAN LETTER CHIN",NULL},
/* 10EA */ { "GEORGIAN LETTER CAN",NULL},
/* 10EB */ { "GEORGIAN LETTER JIL",NULL},
/* 10EC */ { "GEORGIAN LETTER CIL",NULL},
/* 10ED */ { "GEORGIAN LETTER CHAR",NULL},
/* 10EE */ { "GEORGIAN LETTER XAN",NULL},
/* 10EF */ { "GEORGIAN LETTER JHAN",NULL},
/* 10F0 */ { "GEORGIAN LETTER HAE",NULL},
/* 10F1 */ { "GEORGIAN LETTER HE",NULL},
/* 10F2 */ { "GEORGIAN LETTER HIE",NULL},
/* 10F3 */ { "GEORGIAN LETTER WE",NULL},
/* 10F4 */ { "GEORGIAN LETTER HAR",NULL},
/* 10F5 */ { "GEORGIAN LETTER HOE",NULL},
/* 10F6 */ { "GEORGIAN LETTER FI",NULL},
/* 10F7 */ { "GEORGIAN LETTER YN",NULL},
/* 10F8 */ { "GEORGIAN LETTER ELIFI",NULL},
/* 10F9 */ { "GEORGIAN LETTER TURNED GAN",NULL},
/* 10FA */ { "GEORGIAN LETTER AIN",NULL},
/* 10FB */ { "GEORGIAN PARAGRAPH SEPARATOR","	x (three dot punctuation - 2056)"},
/* 10FC */ { "MODIFIER LETTER GEORGIAN NAR","	# <super> 10DC"},
/* 10FD */ { "GEORGIAN LETTER AEN",NULL},
/* 10FE */ { "GEORGIAN LETTER HARD SIGN",NULL},
/* 10FF */ { "GEORGIAN LETTER LABIAL SIGN",NULL}
};

UN_DLL_LOCAL
static const struct unicode_nameannot una_00_11[] = {
/* 1100 */ { "HANGUL CHOSEONG KIYEOK","	= G"},
/* 1101 */ { "HANGUL CHOSEONG SSANGKIYEOK","	= GG"},
/* 1102 */ { "HANGUL CHOSEONG NIEUN","	= N"},
/* 1103 */ { "HANGUL CHOSEONG TIKEUT","	= D"},
/* 1104 */ { "HANGUL CHOSEONG SSANGTIKEUT","	= DD"},
/* 1105 */ { "HANGUL CHOSEONG RIEUL","	= R"},
/* 1106 */ { "HANGUL CHOSEONG MIEUM","	= M"},
/* 1107 */ { "HANGUL CHOSEONG PIEUP","	= B"},
/* 1108 */ { "HANGUL CHOSEONG SSANGPIEUP","	= BB"},
/* 1109 */ { "HANGUL CHOSEONG SIOS","	= S\n"
	"	* voiceless lenis alveolar fricative"},
/* 110A */ { "HANGUL CHOSEONG SSANGSIOS","	= SS\n"
	"	* voiceless fortis alveolar fricative"},
/* 110B */ { "HANGUL CHOSEONG IEUNG","	* zero sound"},
/* 110C */ { "HANGUL CHOSEONG CIEUC","	= J\n"
	"	* voiceless or voiced lenis alveolar affricate"},
/* 110D */ { "HANGUL CHOSEONG SSANGCIEUC","	= JJ\n"
	"	* voiceless unaspirated fortis alveolar affricate"},
/* 110E */ { "HANGUL CHOSEONG CHIEUCH","	= C\n"
	"	* voiceless aspirated alveolar affricate"},
/* 110F */ { "HANGUL CHOSEONG KHIEUKH","	= K"},
/* 1110 */ { "HANGUL CHOSEONG THIEUTH","	= T"},
/* 1111 */ { "HANGUL CHOSEONG PHIEUPH","	= P"},
/* 1112 */ { "HANGUL CHOSEONG HIEUH","	= H\n"
	"	* voiceless glottal fricative"},
/* 1113 */ { "HANGUL CHOSEONG NIEUN-KIYEOK",NULL},
/* 1114 */ { "HANGUL CHOSEONG SSANGNIEUN",NULL},
/* 1115 */ { "HANGUL CHOSEONG NIEUN-TIKEUT",NULL},
/* 1116 */ { "HANGUL CHOSEONG NIEUN-PIEUP",NULL},
/* 1117 */ { "HANGUL CHOSEONG TIKEUT-KIYEOK",NULL},
/* 1118 */ { "HANGUL CHOSEONG RIEUL-NIEUN",NULL},
/* 1119 */ { "HANGUL CHOSEONG SSANGRIEUL",NULL},
/* 111A */ { "HANGUL CHOSEONG RIEUL-HIEUH",NULL},
/* 111B */ { "HANGUL CHOSEONG KAPYEOUNRIEUL",NULL},
/* 111C */ { "HANGUL CHOSEONG MIEUM-PIEUP",NULL},
/* 111D */ { "HANGUL CHOSEONG KAPYEOUNMIEUM",NULL},
/* 111E */ { "HANGUL CHOSEONG PIEUP-KIYEOK",NULL},
/* 111F */ { "HANGUL CHOSEONG PIEUP-NIEUN",NULL},
/* 1120 */ { "HANGUL CHOSEONG PIEUP-TIKEUT",NULL},
/* 1121 */ { "HANGUL CHOSEONG PIEUP-SIOS",NULL},
/* 1122 */ { "HANGUL CHOSEONG PIEUP-SIOS-KIYEOK",NULL},
/* 1123 */ { "HANGUL CHOSEONG PIEUP-SIOS-TIKEUT",NULL},
/* 1124 */ { "HANGUL CHOSEONG PIEUP-SIOS-PIEUP",NULL},
/* 1125 */ { "HANGUL CHOSEONG PIEUP-SSANGSIOS",NULL},
/* 1126 */ { "HANGUL CHOSEONG PIEUP-SIOS-CIEUC",NULL},
/* 1127 */ { "HANGUL CHOSEONG PIEUP-CIEUC",NULL},
/* 1128 */ { "HANGUL CHOSEONG PIEUP-CHIEUCH",NULL},
/* 1129 */ { "HANGUL CHOSEONG PIEUP-THIEUTH",NULL},
/* 112A */ { "HANGUL CHOSEONG PIEUP-PHIEUPH",NULL},
/* 112B */ { "HANGUL CHOSEONG KAPYEOUNPIEUP",NULL},
/* 112C */ { "HANGUL CHOSEONG KAPYEOUNSSANGPIEUP",NULL},
/* 112D */ { "HANGUL CHOSEONG SIOS-KIYEOK",NULL},
/* 112E */ { "HANGUL CHOSEONG SIOS-NIEUN",NULL},
/* 112F */ { "HANGUL CHOSEONG SIOS-TIKEUT",NULL},
/* 1130 */ { "HANGUL CHOSEONG SIOS-RIEUL",NULL},
/* 1131 */ { "HANGUL CHOSEONG SIOS-MIEUM",NULL},
/* 1132 */ { "HANGUL CHOSEONG SIOS-PIEUP",NULL},
/* 1133 */ { "HANGUL CHOSEONG SIOS-PIEUP-KIYEOK",NULL},
/* 1134 */ { "HANGUL CHOSEONG SIOS-SSANGSIOS",NULL},
/* 1135 */ { "HANGUL CHOSEONG SIOS-IEUNG",NULL},
/* 1136 */ { "HANGUL CHOSEONG SIOS-CIEUC",NULL},
/* 1137 */ { "HANGUL CHOSEONG SIOS-CHIEUCH",NULL},
/* 1138 */ { "HANGUL CHOSEONG SIOS-KHIEUKH",NULL},
/* 1139 */ { "HANGUL CHOSEONG SIOS-THIEUTH",NULL},
/* 113A */ { "HANGUL CHOSEONG SIOS-PHIEUPH",NULL},
/* 113B */ { "HANGUL CHOSEONG SIOS-HIEUH",NULL},
/* 113C */ { "HANGUL CHOSEONG CHITUEUMSIOS","	* voiceless lenis dental fricative"},
/* 113D */ { "HANGUL CHOSEONG CHITUEUMSSANGSIOS","	* voiceless fortis dental fricative"},
/* 113E */ { "HANGUL CHOSEONG CEONGCHIEUMSIOS","	* voiceless lenis retroflex fricative"},
/* 113F */ { "HANGUL CHOSEONG CEONGCHIEUMSSANGSIOS","	* voiceless fortis retroflex fricative"},
/* 1140 */ { "HANGUL CHOSEONG PANSIOS","	* voiced alveolar fricative"},
/* 1141 */ { "HANGUL CHOSEONG IEUNG-KIYEOK",NULL},
/* 1142 */ { "HANGUL CHOSEONG IEUNG-TIKEUT",NULL},
/* 1143 */ { "HANGUL CHOSEONG IEUNG-MIEUM",NULL},
/* 1144 */ { "HANGUL CHOSEONG IEUNG-PIEUP",NULL},
/* 1145 */ { "HANGUL CHOSEONG IEUNG-SIOS",NULL},
/* 1146 */ { "HANGUL CHOSEONG IEUNG-PANSIOS",NULL},
/* 1147 */ { "HANGUL CHOSEONG SSANGIEUNG",NULL},
/* 1148 */ { "HANGUL CHOSEONG IEUNG-CIEUC",NULL},
/* 1149 */ { "HANGUL CHOSEONG IEUNG-CHIEUCH",NULL},
/* 114A */ { "HANGUL CHOSEONG IEUNG-THIEUTH",NULL},
/* 114B */ { "HANGUL CHOSEONG IEUNG-PHIEUPH",NULL},
/* 114C */ { "HANGUL CHOSEONG YESIEUNG","	* velar nasal consonant"},
/* 114D */ { "HANGUL CHOSEONG CIEUC-IEUNG",NULL},
/* 114E */ { "HANGUL CHOSEONG CHITUEUMCIEUC","	* voiceless or voiced lenis dental affricate"},
/* 114F */ { "HANGUL CHOSEONG CHITUEUMSSANGCIEUC","	* voiceless unaspirated fortis dental affricate"},
/* 1150 */ { "HANGUL CHOSEONG CEONGCHIEUMCIEUC","	* voiceless or voiced lenis retroflex affricate"},
/* 1151 */ { "HANGUL CHOSEONG CEONGCHIEUMSSANGCIEUC","	* voiceless unaspirated fortis retroflex affricate"},
/* 1152 */ { "HANGUL CHOSEONG CHIEUCH-KHIEUKH",NULL},
/* 1153 */ { "HANGUL CHOSEONG CHIEUCH-HIEUH",NULL},
/* 1154 */ { "HANGUL CHOSEONG CHITUEUMCHIEUCH","	* voiceless aspirated dental affricate"},
/* 1155 */ { "HANGUL CHOSEONG CEONGCHIEUMCHIEUCH","	* voiceless aspirated retroflex affricate"},
/* 1156 */ { "HANGUL CHOSEONG PHIEUPH-PIEUP",NULL},
/* 1157 */ { "HANGUL CHOSEONG KAPYEOUNPHIEUPH",NULL},
/* 1158 */ { "HANGUL CHOSEONG SSANGHIEUH",NULL},
/* 1159 */ { "HANGUL CHOSEONG YEORINHIEUH","	* glottal stop"},
/* 115A */ { "HANGUL CHOSEONG KIYEOK-TIKEUT",NULL},
/* 115B */ { "HANGUL CHOSEONG NIEUN-SIOS",NULL},
/* 115C */ { "HANGUL CHOSEONG NIEUN-CIEUC",NULL},
/* 115D */ { "HANGUL CHOSEONG NIEUN-HIEUH",NULL},
/* 115E */ { "HANGUL CHOSEONG TIKEUT-RIEUL",NULL},
/* 115F */ { "HANGUL CHOSEONG FILLER",NULL},
/* 1160 */ { "HANGUL JUNGSEONG FILLER",NULL},
/* 1161 */ { "HANGUL JUNGSEONG A","	= A"},
/* 1162 */ { "HANGUL JUNGSEONG AE","	= AE"},
/* 1163 */ { "HANGUL JUNGSEONG YA","	= YA"},
/* 1164 */ { "HANGUL JUNGSEONG YAE","	= YAE"},
/* 1165 */ { "HANGUL JUNGSEONG EO","	= EO"},
/* 1166 */ { "HANGUL JUNGSEONG E","	= E"},
/* 1167 */ { "HANGUL JUNGSEONG YEO","	= YEO"},
/* 1168 */ { "HANGUL JUNGSEONG YE","	= YE"},
/* 1169 */ { "HANGUL JUNGSEONG O","	= O"},
/* 116A */ { "HANGUL JUNGSEONG WA","	= WA"},
/* 116B */ { "HANGUL JUNGSEONG WAE","	= WAE"},
/* 116C */ { "HANGUL JUNGSEONG OE","	= OE"},
/* 116D */ { "HANGUL JUNGSEONG YO","	= YO"},
/* 116E */ { "HANGUL JUNGSEONG U","	= U"},
/* 116F */ { "HANGUL JUNGSEONG WEO","	= WEO"},
/* 1170 */ { "HANGUL JUNGSEONG WE","	= WE"},
/* 1171 */ { "HANGUL JUNGSEONG WI","	= WI"},
/* 1172 */ { "HANGUL JUNGSEONG YU","	= YU"},
/* 1173 */ { "HANGUL JUNGSEONG EU","	= EU"},
/* 1174 */ { "HANGUL JUNGSEONG YI","	= YI"},
/* 1175 */ { "HANGUL JUNGSEONG I","	= I"},
/* 1176 */ { "HANGUL JUNGSEONG A-O",NULL},
/* 1177 */ { "HANGUL JUNGSEONG A-U",NULL},
/* 1178 */ { "HANGUL JUNGSEONG YA-O",NULL},
/* 1179 */ { "HANGUL JUNGSEONG YA-YO",NULL},
/* 117A */ { "HANGUL JUNGSEONG EO-O",NULL},
/* 117B */ { "HANGUL JUNGSEONG EO-U",NULL},
/* 117C */ { "HANGUL JUNGSEONG EO-EU",NULL},
/* 117D */ { "HANGUL JUNGSEONG YEO-O",NULL},
/* 117E */ { "HANGUL JUNGSEONG YEO-U",NULL},
/* 117F */ { "HANGUL JUNGSEONG O-EO",NULL},
/* 1180 */ { "HANGUL JUNGSEONG O-E",NULL},
/* 1181 */ { "HANGUL JUNGSEONG O-YE",NULL},
/* 1182 */ { "HANGUL JUNGSEONG O-O",NULL},
/* 1183 */ { "HANGUL JUNGSEONG O-U",NULL},
/* 1184 */ { "HANGUL JUNGSEONG YO-YA",NULL},
/* 1185 */ { "HANGUL JUNGSEONG YO-YAE",NULL},
/* 1186 */ { "HANGUL JUNGSEONG YO-YEO",NULL},
/* 1187 */ { "HANGUL JUNGSEONG YO-O",NULL},
/* 1188 */ { "HANGUL JUNGSEONG YO-I",NULL},
/* 1189 */ { "HANGUL JUNGSEONG U-A",NULL},
/* 118A */ { "HANGUL JUNGSEONG U-AE",NULL},
/* 118B */ { "HANGUL JUNGSEONG U-EO-EU",NULL},
/* 118C */ { "HANGUL JUNGSEONG U-YE",NULL},
/* 118D */ { "HANGUL JUNGSEONG U-U",NULL},
/* 118E */ { "HANGUL JUNGSEONG YU-A",NULL},
/* 118F */ { "HANGUL JUNGSEONG YU-EO",NULL},
/* 1190 */ { "HANGUL JUNGSEONG YU-E",NULL},
/* 1191 */ { "HANGUL JUNGSEONG YU-YEO",NULL},
/* 1192 */ { "HANGUL JUNGSEONG YU-YE",NULL},
/* 1193 */ { "HANGUL JUNGSEONG YU-U",NULL},
/* 1194 */ { "HANGUL JUNGSEONG YU-I",NULL},
/* 1195 */ { "HANGUL JUNGSEONG EU-U",NULL},
/* 1196 */ { "HANGUL JUNGSEONG EU-EU",NULL},
/* 1197 */ { "HANGUL JUNGSEONG YI-U",NULL},
/* 1198 */ { "HANGUL JUNGSEONG I-A",NULL},
/* 1199 */ { "HANGUL JUNGSEONG I-YA",NULL},
/* 119A */ { "HANGUL JUNGSEONG I-O",NULL},
/* 119B */ { "HANGUL JUNGSEONG I-U",NULL},
/* 119C */ { "HANGUL JUNGSEONG I-EU",NULL},
/* 119D */ { "HANGUL JUNGSEONG I-ARAEA",NULL},
/* 119E */ { "HANGUL JUNGSEONG ARAEA","	* rounded open-mid back vowel"},
/* 119F */ { "HANGUL JUNGSEONG ARAEA-EO",NULL},
/* 11A0 */ { "HANGUL JUNGSEONG ARAEA-U",NULL},
/* 11A1 */ { "HANGUL JUNGSEONG ARAEA-I",NULL},
/* 11A2 */ { "HANGUL JUNGSEONG SSANGARAEA",NULL},
/* 11A3 */ { "HANGUL JUNGSEONG A-EU",NULL},
/* 11A4 */ { "HANGUL JUNGSEONG YA-U",NULL},
/* 11A5 */ { "HANGUL JUNGSEONG YEO-YA",NULL},
/* 11A6 */ { "HANGUL JUNGSEONG O-YA",NULL},
/* 11A7 */ { "HANGUL JUNGSEONG O-YAE",NULL},
/* 11A8 */ { "HANGUL JONGSEONG KIYEOK","	= G"},
/* 11A9 */ { "HANGUL JONGSEONG SSANGKIYEOK","	= GG"},
/* 11AA */ { "HANGUL JONGSEONG KIYEOK-SIOS","	= GS"},
/* 11AB */ { "HANGUL JONGSEONG NIEUN","	= N"},
/* 11AC */ { "HANGUL JONGSEONG NIEUN-CIEUC","	= NJ"},
/* 11AD */ { "HANGUL JONGSEONG NIEUN-HIEUH","	= NH"},
/* 11AE */ { "HANGUL JONGSEONG TIKEUT","	= D"},
/* 11AF */ { "HANGUL JONGSEONG RIEUL","	= L"},
/* 11B0 */ { "HANGUL JONGSEONG RIEUL-KIYEOK","	= LG"},
/* 11B1 */ { "HANGUL JONGSEONG RIEUL-MIEUM","	= LM"},
/* 11B2 */ { "HANGUL JONGSEONG RIEUL-PIEUP","	= LB"},
/* 11B3 */ { "HANGUL JONGSEONG RIEUL-SIOS","	= LS"},
/* 11B4 */ { "HANGUL JONGSEONG RIEUL-THIEUTH","	= LT"},
/* 11B5 */ { "HANGUL JONGSEONG RIEUL-PHIEUPH","	= LP"},
/* 11B6 */ { "HANGUL JONGSEONG RIEUL-HIEUH","	= LH"},
/* 11B7 */ { "HANGUL JONGSEONG MIEUM","	= M"},
/* 11B8 */ { "HANGUL JONGSEONG PIEUP","	= B"},
/* 11B9 */ { "HANGUL JONGSEONG PIEUP-SIOS","	= BS"},
/* 11BA */ { "HANGUL JONGSEONG SIOS","	= S"},
/* 11BB */ { "HANGUL JONGSEONG SSANGSIOS","	= SS"},
/* 11BC */ { "HANGUL JONGSEONG IEUNG","	= NG\n"
	"	* velar nasal consonant"},
/* 11BD */ { "HANGUL JONGSEONG CIEUC","	= J"},
/* 11BE */ { "HANGUL JONGSEONG CHIEUCH","	= C"},
/* 11BF */ { "HANGUL JONGSEONG KHIEUKH","	= K"},
/* 11C0 */ { "HANGUL JONGSEONG THIEUTH","	= T"},
/* 11C1 */ { "HANGUL JONGSEONG PHIEUPH","	= P"},
/* 11C2 */ { "HANGUL JONGSEONG HIEUH","	= H"},
/* 11C3 */ { "HANGUL JONGSEONG KIYEOK-RIEUL",NULL},
/* 11C4 */ { "HANGUL JONGSEONG KIYEOK-SIOS-KIYEOK",NULL},
/* 11C5 */ { "HANGUL JONGSEONG NIEUN-KIYEOK",NULL},
/* 11C6 */ { "HANGUL JONGSEONG NIEUN-TIKEUT",NULL},
/* 11C7 */ { "HANGUL JONGSEONG NIEUN-SIOS",NULL},
/* 11C8 */ { "HANGUL JONGSEONG NIEUN-PANSIOS",NULL},
/* 11C9 */ { "HANGUL JONGSEONG NIEUN-THIEUTH",NULL},
/* 11CA */ { "HANGUL JONGSEONG TIKEUT-KIYEOK",NULL},
/* 11CB */ { "HANGUL JONGSEONG TIKEUT-RIEUL",NULL},
/* 11CC */ { "HANGUL JONGSEONG RIEUL-KIYEOK-SIOS",NULL},
/* 11CD */ { "HANGUL JONGSEONG RIEUL-NIEUN",NULL},
/* 11CE */ { "HANGUL JONGSEONG RIEUL-TIKEUT",NULL},
/* 11CF */ { "HANGUL JONGSEONG RIEUL-TIKEUT-HIEUH",NULL},
/* 11D0 */ { "HANGUL JONGSEONG SSANGRIEUL",NULL},
/* 11D1 */ { "HANGUL JONGSEONG RIEUL-MIEUM-KIYEOK",NULL},
/* 11D2 */ { "HANGUL JONGSEONG RIEUL-MIEUM-SIOS",NULL},
/* 11D3 */ { "HANGUL JONGSEONG RIEUL-PIEUP-SIOS",NULL},
/* 11D4 */ { "HANGUL JONGSEONG RIEUL-PIEUP-HIEUH",NULL},
/* 11D5 */ { "HANGUL JONGSEONG RIEUL-KAPYEOUNPIEUP",NULL},
/* 11D6 */ { "HANGUL JONGSEONG RIEUL-SSANGSIOS",NULL},
/* 11D7 */ { "HANGUL JONGSEONG RIEUL-PANSIOS",NULL},
/* 11D8 */ { "HANGUL JONGSEONG RIEUL-KHIEUKH",NULL},
/* 11D9 */ { "HANGUL JONGSEONG RIEUL-YEORINHIEUH",NULL},
/* 11DA */ { "HANGUL JONGSEONG MIEUM-KIYEOK",NULL},
/* 11DB */ { "HANGUL JONGSEONG MIEUM-RIEUL",NULL},
/* 11DC */ { "HANGUL JONGSEONG MIEUM-PIEUP",NULL},
/* 11DD */ { "HANGUL JONGSEONG MIEUM-SIOS",NULL},
/* 11DE */ { "HANGUL JONGSEONG MIEUM-SSANGSIOS",NULL},
/* 11DF */ { "HANGUL JONGSEONG MIEUM-PANSIOS",NULL},
/* 11E0 */ { "HANGUL JONGSEONG MIEUM-CHIEUCH",NULL},
/* 11E1 */ { "HANGUL JONGSEONG MIEUM-HIEUH",NULL},
/* 11E2 */ { "HANGUL JONGSEONG KAPYEOUNMIEUM",NULL},
/* 11E3 */ { "HANGUL JONGSEONG PIEUP-RIEUL",NULL},
/* 11E4 */ { "HANGUL JONGSEONG PIEUP-PHIEUPH",NULL},
/* 11E5 */ { "HANGUL JONGSEONG PIEUP-HIEUH",NULL},
/* 11E6 */ { "HANGUL JONGSEONG KAPYEOUNPIEUP",NULL},
/* 11E7 */ { "HANGUL JONGSEONG SIOS-KIYEOK",NULL},
/* 11E8 */ { "HANGUL JONGSEONG SIOS-TIKEUT",NULL},
/* 11E9 */ { "HANGUL JONGSEONG SIOS-RIEUL",NULL},
/* 11EA */ { "HANGUL JONGSEONG SIOS-PIEUP",NULL},
/* 11EB */ { "HANGUL JONGSEONG PANSIOS",NULL},
/* 11EC */ { "HANGUL JONGSEONG IEUNG-KIYEOK","	% HANGUL JONGSEONG YESIEUNG-KIYEOK"},
/* 11ED */ { "HANGUL JONGSEONG IEUNG-SSANGKIYEOK","	% HANGUL JONGSEONG YESIEUNG-SSANGKIYEOK"},
/* 11EE */ { "HANGUL JONGSEONG SSANGIEUNG","	% HANGUL JONGSEONG SSANGYESIEUNG"},
/* 11EF */ { "HANGUL JONGSEONG IEUNG-KHIEUKH","	% HANGUL JONGSEONG YESIEUNG-KHIEUKH"},
/* 11F0 */ { "HANGUL JONGSEONG YESIEUNG",NULL},
/* 11F1 */ { "HANGUL JONGSEONG YESIEUNG-SIOS",NULL},
/* 11F2 */ { "HANGUL JONGSEONG YESIEUNG-PANSIOS",NULL},
/* 11F3 */ { "HANGUL JONGSEONG PHIEUPH-PIEUP",NULL},
/* 11F4 */ { "HANGUL JONGSEONG KAPYEOUNPHIEUPH",NULL},
/* 11F5 */ { "HANGUL JONGSEONG HIEUH-NIEUN",NULL},
/* 11F6 */ { "HANGUL JONGSEONG HIEUH-RIEUL",NULL},
/* 11F7 */ { "HANGUL JONGSEONG HIEUH-MIEUM",NULL},
/* 11F8 */ { "HANGUL JONGSEONG HIEUH-PIEUP",NULL},
/* 11F9 */ { "HANGUL JONGSEONG YEORINHIEUH",NULL},
/* 11FA */ { "HANGUL JONGSEONG KIYEOK-NIEUN",NULL},
/* 11FB */ { "HANGUL JONGSEONG KIYEOK-PIEUP",NULL},
/* 11FC */ { "HANGUL JONGSEONG KIYEOK-CHIEUCH",NULL},
/* 11FD */ { "HANGUL JONGSEONG KIYEOK-KHIEUKH",NULL},
/* 11FE */ { "HANGUL JONGSEONG KIYEOK-HIEUH",NULL},
/* 11FF */ { "HANGUL JONGSEONG SSANGNIEUN",NULL}
};

UN_DLL_LOCAL
static const struct unicode_nameannot una_00_12[] = {
/* 1200 */ { "ETHIOPIC SYLLABLE HA",NULL},
/* 1201 */ { "ETHIOPIC SYLLABLE HU",NULL},
/* 1202 */ { "ETHIOPIC SYLLABLE HI",NULL},
/* 1203 */ { "ETHIOPIC SYLLABLE HAA",NULL},
/* 1204 */ { "ETHIOPIC SYLLABLE HEE",NULL},
/* 1205 */ { "ETHIOPIC SYLLABLE HE",NULL},
/* 1206 */ { "ETHIOPIC SYLLABLE HO",NULL},
/* 1207 */ { "ETHIOPIC SYLLABLE HOA",NULL},
/* 1208 */ { "ETHIOPIC SYLLABLE LA",NULL},
/* 1209 */ { "ETHIOPIC SYLLABLE LU",NULL},
/* 120A */ { "ETHIOPIC SYLLABLE LI",NULL},
/* 120B */ { "ETHIOPIC SYLLABLE LAA",NULL},
/* 120C */ { "ETHIOPIC SYLLABLE LEE",NULL},
/* 120D */ { "ETHIOPIC SYLLABLE LE",NULL},
/* 120E */ { "ETHIOPIC SYLLABLE LO",NULL},
/* 120F */ { "ETHIOPIC SYLLABLE LWA",NULL},
/* 1210 */ { "ETHIOPIC SYLLABLE HHA",NULL},
/* 1211 */ { "ETHIOPIC SYLLABLE HHU",NULL},
/* 1212 */ { "ETHIOPIC SYLLABLE HHI",NULL},
/* 1213 */ { "ETHIOPIC SYLLABLE HHAA",NULL},
/* 1214 */ { "ETHIOPIC SYLLABLE HHEE",NULL},
/* 1215 */ { "ETHIOPIC SYLLABLE HHE",NULL},
/* 1216 */ { "ETHIOPIC SYLLABLE HHO",NULL},
/* 1217 */ { "ETHIOPIC SYLLABLE HHWA",NULL},
/* 1218 */ { "ETHIOPIC SYLLABLE MA",NULL},
/* 1219 */ { "ETHIOPIC SYLLABLE MU",NULL},
/* 121A */ { "ETHIOPIC SYLLABLE MI",NULL},
/* 121B */ { "ETHIOPIC SYLLABLE MAA",NULL},
/* 121C */ { "ETHIOPIC SYLLABLE MEE",NULL},
/* 121D */ { "ETHIOPIC SYLLABLE ME",NULL},
/* 121E */ { "ETHIOPIC SYLLABLE MO",NULL},
/* 121F */ { "ETHIOPIC SYLLABLE MWA",NULL},
/* 1220 */ { "ETHIOPIC SYLLABLE SZA",NULL},
/* 1221 */ { "ETHIOPIC SYLLABLE SZU",NULL},
/* 1222 */ { "ETHIOPIC SYLLABLE SZI",NULL},
/* 1223 */ { "ETHIOPIC SYLLABLE SZAA",NULL},
/* 1224 */ { "ETHIOPIC SYLLABLE SZEE",NULL},
/* 1225 */ { "ETHIOPIC SYLLABLE SZE",NULL},
/* 1226 */ { "ETHIOPIC SYLLABLE SZO",NULL},
/* 1227 */ { "ETHIOPIC SYLLABLE SZWA",NULL},
/* 1228 */ { "ETHIOPIC SYLLABLE RA",NULL},
/* 1229 */ { "ETHIOPIC SYLLABLE RU",NULL},
/* 122A */ { "ETHIOPIC SYLLABLE RI",NULL},
/* 122B */ { "ETHIOPIC SYLLABLE RAA",NULL},
/* 122C */ { "ETHIOPIC SYLLABLE REE",NULL},
/* 122D */ { "ETHIOPIC SYLLABLE RE",NULL},
/* 122E */ { "ETHIOPIC SYLLABLE RO",NULL},
/* 122F */ { "ETHIOPIC SYLLABLE RWA",NULL},
/* 1230 */ { "ETHIOPIC SYLLABLE SA",NULL},
/* 1231 */ { "ETHIOPIC SYLLABLE SU",NULL},
/* 1232 */ { "ETHIOPIC SYLLABLE SI",NULL},
/* 1233 */ { "ETHIOPIC SYLLABLE SAA",NULL},
/* 1234 */ { "ETHIOPIC SYLLABLE SEE",NULL},
/* 1235 */ { "ETHIOPIC SYLLABLE SE",NULL},
/* 1236 */ { "ETHIOPIC SYLLABLE SO",NULL},
/* 1237 */ { "ETHIOPIC SYLLABLE SWA",NULL},
/* 1238 */ { "ETHIOPIC SYLLABLE SHA",NULL},
/* 1239 */ { "ETHIOPIC SYLLABLE SHU",NULL},
/* 123A */ { "ETHIOPIC SYLLABLE SHI",NULL},
/* 123B */ { "ETHIOPIC SYLLABLE SHAA",NULL},
/* 123C */ { "ETHIOPIC SYLLABLE SHEE",NULL},
/* 123D */ { "ETHIOPIC SYLLABLE SHE",NULL},
/* 123E */ { "ETHIOPIC SYLLABLE SHO",NULL},
/* 123F */ { "ETHIOPIC SYLLABLE SHWA",NULL},
/* 1240 */ { "ETHIOPIC SYLLABLE QA",NULL},
/* 1241 */ { "ETHIOPIC SYLLABLE QU",NULL},
/* 1242 */ { "ETHIOPIC SYLLABLE QI",NULL},
/* 1243 */ { "ETHIOPIC SYLLABLE QAA",NULL},
/* 1244 */ { "ETHIOPIC SYLLABLE QEE",NULL},
/* 1245 */ { "ETHIOPIC SYLLABLE QE",NULL},
/* 1246 */ { "ETHIOPIC SYLLABLE QO",NULL},
/* 1247 */ { "ETHIOPIC SYLLABLE QOA",NULL},
/* 1248 */ { "ETHIOPIC SYLLABLE QWA",NULL},
/* 1249 */ { NULL,NULL},
/* 124A */ { "ETHIOPIC SYLLABLE QWI","	x (ethiopic syllable gurage qwi - 1E7F0)"},
/* 124B */ { "ETHIOPIC SYLLABLE QWAA",NULL},
/* 124C */ { "ETHIOPIC SYLLABLE QWEE","	x (ethiopic syllable gurage qwee - 1E7F1)"},
/* 124D */ { "ETHIOPIC SYLLABLE QWE","	x (ethiopic syllable gurage qwe - 1E7F2)"},
/* 124E */ { NULL,NULL},
/* 124F */ { NULL,NULL},
/* 1250 */ { "ETHIOPIC SYLLABLE QHA",NULL},
/* 1251 */ { "ETHIOPIC SYLLABLE QHU",NULL},
/* 1252 */ { "ETHIOPIC SYLLABLE QHI",NULL},
/* 1253 */ { "ETHIOPIC SYLLABLE QHAA",NULL},
/* 1254 */ { "ETHIOPIC SYLLABLE QHEE",NULL},
/* 1255 */ { "ETHIOPIC SYLLABLE QHE",NULL},
/* 1256 */ { "ETHIOPIC SYLLABLE QHO",NULL},
/* 1257 */ { NULL,NULL},
/* 1258 */ { "ETHIOPIC SYLLABLE QHWA",NULL},
/* 1259 */ { NULL,NULL},
/* 125A */ { "ETHIOPIC SYLLABLE QHWI",NULL},
/* 125B */ { "ETHIOPIC SYLLABLE QHWAA",NULL},
/* 125C */ { "ETHIOPIC SYLLABLE QHWEE",NULL},
/* 125D */ { "ETHIOPIC SYLLABLE QHWE",NULL},
/* 125E */ { NULL,NULL},
/* 125F */ { NULL,NULL},
/* 1260 */ { "ETHIOPIC SYLLABLE BA",NULL},
/* 1261 */ { "ETHIOPIC SYLLABLE BU",NULL},
/* 1262 */ { "ETHIOPIC SYLLABLE BI",NULL},
/* 1263 */ { "ETHIOPIC SYLLABLE BAA",NULL},
/* 1264 */ { "ETHIOPIC SYLLABLE BEE",NULL},
/* 1265 */ { "ETHIOPIC SYLLABLE BE",NULL},
/* 1266 */ { "ETHIOPIC SYLLABLE BO",NULL},
/* 1267 */ { "ETHIOPIC SYLLABLE BWA",NULL},
/* 1268 */ { "ETHIOPIC SYLLABLE VA",NULL},
/* 1269 */ { "ETHIOPIC SYLLABLE VU",NULL},
/* 126A */ { "ETHIOPIC SYLLABLE VI",NULL},
/* 126B */ { "ETHIOPIC SYLLABLE VAA",NULL},
/* 126C */ { "ETHIOPIC SYLLABLE VEE",NULL},
/* 126D */ { "ETHIOPIC SYLLABLE VE",NULL},
/* 126E */ { "ETHIOPIC SYLLABLE VO",NULL},
/* 126F */ { "ETHIOPIC SYLLABLE VWA",NULL},
/* 1270 */ { "ETHIOPIC SYLLABLE TA",NULL},
/* 1271 */ { "ETHIOPIC SYLLABLE TU",NULL},
/* 1272 */ { "ETHIOPIC SYLLABLE TI",NULL},
/* 1273 */ { "ETHIOPIC SYLLABLE TAA",NULL},
/* 1274 */ { "ETHIOPIC SYLLABLE TEE",NULL},
/* 1275 */ { "ETHIOPIC SYLLABLE TE",NULL},
/* 1276 */ { "ETHIOPIC SYLLABLE TO",NULL},
/* 1277 */ { "ETHIOPIC SYLLABLE TWA",NULL},
/* 1278 */ { "ETHIOPIC SYLLABLE CA",NULL},
/* 1279 */ { "ETHIOPIC SYLLABLE CU",NULL},
/* 127A */ { "ETHIOPIC SYLLABLE CI",NULL},
/* 127B */ { "ETHIOPIC SYLLABLE CAA",NULL},
/* 127C */ { "ETHIOPIC SYLLABLE CEE",NULL},
/* 127D */ { "ETHIOPIC SYLLABLE CE",NULL},
/* 127E */ { "ETHIOPIC SYLLABLE CO",NULL},
/* 127F */ { "ETHIOPIC SYLLABLE CWA",NULL},
/* 1280 */ { "ETHIOPIC SYLLABLE XA",NULL},
/* 1281 */ { "ETHIOPIC SYLLABLE XU",NULL},
/* 1282 */ { "ETHIOPIC SYLLABLE XI",NULL},
/* 1283 */ { "ETHIOPIC SYLLABLE XAA",NULL},
/* 1284 */ { "ETHIOPIC SYLLABLE XEE",NULL},
/* 1285 */ { "ETHIOPIC SYLLABLE XE",NULL},
/* 1286 */ { "ETHIOPIC SYLLABLE XO",NULL},
/* 1287 */ { "ETHIOPIC SYLLABLE XOA",NULL},
/* 1288 */ { "ETHIOPIC SYLLABLE XWA",NULL},
/* 1289 */ { NULL,NULL},
/* 128A */ { "ETHIOPIC SYLLABLE XWI",NULL},
/* 128B */ { "ETHIOPIC SYLLABLE XWAA",NULL},
/* 128C */ { "ETHIOPIC SYLLABLE XWEE",NULL},
/* 128D */ { "ETHIOPIC SYLLABLE XWE",NULL},
/* 128E */ { NULL,NULL},
/* 128F */ { NULL,NULL},
/* 1290 */ { "ETHIOPIC SYLLABLE NA",NULL},
/* 1291 */ { "ETHIOPIC SYLLABLE NU",NULL},
/* 1292 */ { "ETHIOPIC SYLLABLE NI",NULL},
/* 1293 */ { "ETHIOPIC SYLLABLE NAA",NULL},
/* 1294 */ { "ETHIOPIC SYLLABLE NEE",NULL},
/* 1295 */ { "ETHIOPIC SYLLABLE NE",NULL},
/* 1296 */ { "ETHIOPIC SYLLABLE NO",NULL},
/* 1297 */ { "ETHIOPIC SYLLABLE NWA",NULL},
/* 1298 */ { "ETHIOPIC SYLLABLE NYA",NULL},
/* 1299 */ { "ETHIOPIC SYLLABLE NYU",NULL},
/* 129A */ { "ETHIOPIC SYLLABLE NYI",NULL},
/* 129B */ { "ETHIOPIC SYLLABLE NYAA",NULL},
/* 129C */ { "ETHIOPIC SYLLABLE NYEE",NULL},
/* 129D */ { "ETHIOPIC SYLLABLE NYE",NULL},
/* 129E */ { "ETHIOPIC SYLLABLE NYO",NULL},
/* 129F */ { "ETHIOPIC SYLLABLE NYWA",NULL},
/* 12A0 */ { "ETHIOPIC SYLLABLE GLOTTAL A",NULL},
/* 12A1 */ { "ETHIOPIC SYLLABLE GLOTTAL U",NULL},
/* 12A2 */ { "ETHIOPIC SYLLABLE GLOTTAL I",NULL},
/* 12A3 */ { "ETHIOPIC SYLLABLE GLOTTAL AA",NULL},
/* 12A4 */ { "ETHIOPIC SYLLABLE GLOTTAL EE",NULL},
/* 12A5 */ { "ETHIOPIC SYLLABLE GLOTTAL E",NULL},
/* 12A6 */ { "ETHIOPIC SYLLABLE GLOTTAL O",NULL},
/* 12A7 */ { "ETHIOPIC SYLLABLE GLOTTAL WA",NULL},
/* 12A8 */ { "ETHIOPIC SYLLABLE KA",NULL},
/* 12A9 */ { "ETHIOPIC SYLLABLE KU",NULL},
/* 12AA */ { "ETHIOPIC SYLLABLE KI",NULL},
/* 12AB */ { "ETHIOPIC SYLLABLE KAA",NULL},
/* 12AC */ { "ETHIOPIC SYLLABLE KEE",NULL},
/* 12AD */ { "ETHIOPIC SYLLABLE KE",NULL},
/* 12AE */ { "ETHIOPIC SYLLABLE KO",NULL},
/* 12AF */ { "ETHIOPIC SYLLABLE KOA",NULL},
/* 12B0 */ { "ETHIOPIC SYLLABLE KWA",NULL},
/* 12B1 */ { NULL,NULL},
/* 12B2 */ { "ETHIOPIC SYLLABLE KWI","	x (ethiopic syllable gurage kwi - 1E7F5)"},
/* 12B3 */ { "ETHIOPIC SYLLABLE KWAA",NULL},
/* 12B4 */ { "ETHIOPIC SYLLABLE KWEE","	x (ethiopic syllable gurage kwee - 1E7F6)"},
/* 12B5 */ { "ETHIOPIC SYLLABLE KWE","	x (ethiopic syllable gurage kwe - 1E7F7)"},
/* 12B6 */ { NULL,NULL},
/* 12B7 */ { NULL,NULL},
/* 12B8 */ { "ETHIOPIC SYLLABLE KXA",NULL},
/* 12B9 */ { "ETHIOPIC SYLLABLE KXU",NULL},
/* 12BA */ { "ETHIOPIC SYLLABLE KXI",NULL},
/* 12BB */ { "ETHIOPIC SYLLABLE KXAA",NULL},
/* 12BC */ { "ETHIOPIC SYLLABLE KXEE",NULL},
/* 12BD */ { "ETHIOPIC SYLLABLE KXE",NULL},
/* 12BE */ { "ETHIOPIC SYLLABLE KXO",NULL},
/* 12BF */ { NULL,NULL},
/* 12C0 */ { "ETHIOPIC SYLLABLE KXWA","	x (ethiopic syllable gurage hhwa - 1E7E8)"},
/* 12C1 */ { NULL,NULL},
/* 12C2 */ { "ETHIOPIC SYLLABLE KXWI","	x (ethiopic syllable hhwi - 1E7E9)"},
/* 12C3 */ { "ETHIOPIC SYLLABLE KXWAA",NULL},
/* 12C4 */ { "ETHIOPIC SYLLABLE KXWEE","	x (ethiopic syllable hhwee - 1E7EA)"},
/* 12C5 */ { "ETHIOPIC SYLLABLE KXWE","	x (ethiopic syllable hhwe - 1E7EB)"},
/* 12C6 */ { NULL,NULL},
/* 12C7 */ { NULL,NULL},
/* 12C8 */ { "ETHIOPIC SYLLABLE WA",NULL},
/* 12C9 */ { "ETHIOPIC SYLLABLE WU",NULL},
/* 12CA */ { "ETHIOPIC SYLLABLE WI",NULL},
/* 12CB */ { "ETHIOPIC SYLLABLE WAA",NULL},
/* 12CC */ { "ETHIOPIC SYLLABLE WEE",NULL},
/* 12CD */ { "ETHIOPIC SYLLABLE WE",NULL},
/* 12CE */ { "ETHIOPIC SYLLABLE WO",NULL},
/* 12CF */ { "ETHIOPIC SYLLABLE WOA",NULL},
/* 12D0 */ { "ETHIOPIC SYLLABLE PHARYNGEAL A",NULL},
/* 12D1 */ { "ETHIOPIC SYLLABLE PHARYNGEAL U",NULL},
/* 12D2 */ { "ETHIOPIC SYLLABLE PHARYNGEAL I",NULL},
/* 12D3 */ { "ETHIOPIC SYLLABLE PHARYNGEAL AA",NULL},
/* 12D4 */ { "ETHIOPIC SYLLABLE PHARYNGEAL EE",NULL},
/* 12D5 */ { "ETHIOPIC SYLLABLE PHARYNGEAL E",NULL},
/* 12D6 */ { "ETHIOPIC SYLLABLE PHARYNGEAL O",NULL},
/* 12D7 */ { NULL,NULL},
/* 12D8 */ { "ETHIOPIC SYLLABLE ZA",NULL},
/* 12D9 */ { "ETHIOPIC SYLLABLE ZU",NULL},
/* 12DA */ { "ETHIOPIC SYLLABLE ZI",NULL},
/* 12DB */ { "ETHIOPIC SYLLABLE ZAA",NULL},
/* 12DC */ { "ETHIOPIC SYLLABLE ZEE",NULL},
/* 12DD */ { "ETHIOPIC SYLLABLE ZE",NULL},
/* 12DE */ { "ETHIOPIC SYLLABLE ZO",NULL},
/* 12DF */ { "ETHIOPIC SYLLABLE ZWA",NULL},
/* 12E0 */ { "ETHIOPIC SYLLABLE ZHA",NULL},
/* 12E1 */ { "ETHIOPIC SYLLABLE ZHU",NULL},
/* 12E2 */ { "ETHIOPIC SYLLABLE ZHI",NULL},
/* 12E3 */ { "ETHIOPIC SYLLABLE ZHAA",NULL},
/* 12E4 */ { "ETHIOPIC SYLLABLE ZHEE",NULL},
/* 12E5 */ { "ETHIOPIC SYLLABLE ZHE",NULL},
/* 12E6 */ { "ETHIOPIC SYLLABLE ZHO",NULL},
/* 12E7 */ { "ETHIOPIC SYLLABLE ZHWA",NULL},
/* 12E8 */ { "ETHIOPIC SYLLABLE YA",NULL},
/* 12E9 */ { "ETHIOPIC SYLLABLE YU",NULL},
/* 12EA */ { "ETHIOPIC SYLLABLE YI",NULL},
/* 12EB */ { "ETHIOPIC SYLLABLE YAA",NULL},
/* 12EC */ { "ETHIOPIC SYLLABLE YEE",NULL},
/* 12ED */ { "ETHIOPIC SYLLABLE YE",NULL},
/* 12EE */ { "ETHIOPIC SYLLABLE YO",NULL},
/* 12EF */ { "ETHIOPIC SYLLABLE YOA",NULL},
/* 12F0 */ { "ETHIOPIC SYLLABLE DA",NULL},
/* 12F1 */ { "ETHIOPIC SYLLABLE DU",NULL},
/* 12F2 */ { "ETHIOPIC SYLLABLE DI",NULL},
/* 12F3 */ { "ETHIOPIC SYLLABLE DAA",NULL},
/* 12F4 */ { "ETHIOPIC SYLLABLE DEE",NULL},
/* 12F5 */ { "ETHIOPIC SYLLABLE DE",NULL},
/* 12F6 */ { "ETHIOPIC SYLLABLE DO",NULL},
/* 12F7 */ { "ETHIOPIC SYLLABLE DWA",NULL},
/* 12F8 */ { "ETHIOPIC SYLLABLE DDA",NULL},
/* 12F9 */ { "ETHIOPIC SYLLABLE DDU",NULL},
/* 12FA */ { "ETHIOPIC SYLLABLE DDI",NULL},
/* 12FB */ { "ETHIOPIC SYLLABLE DDAA",NULL},
/* 12FC */ { "ETHIOPIC SYLLABLE DDEE",NULL},
/* 12FD */ { "ETHIOPIC SYLLABLE DDE",NULL},
/* 12FE */ { "ETHIOPIC SYLLABLE DDO",NULL},
/* 12FF */ { "ETHIOPIC SYLLABLE DDWA",NULL}
};

UN_DLL_LOCAL
static const struct unicode_nameannot una_00_13[] = {
/* 1300 */ { "ETHIOPIC SYLLABLE JA",NULL},
/* 1301 */ { "ETHIOPIC SYLLABLE JU",NULL},
/* 1302 */ { "ETHIOPIC SYLLABLE JI",NULL},
/* 1303 */ { "ETHIOPIC SYLLABLE JAA",NULL},
/* 1304 */ { "ETHIOPIC SYLLABLE JEE",NULL},
/* 1305 */ { "ETHIOPIC SYLLABLE JE",NULL},
/* 1306 */ { "ETHIOPIC SYLLABLE JO",NULL},
/* 1307 */ { "ETHIOPIC SYLLABLE JWA",NULL},
/* 1308 */ { "ETHIOPIC SYLLABLE GA",NULL},
/* 1309 */ { "ETHIOPIC SYLLABLE GU",NULL},
/* 130A */ { "ETHIOPIC SYLLABLE GI",NULL},
/* 130B */ { "ETHIOPIC SYLLABLE GAA",NULL},
/* 130C */ { "ETHIOPIC SYLLABLE GEE",NULL},
/* 130D */ { "ETHIOPIC SYLLABLE GE",NULL},
/* 130E */ { "ETHIOPIC SYLLABLE GO",NULL},
/* 130F */ { "ETHIOPIC SYLLABLE GOA",NULL},
/* 1310 */ { "ETHIOPIC SYLLABLE GWA",NULL},
/* 1311 */ { NULL,NULL},
/* 1312 */ { "ETHIOPIC SYLLABLE GWI","	x (ethiopic syllable gurage gwi - 1E7F8)"},
/* 1313 */ { "ETHIOPIC SYLLABLE GWAA",NULL},
/* 1314 */ { "ETHIOPIC SYLLABLE GWEE","	x (ethiopic syllable gurage gwee - 1E7F9)"},
/* 1315 */ { "ETHIOPIC SYLLABLE GWE","	x (ethiopic syllable gurage gwe - 1E7FA)"},
/* 1316 */ { NULL,NULL},
/* 1317 */ { NULL,NULL},
/* 1318 */ { "ETHIOPIC SYLLABLE GGA",NULL},
/* 1319 */ { "ETHIOPIC SYLLABLE GGU",NULL},
/* 131A */ { "ETHIOPIC SYLLABLE GGI",NULL},
/* 131B */ { "ETHIOPIC SYLLABLE GGAA",NULL},
/* 131C */ { "ETHIOPIC SYLLABLE GGEE",NULL},
/* 131D */ { "ETHIOPIC SYLLABLE GGE",NULL},
/* 131E */ { "ETHIOPIC SYLLABLE GGO",NULL},
/* 131F */ { "ETHIOPIC SYLLABLE GGWAA",NULL},
/* 1320 */ { "ETHIOPIC SYLLABLE THA",NULL},
/* 1321 */ { "ETHIOPIC SYLLABLE THU",NULL},
/* 1322 */ { "ETHIOPIC SYLLABLE THI",NULL},
/* 1323 */ { "ETHIOPIC SYLLABLE THAA",NULL},
/* 1324 */ { "ETHIOPIC SYLLABLE THEE",NULL},
/* 1325 */ { "ETHIOPIC SYLLABLE THE",NULL},
/* 1326 */ { "ETHIOPIC SYLLABLE THO",NULL},
/* 1327 */ { "ETHIOPIC SYLLABLE THWA",NULL},
/* 1328 */ { "ETHIOPIC SYLLABLE CHA",NULL},
/* 1329 */ { "ETHIOPIC SYLLABLE CHU",NULL},
/* 132A */ { "ETHIOPIC SYLLABLE CHI",NULL},
/* 132B */ { "ETHIOPIC SYLLABLE CHAA",NULL},
/* 132C */ { "ETHIOPIC SYLLABLE CHEE",NULL},
/* 132D */ { "ETHIOPIC SYLLABLE CHE",NULL},
/* 132E */ { "ETHIOPIC SYLLABLE CHO",NULL},
/* 132F */ { "ETHIOPIC SYLLABLE CHWA",NULL},
/* 1330 */ { "ETHIOPIC SYLLABLE PHA",NULL},
/* 1331 */ { "ETHIOPIC SYLLABLE PHU",NULL},
/* 1332 */ { "ETHIOPIC SYLLABLE PHI",NULL},
/* 1333 */ { "ETHIOPIC SYLLABLE PHAA",NULL},
/* 1334 */ { "ETHIOPIC SYLLABLE PHEE",NULL},
/* 1335 */ { "ETHIOPIC SYLLABLE PHE",NULL},
/* 1336 */ { "ETHIOPIC SYLLABLE PHO",NULL},
/* 1337 */ { "ETHIOPIC SYLLABLE PHWA",NULL},
/* 1338 */ { "ETHIOPIC SYLLABLE TSA",NULL},
/* 1339 */ { "ETHIOPIC SYLLABLE TSU",NULL},
/* 133A */ { "ETHIOPIC SYLLABLE TSI",NULL},
/* 133B */ { "ETHIOPIC SYLLABLE TSAA",NULL},
/* 133C */ { "ETHIOPIC SYLLABLE TSEE",NULL},
/* 133D */ { "ETHIOPIC SYLLABLE TSE",NULL},
/* 133E */ { "ETHIOPIC SYLLABLE TSO",NULL},
/* 133F */ { "ETHIOPIC SYLLABLE TSWA",NULL},
/* 1340 */ { "ETHIOPIC SYLLABLE TZA",NULL},
/* 1341 */ { "ETHIOPIC SYLLABLE TZU",NULL},
/* 1342 */ { "ETHIOPIC SYLLABLE TZI",NULL},
/* 1343 */ { "ETHIOPIC SYLLABLE TZAA",NULL},
/* 1344 */ { "ETHIOPIC SYLLABLE TZEE",NULL},
/* 1345 */ { "ETHIOPIC SYLLABLE TZE",NULL},
/* 1346 */ { "ETHIOPIC SYLLABLE TZO",NULL},
/* 1347 */ { "ETHIOPIC SYLLABLE TZOA",NULL},
/* 1348 */ { "ETHIOPIC SYLLABLE FA",NULL},
/* 1349 */ { "ETHIOPIC SYLLABLE FU",NULL},
/* 134A */ { "ETHIOPIC SYLLABLE FI",NULL},
/* 134B */ { "ETHIOPIC SYLLABLE FAA",NULL},
/* 134C */ { "ETHIOPIC SYLLABLE FEE",NULL},
/* 134D */ { "ETHIOPIC SYLLABLE FE",NULL},
/* 134E */ { "ETHIOPIC SYLLABLE FO",NULL},
/* 134F */ { "ETHIOPIC SYLLABLE FWA",NULL},
/* 1350 */ { "ETHIOPIC SYLLABLE PA",NULL},
/* 1351 */ { "ETHIOPIC SYLLABLE PU",NULL},
/* 1352 */ { "ETHIOPIC SYLLABLE PI",NULL},
/* 1353 */ { "ETHIOPIC SYLLABLE PAA",NULL},
/* 1354 */ { "ETHIOPIC SYLLABLE PEE",NULL},
/* 1355 */ { "ETHIOPIC SYLLABLE PE",NULL},
/* 1356 */ { "ETHIOPIC SYLLABLE PO",NULL},
/* 1357 */ { "ETHIOPIC SYLLABLE PWA",NULL},
/* 1358 */ { "ETHIOPIC SYLLABLE RYA",NULL},
/* 1359 */ { "ETHIOPIC SYLLABLE MYA",NULL},
/* 135A */ { "ETHIOPIC SYLLABLE FYA",NULL},
/* 135B */ { NULL,NULL},
/* 135C */ { NULL,NULL},
/* 135D */ { "ETHIOPIC COMBINING GEMINATION AND VOWEL LENGTH MARK","	* Basketo"},
/* 135E */ { "ETHIOPIC COMBINING VOWEL LENGTH MARK","	* Basketo"},
/* 135F */ { "ETHIOPIC COMBINING GEMINATION MARK",NULL},
/* 1360 */ { "ETHIOPIC SECTION MARK",NULL},
/* 1361 */ { "ETHIOPIC WORDSPACE",NULL},
/* 1362 */ { "ETHIOPIC FULL STOP",NULL},
/* 1363 */ { "ETHIOPIC COMMA",NULL},
/* 1364 */ { "ETHIOPIC SEMICOLON",NULL},
/* 1365 */ { "ETHIOPIC COLON",NULL},
/* 1366 */ { "ETHIOPIC PREFACE COLON",NULL},
/* 1367 */ { "ETHIOPIC QUESTION MARK",NULL},
/* 1368 */ { "ETHIOPIC PARAGRAPH SEPARATOR",NULL},
/* 1369 */ { "ETHIOPIC DIGIT ONE",NULL},
/* 136A */ { "ETHIOPIC DIGIT TWO",NULL},
/* 136B */ { "ETHIOPIC DIGIT THREE",NULL},
/* 136C */ { "ETHIOPIC DIGIT FOUR",NULL},
/* 136D */ { "ETHIOPIC DIGIT FIVE",NULL},
/* 136E */ { "ETHIOPIC DIGIT SIX",NULL},
/* 136F */ { "ETHIOPIC DIGIT SEVEN",NULL},
/* 1370 */ { "ETHIOPIC DIGIT EIGHT",NULL},
/* 1371 */ { "ETHIOPIC DIGIT NINE",NULL},
/* 1372 */ { "ETHIOPIC NUMBER TEN",NULL},
/* 1373 */ { "ETHIOPIC NUMBER TWENTY",NULL},
/* 1374 */ { "ETHIOPIC NUMBER THIRTY",NULL},
/* 1375 */ { "ETHIOPIC NUMBER FORTY",NULL},
/* 1376 */ { "ETHIOPIC NUMBER FIFTY",NULL},
/* 1377 */ { "ETHIOPIC NUMBER SIXTY",NULL},
/* 1378 */ { "ETHIOPIC NUMBER SEVENTY",NULL},
/* 1379 */ { "ETHIOPIC NUMBER EIGHTY",NULL},
/* 137A */ { "ETHIOPIC NUMBER NINETY",NULL},
/* 137B */ { "ETHIOPIC NUMBER HUNDRED",NULL},
/* 137C */ { "ETHIOPIC NUMBER TEN THOUSAND",NULL},
/* 137D */ { NULL,NULL},
/* 137E */ { NULL,NULL},
/* 137F */ { NULL,NULL},
/* 1380 */ { "ETHIOPIC SYLLABLE SEBATBEIT MWA",NULL},
/* 1381 */ { "ETHIOPIC SYLLABLE MWI","	x (ethiopic syllable gurage mwi - 1E7ED)"},
/* 1382 */ { "ETHIOPIC SYLLABLE MWEE","	x (ethiopic syllable gurage mwee - 1E7EE)"},
/* 1383 */ { "ETHIOPIC SYLLABLE MWE",NULL},
/* 1384 */ { "ETHIOPIC SYLLABLE SEBATBEIT BWA",NULL},
/* 1385 */ { "ETHIOPIC SYLLABLE BWI","	x (ethiopic syllable gurage bwi - 1E7F3)"},
/* 1386 */ { "ETHIOPIC SYLLABLE BWEE","	x (ethiopic syllable gurage bwee - 1E7F4)"},
/* 1387 */ { "ETHIOPIC SYLLABLE BWE",NULL},
/* 1388 */ { "ETHIOPIC SYLLABLE SEBATBEIT FWA",NULL},
/* 1389 */ { "ETHIOPIC SYLLABLE FWI","	x (ethiopic syllable gurage fwi - 1E7FB)"},
/* 138A */ { "ETHIOPIC SYLLABLE FWEE","	x (ethiopic syllable gurage fwee - 1E7FC)"},
/* 138B */ { "ETHIOPIC SYLLABLE FWE",NULL},
/* 138C */ { "ETHIOPIC SYLLABLE SEBATBEIT PWA",NULL},
/* 138D */ { "ETHIOPIC SYLLABLE PWI","	x (ethiopic syllable gurage pwi - 1E7FD)"},
/* 138E */ { "ETHIOPIC SYLLABLE PWEE","	x (ethiopic syllable gurage pwee - 1E7FE)"},
/* 138F */ { "ETHIOPIC SYLLABLE PWE",NULL},
/* 1390 */ { "ETHIOPIC TONAL MARK YIZET",NULL},
/* 1391 */ { "ETHIOPIC TONAL MARK DERET",NULL},
/* 1392 */ { "ETHIOPIC TONAL MARK RIKRIK",NULL},
/* 1393 */ { "ETHIOPIC TONAL MARK SHORT RIKRIK",NULL},
/* 1394 */ { "ETHIOPIC TONAL MARK DIFAT",NULL},
/* 1395 */ { "ETHIOPIC TONAL MARK KENAT",NULL},
/* 1396 */ { "ETHIOPIC TONAL MARK CHIRET",NULL},
/* 1397 */ { "ETHIOPIC TONAL MARK HIDET",NULL},
/* 1398 */ { "ETHIOPIC TONAL MARK DERET-HIDET",NULL},
/* 1399 */ { "ETHIOPIC TONAL MARK KURT",NULL},
/* 139A */ { NULL,NULL},
/* 139B */ { NULL,NULL},
/* 139C */ { NULL,NULL},
/* 139D */ { NULL,NULL},
/* 139E */ { NULL,NULL},
/* 139F */ { NULL,NULL},
/* 13A0 */ { "CHEROKEE LETTER A",NULL},
/* 13A1 */ { "CHEROKEE LETTER E",NULL},
/* 13A2 */ { "CHEROKEE LETTER I",NULL},
/* 13A3 */ { "CHEROKEE LETTER O",NULL},
/* 13A4 */ { "CHEROKEE LETTER U",NULL},
/* 13A5 */ { "CHEROKEE LETTER V",NULL},
/* 13A6 */ { "CHEROKEE LETTER GA",NULL},
/* 13A7 */ { "CHEROKEE LETTER KA",NULL},
/* 13A8 */ { "CHEROKEE LETTER GE",NULL},
/* 13A9 */ { "CHEROKEE LETTER GI",NULL},
/* 13AA */ { "CHEROKEE LETTER GO",NULL},
/* 13AB */ { "CHEROKEE LETTER GU",NULL},
/* 13AC */ { "CHEROKEE LETTER GV",NULL},
/* 13AD */ { "CHEROKEE LETTER HA",NULL},
/* 13AE */ { "CHEROKEE LETTER HE",NULL},
/* 13AF */ { "CHEROKEE LETTER HI",NULL},
/* 13B0 */ { "CHEROKEE LETTER HO",NULL},
/* 13B1 */ { "CHEROKEE LETTER HU",NULL},
/* 13B2 */ { "CHEROKEE LETTER HV",NULL},
/* 13B3 */ { "CHEROKEE LETTER LA",NULL},
/* 13B4 */ { "CHEROKEE LETTER LE",NULL},
/* 13B5 */ { "CHEROKEE LETTER LI",NULL},
/* 13B6 */ { "CHEROKEE LETTER LO",NULL},
/* 13B7 */ { "CHEROKEE LETTER LU",NULL},
/* 13B8 */ { "CHEROKEE LETTER LV",NULL},
/* 13B9 */ { "CHEROKEE LETTER MA",NULL},
/* 13BA */ { "CHEROKEE LETTER ME",NULL},
/* 13BB */ { "CHEROKEE LETTER MI",NULL},
/* 13BC */ { "CHEROKEE LETTER MO",NULL},
/* 13BD */ { "CHEROKEE LETTER MU",NULL},
/* 13BE */ { "CHEROKEE LETTER NA",NULL},
/* 13BF */ { "CHEROKEE LETTER HNA",NULL},
/* 13C0 */ { "CHEROKEE LETTER NAH",NULL},
/* 13C1 */ { "CHEROKEE LETTER NE",NULL},
/* 13C2 */ { "CHEROKEE LETTER NI",NULL},
/* 13C3 */ { "CHEROKEE LETTER NO",NULL},
/* 13C4 */ { "CHEROKEE LETTER NU",NULL},
/* 13C5 */ { "CHEROKEE LETTER NV",NULL},
/* 13C6 */ { "CHEROKEE LETTER QUA",NULL},
/* 13C7 */ { "CHEROKEE LETTER QUE",NULL},
/* 13C8 */ { "CHEROKEE LETTER QUI",NULL},
/* 13C9 */ { "CHEROKEE LETTER QUO",NULL},
/* 13CA */ { "CHEROKEE LETTER QUU",NULL},
/* 13CB */ { "CHEROKEE LETTER QUV",NULL},
/* 13CC */ { "CHEROKEE LETTER SA",NULL},
/* 13CD */ { "CHEROKEE LETTER S",NULL},
/* 13CE */ { "CHEROKEE LETTER SE",NULL},
/* 13CF */ { "CHEROKEE LETTER SI",NULL},
/* 13D0 */ { "CHEROKEE LETTER SO",NULL},
/* 13D1 */ { "CHEROKEE LETTER SU",NULL},
/* 13D2 */ { "CHEROKEE LETTER SV",NULL},
/* 13D3 */ { "CHEROKEE LETTER DA",NULL},
/* 13D4 */ { "CHEROKEE LETTER TA",NULL},
/* 13D5 */ { "CHEROKEE LETTER DE",NULL},
/* 13D6 */ { "CHEROKEE LETTER TE",NULL},
/* 13D7 */ { "CHEROKEE LETTER DI",NULL},
/* 13D8 */ { "CHEROKEE LETTER TI",NULL},
/* 13D9 */ { "CHEROKEE LETTER DO",NULL},
/* 13DA */ { "CHEROKEE LETTER DU",NULL},
/* 13DB */ { "CHEROKEE LETTER DV",NULL},
/* 13DC */ { "CHEROKEE LETTER DLA",NULL},
/* 13DD */ { "CHEROKEE LETTER TLA",NULL},
/* 13DE */ { "CHEROKEE LETTER TLE",NULL},
/* 13DF */ { "CHEROKEE LETTER TLI",NULL},
/* 13E0 */ { "CHEROKEE LETTER TLO",NULL},
/* 13E1 */ { "CHEROKEE LETTER TLU",NULL},
/* 13E2 */ { "CHEROKEE LETTER TLV",NULL},
/* 13E3 */ { "CHEROKEE LETTER TSA",NULL},
/* 13E4 */ { "CHEROKEE LETTER TSE",NULL},
/* 13E5 */ { "CHEROKEE LETTER TSI",NULL},
/* 13E6 */ { "CHEROKEE LETTER TSO",NULL},
/* 13E7 */ { "CHEROKEE LETTER TSU",NULL},
/* 13E8 */ { "CHEROKEE LETTER TSV",NULL},
/* 13E9 */ { "CHEROKEE LETTER WA",NULL},
/* 13EA */ { "CHEROKEE LETTER WE",NULL},
/* 13EB */ { "CHEROKEE LETTER WI",NULL},
/* 13EC */ { "CHEROKEE LETTER WO",NULL},
/* 13ED */ { "CHEROKEE LETTER WU",NULL},
/* 13EE */ { "CHEROKEE LETTER WV",NULL},
/* 13EF */ { "CHEROKEE LETTER YA",NULL},
/* 13F0 */ { "CHEROKEE LETTER YE",NULL},
/* 13F1 */ { "CHEROKEE LETTER YI",NULL},
/* 13F2 */ { "CHEROKEE LETTER YO",NULL},
/* 13F3 */ { "CHEROKEE LETTER YU",NULL},
/* 13F4 */ { "CHEROKEE LETTER YV",NULL},
/* 13F5 */ { "CHEROKEE LETTER MV",NULL},
/* 13F6 */ { NULL,NULL},
/* 13F7 */ { NULL,NULL},
/* 13F8 */ { "CHEROKEE SMALL LETTER YE",NULL},
/* 13F9 */ { "CHEROKEE SMALL LETTER YI",NULL},
/* 13FA */ { "CHEROKEE SMALL LETTER YO",NULL},
/* 13FB */ { "CHEROKEE SMALL LETTER YU",NULL},
/* 13FC */ { "CHEROKEE SMALL LETTER YV",NULL},
/* 13FD */ { "CHEROKEE SMALL LETTER MV",NULL},
/* 13FE */ { NULL,NULL},
/* 13FF */ { NULL,NULL}
};

UN_DLL_LOCAL
static const struct unicode_nameannot una_00_14[] = {
/* 1400 */ { "CANADIAN SYLLABICS HYPHEN",NULL},
/* 1401 */ { "CANADIAN SYLLABICS E","	* Inuktitut (AI), Carrier (U)"},
/* 1402 */ { "CANADIAN SYLLABICS AAI","	* Inuktitut"},
/* 1403 */ { "CANADIAN SYLLABICS I","	* Carrier (O)"},
/* 1404 */ { "CANADIAN SYLLABICS II",NULL},
/* 1405 */ { "CANADIAN SYLLABICS O","	* Inuktitut (U), Carrier (E)"},
/* 1406 */ { "CANADIAN SYLLABICS OO","	* Inuktitut (UU)"},
/* 1407 */ { "CANADIAN SYLLABICS Y-CREE OO",NULL},
/* 1408 */ { "CANADIAN SYLLABICS CARRIER EE",NULL},
/* 1409 */ { "CANADIAN SYLLABICS CARRIER I",NULL},
/* 140A */ { "CANADIAN SYLLABICS A",NULL},
/* 140B */ { "CANADIAN SYLLABICS AA",NULL},
/* 140C */ { "CANADIAN SYLLABICS WE",NULL},
/* 140D */ { "CANADIAN SYLLABICS WEST-CREE WE",NULL},
/* 140E */ { "CANADIAN SYLLABICS WI",NULL},
/* 140F */ { "CANADIAN SYLLABICS WEST-CREE WI",NULL},
/* 1410 */ { "CANADIAN SYLLABICS WII",NULL},
/* 1411 */ { "CANADIAN SYLLABICS WEST-CREE WII",NULL},
/* 1412 */ { "CANADIAN SYLLABICS WO",NULL},
/* 1413 */ { "CANADIAN SYLLABICS WEST-CREE WO",NULL},
/* 1414 */ { "CANADIAN SYLLABICS WOO",NULL},
/* 1415 */ { "CANADIAN SYLLABICS WEST-CREE WOO",NULL},
/* 1416 */ { "CANADIAN SYLLABICS NASKAPI WOO",NULL},
/* 1417 */ { "CANADIAN SYLLABICS WA",NULL},
/* 1418 */ { "CANADIAN SYLLABICS WEST-CREE WA",NULL},
/* 1419 */ { "CANADIAN SYLLABICS WAA",NULL},
/* 141A */ { "CANADIAN SYLLABICS WEST-CREE WAA",NULL},
/* 141B */ { "CANADIAN SYLLABICS NASKAPI WAA",NULL},
/* 141C */ { "CANADIAN SYLLABICS AI","	* East Cree"},
/* 141D */ { "CANADIAN SYLLABICS Y-CREE W",NULL},
/* 141E */ { "CANADIAN SYLLABICS GLOTTAL STOP","	* Moose Cree (Y), Algonquian (GLOTTAL STOP)"},
/* 141F */ { "CANADIAN SYLLABICS FINAL ACUTE","	* West Cree (T), East Cree (Y), Inuktitut (GLOTTAL STOP)\n"
	"	* Athapascan (B/P), Sayisi (I), Carrier (G)"},
/* 1420 */ { "CANADIAN SYLLABICS FINAL GRAVE","	* West Cree (K), Athapascan (K), Carrier (KH)"},
/* 1421 */ { "CANADIAN SYLLABICS FINAL BOTTOM HALF RING","	* N Cree (SH), Sayisi (R), Carrier (NG)"},
/* 1422 */ { "CANADIAN SYLLABICS FINAL TOP HALF RING","	* Algonquian (S), Chipewyan (R), Sayisi (S)"},
/* 1423 */ { "CANADIAN SYLLABICS FINAL RIGHT HALF RING","	* West Cree (N), Athapascan (D/T), Sayisi (N), Carrier (N)"},
/* 1424 */ { "CANADIAN SYLLABICS FINAL RING","	* West Cree (W), Sayisi (O)"},
/* 1425 */ { "CANADIAN SYLLABICS FINAL DOUBLE ACUTE","	* Chipewyan (TT), South Slavey (GH)"},
/* 1426 */ { "CANADIAN SYLLABICS FINAL DOUBLE SHORT VERTICAL STROKES","	* Algonquian (H), Carrier (R)"},
/* 1427 */ { "CANADIAN SYLLABICS FINAL MIDDLE DOT","	* Moose Cree (W), Athapascan (Y), Sayisi (YU), Blackfoot (S)"},
/* 1428 */ { "CANADIAN SYLLABICS FINAL SHORT HORIZONTAL STROKE","	* West Cree (C), Sayisi (D)"},
/* 1429 */ { "CANADIAN SYLLABICS FINAL PLUS","	* Athapascan (N), Sayisi (AI)"},
/* 142A */ { "CANADIAN SYLLABICS FINAL DOWN TACK","	* N Cree (L), Carrier (D)\n"
	"	x (down tack - 22A4)"},
/* 142B */ { "CANADIAN SYLLABICS EN",NULL},
/* 142C */ { "CANADIAN SYLLABICS IN",NULL},
/* 142D */ { "CANADIAN SYLLABICS ON",NULL},
/* 142E */ { "CANADIAN SYLLABICS AN",NULL},
/* 142F */ { "CANADIAN SYLLABICS PE","	* Inuktitut (PAI), Athapascan (BE), Carrier (HU)"},
/* 1430 */ { "CANADIAN SYLLABICS PAAI","	* Inuktitut"},
/* 1431 */ { "CANADIAN SYLLABICS PI",NULL},
/* 1432 */ { "CANADIAN SYLLABICS PII",NULL},
/* 1433 */ { "CANADIAN SYLLABICS PO","	* Inuktitut (PU), Athapascan (BO), Carrier (HE)"},
/* 1434 */ { "CANADIAN SYLLABICS POO","	* Inuktitut (PUU)"},
/* 1435 */ { "CANADIAN SYLLABICS Y-CREE POO",NULL},
/* 1436 */ { "CANADIAN SYLLABICS CARRIER HEE",NULL},
/* 1437 */ { "CANADIAN SYLLABICS CARRIER HI",NULL},
/* 1438 */ { "CANADIAN SYLLABICS PA","	* Athapascan (BA), Carrier (HA)"},
/* 1439 */ { "CANADIAN SYLLABICS PAA",NULL},
/* 143A */ { "CANADIAN SYLLABICS PWE",NULL},
/* 143B */ { "CANADIAN SYLLABICS WEST-CREE PWE",NULL},
/* 143C */ { "CANADIAN SYLLABICS PWI",NULL},
/* 143D */ { "CANADIAN SYLLABICS WEST-CREE PWI",NULL},
/* 143E */ { "CANADIAN SYLLABICS PWII",NULL},
/* 143F */ { "CANADIAN SYLLABICS WEST-CREE PWII",NULL},
/* 1440 */ { "CANADIAN SYLLABICS PWO",NULL},
/* 1441 */ { "CANADIAN SYLLABICS WEST-CREE PWO",NULL},
/* 1442 */ { "CANADIAN SYLLABICS PWOO",NULL},
/* 1443 */ { "CANADIAN SYLLABICS WEST-CREE PWOO",NULL},
/* 1444 */ { "CANADIAN SYLLABICS PWA",NULL},
/* 1445 */ { "CANADIAN SYLLABICS WEST-CREE PWA",NULL},
/* 1446 */ { "CANADIAN SYLLABICS PWAA",NULL},
/* 1447 */ { "CANADIAN SYLLABICS WEST-CREE PWAA",NULL},
/* 1448 */ { "CANADIAN SYLLABICS Y-CREE PWAA",NULL},
/* 1449 */ { "CANADIAN SYLLABICS P","	* Blackfoot (Y)"},
/* 144A */ { "CANADIAN SYLLABICS WEST-CREE P","	* Sayisi (G)"},
/* 144B */ { "CANADIAN SYLLABICS CARRIER H",NULL},
/* 144C */ { "CANADIAN SYLLABICS TE","	* Inuktitut (TAI), Athapascan (DI), Carrier (DU)"},
/* 144D */ { "CANADIAN SYLLABICS TAAI","	* Inuktitut"},
/* 144E */ { "CANADIAN SYLLABICS TI","	* Athapascan (DE), Carrier (DO)"},
/* 144F */ { "CANADIAN SYLLABICS TII",NULL},
/* 1450 */ { "CANADIAN SYLLABICS TO","	* Inuktitut (TU), Athapascan (DO), Carrier (DE), Sayisi (DU)"},
/* 1451 */ { "CANADIAN SYLLABICS TOO","	* Inuktitut (TUU)"},
/* 1452 */ { "CANADIAN SYLLABICS Y-CREE TOO",NULL},
/* 1453 */ { "CANADIAN SYLLABICS CARRIER DEE",NULL},
/* 1454 */ { "CANADIAN SYLLABICS CARRIER DI",NULL},
/* 1455 */ { "CANADIAN SYLLABICS TA","	* Athapascan (DA)"},
/* 1456 */ { "CANADIAN SYLLABICS TAA",NULL},
/* 1457 */ { "CANADIAN SYLLABICS TWE",NULL},
/* 1458 */ { "CANADIAN SYLLABICS WEST-CREE TWE",NULL},
/* 1459 */ { "CANADIAN SYLLABICS TWI",NULL},
/* 145A */ { "CANADIAN SYLLABICS WEST-CREE TWI",NULL},
/* 145B */ { "CANADIAN SYLLABICS TWII",NULL},
/* 145C */ { "CANADIAN SYLLABICS WEST-CREE TWII",NULL},
/* 145D */ { "CANADIAN SYLLABICS TWO",NULL},
/* 145E */ { "CANADIAN SYLLABICS WEST-CREE TWO",NULL},
/* 145F */ { "CANADIAN SYLLABICS TWOO",NULL},
/* 1460 */ { "CANADIAN SYLLABICS WEST-CREE TWOO",NULL},
/* 1461 */ { "CANADIAN SYLLABICS TWA",NULL},
/* 1462 */ { "CANADIAN SYLLABICS WEST-CREE TWA",NULL},
/* 1463 */ { "CANADIAN SYLLABICS TWAA",NULL},
/* 1464 */ { "CANADIAN SYLLABICS WEST-CREE TWAA",NULL},
/* 1465 */ { "CANADIAN SYLLABICS NASKAPI TWAA",NULL},
/* 1466 */ { "CANADIAN SYLLABICS T",NULL},
/* 1467 */ { "CANADIAN SYLLABICS TTE","	* South Slavey (DEH)"},
/* 1468 */ { "CANADIAN SYLLABICS TTI","	* South Slavey (DIH)"},
/* 1469 */ { "CANADIAN SYLLABICS TTO","	* South Slavey (DOH)"},
/* 146A */ { "CANADIAN SYLLABICS TTA","	* South Slavey (DAH)"},
/* 146B */ { "CANADIAN SYLLABICS KE","	* Inuktitut (KAI), Blackfoot (PA)"},
/* 146C */ { "CANADIAN SYLLABICS KAAI","	* Inuktitut"},
/* 146D */ { "CANADIAN SYLLABICS KI","	* Blackfoot (PE)"},
/* 146E */ { "CANADIAN SYLLABICS KII",NULL},
/* 146F */ { "CANADIAN SYLLABICS KO","	* Inuktitut (KU), Sayisi (KU), Blackfoot (PI)"},
/* 1470 */ { "CANADIAN SYLLABICS KOO","	* Inuktitut (KUU)"},
/* 1471 */ { "CANADIAN SYLLABICS Y-CREE KOO",NULL},
/* 1472 */ { "CANADIAN SYLLABICS KA","	* Blackfoot (PO)"},
/* 1473 */ { "CANADIAN SYLLABICS KAA",NULL},
/* 1474 */ { "CANADIAN SYLLABICS KWE",NULL},
/* 1475 */ { "CANADIAN SYLLABICS WEST-CREE KWE",NULL},
/* 1476 */ { "CANADIAN SYLLABICS KWI",NULL},
/* 1477 */ { "CANADIAN SYLLABICS WEST-CREE KWI",NULL},
/* 1478 */ { "CANADIAN SYLLABICS KWII",NULL},
/* 1479 */ { "CANADIAN SYLLABICS WEST-CREE KWII",NULL},
/* 147A */ { "CANADIAN SYLLABICS KWO",NULL},
/* 147B */ { "CANADIAN SYLLABICS WEST-CREE KWO",NULL},
/* 147C */ { "CANADIAN SYLLABICS KWOO",NULL},
/* 147D */ { "CANADIAN SYLLABICS WEST-CREE KWOO",NULL},
/* 147E */ { "CANADIAN SYLLABICS KWA",NULL},
/* 147F */ { "CANADIAN SYLLABICS WEST-CREE KWA",NULL},
/* 1480 */ { "CANADIAN SYLLABICS KWAA",NULL},
/* 1481 */ { "CANADIAN SYLLABICS WEST-CREE KWAA",NULL},
/* 1482 */ { "CANADIAN SYLLABICS NASKAPI KWAA",NULL},
/* 1483 */ { "CANADIAN SYLLABICS K",NULL},
/* 1484 */ { "CANADIAN SYLLABICS KW","	* East Cree"},
/* 1485 */ { "CANADIAN SYLLABICS SOUTH-SLAVEY KEH",NULL},
/* 1486 */ { "CANADIAN SYLLABICS SOUTH-SLAVEY KIH",NULL},
/* 1487 */ { "CANADIAN SYLLABICS SOUTH-SLAVEY KOH",NULL},
/* 1488 */ { "CANADIAN SYLLABICS SOUTH-SLAVEY KAH",NULL},
/* 1489 */ { "CANADIAN SYLLABICS CE","	* Inuktitut (GAI), Athapascan (DHE), Sayisi (THE), Blackfoot (MA)"},
/* 148A */ { "CANADIAN SYLLABICS CAAI","	* Inuktitut (GAAI)"},
/* 148B */ { "CANADIAN SYLLABICS CI","	* Inuktitut (GI), Athapascan (DHI), Sayisi (THI), Blackfoot (ME)"},
/* 148C */ { "CANADIAN SYLLABICS CII","	* Inuktitut (GII)"},
/* 148D */ { "CANADIAN SYLLABICS CO","	* Inuktitut (GU), Athapascan (DHO), Sayisi (THO), Blackfoot (MI)"},
/* 148E */ { "CANADIAN SYLLABICS COO","	* Inuktitut (GUU)"},
/* 148F */ { "CANADIAN SYLLABICS Y-CREE COO",NULL},
/* 1490 */ { "CANADIAN SYLLABICS CA","	* Inuktitut (GA), Athapascan (DHA), Sayisi (THA), Blackfoot (MO)"},
/* 1491 */ { "CANADIAN SYLLABICS CAA","	* Inuktitut (GAA)"},
/* 1492 */ { "CANADIAN SYLLABICS CWE",NULL},
/* 1493 */ { "CANADIAN SYLLABICS WEST-CREE CWE",NULL},
/* 1494 */ { "CANADIAN SYLLABICS CWI",NULL},
/* 1495 */ { "CANADIAN SYLLABICS WEST-CREE CWI",NULL},
/* 1496 */ { "CANADIAN SYLLABICS CWII",NULL},
/* 1497 */ { "CANADIAN SYLLABICS WEST-CREE CWII",NULL},
/* 1498 */ { "CANADIAN SYLLABICS CWO",NULL},
/* 1499 */ { "CANADIAN SYLLABICS WEST-CREE CWO",NULL},
/* 149A */ { "CANADIAN SYLLABICS CWOO",NULL},
/* 149B */ { "CANADIAN SYLLABICS WEST-CREE CWOO",NULL},
/* 149C */ { "CANADIAN SYLLABICS CWA",NULL},
/* 149D */ { "CANADIAN SYLLABICS WEST-CREE CWA",NULL},
/* 149E */ { "CANADIAN SYLLABICS CWAA",NULL},
/* 149F */ { "CANADIAN SYLLABICS WEST-CREE CWAA",NULL},
/* 14A0 */ { "CANADIAN SYLLABICS NASKAPI CWAA",NULL},
/* 14A1 */ { "CANADIAN SYLLABICS C","	* Inuktitut (G), Sayisi (T)"},
/* 14A2 */ { "CANADIAN SYLLABICS SAYISI TH","	* Athapascan (DH)"},
/* 14A3 */ { "CANADIAN SYLLABICS ME","	* Inuktitut (MAI), Blackfoot (TA)"},
/* 14A4 */ { "CANADIAN SYLLABICS MAAI","	* Inuktitut"},
/* 14A5 */ { "CANADIAN SYLLABICS MI","	* Blackfoot (TE)"},
/* 14A6 */ { "CANADIAN SYLLABICS MII",NULL},
/* 14A7 */ { "CANADIAN SYLLABICS MO","	* Inuktitut (MU), Sayisi (MU), Blackfoot (TI)"},
/* 14A8 */ { "CANADIAN SYLLABICS MOO","	* Inuktitut (MUU)"},
/* 14A9 */ { "CANADIAN SYLLABICS Y-CREE MOO",NULL},
/* 14AA */ { "CANADIAN SYLLABICS MA","	* Blackfoot (TO)"},
/* 14AB */ { "CANADIAN SYLLABICS MAA",NULL},
/* 14AC */ { "CANADIAN SYLLABICS MWE",NULL},
/* 14AD */ { "CANADIAN SYLLABICS WEST-CREE MWE",NULL},
/* 14AE */ { "CANADIAN SYLLABICS MWI",NULL},
/* 14AF */ { "CANADIAN SYLLABICS WEST-CREE MWI",NULL},
/* 14B0 */ { "CANADIAN SYLLABICS MWII",NULL},
/* 14B1 */ { "CANADIAN SYLLABICS WEST-CREE MWII",NULL},
/* 14B2 */ { "CANADIAN SYLLABICS MWO",NULL},
/* 14B3 */ { "CANADIAN SYLLABICS WEST-CREE MWO",NULL},
/* 14B4 */ { "CANADIAN SYLLABICS MWOO",NULL},
/* 14B5 */ { "CANADIAN SYLLABICS WEST-CREE MWOO",NULL},
/* 14B6 */ { "CANADIAN SYLLABICS MWA",NULL},
/* 14B7 */ { "CANADIAN SYLLABICS WEST-CREE MWA",NULL},
/* 14B8 */ { "CANADIAN SYLLABICS MWAA",NULL},
/* 14B9 */ { "CANADIAN SYLLABICS WEST-CREE MWAA",NULL},
/* 14BA */ { "CANADIAN SYLLABICS NASKAPI MWAA",NULL},
/* 14BB */ { "CANADIAN SYLLABICS M",NULL},
/* 14BC */ { "CANADIAN SYLLABICS WEST-CREE M","	* Carrier (M)"},
/* 14BD */ { "CANADIAN SYLLABICS MH",NULL},
/* 14BE */ { "CANADIAN SYLLABICS ATHAPASCAN M",NULL},
/* 14BF */ { "CANADIAN SYLLABICS SAYISI M",NULL},
/* 14C0 */ { "CANADIAN SYLLABICS NE","	* Inuktitut (NAI)"},
/* 14C1 */ { "CANADIAN SYLLABICS NAAI","	* Inuktitut"},
/* 14C2 */ { "CANADIAN SYLLABICS NI",NULL},
/* 14C3 */ { "CANADIAN SYLLABICS NII",NULL},
/* 14C4 */ { "CANADIAN SYLLABICS NO","	* Inuktitut (NU), Sayisi (NU)"},
/* 14C5 */ { "CANADIAN SYLLABICS NOO","	* Inuktitut (NUU)"},
/* 14C6 */ { "CANADIAN SYLLABICS Y-CREE NOO",NULL},
/* 14C7 */ { "CANADIAN SYLLABICS NA",NULL},
/* 14C8 */ { "CANADIAN SYLLABICS NAA",NULL},
/* 14C9 */ { "CANADIAN SYLLABICS NWE",NULL},
/* 14CA */ { "CANADIAN SYLLABICS WEST-CREE NWE",NULL},
/* 14CB */ { "CANADIAN SYLLABICS NWA",NULL},
/* 14CC */ { "CANADIAN SYLLABICS WEST-CREE NWA",NULL},
/* 14CD */ { "CANADIAN SYLLABICS NWAA",NULL},
/* 14CE */ { "CANADIAN SYLLABICS WEST-CREE NWAA",NULL},
/* 14CF */ { "CANADIAN SYLLABICS NASKAPI NWAA",NULL},
/* 14D0 */ { "CANADIAN SYLLABICS N",NULL},
/* 14D1 */ { "CANADIAN SYLLABICS CARRIER NG",NULL},
/* 14D2 */ { "CANADIAN SYLLABICS NH",NULL},
/* 14D3 */ { "CANADIAN SYLLABICS LE","	* Inuktitut (LAI)"},
/* 14D4 */ { "CANADIAN SYLLABICS LAAI","	* Inuktitut"},
/* 14D5 */ { "CANADIAN SYLLABICS LI",NULL},
/* 14D6 */ { "CANADIAN SYLLABICS LII",NULL},
/* 14D7 */ { "CANADIAN SYLLABICS LO","	* Inuktitut (LU)"},
/* 14D8 */ { "CANADIAN SYLLABICS LOO","	* Inuktitut (LUU)"},
/* 14D9 */ { "CANADIAN SYLLABICS Y-CREE LOO",NULL},
/* 14DA */ { "CANADIAN SYLLABICS LA",NULL},
/* 14DB */ { "CANADIAN SYLLABICS LAA",NULL},
/* 14DC */ { "CANADIAN SYLLABICS LWE",NULL},
/* 14DD */ { "CANADIAN SYLLABICS WEST-CREE LWE",NULL},
/* 14DE */ { "CANADIAN SYLLABICS LWI",NULL},
/* 14DF */ { "CANADIAN SYLLABICS WEST-CREE LWI",NULL},
/* 14E0 */ { "CANADIAN SYLLABICS LWII",NULL},
/* 14E1 */ { "CANADIAN SYLLABICS WEST-CREE LWII",NULL},
/* 14E2 */ { "CANADIAN SYLLABICS LWO",NULL},
/* 14E3 */ { "CANADIAN SYLLABICS WEST-CREE LWO",NULL},
/* 14E4 */ { "CANADIAN SYLLABICS LWOO",NULL},
/* 14E5 */ { "CANADIAN SYLLABICS WEST-CREE LWOO",NULL},
/* 14E6 */ { "CANADIAN SYLLABICS LWA",NULL},
/* 14E7 */ { "CANADIAN SYLLABICS WEST-CREE LWA",NULL},
/* 14E8 */ { "CANADIAN SYLLABICS LWAA",NULL},
/* 14E9 */ { "CANADIAN SYLLABICS WEST-CREE LWAA",NULL},
/* 14EA */ { "CANADIAN SYLLABICS L",NULL},
/* 14EB */ { "CANADIAN SYLLABICS WEST-CREE L",NULL},
/* 14EC */ { "CANADIAN SYLLABICS MEDIAL L",NULL},
/* 14ED */ { "CANADIAN SYLLABICS SE","	* Inuktitut (SAI), Blackfoot (SA)"},
/* 14EE */ { "CANADIAN SYLLABICS SAAI","	* Inuktitut"},
/* 14EF */ { "CANADIAN SYLLABICS SI","	* Blackfoot (SE)"},
/* 14F0 */ { "CANADIAN SYLLABICS SII",NULL},
/* 14F1 */ { "CANADIAN SYLLABICS SO","	* Inuktitut (SU), Sayisi (SU), Blackfoot (SI)"},
/* 14F2 */ { "CANADIAN SYLLABICS SOO","	* Inuktitut (SUU)"},
/* 14F3 */ { "CANADIAN SYLLABICS Y-CREE SOO",NULL},
/* 14F4 */ { "CANADIAN SYLLABICS SA","	* Blackfoot (SO)"},
/* 14F5 */ { "CANADIAN SYLLABICS SAA",NULL},
/* 14F6 */ { "CANADIAN SYLLABICS SWE",NULL},
/* 14F7 */ { "CANADIAN SYLLABICS WEST-CREE SWE",NULL},
/* 14F8 */ { "CANADIAN SYLLABICS SWI",NULL},
/* 14F9 */ { "CANADIAN SYLLABICS WEST-CREE SWI",NULL},
/* 14FA */ { "CANADIAN SYLLABICS SWII",NULL},
/* 14FB */ { "CANADIAN SYLLABICS WEST-CREE SWII",NULL},
/* 14FC */ { "CANADIAN SYLLABICS SWO",NULL},
/* 14FD */ { "CANADIAN SYLLABICS WEST-CREE SWO",NULL},
/* 14FE */ { "CANADIAN SYLLABICS SWOO",NULL},
/* 14FF */ { "CANADIAN SYLLABICS WEST-CREE SWOO",NULL}
};

UN_DLL_LOCAL
static const struct unicode_nameannot una_00_15[] = {
/* 1500 */ { "CANADIAN SYLLABICS SWA",NULL},
/* 1501 */ { "CANADIAN SYLLABICS WEST-CREE SWA",NULL},
/* 1502 */ { "CANADIAN SYLLABICS SWAA",NULL},
/* 1503 */ { "CANADIAN SYLLABICS WEST-CREE SWAA",NULL},
/* 1504 */ { "CANADIAN SYLLABICS NASKAPI SWAA",NULL},
/* 1505 */ { "CANADIAN SYLLABICS S",NULL},
/* 1506 */ { "CANADIAN SYLLABICS ATHAPASCAN S",NULL},
/* 1507 */ { "CANADIAN SYLLABICS SW",NULL},
/* 1508 */ { "CANADIAN SYLLABICS BLACKFOOT S",NULL},
/* 1509 */ { "CANADIAN SYLLABICS MOOSE-CREE SK",NULL},
/* 150A */ { "CANADIAN SYLLABICS NASKAPI SKW",NULL},
/* 150B */ { "CANADIAN SYLLABICS NASKAPI S-W",NULL},
/* 150C */ { "CANADIAN SYLLABICS NASKAPI SPWA",NULL},
/* 150D */ { "CANADIAN SYLLABICS NASKAPI STWA",NULL},
/* 150E */ { "CANADIAN SYLLABICS NASKAPI SKWA",NULL},
/* 150F */ { "CANADIAN SYLLABICS NASKAPI SCWA",NULL},
/* 1510 */ { "CANADIAN SYLLABICS SHE",NULL},
/* 1511 */ { "CANADIAN SYLLABICS SHI",NULL},
/* 1512 */ { "CANADIAN SYLLABICS SHII",NULL},
/* 1513 */ { "CANADIAN SYLLABICS SHO",NULL},
/* 1514 */ { "CANADIAN SYLLABICS SHOO",NULL},
/* 1515 */ { "CANADIAN SYLLABICS SHA",NULL},
/* 1516 */ { "CANADIAN SYLLABICS SHAA",NULL},
/* 1517 */ { "CANADIAN SYLLABICS SHWE",NULL},
/* 1518 */ { "CANADIAN SYLLABICS WEST-CREE SHWE",NULL},
/* 1519 */ { "CANADIAN SYLLABICS SHWI",NULL},
/* 151A */ { "CANADIAN SYLLABICS WEST-CREE SHWI",NULL},
/* 151B */ { "CANADIAN SYLLABICS SHWII",NULL},
/* 151C */ { "CANADIAN SYLLABICS WEST-CREE SHWII",NULL},
/* 151D */ { "CANADIAN SYLLABICS SHWO",NULL},
/* 151E */ { "CANADIAN SYLLABICS WEST-CREE SHWO",NULL},
/* 151F */ { "CANADIAN SYLLABICS SHWOO",NULL},
/* 1520 */ { "CANADIAN SYLLABICS WEST-CREE SHWOO",NULL},
/* 1521 */ { "CANADIAN SYLLABICS SHWA",NULL},
/* 1522 */ { "CANADIAN SYLLABICS WEST-CREE SHWA",NULL},
/* 1523 */ { "CANADIAN SYLLABICS SHWAA",NULL},
/* 1524 */ { "CANADIAN SYLLABICS WEST-CREE SHWAA",NULL},
/* 1525 */ { "CANADIAN SYLLABICS SH",NULL},
/* 1526 */ { "CANADIAN SYLLABICS YE","	* Inuktitut (YAI), Blackfoot (YA)"},
/* 1527 */ { "CANADIAN SYLLABICS YAAI","	* Inuktitut"},
/* 1528 */ { "CANADIAN SYLLABICS YI","	* Blackfoot (YE)"},
/* 1529 */ { "CANADIAN SYLLABICS YII",NULL},
/* 152A */ { "CANADIAN SYLLABICS YO","	* Inuktitut (YU), Blackfoot (YI)"},
/* 152B */ { "CANADIAN SYLLABICS YOO","	* Inuktitut (YUU)"},
/* 152C */ { "CANADIAN SYLLABICS Y-CREE YOO",NULL},
/* 152D */ { "CANADIAN SYLLABICS YA","	* Blackfoot (YO)"},
/* 152E */ { "CANADIAN SYLLABICS YAA",NULL},
/* 152F */ { "CANADIAN SYLLABICS YWE",NULL},
/* 1530 */ { "CANADIAN SYLLABICS WEST-CREE YWE",NULL},
/* 1531 */ { "CANADIAN SYLLABICS YWI",NULL},
/* 1532 */ { "CANADIAN SYLLABICS WEST-CREE YWI",NULL},
/* 1533 */ { "CANADIAN SYLLABICS YWII",NULL},
/* 1534 */ { "CANADIAN SYLLABICS WEST-CREE YWII",NULL},
/* 1535 */ { "CANADIAN SYLLABICS YWO",NULL},
/* 1536 */ { "CANADIAN SYLLABICS WEST-CREE YWO",NULL},
/* 1537 */ { "CANADIAN SYLLABICS YWOO",NULL},
/* 1538 */ { "CANADIAN SYLLABICS WEST-CREE YWOO",NULL},
/* 1539 */ { "CANADIAN SYLLABICS YWA",NULL},
/* 153A */ { "CANADIAN SYLLABICS WEST-CREE YWA",NULL},
/* 153B */ { "CANADIAN SYLLABICS YWAA",NULL},
/* 153C */ { "CANADIAN SYLLABICS WEST-CREE YWAA",NULL},
/* 153D */ { "CANADIAN SYLLABICS NASKAPI YWAA",NULL},
/* 153E */ { "CANADIAN SYLLABICS Y",NULL},
/* 153F */ { "CANADIAN SYLLABICS BIBLE-CREE Y",NULL},
/* 1540 */ { "CANADIAN SYLLABICS WEST-CREE Y",NULL},
/* 1541 */ { "CANADIAN SYLLABICS SAYISI YI",NULL},
/* 1542 */ { "CANADIAN SYLLABICS RE","	* Inuktitut (RAI)"},
/* 1543 */ { "CANADIAN SYLLABICS R-CREE RE","	* Athapascan (LE)"},
/* 1544 */ { "CANADIAN SYLLABICS WEST-CREE LE","	* Athapascan (LI)"},
/* 1545 */ { "CANADIAN SYLLABICS RAAI","	* Inuktitut"},
/* 1546 */ { "CANADIAN SYLLABICS RI",NULL},
/* 1547 */ { "CANADIAN SYLLABICS RII",NULL},
/* 1548 */ { "CANADIAN SYLLABICS RO","	* Inuktitut (RU)"},
/* 1549 */ { "CANADIAN SYLLABICS ROO","	* Inuktitut (RUU)"},
/* 154A */ { "CANADIAN SYLLABICS WEST-CREE LO","	* Sayisi (LU)"},
/* 154B */ { "CANADIAN SYLLABICS RA",NULL},
/* 154C */ { "CANADIAN SYLLABICS RAA",NULL},
/* 154D */ { "CANADIAN SYLLABICS WEST-CREE LA",NULL},
/* 154E */ { "CANADIAN SYLLABICS RWAA",NULL},
/* 154F */ { "CANADIAN SYLLABICS WEST-CREE RWAA",NULL},
/* 1550 */ { "CANADIAN SYLLABICS R",NULL},
/* 1551 */ { "CANADIAN SYLLABICS WEST-CREE R",NULL},
/* 1552 */ { "CANADIAN SYLLABICS MEDIAL R",NULL},
/* 1553 */ { "CANADIAN SYLLABICS FE","	* Inuktitut (FAI)"},
/* 1554 */ { "CANADIAN SYLLABICS FAAI","	* Inuktitut"},
/* 1555 */ { "CANADIAN SYLLABICS FI",NULL},
/* 1556 */ { "CANADIAN SYLLABICS FII",NULL},
/* 1557 */ { "CANADIAN SYLLABICS FO",NULL},
/* 1558 */ { "CANADIAN SYLLABICS FOO",NULL},
/* 1559 */ { "CANADIAN SYLLABICS FA",NULL},
/* 155A */ { "CANADIAN SYLLABICS FAA",NULL},
/* 155B */ { "CANADIAN SYLLABICS FWAA",NULL},
/* 155C */ { "CANADIAN SYLLABICS WEST-CREE FWAA",NULL},
/* 155D */ { "CANADIAN SYLLABICS F",NULL},
/* 155E */ { "CANADIAN SYLLABICS THE","	* Sayisi (TE)"},
/* 155F */ { "CANADIAN SYLLABICS N-CREE THE",NULL},
/* 1560 */ { "CANADIAN SYLLABICS THI","	* Sayisi (TI)"},
/* 1561 */ { "CANADIAN SYLLABICS N-CREE THI",NULL},
/* 1562 */ { "CANADIAN SYLLABICS THII",NULL},
/* 1563 */ { "CANADIAN SYLLABICS N-CREE THII",NULL},
/* 1564 */ { "CANADIAN SYLLABICS THO","	* Sayisi (TU)"},
/* 1565 */ { "CANADIAN SYLLABICS THOO",NULL},
/* 1566 */ { "CANADIAN SYLLABICS THA","	* Sayisi (TA)"},
/* 1567 */ { "CANADIAN SYLLABICS THAA",NULL},
/* 1568 */ { "CANADIAN SYLLABICS THWAA",NULL},
/* 1569 */ { "CANADIAN SYLLABICS WEST-CREE THWAA",NULL},
/* 156A */ { "CANADIAN SYLLABICS TH",NULL},
/* 156B */ { "CANADIAN SYLLABICS TTHE",NULL},
/* 156C */ { "CANADIAN SYLLABICS TTHI",NULL},
/* 156D */ { "CANADIAN SYLLABICS TTHO","	* Sayisi (TTHU)"},
/* 156E */ { "CANADIAN SYLLABICS TTHA",NULL},
/* 156F */ { "CANADIAN SYLLABICS TTH","	* probably a mistaken interpretation of an asterisk used to mark a proper noun\n"
	"	x (asterisk - 002A)"},
/* 1570 */ { "CANADIAN SYLLABICS TYE",NULL},
/* 1571 */ { "CANADIAN SYLLABICS TYI",NULL},
/* 1572 */ { "CANADIAN SYLLABICS TYO",NULL},
/* 1573 */ { "CANADIAN SYLLABICS TYA",NULL},
/* 1574 */ { "CANADIAN SYLLABICS NUNAVIK HE",NULL},
/* 1575 */ { "CANADIAN SYLLABICS NUNAVIK HI",NULL},
/* 1576 */ { "CANADIAN SYLLABICS NUNAVIK HII",NULL},
/* 1577 */ { "CANADIAN SYLLABICS NUNAVIK HO",NULL},
/* 1578 */ { "CANADIAN SYLLABICS NUNAVIK HOO",NULL},
/* 1579 */ { "CANADIAN SYLLABICS NUNAVIK HA",NULL},
/* 157A */ { "CANADIAN SYLLABICS NUNAVIK HAA",NULL},
/* 157B */ { "CANADIAN SYLLABICS NUNAVIK H",NULL},
/* 157C */ { "CANADIAN SYLLABICS NUNAVUT H",NULL},
/* 157D */ { "CANADIAN SYLLABICS HK","	* Algonquian"},
/* 157E */ { "CANADIAN SYLLABICS QAAI","	* Inuktitut"},
/* 157F */ { "CANADIAN SYLLABICS QI",NULL},
/* 1580 */ { "CANADIAN SYLLABICS QII",NULL},
/* 1581 */ { "CANADIAN SYLLABICS QO","	* Inuktitut (QU)"},
/* 1582 */ { "CANADIAN SYLLABICS QOO","	* Inuktitut (QUU)"},
/* 1583 */ { "CANADIAN SYLLABICS QA",NULL},
/* 1584 */ { "CANADIAN SYLLABICS QAA",NULL},
/* 1585 */ { "CANADIAN SYLLABICS Q",NULL},
/* 1586 */ { "CANADIAN SYLLABICS TLHE","	* Sayisi (KLE)"},
/* 1587 */ { "CANADIAN SYLLABICS TLHI","	* Sayisi (KLI)"},
/* 1588 */ { "CANADIAN SYLLABICS TLHO","	* Sayisi (KLU)"},
/* 1589 */ { "CANADIAN SYLLABICS TLHA","	* Sayisi (KLA)"},
/* 158A */ { "CANADIAN SYLLABICS WEST-CREE RE",NULL},
/* 158B */ { "CANADIAN SYLLABICS WEST-CREE RI",NULL},
/* 158C */ { "CANADIAN SYLLABICS WEST-CREE RO",NULL},
/* 158D */ { "CANADIAN SYLLABICS WEST-CREE RA",NULL},
/* 158E */ { "CANADIAN SYLLABICS NGAAI","	* Inuktitut"},
/* 158F */ { "CANADIAN SYLLABICS NGI",NULL},
/* 1590 */ { "CANADIAN SYLLABICS NGII",NULL},
/* 1591 */ { "CANADIAN SYLLABICS NGO","	* Inuktitut (NGU)"},
/* 1592 */ { "CANADIAN SYLLABICS NGOO","	* Inuktitut (NGUU)"},
/* 1593 */ { "CANADIAN SYLLABICS NGA",NULL},
/* 1594 */ { "CANADIAN SYLLABICS NGAA",NULL},
/* 1595 */ { "CANADIAN SYLLABICS NG",NULL},
/* 1596 */ { "CANADIAN SYLLABICS NNG",NULL},
/* 1597 */ { "CANADIAN SYLLABICS SAYISI SHE",NULL},
/* 1598 */ { "CANADIAN SYLLABICS SAYISI SHI",NULL},
/* 1599 */ { "CANADIAN SYLLABICS SAYISI SHO","	* Sayisi (SHU)"},
/* 159A */ { "CANADIAN SYLLABICS SAYISI SHA",NULL},
/* 159B */ { "CANADIAN SYLLABICS WOODS-CREE THE",NULL},
/* 159C */ { "CANADIAN SYLLABICS WOODS-CREE THI",NULL},
/* 159D */ { "CANADIAN SYLLABICS WOODS-CREE THO",NULL},
/* 159E */ { "CANADIAN SYLLABICS WOODS-CREE THA",NULL},
/* 159F */ { "CANADIAN SYLLABICS WOODS-CREE TH",NULL},
/* 15A0 */ { "CANADIAN SYLLABICS LHI",NULL},
/* 15A1 */ { "CANADIAN SYLLABICS LHII",NULL},
/* 15A2 */ { "CANADIAN SYLLABICS LHO","	* Inuktitut (LHU)"},
/* 15A3 */ { "CANADIAN SYLLABICS LHOO","	* Inuktitut (LHUU)"},
/* 15A4 */ { "CANADIAN SYLLABICS LHA",NULL},
/* 15A5 */ { "CANADIAN SYLLABICS LHAA",NULL},
/* 15A6 */ { "CANADIAN SYLLABICS LH",NULL},
/* 15A7 */ { "CANADIAN SYLLABICS TH-CREE THE",NULL},
/* 15A8 */ { "CANADIAN SYLLABICS TH-CREE THI",NULL},
/* 15A9 */ { "CANADIAN SYLLABICS TH-CREE THII",NULL},
/* 15AA */ { "CANADIAN SYLLABICS TH-CREE THO",NULL},
/* 15AB */ { "CANADIAN SYLLABICS TH-CREE THOO",NULL},
/* 15AC */ { "CANADIAN SYLLABICS TH-CREE THA",NULL},
/* 15AD */ { "CANADIAN SYLLABICS TH-CREE THAA",NULL},
/* 15AE */ { "CANADIAN SYLLABICS TH-CREE TH",NULL},
/* 15AF */ { "CANADIAN SYLLABICS AIVILIK B",NULL},
/* 15B0 */ { "CANADIAN SYLLABICS BLACKFOOT E",NULL},
/* 15B1 */ { "CANADIAN SYLLABICS BLACKFOOT I",NULL},
/* 15B2 */ { "CANADIAN SYLLABICS BLACKFOOT O",NULL},
/* 15B3 */ { "CANADIAN SYLLABICS BLACKFOOT A",NULL},
/* 15B4 */ { "CANADIAN SYLLABICS BLACKFOOT WE",NULL},
/* 15B5 */ { "CANADIAN SYLLABICS BLACKFOOT WI",NULL},
/* 15B6 */ { "CANADIAN SYLLABICS BLACKFOOT WO",NULL},
/* 15B7 */ { "CANADIAN SYLLABICS BLACKFOOT WA",NULL},
/* 15B8 */ { "CANADIAN SYLLABICS BLACKFOOT NE",NULL},
/* 15B9 */ { "CANADIAN SYLLABICS BLACKFOOT NI",NULL},
/* 15BA */ { "CANADIAN SYLLABICS BLACKFOOT NO",NULL},
/* 15BB */ { "CANADIAN SYLLABICS BLACKFOOT NA",NULL},
/* 15BC */ { "CANADIAN SYLLABICS BLACKFOOT KE",NULL},
/* 15BD */ { "CANADIAN SYLLABICS BLACKFOOT KI",NULL},
/* 15BE */ { "CANADIAN SYLLABICS BLACKFOOT KO",NULL},
/* 15BF */ { "CANADIAN SYLLABICS BLACKFOOT KA",NULL},
/* 15C0 */ { "CANADIAN SYLLABICS SAYISI HE",NULL},
/* 15C1 */ { "CANADIAN SYLLABICS SAYISI HI",NULL},
/* 15C2 */ { "CANADIAN SYLLABICS SAYISI HO","	* Sayisi (HU)"},
/* 15C3 */ { "CANADIAN SYLLABICS SAYISI HA",NULL},
/* 15C4 */ { "CANADIAN SYLLABICS CARRIER GHU",NULL},
/* 15C5 */ { "CANADIAN SYLLABICS CARRIER GHO",NULL},
/* 15C6 */ { "CANADIAN SYLLABICS CARRIER GHE",NULL},
/* 15C7 */ { "CANADIAN SYLLABICS CARRIER GHEE",NULL},
/* 15C8 */ { "CANADIAN SYLLABICS CARRIER GHI",NULL},
/* 15C9 */ { "CANADIAN SYLLABICS CARRIER GHA",NULL},
/* 15CA */ { "CANADIAN SYLLABICS CARRIER RU",NULL},
/* 15CB */ { "CANADIAN SYLLABICS CARRIER RO",NULL},
/* 15CC */ { "CANADIAN SYLLABICS CARRIER RE",NULL},
/* 15CD */ { "CANADIAN SYLLABICS CARRIER REE",NULL},
/* 15CE */ { "CANADIAN SYLLABICS CARRIER RI",NULL},
/* 15CF */ { "CANADIAN SYLLABICS CARRIER RA",NULL},
/* 15D0 */ { "CANADIAN SYLLABICS CARRIER WU",NULL},
/* 15D1 */ { "CANADIAN SYLLABICS CARRIER WO",NULL},
/* 15D2 */ { "CANADIAN SYLLABICS CARRIER WE",NULL},
/* 15D3 */ { "CANADIAN SYLLABICS CARRIER WEE",NULL},
/* 15D4 */ { "CANADIAN SYLLABICS CARRIER WI",NULL},
/* 15D5 */ { "CANADIAN SYLLABICS CARRIER WA",NULL},
/* 15D6 */ { "CANADIAN SYLLABICS CARRIER HWU",NULL},
/* 15D7 */ { "CANADIAN SYLLABICS CARRIER HWO",NULL},
/* 15D8 */ { "CANADIAN SYLLABICS CARRIER HWE",NULL},
/* 15D9 */ { "CANADIAN SYLLABICS CARRIER HWEE",NULL},
/* 15DA */ { "CANADIAN SYLLABICS CARRIER HWI",NULL},
/* 15DB */ { "CANADIAN SYLLABICS CARRIER HWA",NULL},
/* 15DC */ { "CANADIAN SYLLABICS CARRIER THU",NULL},
/* 15DD */ { "CANADIAN SYLLABICS CARRIER THO",NULL},
/* 15DE */ { "CANADIAN SYLLABICS CARRIER THE",NULL},
/* 15DF */ { "CANADIAN SYLLABICS CARRIER THEE",NULL},
/* 15E0 */ { "CANADIAN SYLLABICS CARRIER THI",NULL},
/* 15E1 */ { "CANADIAN SYLLABICS CARRIER THA",NULL},
/* 15E2 */ { "CANADIAN SYLLABICS CARRIER TTU",NULL},
/* 15E3 */ { "CANADIAN SYLLABICS CARRIER TTO",NULL},
/* 15E4 */ { "CANADIAN SYLLABICS CARRIER TTE",NULL},
/* 15E5 */ { "CANADIAN SYLLABICS CARRIER TTEE",NULL},
/* 15E6 */ { "CANADIAN SYLLABICS CARRIER TTI",NULL},
/* 15E7 */ { "CANADIAN SYLLABICS CARRIER TTA",NULL},
/* 15E8 */ { "CANADIAN SYLLABICS CARRIER PU",NULL},
/* 15E9 */ { "CANADIAN SYLLABICS CARRIER PO",NULL},
/* 15EA */ { "CANADIAN SYLLABICS CARRIER PE",NULL},
/* 15EB */ { "CANADIAN SYLLABICS CARRIER PEE",NULL},
/* 15EC */ { "CANADIAN SYLLABICS CARRIER PI",NULL},
/* 15ED */ { "CANADIAN SYLLABICS CARRIER PA",NULL},
/* 15EE */ { "CANADIAN SYLLABICS CARRIER P",NULL},
/* 15EF */ { "CANADIAN SYLLABICS CARRIER GU",NULL},
/* 15F0 */ { "CANADIAN SYLLABICS CARRIER GO",NULL},
/* 15F1 */ { "CANADIAN SYLLABICS CARRIER GE",NULL},
/* 15F2 */ { "CANADIAN SYLLABICS CARRIER GEE",NULL},
/* 15F3 */ { "CANADIAN SYLLABICS CARRIER GI",NULL},
/* 15F4 */ { "CANADIAN SYLLABICS CARRIER GA",NULL},
/* 15F5 */ { "CANADIAN SYLLABICS CARRIER KHU",NULL},
/* 15F6 */ { "CANADIAN SYLLABICS CARRIER KHO",NULL},
/* 15F7 */ { "CANADIAN SYLLABICS CARRIER KHE",NULL},
/* 15F8 */ { "CANADIAN SYLLABICS CARRIER KHEE",NULL},
/* 15F9 */ { "CANADIAN SYLLABICS CARRIER KHI",NULL},
/* 15FA */ { "CANADIAN SYLLABICS CARRIER KHA",NULL},
/* 15FB */ { "CANADIAN SYLLABICS CARRIER KKU",NULL},
/* 15FC */ { "CANADIAN SYLLABICS CARRIER KKO",NULL},
/* 15FD */ { "CANADIAN SYLLABICS CARRIER KKE",NULL},
/* 15FE */ { "CANADIAN SYLLABICS CARRIER KKEE",NULL},
/* 15FF */ { "CANADIAN SYLLABICS CARRIER KKI",NULL}
};

UN_DLL_LOCAL
static const struct unicode_nameannot una_00_16[] = {
/* 1600 */ { "CANADIAN SYLLABICS CARRIER KKA",NULL},
/* 1601 */ { "CANADIAN SYLLABICS CARRIER KK",NULL},
/* 1602 */ { "CANADIAN SYLLABICS CARRIER NU",NULL},
/* 1603 */ { "CANADIAN SYLLABICS CARRIER NO",NULL},
/* 1604 */ { "CANADIAN SYLLABICS CARRIER NE",NULL},
/* 1605 */ { "CANADIAN SYLLABICS CARRIER NEE",NULL},
/* 1606 */ { "CANADIAN SYLLABICS CARRIER NI",NULL},
/* 1607 */ { "CANADIAN SYLLABICS CARRIER NA",NULL},
/* 1608 */ { "CANADIAN SYLLABICS CARRIER MU",NULL},
/* 1609 */ { "CANADIAN SYLLABICS CARRIER MO",NULL},
/* 160A */ { "CANADIAN SYLLABICS CARRIER ME",NULL},
/* 160B */ { "CANADIAN SYLLABICS CARRIER MEE",NULL},
/* 160C */ { "CANADIAN SYLLABICS CARRIER MI",NULL},
/* 160D */ { "CANADIAN SYLLABICS CARRIER MA",NULL},
/* 160E */ { "CANADIAN SYLLABICS CARRIER YU",NULL},
/* 160F */ { "CANADIAN SYLLABICS CARRIER YO",NULL},
/* 1610 */ { "CANADIAN SYLLABICS CARRIER YE",NULL},
/* 1611 */ { "CANADIAN SYLLABICS CARRIER YEE",NULL},
/* 1612 */ { "CANADIAN SYLLABICS CARRIER YI",NULL},
/* 1613 */ { "CANADIAN SYLLABICS CARRIER YA",NULL},
/* 1614 */ { "CANADIAN SYLLABICS CARRIER JU","	* Athapascan (ZA), Sayisi (TZO), South Slavey (DHA)"},
/* 1615 */ { "CANADIAN SYLLABICS SAYISI JU","	* Athapascan (ZO), Sayisi (TZU), South Slavey (DHO)"},
/* 1616 */ { "CANADIAN SYLLABICS CARRIER JO",NULL},
/* 1617 */ { "CANADIAN SYLLABICS CARRIER JE",NULL},
/* 1618 */ { "CANADIAN SYLLABICS CARRIER JEE",NULL},
/* 1619 */ { "CANADIAN SYLLABICS CARRIER JI",NULL},
/* 161A */ { "CANADIAN SYLLABICS SAYISI JI","	* Athapascan (ZE), Sayisi (TZE), South Slavey (DHE)"},
/* 161B */ { "CANADIAN SYLLABICS CARRIER JA","	* Athapascan (ZI), Sayisi (TZI), South Slavey (DHI)"},
/* 161C */ { "CANADIAN SYLLABICS CARRIER JJU",NULL},
/* 161D */ { "CANADIAN SYLLABICS CARRIER JJO",NULL},
/* 161E */ { "CANADIAN SYLLABICS CARRIER JJE",NULL},
/* 161F */ { "CANADIAN SYLLABICS CARRIER JJEE",NULL},
/* 1620 */ { "CANADIAN SYLLABICS CARRIER JJI",NULL},
/* 1621 */ { "CANADIAN SYLLABICS CARRIER JJA",NULL},
/* 1622 */ { "CANADIAN SYLLABICS CARRIER LU",NULL},
/* 1623 */ { "CANADIAN SYLLABICS CARRIER LO",NULL},
/* 1624 */ { "CANADIAN SYLLABICS CARRIER LE",NULL},
/* 1625 */ { "CANADIAN SYLLABICS CARRIER LEE",NULL},
/* 1626 */ { "CANADIAN SYLLABICS CARRIER LI",NULL},
/* 1627 */ { "CANADIAN SYLLABICS CARRIER LA",NULL},
/* 1628 */ { "CANADIAN SYLLABICS CARRIER DLU",NULL},
/* 1629 */ { "CANADIAN SYLLABICS CARRIER DLO",NULL},
/* 162A */ { "CANADIAN SYLLABICS CARRIER DLE",NULL},
/* 162B */ { "CANADIAN SYLLABICS CARRIER DLEE",NULL},
/* 162C */ { "CANADIAN SYLLABICS CARRIER DLI",NULL},
/* 162D */ { "CANADIAN SYLLABICS CARRIER DLA",NULL},
/* 162E */ { "CANADIAN SYLLABICS CARRIER LHU",NULL},
/* 162F */ { "CANADIAN SYLLABICS CARRIER LHO",NULL},
/* 1630 */ { "CANADIAN SYLLABICS CARRIER LHE",NULL},
/* 1631 */ { "CANADIAN SYLLABICS CARRIER LHEE",NULL},
/* 1632 */ { "CANADIAN SYLLABICS CARRIER LHI",NULL},
/* 1633 */ { "CANADIAN SYLLABICS CARRIER LHA",NULL},
/* 1634 */ { "CANADIAN SYLLABICS CARRIER TLHU",NULL},
/* 1635 */ { "CANADIAN SYLLABICS CARRIER TLHO",NULL},
/* 1636 */ { "CANADIAN SYLLABICS CARRIER TLHE",NULL},
/* 1637 */ { "CANADIAN SYLLABICS CARRIER TLHEE",NULL},
/* 1638 */ { "CANADIAN SYLLABICS CARRIER TLHI",NULL},
/* 1639 */ { "CANADIAN SYLLABICS CARRIER TLHA",NULL},
/* 163A */ { "CANADIAN SYLLABICS CARRIER TLU",NULL},
/* 163B */ { "CANADIAN SYLLABICS CARRIER TLO",NULL},
/* 163C */ { "CANADIAN SYLLABICS CARRIER TLE",NULL},
/* 163D */ { "CANADIAN SYLLABICS CARRIER TLEE",NULL},
/* 163E */ { "CANADIAN SYLLABICS CARRIER TLI",NULL},
/* 163F */ { "CANADIAN SYLLABICS CARRIER TLA",NULL},
/* 1640 */ { "CANADIAN SYLLABICS CARRIER ZU",NULL},
/* 1641 */ { "CANADIAN SYLLABICS CARRIER ZO",NULL},
/* 1642 */ { "CANADIAN SYLLABICS CARRIER ZE",NULL},
/* 1643 */ { "CANADIAN SYLLABICS CARRIER ZEE",NULL},
/* 1644 */ { "CANADIAN SYLLABICS CARRIER ZI",NULL},
/* 1645 */ { "CANADIAN SYLLABICS CARRIER ZA",NULL},
/* 1646 */ { "CANADIAN SYLLABICS CARRIER Z",NULL},
/* 1647 */ { "CANADIAN SYLLABICS CARRIER INITIAL Z",NULL},
/* 1648 */ { "CANADIAN SYLLABICS CARRIER DZU",NULL},
/* 1649 */ { "CANADIAN SYLLABICS CARRIER DZO",NULL},
/* 164A */ { "CANADIAN SYLLABICS CARRIER DZE",NULL},
/* 164B */ { "CANADIAN SYLLABICS CARRIER DZEE",NULL},
/* 164C */ { "CANADIAN SYLLABICS CARRIER DZI",NULL},
/* 164D */ { "CANADIAN SYLLABICS CARRIER DZA",NULL},
/* 164E */ { "CANADIAN SYLLABICS CARRIER SU",NULL},
/* 164F */ { "CANADIAN SYLLABICS CARRIER SO",NULL},
/* 1650 */ { "CANADIAN SYLLABICS CARRIER SE",NULL},
/* 1651 */ { "CANADIAN SYLLABICS CARRIER SEE",NULL},
/* 1652 */ { "CANADIAN SYLLABICS CARRIER SI",NULL},
/* 1653 */ { "CANADIAN SYLLABICS CARRIER SA",NULL},
/* 1654 */ { "CANADIAN SYLLABICS CARRIER SHU",NULL},
/* 1655 */ { "CANADIAN SYLLABICS CARRIER SHO",NULL},
/* 1656 */ { "CANADIAN SYLLABICS CARRIER SHE",NULL},
/* 1657 */ { "CANADIAN SYLLABICS CARRIER SHEE",NULL},
/* 1658 */ { "CANADIAN SYLLABICS CARRIER SHI",NULL},
/* 1659 */ { "CANADIAN SYLLABICS CARRIER SHA",NULL},
/* 165A */ { "CANADIAN SYLLABICS CARRIER SH",NULL},
/* 165B */ { "CANADIAN SYLLABICS CARRIER TSU",NULL},
/* 165C */ { "CANADIAN SYLLABICS CARRIER TSO",NULL},
/* 165D */ { "CANADIAN SYLLABICS CARRIER TSE",NULL},
/* 165E */ { "CANADIAN SYLLABICS CARRIER TSEE",NULL},
/* 165F */ { "CANADIAN SYLLABICS CARRIER TSI",NULL},
/* 1660 */ { "CANADIAN SYLLABICS CARRIER TSA",NULL},
/* 1661 */ { "CANADIAN SYLLABICS CARRIER CHU",NULL},
/* 1662 */ { "CANADIAN SYLLABICS CARRIER CHO",NULL},
/* 1663 */ { "CANADIAN SYLLABICS CARRIER CHE",NULL},
/* 1664 */ { "CANADIAN SYLLABICS CARRIER CHEE",NULL},
/* 1665 */ { "CANADIAN SYLLABICS CARRIER CHI",NULL},
/* 1666 */ { "CANADIAN SYLLABICS CARRIER CHA",NULL},
/* 1667 */ { "CANADIAN SYLLABICS CARRIER TTSU",NULL},
/* 1668 */ { "CANADIAN SYLLABICS CARRIER TTSO",NULL},
/* 1669 */ { "CANADIAN SYLLABICS CARRIER TTSE",NULL},
/* 166A */ { "CANADIAN SYLLABICS CARRIER TTSEE",NULL},
/* 166B */ { "CANADIAN SYLLABICS CARRIER TTSI",NULL},
/* 166C */ { "CANADIAN SYLLABICS CARRIER TTSA",NULL},
/* 166D */ { "CANADIAN SYLLABICS CHI SIGN","	* Algonquian\n"
	"	* used as a symbol to denote Christ\n"
	"	x (chi rho - 2627)"},
/* 166E */ { "CANADIAN SYLLABICS FULL STOP","	x (stenographic full stop - 2E3C)"},
/* 166F */ { "CANADIAN SYLLABICS QAI",NULL},
/* 1670 */ { "CANADIAN SYLLABICS NGAI",NULL},
/* 1671 */ { "CANADIAN SYLLABICS NNGI",NULL},
/* 1672 */ { "CANADIAN SYLLABICS NNGII",NULL},
/* 1673 */ { "CANADIAN SYLLABICS NNGO","	* Inuktitut (NNGU)"},
/* 1674 */ { "CANADIAN SYLLABICS NNGOO","	* Inuktitut (NNGUU)"},
/* 1675 */ { "CANADIAN SYLLABICS NNGA",NULL},
/* 1676 */ { "CANADIAN SYLLABICS NNGAA",NULL},
/* 1677 */ { "CANADIAN SYLLABICS WOODS-CREE THWEE",NULL},
/* 1678 */ { "CANADIAN SYLLABICS WOODS-CREE THWI",NULL},
/* 1679 */ { "CANADIAN SYLLABICS WOODS-CREE THWII",NULL},
/* 167A */ { "CANADIAN SYLLABICS WOODS-CREE THWO",NULL},
/* 167B */ { "CANADIAN SYLLABICS WOODS-CREE THWOO",NULL},
/* 167C */ { "CANADIAN SYLLABICS WOODS-CREE THWA",NULL},
/* 167D */ { "CANADIAN SYLLABICS WOODS-CREE THWAA",NULL},
/* 167E */ { "CANADIAN SYLLABICS WOODS-CREE FINAL TH",NULL},
/* 167F */ { "CANADIAN SYLLABICS BLACKFOOT W",NULL},
/* 1680 */ { "OGHAM SPACE MARK","	* glyph is blank in \"stemless\" style fonts\n"
	"	x (space - 0020)"},
/* 1681 */ { "OGHAM LETTER BEITH",NULL},
/* 1682 */ { "OGHAM LETTER LUIS",NULL},
/* 1683 */ { "OGHAM LETTER FEARN",NULL},
/* 1684 */ { "OGHAM LETTER SAIL",NULL},
/* 1685 */ { "OGHAM LETTER NION",NULL},
/* 1686 */ { "OGHAM LETTER UATH",NULL},
/* 1687 */ { "OGHAM LETTER DAIR",NULL},
/* 1688 */ { "OGHAM LETTER TINNE",NULL},
/* 1689 */ { "OGHAM LETTER COLL",NULL},
/* 168A */ { "OGHAM LETTER CEIRT",NULL},
/* 168B */ { "OGHAM LETTER MUIN",NULL},
/* 168C */ { "OGHAM LETTER GORT",NULL},
/* 168D */ { "OGHAM LETTER NGEADAL",NULL},
/* 168E */ { "OGHAM LETTER STRAIF",NULL},
/* 168F */ { "OGHAM LETTER RUIS",NULL},
/* 1690 */ { "OGHAM LETTER AILM",NULL},
/* 1691 */ { "OGHAM LETTER ONN",NULL},
/* 1692 */ { "OGHAM LETTER UR",NULL},
/* 1693 */ { "OGHAM LETTER EADHADH",NULL},
/* 1694 */ { "OGHAM LETTER IODHADH",NULL},
/* 1695 */ { "OGHAM LETTER EABHADH",NULL},
/* 1696 */ { "OGHAM LETTER OR",NULL},
/* 1697 */ { "OGHAM LETTER UILLEANN",NULL},
/* 1698 */ { "OGHAM LETTER IFIN",NULL},
/* 1699 */ { "OGHAM LETTER EAMHANCHOLL",NULL},
/* 169A */ { "OGHAM LETTER PEITH",NULL},
/* 169B */ { "OGHAM FEATHER MARK","	* marks beginning of Ogham text"},
/* 169C */ { "OGHAM REVERSED FEATHER MARK","	* marks end of Ogham text"},
/* 169D */ { NULL,NULL},
/* 169E */ { NULL,NULL},
/* 169F */ { NULL,NULL},
/* 16A0 */ { "RUNIC LETTER FEHU FEOH FE F",NULL},
/* 16A1 */ { "RUNIC LETTER V",NULL},
/* 16A2 */ { "RUNIC LETTER URUZ UR U",NULL},
/* 16A3 */ { "RUNIC LETTER YR",NULL},
/* 16A4 */ { "RUNIC LETTER Y",NULL},
/* 16A5 */ { "RUNIC LETTER W",NULL},
/* 16A6 */ { "RUNIC LETTER THURISAZ THURS THORN","	x (latin small letter thorn - 00FE)"},
/* 16A7 */ { "RUNIC LETTER ETH",NULL},
/* 16A8 */ { "RUNIC LETTER ANSUZ A",NULL},
/* 16A9 */ { "RUNIC LETTER OS O",NULL},
/* 16AA */ { "RUNIC LETTER AC A",NULL},
/* 16AB */ { "RUNIC LETTER AESC",NULL},
/* 16AC */ { "RUNIC LETTER LONG-BRANCH-OSS O",NULL},
/* 16AD */ { "RUNIC LETTER SHORT-TWIG-OSS O",NULL},
/* 16AE */ { "RUNIC LETTER O",NULL},
/* 16AF */ { "RUNIC LETTER OE",NULL},
/* 16B0 */ { "RUNIC LETTER ON",NULL},
/* 16B1 */ { "RUNIC LETTER RAIDO RAD REID R",NULL},
/* 16B2 */ { "RUNIC LETTER KAUNA",NULL},
/* 16B3 */ { "RUNIC LETTER CEN",NULL},
/* 16B4 */ { "RUNIC LETTER KAUN K",NULL},
/* 16B5 */ { "RUNIC LETTER G",NULL},
/* 16B6 */ { "RUNIC LETTER ENG",NULL},
/* 16B7 */ { "RUNIC LETTER GEBO GYFU G",NULL},
/* 16B8 */ { "RUNIC LETTER GAR",NULL},
/* 16B9 */ { "RUNIC LETTER WUNJO WYNN W","	x (latin letter wynn - 01BF)"},
/* 16BA */ { "RUNIC LETTER HAGLAZ H",NULL},
/* 16BB */ { "RUNIC LETTER HAEGL H",NULL},
/* 16BC */ { "RUNIC LETTER LONG-BRANCH-HAGALL H",NULL},
/* 16BD */ { "RUNIC LETTER SHORT-TWIG-HAGALL H",NULL},
/* 16BE */ { "RUNIC LETTER NAUDIZ NYD NAUD N",NULL},
/* 16BF */ { "RUNIC LETTER SHORT-TWIG-NAUD N",NULL},
/* 16C0 */ { "RUNIC LETTER DOTTED-N",NULL},
/* 16C1 */ { "RUNIC LETTER ISAZ IS ISS I",NULL},
/* 16C2 */ { "RUNIC LETTER E",NULL},
/* 16C3 */ { "RUNIC LETTER JERAN J",NULL},
/* 16C4 */ { "RUNIC LETTER GER",NULL},
/* 16C5 */ { "RUNIC LETTER LONG-BRANCH-AR AE",NULL},
/* 16C6 */ { "RUNIC LETTER SHORT-TWIG-AR A",NULL},
/* 16C7 */ { "RUNIC LETTER IWAZ EOH",NULL},
/* 16C8 */ { "RUNIC LETTER PERTHO PEORTH P",NULL},
/* 16C9 */ { "RUNIC LETTER ALGIZ EOLHX",NULL},
/* 16CA */ { "RUNIC LETTER SOWILO S",NULL},
/* 16CB */ { "RUNIC LETTER SIGEL LONG-BRANCH-SOL S",NULL},
/* 16CC */ { "RUNIC LETTER SHORT-TWIG-SOL S",NULL},
/* 16CD */ { "RUNIC LETTER C",NULL},
/* 16CE */ { "RUNIC LETTER Z",NULL},
/* 16CF */ { "RUNIC LETTER TIWAZ TIR TYR T",NULL},
/* 16D0 */ { "RUNIC LETTER SHORT-TWIG-TYR T",NULL},
/* 16D1 */ { "RUNIC LETTER D",NULL},
/* 16D2 */ { "RUNIC LETTER BERKANAN BEORC BJARKAN B",NULL},
/* 16D3 */ { "RUNIC LETTER SHORT-TWIG-BJARKAN B",NULL},
/* 16D4 */ { "RUNIC LETTER DOTTED-P",NULL},
/* 16D5 */ { "RUNIC LETTER OPEN-P",NULL},
/* 16D6 */ { "RUNIC LETTER EHWAZ EH E",NULL},
/* 16D7 */ { "RUNIC LETTER MANNAZ MAN M",NULL},
/* 16D8 */ { "RUNIC LETTER LONG-BRANCH-MADR M",NULL},
/* 16D9 */ { "RUNIC LETTER SHORT-TWIG-MADR M",NULL},
/* 16DA */ { "RUNIC LETTER LAUKAZ LAGU LOGR L",NULL},
/* 16DB */ { "RUNIC LETTER DOTTED-L",NULL},
/* 16DC */ { "RUNIC LETTER INGWAZ",NULL},
/* 16DD */ { "RUNIC LETTER ING",NULL},
/* 16DE */ { "RUNIC LETTER DAGAZ DAEG D",NULL},
/* 16DF */ { "RUNIC LETTER OTHALAN ETHEL O",NULL},
/* 16E0 */ { "RUNIC LETTER EAR",NULL},
/* 16E1 */ { "RUNIC LETTER IOR",NULL},
/* 16E2 */ { "RUNIC LETTER CWEORTH",NULL},
/* 16E3 */ { "RUNIC LETTER CALC",NULL},
/* 16E4 */ { "RUNIC LETTER CEALC",NULL},
/* 16E5 */ { "RUNIC LETTER STAN",NULL},
/* 16E6 */ { "RUNIC LETTER LONG-BRANCH-YR",NULL},
/* 16E7 */ { "RUNIC LETTER SHORT-TWIG-YR",NULL},
/* 16E8 */ { "RUNIC LETTER ICELANDIC-YR",NULL},
/* 16E9 */ { "RUNIC LETTER Q",NULL},
/* 16EA */ { "RUNIC LETTER X",NULL},
/* 16EB */ { "RUNIC SINGLE PUNCTUATION",NULL},
/* 16EC */ { "RUNIC MULTIPLE PUNCTUATION",NULL},
/* 16ED */ { "RUNIC CROSS PUNCTUATION",NULL},
/* 16EE */ { "RUNIC ARLAUG SYMBOL","	* golden number 17"},
/* 16EF */ { "RUNIC TVIMADUR SYMBOL","	* golden number 18"},
/* 16F0 */ { "RUNIC BELGTHOR SYMBOL","	* golden number 19"},
/* 16F1 */ { "RUNIC LETTER K",NULL},
/* 16F2 */ { "RUNIC LETTER SH",NULL},
/* 16F3 */ { "RUNIC LETTER OO",NULL},
/* 16F4 */ { "RUNIC LETTER FRANKS CASKET OS",NULL},
/* 16F5 */ { "RUNIC LETTER FRANKS CASKET IS",NULL},
/* 16F6 */ { "RUNIC LETTER FRANKS CASKET EH",NULL},
/* 16F7 */ { "RUNIC LETTER FRANKS CASKET AC",NULL},
/* 16F8 */ { "RUNIC LETTER FRANKS CASKET AESC",NULL},
/* 16F9 */ { NULL,NULL},
/* 16FA */ { NULL,NULL},
/* 16FB */ { NULL,NULL},
/* 16FC */ { NULL,NULL},
/* 16FD */ { NULL,NULL},
/* 16FE */ { NULL,NULL},
/* 16FF */ { NULL,NULL}
};

UN_DLL_LOCAL
static const struct unicode_nameannot una_00_17[] = {
/* 1700 */ { "TAGALOG LETTER A",NULL},
/* 1701 */ { "TAGALOG LETTER I",NULL},
/* 1702 */ { "TAGALOG LETTER U",NULL},
/* 1703 */ { "TAGALOG LETTER KA",NULL},
/* 1704 */ { "TAGALOG LETTER GA",NULL},
/* 1705 */ { "TAGALOG LETTER NGA",NULL},
/* 1706 */ { "TAGALOG LETTER TA",NULL},
/* 1707 */ { "TAGALOG LETTER DA",NULL},
/* 1708 */ { "TAGALOG LETTER NA",NULL},
/* 1709 */ { "TAGALOG LETTER PA",NULL},
/* 170A */ { "TAGALOG LETTER BA",NULL},
/* 170B */ { "TAGALOG LETTER MA",NULL},
/* 170C */ { "TAGALOG LETTER YA",NULL},
/* 170D */ { "TAGALOG LETTER RA","	* modern adaptation, not found in Old Tagalog"},
/* 170E */ { "TAGALOG LETTER LA",NULL},
/* 170F */ { "TAGALOG LETTER WA",NULL},
/* 1710 */ { "TAGALOG LETTER SA",NULL},
/* 1711 */ { "TAGALOG LETTER HA",NULL},
/* 1712 */ { "TAGALOG VOWEL SIGN I",NULL},
/* 1713 */ { "TAGALOG VOWEL SIGN U",NULL},
/* 1714 */ { "TAGALOG SIGN VIRAMA",NULL},
/* 1715 */ { "TAGALOG SIGN PAMUDPOD","	x (hanunoo sign pamudpod - 1734)"},
/* 1716 */ { NULL,NULL},
/* 1717 */ { NULL,NULL},
/* 1718 */ { NULL,NULL},
/* 1719 */ { NULL,NULL},
/* 171A */ { NULL,NULL},
/* 171B */ { NULL,NULL},
/* 171C */ { NULL,NULL},
/* 171D */ { NULL,NULL},
/* 171E */ { NULL,NULL},
/* 171F */ { "TAGALOG LETTER ARCHAIC RA","	= Zambales ra\n"
	"	* modern adaptation, not found in Old Tagalog"},
/* 1720 */ { "HANUNOO LETTER A",NULL},
/* 1721 */ { "HANUNOO LETTER I",NULL},
/* 1722 */ { "HANUNOO LETTER U",NULL},
/* 1723 */ { "HANUNOO LETTER KA",NULL},
/* 1724 */ { "HANUNOO LETTER GA",NULL},
/* 1725 */ { "HANUNOO LETTER NGA",NULL},
/* 1726 */ { "HANUNOO LETTER TA",NULL},
/* 1727 */ { "HANUNOO LETTER DA",NULL},
/* 1728 */ { "HANUNOO LETTER NA",NULL},
/* 1729 */ { "HANUNOO LETTER PA",NULL},
/* 172A */ { "HANUNOO LETTER BA",NULL},
/* 172B */ { "HANUNOO LETTER MA",NULL},
/* 172C */ { "HANUNOO LETTER YA",NULL},
/* 172D */ { "HANUNOO LETTER RA",NULL},
/* 172E */ { "HANUNOO LETTER LA",NULL},
/* 172F */ { "HANUNOO LETTER WA",NULL},
/* 1730 */ { "HANUNOO LETTER SA",NULL},
/* 1731 */ { "HANUNOO LETTER HA",NULL},
/* 1732 */ { "HANUNOO VOWEL SIGN I",NULL},
/* 1733 */ { "HANUNOO VOWEL SIGN U",NULL},
/* 1734 */ { "HANUNOO SIGN PAMUDPOD","	x (tagalog sign pamudpod - 1715)"},
/* 1735 */ { "PHILIPPINE SINGLE PUNCTUATION","	x (devanagari danda - 0964)"},
/* 1736 */ { "PHILIPPINE DOUBLE PUNCTUATION","	x (devanagari double danda - 0965)"},
/* 1737 */ { NULL,NULL},
/* 1738 */ { NULL,NULL},
/* 1739 */ { NULL,NULL},
/* 173A */ { NULL,NULL},
/* 173B */ { NULL,NULL},
/* 173C */ { NULL,NULL},
/* 173D */ { NULL,NULL},
/* 173E */ { NULL,NULL},
/* 173F */ { NULL,NULL},
/* 1740 */ { "BUHID LETTER A",NULL},
/* 1741 */ { "BUHID LETTER I",NULL},
/* 1742 */ { "BUHID LETTER U",NULL},
/* 1743 */ { "BUHID LETTER KA",NULL},
/* 1744 */ { "BUHID LETTER GA",NULL},
/* 1745 */ { "BUHID LETTER NGA",NULL},
/* 1746 */ { "BUHID LETTER TA",NULL},
/* 1747 */ { "BUHID LETTER DA",NULL},
/* 1748 */ { "BUHID LETTER NA",NULL},
/* 1749 */ { "BUHID LETTER PA",NULL},
/* 174A */ { "BUHID LETTER BA",NULL},
/* 174B */ { "BUHID LETTER MA",NULL},
/* 174C */ { "BUHID LETTER YA",NULL},
/* 174D */ { "BUHID LETTER RA",NULL},
/* 174E */ { "BUHID LETTER LA",NULL},
/* 174F */ { "BUHID LETTER WA",NULL},
/* 1750 */ { "BUHID LETTER SA",NULL},
/* 1751 */ { "BUHID LETTER HA",NULL},
/* 1752 */ { "BUHID VOWEL SIGN I",NULL},
/* 1753 */ { "BUHID VOWEL SIGN U",NULL},
/* 1754 */ { NULL,NULL},
/* 1755 */ { NULL,NULL},
/* 1756 */ { NULL,NULL},
/* 1757 */ { NULL,NULL},
/* 1758 */ { NULL,NULL},
/* 1759 */ { NULL,NULL},
/* 175A */ { NULL,NULL},
/* 175B */ { NULL,NULL},
/* 175C */ { NULL,NULL},
/* 175D */ { NULL,NULL},
/* 175E */ { NULL,NULL},
/* 175F */ { NULL,NULL},
/* 1760 */ { "TAGBANWA LETTER A",NULL},
/* 1761 */ { "TAGBANWA LETTER I",NULL},
/* 1762 */ { "TAGBANWA LETTER U",NULL},
/* 1763 */ { "TAGBANWA LETTER KA",NULL},
/* 1764 */ { "TAGBANWA LETTER GA",NULL},
/* 1765 */ { "TAGBANWA LETTER NGA",NULL},
/* 1766 */ { "TAGBANWA LETTER TA",NULL},
/* 1767 */ { "TAGBANWA LETTER DA",NULL},
/* 1768 */ { "TAGBANWA LETTER NA",NULL},
/* 1769 */ { "TAGBANWA LETTER PA",NULL},
/* 176A */ { "TAGBANWA LETTER BA",NULL},
/* 176B */ { "TAGBANWA LETTER MA",NULL},
/* 176C */ { "TAGBANWA LETTER YA",NULL},
/* 176D */ { NULL,NULL},
/* 176E */ { "TAGBANWA LETTER LA",NULL},
/* 176F */ { "TAGBANWA LETTER WA",NULL},
/* 1770 */ { "TAGBANWA LETTER SA",NULL},
/* 1771 */ { NULL,NULL},
/* 1772 */ { "TAGBANWA VOWEL SIGN I",NULL},
/* 1773 */ { "TAGBANWA VOWEL SIGN U",NULL},
/* 1774 */ { NULL,NULL},
/* 1775 */ { NULL,NULL},
/* 1776 */ { NULL,NULL},
/* 1777 */ { NULL,NULL},
/* 1778 */ { NULL,NULL},
/* 1779 */ { NULL,NULL},
/* 177A */ { NULL,NULL},
/* 177B */ { NULL,NULL},
/* 177C */ { NULL,NULL},
/* 177D */ { NULL,NULL},
/* 177E */ { NULL,NULL},
/* 177F */ { NULL,NULL},
/* 1780 */ { "KHMER LETTER KA",NULL},
/* 1781 */ { "KHMER LETTER KHA",NULL},
/* 1782 */ { "KHMER LETTER KO",NULL},
/* 1783 */ { "KHMER LETTER KHO",NULL},
/* 1784 */ { "KHMER LETTER NGO",NULL},
/* 1785 */ { "KHMER LETTER CA",NULL},
/* 1786 */ { "KHMER LETTER CHA",NULL},
/* 1787 */ { "KHMER LETTER CO",NULL},
/* 1788 */ { "KHMER LETTER CHO",NULL},
/* 1789 */ { "KHMER LETTER NYO",NULL},
/* 178A */ { "KHMER LETTER DA",NULL},
/* 178B */ { "KHMER LETTER TTHA",NULL},
/* 178C */ { "KHMER LETTER DO",NULL},
/* 178D */ { "KHMER LETTER TTHO",NULL},
/* 178E */ { "KHMER LETTER NNO","	* as this character belongs to the first register, its correct transliteration is nna, not nno"},
/* 178F */ { "KHMER LETTER TA",NULL},
/* 1790 */ { "KHMER LETTER THA",NULL},
/* 1791 */ { "KHMER LETTER TO",NULL},
/* 1792 */ { "KHMER LETTER THO",NULL},
/* 1793 */ { "KHMER LETTER NO",NULL},
/* 1794 */ { "KHMER LETTER BA",NULL},
/* 1795 */ { "KHMER LETTER PHA",NULL},
/* 1796 */ { "KHMER LETTER PO",NULL},
/* 1797 */ { "KHMER LETTER PHO",NULL},
/* 1798 */ { "KHMER LETTER MO",NULL},
/* 1799 */ { "KHMER LETTER YO",NULL},
/* 179A */ { "KHMER LETTER RO",NULL},
/* 179B */ { "KHMER LETTER LO",NULL},
/* 179C */ { "KHMER LETTER VO",NULL},
/* 179D */ { "KHMER LETTER SHA","	* used only for Pali/Sanskrit transliteration"},
/* 179E */ { "KHMER LETTER SSO","	* used only for Pali/Sanskrit transliteration\n"
	"	* as this character belongs to the first register, its correct transliteration is ssa, not sso"},
/* 179F */ { "KHMER LETTER SA",NULL},
/* 17A0 */ { "KHMER LETTER HA",NULL},
/* 17A1 */ { "KHMER LETTER LA",NULL},
/* 17A2 */ { "KHMER LETTER QA","	* glottal stop"},
/* 17A3 */ { "KHMER INDEPENDENT VOWEL QAQ","	* use of this character is strongly discouraged; 17A2 should be used instead"},
/* 17A4 */ { "KHMER INDEPENDENT VOWEL QAA","	* use of this character is strongly discouraged; the sequence 17A2 17B6 should be used instead"},
/* 17A5 */ { "KHMER INDEPENDENT VOWEL QI",NULL},
/* 17A6 */ { "KHMER INDEPENDENT VOWEL QII",NULL},
/* 17A7 */ { "KHMER INDEPENDENT VOWEL QU",NULL},
/* 17A8 */ { "KHMER INDEPENDENT VOWEL QUK","	* obsolete ligature for the sequence 17A7 1780\n"
	"	* use of the sequence is now preferred"},
/* 17A9 */ { "KHMER INDEPENDENT VOWEL QUU",NULL},
/* 17AA */ { "KHMER INDEPENDENT VOWEL QUUV",NULL},
/* 17AB */ { "KHMER INDEPENDENT VOWEL RY",NULL},
/* 17AC */ { "KHMER INDEPENDENT VOWEL RYY",NULL},
/* 17AD */ { "KHMER INDEPENDENT VOWEL LY",NULL},
/* 17AE */ { "KHMER INDEPENDENT VOWEL LYY",NULL},
/* 17AF */ { "KHMER INDEPENDENT VOWEL QE",NULL},
/* 17B0 */ { "KHMER INDEPENDENT VOWEL QAI",NULL},
/* 17B1 */ { "KHMER INDEPENDENT VOWEL QOO TYPE ONE",NULL},
/* 17B2 */ { "KHMER INDEPENDENT VOWEL QOO TYPE TWO","	* this is a variant for 17B1, used in only two words\n"
	"	* 17B1 is the normal variant of this vowel"},
/* 17B3 */ { "KHMER INDEPENDENT VOWEL QAU",NULL},
/* 17B4 */ { "KHMER VOWEL INHERENT AQ",NULL},
/* 17B5 */ { "KHMER VOWEL INHERENT AA",NULL},
/* 17B6 */ { "KHMER VOWEL SIGN AA",NULL},
/* 17B7 */ { "KHMER VOWEL SIGN I",NULL},
/* 17B8 */ { "KHMER VOWEL SIGN II",NULL},
/* 17B9 */ { "KHMER VOWEL SIGN Y",NULL},
/* 17BA */ { "KHMER VOWEL SIGN YY",NULL},
/* 17BB */ { "KHMER VOWEL SIGN U",NULL},
/* 17BC */ { "KHMER VOWEL SIGN UU",NULL},
/* 17BD */ { "KHMER VOWEL SIGN UA",NULL},
/* 17BE */ { "KHMER VOWEL SIGN OE",NULL},
/* 17BF */ { "KHMER VOWEL SIGN YA",NULL},
/* 17C0 */ { "KHMER VOWEL SIGN IE",NULL},
/* 17C1 */ { "KHMER VOWEL SIGN E",NULL},
/* 17C2 */ { "KHMER VOWEL SIGN AE",NULL},
/* 17C3 */ { "KHMER VOWEL SIGN AI",NULL},
/* 17C4 */ { "KHMER VOWEL SIGN OO",NULL},
/* 17C5 */ { "KHMER VOWEL SIGN AU",NULL},
/* 17C6 */ { "KHMER SIGN NIKAHIT","	= srak am\n"
	"	= anusvara\n"
	"	* final nasalization\n"
	"	* this character is usually regarded as a vowel sign am, along with om and aam\n"
	"	x (thai character nikhahit - 0E4D)\n"
	"	x (myanmar sign anusvara - 1036)"},
/* 17C7 */ { "KHMER SIGN REAHMUK","	= srak ah\n"
	"	= visarga\n"
	"	x (myanmar sign visarga - 1038)"},
/* 17C8 */ { "KHMER SIGN YUUKALEAPINTU","	* inserts a short inherent vowel with abrupt glottal stop\n"
	"	* the preferred transliteration is yukaleakpintu"},
/* 17C9 */ { "KHMER SIGN MUUSIKATOAN","	* changes the second register to the first\n"
	"	* the preferred transliteration is muusekatoan"},
/* 17CA */ { "KHMER SIGN TRIISAP","	* changes the first register to the second\n"
	"	* the preferred transliteration is treisap"},
/* 17CB */ { "KHMER SIGN BANTOC","	* shortens the vowel sound in the previous orthographic syllable\n"
	"	* the preferred transliteration is bantak"},
/* 17CC */ { "KHMER SIGN ROBAT","	* a diacritic historically corresponding to the repha form of ra in Devanagari"},
/* 17CD */ { "KHMER SIGN TOANDAKHIAT","	* indicates that the base character is not pronounced\n"
	"	x (thai character thanthakhat - 0E4C)"},
/* 17CE */ { "KHMER SIGN KAKABAT","	* sign used with some exclamations"},
/* 17CF */ { "KHMER SIGN AHSDA","	* denotes stressed intonation in some single-consonant words"},
/* 17D0 */ { "KHMER SIGN SAMYOK SANNYA","	* denotes deviation from the general rules of pronunciation, mostly used in loan words from Pali/Sanskrit, French, and so on"},
/* 17D1 */ { "KHMER SIGN VIRIAM","	* mostly obsolete, a \"killer\"\n"
	"	* indicates that the base character is the final consonant of a word without its inherent vowel sound"},
/* 17D2 */ { "KHMER SIGN COENG","	* functions to indicate that the following Khmer letter is to be rendered subscripted\n"
	"	* shape shown is arbitrary and is not visibly rendered"},
/* 17D3 */ { "KHMER SIGN BATHAMASAT","	* originally intended as part of lunar date symbols\n"
	"	* use of this character is discouraged in favor of the complete set of lunar date symbols\n"
	"	x (khmer symbol pathamasat - 19E0)"},
/* 17D4 */ { "KHMER SIGN KHAN","	* functions as a full stop, period\n"
	"	x (thai character paiyannoi - 0E2F)\n"
	"	x (myanmar sign little section - 104A)"},
/* 17D5 */ { "KHMER SIGN BARIYOOSAN","	* indicates the end of a section or a text\n"
	"	x (thai character angkhankhu - 0E5A)\n"
	"	x (myanmar sign section - 104B)"},
/* 17D6 */ { "KHMER SIGN CAMNUC PII KUUH","	* functions as colon\n"
	"	* the preferred transliteration is camnoc pii kuuh\n"
	"	x (division sign - 00F7)\n"
	"	x (tibetan mark gter tsheg - 0F14)"},
/* 17D7 */ { "KHMER SIGN LEK TOO","	* repetition sign\n"
	"	x (thai character maiyamok - 0E46)"},
/* 17D8 */ { "KHMER SIGN BEYYAL","	* et cetera\n"
	"	* use of this character is discouraged; other abbreviations for et cetera also exist\n"
	"	* preferred spelling: 17D4 179B 17D4"},
/* 17D9 */ { "KHMER SIGN PHNAEK MUAN","	* indicates the beginning of a book or a treatise\n"
	"	* the preferred transliteration is phnek moan\n"
	"	x (thai character fongman - 0E4F)"},
/* 17DA */ { "KHMER SIGN KOOMUUT","	* indicates the end of a book or treatise\n"
	"	* this forms a pair with 17D9\n"
	"	* the preferred transliteration is koomoot\n"
	"	x (thai character khomut - 0E5B)"},
/* 17DB */ { "KHMER CURRENCY SYMBOL RIEL",NULL},
/* 17DC */ { "KHMER SIGN AVAKRAHASANYA","	* rare, shows an omitted Sanskrit vowel, like an apostrophe\n"
	"	* the preferred transliteration is avakraha sannya\n"
	"	x (devanagari sign avagraha - 093D)"},
/* 17DD */ { "KHMER SIGN ATTHACAN","	* mostly obsolete\n"
	"	* indicates that the base character is the final consonant of a word with its inherent vowel sound\n"
	"	x (khmer sign viriam - 17D1)"},
/* 17DE */ { NULL,NULL},
/* 17DF */ { NULL,NULL},
/* 17E0 */ { "KHMER DIGIT ZERO",NULL},
/* 17E1 */ { "KHMER DIGIT ONE",NULL},
/* 17E2 */ { "KHMER DIGIT TWO",NULL},
/* 17E3 */ { "KHMER DIGIT THREE",NULL},
/* 17E4 */ { "KHMER DIGIT FOUR",NULL},
/* 17E5 */ { "KHMER DIGIT FIVE",NULL},
/* 17E6 */ { "KHMER DIGIT SIX",NULL},
/* 17E7 */ { "KHMER DIGIT SEVEN",NULL},
/* 17E8 */ { "KHMER DIGIT EIGHT",NULL},
/* 17E9 */ { "KHMER DIGIT NINE",NULL},
/* 17EA */ { NULL,NULL},
/* 17EB */ { NULL,NULL},
/* 17EC */ { NULL,NULL},
/* 17ED */ { NULL,NULL},
/* 17EE */ { NULL,NULL},
/* 17EF */ { NULL,NULL},
/* 17F0 */ { "KHMER SYMBOL LEK ATTAK SON",NULL},
/* 17F1 */ { "KHMER SYMBOL LEK ATTAK MUOY",NULL},
/* 17F2 */ { "KHMER SYMBOL LEK ATTAK PII",NULL},
/* 17F3 */ { "KHMER SYMBOL LEK ATTAK BEI",NULL},
/* 17F4 */ { "KHMER SYMBOL LEK ATTAK BUON",NULL},
/* 17F5 */ { "KHMER SYMBOL LEK ATTAK PRAM",NULL},
/* 17F6 */ { "KHMER SYMBOL LEK ATTAK PRAM-MUOY",NULL},
/* 17F7 */ { "KHMER SYMBOL LEK ATTAK PRAM-PII",NULL},
/* 17F8 */ { "KHMER SYMBOL LEK ATTAK PRAM-BEI",NULL},
/* 17F9 */ { "KHMER SYMBOL LEK ATTAK PRAM-BUON",NULL},
/* 17FA */ { NULL,NULL},
/* 17FB */ { NULL,NULL},
/* 17FC */ { NULL,NULL},
/* 17FD */ { NULL,NULL},
/* 17FE */ { NULL,NULL},
/* 17FF */ { NULL,NULL}
};

UN_DLL_LOCAL
static const struct unicode_nameannot una_00_18[] = {
/* 1800 */ { "MONGOLIAN BIRGA","	x (tibetan mark initial yig mgo mdun ma - 0F04)"},
/* 1801 */ { "MONGOLIAN ELLIPSIS",NULL},
/* 1802 */ { "MONGOLIAN COMMA",NULL},
/* 1803 */ { "MONGOLIAN FULL STOP",NULL},
/* 1804 */ { "MONGOLIAN COLON",NULL},
/* 1805 */ { "MONGOLIAN FOUR DOTS","	* marks the end of a chapter"},
/* 1806 */ { "MONGOLIAN TODO SOFT HYPHEN","	* not a format control character, but simply a hyphen for Todo\n"
	"	* also used in the Hudum (modern Mongolian) writing system\n"
	"	x (soft hyphen - 00AD)"},
/* 1807 */ { "MONGOLIAN SIBE SYLLABLE BOUNDARY MARKER",NULL},
/* 1808 */ { "MONGOLIAN MANCHU COMMA",NULL},
/* 1809 */ { "MONGOLIAN MANCHU FULL STOP",NULL},
/* 180A */ { "MONGOLIAN NIRUGU",NULL},
/* 180B */ { "MONGOLIAN FREE VARIATION SELECTOR ONE","	* abbreviated FVS1"},
/* 180C */ { "MONGOLIAN FREE VARIATION SELECTOR TWO","	* abbreviated FVS2"},
/* 180D */ { "MONGOLIAN FREE VARIATION SELECTOR THREE","	* abbreviated FVS3"},
/* 180E */ { "MONGOLIAN VOWEL SEPARATOR","	* abbreviated MVS"},
/* 180F */ { "MONGOLIAN FREE VARIATION SELECTOR FOUR","	* abbreviated FVS4"},
/* 1810 */ { "MONGOLIAN DIGIT ZERO",NULL},
/* 1811 */ { "MONGOLIAN DIGIT ONE",NULL},
/* 1812 */ { "MONGOLIAN DIGIT TWO",NULL},
/* 1813 */ { "MONGOLIAN DIGIT THREE",NULL},
/* 1814 */ { "MONGOLIAN DIGIT FOUR",NULL},
/* 1815 */ { "MONGOLIAN DIGIT FIVE",NULL},
/* 1816 */ { "MONGOLIAN DIGIT SIX",NULL},
/* 1817 */ { "MONGOLIAN DIGIT SEVEN",NULL},
/* 1818 */ { "MONGOLIAN DIGIT EIGHT",NULL},
/* 1819 */ { "MONGOLIAN DIGIT NINE",NULL},
/* 181A */ { NULL,NULL},
/* 181B */ { NULL,NULL},
/* 181C */ { NULL,NULL},
/* 181D */ { NULL,NULL},
/* 181E */ { NULL,NULL},
/* 181F */ { NULL,NULL},
/* 1820 */ { "MONGOLIAN LETTER A","	x (cyrillic small letter a - 0430)"},
/* 1821 */ { "MONGOLIAN LETTER E","	x (cyrillic small letter e - 044D)"},
/* 1822 */ { "MONGOLIAN LETTER I","	x (cyrillic small letter i - 0438)"},
/* 1823 */ { "MONGOLIAN LETTER O","	x (cyrillic small letter o - 043E)"},
/* 1824 */ { "MONGOLIAN LETTER U","	x (cyrillic small letter u - 0443)"},
/* 1825 */ { "MONGOLIAN LETTER OE","	x (cyrillic small letter barred o - 04E9)"},
/* 1826 */ { "MONGOLIAN LETTER UE","	x (cyrillic small letter straight u - 04AF)"},
/* 1827 */ { "MONGOLIAN LETTER EE",NULL},
/* 1828 */ { "MONGOLIAN LETTER NA","	x (cyrillic small letter en - 043D)"},
/* 1829 */ { "MONGOLIAN LETTER ANG",NULL},
/* 182A */ { "MONGOLIAN LETTER BA","	x (cyrillic small letter be - 0431)"},
/* 182B */ { "MONGOLIAN LETTER PA","	x (cyrillic small letter pe - 043F)"},
/* 182C */ { "MONGOLIAN LETTER QA","	x (cyrillic small letter ha - 0445)"},
/* 182D */ { "MONGOLIAN LETTER GA","	x (cyrillic small letter ghe - 0433)"},
/* 182E */ { "MONGOLIAN LETTER MA","	x (cyrillic small letter em - 043C)"},
/* 182F */ { "MONGOLIAN LETTER LA","	x (cyrillic small letter el - 043B)"},
/* 1830 */ { "MONGOLIAN LETTER SA","	x (cyrillic small letter es - 0441)"},
/* 1831 */ { "MONGOLIAN LETTER SHA","	x (cyrillic small letter sha - 0448)"},
/* 1832 */ { "MONGOLIAN LETTER TA","	x (cyrillic small letter te - 0442)"},
/* 1833 */ { "MONGOLIAN LETTER DA","	x (cyrillic small letter de - 0434)"},
/* 1834 */ { "MONGOLIAN LETTER CHA","	x (cyrillic small letter che - 0447)"},
/* 1835 */ { "MONGOLIAN LETTER JA","	x (cyrillic small letter zhe - 0436)"},
/* 1836 */ { "MONGOLIAN LETTER YA","	x (cyrillic small letter short i - 0439)"},
/* 1837 */ { "MONGOLIAN LETTER RA","	x (cyrillic small letter er - 0440)"},
/* 1838 */ { "MONGOLIAN LETTER WA","	x (cyrillic small letter ve - 0432)"},
/* 1839 */ { "MONGOLIAN LETTER FA","	x (cyrillic small letter ef - 0444)"},
/* 183A */ { "MONGOLIAN LETTER KA","	x (cyrillic small letter ha - 0445)"},
/* 183B */ { "MONGOLIAN LETTER KHA","	x (cyrillic small letter ka - 043A)"},
/* 183C */ { "MONGOLIAN LETTER TSA","	x (cyrillic small letter tse - 0446)"},
/* 183D */ { "MONGOLIAN LETTER ZA","	x (cyrillic small letter ze - 0437)"},
/* 183E */ { "MONGOLIAN LETTER HAA","	x (cyrillic small letter ha - 0445)"},
/* 183F */ { "MONGOLIAN LETTER ZRA","	x (cyrillic small letter zhe - 0436)"},
/* 1840 */ { "MONGOLIAN LETTER LHA",NULL},
/* 1841 */ { "MONGOLIAN LETTER ZHI",NULL},
/* 1842 */ { "MONGOLIAN LETTER CHI",NULL},
/* 1843 */ { "MONGOLIAN LETTER TODO LONG VOWEL SIGN",NULL},
/* 1844 */ { "MONGOLIAN LETTER TODO E",NULL},
/* 1845 */ { "MONGOLIAN LETTER TODO I",NULL},
/* 1846 */ { "MONGOLIAN LETTER TODO O",NULL},
/* 1847 */ { "MONGOLIAN LETTER TODO U",NULL},
/* 1848 */ { "MONGOLIAN LETTER TODO OE",NULL},
/* 1849 */ { "MONGOLIAN LETTER TODO UE",NULL},
/* 184A */ { "MONGOLIAN LETTER TODO ANG",NULL},
/* 184B */ { "MONGOLIAN LETTER TODO BA",NULL},
/* 184C */ { "MONGOLIAN LETTER TODO PA",NULL},
/* 184D */ { "MONGOLIAN LETTER TODO QA",NULL},
/* 184E */ { "MONGOLIAN LETTER TODO GA",NULL},
/* 184F */ { "MONGOLIAN LETTER TODO MA",NULL},
/* 1850 */ { "MONGOLIAN LETTER TODO TA",NULL},
/* 1851 */ { "MONGOLIAN LETTER TODO DA",NULL},
/* 1852 */ { "MONGOLIAN LETTER TODO CHA",NULL},
/* 1853 */ { "MONGOLIAN LETTER TODO JA",NULL},
/* 1854 */ { "MONGOLIAN LETTER TODO TSA",NULL},
/* 1855 */ { "MONGOLIAN LETTER TODO YA",NULL},
/* 1856 */ { "MONGOLIAN LETTER TODO WA",NULL},
/* 1857 */ { "MONGOLIAN LETTER TODO KA",NULL},
/* 1858 */ { "MONGOLIAN LETTER TODO GAA",NULL},
/* 1859 */ { "MONGOLIAN LETTER TODO HAA",NULL},
/* 185A */ { "MONGOLIAN LETTER TODO JIA",NULL},
/* 185B */ { "MONGOLIAN LETTER TODO NIA",NULL},
/* 185C */ { "MONGOLIAN LETTER TODO DZA",NULL},
/* 185D */ { "MONGOLIAN LETTER SIBE E",NULL},
/* 185E */ { "MONGOLIAN LETTER SIBE I",NULL},
/* 185F */ { "MONGOLIAN LETTER SIBE IY",NULL},
/* 1860 */ { "MONGOLIAN LETTER SIBE UE",NULL},
/* 1861 */ { "MONGOLIAN LETTER SIBE U",NULL},
/* 1862 */ { "MONGOLIAN LETTER SIBE ANG",NULL},
/* 1863 */ { "MONGOLIAN LETTER SIBE KA",NULL},
/* 1864 */ { "MONGOLIAN LETTER SIBE GA",NULL},
/* 1865 */ { "MONGOLIAN LETTER SIBE HA",NULL},
/* 1866 */ { "MONGOLIAN LETTER SIBE PA",NULL},
/* 1867 */ { "MONGOLIAN LETTER SIBE SHA",NULL},
/* 1868 */ { "MONGOLIAN LETTER SIBE TA",NULL},
/* 1869 */ { "MONGOLIAN LETTER SIBE DA",NULL},
/* 186A */ { "MONGOLIAN LETTER SIBE JA",NULL},
/* 186B */ { "MONGOLIAN LETTER SIBE FA",NULL},
/* 186C */ { "MONGOLIAN LETTER SIBE GAA",NULL},
/* 186D */ { "MONGOLIAN LETTER SIBE HAA",NULL},
/* 186E */ { "MONGOLIAN LETTER SIBE TSA",NULL},
/* 186F */ { "MONGOLIAN LETTER SIBE ZA",NULL},
/* 1870 */ { "MONGOLIAN LETTER SIBE RAA",NULL},
/* 1871 */ { "MONGOLIAN LETTER SIBE CHA",NULL},
/* 1872 */ { "MONGOLIAN LETTER SIBE ZHA",NULL},
/* 1873 */ { "MONGOLIAN LETTER MANCHU I",NULL},
/* 1874 */ { "MONGOLIAN LETTER MANCHU KA",NULL},
/* 1875 */ { "MONGOLIAN LETTER MANCHU RA",NULL},
/* 1876 */ { "MONGOLIAN LETTER MANCHU FA",NULL},
/* 1877 */ { "MONGOLIAN LETTER MANCHU ZHA",NULL},
/* 1878 */ { "MONGOLIAN LETTER CHA WITH TWO DOTS",NULL},
/* 1879 */ { NULL,NULL},
/* 187A */ { NULL,NULL},
/* 187B */ { NULL,NULL},
/* 187C */ { NULL,NULL},
/* 187D */ { NULL,NULL},
/* 187E */ { NULL,NULL},
/* 187F */ { NULL,NULL},
/* 1880 */ { "MONGOLIAN LETTER ALI GALI ANUSVARA ONE","	x (tibetan sign sna ldan - 0F83)"},
/* 1881 */ { "MONGOLIAN LETTER ALI GALI VISARGA ONE","	x (tibetan sign rnam bcad - 0F7F)"},
/* 1882 */ { "MONGOLIAN LETTER ALI GALI DAMARU","	x (tibetan sign lce tsa can - 0F88)"},
/* 1883 */ { "MONGOLIAN LETTER ALI GALI UBADAMA",NULL},
/* 1884 */ { "MONGOLIAN LETTER ALI GALI INVERTED UBADAMA","	x (tibetan sign mchu can - 0F89)"},
/* 1885 */ { "MONGOLIAN LETTER ALI GALI BALUDA","	x (tibetan mark paluta - 0F85)"},
/* 1886 */ { "MONGOLIAN LETTER ALI GALI THREE BALUDA",NULL},
/* 1887 */ { "MONGOLIAN LETTER ALI GALI A",NULL},
/* 1888 */ { "MONGOLIAN LETTER ALI GALI I",NULL},
/* 1889 */ { "MONGOLIAN LETTER ALI GALI KA",NULL},
/* 188A */ { "MONGOLIAN LETTER ALI GALI NGA",NULL},
/* 188B */ { "MONGOLIAN LETTER ALI GALI CA",NULL},
/* 188C */ { "MONGOLIAN LETTER ALI GALI TTA",NULL},
/* 188D */ { "MONGOLIAN LETTER ALI GALI TTHA",NULL},
/* 188E */ { "MONGOLIAN LETTER ALI GALI DDA",NULL},
/* 188F */ { "MONGOLIAN LETTER ALI GALI NNA",NULL},
/* 1890 */ { "MONGOLIAN LETTER ALI GALI TA",NULL},
/* 1891 */ { "MONGOLIAN LETTER ALI GALI DA",NULL},
/* 1892 */ { "MONGOLIAN LETTER ALI GALI PA",NULL},
/* 1893 */ { "MONGOLIAN LETTER ALI GALI PHA",NULL},
/* 1894 */ { "MONGOLIAN LETTER ALI GALI SSA",NULL},
/* 1895 */ { "MONGOLIAN LETTER ALI GALI ZHA",NULL},
/* 1896 */ { "MONGOLIAN LETTER ALI GALI ZA",NULL},
/* 1897 */ { "MONGOLIAN LETTER ALI GALI AH",NULL},
/* 1898 */ { "MONGOLIAN LETTER TODO ALI GALI TA",NULL},
/* 1899 */ { "MONGOLIAN LETTER TODO ALI GALI ZHA",NULL},
/* 189A */ { "MONGOLIAN LETTER MANCHU ALI GALI GHA",NULL},
/* 189B */ { "MONGOLIAN LETTER MANCHU ALI GALI NGA",NULL},
/* 189C */ { "MONGOLIAN LETTER MANCHU ALI GALI CA",NULL},
/* 189D */ { "MONGOLIAN LETTER MANCHU ALI GALI JHA",NULL},
/* 189E */ { "MONGOLIAN LETTER MANCHU ALI GALI TTA",NULL},
/* 189F */ { "MONGOLIAN LETTER MANCHU ALI GALI DDHA",NULL},
/* 18A0 */ { "MONGOLIAN LETTER MANCHU ALI GALI TA",NULL},
/* 18A1 */ { "MONGOLIAN LETTER MANCHU ALI GALI DHA",NULL},
/* 18A2 */ { "MONGOLIAN LETTER MANCHU ALI GALI SSA",NULL},
/* 18A3 */ { "MONGOLIAN LETTER MANCHU ALI GALI CYA",NULL},
/* 18A4 */ { "MONGOLIAN LETTER MANCHU ALI GALI ZHA",NULL},
/* 18A5 */ { "MONGOLIAN LETTER MANCHU ALI GALI ZA",NULL},
/* 18A6 */ { "MONGOLIAN LETTER ALI GALI HALF U",NULL},
/* 18A7 */ { "MONGOLIAN LETTER ALI GALI HALF YA",NULL},
/* 18A8 */ { "MONGOLIAN LETTER MANCHU ALI GALI BHA",NULL},
/* 18A9 */ { "MONGOLIAN LETTER ALI GALI DAGALGA",NULL},
/* 18AA */ { "MONGOLIAN LETTER MANCHU ALI GALI LHA",NULL},
/* 18AB */ { NULL,NULL},
/* 18AC */ { NULL,NULL},
/* 18AD */ { NULL,NULL},
/* 18AE */ { NULL,NULL},
/* 18AF */ { NULL,NULL},
/* 18B0 */ { "CANADIAN SYLLABICS OY",NULL},
/* 18B1 */ { "CANADIAN SYLLABICS AY",NULL},
/* 18B2 */ { "CANADIAN SYLLABICS AAY",NULL},
/* 18B3 */ { "CANADIAN SYLLABICS WAY",NULL},
/* 18B4 */ { "CANADIAN SYLLABICS POY",NULL},
/* 18B5 */ { "CANADIAN SYLLABICS PAY",NULL},
/* 18B6 */ { "CANADIAN SYLLABICS PWOY",NULL},
/* 18B7 */ { "CANADIAN SYLLABICS TAY",NULL},
/* 18B8 */ { "CANADIAN SYLLABICS KAY",NULL},
/* 18B9 */ { "CANADIAN SYLLABICS KWAY",NULL},
/* 18BA */ { "CANADIAN SYLLABICS MAY",NULL},
/* 18BB */ { "CANADIAN SYLLABICS NOY",NULL},
/* 18BC */ { "CANADIAN SYLLABICS NAY",NULL},
/* 18BD */ { "CANADIAN SYLLABICS LAY",NULL},
/* 18BE */ { "CANADIAN SYLLABICS SOY",NULL},
/* 18BF */ { "CANADIAN SYLLABICS SAY",NULL},
/* 18C0 */ { "CANADIAN SYLLABICS SHOY",NULL},
/* 18C1 */ { "CANADIAN SYLLABICS SHAY",NULL},
/* 18C2 */ { "CANADIAN SYLLABICS SHWOY",NULL},
/* 18C3 */ { "CANADIAN SYLLABICS YOY",NULL},
/* 18C4 */ { "CANADIAN SYLLABICS YAY",NULL},
/* 18C5 */ { "CANADIAN SYLLABICS RAY",NULL},
/* 18C6 */ { "CANADIAN SYLLABICS NWI",NULL},
/* 18C7 */ { "CANADIAN SYLLABICS OJIBWAY NWI",NULL},
/* 18C8 */ { "CANADIAN SYLLABICS NWII",NULL},
/* 18C9 */ { "CANADIAN SYLLABICS OJIBWAY NWII",NULL},
/* 18CA */ { "CANADIAN SYLLABICS NWO",NULL},
/* 18CB */ { "CANADIAN SYLLABICS OJIBWAY NWO",NULL},
/* 18CC */ { "CANADIAN SYLLABICS NWOO",NULL},
/* 18CD */ { "CANADIAN SYLLABICS OJIBWAY NWOO",NULL},
/* 18CE */ { "CANADIAN SYLLABICS RWEE",NULL},
/* 18CF */ { "CANADIAN SYLLABICS RWI",NULL},
/* 18D0 */ { "CANADIAN SYLLABICS RWII",NULL},
/* 18D1 */ { "CANADIAN SYLLABICS RWO",NULL},
/* 18D2 */ { "CANADIAN SYLLABICS RWOO",NULL},
/* 18D3 */ { "CANADIAN SYLLABICS RWA",NULL},
/* 18D4 */ { "CANADIAN SYLLABICS OJIBWAY P",NULL},
/* 18D5 */ { "CANADIAN SYLLABICS OJIBWAY T",NULL},
/* 18D6 */ { "CANADIAN SYLLABICS OJIBWAY K",NULL},
/* 18D7 */ { "CANADIAN SYLLABICS OJIBWAY C",NULL},
/* 18D8 */ { "CANADIAN SYLLABICS OJIBWAY M",NULL},
/* 18D9 */ { "CANADIAN SYLLABICS OJIBWAY N",NULL},
/* 18DA */ { "CANADIAN SYLLABICS OJIBWAY S",NULL},
/* 18DB */ { "CANADIAN SYLLABICS OJIBWAY SH",NULL},
/* 18DC */ { "CANADIAN SYLLABICS EASTERN W",NULL},
/* 18DD */ { "CANADIAN SYLLABICS WESTERN W",NULL},
/* 18DE */ { "CANADIAN SYLLABICS FINAL SMALL RING",NULL},
/* 18DF */ { "CANADIAN SYLLABICS FINAL RAISED DOT","	* Carrier (GLOTTAL STOP)"},
/* 18E0 */ { "CANADIAN SYLLABICS R-CREE RWE","	* Beaver Dene (LYE)"},
/* 18E1 */ { "CANADIAN SYLLABICS WEST-CREE LOO","	* Chipewyan Dene (LU)"},
/* 18E2 */ { "CANADIAN SYLLABICS WEST-CREE LAA","	* Beaver Dene (LAI)"},
/* 18E3 */ { "CANADIAN SYLLABICS THWE","	* Beaver Dene (TYE)"},
/* 18E4 */ { "CANADIAN SYLLABICS THWA","	* Beaver Dene (TYA)"},
/* 18E5 */ { "CANADIAN SYLLABICS TTHWE","	* Beaver Dene (HYE)"},
/* 18E6 */ { "CANADIAN SYLLABICS TTHOO","	* Chipewyan Dene (TTHU)"},
/* 18E7 */ { "CANADIAN SYLLABICS TTHAA","	* Beaver Dene (HAI)"},
/* 18E8 */ { "CANADIAN SYLLABICS TLHWE","	* Beaver Dene (TLYE)"},
/* 18E9 */ { "CANADIAN SYLLABICS TLHOO","	* Chipewyan Dene (TLU)"},
/* 18EA */ { "CANADIAN SYLLABICS SAYISI SHWE","	* Beaver Dene (ZYE)"},
/* 18EB */ { "CANADIAN SYLLABICS SAYISI SHOO","	* Chipewyan Dene (SHU)"},
/* 18EC */ { "CANADIAN SYLLABICS SAYISI HOO","	* Chipewyan Dene (GU)"},
/* 18ED */ { "CANADIAN SYLLABICS CARRIER GWU","	* Beaver Dene (CHYE)"},
/* 18EE */ { "CANADIAN SYLLABICS CARRIER DENE GEE","	* Chipewyan Dene (CHU)"},
/* 18EF */ { "CANADIAN SYLLABICS CARRIER GAA","	* Beaver Dene (CHAI)"},
/* 18F0 */ { "CANADIAN SYLLABICS CARRIER GWA","	* Hare Dene (SHYA)"},
/* 18F1 */ { "CANADIAN SYLLABICS SAYISI JUU","	* Chipewyan Dene (TSU)"},
/* 18F2 */ { "CANADIAN SYLLABICS CARRIER JWA","	* Beaver Dene (TSYE)"},
/* 18F3 */ { "CANADIAN SYLLABICS BEAVER DENE L",NULL},
/* 18F4 */ { "CANADIAN SYLLABICS BEAVER DENE R",NULL},
/* 18F5 */ { "CANADIAN SYLLABICS CARRIER DENTAL S",NULL},
/* 18F6 */ { NULL,NULL},
/* 18F7 */ { NULL,NULL},
/* 18F8 */ { NULL,NULL},
/* 18F9 */ { NULL,NULL},
/* 18FA */ { NULL,NULL},
/* 18FB */ { NULL,NULL},
/* 18FC */ { NULL,NULL},
/* 18FD */ { NULL,NULL},
/* 18FE */ { NULL,NULL},
/* 18FF */ { NULL,NULL}
};

UN_DLL_LOCAL
static const struct unicode_nameannot una_00_19[] = {
/* 1900 */ { "LIMBU VOWEL-CARRIER LETTER",NULL},
/* 1901 */ { "LIMBU LETTER KA",NULL},
/* 1902 */ { "LIMBU LETTER KHA",NULL},
/* 1903 */ { "LIMBU LETTER GA",NULL},
/* 1904 */ { "LIMBU LETTER GHA",NULL},
/* 1905 */ { "LIMBU LETTER NGA",NULL},
/* 1906 */ { "LIMBU LETTER CA",NULL},
/* 1907 */ { "LIMBU LETTER CHA",NULL},
/* 1908 */ { "LIMBU LETTER JA",NULL},
/* 1909 */ { "LIMBU LETTER JHA",NULL},
/* 190A */ { "LIMBU LETTER YAN",NULL},
/* 190B */ { "LIMBU LETTER TA",NULL},
/* 190C */ { "LIMBU LETTER THA",NULL},
/* 190D */ { "LIMBU LETTER DA",NULL},
/* 190E */ { "LIMBU LETTER DHA",NULL},
/* 190F */ { "LIMBU LETTER NA",NULL},
/* 1910 */ { "LIMBU LETTER PA",NULL},
/* 1911 */ { "LIMBU LETTER PHA",NULL},
/* 1912 */ { "LIMBU LETTER BA",NULL},
/* 1913 */ { "LIMBU LETTER BHA",NULL},
/* 1914 */ { "LIMBU LETTER MA",NULL},
/* 1915 */ { "LIMBU LETTER YA",NULL},
/* 1916 */ { "LIMBU LETTER RA",NULL},
/* 1917 */ { "LIMBU LETTER LA",NULL},
/* 1918 */ { "LIMBU LETTER WA",NULL},
/* 1919 */ { "LIMBU LETTER SHA",NULL},
/* 191A */ { "LIMBU LETTER SSA",NULL},
/* 191B */ { "LIMBU LETTER SA",NULL},
/* 191C */ { "LIMBU LETTER HA",NULL},
/* 191D */ { "LIMBU LETTER GYAN","	= jnya"},
/* 191E */ { "LIMBU LETTER TRA",NULL},
/* 191F */ { NULL,NULL},
/* 1920 */ { "LIMBU VOWEL SIGN A",NULL},
/* 1921 */ { "LIMBU VOWEL SIGN I",NULL},
/* 1922 */ { "LIMBU VOWEL SIGN U",NULL},
/* 1923 */ { "LIMBU VOWEL SIGN EE",NULL},
/* 1924 */ { "LIMBU VOWEL SIGN AI",NULL},
/* 1925 */ { "LIMBU VOWEL SIGN OO",NULL},
/* 1926 */ { "LIMBU VOWEL SIGN AU",NULL},
/* 1927 */ { "LIMBU VOWEL SIGN E",NULL},
/* 1928 */ { "LIMBU VOWEL SIGN O",NULL},
/* 1929 */ { "LIMBU SUBJOINED LETTER YA",NULL},
/* 192A */ { "LIMBU SUBJOINED LETTER RA",NULL},
/* 192B */ { "LIMBU SUBJOINED LETTER WA",NULL},
/* 192C */ { NULL,NULL},
/* 192D */ { NULL,NULL},
/* 192E */ { NULL,NULL},
/* 192F */ { NULL,NULL},
/* 1930 */ { "LIMBU SMALL LETTER KA",NULL},
/* 1931 */ { "LIMBU SMALL LETTER NGA",NULL},
/* 1932 */ { "LIMBU SMALL LETTER ANUSVARA",NULL},
/* 1933 */ { "LIMBU SMALL LETTER TA",NULL},
/* 1934 */ { "LIMBU SMALL LETTER NA",NULL},
/* 1935 */ { "LIMBU SMALL LETTER PA",NULL},
/* 1936 */ { "LIMBU SMALL LETTER MA",NULL},
/* 1937 */ { "LIMBU SMALL LETTER RA",NULL},
/* 1938 */ { "LIMBU SMALL LETTER LA",NULL},
/* 1939 */ { "LIMBU SIGN MUKPHRENG",NULL},
/* 193A */ { "LIMBU SIGN KEMPHRENG",NULL},
/* 193B */ { "LIMBU SIGN SA-I",NULL},
/* 193C */ { NULL,NULL},
/* 193D */ { NULL,NULL},
/* 193E */ { NULL,NULL},
/* 193F */ { NULL,NULL},
/* 1940 */ { "LIMBU SIGN LOO",NULL},
/* 1941 */ { NULL,NULL},
/* 1942 */ { NULL,NULL},
/* 1943 */ { NULL,NULL},
/* 1944 */ { "LIMBU EXCLAMATION MARK",NULL},
/* 1945 */ { "LIMBU QUESTION MARK",NULL},
/* 1946 */ { "LIMBU DIGIT ZERO",NULL},
/* 1947 */ { "LIMBU DIGIT ONE",NULL},
/* 1948 */ { "LIMBU DIGIT TWO",NULL},
/* 1949 */ { "LIMBU DIGIT THREE",NULL},
/* 194A */ { "LIMBU DIGIT FOUR",NULL},
/* 194B */ { "LIMBU DIGIT FIVE",NULL},
/* 194C */ { "LIMBU DIGIT SIX",NULL},
/* 194D */ { "LIMBU DIGIT SEVEN",NULL},
/* 194E */ { "LIMBU DIGIT EIGHT",NULL},
/* 194F */ { "LIMBU DIGIT NINE",NULL},
/* 1950 */ { "TAI LE LETTER KA",NULL},
/* 1951 */ { "TAI LE LETTER XA",NULL},
/* 1952 */ { "TAI LE LETTER NGA",NULL},
/* 1953 */ { "TAI LE LETTER TSA",NULL},
/* 1954 */ { "TAI LE LETTER SA",NULL},
/* 1955 */ { "TAI LE LETTER YA",NULL},
/* 1956 */ { "TAI LE LETTER TA",NULL},
/* 1957 */ { "TAI LE LETTER THA",NULL},
/* 1958 */ { "TAI LE LETTER LA",NULL},
/* 1959 */ { "TAI LE LETTER PA",NULL},
/* 195A */ { "TAI LE LETTER PHA",NULL},
/* 195B */ { "TAI LE LETTER MA",NULL},
/* 195C */ { "TAI LE LETTER FA",NULL},
/* 195D */ { "TAI LE LETTER VA",NULL},
/* 195E */ { "TAI LE LETTER HA",NULL},
/* 195F */ { "TAI LE LETTER QA",NULL},
/* 1960 */ { "TAI LE LETTER KHA",NULL},
/* 1961 */ { "TAI LE LETTER TSHA",NULL},
/* 1962 */ { "TAI LE LETTER NA",NULL},
/* 1963 */ { "TAI LE LETTER A",NULL},
/* 1964 */ { "TAI LE LETTER I",NULL},
/* 1965 */ { "TAI LE LETTER EE",NULL},
/* 1966 */ { "TAI LE LETTER EH",NULL},
/* 1967 */ { "TAI LE LETTER U",NULL},
/* 1968 */ { "TAI LE LETTER OO",NULL},
/* 1969 */ { "TAI LE LETTER O",NULL},
/* 196A */ { "TAI LE LETTER UE",NULL},
/* 196B */ { "TAI LE LETTER E",NULL},
/* 196C */ { "TAI LE LETTER AUE",NULL},
/* 196D */ { "TAI LE LETTER AI",NULL},
/* 196E */ { NULL,NULL},
/* 196F */ { NULL,NULL},
/* 1970 */ { "TAI LE LETTER TONE-2",NULL},
/* 1971 */ { "TAI LE LETTER TONE-3",NULL},
/* 1972 */ { "TAI LE LETTER TONE-4",NULL},
/* 1973 */ { "TAI LE LETTER TONE-5",NULL},
/* 1974 */ { "TAI LE LETTER TONE-6",NULL},
/* 1975 */ { NULL,NULL},
/* 1976 */ { NULL,NULL},
/* 1977 */ { NULL,NULL},
/* 1978 */ { NULL,NULL},
/* 1979 */ { NULL,NULL},
/* 197A */ { NULL,NULL},
/* 197B */ { NULL,NULL},
/* 197C */ { NULL,NULL},
/* 197D */ { NULL,NULL},
/* 197E */ { NULL,NULL},
/* 197F */ { NULL,NULL},
/* 1980 */ { "NEW TAI LUE LETTER HIGH QA",NULL},
/* 1981 */ { "NEW TAI LUE LETTER LOW QA",NULL},
/* 1982 */ { "NEW TAI LUE LETTER HIGH KA",NULL},
/* 1983 */ { "NEW TAI LUE LETTER HIGH XA",NULL},
/* 1984 */ { "NEW TAI LUE LETTER HIGH NGA",NULL},
/* 1985 */ { "NEW TAI LUE LETTER LOW KA",NULL},
/* 1986 */ { "NEW TAI LUE LETTER LOW XA",NULL},
/* 1987 */ { "NEW TAI LUE LETTER LOW NGA",NULL},
/* 1988 */ { "NEW TAI LUE LETTER HIGH TSA",NULL},
/* 1989 */ { "NEW TAI LUE LETTER HIGH SA",NULL},
/* 198A */ { "NEW TAI LUE LETTER HIGH YA",NULL},
/* 198B */ { "NEW TAI LUE LETTER LOW TSA",NULL},
/* 198C */ { "NEW TAI LUE LETTER LOW SA",NULL},
/* 198D */ { "NEW TAI LUE LETTER LOW YA",NULL},
/* 198E */ { "NEW TAI LUE LETTER HIGH TA",NULL},
/* 198F */ { "NEW TAI LUE LETTER HIGH THA",NULL},
/* 1990 */ { "NEW TAI LUE LETTER HIGH NA",NULL},
/* 1991 */ { "NEW TAI LUE LETTER LOW TA",NULL},
/* 1992 */ { "NEW TAI LUE LETTER LOW THA",NULL},
/* 1993 */ { "NEW TAI LUE LETTER LOW NA",NULL},
/* 1994 */ { "NEW TAI LUE LETTER HIGH PA",NULL},
/* 1995 */ { "NEW TAI LUE LETTER HIGH PHA",NULL},
/* 1996 */ { "NEW TAI LUE LETTER HIGH MA",NULL},
/* 1997 */ { "NEW TAI LUE LETTER LOW PA",NULL},
/* 1998 */ { "NEW TAI LUE LETTER LOW PHA",NULL},
/* 1999 */ { "NEW TAI LUE LETTER LOW MA",NULL},
/* 199A */ { "NEW TAI LUE LETTER HIGH FA",NULL},
/* 199B */ { "NEW TAI LUE LETTER HIGH VA",NULL},
/* 199C */ { "NEW TAI LUE LETTER HIGH LA",NULL},
/* 199D */ { "NEW TAI LUE LETTER LOW FA",NULL},
/* 199E */ { "NEW TAI LUE LETTER LOW VA",NULL},
/* 199F */ { "NEW TAI LUE LETTER LOW LA",NULL},
/* 19A0 */ { "NEW TAI LUE LETTER HIGH HA",NULL},
/* 19A1 */ { "NEW TAI LUE LETTER HIGH DA",NULL},
/* 19A2 */ { "NEW TAI LUE LETTER HIGH BA",NULL},
/* 19A3 */ { "NEW TAI LUE LETTER LOW HA",NULL},
/* 19A4 */ { "NEW TAI LUE LETTER LOW DA",NULL},
/* 19A5 */ { "NEW TAI LUE LETTER LOW BA",NULL},
/* 19A6 */ { "NEW TAI LUE LETTER HIGH KVA",NULL},
/* 19A7 */ { "NEW TAI LUE LETTER HIGH XVA",NULL},
/* 19A8 */ { "NEW TAI LUE LETTER LOW KVA",NULL},
/* 19A9 */ { "NEW TAI LUE LETTER LOW XVA",NULL},
/* 19AA */ { "NEW TAI LUE LETTER HIGH SUA",NULL},
/* 19AB */ { "NEW TAI LUE LETTER LOW SUA","	* a labialized consonant cluster"},
/* 19AC */ { NULL,NULL},
/* 19AD */ { NULL,NULL},
/* 19AE */ { NULL,NULL},
/* 19AF */ { NULL,NULL},
/* 19B0 */ { "NEW TAI LUE VOWEL SIGN VOWEL SHORTENER",NULL},
/* 19B1 */ { "NEW TAI LUE VOWEL SIGN AA",NULL},
/* 19B2 */ { "NEW TAI LUE VOWEL SIGN II",NULL},
/* 19B3 */ { "NEW TAI LUE VOWEL SIGN U",NULL},
/* 19B4 */ { "NEW TAI LUE VOWEL SIGN UU",NULL},
/* 19B5 */ { "NEW TAI LUE VOWEL SIGN E","	* precedes consonant in visual order"},
/* 19B6 */ { "NEW TAI LUE VOWEL SIGN AE","	* precedes consonant in visual order"},
/* 19B7 */ { "NEW TAI LUE VOWEL SIGN O","	* precedes consonant in visual order"},
/* 19B8 */ { "NEW TAI LUE VOWEL SIGN OA",NULL},
/* 19B9 */ { "NEW TAI LUE VOWEL SIGN UE",NULL},
/* 19BA */ { "NEW TAI LUE VOWEL SIGN AY","	* precedes consonant in visual order"},
/* 19BB */ { "NEW TAI LUE VOWEL SIGN AAY",NULL},
/* 19BC */ { "NEW TAI LUE VOWEL SIGN UY",NULL},
/* 19BD */ { "NEW TAI LUE VOWEL SIGN OY",NULL},
/* 19BE */ { "NEW TAI LUE VOWEL SIGN OAY",NULL},
/* 19BF */ { "NEW TAI LUE VOWEL SIGN UEY",NULL},
/* 19C0 */ { "NEW TAI LUE VOWEL SIGN IY",NULL},
/* 19C1 */ { "NEW TAI LUE LETTER FINAL V",NULL},
/* 19C2 */ { "NEW TAI LUE LETTER FINAL NG",NULL},
/* 19C3 */ { "NEW TAI LUE LETTER FINAL N",NULL},
/* 19C4 */ { "NEW TAI LUE LETTER FINAL M",NULL},
/* 19C5 */ { "NEW TAI LUE LETTER FINAL K",NULL},
/* 19C6 */ { "NEW TAI LUE LETTER FINAL D",NULL},
/* 19C7 */ { "NEW TAI LUE LETTER FINAL B",NULL},
/* 19C8 */ { "NEW TAI LUE TONE MARK-1",NULL},
/* 19C9 */ { "NEW TAI LUE TONE MARK-2",NULL},
/* 19CA */ { NULL,NULL},
/* 19CB */ { NULL,NULL},
/* 19CC */ { NULL,NULL},
/* 19CD */ { NULL,NULL},
/* 19CE */ { NULL,NULL},
/* 19CF */ { NULL,NULL},
/* 19D0 */ { "NEW TAI LUE DIGIT ZERO",NULL},
/* 19D1 */ { "NEW TAI LUE DIGIT ONE",NULL},
/* 19D2 */ { "NEW TAI LUE DIGIT TWO",NULL},
/* 19D3 */ { "NEW TAI LUE DIGIT THREE",NULL},
/* 19D4 */ { "NEW TAI LUE DIGIT FOUR",NULL},
/* 19D5 */ { "NEW TAI LUE DIGIT FIVE",NULL},
/* 19D6 */ { "NEW TAI LUE DIGIT SIX",NULL},
/* 19D7 */ { "NEW TAI LUE DIGIT SEVEN",NULL},
/* 19D8 */ { "NEW TAI LUE DIGIT EIGHT",NULL},
/* 19D9 */ { "NEW TAI LUE DIGIT NINE",NULL},
/* 19DA */ { "NEW TAI LUE THAM DIGIT ONE",NULL},
/* 19DB */ { NULL,NULL},
/* 19DC */ { NULL,NULL},
/* 19DD */ { NULL,NULL},
/* 19DE */ { "NEW TAI LUE SIGN LAE","	* conjunction: and"},
/* 19DF */ { "NEW TAI LUE SIGN LAEV","	* perfective"},
/* 19E0 */ { "KHMER SYMBOL PATHAMASAT","	* represents the first Ashadha (eighth month of the lunar calendar)\n"
	"		The following fifteen characters represent the first through the fifteenth waxing days, respectively."},
/* 19E1 */ { "KHMER SYMBOL MUOY KOET",NULL},
/* 19E2 */ { "KHMER SYMBOL PII KOET",NULL},
/* 19E3 */ { "KHMER SYMBOL BEI KOET",NULL},
/* 19E4 */ { "KHMER SYMBOL BUON KOET",NULL},
/* 19E5 */ { "KHMER SYMBOL PRAM KOET",NULL},
/* 19E6 */ { "KHMER SYMBOL PRAM-MUOY KOET",NULL},
/* 19E7 */ { "KHMER SYMBOL PRAM-PII KOET",NULL},
/* 19E8 */ { "KHMER SYMBOL PRAM-BEI KOET",NULL},
/* 19E9 */ { "KHMER SYMBOL PRAM-BUON KOET",NULL},
/* 19EA */ { "KHMER SYMBOL DAP KOET",NULL},
/* 19EB */ { "KHMER SYMBOL DAP-MUOY KOET",NULL},
/* 19EC */ { "KHMER SYMBOL DAP-PII KOET",NULL},
/* 19ED */ { "KHMER SYMBOL DAP-BEI KOET",NULL},
/* 19EE */ { "KHMER SYMBOL DAP-BUON KOET",NULL},
/* 19EF */ { "KHMER SYMBOL DAP-PRAM KOET",NULL},
/* 19F0 */ { "KHMER SYMBOL TUTEYASAT","	* represents the second Ashadha in the lunar calendar during the Adhikameas leap year\n"
	"		The following fifteen characters represent the first through the fifteenth waning days, respectively."},
/* 19F1 */ { "KHMER SYMBOL MUOY ROC",NULL},
/* 19F2 */ { "KHMER SYMBOL PII ROC",NULL},
/* 19F3 */ { "KHMER SYMBOL BEI ROC",NULL},
/* 19F4 */ { "KHMER SYMBOL BUON ROC",NULL},
/* 19F5 */ { "KHMER SYMBOL PRAM ROC",NULL},
/* 19F6 */ { "KHMER SYMBOL PRAM-MUOY ROC",NULL},
/* 19F7 */ { "KHMER SYMBOL PRAM-PII ROC",NULL},
/* 19F8 */ { "KHMER SYMBOL PRAM-BEI ROC",NULL},
/* 19F9 */ { "KHMER SYMBOL PRAM-BUON ROC",NULL},
/* 19FA */ { "KHMER SYMBOL DAP ROC",NULL},
/* 19FB */ { "KHMER SYMBOL DAP-MUOY ROC",NULL},
/* 19FC */ { "KHMER SYMBOL DAP-PII ROC",NULL},
/* 19FD */ { "KHMER SYMBOL DAP-BEI ROC",NULL},
/* 19FE */ { "KHMER SYMBOL DAP-BUON ROC",NULL},
/* 19FF */ { "KHMER SYMBOL DAP-PRAM ROC",NULL}
};

UN_DLL_LOCAL
static const struct unicode_nameannot una_00_1A[] = {
/* 1A00 */ { "BUGINESE LETTER KA",NULL},
/* 1A01 */ { "BUGINESE LETTER GA",NULL},
/* 1A02 */ { "BUGINESE LETTER NGA",NULL},
/* 1A03 */ { "BUGINESE LETTER NGKA",NULL},
/* 1A04 */ { "BUGINESE LETTER PA",NULL},
/* 1A05 */ { "BUGINESE LETTER BA",NULL},
/* 1A06 */ { "BUGINESE LETTER MA",NULL},
/* 1A07 */ { "BUGINESE LETTER MPA",NULL},
/* 1A08 */ { "BUGINESE LETTER TA",NULL},
/* 1A09 */ { "BUGINESE LETTER DA",NULL},
/* 1A0A */ { "BUGINESE LETTER NA",NULL},
/* 1A0B */ { "BUGINESE LETTER NRA",NULL},
/* 1A0C */ { "BUGINESE LETTER CA",NULL},
/* 1A0D */ { "BUGINESE LETTER JA",NULL},
/* 1A0E */ { "BUGINESE LETTER NYA",NULL},
/* 1A0F */ { "BUGINESE LETTER NYCA",NULL},
/* 1A10 */ { "BUGINESE LETTER YA",NULL},
/* 1A11 */ { "BUGINESE LETTER RA",NULL},
/* 1A12 */ { "BUGINESE LETTER LA",NULL},
/* 1A13 */ { "BUGINESE LETTER VA",NULL},
/* 1A14 */ { "BUGINESE LETTER SA",NULL},
/* 1A15 */ { "BUGINESE LETTER A",NULL},
/* 1A16 */ { "BUGINESE LETTER HA",NULL},
/* 1A17 */ { "BUGINESE VOWEL SIGN I",NULL},
/* 1A18 */ { "BUGINESE VOWEL SIGN U",NULL},
/* 1A19 */ { "BUGINESE VOWEL SIGN E",NULL},
/* 1A1A */ { "BUGINESE VOWEL SIGN O",NULL},
/* 1A1B */ { "BUGINESE VOWEL SIGN AE",NULL},
/* 1A1C */ { NULL,NULL},
/* 1A1D */ { NULL,NULL},
/* 1A1E */ { "BUGINESE PALLAWA",NULL},
/* 1A1F */ { "BUGINESE END OF SECTION",NULL},
/* 1A20 */ { "TAI THAM LETTER HIGH KA",NULL},
/* 1A21 */ { "TAI THAM LETTER HIGH KHA",NULL},
/* 1A22 */ { "TAI THAM LETTER HIGH KXA",NULL},
/* 1A23 */ { "TAI THAM LETTER LOW KA",NULL},
/* 1A24 */ { "TAI THAM LETTER LOW KXA",NULL},
/* 1A25 */ { "TAI THAM LETTER LOW KHA",NULL},
/* 1A26 */ { "TAI THAM LETTER NGA",NULL},
/* 1A27 */ { "TAI THAM LETTER HIGH CA",NULL},
/* 1A28 */ { "TAI THAM LETTER HIGH CHA",NULL},
/* 1A29 */ { "TAI THAM LETTER LOW CA",NULL},
/* 1A2A */ { "TAI THAM LETTER LOW SA",NULL},
/* 1A2B */ { "TAI THAM LETTER LOW CHA",NULL},
/* 1A2C */ { "TAI THAM LETTER NYA",NULL},
/* 1A2D */ { "TAI THAM LETTER RATA",NULL},
/* 1A2E */ { "TAI THAM LETTER HIGH RATHA","	* an alternative glyph with the upper part shaped like 1A33 is used in Thailand and Laos\n"
	"	* contrast the sequence 1A2D 1A5B"},
/* 1A2F */ { "TAI THAM LETTER DA",NULL},
/* 1A30 */ { "TAI THAM LETTER LOW RATHA",NULL},
/* 1A31 */ { "TAI THAM LETTER RANA",NULL},
/* 1A32 */ { "TAI THAM LETTER HIGH TA",NULL},
/* 1A33 */ { "TAI THAM LETTER HIGH THA",NULL},
/* 1A34 */ { "TAI THAM LETTER LOW TA",NULL},
/* 1A35 */ { "TAI THAM LETTER LOW THA",NULL},
/* 1A36 */ { "TAI THAM LETTER NA",NULL},
/* 1A37 */ { "TAI THAM LETTER BA",NULL},
/* 1A38 */ { "TAI THAM LETTER HIGH PA",NULL},
/* 1A39 */ { "TAI THAM LETTER HIGH PHA",NULL},
/* 1A3A */ { "TAI THAM LETTER HIGH FA",NULL},
/* 1A3B */ { "TAI THAM LETTER LOW PA",NULL},
/* 1A3C */ { "TAI THAM LETTER LOW FA",NULL},
/* 1A3D */ { "TAI THAM LETTER LOW PHA",NULL},
/* 1A3E */ { "TAI THAM LETTER MA",NULL},
/* 1A3F */ { "TAI THAM LETTER LOW YA",NULL},
/* 1A40 */ { "TAI THAM LETTER HIGH YA",NULL},
/* 1A41 */ { "TAI THAM LETTER RA",NULL},
/* 1A42 */ { "TAI THAM LETTER RUE",NULL},
/* 1A43 */ { "TAI THAM LETTER LA",NULL},
/* 1A44 */ { "TAI THAM LETTER LUE",NULL},
/* 1A45 */ { "TAI THAM LETTER WA",NULL},
/* 1A46 */ { "TAI THAM LETTER HIGH SHA",NULL},
/* 1A47 */ { "TAI THAM LETTER HIGH SSA",NULL},
/* 1A48 */ { "TAI THAM LETTER HIGH SA",NULL},
/* 1A49 */ { "TAI THAM LETTER HIGH HA",NULL},
/* 1A4A */ { "TAI THAM LETTER LLA",NULL},
/* 1A4B */ { "TAI THAM LETTER A","	* indicates a glottal consonant"},
/* 1A4C */ { "TAI THAM LETTER LOW HA",NULL},
/* 1A4D */ { "TAI THAM LETTER I",NULL},
/* 1A4E */ { "TAI THAM LETTER II",NULL},
/* 1A4F */ { "TAI THAM LETTER U",NULL},
/* 1A50 */ { "TAI THAM LETTER UU",NULL},
/* 1A51 */ { "TAI THAM LETTER EE",NULL},
/* 1A52 */ { "TAI THAM LETTER OO",NULL},
/* 1A53 */ { "TAI THAM LETTER LAE",NULL},
/* 1A54 */ { "TAI THAM LETTER GREAT SA",NULL},
/* 1A55 */ { "TAI THAM CONSONANT SIGN MEDIAL RA",NULL},
/* 1A56 */ { "TAI THAM CONSONANT SIGN MEDIAL LA",NULL},
/* 1A57 */ { "TAI THAM CONSONANT SIGN LA TANG LAI",NULL},
/* 1A58 */ { "TAI THAM SIGN MAI KANG LAI",NULL},
/* 1A59 */ { "TAI THAM CONSONANT SIGN FINAL NGA",NULL},
/* 1A5A */ { "TAI THAM CONSONANT SIGN LOW PA",NULL},
/* 1A5B */ { "TAI THAM CONSONANT SIGN HIGH RATHA OR LOW PA",NULL},
/* 1A5C */ { "TAI THAM CONSONANT SIGN MA",NULL},
/* 1A5D */ { "TAI THAM CONSONANT SIGN BA",NULL},
/* 1A5E */ { "TAI THAM CONSONANT SIGN SA",NULL},
/* 1A5F */ { NULL,NULL},
/* 1A60 */ { "TAI THAM SIGN SAKOT",NULL},
/* 1A61 */ { "TAI THAM VOWEL SIGN A",NULL},
/* 1A62 */ { "TAI THAM VOWEL SIGN MAI SAT",NULL},
/* 1A63 */ { "TAI THAM VOWEL SIGN AA",NULL},
/* 1A64 */ { "TAI THAM VOWEL SIGN TALL AA",NULL},
/* 1A65 */ { "TAI THAM VOWEL SIGN I",NULL},
/* 1A66 */ { "TAI THAM VOWEL SIGN II",NULL},
/* 1A67 */ { "TAI THAM VOWEL SIGN UE",NULL},
/* 1A68 */ { "TAI THAM VOWEL SIGN UUE",NULL},
/* 1A69 */ { "TAI THAM VOWEL SIGN U",NULL},
/* 1A6A */ { "TAI THAM VOWEL SIGN UU",NULL},
/* 1A6B */ { "TAI THAM VOWEL SIGN O",NULL},
/* 1A6C */ { "TAI THAM VOWEL SIGN OA BELOW",NULL},
/* 1A6D */ { "TAI THAM VOWEL SIGN OY",NULL},
/* 1A6E */ { "TAI THAM VOWEL SIGN E",NULL},
/* 1A6F */ { "TAI THAM VOWEL SIGN AE",NULL},
/* 1A70 */ { "TAI THAM VOWEL SIGN OO",NULL},
/* 1A71 */ { "TAI THAM VOWEL SIGN AI",NULL},
/* 1A72 */ { "TAI THAM VOWEL SIGN THAM AI",NULL},
/* 1A73 */ { "TAI THAM VOWEL SIGN OA ABOVE",NULL},
/* 1A74 */ { "TAI THAM SIGN MAI KANG",NULL},
/* 1A75 */ { "TAI THAM SIGN TONE-1",NULL},
/* 1A76 */ { "TAI THAM SIGN TONE-2",NULL},
/* 1A77 */ { "TAI THAM SIGN KHUEN TONE-3",NULL},
/* 1A78 */ { "TAI THAM SIGN KHUEN TONE-4",NULL},
/* 1A79 */ { "TAI THAM SIGN KHUEN TONE-5",NULL},
/* 1A7A */ { "TAI THAM SIGN RA HAAM","	* indicates following character(s) are not sounded"},
/* 1A7B */ { "TAI THAM SIGN MAI SAM",NULL},
/* 1A7C */ { "TAI THAM SIGN KHUEN-LUE KARAN",NULL},
/* 1A7D */ { NULL,NULL},
/* 1A7E */ { NULL,NULL},
/* 1A7F */ { "TAI THAM COMBINING CRYPTOGRAMMIC DOT",NULL},
/* 1A80 */ { "TAI THAM HORA DIGIT ZERO",NULL},
/* 1A81 */ { "TAI THAM HORA DIGIT ONE",NULL},
/* 1A82 */ { "TAI THAM HORA DIGIT TWO",NULL},
/* 1A83 */ { "TAI THAM HORA DIGIT THREE",NULL},
/* 1A84 */ { "TAI THAM HORA DIGIT FOUR",NULL},
/* 1A85 */ { "TAI THAM HORA DIGIT FIVE",NULL},
/* 1A86 */ { "TAI THAM HORA DIGIT SIX",NULL},
/* 1A87 */ { "TAI THAM HORA DIGIT SEVEN",NULL},
/* 1A88 */ { "TAI THAM HORA DIGIT EIGHT",NULL},
/* 1A89 */ { "TAI THAM HORA DIGIT NINE",NULL},
/* 1A8A */ { NULL,NULL},
/* 1A8B */ { NULL,NULL},
/* 1A8C */ { NULL,NULL},
/* 1A8D */ { NULL,NULL},
/* 1A8E */ { NULL,NULL},
/* 1A8F */ { NULL,NULL},
/* 1A90 */ { "TAI THAM THAM DIGIT ZERO",NULL},
/* 1A91 */ { "TAI THAM THAM DIGIT ONE",NULL},
/* 1A92 */ { "TAI THAM THAM DIGIT TWO",NULL},
/* 1A93 */ { "TAI THAM THAM DIGIT THREE",NULL},
/* 1A94 */ { "TAI THAM THAM DIGIT FOUR",NULL},
/* 1A95 */ { "TAI THAM THAM DIGIT FIVE",NULL},
/* 1A96 */ { "TAI THAM THAM DIGIT SIX",NULL},
/* 1A97 */ { "TAI THAM THAM DIGIT SEVEN",NULL},
/* 1A98 */ { "TAI THAM THAM DIGIT EIGHT",NULL},
/* 1A99 */ { "TAI THAM THAM DIGIT NINE",NULL},
/* 1A9A */ { NULL,NULL},
/* 1A9B */ { NULL,NULL},
/* 1A9C */ { NULL,NULL},
/* 1A9D */ { NULL,NULL},
/* 1A9E */ { NULL,NULL},
/* 1A9F */ { NULL,NULL},
/* 1AA0 */ { "TAI THAM SIGN WIANG","	* logograph for city"},
/* 1AA1 */ { "TAI THAM SIGN WIANGWAAK","	* logograph for village"},
/* 1AA2 */ { "TAI THAM SIGN SAWAN","	* logograph for heaven"},
/* 1AA3 */ { "TAI THAM SIGN KEOW","	= courtyard"},
/* 1AA4 */ { "TAI THAM SIGN HOY","	= oyster"},
/* 1AA5 */ { "TAI THAM SIGN DOKMAI","	= flower\n"
	"	x (flower punctuation mark - 2055)"},
/* 1AA6 */ { "TAI THAM SIGN REVERSED ROTATED RANA",NULL},
/* 1AA7 */ { "TAI THAM SIGN MAI YAMOK","	x (thai character maiyamok - 0E46)"},
/* 1AA8 */ { "TAI THAM SIGN KAAN","	* danda"},
/* 1AA9 */ { "TAI THAM SIGN KAANKUU","	* double danda"},
/* 1AAA */ { "TAI THAM SIGN SATKAAN",NULL},
/* 1AAB */ { "TAI THAM SIGN SATKAANKUU","	x (thai character angkhankhu - 0E5A)"},
/* 1AAC */ { "TAI THAM SIGN HANG","	* used in combinations with other punctuation marks at ends of sections"},
/* 1AAD */ { "TAI THAM SIGN CAANG",NULL},
/* 1AAE */ { NULL,NULL},
/* 1AAF */ { NULL,NULL},
/* 1AB0 */ { "COMBINING DOUBLED CIRCUMFLEX ACCENT",NULL},
/* 1AB1 */ { "COMBINING DIAERESIS-RING",NULL},
/* 1AB2 */ { "COMBINING INFINITY",NULL},
/* 1AB3 */ { "COMBINING DOWNWARDS ARROW",NULL},
/* 1AB4 */ { "COMBINING TRIPLE DOT",NULL},
/* 1AB5 */ { "COMBINING X-X BELOW",NULL},
/* 1AB6 */ { "COMBINING WIGGLY LINE BELOW",NULL},
/* 1AB7 */ { "COMBINING OPEN MARK BELOW","	x (combining ogonek - 0328)\n"
	"	x (combining greek ypogegrammeni - 0345)"},
/* 1AB8 */ { "COMBINING DOUBLE OPEN MARK BELOW",NULL},
/* 1AB9 */ { "COMBINING LIGHT CENTRALIZATION STROKE BELOW",NULL},
/* 1ABA */ { "COMBINING STRONG CENTRALIZATION STROKE BELOW",NULL},
/* 1ABB */ { "COMBINING PARENTHESES ABOVE","	* intended to surround a diacritic above"},
/* 1ABC */ { "COMBINING DOUBLE PARENTHESES ABOVE",NULL},
/* 1ABD */ { "COMBINING PARENTHESES BELOW","	* intended to surround a diacritic below"},
/* 1ABE */ { "COMBINING PARENTHESES OVERLAY","	* intended to surround a base letter\n"
	"	* exact placement is font dependent"},
/* 1ABF */ { "COMBINING LATIN SMALL LETTER W BELOW","	* indicates voiced labialization\n"
	"	x (modifier letter small w - 02B7)"},
/* 1AC0 */ { "COMBINING LATIN SMALL LETTER TURNED W BELOW","	* indicates voiceless labialization"},
/* 1AC1 */ { "COMBINING LEFT PARENTHESIS ABOVE LEFT",NULL},
/* 1AC2 */ { "COMBINING RIGHT PARENTHESIS ABOVE RIGHT",NULL},
/* 1AC3 */ { "COMBINING LEFT PARENTHESIS BELOW LEFT",NULL},
/* 1AC4 */ { "COMBINING RIGHT PARENTHESIS BELOW RIGHT",NULL},
/* 1AC5 */ { "COMBINING SQUARE BRACKETS ABOVE","	* intended to surround a diacritic above\n"
	"	* used to indicate cancellation of an accent in Japanese transliteration"},
/* 1AC6 */ { "COMBINING NUMBER SIGN ABOVE","	= superscript octothorp"},
/* 1AC7 */ { "COMBINING INVERTED DOUBLE ARCH ABOVE","	x (combining inverted double arch below - 032B)\n"
	"	x (combining latin small letter w - 1DF1)"},
/* 1AC8 */ { "COMBINING PLUS SIGN ABOVE","	x (combining plus sign below - 031F)"},
/* 1AC9 */ { "COMBINING DOUBLE PLUS SIGN ABOVE",NULL},
/* 1ACA */ { "COMBINING DOUBLE PLUS SIGN BELOW",NULL},
/* 1ACB */ { "COMBINING TRIPLE ACUTE ACCENT","	x (combining double acute accent - 030B)"},
/* 1ACC */ { "COMBINING LATIN SMALL LETTER INSULAR G",NULL},
/* 1ACD */ { "COMBINING LATIN SMALL LETTER INSULAR R",NULL},
/* 1ACE */ { "COMBINING LATIN SMALL LETTER INSULAR T",NULL},
/* 1ACF */ { NULL,NULL},
/* 1AD0 */ { NULL,NULL},
/* 1AD1 */ { NULL,NULL},
/* 1AD2 */ { NULL,NULL},
/* 1AD3 */ { NULL,NULL},
/* 1AD4 */ { NULL,NULL},
/* 1AD5 */ { NULL,NULL},
/* 1AD6 */ { NULL,NULL},
/* 1AD7 */ { NULL,NULL},
/* 1AD8 */ { NULL,NULL},
/* 1AD9 */ { NULL,NULL},
/* 1ADA */ { NULL,NULL},
/* 1ADB */ { NULL,NULL},
/* 1ADC */ { NULL,NULL},
/* 1ADD */ { NULL,NULL},
/* 1ADE */ { NULL,NULL},
/* 1ADF */ { NULL,NULL},
/* 1AE0 */ { NULL,NULL},
/* 1AE1 */ { NULL,NULL},
/* 1AE2 */ { NULL,NULL},
/* 1AE3 */ { NULL,NULL},
/* 1AE4 */ { NULL,NULL},
/* 1AE5 */ { NULL,NULL},
/* 1AE6 */ { NULL,NULL},
/* 1AE7 */ { NULL,NULL},
/* 1AE8 */ { NULL,NULL},
/* 1AE9 */ { NULL,NULL},
/* 1AEA */ { NULL,NULL},
/* 1AEB */ { NULL,NULL},
/* 1AEC */ { NULL,NULL},
/* 1AED */ { NULL,NULL},
/* 1AEE */ { NULL,NULL},
/* 1AEF */ { NULL,NULL},
/* 1AF0 */ { NULL,NULL},
/* 1AF1 */ { NULL,NULL},
/* 1AF2 */ { NULL,NULL},
/* 1AF3 */ { NULL,NULL},
/* 1AF4 */ { NULL,NULL},
/* 1AF5 */ { NULL,NULL},
/* 1AF6 */ { NULL,NULL},
/* 1AF7 */ { NULL,NULL},
/* 1AF8 */ { NULL,NULL},
/* 1AF9 */ { NULL,NULL},
/* 1AFA */ { NULL,NULL},
/* 1AFB */ { NULL,NULL},
/* 1AFC */ { NULL,NULL},
/* 1AFD */ { NULL,NULL},
/* 1AFE */ { NULL,NULL},
/* 1AFF */ { NULL,NULL}
};

UN_DLL_LOCAL
static const struct unicode_nameannot una_00_1B[] = {
/* 1B00 */ { "BALINESE SIGN ULU RICEM","	= ardhacandra"},
/* 1B01 */ { "BALINESE SIGN ULU CANDRA","	= candrabindu"},
/* 1B02 */ { "BALINESE SIGN CECEK","	= anusvara"},
/* 1B03 */ { "BALINESE SIGN SURANG","	= final r\n"
	"	* also used for repha in transliteration of Kawi\n"
	"	x (javanese sign layar - A982)"},
/* 1B04 */ { "BALINESE SIGN BISAH","	= visarga"},
/* 1B05 */ { "BALINESE LETTER AKARA","	= a"},
/* 1B06 */ { "BALINESE LETTER AKARA TEDUNG","	= aa\n"
	"	: 1B05 1B35"},
/* 1B07 */ { "BALINESE LETTER IKARA","	= i"},
/* 1B08 */ { "BALINESE LETTER IKARA TEDUNG","	= ii\n"
	"	: 1B07 1B35"},
/* 1B09 */ { "BALINESE LETTER UKARA","	= u"},
/* 1B0A */ { "BALINESE LETTER UKARA TEDUNG","	= uu\n"
	"	: 1B09 1B35"},
/* 1B0B */ { "BALINESE LETTER RA REPA","	= vocalic r"},
/* 1B0C */ { "BALINESE LETTER RA REPA TEDUNG","	= vocalic rr\n"
	"	: 1B0B 1B35"},
/* 1B0D */ { "BALINESE LETTER LA LENGA","	= vocalic l"},
/* 1B0E */ { "BALINESE LETTER LA LENGA TEDUNG","	= vocalic ll\n"
	"	: 1B0D 1B35"},
/* 1B0F */ { "BALINESE LETTER EKARA","	= e"},
/* 1B10 */ { "BALINESE LETTER AIKARA","	= ai"},
/* 1B11 */ { "BALINESE LETTER OKARA","	= o"},
/* 1B12 */ { "BALINESE LETTER OKARA TEDUNG","	= au\n"
	"	: 1B11 1B35"},
/* 1B13 */ { "BALINESE LETTER KA",NULL},
/* 1B14 */ { "BALINESE LETTER KA MAHAPRANA","	= kha"},
/* 1B15 */ { "BALINESE LETTER GA",NULL},
/* 1B16 */ { "BALINESE LETTER GA GORA","	= gha"},
/* 1B17 */ { "BALINESE LETTER NGA",NULL},
/* 1B18 */ { "BALINESE LETTER CA",NULL},
/* 1B19 */ { "BALINESE LETTER CA LACA","	= cha"},
/* 1B1A */ { "BALINESE LETTER JA",NULL},
/* 1B1B */ { "BALINESE LETTER JA JERA","	= jha"},
/* 1B1C */ { "BALINESE LETTER NYA",NULL},
/* 1B1D */ { "BALINESE LETTER TA LATIK","	= tta"},
/* 1B1E */ { "BALINESE LETTER TA MURDA MAHAPRANA","	= ttha"},
/* 1B1F */ { "BALINESE LETTER DA MURDA ALPAPRANA","	= dda"},
/* 1B20 */ { "BALINESE LETTER DA MURDA MAHAPRANA","	= ddha"},
/* 1B21 */ { "BALINESE LETTER NA RAMBAT","	= nna"},
/* 1B22 */ { "BALINESE LETTER TA",NULL},
/* 1B23 */ { "BALINESE LETTER TA TAWA","	= tha"},
/* 1B24 */ { "BALINESE LETTER DA",NULL},
/* 1B25 */ { "BALINESE LETTER DA MADU","	= dha"},
/* 1B26 */ { "BALINESE LETTER NA",NULL},
/* 1B27 */ { "BALINESE LETTER PA",NULL},
/* 1B28 */ { "BALINESE LETTER PA KAPAL","	= pha"},
/* 1B29 */ { "BALINESE LETTER BA",NULL},
/* 1B2A */ { "BALINESE LETTER BA KEMBANG","	= bha"},
/* 1B2B */ { "BALINESE LETTER MA",NULL},
/* 1B2C */ { "BALINESE LETTER YA",NULL},
/* 1B2D */ { "BALINESE LETTER RA",NULL},
/* 1B2E */ { "BALINESE LETTER LA",NULL},
/* 1B2F */ { "BALINESE LETTER WA",NULL},
/* 1B30 */ { "BALINESE LETTER SA SAGA","	= sha"},
/* 1B31 */ { "BALINESE LETTER SA SAPA","	= ssa"},
/* 1B32 */ { "BALINESE LETTER SA",NULL},
/* 1B33 */ { "BALINESE LETTER HA",NULL},
/* 1B34 */ { "BALINESE SIGN REREKAN","	= nukta"},
/* 1B35 */ { "BALINESE VOWEL SIGN TEDUNG","	= aa"},
/* 1B36 */ { "BALINESE VOWEL SIGN ULU","	= i"},
/* 1B37 */ { "BALINESE VOWEL SIGN ULU SARI","	= ii"},
/* 1B38 */ { "BALINESE VOWEL SIGN SUKU","	= u"},
/* 1B39 */ { "BALINESE VOWEL SIGN SUKU ILUT","	= uu"},
/* 1B3A */ { "BALINESE VOWEL SIGN RA REPA","	= vocalic r"},
/* 1B3B */ { "BALINESE VOWEL SIGN RA REPA TEDUNG","	= vocalic rr\n"
	"	: 1B3A 1B35"},
/* 1B3C */ { "BALINESE VOWEL SIGN LA LENGA","	= vocalic l"},
/* 1B3D */ { "BALINESE VOWEL SIGN LA LENGA TEDUNG","	= vocalic ll\n"
	"	: 1B3C 1B35"},
/* 1B3E */ { "BALINESE VOWEL SIGN TALING","	= e"},
/* 1B3F */ { "BALINESE VOWEL SIGN TALING REPA","	= ai"},
/* 1B40 */ { "BALINESE VOWEL SIGN TALING TEDUNG","	= o\n"
	"	: 1B3E 1B35"},
/* 1B41 */ { "BALINESE VOWEL SIGN TALING REPA TEDUNG","	= au\n"
	"	: 1B3F 1B35"},
/* 1B42 */ { "BALINESE VOWEL SIGN PEPET","	= ae"},
/* 1B43 */ { "BALINESE VOWEL SIGN PEPET TEDUNG","	= oe\n"
	"	: 1B42 1B35"},
/* 1B44 */ { "BALINESE ADEG ADEG","	= virama"},
/* 1B45 */ { "BALINESE LETTER KAF SASAK",NULL},
/* 1B46 */ { "BALINESE LETTER KHOT SASAK",NULL},
/* 1B47 */ { "BALINESE LETTER TZIR SASAK",NULL},
/* 1B48 */ { "BALINESE LETTER EF SASAK",NULL},
/* 1B49 */ { "BALINESE LETTER VE SASAK",NULL},
/* 1B4A */ { "BALINESE LETTER ZAL SASAK",NULL},
/* 1B4B */ { "BALINESE LETTER ASYURA SASAK",NULL},
/* 1B4C */ { "BALINESE LETTER ARCHAIC JNYA","	* used in older texts in place of the ja + nya conjunct\n"
	"	x (javanese letter nya murda - A998)"},
/* 1B4D */ { NULL,NULL},
/* 1B4E */ { "BALINESE INVERTED CARIK SIKI",NULL},
/* 1B4F */ { "BALINESE INVERTED CARIK PAREREN",NULL},
/* 1B50 */ { "BALINESE DIGIT ZERO",NULL},
/* 1B51 */ { "BALINESE DIGIT ONE",NULL},
/* 1B52 */ { "BALINESE DIGIT TWO",NULL},
/* 1B53 */ { "BALINESE DIGIT THREE",NULL},
/* 1B54 */ { "BALINESE DIGIT FOUR",NULL},
/* 1B55 */ { "BALINESE DIGIT FIVE",NULL},
/* 1B56 */ { "BALINESE DIGIT SIX",NULL},
/* 1B57 */ { "BALINESE DIGIT SEVEN",NULL},
/* 1B58 */ { "BALINESE DIGIT EIGHT",NULL},
/* 1B59 */ { "BALINESE DIGIT NINE",NULL},
/* 1B5A */ { "BALINESE PANTI","	= section"},
/* 1B5B */ { "BALINESE PAMADA","	= honorific section"},
/* 1B5C */ { "BALINESE WINDU","	= punctuation ring"},
/* 1B5D */ { "BALINESE CARIK PAMUNGKAH","	= colon"},
/* 1B5E */ { "BALINESE CARIK SIKI","	= danda\n"
	"	x (devanagari danda - 0964)"},
/* 1B5F */ { "BALINESE CARIK PAREREN","	= double danda\n"
	"	x (devanagari double danda - 0965)"},
/* 1B60 */ { "BALINESE PAMENENG","	= line-breaking hyphen"},
/* 1B61 */ { "BALINESE MUSICAL SYMBOL DONG",NULL},
/* 1B62 */ { "BALINESE MUSICAL SYMBOL DENG",NULL},
/* 1B63 */ { "BALINESE MUSICAL SYMBOL DUNG",NULL},
/* 1B64 */ { "BALINESE MUSICAL SYMBOL DANG",NULL},
/* 1B65 */ { "BALINESE MUSICAL SYMBOL DANG SURANG",NULL},
/* 1B66 */ { "BALINESE MUSICAL SYMBOL DING",NULL},
/* 1B67 */ { "BALINESE MUSICAL SYMBOL DAENG",NULL},
/* 1B68 */ { "BALINESE MUSICAL SYMBOL DEUNG",NULL},
/* 1B69 */ { "BALINESE MUSICAL SYMBOL DAING",NULL},
/* 1B6A */ { "BALINESE MUSICAL SYMBOL DANG GEDE",NULL},
/* 1B6B */ { "BALINESE MUSICAL SYMBOL COMBINING TEGEH",NULL},
/* 1B6C */ { "BALINESE MUSICAL SYMBOL COMBINING ENDEP",NULL},
/* 1B6D */ { "BALINESE MUSICAL SYMBOL COMBINING KEMPUL",NULL},
/* 1B6E */ { "BALINESE MUSICAL SYMBOL COMBINING KEMPLI",NULL},
/* 1B6F */ { "BALINESE MUSICAL SYMBOL COMBINING JEGOGAN",NULL},
/* 1B70 */ { "BALINESE MUSICAL SYMBOL COMBINING KEMPUL WITH JEGOGAN",NULL},
/* 1B71 */ { "BALINESE MUSICAL SYMBOL COMBINING KEMPLI WITH JEGOGAN",NULL},
/* 1B72 */ { "BALINESE MUSICAL SYMBOL COMBINING BENDE",NULL},
/* 1B73 */ { "BALINESE MUSICAL SYMBOL COMBINING GONG",NULL},
/* 1B74 */ { "BALINESE MUSICAL SYMBOL RIGHT-HAND OPEN DUG",NULL},
/* 1B75 */ { "BALINESE MUSICAL SYMBOL RIGHT-HAND OPEN DAG",NULL},
/* 1B76 */ { "BALINESE MUSICAL SYMBOL RIGHT-HAND CLOSED TUK",NULL},
/* 1B77 */ { "BALINESE MUSICAL SYMBOL RIGHT-HAND CLOSED TAK",NULL},
/* 1B78 */ { "BALINESE MUSICAL SYMBOL LEFT-HAND OPEN PANG",NULL},
/* 1B79 */ { "BALINESE MUSICAL SYMBOL LEFT-HAND OPEN PUNG",NULL},
/* 1B7A */ { "BALINESE MUSICAL SYMBOL LEFT-HAND CLOSED PLAK",NULL},
/* 1B7B */ { "BALINESE MUSICAL SYMBOL LEFT-HAND CLOSED PLUK",NULL},
/* 1B7C */ { "BALINESE MUSICAL SYMBOL LEFT-HAND OPEN PING",NULL},
/* 1B7D */ { "BALINESE PANTI LANTANG","	= end of text"},
/* 1B7E */ { "BALINESE PAMADA LANTANG","	= end of text"},
/* 1B7F */ { "BALINESE PANTI BAWAK","	* truncated form of panti\n"
	"	x (balinese panti - 1B5A)"},
/* 1B80 */ { "SUNDANESE SIGN PANYECEK","	= anusvara"},
/* 1B81 */ { "SUNDANESE SIGN PANGLAYAR","	= final r"},
/* 1B82 */ { "SUNDANESE SIGN PANGWISAD","	= visarga"},
/* 1B83 */ { "SUNDANESE LETTER A",NULL},
/* 1B84 */ { "SUNDANESE LETTER I",NULL},
/* 1B85 */ { "SUNDANESE LETTER U",NULL},
/* 1B86 */ { "SUNDANESE LETTER AE",NULL},
/* 1B87 */ { "SUNDANESE LETTER O",NULL},
/* 1B88 */ { "SUNDANESE LETTER E",NULL},
/* 1B89 */ { "SUNDANESE LETTER EU",NULL},
/* 1B8A */ { "SUNDANESE LETTER KA",NULL},
/* 1B8B */ { "SUNDANESE LETTER QA",NULL},
/* 1B8C */ { "SUNDANESE LETTER GA",NULL},
/* 1B8D */ { "SUNDANESE LETTER NGA",NULL},
/* 1B8E */ { "SUNDANESE LETTER CA",NULL},
/* 1B8F */ { "SUNDANESE LETTER JA",NULL},
/* 1B90 */ { "SUNDANESE LETTER ZA",NULL},
/* 1B91 */ { "SUNDANESE LETTER NYA",NULL},
/* 1B92 */ { "SUNDANESE LETTER TA",NULL},
/* 1B93 */ { "SUNDANESE LETTER DA",NULL},
/* 1B94 */ { "SUNDANESE LETTER NA",NULL},
/* 1B95 */ { "SUNDANESE LETTER PA",NULL},
/* 1B96 */ { "SUNDANESE LETTER FA",NULL},
/* 1B97 */ { "SUNDANESE LETTER VA",NULL},
/* 1B98 */ { "SUNDANESE LETTER BA",NULL},
/* 1B99 */ { "SUNDANESE LETTER MA",NULL},
/* 1B9A */ { "SUNDANESE LETTER YA",NULL},
/* 1B9B */ { "SUNDANESE LETTER RA",NULL},
/* 1B9C */ { "SUNDANESE LETTER LA",NULL},
/* 1B9D */ { "SUNDANESE LETTER WA",NULL},
/* 1B9E */ { "SUNDANESE LETTER SA",NULL},
/* 1B9F */ { "SUNDANESE LETTER XA",NULL},
/* 1BA0 */ { "SUNDANESE LETTER HA",NULL},
/* 1BA1 */ { "SUNDANESE CONSONANT SIGN PAMINGKAL","	= subjoined ya"},
/* 1BA2 */ { "SUNDANESE CONSONANT SIGN PANYAKRA","	= subjoined ra"},
/* 1BA3 */ { "SUNDANESE CONSONANT SIGN PANYIKU","	= subjoined la"},
/* 1BA4 */ { "SUNDANESE VOWEL SIGN PANGHULU","	= i"},
/* 1BA5 */ { "SUNDANESE VOWEL SIGN PANYUKU","	= u"},
/* 1BA6 */ { "SUNDANESE VOWEL SIGN PANAELAENG","	= ae"},
/* 1BA7 */ { "SUNDANESE VOWEL SIGN PANOLONG","	= o"},
/* 1BA8 */ { "SUNDANESE VOWEL SIGN PAMEPET","	= e"},
/* 1BA9 */ { "SUNDANESE VOWEL SIGN PANEULEUNG","	= eu"},
/* 1BAA */ { "SUNDANESE SIGN PAMAAEH","	= virama\n"
	"	* does not form conjuncts"},
/* 1BAB */ { "SUNDANESE SIGN VIRAMA","	* forms conjuncts in older orthography"},
/* 1BAC */ { "SUNDANESE CONSONANT SIGN PASANGAN MA","	= subjoined ma"},
/* 1BAD */ { "SUNDANESE CONSONANT SIGN PASANGAN WA","	= subjoined wa"},
/* 1BAE */ { "SUNDANESE LETTER KHA",NULL},
/* 1BAF */ { "SUNDANESE LETTER SYA",NULL},
/* 1BB0 */ { "SUNDANESE DIGIT ZERO",NULL},
/* 1BB1 */ { "SUNDANESE DIGIT ONE",NULL},
/* 1BB2 */ { "SUNDANESE DIGIT TWO",NULL},
/* 1BB3 */ { "SUNDANESE DIGIT THREE",NULL},
/* 1BB4 */ { "SUNDANESE DIGIT FOUR",NULL},
/* 1BB5 */ { "SUNDANESE DIGIT FIVE",NULL},
/* 1BB6 */ { "SUNDANESE DIGIT SIX",NULL},
/* 1BB7 */ { "SUNDANESE DIGIT SEVEN",NULL},
/* 1BB8 */ { "SUNDANESE DIGIT EIGHT",NULL},
/* 1BB9 */ { "SUNDANESE DIGIT NINE",NULL},
/* 1BBA */ { "SUNDANESE AVAGRAHA","	= gemination mark"},
/* 1BBB */ { "SUNDANESE LETTER REU","	* vocalic r"},
/* 1BBC */ { "SUNDANESE LETTER LEU","	* vocalic l"},
/* 1BBD */ { "SUNDANESE LETTER BHA","	% SUNDANESE LETTER ARCHAIC I\n"
	"	* character was originally misidentified, so its name is a misnomer"},
/* 1BBE */ { "SUNDANESE LETTER FINAL K",NULL},
/* 1BBF */ { "SUNDANESE LETTER FINAL M","	* used in a 21st century document\n"
	"	* for actual final m, use the sequence 1B99 1BAA"},
/* 1BC0 */ { "BATAK LETTER A","	* letter a or ha for Karo and Pakpak"},
/* 1BC1 */ { "BATAK LETTER SIMALUNGUN A",NULL},
/* 1BC2 */ { "BATAK LETTER HA","	* Toba letter ha or ka\n"
	"	* also known as letter ka for Karo and Pakpak"},
/* 1BC3 */ { "BATAK LETTER SIMALUNGUN HA","	* also known as Simalungun letter ka"},
/* 1BC4 */ { "BATAK LETTER MANDAILING HA",NULL},
/* 1BC5 */ { "BATAK LETTER BA",NULL},
/* 1BC6 */ { "BATAK LETTER KARO BA","	* also used in Toba"},
/* 1BC7 */ { "BATAK LETTER PA",NULL},
/* 1BC8 */ { "BATAK LETTER SIMALUNGUN PA",NULL},
/* 1BC9 */ { "BATAK LETTER NA",NULL},
/* 1BCA */ { "BATAK LETTER MANDAILING NA",NULL},
/* 1BCB */ { "BATAK LETTER WA","	* Toba letter wa can be represented by either 1BCB or 1BCD"},
/* 1BCC */ { "BATAK LETTER SIMALUNGUN WA",NULL},
/* 1BCD */ { "BATAK LETTER PAKPAK WA",NULL},
/* 1BCE */ { "BATAK LETTER GA",NULL},
/* 1BCF */ { "BATAK LETTER SIMALUNGUN GA",NULL},
/* 1BD0 */ { "BATAK LETTER JA",NULL},
/* 1BD1 */ { "BATAK LETTER DA",NULL},
/* 1BD2 */ { "BATAK LETTER RA",NULL},
/* 1BD3 */ { "BATAK LETTER SIMALUNGUN RA",NULL},
/* 1BD4 */ { "BATAK LETTER MA",NULL},
/* 1BD5 */ { "BATAK LETTER SIMALUNGUN MA",NULL},
/* 1BD6 */ { "BATAK LETTER SOUTHERN TA","	* Toba letter ta can be represented by either 1BD6 or 1BD7"},
/* 1BD7 */ { "BATAK LETTER NORTHERN TA",NULL},
/* 1BD8 */ { "BATAK LETTER SA","	* Pakpak letter sa or ca"},
/* 1BD9 */ { "BATAK LETTER SIMALUNGUN SA",NULL},
/* 1BDA */ { "BATAK LETTER MANDAILING SA",NULL},
/* 1BDB */ { "BATAK LETTER YA",NULL},
/* 1BDC */ { "BATAK LETTER SIMALUNGUN YA",NULL},
/* 1BDD */ { "BATAK LETTER NGA",NULL},
/* 1BDE */ { "BATAK LETTER LA",NULL},
/* 1BDF */ { "BATAK LETTER SIMALUNGUN LA",NULL},
/* 1BE0 */ { "BATAK LETTER NYA","	* Karo letter ca can be represented by either 1BE0 or 1BE1"},
/* 1BE1 */ { "BATAK LETTER CA",NULL},
/* 1BE2 */ { "BATAK LETTER NDA",NULL},
/* 1BE3 */ { "BATAK LETTER MBA",NULL},
/* 1BE4 */ { "BATAK LETTER I",NULL},
/* 1BE5 */ { "BATAK LETTER U",NULL},
/* 1BE6 */ { "BATAK SIGN TOMPI",NULL},
/* 1BE7 */ { "BATAK VOWEL SIGN E","	= kebereten"},
/* 1BE8 */ { "BATAK VOWEL SIGN PAKPAK E","	= ketolongen"},
/* 1BE9 */ { "BATAK VOWEL SIGN EE","	= talinga"},
/* 1BEA */ { "BATAK VOWEL SIGN I","	= ulua"},
/* 1BEB */ { "BATAK VOWEL SIGN KARO I","	= haluan"},
/* 1BEC */ { "BATAK VOWEL SIGN O","	= siala ulu"},
/* 1BED */ { "BATAK VOWEL SIGN KARO O","	= hatulungan"},
/* 1BEE */ { "BATAK VOWEL SIGN U","	= boruta"},
/* 1BEF */ { "BATAK VOWEL SIGN U FOR SIMALUNGUN SA","	= haboritan for simalungun sa"},
/* 1BF0 */ { "BATAK CONSONANT SIGN NG","	= amisara"},
/* 1BF1 */ { "BATAK CONSONANT SIGN H","	= hajoringan"},
/* 1BF2 */ { "BATAK PANGOLAT","	= virama\n"
	"	* cannot follow a vowel sign"},
/* 1BF3 */ { "BATAK PANONGONAN","	= virama"},
/* 1BF4 */ { NULL,NULL},
/* 1BF5 */ { NULL,NULL},
/* 1BF6 */ { NULL,NULL},
/* 1BF7 */ { NULL,NULL},
/* 1BF8 */ { NULL,NULL},
/* 1BF9 */ { NULL,NULL},
/* 1BFA */ { NULL,NULL},
/* 1BFB */ { NULL,NULL},
/* 1BFC */ { "BATAK SYMBOL BINDU NA METEK",NULL},
/* 1BFD */ { "BATAK SYMBOL BINDU PINARBORAS",NULL},
/* 1BFE */ { "BATAK SYMBOL BINDU JUDUL",NULL},
/* 1BFF */ { "BATAK SYMBOL BINDU PANGOLAT",NULL}
};

UN_DLL_LOCAL
static const struct unicode_nameannot una_00_1C[] = {
/* 1C00 */ { "LEPCHA LETTER KA",NULL},
/* 1C01 */ { "LEPCHA LETTER KLA",NULL},
/* 1C02 */ { "LEPCHA LETTER KHA",NULL},
/* 1C03 */ { "LEPCHA LETTER GA",NULL},
/* 1C04 */ { "LEPCHA LETTER GLA",NULL},
/* 1C05 */ { "LEPCHA LETTER NGA",NULL},
/* 1C06 */ { "LEPCHA LETTER CA",NULL},
/* 1C07 */ { "LEPCHA LETTER CHA",NULL},
/* 1C08 */ { "LEPCHA LETTER JA",NULL},
/* 1C09 */ { "LEPCHA LETTER NYA",NULL},
/* 1C0A */ { "LEPCHA LETTER TA",NULL},
/* 1C0B */ { "LEPCHA LETTER THA",NULL},
/* 1C0C */ { "LEPCHA LETTER DA",NULL},
/* 1C0D */ { "LEPCHA LETTER NA",NULL},
/* 1C0E */ { "LEPCHA LETTER PA",NULL},
/* 1C0F */ { "LEPCHA LETTER PLA",NULL},
/* 1C10 */ { "LEPCHA LETTER PHA",NULL},
/* 1C11 */ { "LEPCHA LETTER FA",NULL},
/* 1C12 */ { "LEPCHA LETTER FLA",NULL},
/* 1C13 */ { "LEPCHA LETTER BA",NULL},
/* 1C14 */ { "LEPCHA LETTER BLA",NULL},
/* 1C15 */ { "LEPCHA LETTER MA",NULL},
/* 1C16 */ { "LEPCHA LETTER MLA",NULL},
/* 1C17 */ { "LEPCHA LETTER TSA",NULL},
/* 1C18 */ { "LEPCHA LETTER TSHA",NULL},
/* 1C19 */ { "LEPCHA LETTER DZA",NULL},
/* 1C1A */ { "LEPCHA LETTER YA",NULL},
/* 1C1B */ { "LEPCHA LETTER RA",NULL},
/* 1C1C */ { "LEPCHA LETTER LA",NULL},
/* 1C1D */ { "LEPCHA LETTER HA",NULL},
/* 1C1E */ { "LEPCHA LETTER HLA",NULL},
/* 1C1F */ { "LEPCHA LETTER VA",NULL},
/* 1C20 */ { "LEPCHA LETTER SA",NULL},
/* 1C21 */ { "LEPCHA LETTER SHA",NULL},
/* 1C22 */ { "LEPCHA LETTER WA",NULL},
/* 1C23 */ { "LEPCHA LETTER A",NULL},
/* 1C24 */ { "LEPCHA SUBJOINED LETTER YA",NULL},
/* 1C25 */ { "LEPCHA SUBJOINED LETTER RA",NULL},
/* 1C26 */ { "LEPCHA VOWEL SIGN AA",NULL},
/* 1C27 */ { "LEPCHA VOWEL SIGN I",NULL},
/* 1C28 */ { "LEPCHA VOWEL SIGN O",NULL},
/* 1C29 */ { "LEPCHA VOWEL SIGN OO",NULL},
/* 1C2A */ { "LEPCHA VOWEL SIGN U",NULL},
/* 1C2B */ { "LEPCHA VOWEL SIGN UU",NULL},
/* 1C2C */ { "LEPCHA VOWEL SIGN E",NULL},
/* 1C2D */ { "LEPCHA CONSONANT SIGN K",NULL},
/* 1C2E */ { "LEPCHA CONSONANT SIGN M",NULL},
/* 1C2F */ { "LEPCHA CONSONANT SIGN L",NULL},
/* 1C30 */ { "LEPCHA CONSONANT SIGN N",NULL},
/* 1C31 */ { "LEPCHA CONSONANT SIGN P",NULL},
/* 1C32 */ { "LEPCHA CONSONANT SIGN R",NULL},
/* 1C33 */ { "LEPCHA CONSONANT SIGN T",NULL},
/* 1C34 */ { "LEPCHA CONSONANT SIGN NYIN-DO",NULL},
/* 1C35 */ { "LEPCHA CONSONANT SIGN KANG",NULL},
/* 1C36 */ { "LEPCHA SIGN RAN",NULL},
/* 1C37 */ { "LEPCHA SIGN NUKTA",NULL},
/* 1C38 */ { NULL,NULL},
/* 1C39 */ { NULL,NULL},
/* 1C3A */ { NULL,NULL},
/* 1C3B */ { "LEPCHA PUNCTUATION TA-ROL",NULL},
/* 1C3C */ { "LEPCHA PUNCTUATION NYET THYOOM TA-ROL",NULL},
/* 1C3D */ { "LEPCHA PUNCTUATION CER-WA",NULL},
/* 1C3E */ { "LEPCHA PUNCTUATION TSHOOK CER-WA",NULL},
/* 1C3F */ { "LEPCHA PUNCTUATION TSHOOK",NULL},
/* 1C40 */ { "LEPCHA DIGIT ZERO",NULL},
/* 1C41 */ { "LEPCHA DIGIT ONE",NULL},
/* 1C42 */ { "LEPCHA DIGIT TWO",NULL},
/* 1C43 */ { "LEPCHA DIGIT THREE",NULL},
/* 1C44 */ { "LEPCHA DIGIT FOUR",NULL},
/* 1C45 */ { "LEPCHA DIGIT FIVE",NULL},
/* 1C46 */ { "LEPCHA DIGIT SIX",NULL},
/* 1C47 */ { "LEPCHA DIGIT SEVEN",NULL},
/* 1C48 */ { "LEPCHA DIGIT EIGHT",NULL},
/* 1C49 */ { "LEPCHA DIGIT NINE",NULL},
/* 1C4A */ { NULL,NULL},
/* 1C4B */ { NULL,NULL},
/* 1C4C */ { NULL,NULL},
/* 1C4D */ { "LEPCHA LETTER TTA",NULL},
/* 1C4E */ { "LEPCHA LETTER TTHA",NULL},
/* 1C4F */ { "LEPCHA LETTER DDA",NULL},
/* 1C50 */ { "OL CHIKI DIGIT ZERO",NULL},
/* 1C51 */ { "OL CHIKI DIGIT ONE",NULL},
/* 1C52 */ { "OL CHIKI DIGIT TWO",NULL},
/* 1C53 */ { "OL CHIKI DIGIT THREE",NULL},
/* 1C54 */ { "OL CHIKI DIGIT FOUR",NULL},
/* 1C55 */ { "OL CHIKI DIGIT FIVE",NULL},
/* 1C56 */ { "OL CHIKI DIGIT SIX",NULL},
/* 1C57 */ { "OL CHIKI DIGIT SEVEN",NULL},
/* 1C58 */ { "OL CHIKI DIGIT EIGHT",NULL},
/* 1C59 */ { "OL CHIKI DIGIT NINE",NULL},
/* 1C5A */ { "OL CHIKI LETTER LA",NULL},
/* 1C5B */ { "OL CHIKI LETTER AT",NULL},
/* 1C5C */ { "OL CHIKI LETTER AG",NULL},
/* 1C5D */ { "OL CHIKI LETTER ANG",NULL},
/* 1C5E */ { "OL CHIKI LETTER AL",NULL},
/* 1C5F */ { "OL CHIKI LETTER LAA",NULL},
/* 1C60 */ { "OL CHIKI LETTER AAK",NULL},
/* 1C61 */ { "OL CHIKI LETTER AAJ",NULL},
/* 1C62 */ { "OL CHIKI LETTER AAM",NULL},
/* 1C63 */ { "OL CHIKI LETTER AAW",NULL},
/* 1C64 */ { "OL CHIKI LETTER LI",NULL},
/* 1C65 */ { "OL CHIKI LETTER IS",NULL},
/* 1C66 */ { "OL CHIKI LETTER IH",NULL},
/* 1C67 */ { "OL CHIKI LETTER INY",NULL},
/* 1C68 */ { "OL CHIKI LETTER IR",NULL},
/* 1C69 */ { "OL CHIKI LETTER LU",NULL},
/* 1C6A */ { "OL CHIKI LETTER UC",NULL},
/* 1C6B */ { "OL CHIKI LETTER UD",NULL},
/* 1C6C */ { "OL CHIKI LETTER UNN",NULL},
/* 1C6D */ { "OL CHIKI LETTER UY",NULL},
/* 1C6E */ { "OL CHIKI LETTER LE",NULL},
/* 1C6F */ { "OL CHIKI LETTER EP",NULL},
/* 1C70 */ { "OL CHIKI LETTER EDD",NULL},
/* 1C71 */ { "OL CHIKI LETTER EN",NULL},
/* 1C72 */ { "OL CHIKI LETTER ERR",NULL},
/* 1C73 */ { "OL CHIKI LETTER LO",NULL},
/* 1C74 */ { "OL CHIKI LETTER OTT",NULL},
/* 1C75 */ { "OL CHIKI LETTER OB",NULL},
/* 1C76 */ { "OL CHIKI LETTER OV",NULL},
/* 1C77 */ { "OL CHIKI LETTER OH",NULL},
/* 1C78 */ { "OL CHIKI MU TTUDDAG","	* nasalization mark"},
/* 1C79 */ { "OL CHIKI GAAHLAA TTUDDAAG","	* vowel modifier"},
/* 1C7A */ { "OL CHIKI MU-GAAHLAA TTUDDAAG",NULL},
/* 1C7B */ { "OL CHIKI RELAA","	* length mark"},
/* 1C7C */ { "OL CHIKI PHAARKAA","	* glottal protector"},
/* 1C7D */ { "OL CHIKI AHAD","	* deglottalizes consonants"},
/* 1C7E */ { "OL CHIKI PUNCTUATION MUCAAD",NULL},
/* 1C7F */ { "OL CHIKI PUNCTUATION DOUBLE MUCAAD",NULL},
/* 1C80 */ { "CYRILLIC SMALL LETTER ROUNDED VE","	x (cyrillic small letter ve - 0432)"},
/* 1C81 */ { "CYRILLIC SMALL LETTER LONG-LEGGED DE","	x (cyrillic small letter de - 0434)"},
/* 1C82 */ { "CYRILLIC SMALL LETTER NARROW O","	x (cyrillic small letter o - 043E)"},
/* 1C83 */ { "CYRILLIC SMALL LETTER WIDE ES","	x (cyrillic small letter es - 0441)"},
/* 1C84 */ { "CYRILLIC SMALL LETTER TALL TE",NULL},
/* 1C85 */ { "CYRILLIC SMALL LETTER THREE-LEGGED TE","	x (cyrillic small letter te - 0442)"},
/* 1C86 */ { "CYRILLIC SMALL LETTER TALL HARD SIGN","	x (cyrillic small letter hard sign - 044A)"},
/* 1C87 */ { "CYRILLIC SMALL LETTER TALL YAT","	x (cyrillic small letter yat - 0463)"},
/* 1C88 */ { "CYRILLIC SMALL LETTER UNBLENDED UK","	x (cyrillic small letter uk - 0479)\n"
	"	x (cyrillic small letter monograph uk - A64B)"},
/* 1C89 */ { "CYRILLIC CAPITAL LETTER TJE",NULL},
/* 1C8A */ { "CYRILLIC SMALL LETTER TJE","	x (cyrillic small letter komi tje - 050F)"},
/* 1C8B */ { NULL,NULL},
/* 1C8C */ { NULL,NULL},
/* 1C8D */ { NULL,NULL},
/* 1C8E */ { NULL,NULL},
/* 1C8F */ { NULL,NULL},
/* 1C90 */ { "GEORGIAN MTAVRULI CAPITAL LETTER AN","	x (georgian letter an - 10D0)"},
/* 1C91 */ { "GEORGIAN MTAVRULI CAPITAL LETTER BAN",NULL},
/* 1C92 */ { "GEORGIAN MTAVRULI CAPITAL LETTER GAN",NULL},
/* 1C93 */ { "GEORGIAN MTAVRULI CAPITAL LETTER DON",NULL},
/* 1C94 */ { "GEORGIAN MTAVRULI CAPITAL LETTER EN",NULL},
/* 1C95 */ { "GEORGIAN MTAVRULI CAPITAL LETTER VIN",NULL},
/* 1C96 */ { "GEORGIAN MTAVRULI CAPITAL LETTER ZEN",NULL},
/* 1C97 */ { "GEORGIAN MTAVRULI CAPITAL LETTER TAN",NULL},
/* 1C98 */ { "GEORGIAN MTAVRULI CAPITAL LETTER IN",NULL},
/* 1C99 */ { "GEORGIAN MTAVRULI CAPITAL LETTER KAN",NULL},
/* 1C9A */ { "GEORGIAN MTAVRULI CAPITAL LETTER LAS",NULL},
/* 1C9B */ { "GEORGIAN MTAVRULI CAPITAL LETTER MAN",NULL},
/* 1C9C */ { "GEORGIAN MTAVRULI CAPITAL LETTER NAR",NULL},
/* 1C9D */ { "GEORGIAN MTAVRULI CAPITAL LETTER ON",NULL},
/* 1C9E */ { "GEORGIAN MTAVRULI CAPITAL LETTER PAR",NULL},
/* 1C9F */ { "GEORGIAN MTAVRULI CAPITAL LETTER ZHAR",NULL},
/* 1CA0 */ { "GEORGIAN MTAVRULI CAPITAL LETTER RAE",NULL},
/* 1CA1 */ { "GEORGIAN MTAVRULI CAPITAL LETTER SAN",NULL},
/* 1CA2 */ { "GEORGIAN MTAVRULI CAPITAL LETTER TAR",NULL},
/* 1CA3 */ { "GEORGIAN MTAVRULI CAPITAL LETTER UN",NULL},
/* 1CA4 */ { "GEORGIAN MTAVRULI CAPITAL LETTER PHAR",NULL},
/* 1CA5 */ { "GEORGIAN MTAVRULI CAPITAL LETTER KHAR",NULL},
/* 1CA6 */ { "GEORGIAN MTAVRULI CAPITAL LETTER GHAN",NULL},
/* 1CA7 */ { "GEORGIAN MTAVRULI CAPITAL LETTER QAR",NULL},
/* 1CA8 */ { "GEORGIAN MTAVRULI CAPITAL LETTER SHIN",NULL},
/* 1CA9 */ { "GEORGIAN MTAVRULI CAPITAL LETTER CHIN",NULL},
/* 1CAA */ { "GEORGIAN MTAVRULI CAPITAL LETTER CAN",NULL},
/* 1CAB */ { "GEORGIAN MTAVRULI CAPITAL LETTER JIL",NULL},
/* 1CAC */ { "GEORGIAN MTAVRULI CAPITAL LETTER CIL",NULL},
/* 1CAD */ { "GEORGIAN MTAVRULI CAPITAL LETTER CHAR",NULL},
/* 1CAE */ { "GEORGIAN MTAVRULI CAPITAL LETTER XAN",NULL},
/* 1CAF */ { "GEORGIAN MTAVRULI CAPITAL LETTER JHAN",NULL},
/* 1CB0 */ { "GEORGIAN MTAVRULI CAPITAL LETTER HAE",NULL},
/* 1CB1 */ { "GEORGIAN MTAVRULI CAPITAL LETTER HE",NULL},
/* 1CB2 */ { "GEORGIAN MTAVRULI CAPITAL LETTER HIE",NULL},
/* 1CB3 */ { "GEORGIAN MTAVRULI CAPITAL LETTER WE",NULL},
/* 1CB4 */ { "GEORGIAN MTAVRULI CAPITAL LETTER HAR",NULL},
/* 1CB5 */ { "GEORGIAN MTAVRULI CAPITAL LETTER HOE",NULL},
/* 1CB6 */ { "GEORGIAN MTAVRULI CAPITAL LETTER FI",NULL},
/* 1CB7 */ { "GEORGIAN MTAVRULI CAPITAL LETTER YN",NULL},
/* 1CB8 */ { "GEORGIAN MTAVRULI CAPITAL LETTER ELIFI",NULL},
/* 1CB9 */ { "GEORGIAN MTAVRULI CAPITAL LETTER TURNED GAN",NULL},
/* 1CBA */ { "GEORGIAN MTAVRULI CAPITAL LETTER AIN",NULL},
/* 1CBB */ { NULL,NULL},
/* 1CBC */ { NULL,NULL},
/* 1CBD */ { "GEORGIAN MTAVRULI CAPITAL LETTER AEN",NULL},
/* 1CBE */ { "GEORGIAN MTAVRULI CAPITAL LETTER HARD SIGN",NULL},
/* 1CBF */ { "GEORGIAN MTAVRULI CAPITAL LETTER LABIAL SIGN",NULL},
/* 1CC0 */ { "SUNDANESE PUNCTUATION BINDU SURYA","	* sun"},
/* 1CC1 */ { "SUNDANESE PUNCTUATION BINDU PANGLONG","	* half moon"},
/* 1CC2 */ { "SUNDANESE PUNCTUATION BINDU PURNAMA","	* full moon"},
/* 1CC3 */ { "SUNDANESE PUNCTUATION BINDU CAKRA","	* wheel"},
/* 1CC4 */ { "SUNDANESE PUNCTUATION BINDU LEU SATANGA",NULL},
/* 1CC5 */ { "SUNDANESE PUNCTUATION BINDU KA SATANGA",NULL},
/* 1CC6 */ { "SUNDANESE PUNCTUATION BINDU DA SATANGA",NULL},
/* 1CC7 */ { "SUNDANESE PUNCTUATION BINDU BA SATANGA",NULL},
/* 1CC8 */ { NULL,NULL},
/* 1CC9 */ { NULL,NULL},
/* 1CCA */ { NULL,NULL},
/* 1CCB */ { NULL,NULL},
/* 1CCC */ { NULL,NULL},
/* 1CCD */ { NULL,NULL},
/* 1CCE */ { NULL,NULL},
/* 1CCF */ { NULL,NULL},
/* 1CD0 */ { "VEDIC TONE KARSHANA","	= vaidika saamasvara karshanna"},
/* 1CD1 */ { "VEDIC TONE SHARA","	= vaidika svarita uurdhva shara"},
/* 1CD2 */ { "VEDIC TONE PRENKHA","	* indicates vibrato\n"
	"	= vaidika saamasvara prenkha"},
/* 1CD3 */ { "VEDIC SIGN NIHSHVASA","	= vaidika saamagaana yogakaala\n"
	"	* separates sections between which a pause is disallowed"},
/* 1CD4 */ { "VEDIC SIGN YAJURVEDIC MIDLINE SVARITA","	* used predominantly in the Maitrayani Samhita and in some manuscripts of the Vajasaneyi Madhyandina Samhita"},
/* 1CD5 */ { "VEDIC TONE YAJURVEDIC AGGRAVATED INDEPENDENT SVARITA","	= vaidika svarita adho nyubja"},
/* 1CD6 */ { "VEDIC TONE YAJURVEDIC INDEPENDENT SVARITA","	= vaidika svarita adhah konna"},
/* 1CD7 */ { "VEDIC TONE YAJURVEDIC KATHAKA INDEPENDENT SVARITA","	= vaidika svarita adho vakra rekhaa"},
/* 1CD8 */ { "VEDIC TONE CANDRA BELOW","	= vaidika svarita adho'rdha vakra"},
/* 1CD9 */ { "VEDIC TONE YAJURVEDIC KATHAKA INDEPENDENT SVARITA SCHROEDER","	= vaidika svarita adhah samyukta rekhaa"},
/* 1CDA */ { "VEDIC TONE DOUBLE SVARITA","	= vaidika svarita uurdhva dvi rekhaa\n"
	"	x (devanagari stress sign udatta - 0951)"},
/* 1CDB */ { "VEDIC TONE TRIPLE SVARITA","	= vaidika svarita uurdhva tri rekhaa"},
/* 1CDC */ { "VEDIC TONE KATHAKA ANUDATTA","	= vaidika svarita adho rekhaa\n"
	"	x (devanagari stress sign anudatta - 0952)"},
/* 1CDD */ { "VEDIC TONE DOT BELOW","	= vaidika svarita adho bindu"},
/* 1CDE */ { "VEDIC TONE TWO DOTS BELOW","	= vaidika svarita adho dvi bindu"},
/* 1CDF */ { "VEDIC TONE THREE DOTS BELOW","	= vaidika svarita adhas tri bindu"},
/* 1CE0 */ { "VEDIC TONE RIGVEDIC KASHMIRI INDEPENDENT SVARITA","	= vaidika uurdhva vakra rekhaa"},
/* 1CE1 */ { "VEDIC TONE ATHARVAVEDIC INDEPENDENT SVARITA","	= vaidika svarita dvi vakra khannda"},
/* 1CE2 */ { "VEDIC SIGN VISARGA SVARITA","	= vaidika madhyarekhaa"},
/* 1CE3 */ { "VEDIC SIGN VISARGA UDATTA","	= vaidika visarga dakshinnatah uurdhvaga"},
/* 1CE4 */ { "VEDIC SIGN REVERSED VISARGA UDATTA","	= vaidika visarga vaamatah uurdhvaga"},
/* 1CE5 */ { "VEDIC SIGN VISARGA ANUDATTA","	= vaidika visarga vaamatah adhoga"},
/* 1CE6 */ { "VEDIC SIGN REVERSED VISARGA ANUDATTA","	= vaidika visarga dakshinnatah adhoga"},
/* 1CE7 */ { "VEDIC SIGN VISARGA UDATTA WITH TAIL","	= vaidika visarga dakshinnatah uurdhva vakra"},
/* 1CE8 */ { "VEDIC SIGN VISARGA ANUDATTA WITH TAIL","	= vaidika visarga vaamatah adho vakra"},
/* 1CE9 */ { "VEDIC SIGN ANUSVARA ANTARGOMUKHA","	= vaidika anusvaara antarmukha"},
/* 1CEA */ { "VEDIC SIGN ANUSVARA BAHIRGOMUKHA","	= vaidika anusvaara naagaphanna"},
/* 1CEB */ { "VEDIC SIGN ANUSVARA VAMAGOMUKHA","	= vaidika anusvaara vaamagomukha"},
/* 1CEC */ { "VEDIC SIGN ANUSVARA VAMAGOMUKHA WITH TAIL","	= vaidika anusvaara vaamagomukha sa-vakra"},
/* 1CED */ { "VEDIC SIGN TIRYAK","	= vaidika tiryak"},
/* 1CEE */ { "VEDIC SIGN HEXIFORM LONG ANUSVARA","	= vaidika anusvaara anugaamii"},
/* 1CEF */ { "VEDIC SIGN LONG ANUSVARA","	= vaidika anusvaara dakshinnamukha"},
/* 1CF0 */ { "VEDIC SIGN RTHANG LONG ANUSVARA","	= vaidika anusvaara ttha-sadrisha"},
/* 1CF1 */ { "VEDIC SIGN ANUSVARA UBHAYATO MUKHA","	= vaidika anusvaara ubhayato mukha"},
/* 1CF2 */ { "VEDIC SIGN ARDHAVISARGA","	= vaidika jihvaamuuliiya upadhmaaniiya"},
/* 1CF3 */ { "VEDIC SIGN ROTATED ARDHAVISARGA",NULL},
/* 1CF4 */ { "VEDIC TONE CANDRA ABOVE",NULL},
/* 1CF5 */ { "VEDIC SIGN JIHVAMULIYA","	* marks a velar fricative occurring only before unvoiced velar stops\n"
	"	x (kannada sign jihvamuliya - 0CF1)\n"
	"	x (tibetan sign lce tsa can - 0F88)\n"
	"	x (brahmi sign jihvamuliya - 11003)\n"
	"	x (sharada sign jihvamuliya - 111C2)\n"
	"	x (newa sign jihvamuliya - 11460)\n"
	"	x (soyombo sign jihvamuliya - 11A84)"},
/* 1CF6 */ { "VEDIC SIGN UPADHMANIYA","	* marks a bilabial fricative occurring only before unvoiced labial stops\n"
	"	x (kannada sign upadhmaniya - 0CF2)\n"
	"	x (tibetan sign mchu can - 0F89)\n"
	"	x (brahmi sign upadhmaniya - 11004)\n"
	"	x (sharada sign upadhmaniya - 111C3)\n"
	"	x (newa sign upadhmaniya - 11461)\n"
	"	x (soyombo sign upadhmaniya - 11A85)"},
/* 1CF7 */ { "VEDIC SIGN ATIKRAMA","	= vaidika saamagaana atikrama\n"
	"	* indicates skipping of a svara"},
/* 1CF8 */ { "VEDIC TONE RING ABOVE",NULL},
/* 1CF9 */ { "VEDIC TONE DOUBLE RING ABOVE",NULL},
/* 1CFA */ { "VEDIC SIGN DOUBLE ANUSVARA ANTARGOMUKHA","	* used as a base for a combining nasal sign"},
/* 1CFB */ { NULL,NULL},
/* 1CFC */ { NULL,NULL},
/* 1CFD */ { NULL,NULL},
/* 1CFE */ { NULL,NULL},
/* 1CFF */ { NULL,NULL}
};

UN_DLL_LOCAL
static const struct unicode_nameannot una_00_1D[] = {
/* 1D00 */ { "LATIN LETTER SMALL CAPITAL A",NULL},
/* 1D01 */ { "LATIN LETTER SMALL CAPITAL AE",NULL},
/* 1D02 */ { "LATIN SMALL LETTER TURNED AE","	* glyph can also have sideways orientation"},
/* 1D03 */ { "LATIN LETTER SMALL CAPITAL BARRED B","	x (latin capital letter b with stroke - 0243)"},
/* 1D04 */ { "LATIN LETTER SMALL CAPITAL C",NULL},
/* 1D05 */ { "LATIN LETTER SMALL CAPITAL D",NULL},
/* 1D06 */ { "LATIN LETTER SMALL CAPITAL ETH",NULL},
/* 1D07 */ { "LATIN LETTER SMALL CAPITAL E",NULL},
/* 1D08 */ { "LATIN SMALL LETTER TURNED OPEN E","	x (latin small letter reversed open e - 025C)\n"
	"	x (modifier letter small turned open e - 1D4C)"},
/* 1D09 */ { "LATIN SMALL LETTER TURNED I",NULL},
/* 1D0A */ { "LATIN LETTER SMALL CAPITAL J",NULL},
/* 1D0B */ { "LATIN LETTER SMALL CAPITAL K","	x (latin small letter kra - 0138)\n"
	"	x (cyrillic small letter ka - 043A)"},
/* 1D0C */ { "LATIN LETTER SMALL CAPITAL L WITH STROKE",NULL},
/* 1D0D */ { "LATIN LETTER SMALL CAPITAL M",NULL},
/* 1D0E */ { "LATIN LETTER SMALL CAPITAL REVERSED N","	x (cyrillic small letter i - 0438)\n"
	"	x (modifier letter capital reversed n - 1D3B)"},
/* 1D0F */ { "LATIN LETTER SMALL CAPITAL O",NULL},
/* 1D10 */ { "LATIN LETTER SMALL CAPITAL OPEN O","	x (latin small letter open o - 0254)\n"
	"	x (latin small letter reversed c - 2184)"},
/* 1D11 */ { "LATIN SMALL LETTER SIDEWAYS O",NULL},
/* 1D12 */ { "LATIN SMALL LETTER SIDEWAYS OPEN O",NULL},
/* 1D13 */ { "LATIN SMALL LETTER SIDEWAYS O WITH STROKE",NULL},
/* 1D14 */ { "LATIN SMALL LETTER TURNED OE","	* glyph can also have sideways orientation"},
/* 1D15 */ { "LATIN LETTER SMALL CAPITAL OU",NULL},
/* 1D16 */ { "LATIN SMALL LETTER TOP HALF O",NULL},
/* 1D17 */ { "LATIN SMALL LETTER BOTTOM HALF O",NULL},
/* 1D18 */ { "LATIN LETTER SMALL CAPITAL P","	* represents a semi-voiced [p]\n"
	"	x (greek letter small capital rho - 1D29)"},
/* 1D19 */ { "LATIN LETTER SMALL CAPITAL REVERSED R",NULL},
/* 1D1A */ { "LATIN LETTER SMALL CAPITAL TURNED R",NULL},
/* 1D1B */ { "LATIN LETTER SMALL CAPITAL T","	x (cyrillic small letter te - 0442)"},
/* 1D1C */ { "LATIN LETTER SMALL CAPITAL U",NULL},
/* 1D1D */ { "LATIN SMALL LETTER SIDEWAYS U",NULL},
/* 1D1E */ { "LATIN SMALL LETTER SIDEWAYS DIAERESIZED U","	* glyph can also have turned orientation"},
/* 1D1F */ { "LATIN SMALL LETTER SIDEWAYS TURNED M",NULL},
/* 1D20 */ { "LATIN LETTER SMALL CAPITAL V",NULL},
/* 1D21 */ { "LATIN LETTER SMALL CAPITAL W",NULL},
/* 1D22 */ { "LATIN LETTER SMALL CAPITAL Z",NULL},
/* 1D23 */ { "LATIN LETTER SMALL CAPITAL EZH","	x (cyrillic small letter abkhasian dze - 04E1)"},
/* 1D24 */ { "LATIN LETTER VOICED LARYNGEAL SPIRANT",NULL},
/* 1D25 */ { "LATIN LETTER AIN","	x (arabic letter ain - 0639)\n"
	"	x (latin small letter egyptological ain - A725)"},
/* 1D26 */ { "GREEK LETTER SMALL CAPITAL GAMMA",NULL},
/* 1D27 */ { "GREEK LETTER SMALL CAPITAL LAMDA",NULL},
/* 1D28 */ { "GREEK LETTER SMALL CAPITAL PI",NULL},
/* 1D29 */ { "GREEK LETTER SMALL CAPITAL RHO","	* represents a voiceless uvular trill\n"
	"	x (latin letter small capital p - 1D18)"},
/* 1D2A */ { "GREEK LETTER SMALL CAPITAL PSI",NULL},
/* 1D2B */ { "CYRILLIC LETTER SMALL CAPITAL EL","	* in italic style, the glyph is obliqued, not italicized\n"
	"	x (cyrillic small letter el - 043B)"},
/* 1D2C */ { "MODIFIER LETTER CAPITAL A","	# <super> 0041"},
/* 1D2D */ { "MODIFIER LETTER CAPITAL AE","	# <super> 00C6"},
/* 1D2E */ { "MODIFIER LETTER CAPITAL B","	# <super> 0042"},
/* 1D2F */ { "MODIFIER LETTER CAPITAL BARRED B",NULL},
/* 1D30 */ { "MODIFIER LETTER CAPITAL D","	# <super> 0044"},
/* 1D31 */ { "MODIFIER LETTER CAPITAL E","	# <super> 0045"},
/* 1D32 */ { "MODIFIER LETTER CAPITAL REVERSED E","	# <super> 018E"},
/* 1D33 */ { "MODIFIER LETTER CAPITAL G","	# <super> 0047"},
/* 1D34 */ { "MODIFIER LETTER CAPITAL H","	# <super> 0048\n"
	"		A typographical distinction between a capital form and a small capital form is not distinctive for this character in IPA usage.\n"
	"	* voiceless epiglottalized release (extended IPA)\n"
	"	x (latin letter small capital h - 029C)\n"
	"	x (modifier letter cyrillic en - 1D78)"},
/* 1D35 */ { "MODIFIER LETTER CAPITAL I","	# <super> 0049"},
/* 1D36 */ { "MODIFIER LETTER CAPITAL J","	# <super> 004A"},
/* 1D37 */ { "MODIFIER LETTER CAPITAL K","	# <super> 004B"},
/* 1D38 */ { "MODIFIER LETTER CAPITAL L","	# <super> 004C"},
/* 1D39 */ { "MODIFIER LETTER CAPITAL M","	# <super> 004D"},
/* 1D3A */ { "MODIFIER LETTER CAPITAL N","	# <super> 004E"},
/* 1D3B */ { "MODIFIER LETTER CAPITAL REVERSED N",NULL},
/* 1D3C */ { "MODIFIER LETTER CAPITAL O","	# <super> 004F"},
/* 1D3D */ { "MODIFIER LETTER CAPITAL OU","	# <super> 0222"},
/* 1D3E */ { "MODIFIER LETTER CAPITAL P","	# <super> 0050"},
/* 1D3F */ { "MODIFIER LETTER CAPITAL R","	# <super> 0052"},
/* 1D40 */ { "MODIFIER LETTER CAPITAL T","	# <super> 0054"},
/* 1D41 */ { "MODIFIER LETTER CAPITAL U","	# <super> 0055"},
/* 1D42 */ { "MODIFIER LETTER CAPITAL W","	# <super> 0057"},
/* 1D43 */ { "MODIFIER LETTER SMALL A","	# <super> 0061"},
/* 1D44 */ { "MODIFIER LETTER SMALL TURNED A","	# <super> 0250"},
/* 1D45 */ { "MODIFIER LETTER SMALL ALPHA","	# <super> 0251"},
/* 1D46 */ { "MODIFIER LETTER SMALL TURNED AE","	# <super> 1D02"},
/* 1D47 */ { "MODIFIER LETTER SMALL B","	# <super> 0062"},
/* 1D48 */ { "MODIFIER LETTER SMALL D","	# <super> 0064"},
/* 1D49 */ { "MODIFIER LETTER SMALL E","	# <super> 0065"},
/* 1D4A */ { "MODIFIER LETTER SMALL SCHWA","	# <super> 0259"},
/* 1D4B */ { "MODIFIER LETTER SMALL OPEN E","	# <super> 025B"},
/* 1D4C */ { "MODIFIER LETTER SMALL TURNED OPEN E","	* more appropriate equivalence would be to 1D08\n"
	"	# <super> 025C"},
/* 1D4D */ { "MODIFIER LETTER SMALL G","	# <super> 0067"},
/* 1D4E */ { "MODIFIER LETTER SMALL TURNED I",NULL},
/* 1D4F */ { "MODIFIER LETTER SMALL K","	# <super> 006B"},
/* 1D50 */ { "MODIFIER LETTER SMALL M","	# <super> 006D"},
/* 1D51 */ { "MODIFIER LETTER SMALL ENG","	# <super> 014B"},
/* 1D52 */ { "MODIFIER LETTER SMALL O","	# <super> 006F"},
/* 1D53 */ { "MODIFIER LETTER SMALL OPEN O","	# <super> 0254"},
/* 1D54 */ { "MODIFIER LETTER SMALL TOP HALF O","	# <super> 1D16"},
/* 1D55 */ { "MODIFIER LETTER SMALL BOTTOM HALF O","	# <super> 1D17"},
/* 1D56 */ { "MODIFIER LETTER SMALL P","	# <super> 0070"},
/* 1D57 */ { "MODIFIER LETTER SMALL T","	# <super> 0074"},
/* 1D58 */ { "MODIFIER LETTER SMALL U","	# <super> 0075"},
/* 1D59 */ { "MODIFIER LETTER SMALL SIDEWAYS U","	# <super> 1D1D"},
/* 1D5A */ { "MODIFIER LETTER SMALL TURNED M","	# <super> 026F"},
/* 1D5B */ { "MODIFIER LETTER SMALL V","	# <super> 0076"},
/* 1D5C */ { "MODIFIER LETTER SMALL AIN","	x (latin small letter egyptological ain - A725)\n"
	"	# <super> 1D25"},
/* 1D5D */ { "MODIFIER LETTER SMALL BETA","	# <super> 03B2"},
/* 1D5E */ { "MODIFIER LETTER SMALL GREEK GAMMA","	# <super> 03B3"},
/* 1D5F */ { "MODIFIER LETTER SMALL DELTA","	# <super> 03B4"},
/* 1D60 */ { "MODIFIER LETTER SMALL GREEK PHI","	# <super> 03C6"},
/* 1D61 */ { "MODIFIER LETTER SMALL CHI","	# <super> 03C7"},
/* 1D62 */ { "LATIN SUBSCRIPT SMALL LETTER I","	# <sub> 0069"},
/* 1D63 */ { "LATIN SUBSCRIPT SMALL LETTER R","	# <sub> 0072"},
/* 1D64 */ { "LATIN SUBSCRIPT SMALL LETTER U","	# <sub> 0075"},
/* 1D65 */ { "LATIN SUBSCRIPT SMALL LETTER V","	# <sub> 0076"},
/* 1D66 */ { "GREEK SUBSCRIPT SMALL LETTER BETA","	# <sub> 03B2"},
/* 1D67 */ { "GREEK SUBSCRIPT SMALL LETTER GAMMA","	# <sub> 03B3"},
/* 1D68 */ { "GREEK SUBSCRIPT SMALL LETTER RHO","	# <sub> 03C1"},
/* 1D69 */ { "GREEK SUBSCRIPT SMALL LETTER PHI","	# <sub> 03C6"},
/* 1D6A */ { "GREEK SUBSCRIPT SMALL LETTER CHI","	# <sub> 03C7"},
/* 1D6B */ { "LATIN SMALL LETTER UE",NULL},
/* 1D6C */ { "LATIN SMALL LETTER B WITH MIDDLE TILDE",NULL},
/* 1D6D */ { "LATIN SMALL LETTER D WITH MIDDLE TILDE",NULL},
/* 1D6E */ { "LATIN SMALL LETTER F WITH MIDDLE TILDE",NULL},
/* 1D6F */ { "LATIN SMALL LETTER M WITH MIDDLE TILDE",NULL},
/* 1D70 */ { "LATIN SMALL LETTER N WITH MIDDLE TILDE",NULL},
/* 1D71 */ { "LATIN SMALL LETTER P WITH MIDDLE TILDE",NULL},
/* 1D72 */ { "LATIN SMALL LETTER R WITH MIDDLE TILDE",NULL},
/* 1D73 */ { "LATIN SMALL LETTER R WITH FISHHOOK AND MIDDLE TILDE",NULL},
/* 1D74 */ { "LATIN SMALL LETTER S WITH MIDDLE TILDE",NULL},
/* 1D75 */ { "LATIN SMALL LETTER T WITH MIDDLE TILDE",NULL},
/* 1D76 */ { "LATIN SMALL LETTER Z WITH MIDDLE TILDE",NULL},
/* 1D77 */ { "LATIN SMALL LETTER TURNED G","	* Georgian transcription\n"
	"	x (georgian letter turned gan - 10F9)"},
/* 1D78 */ { "MODIFIER LETTER CYRILLIC EN","	* sometimes substituted for 1D34 in extended IPA\n"
	"	x (modifier letter capital h - 1D34)\n"
	"	# <super> 043D"},
/* 1D79 */ { "LATIN SMALL LETTER INSULAR G","	* older Irish phonetic notation\n"
	"	* uppercase is A77D\n"
	"	x (latin small letter g - 0067)\n"
	"	x (latin small letter yogh - 021D)\n"
	"	x (latin small letter script g - 0261)\n"
	"	x (latin small letter gamma - 0263)\n"
	"	x (latin small letter closed insular g - A7D1)"},
/* 1D7A */ { "LATIN SMALL LETTER TH WITH STRIKETHROUGH","	* American dictionary usage\n"
	"	x (greek small letter theta - 03B8)"},
/* 1D7B */ { "LATIN SMALL CAPITAL LETTER I WITH STROKE","	* used with different meanings by Americanists and Oxford dictionaries\n"
	"	x (latin capital letter i with stroke - 0197)"},
/* 1D7C */ { "LATIN SMALL LETTER IOTA WITH STROKE","	* used by Russianists"},
/* 1D7D */ { "LATIN SMALL LETTER P WITH STROKE","	* used by Americanists\n"
	"	* uppercase is 2C63"},
/* 1D7E */ { "LATIN SMALL CAPITAL LETTER U WITH STROKE","	* used by Americanists\n"
	"	x (latin small letter u bar - 0289)"},
/* 1D7F */ { "LATIN SMALL LETTER UPSILON WITH STROKE","	* used by Americanists and Oxford dictionaries"},
/* 1D80 */ { "LATIN SMALL LETTER B WITH PALATAL HOOK",NULL},
/* 1D81 */ { "LATIN SMALL LETTER D WITH PALATAL HOOK",NULL},
/* 1D82 */ { "LATIN SMALL LETTER F WITH PALATAL HOOK",NULL},
/* 1D83 */ { "LATIN SMALL LETTER G WITH PALATAL HOOK",NULL},
/* 1D84 */ { "LATIN SMALL LETTER K WITH PALATAL HOOK",NULL},
/* 1D85 */ { "LATIN SMALL LETTER L WITH PALATAL HOOK",NULL},
/* 1D86 */ { "LATIN SMALL LETTER M WITH PALATAL HOOK","	x (latin small letter m with hook - 0271)"},
/* 1D87 */ { "LATIN SMALL LETTER N WITH PALATAL HOOK","	x (latin small letter eng - 014B)"},
/* 1D88 */ { "LATIN SMALL LETTER P WITH PALATAL HOOK",NULL},
/* 1D89 */ { "LATIN SMALL LETTER R WITH PALATAL HOOK",NULL},
/* 1D8A */ { "LATIN SMALL LETTER S WITH PALATAL HOOK",NULL},
/* 1D8B */ { "LATIN SMALL LETTER ESH WITH PALATAL HOOK",NULL},
/* 1D8C */ { "LATIN SMALL LETTER V WITH PALATAL HOOK",NULL},
/* 1D8D */ { "LATIN SMALL LETTER X WITH PALATAL HOOK",NULL},
/* 1D8E */ { "LATIN SMALL LETTER Z WITH PALATAL HOOK","	* uppercase is A7C6\n"
	"	x (latin small letter z with hook - 0225)"},
/* 1D8F */ { "LATIN SMALL LETTER A WITH RETROFLEX HOOK",NULL},
/* 1D90 */ { "LATIN SMALL LETTER ALPHA WITH RETROFLEX HOOK",NULL},
/* 1D91 */ { "LATIN SMALL LETTER D WITH HOOK AND TAIL",NULL},
/* 1D92 */ { "LATIN SMALL LETTER E WITH RETROFLEX HOOK",NULL},
/* 1D93 */ { "LATIN SMALL LETTER OPEN E WITH RETROFLEX HOOK",NULL},
/* 1D94 */ { "LATIN SMALL LETTER REVERSED OPEN E WITH RETROFLEX HOOK","	x (latin small letter reversed open e with hook - 025D)"},
/* 1D95 */ { "LATIN SMALL LETTER SCHWA WITH RETROFLEX HOOK","	x (latin small letter schwa with hook - 025A)"},
/* 1D96 */ { "LATIN SMALL LETTER I WITH RETROFLEX HOOK",NULL},
/* 1D97 */ { "LATIN SMALL LETTER OPEN O WITH RETROFLEX HOOK",NULL},
/* 1D98 */ { "LATIN SMALL LETTER ESH WITH RETROFLEX HOOK",NULL},
/* 1D99 */ { "LATIN SMALL LETTER U WITH RETROFLEX HOOK",NULL},
/* 1D9A */ { "LATIN SMALL LETTER EZH WITH RETROFLEX HOOK",NULL},
/* 1D9B */ { "MODIFIER LETTER SMALL TURNED ALPHA","	# <super> 0252"},
/* 1D9C */ { "MODIFIER LETTER SMALL C","	# <super> 0063"},
/* 1D9D */ { "MODIFIER LETTER SMALL C WITH CURL","	# <super> 0255"},
/* 1D9E */ { "MODIFIER LETTER SMALL ETH","	# <super> 00F0"},
/* 1D9F */ { "MODIFIER LETTER SMALL REVERSED OPEN E","	# <super> 025C"},
/* 1DA0 */ { "MODIFIER LETTER SMALL F","	# <super> 0066"},
/* 1DA1 */ { "MODIFIER LETTER SMALL DOTLESS J WITH STROKE","	# <super> 025F"},
/* 1DA2 */ { "MODIFIER LETTER SMALL SCRIPT G","	# <super> 0261"},
/* 1DA3 */ { "MODIFIER LETTER SMALL TURNED H","	# <super> 0265"},
/* 1DA4 */ { "MODIFIER LETTER SMALL I WITH STROKE","	# <super> 0268"},
/* 1DA5 */ { "MODIFIER LETTER SMALL IOTA","	# <super> 0269"},
/* 1DA6 */ { "MODIFIER LETTER SMALL CAPITAL I","	* not for use in UPA\n"
	"	x (modifier letter capital i - 1D35)\n"
	"	# <super> 026A"},
/* 1DA7 */ { "MODIFIER LETTER SMALL CAPITAL I WITH STROKE","	# <super> 1D7B"},
/* 1DA8 */ { "MODIFIER LETTER SMALL J WITH CROSSED-TAIL","	# <super> 029D"},
/* 1DA9 */ { "MODIFIER LETTER SMALL L WITH RETROFLEX HOOK","	# <super> 026D"},
/* 1DAA */ { "MODIFIER LETTER SMALL L WITH PALATAL HOOK","	# <super> 1D85"},
/* 1DAB */ { "MODIFIER LETTER SMALL CAPITAL L","	* not for use in UPA\n"
	"	x (modifier letter capital l - 1D38)\n"
	"	# <super> 029F"},
/* 1DAC */ { "MODIFIER LETTER SMALL M WITH HOOK","	# <super> 0271"},
/* 1DAD */ { "MODIFIER LETTER SMALL TURNED M WITH LONG LEG","	# <super> 0270"},
/* 1DAE */ { "MODIFIER LETTER SMALL N WITH LEFT HOOK","	# <super> 0272"},
/* 1DAF */ { "MODIFIER LETTER SMALL N WITH RETROFLEX HOOK","	# <super> 0273"},
/* 1DB0 */ { "MODIFIER LETTER SMALL CAPITAL N","	* not for use in UPA\n"
	"	x (modifier letter capital n - 1D3A)\n"
	"	# <super> 0274"},
/* 1DB1 */ { "MODIFIER LETTER SMALL BARRED O","	# <super> 0275"},
/* 1DB2 */ { "MODIFIER LETTER SMALL PHI","	# <super> 0278"},
/* 1DB3 */ { "MODIFIER LETTER SMALL S WITH HOOK","	# <super> 0282"},
/* 1DB4 */ { "MODIFIER LETTER SMALL ESH","	# <super> 0283"},
/* 1DB5 */ { "MODIFIER LETTER SMALL T WITH PALATAL HOOK","	# <super> 01AB"},
/* 1DB6 */ { "MODIFIER LETTER SMALL U BAR","	# <super> 0289"},
/* 1DB7 */ { "MODIFIER LETTER SMALL UPSILON","	# <super> 028A"},
/* 1DB8 */ { "MODIFIER LETTER SMALL CAPITAL U","	* not for use in UPA\n"
	"	x (modifier letter capital u - 1D41)\n"
	"	# <super> 1D1C"},
/* 1DB9 */ { "MODIFIER LETTER SMALL V WITH HOOK","	# <super> 028B"},
/* 1DBA */ { "MODIFIER LETTER SMALL TURNED V","	# <super> 028C"},
/* 1DBB */ { "MODIFIER LETTER SMALL Z","	# <super> 007A"},
/* 1DBC */ { "MODIFIER LETTER SMALL Z WITH RETROFLEX HOOK","	# <super> 0290"},
/* 1DBD */ { "MODIFIER LETTER SMALL Z WITH CURL","	# <super> 0291"},
/* 1DBE */ { "MODIFIER LETTER SMALL EZH","	# <super> 0292"},
/* 1DBF */ { "MODIFIER LETTER SMALL THETA","	# <super> 03B8"},
/* 1DC0 */ { "COMBINING DOTTED GRAVE ACCENT","	x (combining grave accent - 0300)\n"
	"	x (greek dialytika and varia - 1FED)"},
/* 1DC1 */ { "COMBINING DOTTED ACUTE ACCENT","	x (combining acute accent - 0301)\n"
	"	x (combining greek dialytika tonos - 0344)\n"
	"	x (greek dialytika and oxia - 1FEE)"},
/* 1DC2 */ { "COMBINING SNAKE BELOW",NULL},
/* 1DC3 */ { "COMBINING SUSPENSION MARK","	* Glagolitic\n"
	"	x (combining breve - 0306)"},
/* 1DC4 */ { "COMBINING MACRON-ACUTE",NULL},
/* 1DC5 */ { "COMBINING GRAVE-MACRON",NULL},
/* 1DC6 */ { "COMBINING MACRON-GRAVE",NULL},
/* 1DC7 */ { "COMBINING ACUTE-MACRON",NULL},
/* 1DC8 */ { "COMBINING GRAVE-ACUTE-GRAVE",NULL},
/* 1DC9 */ { "COMBINING ACUTE-GRAVE-ACUTE",NULL},
/* 1DCA */ { "COMBINING LATIN SMALL LETTER R BELOW",NULL},
/* 1DCB */ { "COMBINING BREVE-MACRON","	* Lithuanian dialectology"},
/* 1DCC */ { "COMBINING MACRON-BREVE","	* Lithuanian dialectology"},
/* 1DCD */ { "COMBINING DOUBLE CIRCUMFLEX ABOVE",NULL},
/* 1DCE */ { "COMBINING OGONEK ABOVE",NULL},
/* 1DCF */ { "COMBINING ZIGZAG BELOW",NULL},
/* 1DD0 */ { "COMBINING IS BELOW",NULL},
/* 1DD1 */ { "COMBINING UR ABOVE",NULL},
/* 1DD2 */ { "COMBINING US ABOVE",NULL},
/* 1DD3 */ { "COMBINING LATIN SMALL LETTER FLATTENED OPEN A ABOVE",NULL},
/* 1DD4 */ { "COMBINING LATIN SMALL LETTER AE",NULL},
/* 1DD5 */ { "COMBINING LATIN SMALL LETTER AO",NULL},
/* 1DD6 */ { "COMBINING LATIN SMALL LETTER AV",NULL},
/* 1DD7 */ { "COMBINING LATIN SMALL LETTER C CEDILLA",NULL},
/* 1DD8 */ { "COMBINING LATIN SMALL LETTER INSULAR D",NULL},
/* 1DD9 */ { "COMBINING LATIN SMALL LETTER ETH",NULL},
/* 1DDA */ { "COMBINING LATIN SMALL LETTER G",NULL},
/* 1DDB */ { "COMBINING LATIN LETTER SMALL CAPITAL G",NULL},
/* 1DDC */ { "COMBINING LATIN SMALL LETTER K",NULL},
/* 1DDD */ { "COMBINING LATIN SMALL LETTER L",NULL},
/* 1DDE */ { "COMBINING LATIN LETTER SMALL CAPITAL L",NULL},
/* 1DDF */ { "COMBINING LATIN LETTER SMALL CAPITAL M",NULL},
/* 1DE0 */ { "COMBINING LATIN SMALL LETTER N",NULL},
/* 1DE1 */ { "COMBINING LATIN LETTER SMALL CAPITAL N",NULL},
/* 1DE2 */ { "COMBINING LATIN LETTER SMALL CAPITAL R",NULL},
/* 1DE3 */ { "COMBINING LATIN SMALL LETTER R ROTUNDA",NULL},
/* 1DE4 */ { "COMBINING LATIN SMALL LETTER S",NULL},
/* 1DE5 */ { "COMBINING LATIN SMALL LETTER LONG S",NULL},
/* 1DE6 */ { "COMBINING LATIN SMALL LETTER Z",NULL},
/* 1DE7 */ { "COMBINING LATIN SMALL LETTER ALPHA",NULL},
/* 1DE8 */ { "COMBINING LATIN SMALL LETTER B",NULL},
/* 1DE9 */ { "COMBINING LATIN SMALL LETTER BETA",NULL},
/* 1DEA */ { "COMBINING LATIN SMALL LETTER SCHWA",NULL},
/* 1DEB */ { "COMBINING LATIN SMALL LETTER F",NULL},
/* 1DEC */ { "COMBINING LATIN SMALL LETTER L WITH DOUBLE MIDDLE TILDE",NULL},
/* 1DED */ { "COMBINING LATIN SMALL LETTER O WITH LIGHT CENTRALIZATION STROKE",NULL},
/* 1DEE */ { "COMBINING LATIN SMALL LETTER P",NULL},
/* 1DEF */ { "COMBINING LATIN SMALL LETTER ESH",NULL},
/* 1DF0 */ { "COMBINING LATIN SMALL LETTER U WITH LIGHT CENTRALIZATION STROKE",NULL},
/* 1DF1 */ { "COMBINING LATIN SMALL LETTER W","	x (combining latin small letter w below - 1ABF)\n"
	"	x (combining inverted double arch above - 1AC7)"},
/* 1DF2 */ { "COMBINING LATIN SMALL LETTER A WITH DIAERESIS",NULL},
/* 1DF3 */ { "COMBINING LATIN SMALL LETTER O WITH DIAERESIS",NULL},
/* 1DF4 */ { "COMBINING LATIN SMALL LETTER U WITH DIAERESIS",NULL},
/* 1DF5 */ { "COMBINING UP TACK ABOVE",NULL},
/* 1DF6 */ { "COMBINING KAVYKA ABOVE RIGHT",NULL},
/* 1DF7 */ { "COMBINING KAVYKA ABOVE LEFT","	x (combining cyrillic kavyka - A67C)"},
/* 1DF8 */ { "COMBINING DOT ABOVE LEFT","	* used in Syriac as a disambiguation dot\n"
	"	* used in Typicon Cyrillic, where the dot may have a square appearance\n"
	"	* used in the Latin-based Americanist phonetic notation of Franz Boas\n"
	"	x (combining dot above right - 0358)\n"
	"	x (syriac feminine dot - 0740)"},
/* 1DF9 */ { "COMBINING WIDE INVERTED BRIDGE BELOW","	* used as a Typicon mark\n"
	"	x (combining inverted bridge below - 033A)"},
/* 1DFA */ { "COMBINING DOT BELOW LEFT","	* used in Syriac as a disambiguation dot"},
/* 1DFB */ { "COMBINING DELETION MARK","	* used for \"mhusaa\" in Newa scribal tradition"},
/* 1DFC */ { "COMBINING DOUBLE INVERTED BREVE BELOW",NULL},
/* 1DFD */ { "COMBINING ALMOST EQUAL TO BELOW","	* diacritic indicating a strident vowel in Khoisan languages"},
/* 1DFE */ { "COMBINING LEFT ARROWHEAD ABOVE",NULL},
/* 1DFF */ { "COMBINING RIGHT ARROWHEAD AND DOWN ARROWHEAD BELOW",NULL}
};

UN_DLL_LOCAL
static const struct unicode_nameannot una_00_1E[] = {
/* 1E00 */ { "LATIN CAPITAL LETTER A WITH RING BELOW","	: 0041 0325"},
/* 1E01 */ { "LATIN SMALL LETTER A WITH RING BELOW","	: 0061 0325"},
/* 1E02 */ { "LATIN CAPITAL LETTER B WITH DOT ABOVE","	: 0042 0307"},
/* 1E03 */ { "LATIN SMALL LETTER B WITH DOT ABOVE","	* Irish Gaelic (old orthography)\n"
	"	: 0062 0307"},
/* 1E04 */ { "LATIN CAPITAL LETTER B WITH DOT BELOW","	: 0042 0323"},
/* 1E05 */ { "LATIN SMALL LETTER B WITH DOT BELOW","	: 0062 0323"},
/* 1E06 */ { "LATIN CAPITAL LETTER B WITH LINE BELOW","	: 0042 0331"},
/* 1E07 */ { "LATIN SMALL LETTER B WITH LINE BELOW","	: 0062 0331"},
/* 1E08 */ { "LATIN CAPITAL LETTER C WITH CEDILLA AND ACUTE","	: 00C7 0301"},
/* 1E09 */ { "LATIN SMALL LETTER C WITH CEDILLA AND ACUTE","	: 00E7 0301"},
/* 1E0A */ { "LATIN CAPITAL LETTER D WITH DOT ABOVE","	: 0044 0307"},
/* 1E0B */ { "LATIN SMALL LETTER D WITH DOT ABOVE","	* Irish Gaelic (old orthography)\n"
	"	: 0064 0307"},
/* 1E0C */ { "LATIN CAPITAL LETTER D WITH DOT BELOW","	: 0044 0323"},
/* 1E0D */ { "LATIN SMALL LETTER D WITH DOT BELOW","	* Indic transliteration\n"
	"	: 0064 0323"},
/* 1E0E */ { "LATIN CAPITAL LETTER D WITH LINE BELOW","	: 0044 0331"},
/* 1E0F */ { "LATIN SMALL LETTER D WITH LINE BELOW","	: 0064 0331"},
/* 1E10 */ { "LATIN CAPITAL LETTER D WITH CEDILLA","	: 0044 0327"},
/* 1E11 */ { "LATIN SMALL LETTER D WITH CEDILLA","	* Livonian\n"
	"	: 0064 0327"},
/* 1E12 */ { "LATIN CAPITAL LETTER D WITH CIRCUMFLEX BELOW","	: 0044 032D"},
/* 1E13 */ { "LATIN SMALL LETTER D WITH CIRCUMFLEX BELOW","	: 0064 032D"},
/* 1E14 */ { "LATIN CAPITAL LETTER E WITH MACRON AND GRAVE","	: 0112 0300"},
/* 1E15 */ { "LATIN SMALL LETTER E WITH MACRON AND GRAVE","	: 0113 0300"},
/* 1E16 */ { "LATIN CAPITAL LETTER E WITH MACRON AND ACUTE","	: 0112 0301"},
/* 1E17 */ { "LATIN SMALL LETTER E WITH MACRON AND ACUTE","	: 0113 0301"},
/* 1E18 */ { "LATIN CAPITAL LETTER E WITH CIRCUMFLEX BELOW","	: 0045 032D"},
/* 1E19 */ { "LATIN SMALL LETTER E WITH CIRCUMFLEX BELOW","	: 0065 032D"},
/* 1E1A */ { "LATIN CAPITAL LETTER E WITH TILDE BELOW","	: 0045 0330"},
/* 1E1B */ { "LATIN SMALL LETTER E WITH TILDE BELOW","	: 0065 0330"},
/* 1E1C */ { "LATIN CAPITAL LETTER E WITH CEDILLA AND BREVE","	: 0228 0306"},
/* 1E1D */ { "LATIN SMALL LETTER E WITH CEDILLA AND BREVE","	* Hebrew transliteration\n"
	"	: 0229 0306"},
/* 1E1E */ { "LATIN CAPITAL LETTER F WITH DOT ABOVE","	: 0046 0307"},
/* 1E1F */ { "LATIN SMALL LETTER F WITH DOT ABOVE","	* Irish Gaelic (old orthography)\n"
	"	: 0066 0307"},
/* 1E20 */ { "LATIN CAPITAL LETTER G WITH MACRON","	: 0047 0304"},
/* 1E21 */ { "LATIN SMALL LETTER G WITH MACRON","	: 0067 0304"},
/* 1E22 */ { "LATIN CAPITAL LETTER H WITH DOT ABOVE","	: 0048 0307"},
/* 1E23 */ { "LATIN SMALL LETTER H WITH DOT ABOVE","	: 0068 0307"},
/* 1E24 */ { "LATIN CAPITAL LETTER H WITH DOT BELOW","	: 0048 0323"},
/* 1E25 */ { "LATIN SMALL LETTER H WITH DOT BELOW","	* Indic transliteration\n"
	"	: 0068 0323"},
/* 1E26 */ { "LATIN CAPITAL LETTER H WITH DIAERESIS","	: 0048 0308"},
/* 1E27 */ { "LATIN SMALL LETTER H WITH DIAERESIS","	: 0068 0308"},
/* 1E28 */ { "LATIN CAPITAL LETTER H WITH CEDILLA","	: 0048 0327"},
/* 1E29 */ { "LATIN SMALL LETTER H WITH CEDILLA","	: 0068 0327"},
/* 1E2A */ { "LATIN CAPITAL LETTER H WITH BREVE BELOW","	: 0048 032E"},
/* 1E2B */ { "LATIN SMALL LETTER H WITH BREVE BELOW","	* Semitic transliteration\n"
	"	: 0068 032E"},
/* 1E2C */ { "LATIN CAPITAL LETTER I WITH TILDE BELOW","	: 0049 0330"},
/* 1E2D */ { "LATIN SMALL LETTER I WITH TILDE BELOW","	: 0069 0330"},
/* 1E2E */ { "LATIN CAPITAL LETTER I WITH DIAERESIS AND ACUTE","	: 00CF 0301"},
/* 1E2F */ { "LATIN SMALL LETTER I WITH DIAERESIS AND ACUTE","	: 00EF 0301"},
/* 1E30 */ { "LATIN CAPITAL LETTER K WITH ACUTE","	: 004B 0301"},
/* 1E31 */ { "LATIN SMALL LETTER K WITH ACUTE","	* Macedonian transliteration\n"
	"	: 006B 0301"},
/* 1E32 */ { "LATIN CAPITAL LETTER K WITH DOT BELOW","	: 004B 0323"},
/* 1E33 */ { "LATIN SMALL LETTER K WITH DOT BELOW","	: 006B 0323"},
/* 1E34 */ { "LATIN CAPITAL LETTER K WITH LINE BELOW","	: 004B 0331"},
/* 1E35 */ { "LATIN SMALL LETTER K WITH LINE BELOW","	: 006B 0331"},
/* 1E36 */ { "LATIN CAPITAL LETTER L WITH DOT BELOW","	: 004C 0323"},
/* 1E37 */ { "LATIN SMALL LETTER L WITH DOT BELOW","	* Indic transliteration\n"
	"	* see ISO 15919 on the use of dot below versus ring below in Indic transliteration\n"
	"	x (combining ring below - 0325)\n"
	"	: 006C 0323"},
/* 1E38 */ { "LATIN CAPITAL LETTER L WITH DOT BELOW AND MACRON","	: 1E36 0304"},
/* 1E39 */ { "LATIN SMALL LETTER L WITH DOT BELOW AND MACRON","	* Indic transliteration\n"
	"	: 1E37 0304"},
/* 1E3A */ { "LATIN CAPITAL LETTER L WITH LINE BELOW","	: 004C 0331"},
/* 1E3B */ { "LATIN SMALL LETTER L WITH LINE BELOW","	* Indic transliteration\n"
	"	: 006C 0331"},
/* 1E3C */ { "LATIN CAPITAL LETTER L WITH CIRCUMFLEX BELOW","	: 004C 032D"},
/* 1E3D */ { "LATIN SMALL LETTER L WITH CIRCUMFLEX BELOW","	: 006C 032D"},
/* 1E3E */ { "LATIN CAPITAL LETTER M WITH ACUTE","	: 004D 0301"},
/* 1E3F */ { "LATIN SMALL LETTER M WITH ACUTE","	: 006D 0301"},
/* 1E40 */ { "LATIN CAPITAL LETTER M WITH DOT ABOVE","	: 004D 0307"},
/* 1E41 */ { "LATIN SMALL LETTER M WITH DOT ABOVE","	* Irish Gaelic (old orthography)\n"
	"	: 006D 0307"},
/* 1E42 */ { "LATIN CAPITAL LETTER M WITH DOT BELOW","	: 004D 0323"},
/* 1E43 */ { "LATIN SMALL LETTER M WITH DOT BELOW","	* Indic transliteration\n"
	"	: 006D 0323"},
/* 1E44 */ { "LATIN CAPITAL LETTER N WITH DOT ABOVE","	: 004E 0307"},
/* 1E45 */ { "LATIN SMALL LETTER N WITH DOT ABOVE","	* Indic transliteration\n"
	"	: 006E 0307"},
/* 1E46 */ { "LATIN CAPITAL LETTER N WITH DOT BELOW","	: 004E 0323"},
/* 1E47 */ { "LATIN SMALL LETTER N WITH DOT BELOW","	* Indic transliteration\n"
	"	: 006E 0323"},
/* 1E48 */ { "LATIN CAPITAL LETTER N WITH LINE BELOW","	: 004E 0331"},
/* 1E49 */ { "LATIN SMALL LETTER N WITH LINE BELOW","	* Indic transliteration\n"
	"	: 006E 0331"},
/* 1E4A */ { "LATIN CAPITAL LETTER N WITH CIRCUMFLEX BELOW","	: 004E 032D"},
/* 1E4B */ { "LATIN SMALL LETTER N WITH CIRCUMFLEX BELOW","	: 006E 032D"},
/* 1E4C */ { "LATIN CAPITAL LETTER O WITH TILDE AND ACUTE","	: 00D5 0301"},
/* 1E4D */ { "LATIN SMALL LETTER O WITH TILDE AND ACUTE","	: 00F5 0301"},
/* 1E4E */ { "LATIN CAPITAL LETTER O WITH TILDE AND DIAERESIS","	: 00D5 0308"},
/* 1E4F */ { "LATIN SMALL LETTER O WITH TILDE AND DIAERESIS","	: 00F5 0308"},
/* 1E50 */ { "LATIN CAPITAL LETTER O WITH MACRON AND GRAVE","	: 014C 0300"},
/* 1E51 */ { "LATIN SMALL LETTER O WITH MACRON AND GRAVE","	: 014D 0300"},
/* 1E52 */ { "LATIN CAPITAL LETTER O WITH MACRON AND ACUTE","	: 014C 0301"},
/* 1E53 */ { "LATIN SMALL LETTER O WITH MACRON AND ACUTE","	: 014D 0301"},
/* 1E54 */ { "LATIN CAPITAL LETTER P WITH ACUTE","	: 0050 0301"},
/* 1E55 */ { "LATIN SMALL LETTER P WITH ACUTE","	: 0070 0301"},
/* 1E56 */ { "LATIN CAPITAL LETTER P WITH DOT ABOVE","	: 0050 0307"},
/* 1E57 */ { "LATIN SMALL LETTER P WITH DOT ABOVE","	* Irish Gaelic (old orthography)\n"
	"	: 0070 0307"},
/* 1E58 */ { "LATIN CAPITAL LETTER R WITH DOT ABOVE","	: 0052 0307"},
/* 1E59 */ { "LATIN SMALL LETTER R WITH DOT ABOVE","	: 0072 0307"},
/* 1E5A */ { "LATIN CAPITAL LETTER R WITH DOT BELOW","	: 0052 0323"},
/* 1E5B */ { "LATIN SMALL LETTER R WITH DOT BELOW","	* Indic transliteration\n"
	"	* see ISO 15919 on the use of dot below versus ring below in Indic transliteration\n"
	"	x (combining ring below - 0325)\n"
	"	: 0072 0323"},
/* 1E5C */ { "LATIN CAPITAL LETTER R WITH DOT BELOW AND MACRON","	: 1E5A 0304"},
/* 1E5D */ { "LATIN SMALL LETTER R WITH DOT BELOW AND MACRON","	* Indic transliteration\n"
	"	: 1E5B 0304"},
/* 1E5E */ { "LATIN CAPITAL LETTER R WITH LINE BELOW","	: 0052 0331"},
/* 1E5F */ { "LATIN SMALL LETTER R WITH LINE BELOW","	* Indic transliteration\n"
	"	: 0072 0331"},
/* 1E60 */ { "LATIN CAPITAL LETTER S WITH DOT ABOVE","	: 0053 0307"},
/* 1E61 */ { "LATIN SMALL LETTER S WITH DOT ABOVE","	* Irish Gaelic (old orthography)\n"
	"	: 0073 0307"},
/* 1E62 */ { "LATIN CAPITAL LETTER S WITH DOT BELOW","	: 0053 0323"},
/* 1E63 */ { "LATIN SMALL LETTER S WITH DOT BELOW","	* Indic transliteration\n"
	"	: 0073 0323"},
/* 1E64 */ { "LATIN CAPITAL LETTER S WITH ACUTE AND DOT ABOVE","	: 015A 0307"},
/* 1E65 */ { "LATIN SMALL LETTER S WITH ACUTE AND DOT ABOVE","	: 015B 0307"},
/* 1E66 */ { "LATIN CAPITAL LETTER S WITH CARON AND DOT ABOVE","	: 0160 0307"},
/* 1E67 */ { "LATIN SMALL LETTER S WITH CARON AND DOT ABOVE","	: 0161 0307"},
/* 1E68 */ { "LATIN CAPITAL LETTER S WITH DOT BELOW AND DOT ABOVE","	: 1E62 0307"},
/* 1E69 */ { "LATIN SMALL LETTER S WITH DOT BELOW AND DOT ABOVE","	: 1E63 0307"},
/* 1E6A */ { "LATIN CAPITAL LETTER T WITH DOT ABOVE","	: 0054 0307"},
/* 1E6B */ { "LATIN SMALL LETTER T WITH DOT ABOVE","	* Irish Gaelic (old orthography)\n"
	"	: 0074 0307"},
/* 1E6C */ { "LATIN CAPITAL LETTER T WITH DOT BELOW","	: 0054 0323"},
/* 1E6D */ { "LATIN SMALL LETTER T WITH DOT BELOW","	* Indic transliteration\n"
	"	: 0074 0323"},
/* 1E6E */ { "LATIN CAPITAL LETTER T WITH LINE BELOW","	: 0054 0331"},
/* 1E6F */ { "LATIN SMALL LETTER T WITH LINE BELOW","	* Semitic transliteration\n"
	"	: 0074 0331"},
/* 1E70 */ { "LATIN CAPITAL LETTER T WITH CIRCUMFLEX BELOW","	: 0054 032D"},
/* 1E71 */ { "LATIN SMALL LETTER T WITH CIRCUMFLEX BELOW","	: 0074 032D"},
/* 1E72 */ { "LATIN CAPITAL LETTER U WITH DIAERESIS BELOW","	: 0055 0324"},
/* 1E73 */ { "LATIN SMALL LETTER U WITH DIAERESIS BELOW","	: 0075 0324"},
/* 1E74 */ { "LATIN CAPITAL LETTER U WITH TILDE BELOW","	: 0055 0330"},
/* 1E75 */ { "LATIN SMALL LETTER U WITH TILDE BELOW","	: 0075 0330"},
/* 1E76 */ { "LATIN CAPITAL LETTER U WITH CIRCUMFLEX BELOW","	: 0055 032D"},
/* 1E77 */ { "LATIN SMALL LETTER U WITH CIRCUMFLEX BELOW","	: 0075 032D"},
/* 1E78 */ { "LATIN CAPITAL LETTER U WITH TILDE AND ACUTE","	: 0168 0301"},
/* 1E79 */ { "LATIN SMALL LETTER U WITH TILDE AND ACUTE","	: 0169 0301"},
/* 1E7A */ { "LATIN CAPITAL LETTER U WITH MACRON AND DIAERESIS","	: 016A 0308"},
/* 1E7B */ { "LATIN SMALL LETTER U WITH MACRON AND DIAERESIS","	: 016B 0308"},
/* 1E7C */ { "LATIN CAPITAL LETTER V WITH TILDE","	: 0056 0303"},
/* 1E7D */ { "LATIN SMALL LETTER V WITH TILDE","	: 0076 0303"},
/* 1E7E */ { "LATIN CAPITAL LETTER V WITH DOT BELOW","	: 0056 0323"},
/* 1E7F */ { "LATIN SMALL LETTER V WITH DOT BELOW","	: 0076 0323"},
/* 1E80 */ { "LATIN CAPITAL LETTER W WITH GRAVE","	: 0057 0300"},
/* 1E81 */ { "LATIN SMALL LETTER W WITH GRAVE","	* Welsh\n"
	"	: 0077 0300"},
/* 1E82 */ { "LATIN CAPITAL LETTER W WITH ACUTE","	: 0057 0301"},
/* 1E83 */ { "LATIN SMALL LETTER W WITH ACUTE","	* Welsh\n"
	"	: 0077 0301"},
/* 1E84 */ { "LATIN CAPITAL LETTER W WITH DIAERESIS","	: 0057 0308"},
/* 1E85 */ { "LATIN SMALL LETTER W WITH DIAERESIS","	* Welsh\n"
	"	: 0077 0308"},
/* 1E86 */ { "LATIN CAPITAL LETTER W WITH DOT ABOVE","	: 0057 0307"},
/* 1E87 */ { "LATIN SMALL LETTER W WITH DOT ABOVE","	: 0077 0307"},
/* 1E88 */ { "LATIN CAPITAL LETTER W WITH DOT BELOW","	: 0057 0323"},
/* 1E89 */ { "LATIN SMALL LETTER W WITH DOT BELOW","	: 0077 0323"},
/* 1E8A */ { "LATIN CAPITAL LETTER X WITH DOT ABOVE","	: 0058 0307"},
/* 1E8B */ { "LATIN SMALL LETTER X WITH DOT ABOVE","	: 0078 0307"},
/* 1E8C */ { "LATIN CAPITAL LETTER X WITH DIAERESIS","	: 0058 0308"},
/* 1E8D */ { "LATIN SMALL LETTER X WITH DIAERESIS","	: 0078 0308"},
/* 1E8E */ { "LATIN CAPITAL LETTER Y WITH DOT ABOVE","	: 0059 0307"},
/* 1E8F */ { "LATIN SMALL LETTER Y WITH DOT ABOVE","	: 0079 0307"},
/* 1E90 */ { "LATIN CAPITAL LETTER Z WITH CIRCUMFLEX","	: 005A 0302"},
/* 1E91 */ { "LATIN SMALL LETTER Z WITH CIRCUMFLEX","	: 007A 0302"},
/* 1E92 */ { "LATIN CAPITAL LETTER Z WITH DOT BELOW","	: 005A 0323"},
/* 1E93 */ { "LATIN SMALL LETTER Z WITH DOT BELOW","	* Semitic transliteration\n"
	"	x (arabic letter zah - 0638)\n"
	"	: 007A 0323"},
/* 1E94 */ { "LATIN CAPITAL LETTER Z WITH LINE BELOW","	: 005A 0331"},
/* 1E95 */ { "LATIN SMALL LETTER Z WITH LINE BELOW","	* Semitic transliteration\n"
	"	: 007A 0331"},
/* 1E96 */ { "LATIN SMALL LETTER H WITH LINE BELOW","	* Semitic transliteration\n"
	"	: 0068 0331"},
/* 1E97 */ { "LATIN SMALL LETTER T WITH DIAERESIS","	: 0074 0308"},
/* 1E98 */ { "LATIN SMALL LETTER W WITH RING ABOVE","	: 0077 030A"},
/* 1E99 */ { "LATIN SMALL LETTER Y WITH RING ABOVE","	: 0079 030A"},
/* 1E9A */ { "LATIN SMALL LETTER A WITH RIGHT HALF RING","	# 0061 02BE"},
/* 1E9B */ { "LATIN SMALL LETTER LONG S WITH DOT ABOVE","	* in current use in Gaelic types (as glyph variant of 1E61)\n"
	"	: 017F 0307"},
/* 1E9C */ { "LATIN SMALL LETTER LONG S WITH DIAGONAL STROKE","	* medievalist use for several abbreviations\n"
	"	* used in pre-1921 Latvian orthography\n"
	"	* used in Blackletter types of pre-1950 Lower Sorbian orthography\n"
	"	x (latin small letter s with oblique stroke - A7A9)"},
/* 1E9D */ { "LATIN SMALL LETTER LONG S WITH HIGH STROKE",NULL},
/* 1E9E */ { "LATIN CAPITAL LETTER SHARP S","	* not used in Swiss High German\n"
	"	* lowercase is 00DF\n"
	"	x (latin small letter sharp s - 00DF)\n"
	"	x (latin capital letter middle scots s - A7D6)"},
/* 1E9F */ { "LATIN SMALL LETTER DELTA","	* this is a non-casing letter\n"
	"	x (greek small letter delta - 03B4)"},
/* 1EA0 */ { "LATIN CAPITAL LETTER A WITH DOT BELOW","	: 0041 0323"},
/* 1EA1 */ { "LATIN SMALL LETTER A WITH DOT BELOW","	: 0061 0323"},
/* 1EA2 */ { "LATIN CAPITAL LETTER A WITH HOOK ABOVE","	: 0041 0309"},
/* 1EA3 */ { "LATIN SMALL LETTER A WITH HOOK ABOVE","	: 0061 0309"},
/* 1EA4 */ { "LATIN CAPITAL LETTER A WITH CIRCUMFLEX AND ACUTE","	: 00C2 0301"},
/* 1EA5 */ { "LATIN SMALL LETTER A WITH CIRCUMFLEX AND ACUTE","	: 00E2 0301"},
/* 1EA6 */ { "LATIN CAPITAL LETTER A WITH CIRCUMFLEX AND GRAVE","	: 00C2 0300"},
/* 1EA7 */ { "LATIN SMALL LETTER A WITH CIRCUMFLEX AND GRAVE","	: 00E2 0300"},
/* 1EA8 */ { "LATIN CAPITAL LETTER A WITH CIRCUMFLEX AND HOOK ABOVE","	: 00C2 0309"},
/* 1EA9 */ { "LATIN SMALL LETTER A WITH CIRCUMFLEX AND HOOK ABOVE","	: 00E2 0309"},
/* 1EAA */ { "LATIN CAPITAL LETTER A WITH CIRCUMFLEX AND TILDE","	: 00C2 0303"},
/* 1EAB */ { "LATIN SMALL LETTER A WITH CIRCUMFLEX AND TILDE","	: 00E2 0303"},
/* 1EAC */ { "LATIN CAPITAL LETTER A WITH CIRCUMFLEX AND DOT BELOW","	: 1EA0 0302"},
/* 1EAD */ { "LATIN SMALL LETTER A WITH CIRCUMFLEX AND DOT BELOW","	: 1EA1 0302"},
/* 1EAE */ { "LATIN CAPITAL LETTER A WITH BREVE AND ACUTE","	: 0102 0301"},
/* 1EAF */ { "LATIN SMALL LETTER A WITH BREVE AND ACUTE","	: 0103 0301"},
/* 1EB0 */ { "LATIN CAPITAL LETTER A WITH BREVE AND GRAVE","	: 0102 0300"},
/* 1EB1 */ { "LATIN SMALL LETTER A WITH BREVE AND GRAVE","	: 0103 0300"},
/* 1EB2 */ { "LATIN CAPITAL LETTER A WITH BREVE AND HOOK ABOVE","	: 0102 0309"},
/* 1EB3 */ { "LATIN SMALL LETTER A WITH BREVE AND HOOK ABOVE","	: 0103 0309"},
/* 1EB4 */ { "LATIN CAPITAL LETTER A WITH BREVE AND TILDE","	: 0102 0303"},
/* 1EB5 */ { "LATIN SMALL LETTER A WITH BREVE AND TILDE","	: 0103 0303"},
/* 1EB6 */ { "LATIN CAPITAL LETTER A WITH BREVE AND DOT BELOW","	: 1EA0 0306"},
/* 1EB7 */ { "LATIN SMALL LETTER A WITH BREVE AND DOT BELOW","	: 1EA1 0306"},
/* 1EB8 */ { "LATIN CAPITAL LETTER E WITH DOT BELOW","	: 0045 0323"},
/* 1EB9 */ { "LATIN SMALL LETTER E WITH DOT BELOW","	: 0065 0323"},
/* 1EBA */ { "LATIN CAPITAL LETTER E WITH HOOK ABOVE","	: 0045 0309"},
/* 1EBB */ { "LATIN SMALL LETTER E WITH HOOK ABOVE","	: 0065 0309"},
/* 1EBC */ { "LATIN CAPITAL LETTER E WITH TILDE","	: 0045 0303"},
/* 1EBD */ { "LATIN SMALL LETTER E WITH TILDE","	: 0065 0303"},
/* 1EBE */ { "LATIN CAPITAL LETTER E WITH CIRCUMFLEX AND ACUTE","	: 00CA 0301"},
/* 1EBF */ { "LATIN SMALL LETTER E WITH CIRCUMFLEX AND ACUTE","	: 00EA 0301"},
/* 1EC0 */ { "LATIN CAPITAL LETTER E WITH CIRCUMFLEX AND GRAVE","	: 00CA 0300"},
/* 1EC1 */ { "LATIN SMALL LETTER E WITH CIRCUMFLEX AND GRAVE","	: 00EA 0300"},
/* 1EC2 */ { "LATIN CAPITAL LETTER E WITH CIRCUMFLEX AND HOOK ABOVE","	: 00CA 0309"},
/* 1EC3 */ { "LATIN SMALL LETTER E WITH CIRCUMFLEX AND HOOK ABOVE","	: 00EA 0309"},
/* 1EC4 */ { "LATIN CAPITAL LETTER E WITH CIRCUMFLEX AND TILDE","	: 00CA 0303"},
/* 1EC5 */ { "LATIN SMALL LETTER E WITH CIRCUMFLEX AND TILDE","	: 00EA 0303"},
/* 1EC6 */ { "LATIN CAPITAL LETTER E WITH CIRCUMFLEX AND DOT BELOW","	: 1EB8 0302"},
/* 1EC7 */ { "LATIN SMALL LETTER E WITH CIRCUMFLEX AND DOT BELOW","	: 1EB9 0302"},
/* 1EC8 */ { "LATIN CAPITAL LETTER I WITH HOOK ABOVE","	: 0049 0309"},
/* 1EC9 */ { "LATIN SMALL LETTER I WITH HOOK ABOVE","	: 0069 0309"},
/* 1ECA */ { "LATIN CAPITAL LETTER I WITH DOT BELOW","	: 0049 0323"},
/* 1ECB */ { "LATIN SMALL LETTER I WITH DOT BELOW","	: 0069 0323"},
/* 1ECC */ { "LATIN CAPITAL LETTER O WITH DOT BELOW","	: 004F 0323"},
/* 1ECD */ { "LATIN SMALL LETTER O WITH DOT BELOW","	: 006F 0323"},
/* 1ECE */ { "LATIN CAPITAL LETTER O WITH HOOK ABOVE","	: 004F 0309"},
/* 1ECF */ { "LATIN SMALL LETTER O WITH HOOK ABOVE","	: 006F 0309"},
/* 1ED0 */ { "LATIN CAPITAL LETTER O WITH CIRCUMFLEX AND ACUTE","	: 00D4 0301"},
/* 1ED1 */ { "LATIN SMALL LETTER O WITH CIRCUMFLEX AND ACUTE","	: 00F4 0301"},
/* 1ED2 */ { "LATIN CAPITAL LETTER O WITH CIRCUMFLEX AND GRAVE","	: 00D4 0300"},
/* 1ED3 */ { "LATIN SMALL LETTER O WITH CIRCUMFLEX AND GRAVE","	: 00F4 0300"},
/* 1ED4 */ { "LATIN CAPITAL LETTER O WITH CIRCUMFLEX AND HOOK ABOVE","	: 00D4 0309"},
/* 1ED5 */ { "LATIN SMALL LETTER O WITH CIRCUMFLEX AND HOOK ABOVE","	: 00F4 0309"},
/* 1ED6 */ { "LATIN CAPITAL LETTER O WITH CIRCUMFLEX AND TILDE","	: 00D4 0303"},
/* 1ED7 */ { "LATIN SMALL LETTER O WITH CIRCUMFLEX AND TILDE","	: 00F4 0303"},
/* 1ED8 */ { "LATIN CAPITAL LETTER O WITH CIRCUMFLEX AND DOT BELOW","	: 1ECC 0302"},
/* 1ED9 */ { "LATIN SMALL LETTER O WITH CIRCUMFLEX AND DOT BELOW","	: 1ECD 0302"},
/* 1EDA */ { "LATIN CAPITAL LETTER O WITH HORN AND ACUTE","	: 01A0 0301"},
/* 1EDB */ { "LATIN SMALL LETTER O WITH HORN AND ACUTE","	: 01A1 0301"},
/* 1EDC */ { "LATIN CAPITAL LETTER O WITH HORN AND GRAVE","	: 01A0 0300"},
/* 1EDD */ { "LATIN SMALL LETTER O WITH HORN AND GRAVE","	: 01A1 0300"},
/* 1EDE */ { "LATIN CAPITAL LETTER O WITH HORN AND HOOK ABOVE","	: 01A0 0309"},
/* 1EDF */ { "LATIN SMALL LETTER O WITH HORN AND HOOK ABOVE","	: 01A1 0309"},
/* 1EE0 */ { "LATIN CAPITAL LETTER O WITH HORN AND TILDE","	: 01A0 0303"},
/* 1EE1 */ { "LATIN SMALL LETTER O WITH HORN AND TILDE","	: 01A1 0303"},
/* 1EE2 */ { "LATIN CAPITAL LETTER O WITH HORN AND DOT BELOW","	: 01A0 0323"},
/* 1EE3 */ { "LATIN SMALL LETTER O WITH HORN AND DOT BELOW","	: 01A1 0323"},
/* 1EE4 */ { "LATIN CAPITAL LETTER U WITH DOT BELOW","	: 0055 0323"},
/* 1EE5 */ { "LATIN SMALL LETTER U WITH DOT BELOW","	: 0075 0323"},
/* 1EE6 */ { "LATIN CAPITAL LETTER U WITH HOOK ABOVE","	: 0055 0309"},
/* 1EE7 */ { "LATIN SMALL LETTER U WITH HOOK ABOVE","	: 0075 0309"},
/* 1EE8 */ { "LATIN CAPITAL LETTER U WITH HORN AND ACUTE","	: 01AF 0301"},
/* 1EE9 */ { "LATIN SMALL LETTER U WITH HORN AND ACUTE","	: 01B0 0301"},
/* 1EEA */ { "LATIN CAPITAL LETTER U WITH HORN AND GRAVE","	: 01AF 0300"},
/* 1EEB */ { "LATIN SMALL LETTER U WITH HORN AND GRAVE","	: 01B0 0300"},
/* 1EEC */ { "LATIN CAPITAL LETTER U WITH HORN AND HOOK ABOVE","	: 01AF 0309"},
/* 1EED */ { "LATIN SMALL LETTER U WITH HORN AND HOOK ABOVE","	: 01B0 0309"},
/* 1EEE */ { "LATIN CAPITAL LETTER U WITH HORN AND TILDE","	: 01AF 0303"},
/* 1EEF */ { "LATIN SMALL LETTER U WITH HORN AND TILDE","	: 01B0 0303"},
/* 1EF0 */ { "LATIN CAPITAL LETTER U WITH HORN AND DOT BELOW","	: 01AF 0323"},
/* 1EF1 */ { "LATIN SMALL LETTER U WITH HORN AND DOT BELOW","	: 01B0 0323"},
/* 1EF2 */ { "LATIN CAPITAL LETTER Y WITH GRAVE","	: 0059 0300"},
/* 1EF3 */ { "LATIN SMALL LETTER Y WITH GRAVE","	* Welsh\n"
	"	: 0079 0300"},
/* 1EF4 */ { "LATIN CAPITAL LETTER Y WITH DOT BELOW","	: 0059 0323"},
/* 1EF5 */ { "LATIN SMALL LETTER Y WITH DOT BELOW","	: 0079 0323"},
/* 1EF6 */ { "LATIN CAPITAL LETTER Y WITH HOOK ABOVE","	: 0059 0309"},
/* 1EF7 */ { "LATIN SMALL LETTER Y WITH HOOK ABOVE","	: 0079 0309"},
/* 1EF8 */ { "LATIN CAPITAL LETTER Y WITH TILDE","	: 0059 0303"},
/* 1EF9 */ { "LATIN SMALL LETTER Y WITH TILDE","	: 0079 0303"},
/* 1EFA */ { "LATIN CAPITAL LETTER MIDDLE-WELSH LL",NULL},
/* 1EFB */ { "LATIN SMALL LETTER MIDDLE-WELSH LL",NULL},
/* 1EFC */ { "LATIN CAPITAL LETTER MIDDLE-WELSH V",NULL},
/* 1EFD */ { "LATIN SMALL LETTER MIDDLE-WELSH V",NULL},
/* 1EFE */ { "LATIN CAPITAL LETTER Y WITH LOOP",NULL},
/* 1EFF */ { "LATIN SMALL LETTER Y WITH LOOP",NULL}
};

UN_DLL_LOCAL
static const struct unicode_nameannot una_00_1F[] = {
/* 1F00 */ { "GREEK SMALL LETTER ALPHA WITH PSILI","	: 03B1 0313"},
/* 1F01 */ { "GREEK SMALL LETTER ALPHA WITH DASIA","	: 03B1 0314"},
/* 1F02 */ { "GREEK SMALL LETTER ALPHA WITH PSILI AND VARIA","	: 1F00 0300"},
/* 1F03 */ { "GREEK SMALL LETTER ALPHA WITH DASIA AND VARIA","	: 1F01 0300"},
/* 1F04 */ { "GREEK SMALL LETTER ALPHA WITH PSILI AND OXIA","	: 1F00 0301"},
/* 1F05 */ { "GREEK SMALL LETTER ALPHA WITH DASIA AND OXIA","	: 1F01 0301"},
/* 1F06 */ { "GREEK SMALL LETTER ALPHA WITH PSILI AND PERISPOMENI","	: 1F00 0342"},
/* 1F07 */ { "GREEK SMALL LETTER ALPHA WITH DASIA AND PERISPOMENI","	: 1F01 0342"},
/* 1F08 */ { "GREEK CAPITAL LETTER ALPHA WITH PSILI","	: 0391 0313"},
/* 1F09 */ { "GREEK CAPITAL LETTER ALPHA WITH DASIA","	: 0391 0314"},
/* 1F0A */ { "GREEK CAPITAL LETTER ALPHA WITH PSILI AND VARIA","	: 1F08 0300"},
/* 1F0B */ { "GREEK CAPITAL LETTER ALPHA WITH DASIA AND VARIA","	: 1F09 0300"},
/* 1F0C */ { "GREEK CAPITAL LETTER ALPHA WITH PSILI AND OXIA","	: 1F08 0301"},
/* 1F0D */ { "GREEK CAPITAL LETTER ALPHA WITH DASIA AND OXIA","	: 1F09 0301"},
/* 1F0E */ { "GREEK CAPITAL LETTER ALPHA WITH PSILI AND PERISPOMENI","	: 1F08 0342"},
/* 1F0F */ { "GREEK CAPITAL LETTER ALPHA WITH DASIA AND PERISPOMENI","	: 1F09 0342"},
/* 1F10 */ { "GREEK SMALL LETTER EPSILON WITH PSILI","	: 03B5 0313"},
/* 1F11 */ { "GREEK SMALL LETTER EPSILON WITH DASIA","	: 03B5 0314"},
/* 1F12 */ { "GREEK SMALL LETTER EPSILON WITH PSILI AND VARIA","	: 1F10 0300"},
/* 1F13 */ { "GREEK SMALL LETTER EPSILON WITH DASIA AND VARIA","	: 1F11 0300"},
/* 1F14 */ { "GREEK SMALL LETTER EPSILON WITH PSILI AND OXIA","	: 1F10 0301"},
/* 1F15 */ { "GREEK SMALL LETTER EPSILON WITH DASIA AND OXIA","	: 1F11 0301"},
/* 1F16 */ { NULL,NULL},
/* 1F17 */ { NULL,NULL},
/* 1F18 */ { "GREEK CAPITAL LETTER EPSILON WITH PSILI","	: 0395 0313"},
/* 1F19 */ { "GREEK CAPITAL LETTER EPSILON WITH DASIA","	: 0395 0314"},
/* 1F1A */ { "GREEK CAPITAL LETTER EPSILON WITH PSILI AND VARIA","	: 1F18 0300"},
/* 1F1B */ { "GREEK CAPITAL LETTER EPSILON WITH DASIA AND VARIA","	: 1F19 0300"},
/* 1F1C */ { "GREEK CAPITAL LETTER EPSILON WITH PSILI AND OXIA","	: 1F18 0301"},
/* 1F1D */ { "GREEK CAPITAL LETTER EPSILON WITH DASIA AND OXIA","	: 1F19 0301"},
/* 1F1E */ { NULL,NULL},
/* 1F1F */ { NULL,NULL},
/* 1F20 */ { "GREEK SMALL LETTER ETA WITH PSILI","	: 03B7 0313"},
/* 1F21 */ { "GREEK SMALL LETTER ETA WITH DASIA","	: 03B7 0314"},
/* 1F22 */ { "GREEK SMALL LETTER ETA WITH PSILI AND VARIA","	: 1F20 0300"},
/* 1F23 */ { "GREEK SMALL LETTER ETA WITH DASIA AND VARIA","	: 1F21 0300"},
/* 1F24 */ { "GREEK SMALL LETTER ETA WITH PSILI AND OXIA","	: 1F20 0301"},
/* 1F25 */ { "GREEK SMALL LETTER ETA WITH DASIA AND OXIA","	: 1F21 0301"},
/* 1F26 */ { "GREEK SMALL LETTER ETA WITH PSILI AND PERISPOMENI","	: 1F20 0342"},
/* 1F27 */ { "GREEK SMALL LETTER ETA WITH DASIA AND PERISPOMENI","	: 1F21 0342"},
/* 1F28 */ { "GREEK CAPITAL LETTER ETA WITH PSILI","	: 0397 0313"},
/* 1F29 */ { "GREEK CAPITAL LETTER ETA WITH DASIA","	: 0397 0314"},
/* 1F2A */ { "GREEK CAPITAL LETTER ETA WITH PSILI AND VARIA","	: 1F28 0300"},
/* 1F2B */ { "GREEK CAPITAL LETTER ETA WITH DASIA AND VARIA","	: 1F29 0300"},
/* 1F2C */ { "GREEK CAPITAL LETTER ETA WITH PSILI AND OXIA","	: 1F28 0301"},
/* 1F2D */ { "GREEK CAPITAL LETTER ETA WITH DASIA AND OXIA","	: 1F29 0301"},
/* 1F2E */ { "GREEK CAPITAL LETTER ETA WITH PSILI AND PERISPOMENI","	: 1F28 0342"},
/* 1F2F */ { "GREEK CAPITAL LETTER ETA WITH DASIA AND PERISPOMENI","	: 1F29 0342"},
/* 1F30 */ { "GREEK SMALL LETTER IOTA WITH PSILI","	: 03B9 0313"},
/* 1F31 */ { "GREEK SMALL LETTER IOTA WITH DASIA","	: 03B9 0314"},
/* 1F32 */ { "GREEK SMALL LETTER IOTA WITH PSILI AND VARIA","	: 1F30 0300"},
/* 1F33 */ { "GREEK SMALL LETTER IOTA WITH DASIA AND VARIA","	: 1F31 0300"},
/* 1F34 */ { "GREEK SMALL LETTER IOTA WITH PSILI AND OXIA","	: 1F30 0301"},
/* 1F35 */ { "GREEK SMALL LETTER IOTA WITH DASIA AND OXIA","	: 1F31 0301"},
/* 1F36 */ { "GREEK SMALL LETTER IOTA WITH PSILI AND PERISPOMENI","	: 1F30 0342"},
/* 1F37 */ { "GREEK SMALL LETTER IOTA WITH DASIA AND PERISPOMENI","	: 1F31 0342"},
/* 1F38 */ { "GREEK CAPITAL LETTER IOTA WITH PSILI","	: 0399 0313"},
/* 1F39 */ { "GREEK CAPITAL LETTER IOTA WITH DASIA","	: 0399 0314"},
/* 1F3A */ { "GREEK CAPITAL LETTER IOTA WITH PSILI AND VARIA","	: 1F38 0300"},
/* 1F3B */ { "GREEK CAPITAL LETTER IOTA WITH DASIA AND VARIA","	: 1F39 0300"},
/* 1F3C */ { "GREEK CAPITAL LETTER IOTA WITH PSILI AND OXIA","	: 1F38 0301"},
/* 1F3D */ { "GREEK CAPITAL LETTER IOTA WITH DASIA AND OXIA","	: 1F39 0301"},
/* 1F3E */ { "GREEK CAPITAL LETTER IOTA WITH PSILI AND PERISPOMENI","	: 1F38 0342"},
/* 1F3F */ { "GREEK CAPITAL LETTER IOTA WITH DASIA AND PERISPOMENI","	: 1F39 0342"},
/* 1F40 */ { "GREEK SMALL LETTER OMICRON WITH PSILI","	: 03BF 0313"},
/* 1F41 */ { "GREEK SMALL LETTER OMICRON WITH DASIA","	: 03BF 0314"},
/* 1F42 */ { "GREEK SMALL LETTER OMICRON WITH PSILI AND VARIA","	: 1F40 0300"},
/* 1F43 */ { "GREEK SMALL LETTER OMICRON WITH DASIA AND VARIA","	: 1F41 0300"},
/* 1F44 */ { "GREEK SMALL LETTER OMICRON WITH PSILI AND OXIA","	: 1F40 0301"},
/* 1F45 */ { "GREEK SMALL LETTER OMICRON WITH DASIA AND OXIA","	: 1F41 0301"},
/* 1F46 */ { NULL,NULL},
/* 1F47 */ { NULL,NULL},
/* 1F48 */ { "GREEK CAPITAL LETTER OMICRON WITH PSILI","	: 039F 0313"},
/* 1F49 */ { "GREEK CAPITAL LETTER OMICRON WITH DASIA","	: 039F 0314"},
/* 1F4A */ { "GREEK CAPITAL LETTER OMICRON WITH PSILI AND VARIA","	: 1F48 0300"},
/* 1F4B */ { "GREEK CAPITAL LETTER OMICRON WITH DASIA AND VARIA","	: 1F49 0300"},
/* 1F4C */ { "GREEK CAPITAL LETTER OMICRON WITH PSILI AND OXIA","	: 1F48 0301"},
/* 1F4D */ { "GREEK CAPITAL LETTER OMICRON WITH DASIA AND OXIA","	: 1F49 0301"},
/* 1F4E */ { NULL,NULL},
/* 1F4F */ { NULL,NULL},
/* 1F50 */ { "GREEK SMALL LETTER UPSILON WITH PSILI","	: 03C5 0313"},
/* 1F51 */ { "GREEK SMALL LETTER UPSILON WITH DASIA","	: 03C5 0314"},
/* 1F52 */ { "GREEK SMALL LETTER UPSILON WITH PSILI AND VARIA","	: 1F50 0300"},
/* 1F53 */ { "GREEK SMALL LETTER UPSILON WITH DASIA AND VARIA","	: 1F51 0300"},
/* 1F54 */ { "GREEK SMALL LETTER UPSILON WITH PSILI AND OXIA","	: 1F50 0301"},
/* 1F55 */ { "GREEK SMALL LETTER UPSILON WITH DASIA AND OXIA","	: 1F51 0301"},
/* 1F56 */ { "GREEK SMALL LETTER UPSILON WITH PSILI AND PERISPOMENI","	: 1F50 0342"},
/* 1F57 */ { "GREEK SMALL LETTER UPSILON WITH DASIA AND PERISPOMENI","	: 1F51 0342"},
/* 1F58 */ { NULL,NULL},
/* 1F59 */ { "GREEK CAPITAL LETTER UPSILON WITH DASIA","	: 03A5 0314"},
/* 1F5A */ { NULL,NULL},
/* 1F5B */ { "GREEK CAPITAL LETTER UPSILON WITH DASIA AND VARIA","	: 1F59 0300"},
/* 1F5C */ { NULL,NULL},
/* 1F5D */ { "GREEK CAPITAL LETTER UPSILON WITH DASIA AND OXIA","	: 1F59 0301"},
/* 1F5E */ { NULL,NULL},
/* 1F5F */ { "GREEK CAPITAL LETTER UPSILON WITH DASIA AND PERISPOMENI","	: 1F59 0342"},
/* 1F60 */ { "GREEK SMALL LETTER OMEGA WITH PSILI","	: 03C9 0313"},
/* 1F61 */ { "GREEK SMALL LETTER OMEGA WITH DASIA","	: 03C9 0314"},
/* 1F62 */ { "GREEK SMALL LETTER OMEGA WITH PSILI AND VARIA","	: 1F60 0300"},
/* 1F63 */ { "GREEK SMALL LETTER OMEGA WITH DASIA AND VARIA","	: 1F61 0300"},
/* 1F64 */ { "GREEK SMALL LETTER OMEGA WITH PSILI AND OXIA","	: 1F60 0301"},
/* 1F65 */ { "GREEK SMALL LETTER OMEGA WITH DASIA AND OXIA","	: 1F61 0301"},
/* 1F66 */ { "GREEK SMALL LETTER OMEGA WITH PSILI AND PERISPOMENI","	: 1F60 0342"},
/* 1F67 */ { "GREEK SMALL LETTER OMEGA WITH DASIA AND PERISPOMENI","	: 1F61 0342"},
/* 1F68 */ { "GREEK CAPITAL LETTER OMEGA WITH PSILI","	: 03A9 0313"},
/* 1F69 */ { "GREEK CAPITAL LETTER OMEGA WITH DASIA","	: 03A9 0314"},
/* 1F6A */ { "GREEK CAPITAL LETTER OMEGA WITH PSILI AND VARIA","	: 1F68 0300"},
/* 1F6B */ { "GREEK CAPITAL LETTER OMEGA WITH DASIA AND VARIA","	: 1F69 0300"},
/* 1F6C */ { "GREEK CAPITAL LETTER OMEGA WITH PSILI AND OXIA","	: 1F68 0301"},
/* 1F6D */ { "GREEK CAPITAL LETTER OMEGA WITH DASIA AND OXIA","	: 1F69 0301"},
/* 1F6E */ { "GREEK CAPITAL LETTER OMEGA WITH PSILI AND PERISPOMENI","	: 1F68 0342"},
/* 1F6F */ { "GREEK CAPITAL LETTER OMEGA WITH DASIA AND PERISPOMENI","	: 1F69 0342"},
/* 1F70 */ { "GREEK SMALL LETTER ALPHA WITH VARIA","	: 03B1 0300"},
/* 1F71 */ { "GREEK SMALL LETTER ALPHA WITH OXIA","	: 03AC greek small letter alpha with tonos"},
/* 1F72 */ { "GREEK SMALL LETTER EPSILON WITH VARIA","	: 03B5 0300"},
/* 1F73 */ { "GREEK SMALL LETTER EPSILON WITH OXIA","	: 03AD greek small letter epsilon with tonos"},
/* 1F74 */ { "GREEK SMALL LETTER ETA WITH VARIA","	: 03B7 0300"},
/* 1F75 */ { "GREEK SMALL LETTER ETA WITH OXIA","	: 03AE greek small letter eta with tonos"},
/* 1F76 */ { "GREEK SMALL LETTER IOTA WITH VARIA","	: 03B9 0300"},
/* 1F77 */ { "GREEK SMALL LETTER IOTA WITH OXIA","	: 03AF greek small letter iota with tonos"},
/* 1F78 */ { "GREEK SMALL LETTER OMICRON WITH VARIA","	: 03BF 0300"},
/* 1F79 */ { "GREEK SMALL LETTER OMICRON WITH OXIA","	: 03CC greek small letter omicron with tonos"},
/* 1F7A */ { "GREEK SMALL LETTER UPSILON WITH VARIA","	: 03C5 0300"},
/* 1F7B */ { "GREEK SMALL LETTER UPSILON WITH OXIA","	: 03CD greek small letter upsilon with tonos"},
/* 1F7C */ { "GREEK SMALL LETTER OMEGA WITH VARIA","	: 03C9 0300"},
/* 1F7D */ { "GREEK SMALL LETTER OMEGA WITH OXIA","	: 03CE greek small letter omega with tonos"},
/* 1F7E */ { NULL,NULL},
/* 1F7F */ { NULL,NULL},
/* 1F80 */ { "GREEK SMALL LETTER ALPHA WITH PSILI AND YPOGEGRAMMENI","	: 1F00 0345"},
/* 1F81 */ { "GREEK SMALL LETTER ALPHA WITH DASIA AND YPOGEGRAMMENI","	: 1F01 0345"},
/* 1F82 */ { "GREEK SMALL LETTER ALPHA WITH PSILI AND VARIA AND YPOGEGRAMMENI","	: 1F02 0345"},
/* 1F83 */ { "GREEK SMALL LETTER ALPHA WITH DASIA AND VARIA AND YPOGEGRAMMENI","	: 1F03 0345"},
/* 1F84 */ { "GREEK SMALL LETTER ALPHA WITH PSILI AND OXIA AND YPOGEGRAMMENI","	: 1F04 0345"},
/* 1F85 */ { "GREEK SMALL LETTER ALPHA WITH DASIA AND OXIA AND YPOGEGRAMMENI","	: 1F05 0345"},
/* 1F86 */ { "GREEK SMALL LETTER ALPHA WITH PSILI AND PERISPOMENI AND YPOGEGRAMMENI","	: 1F06 0345"},
/* 1F87 */ { "GREEK SMALL LETTER ALPHA WITH DASIA AND PERISPOMENI AND YPOGEGRAMMENI","	: 1F07 0345"},
/* 1F88 */ { "GREEK CAPITAL LETTER ALPHA WITH PSILI AND PROSGEGRAMMENI","	: 1F08 0345"},
/* 1F89 */ { "GREEK CAPITAL LETTER ALPHA WITH DASIA AND PROSGEGRAMMENI","	: 1F09 0345"},
/* 1F8A */ { "GREEK CAPITAL LETTER ALPHA WITH PSILI AND VARIA AND PROSGEGRAMMENI","	: 1F0A 0345"},
/* 1F8B */ { "GREEK CAPITAL LETTER ALPHA WITH DASIA AND VARIA AND PROSGEGRAMMENI","	: 1F0B 0345"},
/* 1F8C */ { "GREEK CAPITAL LETTER ALPHA WITH PSILI AND OXIA AND PROSGEGRAMMENI","	: 1F0C 0345"},
/* 1F8D */ { "GREEK CAPITAL LETTER ALPHA WITH DASIA AND OXIA AND PROSGEGRAMMENI","	: 1F0D 0345"},
/* 1F8E */ { "GREEK CAPITAL LETTER ALPHA WITH PSILI AND PERISPOMENI AND PROSGEGRAMMENI","	: 1F0E 0345"},
/* 1F8F */ { "GREEK CAPITAL LETTER ALPHA WITH DASIA AND PERISPOMENI AND PROSGEGRAMMENI","	: 1F0F 0345"},
/* 1F90 */ { "GREEK SMALL LETTER ETA WITH PSILI AND YPOGEGRAMMENI","	: 1F20 0345"},
/* 1F91 */ { "GREEK SMALL LETTER ETA WITH DASIA AND YPOGEGRAMMENI","	: 1F21 0345"},
/* 1F92 */ { "GREEK SMALL LETTER ETA WITH PSILI AND VARIA AND YPOGEGRAMMENI","	: 1F22 0345"},
/* 1F93 */ { "GREEK SMALL LETTER ETA WITH DASIA AND VARIA AND YPOGEGRAMMENI","	: 1F23 0345"},
/* 1F94 */ { "GREEK SMALL LETTER ETA WITH PSILI AND OXIA AND YPOGEGRAMMENI","	: 1F24 0345"},
/* 1F95 */ { "GREEK SMALL LETTER ETA WITH DASIA AND OXIA AND YPOGEGRAMMENI","	: 1F25 0345"},
/* 1F96 */ { "GREEK SMALL LETTER ETA WITH PSILI AND PERISPOMENI AND YPOGEGRAMMENI","	: 1F26 0345"},
/* 1F97 */ { "GREEK SMALL LETTER ETA WITH DASIA AND PERISPOMENI AND YPOGEGRAMMENI","	: 1F27 0345"},
/* 1F98 */ { "GREEK CAPITAL LETTER ETA WITH PSILI AND PROSGEGRAMMENI","	: 1F28 0345"},
/* 1F99 */ { "GREEK CAPITAL LETTER ETA WITH DASIA AND PROSGEGRAMMENI","	: 1F29 0345"},
/* 1F9A */ { "GREEK CAPITAL LETTER ETA WITH PSILI AND VARIA AND PROSGEGRAMMENI","	: 1F2A 0345"},
/* 1F9B */ { "GREEK CAPITAL LETTER ETA WITH DASIA AND VARIA AND PROSGEGRAMMENI","	: 1F2B 0345"},
/* 1F9C */ { "GREEK CAPITAL LETTER ETA WITH PSILI AND OXIA AND PROSGEGRAMMENI","	: 1F2C 0345"},
/* 1F9D */ { "GREEK CAPITAL LETTER ETA WITH DASIA AND OXIA AND PROSGEGRAMMENI","	: 1F2D 0345"},
/* 1F9E */ { "GREEK CAPITAL LETTER ETA WITH PSILI AND PERISPOMENI AND PROSGEGRAMMENI","	: 1F2E 0345"},
/* 1F9F */ { "GREEK CAPITAL LETTER ETA WITH DASIA AND PERISPOMENI AND PROSGEGRAMMENI","	: 1F2F 0345"},
/* 1FA0 */ { "GREEK SMALL LETTER OMEGA WITH PSILI AND YPOGEGRAMMENI","	: 1F60 0345"},
/* 1FA1 */ { "GREEK SMALL LETTER OMEGA WITH DASIA AND YPOGEGRAMMENI","	: 1F61 0345"},
/* 1FA2 */ { "GREEK SMALL LETTER OMEGA WITH PSILI AND VARIA AND YPOGEGRAMMENI","	: 1F62 0345"},
/* 1FA3 */ { "GREEK SMALL LETTER OMEGA WITH DASIA AND VARIA AND YPOGEGRAMMENI","	: 1F63 0345"},
/* 1FA4 */ { "GREEK SMALL LETTER OMEGA WITH PSILI AND OXIA AND YPOGEGRAMMENI","	: 1F64 0345"},
/* 1FA5 */ { "GREEK SMALL LETTER OMEGA WITH DASIA AND OXIA AND YPOGEGRAMMENI","	: 1F65 0345"},
/* 1FA6 */ { "GREEK SMALL LETTER OMEGA WITH PSILI AND PERISPOMENI AND YPOGEGRAMMENI","	: 1F66 0345"},
/* 1FA7 */ { "GREEK SMALL LETTER OMEGA WITH DASIA AND PERISPOMENI AND YPOGEGRAMMENI","	: 1F67 0345"},
/* 1FA8 */ { "GREEK CAPITAL LETTER OMEGA WITH PSILI AND PROSGEGRAMMENI","	: 1F68 0345"},
/* 1FA9 */ { "GREEK CAPITAL LETTER OMEGA WITH DASIA AND PROSGEGRAMMENI","	: 1F69 0345"},
/* 1FAA */ { "GREEK CAPITAL LETTER OMEGA WITH PSILI AND VARIA AND PROSGEGRAMMENI","	: 1F6A 0345"},
/* 1FAB */ { "GREEK CAPITAL LETTER OMEGA WITH DASIA AND VARIA AND PROSGEGRAMMENI","	: 1F6B 0345"},
/* 1FAC */ { "GREEK CAPITAL LETTER OMEGA WITH PSILI AND OXIA AND PROSGEGRAMMENI","	: 1F6C 0345"},
/* 1FAD */ { "GREEK CAPITAL LETTER OMEGA WITH DASIA AND OXIA AND PROSGEGRAMMENI","	: 1F6D 0345"},
/* 1FAE */ { "GREEK CAPITAL LETTER OMEGA WITH PSILI AND PERISPOMENI AND PROSGEGRAMMENI","	: 1F6E 0345"},
/* 1FAF */ { "GREEK CAPITAL LETTER OMEGA WITH DASIA AND PERISPOMENI AND PROSGEGRAMMENI","	: 1F6F 0345"},
/* 1FB0 */ { "GREEK SMALL LETTER ALPHA WITH VRACHY","	: 03B1 0306"},
/* 1FB1 */ { "GREEK SMALL LETTER ALPHA WITH MACRON","	: 03B1 0304"},
/* 1FB2 */ { "GREEK SMALL LETTER ALPHA WITH VARIA AND YPOGEGRAMMENI","	: 1F70 0345"},
/* 1FB3 */ { "GREEK SMALL LETTER ALPHA WITH YPOGEGRAMMENI","	: 03B1 0345"},
/* 1FB4 */ { "GREEK SMALL LETTER ALPHA WITH OXIA AND YPOGEGRAMMENI","	: 03AC 0345"},
/* 1FB5 */ { NULL,NULL},
/* 1FB6 */ { "GREEK SMALL LETTER ALPHA WITH PERISPOMENI","	: 03B1 0342"},
/* 1FB7 */ { "GREEK SMALL LETTER ALPHA WITH PERISPOMENI AND YPOGEGRAMMENI","	: 1FB6 0345"},
/* 1FB8 */ { "GREEK CAPITAL LETTER ALPHA WITH VRACHY","	: 0391 0306"},
/* 1FB9 */ { "GREEK CAPITAL LETTER ALPHA WITH MACRON","	: 0391 0304"},
/* 1FBA */ { "GREEK CAPITAL LETTER ALPHA WITH VARIA","	: 0391 0300"},
/* 1FBB */ { "GREEK CAPITAL LETTER ALPHA WITH OXIA","	: 0386 greek capital letter alpha with tonos"},
/* 1FBC */ { "GREEK CAPITAL LETTER ALPHA WITH PROSGEGRAMMENI","	: 0391 0345"},
/* 1FBD */ { "GREEK KORONIS","	# 0020 0313"},
/* 1FBE */ { "GREEK PROSGEGRAMMENI","	: 03B9 greek small letter iota"},
/* 1FBF */ { "GREEK PSILI","	x (modifier letter apostrophe - 02BC)\n"
	"	# 0020 0313"},
/* 1FC0 */ { "GREEK PERISPOMENI","	# 0020 0342"},
/* 1FC1 */ { "GREEK DIALYTIKA AND PERISPOMENI","	: 00A8 0342"},
/* 1FC2 */ { "GREEK SMALL LETTER ETA WITH VARIA AND YPOGEGRAMMENI","	: 1F74 0345"},
/* 1FC3 */ { "GREEK SMALL LETTER ETA WITH YPOGEGRAMMENI","	: 03B7 0345"},
/* 1FC4 */ { "GREEK SMALL LETTER ETA WITH OXIA AND YPOGEGRAMMENI","	: 03AE 0345"},
/* 1FC5 */ { NULL,NULL},
/* 1FC6 */ { "GREEK SMALL LETTER ETA WITH PERISPOMENI","	: 03B7 0342"},
/* 1FC7 */ { "GREEK SMALL LETTER ETA WITH PERISPOMENI AND YPOGEGRAMMENI","	: 1FC6 0345"},
/* 1FC8 */ { "GREEK CAPITAL LETTER EPSILON WITH VARIA","	: 0395 0300"},
/* 1FC9 */ { "GREEK CAPITAL LETTER EPSILON WITH OXIA","	: 0388 greek capital letter epsilon with tonos"},
/* 1FCA */ { "GREEK CAPITAL LETTER ETA WITH VARIA","	: 0397 0300"},
/* 1FCB */ { "GREEK CAPITAL LETTER ETA WITH OXIA","	: 0389 greek capital letter eta with tonos"},
/* 1FCC */ { "GREEK CAPITAL LETTER ETA WITH PROSGEGRAMMENI","	: 0397 0345"},
/* 1FCD */ { "GREEK PSILI AND VARIA","	: 1FBF 0300"},
/* 1FCE */ { "GREEK PSILI AND OXIA","	: 1FBF 0301"},
/* 1FCF */ { "GREEK PSILI AND PERISPOMENI","	: 1FBF 0342"},
/* 1FD0 */ { "GREEK SMALL LETTER IOTA WITH VRACHY","	: 03B9 0306"},
/* 1FD1 */ { "GREEK SMALL LETTER IOTA WITH MACRON","	: 03B9 0304"},
/* 1FD2 */ { "GREEK SMALL LETTER IOTA WITH DIALYTIKA AND VARIA","	: 03CA 0300"},
/* 1FD3 */ { "GREEK SMALL LETTER IOTA WITH DIALYTIKA AND OXIA","	: 0390 greek small letter iota with dialytika and tonos"},
/* 1FD4 */ { NULL,NULL},
/* 1FD5 */ { NULL,NULL},
/* 1FD6 */ { "GREEK SMALL LETTER IOTA WITH PERISPOMENI","	: 03B9 0342"},
/* 1FD7 */ { "GREEK SMALL LETTER IOTA WITH DIALYTIKA AND PERISPOMENI","	: 03CA 0342"},
/* 1FD8 */ { "GREEK CAPITAL LETTER IOTA WITH VRACHY","	: 0399 0306"},
/* 1FD9 */ { "GREEK CAPITAL LETTER IOTA WITH MACRON","	: 0399 0304"},
/* 1FDA */ { "GREEK CAPITAL LETTER IOTA WITH VARIA","	: 0399 0300"},
/* 1FDB */ { "GREEK CAPITAL LETTER IOTA WITH OXIA","	: 038A greek capital letter iota with tonos"},
/* 1FDC */ { NULL,NULL},
/* 1FDD */ { "GREEK DASIA AND VARIA","	: 1FFE 0300"},
/* 1FDE */ { "GREEK DASIA AND OXIA","	: 1FFE 0301"},
/* 1FDF */ { "GREEK DASIA AND PERISPOMENI","	: 1FFE 0342"},
/* 1FE0 */ { "GREEK SMALL LETTER UPSILON WITH VRACHY","	: 03C5 0306"},
/* 1FE1 */ { "GREEK SMALL LETTER UPSILON WITH MACRON","	: 03C5 0304"},
/* 1FE2 */ { "GREEK SMALL LETTER UPSILON WITH DIALYTIKA AND VARIA","	: 03CB 0300"},
/* 1FE3 */ { "GREEK SMALL LETTER UPSILON WITH DIALYTIKA AND OXIA","	: 03B0 greek small letter upsilon with dialytika and tonos"},
/* 1FE4 */ { "GREEK SMALL LETTER RHO WITH PSILI","	: 03C1 0313"},
/* 1FE5 */ { "GREEK SMALL LETTER RHO WITH DASIA","	: 03C1 0314"},
/* 1FE6 */ { "GREEK SMALL LETTER UPSILON WITH PERISPOMENI","	: 03C5 0342"},
/* 1FE7 */ { "GREEK SMALL LETTER UPSILON WITH DIALYTIKA AND PERISPOMENI","	: 03CB 0342"},
/* 1FE8 */ { "GREEK CAPITAL LETTER UPSILON WITH VRACHY","	: 03A5 0306"},
/* 1FE9 */ { "GREEK CAPITAL LETTER UPSILON WITH MACRON","	: 03A5 0304"},
/* 1FEA */ { "GREEK CAPITAL LETTER UPSILON WITH VARIA","	: 03A5 0300"},
/* 1FEB */ { "GREEK CAPITAL LETTER UPSILON WITH OXIA","	: 038E greek capital letter upsilon with tonos"},
/* 1FEC */ { "GREEK CAPITAL LETTER RHO WITH DASIA","	: 03A1 0314"},
/* 1FED */ { "GREEK DIALYTIKA AND VARIA","	: 00A8 0300"},
/* 1FEE */ { "GREEK DIALYTIKA AND OXIA","	: 0385 greek dialytika tonos"},
/* 1FEF */ { "GREEK VARIA","	: 0060 grave accent"},
/* 1FF0 */ { NULL,NULL},
/* 1FF1 */ { NULL,NULL},
/* 1FF2 */ { "GREEK SMALL LETTER OMEGA WITH VARIA AND YPOGEGRAMMENI","	: 1F7C 0345"},
/* 1FF3 */ { "GREEK SMALL LETTER OMEGA WITH YPOGEGRAMMENI","	: 03C9 0345"},
/* 1FF4 */ { "GREEK SMALL LETTER OMEGA WITH OXIA AND YPOGEGRAMMENI","	: 03CE 0345"},
/* 1FF5 */ { NULL,NULL},
/* 1FF6 */ { "GREEK SMALL LETTER OMEGA WITH PERISPOMENI","	: 03C9 0342"},
/* 1FF7 */ { "GREEK SMALL LETTER OMEGA WITH PERISPOMENI AND YPOGEGRAMMENI","	: 1FF6 0345"},
/* 1FF8 */ { "GREEK CAPITAL LETTER OMICRON WITH VARIA","	: 039F 0300"},
/* 1FF9 */ { "GREEK CAPITAL LETTER OMICRON WITH OXIA","	: 038C greek capital letter omicron with tonos"},
/* 1FFA */ { "GREEK CAPITAL LETTER OMEGA WITH VARIA","	: 03A9 0300"},
/* 1FFB */ { "GREEK CAPITAL LETTER OMEGA WITH OXIA","	: 038F greek capital letter omega with tonos"},
/* 1FFC */ { "GREEK CAPITAL LETTER OMEGA WITH PROSGEGRAMMENI","	: 03A9 0345"},
/* 1FFD */ { "GREEK OXIA","	: 00B4 acute accent"},
/* 1FFE */ { "GREEK DASIA","	x (modifier letter reversed comma - 02BD)\n"
	"	# 0020 0314"},
/* 1FFF */ { NULL,NULL}
};

UN_DLL_LOCAL
static const struct unicode_nameannot una_00_20[] = {
/* 2000 */ { "EN QUAD","	: 2002 en space"},
/* 2001 */ { "EM QUAD","	= mutton quad\n"
	"	: 2003 em space"},
/* 2002 */ { "EN SPACE","	= nut\n"
	"	* half an em\n"
	"	# 0020 space"},
/* 2003 */ { "EM SPACE","	= mutton\n"
	"	* nominally, a space equal to the type size in points\n"
	"	* may scale by the condensation factor of a font\n"
	"	# 0020 space"},
/* 2004 */ { "THREE-PER-EM SPACE","	= thick space\n"
	"	# 0020 space"},
/* 2005 */ { "FOUR-PER-EM SPACE","	= mid space\n"
	"	# 0020 space"},
/* 2006 */ { "SIX-PER-EM SPACE","	* in computer typography sometimes equated to thin space\n"
	"	# 0020 space"},
/* 2007 */ { "FIGURE SPACE","	* space equal to tabular width of a font\n"
	"	* this is equivalent to the digit width of fonts with fixed-width digits\n"
	"	# <noBreak> 0020"},
/* 2008 */ { "PUNCTUATION SPACE","	* space equal to narrow punctuation of a font\n"
	"	# 0020 space"},
/* 2009 */ { "THIN SPACE","	* a fifth of an em (or sometimes a sixth)\n"
	"	x (narrow no-break space - 202F)\n"
	"	# 0020 space"},
/* 200A */ { "HAIR SPACE","	* thinner than a thin space\n"
	"	* in traditional typography, the thinnest space available\n"
	"	# 0020 space"},
/* 200B */ { "ZERO WIDTH SPACE","	* commonly abbreviated ZWSP\n"
	"	* this character is intended for invisible word separation and for line break control; it has no width, but its presence between two characters does not prevent increased letter spacing in justification"},
/* 200C */ { "ZERO WIDTH NON-JOINER","	* commonly abbreviated ZWNJ"},
/* 200D */ { "ZERO WIDTH JOINER","	* commonly abbreviated ZWJ"},
/* 200E */ { "LEFT-TO-RIGHT MARK","	* commonly abbreviated LRM"},
/* 200F */ { "RIGHT-TO-LEFT MARK","	* commonly abbreviated RLM\n"
	"	x (arabic letter mark - 061C)"},
/* 2010 */ { "HYPHEN","	x (hyphen-minus - 002D)\n"
	"	x (soft hyphen - 00AD)"},
/* 2011 */ { "NON-BREAKING HYPHEN","	# <noBreak> 2010"},
/* 2012 */ { "FIGURE DASH",NULL},
/* 2013 */ { "EN DASH",NULL},
/* 2014 */ { "EM DASH","	* may be used in pairs to offset parenthetical text\n"
	"	x (two-em dash - 2E3A)\n"
	"	x (katakana-hiragana prolonged sound mark - 30FC)"},
/* 2015 */ { "HORIZONTAL BAR","	= quotation dash\n"
	"	* long dash introducing quoted text"},
/* 2016 */ { "DOUBLE VERTICAL LINE","	* used in pairs to indicate norm of a matrix\n"
	"	x (combining double vertical stroke overlay - 20E6)\n"
	"	x (parallel to - 2225)\n"
	"	x (double vertical bar - 23F8)"},
/* 2017 */ { "DOUBLE LOW LINE","	* this is a spacing character\n"
	"	x (low line - 005F)\n"
	"	x (combining double low line - 0333)\n"
	"	# 0020 0333"},
/* 2018 */ { "LEFT SINGLE QUOTATION MARK","	= single turned comma quotation mark\n"
	"	* this is the preferred character (as opposed to 201B)\n"
	"	x (apostrophe - 0027)\n"
	"	x (modifier letter turned comma - 02BB)\n"
	"	x (heavy single turned comma quotation mark ornament - 275B)\n"
	"	~ 2018 FE00 non-fullwidth form\n"
	"	~ 2018 FE01 right-justified fullwidth form"},
/* 2019 */ { "RIGHT SINGLE QUOTATION MARK","	= single comma quotation mark\n"
	"	* this is the preferred character to use for apostrophe\n"
	"	x (apostrophe - 0027)\n"
	"	x (modifier letter apostrophe - 02BC)\n"
	"	x (heavy single comma quotation mark ornament - 275C)\n"
	"	~ 2019 FE00 non-fullwidth form\n"
	"	~ 2019 FE01 left-justified fullwidth form"},
/* 201A */ { "SINGLE LOW-9 QUOTATION MARK","	= low single comma quotation mark\n"
	"	* used as opening single quotation mark in some languages"},
/* 201B */ { "SINGLE HIGH-REVERSED-9 QUOTATION MARK","	= single reversed comma quotation mark\n"
	"	* has same semantic as 2018, but differs in appearance\n"
	"	x (modifier letter reversed comma - 02BD)"},
/* 201C */ { "LEFT DOUBLE QUOTATION MARK","	= double turned comma quotation mark\n"
	"	* this is the preferred character (as opposed to 201F)\n"
	"	x (quotation mark - 0022)\n"
	"	x (heavy double turned comma quotation mark ornament - 275D)\n"
	"	x (reversed double prime quotation mark - 301D)\n"
	"	~ 201C FE00 non-fullwidth form\n"
	"	~ 201C FE01 right-justified fullwidth form"},
/* 201D */ { "RIGHT DOUBLE QUOTATION MARK","	= double comma quotation mark\n"
	"	x (quotation mark - 0022)\n"
	"	x (double prime - 2033)\n"
	"	x (heavy double comma quotation mark ornament - 275E)\n"
	"	x (double prime quotation mark - 301E)\n"
	"	~ 201D FE00 non-fullwidth form\n"
	"	~ 201D FE01 left-justified fullwidth form"},
/* 201E */ { "DOUBLE LOW-9 QUOTATION MARK","	= low double comma quotation mark\n"
	"	* used as opening double quotation mark in some languages\n"
	"	x (double low-reversed-9 quotation mark - 2E42)\n"
	"	x (low double prime quotation mark - 301F)"},
/* 201F */ { "DOUBLE HIGH-REVERSED-9 QUOTATION MARK","	= double reversed comma quotation mark\n"
	"	* has same semantic as 201C, but differs in appearance"},
/* 2020 */ { "DAGGER","	= obelisk, long cross, oblong cross\n"
	"	x (turned dagger - 2E38)"},
/* 2021 */ { "DOUBLE DAGGER","	= diesis, double obelisk\n"
	"	x (triple dagger - 2E4B)"},
/* 2022 */ { "BULLET","	= black small circle\n"
	"	x (middle dot - 00B7)\n"
	"	x (one dot leader - 2024)\n"
	"	x (bullet operator - 2219)\n"
	"	x (inverse bullet - 25D8)\n"
	"	x (white bullet - 25E6)"},
/* 2023 */ { "TRIANGULAR BULLET","	x (end of proof - 220E)\n"
	"	x (black right-pointing small triangle - 25B8)"},
/* 2024 */ { "ONE DOT LEADER","	* also used as an Armenian semicolon (mijaket)\n"
	"	x (middle dot - 00B7)\n"
	"	x (bullet - 2022)\n"
	"	x (bullet operator - 2219)\n"
	"	# 002E full stop"},
/* 2025 */ { "TWO DOT LEADER","	# 002E 002E"},
/* 2026 */ { "HORIZONTAL ELLIPSIS","	= three dot leader\n"
	"	x (vertical ellipsis - 22EE)\n"
	"	x (presentation form for vertical horizontal ellipsis - FE19)\n"
	"	# 002E 002E 002E"},
/* 2027 */ { "HYPHENATION POINT","	* visible symbol used to indicate correct positions for word breaking, as in dic·tion·ar·ies"},
/* 2028 */ { "LINE SEPARATOR","	* may be used to represent this semantic unambiguously"},
/* 2029 */ { "PARAGRAPH SEPARATOR","	* may be used to represent this semantic unambiguously"},
/* 202A */ { "LEFT-TO-RIGHT EMBEDDING","	* commonly abbreviated LRE"},
/* 202B */ { "RIGHT-TO-LEFT EMBEDDING","	* commonly abbreviated RLE"},
/* 202C */ { "POP DIRECTIONAL FORMATTING","	* commonly abbreviated PDF"},
/* 202D */ { "LEFT-TO-RIGHT OVERRIDE","	* commonly abbreviated LRO"},
/* 202E */ { "RIGHT-TO-LEFT OVERRIDE","	* commonly abbreviated RLO"},
/* 202F */ { "NARROW NO-BREAK SPACE","	* commonly abbreviated NNBSP\n"
	"	* a narrow form of a no-break space, typically the width of a thin space or a mid space\n"
	"	x (no-break space - 00A0)\n"
	"	x (four-per-em space - 2005)\n"
	"	x (thin space - 2009)\n"
	"	# <noBreak> 0020"},
/* 2030 */ { "PER MILLE SIGN","	= permille, per thousand\n"
	"	* used, for example, in measures of blood alcohol content, salinity, etc.\n"
	"	x (percent sign - 0025)\n"
	"	x (arabic-indic per mille sign - 0609)"},
/* 2031 */ { "PER TEN THOUSAND SIGN","	= permyriad\n"
	"	* percent of a percent, rarely used\n"
	"	x (percent sign - 0025)\n"
	"	x (arabic-indic per ten thousand sign - 060A)"},
/* 2032 */ { "PRIME","	= minutes, feet\n"
	"	x (apostrophe - 0027)\n"
	"	x (acute accent - 00B4)\n"
	"	x (modifier letter prime - 02B9)"},
/* 2033 */ { "DOUBLE PRIME","	= seconds, inches\n"
	"	x (quotation mark - 0022)\n"
	"	x (modifier letter double prime - 02BA)\n"
	"	x (right double quotation mark - 201D)\n"
	"	x (ditto mark - 3003)\n"
	"	x (double prime quotation mark - 301E)\n"
	"	# 2032 2032"},
/* 2034 */ { "TRIPLE PRIME","	= lines (old measure, 1/12 of an inch)\n"
	"	# 2032 2032 2032"},
/* 2035 */ { "REVERSED PRIME","	x (grave accent - 0060)"},
/* 2036 */ { "REVERSED DOUBLE PRIME","	x (reversed double prime quotation mark - 301D)\n"
	"	# 2035 2035"},
/* 2037 */ { "REVERSED TRIPLE PRIME","	# 2035 2035 2035"},
/* 2038 */ { "CARET","	x (up arrowhead - 2303)\n"
	"	x (modifier letter low circumflex accent - A788)"},
/* 2039 */ { "SINGLE LEFT-POINTING ANGLE QUOTATION MARK","	= left pointing single guillemet\n"
	"	* usually opening, sometimes closing\n"
	"	x (less-than sign - 003C)\n"
	"	x (left-pointing angle bracket - 2329)\n"
	"	x (left angle bracket - 3008)"},
/* 203A */ { "SINGLE RIGHT-POINTING ANGLE QUOTATION MARK","	= right pointing single guillemet\n"
	"	* usually closing, sometimes opening\n"
	"	x (greater-than sign - 003E)\n"
	"	x (right-pointing angle bracket - 232A)\n"
	"	x (right angle bracket - 3009)"},
/* 203B */ { "REFERENCE MARK","	= Japanese kome\n"
	"	= Urdu paragraph separator\n"
	"	x (tibetan ku ru kha bzhi mig can - 0FBF)\n"
	"	x 200AD"},
/* 203C */ { "DOUBLE EXCLAMATION MARK","	x (exclamation mark - 0021)\n"
	"	# 0021 0021"},
/* 203D */ { "INTERROBANG","	x (exclamation mark - 0021)\n"
	"	x (question mark - 003F)\n"
	"	x (inverted interrobang - 2E18)\n"
	"	x (heavy interrobang ornament - 1F679)"},
/* 203E */ { "OVERLINE","	= spacing overscore\n"
	"	# 0020 0305"},
/* 203F */ { "UNDERTIE","	= Greek enotikon\n"
	"	x (smile - 2323)"},
/* 2040 */ { "CHARACTER TIE","	= z notation sequence concatenation\n"
	"	x (frown - 2322)"},
/* 2041 */ { "CARET INSERTION POINT","	* proofreader's mark: insert here\n"
	"	x (right semidirect product - 22CC)"},
/* 2042 */ { "ASTERISM",NULL},
/* 2043 */ { "HYPHEN BULLET","	x (hyphen-minus - 002D)"},
/* 2044 */ { "FRACTION SLASH","	= solidus (in typography)\n"
	"	* for composing arbitrary fractions\n"
	"	x (solidus - 002F)\n"
	"	x (division slash - 2215)"},
/* 2045 */ { "LEFT SQUARE BRACKET WITH QUILL","	x (left vertical bar with quill - 2E20)\n"
	"	x (left square bracket with stroke - 2E55)"},
/* 2046 */ { "RIGHT SQUARE BRACKET WITH QUILL",NULL},
/* 2047 */ { "DOUBLE QUESTION MARK","	# 003F 003F"},
/* 2048 */ { "QUESTION EXCLAMATION MARK","	# 003F 0021"},
/* 2049 */ { "EXCLAMATION QUESTION MARK","	# 0021 003F"},
/* 204A */ { "TIRONIAN SIGN ET","	* Irish Gaelic, Old English, ...\n"
	"	x (ampersand - 0026)\n"
	"	x (tironian sign capital et - 2E52)\n"
	"	x (script ligature et ornament - 1F670)"},
/* 204B */ { "REVERSED PILCROW SIGN","	x (pilcrow sign - 00B6)\n"
	"	x (paragraphus mark - 2E4D)"},
/* 204C */ { "BLACK LEFTWARDS BULLET",NULL},
/* 204D */ { "BLACK RIGHTWARDS BULLET",NULL},
/* 204E */ { "LOW ASTERISK","	x (asterisk - 002A)\n"
	"	x (combining asterisk below - 0359)"},
/* 204F */ { "REVERSED SEMICOLON","	* used occasionally in Sindhi when Sindhi is written in the Arabic script\n"
	"	x (semicolon - 003B)\n"
	"	x (arabic semicolon - 061B)"},
/* 2050 */ { "CLOSE UP","	* editing mark\n"
	"	x (modifier breve with inverted breve - AB5B)"},
/* 2051 */ { "TWO ASTERISKS ALIGNED VERTICALLY",NULL},
/* 2052 */ { "COMMERCIAL MINUS SIGN","	= abzüglich (German), med avdrag av (Swedish), piska (Swedish, \"whip\")\n"
	"	* a common glyph variant and fallback representation looks like ./.\n"
	"	* may also be used as a dingbat to indicate correctness\n"
	"	* used in Finno-Ugric Phonetic Alphabet to indicate a related borrowed form with different sound\n"
	"	x (percent sign - 0025)\n"
	"	x (arabic percent sign - 066A)\n"
	"	x (division sign - 00F7)"},
/* 2053 */ { "SWUNG DASH","	x (tilde - 007E)"},
/* 2054 */ { "INVERTED UNDERTIE",NULL},
/* 2055 */ { "FLOWER PUNCTUATION MARK","	= phul, puspika\n"
	"	* used as a punctuation mark with Syloti Nagri, Bengali and other Indic scripts\n"
	"	x (heavy eight teardrop-spoked propeller asterisk - 274B)"},
/* 2056 */ { "THREE DOT PUNCTUATION","	x (georgian paragraph separator - 10FB)"},
/* 2057 */ { "QUADRUPLE PRIME","	# 2032 2032 2032 2032"},
/* 2058 */ { "FOUR DOT PUNCTUATION",NULL},
/* 2059 */ { "FIVE DOT PUNCTUATION","	= Greek pentonkion\n"
	"	= quincunx\n"
	"	x (die face-5 - 2684)"},
/* 205A */ { "TWO DOT PUNCTUATION","	* historically used to indicate the end of a sentence or change of speaker\n"
	"	* extends from baseline to cap height\n"
	"	x (presentation form for vertical two dot leader - FE30)\n"
	"	x (greek acrophonic epidaurean two - 1015B)"},
/* 205B */ { "FOUR DOT MARK","	* used by scribes in the margin as highlighter mark\n"
	"	* this is centered on the line, but extends beyond top and bottom of the line"},
/* 205C */ { "DOTTED CROSS","	* used by scribes in the margin as highlighter mark"},
/* 205D */ { "TRICOLON","	= Epidaurean acrophonic symbol three\n"
	"	x (vertical ellipsis - 22EE)\n"
	"	x (triple colon operator - 2AF6)\n"
	"	x (presentation form for vertical horizontal ellipsis - FE19)"},
/* 205E */ { "VERTICAL FOUR DOTS","	* used in dictionaries to indicate legal but undesirable word break\n"
	"	* glyph extends the whole height of the line\n"
	"	x (vertical six dots - 2E3D)"},
/* 205F */ { "MEDIUM MATHEMATICAL SPACE","	* abbreviated MMSP\n"
	"	* four-eighteenths of an em\n"
	"	# 0020 space"},
/* 2060 */ { "WORD JOINER","	* commonly abbreviated WJ\n"
	"	* a zero width non-breaking space (only)\n"
	"	* intended for disambiguation of functions for byte order mark\n"
	"	x (zero width no-break space - FEFF)"},
/* 2061 */ { "FUNCTION APPLICATION","	* contiguity operator indicating application of a function"},
/* 2062 */ { "INVISIBLE TIMES","	* contiguity operator indicating multiplication"},
/* 2063 */ { "INVISIBLE SEPARATOR","	= invisible comma\n"
	"	* contiguity operator indicating that adjacent mathematical symbols form a list, e.g. when no visible comma is used between multiple indices"},
/* 2064 */ { "INVISIBLE PLUS","	* contiguity operator indicating addition"},
/* 2065 */ { NULL,NULL},
/* 2066 */ { "LEFT-TO-RIGHT ISOLATE","	* commonly abbreviated LRI"},
/* 2067 */ { "RIGHT-TO-LEFT ISOLATE","	* commonly abbreviated RLI"},
/* 2068 */ { "FIRST STRONG ISOLATE","	* commonly abbreviated FSI"},
/* 2069 */ { "POP DIRECTIONAL ISOLATE","	* commonly abbreviated PDI"},
/* 206A */ { "INHIBIT SYMMETRIC SWAPPING",NULL},
/* 206B */ { "ACTIVATE SYMMETRIC SWAPPING",NULL},
/* 206C */ { "INHIBIT ARABIC FORM SHAPING",NULL},
/* 206D */ { "ACTIVATE ARABIC FORM SHAPING",NULL},
/* 206E */ { "NATIONAL DIGIT SHAPES",NULL},
/* 206F */ { "NOMINAL DIGIT SHAPES",NULL},
/* 2070 */ { "SUPERSCRIPT ZERO","	# <super> 0030"},
/* 2071 */ { "SUPERSCRIPT LATIN SMALL LETTER I","	* functions as a modifier letter\n"
	"	# <super> 0069"},
/* 2072 */ { NULL,"	x (superscript two - 00B2)"},
/* 2073 */ { NULL,"	x (superscript three - 00B3)"},
/* 2074 */ { "SUPERSCRIPT FOUR","	# <super> 0034"},
/* 2075 */ { "SUPERSCRIPT FIVE","	# <super> 0035"},
/* 2076 */ { "SUPERSCRIPT SIX","	# <super> 0036"},
/* 2077 */ { "SUPERSCRIPT SEVEN","	# <super> 0037"},
/* 2078 */ { "SUPERSCRIPT EIGHT","	# <super> 0038"},
/* 2079 */ { "SUPERSCRIPT NINE","	# <super> 0039"},
/* 207A */ { "SUPERSCRIPT PLUS SIGN","	# <super> 002B"},
/* 207B */ { "SUPERSCRIPT MINUS","	# <super> 2212"},
/* 207C */ { "SUPERSCRIPT EQUALS SIGN","	# <super> 003D"},
/* 207D */ { "SUPERSCRIPT LEFT PARENTHESIS","	# <super> 0028"},
/* 207E */ { "SUPERSCRIPT RIGHT PARENTHESIS","	# <super> 0029"},
/* 207F */ { "SUPERSCRIPT LATIN SMALL LETTER N","	* functions as a modifier letter\n"
	"	# <super> 006E"},
/* 2080 */ { "SUBSCRIPT ZERO","	# <sub> 0030"},
/* 2081 */ { "SUBSCRIPT ONE","	# <sub> 0031"},
/* 2082 */ { "SUBSCRIPT TWO","	# <sub> 0032"},
/* 2083 */ { "SUBSCRIPT THREE","	# <sub> 0033"},
/* 2084 */ { "SUBSCRIPT FOUR","	# <sub> 0034"},
/* 2085 */ { "SUBSCRIPT FIVE","	# <sub> 0035"},
/* 2086 */ { "SUBSCRIPT SIX","	# <sub> 0036"},
/* 2087 */ { "SUBSCRIPT SEVEN","	# <sub> 0037"},
/* 2088 */ { "SUBSCRIPT EIGHT","	# <sub> 0038"},
/* 2089 */ { "SUBSCRIPT NINE","	# <sub> 0039"},
/* 208A */ { "SUBSCRIPT PLUS SIGN","	# <sub> 002B"},
/* 208B */ { "SUBSCRIPT MINUS","	# <sub> 2212"},
/* 208C */ { "SUBSCRIPT EQUALS SIGN","	# <sub> 003D"},
/* 208D */ { "SUBSCRIPT LEFT PARENTHESIS","	# <sub> 0028"},
/* 208E */ { "SUBSCRIPT RIGHT PARENTHESIS","	# <sub> 0029"},
/* 208F */ { NULL,NULL},
/* 2090 */ { "LATIN SUBSCRIPT SMALL LETTER A","	# <sub> 0061"},
/* 2091 */ { "LATIN SUBSCRIPT SMALL LETTER E","	# <sub> 0065"},
/* 2092 */ { "LATIN SUBSCRIPT SMALL LETTER O","	# <sub> 006F"},
/* 2093 */ { "LATIN SUBSCRIPT SMALL LETTER X","	# <sub> 0078"},
/* 2094 */ { "LATIN SUBSCRIPT SMALL LETTER SCHWA","	# <sub> 0259"},
/* 2095 */ { "LATIN SUBSCRIPT SMALL LETTER H","	# <sub> 0068"},
/* 2096 */ { "LATIN SUBSCRIPT SMALL LETTER K","	# <sub> 006B"},
/* 2097 */ { "LATIN SUBSCRIPT SMALL LETTER L","	# <sub> 006C"},
/* 2098 */ { "LATIN SUBSCRIPT SMALL LETTER M","	# <sub> 006D"},
/* 2099 */ { "LATIN SUBSCRIPT SMALL LETTER N","	# <sub> 006E"},
/* 209A */ { "LATIN SUBSCRIPT SMALL LETTER P","	# <sub> 0070"},
/* 209B */ { "LATIN SUBSCRIPT SMALL LETTER S","	# <sub> 0073"},
/* 209C */ { "LATIN SUBSCRIPT SMALL LETTER T","	# <sub> 0074"},
/* 209D */ { NULL,NULL},
/* 209E */ { NULL,NULL},
/* 209F */ { NULL,NULL},
/* 20A0 */ { "EURO-CURRENCY SIGN","	* intended for ECU, but not widely used\n"
	"	* historical character; this is NOT the euro!\n"
	"	x (euro sign - 20AC)"},
/* 20A1 */ { "COLON SIGN","	* Costa Rica, El Salvador"},
/* 20A2 */ { "CRUZEIRO SIGN","	* Brazil"},
/* 20A3 */ { "FRENCH FRANC SIGN","	* France\n"
	"	x (latin capital letter f with stroke - A798)"},
/* 20A4 */ { "LIRA SIGN","	* intended for lira, but not widely used\n"
	"	* preferred character for lira is 00A3\n"
	"	x (pound sign - 00A3)\n"
	"	x (turkish lira sign - 20BA)"},
/* 20A5 */ { "MILL SIGN","	* USA (1/10 cent)"},
/* 20A6 */ { "NAIRA SIGN","	* Nigeria"},
/* 20A7 */ { "PESETA SIGN","	* Spain\n"
	"	x (peso sign - 20B1)"},
/* 20A8 */ { "RUPEE SIGN","	* India, unofficial legacy practice\n"
	"	x (indian rupee sign - 20B9)\n"
	"	# 0052 0073"},
/* 20A9 */ { "WON SIGN","	* Korea\n"
	"	* glyph may have one or two crossbars, but the most common form in Korea has only one"},
/* 20AA */ { "NEW SHEQEL SIGN","	= shekel\n"
	"	* Israel"},
/* 20AB */ { "DONG SIGN","	* Vietnam"},
/* 20AC */ { "EURO SIGN","	* currency sign for the European Monetary Union\n"
	"	* euro, not ecu\n"
	"	x (euro-currency sign - 20A0)"},
/* 20AD */ { "KIP SIGN","	* Laos"},
/* 20AE */ { "TUGRIK SIGN","	* Mongolia\n"
	"	* also transliterated as tugrug, tugric, tugrog, togrog, tögrög"},
/* 20AF */ { "DRACHMA SIGN","	* Greece"},
/* 20B0 */ { "GERMAN PENNY SIGN",NULL},
/* 20B1 */ { "PESO SIGN","	= Filipino peso sign\n"
	"	* Philippines\n"
	"	* extant and discontinued Latin-American peso currencies (Mexican, Chilean, Colombian, etc.) use the dollar sign\n"
	"	x (dollar sign - 0024)\n"
	"	x (peseta sign - 20A7)"},
/* 20B2 */ { "GUARANI SIGN","	* Paraguay\n"
	"	* Often represented by G. or Gs."},
/* 20B3 */ { "AUSTRAL SIGN","	* former Argentinian currency"},
/* 20B4 */ { "HRYVNIA SIGN","	* Ukraine"},
/* 20B5 */ { "CEDI SIGN","	* Ghana\n"
	"	* glyph may look like '023B' or like C with a short vertical stroke through the upper arm\n"
	"	x (cent sign - 00A2)\n"
	"	x (latin capital letter c with stroke - 023B)"},
/* 20B6 */ { "LIVRE TOURNOIS SIGN","	* used in France from 13th-18th centuries"},
/* 20B7 */ { "SPESMILO SIGN","	* historical international currency associated with Esperanto"},
/* 20B8 */ { "TENGE SIGN","	* Kazakhstan\n"
	"	x (apl functional symbol up tack overbar - 2351)\n"
	"	x (box drawings down single and horizontal double - 2564)\n"
	"	x (postal mark - 3012)"},
/* 20B9 */ { "INDIAN RUPEE SIGN","	* official rupee currency sign for India\n"
	"	* contrasts with script-specific rupee signs and abbreviations\n"
	"	x (devanagari letter ra - 0930)"},
/* 20BA */ { "TURKISH LIRA SIGN","	* official lira currency sign for Turkey"},
/* 20BB */ { "NORDIC MARK SIGN","	* early representation of the Mark currency used in Denmark and Norway\n"
	"	x (script capital m - 2133)"},
/* 20BC */ { "MANAT SIGN","	* Azerbaijan"},
/* 20BD */ { "RUBLE SIGN","	* Russia"},
/* 20BE */ { "LARI SIGN","	* Georgia"},
/* 20BF */ { "BITCOIN SIGN","	* a cryptocurrency"},
/* 20C0 */ { "SOM SIGN","	* Kyrgyzstan"},
/* 20C1 */ { NULL,NULL},
/* 20C2 */ { NULL,NULL},
/* 20C3 */ { NULL,NULL},
/* 20C4 */ { NULL,NULL},
/* 20C5 */ { NULL,NULL},
/* 20C6 */ { NULL,NULL},
/* 20C7 */ { NULL,NULL},
/* 20C8 */ { NULL,NULL},
/* 20C9 */ { NULL,NULL},
/* 20CA */ { NULL,NULL},
/* 20CB */ { NULL,NULL},
/* 20CC */ { NULL,NULL},
/* 20CD */ { NULL,NULL},
/* 20CE */ { NULL,NULL},
/* 20CF */ { NULL,NULL},
/* 20D0 */ { "COMBINING LEFT HARPOON ABOVE",NULL},
/* 20D1 */ { "COMBINING RIGHT HARPOON ABOVE","	* vector"},
/* 20D2 */ { "COMBINING LONG VERTICAL LINE OVERLAY","	* negation"},
/* 20D3 */ { "COMBINING SHORT VERTICAL LINE OVERLAY","	* occasional variant for negation"},
/* 20D4 */ { "COMBINING ANTICLOCKWISE ARROW ABOVE",NULL},
/* 20D5 */ { "COMBINING CLOCKWISE ARROW ABOVE","	* rotation"},
/* 20D6 */ { "COMBINING LEFT ARROW ABOVE",NULL},
/* 20D7 */ { "COMBINING RIGHT ARROW ABOVE","	* vector"},
/* 20D8 */ { "COMBINING RING OVERLAY",NULL},
/* 20D9 */ { "COMBINING CLOCKWISE RING OVERLAY",NULL},
/* 20DA */ { "COMBINING ANTICLOCKWISE RING OVERLAY",NULL},
/* 20DB */ { "COMBINING THREE DOTS ABOVE","	= third derivative"},
/* 20DC */ { "COMBINING FOUR DOTS ABOVE","	= fourth derivative"},
/* 20DD */ { "COMBINING ENCLOSING CIRCLE","	= JIS composition circle\n"
	"	= Cyrillic combining ten thousands sign\n"
	"	x (white circle - 25CB)\n"
	"	x (large circle - 25EF)\n"
	"	x (ideographic number zero - 3007)"},
/* 20DE */ { "COMBINING ENCLOSING SQUARE","	x (white square - 25A1)"},
/* 20DF */ { "COMBINING ENCLOSING DIAMOND","	x (white diamond - 25C7)"},
/* 20E0 */ { "COMBINING ENCLOSING CIRCLE BACKSLASH","	* prohibition\n"
	"	x (prohibited sign - 1F6C7)"},
/* 20E1 */ { "COMBINING LEFT RIGHT ARROW ABOVE","	* tensor"},
/* 20E2 */ { "COMBINING ENCLOSING SCREEN","	x (clear screen symbol - 239A)\n"
	"	x (screen - 1F5B5)"},
/* 20E3 */ { "COMBINING ENCLOSING KEYCAP",NULL},
/* 20E4 */ { "COMBINING ENCLOSING UPWARD POINTING TRIANGLE","	x (white up-pointing triangle - 25B3)"},
/* 20E5 */ { "COMBINING REVERSE SOLIDUS OVERLAY","	x (reverse solidus - 005C)"},
/* 20E6 */ { "COMBINING DOUBLE VERTICAL STROKE OVERLAY","	= z notation finite function diacritic\n"
	"	x (double vertical line - 2016)\n"
	"	x (rightwards arrow with double vertical stroke - 21FB)\n"
	"	x (rightwards two-headed arrow with double vertical stroke - 2901)\n"
	"	x (rightwards arrow with tail with double vertical stroke - 2915)"},
/* 20E7 */ { "COMBINING ANNUITY SYMBOL","	= actuarial bend\n"
	"	x (right ceiling - 2309)"},
/* 20E8 */ { "COMBINING TRIPLE UNDERDOT",NULL},
/* 20E9 */ { "COMBINING WIDE BRIDGE ABOVE","	= contraction operator\n"
	"	* extends the full width of the base character\n"
	"	x (combining bridge above - 0346)"},
/* 20EA */ { "COMBINING LEFTWARDS ARROW OVERLAY","	x (leftwards arrow - 2190)"},
/* 20EB */ { "COMBINING LONG DOUBLE SOLIDUS OVERLAY","	= long double slash overlay"},
/* 20EC */ { "COMBINING RIGHTWARDS HARPOON WITH BARB DOWNWARDS",NULL},
/* 20ED */ { "COMBINING LEFTWARDS HARPOON WITH BARB DOWNWARDS",NULL},
/* 20EE */ { "COMBINING LEFT ARROW BELOW",NULL},
/* 20EF */ { "COMBINING RIGHT ARROW BELOW",NULL},
/* 20F0 */ { "COMBINING ASTERISK ABOVE",NULL},
/* 20F1 */ { NULL,NULL},
/* 20F2 */ { NULL,NULL},
/* 20F3 */ { NULL,NULL},
/* 20F4 */ { NULL,NULL},
/* 20F5 */ { NULL,NULL},
/* 20F6 */ { NULL,NULL},
/* 20F7 */ { NULL,NULL},
/* 20F8 */ { NULL,NULL},
/* 20F9 */ { NULL,NULL},
/* 20FA */ { NULL,NULL},
/* 20FB */ { NULL,NULL},
/* 20FC */ { NULL,NULL},
/* 20FD */ { NULL,NULL},
/* 20FE */ { NULL,NULL},
/* 20FF */ { NULL,NULL}
};

UN_DLL_LOCAL
static const struct unicode_nameannot una_00_21[] = {
/* 2100 */ { "ACCOUNT OF","	# 0061 002F 0063"},
/* 2101 */ { "ADDRESSED TO THE SUBJECT","	= aux soins (de)\n"
	"	x (aktieselskab - 214D)\n"
	"	# 0061 002F 0073"},
/* 2102 */ { "DOUBLE-STRUCK CAPITAL C","	= the set of complex numbers\n"
	"	# <font> 0043 latin capital letter c"},
/* 2103 */ { "DEGREE CELSIUS","	= degrees centigrade\n"
	"	# 00B0 0043"},
/* 2104 */ { "CENTRE LINE SYMBOL","	= clone"},
/* 2105 */ { "CARE OF","	# 0063 002F 006F"},
/* 2106 */ { "CADA UNA","	* Spanish for \"each one\"\n"
	"	# 0063 002F 0075"},
/* 2107 */ { "EULER CONSTANT","	x (latin capital letter e - 0045)\n"
	"	# 0190 latin capital letter open e"},
/* 2108 */ { "SCRUPLE",NULL},
/* 2109 */ { "DEGREE FAHRENHEIT","	# 00B0 0046"},
/* 210A */ { "SCRIPT SMALL G","	= real number symbol\n"
	"	# <font> 0067 latin small letter g"},
/* 210B */ { "SCRIPT CAPITAL H","	= Hamiltonian operator\n"
	"	~ 210B FE00 chancery style\n"
	"	~ 210B FE01 roundhand style\n"
	"	# <font> 0048 latin capital letter h"},
/* 210C */ { "BLACK-LETTER CAPITAL H","	= Hilbert space\n"
	"	# <font> 0048 latin capital letter h"},
/* 210D */ { "DOUBLE-STRUCK CAPITAL H","	# <font> 0048 latin capital letter h"},
/* 210E */ { "PLANCK CONSTANT","	= height, specific enthalpy, ...\n"
	"	* simply a mathematical italic h; this character's name results from legacy usage\n"
	"	# <font> 0068 latin small letter h"},
/* 210F */ { "PLANCK CONSTANT OVER TWO PI","	= reduced Planck constant, Dirac constant\n"
	"	x (cyrillic small letter tshe - 045B)\n"
	"	# <font> 0127 latin small letter h with stroke"},
/* 2110 */ { "SCRIPT CAPITAL I","	~ 2110 FE00 chancery style\n"
	"	~ 2110 FE01 roundhand style\n"
	"	# <font> 0049 latin capital letter i"},
/* 2111 */ { "BLACK-LETTER CAPITAL I","	= imaginary part\n"
	"	# <font> 0049 latin capital letter i"},
/* 2112 */ { "SCRIPT CAPITAL L","	= Laplace transform\n"
	"	~ 2112 FE00 chancery style\n"
	"	~ 2112 FE01 roundhand style\n"
	"	# <font> 004C latin capital letter l"},
/* 2113 */ { "SCRIPT SMALL L","	= mathematical symbol 'ell'\n"
	"	= liter (traditional symbol)\n"
	"	* despite its character name, this symbol is derived from a special italicized version of the small letter l\n"
	"	* the SI recommended symbol for liter is 006C or 004C\n"
	"	x (mathematical script small l - 1D4C1)\n"
	"	# <font> 006C latin small letter l"},
/* 2114 */ { "L B BAR SYMBOL","	= pounds\n"
	"	x (number sign - 0023)"},
/* 2115 */ { "DOUBLE-STRUCK CAPITAL N","	= natural number\n"
	"	* a glyph variant with doubled vertical strokes exists\n"
	"	# <font> 004E latin capital letter n"},
/* 2116 */ { "NUMERO SIGN","	x (number sign - 0023)\n"
	"	x (masculine ordinal indicator - 00BA)\n"
	"	# 004E 006F"},
/* 2117 */ { "SOUND RECORDING COPYRIGHT","	= published\n"
	"	= phonorecord sign\n"
	"	x (copyright sign - 00A9)\n"
	"	x (circled latin capital letter p - 24C5)"},
/* 2118 */ { "SCRIPT CAPITAL P","	% WEIERSTRASS ELLIPTIC FUNCTION\n"
	"	* actually this has the form of a lowercase calligraphic p, despite its name"},
/* 2119 */ { "DOUBLE-STRUCK CAPITAL P","	# <font> 0050 latin capital letter p"},
/* 211A */ { "DOUBLE-STRUCK CAPITAL Q","	= the set of rational numbers\n"
	"	# <font> 0051 latin capital letter q"},
/* 211B */ { "SCRIPT CAPITAL R","	= Riemann Integral\n"
	"	~ 211B FE00 chancery style\n"
	"	~ 211B FE01 roundhand style\n"
	"	# <font> 0052 latin capital letter r"},
/* 211C */ { "BLACK-LETTER CAPITAL R","	= real part\n"
	"	# <font> 0052 latin capital letter r"},
/* 211D */ { "DOUBLE-STRUCK CAPITAL R","	= the set of real numbers\n"
	"	# <font> 0052 latin capital letter r"},
/* 211E */ { "PRESCRIPTION TAKE","	= recipe\n"
	"	= cross ratio"},
/* 211F */ { "RESPONSE",NULL},
/* 2120 */ { "SERVICE MARK","	# <super> 0053 004D"},
/* 2121 */ { "TELEPHONE SIGN","	* typical forms for this symbol may use lowercase, small caps or superscripted letter shapes\n"
	"	x (black telephone - 260E)\n"
	"	x (telephone location sign - 2706)\n"
	"	x (telephone receiver - 1F4DE)\n"
	"	# 0054 0045 004C"},
/* 2122 */ { "TRADE MARK SIGN","	x (circled wz - 1F12E)\n"
	"	# <super> 0054 004D"},
/* 2123 */ { "VERSICLE",NULL},
/* 2124 */ { "DOUBLE-STRUCK CAPITAL Z","	= the set of integers\n"
	"	# <font> 005A latin capital letter z"},
/* 2125 */ { "OUNCE SIGN","	x (latin small letter ezh - 0292)\n"
	"	x (alchemical symbol for half ounce - 1F773)\n"
	"	x 2CF04"},
/* 2126 */ { "OHM SIGN","	* SI unit of resistance, named after G. S. Ohm, German physicist\n"
	"	* preferred representation is 03A9\n"
	"	x (ascending node - 260A)\n"
	"	: 03A9 greek capital letter omega"},
/* 2127 */ { "INVERTED OHM SIGN","	= mho\n"
	"	* archaic unit of conductance (= the SI unit siemens)\n"
	"	* typographically a turned greek capital letter omega\n"
	"	x (latin capital letter upsilon - 01B1)\n"
	"	x (greek capital letter omega - 03A9)\n"
	"	x (descending node - 260B)"},
/* 2128 */ { "BLACK-LETTER CAPITAL Z","	# <font> 005A latin capital letter z"},
/* 2129 */ { "TURNED GREEK SMALL LETTER IOTA","	* unique element fulfilling a description (logic)\n"
	"	x (greek small letter iota - 03B9)"},
/* 212A */ { "KELVIN SIGN","	: 004B latin capital letter k"},
/* 212B */ { "ANGSTROM SIGN","	* non SI length unit (=0.1 nm) named after A. J. Ångström, Swedish physicist\n"
	"	* preferred representation is 00C5\n"
	"	: 00C5 latin capital letter a with ring above"},
/* 212C */ { "SCRIPT CAPITAL B","	= Bernoulli function\n"
	"	~ 212C FE00 chancery style\n"
	"	~ 212C FE01 roundhand style\n"
	"	# <font> 0042 latin capital letter b"},
/* 212D */ { "BLACK-LETTER CAPITAL C","	# <font> 0043 latin capital letter c"},
/* 212E */ { "ESTIMATED SYMBOL","	* used in European packaging\n"
	"	x (latin small letter e - 0065)"},
/* 212F */ { "SCRIPT SMALL E","	= error\n"
	"	= natural exponent\n"
	"	# <font> 0065 latin small letter e"},
/* 2130 */ { "SCRIPT CAPITAL E","	= emf (electromotive force)\n"
	"	~ 2130 FE00 chancery style\n"
	"	~ 2130 FE01 roundhand style\n"
	"	# <font> 0045 latin capital letter e"},
/* 2131 */ { "SCRIPT CAPITAL F","	= Fourier transform\n"
	"	~ 2131 FE00 chancery style\n"
	"	~ 2131 FE01 roundhand style\n"
	"	# <font> 0046 latin capital letter f"},
/* 2132 */ { "TURNED CAPITAL F","	= Claudian digamma inversum\n"
	"	* lowercase is 214E\n"
	"	x (latin capital letter f - 0046)\n"
	"	x (greek letter digamma - 03DC)"},
/* 2133 */ { "SCRIPT CAPITAL M","	= M-matrix (physics)\n"
	"	= German Mark currency symbol, before WWII\n"
	"	* when used for the Mark symbol, the range of glyph variation is restricted\n"
	"	x (nordic mark sign - 20BB)\n"
	"	~ 2133 FE00 chancery style\n"
	"	~ 2133 FE01 roundhand style\n"
	"	# <font> 004D latin capital letter m"},
/* 2134 */ { "SCRIPT SMALL O","	= order, of inferior order to\n"
	"	# <font> 006F latin small letter o"},
/* 2135 */ { "ALEF SYMBOL","	= first transfinite cardinal (countable)\n"
	"	# 05D0 hebrew letter alef"},
/* 2136 */ { "BET SYMBOL","	= second transfinite cardinal (the continuum)\n"
	"	# 05D1 hebrew letter bet"},
/* 2137 */ { "GIMEL SYMBOL","	= third transfinite cardinal (functions of a real variable)\n"
	"	# 05D2 hebrew letter gimel"},
/* 2138 */ { "DALET SYMBOL","	= fourth transfinite cardinal\n"
	"	# 05D3 hebrew letter dalet"},
/* 2139 */ { "INFORMATION SOURCE","	* intended for use with 20DD\n"
	"	x (circled information source - 1F6C8)\n"
	"	# <font> 0069 latin small letter i"},
/* 213A */ { "ROTATED CAPITAL Q","	* a binding signature mark"},
/* 213B */ { "FACSIMILE SIGN","	* typical forms for this symbol may use lowercase, small caps or superscripted letter shapes\n"
	"	x (telephone sign - 2121)\n"
	"	x (fax machine - 1F4E0)\n"
	"	# 0046 0041 0058"},
/* 213C */ { "DOUBLE-STRUCK SMALL PI","	# <font> 03C0 greek small letter pi"},
/* 213D */ { "DOUBLE-STRUCK SMALL GAMMA","	# <font> 03B3 greek small letter gamma"},
/* 213E */ { "DOUBLE-STRUCK CAPITAL GAMMA","	# <font> 0393 greek capital letter gamma"},
/* 213F */ { "DOUBLE-STRUCK CAPITAL PI","	# <font> 03A0 greek capital letter pi"},
/* 2140 */ { "DOUBLE-STRUCK N-ARY SUMMATION","	# <font> 2211 n-ary summation"},
/* 2141 */ { "TURNED SANS-SERIF CAPITAL G","	= game\n"
	"	x (lisu letter hha - A4E8)"},
/* 2142 */ { "TURNED SANS-SERIF CAPITAL L","	x (lisu letter uh - A4F6)"},
/* 2143 */ { "REVERSED SANS-SERIF CAPITAL L",NULL},
/* 2144 */ { "TURNED SANS-SERIF CAPITAL Y","	x (lisu letter yha - 11FB0)"},
/* 2145 */ { "DOUBLE-STRUCK ITALIC CAPITAL D","	* sometimes used for the differential\n"
	"	# <font> 0044 latin capital letter d"},
/* 2146 */ { "DOUBLE-STRUCK ITALIC SMALL D","	* sometimes used for the differential\n"
	"	# <font> 0064 latin small letter d"},
/* 2147 */ { "DOUBLE-STRUCK ITALIC SMALL E","	* sometimes used for the natural exponent\n"
	"	# <font> 0065 latin small letter e"},
/* 2148 */ { "DOUBLE-STRUCK ITALIC SMALL I","	* sometimes used for the imaginary unit\n"
	"	# <font> 0069 latin small letter i"},
/* 2149 */ { "DOUBLE-STRUCK ITALIC SMALL J","	* sometimes used for the imaginary unit\n"
	"	# <font> 006A latin small letter j"},
/* 214A */ { "PROPERTY LINE",NULL},
/* 214B */ { "TURNED AMPERSAND","	* used in linear logic\n"
	"	x (ampersand - 0026)"},
/* 214C */ { "PER SIGN","	* abbreviates the word 'per'"},
/* 214D */ { "AKTIESELSKAB","	x (addressed to the subject - 2101)"},
/* 214E */ { "TURNED SMALL F","	* uppercase is 2132\n"
	"	x (greek small letter digamma - 03DD)"},
/* 214F */ { "SYMBOL FOR SAMARITAN SOURCE",NULL},
/* 2150 */ { "VULGAR FRACTION ONE SEVENTH","	# <fraction> 0031 2044 0037"},
/* 2151 */ { "VULGAR FRACTION ONE NINTH","	# <fraction> 0031 2044 0039"},
/* 2152 */ { "VULGAR FRACTION ONE TENTH","	# <fraction> 0031 2044 0031 0030"},
/* 2153 */ { "VULGAR FRACTION ONE THIRD","	# <fraction> 0031 2044 0033"},
/* 2154 */ { "VULGAR FRACTION TWO THIRDS","	# <fraction> 0032 2044 0033"},
/* 2155 */ { "VULGAR FRACTION ONE FIFTH","	# <fraction> 0031 2044 0035"},
/* 2156 */ { "VULGAR FRACTION TWO FIFTHS","	# <fraction> 0032 2044 0035"},
/* 2157 */ { "VULGAR FRACTION THREE FIFTHS","	# <fraction> 0033 2044 0035"},
/* 2158 */ { "VULGAR FRACTION FOUR FIFTHS","	# <fraction> 0034 2044 0035"},
/* 2159 */ { "VULGAR FRACTION ONE SIXTH","	# <fraction> 0031 2044 0036"},
/* 215A */ { "VULGAR FRACTION FIVE SIXTHS","	# <fraction> 0035 2044 0036"},
/* 215B */ { "VULGAR FRACTION ONE EIGHTH","	# <fraction> 0031 2044 0038"},
/* 215C */ { "VULGAR FRACTION THREE EIGHTHS","	# <fraction> 0033 2044 0038"},
/* 215D */ { "VULGAR FRACTION FIVE EIGHTHS","	# <fraction> 0035 2044 0038"},
/* 215E */ { "VULGAR FRACTION SEVEN EIGHTHS","	# <fraction> 0037 2044 0038"},
/* 215F */ { "FRACTION NUMERATOR ONE","	# <fraction> 0031 2044"},
/* 2160 */ { "ROMAN NUMERAL ONE","	# 0049 latin capital letter i"},
/* 2161 */ { "ROMAN NUMERAL TWO","	# 0049 0049"},
/* 2162 */ { "ROMAN NUMERAL THREE","	# 0049 0049 0049"},
/* 2163 */ { "ROMAN NUMERAL FOUR","	# 0049 0056"},
/* 2164 */ { "ROMAN NUMERAL FIVE","	# 0056 latin capital letter v"},
/* 2165 */ { "ROMAN NUMERAL SIX","	# 0056 0049"},
/* 2166 */ { "ROMAN NUMERAL SEVEN","	# 0056 0049 0049"},
/* 2167 */ { "ROMAN NUMERAL EIGHT","	# 0056 0049 0049 0049"},
/* 2168 */ { "ROMAN NUMERAL NINE","	# 0049 0058"},
/* 2169 */ { "ROMAN NUMERAL TEN","	# 0058 latin capital letter x"},
/* 216A */ { "ROMAN NUMERAL ELEVEN","	# 0058 0049"},
/* 216B */ { "ROMAN NUMERAL TWELVE","	# 0058 0049 0049"},
/* 216C */ { "ROMAN NUMERAL FIFTY","	# 004C latin capital letter l"},
/* 216D */ { "ROMAN NUMERAL ONE HUNDRED","	# 0043 latin capital letter c"},
/* 216E */ { "ROMAN NUMERAL FIVE HUNDRED","	# 0044 latin capital letter d"},
/* 216F */ { "ROMAN NUMERAL ONE THOUSAND","	# 004D latin capital letter m"},
/* 2170 */ { "SMALL ROMAN NUMERAL ONE","	# 0069 latin small letter i"},
/* 2171 */ { "SMALL ROMAN NUMERAL TWO","	# 0069 0069"},
/* 2172 */ { "SMALL ROMAN NUMERAL THREE","	# 0069 0069 0069"},
/* 2173 */ { "SMALL ROMAN NUMERAL FOUR","	# 0069 0076"},
/* 2174 */ { "SMALL ROMAN NUMERAL FIVE","	# 0076 latin small letter v"},
/* 2175 */ { "SMALL ROMAN NUMERAL SIX","	# 0076 0069"},
/* 2176 */ { "SMALL ROMAN NUMERAL SEVEN","	# 0076 0069 0069"},
/* 2177 */ { "SMALL ROMAN NUMERAL EIGHT","	# 0076 0069 0069 0069"},
/* 2178 */ { "SMALL ROMAN NUMERAL NINE","	# 0069 0078"},
/* 2179 */ { "SMALL ROMAN NUMERAL TEN","	# 0078 latin small letter x"},
/* 217A */ { "SMALL ROMAN NUMERAL ELEVEN","	# 0078 0069"},
/* 217B */ { "SMALL ROMAN NUMERAL TWELVE","	# 0078 0069 0069"},
/* 217C */ { "SMALL ROMAN NUMERAL FIFTY","	# 006C latin small letter l"},
/* 217D */ { "SMALL ROMAN NUMERAL ONE HUNDRED","	# 0063 latin small letter c"},
/* 217E */ { "SMALL ROMAN NUMERAL FIVE HUNDRED","	# 0064 latin small letter d"},
/* 217F */ { "SMALL ROMAN NUMERAL ONE THOUSAND","	# 006D latin small letter m"},
/* 2180 */ { "ROMAN NUMERAL ONE THOUSAND C D",NULL},
/* 2181 */ { "ROMAN NUMERAL FIVE THOUSAND",NULL},
/* 2182 */ { "ROMAN NUMERAL TEN THOUSAND",NULL},
/* 2183 */ { "ROMAN NUMERAL REVERSED ONE HUNDRED","	= apostrophic C\n"
	"	= Claudian antisigma\n"
	"	* used in combination with C and I to form large numbers\n"
	"	* lowercase is 2184\n"
	"	x (latin capital letter open o - 0186)\n"
	"	x (greek capital reversed lunate sigma symbol - 03FD)"},
/* 2184 */ { "LATIN SMALL LETTER REVERSED C","	x (latin small letter open o - 0254)\n"
	"	x (greek small reversed lunate sigma symbol - 037B)"},
/* 2185 */ { "ROMAN NUMERAL SIX LATE FORM","	x (greek letter stigma - 03DA)"},
/* 2186 */ { "ROMAN NUMERAL FIFTY EARLY FORM","	x (runic letter z - 16CE)\n"
	"	x (downwards arrow - 2193)"},
/* 2187 */ { "ROMAN NUMERAL FIFTY THOUSAND",NULL},
/* 2188 */ { "ROMAN NUMERAL ONE HUNDRED THOUSAND",NULL},
/* 2189 */ { "VULGAR FRACTION ZERO THIRDS","	* used in baseball scoring, from ARIB STD B24\n"
	"	# <fraction> 0030 2044 0033"},
/* 218A */ { "TURNED DIGIT TWO","	* digit for 10 in some duodecimal systems"},
/* 218B */ { "TURNED DIGIT THREE","	* digit for 11 in some duodecimal systems\n"
	"	x (latin capital letter open e - 0190)\n"
	"	x (euler constant - 2107)\n"
	"	x (latin small letter tresillo - A72B)"},
/* 218C */ { NULL,NULL},
/* 218D */ { NULL,NULL},
/* 218E */ { NULL,NULL},
/* 218F */ { NULL,NULL},
/* 2190 */ { "LEFTWARDS ARROW","	x (combining leftwards arrow overlay - 20EA)"},
/* 2191 */ { "UPWARDS ARROW","	* IPA: egressive airflow"},
/* 2192 */ { "RIGHTWARDS ARROW","	= z notation total function"},
/* 2193 */ { "DOWNWARDS ARROW","	* IPA: ingressive airflow\n"
	"	x (leftwards arrow from downwards arrow - 1F8C0)"},
/* 2194 */ { "LEFT RIGHT ARROW","	= z notation relation"},
/* 2195 */ { "UP DOWN ARROW",NULL},
/* 2196 */ { "NORTH WEST ARROW",NULL},
/* 2197 */ { "NORTH EAST ARROW",NULL},
/* 2198 */ { "SOUTH EAST ARROW",NULL},
/* 2199 */ { "SOUTH WEST ARROW",NULL},
/* 219A */ { "LEFTWARDS ARROW WITH STROKE","	* negation of 2190\n"
	"	: 2190 0338"},
/* 219B */ { "RIGHTWARDS ARROW WITH STROKE","	* negation of 2192\n"
	"	: 2192 0338"},
/* 219C */ { "LEFTWARDS WAVE ARROW","	x (wave arrow pointing directly left - 2B3F)"},
/* 219D */ { "RIGHTWARDS WAVE ARROW",NULL},
/* 219E */ { "LEFTWARDS TWO HEADED ARROW","	= fast cursor left\n"
	"	x (leftwards two-headed arrow with triangle arrowheads - 2BEC)"},
/* 219F */ { "UPWARDS TWO HEADED ARROW","	= fast cursor up"},
/* 21A0 */ { "RIGHTWARDS TWO HEADED ARROW","	= z notation total surjection\n"
	"	= fast cursor right"},
/* 21A1 */ { "DOWNWARDS TWO HEADED ARROW","	= form feed\n"
	"	= fast cursor down"},
/* 21A2 */ { "LEFTWARDS ARROW WITH TAIL",NULL},
/* 21A3 */ { "RIGHTWARDS ARROW WITH TAIL","	= z notation total injection"},
/* 21A4 */ { "LEFTWARDS ARROW FROM BAR",NULL},
/* 21A5 */ { "UPWARDS ARROW FROM BAR",NULL},
/* 21A6 */ { "RIGHTWARDS ARROW FROM BAR","	= z notation maplet"},
/* 21A7 */ { "DOWNWARDS ARROW FROM BAR","	= depth symbol"},
/* 21A8 */ { "UP DOWN ARROW WITH BASE",NULL},
/* 21A9 */ { "LEFTWARDS ARROW WITH HOOK",NULL},
/* 21AA */ { "RIGHTWARDS ARROW WITH HOOK",NULL},
/* 21AB */ { "LEFTWARDS ARROW WITH LOOP",NULL},
/* 21AC */ { "RIGHTWARDS ARROW WITH LOOP",NULL},
/* 21AD */ { "LEFT RIGHT WAVE ARROW",NULL},
/* 21AE */ { "LEFT RIGHT ARROW WITH STROKE","	* negation of 2194\n"
	"	: 2194 0338"},
/* 21AF */ { "DOWNWARDS ZIGZAG ARROW","	= electrolysis\n"
	"	x (downwards triangle-headed zigzag arrow - 2B4D)"},
/* 21B0 */ { "UPWARDS ARROW WITH TIP LEFTWARDS",NULL},
/* 21B1 */ { "UPWARDS ARROW WITH TIP RIGHTWARDS",NULL},
/* 21B2 */ { "DOWNWARDS ARROW WITH TIP LEFTWARDS","	x (inverse downwards arrow with tip leftwards - 1FBB4)"},
/* 21B3 */ { "DOWNWARDS ARROW WITH TIP RIGHTWARDS",NULL},
/* 21B4 */ { "RIGHTWARDS ARROW WITH CORNER DOWNWARDS","	= line feed"},
/* 21B5 */ { "DOWNWARDS ARROW WITH CORNER LEFTWARDS","	* may indicate a carriage return or new line\n"
	"	x (return symbol - 23CE)"},
/* 21B6 */ { "ANTICLOCKWISE TOP SEMICIRCLE ARROW",NULL},
/* 21B7 */ { "CLOCKWISE TOP SEMICIRCLE ARROW",NULL},
/* 21B8 */ { "NORTH WEST ARROW TO LONG BAR","	= home"},
/* 21B9 */ { "LEFTWARDS ARROW TO BAR OVER RIGHTWARDS ARROW TO BAR","	= tab with shift tab\n"
	"	x (horizontal tab key - 2B7E)"},
/* 21BA */ { "ANTICLOCKWISE OPEN CIRCLE ARROW",NULL},
/* 21BB */ { "CLOCKWISE OPEN CIRCLE ARROW",NULL},
/* 21BC */ { "LEFTWARDS HARPOON WITH BARB UPWARDS",NULL},
/* 21BD */ { "LEFTWARDS HARPOON WITH BARB DOWNWARDS",NULL},
/* 21BE */ { "UPWARDS HARPOON WITH BARB RIGHTWARDS",NULL},
/* 21BF */ { "UPWARDS HARPOON WITH BARB LEFTWARDS",NULL},
/* 21C0 */ { "RIGHTWARDS HARPOON WITH BARB UPWARDS",NULL},
/* 21C1 */ { "RIGHTWARDS HARPOON WITH BARB DOWNWARDS",NULL},
/* 21C2 */ { "DOWNWARDS HARPOON WITH BARB RIGHTWARDS",NULL},
/* 21C3 */ { "DOWNWARDS HARPOON WITH BARB LEFTWARDS",NULL},
/* 21C4 */ { "RIGHTWARDS ARROW OVER LEFTWARDS ARROW",NULL},
/* 21C5 */ { "UPWARDS ARROW LEFTWARDS OF DOWNWARDS ARROW",NULL},
/* 21C6 */ { "LEFTWARDS ARROW OVER RIGHTWARDS ARROW",NULL},
/* 21C7 */ { "LEFTWARDS PAIRED ARROWS",NULL},
/* 21C8 */ { "UPWARDS PAIRED ARROWS",NULL},
/* 21C9 */ { "RIGHTWARDS PAIRED ARROWS",NULL},
/* 21CA */ { "DOWNWARDS PAIRED ARROWS",NULL},
/* 21CB */ { "LEFTWARDS HARPOON OVER RIGHTWARDS HARPOON",NULL},
/* 21CC */ { "RIGHTWARDS HARPOON OVER LEFTWARDS HARPOON",NULL},
/* 21CD */ { "LEFTWARDS DOUBLE ARROW WITH STROKE","	* negation of 21D0\n"
	"	: 21D0 0338"},
/* 21CE */ { "LEFT RIGHT DOUBLE ARROW WITH STROKE","	* negation of 21D4\n"
	"	: 21D4 0338"},
/* 21CF */ { "RIGHTWARDS DOUBLE ARROW WITH STROKE","	* negation of 21D2\n"
	"	: 21D2 0338"},
/* 21D0 */ { "LEFTWARDS DOUBLE ARROW",NULL},
/* 21D1 */ { "UPWARDS DOUBLE ARROW",NULL},
/* 21D2 */ { "RIGHTWARDS DOUBLE ARROW",NULL},
/* 21D3 */ { "DOWNWARDS DOUBLE ARROW",NULL},
/* 21D4 */ { "LEFT RIGHT DOUBLE ARROW",NULL},
/* 21D5 */ { "UP DOWN DOUBLE ARROW",NULL},
/* 21D6 */ { "NORTH WEST DOUBLE ARROW",NULL},
/* 21D7 */ { "NORTH EAST DOUBLE ARROW",NULL},
/* 21D8 */ { "SOUTH EAST DOUBLE ARROW",NULL},
/* 21D9 */ { "SOUTH WEST DOUBLE ARROW",NULL},
/* 21DA */ { "LEFTWARDS TRIPLE ARROW",NULL},
/* 21DB */ { "RIGHTWARDS TRIPLE ARROW",NULL},
/* 21DC */ { "LEFTWARDS SQUIGGLE ARROW","	x (long leftwards squiggle arrow - 2B33)"},
/* 21DD */ { "RIGHTWARDS SQUIGGLE ARROW",NULL},
/* 21DE */ { "UPWARDS ARROW WITH DOUBLE STROKE","	= page up"},
/* 21DF */ { "DOWNWARDS ARROW WITH DOUBLE STROKE","	= page down"},
/* 21E0 */ { "LEFTWARDS DASHED ARROW",NULL},
/* 21E1 */ { "UPWARDS DASHED ARROW",NULL},
/* 21E2 */ { "RIGHTWARDS DASHED ARROW",NULL},
/* 21E3 */ { "DOWNWARDS DASHED ARROW",NULL},
/* 21E4 */ { "LEFTWARDS ARROW TO BAR","	= leftward tab"},
/* 21E5 */ { "RIGHTWARDS ARROW TO BAR","	= rightward tab"},
/* 21E6 */ { "LEFTWARDS WHITE ARROW","	x (north east white arrow - 2B00)"},
/* 21E7 */ { "UPWARDS WHITE ARROW","	= shift\n"
	"	= level 2 select (ISO 9995-7)"},
/* 21E8 */ { "RIGHTWARDS WHITE ARROW","	= group select (ISO 9995-7)"},
/* 21E9 */ { "DOWNWARDS WHITE ARROW",NULL},
/* 21EA */ { "UPWARDS WHITE ARROW FROM BAR","	= caps lock"},
/* 21EB */ { "UPWARDS WHITE ARROW ON PEDESTAL","	= level 2 lock"},
/* 21EC */ { "UPWARDS WHITE ARROW ON PEDESTAL WITH HORIZONTAL BAR","	= capitals (caps) lock\n"
	"	x (upwards white arrow from bar with horizontal bar - 2BB8)"},
/* 21ED */ { "UPWARDS WHITE ARROW ON PEDESTAL WITH VERTICAL BAR","	= numeric lock"},
/* 21EE */ { "UPWARDS WHITE DOUBLE ARROW","	= level 3 select"},
/* 21EF */ { "UPWARDS WHITE DOUBLE ARROW ON PEDESTAL","	= level 3 lock"},
/* 21F0 */ { "RIGHTWARDS WHITE ARROW FROM WALL","	= group lock"},
/* 21F1 */ { "NORTH WEST ARROW TO CORNER","	= home"},
/* 21F2 */ { "SOUTH EAST ARROW TO CORNER","	= end"},
/* 21F3 */ { "UP DOWN WHITE ARROW","	= scrolling\n"
	"	x (left right white arrow - 2B04)"},
/* 21F4 */ { "RIGHT ARROW WITH SMALL CIRCLE","	x (left arrow with small circle - 2B30)"},
/* 21F5 */ { "DOWNWARDS ARROW LEFTWARDS OF UPWARDS ARROW",NULL},
/* 21F6 */ { "THREE RIGHTWARDS ARROWS","	x (three leftwards arrows - 2B31)"},
/* 21F7 */ { "LEFTWARDS ARROW WITH VERTICAL STROKE",NULL},
/* 21F8 */ { "RIGHTWARDS ARROW WITH VERTICAL STROKE","	= z notation partial function"},
/* 21F9 */ { "LEFT RIGHT ARROW WITH VERTICAL STROKE","	= z notation partial relation"},
/* 21FA */ { "LEFTWARDS ARROW WITH DOUBLE VERTICAL STROKE",NULL},
/* 21FB */ { "RIGHTWARDS ARROW WITH DOUBLE VERTICAL STROKE","	= z notation finite function"},
/* 21FC */ { "LEFT RIGHT ARROW WITH DOUBLE VERTICAL STROKE","	= z notation finite relation"},
/* 21FD */ { "LEFTWARDS OPEN-HEADED ARROW",NULL},
/* 21FE */ { "RIGHTWARDS OPEN-HEADED ARROW",NULL},
/* 21FF */ { "LEFT RIGHT OPEN-HEADED ARROW",NULL}
};

UN_DLL_LOCAL
static const struct unicode_nameannot una_00_22[] = {
/* 2200 */ { "FOR ALL","	= universal quantifier"},
/* 2201 */ { "COMPLEMENT","	x (latin letter stretched c - 0297)"},
/* 2202 */ { "PARTIAL DIFFERENTIAL",NULL},
/* 2203 */ { "THERE EXISTS","	= existential quantifier"},
/* 2204 */ { "THERE DOES NOT EXIST","	: 2203 0338"},
/* 2205 */ { "EMPTY SET","	= null set\n"
	"	* used in linguistics to indicate a null morpheme or phonological \"zero\"\n"
	"	x (latin capital letter o with stroke - 00D8)\n"
	"	x (diameter sign - 2300)\n"
	"	~ 2205 FE00 zero with long diagonal stroke overlay form"},
/* 2206 */ { "INCREMENT","	= Laplace operator\n"
	"	= forward difference\n"
	"	= symmetric difference (in set theory)\n"
	"	* other symbols may also be used for symmetric difference\n"
	"	x (greek capital letter delta - 0394)\n"
	"	x (white up-pointing triangle - 25B3)"},
/* 2207 */ { "NABLA","	= backward difference\n"
	"	= gradient, del\n"
	"	* used for Laplacian operator (written with superscript 2)\n"
	"	x (white down-pointing triangle - 25BD)"},
/* 2208 */ { "ELEMENT OF",NULL},
/* 2209 */ { "NOT AN ELEMENT OF","	: 2208 0338"},
/* 220A */ { "SMALL ELEMENT OF","	* originates in math pi fonts; not the straight epsilon\n"
	"	x (greek lunate epsilon symbol - 03F5)"},
/* 220B */ { "CONTAINS AS MEMBER","	= such that"},
/* 220C */ { "DOES NOT CONTAIN AS MEMBER","	: 220B 0338"},
/* 220D */ { "SMALL CONTAINS AS MEMBER","	x (greek reversed lunate epsilon symbol - 03F6)"},
/* 220E */ { "END OF PROOF","	= q.e.d.\n"
	"	x (triangular bullet - 2023)\n"
	"	x (black vertical rectangle - 25AE)"},
/* 220F */ { "N-ARY PRODUCT","	= product sign\n"
	"	x (greek capital letter pi - 03A0)"},
/* 2210 */ { "N-ARY COPRODUCT","	= coproduct sign"},
/* 2211 */ { "N-ARY SUMMATION","	= summation sign\n"
	"	x (greek capital letter sigma - 03A3)\n"
	"	x (double-struck n-ary summation - 2140)"},
/* 2212 */ { "MINUS SIGN","	x (hyphen-minus - 002D)\n"
	"	x (heavy minus sign - 2796)"},
/* 2213 */ { "MINUS-OR-PLUS SIGN","	x (plus-minus sign - 00B1)"},
/* 2214 */ { "DOT PLUS",NULL},
/* 2215 */ { "DIVISION SLASH","	* generic division operator\n"
	"	x (solidus - 002F)\n"
	"	x (fraction slash - 2044)\n"
	"	x (mathematical rising diagonal - 27CB)"},
/* 2216 */ { "SET MINUS","	x (reverse solidus - 005C)\n"
	"	x (mathematical falling diagonal - 27CD)\n"
	"	x (reverse solidus operator - 29F5)"},
/* 2217 */ { "ASTERISK OPERATOR","	* may be used to represent the telephony asterisk seen on keypads\n"
	"	x (asterisk - 002A)"},
/* 2218 */ { "RING OPERATOR","	= composite function\n"
	"	= APL jot\n"
	"	x (degree sign - 00B0)\n"
	"	x (devanagari abbreviation sign - 0970)\n"
	"	x (white bullet - 25E6)\n"
	"	x (avestan abbreviation mark - 10B39)"},
/* 2219 */ { "BULLET OPERATOR","	x (middle dot - 00B7)\n"
	"	x (bullet - 2022)\n"
	"	x (one dot leader - 2024)"},
/* 221A */ { "SQUARE ROOT","	= radical sign\n"
	"	x (check mark - 2713)"},
/* 221B */ { "CUBE ROOT","	x (arabic-indic cube root - 0606)"},
/* 221C */ { "FOURTH ROOT","	x (arabic-indic fourth root - 0607)"},
/* 221D */ { "PROPORTIONAL TO","	x (greek small letter alpha - 03B1)"},
/* 221E */ { "INFINITY",NULL},
/* 221F */ { "RIGHT ANGLE","	= with (in chess notation)\n"
	"	x (reversed right angle - 2BFE)"},
/* 2220 */ { "ANGLE","	* used in astrological contexts for aspect semisquare\n"
	"	x (acute angle - 299F)"},
/* 2221 */ { "MEASURED ANGLE",NULL},
/* 2222 */ { "SPHERICAL ANGLE","	= angle arc"},
/* 2223 */ { "DIVIDES","	= such that\n"
	"	= APL stile\n"
	"	x (vertical line - 007C)\n"
	"	x (latin letter dental click - 01C0)"},
/* 2224 */ { "DOES NOT DIVIDE","	: 2223 0338"},
/* 2225 */ { "PARALLEL TO","	x (latin letter lateral click - 01C1)\n"
	"	x (double vertical line - 2016)"},
/* 2226 */ { "NOT PARALLEL TO","	: 2225 0338"},
/* 2227 */ { "LOGICAL AND","	= wedge, conjunction\n"
	"	x (n-ary logical and - 22C0)\n"
	"	x (up arrowhead - 2303)\n"
	"	x (and with dot - 27D1)"},
/* 2228 */ { "LOGICAL OR","	= vee, disjunction\n"
	"	x (n-ary logical or - 22C1)\n"
	"	x (down arrowhead - 2304)\n"
	"	x (or with dot inside - 27C7)"},
/* 2229 */ { "INTERSECTION","	= cap, hat\n"
	"	x (n-ary intersection - 22C2)\n"
	"	~ 2229 FE00 with serifs"},
/* 222A */ { "UNION","	= cup\n"
	"	x (n-ary union - 22C3)\n"
	"	~ 222A FE00 with serifs"},
/* 222B */ { "INTEGRAL","	x (latin small letter esh - 0283)"},
/* 222C */ { "DOUBLE INTEGRAL","	# 222B 222B"},
/* 222D */ { "TRIPLE INTEGRAL","	x (quadruple integral operator - 2A0C)\n"
	"	# 222B 222B 222B"},
/* 222E */ { "CONTOUR INTEGRAL","	x (integral around a point operator - 2A15)"},
/* 222F */ { "SURFACE INTEGRAL","	# 222E 222E"},
/* 2230 */ { "VOLUME INTEGRAL","	# 222E 222E 222E"},
/* 2231 */ { "CLOCKWISE INTEGRAL",NULL},
/* 2232 */ { "CLOCKWISE CONTOUR INTEGRAL",NULL},
/* 2233 */ { "ANTICLOCKWISE CONTOUR INTEGRAL","	* clockwise or anticlockwise arrows do not reverse during layout mirroring"},
/* 2234 */ { "THEREFORE","	x (historic site - 26EC)"},
/* 2235 */ { "BECAUSE",NULL},
/* 2236 */ { "RATIO","	* preferred to 003A for denotation of division or scale in mathematical use\n"
	"	x (colon - 003A)"},
/* 2237 */ { "PROPORTION",NULL},
/* 2238 */ { "DOT MINUS","	= saturating subtraction\n"
	"	* sometimes claimed as notation for symmetric set difference, but 2206 is preferred"},
/* 2239 */ { "EXCESS",NULL},
/* 223A */ { "GEOMETRIC PROPORTION",NULL},
/* 223B */ { "HOMOTHETIC","	x (tilde operator with rising dots - 2A6B)"},
/* 223C */ { "TILDE OPERATOR","	= varies with (proportional to)\n"
	"	= difference between\n"
	"	= similar to\n"
	"	= not\n"
	"	= cycle\n"
	"	= APL tilde\n"
	"	x (tilde - 007E)\n"
	"	x (not sign - 00AC)\n"
	"	x (small tilde - 02DC)"},
/* 223D */ { "REVERSED TILDE","	= lazy S\n"
	"	* reversed tilde and lazy S are glyph variants"},
/* 223E */ { "INVERTED LAZY S","	= most positive"},
/* 223F */ { "SINE WAVE","	= alternating current"},
/* 2240 */ { "WREATH PRODUCT",NULL},
/* 2241 */ { "NOT TILDE","	: 223C 0338"},
/* 2242 */ { "MINUS TILDE",NULL},
/* 2243 */ { "ASYMPTOTICALLY EQUAL TO",NULL},
/* 2244 */ { "NOT ASYMPTOTICALLY EQUAL TO","	: 2243 0338"},
/* 2245 */ { "APPROXIMATELY EQUAL TO","	x (approximately equal or equal to - 2A70)"},
/* 2246 */ { "APPROXIMATELY BUT NOT ACTUALLY EQUAL TO",NULL},
/* 2247 */ { "NEITHER APPROXIMATELY NOR ACTUALLY EQUAL TO","	: 2245 0338"},
/* 2248 */ { "ALMOST EQUAL TO","	= asymptotic to"},
/* 2249 */ { "NOT ALMOST EQUAL TO","	: 2248 0338"},
/* 224A */ { "ALMOST EQUAL OR EQUAL TO",NULL},
/* 224B */ { "TRIPLE TILDE",NULL},
/* 224C */ { "ALL EQUAL TO","	* reversed tilde and lazy S are glyph variants"},
/* 224D */ { "EQUIVALENT TO",NULL},
/* 224E */ { "GEOMETRICALLY EQUIVALENT TO",NULL},
/* 224F */ { "DIFFERENCE BETWEEN",NULL},
/* 2250 */ { "APPROACHES THE LIMIT",NULL},
/* 2251 */ { "GEOMETRICALLY EQUAL TO",NULL},
/* 2252 */ { "APPROXIMATELY EQUAL TO OR THE IMAGE OF","	= nearly equals"},
/* 2253 */ { "IMAGE OF OR APPROXIMATELY EQUAL TO",NULL},
/* 2254 */ { "COLON EQUALS",NULL},
/* 2255 */ { "EQUALS COLON",NULL},
/* 2256 */ { "RING IN EQUAL TO",NULL},
/* 2257 */ { "RING EQUAL TO","	= approximately equal to"},
/* 2258 */ { "CORRESPONDS TO",NULL},
/* 2259 */ { "ESTIMATES","	= corresponds to"},
/* 225A */ { "EQUIANGULAR TO",NULL},
/* 225B */ { "STAR EQUALS",NULL},
/* 225C */ { "DELTA EQUAL TO","	= equiangular\n"
	"	= equal to by definition"},
/* 225D */ { "EQUAL TO BY DEFINITION",NULL},
/* 225E */ { "MEASURED BY",NULL},
/* 225F */ { "QUESTIONED EQUAL TO",NULL},
/* 2260 */ { "NOT EQUAL TO","	x (equals sign - 003D)\n"
	"	x (latin letter alveolar click - 01C2)\n"
	"	: 003D 0338"},
/* 2261 */ { "IDENTICAL TO",NULL},
/* 2262 */ { "NOT IDENTICAL TO","	: 2261 0338"},
/* 2263 */ { "STRICTLY EQUIVALENT TO",NULL},
/* 2264 */ { "LESS-THAN OR EQUAL TO","	x (less-than sign - 003C)"},
/* 2265 */ { "GREATER-THAN OR EQUAL TO","	x (greater-than sign - 003E)"},
/* 2266 */ { "LESS-THAN OVER EQUAL TO",NULL},
/* 2267 */ { "GREATER-THAN OVER EQUAL TO",NULL},
/* 2268 */ { "LESS-THAN BUT NOT EQUAL TO","	x (less-than and single-line not equal to - 2A87)\n"
	"	~ 2268 FE00 with vertical stroke"},
/* 2269 */ { "GREATER-THAN BUT NOT EQUAL TO","	x (greater-than and single-line not equal to - 2A88)\n"
	"	~ 2269 FE00 with vertical stroke"},
/* 226A */ { "MUCH LESS-THAN","	x (left-pointing double angle quotation mark - 00AB)"},
/* 226B */ { "MUCH GREATER-THAN","	x (right-pointing double angle quotation mark - 00BB)"},
/* 226C */ { "BETWEEN","	= plaintiff, quantic"},
/* 226D */ { "NOT EQUIVALENT TO","	: 224D 0338"},
/* 226E */ { "NOT LESS-THAN","	: 003C 0338"},
/* 226F */ { "NOT GREATER-THAN","	: 003E 0338"},
/* 2270 */ { "NEITHER LESS-THAN NOR EQUAL TO","	: 2264 0338"},
/* 2271 */ { "NEITHER GREATER-THAN NOR EQUAL TO","	: 2265 0338"},
/* 2272 */ { "LESS-THAN OR EQUIVALENT TO","	~ 2272 FE00 following the slant of the lower leg"},
/* 2273 */ { "GREATER-THAN OR EQUIVALENT TO","	~ 2273 FE00 following the slant of the lower leg"},
/* 2274 */ { "NEITHER LESS-THAN NOR EQUIVALENT TO","	: 2272 0338"},
/* 2275 */ { "NEITHER GREATER-THAN NOR EQUIVALENT TO","	: 2273 0338"},
/* 2276 */ { "LESS-THAN OR GREATER-THAN",NULL},
/* 2277 */ { "GREATER-THAN OR LESS-THAN",NULL},
/* 2278 */ { "NEITHER LESS-THAN NOR GREATER-THAN","	: 2276 0338"},
/* 2279 */ { "NEITHER GREATER-THAN NOR LESS-THAN","	: 2277 0338"},
/* 227A */ { "PRECEDES","	= lower rank than\n"
	"	x (precedes under relation - 22B0)"},
/* 227B */ { "SUCCEEDS","	= higher rank than\n"
	"	x (succeeds under relation - 22B1)"},
/* 227C */ { "PRECEDES OR EQUAL TO",NULL},
/* 227D */ { "SUCCEEDS OR EQUAL TO",NULL},
/* 227E */ { "PRECEDES OR EQUIVALENT TO",NULL},
/* 227F */ { "SUCCEEDS OR EQUIVALENT TO",NULL},
/* 2280 */ { "DOES NOT PRECEDE","	: 227A 0338"},
/* 2281 */ { "DOES NOT SUCCEED","	: 227B 0338"},
/* 2282 */ { "SUBSET OF","	= included in set\n"
	"	x (left sideways u bracket - 2E26)"},
/* 2283 */ { "SUPERSET OF","	= includes in set\n"
	"	x (right sideways u bracket - 2E27)"},
/* 2284 */ { "NOT A SUBSET OF","	: 2282 0338"},
/* 2285 */ { "NOT A SUPERSET OF","	: 2283 0338"},
/* 2286 */ { "SUBSET OF OR EQUAL TO",NULL},
/* 2287 */ { "SUPERSET OF OR EQUAL TO",NULL},
/* 2288 */ { "NEITHER A SUBSET OF NOR EQUAL TO","	: 2286 0338"},
/* 2289 */ { "NEITHER A SUPERSET OF NOR EQUAL TO","	: 2287 0338"},
/* 228A */ { "SUBSET OF WITH NOT EQUAL TO","	~ 228A FE00 with stroke through bottom members"},
/* 228B */ { "SUPERSET OF WITH NOT EQUAL TO","	~ 228B FE00 with stroke through bottom members"},
/* 228C */ { "MULTISET",NULL},
/* 228D */ { "MULTISET MULTIPLICATION","	x (n-ary union operator with dot - 2A03)\n"
	"	x (intersection with dot - 2A40)"},
/* 228E */ { "MULTISET UNION","	= z notation bag addition\n"
	"	x (n-ary union operator with plus - 2A04)"},
/* 228F */ { "SQUARE IMAGE OF",NULL},
/* 2290 */ { "SQUARE ORIGINAL OF",NULL},
/* 2291 */ { "SQUARE IMAGE OF OR EQUAL TO",NULL},
/* 2292 */ { "SQUARE ORIGINAL OF OR EQUAL TO",NULL},
/* 2293 */ { "SQUARE CAP","	x (n-ary square intersection operator - 2A05)\n"
	"	~ 2293 FE00 with serifs"},
/* 2294 */ { "SQUARE CUP","	x (n-ary square union operator - 2A06)\n"
	"	~ 2294 FE00 with serifs"},
/* 2295 */ { "CIRCLED PLUS","	= direct sum\n"
	"	= vector pointing into page\n"
	"	x (n-ary circled plus operator - 2A01)\n"
	"	x (alchemical symbol for verdigris - 1F728)\n"
	"	~ 2295 FE00 with white rim"},
/* 2296 */ { "CIRCLED MINUS","	= symmetric difference\n"
	"	x (circle with horizontal bar - 29B5)\n"
	"	x (alchemical symbol for salt - 1F714)"},
/* 2297 */ { "CIRCLED TIMES","	= tensor product\n"
	"	= vector pointing into page\n"
	"	x (circled crossing lanes - 26D2)\n"
	"	x (n-ary circled times operator - 2A02)\n"
	"	x (circled x - 2BBE)\n"
	"	x (lot of fortune - 1F774)\n"
	"	~ 2297 FE00 with white rim"},
/* 2298 */ { "CIRCLED DIVISION SLASH",NULL},
/* 2299 */ { "CIRCLED DOT OPERATOR","	= direct product\n"
	"	= vector pointing out of page\n"
	"	x (latin letter bilabial click - 0298)\n"
	"	x (sun - 2609)\n"
	"	x (n-ary circled dot operator - 2A00)"},
/* 229A */ { "CIRCLED RING OPERATOR","	x (apl functional symbol circle jot - 233E)\n"
	"	x (bullseye - 25CE)"},
/* 229B */ { "CIRCLED ASTERISK OPERATOR","	x (apl functional symbol circle star - 235F)"},
/* 229C */ { "CIRCLED EQUALS","	= NoDerivatives (symbol used for Creative Commons licenses)\n"
	"	x (circled cc - 1F16D)\n"
	"	~ 229C FE00 with equal sign touching the circle"},
/* 229D */ { "CIRCLED DASH",NULL},
/* 229E */ { "SQUARED PLUS",NULL},
/* 229F */ { "SQUARED MINUS",NULL},
/* 22A0 */ { "SQUARED TIMES","	x (ballot box with x - 2612)\n"
	"	x (squared saltire - 26DD)\n"
	"	x (alchemical symbol for month - 1F771)"},
/* 22A1 */ { "SQUARED DOT OPERATOR","	x (alchemical symbol for urine - 1F755)\n"
	"	x (right open squared dot - 1FBBC)"},
/* 22A2 */ { "RIGHT TACK","	= turnstile\n"
	"	= proves, implies, yields\n"
	"	= reducible"},
/* 22A3 */ { "LEFT TACK","	= reverse turnstile\n"
	"	= non-theorem, does not yield"},
/* 22A4 */ { "DOWN TACK","	= top\n"
	"	x (raised interpolation marker - 2E06)\n"
	"	x (alchemical symbol for crucible-4 - 1F768)"},
/* 22A5 */ { "UP TACK","	= base, bottom\n"
	"	x (perpendicular - 27C2)"},
/* 22A6 */ { "ASSERTION","	= reduces to"},
/* 22A7 */ { "MODELS",NULL},
/* 22A8 */ { "TRUE","	= statement is true, valid\n"
	"	= is a tautology\n"
	"	= satisfies\n"
	"	= results in"},
/* 22A9 */ { "FORCES",NULL},
/* 22AA */ { "TRIPLE VERTICAL BAR RIGHT TURNSTILE",NULL},
/* 22AB */ { "DOUBLE VERTICAL BAR DOUBLE RIGHT TURNSTILE",NULL},
/* 22AC */ { "DOES NOT PROVE","	: 22A2 0338"},
/* 22AD */ { "NOT TRUE","	: 22A8 0338"},
/* 22AE */ { "DOES NOT FORCE","	: 22A9 0338"},
/* 22AF */ { "NEGATED DOUBLE VERTICAL BAR DOUBLE RIGHT TURNSTILE","	: 22AB 0338"},
/* 22B0 */ { "PRECEDES UNDER RELATION","	x (precedes - 227A)"},
/* 22B1 */ { "SUCCEEDS UNDER RELATION","	x (succeeds - 227B)"},
/* 22B2 */ { "NORMAL SUBGROUP OF","	x (white left-pointing pointer - 25C5)"},
/* 22B3 */ { "CONTAINS AS NORMAL SUBGROUP","	x (white right-pointing pointer - 25BB)"},
/* 22B4 */ { "NORMAL SUBGROUP OF OR EQUAL TO",NULL},
/* 22B5 */ { "CONTAINS AS NORMAL SUBGROUP OR EQUAL TO",NULL},
/* 22B6 */ { "ORIGINAL OF",NULL},
/* 22B7 */ { "IMAGE OF",NULL},
/* 22B8 */ { "MULTIMAP",NULL},
/* 22B9 */ { "HERMITIAN CONJUGATE MATRIX",NULL},
/* 22BA */ { "INTERCALATE",NULL},
/* 22BB */ { "XOR","	x (semisextile - 26BA)"},
/* 22BC */ { "NAND","	x (projective - 2305)\n"
	"	x (quincunx - 26BB)"},
/* 22BD */ { "NOR",NULL},
/* 22BE */ { "RIGHT ANGLE WITH ARC",NULL},
/* 22BF */ { "RIGHT TRIANGLE",NULL},
/* 22C0 */ { "N-ARY LOGICAL AND","	* also used to denote the universal quantifier\n"
	"	x (logical and - 2227)"},
/* 22C1 */ { "N-ARY LOGICAL OR","	* also used to denote the existential quantifier\n"
	"	x (logical or - 2228)"},
/* 22C2 */ { "N-ARY INTERSECTION","	= z notation generalised intersection\n"
	"	x (intersection - 2229)"},
/* 22C3 */ { "N-ARY UNION","	= z notation generalised union\n"
	"	x (union - 222A)"},
/* 22C4 */ { "DIAMOND OPERATOR","	x (white diamond - 25C7)\n"
	"	x (black small diamond - 2B29)"},
/* 22C5 */ { "DOT OPERATOR","	* preferred to 00B7 for denotation of multiplication\n"
	"	x (middle dot - 00B7)"},
/* 22C6 */ { "STAR OPERATOR","	* APL\n"
	"	x (arabic five pointed star - 066D)\n"
	"	x (black star - 2605)\n"
	"	x (white medium star - 2B50)"},
/* 22C7 */ { "DIVISION TIMES","	x (multiplication sign - 00D7)\n"
	"	x (division sign - 00F7)\n"
	"	x (multiplication x - 2715)"},
/* 22C8 */ { "BOWTIE","	x (ocr bow tie - 2445)"},
/* 22C9 */ { "LEFT NORMAL FACTOR SEMIDIRECT PRODUCT",NULL},
/* 22CA */ { "RIGHT NORMAL FACTOR SEMIDIRECT PRODUCT",NULL},
/* 22CB */ { "LEFT SEMIDIRECT PRODUCT",NULL},
/* 22CC */ { "RIGHT SEMIDIRECT PRODUCT","	x (caret insertion point - 2041)"},
/* 22CD */ { "REVERSED TILDE EQUALS",NULL},
/* 22CE */ { "CURLY LOGICAL OR",NULL},
/* 22CF */ { "CURLY LOGICAL AND",NULL},
/* 22D0 */ { "DOUBLE SUBSET",NULL},
/* 22D1 */ { "DOUBLE SUPERSET",NULL},
/* 22D2 */ { "DOUBLE INTERSECTION",NULL},
/* 22D3 */ { "DOUBLE UNION",NULL},
/* 22D4 */ { "PITCHFORK","	= proper intersection"},
/* 22D5 */ { "EQUAL AND PARALLEL TO","	x (viewdata square - 2317)"},
/* 22D6 */ { "LESS-THAN WITH DOT",NULL},
/* 22D7 */ { "GREATER-THAN WITH DOT",NULL},
/* 22D8 */ { "VERY MUCH LESS-THAN",NULL},
/* 22D9 */ { "VERY MUCH GREATER-THAN",NULL},
/* 22DA */ { "LESS-THAN EQUAL TO OR GREATER-THAN","	~ 22DA FE00 with slanted equal"},
/* 22DB */ { "GREATER-THAN EQUAL TO OR LESS-THAN","	~ 22DB FE00 with slanted equal"},
/* 22DC */ { "EQUAL TO OR LESS-THAN",NULL},
/* 22DD */ { "EQUAL TO OR GREATER-THAN",NULL},
/* 22DE */ { "EQUAL TO OR PRECEDES",NULL},
/* 22DF */ { "EQUAL TO OR SUCCEEDS",NULL},
/* 22E0 */ { "DOES NOT PRECEDE OR EQUAL","	: 227C 0338"},
/* 22E1 */ { "DOES NOT SUCCEED OR EQUAL","	: 227D 0338"},
/* 22E2 */ { "NOT SQUARE IMAGE OF OR EQUAL TO","	: 2291 0338"},
/* 22E3 */ { "NOT SQUARE ORIGINAL OF OR EQUAL TO","	: 2292 0338"},
/* 22E4 */ { "SQUARE IMAGE OF OR NOT EQUAL TO",NULL},
/* 22E5 */ { "SQUARE ORIGINAL OF OR NOT EQUAL TO",NULL},
/* 22E6 */ { "LESS-THAN BUT NOT EQUIVALENT TO",NULL},
/* 22E7 */ { "GREATER-THAN BUT NOT EQUIVALENT TO",NULL},
/* 22E8 */ { "PRECEDES BUT NOT EQUIVALENT TO",NULL},
/* 22E9 */ { "SUCCEEDS BUT NOT EQUIVALENT TO",NULL},
/* 22EA */ { "NOT NORMAL SUBGROUP OF","	: 22B2 0338"},
/* 22EB */ { "DOES NOT CONTAIN AS NORMAL SUBGROUP","	: 22B3 0338"},
/* 22EC */ { "NOT NORMAL SUBGROUP OF OR EQUAL TO","	: 22B4 0338"},
/* 22ED */ { "DOES NOT CONTAIN AS NORMAL SUBGROUP OR EQUAL","	: 22B5 0338"},
/* 22EE */ { "VERTICAL ELLIPSIS","	x (tricolon - 205D)\n"
	"	x (horizontal ellipsis - 2026)\n"
	"	x (presentation form for vertical horizontal ellipsis - FE19)"},
/* 22EF */ { "MIDLINE HORIZONTAL ELLIPSIS",NULL},
/* 22F0 */ { "UP RIGHT DIAGONAL ELLIPSIS",NULL},
/* 22F1 */ { "DOWN RIGHT DIAGONAL ELLIPSIS",NULL},
/* 22F2 */ { "ELEMENT OF WITH LONG HORIZONTAL STROKE",NULL},
/* 22F3 */ { "ELEMENT OF WITH VERTICAL BAR AT END OF HORIZONTAL STROKE",NULL},
/* 22F4 */ { "SMALL ELEMENT OF WITH VERTICAL BAR AT END OF HORIZONTAL STROKE",NULL},
/* 22F5 */ { "ELEMENT OF WITH DOT ABOVE",NULL},
/* 22F6 */ { "ELEMENT OF WITH OVERBAR",NULL},
/* 22F7 */ { "SMALL ELEMENT OF WITH OVERBAR",NULL},
/* 22F8 */ { "ELEMENT OF WITH UNDERBAR",NULL},
/* 22F9 */ { "ELEMENT OF WITH TWO HORIZONTAL STROKES",NULL},
/* 22FA */ { "CONTAINS WITH LONG HORIZONTAL STROKE",NULL},
/* 22FB */ { "CONTAINS WITH VERTICAL BAR AT END OF HORIZONTAL STROKE",NULL},
/* 22FC */ { "SMALL CONTAINS WITH VERTICAL BAR AT END OF HORIZONTAL STROKE",NULL},
/* 22FD */ { "CONTAINS WITH OVERBAR",NULL},
/* 22FE */ { "SMALL CONTAINS WITH OVERBAR",NULL},
/* 22FF */ { "Z NOTATION BAG MEMBERSHIP",NULL}
};

UN_DLL_LOCAL
static const struct unicode_nameannot una_00_23[] = {
/* 2300 */ { "DIAMETER SIGN","	x (latin small letter o with stroke - 00F8)\n"
	"	x (empty set - 2205)"},
/* 2301 */ { "ELECTRIC ARROW","	* from ISO 2047\n"
	"	* symbol for End of Transmission"},
/* 2302 */ { "HOUSE",NULL},
/* 2303 */ { "UP ARROWHEAD","	x (circumflex accent - 005E)\n"
	"	x (modifier letter up arrowhead - 02C4)\n"
	"	x (caret - 2038)\n"
	"	x (logical and - 2227)"},
/* 2304 */ { "DOWN ARROWHEAD","	x (modifier letter down arrowhead - 02C5)\n"
	"	x (logical or - 2228)\n"
	"	x (countersink - 2335)"},
/* 2305 */ { "PROJECTIVE","	x (nand - 22BC)"},
/* 2306 */ { "PERSPECTIVE",NULL},
/* 2307 */ { "WAVY LINE","	x (wiggly vertical line - 2E3E)\n"
	"	x (wavy dash - 3030)"},
/* 2308 */ { "LEFT CEILING","	= APL upstile\n"
	"	x (top left half bracket - 2E22)\n"
	"	x (left corner bracket - 300C)"},
/* 2309 */ { "RIGHT CEILING","	x (combining annuity symbol - 20E7)\n"
	"	x (top right half bracket - 2E23)"},
/* 230A */ { "LEFT FLOOR","	= APL downstile\n"
	"	x (bottom left half bracket - 2E24)"},
/* 230B */ { "RIGHT FLOOR","	x (right corner bracket - 300D)\n"
	"	x (bottom right half bracket - 2E25)"},
/* 230C */ { "BOTTOM RIGHT CROP","	* set of four \"crop\" corners, arranged facing outward"},
/* 230D */ { "BOTTOM LEFT CROP",NULL},
/* 230E */ { "TOP RIGHT CROP",NULL},
/* 230F */ { "TOP LEFT CROP",NULL},
/* 2310 */ { "REVERSED NOT SIGN","	= beginning of line\n"
	"	x (not sign - 00AC)"},
/* 2311 */ { "SQUARE LOZENGE","	= Kissen (pillow)\n"
	"	* used as a command delimiter in some very old computers\n"
	"	x (rotated white four pointed cusp - 2BCF)"},
/* 2312 */ { "ARC","	= position of any line\n"
	"	x (upper half circle - 25E0)"},
/* 2313 */ { "SEGMENT","	= position of a surface"},
/* 2314 */ { "SECTOR",NULL},
/* 2315 */ { "TELEPHONE RECORDER",NULL},
/* 2316 */ { "POSITION INDICATOR","	= true position\n"
	"	x (square position indicator - 2BD0)"},
/* 2317 */ { "VIEWDATA SQUARE","	x (equal and parallel to - 22D5)"},
/* 2318 */ { "PLACE OF INTEREST SIGN","	= command key (1.0)\n"
	"	= operating system key (ISO 9995-7)"},
/* 2319 */ { "TURNED NOT SIGN","	= line marker"},
/* 231A */ { "WATCH","	x (alarm clock - 23F0)\n"
	"	x (clock face one oclock - 1F550)"},
/* 231B */ { "HOURGLASS","	= alchemical symbol for hour"},
/* 231C */ { "TOP LEFT CORNER","	x (right angle substitution marker - 2E00)\n"
	"	x (top left half bracket - 2E22)"},
/* 231D */ { "TOP RIGHT CORNER",NULL},
/* 231E */ { "BOTTOM LEFT CORNER",NULL},
/* 231F */ { "BOTTOM RIGHT CORNER",NULL},
/* 2320 */ { "TOP HALF INTEGRAL","	x (integral extension - 23AE)"},
/* 2321 */ { "BOTTOM HALF INTEGRAL",NULL},
/* 2322 */ { "FROWN","	x (character tie - 2040)"},
/* 2323 */ { "SMILE","	x (undertie - 203F)"},
/* 2324 */ { "UP ARROWHEAD BETWEEN TWO HORIZONTAL BARS","	= enter key"},
/* 2325 */ { "OPTION KEY",NULL},
/* 2326 */ { "ERASE TO THE RIGHT","	= delete to the right key"},
/* 2327 */ { "X IN A RECTANGLE BOX","	= clear key\n"
	"	x (ballot box with x - 2612)"},
/* 2328 */ { "KEYBOARD","	x (keyboard and mouse - 1F5A6)\n"
	"	x (wired keyboard - 1F5AE)"},
/* 2329 */ { "LEFT-POINTING ANGLE BRACKET","	x (less-than sign - 003C)\n"
	"	x (single left-pointing angle quotation mark - 2039)\n"
	"	x (mathematical left angle bracket - 27E8)\n"
	"	: 3008 left angle bracket"},
/* 232A */ { "RIGHT-POINTING ANGLE BRACKET","	x (greater-than sign - 003E)\n"
	"	x (single right-pointing angle quotation mark - 203A)\n"
	"	x (mathematical right angle bracket - 27E9)\n"
	"	: 3009 right angle bracket"},
/* 232B */ { "ERASE TO THE LEFT","	= delete to the left key"},
/* 232C */ { "BENZENE RING",NULL},
/* 232D */ { "CYLINDRICITY",NULL},
/* 232E */ { "ALL AROUND-PROFILE",NULL},
/* 232F */ { "SYMMETRY",NULL},
/* 2330 */ { "TOTAL RUNOUT",NULL},
/* 2331 */ { "DIMENSION ORIGIN",NULL},
/* 2332 */ { "CONICAL TAPER",NULL},
/* 2333 */ { "SLOPE","	x (lower left triangle - 25FA)"},
/* 2334 */ { "COUNTERBORE","	x (open box - 2423)"},
/* 2335 */ { "COUNTERSINK","	x (down arrowhead - 2304)"},
/* 2336 */ { "APL FUNCTIONAL SYMBOL I-BEAM",NULL},
/* 2337 */ { "APL FUNCTIONAL SYMBOL SQUISH QUAD","	x (apl functional symbol quad - 2395)\n"
	"	x (white vertical rectangle - 25AF)\n"
	"	x (n-ary white vertical bar - 2AFF)"},
/* 2338 */ { "APL FUNCTIONAL SYMBOL QUAD EQUAL",NULL},
/* 2339 */ { "APL FUNCTIONAL SYMBOL QUAD DIVIDE",NULL},
/* 233A */ { "APL FUNCTIONAL SYMBOL QUAD DIAMOND","	x (white diamond in square - 26CB)"},
/* 233B */ { "APL FUNCTIONAL SYMBOL QUAD JOT","	x (squared small circle - 29C7)"},
/* 233C */ { "APL FUNCTIONAL SYMBOL QUAD CIRCLE",NULL},
/* 233D */ { "APL FUNCTIONAL SYMBOL CIRCLE STILE",NULL},
/* 233E */ { "APL FUNCTIONAL SYMBOL CIRCLE JOT","	x (circled ring operator - 229A)\n"
	"	x (circled white bullet - 29BE)"},
/* 233F */ { "APL FUNCTIONAL SYMBOL SLASH BAR","	x (roman as sign - 1019A)"},
/* 2340 */ { "APL FUNCTIONAL SYMBOL BACKSLASH BAR","	x (reverse solidus with horizontal stroke - 29F7)"},
/* 2341 */ { "APL FUNCTIONAL SYMBOL QUAD SLASH","	x (squared rising diagonal slash - 29C4)"},
/* 2342 */ { "APL FUNCTIONAL SYMBOL QUAD BACKSLASH","	x (squared falling diagonal slash - 29C5)"},
/* 2343 */ { "APL FUNCTIONAL SYMBOL QUAD LESS-THAN",NULL},
/* 2344 */ { "APL FUNCTIONAL SYMBOL QUAD GREATER-THAN",NULL},
/* 2345 */ { "APL FUNCTIONAL SYMBOL LEFTWARDS VANE",NULL},
/* 2346 */ { "APL FUNCTIONAL SYMBOL RIGHTWARDS VANE",NULL},
/* 2347 */ { "APL FUNCTIONAL SYMBOL QUAD LEFTWARDS ARROW",NULL},
/* 2348 */ { "APL FUNCTIONAL SYMBOL QUAD RIGHTWARDS ARROW",NULL},
/* 2349 */ { "APL FUNCTIONAL SYMBOL CIRCLE BACKSLASH","	x (reversed empty set - 29B0)"},
/* 234A */ { "APL FUNCTIONAL SYMBOL DOWN TACK UNDERBAR","	= up tack underbar\n"
	"	* preferred naming for APL tack symbols now follows the London Convention in ISO/IEC 13751:2000 (APL Extended)\n"
	"	x (up tack - 22A5)"},
/* 234B */ { "APL FUNCTIONAL SYMBOL DELTA STILE",NULL},
/* 234C */ { "APL FUNCTIONAL SYMBOL QUAD DOWN CARET",NULL},
/* 234D */ { "APL FUNCTIONAL SYMBOL QUAD DELTA",NULL},
/* 234E */ { "APL FUNCTIONAL SYMBOL DOWN TACK JOT","	= up tack jot\n"
	"	x (up tack - 22A5)"},
/* 234F */ { "APL FUNCTIONAL SYMBOL UPWARDS VANE",NULL},
/* 2350 */ { "APL FUNCTIONAL SYMBOL QUAD UPWARDS ARROW",NULL},
/* 2351 */ { "APL FUNCTIONAL SYMBOL UP TACK OVERBAR","	= down tack overbar\n"
	"	x (down tack - 22A4)\n"
	"	x (postal mark - 3012)"},
/* 2352 */ { "APL FUNCTIONAL SYMBOL DEL STILE",NULL},
/* 2353 */ { "APL FUNCTIONAL SYMBOL QUAD UP CARET","	x (up arrowhead in a rectangle box - 2BB9)"},
/* 2354 */ { "APL FUNCTIONAL SYMBOL QUAD DEL",NULL},
/* 2355 */ { "APL FUNCTIONAL SYMBOL UP TACK JOT","	= down tack jot\n"
	"	x (down tack - 22A4)"},
/* 2356 */ { "APL FUNCTIONAL SYMBOL DOWNWARDS VANE",NULL},
/* 2357 */ { "APL FUNCTIONAL SYMBOL QUAD DOWNWARDS ARROW",NULL},
/* 2358 */ { "APL FUNCTIONAL SYMBOL QUOTE UNDERBAR",NULL},
/* 2359 */ { "APL FUNCTIONAL SYMBOL DELTA UNDERBAR",NULL},
/* 235A */ { "APL FUNCTIONAL SYMBOL DIAMOND UNDERBAR",NULL},
/* 235B */ { "APL FUNCTIONAL SYMBOL JOT UNDERBAR",NULL},
/* 235C */ { "APL FUNCTIONAL SYMBOL CIRCLE UNDERBAR",NULL},
/* 235D */ { "APL FUNCTIONAL SYMBOL UP SHOE JOT","	x (intersection with dot - 2A40)"},
/* 235E */ { "APL FUNCTIONAL SYMBOL QUOTE QUAD",NULL},
/* 235F */ { "APL FUNCTIONAL SYMBOL CIRCLE STAR","	x (circled asterisk operator - 229B)"},
/* 2360 */ { "APL FUNCTIONAL SYMBOL QUAD COLON",NULL},
/* 2361 */ { "APL FUNCTIONAL SYMBOL UP TACK DIAERESIS","	= down tack diaeresis\n"
	"	x (down tack - 22A4)"},
/* 2362 */ { "APL FUNCTIONAL SYMBOL DEL DIAERESIS",NULL},
/* 2363 */ { "APL FUNCTIONAL SYMBOL STAR DIAERESIS",NULL},
/* 2364 */ { "APL FUNCTIONAL SYMBOL JOT DIAERESIS","	= hoot"},
/* 2365 */ { "APL FUNCTIONAL SYMBOL CIRCLE DIAERESIS","	= holler"},
/* 2366 */ { "APL FUNCTIONAL SYMBOL DOWN SHOE STILE",NULL},
/* 2367 */ { "APL FUNCTIONAL SYMBOL LEFT SHOE STILE",NULL},
/* 2368 */ { "APL FUNCTIONAL SYMBOL TILDE DIAERESIS","	= smirk"},
/* 2369 */ { "APL FUNCTIONAL SYMBOL GREATER-THAN DIAERESIS",NULL},
/* 236A */ { "APL FUNCTIONAL SYMBOL COMMA BAR",NULL},
/* 236B */ { "APL FUNCTIONAL SYMBOL DEL TILDE",NULL},
/* 236C */ { "APL FUNCTIONAL SYMBOL ZILDE",NULL},
/* 236D */ { "APL FUNCTIONAL SYMBOL STILE TILDE",NULL},
/* 236E */ { "APL FUNCTIONAL SYMBOL SEMICOLON UNDERBAR",NULL},
/* 236F */ { "APL FUNCTIONAL SYMBOL QUAD NOT EQUAL",NULL},
/* 2370 */ { "APL FUNCTIONAL SYMBOL QUAD QUESTION","	x (uncertainty sign - 2BD1)"},
/* 2371 */ { "APL FUNCTIONAL SYMBOL DOWN CARET TILDE",NULL},
/* 2372 */ { "APL FUNCTIONAL SYMBOL UP CARET TILDE",NULL},
/* 2373 */ { "APL FUNCTIONAL SYMBOL IOTA",NULL},
/* 2374 */ { "APL FUNCTIONAL SYMBOL RHO",NULL},
/* 2375 */ { "APL FUNCTIONAL SYMBOL OMEGA",NULL},
/* 2376 */ { "APL FUNCTIONAL SYMBOL ALPHA UNDERBAR",NULL},
/* 2377 */ { "APL FUNCTIONAL SYMBOL EPSILON UNDERBAR",NULL},
/* 2378 */ { "APL FUNCTIONAL SYMBOL IOTA UNDERBAR",NULL},
/* 2379 */ { "APL FUNCTIONAL SYMBOL OMEGA UNDERBAR",NULL},
/* 237A */ { "APL FUNCTIONAL SYMBOL ALPHA",NULL},
/* 237B */ { "NOT CHECK MARK","	* from ISO 2047\n"
	"	* symbol for Negative Acknowledge"},
/* 237C */ { "RIGHT ANGLE WITH DOWNWARDS ZIGZAG ARROW",NULL},
/* 237D */ { "SHOULDERED OPEN BOX","	* from ISO 9995-7\n"
	"	* keyboard symbol for No Break Space\n"
	"	x (open box - 2423)"},
/* 237E */ { "BELL SYMBOL","	* from ISO 2047"},
/* 237F */ { "VERTICAL LINE WITH MIDDLE DOT","	* from ISO 2047\n"
	"	* symbol for End of Medium"},
/* 2380 */ { "INSERTION SYMBOL",NULL},
/* 2381 */ { "CONTINUOUS UNDERLINE SYMBOL",NULL},
/* 2382 */ { "DISCONTINUOUS UNDERLINE SYMBOL",NULL},
/* 2383 */ { "EMPHASIS SYMBOL",NULL},
/* 2384 */ { "COMPOSITION SYMBOL",NULL},
/* 2385 */ { "WHITE SQUARE WITH CENTRE VERTICAL LINE","	= center"},
/* 2386 */ { "ENTER SYMBOL",NULL},
/* 2387 */ { "ALTERNATIVE KEY SYMBOL",NULL},
/* 2388 */ { "HELM SYMBOL","	= control\n"
	"	x (wheel of dharma - 2638)"},
/* 2389 */ { "CIRCLED HORIZONTAL BAR WITH NOTCH","	= pause"},
/* 238A */ { "CIRCLED TRIANGLE DOWN","	= interrupt, break"},
/* 238B */ { "BROKEN CIRCLE WITH NORTHWEST ARROW","	= escape"},
/* 238C */ { "UNDO SYMBOL",NULL},
/* 238D */ { "MONOSTABLE SYMBOL",NULL},
/* 238E */ { "HYSTERESIS SYMBOL",NULL},
/* 238F */ { "OPEN-CIRCUIT-OUTPUT H-TYPE SYMBOL",NULL},
/* 2390 */ { "OPEN-CIRCUIT-OUTPUT L-TYPE SYMBOL",NULL},
/* 2391 */ { "PASSIVE-PULL-DOWN-OUTPUT SYMBOL",NULL},
/* 2392 */ { "PASSIVE-PULL-UP-OUTPUT SYMBOL",NULL},
/* 2393 */ { "DIRECT CURRENT SYMBOL FORM TWO",NULL},
/* 2394 */ { "SOFTWARE-FUNCTION SYMBOL","	x (white hexagon - 2B21)"},
/* 2395 */ { "APL FUNCTIONAL SYMBOL QUAD","	x (apl functional symbol squish quad - 2337)\n"
	"	x (white vertical rectangle - 25AF)"},
/* 2396 */ { "DECIMAL SEPARATOR KEY SYMBOL",NULL},
/* 2397 */ { "PREVIOUS PAGE","	x (back with leftwards arrow above - 1F519)"},
/* 2398 */ { "NEXT PAGE",NULL},
/* 2399 */ { "PRINT SCREEN SYMBOL",NULL},
/* 239A */ { "CLEAR SCREEN SYMBOL","	x (combining enclosing screen - 20E2)"},
/* 239B */ { "LEFT PARENTHESIS UPPER HOOK",NULL},
/* 239C */ { "LEFT PARENTHESIS EXTENSION",NULL},
/* 239D */ { "LEFT PARENTHESIS LOWER HOOK",NULL},
/* 239E */ { "RIGHT PARENTHESIS UPPER HOOK",NULL},
/* 239F */ { "RIGHT PARENTHESIS EXTENSION",NULL},
/* 23A0 */ { "RIGHT PARENTHESIS LOWER HOOK",NULL},
/* 23A1 */ { "LEFT SQUARE BRACKET UPPER CORNER",NULL},
/* 23A2 */ { "LEFT SQUARE BRACKET EXTENSION",NULL},
/* 23A3 */ { "LEFT SQUARE BRACKET LOWER CORNER",NULL},
/* 23A4 */ { "RIGHT SQUARE BRACKET UPPER CORNER",NULL},
/* 23A5 */ { "RIGHT SQUARE BRACKET EXTENSION",NULL},
/* 23A6 */ { "RIGHT SQUARE BRACKET LOWER CORNER",NULL},
/* 23A7 */ { "LEFT CURLY BRACKET UPPER HOOK",NULL},
/* 23A8 */ { "LEFT CURLY BRACKET MIDDLE PIECE",NULL},
/* 23A9 */ { "LEFT CURLY BRACKET LOWER HOOK",NULL},
/* 23AA */ { "CURLY BRACKET EXTENSION",NULL},
/* 23AB */ { "RIGHT CURLY BRACKET UPPER HOOK",NULL},
/* 23AC */ { "RIGHT CURLY BRACKET MIDDLE PIECE",NULL},
/* 23AD */ { "RIGHT CURLY BRACKET LOWER HOOK",NULL},
/* 23AE */ { "INTEGRAL EXTENSION","	x (top half integral - 2320)\n"
	"	x (bottom half integral - 2321)"},
/* 23AF */ { "HORIZONTAL LINE EXTENSION","	* used for extension of arrows\n"
	"	x (vertical line extension - 23D0)"},
/* 23B0 */ { "UPPER LEFT OR LOWER RIGHT CURLY BRACKET SECTION","	= left moustache"},
/* 23B1 */ { "UPPER RIGHT OR LOWER LEFT CURLY BRACKET SECTION","	= right moustache"},
/* 23B2 */ { "SUMMATION TOP",NULL},
/* 23B3 */ { "SUMMATION BOTTOM","	x (n-ary summation - 2211)"},
/* 23B4 */ { "TOP SQUARE BRACKET","	x (presentation form for vertical left square bracket - FE47)"},
/* 23B5 */ { "BOTTOM SQUARE BRACKET","	x (presentation form for vertical right square bracket - FE48)"},
/* 23B6 */ { "BOTTOM SQUARE BRACKET OVER TOP SQUARE BRACKET","	* only used for terminal emulation"},
/* 23B7 */ { "RADICAL SYMBOL BOTTOM",NULL},
/* 23B8 */ { "LEFT VERTICAL BOX LINE",NULL},
/* 23B9 */ { "RIGHT VERTICAL BOX LINE",NULL},
/* 23BA */ { "HORIZONTAL SCAN LINE-1",NULL},
/* 23BB */ { "HORIZONTAL SCAN LINE-3","	* scan line-5 is unified with the box-drawing graphic character 2500"},
/* 23BC */ { "HORIZONTAL SCAN LINE-7",NULL},
/* 23BD */ { "HORIZONTAL SCAN LINE-9",NULL},
/* 23BE */ { "DENTISTRY SYMBOL LIGHT VERTICAL AND TOP RIGHT","	x (left ceiling - 2308)"},
/* 23BF */ { "DENTISTRY SYMBOL LIGHT VERTICAL AND BOTTOM RIGHT","	x (left floor - 230A)"},
/* 23C0 */ { "DENTISTRY SYMBOL LIGHT VERTICAL WITH CIRCLE",NULL},
/* 23C1 */ { "DENTISTRY SYMBOL LIGHT DOWN AND HORIZONTAL WITH CIRCLE",NULL},
/* 23C2 */ { "DENTISTRY SYMBOL LIGHT UP AND HORIZONTAL WITH CIRCLE",NULL},
/* 23C3 */ { "DENTISTRY SYMBOL LIGHT VERTICAL WITH TRIANGLE",NULL},
/* 23C4 */ { "DENTISTRY SYMBOL LIGHT DOWN AND HORIZONTAL WITH TRIANGLE",NULL},
/* 23C5 */ { "DENTISTRY SYMBOL LIGHT UP AND HORIZONTAL WITH TRIANGLE",NULL},
/* 23C6 */ { "DENTISTRY SYMBOL LIGHT VERTICAL AND WAVE",NULL},
/* 23C7 */ { "DENTISTRY SYMBOL LIGHT DOWN AND HORIZONTAL WITH WAVE",NULL},
/* 23C8 */ { "DENTISTRY SYMBOL LIGHT UP AND HORIZONTAL WITH WAVE",NULL},
/* 23C9 */ { "DENTISTRY SYMBOL LIGHT DOWN AND HORIZONTAL","	x (down tack - 22A4)\n"
	"	x (box drawings light down and horizontal - 252C)"},
/* 23CA */ { "DENTISTRY SYMBOL LIGHT UP AND HORIZONTAL","	x (up tack - 22A5)\n"
	"	x (box drawings light up and horizontal - 2534)"},
/* 23CB */ { "DENTISTRY SYMBOL LIGHT VERTICAL AND TOP LEFT","	x (right ceiling - 2309)"},
/* 23CC */ { "DENTISTRY SYMBOL LIGHT VERTICAL AND BOTTOM LEFT","	x (right floor - 230B)"},
/* 23CD */ { "SQUARE FOOT",NULL},
/* 23CE */ { "RETURN SYMBOL","	* may be shown with either hollow or filled glyph\n"
	"	x (downwards arrow with corner leftwards - 21B5)\n"
	"	x (return left - 2B90)"},
/* 23CF */ { "EJECT SYMBOL","	* UI symbol to eject media"},
/* 23D0 */ { "VERTICAL LINE EXTENSION","	* used for extension of arrows\n"
	"	x (horizontal line extension - 23AF)"},
/* 23D1 */ { "METRICAL BREVE",NULL},
/* 23D2 */ { "METRICAL LONG OVER SHORT",NULL},
/* 23D3 */ { "METRICAL SHORT OVER LONG",NULL},
/* 23D4 */ { "METRICAL LONG OVER TWO SHORTS",NULL},
/* 23D5 */ { "METRICAL TWO SHORTS OVER LONG",NULL},
/* 23D6 */ { "METRICAL TWO SHORTS JOINED",NULL},
/* 23D7 */ { "METRICAL TRISEME","	x (combining greek musical triseme - 1D242)"},
/* 23D8 */ { "METRICAL TETRASEME","	x (combining greek musical tetraseme - 1D243)"},
/* 23D9 */ { "METRICAL PENTASEME","	x (combining greek musical pentaseme - 1D244)"},
/* 23DA */ { "EARTH GROUND",NULL},
/* 23DB */ { "FUSE",NULL},
/* 23DC */ { "TOP PARENTHESIS","	x (presentation form for vertical left parenthesis - FE35)"},
/* 23DD */ { "BOTTOM PARENTHESIS","	x (presentation form for vertical right parenthesis - FE36)"},
/* 23DE */ { "TOP CURLY BRACKET","	x (presentation form for vertical left curly bracket - FE37)"},
/* 23DF */ { "BOTTOM CURLY BRACKET","	x (presentation form for vertical right curly bracket - FE38)"},
/* 23E0 */ { "TOP TORTOISE SHELL BRACKET","	x (presentation form for vertical left tortoise shell bracket - FE39)"},
/* 23E1 */ { "BOTTOM TORTOISE SHELL BRACKET","	x (presentation form for vertical right tortoise shell bracket - FE3A)"},
/* 23E2 */ { "WHITE TRAPEZIUM",NULL},
/* 23E3 */ { "BENZENE RING WITH CIRCLE",NULL},
/* 23E4 */ { "STRAIGHTNESS",NULL},
/* 23E5 */ { "FLATNESS",NULL},
/* 23E6 */ { "AC CURRENT",NULL},
/* 23E7 */ { "ELECTRICAL INTERSECTION",NULL},
/* 23E8 */ { "DECIMAL EXPONENT SYMBOL","	* Algol-60 token for scientific notation literals"},
/* 23E9 */ { "BLACK RIGHT-POINTING DOUBLE TRIANGLE","	= fast forward"},
/* 23EA */ { "BLACK LEFT-POINTING DOUBLE TRIANGLE","	= rewind, fast backwards"},
/* 23EB */ { "BLACK UP-POINTING DOUBLE TRIANGLE",NULL},
/* 23EC */ { "BLACK DOWN-POINTING DOUBLE TRIANGLE",NULL},
/* 23ED */ { "BLACK RIGHT-POINTING DOUBLE TRIANGLE WITH VERTICAL BAR","	= skip to the end or next file/track/chapter"},
/* 23EE */ { "BLACK LEFT-POINTING DOUBLE TRIANGLE WITH VERTICAL BAR","	= skip to the start or previous file/track/chapter"},
/* 23EF */ { "BLACK RIGHT-POINTING TRIANGLE WITH DOUBLE VERTICAL BAR","	= play/pause toggle"},
/* 23F0 */ { "ALARM CLOCK","	x (watch - 231A)\n"
	"	x (clock face one oclock - 1F550)\n"
	"	x (mantelpiece clock - 1F570)"},
/* 23F1 */ { "STOPWATCH",NULL},
/* 23F2 */ { "TIMER CLOCK",NULL},
/* 23F3 */ { "HOURGLASS WITH FLOWING SAND","	x (hourglass - 231B)"},
/* 23F4 */ { "BLACK MEDIUM LEFT-POINTING TRIANGLE","	= reverse, back\n"
	"	* preferred to 25C0 black left-pointing triangle\n"
	"	x (black medium left-pointing triangle centred - 2BC7)\n"
	"	x (black left-pointing isosceles right triangle - 1F780)"},
/* 23F5 */ { "BLACK MEDIUM RIGHT-POINTING TRIANGLE","	= forward\n"
	"	* preferred to 25B6 black right-pointing triangle\n"
	"	x (black medium right-pointing triangle centred - 2BC8)\n"
	"	x (black right-pointing isosceles right triangle - 1F782)"},
/* 23F6 */ { "BLACK MEDIUM UP-POINTING TRIANGLE","	= increase\n"
	"	* preferred to 25B2 black up-pointing triangle\n"
	"	x (black medium up-pointing triangle centred - 2BC5)\n"
	"	x (black up-pointing isosceles right triangle - 1F781)"},
/* 23F7 */ { "BLACK MEDIUM DOWN-POINTING TRIANGLE","	= decrease\n"
	"	* preferred to 25BC black down-pointing triangle\n"
	"	x (black medium down-pointing triangle centred - 2BC6)\n"
	"	x (black down-pointing isosceles right triangle - 1F783)"},
/* 23F8 */ { "DOUBLE VERTICAL BAR","	= pause\n"
	"	x (double vertical line - 2016)"},
/* 23F9 */ { "BLACK SQUARE FOR STOP","	= stop\n"
	"	x (black medium square - 25FC)"},
/* 23FA */ { "BLACK CIRCLE FOR RECORD","	= record\n"
	"	x (medium black circle - 26AB)\n"
	"	x (large red circle - 1F534)"},
/* 23FB */ { "POWER SYMBOL","	* IEC 5009 standby symbol\n"
	"	* IEEE 1621 power symbol"},
/* 23FC */ { "POWER ON-OFF SYMBOL","	* IEC 5010 power on-off symbol"},
/* 23FD */ { "POWER ON SYMBOL","	* use 2B58 for power off symbol\n"
	"	x (heavy circle - 2B58)"},
/* 23FE */ { "POWER SLEEP SYMBOL","	x (last quarter moon - 263E)"},
/* 23FF */ { "OBSERVER EYE SYMBOL",NULL}
};

UN_DLL_LOCAL
static const struct unicode_nameannot una_00_24[] = {
/* 2400 */ { "SYMBOL FOR NULL",NULL},
/* 2401 */ { "SYMBOL FOR START OF HEADING",NULL},
/* 2402 */ { "SYMBOL FOR START OF TEXT",NULL},
/* 2403 */ { "SYMBOL FOR END OF TEXT",NULL},
/* 2404 */ { "SYMBOL FOR END OF TRANSMISSION",NULL},
/* 2405 */ { "SYMBOL FOR ENQUIRY",NULL},
/* 2406 */ { "SYMBOL FOR ACKNOWLEDGE",NULL},
/* 2407 */ { "SYMBOL FOR BELL",NULL},
/* 2408 */ { "SYMBOL FOR BACKSPACE",NULL},
/* 2409 */ { "SYMBOL FOR HORIZONTAL TABULATION",NULL},
/* 240A */ { "SYMBOL FOR LINE FEED",NULL},
/* 240B */ { "SYMBOL FOR VERTICAL TABULATION",NULL},
/* 240C */ { "SYMBOL FOR FORM FEED",NULL},
/* 240D */ { "SYMBOL FOR CARRIAGE RETURN",NULL},
/* 240E */ { "SYMBOL FOR SHIFT OUT",NULL},
/* 240F */ { "SYMBOL FOR SHIFT IN",NULL},
/* 2410 */ { "SYMBOL FOR DATA LINK ESCAPE",NULL},
/* 2411 */ { "SYMBOL FOR DEVICE CONTROL ONE",NULL},
/* 2412 */ { "SYMBOL FOR DEVICE CONTROL TWO",NULL},
/* 2413 */ { "SYMBOL FOR DEVICE CONTROL THREE",NULL},
/* 2414 */ { "SYMBOL FOR DEVICE CONTROL FOUR",NULL},
/* 2415 */ { "SYMBOL FOR NEGATIVE ACKNOWLEDGE",NULL},
/* 2416 */ { "SYMBOL FOR SYNCHRONOUS IDLE",NULL},
/* 2417 */ { "SYMBOL FOR END OF TRANSMISSION BLOCK",NULL},
/* 2418 */ { "SYMBOL FOR CANCEL",NULL},
/* 2419 */ { "SYMBOL FOR END OF MEDIUM",NULL},
/* 241A */ { "SYMBOL FOR SUBSTITUTE",NULL},
/* 241B */ { "SYMBOL FOR ESCAPE",NULL},
/* 241C */ { "SYMBOL FOR FILE SEPARATOR",NULL},
/* 241D */ { "SYMBOL FOR GROUP SEPARATOR",NULL},
/* 241E */ { "SYMBOL FOR RECORD SEPARATOR",NULL},
/* 241F */ { "SYMBOL FOR UNIT SEPARATOR",NULL},
/* 2420 */ { "SYMBOL FOR SPACE",NULL},
/* 2421 */ { "SYMBOL FOR DELETE",NULL},
/* 2422 */ { "BLANK SYMBOL","	* graphic for space\n"
	"	x (latin small letter b with stroke - 0180)"},
/* 2423 */ { "OPEN BOX","	* graphic for space\n"
	"	x (counterbore - 2334)"},
/* 2424 */ { "SYMBOL FOR NEWLINE",NULL},
/* 2425 */ { "SYMBOL FOR DELETE FORM TWO","	* from ISO 9995-7\n"
	"	* keyboard symbol for undoable delete"},
/* 2426 */ { "SYMBOL FOR SUBSTITUTE FORM TWO","	* from ISO 2047\n"
	"	x (arabic question mark - 061F)"},
/* 2427 */ { "SYMBOL FOR DELETE SQUARE CHECKER BOARD FORM",NULL},
/* 2428 */ { "SYMBOL FOR DELETE RECTANGULAR CHECKER BOARD FORM",NULL},
/* 2429 */ { "SYMBOL FOR DELETE MEDIUM SHADE FORM",NULL},
/* 242A */ { NULL,NULL},
/* 242B */ { NULL,NULL},
/* 242C */ { NULL,NULL},
/* 242D */ { NULL,NULL},
/* 242E */ { NULL,NULL},
/* 242F */ { NULL,NULL},
/* 2430 */ { NULL,NULL},
/* 2431 */ { NULL,NULL},
/* 2432 */ { NULL,NULL},
/* 2433 */ { NULL,NULL},
/* 2434 */ { NULL,NULL},
/* 2435 */ { NULL,NULL},
/* 2436 */ { NULL,NULL},
/* 2437 */ { NULL,NULL},
/* 2438 */ { NULL,NULL},
/* 2439 */ { NULL,NULL},
/* 243A */ { NULL,NULL},
/* 243B */ { NULL,NULL},
/* 243C */ { NULL,NULL},
/* 243D */ { NULL,NULL},
/* 243E */ { NULL,NULL},
/* 243F */ { NULL,NULL},
/* 2440 */ { "OCR HOOK",NULL},
/* 2441 */ { "OCR CHAIR",NULL},
/* 2442 */ { "OCR FORK",NULL},
/* 2443 */ { "OCR INVERTED FORK",NULL},
/* 2444 */ { "OCR BELT BUCKLE",NULL},
/* 2445 */ { "OCR BOW TIE","	= unique asterisk\n"
	"	x (bowtie - 22C8)"},
/* 2446 */ { "OCR BRANCH BANK IDENTIFICATION","	= transit"},
/* 2447 */ { "OCR AMOUNT OF CHECK","	= amount"},
/* 2448 */ { "OCR DASH","	% MICR ON US SYMBOL\n"
	"	= on us"},
/* 2449 */ { "OCR CUSTOMER ACCOUNT NUMBER","	% MICR DASH SYMBOL\n"
	"	= dash"},
/* 244A */ { "OCR DOUBLE BACKSLASH",NULL},
/* 244B */ { NULL,NULL},
/* 244C */ { NULL,NULL},
/* 244D */ { NULL,NULL},
/* 244E */ { NULL,NULL},
/* 244F */ { NULL,NULL},
/* 2450 */ { NULL,NULL},
/* 2451 */ { NULL,NULL},
/* 2452 */ { NULL,NULL},
/* 2453 */ { NULL,NULL},
/* 2454 */ { NULL,NULL},
/* 2455 */ { NULL,NULL},
/* 2456 */ { NULL,NULL},
/* 2457 */ { NULL,NULL},
/* 2458 */ { NULL,NULL},
/* 2459 */ { NULL,NULL},
/* 245A */ { NULL,NULL},
/* 245B */ { NULL,NULL},
/* 245C */ { NULL,NULL},
/* 245D */ { NULL,NULL},
/* 245E */ { NULL,NULL},
/* 245F */ { NULL,NULL},
/* 2460 */ { "CIRCLED DIGIT ONE","	# <circle> 0031"},
/* 2461 */ { "CIRCLED DIGIT TWO","	# <circle> 0032"},
/* 2462 */ { "CIRCLED DIGIT THREE","	# <circle> 0033"},
/* 2463 */ { "CIRCLED DIGIT FOUR","	# <circle> 0034"},
/* 2464 */ { "CIRCLED DIGIT FIVE","	# <circle> 0035"},
/* 2465 */ { "CIRCLED DIGIT SIX","	# <circle> 0036"},
/* 2466 */ { "CIRCLED DIGIT SEVEN","	# <circle> 0037"},
/* 2467 */ { "CIRCLED DIGIT EIGHT","	# <circle> 0038"},
/* 2468 */ { "CIRCLED DIGIT NINE","	# <circle> 0039"},
/* 2469 */ { "CIRCLED NUMBER TEN","	# <circle> 0031 0030"},
/* 246A */ { "CIRCLED NUMBER ELEVEN","	# <circle> 0031 0031"},
/* 246B */ { "CIRCLED NUMBER TWELVE","	# <circle> 0031 0032"},
/* 246C */ { "CIRCLED NUMBER THIRTEEN","	# <circle> 0031 0033"},
/* 246D */ { "CIRCLED NUMBER FOURTEEN","	# <circle> 0031 0034"},
/* 246E */ { "CIRCLED NUMBER FIFTEEN","	# <circle> 0031 0035"},
/* 246F */ { "CIRCLED NUMBER SIXTEEN","	# <circle> 0031 0036"},
/* 2470 */ { "CIRCLED NUMBER SEVENTEEN","	# <circle> 0031 0037"},
/* 2471 */ { "CIRCLED NUMBER EIGHTEEN","	# <circle> 0031 0038"},
/* 2472 */ { "CIRCLED NUMBER NINETEEN","	# <circle> 0031 0039"},
/* 2473 */ { "CIRCLED NUMBER TWENTY","	# <circle> 0032 0030"},
/* 2474 */ { "PARENTHESIZED DIGIT ONE","	# 0028 0031 0029"},
/* 2475 */ { "PARENTHESIZED DIGIT TWO","	# 0028 0032 0029"},
/* 2476 */ { "PARENTHESIZED DIGIT THREE","	# 0028 0033 0029"},
/* 2477 */ { "PARENTHESIZED DIGIT FOUR","	# 0028 0034 0029"},
/* 2478 */ { "PARENTHESIZED DIGIT FIVE","	# 0028 0035 0029"},
/* 2479 */ { "PARENTHESIZED DIGIT SIX","	# 0028 0036 0029"},
/* 247A */ { "PARENTHESIZED DIGIT SEVEN","	# 0028 0037 0029"},
/* 247B */ { "PARENTHESIZED DIGIT EIGHT","	# 0028 0038 0029"},
/* 247C */ { "PARENTHESIZED DIGIT NINE","	# 0028 0039 0029"},
/* 247D */ { "PARENTHESIZED NUMBER TEN","	# 0028 0031 0030 0029"},
/* 247E */ { "PARENTHESIZED NUMBER ELEVEN","	# 0028 0031 0031 0029"},
/* 247F */ { "PARENTHESIZED NUMBER TWELVE","	# 0028 0031 0032 0029"},
/* 2480 */ { "PARENTHESIZED NUMBER THIRTEEN","	# 0028 0031 0033 0029"},
/* 2481 */ { "PARENTHESIZED NUMBER FOURTEEN","	# 0028 0031 0034 0029"},
/* 2482 */ { "PARENTHESIZED NUMBER FIFTEEN","	# 0028 0031 0035 0029"},
/* 2483 */ { "PARENTHESIZED NUMBER SIXTEEN","	# 0028 0031 0036 0029"},
/* 2484 */ { "PARENTHESIZED NUMBER SEVENTEEN","	# 0028 0031 0037 0029"},
/* 2485 */ { "PARENTHESIZED NUMBER EIGHTEEN","	# 0028 0031 0038 0029"},
/* 2486 */ { "PARENTHESIZED NUMBER NINETEEN","	# 0028 0031 0039 0029"},
/* 2487 */ { "PARENTHESIZED NUMBER TWENTY","	# 0028 0032 0030 0029"},
/* 2488 */ { "DIGIT ONE FULL STOP","	# 0031 002E"},
/* 2489 */ { "DIGIT TWO FULL STOP","	# 0032 002E"},
/* 248A */ { "DIGIT THREE FULL STOP","	# 0033 002E"},
/* 248B */ { "DIGIT FOUR FULL STOP","	# 0034 002E"},
/* 248C */ { "DIGIT FIVE FULL STOP","	# 0035 002E"},
/* 248D */ { "DIGIT SIX FULL STOP","	# 0036 002E"},
/* 248E */ { "DIGIT SEVEN FULL STOP","	# 0037 002E"},
/* 248F */ { "DIGIT EIGHT FULL STOP","	# 0038 002E"},
/* 2490 */ { "DIGIT NINE FULL STOP","	# 0039 002E"},
/* 2491 */ { "NUMBER TEN FULL STOP","	# 0031 0030 002E"},
/* 2492 */ { "NUMBER ELEVEN FULL STOP","	# 0031 0031 002E"},
/* 2493 */ { "NUMBER TWELVE FULL STOP","	# 0031 0032 002E"},
/* 2494 */ { "NUMBER THIRTEEN FULL STOP","	# 0031 0033 002E"},
/* 2495 */ { "NUMBER FOURTEEN FULL STOP","	# 0031 0034 002E"},
/* 2496 */ { "NUMBER FIFTEEN FULL STOP","	# 0031 0035 002E"},
/* 2497 */ { "NUMBER SIXTEEN FULL STOP","	# 0031 0036 002E"},
/* 2498 */ { "NUMBER SEVENTEEN FULL STOP","	# 0031 0037 002E"},
/* 2499 */ { "NUMBER EIGHTEEN FULL STOP","	# 0031 0038 002E"},
/* 249A */ { "NUMBER NINETEEN FULL STOP","	# 0031 0039 002E"},
/* 249B */ { "NUMBER TWENTY FULL STOP","	# 0032 0030 002E"},
/* 249C */ { "PARENTHESIZED LATIN SMALL LETTER A","	# 0028 0061 0029"},
/* 249D */ { "PARENTHESIZED LATIN SMALL LETTER B","	# 0028 0062 0029"},
/* 249E */ { "PARENTHESIZED LATIN SMALL LETTER C","	# 0028 0063 0029"},
/* 249F */ { "PARENTHESIZED LATIN SMALL LETTER D","	# 0028 0064 0029"},
/* 24A0 */ { "PARENTHESIZED LATIN SMALL LETTER E","	# 0028 0065 0029"},
/* 24A1 */ { "PARENTHESIZED LATIN SMALL LETTER F","	# 0028 0066 0029"},
/* 24A2 */ { "PARENTHESIZED LATIN SMALL LETTER G","	# 0028 0067 0029"},
/* 24A3 */ { "PARENTHESIZED LATIN SMALL LETTER H","	# 0028 0068 0029"},
/* 24A4 */ { "PARENTHESIZED LATIN SMALL LETTER I","	# 0028 0069 0029"},
/* 24A5 */ { "PARENTHESIZED LATIN SMALL LETTER J","	# 0028 006A 0029"},
/* 24A6 */ { "PARENTHESIZED LATIN SMALL LETTER K","	# 0028 006B 0029"},
/* 24A7 */ { "PARENTHESIZED LATIN SMALL LETTER L","	# 0028 006C 0029"},
/* 24A8 */ { "PARENTHESIZED LATIN SMALL LETTER M","	# 0028 006D 0029"},
/* 24A9 */ { "PARENTHESIZED LATIN SMALL LETTER N","	# 0028 006E 0029"},
/* 24AA */ { "PARENTHESIZED LATIN SMALL LETTER O","	# 0028 006F 0029"},
/* 24AB */ { "PARENTHESIZED LATIN SMALL LETTER P","	# 0028 0070 0029"},
/* 24AC */ { "PARENTHESIZED LATIN SMALL LETTER Q","	# 0028 0071 0029"},
/* 24AD */ { "PARENTHESIZED LATIN SMALL LETTER R","	# 0028 0072 0029"},
/* 24AE */ { "PARENTHESIZED LATIN SMALL LETTER S","	# 0028 0073 0029"},
/* 24AF */ { "PARENTHESIZED LATIN SMALL LETTER T","	# 0028 0074 0029"},
/* 24B0 */ { "PARENTHESIZED LATIN SMALL LETTER U","	# 0028 0075 0029"},
/* 24B1 */ { "PARENTHESIZED LATIN SMALL LETTER V","	# 0028 0076 0029"},
/* 24B2 */ { "PARENTHESIZED LATIN SMALL LETTER W","	# 0028 0077 0029"},
/* 24B3 */ { "PARENTHESIZED LATIN SMALL LETTER X","	# 0028 0078 0029"},
/* 24B4 */ { "PARENTHESIZED LATIN SMALL LETTER Y","	# 0028 0079 0029"},
/* 24B5 */ { "PARENTHESIZED LATIN SMALL LETTER Z","	# 0028 007A 0029"},
/* 24B6 */ { "CIRCLED LATIN CAPITAL LETTER A","	# <circle> 0041"},
/* 24B7 */ { "CIRCLED LATIN CAPITAL LETTER B","	# <circle> 0042"},
/* 24B8 */ { "CIRCLED LATIN CAPITAL LETTER C","	x (copyright sign - 00A9)\n"
	"	# <circle> 0043"},
/* 24B9 */ { "CIRCLED LATIN CAPITAL LETTER D","	# <circle> 0044"},
/* 24BA */ { "CIRCLED LATIN CAPITAL LETTER E","	# <circle> 0045"},
/* 24BB */ { "CIRCLED LATIN CAPITAL LETTER F","	# <circle> 0046"},
/* 24BC */ { "CIRCLED LATIN CAPITAL LETTER G","	# <circle> 0047"},
/* 24BD */ { "CIRCLED LATIN CAPITAL LETTER H","	# <circle> 0048"},
/* 24BE */ { "CIRCLED LATIN CAPITAL LETTER I","	# <circle> 0049"},
/* 24BF */ { "CIRCLED LATIN CAPITAL LETTER J","	# <circle> 004A"},
/* 24C0 */ { "CIRCLED LATIN CAPITAL LETTER K","	# <circle> 004B"},
/* 24C1 */ { "CIRCLED LATIN CAPITAL LETTER L","	# <circle> 004C"},
/* 24C2 */ { "CIRCLED LATIN CAPITAL LETTER M","	x (mask work symbol - 1F1AD)\n"
	"	# <circle> 004D"},
/* 24C3 */ { "CIRCLED LATIN CAPITAL LETTER N","	# <circle> 004E"},
/* 24C4 */ { "CIRCLED LATIN CAPITAL LETTER O","	# <circle> 004F"},
/* 24C5 */ { "CIRCLED LATIN CAPITAL LETTER P","	x (sound recording copyright - 2117)\n"
	"	# <circle> 0050"},
/* 24C6 */ { "CIRCLED LATIN CAPITAL LETTER Q","	# <circle> 0051"},
/* 24C7 */ { "CIRCLED LATIN CAPITAL LETTER R","	x (registered sign - 00AE)\n"
	"	# <circle> 0052"},
/* 24C8 */ { "CIRCLED LATIN CAPITAL LETTER S","	# <circle> 0053"},
/* 24C9 */ { "CIRCLED LATIN CAPITAL LETTER T","	# <circle> 0054"},
/* 24CA */ { "CIRCLED LATIN CAPITAL LETTER U","	# <circle> 0055"},
/* 24CB */ { "CIRCLED LATIN CAPITAL LETTER V","	# <circle> 0056"},
/* 24CC */ { "CIRCLED LATIN CAPITAL LETTER W","	# <circle> 0057"},
/* 24CD */ { "CIRCLED LATIN CAPITAL LETTER X","	# <circle> 0058"},
/* 24CE */ { "CIRCLED LATIN CAPITAL LETTER Y","	# <circle> 0059"},
/* 24CF */ { "CIRCLED LATIN CAPITAL LETTER Z","	# <circle> 005A"},
/* 24D0 */ { "CIRCLED LATIN SMALL LETTER A","	# <circle> 0061"},
/* 24D1 */ { "CIRCLED LATIN SMALL LETTER B","	# <circle> 0062"},
/* 24D2 */ { "CIRCLED LATIN SMALL LETTER C","	# <circle> 0063"},
/* 24D3 */ { "CIRCLED LATIN SMALL LETTER D","	# <circle> 0064"},
/* 24D4 */ { "CIRCLED LATIN SMALL LETTER E","	# <circle> 0065"},
/* 24D5 */ { "CIRCLED LATIN SMALL LETTER F","	# <circle> 0066"},
/* 24D6 */ { "CIRCLED LATIN SMALL LETTER G","	# <circle> 0067"},
/* 24D7 */ { "CIRCLED LATIN SMALL LETTER H","	# <circle> 0068"},
/* 24D8 */ { "CIRCLED LATIN SMALL LETTER I","	# <circle> 0069"},
/* 24D9 */ { "CIRCLED LATIN SMALL LETTER J","	# <circle> 006A"},
/* 24DA */ { "CIRCLED LATIN SMALL LETTER K","	# <circle> 006B"},
/* 24DB */ { "CIRCLED LATIN SMALL LETTER L","	# <circle> 006C"},
/* 24DC */ { "CIRCLED LATIN SMALL LETTER M","	# <circle> 006D"},
/* 24DD */ { "CIRCLED LATIN SMALL LETTER N","	# <circle> 006E"},
/* 24DE */ { "CIRCLED LATIN SMALL LETTER O","	# <circle> 006F"},
/* 24DF */ { "CIRCLED LATIN SMALL LETTER P","	# <circle> 0070"},
/* 24E0 */ { "CIRCLED LATIN SMALL LETTER Q","	# <circle> 0071"},
/* 24E1 */ { "CIRCLED LATIN SMALL LETTER R","	# <circle> 0072"},
/* 24E2 */ { "CIRCLED LATIN SMALL LETTER S","	# <circle> 0073"},
/* 24E3 */ { "CIRCLED LATIN SMALL LETTER T","	# <circle> 0074"},
/* 24E4 */ { "CIRCLED LATIN SMALL LETTER U","	# <circle> 0075"},
/* 24E5 */ { "CIRCLED LATIN SMALL LETTER V","	# <circle> 0076"},
/* 24E6 */ { "CIRCLED LATIN SMALL LETTER W","	# <circle> 0077"},
/* 24E7 */ { "CIRCLED LATIN SMALL LETTER X","	# <circle> 0078"},
/* 24E8 */ { "CIRCLED LATIN SMALL LETTER Y","	# <circle> 0079"},
/* 24E9 */ { "CIRCLED LATIN SMALL LETTER Z","	# <circle> 007A"},
/* 24EA */ { "CIRCLED DIGIT ZERO","	# <circle> 0030"},
/* 24EB */ { "NEGATIVE CIRCLED NUMBER ELEVEN",NULL},
/* 24EC */ { "NEGATIVE CIRCLED NUMBER TWELVE",NULL},
/* 24ED */ { "NEGATIVE CIRCLED NUMBER THIRTEEN",NULL},
/* 24EE */ { "NEGATIVE CIRCLED NUMBER FOURTEEN",NULL},
/* 24EF */ { "NEGATIVE CIRCLED NUMBER FIFTEEN",NULL},
/* 24F0 */ { "NEGATIVE CIRCLED NUMBER SIXTEEN",NULL},
/* 24F1 */ { "NEGATIVE CIRCLED NUMBER SEVENTEEN",NULL},
/* 24F2 */ { "NEGATIVE CIRCLED NUMBER EIGHTEEN",NULL},
/* 24F3 */ { "NEGATIVE CIRCLED NUMBER NINETEEN",NULL},
/* 24F4 */ { "NEGATIVE CIRCLED NUMBER TWENTY",NULL},
/* 24F5 */ { "DOUBLE CIRCLED DIGIT ONE",NULL},
/* 24F6 */ { "DOUBLE CIRCLED DIGIT TWO",NULL},
/* 24F7 */ { "DOUBLE CIRCLED DIGIT THREE",NULL},
/* 24F8 */ { "DOUBLE CIRCLED DIGIT FOUR",NULL},
/* 24F9 */ { "DOUBLE CIRCLED DIGIT FIVE",NULL},
/* 24FA */ { "DOUBLE CIRCLED DIGIT SIX",NULL},
/* 24FB */ { "DOUBLE CIRCLED DIGIT SEVEN",NULL},
/* 24FC */ { "DOUBLE CIRCLED DIGIT EIGHT",NULL},
/* 24FD */ { "DOUBLE CIRCLED DIGIT NINE",NULL},
/* 24FE */ { "DOUBLE CIRCLED NUMBER TEN",NULL},
/* 24FF */ { "NEGATIVE CIRCLED DIGIT ZERO","	x (dingbat negative circled digit one - 2776)"}
};

UN_DLL_LOCAL
static const struct unicode_nameannot una_00_25[] = {
/* 2500 */ { "BOX DRAWINGS LIGHT HORIZONTAL","	= Videotex Mosaic DG 15\n"
	"	x (box drawings light horizontal with vertical stroke - 1FBAF)"},
/* 2501 */ { "BOX DRAWINGS HEAVY HORIZONTAL",NULL},
/* 2502 */ { "BOX DRAWINGS LIGHT VERTICAL","	= Videotex Mosaic DG 14"},
/* 2503 */ { "BOX DRAWINGS HEAVY VERTICAL",NULL},
/* 2504 */ { "BOX DRAWINGS LIGHT TRIPLE DASH HORIZONTAL",NULL},
/* 2505 */ { "BOX DRAWINGS HEAVY TRIPLE DASH HORIZONTAL",NULL},
/* 2506 */ { "BOX DRAWINGS LIGHT TRIPLE DASH VERTICAL",NULL},
/* 2507 */ { "BOX DRAWINGS HEAVY TRIPLE DASH VERTICAL",NULL},
/* 2508 */ { "BOX DRAWINGS LIGHT QUADRUPLE DASH HORIZONTAL",NULL},
/* 2509 */ { "BOX DRAWINGS HEAVY QUADRUPLE DASH HORIZONTAL",NULL},
/* 250A */ { "BOX DRAWINGS LIGHT QUADRUPLE DASH VERTICAL",NULL},
/* 250B */ { "BOX DRAWINGS HEAVY QUADRUPLE DASH VERTICAL",NULL},
/* 250C */ { "BOX DRAWINGS LIGHT DOWN AND RIGHT","	= Videotex Mosaic DG 16"},
/* 250D */ { "BOX DRAWINGS DOWN LIGHT AND RIGHT HEAVY",NULL},
/* 250E */ { "BOX DRAWINGS DOWN HEAVY AND RIGHT LIGHT",NULL},
/* 250F */ { "BOX DRAWINGS HEAVY DOWN AND RIGHT",NULL},
/* 2510 */ { "BOX DRAWINGS LIGHT DOWN AND LEFT","	= Videotex Mosaic DG 17"},
/* 2511 */ { "BOX DRAWINGS DOWN LIGHT AND LEFT HEAVY",NULL},
/* 2512 */ { "BOX DRAWINGS DOWN HEAVY AND LEFT LIGHT",NULL},
/* 2513 */ { "BOX DRAWINGS HEAVY DOWN AND LEFT",NULL},
/* 2514 */ { "BOX DRAWINGS LIGHT UP AND RIGHT","	= Videotex Mosaic DG 18"},
/* 2515 */ { "BOX DRAWINGS UP LIGHT AND RIGHT HEAVY",NULL},
/* 2516 */ { "BOX DRAWINGS UP HEAVY AND RIGHT LIGHT",NULL},
/* 2517 */ { "BOX DRAWINGS HEAVY UP AND RIGHT",NULL},
/* 2518 */ { "BOX DRAWINGS LIGHT UP AND LEFT","	= Videotex Mosaic DG 19"},
/* 2519 */ { "BOX DRAWINGS UP LIGHT AND LEFT HEAVY",NULL},
/* 251A */ { "BOX DRAWINGS UP HEAVY AND LEFT LIGHT",NULL},
/* 251B */ { "BOX DRAWINGS HEAVY UP AND LEFT",NULL},
/* 251C */ { "BOX DRAWINGS LIGHT VERTICAL AND RIGHT","	= Videotex Mosaic DG 20"},
/* 251D */ { "BOX DRAWINGS VERTICAL LIGHT AND RIGHT HEAVY","	= Videotex Mosaic DG 03"},
/* 251E */ { "BOX DRAWINGS UP HEAVY AND RIGHT DOWN LIGHT",NULL},
/* 251F */ { "BOX DRAWINGS DOWN HEAVY AND RIGHT UP LIGHT",NULL},
/* 2520 */ { "BOX DRAWINGS VERTICAL HEAVY AND RIGHT LIGHT",NULL},
/* 2521 */ { "BOX DRAWINGS DOWN LIGHT AND RIGHT UP HEAVY",NULL},
/* 2522 */ { "BOX DRAWINGS UP LIGHT AND RIGHT DOWN HEAVY",NULL},
/* 2523 */ { "BOX DRAWINGS HEAVY VERTICAL AND RIGHT",NULL},
/* 2524 */ { "BOX DRAWINGS LIGHT VERTICAL AND LEFT","	= Videotex Mosaic DG 21"},
/* 2525 */ { "BOX DRAWINGS VERTICAL LIGHT AND LEFT HEAVY","	= Videotex Mosaic DG 04"},
/* 2526 */ { "BOX DRAWINGS UP HEAVY AND LEFT DOWN LIGHT",NULL},
/* 2527 */ { "BOX DRAWINGS DOWN HEAVY AND LEFT UP LIGHT",NULL},
/* 2528 */ { "BOX DRAWINGS VERTICAL HEAVY AND LEFT LIGHT",NULL},
/* 2529 */ { "BOX DRAWINGS DOWN LIGHT AND LEFT UP HEAVY",NULL},
/* 252A */ { "BOX DRAWINGS UP LIGHT AND LEFT DOWN HEAVY",NULL},
/* 252B */ { "BOX DRAWINGS HEAVY VERTICAL AND LEFT",NULL},
/* 252C */ { "BOX DRAWINGS LIGHT DOWN AND HORIZONTAL","	= Videotex Mosaic DG 22"},
/* 252D */ { "BOX DRAWINGS LEFT HEAVY AND RIGHT DOWN LIGHT",NULL},
/* 252E */ { "BOX DRAWINGS RIGHT HEAVY AND LEFT DOWN LIGHT",NULL},
/* 252F */ { "BOX DRAWINGS DOWN LIGHT AND HORIZONTAL HEAVY","	= Videotex Mosaic DG 02"},
/* 2530 */ { "BOX DRAWINGS DOWN HEAVY AND HORIZONTAL LIGHT",NULL},
/* 2531 */ { "BOX DRAWINGS RIGHT LIGHT AND LEFT DOWN HEAVY",NULL},
/* 2532 */ { "BOX DRAWINGS LEFT LIGHT AND RIGHT DOWN HEAVY",NULL},
/* 2533 */ { "BOX DRAWINGS HEAVY DOWN AND HORIZONTAL",NULL},
/* 2534 */ { "BOX DRAWINGS LIGHT UP AND HORIZONTAL","	= Videotex Mosaic DG 23"},
/* 2535 */ { "BOX DRAWINGS LEFT HEAVY AND RIGHT UP LIGHT",NULL},
/* 2536 */ { "BOX DRAWINGS RIGHT HEAVY AND LEFT UP LIGHT",NULL},
/* 2537 */ { "BOX DRAWINGS UP LIGHT AND HORIZONTAL HEAVY","	= Videotex Mosaic DG 01"},
/* 2538 */ { "BOX DRAWINGS UP HEAVY AND HORIZONTAL LIGHT",NULL},
/* 2539 */ { "BOX DRAWINGS RIGHT LIGHT AND LEFT UP HEAVY",NULL},
/* 253A */ { "BOX DRAWINGS LEFT LIGHT AND RIGHT UP HEAVY",NULL},
/* 253B */ { "BOX DRAWINGS HEAVY UP AND HORIZONTAL",NULL},
/* 253C */ { "BOX DRAWINGS LIGHT VERTICAL AND HORIZONTAL","	= Videotex Mosaic DG 24"},
/* 253D */ { "BOX DRAWINGS LEFT HEAVY AND RIGHT VERTICAL LIGHT",NULL},
/* 253E */ { "BOX DRAWINGS RIGHT HEAVY AND LEFT VERTICAL LIGHT",NULL},
/* 253F */ { "BOX DRAWINGS VERTICAL LIGHT AND HORIZONTAL HEAVY","	= Videotex Mosaic DG 13"},
/* 2540 */ { "BOX DRAWINGS UP HEAVY AND DOWN HORIZONTAL LIGHT",NULL},
/* 2541 */ { "BOX DRAWINGS DOWN HEAVY AND UP HORIZONTAL LIGHT",NULL},
/* 2542 */ { "BOX DRAWINGS VERTICAL HEAVY AND HORIZONTAL LIGHT",NULL},
/* 2543 */ { "BOX DRAWINGS LEFT UP HEAVY AND RIGHT DOWN LIGHT",NULL},
/* 2544 */ { "BOX DRAWINGS RIGHT UP HEAVY AND LEFT DOWN LIGHT",NULL},
/* 2545 */ { "BOX DRAWINGS LEFT DOWN HEAVY AND RIGHT UP LIGHT",NULL},
/* 2546 */ { "BOX DRAWINGS RIGHT DOWN HEAVY AND LEFT UP LIGHT",NULL},
/* 2547 */ { "BOX DRAWINGS DOWN LIGHT AND UP HORIZONTAL HEAVY",NULL},
/* 2548 */ { "BOX DRAWINGS UP LIGHT AND DOWN HORIZONTAL HEAVY",NULL},
/* 2549 */ { "BOX DRAWINGS RIGHT LIGHT AND LEFT VERTICAL HEAVY",NULL},
/* 254A */ { "BOX DRAWINGS LEFT LIGHT AND RIGHT VERTICAL HEAVY",NULL},
/* 254B */ { "BOX DRAWINGS HEAVY VERTICAL AND HORIZONTAL",NULL},
/* 254C */ { "BOX DRAWINGS LIGHT DOUBLE DASH HORIZONTAL",NULL},
/* 254D */ { "BOX DRAWINGS HEAVY DOUBLE DASH HORIZONTAL",NULL},
/* 254E */ { "BOX DRAWINGS LIGHT DOUBLE DASH VERTICAL",NULL},
/* 254F */ { "BOX DRAWINGS HEAVY DOUBLE DASH VERTICAL",NULL},
/* 2550 */ { "BOX DRAWINGS DOUBLE HORIZONTAL",NULL},
/* 2551 */ { "BOX DRAWINGS DOUBLE VERTICAL",NULL},
/* 2552 */ { "BOX DRAWINGS DOWN SINGLE AND RIGHT DOUBLE",NULL},
/* 2553 */ { "BOX DRAWINGS DOWN DOUBLE AND RIGHT SINGLE",NULL},
/* 2554 */ { "BOX DRAWINGS DOUBLE DOWN AND RIGHT",NULL},
/* 2555 */ { "BOX DRAWINGS DOWN SINGLE AND LEFT DOUBLE",NULL},
/* 2556 */ { "BOX DRAWINGS DOWN DOUBLE AND LEFT SINGLE",NULL},
/* 2557 */ { "BOX DRAWINGS DOUBLE DOWN AND LEFT",NULL},
/* 2558 */ { "BOX DRAWINGS UP SINGLE AND RIGHT DOUBLE",NULL},
/* 2559 */ { "BOX DRAWINGS UP DOUBLE AND RIGHT SINGLE",NULL},
/* 255A */ { "BOX DRAWINGS DOUBLE UP AND RIGHT",NULL},
/* 255B */ { "BOX DRAWINGS UP SINGLE AND LEFT DOUBLE",NULL},
/* 255C */ { "BOX DRAWINGS UP DOUBLE AND LEFT SINGLE",NULL},
/* 255D */ { "BOX DRAWINGS DOUBLE UP AND LEFT",NULL},
/* 255E */ { "BOX DRAWINGS VERTICAL SINGLE AND RIGHT DOUBLE",NULL},
/* 255F */ { "BOX DRAWINGS VERTICAL DOUBLE AND RIGHT SINGLE",NULL},
/* 2560 */ { "BOX DRAWINGS DOUBLE VERTICAL AND RIGHT",NULL},
/* 2561 */ { "BOX DRAWINGS VERTICAL SINGLE AND LEFT DOUBLE",NULL},
/* 2562 */ { "BOX DRAWINGS VERTICAL DOUBLE AND LEFT SINGLE",NULL},
/* 2563 */ { "BOX DRAWINGS DOUBLE VERTICAL AND LEFT",NULL},
/* 2564 */ { "BOX DRAWINGS DOWN SINGLE AND HORIZONTAL DOUBLE",NULL},
/* 2565 */ { "BOX DRAWINGS DOWN DOUBLE AND HORIZONTAL SINGLE",NULL},
/* 2566 */ { "BOX DRAWINGS DOUBLE DOWN AND HORIZONTAL",NULL},
/* 2567 */ { "BOX DRAWINGS UP SINGLE AND HORIZONTAL DOUBLE",NULL},
/* 2568 */ { "BOX DRAWINGS UP DOUBLE AND HORIZONTAL SINGLE",NULL},
/* 2569 */ { "BOX DRAWINGS DOUBLE UP AND HORIZONTAL",NULL},
/* 256A */ { "BOX DRAWINGS VERTICAL SINGLE AND HORIZONTAL DOUBLE",NULL},
/* 256B */ { "BOX DRAWINGS VERTICAL DOUBLE AND HORIZONTAL SINGLE",NULL},
/* 256C */ { "BOX DRAWINGS DOUBLE VERTICAL AND HORIZONTAL",NULL},
/* 256D */ { "BOX DRAWINGS LIGHT ARC DOWN AND RIGHT",NULL},
/* 256E */ { "BOX DRAWINGS LIGHT ARC DOWN AND LEFT",NULL},
/* 256F */ { "BOX DRAWINGS LIGHT ARC UP AND LEFT",NULL},
/* 2570 */ { "BOX DRAWINGS LIGHT ARC UP AND RIGHT",NULL},
/* 2571 */ { "BOX DRAWINGS LIGHT DIAGONAL UPPER RIGHT TO LOWER LEFT","	* not intended for mathematical symbol \\diagup\n"
	"	x (solidus - 002F)\n"
	"	x (fraction slash - 2044)\n"
	"	x (division slash - 2215)"},
/* 2572 */ { "BOX DRAWINGS LIGHT DIAGONAL UPPER LEFT TO LOWER RIGHT","	* not intended for mathematical symbol \\diagdown\n"
	"	x (reverse solidus - 005C)\n"
	"	x (set minus - 2216)\n"
	"	x (reverse solidus operator - 29F5)"},
/* 2573 */ { "BOX DRAWINGS LIGHT DIAGONAL CROSS","	x (saltire - 2613)\n"
	"	x (multiplication x - 2715)"},
/* 2574 */ { "BOX DRAWINGS LIGHT LEFT",NULL},
/* 2575 */ { "BOX DRAWINGS LIGHT UP",NULL},
/* 2576 */ { "BOX DRAWINGS LIGHT RIGHT",NULL},
/* 2577 */ { "BOX DRAWINGS LIGHT DOWN",NULL},
/* 2578 */ { "BOX DRAWINGS HEAVY LEFT",NULL},
/* 2579 */ { "BOX DRAWINGS HEAVY UP",NULL},
/* 257A */ { "BOX DRAWINGS HEAVY RIGHT",NULL},
/* 257B */ { "BOX DRAWINGS HEAVY DOWN",NULL},
/* 257C */ { "BOX DRAWINGS LIGHT LEFT AND HEAVY RIGHT",NULL},
/* 257D */ { "BOX DRAWINGS LIGHT UP AND HEAVY DOWN",NULL},
/* 257E */ { "BOX DRAWINGS HEAVY LEFT AND LIGHT RIGHT",NULL},
/* 257F */ { "BOX DRAWINGS HEAVY UP AND LIGHT DOWN",NULL},
/* 2580 */ { "UPPER HALF BLOCK",NULL},
/* 2581 */ { "LOWER ONE EIGHTH BLOCK",NULL},
/* 2582 */ { "LOWER ONE QUARTER BLOCK",NULL},
/* 2583 */ { "LOWER THREE EIGHTHS BLOCK",NULL},
/* 2584 */ { "LOWER HALF BLOCK",NULL},
/* 2585 */ { "LOWER FIVE EIGHTHS BLOCK",NULL},
/* 2586 */ { "LOWER THREE QUARTERS BLOCK",NULL},
/* 2587 */ { "LOWER SEVEN EIGHTHS BLOCK",NULL},
/* 2588 */ { "FULL BLOCK","	= solid\n"
	"	x (black square - 25A0)"},
/* 2589 */ { "LEFT SEVEN EIGHTHS BLOCK",NULL},
/* 258A */ { "LEFT THREE QUARTERS BLOCK",NULL},
/* 258B */ { "LEFT FIVE EIGHTHS BLOCK",NULL},
/* 258C */ { "LEFT HALF BLOCK",NULL},
/* 258D */ { "LEFT THREE EIGHTHS BLOCK",NULL},
/* 258E */ { "LEFT ONE QUARTER BLOCK",NULL},
/* 258F */ { "LEFT ONE EIGHTH BLOCK",NULL},
/* 2590 */ { "RIGHT HALF BLOCK",NULL},
/* 2591 */ { "LIGHT SHADE","	* 25%"},
/* 2592 */ { "MEDIUM SHADE","	= speckles fill, dotted fill\n"
	"	* 50%\n"
	"	* used in mapping to cp949\n"
	"	x (inverse medium shade - 1FB90)"},
/* 2593 */ { "DARK SHADE","	* 75%"},
/* 2594 */ { "UPPER ONE EIGHTH BLOCK",NULL},
/* 2595 */ { "RIGHT ONE EIGHTH BLOCK",NULL},
/* 2596 */ { "QUADRANT LOWER LEFT",NULL},
/* 2597 */ { "QUADRANT LOWER RIGHT",NULL},
/* 2598 */ { "QUADRANT UPPER LEFT",NULL},
/* 2599 */ { "QUADRANT UPPER LEFT AND LOWER LEFT AND LOWER RIGHT",NULL},
/* 259A */ { "QUADRANT UPPER LEFT AND LOWER RIGHT","	x (reverse checker board - 1F67F)\n"
	"	x (checker board fill - 1FB95)"},
/* 259B */ { "QUADRANT UPPER LEFT AND UPPER RIGHT AND LOWER LEFT",NULL},
/* 259C */ { "QUADRANT UPPER LEFT AND UPPER RIGHT AND LOWER RIGHT",NULL},
/* 259D */ { "QUADRANT UPPER RIGHT",NULL},
/* 259E */ { "QUADRANT UPPER RIGHT AND LOWER LEFT","	x (checker board - 1F67E)\n"
	"	x (inverse checker board fill - 1FB96)"},
/* 259F */ { "QUADRANT UPPER RIGHT AND LOWER LEFT AND LOWER RIGHT",NULL},
/* 25A0 */ { "BLACK SQUARE","	= moding mark (in ideographic text)\n"
	"	x (full block - 2588)\n"
	"	x (black large square - 2B1B)"},
/* 25A1 */ { "WHITE SQUARE","	= quadrature\n"
	"	= alchemical symbol for salt\n"
	"	* used in astrological contexts for aspect square\n"
	"	* may be used to represent a missing ideograph\n"
	"	x (combining enclosing square - 20DE)\n"
	"	x (white medium square - 25FB)\n"
	"	x (ballot box - 2610)\n"
	"	x (white large square - 2B1C)\n"
	"	x (geta mark - 3013)"},
/* 25A2 */ { "WHITE SQUARE WITH ROUNDED CORNERS",NULL},
/* 25A3 */ { "WHITE SQUARE CONTAINING BLACK SMALL SQUARE",NULL},
/* 25A4 */ { "SQUARE WITH HORIZONTAL FILL",NULL},
/* 25A5 */ { "SQUARE WITH VERTICAL FILL",NULL},
/* 25A6 */ { "SQUARE WITH ORTHOGONAL CROSSHATCH FILL",NULL},
/* 25A7 */ { "SQUARE WITH UPPER LEFT TO LOWER RIGHT FILL","	x (upper left to lower right fill - 1FB98)"},
/* 25A8 */ { "SQUARE WITH UPPER RIGHT TO LOWER LEFT FILL","	x (upper right to lower left fill - 1FB99)"},
/* 25A9 */ { "SQUARE WITH DIAGONAL CROSSHATCH FILL",NULL},
/* 25AA */ { "BLACK SMALL SQUARE","	= square bullet\n"
	"	x (black very small square - 2B1D)"},
/* 25AB */ { "WHITE SMALL SQUARE","	x (white very small square - 2B1E)"},
/* 25AC */ { "BLACK RECTANGLE",NULL},
/* 25AD */ { "WHITE RECTANGLE",NULL},
/* 25AE */ { "BLACK VERTICAL RECTANGLE","	= histogram marker\n"
	"	x (end of proof - 220E)"},
/* 25AF */ { "WHITE VERTICAL RECTANGLE","	x (apl functional symbol squish quad - 2337)\n"
	"	x (apl functional symbol quad - 2395)"},
/* 25B0 */ { "BLACK PARALLELOGRAM",NULL},
/* 25B1 */ { "WHITE PARALLELOGRAM",NULL},
/* 25B2 */ { "BLACK UP-POINTING TRIANGLE","	x (black medium up-pointing triangle - 23F6)\n"
	"	x (black medium up-pointing triangle centred - 2BC5)\n"
	"	x (up-pointing red triangle - 1F53A)"},
/* 25B3 */ { "WHITE UP-POINTING TRIANGLE","	= trine\n"
	"	x (combining enclosing upward pointing triangle - 20E4)\n"
	"	x (increment - 2206)\n"
	"	x (triangle with rounded corners - 1F6C6)\n"
	"	x (alchemical symbol for fire - 1F702)"},
/* 25B4 */ { "BLACK UP-POINTING SMALL TRIANGLE","	x (up-pointing small red triangle - 1F53C)"},
/* 25B5 */ { "WHITE UP-POINTING SMALL TRIANGLE",NULL},
/* 25B6 */ { "BLACK RIGHT-POINTING TRIANGLE","	x (black medium right-pointing triangle - 23F5)\n"
	"	x (black medium right-pointing triangle centred - 2BC8)"},
/* 25B7 */ { "WHITE RIGHT-POINTING TRIANGLE","	= z notation range restriction"},
/* 25B8 */ { "BLACK RIGHT-POINTING SMALL TRIANGLE","	x (triangular bullet - 2023)"},
/* 25B9 */ { "WHITE RIGHT-POINTING SMALL TRIANGLE",NULL},
/* 25BA */ { "BLACK RIGHT-POINTING POINTER",NULL},
/* 25BB */ { "WHITE RIGHT-POINTING POINTER","	= forward arrow indicator\n"
	"	x (contains as normal subgroup - 22B3)"},
/* 25BC */ { "BLACK DOWN-POINTING TRIANGLE","	x (black medium down-pointing triangle - 23F7)\n"
	"	x (black medium down-pointing triangle centred - 2BC6)\n"
	"	x (down-pointing red triangle - 1F53B)"},
/* 25BD */ { "WHITE DOWN-POINTING TRIANGLE","	= Hamilton operator\n"
	"	x (nabla - 2207)\n"
	"	x (heavy white down-pointing triangle - 26DB)\n"
	"	x (alchemical symbol for water - 1F704)"},
/* 25BE */ { "BLACK DOWN-POINTING SMALL TRIANGLE","	x (down-pointing small red triangle - 1F53D)"},
/* 25BF */ { "WHITE DOWN-POINTING SMALL TRIANGLE",NULL},
/* 25C0 */ { "BLACK LEFT-POINTING TRIANGLE","	x (black medium left-pointing triangle - 23F4)\n"
	"	x (black medium left-pointing triangle centred - 2BC7)"},
/* 25C1 */ { "WHITE LEFT-POINTING TRIANGLE","	= z notation domain restriction"},
/* 25C2 */ { "BLACK LEFT-POINTING SMALL TRIANGLE",NULL},
/* 25C3 */ { "WHITE LEFT-POINTING SMALL TRIANGLE",NULL},
/* 25C4 */ { "BLACK LEFT-POINTING POINTER",NULL},
/* 25C5 */ { "WHITE LEFT-POINTING POINTER","	= backward arrow indicator\n"
	"	x (normal subgroup of - 22B2)"},
/* 25C6 */ { "BLACK DIAMOND","	x (black diamond suit - 2666)\n"
	"	x (black medium diamond - 2B25)\n"
	"	x (large orange diamond - 1F536)"},
/* 25C7 */ { "WHITE DIAMOND","	x (combining enclosing diamond - 20DF)\n"
	"	x (diamond operator - 22C4)\n"
	"	x (white diamond suit - 2662)\n"
	"	x (white concave-sided diamond - 27E1)\n"
	"	x (white medium diamond - 2B26)\n"
	"	x (alchemical symbol for soap - 1F754)\n"
	"	x (box drawings light diagonal diamond - 1FBAE)"},
/* 25C8 */ { "WHITE DIAMOND CONTAINING BLACK SMALL DIAMOND",NULL},
/* 25C9 */ { "FISHEYE","	= tainome (Japanese, a kind of bullet)"},
/* 25CA */ { "LOZENGE","	x (white diamond suit - 2662)"},
/* 25CB */ { "WHITE CIRCLE","	x (combining enclosing circle - 20DD)\n"
	"	x (large circle - 25EF)\n"
	"	x (heavy circle - 2B58)\n"
	"	x (ideographic number zero - 3007)\n"
	"	x (full moon symbol - 1F315)"},
/* 25CC */ { "DOTTED CIRCLE","	* note that the reference glyph for this character is intentionally larger than the dotted circle glyph used to indicate combining characters in this standard; see, for example, 0300"},
/* 25CD */ { "CIRCLE WITH VERTICAL FILL",NULL},
/* 25CE */ { "BULLSEYE","	x (circled ring operator - 229A)\n"
	"	x (heavy circle with circle inside - 2B57)\n"
	"	x (direct hit - 1F3AF)\n"
	"	x (round target - 1F78B)"},
/* 25CF */ { "BLACK CIRCLE","	x (black large circle - 2B24)\n"
	"	x (new moon symbol - 1F311)\n"
	"	x (large red circle - 1F534)"},
/* 25D0 */ { "CIRCLE WITH LEFT HALF BLACK","	x (first quarter moon symbol - 1F313)"},
/* 25D1 */ { "CIRCLE WITH RIGHT HALF BLACK","	x (last quarter moon symbol - 1F317)"},
/* 25D2 */ { "CIRCLE WITH LOWER HALF BLACK",NULL},
/* 25D3 */ { "CIRCLE WITH UPPER HALF BLACK",NULL},
/* 25D4 */ { "CIRCLE WITH UPPER RIGHT QUADRANT BLACK",NULL},
/* 25D5 */ { "CIRCLE WITH ALL BUT UPPER LEFT QUADRANT BLACK",NULL},
/* 25D6 */ { "LEFT HALF BLACK CIRCLE",NULL},
/* 25D7 */ { "RIGHT HALF BLACK CIRCLE",NULL},
/* 25D8 */ { "INVERSE BULLET","	x (bullet - 2022)\n"
	"	x (white bullet - 25E6)"},
/* 25D9 */ { "INVERSE WHITE CIRCLE",NULL},
/* 25DA */ { "UPPER HALF INVERSE WHITE CIRCLE",NULL},
/* 25DB */ { "LOWER HALF INVERSE WHITE CIRCLE",NULL},
/* 25DC */ { "UPPER LEFT QUADRANT CIRCULAR ARC",NULL},
/* 25DD */ { "UPPER RIGHT QUADRANT CIRCULAR ARC",NULL},
/* 25DE */ { "LOWER RIGHT QUADRANT CIRCULAR ARC",NULL},
/* 25DF */ { "LOWER LEFT QUADRANT CIRCULAR ARC",NULL},
/* 25E0 */ { "UPPER HALF CIRCLE","	x (arc - 2312)"},
/* 25E1 */ { "LOWER HALF CIRCLE",NULL},
/* 25E2 */ { "BLACK LOWER RIGHT TRIANGLE","	x (lower right triangular medium shade - 1FB9E)"},
/* 25E3 */ { "BLACK LOWER LEFT TRIANGLE","	x (lower left triangular medium shade - 1FB9F)"},
/* 25E4 */ { "BLACK UPPER LEFT TRIANGLE","	x (upper left triangular medium shade - 1FB9C)"},
/* 25E5 */ { "BLACK UPPER RIGHT TRIANGLE","	x (upper right triangular medium shade - 1FB9D)"},
/* 25E6 */ { "WHITE BULLET","	x (bullet - 2022)\n"
	"	x (ring operator - 2218)\n"
	"	x (inverse bullet - 25D8)"},
/* 25E7 */ { "SQUARE WITH LEFT HALF BLACK","	x (square with top half black - 2B12)"},
/* 25E8 */ { "SQUARE WITH RIGHT HALF BLACK",NULL},
/* 25E9 */ { "SQUARE WITH UPPER LEFT DIAGONAL HALF BLACK",NULL},
/* 25EA */ { "SQUARE WITH LOWER RIGHT DIAGONAL HALF BLACK",NULL},
/* 25EB */ { "WHITE SQUARE WITH VERTICAL BISECTING LINE",NULL},
/* 25EC */ { "WHITE UP-POINTING TRIANGLE WITH DOT",NULL},
/* 25ED */ { "UP-POINTING TRIANGLE WITH LEFT HALF BLACK",NULL},
/* 25EE */ { "UP-POINTING TRIANGLE WITH RIGHT HALF BLACK",NULL},
/* 25EF */ { "LARGE CIRCLE","	x (combining enclosing circle - 20DD)\n"
	"	x (white circle - 25CB)\n"
	"	x (black large circle - 2B24)\n"
	"	x (heavy large circle - 2B55)\n"
	"	x (ideographic number zero - 3007)"},
/* 25F0 */ { "WHITE SQUARE WITH UPPER LEFT QUADRANT",NULL},
/* 25F1 */ { "WHITE SQUARE WITH LOWER LEFT QUADRANT",NULL},
/* 25F2 */ { "WHITE SQUARE WITH LOWER RIGHT QUADRANT",NULL},
/* 25F3 */ { "WHITE SQUARE WITH UPPER RIGHT QUADRANT",NULL},
/* 25F4 */ { "WHITE CIRCLE WITH UPPER LEFT QUADRANT",NULL},
/* 25F5 */ { "WHITE CIRCLE WITH LOWER LEFT QUADRANT",NULL},
/* 25F6 */ { "WHITE CIRCLE WITH LOWER RIGHT QUADRANT",NULL},
/* 25F7 */ { "WHITE CIRCLE WITH UPPER RIGHT QUADRANT",NULL},
/* 25F8 */ { "UPPER LEFT TRIANGLE",NULL},
/* 25F9 */ { "UPPER RIGHT TRIANGLE",NULL},
/* 25FA */ { "LOWER LEFT TRIANGLE","	x (right triangle - 22BF)"},
/* 25FB */ { "WHITE MEDIUM SQUARE","	= always (modal operator)\n"
	"	x (white square - 25A1)\n"
	"	x (white square with leftwards tick - 27E4)"},
/* 25FC */ { "BLACK MEDIUM SQUARE","	x (black square for stop - 23F9)\n"
	"	x (black square - 25A0)"},
/* 25FD */ { "WHITE MEDIUM SMALL SQUARE","	x (white small square - 25AB)"},
/* 25FE */ { "BLACK MEDIUM SMALL SQUARE","	x (black small square - 25AA)"},
/* 25FF */ { "LOWER RIGHT TRIANGLE","	x (right triangle - 22BF)"}
};

UN_DLL_LOCAL
static const struct unicode_nameannot una_00_26[] = {
/* 2600 */ { "BLACK SUN WITH RAYS","	= clear weather\n"
	"	x (sun - 2609)\n"
	"	x (high brightness symbol - 1F506)"},
/* 2601 */ { "CLOUD","	= cloudy weather"},
/* 2602 */ { "UMBRELLA","	= rainy weather\n"
	"	x (closed umbrella - 1F302)\n"
	"	x (cloud with rain - 1F327)"},
/* 2603 */ { "SNOWMAN","	= snowy weather"},
/* 2604 */ { "COMET",NULL},
/* 2605 */ { "BLACK STAR","	x (star operator - 22C6)\n"
	"	x (black small star - 2B51)\n"
	"	x (left half black star - 2BE8)\n"
	"	x (light five pointed black star - 1F7C9)"},
/* 2606 */ { "WHITE STAR","	x (stress outlined white star - 2729)\n"
	"	x (white medium star - 2B50)\n"
	"	x (glowing star - 1F31F)"},
/* 2607 */ { "LIGHTNING",NULL},
/* 2608 */ { "THUNDERSTORM",NULL},
/* 2609 */ { "SUN","	= alchemical symbol for gold\n"
	"	x (circled dot operator - 2299)\n"
	"	x (black sun with rays - 2600)\n"
	"	x (white sun with rays - 263C)\n"
	"	x (sun with face - 1F31E)"},
/* 260A */ { "ASCENDING NODE","	= alchemical symbol for sublimation\n"
	"	x (alchemical symbol for sublimation - 1F75E)"},
/* 260B */ { "DESCENDING NODE","	= alchemical symbol for purify\n"
	"	x (alchemical symbol for purify - 1F763)"},
/* 260C */ { "CONJUNCTION","	= alchemical symbol for day\n"
	"	x (occultation - 1F775)"},
/* 260D */ { "OPPOSITION","	x (lunar eclipse - 1F776)"},
/* 260E */ { "BLACK TELEPHONE","	x (telephone sign - 2121)\n"
	"	x (telephone location sign - 2706)\n"
	"	x (telephone receiver - 1F4DE)\n"
	"	x (black touchtone telephone - 1F57F)"},
/* 260F */ { "WHITE TELEPHONE","	x (white touchtone telephone - 1F57E)"},
/* 2610 */ { "BALLOT BOX","	x (white square - 25A1)"},
/* 2611 */ { "BALLOT BOX WITH CHECK","	x (ballot box with bold check - 1F5F9)"},
/* 2612 */ { "BALLOT BOX WITH X","	x (squared times - 22A0)\n"
	"	x (x in a rectangle box - 2327)\n"
	"	x (ballot box with light x - 2BBD)\n"
	"	x (ballot box with ballot - 1F5F3)\n"
	"	x (ballot box with script x - 1F5F5)"},
/* 2613 */ { "SALTIRE","	= St. Andrew's Cross\n"
	"	x (ballot x - 2717)\n"
	"	x (light saltire - 1F7A9)"},
/* 2614 */ { "UMBRELLA WITH RAIN DROPS","	= showery weather"},
/* 2615 */ { "HOT BEVERAGE","	= tea or coffee, depending on locale\n"
	"	* can be used to indicate a wait\n"
	"	x (watch - 231A)\n"
	"	x (hourglass - 231B)\n"
	"	x (teacup without handle - 1F375)"},
/* 2616 */ { "WHITE SHOGI PIECE",NULL},
/* 2617 */ { "BLACK SHOGI PIECE",NULL},
/* 2618 */ { "SHAMROCK","	x (four leaf clover - 1F340)"},
/* 2619 */ { "REVERSED ROTATED FLORAL HEART BULLET","	* a binding signature mark\n"
	"	x (rotated floral heart bullet - 2767)\n"
	"	x (south west pointing vine leaf - 1F659)"},
/* 261A */ { "BLACK LEFT POINTING INDEX",NULL},
/* 261B */ { "BLACK RIGHT POINTING INDEX",NULL},
/* 261C */ { "WHITE LEFT POINTING INDEX","	x (white left pointing backhand index - 1F448)\n"
	"	x (sideways white left pointing index - 1F598)"},
/* 261D */ { "WHITE UP POINTING INDEX","	x (sideways white up pointing index - 1F59E)"},
/* 261E */ { "WHITE RIGHT POINTING INDEX","	= fist (typographic term)\n"
	"	x (sideways white right pointing index - 1F599)\n"
	"	x (left third white right pointing index - 1FBC1)"},
/* 261F */ { "WHITE DOWN POINTING INDEX","	x (white down pointing left hand index - 1F597)\n"
	"	x (sideways white down pointing index - 1F59F)"},
/* 2620 */ { "SKULL AND CROSSBONES","	= poison\n"
	"	x (black skull and crossbones - 1F571)"},
/* 2621 */ { "CAUTION SIGN",NULL},
/* 2622 */ { "RADIOACTIVE SIGN",NULL},
/* 2623 */ { "BIOHAZARD SIGN",NULL},
/* 2624 */ { "CADUCEUS","	x (staff of aesculapius - 2695)\n"
	"	x (alchemical symbol for caduceus - 1F750)"},
/* 2625 */ { "ANKH",NULL},
/* 2626 */ { "ORTHODOX CROSS",NULL},
/* 2627 */ { "CHI RHO","	= Constantine's cross, Christogram\n"
	"	x (coptic symbol khi ro - 2CE9)"},
/* 2628 */ { "CROSS OF LORRAINE",NULL},
/* 2629 */ { "CROSS OF JERUSALEM","	= simple cross potent\n"
	"	* contrasts with the actual cross of Jerusalem, which adds a small crosslet at each corner\n"
	"	x (alchemical symbol for vinegar - 1F70A)"},
/* 262A */ { "STAR AND CRESCENT",NULL},
/* 262B */ { "FARSI SYMBOL","	= symbol of iran (1.0)"},
/* 262C */ { "ADI SHAKTI","	= khanda\n"
	"	* Sikh religious symbol\n"
	"	x (khanda - 1FAAF)"},
/* 262D */ { "HAMMER AND SICKLE",NULL},
/* 262E */ { "PEACE SYMBOL",NULL},
/* 262F */ { "YIN YANG","	x (tibetan symbol nor bu nyis -khyil - 0FCA)"},
/* 2630 */ { "TRIGRAM FOR HEAVEN","	= qian2"},
/* 2631 */ { "TRIGRAM FOR LAKE","	= dui4"},
/* 2632 */ { "TRIGRAM FOR FIRE","	= li2"},
/* 2633 */ { "TRIGRAM FOR THUNDER","	= zhen4"},
/* 2634 */ { "TRIGRAM FOR WIND","	= xun4"},
/* 2635 */ { "TRIGRAM FOR WATER","	= kan3"},
/* 2636 */ { "TRIGRAM FOR MOUNTAIN","	= gen4"},
/* 2637 */ { "TRIGRAM FOR EARTH","	= kun1"},
/* 2638 */ { "WHEEL OF DHARMA","	x (helm symbol - 2388)"},
/* 2639 */ { "WHITE FROWNING FACE","	x (slightly frowning face - 1F641)"},
/* 263A */ { "WHITE SMILING FACE","	x (slightly smiling face - 1F642)\n"
	"	= have a nice day!"},
/* 263B */ { "BLACK SMILING FACE",NULL},
/* 263C */ { "WHITE SUN WITH RAYS","	= compass\n"
	"	x (sun - 2609)\n"
	"	x (sun with face - 1F31E)\n"
	"	x (white sun - 1F323)\n"
	"	x (high brightness symbol - 1F506)"},
/* 263D */ { "FIRST QUARTER MOON","	= alchemical symbol for silver\n"
	"	x (first quarter moon symbol - 1F313)"},
/* 263E */ { "LAST QUARTER MOON","	= alchemical symbol for silver\n"
	"	x (power sleep symbol - 23FE)\n"
	"	x (last quarter moon symbol - 1F317)\n"
	"	x (crescent moon - 1F319)"},
/* 263F */ { "MERCURY","	= alchemical symbol for quicksilver\n"
	"	= intersexuality"},
/* 2640 */ { "FEMALE SIGN","	= Venus\n"
	"	= alchemical symbol for copper\n"
	"	x (woman - 1F469)\n"
	"	x (womens symbol - 1F6BA)"},
/* 2641 */ { "EARTH","	= alchemical symbol for antimony\n"
	"	x (alchemical symbol for verdigris - 1F728)"},
/* 2642 */ { "MALE SIGN","	= Mars\n"
	"	= alchemical symbol for iron\n"
	"	x (passed symbol - 2BFD)\n"
	"	x (man - 1F468)\n"
	"	x (mens symbol - 1F6B9)"},
/* 2643 */ { "JUPITER","	= alchemical symbol for tin"},
/* 2644 */ { "SATURN","	= alchemical symbol for lead"},
/* 2645 */ { "URANUS","	x (astronomical symbol for uranus - 26E2)"},
/* 2646 */ { "NEPTUNE","	= alchemical symbol for bismuth/tinglass\n"
	"	x (neptune form two - 2BC9)"},
/* 2647 */ { "PLUTO","	x (pluto form two - 2BD3)"},
/* 2648 */ { "ARIES","	x (ram - 1F40F)"},
/* 2649 */ { "TAURUS",NULL},
/* 264A */ { "GEMINI",NULL},
/* 264B */ { "CANCER","	x (crab - 1F980)"},
/* 264C */ { "LEO","	x (lion face - 1F981)"},
/* 264D */ { "VIRGO","	= minim (alternate glyph)"},
/* 264E */ { "LIBRA","	x (scales - 2696)\n"
	"	x (alchemical symbol for sublimation - 1F75E)"},
/* 264F */ { "SCORPIUS","	= Scorpio\n"
	"	= minim, drop\n"
	"	x (scorpion - 1F982)"},
/* 2650 */ { "SAGITTARIUS","	x (bow and arrow - 1F3F9)"},
/* 2651 */ { "CAPRICORN",NULL},
/* 2652 */ { "AQUARIUS","	x (amphora - 1F3FA)"},
/* 2653 */ { "PISCES",NULL},
/* 2654 */ { "WHITE CHESS KING",NULL},
/* 2655 */ { "WHITE CHESS QUEEN",NULL},
/* 2656 */ { "WHITE CHESS ROOK",NULL},
/* 2657 */ { "WHITE CHESS BISHOP",NULL},
/* 2658 */ { "WHITE CHESS KNIGHT",NULL},
/* 2659 */ { "WHITE CHESS PAWN",NULL},
/* 265A */ { "BLACK CHESS KING",NULL},
/* 265B */ { "BLACK CHESS QUEEN",NULL},
/* 265C */ { "BLACK CHESS ROOK",NULL},
/* 265D */ { "BLACK CHESS BISHOP",NULL},
/* 265E */ { "BLACK CHESS KNIGHT",NULL},
/* 265F */ { "BLACK CHESS PAWN",NULL},
/* 2660 */ { "BLACK SPADE SUIT",NULL},
/* 2661 */ { "WHITE HEART SUIT",NULL},
/* 2662 */ { "WHITE DIAMOND SUIT","	x (white diamond - 25C7)\n"
	"	x (lozenge - 25CA)"},
/* 2663 */ { "BLACK CLUB SUIT","	x (shamrock - 2618)"},
/* 2664 */ { "WHITE SPADE SUIT",NULL},
/* 2665 */ { "BLACK HEART SUIT","	= valentine\n"
	"	x (heavy black heart - 2764)\n"
	"	x (heart with tip on the left - 1F394)\n"
	"	x (blue heart - 1F499)\n"
	"	x (black heart - 1F5A4)"},
/* 2666 */ { "BLACK DIAMOND SUIT","	x (black diamond - 25C6)"},
/* 2667 */ { "WHITE CLUB SUIT",NULL},
/* 2668 */ { "HOT SPRINGS",NULL},
/* 2669 */ { "QUARTER NOTE","	= crotchet\n"
	"	x (musical symbol quarter note - 1D15F)"},
/* 266A */ { "EIGHTH NOTE","	= quaver\n"
	"	x (musical symbol eighth note - 1D160)\n"
	"	x (musical note - 1F3B5)"},
/* 266B */ { "BEAMED EIGHTH NOTES","	= beamed quavers\n"
	"	x (beamed descending musical notes - 1F39D)\n"
	"	x (multiple musical notes - 1F3B6)"},
/* 266C */ { "BEAMED SIXTEENTH NOTES","	= beamed semiquavers"},
/* 266D */ { "MUSIC FLAT SIGN",NULL},
/* 266E */ { "MUSIC NATURAL SIGN",NULL},
/* 266F */ { "MUSIC SHARP SIGN","	= z notation infix bag count\n"
	"	x (number sign - 0023)"},
/* 2670 */ { "WEST SYRIAC CROSS",NULL},
/* 2671 */ { "EAST SYRIAC CROSS",NULL},
/* 2672 */ { "UNIVERSAL RECYCLING SYMBOL","	* used as generic symbol for recycling or to indicate that material is recyclable"},
/* 2673 */ { "RECYCLING SYMBOL FOR TYPE-1 PLASTICS","	* polyethylene terephthalate"},
/* 2674 */ { "RECYCLING SYMBOL FOR TYPE-2 PLASTICS","	* high density polyethylene"},
/* 2675 */ { "RECYCLING SYMBOL FOR TYPE-3 PLASTICS","	* vinyl, polyvinyl chloride"},
/* 2676 */ { "RECYCLING SYMBOL FOR TYPE-4 PLASTICS","	* low density polyethylene"},
/* 2677 */ { "RECYCLING SYMBOL FOR TYPE-5 PLASTICS","	* polypropylene"},
/* 2678 */ { "RECYCLING SYMBOL FOR TYPE-6 PLASTICS","	* polystyrene"},
/* 2679 */ { "RECYCLING SYMBOL FOR TYPE-7 PLASTICS","	* other plastics"},
/* 267A */ { "RECYCLING SYMBOL FOR GENERIC MATERIALS","	* used together with other text and labels to indicate the type of material to be recycled"},
/* 267B */ { "BLACK UNIVERSAL RECYCLING SYMBOL","	x (clockwise rightwards and leftwards open circle arrows - 1F501)"},
/* 267C */ { "RECYCLED PAPER SYMBOL","	* used to indicate 100% recycled paper content"},
/* 267D */ { "PARTIALLY-RECYCLED PAPER SYMBOL","	* percentage of recycled paper content indicated in overlay or next to this symbol"},
/* 267E */ { "PERMANENT PAPER SIGN","	= non-acid paper"},
/* 267F */ { "WHEELCHAIR SYMBOL","	= accessible place"},
/* 2680 */ { "DIE FACE-1","	x (game die - 1F3B2)"},
/* 2681 */ { "DIE FACE-2",NULL},
/* 2682 */ { "DIE FACE-3",NULL},
/* 2683 */ { "DIE FACE-4",NULL},
/* 2684 */ { "DIE FACE-5",NULL},
/* 2685 */ { "DIE FACE-6",NULL},
/* 2686 */ { "WHITE CIRCLE WITH DOT RIGHT",NULL},
/* 2687 */ { "WHITE CIRCLE WITH TWO DOTS","	x (face without mouth - 1F636)"},
/* 2688 */ { "BLACK CIRCLE WITH WHITE DOT RIGHT",NULL},
/* 2689 */ { "BLACK CIRCLE WITH TWO WHITE DOTS",NULL},
/* 268A */ { "MONOGRAM FOR YANG",NULL},
/* 268B */ { "MONOGRAM FOR YIN",NULL},
/* 268C */ { "DIGRAM FOR GREATER YANG",NULL},
/* 268D */ { "DIGRAM FOR LESSER YIN",NULL},
/* 268E */ { "DIGRAM FOR LESSER YANG",NULL},
/* 268F */ { "DIGRAM FOR GREATER YIN",NULL},
/* 2690 */ { "WHITE FLAG","	x (waving white flag - 1F3F3)"},
/* 2691 */ { "BLACK FLAG",NULL},
/* 2692 */ { "HAMMER AND PICK","	= mining, working day (in timetables)\n"
	"	x (pick - 26CF)\n"
	"	x (hammer and wrench - 1F6E0)"},
/* 2693 */ { "ANCHOR","	= nautical term, harbor (on maps)"},
/* 2694 */ { "CROSSED SWORDS","	= military term, battleground (on maps), killed in action\n"
	"	x (fencer - 1F93A)"},
/* 2695 */ { "STAFF OF AESCULAPIUS","	= medical term\n"
	"	* both inclined or upright renderings of this symbol are common\n"
	"	x (caduceus - 2624)\n"
	"	x (bowl of hygieia - 1F54F)"},
/* 2696 */ { "SCALES","	= legal term, jurisprudence\n"
	"	x (libra - 264E)"},
/* 2697 */ { "ALEMBIC","	= chemical term, chemistry\n"
	"	x (alchemical symbol for retort - 1F76D)"},
/* 2698 */ { "FLOWER","	= botanical term\n"
	"	x (flower punctuation mark - 2055)\n"
	"	x (white florette - 2740)\n"
	"	x (tulip - 1F337)"},
/* 2699 */ { "GEAR","	= technology, tools"},
/* 269A */ { "STAFF OF HERMES","	* signifies a commercial term or commerce\n"
	"	* glyph shows a heraldic staff with a winged wheel"},
/* 269B */ { "ATOM SYMBOL","	= nuclear installation (on maps)"},
/* 269C */ { "FLEUR-DE-LIS",NULL},
/* 269D */ { "OUTLINED WHITE STAR","	* alternative form of the interlaced pentagram (a symbol of Morocco)\n"
	"	x (right-handed interlaced pentagram - 26E5)"},
/* 269E */ { "THREE LINES CONVERGING RIGHT","	= someone speaking\n"
	"	x (three rays left - 1F5E6)"},
/* 269F */ { "THREE LINES CONVERGING LEFT","	= background speaking\n"
	"	x (three rays right - 1F5E7)"},
/* 26A0 */ { "WARNING SIGN","	x (octagonal sign - 1F6D1)"},
/* 26A1 */ { "HIGH VOLTAGE SIGN","	= thunder\n"
	"	= lightning symbol\n"
	"	x (lightning mood - 1F5F2)"},
/* 26A2 */ { "DOUBLED FEMALE SIGN","	= female homosexuality\n"
	"	x (two women holding hands - 1F46D)"},
/* 26A3 */ { "DOUBLED MALE SIGN","	* a glyph variant has the two circles on the same line\n"
	"	= male homosexuality\n"
	"	x (two men holding hands - 1F46C)"},
/* 26A4 */ { "INTERLOCKED FEMALE AND MALE SIGN","	* a glyph variant has the two circles on the same line\n"
	"	= heterosexuality"},
/* 26A5 */ { "MALE AND FEMALE SIGN","	= intersex, androgynous\n"
	"	= hermaphroditic (in botany)"},
/* 26A6 */ { "MALE WITH STROKE SIGN","	= transgender\n"
	"	= alchemical symbol for iron or crocus of iron"},
/* 26A7 */ { "MALE WITH STROKE AND MALE AND FEMALE SIGN","	= transgender"},
/* 26A8 */ { "VERTICAL MALE WITH STROKE SIGN","	= alchemical symbol for iron"},
/* 26A9 */ { "HORIZONTAL MALE WITH STROKE SIGN","	= alchemical symbol for iron"},
/* 26AA */ { "MEDIUM WHITE CIRCLE","	= asexuality, sexless, genderless\n"
	"	= engaged, betrothed\n"
	"	* base for male or female sign\n"
	"	x (medium bold white circle - 1F785)"},
/* 26AB */ { "MEDIUM BLACK CIRCLE","	* UI symbol for record function\n"
	"	x (black circle for record - 23FA)"},
/* 26AC */ { "MEDIUM SMALL WHITE CIRCLE","	= engaged, betrothed (genealogy)\n"
	"	* can represent wedding ring"},
/* 26AD */ { "MARRIAGE SYMBOL","	x (infinity - 221E)\n"
	"	x (wedding - 1F492)"},
/* 26AE */ { "DIVORCE SYMBOL","	x (infinity negated with vertical bar - 29DE)\n"
	"	x (united symbol - 2BFA)"},
/* 26AF */ { "UNMARRIED PARTNERSHIP SYMBOL","	x (double-ended multimap - 29DF)\n"
	"	x (separated symbol - 2BFB)"},
/* 26B0 */ { "COFFIN","	= buried (genealogy)\n"
	"	x (white rectangle - 25AD)"},
/* 26B1 */ { "FUNERAL URN","	= cremated (genealogy)"},
/* 26B2 */ { "NEUTER",NULL},
/* 26B3 */ { "CERES",NULL},
/* 26B4 */ { "PALLAS",NULL},
/* 26B5 */ { "JUNO",NULL},
/* 26B6 */ { "VESTA",NULL},
/* 26B7 */ { "CHIRON",NULL},
/* 26B8 */ { "BLACK MOON LILITH",NULL},
/* 26B9 */ { "SEXTILE","	* an aspect of 60 degrees\n"
	"	x (asterisk - 002A)\n"
	"	x (six pointed black star - 2736)\n"
	"	x (alchemical symbol for sal-ammoniac - 1F739)"},
/* 26BA */ { "SEMISEXTILE","	* an aspect of 30 degrees\n"
	"	x (xor - 22BB)"},
/* 26BB */ { "QUINCUNX","	* an aspect of 150 degrees\n"
	"	x (nand - 22BC)"},
/* 26BC */ { "SESQUIQUADRATE","	* an aspect of 135 degrees"},
/* 26BD */ { "SOCCER BALL",NULL},
/* 26BE */ { "BASEBALL",NULL},
/* 26BF */ { "SQUARED KEY","	= parental lock\n"
	"	x (key - 1F511)"},
/* 26C0 */ { "WHITE DRAUGHTS MAN",NULL},
/* 26C1 */ { "WHITE DRAUGHTS KING",NULL},
/* 26C2 */ { "BLACK DRAUGHTS MAN",NULL},
/* 26C3 */ { "BLACK DRAUGHTS KING",NULL},
/* 26C4 */ { "SNOWMAN WITHOUT SNOW","	= light snow"},
/* 26C5 */ { "SUN BEHIND CLOUD","	= partly cloudy\n"
	"	x (white sun with small cloud - 1F324)"},
/* 26C6 */ { "RAIN","	= rainy weather"},
/* 26C7 */ { "BLACK SNOWMAN","	= heavy snow"},
/* 26C8 */ { "THUNDER CLOUD AND RAIN","	= thunderstorm\n"
	"	x (cloud with lightning - 1F329)"},
/* 26C9 */ { "TURNED WHITE SHOGI PIECE",NULL},
/* 26CA */ { "TURNED BLACK SHOGI PIECE",NULL},
/* 26CB */ { "WHITE DIAMOND IN SQUARE","	x (apl functional symbol quad diamond - 233A)"},
/* 26CC */ { "CROSSING LANES","	= accident\n"
	"	x (falling diagonal crossing rising diagonal - 292C)"},
/* 26CD */ { "DISABLED CAR",NULL},
/* 26CE */ { "OPHIUCHUS",NULL},
/* 26CF */ { "PICK","	= under construction\n"
	"	x (hammer and pick - 2692)\n"
	"	x (construction sign - 1F6A7)"},
/* 26D0 */ { "CAR SLIDING","	= icy road"},
/* 26D1 */ { "HELMET WITH WHITE CROSS","	= maintenance\n"
	"	x (construction worker - 1F477)"},
/* 26D2 */ { "CIRCLED CROSSING LANES","	= road closed\n"
	"	x (circled times - 2297)"},
/* 26D3 */ { "CHAINS","	= tyre chains required"},
/* 26D4 */ { "NO ENTRY","	x (no entry sign - 1F6AB)"},
/* 26D5 */ { "ALTERNATE ONE-WAY LEFT WAY TRAFFIC","	* left side traffic"},
/* 26D6 */ { "BLACK TWO-WAY LEFT WAY TRAFFIC","	* left side traffic"},
/* 26D7 */ { "WHITE TWO-WAY LEFT WAY TRAFFIC","	* left side traffic"},
/* 26D8 */ { "BLACK LEFT LANE MERGE","	* left side traffic"},
/* 26D9 */ { "WHITE LEFT LANE MERGE","	* left side traffic"},
/* 26DA */ { "DRIVE SLOW SIGN",NULL},
/* 26DB */ { "HEAVY WHITE DOWN-POINTING TRIANGLE","	= drive slow\n"
	"	x (white down-pointing triangle - 25BD)"},
/* 26DC */ { "LEFT CLOSED ENTRY",NULL},
/* 26DD */ { "SQUARED SALTIRE","	= closed entry\n"
	"	x (squared times - 22A0)"},
/* 26DE */ { "FALLING DIAGONAL IN WHITE CIRCLE IN BLACK SQUARE","	= closed to large vehicles"},
/* 26DF */ { "BLACK TRUCK","	= black lorry\n"
	"	= closed to large vehicles, alternate\n"
	"	x (delivery truck - 1F69A)"},
/* 26E0 */ { "RESTRICTED LEFT ENTRY-1",NULL},
/* 26E1 */ { "RESTRICTED LEFT ENTRY-2",NULL},
/* 26E2 */ { "ASTRONOMICAL SYMBOL FOR URANUS","	x (uranus - 2645)"},
/* 26E3 */ { "HEAVY CIRCLE WITH STROKE AND TWO DOTS ABOVE","	= public office"},
/* 26E4 */ { "PENTAGRAM","	= pentalpha, pentangle\n"
	"	* symbol of various uses, including symbol on Moroccan flag\n"
	"	x (white star - 2606)"},
/* 26E5 */ { "RIGHT-HANDED INTERLACED PENTAGRAM","	* symbol of various countries, including Morocco\n"
	"	x (outlined white star - 269D)"},
/* 26E6 */ { "LEFT-HANDED INTERLACED PENTAGRAM","	* symbol of various countries, including Ethiopia"},
/* 26E7 */ { "INVERTED PENTAGRAM",NULL},
/* 26E8 */ { "BLACK CROSS ON SHIELD","	= hospital\n"
	"	x (hospital - 1F3E5)"},
/* 26E9 */ { "SHINTO SHRINE","	= torii"},
/* 26EA */ { "CHURCH",NULL},
/* 26EB */ { "CASTLE","	x (european castle - 1F3F0)"},
/* 26EC */ { "HISTORIC SITE","	x (therefore - 2234)"},
/* 26ED */ { "GEAR WITHOUT HUB","	= factory\n"
	"	x (gear - 2699)\n"
	"	x (factory - 1F3ED)"},
/* 26EE */ { "GEAR WITH HANDLES","	= power plant, power substation"},
/* 26EF */ { "MAP SYMBOL FOR LIGHTHOUSE",NULL},
/* 26F0 */ { "MOUNTAIN","	x (snow capped mountain - 1F3D4)"},
/* 26F1 */ { "UMBRELLA ON GROUND","	= bathing beach\n"
	"	x (umbrella - 2602)\n"
	"	x (beach with umbrella - 1F3D6)"},
/* 26F2 */ { "FOUNTAIN","	= park"},
/* 26F3 */ { "FLAG IN HOLE","	= golf course\n"
	"	x (triangular flag on post - 1F6A9)"},
/* 26F4 */ { "FERRY","	= ferry boat terminal\n"
	"	x (ship - 1F6A2)"},
/* 26F5 */ { "SAILBOAT","	= marina or yacht harbour\n"
	"	x (rowboat - 1F6A3)"},
/* 26F6 */ { "SQUARE FOUR CORNERS","	= intersection"},
/* 26F7 */ { "SKIER","	= ski resort\n"
	"	x (ski and ski boot - 1F3BF)"},
/* 26F8 */ { "ICE SKATE","	= ice skating rink"},
/* 26F9 */ { "PERSON WITH BALL","	= track and field, gymnasium"},
/* 26FA */ { "TENT","	= camping site\n"
	"	x (camping - 1F3D5)"},
/* 26FB */ { "JAPANESE BANK SYMBOL","	x (bank - 1F3E6)"},
/* 26FC */ { "HEADSTONE GRAVEYARD SYMBOL","	= graveyard, memorial park, cemetery"},
/* 26FD */ { "FUEL PUMP","	= petrol station, gas station"},
/* 26FE */ { "CUP ON BLACK SQUARE","	= drive-in restaurant\n"
	"	x (hot beverage - 2615)\n"
	"	x (teacup without handle - 1F375)"},
/* 26FF */ { "WHITE FLAG WITH HORIZONTAL MIDDLE BLACK STRIPE","	= Japanese self-defence force site"}
};

UN_DLL_LOCAL
static const struct unicode_nameannot una_00_27[] = {
/* 2700 */ { "BLACK SAFETY SCISSORS","	* forms a pair with 2704 white scissors"},
/* 2701 */ { "UPPER BLADE SCISSORS",NULL},
/* 2702 */ { "BLACK SCISSORS",NULL},
/* 2703 */ { "LOWER BLADE SCISSORS",NULL},
/* 2704 */ { "WHITE SCISSORS",NULL},
/* 2705 */ { "WHITE HEAVY CHECK MARK","	x (heavy check mark - 2714)"},
/* 2706 */ { "TELEPHONE LOCATION SIGN","	x (telephone sign - 2121)\n"
	"	x (telephone receiver - 1F4DE)"},
/* 2707 */ { "TAPE DRIVE",NULL},
/* 2708 */ { "AIRPLANE","	x (up-pointing airplane - 1F6E7)"},
/* 2709 */ { "ENVELOPE","	= sealed envelope\n"
	"	x (incoming envelope - 1F4E8)\n"
	"	x (back of envelope - 1F582)"},
/* 270A */ { "RAISED FIST","	= rock in Rock, Paper, Scissors game\n"
	"	x (fisted hand sign - 1F44A)"},
/* 270B */ { "RAISED HAND","	= paper in Rock, Paper, Scissors game\n"
	"	x (waving hand sign - 1F44B)\n"
	"	x (raised hand with fingers splayed - 1F590)"},
/* 270C */ { "VICTORY HAND","	= scissors in Rock, Paper, Scissors game\n"
	"	x (reversed victory hand - 1F594)"},
/* 270D */ { "WRITING HAND","	x (left writing hand - 1F58E)"},
/* 270E */ { "LOWER RIGHT PENCIL","	x (lower left pencil - 1F589)"},
/* 270F */ { "PENCIL",NULL},
/* 2710 */ { "UPPER RIGHT PENCIL",NULL},
/* 2711 */ { "WHITE NIB","	x (lower left fountain pen - 1F58B)"},
/* 2712 */ { "BLACK NIB",NULL},
/* 2713 */ { "CHECK MARK","	x (square root - 221A)\n"
	"	x (light check mark - 1F5F8)\n"
	"	x (inverse check mark - 1FBB1)"},
/* 2714 */ { "HEAVY CHECK MARK",NULL},
/* 2715 */ { "MULTIPLICATION X","	x (multiplication sign - 00D7)\n"
	"	x (box drawings light diagonal cross - 2573)\n"
	"	x (thin saltire - 1F7A8)"},
/* 2716 */ { "HEAVY MULTIPLICATION X","	x (extremely heavy saltire - 1F7AE)"},
/* 2717 */ { "BALLOT X","	x (saltire - 2613)\n"
	"	x (ballot script x - 1F5F4)"},
/* 2718 */ { "HEAVY BALLOT X","	x (ballot bold script x - 1F5F6)"},
/* 2719 */ { "OUTLINED GREEK CROSS",NULL},
/* 271A */ { "HEAVY GREEK CROSS","	x (very bold greek cross - 1F7A5)\n"
	"	x (voided greek cross - 1FBBB)"},
/* 271B */ { "OPEN CENTRE CROSS",NULL},
/* 271C */ { "HEAVY OPEN CENTRE CROSS",NULL},
/* 271D */ { "LATIN CROSS",NULL},
/* 271E */ { "SHADOWED WHITE LATIN CROSS","	x (white latin cross - 1F546)"},
/* 271F */ { "OUTLINED LATIN CROSS",NULL},
/* 2720 */ { "MALTESE CROSS","	* historically, the Maltese cross took many forms; the shape shown in the Zapf Dingbats is similar to one known as the Cross Formée\n"
	"	x (cross patty with right crossbar - 2E50)\n"
	"	x (circled cross formee - 1F902)"},
/* 2721 */ { "STAR OF DAVID","	x (six pointed star with middle dot - 1F52F)"},
/* 2722 */ { "FOUR TEARDROP-SPOKED ASTERISK",NULL},
/* 2723 */ { "FOUR BALLOON-SPOKED ASTERISK",NULL},
/* 2724 */ { "HEAVY FOUR BALLOON-SPOKED ASTERISK",NULL},
/* 2725 */ { "FOUR CLUB-SPOKED ASTERISK",NULL},
/* 2726 */ { "BLACK FOUR POINTED STAR","	x (light four pointed black cusp - 2BCC)\n"
	"	x (four pointed black star - 1F7C6)"},
/* 2727 */ { "WHITE FOUR POINTED STAR","	x (white four pointed cusp - 2BCE)"},
/* 2728 */ { "SPARKLES",NULL},
/* 2729 */ { "STRESS OUTLINED WHITE STAR","	x (white star - 2606)"},
/* 272A */ { "CIRCLED WHITE STAR",NULL},
/* 272B */ { "OPEN CENTRE BLACK STAR",NULL},
/* 272C */ { "BLACK CENTRE WHITE STAR",NULL},
/* 272D */ { "OUTLINED BLACK STAR",NULL},
/* 272E */ { "HEAVY OUTLINED BLACK STAR",NULL},
/* 272F */ { "PINWHEEL STAR",NULL},
/* 2730 */ { "SHADOWED WHITE STAR",NULL},
/* 2731 */ { "HEAVY ASTERISK","	x (asterisk - 002A)\n"
	"	x (heavy six spoked asterisk - 1F7B8)"},
/* 2732 */ { "OPEN CENTRE ASTERISK",NULL},
/* 2733 */ { "EIGHT SPOKED ASTERISK","	x (light eight spoked asterisk - 1F7BB)"},
/* 2734 */ { "EIGHT POINTED BLACK STAR",NULL},
/* 2735 */ { "EIGHT POINTED PINWHEEL STAR","	x (heavy eight pointed pinwheel star - 1F7D1)"},
/* 2736 */ { "SIX POINTED BLACK STAR","	x (sextile - 26B9)"},
/* 2737 */ { "EIGHT POINTED RECTILINEAR BLACK STAR","	x (medium eight pointed black star - 1F7CE)"},
/* 2738 */ { "HEAVY EIGHT POINTED RECTILINEAR BLACK STAR","	x (heavy eight pointed black star - 1F7CF)"},
/* 2739 */ { "TWELVE POINTED BLACK STAR",NULL},
/* 273A */ { "SIXTEEN POINTED ASTERISK","	= starburst"},
/* 273B */ { "TEARDROP-SPOKED ASTERISK",NULL},
/* 273C */ { "OPEN CENTRE TEARDROP-SPOKED ASTERISK",NULL},
/* 273D */ { "HEAVY TEARDROP-SPOKED ASTERISK",NULL},
/* 273E */ { "SIX PETALLED BLACK AND WHITE FLORETTE",NULL},
/* 273F */ { "BLACK FLORETTE",NULL},
/* 2740 */ { "WHITE FLORETTE","	x (rosette - 1F3F5)\n"
	"	x (white flower - 1F4AE)"},
/* 2741 */ { "EIGHT PETALLED OUTLINED BLACK FLORETTE","	x (black rosette - 1F3F6)"},
/* 2742 */ { "CIRCLED OPEN CENTRE EIGHT POINTED STAR",NULL},
/* 2743 */ { "HEAVY TEARDROP-SPOKED PINWHEEL ASTERISK",NULL},
/* 2744 */ { "SNOWFLAKE",NULL},
/* 2745 */ { "TIGHT TRIFOLIATE SNOWFLAKE",NULL},
/* 2746 */ { "HEAVY CHEVRON SNOWFLAKE",NULL},
/* 2747 */ { "SPARKLE",NULL},
/* 2748 */ { "HEAVY SPARKLE",NULL},
/* 2749 */ { "BALLOON-SPOKED ASTERISK","	= jack"},
/* 274A */ { "EIGHT TEARDROP-SPOKED PROPELLER ASTERISK",NULL},
/* 274B */ { "HEAVY EIGHT TEARDROP-SPOKED PROPELLER ASTERISK","	= turbofan"},
/* 274C */ { "CROSS MARK","	* forms a game tally pair with 2B55\n"
	"	x (multiplication sign - 00D7)\n"
	"	x (cancellation x - 1F5D9)\n"
	"	x (heavy saltire - 1F7AC)\n"
	"	x (white heavy saltire with rounded corners - 1FBC0)"},
/* 274D */ { "SHADOWED WHITE CIRCLE","	x (lower right shadowed white circle - 1F53E)"},
/* 274E */ { "NEGATIVE SQUARED CROSS MARK","	x (negative diagonal cross - 1FBBD)"},
/* 274F */ { "LOWER RIGHT DROP-SHADOWED WHITE SQUARE",NULL},
/* 2750 */ { "UPPER RIGHT DROP-SHADOWED WHITE SQUARE",NULL},
/* 2751 */ { "LOWER RIGHT SHADOWED WHITE SQUARE",NULL},
/* 2752 */ { "UPPER RIGHT SHADOWED WHITE SQUARE",NULL},
/* 2753 */ { "BLACK QUESTION MARK ORNAMENT",NULL},
/* 2754 */ { "WHITE QUESTION MARK ORNAMENT",NULL},
/* 2755 */ { "WHITE EXCLAMATION MARK ORNAMENT",NULL},
/* 2756 */ { "BLACK DIAMOND MINUS WHITE X",NULL},
/* 2757 */ { "HEAVY EXCLAMATION MARK SYMBOL","	= obstacles on the road, ARIB STD B24"},
/* 2758 */ { "LIGHT VERTICAL BAR","	x (vertical line - 007C)"},
/* 2759 */ { "MEDIUM VERTICAL BAR",NULL},
/* 275A */ { "HEAVY VERTICAL BAR",NULL},
/* 275B */ { "HEAVY SINGLE TURNED COMMA QUOTATION MARK ORNAMENT","	x (left single quotation mark - 2018)"},
/* 275C */ { "HEAVY SINGLE COMMA QUOTATION MARK ORNAMENT","	x (right single quotation mark - 2019)"},
/* 275D */ { "HEAVY DOUBLE TURNED COMMA QUOTATION MARK ORNAMENT","	x (left double quotation mark - 201C)\n"
	"	x (sans-serif heavy double turned comma quotation mark ornament - 1F676)"},
/* 275E */ { "HEAVY DOUBLE COMMA QUOTATION MARK ORNAMENT","	x (right double quotation mark - 201D)\n"
	"	x (sans-serif heavy double comma quotation mark ornament - 1F677)"},
/* 275F */ { "HEAVY LOW SINGLE COMMA QUOTATION MARK ORNAMENT","	x (single low-9 quotation mark - 201A)"},
/* 2760 */ { "HEAVY LOW DOUBLE COMMA QUOTATION MARK ORNAMENT","	x (double low-9 quotation mark - 201E)\n"
	"	x (sans-serif heavy low double comma quotation mark ornament - 1F678)"},
/* 2761 */ { "CURVED STEM PARAGRAPH SIGN ORNAMENT","	x (pilcrow sign - 00B6)"},
/* 2762 */ { "HEAVY EXCLAMATION MARK ORNAMENT","	x (exclamation mark - 0021)"},
/* 2763 */ { "HEAVY HEART EXCLAMATION MARK ORNAMENT",NULL},
/* 2764 */ { "HEAVY BLACK HEART","	* displayed with a red color when used in emoji style\n"
	"	x (black heart suit - 2665)\n"
	"	x (black heart - 1F5A4)\n"
	"	x (white heart - 1F90D)"},
/* 2765 */ { "ROTATED HEAVY BLACK HEART BULLET",NULL},
/* 2766 */ { "FLORAL HEART","	= Aldus leaf"},
/* 2767 */ { "ROTATED FLORAL HEART BULLET","	= hedera, ivy leaf\n"
	"	x (reversed rotated floral heart bullet - 2619)\n"
	"	x (south east pointing vine leaf - 1F65B)"},
/* 2768 */ { "MEDIUM LEFT PARENTHESIS ORNAMENT","	x (left parenthesis - 0028)"},
/* 2769 */ { "MEDIUM RIGHT PARENTHESIS ORNAMENT","	x (right parenthesis - 0029)"},
/* 276A */ { "MEDIUM FLATTENED LEFT PARENTHESIS ORNAMENT",NULL},
/* 276B */ { "MEDIUM FLATTENED RIGHT PARENTHESIS ORNAMENT",NULL},
/* 276C */ { "MEDIUM LEFT-POINTING ANGLE BRACKET ORNAMENT","	x (left-pointing angle bracket - 2329)"},
/* 276D */ { "MEDIUM RIGHT-POINTING ANGLE BRACKET ORNAMENT","	x (right-pointing angle bracket - 232A)"},
/* 276E */ { "HEAVY LEFT-POINTING ANGLE QUOTATION MARK ORNAMENT","	x (single left-pointing angle quotation mark - 2039)"},
/* 276F */ { "HEAVY RIGHT-POINTING ANGLE QUOTATION MARK ORNAMENT","	x (single right-pointing angle quotation mark - 203A)"},
/* 2770 */ { "HEAVY LEFT-POINTING ANGLE BRACKET ORNAMENT",NULL},
/* 2771 */ { "HEAVY RIGHT-POINTING ANGLE BRACKET ORNAMENT",NULL},
/* 2772 */ { "LIGHT LEFT TORTOISE SHELL BRACKET ORNAMENT","	x (left tortoise shell bracket - 3014)"},
/* 2773 */ { "LIGHT RIGHT TORTOISE SHELL BRACKET ORNAMENT","	x (right tortoise shell bracket - 3015)"},
/* 2774 */ { "MEDIUM LEFT CURLY BRACKET ORNAMENT","	x (left curly bracket - 007B)"},
/* 2775 */ { "MEDIUM RIGHT CURLY BRACKET ORNAMENT","	x (right curly bracket - 007D)"},
/* 2776 */ { "DINGBAT NEGATIVE CIRCLED DIGIT ONE",NULL},
/* 2777 */ { "DINGBAT NEGATIVE CIRCLED DIGIT TWO",NULL},
/* 2778 */ { "DINGBAT NEGATIVE CIRCLED DIGIT THREE",NULL},
/* 2779 */ { "DINGBAT NEGATIVE CIRCLED DIGIT FOUR",NULL},
/* 277A */ { "DINGBAT NEGATIVE CIRCLED DIGIT FIVE",NULL},
/* 277B */ { "DINGBAT NEGATIVE CIRCLED DIGIT SIX",NULL},
/* 277C */ { "DINGBAT NEGATIVE CIRCLED DIGIT SEVEN",NULL},
/* 277D */ { "DINGBAT NEGATIVE CIRCLED DIGIT EIGHT",NULL},
/* 277E */ { "DINGBAT NEGATIVE CIRCLED DIGIT NINE",NULL},
/* 277F */ { "DINGBAT NEGATIVE CIRCLED NUMBER TEN",NULL},
/* 2780 */ { "DINGBAT CIRCLED SANS-SERIF DIGIT ONE",NULL},
/* 2781 */ { "DINGBAT CIRCLED SANS-SERIF DIGIT TWO",NULL},
/* 2782 */ { "DINGBAT CIRCLED SANS-SERIF DIGIT THREE",NULL},
/* 2783 */ { "DINGBAT CIRCLED SANS-SERIF DIGIT FOUR",NULL},
/* 2784 */ { "DINGBAT CIRCLED SANS-SERIF DIGIT FIVE",NULL},
/* 2785 */ { "DINGBAT CIRCLED SANS-SERIF DIGIT SIX",NULL},
/* 2786 */ { "DINGBAT CIRCLED SANS-SERIF DIGIT SEVEN",NULL},
/* 2787 */ { "DINGBAT CIRCLED SANS-SERIF DIGIT EIGHT",NULL},
/* 2788 */ { "DINGBAT CIRCLED SANS-SERIF DIGIT NINE",NULL},
/* 2789 */ { "DINGBAT CIRCLED SANS-SERIF NUMBER TEN",NULL},
/* 278A */ { "DINGBAT NEGATIVE CIRCLED SANS-SERIF DIGIT ONE",NULL},
/* 278B */ { "DINGBAT NEGATIVE CIRCLED SANS-SERIF DIGIT TWO",NULL},
/* 278C */ { "DINGBAT NEGATIVE CIRCLED SANS-SERIF DIGIT THREE",NULL},
/* 278D */ { "DINGBAT NEGATIVE CIRCLED SANS-SERIF DIGIT FOUR",NULL},
/* 278E */ { "DINGBAT NEGATIVE CIRCLED SANS-SERIF DIGIT FIVE",NULL},
/* 278F */ { "DINGBAT NEGATIVE CIRCLED SANS-SERIF DIGIT SIX",NULL},
/* 2790 */ { "DINGBAT NEGATIVE CIRCLED SANS-SERIF DIGIT SEVEN",NULL},
/* 2791 */ { "DINGBAT NEGATIVE CIRCLED SANS-SERIF DIGIT EIGHT",NULL},
/* 2792 */ { "DINGBAT NEGATIVE CIRCLED SANS-SERIF DIGIT NINE",NULL},
/* 2793 */ { "DINGBAT NEGATIVE CIRCLED SANS-SERIF NUMBER TEN",NULL},
/* 2794 */ { "HEAVY WIDE-HEADED RIGHTWARDS ARROW","	x (wide-headed rightwards medium barb arrow - 1F872)"},
/* 2795 */ { "HEAVY PLUS SIGN","	x (plus sign - 002B)"},
/* 2796 */ { "HEAVY MINUS SIGN","	x (minus sign - 2212)"},
/* 2797 */ { "HEAVY DIVISION SIGN","	x (division sign - 00F7)"},
/* 2798 */ { "HEAVY SOUTH EAST ARROW",NULL},
/* 2799 */ { "HEAVY RIGHTWARDS ARROW",NULL},
/* 279A */ { "HEAVY NORTH EAST ARROW",NULL},
/* 279B */ { "DRAFTING POINT RIGHTWARDS ARROW",NULL},
/* 279C */ { "HEAVY ROUND-TIPPED RIGHTWARDS ARROW",NULL},
/* 279D */ { "TRIANGLE-HEADED RIGHTWARDS ARROW",NULL},
/* 279E */ { "HEAVY TRIANGLE-HEADED RIGHTWARDS ARROW",NULL},
/* 279F */ { "DASHED TRIANGLE-HEADED RIGHTWARDS ARROW",NULL},
/* 27A0 */ { "HEAVY DASHED TRIANGLE-HEADED RIGHTWARDS ARROW",NULL},
/* 27A1 */ { "BLACK RIGHTWARDS ARROW","	* fonts may harmonize this glyph with the style for other black arrows\n"
	"	x (leftwards black arrow - 2B05)\n"
	"	x (rightwards black arrow - 2B95)"},
/* 27A2 */ { "THREE-D TOP-LIGHTED RIGHTWARDS ARROWHEAD","	x (three-d top-lighted rightwards equilateral arrowhead - 2B9A)"},
/* 27A3 */ { "THREE-D BOTTOM-LIGHTED RIGHTWARDS ARROWHEAD",NULL},
/* 27A4 */ { "BLACK RIGHTWARDS ARROWHEAD","	x (black rightwards equilateral arrowhead - 2B9E)\n"
	"	x (arrowhead-shaped pointer - 1FBB0)"},
/* 27A5 */ { "HEAVY BLACK CURVED DOWNWARDS AND RIGHTWARDS ARROW","	x (black curved downwards and rightwards arrow - 2BA9)"},
/* 27A6 */ { "HEAVY BLACK CURVED UPWARDS AND RIGHTWARDS ARROW","	x (black curved upwards and rightwards arrow - 2BAB)"},
/* 27A7 */ { "SQUAT BLACK RIGHTWARDS ARROW","	x (rightwards heavy compressed arrow - 1F842)"},
/* 27A8 */ { "HEAVY CONCAVE-POINTED BLACK RIGHTWARDS ARROW",NULL},
/* 27A9 */ { "RIGHT-SHADED WHITE RIGHTWARDS ARROW","	x (rightwards right-shaded white arrow - 1F8A5)"},
/* 27AA */ { "LEFT-SHADED WHITE RIGHTWARDS ARROW","	x (rightwards left-shaded white arrow - 1F8A7)"},
/* 27AB */ { "BACK-TILTED SHADOWED WHITE RIGHTWARDS ARROW","	x (rightwards back-tilted shadowed white arrow - 1F8A9)"},
/* 27AC */ { "FRONT-TILTED SHADOWED WHITE RIGHTWARDS ARROW","	x (rightwards front-tilted shadowed white arrow - 1F8AB)"},
/* 27AD */ { "HEAVY LOWER RIGHT-SHADOWED WHITE RIGHTWARDS ARROW",NULL},
/* 27AE */ { "HEAVY UPPER RIGHT-SHADOWED WHITE RIGHTWARDS ARROW",NULL},
/* 27AF */ { "NOTCHED LOWER RIGHT-SHADOWED WHITE RIGHTWARDS ARROW","	x (rightwards arrow with notched tail - 1F89A)"},
/* 27B0 */ { "CURLY LOOP","	= kururi"},
/* 27B1 */ { "NOTCHED UPPER RIGHT-SHADOWED WHITE RIGHTWARDS ARROW",NULL},
/* 27B2 */ { "CIRCLED HEAVY WHITE RIGHTWARDS ARROW",NULL},
/* 27B3 */ { "WHITE-FEATHERED RIGHTWARDS ARROW",NULL},
/* 27B4 */ { "BLACK-FEATHERED SOUTH EAST ARROW",NULL},
/* 27B5 */ { "BLACK-FEATHERED RIGHTWARDS ARROW",NULL},
/* 27B6 */ { "BLACK-FEATHERED NORTH EAST ARROW",NULL},
/* 27B7 */ { "HEAVY BLACK-FEATHERED SOUTH EAST ARROW",NULL},
/* 27B8 */ { "HEAVY BLACK-FEATHERED RIGHTWARDS ARROW",NULL},
/* 27B9 */ { "HEAVY BLACK-FEATHERED NORTH EAST ARROW",NULL},
/* 27BA */ { "TEARDROP-BARBED RIGHTWARDS ARROW",NULL},
/* 27BB */ { "HEAVY TEARDROP-SHANKED RIGHTWARDS ARROW",NULL},
/* 27BC */ { "WEDGE-TAILED RIGHTWARDS ARROW",NULL},
/* 27BD */ { "HEAVY WEDGE-TAILED RIGHTWARDS ARROW",NULL},
/* 27BE */ { "OPEN-OUTLINED RIGHTWARDS ARROW",NULL},
/* 27BF */ { "DOUBLE CURLY LOOP",NULL},
/* 27C0 */ { "THREE DIMENSIONAL ANGLE","	* used by Euclid"},
/* 27C1 */ { "WHITE TRIANGLE CONTAINING SMALL WHITE TRIANGLE","	* used by Euclid"},
/* 27C2 */ { "PERPENDICULAR","	= orthogonal to\n"
	"	* relation, typeset with additional spacing\n"
	"	x (up tack - 22A5)"},
/* 27C3 */ { "OPEN SUBSET",NULL},
/* 27C4 */ { "OPEN SUPERSET",NULL},
/* 27C5 */ { "LEFT S-SHAPED BAG DELIMITER",NULL},
/* 27C6 */ { "RIGHT S-SHAPED BAG DELIMITER",NULL},
/* 27C7 */ { "OR WITH DOT INSIDE","	x (logical or - 2228)\n"
	"	x (multiset multiplication - 228D)\n"
	"	x (and with dot - 27D1)"},
/* 27C8 */ { "REVERSE SOLIDUS PRECEDING SUBSET",NULL},
/* 27C9 */ { "SUPERSET PRECEDING SOLIDUS",NULL},
/* 27CA */ { "VERTICAL BAR WITH HORIZONTAL STROKE","	x (parallel with horizontal stroke - 2AF2)\n"
	"	x (triple vertical bar with horizontal stroke - 2AF5)"},
/* 27CB */ { "MATHEMATICAL RISING DIAGONAL","	= \\diagup\n"
	"	x (division slash - 2215)"},
/* 27CC */ { "LONG DIVISION","	* graphically extends over the dividend\n"
	"	x (division sign - 00F7)\n"
	"	x (division slash - 2215)\n"
	"	x (square root - 221A)"},
/* 27CD */ { "MATHEMATICAL FALLING DIAGONAL","	= \\diagdown\n"
	"	x (set minus - 2216)\n"
	"	x (reverse solidus operator - 29F5)"},
/* 27CE */ { "SQUARED LOGICAL AND","	= box min\n"
	"	* morphological min product operator\n"
	"	* morphological erosion operator\n"
	"	* additive minimum operator"},
/* 27CF */ { "SQUARED LOGICAL OR","	= box max\n"
	"	* morphological max product operator\n"
	"	* morphological dilation operator\n"
	"	* additive maximum operator"},
/* 27D0 */ { "WHITE DIAMOND WITH CENTRED DOT","	x (diamond shape with a dot inside - 1F4A0)"},
/* 27D1 */ { "AND WITH DOT","	x (logical and - 2227)\n"
	"	x (or with dot inside - 27C7)\n"
	"	x (intersection with dot - 2A40)"},
/* 27D2 */ { "ELEMENT OF OPENING UPWARDS","	x (element of opening downwards - 2AD9)"},
/* 27D3 */ { "LOWER RIGHT CORNER WITH DOT","	= pullback\n"
	"	x (right floor - 230B)"},
/* 27D4 */ { "UPPER LEFT CORNER WITH DOT","	= pushout\n"
	"	x (left ceiling - 2308)"},
/* 27D5 */ { "LEFT OUTER JOIN",NULL},
/* 27D6 */ { "RIGHT OUTER JOIN",NULL},
/* 27D7 */ { "FULL OUTER JOIN","	x (join - 2A1D)"},
/* 27D8 */ { "LARGE UP TACK","	x (up tack - 22A5)"},
/* 27D9 */ { "LARGE DOWN TACK","	x (down tack - 22A4)"},
/* 27DA */ { "LEFT AND RIGHT DOUBLE TURNSTILE","	x (true - 22A8)\n"
	"	x (vertical bar double left turnstile - 2AE4)"},
/* 27DB */ { "LEFT AND RIGHT TACK","	x (right tack - 22A2)"},
/* 27DC */ { "LEFT MULTIMAP","	x (multimap - 22B8)"},
/* 27DD */ { "LONG RIGHT TACK","	x (right tack - 22A2)"},
/* 27DE */ { "LONG LEFT TACK","	x (left tack - 22A3)"},
/* 27DF */ { "UP TACK WITH CIRCLE ABOVE","	= radial component\n"
	"	x (down tack with circle below - 2AF1)"},
/* 27E0 */ { "LOZENGE DIVIDED BY HORIZONTAL RULE","	* used as form of possibility in modal logic\n"
	"	x (lozenge - 25CA)"},
/* 27E1 */ { "WHITE CONCAVE-SIDED DIAMOND","	= never (modal operator)\n"
	"	x (white diamond - 25C7)"},
/* 27E2 */ { "WHITE CONCAVE-SIDED DIAMOND WITH LEFTWARDS TICK","	= was never (modal operator)"},
/* 27E3 */ { "WHITE CONCAVE-SIDED DIAMOND WITH RIGHTWARDS TICK","	= will never be (modal operator)"},
/* 27E4 */ { "WHITE SQUARE WITH LEFTWARDS TICK","	= was always (modal operator)\n"
	"	x (white square - 25A1)\n"
	"	x (white medium square - 25FB)"},
/* 27E5 */ { "WHITE SQUARE WITH RIGHTWARDS TICK","	= will always be (modal operator)"},
/* 27E6 */ { "MATHEMATICAL LEFT WHITE SQUARE BRACKET","	= z notation left bag bracket\n"
	"	x (left white square bracket - 301A)"},
/* 27E7 */ { "MATHEMATICAL RIGHT WHITE SQUARE BRACKET","	= z notation right bag bracket\n"
	"	x (right white square bracket - 301B)"},
/* 27E8 */ { "MATHEMATICAL LEFT ANGLE BRACKET","	= bra\n"
	"	= z notation left sequence bracket\n"
	"	x (left-pointing angle bracket - 2329)\n"
	"	x (left angle bracket - 3008)"},
/* 27E9 */ { "MATHEMATICAL RIGHT ANGLE BRACKET","	= ket\n"
	"	= z notation right sequence bracket\n"
	"	x (right-pointing angle bracket - 232A)\n"
	"	x (right angle bracket - 3009)"},
/* 27EA */ { "MATHEMATICAL LEFT DOUBLE ANGLE BRACKET","	= z notation left chevron bracket\n"
	"	x (left double angle bracket - 300A)"},
/* 27EB */ { "MATHEMATICAL RIGHT DOUBLE ANGLE BRACKET","	= z notation right chevron bracket\n"
	"	x (right double angle bracket - 300B)"},
/* 27EC */ { "MATHEMATICAL LEFT WHITE TORTOISE SHELL BRACKET","	x (left black tortoise shell bracket - 2997)\n"
	"	x (left white tortoise shell bracket - 3018)"},
/* 27ED */ { "MATHEMATICAL RIGHT WHITE TORTOISE SHELL BRACKET","	x (right black tortoise shell bracket - 2998)\n"
	"	x (right white tortoise shell bracket - 3019)"},
/* 27EE */ { "MATHEMATICAL LEFT FLATTENED PARENTHESIS","	= lgroup"},
/* 27EF */ { "MATHEMATICAL RIGHT FLATTENED PARENTHESIS","	= rgroup"},
/* 27F0 */ { "UPWARDS QUADRUPLE ARROW","	x (upwards triple arrow - 290A)\n"
	"	x (leftwards quadruple arrow - 2B45)"},
/* 27F1 */ { "DOWNWARDS QUADRUPLE ARROW","	x (downwards triple arrow - 290B)"},
/* 27F2 */ { "ANTICLOCKWISE GAPPED CIRCLE ARROW","	x (anticlockwise open circle arrow - 21BA)\n"
	"	x (anticlockwise closed circle arrow - 2940)"},
/* 27F3 */ { "CLOCKWISE GAPPED CIRCLE ARROW","	x (clockwise open circle arrow - 21BB)\n"
	"	x (clockwise closed circle arrow - 2941)"},
/* 27F4 */ { "RIGHT ARROW WITH CIRCLED PLUS","	x (left arrow with circled plus - 2B32)"},
/* 27F5 */ { "LONG LEFTWARDS ARROW","	x (leftwards arrow - 2190)"},
/* 27F6 */ { "LONG RIGHTWARDS ARROW","	x (rightwards arrow - 2192)"},
/* 27F7 */ { "LONG LEFT RIGHT ARROW","	x (left right arrow - 2194)"},
/* 27F8 */ { "LONG LEFTWARDS DOUBLE ARROW","	x (leftwards double arrow - 21D0)"},
/* 27F9 */ { "LONG RIGHTWARDS DOUBLE ARROW","	x (rightwards double arrow - 21D2)"},
/* 27FA */ { "LONG LEFT RIGHT DOUBLE ARROW","	x (left right double arrow - 21D4)"},
/* 27FB */ { "LONG LEFTWARDS ARROW FROM BAR","	= maps from\n"
	"	x (leftwards arrow from bar - 21A4)"},
/* 27FC */ { "LONG RIGHTWARDS ARROW FROM BAR","	= maps to\n"
	"	x (rightwards arrow from bar - 21A6)"},
/* 27FD */ { "LONG LEFTWARDS DOUBLE ARROW FROM BAR","	x (leftwards double arrow from bar - 2906)"},
/* 27FE */ { "LONG RIGHTWARDS DOUBLE ARROW FROM BAR","	x (rightwards double arrow from bar - 2907)"},
/* 27FF */ { "LONG RIGHTWARDS SQUIGGLE ARROW","	x (rightwards squiggle arrow - 21DD)\n"
	"	x (long leftwards squiggle arrow - 2B33)"}
};

UN_DLL_LOCAL
static const struct unicode_nameannot una_00_28[] = {
/* 2800 */ { "BRAILLE PATTERN BLANK","	* while this character is imaged as a fixed-width blank in many fonts, it does not act as a space"},
/* 2801 */ { "BRAILLE PATTERN DOTS-1",NULL},
/* 2802 */ { "BRAILLE PATTERN DOTS-2",NULL},
/* 2803 */ { "BRAILLE PATTERN DOTS-12",NULL},
/* 2804 */ { "BRAILLE PATTERN DOTS-3",NULL},
/* 2805 */ { "BRAILLE PATTERN DOTS-13",NULL},
/* 2806 */ { "BRAILLE PATTERN DOTS-23",NULL},
/* 2807 */ { "BRAILLE PATTERN DOTS-123",NULL},
/* 2808 */ { "BRAILLE PATTERN DOTS-4",NULL},
/* 2809 */ { "BRAILLE PATTERN DOTS-14",NULL},
/* 280A */ { "BRAILLE PATTERN DOTS-24",NULL},
/* 280B */ { "BRAILLE PATTERN DOTS-124",NULL},
/* 280C */ { "BRAILLE PATTERN DOTS-34",NULL},
/* 280D */ { "BRAILLE PATTERN DOTS-134",NULL},
/* 280E */ { "BRAILLE PATTERN DOTS-234",NULL},
/* 280F */ { "BRAILLE PATTERN DOTS-1234",NULL},
/* 2810 */ { "BRAILLE PATTERN DOTS-5",NULL},
/* 2811 */ { "BRAILLE PATTERN DOTS-15",NULL},
/* 2812 */ { "BRAILLE PATTERN DOTS-25",NULL},
/* 2813 */ { "BRAILLE PATTERN DOTS-125",NULL},
/* 2814 */ { "BRAILLE PATTERN DOTS-35",NULL},
/* 2815 */ { "BRAILLE PATTERN DOTS-135",NULL},
/* 2816 */ { "BRAILLE PATTERN DOTS-235",NULL},
/* 2817 */ { "BRAILLE PATTERN DOTS-1235",NULL},
/* 2818 */ { "BRAILLE PATTERN DOTS-45",NULL},
/* 2819 */ { "BRAILLE PATTERN DOTS-145",NULL},
/* 281A */ { "BRAILLE PATTERN DOTS-245",NULL},
/* 281B */ { "BRAILLE PATTERN DOTS-1245",NULL},
/* 281C */ { "BRAILLE PATTERN DOTS-345",NULL},
/* 281D */ { "BRAILLE PATTERN DOTS-1345",NULL},
/* 281E */ { "BRAILLE PATTERN DOTS-2345",NULL},
/* 281F */ { "BRAILLE PATTERN DOTS-12345",NULL},
/* 2820 */ { "BRAILLE PATTERN DOTS-6",NULL},
/* 2821 */ { "BRAILLE PATTERN DOTS-16",NULL},
/* 2822 */ { "BRAILLE PATTERN DOTS-26",NULL},
/* 2823 */ { "BRAILLE PATTERN DOTS-126",NULL},
/* 2824 */ { "BRAILLE PATTERN DOTS-36",NULL},
/* 2825 */ { "BRAILLE PATTERN DOTS-136",NULL},
/* 2826 */ { "BRAILLE PATTERN DOTS-236",NULL},
/* 2827 */ { "BRAILLE PATTERN DOTS-1236",NULL},
/* 2828 */ { "BRAILLE PATTERN DOTS-46",NULL},
/* 2829 */ { "BRAILLE PATTERN DOTS-146",NULL},
/* 282A */ { "BRAILLE PATTERN DOTS-246",NULL},
/* 282B */ { "BRAILLE PATTERN DOTS-1246",NULL},
/* 282C */ { "BRAILLE PATTERN DOTS-346",NULL},
/* 282D */ { "BRAILLE PATTERN DOTS-1346",NULL},
/* 282E */ { "BRAILLE PATTERN DOTS-2346",NULL},
/* 282F */ { "BRAILLE PATTERN DOTS-12346",NULL},
/* 2830 */ { "BRAILLE PATTERN DOTS-56",NULL},
/* 2831 */ { "BRAILLE PATTERN DOTS-156",NULL},
/* 2832 */ { "BRAILLE PATTERN DOTS-256",NULL},
/* 2833 */ { "BRAILLE PATTERN DOTS-1256",NULL},
/* 2834 */ { "BRAILLE PATTERN DOTS-356",NULL},
/* 2835 */ { "BRAILLE PATTERN DOTS-1356",NULL},
/* 2836 */ { "BRAILLE PATTERN DOTS-2356",NULL},
/* 2837 */ { "BRAILLE PATTERN DOTS-12356",NULL},
/* 2838 */ { "BRAILLE PATTERN DOTS-456",NULL},
/* 2839 */ { "BRAILLE PATTERN DOTS-1456",NULL},
/* 283A */ { "BRAILLE PATTERN DOTS-2456",NULL},
/* 283B */ { "BRAILLE PATTERN DOTS-12456",NULL},
/* 283C */ { "BRAILLE PATTERN DOTS-3456",NULL},
/* 283D */ { "BRAILLE PATTERN DOTS-13456",NULL},
/* 283E */ { "BRAILLE PATTERN DOTS-23456",NULL},
/* 283F */ { "BRAILLE PATTERN DOTS-123456",NULL},
/* 2840 */ { "BRAILLE PATTERN DOTS-7",NULL},
/* 2841 */ { "BRAILLE PATTERN DOTS-17",NULL},
/* 2842 */ { "BRAILLE PATTERN DOTS-27",NULL},
/* 2843 */ { "BRAILLE PATTERN DOTS-127",NULL},
/* 2844 */ { "BRAILLE PATTERN DOTS-37",NULL},
/* 2845 */ { "BRAILLE PATTERN DOTS-137",NULL},
/* 2846 */ { "BRAILLE PATTERN DOTS-237",NULL},
/* 2847 */ { "BRAILLE PATTERN DOTS-1237",NULL},
/* 2848 */ { "BRAILLE PATTERN DOTS-47",NULL},
/* 2849 */ { "BRAILLE PATTERN DOTS-147",NULL},
/* 284A */ { "BRAILLE PATTERN DOTS-247",NULL},
/* 284B */ { "BRAILLE PATTERN DOTS-1247",NULL},
/* 284C */ { "BRAILLE PATTERN DOTS-347",NULL},
/* 284D */ { "BRAILLE PATTERN DOTS-1347",NULL},
/* 284E */ { "BRAILLE PATTERN DOTS-2347",NULL},
/* 284F */ { "BRAILLE PATTERN DOTS-12347",NULL},
/* 2850 */ { "BRAILLE PATTERN DOTS-57",NULL},
/* 2851 */ { "BRAILLE PATTERN DOTS-157",NULL},
/* 2852 */ { "BRAILLE PATTERN DOTS-257",NULL},
/* 2853 */ { "BRAILLE PATTERN DOTS-1257",NULL},
/* 2854 */ { "BRAILLE PATTERN DOTS-357",NULL},
/* 2855 */ { "BRAILLE PATTERN DOTS-1357",NULL},
/* 2856 */ { "BRAILLE PATTERN DOTS-2357",NULL},
/* 2857 */ { "BRAILLE PATTERN DOTS-12357",NULL},
/* 2858 */ { "BRAILLE PATTERN DOTS-457",NULL},
/* 2859 */ { "BRAILLE PATTERN DOTS-1457",NULL},
/* 285A */ { "BRAILLE PATTERN DOTS-2457",NULL},
/* 285B */ { "BRAILLE PATTERN DOTS-12457",NULL},
/* 285C */ { "BRAILLE PATTERN DOTS-3457",NULL},
/* 285D */ { "BRAILLE PATTERN DOTS-13457",NULL},
/* 285E */ { "BRAILLE PATTERN DOTS-23457",NULL},
/* 285F */ { "BRAILLE PATTERN DOTS-123457",NULL},
/* 2860 */ { "BRAILLE PATTERN DOTS-67",NULL},
/* 2861 */ { "BRAILLE PATTERN DOTS-167",NULL},
/* 2862 */ { "BRAILLE PATTERN DOTS-267",NULL},
/* 2863 */ { "BRAILLE PATTERN DOTS-1267",NULL},
/* 2864 */ { "BRAILLE PATTERN DOTS-367",NULL},
/* 2865 */ { "BRAILLE PATTERN DOTS-1367",NULL},
/* 2866 */ { "BRAILLE PATTERN DOTS-2367",NULL},
/* 2867 */ { "BRAILLE PATTERN DOTS-12367",NULL},
/* 2868 */ { "BRAILLE PATTERN DOTS-467",NULL},
/* 2869 */ { "BRAILLE PATTERN DOTS-1467",NULL},
/* 286A */ { "BRAILLE PATTERN DOTS-2467",NULL},
/* 286B */ { "BRAILLE PATTERN DOTS-12467",NULL},
/* 286C */ { "BRAILLE PATTERN DOTS-3467",NULL},
/* 286D */ { "BRAILLE PATTERN DOTS-13467",NULL},
/* 286E */ { "BRAILLE PATTERN DOTS-23467",NULL},
/* 286F */ { "BRAILLE PATTERN DOTS-123467",NULL},
/* 2870 */ { "BRAILLE PATTERN DOTS-567",NULL},
/* 2871 */ { "BRAILLE PATTERN DOTS-1567",NULL},
/* 2872 */ { "BRAILLE PATTERN DOTS-2567",NULL},
/* 2873 */ { "BRAILLE PATTERN DOTS-12567",NULL},
/* 2874 */ { "BRAILLE PATTERN DOTS-3567",NULL},
/* 2875 */ { "BRAILLE PATTERN DOTS-13567",NULL},
/* 2876 */ { "BRAILLE PATTERN DOTS-23567",NULL},
/* 2877 */ { "BRAILLE PATTERN DOTS-123567",NULL},
/* 2878 */ { "BRAILLE PATTERN DOTS-4567",NULL},
/* 2879 */ { "BRAILLE PATTERN DOTS-14567",NULL},
/* 287A */ { "BRAILLE PATTERN DOTS-24567",NULL},
/* 287B */ { "BRAILLE PATTERN DOTS-124567",NULL},
/* 287C */ { "BRAILLE PATTERN DOTS-34567",NULL},
/* 287D */ { "BRAILLE PATTERN DOTS-134567",NULL},
/* 287E */ { "BRAILLE PATTERN DOTS-234567",NULL},
/* 287F */ { "BRAILLE PATTERN DOTS-1234567",NULL},
/* 2880 */ { "BRAILLE PATTERN DOTS-8",NULL},
/* 2881 */ { "BRAILLE PATTERN DOTS-18",NULL},
/* 2882 */ { "BRAILLE PATTERN DOTS-28",NULL},
/* 2883 */ { "BRAILLE PATTERN DOTS-128",NULL},
/* 2884 */ { "BRAILLE PATTERN DOTS-38",NULL},
/* 2885 */ { "BRAILLE PATTERN DOTS-138",NULL},
/* 2886 */ { "BRAILLE PATTERN DOTS-238",NULL},
/* 2887 */ { "BRAILLE PATTERN DOTS-1238",NULL},
/* 2888 */ { "BRAILLE PATTERN DOTS-48",NULL},
/* 2889 */ { "BRAILLE PATTERN DOTS-148",NULL},
/* 288A */ { "BRAILLE PATTERN DOTS-248",NULL},
/* 288B */ { "BRAILLE PATTERN DOTS-1248",NULL},
/* 288C */ { "BRAILLE PATTERN DOTS-348",NULL},
/* 288D */ { "BRAILLE PATTERN DOTS-1348",NULL},
/* 288E */ { "BRAILLE PATTERN DOTS-2348",NULL},
/* 288F */ { "BRAILLE PATTERN DOTS-12348",NULL},
/* 2890 */ { "BRAILLE PATTERN DOTS-58",NULL},
/* 2891 */ { "BRAILLE PATTERN DOTS-158",NULL},
/* 2892 */ { "BRAILLE PATTERN DOTS-258",NULL},
/* 2893 */ { "BRAILLE PATTERN DOTS-1258",NULL},
/* 2894 */ { "BRAILLE PATTERN DOTS-358",NULL},
/* 2895 */ { "BRAILLE PATTERN DOTS-1358",NULL},
/* 2896 */ { "BRAILLE PATTERN DOTS-2358",NULL},
/* 2897 */ { "BRAILLE PATTERN DOTS-12358",NULL},
/* 2898 */ { "BRAILLE PATTERN DOTS-458",NULL},
/* 2899 */ { "BRAILLE PATTERN DOTS-1458",NULL},
/* 289A */ { "BRAILLE PATTERN DOTS-2458",NULL},
/* 289B */ { "BRAILLE PATTERN DOTS-12458",NULL},
/* 289C */ { "BRAILLE PATTERN DOTS-3458",NULL},
/* 289D */ { "BRAILLE PATTERN DOTS-13458",NULL},
/* 289E */ { "BRAILLE PATTERN DOTS-23458",NULL},
/* 289F */ { "BRAILLE PATTERN DOTS-123458",NULL},
/* 28A0 */ { "BRAILLE PATTERN DOTS-68",NULL},
/* 28A1 */ { "BRAILLE PATTERN DOTS-168",NULL},
/* 28A2 */ { "BRAILLE PATTERN DOTS-268",NULL},
/* 28A3 */ { "BRAILLE PATTERN DOTS-1268",NULL},
/* 28A4 */ { "BRAILLE PATTERN DOTS-368",NULL},
/* 28A5 */ { "BRAILLE PATTERN DOTS-1368",NULL},
/* 28A6 */ { "BRAILLE PATTERN DOTS-2368",NULL},
/* 28A7 */ { "BRAILLE PATTERN DOTS-12368",NULL},
/* 28A8 */ { "BRAILLE PATTERN DOTS-468",NULL},
/* 28A9 */ { "BRAILLE PATTERN DOTS-1468",NULL},
/* 28AA */ { "BRAILLE PATTERN DOTS-2468",NULL},
/* 28AB */ { "BRAILLE PATTERN DOTS-12468",NULL},
/* 28AC */ { "BRAILLE PATTERN DOTS-3468",NULL},
/* 28AD */ { "BRAILLE PATTERN DOTS-13468",NULL},
/* 28AE */ { "BRAILLE PATTERN DOTS-23468",NULL},
/* 28AF */ { "BRAILLE PATTERN DOTS-123468",NULL},
/* 28B0 */ { "BRAILLE PATTERN DOTS-568",NULL},
/* 28B1 */ { "BRAILLE PATTERN DOTS-1568",NULL},
/* 28B2 */ { "BRAILLE PATTERN DOTS-2568",NULL},
/* 28B3 */ { "BRAILLE PATTERN DOTS-12568",NULL},
/* 28B4 */ { "BRAILLE PATTERN DOTS-3568",NULL},
/* 28B5 */ { "BRAILLE PATTERN DOTS-13568",NULL},
/* 28B6 */ { "BRAILLE PATTERN DOTS-23568",NULL},
/* 28B7 */ { "BRAILLE PATTERN DOTS-123568",NULL},
/* 28B8 */ { "BRAILLE PATTERN DOTS-4568",NULL},
/* 28B9 */ { "BRAILLE PATTERN DOTS-14568",NULL},
/* 28BA */ { "BRAILLE PATTERN DOTS-24568",NULL},
/* 28BB */ { "BRAILLE PATTERN DOTS-124568",NULL},
/* 28BC */ { "BRAILLE PATTERN DOTS-34568",NULL},
/* 28BD */ { "BRAILLE PATTERN DOTS-134568",NULL},
/* 28BE */ { "BRAILLE PATTERN DOTS-234568",NULL},
/* 28BF */ { "BRAILLE PATTERN DOTS-1234568",NULL},
/* 28C0 */ { "BRAILLE PATTERN DOTS-78",NULL},
/* 28C1 */ { "BRAILLE PATTERN DOTS-178",NULL},
/* 28C2 */ { "BRAILLE PATTERN DOTS-278",NULL},
/* 28C3 */ { "BRAILLE PATTERN DOTS-1278",NULL},
/* 28C4 */ { "BRAILLE PATTERN DOTS-378",NULL},
/* 28C5 */ { "BRAILLE PATTERN DOTS-1378",NULL},
/* 28C6 */ { "BRAILLE PATTERN DOTS-2378",NULL},
/* 28C7 */ { "BRAILLE PATTERN DOTS-12378",NULL},
/* 28C8 */ { "BRAILLE PATTERN DOTS-478",NULL},
/* 28C9 */ { "BRAILLE PATTERN DOTS-1478",NULL},
/* 28CA */ { "BRAILLE PATTERN DOTS-2478",NULL},
/* 28CB */ { "BRAILLE PATTERN DOTS-12478",NULL},
/* 28CC */ { "BRAILLE PATTERN DOTS-3478",NULL},
/* 28CD */ { "BRAILLE PATTERN DOTS-13478",NULL},
/* 28CE */ { "BRAILLE PATTERN DOTS-23478",NULL},
/* 28CF */ { "BRAILLE PATTERN DOTS-123478",NULL},
/* 28D0 */ { "BRAILLE PATTERN DOTS-578",NULL},
/* 28D1 */ { "BRAILLE PATTERN DOTS-1578",NULL},
/* 28D2 */ { "BRAILLE PATTERN DOTS-2578",NULL},
/* 28D3 */ { "BRAILLE PATTERN DOTS-12578",NULL},
/* 28D4 */ { "BRAILLE PATTERN DOTS-3578",NULL},
/* 28D5 */ { "BRAILLE PATTERN DOTS-13578",NULL},
/* 28D6 */ { "BRAILLE PATTERN DOTS-23578",NULL},
/* 28D7 */ { "BRAILLE PATTERN DOTS-123578",NULL},
/* 28D8 */ { "BRAILLE PATTERN DOTS-4578",NULL},
/* 28D9 */ { "BRAILLE PATTERN DOTS-14578",NULL},
/* 28DA */ { "BRAILLE PATTERN DOTS-24578",NULL},
/* 28DB */ { "BRAILLE PATTERN DOTS-124578",NULL},
/* 28DC */ { "BRAILLE PATTERN DOTS-34578",NULL},
/* 28DD */ { "BRAILLE PATTERN DOTS-134578",NULL},
/* 28DE */ { "BRAILLE PATTERN DOTS-234578",NULL},
/* 28DF */ { "BRAILLE PATTERN DOTS-1234578",NULL},
/* 28E0 */ { "BRAILLE PATTERN DOTS-678",NULL},
/* 28E1 */ { "BRAILLE PATTERN DOTS-1678",NULL},
/* 28E2 */ { "BRAILLE PATTERN DOTS-2678",NULL},
/* 28E3 */ { "BRAILLE PATTERN DOTS-12678",NULL},
/* 28E4 */ { "BRAILLE PATTERN DOTS-3678",NULL},
/* 28E5 */ { "BRAILLE PATTERN DOTS-13678",NULL},
/* 28E6 */ { "BRAILLE PATTERN DOTS-23678",NULL},
/* 28E7 */ { "BRAILLE PATTERN DOTS-123678",NULL},
/* 28E8 */ { "BRAILLE PATTERN DOTS-4678",NULL},
/* 28E9 */ { "BRAILLE PATTERN DOTS-14678",NULL},
/* 28EA */ { "BRAILLE PATTERN DOTS-24678",NULL},
/* 28EB */ { "BRAILLE PATTERN DOTS-124678",NULL},
/* 28EC */ { "BRAILLE PATTERN DOTS-34678",NULL},
/* 28ED */ { "BRAILLE PATTERN DOTS-134678",NULL},
/* 28EE */ { "BRAILLE PATTERN DOTS-234678",NULL},
/* 28EF */ { "BRAILLE PATTERN DOTS-1234678",NULL},
/* 28F0 */ { "BRAILLE PATTERN DOTS-5678",NULL},
/* 28F1 */ { "BRAILLE PATTERN DOTS-15678",NULL},
/* 28F2 */ { "BRAILLE PATTERN DOTS-25678",NULL},
/* 28F3 */ { "BRAILLE PATTERN DOTS-125678",NULL},
/* 28F4 */ { "BRAILLE PATTERN DOTS-35678",NULL},
/* 28F5 */ { "BRAILLE PATTERN DOTS-135678",NULL},
/* 28F6 */ { "BRAILLE PATTERN DOTS-235678",NULL},
/* 28F7 */ { "BRAILLE PATTERN DOTS-1235678",NULL},
/* 28F8 */ { "BRAILLE PATTERN DOTS-45678",NULL},
/* 28F9 */ { "BRAILLE PATTERN DOTS-145678",NULL},
/* 28FA */ { "BRAILLE PATTERN DOTS-245678",NULL},
/* 28FB */ { "BRAILLE PATTERN DOTS-1245678",NULL},
/* 28FC */ { "BRAILLE PATTERN DOTS-345678",NULL},
/* 28FD */ { "BRAILLE PATTERN DOTS-1345678",NULL},
/* 28FE */ { "BRAILLE PATTERN DOTS-2345678",NULL},
/* 28FF */ { "BRAILLE PATTERN DOTS-12345678",NULL}
};

UN_DLL_LOCAL
static const struct unicode_nameannot una_00_29[] = {
/* 2900 */ { "RIGHTWARDS TWO-HEADED ARROW WITH VERTICAL STROKE","	= z notation partial surjection\n"
	"	x (leftwards two-headed arrow with vertical stroke - 2B34)"},
/* 2901 */ { "RIGHTWARDS TWO-HEADED ARROW WITH DOUBLE VERTICAL STROKE","	= z notation finite surjection\n"
	"	x (leftwards two-headed arrow with double vertical stroke - 2B35)"},
/* 2902 */ { "LEFTWARDS DOUBLE ARROW WITH VERTICAL STROKE",NULL},
/* 2903 */ { "RIGHTWARDS DOUBLE ARROW WITH VERTICAL STROKE",NULL},
/* 2904 */ { "LEFT RIGHT DOUBLE ARROW WITH VERTICAL STROKE",NULL},
/* 2905 */ { "RIGHTWARDS TWO-HEADED ARROW FROM BAR","	= maps to\n"
	"	x (leftwards two-headed arrow from bar - 2B36)"},
/* 2906 */ { "LEFTWARDS DOUBLE ARROW FROM BAR","	= maps from\n"
	"	x (long leftwards arrow from bar - 27FB)"},
/* 2907 */ { "RIGHTWARDS DOUBLE ARROW FROM BAR","	= maps to\n"
	"	x (long rightwards arrow from bar - 27FC)"},
/* 2908 */ { "DOWNWARDS ARROW WITH HORIZONTAL STROKE",NULL},
/* 2909 */ { "UPWARDS ARROW WITH HORIZONTAL STROKE",NULL},
/* 290A */ { "UPWARDS TRIPLE ARROW","	x (upwards double arrow - 21D1)\n"
	"	x (upwards quadruple arrow - 27F0)"},
/* 290B */ { "DOWNWARDS TRIPLE ARROW","	x (downwards double arrow - 21D3)\n"
	"	x (downwards quadruple arrow - 27F1)"},
/* 290C */ { "LEFTWARDS DOUBLE DASH ARROW",NULL},
/* 290D */ { "RIGHTWARDS DOUBLE DASH ARROW",NULL},
/* 290E */ { "LEFTWARDS TRIPLE DASH ARROW",NULL},
/* 290F */ { "RIGHTWARDS TRIPLE DASH ARROW",NULL},
/* 2910 */ { "RIGHTWARDS TWO-HEADED TRIPLE DASH ARROW","	x (leftwards two-headed triple dash arrow - 2B37)"},
/* 2911 */ { "RIGHTWARDS ARROW WITH DOTTED STEM","	x (leftwards arrow with dotted stem - 2B38)"},
/* 2912 */ { "UPWARDS ARROW TO BAR",NULL},
/* 2913 */ { "DOWNWARDS ARROW TO BAR",NULL},
/* 2914 */ { "RIGHTWARDS ARROW WITH TAIL WITH VERTICAL STROKE","	= z notation partial injection\n"
	"	x (leftwards arrow with tail with vertical stroke - 2B39)"},
/* 2915 */ { "RIGHTWARDS ARROW WITH TAIL WITH DOUBLE VERTICAL STROKE","	= z notation finite injection\n"
	"	x (leftwards arrow with tail with double vertical stroke - 2B3A)"},
/* 2916 */ { "RIGHTWARDS TWO-HEADED ARROW WITH TAIL","	= bijective mapping\n"
	"	= z notation bijection\n"
	"	x (leftwards two-headed arrow with tail - 2B3B)"},
/* 2917 */ { "RIGHTWARDS TWO-HEADED ARROW WITH TAIL WITH VERTICAL STROKE","	= z notation surjective injection\n"
	"	x (leftwards two-headed arrow with tail with vertical stroke - 2B3C)"},
/* 2918 */ { "RIGHTWARDS TWO-HEADED ARROW WITH TAIL WITH DOUBLE VERTICAL STROKE","	= z notation finite surjective injection\n"
	"	x (leftwards two-headed arrow with tail with double vertical stroke - 2B3D)"},
/* 2919 */ { "LEFTWARDS ARROW-TAIL",NULL},
/* 291A */ { "RIGHTWARDS ARROW-TAIL",NULL},
/* 291B */ { "LEFTWARDS DOUBLE ARROW-TAIL",NULL},
/* 291C */ { "RIGHTWARDS DOUBLE ARROW-TAIL",NULL},
/* 291D */ { "LEFTWARDS ARROW TO BLACK DIAMOND",NULL},
/* 291E */ { "RIGHTWARDS ARROW TO BLACK DIAMOND",NULL},
/* 291F */ { "LEFTWARDS ARROW FROM BAR TO BLACK DIAMOND",NULL},
/* 2920 */ { "RIGHTWARDS ARROW FROM BAR TO BLACK DIAMOND",NULL},
/* 2921 */ { "NORTH WEST AND SOUTH EAST ARROW",NULL},
/* 2922 */ { "NORTH EAST AND SOUTH WEST ARROW",NULL},
/* 2923 */ { "NORTH WEST ARROW WITH HOOK",NULL},
/* 2924 */ { "NORTH EAST ARROW WITH HOOK",NULL},
/* 2925 */ { "SOUTH EAST ARROW WITH HOOK",NULL},
/* 2926 */ { "SOUTH WEST ARROW WITH HOOK",NULL},
/* 2927 */ { "NORTH WEST ARROW AND NORTH EAST ARROW",NULL},
/* 2928 */ { "NORTH EAST ARROW AND SOUTH EAST ARROW",NULL},
/* 2929 */ { "SOUTH EAST ARROW AND SOUTH WEST ARROW",NULL},
/* 292A */ { "SOUTH WEST ARROW AND NORTH WEST ARROW",NULL},
/* 292B */ { "RISING DIAGONAL CROSSING FALLING DIAGONAL",NULL},
/* 292C */ { "FALLING DIAGONAL CROSSING RISING DIAGONAL","	x (crossing lanes - 26CC)"},
/* 292D */ { "SOUTH EAST ARROW CROSSING NORTH EAST ARROW",NULL},
/* 292E */ { "NORTH EAST ARROW CROSSING SOUTH EAST ARROW",NULL},
/* 292F */ { "FALLING DIAGONAL CROSSING NORTH EAST ARROW",NULL},
/* 2930 */ { "RISING DIAGONAL CROSSING SOUTH EAST ARROW",NULL},
/* 2931 */ { "NORTH EAST ARROW CROSSING NORTH WEST ARROW",NULL},
/* 2932 */ { "NORTH WEST ARROW CROSSING NORTH EAST ARROW",NULL},
/* 2933 */ { "WAVE ARROW POINTING DIRECTLY RIGHT","	x (rightwards wave arrow - 219D)\n"
	"	x (wave arrow pointing directly left - 2B3F)"},
/* 2934 */ { "ARROW POINTING RIGHTWARDS THEN CURVING UPWARDS",NULL},
/* 2935 */ { "ARROW POINTING RIGHTWARDS THEN CURVING DOWNWARDS",NULL},
/* 2936 */ { "ARROW POINTING DOWNWARDS THEN CURVING LEFTWARDS",NULL},
/* 2937 */ { "ARROW POINTING DOWNWARDS THEN CURVING RIGHTWARDS",NULL},
/* 2938 */ { "RIGHT-SIDE ARC CLOCKWISE ARROW",NULL},
/* 2939 */ { "LEFT-SIDE ARC ANTICLOCKWISE ARROW",NULL},
/* 293A */ { "TOP ARC ANTICLOCKWISE ARROW",NULL},
/* 293B */ { "BOTTOM ARC ANTICLOCKWISE ARROW",NULL},
/* 293C */ { "TOP ARC CLOCKWISE ARROW WITH MINUS",NULL},
/* 293D */ { "TOP ARC ANTICLOCKWISE ARROW WITH PLUS",NULL},
/* 293E */ { "LOWER RIGHT SEMICIRCULAR CLOCKWISE ARROW",NULL},
/* 293F */ { "LOWER LEFT SEMICIRCULAR ANTICLOCKWISE ARROW",NULL},
/* 2940 */ { "ANTICLOCKWISE CLOSED CIRCLE ARROW","	x (combining anticlockwise ring overlay - 20DA)"},
/* 2941 */ { "CLOCKWISE CLOSED CIRCLE ARROW","	x (combining clockwise ring overlay - 20D9)"},
/* 2942 */ { "RIGHTWARDS ARROW ABOVE SHORT LEFTWARDS ARROW",NULL},
/* 2943 */ { "LEFTWARDS ARROW ABOVE SHORT RIGHTWARDS ARROW",NULL},
/* 2944 */ { "SHORT RIGHTWARDS ARROW ABOVE LEFTWARDS ARROW",NULL},
/* 2945 */ { "RIGHTWARDS ARROW WITH PLUS BELOW",NULL},
/* 2946 */ { "LEFTWARDS ARROW WITH PLUS BELOW",NULL},
/* 2947 */ { "RIGHTWARDS ARROW THROUGH X","	x (leftwards arrow through x - 2B3E)"},
/* 2948 */ { "LEFT RIGHT ARROW THROUGH SMALL CIRCLE",NULL},
/* 2949 */ { "UPWARDS TWO-HEADED ARROW FROM SMALL CIRCLE",NULL},
/* 294A */ { "LEFT BARB UP RIGHT BARB DOWN HARPOON",NULL},
/* 294B */ { "LEFT BARB DOWN RIGHT BARB UP HARPOON",NULL},
/* 294C */ { "UP BARB RIGHT DOWN BARB LEFT HARPOON",NULL},
/* 294D */ { "UP BARB LEFT DOWN BARB RIGHT HARPOON",NULL},
/* 294E */ { "LEFT BARB UP RIGHT BARB UP HARPOON",NULL},
/* 294F */ { "UP BARB RIGHT DOWN BARB RIGHT HARPOON",NULL},
/* 2950 */ { "LEFT BARB DOWN RIGHT BARB DOWN HARPOON",NULL},
/* 2951 */ { "UP BARB LEFT DOWN BARB LEFT HARPOON",NULL},
/* 2952 */ { "LEFTWARDS HARPOON WITH BARB UP TO BAR",NULL},
/* 2953 */ { "RIGHTWARDS HARPOON WITH BARB UP TO BAR",NULL},
/* 2954 */ { "UPWARDS HARPOON WITH BARB RIGHT TO BAR",NULL},
/* 2955 */ { "DOWNWARDS HARPOON WITH BARB RIGHT TO BAR",NULL},
/* 2956 */ { "LEFTWARDS HARPOON WITH BARB DOWN TO BAR",NULL},
/* 2957 */ { "RIGHTWARDS HARPOON WITH BARB DOWN TO BAR",NULL},
/* 2958 */ { "UPWARDS HARPOON WITH BARB LEFT TO BAR",NULL},
/* 2959 */ { "DOWNWARDS HARPOON WITH BARB LEFT TO BAR",NULL},
/* 295A */ { "LEFTWARDS HARPOON WITH BARB UP FROM BAR",NULL},
/* 295B */ { "RIGHTWARDS HARPOON WITH BARB UP FROM BAR",NULL},
/* 295C */ { "UPWARDS HARPOON WITH BARB RIGHT FROM BAR",NULL},
/* 295D */ { "DOWNWARDS HARPOON WITH BARB RIGHT FROM BAR",NULL},
/* 295E */ { "LEFTWARDS HARPOON WITH BARB DOWN FROM BAR",NULL},
/* 295F */ { "RIGHTWARDS HARPOON WITH BARB DOWN FROM BAR",NULL},
/* 2960 */ { "UPWARDS HARPOON WITH BARB LEFT FROM BAR",NULL},
/* 2961 */ { "DOWNWARDS HARPOON WITH BARB LEFT FROM BAR",NULL},
/* 2962 */ { "LEFTWARDS HARPOON WITH BARB UP ABOVE LEFTWARDS HARPOON WITH BARB DOWN",NULL},
/* 2963 */ { "UPWARDS HARPOON WITH BARB LEFT BESIDE UPWARDS HARPOON WITH BARB RIGHT",NULL},
/* 2964 */ { "RIGHTWARDS HARPOON WITH BARB UP ABOVE RIGHTWARDS HARPOON WITH BARB DOWN",NULL},
/* 2965 */ { "DOWNWARDS HARPOON WITH BARB LEFT BESIDE DOWNWARDS HARPOON WITH BARB RIGHT",NULL},
/* 2966 */ { "LEFTWARDS HARPOON WITH BARB UP ABOVE RIGHTWARDS HARPOON WITH BARB UP",NULL},
/* 2967 */ { "LEFTWARDS HARPOON WITH BARB DOWN ABOVE RIGHTWARDS HARPOON WITH BARB DOWN",NULL},
/* 2968 */ { "RIGHTWARDS HARPOON WITH BARB UP ABOVE LEFTWARDS HARPOON WITH BARB UP",NULL},
/* 2969 */ { "RIGHTWARDS HARPOON WITH BARB DOWN ABOVE LEFTWARDS HARPOON WITH BARB DOWN",NULL},
/* 296A */ { "LEFTWARDS HARPOON WITH BARB UP ABOVE LONG DASH",NULL},
/* 296B */ { "LEFTWARDS HARPOON WITH BARB DOWN BELOW LONG DASH",NULL},
/* 296C */ { "RIGHTWARDS HARPOON WITH BARB UP ABOVE LONG DASH",NULL},
/* 296D */ { "RIGHTWARDS HARPOON WITH BARB DOWN BELOW LONG DASH",NULL},
/* 296E */ { "UPWARDS HARPOON WITH BARB LEFT BESIDE DOWNWARDS HARPOON WITH BARB RIGHT",NULL},
/* 296F */ { "DOWNWARDS HARPOON WITH BARB LEFT BESIDE UPWARDS HARPOON WITH BARB RIGHT",NULL},
/* 2970 */ { "RIGHT DOUBLE ARROW WITH ROUNDED HEAD","	x (superset of - 2283)"},
/* 2971 */ { "EQUALS SIGN ABOVE RIGHTWARDS ARROW","	x (equals sign above leftwards arrow - 2B40)"},
/* 2972 */ { "TILDE OPERATOR ABOVE RIGHTWARDS ARROW","	x (reverse tilde operator above leftwards arrow - 2B41)"},
/* 2973 */ { "LEFTWARDS ARROW ABOVE TILDE OPERATOR","	x (rightwards arrow above reverse tilde operator - 2B4C)"},
/* 2974 */ { "RIGHTWARDS ARROW ABOVE TILDE OPERATOR","	x (leftwards arrow above reverse tilde operator - 2B4B)"},
/* 2975 */ { "RIGHTWARDS ARROW ABOVE ALMOST EQUAL TO","	x (leftwards arrow above reverse almost equal to - 2B42)"},
/* 2976 */ { "LESS-THAN ABOVE LEFTWARDS ARROW",NULL},
/* 2977 */ { "LEFTWARDS ARROW THROUGH LESS-THAN","	x (rightwards arrow through greater-than - 2B43)"},
/* 2978 */ { "GREATER-THAN ABOVE RIGHTWARDS ARROW",NULL},
/* 2979 */ { "SUBSET ABOVE RIGHTWARDS ARROW",NULL},
/* 297A */ { "LEFTWARDS ARROW THROUGH SUBSET","	x (rightwards arrow through superset - 2B44)"},
/* 297B */ { "SUPERSET ABOVE LEFTWARDS ARROW",NULL},
/* 297C */ { "LEFT FISH TAIL",NULL},
/* 297D */ { "RIGHT FISH TAIL",NULL},
/* 297E */ { "UP FISH TAIL",NULL},
/* 297F */ { "DOWN FISH TAIL",NULL},
/* 2980 */ { "TRIPLE VERTICAL BAR DELIMITER","	x (triple vertical bar binary relation - 2AF4)\n"
	"	x (large triple vertical bar operator - 2AFC)"},
/* 2981 */ { "Z NOTATION SPOT","	* medium-small-sized black circle\n"
	"	x (bullet operator - 2219)\n"
	"	x (black circle - 25CF)"},
/* 2982 */ { "Z NOTATION TYPE COLON","	x (tibetan sign rnam bcad - 0F7F)"},
/* 2983 */ { "LEFT WHITE CURLY BRACKET",NULL},
/* 2984 */ { "RIGHT WHITE CURLY BRACKET",NULL},
/* 2985 */ { "LEFT WHITE PARENTHESIS","	x (left double parenthesis - 2E28)\n"
	"	x (fullwidth left white parenthesis - FF5F)"},
/* 2986 */ { "RIGHT WHITE PARENTHESIS","	* used for Bourbakist intervals\n"
	"	x (right double parenthesis - 2E29)\n"
	"	x (fullwidth right white parenthesis - FF60)"},
/* 2987 */ { "Z NOTATION LEFT IMAGE BRACKET",NULL},
/* 2988 */ { "Z NOTATION RIGHT IMAGE BRACKET",NULL},
/* 2989 */ { "Z NOTATION LEFT BINDING BRACKET",NULL},
/* 298A */ { "Z NOTATION RIGHT BINDING BRACKET",NULL},
/* 298B */ { "LEFT SQUARE BRACKET WITH UNDERBAR",NULL},
/* 298C */ { "RIGHT SQUARE BRACKET WITH UNDERBAR",NULL},
/* 298D */ { "LEFT SQUARE BRACKET WITH TICK IN TOP CORNER",NULL},
/* 298E */ { "RIGHT SQUARE BRACKET WITH TICK IN BOTTOM CORNER",NULL},
/* 298F */ { "LEFT SQUARE BRACKET WITH TICK IN BOTTOM CORNER",NULL},
/* 2990 */ { "RIGHT SQUARE BRACKET WITH TICK IN TOP CORNER",NULL},
/* 2991 */ { "LEFT ANGLE BRACKET WITH DOT",NULL},
/* 2992 */ { "RIGHT ANGLE BRACKET WITH DOT",NULL},
/* 2993 */ { "LEFT ARC LESS-THAN BRACKET",NULL},
/* 2994 */ { "RIGHT ARC GREATER-THAN BRACKET",NULL},
/* 2995 */ { "DOUBLE LEFT ARC GREATER-THAN BRACKET",NULL},
/* 2996 */ { "DOUBLE RIGHT ARC LESS-THAN BRACKET",NULL},
/* 2997 */ { "LEFT BLACK TORTOISE SHELL BRACKET","	x (mathematical left white tortoise shell bracket - 27EC)\n"
	"	x (left tortoise shell bracket - 3014)\n"
	"	x (left white tortoise shell bracket - 3018)"},
/* 2998 */ { "RIGHT BLACK TORTOISE SHELL BRACKET",NULL},
/* 2999 */ { "DOTTED FENCE","	* four close dots vertical"},
/* 299A */ { "VERTICAL ZIGZAG LINE","	x (wavy line - 2307)"},
/* 299B */ { "MEASURED ANGLE OPENING LEFT","	x (measured angle - 2221)"},
/* 299C */ { "RIGHT ANGLE VARIANT WITH SQUARE","	x (right angle - 221F)"},
/* 299D */ { "MEASURED RIGHT ANGLE WITH DOT",NULL},
/* 299E */ { "ANGLE WITH S INSIDE",NULL},
/* 299F */ { "ACUTE ANGLE","	x (angle - 2220)"},
/* 29A0 */ { "SPHERICAL ANGLE OPENING LEFT","	x (spherical angle - 2222)"},
/* 29A1 */ { "SPHERICAL ANGLE OPENING UP",NULL},
/* 29A2 */ { "TURNED ANGLE","	x (angle - 2220)"},
/* 29A3 */ { "REVERSED ANGLE",NULL},
/* 29A4 */ { "ANGLE WITH UNDERBAR",NULL},
/* 29A5 */ { "REVERSED ANGLE WITH UNDERBAR",NULL},
/* 29A6 */ { "OBLIQUE ANGLE OPENING UP",NULL},
/* 29A7 */ { "OBLIQUE ANGLE OPENING DOWN",NULL},
/* 29A8 */ { "MEASURED ANGLE WITH OPEN ARM ENDING IN ARROW POINTING UP AND RIGHT",NULL},
/* 29A9 */ { "MEASURED ANGLE WITH OPEN ARM ENDING IN ARROW POINTING UP AND LEFT",NULL},
/* 29AA */ { "MEASURED ANGLE WITH OPEN ARM ENDING IN ARROW POINTING DOWN AND RIGHT",NULL},
/* 29AB */ { "MEASURED ANGLE WITH OPEN ARM ENDING IN ARROW POINTING DOWN AND LEFT",NULL},
/* 29AC */ { "MEASURED ANGLE WITH OPEN ARM ENDING IN ARROW POINTING RIGHT AND UP",NULL},
/* 29AD */ { "MEASURED ANGLE WITH OPEN ARM ENDING IN ARROW POINTING LEFT AND UP",NULL},
/* 29AE */ { "MEASURED ANGLE WITH OPEN ARM ENDING IN ARROW POINTING RIGHT AND DOWN",NULL},
/* 29AF */ { "MEASURED ANGLE WITH OPEN ARM ENDING IN ARROW POINTING LEFT AND DOWN",NULL},
/* 29B0 */ { "REVERSED EMPTY SET","	x (empty set - 2205)\n"
	"	x (apl functional symbol circle backslash - 2349)"},
/* 29B1 */ { "EMPTY SET WITH OVERBAR",NULL},
/* 29B2 */ { "EMPTY SET WITH SMALL CIRCLE ABOVE",NULL},
/* 29B3 */ { "EMPTY SET WITH RIGHT ARROW ABOVE",NULL},
/* 29B4 */ { "EMPTY SET WITH LEFT ARROW ABOVE",NULL},
/* 29B5 */ { "CIRCLE WITH HORIZONTAL BAR","	* used in superscripted form to mean standard state (chemistry)\n"
	"	x (circled minus - 2296)"},
/* 29B6 */ { "CIRCLED VERTICAL BAR","	x (alchemical symbol for nitre - 1F715)"},
/* 29B7 */ { "CIRCLED PARALLEL",NULL},
/* 29B8 */ { "CIRCLED REVERSE SOLIDUS",NULL},
/* 29B9 */ { "CIRCLED PERPENDICULAR",NULL},
/* 29BA */ { "CIRCLE DIVIDED BY HORIZONTAL BAR AND TOP HALF DIVIDED BY VERTICAL BAR",NULL},
/* 29BB */ { "CIRCLE WITH SUPERIMPOSED X","	x (circled times - 2297)"},
/* 29BC */ { "CIRCLED ANTICLOCKWISE-ROTATED DIVISION SIGN",NULL},
/* 29BD */ { "UP ARROW THROUGH CIRCLE",NULL},
/* 29BE */ { "CIRCLED WHITE BULLET","	x (circled ring operator - 229A)\n"
	"	x (apl functional symbol circle jot - 233E)\n"
	"	x (bullseye - 25CE)"},
/* 29BF */ { "CIRCLED BULLET","	* forms part of a graduated set of circles with enclosed black circle of different sizes\n"
	"	x (circled dot operator - 2299)\n"
	"	x (fisheye - 25C9)\n"
	"	x (n-ary circled dot operator - 2A00)"},
/* 29C0 */ { "CIRCLED LESS-THAN",NULL},
/* 29C1 */ { "CIRCLED GREATER-THAN",NULL},
/* 29C2 */ { "CIRCLE WITH SMALL CIRCLE TO THE RIGHT",NULL},
/* 29C3 */ { "CIRCLE WITH TWO HORIZONTAL STROKES TO THE RIGHT",NULL},
/* 29C4 */ { "SQUARED RISING DIAGONAL SLASH","	x (apl functional symbol quad slash - 2341)\n"
	"	x (masu mark - 303C)"},
/* 29C5 */ { "SQUARED FALLING DIAGONAL SLASH","	x (apl functional symbol quad backslash - 2342)"},
/* 29C6 */ { "SQUARED ASTERISK",NULL},
/* 29C7 */ { "SQUARED SMALL CIRCLE","	x (apl functional symbol quad jot - 233B)"},
/* 29C8 */ { "SQUARED SQUARE",NULL},
/* 29C9 */ { "TWO JOINED SQUARES","	x (overlapping white squares - 2BBA)"},
/* 29CA */ { "TRIANGLE WITH DOT ABOVE",NULL},
/* 29CB */ { "TRIANGLE WITH UNDERBAR",NULL},
/* 29CC */ { "S IN TRIANGLE",NULL},
/* 29CD */ { "TRIANGLE WITH SERIFS AT BOTTOM","	x (white up-pointing triangle - 25B3)"},
/* 29CE */ { "RIGHT TRIANGLE ABOVE LEFT TRIANGLE",NULL},
/* 29CF */ { "LEFT TRIANGLE BESIDE VERTICAL BAR",NULL},
/* 29D0 */ { "VERTICAL BAR BESIDE RIGHT TRIANGLE",NULL},
/* 29D1 */ { "BOWTIE WITH LEFT HALF BLACK","	x (bowtie - 22C8)"},
/* 29D2 */ { "BOWTIE WITH RIGHT HALF BLACK",NULL},
/* 29D3 */ { "BLACK BOWTIE","	x (left and right triangular half block - 1FB9B)"},
/* 29D4 */ { "TIMES WITH LEFT HALF BLACK","	x (left normal factor semidirect product - 22C9)"},
/* 29D5 */ { "TIMES WITH RIGHT HALF BLACK","	x (right normal factor semidirect product - 22CA)"},
/* 29D6 */ { "WHITE HOURGLASS","	= vertical bowtie\n"
	"	= white framus\n"
	"	x (hourglass - 231B)"},
/* 29D7 */ { "BLACK HOURGLASS","	x (upper and lower triangular half block - 1FB9A)"},
/* 29D8 */ { "LEFT WIGGLY FENCE","	x (presentation form for vertical wavy low line - FE34)"},
/* 29D9 */ { "RIGHT WIGGLY FENCE",NULL},
/* 29DA */ { "LEFT DOUBLE WIGGLY FENCE",NULL},
/* 29DB */ { "RIGHT DOUBLE WIGGLY FENCE",NULL},
/* 29DC */ { "INCOMPLETE INFINITY","	= ISOtech entity &iinfin;\n"
	"	x (infinity - 221E)"},
/* 29DD */ { "TIE OVER INFINITY",NULL},
/* 29DE */ { "INFINITY NEGATED WITH VERTICAL BAR",NULL},
/* 29DF */ { "DOUBLE-ENDED MULTIMAP","	x (multimap - 22B8)\n"
	"	x (alchemical symbol for arsenic - 1F73A)"},
/* 29E0 */ { "SQUARE WITH CONTOURED OUTLINE","	= D'Alembertian\n"
	"	x (lower right drop-shadowed white square - 274F)"},
/* 29E1 */ { "INCREASES AS","	x (normal subgroup of or equal to - 22B4)"},
/* 29E2 */ { "SHUFFLE PRODUCT","	x (z notation bag membership - 22FF)"},
/* 29E3 */ { "EQUALS SIGN AND SLANTED PARALLEL","	x (number sign - 0023)\n"
	"	x (equal and parallel to - 22D5)"},
/* 29E4 */ { "EQUALS SIGN AND SLANTED PARALLEL WITH TILDE ABOVE",NULL},
/* 29E5 */ { "IDENTICAL TO AND SLANTED PARALLEL","	x (triple horizontal bar with double vertical stroke - 2A68)"},
/* 29E6 */ { "GLEICH STARK","	= tautological equivalent"},
/* 29E7 */ { "THERMODYNAMIC","	= record mark\n"
	"	* vertical bar crossed by two horizontals\n"
	"	x (not equal to - 2260)\n"
	"	x (group mark - 2BD2)"},
/* 29E8 */ { "DOWN-POINTING TRIANGLE WITH LEFT HALF BLACK","	x (up-pointing triangle with left half black - 25ED)"},
/* 29E9 */ { "DOWN-POINTING TRIANGLE WITH RIGHT HALF BLACK","	x (up-pointing triangle with right half black - 25EE)"},
/* 29EA */ { "BLACK DIAMOND WITH DOWN ARROW",NULL},
/* 29EB */ { "BLACK LOZENGE","	x (lozenge - 25CA)"},
/* 29EC */ { "WHITE CIRCLE WITH DOWN ARROW",NULL},
/* 29ED */ { "BLACK CIRCLE WITH DOWN ARROW",NULL},
/* 29EE */ { "ERROR-BARRED WHITE SQUARE",NULL},
/* 29EF */ { "ERROR-BARRED BLACK SQUARE",NULL},
/* 29F0 */ { "ERROR-BARRED WHITE DIAMOND",NULL},
/* 29F1 */ { "ERROR-BARRED BLACK DIAMOND",NULL},
/* 29F2 */ { "ERROR-BARRED WHITE CIRCLE",NULL},
/* 29F3 */ { "ERROR-BARRED BLACK CIRCLE",NULL},
/* 29F4 */ { "RULE-DELAYED","	= colon right arrow"},
/* 29F5 */ { "REVERSE SOLIDUS OPERATOR","	x (reverse solidus - 005C)\n"
	"	x (set minus - 2216)\n"
	"	x (mathematical falling diagonal - 27CD)"},
/* 29F6 */ { "SOLIDUS WITH OVERBAR",NULL},
/* 29F7 */ { "REVERSE SOLIDUS WITH HORIZONTAL STROKE","	x (apl functional symbol backslash bar - 2340)"},
/* 29F8 */ { "BIG SOLIDUS","	x (division slash - 2215)"},
/* 29F9 */ { "BIG REVERSE SOLIDUS","	= z notation schema hiding\n"
	"	x (set minus - 2216)"},
/* 29FA */ { "DOUBLE PLUS",NULL},
/* 29FB */ { "TRIPLE PLUS",NULL},
/* 29FC */ { "LEFT-POINTING CURVED ANGLE BRACKET","	x (precedes - 227A)\n"
	"	x (left-pointing angle bracket - 2329)"},
/* 29FD */ { "RIGHT-POINTING CURVED ANGLE BRACKET","	x (succeeds - 227B)\n"
	"	x (right-pointing angle bracket - 232A)"},
/* 29FE */ { "TINY","	x (plus sign - 002B)"},
/* 29FF */ { "MINY","	x (minus sign - 2212)"}
};

UN_DLL_LOCAL
static const struct unicode_nameannot una_00_2A[] = {
/* 2A00 */ { "N-ARY CIRCLED DOT OPERATOR","	x (circled dot operator - 2299)\n"
	"	x (fisheye - 25C9)"},
/* 2A01 */ { "N-ARY CIRCLED PLUS OPERATOR","	x (circled plus - 2295)"},
/* 2A02 */ { "N-ARY CIRCLED TIMES OPERATOR","	x (circled times - 2297)\n"
	"	x (heavy circled saltire - 2B59)"},
/* 2A03 */ { "N-ARY UNION OPERATOR WITH DOT","	x (multiset multiplication - 228D)"},
/* 2A04 */ { "N-ARY UNION OPERATOR WITH PLUS","	x (multiset union - 228E)"},
/* 2A05 */ { "N-ARY SQUARE INTERSECTION OPERATOR","	x (square cap - 2293)"},
/* 2A06 */ { "N-ARY SQUARE UNION OPERATOR","	x (square cup - 2294)"},
/* 2A07 */ { "TWO LOGICAL AND OPERATOR","	= merge\n"
	"	x (two intersecting logical and - 2A55)"},
/* 2A08 */ { "TWO LOGICAL OR OPERATOR","	x (two intersecting logical or - 2A56)"},
/* 2A09 */ { "N-ARY TIMES OPERATOR","	x (multiplication sign - 00D7)"},
/* 2A0A */ { "MODULO TWO SUM","	x (n-ary summation - 2211)"},
/* 2A0B */ { "SUMMATION WITH INTEGRAL",NULL},
/* 2A0C */ { "QUADRUPLE INTEGRAL OPERATOR","	x (triple integral - 222D)\n"
	"	# 222B 222B 222B 222B"},
/* 2A0D */ { "FINITE PART INTEGRAL",NULL},
/* 2A0E */ { "INTEGRAL WITH DOUBLE STROKE",NULL},
/* 2A0F */ { "INTEGRAL AVERAGE WITH SLASH",NULL},
/* 2A10 */ { "CIRCULATION FUNCTION",NULL},
/* 2A11 */ { "ANTICLOCKWISE INTEGRATION",NULL},
/* 2A12 */ { "LINE INTEGRATION WITH RECTANGULAR PATH AROUND POLE",NULL},
/* 2A13 */ { "LINE INTEGRATION WITH SEMICIRCULAR PATH AROUND POLE",NULL},
/* 2A14 */ { "LINE INTEGRATION NOT INCLUDING THE POLE",NULL},
/* 2A15 */ { "INTEGRAL AROUND A POINT OPERATOR","	x (contour integral - 222E)"},
/* 2A16 */ { "QUATERNION INTEGRAL OPERATOR",NULL},
/* 2A17 */ { "INTEGRAL WITH LEFTWARDS ARROW WITH HOOK",NULL},
/* 2A18 */ { "INTEGRAL WITH TIMES SIGN",NULL},
/* 2A19 */ { "INTEGRAL WITH INTERSECTION",NULL},
/* 2A1A */ { "INTEGRAL WITH UNION",NULL},
/* 2A1B */ { "INTEGRAL WITH OVERBAR","	= upper integral"},
/* 2A1C */ { "INTEGRAL WITH UNDERBAR","	= lower integral"},
/* 2A1D */ { "JOIN","	= large bowtie\n"
	"	* relational database theory\n"
	"	x (bowtie - 22C8)\n"
	"	x (full outer join - 27D7)"},
/* 2A1E */ { "LARGE LEFT TRIANGLE OPERATOR","	* relational database theory\n"
	"	x (white left-pointing triangle - 25C1)"},
/* 2A1F */ { "Z NOTATION SCHEMA COMPOSITION","	x (z notation relational composition - 2A3E)"},
/* 2A20 */ { "Z NOTATION SCHEMA PIPING","	x (much greater-than - 226B)"},
/* 2A21 */ { "Z NOTATION SCHEMA PROJECTION","	x (upwards harpoon with barb rightwards - 21BE)"},
/* 2A22 */ { "PLUS SIGN WITH SMALL CIRCLE ABOVE",NULL},
/* 2A23 */ { "PLUS SIGN WITH CIRCUMFLEX ACCENT ABOVE",NULL},
/* 2A24 */ { "PLUS SIGN WITH TILDE ABOVE","	= positive difference or sum"},
/* 2A25 */ { "PLUS SIGN WITH DOT BELOW","	x (dot plus - 2214)"},
/* 2A26 */ { "PLUS SIGN WITH TILDE BELOW","	= sum or positive difference"},
/* 2A27 */ { "PLUS SIGN WITH SUBSCRIPT TWO","	= nim-addition"},
/* 2A28 */ { "PLUS SIGN WITH BLACK TRIANGLE",NULL},
/* 2A29 */ { "MINUS SIGN WITH COMMA ABOVE",NULL},
/* 2A2A */ { "MINUS SIGN WITH DOT BELOW","	x (dot minus - 2238)"},
/* 2A2B */ { "MINUS SIGN WITH FALLING DOTS",NULL},
/* 2A2C */ { "MINUS SIGN WITH RISING DOTS",NULL},
/* 2A2D */ { "PLUS SIGN IN LEFT HALF CIRCLE",NULL},
/* 2A2E */ { "PLUS SIGN IN RIGHT HALF CIRCLE",NULL},
/* 2A2F */ { "VECTOR OR CROSS PRODUCT","	x (multiplication sign - 00D7)"},
/* 2A30 */ { "MULTIPLICATION SIGN WITH DOT ABOVE",NULL},
/* 2A31 */ { "MULTIPLICATION SIGN WITH UNDERBAR",NULL},
/* 2A32 */ { "SEMIDIRECT PRODUCT WITH BOTTOM CLOSED",NULL},
/* 2A33 */ { "SMASH PRODUCT",NULL},
/* 2A34 */ { "MULTIPLICATION SIGN IN LEFT HALF CIRCLE",NULL},
/* 2A35 */ { "MULTIPLICATION SIGN IN RIGHT HALF CIRCLE",NULL},
/* 2A36 */ { "CIRCLED MULTIPLICATION SIGN WITH CIRCUMFLEX ACCENT",NULL},
/* 2A37 */ { "MULTIPLICATION SIGN IN DOUBLE CIRCLE",NULL},
/* 2A38 */ { "CIRCLED DIVISION SIGN",NULL},
/* 2A39 */ { "PLUS SIGN IN TRIANGLE",NULL},
/* 2A3A */ { "MINUS SIGN IN TRIANGLE",NULL},
/* 2A3B */ { "MULTIPLICATION SIGN IN TRIANGLE",NULL},
/* 2A3C */ { "INTERIOR PRODUCT","	x (right floor - 230B)\n"
	"	~ 2A3C FE00 tall variant with narrow foot"},
/* 2A3D */ { "RIGHTHAND INTERIOR PRODUCT","	x (left floor - 230A)\n"
	"	x (turned not sign - 2319)\n"
	"	~ 2A3D FE00 tall variant with narrow foot"},
/* 2A3E */ { "Z NOTATION RELATIONAL COMPOSITION","	x (z notation schema composition - 2A1F)"},
/* 2A3F */ { "AMALGAMATION OR COPRODUCT","	x (n-ary coproduct - 2210)"},
/* 2A40 */ { "INTERSECTION WITH DOT","	x (logical and - 2227)\n"
	"	x (and with dot - 27D1)"},
/* 2A41 */ { "UNION WITH MINUS SIGN","	= z notation bag subtraction\n"
	"	x (multiset union - 228E)"},
/* 2A42 */ { "UNION WITH OVERBAR",NULL},
/* 2A43 */ { "INTERSECTION WITH OVERBAR",NULL},
/* 2A44 */ { "INTERSECTION WITH LOGICAL AND",NULL},
/* 2A45 */ { "UNION WITH LOGICAL OR",NULL},
/* 2A46 */ { "UNION ABOVE INTERSECTION",NULL},
/* 2A47 */ { "INTERSECTION ABOVE UNION",NULL},
/* 2A48 */ { "UNION ABOVE BAR ABOVE INTERSECTION",NULL},
/* 2A49 */ { "INTERSECTION ABOVE BAR ABOVE UNION",NULL},
/* 2A4A */ { "UNION BESIDE AND JOINED WITH UNION",NULL},
/* 2A4B */ { "INTERSECTION BESIDE AND JOINED WITH INTERSECTION",NULL},
/* 2A4C */ { "CLOSED UNION WITH SERIFS","	x (union - 222A)"},
/* 2A4D */ { "CLOSED INTERSECTION WITH SERIFS","	x (intersection - 2229)"},
/* 2A4E */ { "DOUBLE SQUARE INTERSECTION",NULL},
/* 2A4F */ { "DOUBLE SQUARE UNION",NULL},
/* 2A50 */ { "CLOSED UNION WITH SERIFS AND SMASH PRODUCT",NULL},
/* 2A51 */ { "LOGICAL AND WITH DOT ABOVE",NULL},
/* 2A52 */ { "LOGICAL OR WITH DOT ABOVE",NULL},
/* 2A53 */ { "DOUBLE LOGICAL AND",NULL},
/* 2A54 */ { "DOUBLE LOGICAL OR",NULL},
/* 2A55 */ { "TWO INTERSECTING LOGICAL AND","	x (two logical and operator - 2A07)"},
/* 2A56 */ { "TWO INTERSECTING LOGICAL OR","	x (two logical or operator - 2A08)"},
/* 2A57 */ { "SLOPING LARGE OR",NULL},
/* 2A58 */ { "SLOPING LARGE AND",NULL},
/* 2A59 */ { "LOGICAL OR OVERLAPPING LOGICAL AND",NULL},
/* 2A5A */ { "LOGICAL AND WITH MIDDLE STEM",NULL},
/* 2A5B */ { "LOGICAL OR WITH MIDDLE STEM",NULL},
/* 2A5C */ { "LOGICAL AND WITH HORIZONTAL DASH",NULL},
/* 2A5D */ { "LOGICAL OR WITH HORIZONTAL DASH",NULL},
/* 2A5E */ { "LOGICAL AND WITH DOUBLE OVERBAR","	x (perspective - 2306)"},
/* 2A5F */ { "LOGICAL AND WITH UNDERBAR",NULL},
/* 2A60 */ { "LOGICAL AND WITH DOUBLE UNDERBAR","	x (estimates - 2259)"},
/* 2A61 */ { "SMALL VEE WITH UNDERBAR","	x (equiangular to - 225A)"},
/* 2A62 */ { "LOGICAL OR WITH DOUBLE OVERBAR",NULL},
/* 2A63 */ { "LOGICAL OR WITH DOUBLE UNDERBAR","	x (equiangular to - 225A)"},
/* 2A64 */ { "Z NOTATION DOMAIN ANTIRESTRICTION",NULL},
/* 2A65 */ { "Z NOTATION RANGE ANTIRESTRICTION","	x (conical taper - 2332)"},
/* 2A66 */ { "EQUALS SIGN WITH DOT BELOW","	x (approaches the limit - 2250)"},
/* 2A67 */ { "IDENTICAL WITH DOT ABOVE",NULL},
/* 2A68 */ { "TRIPLE HORIZONTAL BAR WITH DOUBLE VERTICAL STROKE","	= identical and parallel to\n"
	"	x (equal and parallel to - 22D5)\n"
	"	x (identical to and slanted parallel - 29E5)"},
/* 2A69 */ { "TRIPLE HORIZONTAL BAR WITH TRIPLE VERTICAL STROKE",NULL},
/* 2A6A */ { "TILDE OPERATOR WITH DOT ABOVE",NULL},
/* 2A6B */ { "TILDE OPERATOR WITH RISING DOTS","	x (homothetic - 223B)"},
/* 2A6C */ { "SIMILAR MINUS SIMILAR",NULL},
/* 2A6D */ { "CONGRUENT WITH DOT ABOVE","	x (approximately equal to - 2245)"},
/* 2A6E */ { "EQUALS WITH ASTERISK","	x (star equals - 225B)"},
/* 2A6F */ { "ALMOST EQUAL TO WITH CIRCUMFLEX ACCENT",NULL},
/* 2A70 */ { "APPROXIMATELY EQUAL OR EQUAL TO","	x (approximately equal to - 2245)"},
/* 2A71 */ { "EQUALS SIGN ABOVE PLUS SIGN","	* black stands slightly better (chess notation)"},
/* 2A72 */ { "PLUS SIGN ABOVE EQUALS SIGN","	* white stands slightly better (chess notation)"},
/* 2A73 */ { "EQUALS SIGN ABOVE TILDE OPERATOR",NULL},
/* 2A74 */ { "DOUBLE COLON EQUAL","	# 003A 003A 003D"},
/* 2A75 */ { "TWO CONSECUTIVE EQUALS SIGNS","	# 003D 003D"},
/* 2A76 */ { "THREE CONSECUTIVE EQUALS SIGNS","	# 003D 003D 003D"},
/* 2A77 */ { "EQUALS SIGN WITH TWO DOTS ABOVE AND TWO DOTS BELOW",NULL},
/* 2A78 */ { "EQUIVALENT WITH FOUR DOTS ABOVE",NULL},
/* 2A79 */ { "LESS-THAN WITH CIRCLE INSIDE",NULL},
/* 2A7A */ { "GREATER-THAN WITH CIRCLE INSIDE",NULL},
/* 2A7B */ { "LESS-THAN WITH QUESTION MARK ABOVE",NULL},
/* 2A7C */ { "GREATER-THAN WITH QUESTION MARK ABOVE",NULL},
/* 2A7D */ { "LESS-THAN OR SLANTED EQUAL TO","	x (less-than or equal to - 2264)"},
/* 2A7E */ { "GREATER-THAN OR SLANTED EQUAL TO","	x (greater-than or equal to - 2265)"},
/* 2A7F */ { "LESS-THAN OR SLANTED EQUAL TO WITH DOT INSIDE",NULL},
/* 2A80 */ { "GREATER-THAN OR SLANTED EQUAL TO WITH DOT INSIDE",NULL},
/* 2A81 */ { "LESS-THAN OR SLANTED EQUAL TO WITH DOT ABOVE",NULL},
/* 2A82 */ { "GREATER-THAN OR SLANTED EQUAL TO WITH DOT ABOVE",NULL},
/* 2A83 */ { "LESS-THAN OR SLANTED EQUAL TO WITH DOT ABOVE RIGHT",NULL},
/* 2A84 */ { "GREATER-THAN OR SLANTED EQUAL TO WITH DOT ABOVE LEFT",NULL},
/* 2A85 */ { "LESS-THAN OR APPROXIMATE",NULL},
/* 2A86 */ { "GREATER-THAN OR APPROXIMATE",NULL},
/* 2A87 */ { "LESS-THAN AND SINGLE-LINE NOT EQUAL TO","	x (less-than but not equal to - 2268)"},
/* 2A88 */ { "GREATER-THAN AND SINGLE-LINE NOT EQUAL TO","	x (greater-than but not equal to - 2269)"},
/* 2A89 */ { "LESS-THAN AND NOT APPROXIMATE",NULL},
/* 2A8A */ { "GREATER-THAN AND NOT APPROXIMATE",NULL},
/* 2A8B */ { "LESS-THAN ABOVE DOUBLE-LINE EQUAL ABOVE GREATER-THAN","	x (less-than equal to or greater-than - 22DA)"},
/* 2A8C */ { "GREATER-THAN ABOVE DOUBLE-LINE EQUAL ABOVE LESS-THAN","	x (greater-than equal to or less-than - 22DB)"},
/* 2A8D */ { "LESS-THAN ABOVE SIMILAR OR EQUAL",NULL},
/* 2A8E */ { "GREATER-THAN ABOVE SIMILAR OR EQUAL",NULL},
/* 2A8F */ { "LESS-THAN ABOVE SIMILAR ABOVE GREATER-THAN",NULL},
/* 2A90 */ { "GREATER-THAN ABOVE SIMILAR ABOVE LESS-THAN",NULL},
/* 2A91 */ { "LESS-THAN ABOVE GREATER-THAN ABOVE DOUBLE-LINE EQUAL",NULL},
/* 2A92 */ { "GREATER-THAN ABOVE LESS-THAN ABOVE DOUBLE-LINE EQUAL",NULL},
/* 2A93 */ { "LESS-THAN ABOVE SLANTED EQUAL ABOVE GREATER-THAN ABOVE SLANTED EQUAL",NULL},
/* 2A94 */ { "GREATER-THAN ABOVE SLANTED EQUAL ABOVE LESS-THAN ABOVE SLANTED EQUAL",NULL},
/* 2A95 */ { "SLANTED EQUAL TO OR LESS-THAN","	x (equal to or less-than - 22DC)"},
/* 2A96 */ { "SLANTED EQUAL TO OR GREATER-THAN","	x (equal to or greater-than - 22DD)"},
/* 2A97 */ { "SLANTED EQUAL TO OR LESS-THAN WITH DOT INSIDE",NULL},
/* 2A98 */ { "SLANTED EQUAL TO OR GREATER-THAN WITH DOT INSIDE",NULL},
/* 2A99 */ { "DOUBLE-LINE EQUAL TO OR LESS-THAN","	x (equal to or less-than - 22DC)"},
/* 2A9A */ { "DOUBLE-LINE EQUAL TO OR GREATER-THAN","	x (equal to or greater-than - 22DD)"},
/* 2A9B */ { "DOUBLE-LINE SLANTED EQUAL TO OR LESS-THAN",NULL},
/* 2A9C */ { "DOUBLE-LINE SLANTED EQUAL TO OR GREATER-THAN",NULL},
/* 2A9D */ { "SIMILAR OR LESS-THAN","	~ 2A9D FE00 with similar following the slant of the upper leg"},
/* 2A9E */ { "SIMILAR OR GREATER-THAN","	~ 2A9E FE00 with similar following the slant of the upper leg"},
/* 2A9F */ { "SIMILAR ABOVE LESS-THAN ABOVE EQUALS SIGN",NULL},
/* 2AA0 */ { "SIMILAR ABOVE GREATER-THAN ABOVE EQUALS SIGN",NULL},
/* 2AA1 */ { "DOUBLE NESTED LESS-THAN","	= absolute continuity\n"
	"	x (much less-than - 226A)"},
/* 2AA2 */ { "DOUBLE NESTED GREATER-THAN","	x (much greater-than - 226B)"},
/* 2AA3 */ { "DOUBLE NESTED LESS-THAN WITH UNDERBAR",NULL},
/* 2AA4 */ { "GREATER-THAN OVERLAPPING LESS-THAN",NULL},
/* 2AA5 */ { "GREATER-THAN BESIDE LESS-THAN",NULL},
/* 2AA6 */ { "LESS-THAN CLOSED BY CURVE",NULL},
/* 2AA7 */ { "GREATER-THAN CLOSED BY CURVE",NULL},
/* 2AA8 */ { "LESS-THAN CLOSED BY CURVE ABOVE SLANTED EQUAL",NULL},
/* 2AA9 */ { "GREATER-THAN CLOSED BY CURVE ABOVE SLANTED EQUAL",NULL},
/* 2AAA */ { "SMALLER THAN",NULL},
/* 2AAB */ { "LARGER THAN",NULL},
/* 2AAC */ { "SMALLER THAN OR EQUAL TO","	~ 2AAC FE00 with slanted equal"},
/* 2AAD */ { "LARGER THAN OR EQUAL TO","	~ 2AAD FE00 with slanted equal"},
/* 2AAE */ { "EQUALS SIGN WITH BUMPY ABOVE","	x (difference between - 224F)"},
/* 2AAF */ { "PRECEDES ABOVE SINGLE-LINE EQUALS SIGN","	x (precedes or equal to - 227C)"},
/* 2AB0 */ { "SUCCEEDS ABOVE SINGLE-LINE EQUALS SIGN","	x (succeeds or equal to - 227D)"},
/* 2AB1 */ { "PRECEDES ABOVE SINGLE-LINE NOT EQUAL TO",NULL},
/* 2AB2 */ { "SUCCEEDS ABOVE SINGLE-LINE NOT EQUAL TO",NULL},
/* 2AB3 */ { "PRECEDES ABOVE EQUALS SIGN",NULL},
/* 2AB4 */ { "SUCCEEDS ABOVE EQUALS SIGN",NULL},
/* 2AB5 */ { "PRECEDES ABOVE NOT EQUAL TO",NULL},
/* 2AB6 */ { "SUCCEEDS ABOVE NOT EQUAL TO",NULL},
/* 2AB7 */ { "PRECEDES ABOVE ALMOST EQUAL TO",NULL},
/* 2AB8 */ { "SUCCEEDS ABOVE ALMOST EQUAL TO",NULL},
/* 2AB9 */ { "PRECEDES ABOVE NOT ALMOST EQUAL TO",NULL},
/* 2ABA */ { "SUCCEEDS ABOVE NOT ALMOST EQUAL TO",NULL},
/* 2ABB */ { "DOUBLE PRECEDES",NULL},
/* 2ABC */ { "DOUBLE SUCCEEDS",NULL},
/* 2ABD */ { "SUBSET WITH DOT",NULL},
/* 2ABE */ { "SUPERSET WITH DOT",NULL},
/* 2ABF */ { "SUBSET WITH PLUS SIGN BELOW",NULL},
/* 2AC0 */ { "SUPERSET WITH PLUS SIGN BELOW",NULL},
/* 2AC1 */ { "SUBSET WITH MULTIPLICATION SIGN BELOW",NULL},
/* 2AC2 */ { "SUPERSET WITH MULTIPLICATION SIGN BELOW",NULL},
/* 2AC3 */ { "SUBSET OF OR EQUAL TO WITH DOT ABOVE",NULL},
/* 2AC4 */ { "SUPERSET OF OR EQUAL TO WITH DOT ABOVE",NULL},
/* 2AC5 */ { "SUBSET OF ABOVE EQUALS SIGN",NULL},
/* 2AC6 */ { "SUPERSET OF ABOVE EQUALS SIGN",NULL},
/* 2AC7 */ { "SUBSET OF ABOVE TILDE OPERATOR",NULL},
/* 2AC8 */ { "SUPERSET OF ABOVE TILDE OPERATOR",NULL},
/* 2AC9 */ { "SUBSET OF ABOVE ALMOST EQUAL TO",NULL},
/* 2ACA */ { "SUPERSET OF ABOVE ALMOST EQUAL TO",NULL},
/* 2ACB */ { "SUBSET OF ABOVE NOT EQUAL TO","	~ 2ACB FE00 with stroke through bottom members"},
/* 2ACC */ { "SUPERSET OF ABOVE NOT EQUAL TO","	~ 2ACC FE00 with stroke through bottom members"},
/* 2ACD */ { "SQUARE LEFT OPEN BOX OPERATOR",NULL},
/* 2ACE */ { "SQUARE RIGHT OPEN BOX OPERATOR",NULL},
/* 2ACF */ { "CLOSED SUBSET","	x (subset of - 2282)"},
/* 2AD0 */ { "CLOSED SUPERSET","	x (superset of - 2283)"},
/* 2AD1 */ { "CLOSED SUBSET OR EQUAL TO",NULL},
/* 2AD2 */ { "CLOSED SUPERSET OR EQUAL TO",NULL},
/* 2AD3 */ { "SUBSET ABOVE SUPERSET",NULL},
/* 2AD4 */ { "SUPERSET ABOVE SUBSET",NULL},
/* 2AD5 */ { "SUBSET ABOVE SUBSET",NULL},
/* 2AD6 */ { "SUPERSET ABOVE SUPERSET",NULL},
/* 2AD7 */ { "SUPERSET BESIDE SUBSET",NULL},
/* 2AD8 */ { "SUPERSET BESIDE AND JOINED BY DASH WITH SUBSET",NULL},
/* 2AD9 */ { "ELEMENT OF OPENING DOWNWARDS","	x (element of - 2208)\n"
	"	x (element of opening upwards - 27D2)"},
/* 2ADA */ { "PITCHFORK WITH TEE TOP","	x (pitchfork - 22D4)"},
/* 2ADB */ { "TRANSVERSAL INTERSECTION","	x (pitchfork - 22D4)"},
/* 2ADC */ { "FORKING","	= not independent\n"
	"	* an equational logic symbol, not a computing science symbol\n"
	"	* non-independence (original concept) is related to forking\n"
	"	: 2ADD 0338"},
/* 2ADD */ { "NONFORKING","	= independent\n"
	"	* an equational logic symbol, not a computing science symbol\n"
	"	* independence (original concept) is related to non-forking"},
/* 2ADE */ { "SHORT LEFT TACK","	x (left tack - 22A3)"},
/* 2ADF */ { "SHORT DOWN TACK","	x (down tack - 22A4)"},
/* 2AE0 */ { "SHORT UP TACK","	x (up tack - 22A5)"},
/* 2AE1 */ { "PERPENDICULAR WITH S",NULL},
/* 2AE2 */ { "VERTICAL BAR TRIPLE RIGHT TURNSTILE","	= ordinarily satisfies"},
/* 2AE3 */ { "DOUBLE VERTICAL BAR LEFT TURNSTILE","	x (forces - 22A9)"},
/* 2AE4 */ { "VERTICAL BAR DOUBLE LEFT TURNSTILE","	x (true - 22A8)"},
/* 2AE5 */ { "DOUBLE VERTICAL BAR DOUBLE LEFT TURNSTILE",NULL},
/* 2AE6 */ { "LONG DASH FROM LEFT MEMBER OF DOUBLE VERTICAL","	x (forces - 22A9)"},
/* 2AE7 */ { "SHORT DOWN TACK WITH OVERBAR","	x (down tack - 22A4)\n"
	"	x (apl functional symbol up tack overbar - 2351)\n"
	"	x (postal mark - 3012)"},
/* 2AE8 */ { "SHORT UP TACK WITH UNDERBAR","	x (up tack - 22A5)\n"
	"	x (apl functional symbol down tack underbar - 234A)"},
/* 2AE9 */ { "SHORT UP TACK ABOVE SHORT DOWN TACK",NULL},
/* 2AEA */ { "DOUBLE DOWN TACK",NULL},
/* 2AEB */ { "DOUBLE UP TACK","	= independence\n"
	"	* probability theory"},
/* 2AEC */ { "DOUBLE STROKE NOT SIGN","	x (not sign - 00AC)"},
/* 2AED */ { "REVERSED DOUBLE STROKE NOT SIGN","	x (reversed not sign - 2310)"},
/* 2AEE */ { "DOES NOT DIVIDE WITH REVERSED NEGATION SLASH","	x (does not divide - 2224)"},
/* 2AEF */ { "VERTICAL LINE WITH CIRCLE ABOVE",NULL},
/* 2AF0 */ { "VERTICAL LINE WITH CIRCLE BELOW",NULL},
/* 2AF1 */ { "DOWN TACK WITH CIRCLE BELOW","	= necessarily satisfies\n"
	"	x (up tack with circle above - 27DF)"},
/* 2AF2 */ { "PARALLEL WITH HORIZONTAL STROKE","	x (not parallel to - 2226)\n"
	"	x (vertical bar with horizontal stroke - 27CA)"},
/* 2AF3 */ { "PARALLEL WITH TILDE OPERATOR",NULL},
/* 2AF4 */ { "TRIPLE VERTICAL BAR BINARY RELATION","	= interleave\n"
	"	x (triple vertical bar delimiter - 2980)"},
/* 2AF5 */ { "TRIPLE VERTICAL BAR WITH HORIZONTAL STROKE","	x (vertical bar with horizontal stroke - 27CA)"},
/* 2AF6 */ { "TRIPLE COLON OPERATOR","	* logic\n"
	"	x (tricolon - 205D)\n"
	"	x (vertical ellipsis - 22EE)"},
/* 2AF7 */ { "TRIPLE NESTED LESS-THAN","	x (very much less-than - 22D8)"},
/* 2AF8 */ { "TRIPLE NESTED GREATER-THAN","	x (very much greater-than - 22D9)"},
/* 2AF9 */ { "DOUBLE-LINE SLANTED LESS-THAN OR EQUAL TO","	x (less-than over equal to - 2266)"},
/* 2AFA */ { "DOUBLE-LINE SLANTED GREATER-THAN OR EQUAL TO","	x (greater-than over equal to - 2267)"},
/* 2AFB */ { "TRIPLE SOLIDUS BINARY RELATION","	x (triple vertical bar binary relation - 2AF4)"},
/* 2AFC */ { "LARGE TRIPLE VERTICAL BAR OPERATOR","	* often n-ary\n"
	"	x (triple vertical bar binary relation - 2AF4)\n"
	"	x (triple vertical bar delimiter - 2980)"},
/* 2AFD */ { "DOUBLE SOLIDUS OPERATOR","	x (parallel to - 2225)"},
/* 2AFE */ { "WHITE VERTICAL BAR","	= Dijkstra choice"},
/* 2AFF */ { "N-ARY WHITE VERTICAL BAR","	= n-ary Dijkstra choice"}
};

UN_DLL_LOCAL
static const struct unicode_nameannot una_00_2B[] = {
/* 2B00 */ { "NORTH EAST WHITE ARROW",NULL},
/* 2B01 */ { "NORTH WEST WHITE ARROW",NULL},
/* 2B02 */ { "SOUTH EAST WHITE ARROW",NULL},
/* 2B03 */ { "SOUTH WEST WHITE ARROW",NULL},
/* 2B04 */ { "LEFT RIGHT WHITE ARROW","	x (leftwards white arrow - 21E6)\n"
	"	x (up down white arrow - 21F3)"},
/* 2B05 */ { "LEFTWARDS BLACK ARROW","	x (black rightwards arrow - 27A1)\n"
	"	x (rightwards black arrow - 2B95)"},
/* 2B06 */ { "UPWARDS BLACK ARROW",NULL},
/* 2B07 */ { "DOWNWARDS BLACK ARROW",NULL},
/* 2B08 */ { "NORTH EAST BLACK ARROW",NULL},
/* 2B09 */ { "NORTH WEST BLACK ARROW",NULL},
/* 2B0A */ { "SOUTH EAST BLACK ARROW",NULL},
/* 2B0B */ { "SOUTH WEST BLACK ARROW",NULL},
/* 2B0C */ { "LEFT RIGHT BLACK ARROW",NULL},
/* 2B0D */ { "UP DOWN BLACK ARROW",NULL},
/* 2B0E */ { "RIGHTWARDS ARROW WITH TIP DOWNWARDS",NULL},
/* 2B0F */ { "RIGHTWARDS ARROW WITH TIP UPWARDS",NULL},
/* 2B10 */ { "LEFTWARDS ARROW WITH TIP DOWNWARDS",NULL},
/* 2B11 */ { "LEFTWARDS ARROW WITH TIP UPWARDS",NULL},
/* 2B12 */ { "SQUARE WITH TOP HALF BLACK","	x (square with left half black - 25E7)"},
/* 2B13 */ { "SQUARE WITH BOTTOM HALF BLACK",NULL},
/* 2B14 */ { "SQUARE WITH UPPER RIGHT DIAGONAL HALF BLACK",NULL},
/* 2B15 */ { "SQUARE WITH LOWER LEFT DIAGONAL HALF BLACK",NULL},
/* 2B16 */ { "DIAMOND WITH LEFT HALF BLACK",NULL},
/* 2B17 */ { "DIAMOND WITH RIGHT HALF BLACK",NULL},
/* 2B18 */ { "DIAMOND WITH TOP HALF BLACK",NULL},
/* 2B19 */ { "DIAMOND WITH BOTTOM HALF BLACK",NULL},
/* 2B1A */ { "DOTTED SQUARE",NULL},
/* 2B1B */ { "BLACK LARGE SQUARE","	x (black square - 25A0)"},
/* 2B1C */ { "WHITE LARGE SQUARE","	x (white square - 25A1)"},
/* 2B1D */ { "BLACK VERY SMALL SQUARE","	x (black small square - 25AA)"},
/* 2B1E */ { "WHITE VERY SMALL SQUARE","	x (white small square - 25AB)"},
/* 2B1F */ { "BLACK PENTAGON",NULL},
/* 2B20 */ { "WHITE PENTAGON",NULL},
/* 2B21 */ { "WHITE HEXAGON","	x (software-function symbol - 2394)"},
/* 2B22 */ { "BLACK HEXAGON",NULL},
/* 2B23 */ { "HORIZONTAL BLACK HEXAGON",NULL},
/* 2B24 */ { "BLACK LARGE CIRCLE","	x (black circle - 25CF)\n"
	"	x (large circle - 25EF)\n"
	"	x (large red circle - 1F534)"},
/* 2B25 */ { "BLACK MEDIUM DIAMOND","	x (black diamond - 25C6)"},
/* 2B26 */ { "WHITE MEDIUM DIAMOND","	x (white diamond - 25C7)\n"
	"	x (alchemical symbol for soap - 1F754)"},
/* 2B27 */ { "BLACK MEDIUM LOZENGE",NULL},
/* 2B28 */ { "WHITE MEDIUM LOZENGE","	x (lozenge - 25CA)"},
/* 2B29 */ { "BLACK SMALL DIAMOND","	x (diamond operator - 22C4)\n"
	"	x (small orange diamond - 1F538)"},
/* 2B2A */ { "BLACK SMALL LOZENGE",NULL},
/* 2B2B */ { "WHITE SMALL LOZENGE",NULL},
/* 2B2C */ { "BLACK HORIZONTAL ELLIPSE",NULL},
/* 2B2D */ { "WHITE HORIZONTAL ELLIPSE",NULL},
/* 2B2E */ { "BLACK VERTICAL ELLIPSE",NULL},
/* 2B2F */ { "WHITE VERTICAL ELLIPSE",NULL},
/* 2B30 */ { "LEFT ARROW WITH SMALL CIRCLE","	x (right arrow with small circle - 21F4)"},
/* 2B31 */ { "THREE LEFTWARDS ARROWS","	x (three rightwards arrows - 21F6)"},
/* 2B32 */ { "LEFT ARROW WITH CIRCLED PLUS","	x (right arrow with circled plus - 27F4)"},
/* 2B33 */ { "LONG LEFTWARDS SQUIGGLE ARROW","	x (long rightwards squiggle arrow - 27FF)\n"
	"	x (leftwards squiggle arrow - 21DC)"},
/* 2B34 */ { "LEFTWARDS TWO-HEADED ARROW WITH VERTICAL STROKE","	x (rightwards two-headed arrow with vertical stroke - 2900)"},
/* 2B35 */ { "LEFTWARDS TWO-HEADED ARROW WITH DOUBLE VERTICAL STROKE","	x (rightwards two-headed arrow with double vertical stroke - 2901)"},
/* 2B36 */ { "LEFTWARDS TWO-HEADED ARROW FROM BAR","	x (rightwards two-headed arrow from bar - 2905)"},
/* 2B37 */ { "LEFTWARDS TWO-HEADED TRIPLE DASH ARROW","	x (rightwards two-headed triple dash arrow - 2910)"},
/* 2B38 */ { "LEFTWARDS ARROW WITH DOTTED STEM","	x (rightwards arrow with dotted stem - 2911)"},
/* 2B39 */ { "LEFTWARDS ARROW WITH TAIL WITH VERTICAL STROKE","	x (rightwards arrow with tail with vertical stroke - 2914)"},
/* 2B3A */ { "LEFTWARDS ARROW WITH TAIL WITH DOUBLE VERTICAL STROKE","	x (rightwards arrow with tail with double vertical stroke - 2915)"},
/* 2B3B */ { "LEFTWARDS TWO-HEADED ARROW WITH TAIL","	x (rightwards two-headed arrow with tail - 2916)"},
/* 2B3C */ { "LEFTWARDS TWO-HEADED ARROW WITH TAIL WITH VERTICAL STROKE","	x (rightwards two-headed arrow with tail with vertical stroke - 2917)"},
/* 2B3D */ { "LEFTWARDS TWO-HEADED ARROW WITH TAIL WITH DOUBLE VERTICAL STROKE","	x (rightwards two-headed arrow with tail with double vertical stroke - 2918)"},
/* 2B3E */ { "LEFTWARDS ARROW THROUGH X","	x (rightwards arrow through x - 2947)"},
/* 2B3F */ { "WAVE ARROW POINTING DIRECTLY LEFT","	x (wave arrow pointing directly right - 2933)\n"
	"	x (leftwards wave arrow - 219C)"},
/* 2B40 */ { "EQUALS SIGN ABOVE LEFTWARDS ARROW","	x (equals sign above rightwards arrow - 2971)"},
/* 2B41 */ { "REVERSE TILDE OPERATOR ABOVE LEFTWARDS ARROW","	* mirror image of \"2972\"\n"
	"	x (tilde operator above rightwards arrow - 2972)"},
/* 2B42 */ { "LEFTWARDS ARROW ABOVE REVERSE ALMOST EQUAL TO","	* mirror image of \"2975\"\n"
	"	x (rightwards arrow above almost equal to - 2975)"},
/* 2B43 */ { "RIGHTWARDS ARROW THROUGH GREATER-THAN","	* mirror image of \"2977\"\n"
	"	x (leftwards arrow through less-than - 2977)"},
/* 2B44 */ { "RIGHTWARDS ARROW THROUGH SUPERSET","	* mirror image of \"297A\"\n"
	"	x (leftwards arrow through subset - 297A)"},
/* 2B45 */ { "LEFTWARDS QUADRUPLE ARROW","	x (upwards quadruple arrow - 27F0)"},
/* 2B46 */ { "RIGHTWARDS QUADRUPLE ARROW",NULL},
/* 2B47 */ { "REVERSE TILDE OPERATOR ABOVE RIGHTWARDS ARROW",NULL},
/* 2B48 */ { "RIGHTWARDS ARROW ABOVE REVERSE ALMOST EQUAL TO",NULL},
/* 2B49 */ { "TILDE OPERATOR ABOVE LEFTWARDS ARROW",NULL},
/* 2B4A */ { "LEFTWARDS ARROW ABOVE ALMOST EQUAL TO",NULL},
/* 2B4B */ { "LEFTWARDS ARROW ABOVE REVERSE TILDE OPERATOR","	* mirror image of \"2974\"\n"
	"	x (rightwards arrow above tilde operator - 2974)"},
/* 2B4C */ { "RIGHTWARDS ARROW ABOVE REVERSE TILDE OPERATOR","	* mirror image of \"2973\"\n"
	"	x (leftwards arrow above tilde operator - 2973)"},
/* 2B4D */ { "DOWNWARDS TRIANGLE-HEADED ZIGZAG ARROW","	x (downwards zigzag arrow - 21AF)"},
/* 2B4E */ { "SHORT SLANTED NORTH ARROW","	* slight rise in tone"},
/* 2B4F */ { "SHORT BACKSLANTED SOUTH ARROW","	* slight fall or overall fall in tone when at the end of a word or at the beginning of a phrase, respectively"},
/* 2B50 */ { "WHITE MEDIUM STAR","	x (white star - 2606)"},
/* 2B51 */ { "BLACK SMALL STAR","	x (arabic five pointed star - 066D)\n"
	"	x (star operator - 22C6)\n"
	"	x (black star - 2605)"},
/* 2B52 */ { "WHITE SMALL STAR",NULL},
/* 2B53 */ { "BLACK RIGHT-POINTING PENTAGON",NULL},
/* 2B54 */ { "WHITE RIGHT-POINTING PENTAGON",NULL},
/* 2B55 */ { "HEAVY LARGE CIRCLE","	= basic symbol for speed limit\n"
	"	* forms a game tally pair with 274C\n"
	"	x (large circle - 25EF)"},
/* 2B56 */ { "HEAVY OVAL WITH OVAL INSIDE","	= prefectural office"},
/* 2B57 */ { "HEAVY CIRCLE WITH CIRCLE INSIDE","	= municipal office\n"
	"	x (bullseye - 25CE)"},
/* 2B58 */ { "HEAVY CIRCLE","	= town or village office\n"
	"	= power off symbol\n"
	"	x (power on symbol - 23FD)\n"
	"	x (white circle - 25CB)"},
/* 2B59 */ { "HEAVY CIRCLED SALTIRE","	= police station\n"
	"	x (n-ary circled times operator - 2A02)"},
/* 2B5A */ { "SLANTED NORTH ARROW WITH HOOKED HEAD","	* increasing tone with falling trend at the end"},
/* 2B5B */ { "BACKSLANTED SOUTH ARROW WITH HOOKED TAIL","	* sharp rise and fall in tone"},
/* 2B5C */ { "SLANTED NORTH ARROW WITH HORIZONTAL TAIL","	* continued rise in tone"},
/* 2B5D */ { "BACKSLANTED SOUTH ARROW WITH HORIZONTAL TAIL","	* continued fall in tone"},
/* 2B5E */ { "BENT ARROW POINTING DOWNWARDS THEN NORTH EAST","	* sharp fall in tone with rising trend at the end"},
/* 2B5F */ { "SHORT BENT ARROW POINTING DOWNWARDS THEN NORTH EAST","	* slight fall in tone with rising trend at the end"},
/* 2B60 */ { "LEFTWARDS TRIANGLE-HEADED ARROW","	x (leftwards arrow - 2190)"},
/* 2B61 */ { "UPWARDS TRIANGLE-HEADED ARROW","	x (upwards arrow - 2191)"},
/* 2B62 */ { "RIGHTWARDS TRIANGLE-HEADED ARROW","	x (rightwards arrow - 2192)\n"
	"	x (triangle-headed rightwards arrow - 279D)"},
/* 2B63 */ { "DOWNWARDS TRIANGLE-HEADED ARROW","	x (downwards arrow - 2193)"},
/* 2B64 */ { "LEFT RIGHT TRIANGLE-HEADED ARROW","	x (left right arrow - 2194)"},
/* 2B65 */ { "UP DOWN TRIANGLE-HEADED ARROW","	x (up down arrow - 2195)"},
/* 2B66 */ { "NORTH WEST TRIANGLE-HEADED ARROW","	x (north west arrow - 2196)"},
/* 2B67 */ { "NORTH EAST TRIANGLE-HEADED ARROW","	x (north east arrow - 2197)"},
/* 2B68 */ { "SOUTH EAST TRIANGLE-HEADED ARROW","	x (south east arrow - 2198)"},
/* 2B69 */ { "SOUTH WEST TRIANGLE-HEADED ARROW","	x (south west arrow - 2199)"},
/* 2B6A */ { "LEFTWARDS TRIANGLE-HEADED DASHED ARROW","	x (leftwards dashed arrow - 21E0)"},
/* 2B6B */ { "UPWARDS TRIANGLE-HEADED DASHED ARROW","	x (upwards dashed arrow - 21E1)"},
/* 2B6C */ { "RIGHTWARDS TRIANGLE-HEADED DASHED ARROW","	x (rightwards dashed arrow - 21E2)"},
/* 2B6D */ { "DOWNWARDS TRIANGLE-HEADED DASHED ARROW","	x (downwards dashed arrow - 21E3)"},
/* 2B6E */ { "CLOCKWISE TRIANGLE-HEADED OPEN CIRCLE ARROW","	x (clockwise open circle arrow - 21BB)"},
/* 2B6F */ { "ANTICLOCKWISE TRIANGLE-HEADED OPEN CIRCLE ARROW","	x (anticlockwise open circle arrow - 21BA)"},
/* 2B70 */ { "LEFTWARDS TRIANGLE-HEADED ARROW TO BAR","	= left tab\n"
	"	x (leftwards arrow to bar - 21E4)"},
/* 2B71 */ { "UPWARDS TRIANGLE-HEADED ARROW TO BAR","	= up tab\n"
	"	x (upwards arrow to bar - 2912)"},
/* 2B72 */ { "RIGHTWARDS TRIANGLE-HEADED ARROW TO BAR","	= right tab\n"
	"	x (rightwards arrow to bar - 21E5)"},
/* 2B73 */ { "DOWNWARDS TRIANGLE-HEADED ARROW TO BAR","	= down tab\n"
	"	x (downwards arrow to bar - 2913)"},
/* 2B74 */ { NULL,NULL},
/* 2B75 */ { NULL,NULL},
/* 2B76 */ { "NORTH WEST TRIANGLE-HEADED ARROW TO BAR","	= home"},
/* 2B77 */ { "NORTH EAST TRIANGLE-HEADED ARROW TO BAR",NULL},
/* 2B78 */ { "SOUTH EAST TRIANGLE-HEADED ARROW TO BAR","	= end"},
/* 2B79 */ { "SOUTH WEST TRIANGLE-HEADED ARROW TO BAR",NULL},
/* 2B7A */ { "LEFTWARDS TRIANGLE-HEADED ARROW WITH DOUBLE HORIZONTAL STROKE","	% LEFTWARDS TRIANGLE-HEADED ARROW WITH DOUBLE VERTICAL STROKE"},
/* 2B7B */ { "UPWARDS TRIANGLE-HEADED ARROW WITH DOUBLE HORIZONTAL STROKE","	= page up"},
/* 2B7C */ { "RIGHTWARDS TRIANGLE-HEADED ARROW WITH DOUBLE HORIZONTAL STROKE","	% RIGHTWARDS TRIANGLE-HEADED ARROW WITH DOUBLE VERTICAL STROKE"},
/* 2B7D */ { "DOWNWARDS TRIANGLE-HEADED ARROW WITH DOUBLE HORIZONTAL STROKE","	= page down"},
/* 2B7E */ { "HORIZONTAL TAB KEY","	x (leftwards arrow to bar over rightwards arrow to bar - 21B9)"},
/* 2B7F */ { "VERTICAL TAB KEY",NULL},
/* 2B80 */ { "LEFTWARDS TRIANGLE-HEADED ARROW OVER RIGHTWARDS TRIANGLE-HEADED ARROW","	x (leftwards arrow over rightwards arrow - 21C6)"},
/* 2B81 */ { "UPWARDS TRIANGLE-HEADED ARROW LEFTWARDS OF DOWNWARDS TRIANGLE-HEADED ARROW",NULL},
/* 2B82 */ { "RIGHTWARDS TRIANGLE-HEADED ARROW OVER LEFTWARDS TRIANGLE-HEADED ARROW",NULL},
/* 2B83 */ { "DOWNWARDS TRIANGLE-HEADED ARROW LEFTWARDS OF UPWARDS TRIANGLE-HEADED ARROW","	x (downwards arrow leftwards of upwards arrow - 21F5)"},
/* 2B84 */ { "LEFTWARDS TRIANGLE-HEADED PAIRED ARROWS","	x (leftwards paired arrows - 21C7)"},
/* 2B85 */ { "UPWARDS TRIANGLE-HEADED PAIRED ARROWS","	x (upwards paired arrows - 21C8)"},
/* 2B86 */ { "RIGHTWARDS TRIANGLE-HEADED PAIRED ARROWS","	x (rightwards paired arrows - 21C9)"},
/* 2B87 */ { "DOWNWARDS TRIANGLE-HEADED PAIRED ARROWS","	x (downwards paired arrows - 21CA)"},
/* 2B88 */ { "LEFTWARDS BLACK CIRCLED WHITE ARROW",NULL},
/* 2B89 */ { "UPWARDS BLACK CIRCLED WHITE ARROW",NULL},
/* 2B8A */ { "RIGHTWARDS BLACK CIRCLED WHITE ARROW","	x (circled heavy white rightwards arrow - 27B2)"},
/* 2B8B */ { "DOWNWARDS BLACK CIRCLED WHITE ARROW",NULL},
/* 2B8C */ { "ANTICLOCKWISE TRIANGLE-HEADED RIGHT U-SHAPED ARROW",NULL},
/* 2B8D */ { "ANTICLOCKWISE TRIANGLE-HEADED BOTTOM U-SHAPED ARROW",NULL},
/* 2B8E */ { "ANTICLOCKWISE TRIANGLE-HEADED LEFT U-SHAPED ARROW",NULL},
/* 2B8F */ { "ANTICLOCKWISE TRIANGLE-HEADED TOP U-SHAPED ARROW","	x (anticlockwise top semicircle arrow - 21B6)"},
/* 2B90 */ { "RETURN LEFT","	x (return symbol - 23CE)"},
/* 2B91 */ { "RETURN RIGHT",NULL},
/* 2B92 */ { "NEWLINE LEFT",NULL},
/* 2B93 */ { "NEWLINE RIGHT",NULL},
/* 2B94 */ { "FOUR CORNER ARROWS CIRCLING ANTICLOCKWISE","	= loop"},
/* 2B95 */ { "RIGHTWARDS BLACK ARROW","	x (black rightwards arrow - 27A1)\n"
	"	x (leftwards black arrow - 2B05)"},
/* 2B96 */ { NULL,NULL},
/* 2B97 */ { "SYMBOL FOR TYPE A ELECTRONICS","	* for type B electronics use 3036\n"
	"	x (circled postal mark - 3036)"},
/* 2B98 */ { "THREE-D TOP-LIGHTED LEFTWARDS EQUILATERAL ARROWHEAD",NULL},
/* 2B99 */ { "THREE-D RIGHT-LIGHTED UPWARDS EQUILATERAL ARROWHEAD",NULL},
/* 2B9A */ { "THREE-D TOP-LIGHTED RIGHTWARDS EQUILATERAL ARROWHEAD","	x (three-d top-lighted rightwards arrowhead - 27A2)"},
/* 2B9B */ { "THREE-D LEFT-LIGHTED DOWNWARDS EQUILATERAL ARROWHEAD",NULL},
/* 2B9C */ { "BLACK LEFTWARDS EQUILATERAL ARROWHEAD",NULL},
/* 2B9D */ { "BLACK UPWARDS EQUILATERAL ARROWHEAD",NULL},
/* 2B9E */ { "BLACK RIGHTWARDS EQUILATERAL ARROWHEAD","	x (black rightwards arrowhead - 27A4)"},
/* 2B9F */ { "BLACK DOWNWARDS EQUILATERAL ARROWHEAD",NULL},
/* 2BA0 */ { "DOWNWARDS TRIANGLE-HEADED ARROW WITH LONG TIP LEFTWARDS","	x (downwards arrow with tip leftwards - 21B2)"},
/* 2BA1 */ { "DOWNWARDS TRIANGLE-HEADED ARROW WITH LONG TIP RIGHTWARDS","	x (downwards arrow with tip rightwards - 21B3)"},
/* 2BA2 */ { "UPWARDS TRIANGLE-HEADED ARROW WITH LONG TIP LEFTWARDS","	x (upwards arrow with tip leftwards - 21B0)"},
/* 2BA3 */ { "UPWARDS TRIANGLE-HEADED ARROW WITH LONG TIP RIGHTWARDS","	x (upwards arrow with tip rightwards - 21B1)"},
/* 2BA4 */ { "LEFTWARDS TRIANGLE-HEADED ARROW WITH LONG TIP UPWARDS","	x (leftwards arrow with tip upwards - 2B11)"},
/* 2BA5 */ { "RIGHTWARDS TRIANGLE-HEADED ARROW WITH LONG TIP UPWARDS","	x (rightwards arrow with tip upwards - 2B0F)"},
/* 2BA6 */ { "LEFTWARDS TRIANGLE-HEADED ARROW WITH LONG TIP DOWNWARDS","	x (leftwards arrow with tip downwards - 2B10)"},
/* 2BA7 */ { "RIGHTWARDS TRIANGLE-HEADED ARROW WITH LONG TIP DOWNWARDS","	x (rightwards arrow with tip downwards - 2B0E)"},
/* 2BA8 */ { "BLACK CURVED DOWNWARDS AND LEFTWARDS ARROW",NULL},
/* 2BA9 */ { "BLACK CURVED DOWNWARDS AND RIGHTWARDS ARROW","	x (heavy black curved downwards and rightwards arrow - 27A5)"},
/* 2BAA */ { "BLACK CURVED UPWARDS AND LEFTWARDS ARROW",NULL},
/* 2BAB */ { "BLACK CURVED UPWARDS AND RIGHTWARDS ARROW","	x (heavy black curved upwards and rightwards arrow - 27A6)"},
/* 2BAC */ { "BLACK CURVED LEFTWARDS AND UPWARDS ARROW",NULL},
/* 2BAD */ { "BLACK CURVED RIGHTWARDS AND UPWARDS ARROW",NULL},
/* 2BAE */ { "BLACK CURVED LEFTWARDS AND DOWNWARDS ARROW",NULL},
/* 2BAF */ { "BLACK CURVED RIGHTWARDS AND DOWNWARDS ARROW",NULL},
/* 2BB0 */ { "RIBBON ARROW DOWN LEFT",NULL},
/* 2BB1 */ { "RIBBON ARROW DOWN RIGHT",NULL},
/* 2BB2 */ { "RIBBON ARROW UP LEFT",NULL},
/* 2BB3 */ { "RIBBON ARROW UP RIGHT",NULL},
/* 2BB4 */ { "RIBBON ARROW LEFT UP",NULL},
/* 2BB5 */ { "RIBBON ARROW RIGHT UP",NULL},
/* 2BB6 */ { "RIBBON ARROW LEFT DOWN",NULL},
/* 2BB7 */ { "RIBBON ARROW RIGHT DOWN",NULL},
/* 2BB8 */ { "UPWARDS WHITE ARROW FROM BAR WITH HORIZONTAL BAR","	= caps lock\n"
	"	x (upwards white arrow on pedestal with horizontal bar - 21EC)"},
/* 2BB9 */ { "UP ARROWHEAD IN A RECTANGLE BOX","	= escape\n"
	"	x (apl functional symbol quad up caret - 2353)"},
/* 2BBA */ { "OVERLAPPING WHITE SQUARES","	= pair of bishops\n"
	"	x (two joined squares - 29C9)"},
/* 2BBB */ { "OVERLAPPING WHITE AND BLACK SQUARES","	= bishops of opposite colour"},
/* 2BBC */ { "OVERLAPPING BLACK SQUARES","	= bishops of the same colour"},
/* 2BBD */ { "BALLOT BOX WITH LIGHT X","	x (ballot box with x - 2612)"},
/* 2BBE */ { "CIRCLED X","	x (circled times - 2297)"},
/* 2BBF */ { "CIRCLED BOLD X",NULL},
/* 2BC0 */ { "BLACK SQUARE CENTRED",NULL},
/* 2BC1 */ { "BLACK DIAMOND CENTRED",NULL},
/* 2BC2 */ { "TURNED BLACK PENTAGON",NULL},
/* 2BC3 */ { "HORIZONTAL BLACK OCTAGON","	x (octagonal sign - 1F6D1)"},
/* 2BC4 */ { "BLACK OCTAGON",NULL},
/* 2BC5 */ { "BLACK MEDIUM UP-POINTING TRIANGLE CENTRED","	x (black up-pointing triangle - 25B2)"},
/* 2BC6 */ { "BLACK MEDIUM DOWN-POINTING TRIANGLE CENTRED","	x (black down-pointing triangle - 25BC)"},
/* 2BC7 */ { "BLACK MEDIUM LEFT-POINTING TRIANGLE CENTRED","	x (black left-pointing triangle - 25C0)"},
/* 2BC8 */ { "BLACK MEDIUM RIGHT-POINTING TRIANGLE CENTRED","	x (black right-pointing triangle - 25B6)"},
/* 2BC9 */ { "NEPTUNE FORM TWO","	x (neptune - 2646)"},
/* 2BCA */ { "TOP HALF BLACK CIRCLE",NULL},
/* 2BCB */ { "BOTTOM HALF BLACK CIRCLE",NULL},
/* 2BCC */ { "LIGHT FOUR POINTED BLACK CUSP","	x (black four pointed star - 2726)"},
/* 2BCD */ { "ROTATED LIGHT FOUR POINTED BLACK CUSP",NULL},
/* 2BCE */ { "WHITE FOUR POINTED CUSP","	x (white four pointed star - 2727)"},
/* 2BCF */ { "ROTATED WHITE FOUR POINTED CUSP","	x (square lozenge - 2311)"},
/* 2BD0 */ { "SQUARE POSITION INDICATOR","	x (position indicator - 2316)"},
/* 2BD1 */ { "UNCERTAINTY SIGN","	= query\n"
	"	x (apl functional symbol quad question - 2370)\n"
	"	x (replacement character - FFFD)"},
/* 2BD2 */ { "GROUP MARK","	* formerly used as a separator character for I/O operations\n"
	"	x (double dagger - 2021)\n"
	"	x (thermodynamic - 29E7)\n"
	"	x (triple dagger - 2E4B)"},
/* 2BD3 */ { "PLUTO FORM TWO","	x (pluto - 2647)"},
/* 2BD4 */ { "PLUTO FORM THREE",NULL},
/* 2BD5 */ { "PLUTO FORM FOUR",NULL},
/* 2BD6 */ { "PLUTO FORM FIVE",NULL},
/* 2BD7 */ { "TRANSPLUTO",NULL},
/* 2BD8 */ { "PROSERPINA",NULL},
/* 2BD9 */ { "ASTRAEA",NULL},
/* 2BDA */ { "HYGIEA",NULL},
/* 2BDB */ { "PHOLUS",NULL},
/* 2BDC */ { "NESSUS",NULL},
/* 2BDD */ { "WHITE MOON SELENA",NULL},
/* 2BDE */ { "BLACK DIAMOND ON CROSS","	= true black moon Lilith"},
/* 2BDF */ { "TRUE LIGHT MOON ARTA",NULL},
/* 2BE0 */ { "CUPIDO",NULL},
/* 2BE1 */ { "HADES",NULL},
/* 2BE2 */ { "ZEUS",NULL},
/* 2BE3 */ { "KRONOS",NULL},
/* 2BE4 */ { "APOLLON",NULL},
/* 2BE5 */ { "ADMETOS",NULL},
/* 2BE6 */ { "VULCANUS",NULL},
/* 2BE7 */ { "POSEIDON",NULL},
/* 2BE8 */ { "LEFT HALF BLACK STAR","	x (black star - 2605)"},
/* 2BE9 */ { "RIGHT HALF BLACK STAR",NULL},
/* 2BEA */ { "STAR WITH LEFT HALF BLACK",NULL},
/* 2BEB */ { "STAR WITH RIGHT HALF BLACK",NULL},
/* 2BEC */ { "LEFTWARDS TWO-HEADED ARROW WITH TRIANGLE ARROWHEADS","	= fast cursor left\n"
	"	x (leftwards two headed arrow - 219E)"},
/* 2BED */ { "UPWARDS TWO-HEADED ARROW WITH TRIANGLE ARROWHEADS","	= fast cursor up"},
/* 2BEE */ { "RIGHTWARDS TWO-HEADED ARROW WITH TRIANGLE ARROWHEADS","	= fast cursor right"},
/* 2BEF */ { "DOWNWARDS TWO-HEADED ARROW WITH TRIANGLE ARROWHEADS","	= fast cursor down"},
/* 2BF0 */ { "ERIS FORM ONE",NULL},
/* 2BF1 */ { "ERIS FORM TWO",NULL},
/* 2BF2 */ { "SEDNA",NULL},
/* 2BF3 */ { "RUSSIAN ASTROLOGICAL SYMBOL VIGINTILE","	* the vigintile aspect is usually represented by the letter V"},
/* 2BF4 */ { "RUSSIAN ASTROLOGICAL SYMBOL NOVILE","	* the novile aspect is usually represented by the letter N"},
/* 2BF5 */ { "RUSSIAN ASTROLOGICAL SYMBOL QUINTILE","	* the quintile aspect is usually represented by the letter Q"},
/* 2BF6 */ { "RUSSIAN ASTROLOGICAL SYMBOL BINOVILE","	* the binovile aspect is usually represented by the term N²"},
/* 2BF7 */ { "RUSSIAN ASTROLOGICAL SYMBOL SENTAGON","	* represents a 100-degree aspect"},
/* 2BF8 */ { "RUSSIAN ASTROLOGICAL SYMBOL TREDECILE","	* the tredecile aspect is usually represented by the term D³"},
/* 2BF9 */ { "EQUALS SIGN WITH INFINITY BELOW","	= with compensation for the material"},
/* 2BFA */ { "UNITED SYMBOL","	= united pawns\n"
	"	x (divorce symbol - 26AE)\n"
	"	x (two rings aligned horizontally - 1CC88)"},
/* 2BFB */ { "SEPARATED SYMBOL","	= separated pawns\n"
	"	x (unmarried partnership symbol - 26AF)\n"
	"	x (double-ended multimap - 29DF)\n"
	"	x (alchemical symbol for arsenic - 1F73A)"},
/* 2BFC */ { "DOUBLED SYMBOL","	= doubled pawns"},
/* 2BFD */ { "PASSED SYMBOL","	= passed pawn\n"
	"	x (male sign - 2642)"},
/* 2BFE */ { "REVERSED RIGHT ANGLE","	= without\n"
	"	x (right angle - 221F)"},
/* 2BFF */ { "HELLSCHREIBER PAUSE SYMBOL",NULL}
};

UN_DLL_LOCAL
static const struct unicode_nameannot una_00_2C[] = {
/* 2C00 */ { "GLAGOLITIC CAPITAL LETTER AZU",NULL},
/* 2C01 */ { "GLAGOLITIC CAPITAL LETTER BUKY",NULL},
/* 2C02 */ { "GLAGOLITIC CAPITAL LETTER VEDE",NULL},
/* 2C03 */ { "GLAGOLITIC CAPITAL LETTER GLAGOLI",NULL},
/* 2C04 */ { "GLAGOLITIC CAPITAL LETTER DOBRO",NULL},
/* 2C05 */ { "GLAGOLITIC CAPITAL LETTER YESTU",NULL},
/* 2C06 */ { "GLAGOLITIC CAPITAL LETTER ZHIVETE",NULL},
/* 2C07 */ { "GLAGOLITIC CAPITAL LETTER DZELO",NULL},
/* 2C08 */ { "GLAGOLITIC CAPITAL LETTER ZEMLJA",NULL},
/* 2C09 */ { "GLAGOLITIC CAPITAL LETTER IZHE",NULL},
/* 2C0A */ { "GLAGOLITIC CAPITAL LETTER INITIAL IZHE",NULL},
/* 2C0B */ { "GLAGOLITIC CAPITAL LETTER I",NULL},
/* 2C0C */ { "GLAGOLITIC CAPITAL LETTER DJERVI",NULL},
/* 2C0D */ { "GLAGOLITIC CAPITAL LETTER KAKO",NULL},
/* 2C0E */ { "GLAGOLITIC CAPITAL LETTER LJUDIJE",NULL},
/* 2C0F */ { "GLAGOLITIC CAPITAL LETTER MYSLITE",NULL},
/* 2C10 */ { "GLAGOLITIC CAPITAL LETTER NASHI",NULL},
/* 2C11 */ { "GLAGOLITIC CAPITAL LETTER ONU",NULL},
/* 2C12 */ { "GLAGOLITIC CAPITAL LETTER POKOJI",NULL},
/* 2C13 */ { "GLAGOLITIC CAPITAL LETTER RITSI",NULL},
/* 2C14 */ { "GLAGOLITIC CAPITAL LETTER SLOVO",NULL},
/* 2C15 */ { "GLAGOLITIC CAPITAL LETTER TVRIDO",NULL},
/* 2C16 */ { "GLAGOLITIC CAPITAL LETTER UKU",NULL},
/* 2C17 */ { "GLAGOLITIC CAPITAL LETTER FRITU",NULL},
/* 2C18 */ { "GLAGOLITIC CAPITAL LETTER HERU",NULL},
/* 2C19 */ { "GLAGOLITIC CAPITAL LETTER OTU",NULL},
/* 2C1A */ { "GLAGOLITIC CAPITAL LETTER PE",NULL},
/* 2C1B */ { "GLAGOLITIC CAPITAL LETTER SHTA",NULL},
/* 2C1C */ { "GLAGOLITIC CAPITAL LETTER TSI",NULL},
/* 2C1D */ { "GLAGOLITIC CAPITAL LETTER CHRIVI",NULL},
/* 2C1E */ { "GLAGOLITIC CAPITAL LETTER SHA",NULL},
/* 2C1F */ { "GLAGOLITIC CAPITAL LETTER YERU",NULL},
/* 2C20 */ { "GLAGOLITIC CAPITAL LETTER YERI",NULL},
/* 2C21 */ { "GLAGOLITIC CAPITAL LETTER YATI",NULL},
/* 2C22 */ { "GLAGOLITIC CAPITAL LETTER SPIDERY HA",NULL},
/* 2C23 */ { "GLAGOLITIC CAPITAL LETTER YU",NULL},
/* 2C24 */ { "GLAGOLITIC CAPITAL LETTER SMALL YUS",NULL},
/* 2C25 */ { "GLAGOLITIC CAPITAL LETTER SMALL YUS WITH TAIL",NULL},
/* 2C26 */ { "GLAGOLITIC CAPITAL LETTER YO",NULL},
/* 2C27 */ { "GLAGOLITIC CAPITAL LETTER IOTATED SMALL YUS",NULL},
/* 2C28 */ { "GLAGOLITIC CAPITAL LETTER BIG YUS",NULL},
/* 2C29 */ { "GLAGOLITIC CAPITAL LETTER IOTATED BIG YUS",NULL},
/* 2C2A */ { "GLAGOLITIC CAPITAL LETTER FITA",NULL},
/* 2C2B */ { "GLAGOLITIC CAPITAL LETTER IZHITSA",NULL},
/* 2C2C */ { "GLAGOLITIC CAPITAL LETTER SHTAPIC",NULL},
/* 2C2D */ { "GLAGOLITIC CAPITAL LETTER TROKUTASTI A",NULL},
/* 2C2E */ { "GLAGOLITIC CAPITAL LETTER LATINATE MYSLITE",NULL},
/* 2C2F */ { "GLAGOLITIC CAPITAL LETTER CAUDATE CHRIVI","		This is used as an alternative to Glagolitic letter chrivi, combined with 0303 combining tilde or with 0483 combining Cyrillic titlo as an et cetera sign."},
/* 2C30 */ { "GLAGOLITIC SMALL LETTER AZU",NULL},
/* 2C31 */ { "GLAGOLITIC SMALL LETTER BUKY",NULL},
/* 2C32 */ { "GLAGOLITIC SMALL LETTER VEDE",NULL},
/* 2C33 */ { "GLAGOLITIC SMALL LETTER GLAGOLI",NULL},
/* 2C34 */ { "GLAGOLITIC SMALL LETTER DOBRO",NULL},
/* 2C35 */ { "GLAGOLITIC SMALL LETTER YESTU",NULL},
/* 2C36 */ { "GLAGOLITIC SMALL LETTER ZHIVETE",NULL},
/* 2C37 */ { "GLAGOLITIC SMALL LETTER DZELO",NULL},
/* 2C38 */ { "GLAGOLITIC SMALL LETTER ZEMLJA",NULL},
/* 2C39 */ { "GLAGOLITIC SMALL LETTER IZHE",NULL},
/* 2C3A */ { "GLAGOLITIC SMALL LETTER INITIAL IZHE","	x (cyrillic small letter iota - A647)"},
/* 2C3B */ { "GLAGOLITIC SMALL LETTER I",NULL},
/* 2C3C */ { "GLAGOLITIC SMALL LETTER DJERVI",NULL},
/* 2C3D */ { "GLAGOLITIC SMALL LETTER KAKO",NULL},
/* 2C3E */ { "GLAGOLITIC SMALL LETTER LJUDIJE",NULL},
/* 2C3F */ { "GLAGOLITIC SMALL LETTER MYSLITE",NULL},
/* 2C40 */ { "GLAGOLITIC SMALL LETTER NASHI",NULL},
/* 2C41 */ { "GLAGOLITIC SMALL LETTER ONU",NULL},
/* 2C42 */ { "GLAGOLITIC SMALL LETTER POKOJI",NULL},
/* 2C43 */ { "GLAGOLITIC SMALL LETTER RITSI",NULL},
/* 2C44 */ { "GLAGOLITIC SMALL LETTER SLOVO",NULL},
/* 2C45 */ { "GLAGOLITIC SMALL LETTER TVRIDO",NULL},
/* 2C46 */ { "GLAGOLITIC SMALL LETTER UKU",NULL},
/* 2C47 */ { "GLAGOLITIC SMALL LETTER FRITU",NULL},
/* 2C48 */ { "GLAGOLITIC SMALL LETTER HERU",NULL},
/* 2C49 */ { "GLAGOLITIC SMALL LETTER OTU",NULL},
/* 2C4A */ { "GLAGOLITIC SMALL LETTER PE",NULL},
/* 2C4B */ { "GLAGOLITIC SMALL LETTER SHTA",NULL},
/* 2C4C */ { "GLAGOLITIC SMALL LETTER TSI",NULL},
/* 2C4D */ { "GLAGOLITIC SMALL LETTER CHRIVI",NULL},
/* 2C4E */ { "GLAGOLITIC SMALL LETTER SHA",NULL},
/* 2C4F */ { "GLAGOLITIC SMALL LETTER YERU",NULL},
/* 2C50 */ { "GLAGOLITIC SMALL LETTER YERI",NULL},
/* 2C51 */ { "GLAGOLITIC SMALL LETTER YATI",NULL},
/* 2C52 */ { "GLAGOLITIC SMALL LETTER SPIDERY HA",NULL},
/* 2C53 */ { "GLAGOLITIC SMALL LETTER YU",NULL},
/* 2C54 */ { "GLAGOLITIC SMALL LETTER SMALL YUS",NULL},
/* 2C55 */ { "GLAGOLITIC SMALL LETTER SMALL YUS WITH TAIL",NULL},
/* 2C56 */ { "GLAGOLITIC SMALL LETTER YO",NULL},
/* 2C57 */ { "GLAGOLITIC SMALL LETTER IOTATED SMALL YUS",NULL},
/* 2C58 */ { "GLAGOLITIC SMALL LETTER BIG YUS",NULL},
/* 2C59 */ { "GLAGOLITIC SMALL LETTER IOTATED BIG YUS",NULL},
/* 2C5A */ { "GLAGOLITIC SMALL LETTER FITA",NULL},
/* 2C5B */ { "GLAGOLITIC SMALL LETTER IZHITSA",NULL},
/* 2C5C */ { "GLAGOLITIC SMALL LETTER SHTAPIC",NULL},
/* 2C5D */ { "GLAGOLITIC SMALL LETTER TROKUTASTI A",NULL},
/* 2C5E */ { "GLAGOLITIC SMALL LETTER LATINATE MYSLITE",NULL},
/* 2C5F */ { "GLAGOLITIC SMALL LETTER CAUDATE CHRIVI",NULL},
/* 2C60 */ { "LATIN CAPITAL LETTER L WITH DOUBLE BAR","	x (latin capital letter l with bar - 023D)"},
/* 2C61 */ { "LATIN SMALL LETTER L WITH DOUBLE BAR","	x (latin small letter l with bar - 019A)\n"
	"	x (latin small letter l with double middle tilde - AB38)"},
/* 2C62 */ { "LATIN CAPITAL LETTER L WITH MIDDLE TILDE","	* lowercase is 026B\n"
	"	x (latin capital letter l with stroke - 0141)"},
/* 2C63 */ { "LATIN CAPITAL LETTER P WITH STROKE","	* lowercase is 1D7D"},
/* 2C64 */ { "LATIN CAPITAL LETTER R WITH TAIL","	* lowercase is 027D"},
/* 2C65 */ { "LATIN SMALL LETTER A WITH STROKE","	* uppercase is 023A"},
/* 2C66 */ { "LATIN SMALL LETTER T WITH DIAGONAL STROKE","	* uppercase is 023E"},
/* 2C67 */ { "LATIN CAPITAL LETTER H WITH DESCENDER","	x (cyrillic capital letter en with descender - 04A2)"},
/* 2C68 */ { "LATIN SMALL LETTER H WITH DESCENDER",NULL},
/* 2C69 */ { "LATIN CAPITAL LETTER K WITH DESCENDER","	x (cyrillic capital letter ka with descender - 049A)"},
/* 2C6A */ { "LATIN SMALL LETTER K WITH DESCENDER",NULL},
/* 2C6B */ { "LATIN CAPITAL LETTER Z WITH DESCENDER","	x (latin capital letter z with hook - 0224)"},
/* 2C6C */ { "LATIN SMALL LETTER Z WITH DESCENDER",NULL},
/* 2C6D */ { "LATIN CAPITAL LETTER ALPHA","	= capital script a\n"
	"	* lowercase is 0251"},
/* 2C6E */ { "LATIN CAPITAL LETTER M WITH HOOK","	* lowercase is 0271"},
/* 2C6F */ { "LATIN CAPITAL LETTER TURNED A","	* lowercase is 0250\n"
	"	x (for all - 2200)"},
/* 2C70 */ { "LATIN CAPITAL LETTER TURNED ALPHA","	* lowercase is 0252"},
/* 2C71 */ { "LATIN SMALL LETTER V WITH RIGHT HOOK","	* IPA: voiced labiodental flap"},
/* 2C72 */ { "LATIN CAPITAL LETTER W WITH HOOK",NULL},
/* 2C73 */ { "LATIN SMALL LETTER W WITH HOOK",NULL},
/* 2C74 */ { "LATIN SMALL LETTER V WITH CURL",NULL},
/* 2C75 */ { "LATIN CAPITAL LETTER HALF H","	x (greek capital letter heta - 0370)\n"
	"	x (turned capital f - 2132)\n"
	"	x (roman numeral reversed one hundred - 2183)\n"
	"	x (latin capital letter reversed half h - A7F5)"},
/* 2C76 */ { "LATIN SMALL LETTER HALF H","	x (greek small letter heta - 0371)\n"
	"	x (latin small letter reversed half h - A7F6)"},
/* 2C77 */ { "LATIN SMALL LETTER TAILLESS PHI","	* medium rounded o\n"
	"	x (greek small letter phi - 03C6)"},
/* 2C78 */ { "LATIN SMALL LETTER E WITH NOTCH",NULL},
/* 2C79 */ { "LATIN SMALL LETTER TURNED R WITH TAIL",NULL},
/* 2C7A */ { "LATIN SMALL LETTER O WITH LOW RING INSIDE",NULL},
/* 2C7B */ { "LATIN LETTER SMALL CAPITAL TURNED E","	x (latin capital letter reversed e - 018E)"},
/* 2C7C */ { "LATIN SUBSCRIPT SMALL LETTER J","	# <sub> 006A"},
/* 2C7D */ { "MODIFIER LETTER CAPITAL V","	# <super> 0056"},
/* 2C7E */ { "LATIN CAPITAL LETTER S WITH SWASH TAIL","	* lowercase is 023F\n"
	"	x (latin capital letter s with hook - A7C5)"},
/* 2C7F */ { "LATIN CAPITAL LETTER Z WITH SWASH TAIL","	* lowercase is 0240"},
/* 2C80 */ { "COPTIC CAPITAL LETTER ALFA",NULL},
/* 2C81 */ { "COPTIC SMALL LETTER ALFA",NULL},
/* 2C82 */ { "COPTIC CAPITAL LETTER VIDA",NULL},
/* 2C83 */ { "COPTIC SMALL LETTER VIDA",NULL},
/* 2C84 */ { "COPTIC CAPITAL LETTER GAMMA",NULL},
/* 2C85 */ { "COPTIC SMALL LETTER GAMMA",NULL},
/* 2C86 */ { "COPTIC CAPITAL LETTER DALDA",NULL},
/* 2C87 */ { "COPTIC SMALL LETTER DALDA",NULL},
/* 2C88 */ { "COPTIC CAPITAL LETTER EIE",NULL},
/* 2C89 */ { "COPTIC SMALL LETTER EIE",NULL},
/* 2C8A */ { "COPTIC CAPITAL LETTER SOU",NULL},
/* 2C8B */ { "COPTIC SMALL LETTER SOU",NULL},
/* 2C8C */ { "COPTIC CAPITAL LETTER ZATA",NULL},
/* 2C8D */ { "COPTIC SMALL LETTER ZATA",NULL},
/* 2C8E */ { "COPTIC CAPITAL LETTER HATE",NULL},
/* 2C8F */ { "COPTIC SMALL LETTER HATE",NULL},
/* 2C90 */ { "COPTIC CAPITAL LETTER THETHE",NULL},
/* 2C91 */ { "COPTIC SMALL LETTER THETHE",NULL},
/* 2C92 */ { "COPTIC CAPITAL LETTER IAUDA",NULL},
/* 2C93 */ { "COPTIC SMALL LETTER IAUDA",NULL},
/* 2C94 */ { "COPTIC CAPITAL LETTER KAPA",NULL},
/* 2C95 */ { "COPTIC SMALL LETTER KAPA",NULL},
/* 2C96 */ { "COPTIC CAPITAL LETTER LAULA",NULL},
/* 2C97 */ { "COPTIC SMALL LETTER LAULA",NULL},
/* 2C98 */ { "COPTIC CAPITAL LETTER MI",NULL},
/* 2C99 */ { "COPTIC SMALL LETTER MI",NULL},
/* 2C9A */ { "COPTIC CAPITAL LETTER NI",NULL},
/* 2C9B */ { "COPTIC SMALL LETTER NI",NULL},
/* 2C9C */ { "COPTIC CAPITAL LETTER KSI",NULL},
/* 2C9D */ { "COPTIC SMALL LETTER KSI",NULL},
/* 2C9E */ { "COPTIC CAPITAL LETTER O",NULL},
/* 2C9F */ { "COPTIC SMALL LETTER O",NULL},
/* 2CA0 */ { "COPTIC CAPITAL LETTER PI",NULL},
/* 2CA1 */ { "COPTIC SMALL LETTER PI",NULL},
/* 2CA2 */ { "COPTIC CAPITAL LETTER RO",NULL},
/* 2CA3 */ { "COPTIC SMALL LETTER RO",NULL},
/* 2CA4 */ { "COPTIC CAPITAL LETTER SIMA",NULL},
/* 2CA5 */ { "COPTIC SMALL LETTER SIMA",NULL},
/* 2CA6 */ { "COPTIC CAPITAL LETTER TAU",NULL},
/* 2CA7 */ { "COPTIC SMALL LETTER TAU",NULL},
/* 2CA8 */ { "COPTIC CAPITAL LETTER UA",NULL},
/* 2CA9 */ { "COPTIC SMALL LETTER UA",NULL},
/* 2CAA */ { "COPTIC CAPITAL LETTER FI",NULL},
/* 2CAB */ { "COPTIC SMALL LETTER FI",NULL},
/* 2CAC */ { "COPTIC CAPITAL LETTER KHI",NULL},
/* 2CAD */ { "COPTIC SMALL LETTER KHI",NULL},
/* 2CAE */ { "COPTIC CAPITAL LETTER PSI",NULL},
/* 2CAF */ { "COPTIC SMALL LETTER PSI",NULL},
/* 2CB0 */ { "COPTIC CAPITAL LETTER OOU",NULL},
/* 2CB1 */ { "COPTIC SMALL LETTER OOU",NULL},
/* 2CB2 */ { "COPTIC CAPITAL LETTER DIALECT-P ALEF",NULL},
/* 2CB3 */ { "COPTIC SMALL LETTER DIALECT-P ALEF",NULL},
/* 2CB4 */ { "COPTIC CAPITAL LETTER OLD COPTIC AIN",NULL},
/* 2CB5 */ { "COPTIC SMALL LETTER OLD COPTIC AIN",NULL},
/* 2CB6 */ { "COPTIC CAPITAL LETTER CRYPTOGRAMMIC EIE",NULL},
/* 2CB7 */ { "COPTIC SMALL LETTER CRYPTOGRAMMIC EIE",NULL},
/* 2CB8 */ { "COPTIC CAPITAL LETTER DIALECT-P KAPA",NULL},
/* 2CB9 */ { "COPTIC SMALL LETTER DIALECT-P KAPA",NULL},
/* 2CBA */ { "COPTIC CAPITAL LETTER DIALECT-P NI",NULL},
/* 2CBB */ { "COPTIC SMALL LETTER DIALECT-P NI",NULL},
/* 2CBC */ { "COPTIC CAPITAL LETTER CRYPTOGRAMMIC NI",NULL},
/* 2CBD */ { "COPTIC SMALL LETTER CRYPTOGRAMMIC NI",NULL},
/* 2CBE */ { "COPTIC CAPITAL LETTER OLD COPTIC OOU",NULL},
/* 2CBF */ { "COPTIC SMALL LETTER OLD COPTIC OOU",NULL},
/* 2CC0 */ { "COPTIC CAPITAL LETTER SAMPI",NULL},
/* 2CC1 */ { "COPTIC SMALL LETTER SAMPI",NULL},
/* 2CC2 */ { "COPTIC CAPITAL LETTER CROSSED SHEI",NULL},
/* 2CC3 */ { "COPTIC SMALL LETTER CROSSED SHEI",NULL},
/* 2CC4 */ { "COPTIC CAPITAL LETTER OLD COPTIC SHEI",NULL},
/* 2CC5 */ { "COPTIC SMALL LETTER OLD COPTIC SHEI",NULL},
/* 2CC6 */ { "COPTIC CAPITAL LETTER OLD COPTIC ESH",NULL},
/* 2CC7 */ { "COPTIC SMALL LETTER OLD COPTIC ESH",NULL},
/* 2CC8 */ { "COPTIC CAPITAL LETTER AKHMIMIC KHEI",NULL},
/* 2CC9 */ { "COPTIC SMALL LETTER AKHMIMIC KHEI",NULL},
/* 2CCA */ { "COPTIC CAPITAL LETTER DIALECT-P HORI",NULL},
/* 2CCB */ { "COPTIC SMALL LETTER DIALECT-P HORI",NULL},
/* 2CCC */ { "COPTIC CAPITAL LETTER OLD COPTIC HORI",NULL},
/* 2CCD */ { "COPTIC SMALL LETTER OLD COPTIC HORI",NULL},
/* 2CCE */ { "COPTIC CAPITAL LETTER OLD COPTIC HA",NULL},
/* 2CCF */ { "COPTIC SMALL LETTER OLD COPTIC HA",NULL},
/* 2CD0 */ { "COPTIC CAPITAL LETTER L-SHAPED HA",NULL},
/* 2CD1 */ { "COPTIC SMALL LETTER L-SHAPED HA",NULL},
/* 2CD2 */ { "COPTIC CAPITAL LETTER OLD COPTIC HEI",NULL},
/* 2CD3 */ { "COPTIC SMALL LETTER OLD COPTIC HEI",NULL},
/* 2CD4 */ { "COPTIC CAPITAL LETTER OLD COPTIC HAT",NULL},
/* 2CD5 */ { "COPTIC SMALL LETTER OLD COPTIC HAT",NULL},
/* 2CD6 */ { "COPTIC CAPITAL LETTER OLD COPTIC GANGIA",NULL},
/* 2CD7 */ { "COPTIC SMALL LETTER OLD COPTIC GANGIA",NULL},
/* 2CD8 */ { "COPTIC CAPITAL LETTER OLD COPTIC DJA",NULL},
/* 2CD9 */ { "COPTIC SMALL LETTER OLD COPTIC DJA",NULL},
/* 2CDA */ { "COPTIC CAPITAL LETTER OLD COPTIC SHIMA",NULL},
/* 2CDB */ { "COPTIC SMALL LETTER OLD COPTIC SHIMA",NULL},
/* 2CDC */ { "COPTIC CAPITAL LETTER OLD NUBIAN SHIMA",NULL},
/* 2CDD */ { "COPTIC SMALL LETTER OLD NUBIAN SHIMA",NULL},
/* 2CDE */ { "COPTIC CAPITAL LETTER OLD NUBIAN NGI",NULL},
/* 2CDF */ { "COPTIC SMALL LETTER OLD NUBIAN NGI",NULL},
/* 2CE0 */ { "COPTIC CAPITAL LETTER OLD NUBIAN NYI",NULL},
/* 2CE1 */ { "COPTIC SMALL LETTER OLD NUBIAN NYI",NULL},
/* 2CE2 */ { "COPTIC CAPITAL LETTER OLD NUBIAN WAU",NULL},
/* 2CE3 */ { "COPTIC SMALL LETTER OLD NUBIAN WAU",NULL},
/* 2CE4 */ { "COPTIC SYMBOL KAI","	x (greek kai symbol - 03D7)"},
/* 2CE5 */ { "COPTIC SYMBOL MI RO",NULL},
/* 2CE6 */ { "COPTIC SYMBOL PI RO",NULL},
/* 2CE7 */ { "COPTIC SYMBOL STAUROS",NULL},
/* 2CE8 */ { "COPTIC SYMBOL TAU RO","	x (greek symbol tau rho - 101A0)"},
/* 2CE9 */ { "COPTIC SYMBOL KHI RO","	x (chi rho - 2627)"},
/* 2CEA */ { "COPTIC SYMBOL SHIMA SIMA",NULL},
/* 2CEB */ { "COPTIC CAPITAL LETTER CRYPTOGRAMMIC SHEI",NULL},
/* 2CEC */ { "COPTIC SMALL LETTER CRYPTOGRAMMIC SHEI",NULL},
/* 2CED */ { "COPTIC CAPITAL LETTER CRYPTOGRAMMIC GANGIA",NULL},
/* 2CEE */ { "COPTIC SMALL LETTER CRYPTOGRAMMIC GANGIA",NULL},
/* 2CEF */ { "COPTIC COMBINING NI ABOVE","	* this mark is used in final position and extends above the following character (usually a space)"},
/* 2CF0 */ { "COPTIC COMBINING SPIRITUS ASPER","	x (combining reversed comma above - 0314)\n"
	"	x (combining cyrillic dasia pneumata - 0485)"},
/* 2CF1 */ { "COPTIC COMBINING SPIRITUS LENIS","	x (combining comma above - 0313)\n"
	"	x (combining cyrillic psili pneumata - 0486)"},
/* 2CF2 */ { "COPTIC CAPITAL LETTER BOHAIRIC KHEI","	x (coptic capital letter khei - 03E6)"},
/* 2CF3 */ { "COPTIC SMALL LETTER BOHAIRIC KHEI","	x (coptic small letter khei - 03E7)"},
/* 2CF4 */ { NULL,NULL},
/* 2CF5 */ { NULL,NULL},
/* 2CF6 */ { NULL,NULL},
/* 2CF7 */ { NULL,NULL},
/* 2CF8 */ { NULL,NULL},
/* 2CF9 */ { "COPTIC OLD NUBIAN FULL STOP",NULL},
/* 2CFA */ { "COPTIC OLD NUBIAN DIRECT QUESTION MARK",NULL},
/* 2CFB */ { "COPTIC OLD NUBIAN INDIRECT QUESTION MARK",NULL},
/* 2CFC */ { "COPTIC OLD NUBIAN VERSE DIVIDER",NULL},
/* 2CFD */ { "COPTIC FRACTION ONE HALF",NULL},
/* 2CFE */ { "COPTIC FULL STOP",NULL},
/* 2CFF */ { "COPTIC MORPHOLOGICAL DIVIDER",NULL}
};

UN_DLL_LOCAL
static const struct unicode_nameannot una_00_2D[] = {
/* 2D00 */ { "GEORGIAN SMALL LETTER AN","	x (georgian capital letter an - 10A0)"},
/* 2D01 */ { "GEORGIAN SMALL LETTER BAN",NULL},
/* 2D02 */ { "GEORGIAN SMALL LETTER GAN",NULL},
/* 2D03 */ { "GEORGIAN SMALL LETTER DON",NULL},
/* 2D04 */ { "GEORGIAN SMALL LETTER EN",NULL},
/* 2D05 */ { "GEORGIAN SMALL LETTER VIN",NULL},
/* 2D06 */ { "GEORGIAN SMALL LETTER ZEN",NULL},
/* 2D07 */ { "GEORGIAN SMALL LETTER TAN",NULL},
/* 2D08 */ { "GEORGIAN SMALL LETTER IN",NULL},
/* 2D09 */ { "GEORGIAN SMALL LETTER KAN",NULL},
/* 2D0A */ { "GEORGIAN SMALL LETTER LAS",NULL},
/* 2D0B */ { "GEORGIAN SMALL LETTER MAN",NULL},
/* 2D0C */ { "GEORGIAN SMALL LETTER NAR",NULL},
/* 2D0D */ { "GEORGIAN SMALL LETTER ON",NULL},
/* 2D0E */ { "GEORGIAN SMALL LETTER PAR",NULL},
/* 2D0F */ { "GEORGIAN SMALL LETTER ZHAR",NULL},
/* 2D10 */ { "GEORGIAN SMALL LETTER RAE",NULL},
/* 2D11 */ { "GEORGIAN SMALL LETTER SAN",NULL},
/* 2D12 */ { "GEORGIAN SMALL LETTER TAR",NULL},
/* 2D13 */ { "GEORGIAN SMALL LETTER UN",NULL},
/* 2D14 */ { "GEORGIAN SMALL LETTER PHAR",NULL},
/* 2D15 */ { "GEORGIAN SMALL LETTER KHAR",NULL},
/* 2D16 */ { "GEORGIAN SMALL LETTER GHAN",NULL},
/* 2D17 */ { "GEORGIAN SMALL LETTER QAR",NULL},
/* 2D18 */ { "GEORGIAN SMALL LETTER SHIN",NULL},
/* 2D19 */ { "GEORGIAN SMALL LETTER CHIN",NULL},
/* 2D1A */ { "GEORGIAN SMALL LETTER CAN",NULL},
/* 2D1B */ { "GEORGIAN SMALL LETTER JIL",NULL},
/* 2D1C */ { "GEORGIAN SMALL LETTER CIL",NULL},
/* 2D1D */ { "GEORGIAN SMALL LETTER CHAR",NULL},
/* 2D1E */ { "GEORGIAN SMALL LETTER XAN",NULL},
/* 2D1F */ { "GEORGIAN SMALL LETTER JHAN",NULL},
/* 2D20 */ { "GEORGIAN SMALL LETTER HAE",NULL},
/* 2D21 */ { "GEORGIAN SMALL LETTER HE",NULL},
/* 2D22 */ { "GEORGIAN SMALL LETTER HIE",NULL},
/* 2D23 */ { "GEORGIAN SMALL LETTER WE",NULL},
/* 2D24 */ { "GEORGIAN SMALL LETTER HAR",NULL},
/* 2D25 */ { "GEORGIAN SMALL LETTER HOE",NULL},
/* 2D26 */ { NULL,NULL},
/* 2D27 */ { "GEORGIAN SMALL LETTER YN",NULL},
/* 2D28 */ { NULL,NULL},
/* 2D29 */ { NULL,NULL},
/* 2D2A */ { NULL,NULL},
/* 2D2B */ { NULL,NULL},
/* 2D2C */ { NULL,NULL},
/* 2D2D */ { "GEORGIAN SMALL LETTER AEN",NULL},
/* 2D2E */ { NULL,NULL},
/* 2D2F */ { NULL,NULL},
/* 2D30 */ { "TIFINAGH LETTER YA",NULL},
/* 2D31 */ { "TIFINAGH LETTER YAB",NULL},
/* 2D32 */ { "TIFINAGH LETTER YABH",NULL},
/* 2D33 */ { "TIFINAGH LETTER YAG",NULL},
/* 2D34 */ { "TIFINAGH LETTER YAGHH",NULL},
/* 2D35 */ { "TIFINAGH LETTER BERBER ACADEMY YAJ",NULL},
/* 2D36 */ { "TIFINAGH LETTER YAJ",NULL},
/* 2D37 */ { "TIFINAGH LETTER YAD",NULL},
/* 2D38 */ { "TIFINAGH LETTER YADH",NULL},
/* 2D39 */ { "TIFINAGH LETTER YADD",NULL},
/* 2D3A */ { "TIFINAGH LETTER YADDH",NULL},
/* 2D3B */ { "TIFINAGH LETTER YEY",NULL},
/* 2D3C */ { "TIFINAGH LETTER YAF",NULL},
/* 2D3D */ { "TIFINAGH LETTER YAK",NULL},
/* 2D3E */ { "TIFINAGH LETTER TUAREG YAK",NULL},
/* 2D3F */ { "TIFINAGH LETTER YAKHH",NULL},
/* 2D40 */ { "TIFINAGH LETTER YAH","	= Tuareg yab"},
/* 2D41 */ { "TIFINAGH LETTER BERBER ACADEMY YAH",NULL},
/* 2D42 */ { "TIFINAGH LETTER TUAREG YAH",NULL},
/* 2D43 */ { "TIFINAGH LETTER YAHH",NULL},
/* 2D44 */ { "TIFINAGH LETTER YAA",NULL},
/* 2D45 */ { "TIFINAGH LETTER YAKH",NULL},
/* 2D46 */ { "TIFINAGH LETTER TUAREG YAKH",NULL},
/* 2D47 */ { "TIFINAGH LETTER YAQ",NULL},
/* 2D48 */ { "TIFINAGH LETTER TUAREG YAQ",NULL},
/* 2D49 */ { "TIFINAGH LETTER YI",NULL},
/* 2D4A */ { "TIFINAGH LETTER YAZH",NULL},
/* 2D4B */ { "TIFINAGH LETTER AHAGGAR YAZH",NULL},
/* 2D4C */ { "TIFINAGH LETTER TUAREG YAZH",NULL},
/* 2D4D */ { "TIFINAGH LETTER YAL",NULL},
/* 2D4E */ { "TIFINAGH LETTER YAM",NULL},
/* 2D4F */ { "TIFINAGH LETTER YAN",NULL},
/* 2D50 */ { "TIFINAGH LETTER TUAREG YAGN",NULL},
/* 2D51 */ { "TIFINAGH LETTER TUAREG YANG",NULL},
/* 2D52 */ { "TIFINAGH LETTER YAP",NULL},
/* 2D53 */ { "TIFINAGH LETTER YU","	= Tuareg yaw"},
/* 2D54 */ { "TIFINAGH LETTER YAR",NULL},
/* 2D55 */ { "TIFINAGH LETTER YARR",NULL},
/* 2D56 */ { "TIFINAGH LETTER YAGH",NULL},
/* 2D57 */ { "TIFINAGH LETTER TUAREG YAGH",NULL},
/* 2D58 */ { "TIFINAGH LETTER AYER YAGH","	= Adrar yaj"},
/* 2D59 */ { "TIFINAGH LETTER YAS",NULL},
/* 2D5A */ { "TIFINAGH LETTER YASS",NULL},
/* 2D5B */ { "TIFINAGH LETTER YASH",NULL},
/* 2D5C */ { "TIFINAGH LETTER YAT",NULL},
/* 2D5D */ { "TIFINAGH LETTER YATH",NULL},
/* 2D5E */ { "TIFINAGH LETTER YACH",NULL},
/* 2D5F */ { "TIFINAGH LETTER YATT",NULL},
/* 2D60 */ { "TIFINAGH LETTER YAV",NULL},
/* 2D61 */ { "TIFINAGH LETTER YAW",NULL},
/* 2D62 */ { "TIFINAGH LETTER YAY",NULL},
/* 2D63 */ { "TIFINAGH LETTER YAZ",NULL},
/* 2D64 */ { "TIFINAGH LETTER TAWELLEMET YAZ","	= harpoon yaz"},
/* 2D65 */ { "TIFINAGH LETTER YAZZ",NULL},
/* 2D66 */ { "TIFINAGH LETTER YE",NULL},
/* 2D67 */ { "TIFINAGH LETTER YO",NULL},
/* 2D68 */ { NULL,NULL},
/* 2D69 */ { NULL,NULL},
/* 2D6A */ { NULL,NULL},
/* 2D6B */ { NULL,NULL},
/* 2D6C */ { NULL,NULL},
/* 2D6D */ { NULL,NULL},
/* 2D6E */ { NULL,NULL},
/* 2D6F */ { "TIFINAGH MODIFIER LETTER LABIALIZATION MARK","	= tamatart\n"
	"	# <super> 2D61"},
/* 2D70 */ { "TIFINAGH SEPARATOR MARK","	= tazarast"},
/* 2D71 */ { NULL,NULL},
/* 2D72 */ { NULL,NULL},
/* 2D73 */ { NULL,NULL},
/* 2D74 */ { NULL,NULL},
/* 2D75 */ { NULL,NULL},
/* 2D76 */ { NULL,NULL},
/* 2D77 */ { NULL,NULL},
/* 2D78 */ { NULL,NULL},
/* 2D79 */ { NULL,NULL},
/* 2D7A */ { NULL,NULL},
/* 2D7B */ { NULL,NULL},
/* 2D7C */ { NULL,NULL},
/* 2D7D */ { NULL,NULL},
/* 2D7E */ { NULL,NULL},
/* 2D7F */ { "TIFINAGH CONSONANT JOINER","	* suppresses an inherent vowel\n"
	"	* functions to indicate that previous and following character are part of a bi-consonant cluster\n"
	"	* shape shown is arbitrary and is not visibly rendered"},
/* 2D80 */ { "ETHIOPIC SYLLABLE LOA",NULL},
/* 2D81 */ { "ETHIOPIC SYLLABLE MOA",NULL},
/* 2D82 */ { "ETHIOPIC SYLLABLE ROA",NULL},
/* 2D83 */ { "ETHIOPIC SYLLABLE SOA",NULL},
/* 2D84 */ { "ETHIOPIC SYLLABLE SHOA",NULL},
/* 2D85 */ { "ETHIOPIC SYLLABLE BOA",NULL},
/* 2D86 */ { "ETHIOPIC SYLLABLE TOA",NULL},
/* 2D87 */ { "ETHIOPIC SYLLABLE COA",NULL},
/* 2D88 */ { "ETHIOPIC SYLLABLE NOA",NULL},
/* 2D89 */ { "ETHIOPIC SYLLABLE NYOA",NULL},
/* 2D8A */ { "ETHIOPIC SYLLABLE GLOTTAL OA",NULL},
/* 2D8B */ { "ETHIOPIC SYLLABLE ZOA",NULL},
/* 2D8C */ { "ETHIOPIC SYLLABLE DOA",NULL},
/* 2D8D */ { "ETHIOPIC SYLLABLE DDOA",NULL},
/* 2D8E */ { "ETHIOPIC SYLLABLE JOA",NULL},
/* 2D8F */ { "ETHIOPIC SYLLABLE THOA",NULL},
/* 2D90 */ { "ETHIOPIC SYLLABLE CHOA",NULL},
/* 2D91 */ { "ETHIOPIC SYLLABLE PHOA",NULL},
/* 2D92 */ { "ETHIOPIC SYLLABLE POA",NULL},
/* 2D93 */ { "ETHIOPIC SYLLABLE GGWA",NULL},
/* 2D94 */ { "ETHIOPIC SYLLABLE GGWI",NULL},
/* 2D95 */ { "ETHIOPIC SYLLABLE GGWEE",NULL},
/* 2D96 */ { "ETHIOPIC SYLLABLE GGWE",NULL},
/* 2D97 */ { NULL,NULL},
/* 2D98 */ { NULL,NULL},
/* 2D99 */ { NULL,NULL},
/* 2D9A */ { NULL,NULL},
/* 2D9B */ { NULL,NULL},
/* 2D9C */ { NULL,NULL},
/* 2D9D */ { NULL,NULL},
/* 2D9E */ { NULL,NULL},
/* 2D9F */ { NULL,NULL},
/* 2DA0 */ { "ETHIOPIC SYLLABLE SSA",NULL},
/* 2DA1 */ { "ETHIOPIC SYLLABLE SSU",NULL},
/* 2DA2 */ { "ETHIOPIC SYLLABLE SSI",NULL},
/* 2DA3 */ { "ETHIOPIC SYLLABLE SSAA",NULL},
/* 2DA4 */ { "ETHIOPIC SYLLABLE SSEE",NULL},
/* 2DA5 */ { "ETHIOPIC SYLLABLE SSE",NULL},
/* 2DA6 */ { "ETHIOPIC SYLLABLE SSO",NULL},
/* 2DA7 */ { NULL,NULL},
/* 2DA8 */ { "ETHIOPIC SYLLABLE CCA",NULL},
/* 2DA9 */ { "ETHIOPIC SYLLABLE CCU",NULL},
/* 2DAA */ { "ETHIOPIC SYLLABLE CCI",NULL},
/* 2DAB */ { "ETHIOPIC SYLLABLE CCAA",NULL},
/* 2DAC */ { "ETHIOPIC SYLLABLE CCEE",NULL},
/* 2DAD */ { "ETHIOPIC SYLLABLE CCE",NULL},
/* 2DAE */ { "ETHIOPIC SYLLABLE CCO",NULL},
/* 2DAF */ { NULL,NULL},
/* 2DB0 */ { "ETHIOPIC SYLLABLE ZZA",NULL},
/* 2DB1 */ { "ETHIOPIC SYLLABLE ZZU",NULL},
/* 2DB2 */ { "ETHIOPIC SYLLABLE ZZI",NULL},
/* 2DB3 */ { "ETHIOPIC SYLLABLE ZZAA",NULL},
/* 2DB4 */ { "ETHIOPIC SYLLABLE ZZEE",NULL},
/* 2DB5 */ { "ETHIOPIC SYLLABLE ZZE",NULL},
/* 2DB6 */ { "ETHIOPIC SYLLABLE ZZO",NULL},
/* 2DB7 */ { NULL,NULL},
/* 2DB8 */ { "ETHIOPIC SYLLABLE CCHA",NULL},
/* 2DB9 */ { "ETHIOPIC SYLLABLE CCHU",NULL},
/* 2DBA */ { "ETHIOPIC SYLLABLE CCHI",NULL},
/* 2DBB */ { "ETHIOPIC SYLLABLE CCHAA",NULL},
/* 2DBC */ { "ETHIOPIC SYLLABLE CCHEE",NULL},
/* 2DBD */ { "ETHIOPIC SYLLABLE CCHE",NULL},
/* 2DBE */ { "ETHIOPIC SYLLABLE CCHO",NULL},
/* 2DBF */ { NULL,NULL},
/* 2DC0 */ { "ETHIOPIC SYLLABLE QYA",NULL},
/* 2DC1 */ { "ETHIOPIC SYLLABLE QYU",NULL},
/* 2DC2 */ { "ETHIOPIC SYLLABLE QYI",NULL},
/* 2DC3 */ { "ETHIOPIC SYLLABLE QYAA",NULL},
/* 2DC4 */ { "ETHIOPIC SYLLABLE QYEE",NULL},
/* 2DC5 */ { "ETHIOPIC SYLLABLE QYE",NULL},
/* 2DC6 */ { "ETHIOPIC SYLLABLE QYO",NULL},
/* 2DC7 */ { NULL,NULL},
/* 2DC8 */ { "ETHIOPIC SYLLABLE KYA",NULL},
/* 2DC9 */ { "ETHIOPIC SYLLABLE KYU",NULL},
/* 2DCA */ { "ETHIOPIC SYLLABLE KYI",NULL},
/* 2DCB */ { "ETHIOPIC SYLLABLE KYAA",NULL},
/* 2DCC */ { "ETHIOPIC SYLLABLE KYEE",NULL},
/* 2DCD */ { "ETHIOPIC SYLLABLE KYE",NULL},
/* 2DCE */ { "ETHIOPIC SYLLABLE KYO",NULL},
/* 2DCF */ { NULL,NULL},
/* 2DD0 */ { "ETHIOPIC SYLLABLE XYA","	x (ethiopic syllable hhya - 1E7E0)"},
/* 2DD1 */ { "ETHIOPIC SYLLABLE XYU","	x (ethiopic syllable hhyu - 1E7E1)"},
/* 2DD2 */ { "ETHIOPIC SYLLABLE XYI","	x (ethiopic syllable hhyi - 1E7E2)"},
/* 2DD3 */ { "ETHIOPIC SYLLABLE XYAA","	x (ethiopic syllable hhyaa - 1E7E3)"},
/* 2DD4 */ { "ETHIOPIC SYLLABLE XYEE","	x (ethiopic syllable hhyee - 1E7E4)"},
/* 2DD5 */ { "ETHIOPIC SYLLABLE XYE","	x (ethiopic syllable hhye - 1E7E5)"},
/* 2DD6 */ { "ETHIOPIC SYLLABLE XYO","	x (ethiopic syllable hhyo - 1E7E6)"},
/* 2DD7 */ { NULL,NULL},
/* 2DD8 */ { "ETHIOPIC SYLLABLE GYA",NULL},
/* 2DD9 */ { "ETHIOPIC SYLLABLE GYU",NULL},
/* 2DDA */ { "ETHIOPIC SYLLABLE GYI",NULL},
/* 2DDB */ { "ETHIOPIC SYLLABLE GYAA",NULL},
/* 2DDC */ { "ETHIOPIC SYLLABLE GYEE",NULL},
/* 2DDD */ { "ETHIOPIC SYLLABLE GYE",NULL},
/* 2DDE */ { "ETHIOPIC SYLLABLE GYO",NULL},
/* 2DDF */ { NULL,NULL},
/* 2DE0 */ { "COMBINING CYRILLIC LETTER BE",NULL},
/* 2DE1 */ { "COMBINING CYRILLIC LETTER VE",NULL},
/* 2DE2 */ { "COMBINING CYRILLIC LETTER GHE",NULL},
/* 2DE3 */ { "COMBINING CYRILLIC LETTER DE",NULL},
/* 2DE4 */ { "COMBINING CYRILLIC LETTER ZHE",NULL},
/* 2DE5 */ { "COMBINING CYRILLIC LETTER ZE",NULL},
/* 2DE6 */ { "COMBINING CYRILLIC LETTER KA",NULL},
/* 2DE7 */ { "COMBINING CYRILLIC LETTER EL",NULL},
/* 2DE8 */ { "COMBINING CYRILLIC LETTER EM","	x (combining latin letter small capital m - 1DDF)"},
/* 2DE9 */ { "COMBINING CYRILLIC LETTER EN",NULL},
/* 2DEA */ { "COMBINING CYRILLIC LETTER O","	x (combining latin small letter o - 0366)"},
/* 2DEB */ { "COMBINING CYRILLIC LETTER PE",NULL},
/* 2DEC */ { "COMBINING CYRILLIC LETTER ER",NULL},
/* 2DED */ { "COMBINING CYRILLIC LETTER ES","	x (combining latin small letter c - 0368)"},
/* 2DEE */ { "COMBINING CYRILLIC LETTER TE",NULL},
/* 2DEF */ { "COMBINING CYRILLIC LETTER HA","	x (combining latin small letter x - 036F)"},
/* 2DF0 */ { "COMBINING CYRILLIC LETTER TSE",NULL},
/* 2DF1 */ { "COMBINING CYRILLIC LETTER CHE",NULL},
/* 2DF2 */ { "COMBINING CYRILLIC LETTER SHA",NULL},
/* 2DF3 */ { "COMBINING CYRILLIC LETTER SHCHA",NULL},
/* 2DF4 */ { "COMBINING CYRILLIC LETTER FITA",NULL},
/* 2DF5 */ { "COMBINING CYRILLIC LETTER ES-TE","	* preferred representation is the sequence: 2DED 2DEE"},
/* 2DF6 */ { "COMBINING CYRILLIC LETTER A","	x (combining latin small letter a - 0363)"},
/* 2DF7 */ { "COMBINING CYRILLIC LETTER IE","	x (combining latin small letter e - 0364)"},
/* 2DF8 */ { "COMBINING CYRILLIC LETTER DJERV",NULL},
/* 2DF9 */ { "COMBINING CYRILLIC LETTER MONOGRAPH UK",NULL},
/* 2DFA */ { "COMBINING CYRILLIC LETTER YAT",NULL},
/* 2DFB */ { "COMBINING CYRILLIC LETTER YU",NULL},
/* 2DFC */ { "COMBINING CYRILLIC LETTER IOTIFIED A",NULL},
/* 2DFD */ { "COMBINING CYRILLIC LETTER LITTLE YUS",NULL},
/* 2DFE */ { "COMBINING CYRILLIC LETTER BIG YUS",NULL},
/* 2DFF */ { "COMBINING CYRILLIC LETTER IOTIFIED BIG YUS",NULL}
};

UN_DLL_LOCAL
static const struct unicode_nameannot una_00_2E[] = {
/* 2E00 */ { "RIGHT ANGLE SUBSTITUTION MARKER","	x (top left corner - 231C)"},
/* 2E01 */ { "RIGHT ANGLE DOTTED SUBSTITUTION MARKER",NULL},
/* 2E02 */ { "LEFT SUBSTITUTION BRACKET",NULL},
/* 2E03 */ { "RIGHT SUBSTITUTION BRACKET",NULL},
/* 2E04 */ { "LEFT DOTTED SUBSTITUTION BRACKET",NULL},
/* 2E05 */ { "RIGHT DOTTED SUBSTITUTION BRACKET",NULL},
/* 2E06 */ { "RAISED INTERPOLATION MARKER","	x (down tack - 22A4)"},
/* 2E07 */ { "RAISED DOTTED INTERPOLATION MARKER",NULL},
/* 2E08 */ { "DOTTED TRANSPOSITION MARKER",NULL},
/* 2E09 */ { "LEFT TRANSPOSITION BRACKET",NULL},
/* 2E0A */ { "RIGHT TRANSPOSITION BRACKET",NULL},
/* 2E0B */ { "RAISED SQUARE","	* used as an opening raised omission bracket"},
/* 2E0C */ { "LEFT RAISED OMISSION BRACKET","	* used as an opening or closing raised omission bracket"},
/* 2E0D */ { "RIGHT RAISED OMISSION BRACKET","	* used as a closing or opening raised omission bracket"},
/* 2E0E */ { "EDITORIAL CORONIS","	x (greek koronis - 1FBD)"},
/* 2E0F */ { "PARAGRAPHOS",NULL},
/* 2E10 */ { "FORKED PARAGRAPHOS",NULL},
/* 2E11 */ { "REVERSED FORKED PARAGRAPHOS",NULL},
/* 2E12 */ { "HYPODIASTOLE","	= ypodiastoli"},
/* 2E13 */ { "DOTTED OBELOS","	* glyph variants may look like '00F7' or '2238'\n"
	"	x (commercial minus sign - 2052)"},
/* 2E14 */ { "DOWNWARDS ANCORA","	* contrary to its formal name this symbol points upwards"},
/* 2E15 */ { "UPWARDS ANCORA","	* contrary to its formal name this symbol points downwards"},
/* 2E16 */ { "DOTTED RIGHT-POINTING ANGLE","	= diple periestigmene"},
/* 2E17 */ { "DOUBLE OBLIQUE HYPHEN","	* used in ancient Near-Eastern linguistics\n"
	"	* hyphen in Fraktur text uses 002D or 2010, but with a '2E17' glyph in Fraktur fonts\n"
	"	x (hyphen-minus - 002D)\n"
	"	x (equals sign - 003D)\n"
	"	x (hyphen - 2010)\n"
	"	x (double hyphen - 2E40)\n"
	"	x (oblique hyphen - 2E5D)"},
/* 2E18 */ { "INVERTED INTERROBANG","	= gnaborretni\n"
	"	x (interrobang - 203D)"},
/* 2E19 */ { "PALM BRANCH","	* used as a separator"},
/* 2E1A */ { "HYPHEN WITH DIAERESIS","	* indicates umlaut of the stem vowel of a plural form"},
/* 2E1B */ { "TILDE WITH RING ABOVE","	* indicates change in case for derived form"},
/* 2E1C */ { "LEFT LOW PARAPHRASE BRACKET",NULL},
/* 2E1D */ { "RIGHT LOW PARAPHRASE BRACKET","	* used in N'Ko"},
/* 2E1E */ { "TILDE WITH DOT ABOVE","	* indicates derived form changes to uppercase"},
/* 2E1F */ { "TILDE WITH DOT BELOW","	* indicates derived form changes to lowercase"},
/* 2E20 */ { "LEFT VERTICAL BAR WITH QUILL","	x (left square bracket with quill - 2045)"},
/* 2E21 */ { "RIGHT VERTICAL BAR WITH QUILL",NULL},
/* 2E22 */ { "TOP LEFT HALF BRACKET","	x (left ceiling - 2308)\n"
	"	x (top left corner - 231C)\n"
	"	x (left corner bracket - 300C)"},
/* 2E23 */ { "TOP RIGHT HALF BRACKET",NULL},
/* 2E24 */ { "BOTTOM LEFT HALF BRACKET",NULL},
/* 2E25 */ { "BOTTOM RIGHT HALF BRACKET",NULL},
/* 2E26 */ { "LEFT SIDEWAYS U BRACKET","	x (subset of - 2282)"},
/* 2E27 */ { "RIGHT SIDEWAYS U BRACKET","	x (superset of - 2283)"},
/* 2E28 */ { "LEFT DOUBLE PARENTHESIS","	x (left white parenthesis - 2985)\n"
	"	x (fullwidth left white parenthesis - FF5F)"},
/* 2E29 */ { "RIGHT DOUBLE PARENTHESIS",NULL},
/* 2E2A */ { "TWO DOTS OVER ONE DOT PUNCTUATION",NULL},
/* 2E2B */ { "ONE DOT OVER TWO DOTS PUNCTUATION",NULL},
/* 2E2C */ { "SQUARED FOUR DOT PUNCTUATION",NULL},
/* 2E2D */ { "FIVE DOT MARK",NULL},
/* 2E2E */ { "REVERSED QUESTION MARK","	= punctus percontativus\n"
	"	x (question mark - 003F)\n"
	"	x (inverted question mark - 00BF)\n"
	"	x (arabic question mark - 061F)"},
/* 2E2F */ { "VERTICAL TILDE","	* used for Cyrillic yerik\n"
	"	x (combining vertical tilde - 033E)\n"
	"	x (cyrillic payerok - A67F)"},
/* 2E30 */ { "RING POINT","	* used in Avestan\n"
	"	x (ring operator - 2218)\n"
	"	x (white bullet - 25E6)"},
/* 2E31 */ { "WORD SEPARATOR MIDDLE DOT","	* used in Avestan, Samaritan, ...\n"
	"	x (middle dot - 00B7)"},
/* 2E32 */ { "TURNED COMMA","	* indicates nasalization\n"
	"	x (arabic comma - 060C)"},
/* 2E33 */ { "RAISED DOT","	* glyph position intermediate between 002E and 00B7\n"
	"	x (full stop - 002E)\n"
	"	x (middle dot - 00B7)"},
/* 2E34 */ { "RAISED COMMA","	x (comma - 002C)"},
/* 2E35 */ { "TURNED SEMICOLON","	* indicates sudden glottal closure\n"
	"	x (arabic semicolon - 061B)"},
/* 2E36 */ { "DAGGER WITH LEFT GUARD","	* indicates retracted pronunciation"},
/* 2E37 */ { "DAGGER WITH RIGHT GUARD","	* indicates advanced pronunciation"},
/* 2E38 */ { "TURNED DAGGER","	* indicates retroflex pronunciation\n"
	"	x (dagger - 2020)"},
/* 2E39 */ { "TOP HALF SECTION SIGN","	* indicates pronunciation on one side of the mouth only\n"
	"	x (section sign - 00A7)"},
/* 2E3A */ { "TWO-EM DASH","	= omission dash\n"
	"	* may be used in Chinese for abrupt change of thought, inserting new content, or continuation of tone or sound\n"
	"	x (em dash - 2014)"},
/* 2E3B */ { "THREE-EM DASH",NULL},
/* 2E3C */ { "STENOGRAPHIC FULL STOP","	* used in shorthands and stenographies\n"
	"	x (full stop - 002E)\n"
	"	x (canadian syllabics full stop - 166E)"},
/* 2E3D */ { "VERTICAL SIX DOTS","	x (vertical four dots - 205E)\n"
	"	x (dotted fence - 2999)"},
/* 2E3E */ { "WIGGLY VERTICAL LINE","	x (wavy line - 2307)\n"
	"	x (vertical zigzag line - 299A)"},
/* 2E3F */ { "CAPITULUM","	* ancestor of the pilcrow sign\n"
	"	x (pilcrow sign - 00B6)"},
/* 2E40 */ { "DOUBLE HYPHEN","	x (equals sign - 003D)\n"
	"	x (hyphen - 2010)\n"
	"	x (double oblique hyphen - 2E17)\n"
	"	x (katakana-hiragana double hyphen - 30A0)\n"
	"	x (modifier letter short equals sign - A78A)"},
/* 2E41 */ { "REVERSED COMMA","	* Old Hungarian\n"
	"	* used occasionally in Sindhi when Sindhi is written in the Arabic script\n"
	"	x (comma - 002C)\n"
	"	x (arabic comma - 060C)"},
/* 2E42 */ { "DOUBLE LOW-REVERSED-9 QUOTATION MARK","	x (double low-9 quotation mark - 201E)"},
/* 2E43 */ { "DASH WITH LEFT UPTURN","	* functions as a paragraphos in some Slavonic texts\n"
	"	x (paragraphos - 2E0F)"},
/* 2E44 */ { "DOUBLE SUSPENSION MARK","	* indicates omission of letters in a word on some Byzantine Greek seals and coins\n"
	"	x (greek numeral sign - 0374)"},
/* 2E45 */ { "INVERTED LOW KAVYKA",NULL},
/* 2E46 */ { "INVERTED LOW KAVYKA WITH KAVYKA ABOVE","	x (union above intersection - 2A46)\n"
	"	x (modifier breve with inverted breve - AB5B)"},
/* 2E47 */ { "LOW KAVYKA","	x (cyrillic kavyka - A67E)"},
/* 2E48 */ { "LOW KAVYKA WITH DOT",NULL},
/* 2E49 */ { "DOUBLE STACKED COMMA",NULL},
/* 2E4A */ { "DOTTED SOLIDUS","	= virgula suspensiva\n"
	"	* indicates a medial disjunction more than solidus but less than punctus elevatus"},
/* 2E4B */ { "TRIPLE DAGGER","	x (dagger - 2020)\n"
	"	x (double dagger - 2021)"},
/* 2E4C */ { "MEDIEVAL COMMA","	* indicates a minor medial pause or disjunction of sense"},
/* 2E4D */ { "PARAGRAPHUS MARK","	* indicates the beginning of a paragraph, section, stanza, or proposition\n"
	"	x (pilcrow sign - 00B6)\n"
	"	x (reversed pilcrow sign - 204B)\n"
	"	x (paragraphos - 2E0F)\n"
	"	x (capitulum - 2E3F)"},
/* 2E4E */ { "PUNCTUS ELEVATUS MARK","		* This mark indicates a major intermediate pause where the sensus is complete but the sentence is not; this is similar in some regards to the modern use of a semicolon."},
/* 2E4F */ { "CORNISH VERSE DIVIDER",NULL},
/* 2E50 */ { "CROSS PATTY WITH RIGHT CROSSBAR","	x (maltese cross - 2720)\n"
	"	x (latin capital letter half h - 2C75)"},
/* 2E51 */ { "CROSS PATTY WITH LEFT CROSSBAR","	x (latin capital letter reversed half h - A7F5)"},
/* 2E52 */ { "TIRONIAN SIGN CAPITAL ET","	x (tironian sign et - 204A)"},
/* 2E53 */ { "MEDIEVAL EXCLAMATION MARK",NULL},
/* 2E54 */ { "MEDIEVAL QUESTION MARK",NULL},
/* 2E55 */ { "LEFT SQUARE BRACKET WITH STROKE","	* optional ellipsis\n"
	"	x (left square bracket with quill - 2045)"},
/* 2E56 */ { "RIGHT SQUARE BRACKET WITH STROKE",NULL},
/* 2E57 */ { "LEFT SQUARE BRACKET WITH DOUBLE STROKE","	* obligatory ellipsis"},
/* 2E58 */ { "RIGHT SQUARE BRACKET WITH DOUBLE STROKE",NULL},
/* 2E59 */ { "TOP HALF LEFT PARENTHESIS",NULL},
/* 2E5A */ { "TOP HALF RIGHT PARENTHESIS",NULL},
/* 2E5B */ { "BOTTOM HALF LEFT PARENTHESIS",NULL},
/* 2E5C */ { "BOTTOM HALF RIGHT PARENTHESIS",NULL},
/* 2E5D */ { "OBLIQUE HYPHEN","	x (hyphen - 2010)\n"
	"	x (double oblique hyphen - 2E17)"},
/* 2E5E */ { NULL,NULL},
/* 2E5F */ { NULL,NULL},
/* 2E60 */ { NULL,NULL},
/* 2E61 */ { NULL,NULL},
/* 2E62 */ { NULL,NULL},
/* 2E63 */ { NULL,NULL},
/* 2E64 */ { NULL,NULL},
/* 2E65 */ { NULL,NULL},
/* 2E66 */ { NULL,NULL},
/* 2E67 */ { NULL,NULL},
/* 2E68 */ { NULL,NULL},
/* 2E69 */ { NULL,NULL},
/* 2E6A */ { NULL,NULL},
/* 2E6B */ { NULL,NULL},
/* 2E6C */ { NULL,NULL},
/* 2E6D */ { NULL,NULL},
/* 2E6E */ { NULL,NULL},
/* 2E6F */ { NULL,NULL},
/* 2E70 */ { NULL,NULL},
/* 2E71 */ { NULL,NULL},
/* 2E72 */ { NULL,NULL},
/* 2E73 */ { NULL,NULL},
/* 2E74 */ { NULL,NULL},
/* 2E75 */ { NULL,NULL},
/* 2E76 */ { NULL,NULL},
/* 2E77 */ { NULL,NULL},
/* 2E78 */ { NULL,NULL},
/* 2E79 */ { NULL,NULL},
/* 2E7A */ { NULL,NULL},
/* 2E7B */ { NULL,NULL},
/* 2E7C */ { NULL,NULL},
/* 2E7D */ { NULL,NULL},
/* 2E7E */ { NULL,NULL},
/* 2E7F */ { NULL,NULL},
/* 2E80 */ { "CJK RADICAL REPEAT","	* variant of Kangxi Radical 3"},
/* 2E81 */ { "CJK RADICAL CLIFF","	* variant of Kangxi Radical 27\n"
	"	x 5382\n"
	"	x 20086"},
/* 2E82 */ { "CJK RADICAL SECOND ONE","	* variant of Kangxi Radical 5\n"
	"	x 4E5B"},
/* 2E83 */ { "CJK RADICAL SECOND TWO","	* variant of Kangxi Radical 5\n"
	"	x 4E5A"},
/* 2E84 */ { "CJK RADICAL SECOND THREE","	* variant of Kangxi Radical 5\n"
	"	x 4E59"},
/* 2E85 */ { "CJK RADICAL PERSON","	* variant of Kangxi Radical 9\n"
	"	* form used on left side\n"
	"	x 4EBB"},
/* 2E86 */ { "CJK RADICAL BOX","	* variant of Kangxi Radical 13\n"
	"	x 5182"},
/* 2E87 */ { "CJK RADICAL TABLE","	* variant of Kangxi Radical 16\n"
	"	x 51E0\n"
	"	x 20628"},
/* 2E88 */ { "CJK RADICAL KNIFE ONE","	* variant of Kangxi Radical 18\n"
	"	* form used at top\n"
	"	x 5200\n"
	"	x 2008A"},
/* 2E89 */ { "CJK RADICAL KNIFE TWO","	* variant of Kangxi Radical 18\n"
	"	* form used on right side\n"
	"	x 5202"},
/* 2E8A */ { "CJK RADICAL DIVINATION","	* variant of Kangxi Radical 25\n"
	"	* form used at top\n"
	"	x 535C"},
/* 2E8B */ { "CJK RADICAL SEAL","	* variant of Kangxi Radical 26\n"
	"	* form used at bottom\n"
	"	x 353E"},
/* 2E8C */ { "CJK RADICAL SMALL ONE","	* variant of Kangxi Radical 42\n"
	"	* form used at top\n"
	"	x 5C0F"},
/* 2E8D */ { "CJK RADICAL SMALL TWO","	* variant of Kangxi Radical 42\n"
	"	* form used at top\n"
	"	x 5C0F\n"
	"	x 2D544"},
/* 2E8E */ { "CJK RADICAL LAME ONE","	* variant of Kangxi Radical 43\n"
	"	x 5C22\n"
	"	x 5140"},
/* 2E8F */ { "CJK RADICAL LAME TWO","	* variant of Kangxi Radical 43\n"
	"	x 5C23"},
/* 2E90 */ { "CJK RADICAL LAME THREE","	* variant of Kangxi Radical 43\n"
	"	x 5C22"},
/* 2E91 */ { "CJK RADICAL LAME FOUR","	* variant of Kangxi Radical 43\n"
	"	x 5C23\n"
	"	x 21BC2"},
/* 2E92 */ { "CJK RADICAL SNAKE","	* variant of Kangxi Radical 49\n"
	"	x 5DF3"},
/* 2E93 */ { "CJK RADICAL THREAD","	* variant of Kangxi Radical 52\n"
	"	x 5E7A"},
/* 2E94 */ { "CJK RADICAL SNOUT ONE","	* variant of Kangxi Radical 58\n"
	"	x 5F51"},
/* 2E95 */ { "CJK RADICAL SNOUT TWO","	* variant of Kangxi Radical 58\n"
	"	* actually a form of the radical for hand, despite its resemblance in shape to the radical for snout\n"
	"	x 5F50\n"
	"	x 2B739"},
/* 2E96 */ { "CJK RADICAL HEART ONE","	* variant of Kangxi Radical 61\n"
	"	* form used on left side\n"
	"	x 5FC4"},
/* 2E97 */ { "CJK RADICAL HEART TWO","	* variant of Kangxi Radical 61\n"
	"	* form used at bottom\n"
	"	x 38FA\n"
	"	x 5FC3"},
/* 2E98 */ { "CJK RADICAL HAND","	* variant of Kangxi Radical 64\n"
	"	* form used on left side\n"
	"	x 624C"},
/* 2E99 */ { "CJK RADICAL RAP","	* variant of Kangxi Radical 66\n"
	"	* form used on right side\n"
	"	x 6535"},
/* 2E9A */ { NULL,"	x (kangxi radical not - 2F46)"},
/* 2E9B */ { "CJK RADICAL CHOKE","	* variant of Kangxi Radical 71\n"
	"	x 65E1"},
/* 2E9C */ { "CJK RADICAL SUN","	* variant of Kangxi Radical 72\n"
	"	* actually a form of the radical for hat, despite its resemblance in shape to the radical for sun\n"
	"	x 5183\n"
	"	x 65E5"},
/* 2E9D */ { "CJK RADICAL MOON","	* variant of Kangxi Radical 74\n"
	"	x 6708"},
/* 2E9E */ { "CJK RADICAL DEATH","	* variant of Kangxi Radical 78\n"
	"	x 6B7A"},
/* 2E9F */ { "CJK RADICAL MOTHER","	* variant of Kangxi Radical 80\n"
	"	# 6BCD"},
/* 2EA0 */ { "CJK RADICAL CIVILIAN","	* variant of Kangxi Radical 83\n"
	"	x 6C11"},
/* 2EA1 */ { "CJK RADICAL WATER ONE","	* variant of Kangxi Radical 85\n"
	"	* form used on left side\n"
	"	x 6C35"},
/* 2EA2 */ { "CJK RADICAL WATER TWO","	* variant of Kangxi Radical 85\n"
	"	* form used (rarely) at bottom\n"
	"	x 6C3A"},
/* 2EA3 */ { "CJK RADICAL FIRE","	* variant of Kangxi Radical 86\n"
	"	* form used at bottom\n"
	"	x 706C"},
/* 2EA4 */ { "CJK RADICAL PAW ONE","	* variant of Kangxi Radical 87\n"
	"	* form used at top\n"
	"	x 722B"},
/* 2EA5 */ { "CJK RADICAL PAW TWO","	* variant of Kangxi Radical 87\n"
	"	* form used at top\n"
	"	x 722B"},
/* 2EA6 */ { "CJK RADICAL SIMPLIFIED HALF TREE TRUNK","	* simplified variant of Kangxi Radical 90\n"
	"	x 4E2C"},
/* 2EA7 */ { "CJK RADICAL COW","	* variant of Kangxi Radical 93\n"
	"	x 725B\n"
	"	x 20092"},
/* 2EA8 */ { "CJK RADICAL DOG","	* variant of Kangxi Radical 94\n"
	"	* form used on left side\n"
	"	x 72AD"},
/* 2EA9 */ { "CJK RADICAL JADE","	* variant of Kangxi Radical 96\n"
	"	* form used on left side\n"
	"	x 738B\n"
	"	x 248E9"},
/* 2EAA */ { "CJK RADICAL BOLT OF CLOTH","	* variant of Kangxi Radical 103\n"
	"	* form used on left side\n"
	"	x 758B\n"
	"	x 24D14"},
/* 2EAB */ { "CJK RADICAL EYE","	* variant of Kangxi Radical 109\n"
	"	* variant of Kangxi Radical 122\n"
	"	* form used at top\n"
	"	x (cjk radical net two - 2EB2)\n"
	"	x 76EE\n"
	"	x 7F52"},
/* 2EAC */ { "CJK RADICAL SPIRIT ONE","	* variant of Kangxi Radical 113\n"
	"	x 793A"},
/* 2EAD */ { "CJK RADICAL SPIRIT TWO","	* variant of Kangxi Radical 113\n"
	"	x 793B"},
/* 2EAE */ { "CJK RADICAL BAMBOO","	* variant of Kangxi Radical 118\n"
	"	x 7AF9\n"
	"	x 25AD7"},
/* 2EAF */ { "CJK RADICAL SILK","	* variant of Kangxi Radical 120\n"
	"	* form used on left side\n"
	"	x 7CF9"},
/* 2EB0 */ { "CJK RADICAL C-SIMPLIFIED SILK","	* simplified Chinese variant of Kangxi Radical 120\n"
	"	* form used on left side\n"
	"	x 7E9F"},
/* 2EB1 */ { "CJK RADICAL NET ONE","	* variant of Kangxi Radical 122\n"
	"	x 7F53"},
/* 2EB2 */ { "CJK RADICAL NET TWO","	* variant of Kangxi Radical 109\n"
	"	* variant of Kangxi Radical 122\n"
	"	x (cjk radical eye - 2EAB)\n"
	"	x 7F52\n"
	"	x 26270"},
/* 2EB3 */ { "CJK RADICAL NET THREE","	* variant of Kangxi Radical 122\n"
	"	x 34C1\n"
	"	x 7F51"},
/* 2EB4 */ { "CJK RADICAL NET FOUR","	* variant of Kangxi Radical 122\n"
	"	x 34C1\n"
	"	x 7F51"},
/* 2EB5 */ { "CJK RADICAL MESH","	* variant of Kangxi Radical 122\n"
	"	x 2626B"},
/* 2EB6 */ { "CJK RADICAL SHEEP","	* variant of Kangxi Radical 123\n"
	"	* form used on left side\n"
	"	x 7F8A"},
/* 2EB7 */ { "CJK RADICAL RAM","	* variant of Kangxi Radical 123\n"
	"	* form used at top\n"
	"	x 7F8A\n"
	"	x 2634C"},
/* 2EB8 */ { "CJK RADICAL EWE","	* variant of Kangxi Radical 123\n"
	"	x 7F8B\n"
	"	x 2634B"},
/* 2EB9 */ { "CJK RADICAL OLD","	* variant of Kangxi Radical 125\n"
	"	x 8002"},
/* 2EBA */ { "CJK RADICAL BRUSH ONE","	* variant of Kangxi Radical 129\n"
	"	x 8080"},
/* 2EBB */ { "CJK RADICAL BRUSH TWO","	* variant of Kangxi Radical 129\n"
	"	x 807F"},
/* 2EBC */ { "CJK RADICAL MEAT","	* variant of Kangxi Radical 130\n"
	"	x 8089"},
/* 2EBD */ { "CJK RADICAL MORTAR","	* variant of Kangxi Radical 134\n"
	"	x 81FC\n"
	"	x 26951"},
/* 2EBE */ { "CJK RADICAL GRASS ONE","	* simplified variant of Kangxi Radical 140\n"
	"	x 8279"},
/* 2EBF */ { "CJK RADICAL GRASS TWO","	* variant of Kangxi Radical 140\n"
	"	x 8279"},
/* 2EC0 */ { "CJK RADICAL GRASS THREE","	* variant of Kangxi Radical 140\n"
	"	x 8279"},
/* 2EC1 */ { "CJK RADICAL TIGER","	* variant of Kangxi Radical 141\n"
	"	x 864E"},
/* 2EC2 */ { "CJK RADICAL CLOTHES","	* variant of Kangxi Radical 145\n"
	"	* form used on left side\n"
	"	x 8864"},
/* 2EC3 */ { "CJK RADICAL WEST ONE","	* variant of Kangxi Radical 146\n"
	"	* form used at top\n"
	"	x 8980"},
/* 2EC4 */ { "CJK RADICAL WEST TWO","	* variant of Kangxi Radical 146\n"
	"	* form used on left side\n"
	"	x 897F"},
/* 2EC5 */ { "CJK RADICAL C-SIMPLIFIED SEE","	* simplified Chinese variant of Kangxi Radical 147\n"
	"	x 89C1"},
/* 2EC6 */ { "CJK RADICAL SIMPLIFIED HORN","	* simplified Chinese variant of Kangxi Radical 148\n"
	"	x 89D2"},
/* 2EC7 */ { "CJK RADICAL HORN","	* variant of Kangxi Radical 148\n"
	"	x 278B2"},
/* 2EC8 */ { "CJK RADICAL C-SIMPLIFIED SPEECH","	* simplified Chinese variant of Kangxi Radical 149\n"
	"	x 8BA0"},
/* 2EC9 */ { "CJK RADICAL C-SIMPLIFIED SHELL","	* simplified Chinese variant of Kangxi Radical 154\n"
	"	x 8D1D"},
/* 2ECA */ { "CJK RADICAL FOOT","	* variant of Kangxi Radical 157\n"
	"	* form used on left side\n"
	"	x 8DB3\n"
	"	x 27FB7"},
/* 2ECB */ { "CJK RADICAL C-SIMPLIFIED CART","	* simplified Chinese variant of Kangxi Radical 159\n"
	"	x 8F66"},
/* 2ECC */ { "CJK RADICAL SIMPLIFIED WALK","	* simplified variant of Kangxi Radical 162\n"
	"	x 8FB6"},
/* 2ECD */ { "CJK RADICAL WALK ONE","	* variant of Kangxi Radical 162\n"
	"	x 8FB6"},
/* 2ECE */ { "CJK RADICAL WALK TWO","	* variant of Kangxi Radical 162\n"
	"	x 8FB6"},
/* 2ECF */ { "CJK RADICAL CITY","	* variant of Kangxi Radical 163\n"
	"	* form used on right side\n"
	"	x 9091"},
/* 2ED0 */ { "CJK RADICAL C-SIMPLIFIED GOLD","	* simplified Chinese variant of Kangxi Radical 167\n"
	"	x 9485"},
/* 2ED1 */ { "CJK RADICAL LONG ONE","	* variant of Kangxi Radical 168\n"
	"	x 9577"},
/* 2ED2 */ { "CJK RADICAL LONG TWO","	* variant of Kangxi Radical 168\n"
	"	* form used on left side\n"
	"	x 9578"},
/* 2ED3 */ { "CJK RADICAL C-SIMPLIFIED LONG","	* simplified Chinese variant of Kangxi Radical 168\n"
	"	x 957F"},
/* 2ED4 */ { "CJK RADICAL C-SIMPLIFIED GATE","	* simplified Chinese variant of Kangxi Radical 169\n"
	"	x 95E8"},
/* 2ED5 */ { "CJK RADICAL MOUND ONE","	* variant of Kangxi Radical 170\n"
	"	x 961C\n"
	"	x 28E0F"},
/* 2ED6 */ { "CJK RADICAL MOUND TWO","	* variant of Kangxi Radical 170\n"
	"	* form used on left side\n"
	"	x 961D"},
/* 2ED7 */ { "CJK RADICAL RAIN","	* variant of Kangxi Radical 173\n"
	"	x 96E8"},
/* 2ED8 */ { "CJK RADICAL BLUE","	* variant of Kangxi Radical 174\n"
	"	x 9752"},
/* 2ED9 */ { "CJK RADICAL C-SIMPLIFIED TANNED LEATHER","	* simplified Chinese variant of Kangxi Radical 178\n"
	"	x 97E6"},
/* 2EDA */ { "CJK RADICAL C-SIMPLIFIED LEAF","	* simplified Chinese variant of Kangxi Radical 181\n"
	"	x 9875"},
/* 2EDB */ { "CJK RADICAL C-SIMPLIFIED WIND","	* simplified Chinese variant of Kangxi Radical 182\n"
	"	x 98CE"},
/* 2EDC */ { "CJK RADICAL C-SIMPLIFIED FLY","	* simplified Chinese variant of Kangxi Radical 183\n"
	"	x 98DE"},
/* 2EDD */ { "CJK RADICAL EAT ONE","	* variant of Kangxi Radical 184\n"
	"	* form used at bottom\n"
	"	x 98DF"},
/* 2EDE */ { "CJK RADICAL EAT TWO","	* variant of Kangxi Radical 184\n"
	"	* form used on left side\n"
	"	x 2967F"},
/* 2EDF */ { "CJK RADICAL EAT THREE","	* variant of Kangxi Radical 184\n"
	"	* form used on left side\n"
	"	x 98E0"},
/* 2EE0 */ { "CJK RADICAL C-SIMPLIFIED EAT","	* simplified Chinese variant of Kangxi Radical 184\n"
	"	* form used on left side\n"
	"	x 9963"},
/* 2EE1 */ { "CJK RADICAL HEAD","	* variant of Kangxi Radical 185\n"
	"	x 29810"},
/* 2EE2 */ { "CJK RADICAL C-SIMPLIFIED HORSE","	* simplified Chinese variant of Kangxi Radical 187\n"
	"	x 9A6C"},
/* 2EE3 */ { "CJK RADICAL BONE","	* simplified Chinese variant of Kangxi Radical 188\n"
	"	x 9AA8"},
/* 2EE4 */ { "CJK RADICAL GHOST","	* variant of Kangxi Radical 194\n"
	"	x 9B3C"},
/* 2EE5 */ { "CJK RADICAL C-SIMPLIFIED FISH","	* simplified Chinese variant of Kangxi Radical 195\n"
	"	x 9C7C"},
/* 2EE6 */ { "CJK RADICAL C-SIMPLIFIED BIRD","	* simplified Chinese variant of Kangxi Radical 196\n"
	"	x 9E1F"},
/* 2EE7 */ { "CJK RADICAL C-SIMPLIFIED SALT","	* simplified Chinese variant of Kangxi Radical 197\n"
	"	x 5364"},
/* 2EE8 */ { "CJK RADICAL SIMPLIFIED WHEAT","	* simplified variant of Kangxi Radical 199\n"
	"	x 9EA6"},
/* 2EE9 */ { "CJK RADICAL SIMPLIFIED YELLOW","	* simplified variant of Kangxi Radical 201\n"
	"	x 9EC4"},
/* 2EEA */ { "CJK RADICAL C-SIMPLIFIED FROG","	* simplified Chinese variant of Kangxi Radical 205\n"
	"	x 9EFE"},
/* 2EEB */ { "CJK RADICAL J-SIMPLIFIED EVEN","	* simplified Japanese variant of Kangxi Radical 210\n"
	"	x 6589"},
/* 2EEC */ { "CJK RADICAL C-SIMPLIFIED EVEN","	* simplified Chinese variant of Kangxi Radical 210\n"
	"	x 9F50"},
/* 2EED */ { "CJK RADICAL J-SIMPLIFIED TOOTH","	* simplified Japanese variant of Kangxi Radical 211\n"
	"	x 6B6F"},
/* 2EEE */ { "CJK RADICAL C-SIMPLIFIED TOOTH","	* simplified Chinese variant of Kangxi Radical 211\n"
	"	x 9F7F"},
/* 2EEF */ { "CJK RADICAL J-SIMPLIFIED DRAGON","	* simplified Japanese variant of Kangxi Radical 212\n"
	"	x 7ADC\n"
	"	x 9F8D"},
/* 2EF0 */ { "CJK RADICAL C-SIMPLIFIED DRAGON","	* simplified Chinese variant of Kangxi Radical 212\n"
	"	x 9F99"},
/* 2EF1 */ { "CJK RADICAL TURTLE","	* variant of Kangxi Radical 213\n"
	"	x 9F9C"},
/* 2EF2 */ { "CJK RADICAL J-SIMPLIFIED TURTLE","	* simplified Japanese variant of Kangxi Radical 213\n"
	"	x 4E80"},
/* 2EF3 */ { "CJK RADICAL C-SIMPLIFIED TURTLE","	* simplified Chinese variant of Kangxi Radical 213\n"
	"	# 9F9F"},
/* 2EF4 */ { NULL,NULL},
/* 2EF5 */ { NULL,NULL},
/* 2EF6 */ { NULL,NULL},
/* 2EF7 */ { NULL,NULL},
/* 2EF8 */ { NULL,NULL},
/* 2EF9 */ { NULL,NULL},
/* 2EFA */ { NULL,NULL},
/* 2EFB */ { NULL,NULL},
/* 2EFC */ { NULL,NULL},
/* 2EFD */ { NULL,NULL},
/* 2EFE */ { NULL,NULL},
/* 2EFF */ { NULL,NULL}
};

UN_DLL_LOCAL
static const struct unicode_nameannot una_00_2F[] = {
/* 2F00 */ { "KANGXI RADICAL ONE","	* Kangxi Radical 1\n"
	"	# 4E00"},
/* 2F01 */ { "KANGXI RADICAL LINE","	* Kangxi Radical 2\n"
	"	# 4E28"},
/* 2F02 */ { "KANGXI RADICAL DOT","	* Kangxi Radical 3\n"
	"	# 4E36"},
/* 2F03 */ { "KANGXI RADICAL SLASH","	* Kangxi Radical 4\n"
	"	# 4E3F"},
/* 2F04 */ { "KANGXI RADICAL SECOND","	* Kangxi Radical 5\n"
	"	# 4E59"},
/* 2F05 */ { "KANGXI RADICAL HOOK","	* Kangxi Radical 6\n"
	"	# 4E85"},
/* 2F06 */ { "KANGXI RADICAL TWO","	* Kangxi Radical 7\n"
	"	# 4E8C"},
/* 2F07 */ { "KANGXI RADICAL LID","	* Kangxi Radical 8\n"
	"	# 4EA0"},
/* 2F08 */ { "KANGXI RADICAL MAN","	* Kangxi Radical 9\n"
	"	# 4EBA"},
/* 2F09 */ { "KANGXI RADICAL LEGS","	* Kangxi Radical 10\n"
	"	# 513F"},
/* 2F0A */ { "KANGXI RADICAL ENTER","	* Kangxi Radical 11\n"
	"	# 5165"},
/* 2F0B */ { "KANGXI RADICAL EIGHT","	* Kangxi Radical 12\n"
	"	# 516B"},
/* 2F0C */ { "KANGXI RADICAL DOWN BOX","	* Kangxi Radical 13\n"
	"	# 5182"},
/* 2F0D */ { "KANGXI RADICAL COVER","	* Kangxi Radical 14\n"
	"	# 5196"},
/* 2F0E */ { "KANGXI RADICAL ICE","	* Kangxi Radical 15\n"
	"	# 51AB"},
/* 2F0F */ { "KANGXI RADICAL TABLE","	* Kangxi Radical 16\n"
	"	# 51E0"},
/* 2F10 */ { "KANGXI RADICAL OPEN BOX","	* Kangxi Radical 17\n"
	"	# 51F5"},
/* 2F11 */ { "KANGXI RADICAL KNIFE","	* Kangxi Radical 18\n"
	"	# 5200"},
/* 2F12 */ { "KANGXI RADICAL POWER","	* Kangxi Radical 19\n"
	"	# 529B"},
/* 2F13 */ { "KANGXI RADICAL WRAP","	* Kangxi Radical 20\n"
	"	# 52F9"},
/* 2F14 */ { "KANGXI RADICAL SPOON","	* Kangxi Radical 21\n"
	"	# 5315"},
/* 2F15 */ { "KANGXI RADICAL RIGHT OPEN BOX","	* Kangxi Radical 22\n"
	"	# 531A"},
/* 2F16 */ { "KANGXI RADICAL HIDING ENCLOSURE","	* Kangxi Radical 23\n"
	"	# 5338"},
/* 2F17 */ { "KANGXI RADICAL TEN","	* Kangxi Radical 24\n"
	"	# 5341"},
/* 2F18 */ { "KANGXI RADICAL DIVINATION","	* Kangxi Radical 25\n"
	"	# 535C"},
/* 2F19 */ { "KANGXI RADICAL SEAL","	* Kangxi Radical 26\n"
	"	# 5369"},
/* 2F1A */ { "KANGXI RADICAL CLIFF","	* Kangxi Radical 27\n"
	"	# 5382"},
/* 2F1B */ { "KANGXI RADICAL PRIVATE","	* Kangxi Radical 28\n"
	"	# 53B6"},
/* 2F1C */ { "KANGXI RADICAL AGAIN","	* Kangxi Radical 29\n"
	"	# 53C8"},
/* 2F1D */ { "KANGXI RADICAL MOUTH","	* Kangxi Radical 30\n"
	"	# 53E3"},
/* 2F1E */ { "KANGXI RADICAL ENCLOSURE","	* Kangxi Radical 31\n"
	"	# 56D7"},
/* 2F1F */ { "KANGXI RADICAL EARTH","	* Kangxi Radical 32\n"
	"	# 571F"},
/* 2F20 */ { "KANGXI RADICAL SCHOLAR","	* Kangxi Radical 33\n"
	"	# 58EB"},
/* 2F21 */ { "KANGXI RADICAL GO","	* Kangxi Radical 34\n"
	"	# 5902"},
/* 2F22 */ { "KANGXI RADICAL GO SLOWLY","	* Kangxi Radical 35\n"
	"	# 590A"},
/* 2F23 */ { "KANGXI RADICAL EVENING","	* Kangxi Radical 36\n"
	"	# 5915"},
/* 2F24 */ { "KANGXI RADICAL BIG","	* Kangxi Radical 37\n"
	"	# 5927"},
/* 2F25 */ { "KANGXI RADICAL WOMAN","	* Kangxi Radical 38\n"
	"	# 5973"},
/* 2F26 */ { "KANGXI RADICAL CHILD","	* Kangxi Radical 39\n"
	"	# 5B50"},
/* 2F27 */ { "KANGXI RADICAL ROOF","	* Kangxi Radical 40\n"
	"	# 5B80"},
/* 2F28 */ { "KANGXI RADICAL INCH","	* Kangxi Radical 41\n"
	"	# 5BF8"},
/* 2F29 */ { "KANGXI RADICAL SMALL","	* Kangxi Radical 42\n"
	"	# 5C0F"},
/* 2F2A */ { "KANGXI RADICAL LAME","	* Kangxi Radical 43\n"
	"	x 21BC1\n"
	"	# 5C22"},
/* 2F2B */ { "KANGXI RADICAL CORPSE","	* Kangxi Radical 44\n"
	"	# 5C38"},
/* 2F2C */ { "KANGXI RADICAL SPROUT","	* Kangxi Radical 45\n"
	"	# 5C6E"},
/* 2F2D */ { "KANGXI RADICAL MOUNTAIN","	* Kangxi Radical 46\n"
	"	# 5C71"},
/* 2F2E */ { "KANGXI RADICAL RIVER","	* Kangxi Radical 47\n"
	"	# 5DDB"},
/* 2F2F */ { "KANGXI RADICAL WORK","	* Kangxi Radical 48\n"
	"	# 5DE5"},
/* 2F30 */ { "KANGXI RADICAL ONESELF","	* Kangxi Radical 49\n"
	"	# 5DF1"},
/* 2F31 */ { "KANGXI RADICAL TURBAN","	* Kangxi Radical 50\n"
	"	# 5DFE"},
/* 2F32 */ { "KANGXI RADICAL DRY","	* Kangxi Radical 51\n"
	"	# 5E72"},
/* 2F33 */ { "KANGXI RADICAL SHORT THREAD","	* Kangxi Radical 52\n"
	"	# 5E7A"},
/* 2F34 */ { "KANGXI RADICAL DOTTED CLIFF","	* Kangxi Radical 53\n"
	"	# 5E7F"},
/* 2F35 */ { "KANGXI RADICAL LONG STRIDE","	* Kangxi Radical 54\n"
	"	# 5EF4"},
/* 2F36 */ { "KANGXI RADICAL TWO HANDS","	* Kangxi Radical 55\n"
	"	# 5EFE"},
/* 2F37 */ { "KANGXI RADICAL SHOOT","	* Kangxi Radical 56\n"
	"	# 5F0B"},
/* 2F38 */ { "KANGXI RADICAL BOW","	* Kangxi Radical 57\n"
	"	# 5F13"},
/* 2F39 */ { "KANGXI RADICAL SNOUT","	* Kangxi Radical 58\n"
	"	# 5F50"},
/* 2F3A */ { "KANGXI RADICAL BRISTLE","	* Kangxi Radical 59\n"
	"	# 5F61"},
/* 2F3B */ { "KANGXI RADICAL STEP","	* Kangxi Radical 60\n"
	"	# 5F73"},
/* 2F3C */ { "KANGXI RADICAL HEART","	* Kangxi Radical 61\n"
	"	# 5FC3"},
/* 2F3D */ { "KANGXI RADICAL HALBERD","	* Kangxi Radical 62\n"
	"	# 6208"},
/* 2F3E */ { "KANGXI RADICAL DOOR","	* Kangxi Radical 63\n"
	"	# 6236"},
/* 2F3F */ { "KANGXI RADICAL HAND","	* Kangxi Radical 64\n"
	"	# 624B"},
/* 2F40 */ { "KANGXI RADICAL BRANCH","	* Kangxi Radical 65\n"
	"	# 652F"},
/* 2F41 */ { "KANGXI RADICAL RAP","	* Kangxi Radical 66\n"
	"	# 6534"},
/* 2F42 */ { "KANGXI RADICAL SCRIPT","	* Kangxi Radical 67\n"
	"	# 6587"},
/* 2F43 */ { "KANGXI RADICAL DIPPER","	* Kangxi Radical 68\n"
	"	# 6597"},
/* 2F44 */ { "KANGXI RADICAL AXE","	* Kangxi Radical 69\n"
	"	# 65A4"},
/* 2F45 */ { "KANGXI RADICAL SQUARE","	* Kangxi Radical 70\n"
	"	# 65B9"},
/* 2F46 */ { "KANGXI RADICAL NOT","	* Kangxi Radical 71\n"
	"	# 65E0"},
/* 2F47 */ { "KANGXI RADICAL SUN","	* Kangxi Radical 72\n"
	"	# 65E5"},
/* 2F48 */ { "KANGXI RADICAL SAY","	* Kangxi Radical 73\n"
	"	# 66F0"},
/* 2F49 */ { "KANGXI RADICAL MOON","	* Kangxi Radical 74\n"
	"	# 6708"},
/* 2F4A */ { "KANGXI RADICAL TREE","	* Kangxi Radical 75\n"
	"	# 6728"},
/* 2F4B */ { "KANGXI RADICAL LACK","	* Kangxi Radical 76\n"
	"	# 6B20"},
/* 2F4C */ { "KANGXI RADICAL STOP","	* Kangxi Radical 77\n"
	"	# 6B62"},
/* 2F4D */ { "KANGXI RADICAL DEATH","	* Kangxi Radical 78\n"
	"	# 6B79"},
/* 2F4E */ { "KANGXI RADICAL WEAPON","	* Kangxi Radical 79\n"
	"	# 6BB3"},
/* 2F4F */ { "KANGXI RADICAL DO NOT","	* Kangxi Radical 80\n"
	"	# 6BCB"},
/* 2F50 */ { "KANGXI RADICAL COMPARE","	* Kangxi Radical 81\n"
	"	# 6BD4"},
/* 2F51 */ { "KANGXI RADICAL FUR","	* Kangxi Radical 82\n"
	"	# 6BDB"},
/* 2F52 */ { "KANGXI RADICAL CLAN","	* Kangxi Radical 83\n"
	"	# 6C0F"},
/* 2F53 */ { "KANGXI RADICAL STEAM","	* Kangxi Radical 84\n"
	"	# 6C14"},
/* 2F54 */ { "KANGXI RADICAL WATER","	* Kangxi Radical 85\n"
	"	# 6C34"},
/* 2F55 */ { "KANGXI RADICAL FIRE","	* Kangxi Radical 86\n"
	"	# 706B"},
/* 2F56 */ { "KANGXI RADICAL CLAW","	* Kangxi Radical 87\n"
	"	# 722A"},
/* 2F57 */ { "KANGXI RADICAL FATHER","	* Kangxi Radical 88\n"
	"	# 7236"},
/* 2F58 */ { "KANGXI RADICAL DOUBLE X","	* Kangxi Radical 89\n"
	"	# 723B"},
/* 2F59 */ { "KANGXI RADICAL HALF TREE TRUNK","	* Kangxi Radical 90\n"
	"	# 723F"},
/* 2F5A */ { "KANGXI RADICAL SLICE","	* Kangxi Radical 91\n"
	"	# 7247"},
/* 2F5B */ { "KANGXI RADICAL FANG","	* Kangxi Radical 92\n"
	"	# 7259"},
/* 2F5C */ { "KANGXI RADICAL COW","	* Kangxi Radical 93\n"
	"	# 725B"},
/* 2F5D */ { "KANGXI RADICAL DOG","	* Kangxi Radical 94\n"
	"	# 72AC"},
/* 2F5E */ { "KANGXI RADICAL PROFOUND","	* Kangxi Radical 95\n"
	"	# 7384"},
/* 2F5F */ { "KANGXI RADICAL JADE","	* Kangxi Radical 96\n"
	"	# 7389"},
/* 2F60 */ { "KANGXI RADICAL MELON","	* Kangxi Radical 97\n"
	"	# 74DC"},
/* 2F61 */ { "KANGXI RADICAL TILE","	* Kangxi Radical 98\n"
	"	# 74E6"},
/* 2F62 */ { "KANGXI RADICAL SWEET","	* Kangxi Radical 99\n"
	"	# 7518"},
/* 2F63 */ { "KANGXI RADICAL LIFE","	* Kangxi Radical 100\n"
	"	# 751F"},
/* 2F64 */ { "KANGXI RADICAL USE","	* Kangxi Radical 101\n"
	"	# 7528"},
/* 2F65 */ { "KANGXI RADICAL FIELD","	* Kangxi Radical 102\n"
	"	# 7530"},
/* 2F66 */ { "KANGXI RADICAL BOLT OF CLOTH","	* Kangxi Radical 103\n"
	"	# 758B"},
/* 2F67 */ { "KANGXI RADICAL SICKNESS","	* Kangxi Radical 104\n"
	"	# 7592"},
/* 2F68 */ { "KANGXI RADICAL DOTTED TENT","	* Kangxi Radical 105\n"
	"	# 7676"},
/* 2F69 */ { "KANGXI RADICAL WHITE","	* Kangxi Radical 106\n"
	"	# 767D"},
/* 2F6A */ { "KANGXI RADICAL SKIN","	* Kangxi Radical 107\n"
	"	# 76AE"},
/* 2F6B */ { "KANGXI RADICAL DISH","	* Kangxi Radical 108\n"
	"	# 76BF"},
/* 2F6C */ { "KANGXI RADICAL EYE","	* Kangxi Radical 109\n"
	"	# 76EE"},
/* 2F6D */ { "KANGXI RADICAL SPEAR","	* Kangxi Radical 110\n"
	"	# 77DB"},
/* 2F6E */ { "KANGXI RADICAL ARROW","	* Kangxi Radical 111\n"
	"	# 77E2"},
/* 2F6F */ { "KANGXI RADICAL STONE","	* Kangxi Radical 112\n"
	"	# 77F3"},
/* 2F70 */ { "KANGXI RADICAL SPIRIT","	* Kangxi Radical 113\n"
	"	# 793A"},
/* 2F71 */ { "KANGXI RADICAL TRACK","	* Kangxi Radical 114\n"
	"	# 79B8"},
/* 2F72 */ { "KANGXI RADICAL GRAIN","	* Kangxi Radical 115\n"
	"	# 79BE"},
/* 2F73 */ { "KANGXI RADICAL CAVE","	* Kangxi Radical 116\n"
	"	# 7A74"},
/* 2F74 */ { "KANGXI RADICAL STAND","	* Kangxi Radical 117\n"
	"	# 7ACB"},
/* 2F75 */ { "KANGXI RADICAL BAMBOO","	* Kangxi Radical 118\n"
	"	# 7AF9"},
/* 2F76 */ { "KANGXI RADICAL RICE","	* Kangxi Radical 119\n"
	"	# 7C73"},
/* 2F77 */ { "KANGXI RADICAL SILK","	* Kangxi Radical 120\n"
	"	# 7CF8"},
/* 2F78 */ { "KANGXI RADICAL JAR","	* Kangxi Radical 121\n"
	"	# 7F36"},
/* 2F79 */ { "KANGXI RADICAL NET","	* Kangxi Radical 122\n"
	"	# 7F51"},
/* 2F7A */ { "KANGXI RADICAL SHEEP","	* Kangxi Radical 123\n"
	"	# 7F8A"},
/* 2F7B */ { "KANGXI RADICAL FEATHER","	* Kangxi Radical 124\n"
	"	# 7FBD"},
/* 2F7C */ { "KANGXI RADICAL OLD","	* Kangxi Radical 125\n"
	"	# 8001"},
/* 2F7D */ { "KANGXI RADICAL AND","	* Kangxi Radical 126\n"
	"	# 800C"},
/* 2F7E */ { "KANGXI RADICAL PLOW","	* Kangxi Radical 127\n"
	"	# 8012"},
/* 2F7F */ { "KANGXI RADICAL EAR","	* Kangxi Radical 128\n"
	"	# 8033"},
/* 2F80 */ { "KANGXI RADICAL BRUSH","	* Kangxi Radical 129\n"
	"	# 807F"},
/* 2F81 */ { "KANGXI RADICAL MEAT","	* Kangxi Radical 130\n"
	"	# 8089"},
/* 2F82 */ { "KANGXI RADICAL MINISTER","	* Kangxi Radical 131\n"
	"	# 81E3"},
/* 2F83 */ { "KANGXI RADICAL SELF","	* Kangxi Radical 132\n"
	"	# 81EA"},
/* 2F84 */ { "KANGXI RADICAL ARRIVE","	* Kangxi Radical 133\n"
	"	# 81F3"},
/* 2F85 */ { "KANGXI RADICAL MORTAR","	* Kangxi Radical 134\n"
	"	# 81FC"},
/* 2F86 */ { "KANGXI RADICAL TONGUE","	* Kangxi Radical 135\n"
	"	# 820C"},
/* 2F87 */ { "KANGXI RADICAL OPPOSE","	* Kangxi Radical 136\n"
	"	# 821B"},
/* 2F88 */ { "KANGXI RADICAL BOAT","	* Kangxi Radical 137\n"
	"	# 821F"},
/* 2F89 */ { "KANGXI RADICAL STOPPING","	* Kangxi Radical 138\n"
	"	# 826E"},
/* 2F8A */ { "KANGXI RADICAL COLOR","	* Kangxi Radical 139\n"
	"	# 8272"},
/* 2F8B */ { "KANGXI RADICAL GRASS","	* Kangxi Radical 140\n"
	"	# 8278"},
/* 2F8C */ { "KANGXI RADICAL TIGER","	* Kangxi Radical 141\n"
	"	# 864D"},
/* 2F8D */ { "KANGXI RADICAL INSECT","	* Kangxi Radical 142\n"
	"	# 866B"},
/* 2F8E */ { "KANGXI RADICAL BLOOD","	* Kangxi Radical 143\n"
	"	# 8840"},
/* 2F8F */ { "KANGXI RADICAL WALK ENCLOSURE","	* Kangxi Radical 144\n"
	"	# 884C"},
/* 2F90 */ { "KANGXI RADICAL CLOTHES","	* Kangxi Radical 145\n"
	"	# 8863"},
/* 2F91 */ { "KANGXI RADICAL WEST","	* Kangxi Radical 146\n"
	"	# 897E"},
/* 2F92 */ { "KANGXI RADICAL SEE","	* Kangxi Radical 147\n"
	"	# 898B"},
/* 2F93 */ { "KANGXI RADICAL HORN","	* Kangxi Radical 148\n"
	"	# 89D2"},
/* 2F94 */ { "KANGXI RADICAL SPEECH","	* Kangxi Radical 149\n"
	"	# 8A00"},
/* 2F95 */ { "KANGXI RADICAL VALLEY","	* Kangxi Radical 150\n"
	"	# 8C37"},
/* 2F96 */ { "KANGXI RADICAL BEAN","	* Kangxi Radical 151\n"
	"	# 8C46"},
/* 2F97 */ { "KANGXI RADICAL PIG","	* Kangxi Radical 152\n"
	"	# 8C55"},
/* 2F98 */ { "KANGXI RADICAL BADGER","	* Kangxi Radical 153\n"
	"	# 8C78"},
/* 2F99 */ { "KANGXI RADICAL SHELL","	* Kangxi Radical 154\n"
	"	# 8C9D"},
/* 2F9A */ { "KANGXI RADICAL RED","	* Kangxi Radical 155\n"
	"	# 8D64"},
/* 2F9B */ { "KANGXI RADICAL RUN","	* Kangxi Radical 156\n"
	"	# 8D70"},
/* 2F9C */ { "KANGXI RADICAL FOOT","	* Kangxi Radical 157\n"
	"	# 8DB3"},
/* 2F9D */ { "KANGXI RADICAL BODY","	* Kangxi Radical 158\n"
	"	# 8EAB"},
/* 2F9E */ { "KANGXI RADICAL CART","	* Kangxi Radical 159\n"
	"	# 8ECA"},
/* 2F9F */ { "KANGXI RADICAL BITTER","	* Kangxi Radical 160\n"
	"	# 8F9B"},
/* 2FA0 */ { "KANGXI RADICAL MORNING","	* Kangxi Radical 161\n"
	"	# 8FB0"},
/* 2FA1 */ { "KANGXI RADICAL WALK","	* Kangxi Radical 162\n"
	"	# 8FB5"},
/* 2FA2 */ { "KANGXI RADICAL CITY","	* Kangxi Radical 163\n"
	"	# 9091"},
/* 2FA3 */ { "KANGXI RADICAL WINE","	* Kangxi Radical 164\n"
	"	# 9149"},
/* 2FA4 */ { "KANGXI RADICAL DISTINGUISH","	* Kangxi Radical 165\n"
	"	# 91C6"},
/* 2FA5 */ { "KANGXI RADICAL VILLAGE","	* Kangxi Radical 166\n"
	"	# 91CC"},
/* 2FA6 */ { "KANGXI RADICAL GOLD","	* Kangxi Radical 167\n"
	"	# 91D1"},
/* 2FA7 */ { "KANGXI RADICAL LONG","	* Kangxi Radical 168\n"
	"	# 9577"},
/* 2FA8 */ { "KANGXI RADICAL GATE","	* Kangxi Radical 169\n"
	"	# 9580"},
/* 2FA9 */ { "KANGXI RADICAL MOUND","	* Kangxi Radical 170\n"
	"	# 961C"},
/* 2FAA */ { "KANGXI RADICAL SLAVE","	* Kangxi Radical 171\n"
	"	# 96B6"},
/* 2FAB */ { "KANGXI RADICAL SHORT TAILED BIRD","	* Kangxi Radical 172\n"
	"	# 96B9"},
/* 2FAC */ { "KANGXI RADICAL RAIN","	* Kangxi Radical 173\n"
	"	# 96E8"},
/* 2FAD */ { "KANGXI RADICAL BLUE","	* Kangxi Radical 174\n"
	"	# 9751"},
/* 2FAE */ { "KANGXI RADICAL WRONG","	* Kangxi Radical 175\n"
	"	# 975E"},
/* 2FAF */ { "KANGXI RADICAL FACE","	* Kangxi Radical 176\n"
	"	# 9762"},
/* 2FB0 */ { "KANGXI RADICAL LEATHER","	* Kangxi Radical 177\n"
	"	# 9769"},
/* 2FB1 */ { "KANGXI RADICAL TANNED LEATHER","	* Kangxi Radical 178\n"
	"	# 97CB"},
/* 2FB2 */ { "KANGXI RADICAL LEEK","	* Kangxi Radical 179\n"
	"	# 97ED"},
/* 2FB3 */ { "KANGXI RADICAL SOUND","	* Kangxi Radical 180\n"
	"	# 97F3"},
/* 2FB4 */ { "KANGXI RADICAL LEAF","	* Kangxi Radical 181\n"
	"	# 9801"},
/* 2FB5 */ { "KANGXI RADICAL WIND","	* Kangxi Radical 182\n"
	"	# 98A8"},
/* 2FB6 */ { "KANGXI RADICAL FLY","	* Kangxi Radical 183\n"
	"	# 98DB"},
/* 2FB7 */ { "KANGXI RADICAL EAT","	* Kangxi Radical 184\n"
	"	# 98DF"},
/* 2FB8 */ { "KANGXI RADICAL HEAD","	* Kangxi Radical 185\n"
	"	# 9996"},
/* 2FB9 */ { "KANGXI RADICAL FRAGRANT","	* Kangxi Radical 186\n"
	"	# 9999"},
/* 2FBA */ { "KANGXI RADICAL HORSE","	* Kangxi Radical 187\n"
	"	# 99AC"},
/* 2FBB */ { "KANGXI RADICAL BONE","	* Kangxi Radical 188\n"
	"	# 9AA8"},
/* 2FBC */ { "KANGXI RADICAL TALL","	* Kangxi Radical 189\n"
	"	# 9AD8"},
/* 2FBD */ { "KANGXI RADICAL HAIR","	* Kangxi Radical 190\n"
	"	# 9ADF"},
/* 2FBE */ { "KANGXI RADICAL FIGHT","	* Kangxi Radical 191\n"
	"	# 9B25"},
/* 2FBF */ { "KANGXI RADICAL SACRIFICIAL WINE","	* Kangxi Radical 192\n"
	"	# 9B2F"},
/* 2FC0 */ { "KANGXI RADICAL CAULDRON","	* Kangxi Radical 193\n"
	"	# 9B32"},
/* 2FC1 */ { "KANGXI RADICAL GHOST","	* Kangxi Radical 194\n"
	"	# 9B3C"},
/* 2FC2 */ { "KANGXI RADICAL FISH","	* Kangxi Radical 195\n"
	"	# 9B5A"},
/* 2FC3 */ { "KANGXI RADICAL BIRD","	* Kangxi Radical 196\n"
	"	# 9CE5"},
/* 2FC4 */ { "KANGXI RADICAL SALT","	* Kangxi Radical 197\n"
	"	# 9E75"},
/* 2FC5 */ { "KANGXI RADICAL DEER","	* Kangxi Radical 198\n"
	"	# 9E7F"},
/* 2FC6 */ { "KANGXI RADICAL WHEAT","	* Kangxi Radical 199\n"
	"	# 9EA5"},
/* 2FC7 */ { "KANGXI RADICAL HEMP","	* Kangxi Radical 200\n"
	"	# 9EBB"},
/* 2FC8 */ { "KANGXI RADICAL YELLOW","	* Kangxi Radical 201\n"
	"	# 9EC3"},
/* 2FC9 */ { "KANGXI RADICAL MILLET","	* Kangxi Radical 202\n"
	"	# 9ECD"},
/* 2FCA */ { "KANGXI RADICAL BLACK","	* Kangxi Radical 203\n"
	"	# 9ED1"},
/* 2FCB */ { "KANGXI RADICAL EMBROIDERY","	* Kangxi Radical 204\n"
	"	# 9EF9"},
/* 2FCC */ { "KANGXI RADICAL FROG","	* Kangxi Radical 205\n"
	"	# 9EFD"},
/* 2FCD */ { "KANGXI RADICAL TRIPOD","	* Kangxi Radical 206\n"
	"	# 9F0E"},
/* 2FCE */ { "KANGXI RADICAL DRUM","	* Kangxi Radical 207\n"
	"	# 9F13"},
/* 2FCF */ { "KANGXI RADICAL RAT","	* Kangxi Radical 208\n"
	"	# 9F20"},
/* 2FD0 */ { "KANGXI RADICAL NOSE","	* Kangxi Radical 209\n"
	"	# 9F3B"},
/* 2FD1 */ { "KANGXI RADICAL EVEN","	* Kangxi Radical 210\n"
	"	# 9F4A"},
/* 2FD2 */ { "KANGXI RADICAL TOOTH","	* Kangxi Radical 211\n"
	"	# 9F52"},
/* 2FD3 */ { "KANGXI RADICAL DRAGON","	* Kangxi Radical 212\n"
	"	# 9F8D"},
/* 2FD4 */ { "KANGXI RADICAL TURTLE","	* Kangxi Radical 213\n"
	"	# 9F9C"},
/* 2FD5 */ { "KANGXI RADICAL FLUTE","	* Kangxi Radical 214\n"
	"	# 9FA0"},
/* 2FD6 */ { NULL,NULL},
/* 2FD7 */ { NULL,NULL},
/* 2FD8 */ { NULL,NULL},
/* 2FD9 */ { NULL,NULL},
/* 2FDA */ { NULL,NULL},
/* 2FDB */ { NULL,NULL},
/* 2FDC */ { NULL,NULL},
/* 2FDD */ { NULL,NULL},
/* 2FDE */ { NULL,NULL},
/* 2FDF */ { NULL,NULL},
/* 2FE0 */ { NULL,NULL},
/* 2FE1 */ { NULL,NULL},
/* 2FE2 */ { NULL,NULL},
/* 2FE3 */ { NULL,NULL},
/* 2FE4 */ { NULL,NULL},
/* 2FE5 */ { NULL,NULL},
/* 2FE6 */ { NULL,NULL},
/* 2FE7 */ { NULL,NULL},
/* 2FE8 */ { NULL,NULL},
/* 2FE9 */ { NULL,NULL},
/* 2FEA */ { NULL,NULL},
/* 2FEB */ { NULL,NULL},
/* 2FEC */ { NULL,NULL},
/* 2FED */ { NULL,NULL},
/* 2FEE */ { NULL,NULL},
/* 2FEF */ { NULL,NULL},
/* 2FF0 */ { "IDEOGRAPHIC DESCRIPTION CHARACTER LEFT TO RIGHT",NULL},
/* 2FF1 */ { "IDEOGRAPHIC DESCRIPTION CHARACTER ABOVE TO BELOW",NULL},
/* 2FF2 */ { "IDEOGRAPHIC DESCRIPTION CHARACTER LEFT TO MIDDLE AND RIGHT",NULL},
/* 2FF3 */ { "IDEOGRAPHIC DESCRIPTION CHARACTER ABOVE TO MIDDLE AND BELOW",NULL},
/* 2FF4 */ { "IDEOGRAPHIC DESCRIPTION CHARACTER FULL SURROUND",NULL},
/* 2FF5 */ { "IDEOGRAPHIC DESCRIPTION CHARACTER SURROUND FROM ABOVE",NULL},
/* 2FF6 */ { "IDEOGRAPHIC DESCRIPTION CHARACTER SURROUND FROM BELOW",NULL},
/* 2FF7 */ { "IDEOGRAPHIC DESCRIPTION CHARACTER SURROUND FROM LEFT",NULL},
/* 2FF8 */ { "IDEOGRAPHIC DESCRIPTION CHARACTER SURROUND FROM UPPER LEFT",NULL},
/* 2FF9 */ { "IDEOGRAPHIC DESCRIPTION CHARACTER SURROUND FROM UPPER RIGHT",NULL},
/* 2FFA */ { "IDEOGRAPHIC DESCRIPTION CHARACTER SURROUND FROM LOWER LEFT",NULL},
/* 2FFB */ { "IDEOGRAPHIC DESCRIPTION CHARACTER OVERLAID",NULL},
/* 2FFC */ { "IDEOGRAPHIC DESCRIPTION CHARACTER SURROUND FROM RIGHT",NULL},
/* 2FFD */ { "IDEOGRAPHIC DESCRIPTION CHARACTER SURROUND FROM LOWER RIGHT",NULL},
/* 2FFE */ { "IDEOGRAPHIC DESCRIPTION CHARACTER HORIZONTAL REFLECTION",NULL},
/* 2FFF */ { "IDEOGRAPHIC DESCRIPTION CHARACTER ROTATION",NULL}
};

UN_DLL_LOCAL
static const struct unicode_nameannot una_00_30[] = {
/* 3000 */ { "IDEOGRAPHIC SPACE","	x (space - 0020)\n"
	"	# <wide> 0020"},
/* 3001 */ { "IDEOGRAPHIC COMMA","	* in Chinese, delimits items in a list or series\n"
	"	x (comma - 002C)\n"
	"	x (fullwidth comma - FF0C)\n"
	"	x (halfwidth ideographic comma - FF64)\n"
	"	~ 3001 FE00 corner-justified form\n"
	"	~ 3001 FE01 centered form"},
/* 3002 */ { "IDEOGRAPHIC FULL STOP","	x (full stop - 002E)\n"
	"	x (fullwidth full stop - FF0E)\n"
	"	x (halfwidth ideographic full stop - FF61)\n"
	"	~ 3002 FE00 corner-justified form\n"
	"	~ 3002 FE01 centered form"},
/* 3003 */ { "DITTO MARK","	x (double prime - 2033)"},
/* 3004 */ { "JAPANESE INDUSTRIAL STANDARD SYMBOL",NULL},
/* 3005 */ { "IDEOGRAPHIC ITERATION MARK","	x 206A4\n"
	"	x (tangut iteration mark - 16FE0)"},
/* 3006 */ { "IDEOGRAPHIC CLOSING MARK","	x 4E44"},
/* 3007 */ { "IDEOGRAPHIC NUMBER ZERO","	x (combining enclosing circle - 20DD)\n"
	"	x (white circle - 25CB)\n"
	"	x (large circle - 25EF)"},
/* 3008 */ { "LEFT ANGLE BRACKET","	x (less-than sign - 003C)\n"
	"	x (single left-pointing angle quotation mark - 2039)\n"
	"	x (left-pointing angle bracket - 2329)\n"
	"	x (mathematical left angle bracket - 27E8)"},
/* 3009 */ { "RIGHT ANGLE BRACKET","	x (greater-than sign - 003E)\n"
	"	x (single right-pointing angle quotation mark - 203A)\n"
	"	x (right-pointing angle bracket - 232A)\n"
	"	x (mathematical right angle bracket - 27E9)"},
/* 300A */ { "LEFT DOUBLE ANGLE BRACKET","	x (left-pointing double angle quotation mark - 00AB)\n"
	"	x (mathematical left double angle bracket - 27EA)"},
/* 300B */ { "RIGHT DOUBLE ANGLE BRACKET","	x (right-pointing double angle quotation mark - 00BB)\n"
	"	x (mathematical right double angle bracket - 27EB)"},
/* 300C */ { "LEFT CORNER BRACKET","	x (left ceiling - 2308)"},
/* 300D */ { "RIGHT CORNER BRACKET","	x (right floor - 230B)"},
/* 300E */ { "LEFT WHITE CORNER BRACKET",NULL},
/* 300F */ { "RIGHT WHITE CORNER BRACKET",NULL},
/* 3010 */ { "LEFT BLACK LENTICULAR BRACKET",NULL},
/* 3011 */ { "RIGHT BLACK LENTICULAR BRACKET",NULL},
/* 3012 */ { "POSTAL MARK","	x (tenge sign - 20B8)\n"
	"	x (short down tack with overbar - 2AE7)\n"
	"	x (japanese post office - 1F3E3)"},
/* 3013 */ { "GETA MARK","	* substitute for ideograph not in font\n"
	"	* editorial convention to represent ideographic lacuna\n"
	"	x (white square - 25A1)\n"
	"	x (heavy equals sign - 1F7F0)"},
/* 3014 */ { "LEFT TORTOISE SHELL BRACKET",NULL},
/* 3015 */ { "RIGHT TORTOISE SHELL BRACKET",NULL},
/* 3016 */ { "LEFT WHITE LENTICULAR BRACKET",NULL},
/* 3017 */ { "RIGHT WHITE LENTICULAR BRACKET",NULL},
/* 3018 */ { "LEFT WHITE TORTOISE SHELL BRACKET","	x (mathematical left white tortoise shell bracket - 27EC)"},
/* 3019 */ { "RIGHT WHITE TORTOISE SHELL BRACKET","	x (mathematical right white tortoise shell bracket - 27ED)"},
/* 301A */ { "LEFT WHITE SQUARE BRACKET","	= left abstract syntax bracket\n"
	"	x (mathematical left white square bracket - 27E6)"},
/* 301B */ { "RIGHT WHITE SQUARE BRACKET","	= right abstract syntax bracket\n"
	"	x (mathematical right white square bracket - 27E7)"},
/* 301C */ { "WAVE DASH","		This character was encoded to match JIS C 6226-1978 1-33 \"wave dash\".\n"
	"	x (swung dash - 2053)\n"
	"	x (wavy dash - 3030)\n"
	"	x (fullwidth tilde - FF5E)"},
/* 301D */ { "REVERSED DOUBLE PRIME QUOTATION MARK","	* sometimes depicted as double prime quotation mark\n"
	"	x (left double quotation mark - 201C)\n"
	"	x (reversed double prime - 2036)"},
/* 301E */ { "DOUBLE PRIME QUOTATION MARK","	* this is a mistaken analogue to 201D; 301F is preferred\n"
	"	x (right double quotation mark - 201D)\n"
	"	x (double prime - 2033)"},
/* 301F */ { "LOW DOUBLE PRIME QUOTATION MARK","	* may be depicted as low inverse double prime quotation mark"},
/* 3020 */ { "POSTAL MARK FACE",NULL},
/* 3021 */ { "HANGZHOU NUMERAL ONE",NULL},
/* 3022 */ { "HANGZHOU NUMERAL TWO",NULL},
/* 3023 */ { "HANGZHOU NUMERAL THREE",NULL},
/* 3024 */ { "HANGZHOU NUMERAL FOUR",NULL},
/* 3025 */ { "HANGZHOU NUMERAL FIVE",NULL},
/* 3026 */ { "HANGZHOU NUMERAL SIX",NULL},
/* 3027 */ { "HANGZHOU NUMERAL SEVEN",NULL},
/* 3028 */ { "HANGZHOU NUMERAL EIGHT",NULL},
/* 3029 */ { "HANGZHOU NUMERAL NINE",NULL},
/* 302A */ { "IDEOGRAPHIC LEVEL TONE MARK",NULL},
/* 302B */ { "IDEOGRAPHIC RISING TONE MARK",NULL},
/* 302C */ { "IDEOGRAPHIC DEPARTING TONE MARK",NULL},
/* 302D */ { "IDEOGRAPHIC ENTERING TONE MARK",NULL},
/* 302E */ { "HANGUL SINGLE DOT TONE MARK","	= single dot Bangjeom"},
/* 302F */ { "HANGUL DOUBLE DOT TONE MARK","	= double dot Bangjeom"},
/* 3030 */ { "WAVY DASH","	x (wavy line - 2307)\n"
	"	x (wave dash - 301C)"},
/* 3031 */ { "VERTICAL KANA REPEAT MARK",NULL},
/* 3032 */ { "VERTICAL KANA REPEAT WITH VOICED SOUND MARK","	* implemented as glyphs that are two-em tall"},
/* 3033 */ { "VERTICAL KANA REPEAT MARK UPPER HALF",NULL},
/* 3034 */ { "VERTICAL KANA REPEAT WITH VOICED SOUND MARK UPPER HALF","	* implemented as glyphs that are one-em tall and that combine with the following character to form ligated two-em glyphs for the complete repeat marks"},
/* 3035 */ { "VERTICAL KANA REPEAT MARK LOWER HALF",NULL},
/* 3036 */ { "CIRCLED POSTAL MARK","	= symbol for type B electronics\n"
	"	* not used as a postal mark\n"
	"	x (symbol for type a electronics - 2B97)\n"
	"	# 3012 postal mark"},
/* 3037 */ { "IDEOGRAPHIC TELEGRAPH LINE FEED SEPARATOR SYMBOL",NULL},
/* 3038 */ { "HANGZHOU NUMERAL TEN","	# 5341"},
/* 3039 */ { "HANGZHOU NUMERAL TWENTY","	# 5344"},
/* 303A */ { "HANGZHOU NUMERAL THIRTY","	# 5345"},
/* 303B */ { "VERTICAL IDEOGRAPHIC ITERATION MARK",NULL},
/* 303C */ { "MASU MARK","	* informal abbreviation for Japanese -masu ending\n"
	"	x (squared rising diagonal slash - 29C4)"},
/* 303D */ { "PART ALTERNATION MARK","	* marks the start of a song part in Japanese"},
/* 303E */ { "IDEOGRAPHIC VARIATION INDICATOR","	* visual indicator that the following ideograph is to be taken as a variant of the intended character"},
/* 303F */ { "IDEOGRAPHIC HALF FILL SPACE","	* visual indicator of a screen space for half of an ideograph"},
/* 3040 */ { NULL,NULL},
/* 3041 */ { "HIRAGANA LETTER SMALL A",NULL},
/* 3042 */ { "HIRAGANA LETTER A",NULL},
/* 3043 */ { "HIRAGANA LETTER SMALL I",NULL},
/* 3044 */ { "HIRAGANA LETTER I",NULL},
/* 3045 */ { "HIRAGANA LETTER SMALL U",NULL},
/* 3046 */ { "HIRAGANA LETTER U",NULL},
/* 3047 */ { "HIRAGANA LETTER SMALL E",NULL},
/* 3048 */ { "HIRAGANA LETTER E",NULL},
/* 3049 */ { "HIRAGANA LETTER SMALL O",NULL},
/* 304A */ { "HIRAGANA LETTER O",NULL},
/* 304B */ { "HIRAGANA LETTER KA",NULL},
/* 304C */ { "HIRAGANA LETTER GA","	: 304B 3099"},
/* 304D */ { "HIRAGANA LETTER KI",NULL},
/* 304E */ { "HIRAGANA LETTER GI","	: 304D 3099"},
/* 304F */ { "HIRAGANA LETTER KU",NULL},
/* 3050 */ { "HIRAGANA LETTER GU","	: 304F 3099"},
/* 3051 */ { "HIRAGANA LETTER KE",NULL},
/* 3052 */ { "HIRAGANA LETTER GE","	: 3051 3099"},
/* 3053 */ { "HIRAGANA LETTER KO",NULL},
/* 3054 */ { "HIRAGANA LETTER GO","	: 3053 3099"},
/* 3055 */ { "HIRAGANA LETTER SA",NULL},
/* 3056 */ { "HIRAGANA LETTER ZA","	: 3055 3099"},
/* 3057 */ { "HIRAGANA LETTER SI","	= SHI"},
/* 3058 */ { "HIRAGANA LETTER ZI","	= JI (not unique)\n"
	"	: 3057 3099"},
/* 3059 */ { "HIRAGANA LETTER SU",NULL},
/* 305A */ { "HIRAGANA LETTER ZU","	: 3059 3099"},
/* 305B */ { "HIRAGANA LETTER SE",NULL},
/* 305C */ { "HIRAGANA LETTER ZE","	: 305B 3099"},
/* 305D */ { "HIRAGANA LETTER SO",NULL},
/* 305E */ { "HIRAGANA LETTER ZO","	: 305D 3099"},
/* 305F */ { "HIRAGANA LETTER TA",NULL},
/* 3060 */ { "HIRAGANA LETTER DA","	: 305F 3099"},
/* 3061 */ { "HIRAGANA LETTER TI","	= CHI"},
/* 3062 */ { "HIRAGANA LETTER DI","	= JI (not unique)\n"
	"	: 3061 3099"},
/* 3063 */ { "HIRAGANA LETTER SMALL TU","	= SMALL TSU"},
/* 3064 */ { "HIRAGANA LETTER TU","	= TSU"},
/* 3065 */ { "HIRAGANA LETTER DU","	= ZU (not unique)\n"
	"	: 3064 3099"},
/* 3066 */ { "HIRAGANA LETTER TE",NULL},
/* 3067 */ { "HIRAGANA LETTER DE","	: 3066 3099"},
/* 3068 */ { "HIRAGANA LETTER TO",NULL},
/* 3069 */ { "HIRAGANA LETTER DO","	: 3068 3099"},
/* 306A */ { "HIRAGANA LETTER NA",NULL},
/* 306B */ { "HIRAGANA LETTER NI",NULL},
/* 306C */ { "HIRAGANA LETTER NU",NULL},
/* 306D */ { "HIRAGANA LETTER NE",NULL},
/* 306E */ { "HIRAGANA LETTER NO",NULL},
/* 306F */ { "HIRAGANA LETTER HA",NULL},
/* 3070 */ { "HIRAGANA LETTER BA","	: 306F 3099"},
/* 3071 */ { "HIRAGANA LETTER PA","	: 306F 309A"},
/* 3072 */ { "HIRAGANA LETTER HI",NULL},
/* 3073 */ { "HIRAGANA LETTER BI","	: 3072 3099"},
/* 3074 */ { "HIRAGANA LETTER PI","	: 3072 309A"},
/* 3075 */ { "HIRAGANA LETTER HU","	= FU"},
/* 3076 */ { "HIRAGANA LETTER BU","	: 3075 3099"},
/* 3077 */ { "HIRAGANA LETTER PU","	: 3075 309A"},
/* 3078 */ { "HIRAGANA LETTER HE",NULL},
/* 3079 */ { "HIRAGANA LETTER BE","	: 3078 3099"},
/* 307A */ { "HIRAGANA LETTER PE","	: 3078 309A"},
/* 307B */ { "HIRAGANA LETTER HO",NULL},
/* 307C */ { "HIRAGANA LETTER BO","	: 307B 3099"},
/* 307D */ { "HIRAGANA LETTER PO","	: 307B 309A"},
/* 307E */ { "HIRAGANA LETTER MA",NULL},
/* 307F */ { "HIRAGANA LETTER MI",NULL},
/* 3080 */ { "HIRAGANA LETTER MU",NULL},
/* 3081 */ { "HIRAGANA LETTER ME",NULL},
/* 3082 */ { "HIRAGANA LETTER MO",NULL},
/* 3083 */ { "HIRAGANA LETTER SMALL YA",NULL},
/* 3084 */ { "HIRAGANA LETTER YA",NULL},
/* 3085 */ { "HIRAGANA LETTER SMALL YU",NULL},
/* 3086 */ { "HIRAGANA LETTER YU",NULL},
/* 3087 */ { "HIRAGANA LETTER SMALL YO",NULL},
/* 3088 */ { "HIRAGANA LETTER YO",NULL},
/* 3089 */ { "HIRAGANA LETTER RA",NULL},
/* 308A */ { "HIRAGANA LETTER RI",NULL},
/* 308B */ { "HIRAGANA LETTER RU",NULL},
/* 308C */ { "HIRAGANA LETTER RE",NULL},
/* 308D */ { "HIRAGANA LETTER RO",NULL},
/* 308E */ { "HIRAGANA LETTER SMALL WA",NULL},
/* 308F */ { "HIRAGANA LETTER WA",NULL},
/* 3090 */ { "HIRAGANA LETTER WI",NULL},
/* 3091 */ { "HIRAGANA LETTER WE",NULL},
/* 3092 */ { "HIRAGANA LETTER WO",NULL},
/* 3093 */ { "HIRAGANA LETTER N",NULL},
/* 3094 */ { "HIRAGANA LETTER VU","	: 3046 3099"},
/* 3095 */ { "HIRAGANA LETTER SMALL KA",NULL},
/* 3096 */ { "HIRAGANA LETTER SMALL KE",NULL},
/* 3097 */ { NULL,NULL},
/* 3098 */ { NULL,NULL},
/* 3099 */ { "COMBINING KATAKANA-HIRAGANA VOICED SOUND MARK",NULL},
/* 309A */ { "COMBINING KATAKANA-HIRAGANA SEMI-VOICED SOUND MARK",NULL},
/* 309B */ { "KATAKANA-HIRAGANA VOICED SOUND MARK","	# 0020 3099"},
/* 309C */ { "KATAKANA-HIRAGANA SEMI-VOICED SOUND MARK","	# 0020 309A"},
/* 309D */ { "HIRAGANA ITERATION MARK",NULL},
/* 309E */ { "HIRAGANA VOICED ITERATION MARK","	: 309D 3099"},
/* 309F */ { "HIRAGANA DIGRAPH YORI","	* historically used in vertical contexts, but now found also in horizontal layout\n"
	"	# <vertical> 3088 308A"},
/* 30A0 */ { "KATAKANA-HIRAGANA DOUBLE HYPHEN","	x (equals sign - 003D)\n"
	"	x (double hyphen - 2E40)"},
/* 30A1 */ { "KATAKANA LETTER SMALL A",NULL},
/* 30A2 */ { "KATAKANA LETTER A",NULL},
/* 30A3 */ { "KATAKANA LETTER SMALL I",NULL},
/* 30A4 */ { "KATAKANA LETTER I",NULL},
/* 30A5 */ { "KATAKANA LETTER SMALL U",NULL},
/* 30A6 */ { "KATAKANA LETTER U",NULL},
/* 30A7 */ { "KATAKANA LETTER SMALL E",NULL},
/* 30A8 */ { "KATAKANA LETTER E",NULL},
/* 30A9 */ { "KATAKANA LETTER SMALL O",NULL},
/* 30AA */ { "KATAKANA LETTER O",NULL},
/* 30AB */ { "KATAKANA LETTER KA",NULL},
/* 30AC */ { "KATAKANA LETTER GA","	: 30AB 3099"},
/* 30AD */ { "KATAKANA LETTER KI",NULL},
/* 30AE */ { "KATAKANA LETTER GI","	: 30AD 3099"},
/* 30AF */ { "KATAKANA LETTER KU",NULL},
/* 30B0 */ { "KATAKANA LETTER GU","	: 30AF 3099"},
/* 30B1 */ { "KATAKANA LETTER KE",NULL},
/* 30B2 */ { "KATAKANA LETTER GE","	: 30B1 3099"},
/* 30B3 */ { "KATAKANA LETTER KO",NULL},
/* 30B4 */ { "KATAKANA LETTER GO","	: 30B3 3099"},
/* 30B5 */ { "KATAKANA LETTER SA",NULL},
/* 30B6 */ { "KATAKANA LETTER ZA","	: 30B5 3099"},
/* 30B7 */ { "KATAKANA LETTER SI","	= SHI"},
/* 30B8 */ { "KATAKANA LETTER ZI","	= JI (not unique)\n"
	"	: 30B7 3099"},
/* 30B9 */ { "KATAKANA LETTER SU",NULL},
/* 30BA */ { "KATAKANA LETTER ZU","	: 30B9 3099"},
/* 30BB */ { "KATAKANA LETTER SE",NULL},
/* 30BC */ { "KATAKANA LETTER ZE","	: 30BB 3099"},
/* 30BD */ { "KATAKANA LETTER SO",NULL},
/* 30BE */ { "KATAKANA LETTER ZO","	: 30BD 3099"},
/* 30BF */ { "KATAKANA LETTER TA",NULL},
/* 30C0 */ { "KATAKANA LETTER DA","	: 30BF 3099"},
/* 30C1 */ { "KATAKANA LETTER TI","	= CHI"},
/* 30C2 */ { "KATAKANA LETTER DI","	= JI (not unique)\n"
	"	: 30C1 3099"},
/* 30C3 */ { "KATAKANA LETTER SMALL TU","	= SMALL TSU"},
/* 30C4 */ { "KATAKANA LETTER TU","	= TSU"},
/* 30C5 */ { "KATAKANA LETTER DU","	= ZU (not unique)\n"
	"	: 30C4 3099"},
/* 30C6 */ { "KATAKANA LETTER TE",NULL},
/* 30C7 */ { "KATAKANA LETTER DE","	: 30C6 3099"},
/* 30C8 */ { "KATAKANA LETTER TO",NULL},
/* 30C9 */ { "KATAKANA LETTER DO","	: 30C8 3099"},
/* 30CA */ { "KATAKANA LETTER NA",NULL},
/* 30CB */ { "KATAKANA LETTER NI",NULL},
/* 30CC */ { "KATAKANA LETTER NU",NULL},
/* 30CD */ { "KATAKANA LETTER NE",NULL},
/* 30CE */ { "KATAKANA LETTER NO",NULL},
/* 30CF */ { "KATAKANA LETTER HA",NULL},
/* 30D0 */ { "KATAKANA LETTER BA","	: 30CF 3099"},
/* 30D1 */ { "KATAKANA LETTER PA","	: 30CF 309A"},
/* 30D2 */ { "KATAKANA LETTER HI",NULL},
/* 30D3 */ { "KATAKANA LETTER BI","	: 30D2 3099"},
/* 30D4 */ { "KATAKANA LETTER PI","	: 30D2 309A"},
/* 30D5 */ { "KATAKANA LETTER HU","	= FU"},
/* 30D6 */ { "KATAKANA LETTER BU","	: 30D5 3099"},
/* 30D7 */ { "KATAKANA LETTER PU","	: 30D5 309A"},
/* 30D8 */ { "KATAKANA LETTER HE",NULL},
/* 30D9 */ { "KATAKANA LETTER BE","	: 30D8 3099"},
/* 30DA */ { "KATAKANA LETTER PE","	: 30D8 309A"},
/* 30DB */ { "KATAKANA LETTER HO",NULL},
/* 30DC */ { "KATAKANA LETTER BO","	: 30DB 3099"},
/* 30DD */ { "KATAKANA LETTER PO","	: 30DB 309A"},
/* 30DE */ { "KATAKANA LETTER MA",NULL},
/* 30DF */ { "KATAKANA LETTER MI",NULL},
/* 30E0 */ { "KATAKANA LETTER MU",NULL},
/* 30E1 */ { "KATAKANA LETTER ME",NULL},
/* 30E2 */ { "KATAKANA LETTER MO",NULL},
/* 30E3 */ { "KATAKANA LETTER SMALL YA",NULL},
/* 30E4 */ { "KATAKANA LETTER YA",NULL},
/* 30E5 */ { "KATAKANA LETTER SMALL YU",NULL},
/* 30E6 */ { "KATAKANA LETTER YU",NULL},
/* 30E7 */ { "KATAKANA LETTER SMALL YO",NULL},
/* 30E8 */ { "KATAKANA LETTER YO",NULL},
/* 30E9 */ { "KATAKANA LETTER RA",NULL},
/* 30EA */ { "KATAKANA LETTER RI",NULL},
/* 30EB */ { "KATAKANA LETTER RU",NULL},
/* 30EC */ { "KATAKANA LETTER RE",NULL},
/* 30ED */ { "KATAKANA LETTER RO",NULL},
/* 30EE */ { "KATAKANA LETTER SMALL WA",NULL},
/* 30EF */ { "KATAKANA LETTER WA",NULL},
/* 30F0 */ { "KATAKANA LETTER WI",NULL},
/* 30F1 */ { "KATAKANA LETTER WE",NULL},
/* 30F2 */ { "KATAKANA LETTER WO",NULL},
/* 30F3 */ { "KATAKANA LETTER N",NULL},
/* 30F4 */ { "KATAKANA LETTER VU","	: 30A6 3099"},
/* 30F5 */ { "KATAKANA LETTER SMALL KA",NULL},
/* 30F6 */ { "KATAKANA LETTER SMALL KE",NULL},
/* 30F7 */ { "KATAKANA LETTER VA","	: 30EF 3099"},
/* 30F8 */ { "KATAKANA LETTER VI","	: 30F0 3099"},
/* 30F9 */ { "KATAKANA LETTER VE","	: 30F1 3099"},
/* 30FA */ { "KATAKANA LETTER VO","	: 30F2 3099"},
/* 30FB */ { "KATAKANA MIDDLE DOT","	x (middle dot - 00B7)"},
/* 30FC */ { "KATAKANA-HIRAGANA PROLONGED SOUND MARK","	x (em dash - 2014)"},
/* 30FD */ { "KATAKANA ITERATION MARK",NULL},
/* 30FE */ { "KATAKANA VOICED ITERATION MARK","	: 30FD 3099"},
/* 30FF */ { "KATAKANA DIGRAPH KOTO","	* historically used in vertical contexts, but now found also in horizontal layout\n"
	"	# <vertical> 30B3 30C8"}
};

UN_DLL_LOCAL
static const struct unicode_nameannot una_00_31[] = {
/* 3100 */ { NULL,NULL},
/* 3101 */ { NULL,NULL},
/* 3102 */ { NULL,NULL},
/* 3103 */ { NULL,NULL},
/* 3104 */ { NULL,NULL},
/* 3105 */ { "BOPOMOFO LETTER B",NULL},
/* 3106 */ { "BOPOMOFO LETTER P",NULL},
/* 3107 */ { "BOPOMOFO LETTER M",NULL},
/* 3108 */ { "BOPOMOFO LETTER F",NULL},
/* 3109 */ { "BOPOMOFO LETTER D",NULL},
/* 310A */ { "BOPOMOFO LETTER T",NULL},
/* 310B */ { "BOPOMOFO LETTER N",NULL},
/* 310C */ { "BOPOMOFO LETTER L",NULL},
/* 310D */ { "BOPOMOFO LETTER G",NULL},
/* 310E */ { "BOPOMOFO LETTER K",NULL},
/* 310F */ { "BOPOMOFO LETTER H",NULL},
/* 3110 */ { "BOPOMOFO LETTER J",NULL},
/* 3111 */ { "BOPOMOFO LETTER Q",NULL},
/* 3112 */ { "BOPOMOFO LETTER X",NULL},
/* 3113 */ { "BOPOMOFO LETTER ZH",NULL},
/* 3114 */ { "BOPOMOFO LETTER CH",NULL},
/* 3115 */ { "BOPOMOFO LETTER SH",NULL},
/* 3116 */ { "BOPOMOFO LETTER R",NULL},
/* 3117 */ { "BOPOMOFO LETTER Z",NULL},
/* 3118 */ { "BOPOMOFO LETTER C",NULL},
/* 3119 */ { "BOPOMOFO LETTER S",NULL},
/* 311A */ { "BOPOMOFO LETTER A",NULL},
/* 311B */ { "BOPOMOFO LETTER O",NULL},
/* 311C */ { "BOPOMOFO LETTER E",NULL},
/* 311D */ { "BOPOMOFO LETTER EH",NULL},
/* 311E */ { "BOPOMOFO LETTER AI",NULL},
/* 311F */ { "BOPOMOFO LETTER EI",NULL},
/* 3120 */ { "BOPOMOFO LETTER AU",NULL},
/* 3121 */ { "BOPOMOFO LETTER OU",NULL},
/* 3122 */ { "BOPOMOFO LETTER AN",NULL},
/* 3123 */ { "BOPOMOFO LETTER EN",NULL},
/* 3124 */ { "BOPOMOFO LETTER ANG",NULL},
/* 3125 */ { "BOPOMOFO LETTER ENG",NULL},
/* 3126 */ { "BOPOMOFO LETTER ER",NULL},
/* 3127 */ { "BOPOMOFO LETTER I","	* the vertical stroke form is considered a rendering variant"},
/* 3128 */ { "BOPOMOFO LETTER U",NULL},
/* 3129 */ { "BOPOMOFO LETTER IU",NULL},
/* 312A */ { "BOPOMOFO LETTER V",NULL},
/* 312B */ { "BOPOMOFO LETTER NG",NULL},
/* 312C */ { "BOPOMOFO LETTER GN",NULL},
/* 312D */ { "BOPOMOFO LETTER IH","	* for analytic representation of apical vowel"},
/* 312E */ { "BOPOMOFO LETTER O WITH DOT ABOVE","	* early form of letter e"},
/* 312F */ { "BOPOMOFO LETTER NN","	* syllabic nasal"},
/* 3130 */ { NULL,NULL},
/* 3131 */ { "HANGUL LETTER KIYEOK","	# 1100 hangul choseong kiyeok"},
/* 3132 */ { "HANGUL LETTER SSANGKIYEOK","	# 1101 hangul choseong ssangkiyeok"},
/* 3133 */ { "HANGUL LETTER KIYEOK-SIOS","	# 11AA hangul jongseong kiyeok-sios"},
/* 3134 */ { "HANGUL LETTER NIEUN","	# 1102 hangul choseong nieun"},
/* 3135 */ { "HANGUL LETTER NIEUN-CIEUC","	# 11AC hangul jongseong nieun-cieuc"},
/* 3136 */ { "HANGUL LETTER NIEUN-HIEUH","	# 11AD hangul jongseong nieun-hieuh"},
/* 3137 */ { "HANGUL LETTER TIKEUT","	# 1103 hangul choseong tikeut"},
/* 3138 */ { "HANGUL LETTER SSANGTIKEUT","	# 1104 hangul choseong ssangtikeut"},
/* 3139 */ { "HANGUL LETTER RIEUL","	# 1105 hangul choseong rieul"},
/* 313A */ { "HANGUL LETTER RIEUL-KIYEOK","	# 11B0 hangul jongseong rieul-kiyeok"},
/* 313B */ { "HANGUL LETTER RIEUL-MIEUM","	# 11B1 hangul jongseong rieul-mieum"},
/* 313C */ { "HANGUL LETTER RIEUL-PIEUP","	# 11B2 hangul jongseong rieul-pieup"},
/* 313D */ { "HANGUL LETTER RIEUL-SIOS","	# 11B3 hangul jongseong rieul-sios"},
/* 313E */ { "HANGUL LETTER RIEUL-THIEUTH","	# 11B4 hangul jongseong rieul-thieuth"},
/* 313F */ { "HANGUL LETTER RIEUL-PHIEUPH","	# 11B5 hangul jongseong rieul-phieuph"},
/* 3140 */ { "HANGUL LETTER RIEUL-HIEUH","	# 111A hangul choseong rieul-hieuh"},
/* 3141 */ { "HANGUL LETTER MIEUM","	# 1106 hangul choseong mieum"},
/* 3142 */ { "HANGUL LETTER PIEUP","	# 1107 hangul choseong pieup"},
/* 3143 */ { "HANGUL LETTER SSANGPIEUP","	# 1108 hangul choseong ssangpieup"},
/* 3144 */ { "HANGUL LETTER PIEUP-SIOS","	# 1121 hangul choseong pieup-sios"},
/* 3145 */ { "HANGUL LETTER SIOS","	# 1109 hangul choseong sios"},
/* 3146 */ { "HANGUL LETTER SSANGSIOS","	# 110A hangul choseong ssangsios"},
/* 3147 */ { "HANGUL LETTER IEUNG","	* zero sound as initial or velar nasal consonant as final\n"
	"	# 110B hangul choseong ieung"},
/* 3148 */ { "HANGUL LETTER CIEUC","	# 110C hangul choseong cieuc"},
/* 3149 */ { "HANGUL LETTER SSANGCIEUC","	# 110D hangul choseong ssangcieuc"},
/* 314A */ { "HANGUL LETTER CHIEUCH","	# 110E hangul choseong chieuch"},
/* 314B */ { "HANGUL LETTER KHIEUKH","	# 110F hangul choseong khieukh"},
/* 314C */ { "HANGUL LETTER THIEUTH","	# 1110 hangul choseong thieuth"},
/* 314D */ { "HANGUL LETTER PHIEUPH","	# 1111 hangul choseong phieuph"},
/* 314E */ { "HANGUL LETTER HIEUH","	* voiceless glottal fricative\n"
	"	# 1112 hangul choseong hieuh"},
/* 314F */ { "HANGUL LETTER A","	# 1161 hangul jungseong a"},
/* 3150 */ { "HANGUL LETTER AE","	# 1162 hangul jungseong ae"},
/* 3151 */ { "HANGUL LETTER YA","	# 1163 hangul jungseong ya"},
/* 3152 */ { "HANGUL LETTER YAE","	# 1164 hangul jungseong yae"},
/* 3153 */ { "HANGUL LETTER EO","	# 1165 hangul jungseong eo"},
/* 3154 */ { "HANGUL LETTER E","	# 1166 hangul jungseong e"},
/* 3155 */ { "HANGUL LETTER YEO","	# 1167 hangul jungseong yeo"},
/* 3156 */ { "HANGUL LETTER YE","	# 1168 hangul jungseong ye"},
/* 3157 */ { "HANGUL LETTER O","	# 1169 hangul jungseong o"},
/* 3158 */ { "HANGUL LETTER WA","	# 116A hangul jungseong wa"},
/* 3159 */ { "HANGUL LETTER WAE","	# 116B hangul jungseong wae"},
/* 315A */ { "HANGUL LETTER OE","	# 116C hangul jungseong oe"},
/* 315B */ { "HANGUL LETTER YO","	# 116D hangul jungseong yo"},
/* 315C */ { "HANGUL LETTER U","	# 116E hangul jungseong u"},
/* 315D */ { "HANGUL LETTER WEO","	# 116F hangul jungseong weo"},
/* 315E */ { "HANGUL LETTER WE","	# 1170 hangul jungseong we"},
/* 315F */ { "HANGUL LETTER WI","	# 1171 hangul jungseong wi"},
/* 3160 */ { "HANGUL LETTER YU","	# 1172 hangul jungseong yu"},
/* 3161 */ { "HANGUL LETTER EU","	# 1173 hangul jungseong eu"},
/* 3162 */ { "HANGUL LETTER YI","	# 1174 hangul jungseong yi"},
/* 3163 */ { "HANGUL LETTER I","	# 1175 hangul jungseong i"},
/* 3164 */ { "HANGUL FILLER","	= chaeum\n"
	"	# 1160 hangul jungseong filler"},
/* 3165 */ { "HANGUL LETTER SSANGNIEUN","	# 1114 hangul choseong ssangnieun"},
/* 3166 */ { "HANGUL LETTER NIEUN-TIKEUT","	# 1115 hangul choseong nieun-tikeut"},
/* 3167 */ { "HANGUL LETTER NIEUN-SIOS","	# 11C7 hangul jongseong nieun-sios"},
/* 3168 */ { "HANGUL LETTER NIEUN-PANSIOS","	# 11C8 hangul jongseong nieun-pansios"},
/* 3169 */ { "HANGUL LETTER RIEUL-KIYEOK-SIOS","	# 11CC hangul jongseong rieul-kiyeok-sios"},
/* 316A */ { "HANGUL LETTER RIEUL-TIKEUT","	# 11CE hangul jongseong rieul-tikeut"},
/* 316B */ { "HANGUL LETTER RIEUL-PIEUP-SIOS","	# 11D3 hangul jongseong rieul-pieup-sios"},
/* 316C */ { "HANGUL LETTER RIEUL-PANSIOS","	# 11D7 hangul jongseong rieul-pansios"},
/* 316D */ { "HANGUL LETTER RIEUL-YEORINHIEUH","	# 11D9 hangul jongseong rieul-yeorinhieuh"},
/* 316E */ { "HANGUL LETTER MIEUM-PIEUP","	# 111C hangul choseong mieum-pieup"},
/* 316F */ { "HANGUL LETTER MIEUM-SIOS","	# 11DD hangul jongseong mieum-sios"},
/* 3170 */ { "HANGUL LETTER MIEUM-PANSIOS","	# 11DF hangul jongseong mieum-pansios"},
/* 3171 */ { "HANGUL LETTER KAPYEOUNMIEUM","	# 111D hangul choseong kapyeounmieum"},
/* 3172 */ { "HANGUL LETTER PIEUP-KIYEOK","	# 111E hangul choseong pieup-kiyeok"},
/* 3173 */ { "HANGUL LETTER PIEUP-TIKEUT","	# 1120 hangul choseong pieup-tikeut"},
/* 3174 */ { "HANGUL LETTER PIEUP-SIOS-KIYEOK","	# 1122 hangul choseong pieup-sios-kiyeok"},
/* 3175 */ { "HANGUL LETTER PIEUP-SIOS-TIKEUT","	# 1123 hangul choseong pieup-sios-tikeut"},
/* 3176 */ { "HANGUL LETTER PIEUP-CIEUC","	# 1127 hangul choseong pieup-cieuc"},
/* 3177 */ { "HANGUL LETTER PIEUP-THIEUTH","	# 1129 hangul choseong pieup-thieuth"},
/* 3178 */ { "HANGUL LETTER KAPYEOUNPIEUP","	# 112B hangul choseong kapyeounpieup"},
/* 3179 */ { "HANGUL LETTER KAPYEOUNSSANGPIEUP","	# 112C hangul choseong kapyeounssangpieup"},
/* 317A */ { "HANGUL LETTER SIOS-KIYEOK","	# 112D hangul choseong sios-kiyeok"},
/* 317B */ { "HANGUL LETTER SIOS-NIEUN","	# 112E hangul choseong sios-nieun"},
/* 317C */ { "HANGUL LETTER SIOS-TIKEUT","	# 112F hangul choseong sios-tikeut"},
/* 317D */ { "HANGUL LETTER SIOS-PIEUP","	# 1132 hangul choseong sios-pieup"},
/* 317E */ { "HANGUL LETTER SIOS-CIEUC","	# 1136 hangul choseong sios-cieuc"},
/* 317F */ { "HANGUL LETTER PANSIOS","	* voiced alveolar fricative\n"
	"	# 1140 hangul choseong pansios"},
/* 3180 */ { "HANGUL LETTER SSANGIEUNG","	= ssangyesieung\n"
	"	# 1147 hangul choseong ssangieung"},
/* 3181 */ { "HANGUL LETTER YESIEUNG","	* velar nasal consonant\n"
	"	# 114C hangul choseong yesieung"},
/* 3182 */ { "HANGUL LETTER YESIEUNG-SIOS","	# 11F1 hangul jongseong yesieung-sios"},
/* 3183 */ { "HANGUL LETTER YESIEUNG-PANSIOS","	# 11F2 hangul jongseong yesieung-pansios"},
/* 3184 */ { "HANGUL LETTER KAPYEOUNPHIEUPH","	# 1157 hangul choseong kapyeounphieuph"},
/* 3185 */ { "HANGUL LETTER SSANGHIEUH","	# 1158 hangul choseong ssanghieuh"},
/* 3186 */ { "HANGUL LETTER YEORINHIEUH","	* glottal stop\n"
	"	# 1159 hangul choseong yeorinhieuh"},
/* 3187 */ { "HANGUL LETTER YO-YA","	# 1184 hangul jungseong yo-ya"},
/* 3188 */ { "HANGUL LETTER YO-YAE","	# 1185 hangul jungseong yo-yae"},
/* 3189 */ { "HANGUL LETTER YO-I","	# 1188 hangul jungseong yo-i"},
/* 318A */ { "HANGUL LETTER YU-YEO","	# 1191 hangul jungseong yu-yeo"},
/* 318B */ { "HANGUL LETTER YU-YE","	# 1192 hangul jungseong yu-ye"},
/* 318C */ { "HANGUL LETTER YU-I","	# 1194 hangul jungseong yu-i"},
/* 318D */ { "HANGUL LETTER ARAEA","	* rounded open-mid back vowel\n"
	"	# 119E hangul jungseong araea"},
/* 318E */ { "HANGUL LETTER ARAEAE","	# 11A1 hangul jungseong araea-i"},
/* 318F */ { NULL,NULL},
/* 3190 */ { "IDEOGRAPHIC ANNOTATION LINKING MARK","	= tateten"},
/* 3191 */ { "IDEOGRAPHIC ANNOTATION REVERSE MARK","	= kaeriten re"},
/* 3192 */ { "IDEOGRAPHIC ANNOTATION ONE MARK","	# <super> 4E00"},
/* 3193 */ { "IDEOGRAPHIC ANNOTATION TWO MARK","	# <super> 4E8C"},
/* 3194 */ { "IDEOGRAPHIC ANNOTATION THREE MARK","	# <super> 4E09"},
/* 3195 */ { "IDEOGRAPHIC ANNOTATION FOUR MARK","	# <super> 56DB"},
/* 3196 */ { "IDEOGRAPHIC ANNOTATION TOP MARK","	# <super> 4E0A"},
/* 3197 */ { "IDEOGRAPHIC ANNOTATION MIDDLE MARK","	# <super> 4E2D"},
/* 3198 */ { "IDEOGRAPHIC ANNOTATION BOTTOM MARK","	# <super> 4E0B"},
/* 3199 */ { "IDEOGRAPHIC ANNOTATION FIRST MARK","	# <super> 7532"},
/* 319A */ { "IDEOGRAPHIC ANNOTATION SECOND MARK","	# <super> 4E59"},
/* 319B */ { "IDEOGRAPHIC ANNOTATION THIRD MARK","	# <super> 4E19"},
/* 319C */ { "IDEOGRAPHIC ANNOTATION FOURTH MARK","	# <super> 4E01"},
/* 319D */ { "IDEOGRAPHIC ANNOTATION HEAVEN MARK","	# <super> 5929"},
/* 319E */ { "IDEOGRAPHIC ANNOTATION EARTH MARK","	# <super> 5730"},
/* 319F */ { "IDEOGRAPHIC ANNOTATION MAN MARK","	# <super> 4EBA"},
/* 31A0 */ { "BOPOMOFO LETTER BU",NULL},
/* 31A1 */ { "BOPOMOFO LETTER ZI",NULL},
/* 31A2 */ { "BOPOMOFO LETTER JI",NULL},
/* 31A3 */ { "BOPOMOFO LETTER GU",NULL},
/* 31A4 */ { "BOPOMOFO LETTER EE",NULL},
/* 31A5 */ { "BOPOMOFO LETTER ENN",NULL},
/* 31A6 */ { "BOPOMOFO LETTER OO",NULL},
/* 31A7 */ { "BOPOMOFO LETTER ONN",NULL},
/* 31A8 */ { "BOPOMOFO LETTER IR",NULL},
/* 31A9 */ { "BOPOMOFO LETTER ANN",NULL},
/* 31AA */ { "BOPOMOFO LETTER INN",NULL},
/* 31AB */ { "BOPOMOFO LETTER UNN",NULL},
/* 31AC */ { "BOPOMOFO LETTER IM",NULL},
/* 31AD */ { "BOPOMOFO LETTER NGG",NULL},
/* 31AE */ { "BOPOMOFO LETTER AINN",NULL},
/* 31AF */ { "BOPOMOFO LETTER AUNN",NULL},
/* 31B0 */ { "BOPOMOFO LETTER AM",NULL},
/* 31B1 */ { "BOPOMOFO LETTER OM",NULL},
/* 31B2 */ { "BOPOMOFO LETTER ONG",NULL},
/* 31B3 */ { "BOPOMOFO LETTER INNN",NULL},
/* 31B4 */ { "BOPOMOFO FINAL LETTER P",NULL},
/* 31B5 */ { "BOPOMOFO FINAL LETTER T",NULL},
/* 31B6 */ { "BOPOMOFO FINAL LETTER K","	* use of 31BB is preferred"},
/* 31B7 */ { "BOPOMOFO FINAL LETTER H",NULL},
/* 31B8 */ { "BOPOMOFO LETTER GH",NULL},
/* 31B9 */ { "BOPOMOFO LETTER LH",NULL},
/* 31BA */ { "BOPOMOFO LETTER ZY",NULL},
/* 31BB */ { "BOPOMOFO FINAL LETTER G",NULL},
/* 31BC */ { "BOPOMOFO LETTER GW",NULL},
/* 31BD */ { "BOPOMOFO LETTER KW",NULL},
/* 31BE */ { "BOPOMOFO LETTER OE",NULL},
/* 31BF */ { "BOPOMOFO LETTER AH",NULL},
/* 31C0 */ { "CJK STROKE T","	* 2nd stroke of 51B0"},
/* 31C1 */ { "CJK STROKE WG","	* 2nd stroke of 72D0"},
/* 31C2 */ { "CJK STROKE XG","	* 5th stroke of 6211"},
/* 31C3 */ { "CJK STROKE BXG","	* occurs only in Kaiti typefaces\n"
	"	* equivalent to SWG stroke in Songti typefaces\n"
	"	* 2nd stroke of 5FC3"},
/* 31C4 */ { "CJK STROKE SW","	* 3rd stroke of 4EA1\n"
	"	* 4th stroke of 56DB"},
/* 31C5 */ { "CJK STROKE HZZ","	* 1st stroke of 534D"},
/* 31C6 */ { "CJK STROKE HZG","	* 1st stroke of 7FBD\n"
	"	* 1st stroke of 4E5F"},
/* 31C7 */ { "CJK STROKE HP","	* 1st stroke of 53C8\n"
	"	* 4th stroke of 4ECA"},
/* 31C8 */ { "CJK STROKE HZWG","	* 2nd stroke of 4E5D"},
/* 31C9 */ { "CJK STROKE SZWG","	* 3rd stroke of 5F13\n"
	"	* 2nd stroke of 9A6C"},
/* 31CA */ { "CJK STROKE HZT","	* 2nd stroke of 8BA1\n"
	"	* 2nd stroke of 9CE9"},
/* 31CB */ { "CJK STROKE HZZP","	* 1st stroke of 53CA"},
/* 31CC */ { "CJK STROKE HPWG","	* 1st stroke of 961D\n"
	"	* 6th stroke of 90AE"},
/* 31CD */ { "CJK STROKE HZW","	* 5th stroke of 6295"},
/* 31CE */ { "CJK STROKE HZZZ","	* 1st stroke of 51F8"},
/* 31CF */ { "CJK STROKE N","	* 3rd stroke of 5927"},
/* 31D0 */ { "CJK STROKE H","	* 1st stroke of 5927\n"
	"	* 1st stroke of 4E03"},
/* 31D1 */ { "CJK STROKE S","	* 4th stroke of 4E2D"},
/* 31D2 */ { "CJK STROKE P","	* 1st stroke of 4E4F"},
/* 31D3 */ { "CJK STROKE SP","	* 1st stroke of 6708"},
/* 31D4 */ { "CJK STROKE D","	* 3rd stroke of 4E38"},
/* 31D5 */ { "CJK STROKE HZ","	* 2nd stroke of 56DB"},
/* 31D6 */ { "CJK STROKE HG","	* 1st stroke of 758B\n"
	"	* 1st stroke of 5B50"},
/* 31D7 */ { "CJK STROKE SZ","	* 2nd stroke of 5C71\n"
	"	* 2nd stroke of 4E1C"},
/* 31D8 */ { "CJK STROKE SWZ","	* 6th stroke of 8085"},
/* 31D9 */ { "CJK STROKE ST","	* 3rd stroke of 6C11"},
/* 31DA */ { "CJK STROKE SG","	* 1st stroke of 6C34"},
/* 31DB */ { "CJK STROKE PD","	* 1st stroke of 5DE1\n"
	"	* 1st stroke of 5973"},
/* 31DC */ { "CJK STROKE PZ","	* 3rd stroke of 516C\n"
	"	* 4th stroke of 5F18"},
/* 31DD */ { "CJK STROKE TN","	* 8th stroke of 5EFB"},
/* 31DE */ { "CJK STROKE SZZ","	* 1st stroke of 5350\n"
	"	* 4th stroke of 4E9E"},
/* 31DF */ { "CJK STROKE SWG","	* 7th stroke of 4E71\n"
	"	* 3rd stroke of 5DF1"},
/* 31E0 */ { "CJK STROKE HXWG","	* 1st stroke of 4E59"},
/* 31E1 */ { "CJK STROKE HZZZG","	* 1st stroke of 4E43"},
/* 31E2 */ { "CJK STROKE PG","	* 1st stroke of 4E44"},
/* 31E3 */ { "CJK STROKE Q","	* 6th stroke of 3514"},
/* 31E4 */ { "CJK STROKE HXG","	* 1st stroke of 98DE"},
/* 31E5 */ { "CJK STROKE SZP","	* 8th stroke of 594A"},
/* 31E6 */ { NULL,NULL},
/* 31E7 */ { NULL,NULL},
/* 31E8 */ { NULL,NULL},
/* 31E9 */ { NULL,NULL},
/* 31EA */ { NULL,NULL},
/* 31EB */ { NULL,NULL},
/* 31EC */ { NULL,NULL},
/* 31ED */ { NULL,NULL},
/* 31EE */ { NULL,NULL},
/* 31EF */ { "IDEOGRAPHIC DESCRIPTION CHARACTER SUBTRACTION","	* used to describe a character or a component with a stroke (or other component) removed"},
/* 31F0 */ { "KATAKANA LETTER SMALL KU",NULL},
/* 31F1 */ { "KATAKANA LETTER SMALL SI",NULL},
/* 31F2 */ { "KATAKANA LETTER SMALL SU",NULL},
/* 31F3 */ { "KATAKANA LETTER SMALL TO",NULL},
/* 31F4 */ { "KATAKANA LETTER SMALL NU",NULL},
/* 31F5 */ { "KATAKANA LETTER SMALL HA",NULL},
/* 31F6 */ { "KATAKANA LETTER SMALL HI",NULL},
/* 31F7 */ { "KATAKANA LETTER SMALL HU",NULL},
/* 31F8 */ { "KATAKANA LETTER SMALL HE",NULL},
/* 31F9 */ { "KATAKANA LETTER SMALL HO",NULL},
/* 31FA */ { "KATAKANA LETTER SMALL MU",NULL},
/* 31FB */ { "KATAKANA LETTER SMALL RA",NULL},
/* 31FC */ { "KATAKANA LETTER SMALL RI",NULL},
/* 31FD */ { "KATAKANA LETTER SMALL RU",NULL},
/* 31FE */ { "KATAKANA LETTER SMALL RE",NULL},
/* 31FF */ { "KATAKANA LETTER SMALL RO",NULL}
};

UN_DLL_LOCAL
static const struct unicode_nameannot una_00_32[] = {
/* 3200 */ { "PARENTHESIZED HANGUL KIYEOK","	# 0028 1100 0029"},
/* 3201 */ { "PARENTHESIZED HANGUL NIEUN","	# 0028 1102 0029"},
/* 3202 */ { "PARENTHESIZED HANGUL TIKEUT","	# 0028 1103 0029"},
/* 3203 */ { "PARENTHESIZED HANGUL RIEUL","	# 0028 1105 0029"},
/* 3204 */ { "PARENTHESIZED HANGUL MIEUM","	# 0028 1106 0029"},
/* 3205 */ { "PARENTHESIZED HANGUL PIEUP","	# 0028 1107 0029"},
/* 3206 */ { "PARENTHESIZED HANGUL SIOS","	# 0028 1109 0029"},
/* 3207 */ { "PARENTHESIZED HANGUL IEUNG","	# 0028 110B 0029"},
/* 3208 */ { "PARENTHESIZED HANGUL CIEUC","	# 0028 110C 0029"},
/* 3209 */ { "PARENTHESIZED HANGUL CHIEUCH","	# 0028 110E 0029"},
/* 320A */ { "PARENTHESIZED HANGUL KHIEUKH","	# 0028 110F 0029"},
/* 320B */ { "PARENTHESIZED HANGUL THIEUTH","	# 0028 1110 0029"},
/* 320C */ { "PARENTHESIZED HANGUL PHIEUPH","	# 0028 1111 0029"},
/* 320D */ { "PARENTHESIZED HANGUL HIEUH","	# 0028 1112 0029"},
/* 320E */ { "PARENTHESIZED HANGUL KIYEOK A","	# 0028 1100 1161 0029"},
/* 320F */ { "PARENTHESIZED HANGUL NIEUN A","	# 0028 1102 1161 0029"},
/* 3210 */ { "PARENTHESIZED HANGUL TIKEUT A","	# 0028 1103 1161 0029"},
/* 3211 */ { "PARENTHESIZED HANGUL RIEUL A","	# 0028 1105 1161 0029"},
/* 3212 */ { "PARENTHESIZED HANGUL MIEUM A","	# 0028 1106 1161 0029"},
/* 3213 */ { "PARENTHESIZED HANGUL PIEUP A","	# 0028 1107 1161 0029"},
/* 3214 */ { "PARENTHESIZED HANGUL SIOS A","	# 0028 1109 1161 0029"},
/* 3215 */ { "PARENTHESIZED HANGUL IEUNG A","	# 0028 110B 1161 0029"},
/* 3216 */ { "PARENTHESIZED HANGUL CIEUC A","	# 0028 110C 1161 0029"},
/* 3217 */ { "PARENTHESIZED HANGUL CHIEUCH A","	# 0028 110E 1161 0029"},
/* 3218 */ { "PARENTHESIZED HANGUL KHIEUKH A","	# 0028 110F 1161 0029"},
/* 3219 */ { "PARENTHESIZED HANGUL THIEUTH A","	# 0028 1110 1161 0029"},
/* 321A */ { "PARENTHESIZED HANGUL PHIEUPH A","	# 0028 1111 1161 0029"},
/* 321B */ { "PARENTHESIZED HANGUL HIEUH A","	# 0028 1112 1161 0029"},
/* 321C */ { "PARENTHESIZED HANGUL CIEUC U","	# 0028 110C 116E 0029"},
/* 321D */ { "PARENTHESIZED KOREAN CHARACTER OJEON","	# 0028 110B 1169 110C 1165 11AB 0029"},
/* 321E */ { "PARENTHESIZED KOREAN CHARACTER O HU","	* preferred spelling for the name is ohu\n"
	"	# 0028 110B 1169 1112 116E 0029"},
/* 321F */ { NULL,NULL},
/* 3220 */ { "PARENTHESIZED IDEOGRAPH ONE","	# 0028 4E00 0029"},
/* 3221 */ { "PARENTHESIZED IDEOGRAPH TWO","	# 0028 4E8C 0029"},
/* 3222 */ { "PARENTHESIZED IDEOGRAPH THREE","	# 0028 4E09 0029"},
/* 3223 */ { "PARENTHESIZED IDEOGRAPH FOUR","	# 0028 56DB 0029"},
/* 3224 */ { "PARENTHESIZED IDEOGRAPH FIVE","	# 0028 4E94 0029"},
/* 3225 */ { "PARENTHESIZED IDEOGRAPH SIX","	# 0028 516D 0029"},
/* 3226 */ { "PARENTHESIZED IDEOGRAPH SEVEN","	# 0028 4E03 0029"},
/* 3227 */ { "PARENTHESIZED IDEOGRAPH EIGHT","	# 0028 516B 0029"},
/* 3228 */ { "PARENTHESIZED IDEOGRAPH NINE","	# 0028 4E5D 0029"},
/* 3229 */ { "PARENTHESIZED IDEOGRAPH TEN","	# 0028 5341 0029"},
/* 322A */ { "PARENTHESIZED IDEOGRAPH MOON","	* Monday\n"
	"	# 0028 6708 0029"},
/* 322B */ { "PARENTHESIZED IDEOGRAPH FIRE","	* Tuesday\n"
	"	# 0028 706B 0029"},
/* 322C */ { "PARENTHESIZED IDEOGRAPH WATER","	* Wednesday\n"
	"	# 0028 6C34 0029"},
/* 322D */ { "PARENTHESIZED IDEOGRAPH WOOD","	* Thursday\n"
	"	# 0028 6728 0029"},
/* 322E */ { "PARENTHESIZED IDEOGRAPH METAL","	* Friday\n"
	"	# 0028 91D1 0029"},
/* 322F */ { "PARENTHESIZED IDEOGRAPH EARTH","	* Saturday\n"
	"	# 0028 571F 0029"},
/* 3230 */ { "PARENTHESIZED IDEOGRAPH SUN","	* Sunday\n"
	"	# 0028 65E5 0029"},
/* 3231 */ { "PARENTHESIZED IDEOGRAPH STOCK","	* incorporated\n"
	"	# 0028 682A 0029"},
/* 3232 */ { "PARENTHESIZED IDEOGRAPH HAVE","	* limited\n"
	"	# 0028 6709 0029"},
/* 3233 */ { "PARENTHESIZED IDEOGRAPH SOCIETY","	* company\n"
	"	# 0028 793E 0029"},
/* 3234 */ { "PARENTHESIZED IDEOGRAPH NAME","	# 0028 540D 0029"},
/* 3235 */ { "PARENTHESIZED IDEOGRAPH SPECIAL","	# 0028 7279 0029"},
/* 3236 */ { "PARENTHESIZED IDEOGRAPH FINANCIAL","	# 0028 8CA1 0029"},
/* 3237 */ { "PARENTHESIZED IDEOGRAPH CONGRATULATION","	# 0028 795D 0029"},
/* 3238 */ { "PARENTHESIZED IDEOGRAPH LABOR","	# 0028 52B4 0029"},
/* 3239 */ { "PARENTHESIZED IDEOGRAPH REPRESENT","	# 0028 4EE3 0029"},
/* 323A */ { "PARENTHESIZED IDEOGRAPH CALL","	# 0028 547C 0029"},
/* 323B */ { "PARENTHESIZED IDEOGRAPH STUDY","	# 0028 5B66 0029"},
/* 323C */ { "PARENTHESIZED IDEOGRAPH SUPERVISE","	# 0028 76E3 0029"},
/* 323D */ { "PARENTHESIZED IDEOGRAPH ENTERPRISE","	# 0028 4F01 0029"},
/* 323E */ { "PARENTHESIZED IDEOGRAPH RESOURCE","	# 0028 8CC7 0029"},
/* 323F */ { "PARENTHESIZED IDEOGRAPH ALLIANCE","	# 0028 5354 0029"},
/* 3240 */ { "PARENTHESIZED IDEOGRAPH FESTIVAL","	# 0028 796D 0029"},
/* 3241 */ { "PARENTHESIZED IDEOGRAPH REST","	# 0028 4F11 0029"},
/* 3242 */ { "PARENTHESIZED IDEOGRAPH SELF","	* from\n"
	"	# 0028 81EA 0029"},
/* 3243 */ { "PARENTHESIZED IDEOGRAPH REACH","	* to\n"
	"	# 0028 81F3 0029"},
/* 3244 */ { "CIRCLED IDEOGRAPH QUESTION","	# <circle> 554F"},
/* 3245 */ { "CIRCLED IDEOGRAPH KINDERGARTEN","	# <circle> 5E7C"},
/* 3246 */ { "CIRCLED IDEOGRAPH SCHOOL","	# <circle> 6587"},
/* 3247 */ { "CIRCLED IDEOGRAPH KOTO","	# <circle> 7B8F"},
/* 3248 */ { "CIRCLED NUMBER TEN ON BLACK SQUARE","	= speed limit 10 km/h"},
/* 3249 */ { "CIRCLED NUMBER TWENTY ON BLACK SQUARE","	= speed limit 20 km/h"},
/* 324A */ { "CIRCLED NUMBER THIRTY ON BLACK SQUARE","	= speed limit 30 km/h"},
/* 324B */ { "CIRCLED NUMBER FORTY ON BLACK SQUARE","	= speed limit 40 km/h"},
/* 324C */ { "CIRCLED NUMBER FIFTY ON BLACK SQUARE","	= speed limit 50 km/h"},
/* 324D */ { "CIRCLED NUMBER SIXTY ON BLACK SQUARE","	= speed limit 60 km/h"},
/* 324E */ { "CIRCLED NUMBER SEVENTY ON BLACK SQUARE","	= speed limit 70 km/h"},
/* 324F */ { "CIRCLED NUMBER EIGHTY ON BLACK SQUARE","	= speed limit 80 km/h"},
/* 3250 */ { "PARTNERSHIP SIGN","	# <square> 0050 0054 0045"},
/* 3251 */ { "CIRCLED NUMBER TWENTY ONE","	# <circle> 0032 0031"},
/* 3252 */ { "CIRCLED NUMBER TWENTY TWO","	# <circle> 0032 0032"},
/* 3253 */ { "CIRCLED NUMBER TWENTY THREE","	# <circle> 0032 0033"},
/* 3254 */ { "CIRCLED NUMBER TWENTY FOUR","	# <circle> 0032 0034"},
/* 3255 */ { "CIRCLED NUMBER TWENTY FIVE","	# <circle> 0032 0035"},
/* 3256 */ { "CIRCLED NUMBER TWENTY SIX","	# <circle> 0032 0036"},
/* 3257 */ { "CIRCLED NUMBER TWENTY SEVEN","	# <circle> 0032 0037"},
/* 3258 */ { "CIRCLED NUMBER TWENTY EIGHT","	# <circle> 0032 0038"},
/* 3259 */ { "CIRCLED NUMBER TWENTY NINE","	# <circle> 0032 0039"},
/* 325A */ { "CIRCLED NUMBER THIRTY","	# <circle> 0033 0030"},
/* 325B */ { "CIRCLED NUMBER THIRTY ONE","	# <circle> 0033 0031"},
/* 325C */ { "CIRCLED NUMBER THIRTY TWO","	# <circle> 0033 0032"},
/* 325D */ { "CIRCLED NUMBER THIRTY THREE","	# <circle> 0033 0033"},
/* 325E */ { "CIRCLED NUMBER THIRTY FOUR","	# <circle> 0033 0034"},
/* 325F */ { "CIRCLED NUMBER THIRTY FIVE","	# <circle> 0033 0035"},
/* 3260 */ { "CIRCLED HANGUL KIYEOK","	# <circle> 1100"},
/* 3261 */ { "CIRCLED HANGUL NIEUN","	# <circle> 1102"},
/* 3262 */ { "CIRCLED HANGUL TIKEUT","	# <circle> 1103"},
/* 3263 */ { "CIRCLED HANGUL RIEUL","	# <circle> 1105"},
/* 3264 */ { "CIRCLED HANGUL MIEUM","	# <circle> 1106"},
/* 3265 */ { "CIRCLED HANGUL PIEUP","	# <circle> 1107"},
/* 3266 */ { "CIRCLED HANGUL SIOS","	# <circle> 1109"},
/* 3267 */ { "CIRCLED HANGUL IEUNG","	# <circle> 110B"},
/* 3268 */ { "CIRCLED HANGUL CIEUC","	# <circle> 110C"},
/* 3269 */ { "CIRCLED HANGUL CHIEUCH","	# <circle> 110E"},
/* 326A */ { "CIRCLED HANGUL KHIEUKH","	# <circle> 110F"},
/* 326B */ { "CIRCLED HANGUL THIEUTH","	# <circle> 1110"},
/* 326C */ { "CIRCLED HANGUL PHIEUPH","	# <circle> 1111"},
/* 326D */ { "CIRCLED HANGUL HIEUH","	# <circle> 1112"},
/* 326E */ { "CIRCLED HANGUL KIYEOK A","	# <circle> 1100 1161"},
/* 326F */ { "CIRCLED HANGUL NIEUN A","	# <circle> 1102 1161"},
/* 3270 */ { "CIRCLED HANGUL TIKEUT A","	# <circle> 1103 1161"},
/* 3271 */ { "CIRCLED HANGUL RIEUL A","	# <circle> 1105 1161"},
/* 3272 */ { "CIRCLED HANGUL MIEUM A","	# <circle> 1106 1161"},
/* 3273 */ { "CIRCLED HANGUL PIEUP A","	# <circle> 1107 1161"},
/* 3274 */ { "CIRCLED HANGUL SIOS A","	# <circle> 1109 1161"},
/* 3275 */ { "CIRCLED HANGUL IEUNG A","	# <circle> 110B 1161"},
/* 3276 */ { "CIRCLED HANGUL CIEUC A","	# <circle> 110C 1161"},
/* 3277 */ { "CIRCLED HANGUL CHIEUCH A","	# <circle> 110E 1161"},
/* 3278 */ { "CIRCLED HANGUL KHIEUKH A","	# <circle> 110F 1161"},
/* 3279 */ { "CIRCLED HANGUL THIEUTH A","	# <circle> 1110 1161"},
/* 327A */ { "CIRCLED HANGUL PHIEUPH A","	# <circle> 1111 1161"},
/* 327B */ { "CIRCLED HANGUL HIEUH A","	# <circle> 1112 1161"},
/* 327C */ { "CIRCLED KOREAN CHARACTER CHAMKO","	# <circle> 110E 1161 11B7 1100 1169"},
/* 327D */ { "CIRCLED KOREAN CHARACTER JUEUI","	# <circle> 110C 116E 110B 1174"},
/* 327E */ { "CIRCLED HANGUL IEUNG U","	* postal code mark\n"
	"	# <circle> 110B 116E"},
/* 327F */ { "KOREAN STANDARD SYMBOL",NULL},
/* 3280 */ { "CIRCLED IDEOGRAPH ONE","	= maru-iti, symbol of unification\n"
	"	# <circle> 4E00"},
/* 3281 */ { "CIRCLED IDEOGRAPH TWO","	# <circle> 4E8C"},
/* 3282 */ { "CIRCLED IDEOGRAPH THREE","	# <circle> 4E09"},
/* 3283 */ { "CIRCLED IDEOGRAPH FOUR","	# <circle> 56DB"},
/* 3284 */ { "CIRCLED IDEOGRAPH FIVE","	# <circle> 4E94"},
/* 3285 */ { "CIRCLED IDEOGRAPH SIX","	# <circle> 516D"},
/* 3286 */ { "CIRCLED IDEOGRAPH SEVEN","	# <circle> 4E03"},
/* 3287 */ { "CIRCLED IDEOGRAPH EIGHT","	# <circle> 516B"},
/* 3288 */ { "CIRCLED IDEOGRAPH NINE","	# <circle> 4E5D"},
/* 3289 */ { "CIRCLED IDEOGRAPH TEN","	# <circle> 5341"},
/* 328A */ { "CIRCLED IDEOGRAPH MOON","	* Monday\n"
	"	# <circle> 6708"},
/* 328B */ { "CIRCLED IDEOGRAPH FIRE","	* Tuesday\n"
	"	# <circle> 706B"},
/* 328C */ { "CIRCLED IDEOGRAPH WATER","	* Wednesday\n"
	"	# <circle> 6C34"},
/* 328D */ { "CIRCLED IDEOGRAPH WOOD","	* Thursday\n"
	"	# <circle> 6728"},
/* 328E */ { "CIRCLED IDEOGRAPH METAL","	* Friday\n"
	"	# <circle> 91D1"},
/* 328F */ { "CIRCLED IDEOGRAPH EARTH","	* Saturday\n"
	"	# <circle> 571F"},
/* 3290 */ { "CIRCLED IDEOGRAPH SUN","	* Sunday\n"
	"	# <circle> 65E5"},
/* 3291 */ { "CIRCLED IDEOGRAPH STOCK","	* incorporated\n"
	"	# <circle> 682A"},
/* 3292 */ { "CIRCLED IDEOGRAPH HAVE","	* limited\n"
	"	# <circle> 6709"},
/* 3293 */ { "CIRCLED IDEOGRAPH SOCIETY","	* company\n"
	"	# <circle> 793E"},
/* 3294 */ { "CIRCLED IDEOGRAPH NAME","	# <circle> 540D"},
/* 3295 */ { "CIRCLED IDEOGRAPH SPECIAL","	# <circle> 7279"},
/* 3296 */ { "CIRCLED IDEOGRAPH FINANCIAL","	# <circle> 8CA1"},
/* 3297 */ { "CIRCLED IDEOGRAPH CONGRATULATION","	# <circle> 795D"},
/* 3298 */ { "CIRCLED IDEOGRAPH LABOR","	# <circle> 52B4"},
/* 3299 */ { "CIRCLED IDEOGRAPH SECRET","	# <circle> 79D8"},
/* 329A */ { "CIRCLED IDEOGRAPH MALE","	# <circle> 7537"},
/* 329B */ { "CIRCLED IDEOGRAPH FEMALE","	# <circle> 5973"},
/* 329C */ { "CIRCLED IDEOGRAPH SUITABLE","	# <circle> 9069"},
/* 329D */ { "CIRCLED IDEOGRAPH EXCELLENT","	# <circle> 512A"},
/* 329E */ { "CIRCLED IDEOGRAPH PRINT","	* name seal\n"
	"	# <circle> 5370"},
/* 329F */ { "CIRCLED IDEOGRAPH ATTENTION","	# <circle> 6CE8"},
/* 32A0 */ { "CIRCLED IDEOGRAPH ITEM","	# <circle> 9805"},
/* 32A1 */ { "CIRCLED IDEOGRAPH REST","	* holiday\n"
	"	# <circle> 4F11"},
/* 32A2 */ { "CIRCLED IDEOGRAPH COPY","	# <circle> 5199"},
/* 32A3 */ { "CIRCLED IDEOGRAPH CORRECT","	# <circle> 6B63"},
/* 32A4 */ { "CIRCLED IDEOGRAPH HIGH","	# <circle> 4E0A"},
/* 32A5 */ { "CIRCLED IDEOGRAPH CENTRE","	# <circle> 4E2D"},
/* 32A6 */ { "CIRCLED IDEOGRAPH LOW","	# <circle> 4E0B"},
/* 32A7 */ { "CIRCLED IDEOGRAPH LEFT","	# <circle> 5DE6"},
/* 32A8 */ { "CIRCLED IDEOGRAPH RIGHT","	# <circle> 53F3"},
/* 32A9 */ { "CIRCLED IDEOGRAPH MEDICINE","	# <circle> 533B"},
/* 32AA */ { "CIRCLED IDEOGRAPH RELIGION","	# <circle> 5B97"},
/* 32AB */ { "CIRCLED IDEOGRAPH STUDY","	# <circle> 5B66"},
/* 32AC */ { "CIRCLED IDEOGRAPH SUPERVISE","	# <circle> 76E3"},
/* 32AD */ { "CIRCLED IDEOGRAPH ENTERPRISE","	# <circle> 4F01"},
/* 32AE */ { "CIRCLED IDEOGRAPH RESOURCE","	# <circle> 8CC7"},
/* 32AF */ { "CIRCLED IDEOGRAPH ALLIANCE","	# <circle> 5354"},
/* 32B0 */ { "CIRCLED IDEOGRAPH NIGHT","	# <circle> 591C"},
/* 32B1 */ { "CIRCLED NUMBER THIRTY SIX","	# <circle> 0033 0036"},
/* 32B2 */ { "CIRCLED NUMBER THIRTY SEVEN","	# <circle> 0033 0037"},
/* 32B3 */ { "CIRCLED NUMBER THIRTY EIGHT","	# <circle> 0033 0038"},
/* 32B4 */ { "CIRCLED NUMBER THIRTY NINE","	# <circle> 0033 0039"},
/* 32B5 */ { "CIRCLED NUMBER FORTY","	# <circle> 0034 0030"},
/* 32B6 */ { "CIRCLED NUMBER FORTY ONE","	# <circle> 0034 0031"},
/* 32B7 */ { "CIRCLED NUMBER FORTY TWO","	# <circle> 0034 0032"},
/* 32B8 */ { "CIRCLED NUMBER FORTY THREE","	# <circle> 0034 0033"},
/* 32B9 */ { "CIRCLED NUMBER FORTY FOUR","	# <circle> 0034 0034"},
/* 32BA */ { "CIRCLED NUMBER FORTY FIVE","	# <circle> 0034 0035"},
/* 32BB */ { "CIRCLED NUMBER FORTY SIX","	# <circle> 0034 0036"},
/* 32BC */ { "CIRCLED NUMBER FORTY SEVEN","	# <circle> 0034 0037"},
/* 32BD */ { "CIRCLED NUMBER FORTY EIGHT","	# <circle> 0034 0038"},
/* 32BE */ { "CIRCLED NUMBER FORTY NINE","	# <circle> 0034 0039"},
/* 32BF */ { "CIRCLED NUMBER FIFTY","	# <circle> 0035 0030"},
/* 32C0 */ { "IDEOGRAPHIC TELEGRAPH SYMBOL FOR JANUARY","	# 0031 6708"},
/* 32C1 */ { "IDEOGRAPHIC TELEGRAPH SYMBOL FOR FEBRUARY","	# 0032 6708"},
/* 32C2 */ { "IDEOGRAPHIC TELEGRAPH SYMBOL FOR MARCH","	# 0033 6708"},
/* 32C3 */ { "IDEOGRAPHIC TELEGRAPH SYMBOL FOR APRIL","	# 0034 6708"},
/* 32C4 */ { "IDEOGRAPHIC TELEGRAPH SYMBOL FOR MAY","	# 0035 6708"},
/* 32C5 */ { "IDEOGRAPHIC TELEGRAPH SYMBOL FOR JUNE","	# 0036 6708"},
/* 32C6 */ { "IDEOGRAPHIC TELEGRAPH SYMBOL FOR JULY","	# 0037 6708"},
/* 32C7 */ { "IDEOGRAPHIC TELEGRAPH SYMBOL FOR AUGUST","	# 0038 6708"},
/* 32C8 */ { "IDEOGRAPHIC TELEGRAPH SYMBOL FOR SEPTEMBER","	# 0039 6708"},
/* 32C9 */ { "IDEOGRAPHIC TELEGRAPH SYMBOL FOR OCTOBER","	# 0031 0030 6708"},
/* 32CA */ { "IDEOGRAPHIC TELEGRAPH SYMBOL FOR NOVEMBER","	# 0031 0031 6708"},
/* 32CB */ { "IDEOGRAPHIC TELEGRAPH SYMBOL FOR DECEMBER","	# 0031 0032 6708"},
/* 32CC */ { "SQUARE HG","	# <square> 0048 0067"},
/* 32CD */ { "SQUARE ERG","	# <square> 0065 0072 0067"},
/* 32CE */ { "SQUARE EV","	# <square> 0065 0056"},
/* 32CF */ { "LIMITED LIABILITY SIGN","	# <square> 004C 0054 0044"},
/* 32D0 */ { "CIRCLED KATAKANA A","	# <circle> 30A2"},
/* 32D1 */ { "CIRCLED KATAKANA I","	# <circle> 30A4"},
/* 32D2 */ { "CIRCLED KATAKANA U","	# <circle> 30A6"},
/* 32D3 */ { "CIRCLED KATAKANA E","	# <circle> 30A8"},
/* 32D4 */ { "CIRCLED KATAKANA O","	# <circle> 30AA"},
/* 32D5 */ { "CIRCLED KATAKANA KA","	# <circle> 30AB"},
/* 32D6 */ { "CIRCLED KATAKANA KI","	# <circle> 30AD"},
/* 32D7 */ { "CIRCLED KATAKANA KU","	# <circle> 30AF"},
/* 32D8 */ { "CIRCLED KATAKANA KE","	# <circle> 30B1"},
/* 32D9 */ { "CIRCLED KATAKANA KO","	# <circle> 30B3"},
/* 32DA */ { "CIRCLED KATAKANA SA","	# <circle> 30B5"},
/* 32DB */ { "CIRCLED KATAKANA SI","	# <circle> 30B7"},
/* 32DC */ { "CIRCLED KATAKANA SU","	# <circle> 30B9"},
/* 32DD */ { "CIRCLED KATAKANA SE","	# <circle> 30BB"},
/* 32DE */ { "CIRCLED KATAKANA SO","	# <circle> 30BD"},
/* 32DF */ { "CIRCLED KATAKANA TA","	# <circle> 30BF"},
/* 32E0 */ { "CIRCLED KATAKANA TI","	# <circle> 30C1"},
/* 32E1 */ { "CIRCLED KATAKANA TU","	# <circle> 30C4"},
/* 32E2 */ { "CIRCLED KATAKANA TE","	# <circle> 30C6"},
/* 32E3 */ { "CIRCLED KATAKANA TO","	# <circle> 30C8"},
/* 32E4 */ { "CIRCLED KATAKANA NA","	# <circle> 30CA"},
/* 32E5 */ { "CIRCLED KATAKANA NI","	# <circle> 30CB"},
/* 32E6 */ { "CIRCLED KATAKANA NU","	# <circle> 30CC"},
/* 32E7 */ { "CIRCLED KATAKANA NE","	# <circle> 30CD"},
/* 32E8 */ { "CIRCLED KATAKANA NO","	# <circle> 30CE"},
/* 32E9 */ { "CIRCLED KATAKANA HA","	# <circle> 30CF"},
/* 32EA */ { "CIRCLED KATAKANA HI","	# <circle> 30D2"},
/* 32EB */ { "CIRCLED KATAKANA HU","	# <circle> 30D5"},
/* 32EC */ { "CIRCLED KATAKANA HE","	# <circle> 30D8"},
/* 32ED */ { "CIRCLED KATAKANA HO","	# <circle> 30DB"},
/* 32EE */ { "CIRCLED KATAKANA MA","	# <circle> 30DE"},
/* 32EF */ { "CIRCLED KATAKANA MI","	# <circle> 30DF"},
/* 32F0 */ { "CIRCLED KATAKANA MU","	# <circle> 30E0"},
/* 32F1 */ { "CIRCLED KATAKANA ME","	# <circle> 30E1"},
/* 32F2 */ { "CIRCLED KATAKANA MO","	# <circle> 30E2"},
/* 32F3 */ { "CIRCLED KATAKANA YA","	# <circle> 30E4"},
/* 32F4 */ { "CIRCLED KATAKANA YU","	# <circle> 30E6"},
/* 32F5 */ { "CIRCLED KATAKANA YO","	# <circle> 30E8"},
/* 32F6 */ { "CIRCLED KATAKANA RA","	# <circle> 30E9"},
/* 32F7 */ { "CIRCLED KATAKANA RI","	# <circle> 30EA"},
/* 32F8 */ { "CIRCLED KATAKANA RU","	# <circle> 30EB"},
/* 32F9 */ { "CIRCLED KATAKANA RE","	# <circle> 30EC"},
/* 32FA */ { "CIRCLED KATAKANA RO","	# <circle> 30ED"},
/* 32FB */ { "CIRCLED KATAKANA WA","	# <circle> 30EF"},
/* 32FC */ { "CIRCLED KATAKANA WI","	# <circle> 30F0"},
/* 32FD */ { "CIRCLED KATAKANA WE","	# <circle> 30F1"},
/* 32FE */ { "CIRCLED KATAKANA WO","	# <circle> 30F2"},
/* 32FF */ { "SQUARE ERA NAME REIWA","	# <square> 4EE4 548C"}
};

UN_DLL_LOCAL
static const struct unicode_nameannot una_00_33[] = {
/* 3300 */ { "SQUARE APAATO","	* apartment\n"
	"	# <square> 30A2 30D1 30FC 30C8"},
/* 3301 */ { "SQUARE ARUHUA","	* alpha\n"
	"	# <square> 30A2 30EB 30D5 30A1"},
/* 3302 */ { "SQUARE ANPEA","	* ampere\n"
	"	# <square> 30A2 30F3 30DA 30A2"},
/* 3303 */ { "SQUARE AARU","	* are (unit of area)\n"
	"	# <square> 30A2 30FC 30EB"},
/* 3304 */ { "SQUARE ININGU","	* inning\n"
	"	# <square> 30A4 30CB 30F3 30B0"},
/* 3305 */ { "SQUARE INTI","	* inch\n"
	"	# <square> 30A4 30F3 30C1"},
/* 3306 */ { "SQUARE UON","	* won (Korean currency)\n"
	"	# <square> 30A6 30A9 30F3"},
/* 3307 */ { "SQUARE ESUKUUDO","	* escudo (Portuguese currency)\n"
	"	# <square> 30A8 30B9 30AF 30FC 30C9"},
/* 3308 */ { "SQUARE EEKAA","	* acre\n"
	"	# <square> 30A8 30FC 30AB 30FC"},
/* 3309 */ { "SQUARE ONSU","	* ounce\n"
	"	# <square> 30AA 30F3 30B9"},
/* 330A */ { "SQUARE OOMU","	* ohm\n"
	"	# <square> 30AA 30FC 30E0"},
/* 330B */ { "SQUARE KAIRI","	* kai-ri: nautical mile\n"
	"	# <square> 30AB 30A4 30EA"},
/* 330C */ { "SQUARE KARATTO","	* carat\n"
	"	# <square> 30AB 30E9 30C3 30C8"},
/* 330D */ { "SQUARE KARORII","	* calorie\n"
	"	# <square> 30AB 30ED 30EA 30FC"},
/* 330E */ { "SQUARE GARON","	* gallon\n"
	"	# <square> 30AC 30ED 30F3"},
/* 330F */ { "SQUARE GANMA","	* gamma\n"
	"	# <square> 30AC 30F3 30DE"},
/* 3310 */ { "SQUARE GIGA","	* giga-\n"
	"	# <square> 30AE 30AC"},
/* 3311 */ { "SQUARE GINII","	* guinea\n"
	"	# <square> 30AE 30CB 30FC"},
/* 3312 */ { "SQUARE KYURII","	* curie\n"
	"	# <square> 30AD 30E5 30EA 30FC"},
/* 3313 */ { "SQUARE GIRUDAA","	* guilder\n"
	"	# <square> 30AE 30EB 30C0 30FC"},
/* 3314 */ { "SQUARE KIRO","	* kilo-\n"
	"	# <square> 30AD 30ED"},
/* 3315 */ { "SQUARE KIROGURAMU","	* kilogram\n"
	"	# <square> 30AD 30ED 30B0 30E9 30E0"},
/* 3316 */ { "SQUARE KIROMEETORU","	* kilometer\n"
	"	# <square> 30AD 30ED 30E1 30FC 30C8 30EB"},
/* 3317 */ { "SQUARE KIROWATTO","	* kilowatt\n"
	"	# <square> 30AD 30ED 30EF 30C3 30C8"},
/* 3318 */ { "SQUARE GURAMU","	* gram\n"
	"	# <square> 30B0 30E9 30E0"},
/* 3319 */ { "SQUARE GURAMUTON","	* gram ton\n"
	"	# <square> 30B0 30E9 30E0 30C8 30F3"},
/* 331A */ { "SQUARE KURUZEIRO","	* cruzeiro (Brazilian currency)\n"
	"	# <square> 30AF 30EB 30BC 30A4 30ED"},
/* 331B */ { "SQUARE KUROONE","	* krone\n"
	"	# <square> 30AF 30ED 30FC 30CD"},
/* 331C */ { "SQUARE KEESU","	* case\n"
	"	# <square> 30B1 30FC 30B9"},
/* 331D */ { "SQUARE KORUNA","	* koruna (Czech currency)\n"
	"	# <square> 30B3 30EB 30CA"},
/* 331E */ { "SQUARE KOOPO","	* co-op\n"
	"	# <square> 30B3 30FC 30DD"},
/* 331F */ { "SQUARE SAIKURU","	* cycle\n"
	"	# <square> 30B5 30A4 30AF 30EB"},
/* 3320 */ { "SQUARE SANTIIMU","	* centime\n"
	"	# <square> 30B5 30F3 30C1 30FC 30E0"},
/* 3321 */ { "SQUARE SIRINGU","	* shilling\n"
	"	# <square> 30B7 30EA 30F3 30B0"},
/* 3322 */ { "SQUARE SENTI","	* centi-\n"
	"	# <square> 30BB 30F3 30C1"},
/* 3323 */ { "SQUARE SENTO","	* cent\n"
	"	# <square> 30BB 30F3 30C8"},
/* 3324 */ { "SQUARE DAASU","	* dozen\n"
	"	# <square> 30C0 30FC 30B9"},
/* 3325 */ { "SQUARE DESI","	* deci-\n"
	"	# <square> 30C7 30B7"},
/* 3326 */ { "SQUARE DORU","	* dollar\n"
	"	# <square> 30C9 30EB"},
/* 3327 */ { "SQUARE TON","	* ton\n"
	"	# <square> 30C8 30F3"},
/* 3328 */ { "SQUARE NANO","	* nano-\n"
	"	# <square> 30CA 30CE"},
/* 3329 */ { "SQUARE NOTTO","	* knot, nautical mile\n"
	"	# <square> 30CE 30C3 30C8"},
/* 332A */ { "SQUARE HAITU","	* heights\n"
	"	# <square> 30CF 30A4 30C4"},
/* 332B */ { "SQUARE PAASENTO","	* percent\n"
	"	# <square> 30D1 30FC 30BB 30F3 30C8"},
/* 332C */ { "SQUARE PAATU","	* a mistaken, unused representation originally intended for the Thai baht currency sign\n"
	"	x (thai currency symbol baht - 0E3F)\n"
	"	# <square> 30D1 30FC 30C4"},
/* 332D */ { "SQUARE BAARERU","	* barrel\n"
	"	# <square> 30D0 30FC 30EC 30EB"},
/* 332E */ { "SQUARE PIASUTORU","	* piaster\n"
	"	# <square> 30D4 30A2 30B9 30C8 30EB"},
/* 332F */ { "SQUARE PIKURU","	* picul (unit of weight)\n"
	"	# <square> 30D4 30AF 30EB"},
/* 3330 */ { "SQUARE PIKO","	* pico-\n"
	"	# <square> 30D4 30B3"},
/* 3331 */ { "SQUARE BIRU","	* building\n"
	"	# <square> 30D3 30EB"},
/* 3332 */ { "SQUARE HUARADDO","	* farad\n"
	"	# <square> 30D5 30A1 30E9 30C3 30C9"},
/* 3333 */ { "SQUARE HUIITO","	* feet\n"
	"	# <square> 30D5 30A3 30FC 30C8"},
/* 3334 */ { "SQUARE BUSSYERU","	* bushel\n"
	"	# <square> 30D6 30C3 30B7 30A7 30EB"},
/* 3335 */ { "SQUARE HURAN","	* franc\n"
	"	# <square> 30D5 30E9 30F3"},
/* 3336 */ { "SQUARE HEKUTAARU","	* hectare\n"
	"	# <square> 30D8 30AF 30BF 30FC 30EB"},
/* 3337 */ { "SQUARE PESO","	* peso\n"
	"	# <square> 30DA 30BD"},
/* 3338 */ { "SQUARE PENIHI","	* pfennig\n"
	"	# <square> 30DA 30CB 30D2"},
/* 3339 */ { "SQUARE HERUTU","	* hertz\n"
	"	# <square> 30D8 30EB 30C4"},
/* 333A */ { "SQUARE PENSU","	* pence\n"
	"	# <square> 30DA 30F3 30B9"},
/* 333B */ { "SQUARE PEEZI","	* page\n"
	"	# <square> 30DA 30FC 30B8"},
/* 333C */ { "SQUARE BEETA","	* beta\n"
	"	# <square> 30D9 30FC 30BF"},
/* 333D */ { "SQUARE POINTO","	* point\n"
	"	# <square> 30DD 30A4 30F3 30C8"},
/* 333E */ { "SQUARE BORUTO","	* volt, bolt\n"
	"	# <square> 30DC 30EB 30C8"},
/* 333F */ { "SQUARE HON","	* hon: volume\n"
	"	# <square> 30DB 30F3"},
/* 3340 */ { "SQUARE PONDO","	* pound\n"
	"	# <square> 30DD 30F3 30C9"},
/* 3341 */ { "SQUARE HOORU","	* hall\n"
	"	# <square> 30DB 30FC 30EB"},
/* 3342 */ { "SQUARE HOON","	* horn\n"
	"	# <square> 30DB 30FC 30F3"},
/* 3343 */ { "SQUARE MAIKURO","	* micro-\n"
	"	# <square> 30DE 30A4 30AF 30ED"},
/* 3344 */ { "SQUARE MAIRU","	* mile\n"
	"	# <square> 30DE 30A4 30EB"},
/* 3345 */ { "SQUARE MAHHA","	* mach\n"
	"	# <square> 30DE 30C3 30CF"},
/* 3346 */ { "SQUARE MARUKU","	* mark\n"
	"	# <square> 30DE 30EB 30AF"},
/* 3347 */ { "SQUARE MANSYON","	* mansion (i.e. better quality apartment)\n"
	"	# <square> 30DE 30F3 30B7 30E7 30F3"},
/* 3348 */ { "SQUARE MIKURON","	* micron\n"
	"	# <square> 30DF 30AF 30ED 30F3"},
/* 3349 */ { "SQUARE MIRI","	* milli-\n"
	"	# <square> 30DF 30EA"},
/* 334A */ { "SQUARE MIRIBAARU","	* millibar\n"
	"	# <square> 30DF 30EA 30D0 30FC 30EB"},
/* 334B */ { "SQUARE MEGA","	* mega-\n"
	"	# <square> 30E1 30AC"},
/* 334C */ { "SQUARE MEGATON","	* megaton\n"
	"	# <square> 30E1 30AC 30C8 30F3"},
/* 334D */ { "SQUARE MEETORU","	* meter\n"
	"	# <square> 30E1 30FC 30C8 30EB"},
/* 334E */ { "SQUARE YAADO","	* yard\n"
	"	# <square> 30E4 30FC 30C9"},
/* 334F */ { "SQUARE YAARU","	* yard\n"
	"	# <square> 30E4 30FC 30EB"},
/* 3350 */ { "SQUARE YUAN","	* yuan (Chinese currency)\n"
	"	# <square> 30E6 30A2 30F3"},
/* 3351 */ { "SQUARE RITTORU","	* liter\n"
	"	# <square> 30EA 30C3 30C8 30EB"},
/* 3352 */ { "SQUARE RIRA","	* lira\n"
	"	# <square> 30EA 30E9"},
/* 3353 */ { "SQUARE RUPII","	* rupee\n"
	"	# <square> 30EB 30D4 30FC"},
/* 3354 */ { "SQUARE RUUBURU","	* ruble\n"
	"	# <square> 30EB 30FC 30D6 30EB"},
/* 3355 */ { "SQUARE REMU","	* rem (unit of radiation)\n"
	"	# <square> 30EC 30E0"},
/* 3356 */ { "SQUARE RENTOGEN","	* roentgen\n"
	"	# <square> 30EC 30F3 30C8 30B2 30F3"},
/* 3357 */ { "SQUARE WATTO","	* watt\n"
	"	# <square> 30EF 30C3 30C8"},
/* 3358 */ { "IDEOGRAPHIC TELEGRAPH SYMBOL FOR HOUR ZERO","	# 0030 70B9"},
/* 3359 */ { "IDEOGRAPHIC TELEGRAPH SYMBOL FOR HOUR ONE","	# 0031 70B9"},
/* 335A */ { "IDEOGRAPHIC TELEGRAPH SYMBOL FOR HOUR TWO","	# 0032 70B9"},
/* 335B */ { "IDEOGRAPHIC TELEGRAPH SYMBOL FOR HOUR THREE","	# 0033 70B9"},
/* 335C */ { "IDEOGRAPHIC TELEGRAPH SYMBOL FOR HOUR FOUR","	# 0034 70B9"},
/* 335D */ { "IDEOGRAPHIC TELEGRAPH SYMBOL FOR HOUR FIVE","	# 0035 70B9"},
/* 335E */ { "IDEOGRAPHIC TELEGRAPH SYMBOL FOR HOUR SIX","	# 0036 70B9"},
/* 335F */ { "IDEOGRAPHIC TELEGRAPH SYMBOL FOR HOUR SEVEN","	# 0037 70B9"},
/* 3360 */ { "IDEOGRAPHIC TELEGRAPH SYMBOL FOR HOUR EIGHT","	# 0038 70B9"},
/* 3361 */ { "IDEOGRAPHIC TELEGRAPH SYMBOL FOR HOUR NINE","	# 0039 70B9"},
/* 3362 */ { "IDEOGRAPHIC TELEGRAPH SYMBOL FOR HOUR TEN","	# 0031 0030 70B9"},
/* 3363 */ { "IDEOGRAPHIC TELEGRAPH SYMBOL FOR HOUR ELEVEN","	# 0031 0031 70B9"},
/* 3364 */ { "IDEOGRAPHIC TELEGRAPH SYMBOL FOR HOUR TWELVE","	# 0031 0032 70B9"},
/* 3365 */ { "IDEOGRAPHIC TELEGRAPH SYMBOL FOR HOUR THIRTEEN","	# 0031 0033 70B9"},
/* 3366 */ { "IDEOGRAPHIC TELEGRAPH SYMBOL FOR HOUR FOURTEEN","	# 0031 0034 70B9"},
/* 3367 */ { "IDEOGRAPHIC TELEGRAPH SYMBOL FOR HOUR FIFTEEN","	# 0031 0035 70B9"},
/* 3368 */ { "IDEOGRAPHIC TELEGRAPH SYMBOL FOR HOUR SIXTEEN","	# 0031 0036 70B9"},
/* 3369 */ { "IDEOGRAPHIC TELEGRAPH SYMBOL FOR HOUR SEVENTEEN","	# 0031 0037 70B9"},
/* 336A */ { "IDEOGRAPHIC TELEGRAPH SYMBOL FOR HOUR EIGHTEEN","	# 0031 0038 70B9"},
/* 336B */ { "IDEOGRAPHIC TELEGRAPH SYMBOL FOR HOUR NINETEEN","	# 0031 0039 70B9"},
/* 336C */ { "IDEOGRAPHIC TELEGRAPH SYMBOL FOR HOUR TWENTY","	# 0032 0030 70B9"},
/* 336D */ { "IDEOGRAPHIC TELEGRAPH SYMBOL FOR HOUR TWENTY-ONE","	# 0032 0031 70B9"},
/* 336E */ { "IDEOGRAPHIC TELEGRAPH SYMBOL FOR HOUR TWENTY-TWO","	# 0032 0032 70B9"},
/* 336F */ { "IDEOGRAPHIC TELEGRAPH SYMBOL FOR HOUR TWENTY-THREE","	# 0032 0033 70B9"},
/* 3370 */ { "IDEOGRAPHIC TELEGRAPH SYMBOL FOR HOUR TWENTY-FOUR","	# 0032 0034 70B9"},
/* 3371 */ { "SQUARE HPA","	# <square> 0068 0050 0061"},
/* 3372 */ { "SQUARE DA","	# <square> 0064 0061"},
/* 3373 */ { "SQUARE AU","	# <square> 0041 0055"},
/* 3374 */ { "SQUARE BAR","	# <square> 0062 0061 0072"},
/* 3375 */ { "SQUARE OV","	# <square> 006F 0056"},
/* 3376 */ { "SQUARE PC","	# <square> 0070 0063"},
/* 3377 */ { "SQUARE DM","	# <square> 0064 006D"},
/* 3378 */ { "SQUARE DM SQUARED","	# <square> 0064 006D 00B2"},
/* 3379 */ { "SQUARE DM CUBED","	# <square> 0064 006D 00B3"},
/* 337A */ { "SQUARE IU","	# <square> 0049 0055"},
/* 337B */ { "SQUARE ERA NAME HEISEI","	# <square> 5E73 6210"},
/* 337C */ { "SQUARE ERA NAME SYOUWA","	# <square> 662D 548C"},
/* 337D */ { "SQUARE ERA NAME TAISYOU","	# <square> 5927 6B63"},
/* 337E */ { "SQUARE ERA NAME MEIZI","	# <square> 660E 6CBB"},
/* 337F */ { "SQUARE CORPORATION","	= kabusiki-gaisya\n"
	"	* incorporated\n"
	"	# <square> 682A 5F0F 4F1A 793E"},
/* 3380 */ { "SQUARE PA AMPS","	# <square> 0070 0041"},
/* 3381 */ { "SQUARE NA","	# <square> 006E 0041"},
/* 3382 */ { "SQUARE MU A","	# <square> 03BC 0041"},
/* 3383 */ { "SQUARE MA","	# <square> 006D 0041"},
/* 3384 */ { "SQUARE KA","	# <square> 006B 0041"},
/* 3385 */ { "SQUARE KB","	# <square> 004B 0042"},
/* 3386 */ { "SQUARE MB","	# <square> 004D 0042"},
/* 3387 */ { "SQUARE GB","	# <square> 0047 0042"},
/* 3388 */ { "SQUARE CAL","	# <square> 0063 0061 006C"},
/* 3389 */ { "SQUARE KCAL","	# <square> 006B 0063 0061 006C"},
/* 338A */ { "SQUARE PF","	# <square> 0070 0046"},
/* 338B */ { "SQUARE NF","	# <square> 006E 0046"},
/* 338C */ { "SQUARE MU F","	# <square> 03BC 0046"},
/* 338D */ { "SQUARE MU G","	# <square> 03BC 0067"},
/* 338E */ { "SQUARE MG","	# <square> 006D 0067"},
/* 338F */ { "SQUARE KG","	# <square> 006B 0067"},
/* 3390 */ { "SQUARE HZ","	# <square> 0048 007A"},
/* 3391 */ { "SQUARE KHZ","	# <square> 006B 0048 007A"},
/* 3392 */ { "SQUARE MHZ","	# <square> 004D 0048 007A"},
/* 3393 */ { "SQUARE GHZ","	# <square> 0047 0048 007A"},
/* 3394 */ { "SQUARE THZ","	# <square> 0054 0048 007A"},
/* 3395 */ { "SQUARE MU L","	# <square> 03BC 2113"},
/* 3396 */ { "SQUARE ML","	# <square> 006D 2113"},
/* 3397 */ { "SQUARE DL","	# <square> 0064 2113"},
/* 3398 */ { "SQUARE KL","	# <square> 006B 2113"},
/* 3399 */ { "SQUARE FM","	# <square> 0066 006D"},
/* 339A */ { "SQUARE NM","	# <square> 006E 006D"},
/* 339B */ { "SQUARE MU M","	# <square> 03BC 006D"},
/* 339C */ { "SQUARE MM","	# <square> 006D 006D"},
/* 339D */ { "SQUARE CM","	# <square> 0063 006D"},
/* 339E */ { "SQUARE KM","	# <square> 006B 006D"},
/* 339F */ { "SQUARE MM SQUARED","	# <square> 006D 006D 00B2"},
/* 33A0 */ { "SQUARE CM SQUARED","	# <square> 0063 006D 00B2"},
/* 33A1 */ { "SQUARE M SQUARED","	# <square> 006D 00B2"},
/* 33A2 */ { "SQUARE KM SQUARED","	# <square> 006B 006D 00B2"},
/* 33A3 */ { "SQUARE MM CUBED","	# <square> 006D 006D 00B3"},
/* 33A4 */ { "SQUARE CM CUBED","	# <square> 0063 006D 00B3"},
/* 33A5 */ { "SQUARE M CUBED","	# <square> 006D 00B3"},
/* 33A6 */ { "SQUARE KM CUBED","	# <square> 006B 006D 00B3"},
/* 33A7 */ { "SQUARE M OVER S","	# <square> 006D 2215 0073"},
/* 33A8 */ { "SQUARE M OVER S SQUARED","	# <square> 006D 2215 0073 00B2"},
/* 33A9 */ { "SQUARE PA","	# <square> 0050 0061"},
/* 33AA */ { "SQUARE KPA","	# <square> 006B 0050 0061"},
/* 33AB */ { "SQUARE MPA","	# <square> 004D 0050 0061"},
/* 33AC */ { "SQUARE GPA","	# <square> 0047 0050 0061"},
/* 33AD */ { "SQUARE RAD","	# <square> 0072 0061 0064"},
/* 33AE */ { "SQUARE RAD OVER S","	# <square> 0072 0061 0064 2215 0073"},
/* 33AF */ { "SQUARE RAD OVER S SQUARED","	# <square> 0072 0061 0064 2215 0073 00B2"},
/* 33B0 */ { "SQUARE PS","	# <square> 0070 0073"},
/* 33B1 */ { "SQUARE NS","	# <square> 006E 0073"},
/* 33B2 */ { "SQUARE MU S","	# <square> 03BC 0073"},
/* 33B3 */ { "SQUARE MS","	# <square> 006D 0073"},
/* 33B4 */ { "SQUARE PV","	# <square> 0070 0056"},
/* 33B5 */ { "SQUARE NV","	# <square> 006E 0056"},
/* 33B6 */ { "SQUARE MU V","	# <square> 03BC 0056"},
/* 33B7 */ { "SQUARE MV","	# <square> 006D 0056"},
/* 33B8 */ { "SQUARE KV","	# <square> 006B 0056"},
/* 33B9 */ { "SQUARE MV MEGA","	# <square> 004D 0056"},
/* 33BA */ { "SQUARE PW","	# <square> 0070 0057"},
/* 33BB */ { "SQUARE NW","	# <square> 006E 0057"},
/* 33BC */ { "SQUARE MU W","	# <square> 03BC 0057"},
/* 33BD */ { "SQUARE MW","	# <square> 006D 0057"},
/* 33BE */ { "SQUARE KW","	# <square> 006B 0057"},
/* 33BF */ { "SQUARE MW MEGA","	# <square> 004D 0057"},
/* 33C0 */ { "SQUARE K OHM","	# <square> 006B 03A9"},
/* 33C1 */ { "SQUARE M OHM","	# <square> 004D 03A9"},
/* 33C2 */ { "SQUARE AM","	# <square> 0061 002E 006D 002E"},
/* 33C3 */ { "SQUARE BQ","	# <square> 0042 0071"},
/* 33C4 */ { "SQUARE CC","	# <square> 0063 0063"},
/* 33C5 */ { "SQUARE CD","	x (circled cd - 1F12D)\n"
	"	# <square> 0063 0064"},
/* 33C6 */ { "SQUARE C OVER KG","	# <square> 0043 2215 006B 0067"},
/* 33C7 */ { "SQUARE CO","	# <square> 0043 006F 002E"},
/* 33C8 */ { "SQUARE DB","	# <square> 0064 0042"},
/* 33C9 */ { "SQUARE GY","	# <square> 0047 0079"},
/* 33CA */ { "SQUARE HA","	# <square> 0068 0061"},
/* 33CB */ { "SQUARE HP","	# <square> 0048 0050"},
/* 33CC */ { "SQUARE IN","	# <square> 0069 006E"},
/* 33CD */ { "SQUARE KK","	# <square> 004B 004B"},
/* 33CE */ { "SQUARE KM CAPITAL","	# <square> 004B 004D"},
/* 33CF */ { "SQUARE KT","	# <square> 006B 0074"},
/* 33D0 */ { "SQUARE LM","	# <square> 006C 006D"},
/* 33D1 */ { "SQUARE LN","	# <square> 006C 006E"},
/* 33D2 */ { "SQUARE LOG","	# <square> 006C 006F 0067"},
/* 33D3 */ { "SQUARE LX","	# <square> 006C 0078"},
/* 33D4 */ { "SQUARE MB SMALL","	# <square> 006D 0062"},
/* 33D5 */ { "SQUARE MIL","	# <square> 006D 0069 006C"},
/* 33D6 */ { "SQUARE MOL","	# <square> 006D 006F 006C"},
/* 33D7 */ { "SQUARE PH","	* casing inconsistency in the compatibility decomposition is a known anomaly dating from glyph presentations in very early versions of the standard\n"
	"	# <square> 0050 0048"},
/* 33D8 */ { "SQUARE PM","	# <square> 0070 002E 006D 002E"},
/* 33D9 */ { "SQUARE PPM","	# <square> 0050 0050 004D"},
/* 33DA */ { "SQUARE PR","	# <square> 0050 0052"},
/* 33DB */ { "SQUARE SR","	# <square> 0073 0072"},
/* 33DC */ { "SQUARE SV","	# <square> 0053 0076"},
/* 33DD */ { "SQUARE WB","	# <square> 0057 0062"},
/* 33DE */ { "SQUARE V OVER M","	# <square> 0056 2215 006D"},
/* 33DF */ { "SQUARE A OVER M","	# <square> 0041 2215 006D"},
/* 33E0 */ { "IDEOGRAPHIC TELEGRAPH SYMBOL FOR DAY ONE","	# 0031 65E5"},
/* 33E1 */ { "IDEOGRAPHIC TELEGRAPH SYMBOL FOR DAY TWO","	# 0032 65E5"},
/* 33E2 */ { "IDEOGRAPHIC TELEGRAPH SYMBOL FOR DAY THREE","	# 0033 65E5"},
/* 33E3 */ { "IDEOGRAPHIC TELEGRAPH SYMBOL FOR DAY FOUR","	# 0034 65E5"},
/* 33E4 */ { "IDEOGRAPHIC TELEGRAPH SYMBOL FOR DAY FIVE","	# 0035 65E5"},
/* 33E5 */ { "IDEOGRAPHIC TELEGRAPH SYMBOL FOR DAY SIX","	# 0036 65E5"},
/* 33E6 */ { "IDEOGRAPHIC TELEGRAPH SYMBOL FOR DAY SEVEN","	# 0037 65E5"},
/* 33E7 */ { "IDEOGRAPHIC TELEGRAPH SYMBOL FOR DAY EIGHT","	# 0038 65E5"},
/* 33E8 */ { "IDEOGRAPHIC TELEGRAPH SYMBOL FOR DAY NINE","	# 0039 65E5"},
/* 33E9 */ { "IDEOGRAPHIC TELEGRAPH SYMBOL FOR DAY TEN","	# 0031 0030 65E5"},
/* 33EA */ { "IDEOGRAPHIC TELEGRAPH SYMBOL FOR DAY ELEVEN","	# 0031 0031 65E5"},
/* 33EB */ { "IDEOGRAPHIC TELEGRAPH SYMBOL FOR DAY TWELVE","	# 0031 0032 65E5"},
/* 33EC */ { "IDEOGRAPHIC TELEGRAPH SYMBOL FOR DAY THIRTEEN","	# 0031 0033 65E5"},
/* 33ED */ { "IDEOGRAPHIC TELEGRAPH SYMBOL FOR DAY FOURTEEN","	# 0031 0034 65E5"},
/* 33EE */ { "IDEOGRAPHIC TELEGRAPH SYMBOL FOR DAY FIFTEEN","	# 0031 0035 65E5"},
/* 33EF */ { "IDEOGRAPHIC TELEGRAPH SYMBOL FOR DAY SIXTEEN","	# 0031 0036 65E5"},
/* 33F0 */ { "IDEOGRAPHIC TELEGRAPH SYMBOL FOR DAY SEVENTEEN","	# 0031 0037 65E5"},
/* 33F1 */ { "IDEOGRAPHIC TELEGRAPH SYMBOL FOR DAY EIGHTEEN","	# 0031 0038 65E5"},
/* 33F2 */ { "IDEOGRAPHIC TELEGRAPH SYMBOL FOR DAY NINETEEN","	# 0031 0039 65E5"},
/* 33F3 */ { "IDEOGRAPHIC TELEGRAPH SYMBOL FOR DAY TWENTY","	# 0032 0030 65E5"},
/* 33F4 */ { "IDEOGRAPHIC TELEGRAPH SYMBOL FOR DAY TWENTY-ONE","	# 0032 0031 65E5"},
/* 33F5 */ { "IDEOGRAPHIC TELEGRAPH SYMBOL FOR DAY TWENTY-TWO","	# 0032 0032 65E5"},
/* 33F6 */ { "IDEOGRAPHIC TELEGRAPH SYMBOL FOR DAY TWENTY-THREE","	# 0032 0033 65E5"},
/* 33F7 */ { "IDEOGRAPHIC TELEGRAPH SYMBOL FOR DAY TWENTY-FOUR","	# 0032 0034 65E5"},
/* 33F8 */ { "IDEOGRAPHIC TELEGRAPH SYMBOL FOR DAY TWENTY-FIVE","	# 0032 0035 65E5"},
/* 33F9 */ { "IDEOGRAPHIC TELEGRAPH SYMBOL FOR DAY TWENTY-SIX","	# 0032 0036 65E5"},
/* 33FA */ { "IDEOGRAPHIC TELEGRAPH SYMBOL FOR DAY TWENTY-SEVEN","	# 0032 0037 65E5"},
/* 33FB */ { "IDEOGRAPHIC TELEGRAPH SYMBOL FOR DAY TWENTY-EIGHT","	# 0032 0038 65E5"},
/* 33FC */ { "IDEOGRAPHIC TELEGRAPH SYMBOL FOR DAY TWENTY-NINE","	# 0032 0039 65E5"},
/* 33FD */ { "IDEOGRAPHIC TELEGRAPH SYMBOL FOR DAY THIRTY","	# 0033 0030 65E5"},
/* 33FE */ { "IDEOGRAPHIC TELEGRAPH SYMBOL FOR DAY THIRTY-ONE","	# 0033 0031 65E5"},
/* 33FF */ { "SQUARE GAL","	# <square> 0067 0061 006C"}
};

UN_DLL_LOCAL
static const struct unicode_nameannot una_00_4D[] = {
/* 4D00 */ { NULL,NULL},
/* 4D01 */ { NULL,NULL},
/* 4D02 */ { NULL,NULL},
/* 4D03 */ { NULL,NULL},
/* 4D04 */ { NULL,NULL},
/* 4D05 */ { NULL,NULL},
/* 4D06 */ { NULL,NULL},
/* 4D07 */ { NULL,NULL},
/* 4D08 */ { NULL,NULL},
/* 4D09 */ { NULL,NULL},
/* 4D0A */ { NULL,NULL},
/* 4D0B */ { NULL,NULL},
/* 4D0C */ { NULL,NULL},
/* 4D0D */ { NULL,NULL},
/* 4D0E */ { NULL,NULL},
/* 4D0F */ { NULL,NULL},
/* 4D10 */ { NULL,NULL},
/* 4D11 */ { NULL,NULL},
/* 4D12 */ { NULL,NULL},
/* 4D13 */ { NULL,NULL},
/* 4D14 */ { NULL,NULL},
/* 4D15 */ { NULL,NULL},
/* 4D16 */ { NULL,NULL},
/* 4D17 */ { NULL,NULL},
/* 4D18 */ { NULL,NULL},
/* 4D19 */ { NULL,NULL},
/* 4D1A */ { NULL,NULL},
/* 4D1B */ { NULL,NULL},
/* 4D1C */ { NULL,NULL},
/* 4D1D */ { NULL,NULL},
/* 4D1E */ { NULL,NULL},
/* 4D1F */ { NULL,NULL},
/* 4D20 */ { NULL,NULL},
/* 4D21 */ { NULL,NULL},
/* 4D22 */ { NULL,NULL},
/* 4D23 */ { NULL,NULL},
/* 4D24 */ { NULL,NULL},
/* 4D25 */ { NULL,NULL},
/* 4D26 */ { NULL,NULL},
/* 4D27 */ { NULL,NULL},
/* 4D28 */ { NULL,NULL},
/* 4D29 */ { NULL,NULL},
/* 4D2A */ { NULL,NULL},
/* 4D2B */ { NULL,NULL},
/* 4D2C */ { NULL,NULL},
/* 4D2D */ { NULL,NULL},
/* 4D2E */ { NULL,NULL},
/* 4D2F */ { NULL,NULL},
/* 4D30 */ { NULL,NULL},
/* 4D31 */ { NULL,NULL},
/* 4D32 */ { NULL,NULL},
/* 4D33 */ { NULL,NULL},
/* 4D34 */ { NULL,NULL},
/* 4D35 */ { NULL,NULL},
/* 4D36 */ { NULL,NULL},
/* 4D37 */ { NULL,NULL},
/* 4D38 */ { NULL,NULL},
/* 4D39 */ { NULL,NULL},
/* 4D3A */ { NULL,NULL},
/* 4D3B */ { NULL,NULL},
/* 4D3C */ { NULL,NULL},
/* 4D3D */ { NULL,NULL},
/* 4D3E */ { NULL,NULL},
/* 4D3F */ { NULL,NULL},
/* 4D40 */ { NULL,NULL},
/* 4D41 */ { NULL,NULL},
/* 4D42 */ { NULL,NULL},
/* 4D43 */ { NULL,NULL},
/* 4D44 */ { NULL,NULL},
/* 4D45 */ { NULL,NULL},
/* 4D46 */ { NULL,NULL},
/* 4D47 */ { NULL,NULL},
/* 4D48 */ { NULL,NULL},
/* 4D49 */ { NULL,NULL},
/* 4D4A */ { NULL,NULL},
/* 4D4B */ { NULL,NULL},
/* 4D4C */ { NULL,NULL},
/* 4D4D */ { NULL,NULL},
/* 4D4E */ { NULL,NULL},
/* 4D4F */ { NULL,NULL},
/* 4D50 */ { NULL,NULL},
/* 4D51 */ { NULL,NULL},
/* 4D52 */ { NULL,NULL},
/* 4D53 */ { NULL,NULL},
/* 4D54 */ { NULL,NULL},
/* 4D55 */ { NULL,NULL},
/* 4D56 */ { NULL,NULL},
/* 4D57 */ { NULL,NULL},
/* 4D58 */ { NULL,NULL},
/* 4D59 */ { NULL,NULL},
/* 4D5A */ { NULL,NULL},
/* 4D5B */ { NULL,NULL},
/* 4D5C */ { NULL,NULL},
/* 4D5D */ { NULL,NULL},
/* 4D5E */ { NULL,NULL},
/* 4D5F */ { NULL,NULL},
/* 4D60 */ { NULL,NULL},
/* 4D61 */ { NULL,NULL},
/* 4D62 */ { NULL,NULL},
/* 4D63 */ { NULL,NULL},
/* 4D64 */ { NULL,NULL},
/* 4D65 */ { NULL,NULL},
/* 4D66 */ { NULL,NULL},
/* 4D67 */ { NULL,NULL},
/* 4D68 */ { NULL,NULL},
/* 4D69 */ { NULL,NULL},
/* 4D6A */ { NULL,NULL},
/* 4D6B */ { NULL,NULL},
/* 4D6C */ { NULL,NULL},
/* 4D6D */ { NULL,NULL},
/* 4D6E */ { NULL,NULL},
/* 4D6F */ { NULL,NULL},
/* 4D70 */ { NULL,NULL},
/* 4D71 */ { NULL,NULL},
/* 4D72 */ { NULL,NULL},
/* 4D73 */ { NULL,NULL},
/* 4D74 */ { NULL,NULL},
/* 4D75 */ { NULL,NULL},
/* 4D76 */ { NULL,NULL},
/* 4D77 */ { NULL,NULL},
/* 4D78 */ { NULL,NULL},
/* 4D79 */ { NULL,NULL},
/* 4D7A */ { NULL,NULL},
/* 4D7B */ { NULL,NULL},
/* 4D7C */ { NULL,NULL},
/* 4D7D */ { NULL,NULL},
/* 4D7E */ { NULL,NULL},
/* 4D7F */ { NULL,NULL},
/* 4D80 */ { NULL,NULL},
/* 4D81 */ { NULL,NULL},
/* 4D82 */ { NULL,NULL},
/* 4D83 */ { NULL,NULL},
/* 4D84 */ { NULL,NULL},
/* 4D85 */ { NULL,NULL},
/* 4D86 */ { NULL,NULL},
/* 4D87 */ { NULL,NULL},
/* 4D88 */ { NULL,NULL},
/* 4D89 */ { NULL,NULL},
/* 4D8A */ { NULL,NULL},
/* 4D8B */ { NULL,NULL},
/* 4D8C */ { NULL,NULL},
/* 4D8D */ { NULL,NULL},
/* 4D8E */ { NULL,NULL},
/* 4D8F */ { NULL,NULL},
/* 4D90 */ { NULL,NULL},
/* 4D91 */ { NULL,NULL},
/* 4D92 */ { NULL,NULL},
/* 4D93 */ { NULL,NULL},
/* 4D94 */ { NULL,NULL},
/* 4D95 */ { NULL,NULL},
/* 4D96 */ { NULL,NULL},
/* 4D97 */ { NULL,NULL},
/* 4D98 */ { NULL,NULL},
/* 4D99 */ { NULL,NULL},
/* 4D9A */ { NULL,NULL},
/* 4D9B */ { NULL,NULL},
/* 4D9C */ { NULL,NULL},
/* 4D9D */ { NULL,NULL},
/* 4D9E */ { NULL,NULL},
/* 4D9F */ { NULL,NULL},
/* 4DA0 */ { NULL,NULL},
/* 4DA1 */ { NULL,NULL},
/* 4DA2 */ { NULL,NULL},
/* 4DA3 */ { NULL,NULL},
/* 4DA4 */ { NULL,NULL},
/* 4DA5 */ { NULL,NULL},
/* 4DA6 */ { NULL,NULL},
/* 4DA7 */ { NULL,NULL},
/* 4DA8 */ { NULL,NULL},
/* 4DA9 */ { NULL,NULL},
/* 4DAA */ { NULL,NULL},
/* 4DAB */ { NULL,NULL},
/* 4DAC */ { NULL,NULL},
/* 4DAD */ { NULL,NULL},
/* 4DAE */ { NULL,NULL},
/* 4DAF */ { NULL,NULL},
/* 4DB0 */ { NULL,NULL},
/* 4DB1 */ { NULL,NULL},
/* 4DB2 */ { NULL,NULL},
/* 4DB3 */ { NULL,NULL},
/* 4DB4 */ { NULL,NULL},
/* 4DB5 */ { NULL,NULL},
/* 4DB6 */ { NULL,NULL},
/* 4DB7 */ { NULL,NULL},
/* 4DB8 */ { NULL,NULL},
/* 4DB9 */ { NULL,NULL},
/* 4DBA */ { NULL,NULL},
/* 4DBB */ { NULL,NULL},
/* 4DBC */ { NULL,NULL},
/* 4DBD */ { NULL,NULL},
/* 4DBE */ { NULL,NULL},
/* 4DBF */ { NULL,NULL},
/* 4DC0 */ { "HEXAGRAM FOR THE CREATIVE HEAVEN",NULL},
/* 4DC1 */ { "HEXAGRAM FOR THE RECEPTIVE EARTH",NULL},
/* 4DC2 */ { "HEXAGRAM FOR DIFFICULTY AT THE BEGINNING",NULL},
/* 4DC3 */ { "HEXAGRAM FOR YOUTHFUL FOLLY",NULL},
/* 4DC4 */ { "HEXAGRAM FOR WAITING",NULL},
/* 4DC5 */ { "HEXAGRAM FOR CONFLICT",NULL},
/* 4DC6 */ { "HEXAGRAM FOR THE ARMY",NULL},
/* 4DC7 */ { "HEXAGRAM FOR HOLDING TOGETHER",NULL},
/* 4DC8 */ { "HEXAGRAM FOR SMALL TAMING",NULL},
/* 4DC9 */ { "HEXAGRAM FOR TREADING",NULL},
/* 4DCA */ { "HEXAGRAM FOR PEACE",NULL},
/* 4DCB */ { "HEXAGRAM FOR STANDSTILL",NULL},
/* 4DCC */ { "HEXAGRAM FOR FELLOWSHIP",NULL},
/* 4DCD */ { "HEXAGRAM FOR GREAT POSSESSION",NULL},
/* 4DCE */ { "HEXAGRAM FOR MODESTY",NULL},
/* 4DCF */ { "HEXAGRAM FOR ENTHUSIASM",NULL},
/* 4DD0 */ { "HEXAGRAM FOR FOLLOWING",NULL},
/* 4DD1 */ { "HEXAGRAM FOR WORK ON THE DECAYED",NULL},
/* 4DD2 */ { "HEXAGRAM FOR APPROACH",NULL},
/* 4DD3 */ { "HEXAGRAM FOR CONTEMPLATION",NULL},
/* 4DD4 */ { "HEXAGRAM FOR BITING THROUGH",NULL},
/* 4DD5 */ { "HEXAGRAM FOR GRACE",NULL},
/* 4DD6 */ { "HEXAGRAM FOR SPLITTING APART",NULL},
/* 4DD7 */ { "HEXAGRAM FOR RETURN",NULL},
/* 4DD8 */ { "HEXAGRAM FOR INNOCENCE",NULL},
/* 4DD9 */ { "HEXAGRAM FOR GREAT TAMING",NULL},
/* 4DDA */ { "HEXAGRAM FOR MOUTH CORNERS",NULL},
/* 4DDB */ { "HEXAGRAM FOR GREAT PREPONDERANCE",NULL},
/* 4DDC */ { "HEXAGRAM FOR THE ABYSMAL WATER",NULL},
/* 4DDD */ { "HEXAGRAM FOR THE CLINGING FIRE",NULL},
/* 4DDE */ { "HEXAGRAM FOR INFLUENCE",NULL},
/* 4DDF */ { "HEXAGRAM FOR DURATION",NULL},
/* 4DE0 */ { "HEXAGRAM FOR RETREAT",NULL},
/* 4DE1 */ { "HEXAGRAM FOR GREAT POWER",NULL},
/* 4DE2 */ { "HEXAGRAM FOR PROGRESS",NULL},
/* 4DE3 */ { "HEXAGRAM FOR DARKENING OF THE LIGHT",NULL},
/* 4DE4 */ { "HEXAGRAM FOR THE FAMILY",NULL},
/* 4DE5 */ { "HEXAGRAM FOR OPPOSITION",NULL},
/* 4DE6 */ { "HEXAGRAM FOR OBSTRUCTION",NULL},
/* 4DE7 */ { "HEXAGRAM FOR DELIVERANCE",NULL},
/* 4DE8 */ { "HEXAGRAM FOR DECREASE",NULL},
/* 4DE9 */ { "HEXAGRAM FOR INCREASE",NULL},
/* 4DEA */ { "HEXAGRAM FOR BREAKTHROUGH",NULL},
/* 4DEB */ { "HEXAGRAM FOR COMING TO MEET",NULL},
/* 4DEC */ { "HEXAGRAM FOR GATHERING TOGETHER",NULL},
/* 4DED */ { "HEXAGRAM FOR PUSHING UPWARD",NULL},
/* 4DEE */ { "HEXAGRAM FOR OPPRESSION",NULL},
/* 4DEF */ { "HEXAGRAM FOR THE WELL",NULL},
/* 4DF0 */ { "HEXAGRAM FOR REVOLUTION",NULL},
/* 4DF1 */ { "HEXAGRAM FOR THE CAULDRON",NULL},
/* 4DF2 */ { "HEXAGRAM FOR THE AROUSING THUNDER",NULL},
/* 4DF3 */ { "HEXAGRAM FOR THE KEEPING STILL MOUNTAIN",NULL},
/* 4DF4 */ { "HEXAGRAM FOR DEVELOPMENT",NULL},
/* 4DF5 */ { "HEXAGRAM FOR THE MARRYING MAIDEN",NULL},
/* 4DF6 */ { "HEXAGRAM FOR ABUNDANCE",NULL},
/* 4DF7 */ { "HEXAGRAM FOR THE WANDERER",NULL},
/* 4DF8 */ { "HEXAGRAM FOR THE GENTLE WIND",NULL},
/* 4DF9 */ { "HEXAGRAM FOR THE JOYOUS LAKE",NULL},
/* 4DFA */ { "HEXAGRAM FOR DISPERSION",NULL},
/* 4DFB */ { "HEXAGRAM FOR LIMITATION",NULL},
/* 4DFC */ { "HEXAGRAM FOR INNER TRUTH",NULL},
/* 4DFD */ { "HEXAGRAM FOR SMALL PREPONDERANCE",NULL},
/* 4DFE */ { "HEXAGRAM FOR AFTER COMPLETION",NULL},
/* 4DFF */ { "HEXAGRAM FOR BEFORE COMPLETION",NULL}
};

UN_DLL_LOCAL
static const struct unicode_nameannot una_00_A0[] = {
/* A000 */ { "YI SYLLABLE IT",NULL},
/* A001 */ { "YI SYLLABLE IX",NULL},
/* A002 */ { "YI SYLLABLE I",NULL},
/* A003 */ { "YI SYLLABLE IP",NULL},
/* A004 */ { "YI SYLLABLE IET",NULL},
/* A005 */ { "YI SYLLABLE IEX",NULL},
/* A006 */ { "YI SYLLABLE IE",NULL},
/* A007 */ { "YI SYLLABLE IEP",NULL},
/* A008 */ { "YI SYLLABLE AT",NULL},
/* A009 */ { "YI SYLLABLE AX",NULL},
/* A00A */ { "YI SYLLABLE A",NULL},
/* A00B */ { "YI SYLLABLE AP",NULL},
/* A00C */ { "YI SYLLABLE UOX",NULL},
/* A00D */ { "YI SYLLABLE UO",NULL},
/* A00E */ { "YI SYLLABLE UOP",NULL},
/* A00F */ { "YI SYLLABLE OT",NULL},
/* A010 */ { "YI SYLLABLE OX",NULL},
/* A011 */ { "YI SYLLABLE O",NULL},
/* A012 */ { "YI SYLLABLE OP",NULL},
/* A013 */ { "YI SYLLABLE EX",NULL},
/* A014 */ { "YI SYLLABLE E",NULL},
/* A015 */ { "YI SYLLABLE WU","	% YI SYLLABLE ITERATION MARK\n"
	"	* name is a misnomer"},
/* A016 */ { "YI SYLLABLE BIT",NULL},
/* A017 */ { "YI SYLLABLE BIX",NULL},
/* A018 */ { "YI SYLLABLE BI",NULL},
/* A019 */ { "YI SYLLABLE BIP",NULL},
/* A01A */ { "YI SYLLABLE BIET",NULL},
/* A01B */ { "YI SYLLABLE BIEX",NULL},
/* A01C */ { "YI SYLLABLE BIE",NULL},
/* A01D */ { "YI SYLLABLE BIEP",NULL},
/* A01E */ { "YI SYLLABLE BAT",NULL},
/* A01F */ { "YI SYLLABLE BAX",NULL},
/* A020 */ { "YI SYLLABLE BA",NULL},
/* A021 */ { "YI SYLLABLE BAP",NULL},
/* A022 */ { "YI SYLLABLE BUOX",NULL},
/* A023 */ { "YI SYLLABLE BUO",NULL},
/* A024 */ { "YI SYLLABLE BUOP",NULL},
/* A025 */ { "YI SYLLABLE BOT",NULL},
/* A026 */ { "YI SYLLABLE BOX",NULL},
/* A027 */ { "YI SYLLABLE BO",NULL},
/* A028 */ { "YI SYLLABLE BOP",NULL},
/* A029 */ { "YI SYLLABLE BEX",NULL},
/* A02A */ { "YI SYLLABLE BE",NULL},
/* A02B */ { "YI SYLLABLE BEP",NULL},
/* A02C */ { "YI SYLLABLE BUT",NULL},
/* A02D */ { "YI SYLLABLE BUX",NULL},
/* A02E */ { "YI SYLLABLE BU",NULL},
/* A02F */ { "YI SYLLABLE BUP",NULL},
/* A030 */ { "YI SYLLABLE BURX",NULL},
/* A031 */ { "YI SYLLABLE BUR",NULL},
/* A032 */ { "YI SYLLABLE BYT",NULL},
/* A033 */ { "YI SYLLABLE BYX",NULL},
/* A034 */ { "YI SYLLABLE BY",NULL},
/* A035 */ { "YI SYLLABLE BYP",NULL},
/* A036 */ { "YI SYLLABLE BYRX",NULL},
/* A037 */ { "YI SYLLABLE BYR",NULL},
/* A038 */ { "YI SYLLABLE PIT",NULL},
/* A039 */ { "YI SYLLABLE PIX",NULL},
/* A03A */ { "YI SYLLABLE PI",NULL},
/* A03B */ { "YI SYLLABLE PIP",NULL},
/* A03C */ { "YI SYLLABLE PIEX",NULL},
/* A03D */ { "YI SYLLABLE PIE",NULL},
/* A03E */ { "YI SYLLABLE PIEP",NULL},
/* A03F */ { "YI SYLLABLE PAT",NULL},
/* A040 */ { "YI SYLLABLE PAX",NULL},
/* A041 */ { "YI SYLLABLE PA",NULL},
/* A042 */ { "YI SYLLABLE PAP",NULL},
/* A043 */ { "YI SYLLABLE PUOX",NULL},
/* A044 */ { "YI SYLLABLE PUO",NULL},
/* A045 */ { "YI SYLLABLE PUOP",NULL},
/* A046 */ { "YI SYLLABLE POT",NULL},
/* A047 */ { "YI SYLLABLE POX",NULL},
/* A048 */ { "YI SYLLABLE PO",NULL},
/* A049 */ { "YI SYLLABLE POP",NULL},
/* A04A */ { "YI SYLLABLE PUT",NULL},
/* A04B */ { "YI SYLLABLE PUX",NULL},
/* A04C */ { "YI SYLLABLE PU",NULL},
/* A04D */ { "YI SYLLABLE PUP",NULL},
/* A04E */ { "YI SYLLABLE PURX",NULL},
/* A04F */ { "YI SYLLABLE PUR",NULL},
/* A050 */ { "YI SYLLABLE PYT",NULL},
/* A051 */ { "YI SYLLABLE PYX",NULL},
/* A052 */ { "YI SYLLABLE PY",NULL},
/* A053 */ { "YI SYLLABLE PYP",NULL},
/* A054 */ { "YI SYLLABLE PYRX",NULL},
/* A055 */ { "YI SYLLABLE PYR",NULL},
/* A056 */ { "YI SYLLABLE BBIT",NULL},
/* A057 */ { "YI SYLLABLE BBIX",NULL},
/* A058 */ { "YI SYLLABLE BBI",NULL},
/* A059 */ { "YI SYLLABLE BBIP",NULL},
/* A05A */ { "YI SYLLABLE BBIET",NULL},
/* A05B */ { "YI SYLLABLE BBIEX",NULL},
/* A05C */ { "YI SYLLABLE BBIE",NULL},
/* A05D */ { "YI SYLLABLE BBIEP",NULL},
/* A05E */ { "YI SYLLABLE BBAT",NULL},
/* A05F */ { "YI SYLLABLE BBAX",NULL},
/* A060 */ { "YI SYLLABLE BBA",NULL},
/* A061 */ { "YI SYLLABLE BBAP",NULL},
/* A062 */ { "YI SYLLABLE BBUOX",NULL},
/* A063 */ { "YI SYLLABLE BBUO",NULL},
/* A064 */ { "YI SYLLABLE BBUOP",NULL},
/* A065 */ { "YI SYLLABLE BBOT",NULL},
/* A066 */ { "YI SYLLABLE BBOX",NULL},
/* A067 */ { "YI SYLLABLE BBO",NULL},
/* A068 */ { "YI SYLLABLE BBOP",NULL},
/* A069 */ { "YI SYLLABLE BBEX",NULL},
/* A06A */ { "YI SYLLABLE BBE",NULL},
/* A06B */ { "YI SYLLABLE BBEP",NULL},
/* A06C */ { "YI SYLLABLE BBUT",NULL},
/* A06D */ { "YI SYLLABLE BBUX",NULL},
/* A06E */ { "YI SYLLABLE BBU",NULL},
/* A06F */ { "YI SYLLABLE BBUP",NULL},
/* A070 */ { "YI SYLLABLE BBURX",NULL},
/* A071 */ { "YI SYLLABLE BBUR",NULL},
/* A072 */ { "YI SYLLABLE BBYT",NULL},
/* A073 */ { "YI SYLLABLE BBYX",NULL},
/* A074 */ { "YI SYLLABLE BBY",NULL},
/* A075 */ { "YI SYLLABLE BBYP",NULL},
/* A076 */ { "YI SYLLABLE NBIT",NULL},
/* A077 */ { "YI SYLLABLE NBIX",NULL},
/* A078 */ { "YI SYLLABLE NBI",NULL},
/* A079 */ { "YI SYLLABLE NBIP",NULL},
/* A07A */ { "YI SYLLABLE NBIEX",NULL},
/* A07B */ { "YI SYLLABLE NBIE",NULL},
/* A07C */ { "YI SYLLABLE NBIEP",NULL},
/* A07D */ { "YI SYLLABLE NBAT",NULL},
/* A07E */ { "YI SYLLABLE NBAX",NULL},
/* A07F */ { "YI SYLLABLE NBA",NULL},
/* A080 */ { "YI SYLLABLE NBAP",NULL},
/* A081 */ { "YI SYLLABLE NBOT",NULL},
/* A082 */ { "YI SYLLABLE NBOX",NULL},
/* A083 */ { "YI SYLLABLE NBO",NULL},
/* A084 */ { "YI SYLLABLE NBOP",NULL},
/* A085 */ { "YI SYLLABLE NBUT",NULL},
/* A086 */ { "YI SYLLABLE NBUX",NULL},
/* A087 */ { "YI SYLLABLE NBU",NULL},
/* A088 */ { "YI SYLLABLE NBUP",NULL},
/* A089 */ { "YI SYLLABLE NBURX",NULL},
/* A08A */ { "YI SYLLABLE NBUR",NULL},
/* A08B */ { "YI SYLLABLE NBYT",NULL},
/* A08C */ { "YI SYLLABLE NBYX",NULL},
/* A08D */ { "YI SYLLABLE NBY",NULL},
/* A08E */ { "YI SYLLABLE NBYP",NULL},
/* A08F */ { "YI SYLLABLE NBYRX",NULL},
/* A090 */ { "YI SYLLABLE NBYR",NULL},
/* A091 */ { "YI SYLLABLE HMIT",NULL},
/* A092 */ { "YI SYLLABLE HMIX",NULL},
/* A093 */ { "YI SYLLABLE HMI",NULL},
/* A094 */ { "YI SYLLABLE HMIP",NULL},
/* A095 */ { "YI SYLLABLE HMIEX",NULL},
/* A096 */ { "YI SYLLABLE HMIE",NULL},
/* A097 */ { "YI SYLLABLE HMIEP",NULL},
/* A098 */ { "YI SYLLABLE HMAT",NULL},
/* A099 */ { "YI SYLLABLE HMAX",NULL},
/* A09A */ { "YI SYLLABLE HMA",NULL},
/* A09B */ { "YI SYLLABLE HMAP",NULL},
/* A09C */ { "YI SYLLABLE HMUOX",NULL},
/* A09D */ { "YI SYLLABLE HMUO",NULL},
/* A09E */ { "YI SYLLABLE HMUOP",NULL},
/* A09F */ { "YI SYLLABLE HMOT",NULL},
/* A0A0 */ { "YI SYLLABLE HMOX",NULL},
/* A0A1 */ { "YI SYLLABLE HMO",NULL},
/* A0A2 */ { "YI SYLLABLE HMOP",NULL},
/* A0A3 */ { "YI SYLLABLE HMUT",NULL},
/* A0A4 */ { "YI SYLLABLE HMUX",NULL},
/* A0A5 */ { "YI SYLLABLE HMU",NULL},
/* A0A6 */ { "YI SYLLABLE HMUP",NULL},
/* A0A7 */ { "YI SYLLABLE HMURX",NULL},
/* A0A8 */ { "YI SYLLABLE HMUR",NULL},
/* A0A9 */ { "YI SYLLABLE HMYX",NULL},
/* A0AA */ { "YI SYLLABLE HMY",NULL},
/* A0AB */ { "YI SYLLABLE HMYP",NULL},
/* A0AC */ { "YI SYLLABLE HMYRX",NULL},
/* A0AD */ { "YI SYLLABLE HMYR",NULL},
/* A0AE */ { "YI SYLLABLE MIT",NULL},
/* A0AF */ { "YI SYLLABLE MIX",NULL},
/* A0B0 */ { "YI SYLLABLE MI",NULL},
/* A0B1 */ { "YI SYLLABLE MIP",NULL},
/* A0B2 */ { "YI SYLLABLE MIEX",NULL},
/* A0B3 */ { "YI SYLLABLE MIE",NULL},
/* A0B4 */ { "YI SYLLABLE MIEP",NULL},
/* A0B5 */ { "YI SYLLABLE MAT",NULL},
/* A0B6 */ { "YI SYLLABLE MAX",NULL},
/* A0B7 */ { "YI SYLLABLE MA",NULL},
/* A0B8 */ { "YI SYLLABLE MAP",NULL},
/* A0B9 */ { "YI SYLLABLE MUOT",NULL},
/* A0BA */ { "YI SYLLABLE MUOX",NULL},
/* A0BB */ { "YI SYLLABLE MUO",NULL},
/* A0BC */ { "YI SYLLABLE MUOP",NULL},
/* A0BD */ { "YI SYLLABLE MOT",NULL},
/* A0BE */ { "YI SYLLABLE MOX",NULL},
/* A0BF */ { "YI SYLLABLE MO",NULL},
/* A0C0 */ { "YI SYLLABLE MOP",NULL},
/* A0C1 */ { "YI SYLLABLE MEX",NULL},
/* A0C2 */ { "YI SYLLABLE ME",NULL},
/* A0C3 */ { "YI SYLLABLE MUT",NULL},
/* A0C4 */ { "YI SYLLABLE MUX",NULL},
/* A0C5 */ { "YI SYLLABLE MU",NULL},
/* A0C6 */ { "YI SYLLABLE MUP",NULL},
/* A0C7 */ { "YI SYLLABLE MURX",NULL},
/* A0C8 */ { "YI SYLLABLE MUR",NULL},
/* A0C9 */ { "YI SYLLABLE MYT",NULL},
/* A0CA */ { "YI SYLLABLE MYX",NULL},
/* A0CB */ { "YI SYLLABLE MY",NULL},
/* A0CC */ { "YI SYLLABLE MYP",NULL},
/* A0CD */ { "YI SYLLABLE FIT",NULL},
/* A0CE */ { "YI SYLLABLE FIX",NULL},
/* A0CF */ { "YI SYLLABLE FI",NULL},
/* A0D0 */ { "YI SYLLABLE FIP",NULL},
/* A0D1 */ { "YI SYLLABLE FAT",NULL},
/* A0D2 */ { "YI SYLLABLE FAX",NULL},
/* A0D3 */ { "YI SYLLABLE FA",NULL},
/* A0D4 */ { "YI SYLLABLE FAP",NULL},
/* A0D5 */ { "YI SYLLABLE FOX",NULL},
/* A0D6 */ { "YI SYLLABLE FO",NULL},
/* A0D7 */ { "YI SYLLABLE FOP",NULL},
/* A0D8 */ { "YI SYLLABLE FUT",NULL},
/* A0D9 */ { "YI SYLLABLE FUX",NULL},
/* A0DA */ { "YI SYLLABLE FU",NULL},
/* A0DB */ { "YI SYLLABLE FUP",NULL},
/* A0DC */ { "YI SYLLABLE FURX",NULL},
/* A0DD */ { "YI SYLLABLE FUR",NULL},
/* A0DE */ { "YI SYLLABLE FYT",NULL},
/* A0DF */ { "YI SYLLABLE FYX",NULL},
/* A0E0 */ { "YI SYLLABLE FY",NULL},
/* A0E1 */ { "YI SYLLABLE FYP",NULL},
/* A0E2 */ { "YI SYLLABLE VIT",NULL},
/* A0E3 */ { "YI SYLLABLE VIX",NULL},
/* A0E4 */ { "YI SYLLABLE VI",NULL},
/* A0E5 */ { "YI SYLLABLE VIP",NULL},
/* A0E6 */ { "YI SYLLABLE VIET",NULL},
/* A0E7 */ { "YI SYLLABLE VIEX",NULL},
/* A0E8 */ { "YI SYLLABLE VIE",NULL},
/* A0E9 */ { "YI SYLLABLE VIEP",NULL},
/* A0EA */ { "YI SYLLABLE VAT",NULL},
/* A0EB */ { "YI SYLLABLE VAX",NULL},
/* A0EC */ { "YI SYLLABLE VA",NULL},
/* A0ED */ { "YI SYLLABLE VAP",NULL},
/* A0EE */ { "YI SYLLABLE VOT",NULL},
/* A0EF */ { "YI SYLLABLE VOX",NULL},
/* A0F0 */ { "YI SYLLABLE VO",NULL},
/* A0F1 */ { "YI SYLLABLE VOP",NULL},
/* A0F2 */ { "YI SYLLABLE VEX",NULL},
/* A0F3 */ { "YI SYLLABLE VEP",NULL},
/* A0F4 */ { "YI SYLLABLE VUT",NULL},
/* A0F5 */ { "YI SYLLABLE VUX",NULL},
/* A0F6 */ { "YI SYLLABLE VU",NULL},
/* A0F7 */ { "YI SYLLABLE VUP",NULL},
/* A0F8 */ { "YI SYLLABLE VURX",NULL},
/* A0F9 */ { "YI SYLLABLE VUR",NULL},
/* A0FA */ { "YI SYLLABLE VYT",NULL},
/* A0FB */ { "YI SYLLABLE VYX",NULL},
/* A0FC */ { "YI SYLLABLE VY",NULL},
/* A0FD */ { "YI SYLLABLE VYP",NULL},
/* A0FE */ { "YI SYLLABLE VYRX",NULL},
/* A0FF */ { "YI SYLLABLE VYR",NULL}
};

UN_DLL_LOCAL
static const struct unicode_nameannot una_00_A1[] = {
/* A100 */ { "YI SYLLABLE DIT",NULL},
/* A101 */ { "YI SYLLABLE DIX",NULL},
/* A102 */ { "YI SYLLABLE DI",NULL},
/* A103 */ { "YI SYLLABLE DIP",NULL},
/* A104 */ { "YI SYLLABLE DIEX",NULL},
/* A105 */ { "YI SYLLABLE DIE",NULL},
/* A106 */ { "YI SYLLABLE DIEP",NULL},
/* A107 */ { "YI SYLLABLE DAT",NULL},
/* A108 */ { "YI SYLLABLE DAX",NULL},
/* A109 */ { "YI SYLLABLE DA",NULL},
/* A10A */ { "YI SYLLABLE DAP",NULL},
/* A10B */ { "YI SYLLABLE DUOX",NULL},
/* A10C */ { "YI SYLLABLE DUO",NULL},
/* A10D */ { "YI SYLLABLE DOT",NULL},
/* A10E */ { "YI SYLLABLE DOX",NULL},
/* A10F */ { "YI SYLLABLE DO",NULL},
/* A110 */ { "YI SYLLABLE DOP",NULL},
/* A111 */ { "YI SYLLABLE DEX",NULL},
/* A112 */ { "YI SYLLABLE DE",NULL},
/* A113 */ { "YI SYLLABLE DEP",NULL},
/* A114 */ { "YI SYLLABLE DUT",NULL},
/* A115 */ { "YI SYLLABLE DUX",NULL},
/* A116 */ { "YI SYLLABLE DU",NULL},
/* A117 */ { "YI SYLLABLE DUP",NULL},
/* A118 */ { "YI SYLLABLE DURX",NULL},
/* A119 */ { "YI SYLLABLE DUR",NULL},
/* A11A */ { "YI SYLLABLE TIT",NULL},
/* A11B */ { "YI SYLLABLE TIX",NULL},
/* A11C */ { "YI SYLLABLE TI",NULL},
/* A11D */ { "YI SYLLABLE TIP",NULL},
/* A11E */ { "YI SYLLABLE TIEX",NULL},
/* A11F */ { "YI SYLLABLE TIE",NULL},
/* A120 */ { "YI SYLLABLE TIEP",NULL},
/* A121 */ { "YI SYLLABLE TAT",NULL},
/* A122 */ { "YI SYLLABLE TAX",NULL},
/* A123 */ { "YI SYLLABLE TA",NULL},
/* A124 */ { "YI SYLLABLE TAP",NULL},
/* A125 */ { "YI SYLLABLE TUOT",NULL},
/* A126 */ { "YI SYLLABLE TUOX",NULL},
/* A127 */ { "YI SYLLABLE TUO",NULL},
/* A128 */ { "YI SYLLABLE TUOP",NULL},
/* A129 */ { "YI SYLLABLE TOT",NULL},
/* A12A */ { "YI SYLLABLE TOX",NULL},
/* A12B */ { "YI SYLLABLE TO",NULL},
/* A12C */ { "YI SYLLABLE TOP",NULL},
/* A12D */ { "YI SYLLABLE TEX",NULL},
/* A12E */ { "YI SYLLABLE TE",NULL},
/* A12F */ { "YI SYLLABLE TEP",NULL},
/* A130 */ { "YI SYLLABLE TUT",NULL},
/* A131 */ { "YI SYLLABLE TUX",NULL},
/* A132 */ { "YI SYLLABLE TU",NULL},
/* A133 */ { "YI SYLLABLE TUP",NULL},
/* A134 */ { "YI SYLLABLE TURX",NULL},
/* A135 */ { "YI SYLLABLE TUR",NULL},
/* A136 */ { "YI SYLLABLE DDIT",NULL},
/* A137 */ { "YI SYLLABLE DDIX",NULL},
/* A138 */ { "YI SYLLABLE DDI",NULL},
/* A139 */ { "YI SYLLABLE DDIP",NULL},
/* A13A */ { "YI SYLLABLE DDIEX",NULL},
/* A13B */ { "YI SYLLABLE DDIE",NULL},
/* A13C */ { "YI SYLLABLE DDIEP",NULL},
/* A13D */ { "YI SYLLABLE DDAT",NULL},
/* A13E */ { "YI SYLLABLE DDAX",NULL},
/* A13F */ { "YI SYLLABLE DDA",NULL},
/* A140 */ { "YI SYLLABLE DDAP",NULL},
/* A141 */ { "YI SYLLABLE DDUOX",NULL},
/* A142 */ { "YI SYLLABLE DDUO",NULL},
/* A143 */ { "YI SYLLABLE DDUOP",NULL},
/* A144 */ { "YI SYLLABLE DDOT",NULL},
/* A145 */ { "YI SYLLABLE DDOX",NULL},
/* A146 */ { "YI SYLLABLE DDO",NULL},
/* A147 */ { "YI SYLLABLE DDOP",NULL},
/* A148 */ { "YI SYLLABLE DDEX",NULL},
/* A149 */ { "YI SYLLABLE DDE",NULL},
/* A14A */ { "YI SYLLABLE DDEP",NULL},
/* A14B */ { "YI SYLLABLE DDUT",NULL},
/* A14C */ { "YI SYLLABLE DDUX",NULL},
/* A14D */ { "YI SYLLABLE DDU",NULL},
/* A14E */ { "YI SYLLABLE DDUP",NULL},
/* A14F */ { "YI SYLLABLE DDURX",NULL},
/* A150 */ { "YI SYLLABLE DDUR",NULL},
/* A151 */ { "YI SYLLABLE NDIT",NULL},
/* A152 */ { "YI SYLLABLE NDIX",NULL},
/* A153 */ { "YI SYLLABLE NDI",NULL},
/* A154 */ { "YI SYLLABLE NDIP",NULL},
/* A155 */ { "YI SYLLABLE NDIEX",NULL},
/* A156 */ { "YI SYLLABLE NDIE",NULL},
/* A157 */ { "YI SYLLABLE NDAT",NULL},
/* A158 */ { "YI SYLLABLE NDAX",NULL},
/* A159 */ { "YI SYLLABLE NDA",NULL},
/* A15A */ { "YI SYLLABLE NDAP",NULL},
/* A15B */ { "YI SYLLABLE NDOT",NULL},
/* A15C */ { "YI SYLLABLE NDOX",NULL},
/* A15D */ { "YI SYLLABLE NDO",NULL},
/* A15E */ { "YI SYLLABLE NDOP",NULL},
/* A15F */ { "YI SYLLABLE NDEX",NULL},
/* A160 */ { "YI SYLLABLE NDE",NULL},
/* A161 */ { "YI SYLLABLE NDEP",NULL},
/* A162 */ { "YI SYLLABLE NDUT",NULL},
/* A163 */ { "YI SYLLABLE NDUX",NULL},
/* A164 */ { "YI SYLLABLE NDU",NULL},
/* A165 */ { "YI SYLLABLE NDUP",NULL},
/* A166 */ { "YI SYLLABLE NDURX",NULL},
/* A167 */ { "YI SYLLABLE NDUR",NULL},
/* A168 */ { "YI SYLLABLE HNIT",NULL},
/* A169 */ { "YI SYLLABLE HNIX",NULL},
/* A16A */ { "YI SYLLABLE HNI",NULL},
/* A16B */ { "YI SYLLABLE HNIP",NULL},
/* A16C */ { "YI SYLLABLE HNIET",NULL},
/* A16D */ { "YI SYLLABLE HNIEX",NULL},
/* A16E */ { "YI SYLLABLE HNIE",NULL},
/* A16F */ { "YI SYLLABLE HNIEP",NULL},
/* A170 */ { "YI SYLLABLE HNAT",NULL},
/* A171 */ { "YI SYLLABLE HNAX",NULL},
/* A172 */ { "YI SYLLABLE HNA",NULL},
/* A173 */ { "YI SYLLABLE HNAP",NULL},
/* A174 */ { "YI SYLLABLE HNUOX",NULL},
/* A175 */ { "YI SYLLABLE HNUO",NULL},
/* A176 */ { "YI SYLLABLE HNOT",NULL},
/* A177 */ { "YI SYLLABLE HNOX",NULL},
/* A178 */ { "YI SYLLABLE HNOP",NULL},
/* A179 */ { "YI SYLLABLE HNEX",NULL},
/* A17A */ { "YI SYLLABLE HNE",NULL},
/* A17B */ { "YI SYLLABLE HNEP",NULL},
/* A17C */ { "YI SYLLABLE HNUT",NULL},
/* A17D */ { "YI SYLLABLE NIT",NULL},
/* A17E */ { "YI SYLLABLE NIX",NULL},
/* A17F */ { "YI SYLLABLE NI",NULL},
/* A180 */ { "YI SYLLABLE NIP",NULL},
/* A181 */ { "YI SYLLABLE NIEX",NULL},
/* A182 */ { "YI SYLLABLE NIE",NULL},
/* A183 */ { "YI SYLLABLE NIEP",NULL},
/* A184 */ { "YI SYLLABLE NAX",NULL},
/* A185 */ { "YI SYLLABLE NA",NULL},
/* A186 */ { "YI SYLLABLE NAP",NULL},
/* A187 */ { "YI SYLLABLE NUOX",NULL},
/* A188 */ { "YI SYLLABLE NUO",NULL},
/* A189 */ { "YI SYLLABLE NUOP",NULL},
/* A18A */ { "YI SYLLABLE NOT",NULL},
/* A18B */ { "YI SYLLABLE NOX",NULL},
/* A18C */ { "YI SYLLABLE NO",NULL},
/* A18D */ { "YI SYLLABLE NOP",NULL},
/* A18E */ { "YI SYLLABLE NEX",NULL},
/* A18F */ { "YI SYLLABLE NE",NULL},
/* A190 */ { "YI SYLLABLE NEP",NULL},
/* A191 */ { "YI SYLLABLE NUT",NULL},
/* A192 */ { "YI SYLLABLE NUX",NULL},
/* A193 */ { "YI SYLLABLE NU",NULL},
/* A194 */ { "YI SYLLABLE NUP",NULL},
/* A195 */ { "YI SYLLABLE NURX",NULL},
/* A196 */ { "YI SYLLABLE NUR",NULL},
/* A197 */ { "YI SYLLABLE HLIT",NULL},
/* A198 */ { "YI SYLLABLE HLIX",NULL},
/* A199 */ { "YI SYLLABLE HLI",NULL},
/* A19A */ { "YI SYLLABLE HLIP",NULL},
/* A19B */ { "YI SYLLABLE HLIEX",NULL},
/* A19C */ { "YI SYLLABLE HLIE",NULL},
/* A19D */ { "YI SYLLABLE HLIEP",NULL},
/* A19E */ { "YI SYLLABLE HLAT",NULL},
/* A19F */ { "YI SYLLABLE HLAX",NULL},
/* A1A0 */ { "YI SYLLABLE HLA",NULL},
/* A1A1 */ { "YI SYLLABLE HLAP",NULL},
/* A1A2 */ { "YI SYLLABLE HLUOX",NULL},
/* A1A3 */ { "YI SYLLABLE HLUO",NULL},
/* A1A4 */ { "YI SYLLABLE HLUOP",NULL},
/* A1A5 */ { "YI SYLLABLE HLOX",NULL},
/* A1A6 */ { "YI SYLLABLE HLO",NULL},
/* A1A7 */ { "YI SYLLABLE HLOP",NULL},
/* A1A8 */ { "YI SYLLABLE HLEX",NULL},
/* A1A9 */ { "YI SYLLABLE HLE",NULL},
/* A1AA */ { "YI SYLLABLE HLEP",NULL},
/* A1AB */ { "YI SYLLABLE HLUT",NULL},
/* A1AC */ { "YI SYLLABLE HLUX",NULL},
/* A1AD */ { "YI SYLLABLE HLU",NULL},
/* A1AE */ { "YI SYLLABLE HLUP",NULL},
/* A1AF */ { "YI SYLLABLE HLURX",NULL},
/* A1B0 */ { "YI SYLLABLE HLUR",NULL},
/* A1B1 */ { "YI SYLLABLE HLYT",NULL},
/* A1B2 */ { "YI SYLLABLE HLYX",NULL},
/* A1B3 */ { "YI SYLLABLE HLY",NULL},
/* A1B4 */ { "YI SYLLABLE HLYP",NULL},
/* A1B5 */ { "YI SYLLABLE HLYRX",NULL},
/* A1B6 */ { "YI SYLLABLE HLYR",NULL},
/* A1B7 */ { "YI SYLLABLE LIT",NULL},
/* A1B8 */ { "YI SYLLABLE LIX",NULL},
/* A1B9 */ { "YI SYLLABLE LI",NULL},
/* A1BA */ { "YI SYLLABLE LIP",NULL},
/* A1BB */ { "YI SYLLABLE LIET",NULL},
/* A1BC */ { "YI SYLLABLE LIEX",NULL},
/* A1BD */ { "YI SYLLABLE LIE",NULL},
/* A1BE */ { "YI SYLLABLE LIEP",NULL},
/* A1BF */ { "YI SYLLABLE LAT",NULL},
/* A1C0 */ { "YI SYLLABLE LAX",NULL},
/* A1C1 */ { "YI SYLLABLE LA",NULL},
/* A1C2 */ { "YI SYLLABLE LAP",NULL},
/* A1C3 */ { "YI SYLLABLE LUOT",NULL},
/* A1C4 */ { "YI SYLLABLE LUOX",NULL},
/* A1C5 */ { "YI SYLLABLE LUO",NULL},
/* A1C6 */ { "YI SYLLABLE LUOP",NULL},
/* A1C7 */ { "YI SYLLABLE LOT",NULL},
/* A1C8 */ { "YI SYLLABLE LOX",NULL},
/* A1C9 */ { "YI SYLLABLE LO",NULL},
/* A1CA */ { "YI SYLLABLE LOP",NULL},
/* A1CB */ { "YI SYLLABLE LEX",NULL},
/* A1CC */ { "YI SYLLABLE LE",NULL},
/* A1CD */ { "YI SYLLABLE LEP",NULL},
/* A1CE */ { "YI SYLLABLE LUT",NULL},
/* A1CF */ { "YI SYLLABLE LUX",NULL},
/* A1D0 */ { "YI SYLLABLE LU",NULL},
/* A1D1 */ { "YI SYLLABLE LUP",NULL},
/* A1D2 */ { "YI SYLLABLE LURX",NULL},
/* A1D3 */ { "YI SYLLABLE LUR",NULL},
/* A1D4 */ { "YI SYLLABLE LYT",NULL},
/* A1D5 */ { "YI SYLLABLE LYX",NULL},
/* A1D6 */ { "YI SYLLABLE LY",NULL},
/* A1D7 */ { "YI SYLLABLE LYP",NULL},
/* A1D8 */ { "YI SYLLABLE LYRX",NULL},
/* A1D9 */ { "YI SYLLABLE LYR",NULL},
/* A1DA */ { "YI SYLLABLE GIT",NULL},
/* A1DB */ { "YI SYLLABLE GIX",NULL},
/* A1DC */ { "YI SYLLABLE GI",NULL},
/* A1DD */ { "YI SYLLABLE GIP",NULL},
/* A1DE */ { "YI SYLLABLE GIET",NULL},
/* A1DF */ { "YI SYLLABLE GIEX",NULL},
/* A1E0 */ { "YI SYLLABLE GIE",NULL},
/* A1E1 */ { "YI SYLLABLE GIEP",NULL},
/* A1E2 */ { "YI SYLLABLE GAT",NULL},
/* A1E3 */ { "YI SYLLABLE GAX",NULL},
/* A1E4 */ { "YI SYLLABLE GA",NULL},
/* A1E5 */ { "YI SYLLABLE GAP",NULL},
/* A1E6 */ { "YI SYLLABLE GUOT",NULL},
/* A1E7 */ { "YI SYLLABLE GUOX",NULL},
/* A1E8 */ { "YI SYLLABLE GUO",NULL},
/* A1E9 */ { "YI SYLLABLE GUOP",NULL},
/* A1EA */ { "YI SYLLABLE GOT",NULL},
/* A1EB */ { "YI SYLLABLE GOX",NULL},
/* A1EC */ { "YI SYLLABLE GO",NULL},
/* A1ED */ { "YI SYLLABLE GOP",NULL},
/* A1EE */ { "YI SYLLABLE GET",NULL},
/* A1EF */ { "YI SYLLABLE GEX",NULL},
/* A1F0 */ { "YI SYLLABLE GE",NULL},
/* A1F1 */ { "YI SYLLABLE GEP",NULL},
/* A1F2 */ { "YI SYLLABLE GUT",NULL},
/* A1F3 */ { "YI SYLLABLE GUX",NULL},
/* A1F4 */ { "YI SYLLABLE GU",NULL},
/* A1F5 */ { "YI SYLLABLE GUP",NULL},
/* A1F6 */ { "YI SYLLABLE GURX",NULL},
/* A1F7 */ { "YI SYLLABLE GUR",NULL},
/* A1F8 */ { "YI SYLLABLE KIT",NULL},
/* A1F9 */ { "YI SYLLABLE KIX",NULL},
/* A1FA */ { "YI SYLLABLE KI",NULL},
/* A1FB */ { "YI SYLLABLE KIP",NULL},
/* A1FC */ { "YI SYLLABLE KIEX",NULL},
/* A1FD */ { "YI SYLLABLE KIE",NULL},
/* A1FE */ { "YI SYLLABLE KIEP",NULL},
/* A1FF */ { "YI SYLLABLE KAT",NULL}
};

UN_DLL_LOCAL
static const struct unicode_nameannot una_00_A2[] = {
/* A200 */ { "YI SYLLABLE KAX",NULL},
/* A201 */ { "YI SYLLABLE KA",NULL},
/* A202 */ { "YI SYLLABLE KAP",NULL},
/* A203 */ { "YI SYLLABLE KUOX",NULL},
/* A204 */ { "YI SYLLABLE KUO",NULL},
/* A205 */ { "YI SYLLABLE KUOP",NULL},
/* A206 */ { "YI SYLLABLE KOT",NULL},
/* A207 */ { "YI SYLLABLE KOX",NULL},
/* A208 */ { "YI SYLLABLE KO",NULL},
/* A209 */ { "YI SYLLABLE KOP",NULL},
/* A20A */ { "YI SYLLABLE KET",NULL},
/* A20B */ { "YI SYLLABLE KEX",NULL},
/* A20C */ { "YI SYLLABLE KE",NULL},
/* A20D */ { "YI SYLLABLE KEP",NULL},
/* A20E */ { "YI SYLLABLE KUT",NULL},
/* A20F */ { "YI SYLLABLE KUX",NULL},
/* A210 */ { "YI SYLLABLE KU",NULL},
/* A211 */ { "YI SYLLABLE KUP",NULL},
/* A212 */ { "YI SYLLABLE KURX",NULL},
/* A213 */ { "YI SYLLABLE KUR",NULL},
/* A214 */ { "YI SYLLABLE GGIT",NULL},
/* A215 */ { "YI SYLLABLE GGIX",NULL},
/* A216 */ { "YI SYLLABLE GGI",NULL},
/* A217 */ { "YI SYLLABLE GGIEX",NULL},
/* A218 */ { "YI SYLLABLE GGIE",NULL},
/* A219 */ { "YI SYLLABLE GGIEP",NULL},
/* A21A */ { "YI SYLLABLE GGAT",NULL},
/* A21B */ { "YI SYLLABLE GGAX",NULL},
/* A21C */ { "YI SYLLABLE GGA",NULL},
/* A21D */ { "YI SYLLABLE GGAP",NULL},
/* A21E */ { "YI SYLLABLE GGUOT",NULL},
/* A21F */ { "YI SYLLABLE GGUOX",NULL},
/* A220 */ { "YI SYLLABLE GGUO",NULL},
/* A221 */ { "YI SYLLABLE GGUOP",NULL},
/* A222 */ { "YI SYLLABLE GGOT",NULL},
/* A223 */ { "YI SYLLABLE GGOX",NULL},
/* A224 */ { "YI SYLLABLE GGO",NULL},
/* A225 */ { "YI SYLLABLE GGOP",NULL},
/* A226 */ { "YI SYLLABLE GGET",NULL},
/* A227 */ { "YI SYLLABLE GGEX",NULL},
/* A228 */ { "YI SYLLABLE GGE",NULL},
/* A229 */ { "YI SYLLABLE GGEP",NULL},
/* A22A */ { "YI SYLLABLE GGUT",NULL},
/* A22B */ { "YI SYLLABLE GGUX",NULL},
/* A22C */ { "YI SYLLABLE GGU",NULL},
/* A22D */ { "YI SYLLABLE GGUP",NULL},
/* A22E */ { "YI SYLLABLE GGURX",NULL},
/* A22F */ { "YI SYLLABLE GGUR",NULL},
/* A230 */ { "YI SYLLABLE MGIEX",NULL},
/* A231 */ { "YI SYLLABLE MGIE",NULL},
/* A232 */ { "YI SYLLABLE MGAT",NULL},
/* A233 */ { "YI SYLLABLE MGAX",NULL},
/* A234 */ { "YI SYLLABLE MGA",NULL},
/* A235 */ { "YI SYLLABLE MGAP",NULL},
/* A236 */ { "YI SYLLABLE MGUOX",NULL},
/* A237 */ { "YI SYLLABLE MGUO",NULL},
/* A238 */ { "YI SYLLABLE MGUOP",NULL},
/* A239 */ { "YI SYLLABLE MGOT",NULL},
/* A23A */ { "YI SYLLABLE MGOX",NULL},
/* A23B */ { "YI SYLLABLE MGO",NULL},
/* A23C */ { "YI SYLLABLE MGOP",NULL},
/* A23D */ { "YI SYLLABLE MGEX",NULL},
/* A23E */ { "YI SYLLABLE MGE",NULL},
/* A23F */ { "YI SYLLABLE MGEP",NULL},
/* A240 */ { "YI SYLLABLE MGUT",NULL},
/* A241 */ { "YI SYLLABLE MGUX",NULL},
/* A242 */ { "YI SYLLABLE MGU",NULL},
/* A243 */ { "YI SYLLABLE MGUP",NULL},
/* A244 */ { "YI SYLLABLE MGURX",NULL},
/* A245 */ { "YI SYLLABLE MGUR",NULL},
/* A246 */ { "YI SYLLABLE HXIT",NULL},
/* A247 */ { "YI SYLLABLE HXIX",NULL},
/* A248 */ { "YI SYLLABLE HXI",NULL},
/* A249 */ { "YI SYLLABLE HXIP",NULL},
/* A24A */ { "YI SYLLABLE HXIET",NULL},
/* A24B */ { "YI SYLLABLE HXIEX",NULL},
/* A24C */ { "YI SYLLABLE HXIE",NULL},
/* A24D */ { "YI SYLLABLE HXIEP",NULL},
/* A24E */ { "YI SYLLABLE HXAT",NULL},
/* A24F */ { "YI SYLLABLE HXAX",NULL},
/* A250 */ { "YI SYLLABLE HXA",NULL},
/* A251 */ { "YI SYLLABLE HXAP",NULL},
/* A252 */ { "YI SYLLABLE HXUOT",NULL},
/* A253 */ { "YI SYLLABLE HXUOX",NULL},
/* A254 */ { "YI SYLLABLE HXUO",NULL},
/* A255 */ { "YI SYLLABLE HXUOP",NULL},
/* A256 */ { "YI SYLLABLE HXOT",NULL},
/* A257 */ { "YI SYLLABLE HXOX",NULL},
/* A258 */ { "YI SYLLABLE HXO",NULL},
/* A259 */ { "YI SYLLABLE HXOP",NULL},
/* A25A */ { "YI SYLLABLE HXEX",NULL},
/* A25B */ { "YI SYLLABLE HXE",NULL},
/* A25C */ { "YI SYLLABLE HXEP",NULL},
/* A25D */ { "YI SYLLABLE NGIEX",NULL},
/* A25E */ { "YI SYLLABLE NGIE",NULL},
/* A25F */ { "YI SYLLABLE NGIEP",NULL},
/* A260 */ { "YI SYLLABLE NGAT",NULL},
/* A261 */ { "YI SYLLABLE NGAX",NULL},
/* A262 */ { "YI SYLLABLE NGA",NULL},
/* A263 */ { "YI SYLLABLE NGAP",NULL},
/* A264 */ { "YI SYLLABLE NGUOT",NULL},
/* A265 */ { "YI SYLLABLE NGUOX",NULL},
/* A266 */ { "YI SYLLABLE NGUO",NULL},
/* A267 */ { "YI SYLLABLE NGOT",NULL},
/* A268 */ { "YI SYLLABLE NGOX",NULL},
/* A269 */ { "YI SYLLABLE NGO",NULL},
/* A26A */ { "YI SYLLABLE NGOP",NULL},
/* A26B */ { "YI SYLLABLE NGEX",NULL},
/* A26C */ { "YI SYLLABLE NGE",NULL},
/* A26D */ { "YI SYLLABLE NGEP",NULL},
/* A26E */ { "YI SYLLABLE HIT",NULL},
/* A26F */ { "YI SYLLABLE HIEX",NULL},
/* A270 */ { "YI SYLLABLE HIE",NULL},
/* A271 */ { "YI SYLLABLE HAT",NULL},
/* A272 */ { "YI SYLLABLE HAX",NULL},
/* A273 */ { "YI SYLLABLE HA",NULL},
/* A274 */ { "YI SYLLABLE HAP",NULL},
/* A275 */ { "YI SYLLABLE HUOT",NULL},
/* A276 */ { "YI SYLLABLE HUOX",NULL},
/* A277 */ { "YI SYLLABLE HUO",NULL},
/* A278 */ { "YI SYLLABLE HUOP",NULL},
/* A279 */ { "YI SYLLABLE HOT",NULL},
/* A27A */ { "YI SYLLABLE HOX",NULL},
/* A27B */ { "YI SYLLABLE HO",NULL},
/* A27C */ { "YI SYLLABLE HOP",NULL},
/* A27D */ { "YI SYLLABLE HEX",NULL},
/* A27E */ { "YI SYLLABLE HE",NULL},
/* A27F */ { "YI SYLLABLE HEP",NULL},
/* A280 */ { "YI SYLLABLE WAT",NULL},
/* A281 */ { "YI SYLLABLE WAX",NULL},
/* A282 */ { "YI SYLLABLE WA",NULL},
/* A283 */ { "YI SYLLABLE WAP",NULL},
/* A284 */ { "YI SYLLABLE WUOX",NULL},
/* A285 */ { "YI SYLLABLE WUO",NULL},
/* A286 */ { "YI SYLLABLE WUOP",NULL},
/* A287 */ { "YI SYLLABLE WOX",NULL},
/* A288 */ { "YI SYLLABLE WO",NULL},
/* A289 */ { "YI SYLLABLE WOP",NULL},
/* A28A */ { "YI SYLLABLE WEX",NULL},
/* A28B */ { "YI SYLLABLE WE",NULL},
/* A28C */ { "YI SYLLABLE WEP",NULL},
/* A28D */ { "YI SYLLABLE ZIT",NULL},
/* A28E */ { "YI SYLLABLE ZIX",NULL},
/* A28F */ { "YI SYLLABLE ZI",NULL},
/* A290 */ { "YI SYLLABLE ZIP",NULL},
/* A291 */ { "YI SYLLABLE ZIEX",NULL},
/* A292 */ { "YI SYLLABLE ZIE",NULL},
/* A293 */ { "YI SYLLABLE ZIEP",NULL},
/* A294 */ { "YI SYLLABLE ZAT",NULL},
/* A295 */ { "YI SYLLABLE ZAX",NULL},
/* A296 */ { "YI SYLLABLE ZA",NULL},
/* A297 */ { "YI SYLLABLE ZAP",NULL},
/* A298 */ { "YI SYLLABLE ZUOX",NULL},
/* A299 */ { "YI SYLLABLE ZUO",NULL},
/* A29A */ { "YI SYLLABLE ZUOP",NULL},
/* A29B */ { "YI SYLLABLE ZOT",NULL},
/* A29C */ { "YI SYLLABLE ZOX",NULL},
/* A29D */ { "YI SYLLABLE ZO",NULL},
/* A29E */ { "YI SYLLABLE ZOP",NULL},
/* A29F */ { "YI SYLLABLE ZEX",NULL},
/* A2A0 */ { "YI SYLLABLE ZE",NULL},
/* A2A1 */ { "YI SYLLABLE ZEP",NULL},
/* A2A2 */ { "YI SYLLABLE ZUT",NULL},
/* A2A3 */ { "YI SYLLABLE ZUX",NULL},
/* A2A4 */ { "YI SYLLABLE ZU",NULL},
/* A2A5 */ { "YI SYLLABLE ZUP",NULL},
/* A2A6 */ { "YI SYLLABLE ZURX",NULL},
/* A2A7 */ { "YI SYLLABLE ZUR",NULL},
/* A2A8 */ { "YI SYLLABLE ZYT",NULL},
/* A2A9 */ { "YI SYLLABLE ZYX",NULL},
/* A2AA */ { "YI SYLLABLE ZY",NULL},
/* A2AB */ { "YI SYLLABLE ZYP",NULL},
/* A2AC */ { "YI SYLLABLE ZYRX",NULL},
/* A2AD */ { "YI SYLLABLE ZYR",NULL},
/* A2AE */ { "YI SYLLABLE CIT",NULL},
/* A2AF */ { "YI SYLLABLE CIX",NULL},
/* A2B0 */ { "YI SYLLABLE CI",NULL},
/* A2B1 */ { "YI SYLLABLE CIP",NULL},
/* A2B2 */ { "YI SYLLABLE CIET",NULL},
/* A2B3 */ { "YI SYLLABLE CIEX",NULL},
/* A2B4 */ { "YI SYLLABLE CIE",NULL},
/* A2B5 */ { "YI SYLLABLE CIEP",NULL},
/* A2B6 */ { "YI SYLLABLE CAT",NULL},
/* A2B7 */ { "YI SYLLABLE CAX",NULL},
/* A2B8 */ { "YI SYLLABLE CA",NULL},
/* A2B9 */ { "YI SYLLABLE CAP",NULL},
/* A2BA */ { "YI SYLLABLE CUOX",NULL},
/* A2BB */ { "YI SYLLABLE CUO",NULL},
/* A2BC */ { "YI SYLLABLE CUOP",NULL},
/* A2BD */ { "YI SYLLABLE COT",NULL},
/* A2BE */ { "YI SYLLABLE COX",NULL},
/* A2BF */ { "YI SYLLABLE CO",NULL},
/* A2C0 */ { "YI SYLLABLE COP",NULL},
/* A2C1 */ { "YI SYLLABLE CEX",NULL},
/* A2C2 */ { "YI SYLLABLE CE",NULL},
/* A2C3 */ { "YI SYLLABLE CEP",NULL},
/* A2C4 */ { "YI SYLLABLE CUT",NULL},
/* A2C5 */ { "YI SYLLABLE CUX",NULL},
/* A2C6 */ { "YI SYLLABLE CU",NULL},
/* A2C7 */ { "YI SYLLABLE CUP",NULL},
/* A2C8 */ { "YI SYLLABLE CURX",NULL},
/* A2C9 */ { "YI SYLLABLE CUR",NULL},
/* A2CA */ { "YI SYLLABLE CYT",NULL},
/* A2CB */ { "YI SYLLABLE CYX",NULL},
/* A2CC */ { "YI SYLLABLE CY",NULL},
/* A2CD */ { "YI SYLLABLE CYP",NULL},
/* A2CE */ { "YI SYLLABLE CYRX",NULL},
/* A2CF */ { "YI SYLLABLE CYR",NULL},
/* A2D0 */ { "YI SYLLABLE ZZIT",NULL},
/* A2D1 */ { "YI SYLLABLE ZZIX",NULL},
/* A2D2 */ { "YI SYLLABLE ZZI",NULL},
/* A2D3 */ { "YI SYLLABLE ZZIP",NULL},
/* A2D4 */ { "YI SYLLABLE ZZIET",NULL},
/* A2D5 */ { "YI SYLLABLE ZZIEX",NULL},
/* A2D6 */ { "YI SYLLABLE ZZIE",NULL},
/* A2D7 */ { "YI SYLLABLE ZZIEP",NULL},
/* A2D8 */ { "YI SYLLABLE ZZAT",NULL},
/* A2D9 */ { "YI SYLLABLE ZZAX",NULL},
/* A2DA */ { "YI SYLLABLE ZZA",NULL},
/* A2DB */ { "YI SYLLABLE ZZAP",NULL},
/* A2DC */ { "YI SYLLABLE ZZOX",NULL},
/* A2DD */ { "YI SYLLABLE ZZO",NULL},
/* A2DE */ { "YI SYLLABLE ZZOP",NULL},
/* A2DF */ { "YI SYLLABLE ZZEX",NULL},
/* A2E0 */ { "YI SYLLABLE ZZE",NULL},
/* A2E1 */ { "YI SYLLABLE ZZEP",NULL},
/* A2E2 */ { "YI SYLLABLE ZZUX",NULL},
/* A2E3 */ { "YI SYLLABLE ZZU",NULL},
/* A2E4 */ { "YI SYLLABLE ZZUP",NULL},
/* A2E5 */ { "YI SYLLABLE ZZURX",NULL},
/* A2E6 */ { "YI SYLLABLE ZZUR",NULL},
/* A2E7 */ { "YI SYLLABLE ZZYT",NULL},
/* A2E8 */ { "YI SYLLABLE ZZYX",NULL},
/* A2E9 */ { "YI SYLLABLE ZZY",NULL},
/* A2EA */ { "YI SYLLABLE ZZYP",NULL},
/* A2EB */ { "YI SYLLABLE ZZYRX",NULL},
/* A2EC */ { "YI SYLLABLE ZZYR",NULL},
/* A2ED */ { "YI SYLLABLE NZIT",NULL},
/* A2EE */ { "YI SYLLABLE NZIX",NULL},
/* A2EF */ { "YI SYLLABLE NZI",NULL},
/* A2F0 */ { "YI SYLLABLE NZIP",NULL},
/* A2F1 */ { "YI SYLLABLE NZIEX",NULL},
/* A2F2 */ { "YI SYLLABLE NZIE",NULL},
/* A2F3 */ { "YI SYLLABLE NZIEP",NULL},
/* A2F4 */ { "YI SYLLABLE NZAT",NULL},
/* A2F5 */ { "YI SYLLABLE NZAX",NULL},
/* A2F6 */ { "YI SYLLABLE NZA",NULL},
/* A2F7 */ { "YI SYLLABLE NZAP",NULL},
/* A2F8 */ { "YI SYLLABLE NZUOX",NULL},
/* A2F9 */ { "YI SYLLABLE NZUO",NULL},
/* A2FA */ { "YI SYLLABLE NZOX",NULL},
/* A2FB */ { "YI SYLLABLE NZOP",NULL},
/* A2FC */ { "YI SYLLABLE NZEX",NULL},
/* A2FD */ { "YI SYLLABLE NZE",NULL},
/* A2FE */ { "YI SYLLABLE NZUX",NULL},
/* A2FF */ { "YI SYLLABLE NZU",NULL}
};

UN_DLL_LOCAL
static const struct unicode_nameannot una_00_A3[] = {
/* A300 */ { "YI SYLLABLE NZUP",NULL},
/* A301 */ { "YI SYLLABLE NZURX",NULL},
/* A302 */ { "YI SYLLABLE NZUR",NULL},
/* A303 */ { "YI SYLLABLE NZYT",NULL},
/* A304 */ { "YI SYLLABLE NZYX",NULL},
/* A305 */ { "YI SYLLABLE NZY",NULL},
/* A306 */ { "YI SYLLABLE NZYP",NULL},
/* A307 */ { "YI SYLLABLE NZYRX",NULL},
/* A308 */ { "YI SYLLABLE NZYR",NULL},
/* A309 */ { "YI SYLLABLE SIT",NULL},
/* A30A */ { "YI SYLLABLE SIX",NULL},
/* A30B */ { "YI SYLLABLE SI",NULL},
/* A30C */ { "YI SYLLABLE SIP",NULL},
/* A30D */ { "YI SYLLABLE SIEX",NULL},
/* A30E */ { "YI SYLLABLE SIE",NULL},
/* A30F */ { "YI SYLLABLE SIEP",NULL},
/* A310 */ { "YI SYLLABLE SAT",NULL},
/* A311 */ { "YI SYLLABLE SAX",NULL},
/* A312 */ { "YI SYLLABLE SA",NULL},
/* A313 */ { "YI SYLLABLE SAP",NULL},
/* A314 */ { "YI SYLLABLE SUOX",NULL},
/* A315 */ { "YI SYLLABLE SUO",NULL},
/* A316 */ { "YI SYLLABLE SUOP",NULL},
/* A317 */ { "YI SYLLABLE SOT",NULL},
/* A318 */ { "YI SYLLABLE SOX",NULL},
/* A319 */ { "YI SYLLABLE SO",NULL},
/* A31A */ { "YI SYLLABLE SOP",NULL},
/* A31B */ { "YI SYLLABLE SEX",NULL},
/* A31C */ { "YI SYLLABLE SE",NULL},
/* A31D */ { "YI SYLLABLE SEP",NULL},
/* A31E */ { "YI SYLLABLE SUT",NULL},
/* A31F */ { "YI SYLLABLE SUX",NULL},
/* A320 */ { "YI SYLLABLE SU",NULL},
/* A321 */ { "YI SYLLABLE SUP",NULL},
/* A322 */ { "YI SYLLABLE SURX",NULL},
/* A323 */ { "YI SYLLABLE SUR",NULL},
/* A324 */ { "YI SYLLABLE SYT",NULL},
/* A325 */ { "YI SYLLABLE SYX",NULL},
/* A326 */ { "YI SYLLABLE SY",NULL},
/* A327 */ { "YI SYLLABLE SYP",NULL},
/* A328 */ { "YI SYLLABLE SYRX",NULL},
/* A329 */ { "YI SYLLABLE SYR",NULL},
/* A32A */ { "YI SYLLABLE SSIT",NULL},
/* A32B */ { "YI SYLLABLE SSIX",NULL},
/* A32C */ { "YI SYLLABLE SSI",NULL},
/* A32D */ { "YI SYLLABLE SSIP",NULL},
/* A32E */ { "YI SYLLABLE SSIEX",NULL},
/* A32F */ { "YI SYLLABLE SSIE",NULL},
/* A330 */ { "YI SYLLABLE SSIEP",NULL},
/* A331 */ { "YI SYLLABLE SSAT",NULL},
/* A332 */ { "YI SYLLABLE SSAX",NULL},
/* A333 */ { "YI SYLLABLE SSA",NULL},
/* A334 */ { "YI SYLLABLE SSAP",NULL},
/* A335 */ { "YI SYLLABLE SSOT",NULL},
/* A336 */ { "YI SYLLABLE SSOX",NULL},
/* A337 */ { "YI SYLLABLE SSO",NULL},
/* A338 */ { "YI SYLLABLE SSOP",NULL},
/* A339 */ { "YI SYLLABLE SSEX",NULL},
/* A33A */ { "YI SYLLABLE SSE",NULL},
/* A33B */ { "YI SYLLABLE SSEP",NULL},
/* A33C */ { "YI SYLLABLE SSUT",NULL},
/* A33D */ { "YI SYLLABLE SSUX",NULL},
/* A33E */ { "YI SYLLABLE SSU",NULL},
/* A33F */ { "YI SYLLABLE SSUP",NULL},
/* A340 */ { "YI SYLLABLE SSYT",NULL},
/* A341 */ { "YI SYLLABLE SSYX",NULL},
/* A342 */ { "YI SYLLABLE SSY",NULL},
/* A343 */ { "YI SYLLABLE SSYP",NULL},
/* A344 */ { "YI SYLLABLE SSYRX",NULL},
/* A345 */ { "YI SYLLABLE SSYR",NULL},
/* A346 */ { "YI SYLLABLE ZHAT",NULL},
/* A347 */ { "YI SYLLABLE ZHAX",NULL},
/* A348 */ { "YI SYLLABLE ZHA",NULL},
/* A349 */ { "YI SYLLABLE ZHAP",NULL},
/* A34A */ { "YI SYLLABLE ZHUOX",NULL},
/* A34B */ { "YI SYLLABLE ZHUO",NULL},
/* A34C */ { "YI SYLLABLE ZHUOP",NULL},
/* A34D */ { "YI SYLLABLE ZHOT",NULL},
/* A34E */ { "YI SYLLABLE ZHOX",NULL},
/* A34F */ { "YI SYLLABLE ZHO",NULL},
/* A350 */ { "YI SYLLABLE ZHOP",NULL},
/* A351 */ { "YI SYLLABLE ZHET",NULL},
/* A352 */ { "YI SYLLABLE ZHEX",NULL},
/* A353 */ { "YI SYLLABLE ZHE",NULL},
/* A354 */ { "YI SYLLABLE ZHEP",NULL},
/* A355 */ { "YI SYLLABLE ZHUT",NULL},
/* A356 */ { "YI SYLLABLE ZHUX",NULL},
/* A357 */ { "YI SYLLABLE ZHU",NULL},
/* A358 */ { "YI SYLLABLE ZHUP",NULL},
/* A359 */ { "YI SYLLABLE ZHURX",NULL},
/* A35A */ { "YI SYLLABLE ZHUR",NULL},
/* A35B */ { "YI SYLLABLE ZHYT",NULL},
/* A35C */ { "YI SYLLABLE ZHYX",NULL},
/* A35D */ { "YI SYLLABLE ZHY",NULL},
/* A35E */ { "YI SYLLABLE ZHYP",NULL},
/* A35F */ { "YI SYLLABLE ZHYRX",NULL},
/* A360 */ { "YI SYLLABLE ZHYR",NULL},
/* A361 */ { "YI SYLLABLE CHAT",NULL},
/* A362 */ { "YI SYLLABLE CHAX",NULL},
/* A363 */ { "YI SYLLABLE CHA",NULL},
/* A364 */ { "YI SYLLABLE CHAP",NULL},
/* A365 */ { "YI SYLLABLE CHUOT",NULL},
/* A366 */ { "YI SYLLABLE CHUOX",NULL},
/* A367 */ { "YI SYLLABLE CHUO",NULL},
/* A368 */ { "YI SYLLABLE CHUOP",NULL},
/* A369 */ { "YI SYLLABLE CHOT",NULL},
/* A36A */ { "YI SYLLABLE CHOX",NULL},
/* A36B */ { "YI SYLLABLE CHO",NULL},
/* A36C */ { "YI SYLLABLE CHOP",NULL},
/* A36D */ { "YI SYLLABLE CHET",NULL},
/* A36E */ { "YI SYLLABLE CHEX",NULL},
/* A36F */ { "YI SYLLABLE CHE",NULL},
/* A370 */ { "YI SYLLABLE CHEP",NULL},
/* A371 */ { "YI SYLLABLE CHUX",NULL},
/* A372 */ { "YI SYLLABLE CHU",NULL},
/* A373 */ { "YI SYLLABLE CHUP",NULL},
/* A374 */ { "YI SYLLABLE CHURX",NULL},
/* A375 */ { "YI SYLLABLE CHUR",NULL},
/* A376 */ { "YI SYLLABLE CHYT",NULL},
/* A377 */ { "YI SYLLABLE CHYX",NULL},
/* A378 */ { "YI SYLLABLE CHY",NULL},
/* A379 */ { "YI SYLLABLE CHYP",NULL},
/* A37A */ { "YI SYLLABLE CHYRX",NULL},
/* A37B */ { "YI SYLLABLE CHYR",NULL},
/* A37C */ { "YI SYLLABLE RRAX",NULL},
/* A37D */ { "YI SYLLABLE RRA",NULL},
/* A37E */ { "YI SYLLABLE RRUOX",NULL},
/* A37F */ { "YI SYLLABLE RRUO",NULL},
/* A380 */ { "YI SYLLABLE RROT",NULL},
/* A381 */ { "YI SYLLABLE RROX",NULL},
/* A382 */ { "YI SYLLABLE RRO",NULL},
/* A383 */ { "YI SYLLABLE RROP",NULL},
/* A384 */ { "YI SYLLABLE RRET",NULL},
/* A385 */ { "YI SYLLABLE RREX",NULL},
/* A386 */ { "YI SYLLABLE RRE",NULL},
/* A387 */ { "YI SYLLABLE RREP",NULL},
/* A388 */ { "YI SYLLABLE RRUT",NULL},
/* A389 */ { "YI SYLLABLE RRUX",NULL},
/* A38A */ { "YI SYLLABLE RRU",NULL},
/* A38B */ { "YI SYLLABLE RRUP",NULL},
/* A38C */ { "YI SYLLABLE RRURX",NULL},
/* A38D */ { "YI SYLLABLE RRUR",NULL},
/* A38E */ { "YI SYLLABLE RRYT",NULL},
/* A38F */ { "YI SYLLABLE RRYX",NULL},
/* A390 */ { "YI SYLLABLE RRY",NULL},
/* A391 */ { "YI SYLLABLE RRYP",NULL},
/* A392 */ { "YI SYLLABLE RRYRX",NULL},
/* A393 */ { "YI SYLLABLE RRYR",NULL},
/* A394 */ { "YI SYLLABLE NRAT",NULL},
/* A395 */ { "YI SYLLABLE NRAX",NULL},
/* A396 */ { "YI SYLLABLE NRA",NULL},
/* A397 */ { "YI SYLLABLE NRAP",NULL},
/* A398 */ { "YI SYLLABLE NROX",NULL},
/* A399 */ { "YI SYLLABLE NRO",NULL},
/* A39A */ { "YI SYLLABLE NROP",NULL},
/* A39B */ { "YI SYLLABLE NRET",NULL},
/* A39C */ { "YI SYLLABLE NREX",NULL},
/* A39D */ { "YI SYLLABLE NRE",NULL},
/* A39E */ { "YI SYLLABLE NREP",NULL},
/* A39F */ { "YI SYLLABLE NRUT",NULL},
/* A3A0 */ { "YI SYLLABLE NRUX",NULL},
/* A3A1 */ { "YI SYLLABLE NRU",NULL},
/* A3A2 */ { "YI SYLLABLE NRUP",NULL},
/* A3A3 */ { "YI SYLLABLE NRURX",NULL},
/* A3A4 */ { "YI SYLLABLE NRUR",NULL},
/* A3A5 */ { "YI SYLLABLE NRYT",NULL},
/* A3A6 */ { "YI SYLLABLE NRYX",NULL},
/* A3A7 */ { "YI SYLLABLE NRY",NULL},
/* A3A8 */ { "YI SYLLABLE NRYP",NULL},
/* A3A9 */ { "YI SYLLABLE NRYRX",NULL},
/* A3AA */ { "YI SYLLABLE NRYR",NULL},
/* A3AB */ { "YI SYLLABLE SHAT",NULL},
/* A3AC */ { "YI SYLLABLE SHAX",NULL},
/* A3AD */ { "YI SYLLABLE SHA",NULL},
/* A3AE */ { "YI SYLLABLE SHAP",NULL},
/* A3AF */ { "YI SYLLABLE SHUOX",NULL},
/* A3B0 */ { "YI SYLLABLE SHUO",NULL},
/* A3B1 */ { "YI SYLLABLE SHUOP",NULL},
/* A3B2 */ { "YI SYLLABLE SHOT",NULL},
/* A3B3 */ { "YI SYLLABLE SHOX",NULL},
/* A3B4 */ { "YI SYLLABLE SHO",NULL},
/* A3B5 */ { "YI SYLLABLE SHOP",NULL},
/* A3B6 */ { "YI SYLLABLE SHET",NULL},
/* A3B7 */ { "YI SYLLABLE SHEX",NULL},
/* A3B8 */ { "YI SYLLABLE SHE",NULL},
/* A3B9 */ { "YI SYLLABLE SHEP",NULL},
/* A3BA */ { "YI SYLLABLE SHUT",NULL},
/* A3BB */ { "YI SYLLABLE SHUX",NULL},
/* A3BC */ { "YI SYLLABLE SHU",NULL},
/* A3BD */ { "YI SYLLABLE SHUP",NULL},
/* A3BE */ { "YI SYLLABLE SHURX",NULL},
/* A3BF */ { "YI SYLLABLE SHUR",NULL},
/* A3C0 */ { "YI SYLLABLE SHYT",NULL},
/* A3C1 */ { "YI SYLLABLE SHYX",NULL},
/* A3C2 */ { "YI SYLLABLE SHY",NULL},
/* A3C3 */ { "YI SYLLABLE SHYP",NULL},
/* A3C4 */ { "YI SYLLABLE SHYRX",NULL},
/* A3C5 */ { "YI SYLLABLE SHYR",NULL},
/* A3C6 */ { "YI SYLLABLE RAT",NULL},
/* A3C7 */ { "YI SYLLABLE RAX",NULL},
/* A3C8 */ { "YI SYLLABLE RA",NULL},
/* A3C9 */ { "YI SYLLABLE RAP",NULL},
/* A3CA */ { "YI SYLLABLE RUOX",NULL},
/* A3CB */ { "YI SYLLABLE RUO",NULL},
/* A3CC */ { "YI SYLLABLE RUOP",NULL},
/* A3CD */ { "YI SYLLABLE ROT",NULL},
/* A3CE */ { "YI SYLLABLE ROX",NULL},
/* A3CF */ { "YI SYLLABLE RO",NULL},
/* A3D0 */ { "YI SYLLABLE ROP",NULL},
/* A3D1 */ { "YI SYLLABLE REX",NULL},
/* A3D2 */ { "YI SYLLABLE RE",NULL},
/* A3D3 */ { "YI SYLLABLE REP",NULL},
/* A3D4 */ { "YI SYLLABLE RUT",NULL},
/* A3D5 */ { "YI SYLLABLE RUX",NULL},
/* A3D6 */ { "YI SYLLABLE RU",NULL},
/* A3D7 */ { "YI SYLLABLE RUP",NULL},
/* A3D8 */ { "YI SYLLABLE RURX",NULL},
/* A3D9 */ { "YI SYLLABLE RUR",NULL},
/* A3DA */ { "YI SYLLABLE RYT",NULL},
/* A3DB */ { "YI SYLLABLE RYX",NULL},
/* A3DC */ { "YI SYLLABLE RY",NULL},
/* A3DD */ { "YI SYLLABLE RYP",NULL},
/* A3DE */ { "YI SYLLABLE RYRX",NULL},
/* A3DF */ { "YI SYLLABLE RYR",NULL},
/* A3E0 */ { "YI SYLLABLE JIT",NULL},
/* A3E1 */ { "YI SYLLABLE JIX",NULL},
/* A3E2 */ { "YI SYLLABLE JI",NULL},
/* A3E3 */ { "YI SYLLABLE JIP",NULL},
/* A3E4 */ { "YI SYLLABLE JIET",NULL},
/* A3E5 */ { "YI SYLLABLE JIEX",NULL},
/* A3E6 */ { "YI SYLLABLE JIE",NULL},
/* A3E7 */ { "YI SYLLABLE JIEP",NULL},
/* A3E8 */ { "YI SYLLABLE JUOT",NULL},
/* A3E9 */ { "YI SYLLABLE JUOX",NULL},
/* A3EA */ { "YI SYLLABLE JUO",NULL},
/* A3EB */ { "YI SYLLABLE JUOP",NULL},
/* A3EC */ { "YI SYLLABLE JOT",NULL},
/* A3ED */ { "YI SYLLABLE JOX",NULL},
/* A3EE */ { "YI SYLLABLE JO",NULL},
/* A3EF */ { "YI SYLLABLE JOP",NULL},
/* A3F0 */ { "YI SYLLABLE JUT",NULL},
/* A3F1 */ { "YI SYLLABLE JUX",NULL},
/* A3F2 */ { "YI SYLLABLE JU",NULL},
/* A3F3 */ { "YI SYLLABLE JUP",NULL},
/* A3F4 */ { "YI SYLLABLE JURX",NULL},
/* A3F5 */ { "YI SYLLABLE JUR",NULL},
/* A3F6 */ { "YI SYLLABLE JYT",NULL},
/* A3F7 */ { "YI SYLLABLE JYX",NULL},
/* A3F8 */ { "YI SYLLABLE JY",NULL},
/* A3F9 */ { "YI SYLLABLE JYP",NULL},
/* A3FA */ { "YI SYLLABLE JYRX",NULL},
/* A3FB */ { "YI SYLLABLE JYR",NULL},
/* A3FC */ { "YI SYLLABLE QIT",NULL},
/* A3FD */ { "YI SYLLABLE QIX",NULL},
/* A3FE */ { "YI SYLLABLE QI",NULL},
/* A3FF */ { "YI SYLLABLE QIP",NULL}
};

UN_DLL_LOCAL
static const struct unicode_nameannot una_00_A4[] = {
/* A400 */ { "YI SYLLABLE QIET",NULL},
/* A401 */ { "YI SYLLABLE QIEX",NULL},
/* A402 */ { "YI SYLLABLE QIE",NULL},
/* A403 */ { "YI SYLLABLE QIEP",NULL},
/* A404 */ { "YI SYLLABLE QUOT",NULL},
/* A405 */ { "YI SYLLABLE QUOX",NULL},
/* A406 */ { "YI SYLLABLE QUO",NULL},
/* A407 */ { "YI SYLLABLE QUOP",NULL},
/* A408 */ { "YI SYLLABLE QOT",NULL},
/* A409 */ { "YI SYLLABLE QOX",NULL},
/* A40A */ { "YI SYLLABLE QO",NULL},
/* A40B */ { "YI SYLLABLE QOP",NULL},
/* A40C */ { "YI SYLLABLE QUT",NULL},
/* A40D */ { "YI SYLLABLE QUX",NULL},
/* A40E */ { "YI SYLLABLE QU",NULL},
/* A40F */ { "YI SYLLABLE QUP",NULL},
/* A410 */ { "YI SYLLABLE QURX",NULL},
/* A411 */ { "YI SYLLABLE QUR",NULL},
/* A412 */ { "YI SYLLABLE QYT",NULL},
/* A413 */ { "YI SYLLABLE QYX",NULL},
/* A414 */ { "YI SYLLABLE QY",NULL},
/* A415 */ { "YI SYLLABLE QYP",NULL},
/* A416 */ { "YI SYLLABLE QYRX",NULL},
/* A417 */ { "YI SYLLABLE QYR",NULL},
/* A418 */ { "YI SYLLABLE JJIT",NULL},
/* A419 */ { "YI SYLLABLE JJIX",NULL},
/* A41A */ { "YI SYLLABLE JJI",NULL},
/* A41B */ { "YI SYLLABLE JJIP",NULL},
/* A41C */ { "YI SYLLABLE JJIET",NULL},
/* A41D */ { "YI SYLLABLE JJIEX",NULL},
/* A41E */ { "YI SYLLABLE JJIE",NULL},
/* A41F */ { "YI SYLLABLE JJIEP",NULL},
/* A420 */ { "YI SYLLABLE JJUOX",NULL},
/* A421 */ { "YI SYLLABLE JJUO",NULL},
/* A422 */ { "YI SYLLABLE JJUOP",NULL},
/* A423 */ { "YI SYLLABLE JJOT",NULL},
/* A424 */ { "YI SYLLABLE JJOX",NULL},
/* A425 */ { "YI SYLLABLE JJO",NULL},
/* A426 */ { "YI SYLLABLE JJOP",NULL},
/* A427 */ { "YI SYLLABLE JJUT",NULL},
/* A428 */ { "YI SYLLABLE JJUX",NULL},
/* A429 */ { "YI SYLLABLE JJU",NULL},
/* A42A */ { "YI SYLLABLE JJUP",NULL},
/* A42B */ { "YI SYLLABLE JJURX",NULL},
/* A42C */ { "YI SYLLABLE JJUR",NULL},
/* A42D */ { "YI SYLLABLE JJYT",NULL},
/* A42E */ { "YI SYLLABLE JJYX",NULL},
/* A42F */ { "YI SYLLABLE JJY",NULL},
/* A430 */ { "YI SYLLABLE JJYP",NULL},
/* A431 */ { "YI SYLLABLE NJIT",NULL},
/* A432 */ { "YI SYLLABLE NJIX",NULL},
/* A433 */ { "YI SYLLABLE NJI",NULL},
/* A434 */ { "YI SYLLABLE NJIP",NULL},
/* A435 */ { "YI SYLLABLE NJIET",NULL},
/* A436 */ { "YI SYLLABLE NJIEX",NULL},
/* A437 */ { "YI SYLLABLE NJIE",NULL},
/* A438 */ { "YI SYLLABLE NJIEP",NULL},
/* A439 */ { "YI SYLLABLE NJUOX",NULL},
/* A43A */ { "YI SYLLABLE NJUO",NULL},
/* A43B */ { "YI SYLLABLE NJOT",NULL},
/* A43C */ { "YI SYLLABLE NJOX",NULL},
/* A43D */ { "YI SYLLABLE NJO",NULL},
/* A43E */ { "YI SYLLABLE NJOP",NULL},
/* A43F */ { "YI SYLLABLE NJUX",NULL},
/* A440 */ { "YI SYLLABLE NJU",NULL},
/* A441 */ { "YI SYLLABLE NJUP",NULL},
/* A442 */ { "YI SYLLABLE NJURX",NULL},
/* A443 */ { "YI SYLLABLE NJUR",NULL},
/* A444 */ { "YI SYLLABLE NJYT",NULL},
/* A445 */ { "YI SYLLABLE NJYX",NULL},
/* A446 */ { "YI SYLLABLE NJY",NULL},
/* A447 */ { "YI SYLLABLE NJYP",NULL},
/* A448 */ { "YI SYLLABLE NJYRX",NULL},
/* A449 */ { "YI SYLLABLE NJYR",NULL},
/* A44A */ { "YI SYLLABLE NYIT",NULL},
/* A44B */ { "YI SYLLABLE NYIX",NULL},
/* A44C */ { "YI SYLLABLE NYI",NULL},
/* A44D */ { "YI SYLLABLE NYIP",NULL},
/* A44E */ { "YI SYLLABLE NYIET",NULL},
/* A44F */ { "YI SYLLABLE NYIEX",NULL},
/* A450 */ { "YI SYLLABLE NYIE",NULL},
/* A451 */ { "YI SYLLABLE NYIEP",NULL},
/* A452 */ { "YI SYLLABLE NYUOX",NULL},
/* A453 */ { "YI SYLLABLE NYUO",NULL},
/* A454 */ { "YI SYLLABLE NYUOP",NULL},
/* A455 */ { "YI SYLLABLE NYOT",NULL},
/* A456 */ { "YI SYLLABLE NYOX",NULL},
/* A457 */ { "YI SYLLABLE NYO",NULL},
/* A458 */ { "YI SYLLABLE NYOP",NULL},
/* A459 */ { "YI SYLLABLE NYUT",NULL},
/* A45A */ { "YI SYLLABLE NYUX",NULL},
/* A45B */ { "YI SYLLABLE NYU",NULL},
/* A45C */ { "YI SYLLABLE NYUP",NULL},
/* A45D */ { "YI SYLLABLE XIT",NULL},
/* A45E */ { "YI SYLLABLE XIX",NULL},
/* A45F */ { "YI SYLLABLE XI",NULL},
/* A460 */ { "YI SYLLABLE XIP",NULL},
/* A461 */ { "YI SYLLABLE XIET",NULL},
/* A462 */ { "YI SYLLABLE XIEX",NULL},
/* A463 */ { "YI SYLLABLE XIE",NULL},
/* A464 */ { "YI SYLLABLE XIEP",NULL},
/* A465 */ { "YI SYLLABLE XUOX",NULL},
/* A466 */ { "YI SYLLABLE XUO",NULL},
/* A467 */ { "YI SYLLABLE XOT",NULL},
/* A468 */ { "YI SYLLABLE XOX",NULL},
/* A469 */ { "YI SYLLABLE XO",NULL},
/* A46A */ { "YI SYLLABLE XOP",NULL},
/* A46B */ { "YI SYLLABLE XYT",NULL},
/* A46C */ { "YI SYLLABLE XYX",NULL},
/* A46D */ { "YI SYLLABLE XY",NULL},
/* A46E */ { "YI SYLLABLE XYP",NULL},
/* A46F */ { "YI SYLLABLE XYRX",NULL},
/* A470 */ { "YI SYLLABLE XYR",NULL},
/* A471 */ { "YI SYLLABLE YIT",NULL},
/* A472 */ { "YI SYLLABLE YIX",NULL},
/* A473 */ { "YI SYLLABLE YI",NULL},
/* A474 */ { "YI SYLLABLE YIP",NULL},
/* A475 */ { "YI SYLLABLE YIET",NULL},
/* A476 */ { "YI SYLLABLE YIEX",NULL},
/* A477 */ { "YI SYLLABLE YIE",NULL},
/* A478 */ { "YI SYLLABLE YIEP",NULL},
/* A479 */ { "YI SYLLABLE YUOT",NULL},
/* A47A */ { "YI SYLLABLE YUOX",NULL},
/* A47B */ { "YI SYLLABLE YUO",NULL},
/* A47C */ { "YI SYLLABLE YUOP",NULL},
/* A47D */ { "YI SYLLABLE YOT",NULL},
/* A47E */ { "YI SYLLABLE YOX",NULL},
/* A47F */ { "YI SYLLABLE YO",NULL},
/* A480 */ { "YI SYLLABLE YOP",NULL},
/* A481 */ { "YI SYLLABLE YUT",NULL},
/* A482 */ { "YI SYLLABLE YUX",NULL},
/* A483 */ { "YI SYLLABLE YU",NULL},
/* A484 */ { "YI SYLLABLE YUP",NULL},
/* A485 */ { "YI SYLLABLE YURX",NULL},
/* A486 */ { "YI SYLLABLE YUR",NULL},
/* A487 */ { "YI SYLLABLE YYT",NULL},
/* A488 */ { "YI SYLLABLE YYX",NULL},
/* A489 */ { "YI SYLLABLE YY",NULL},
/* A48A */ { "YI SYLLABLE YYP",NULL},
/* A48B */ { "YI SYLLABLE YYRX",NULL},
/* A48C */ { "YI SYLLABLE YYR",NULL},
/* A48D */ { NULL,NULL},
/* A48E */ { NULL,NULL},
/* A48F */ { NULL,NULL},
/* A490 */ { "YI RADICAL QOT","	x (yi syllable qot - A408)"},
/* A491 */ { "YI RADICAL LI","	x (yi syllable li - A1B9)"},
/* A492 */ { "YI RADICAL KIT",NULL},
/* A493 */ { "YI RADICAL NYIP",NULL},
/* A494 */ { "YI RADICAL CYP",NULL},
/* A495 */ { "YI RADICAL SSI",NULL},
/* A496 */ { "YI RADICAL GGOP",NULL},
/* A497 */ { "YI RADICAL GEP",NULL},
/* A498 */ { "YI RADICAL MI",NULL},
/* A499 */ { "YI RADICAL HXIT",NULL},
/* A49A */ { "YI RADICAL LYR",NULL},
/* A49B */ { "YI RADICAL BBUT",NULL},
/* A49C */ { "YI RADICAL MOP",NULL},
/* A49D */ { "YI RADICAL YO",NULL},
/* A49E */ { "YI RADICAL PUT",NULL},
/* A49F */ { "YI RADICAL HXUO",NULL},
/* A4A0 */ { "YI RADICAL TAT",NULL},
/* A4A1 */ { "YI RADICAL GA",NULL},
/* A4A2 */ { "YI RADICAL ZUP",NULL},
/* A4A3 */ { "YI RADICAL CYT",NULL},
/* A4A4 */ { "YI RADICAL DDUR",NULL},
/* A4A5 */ { "YI RADICAL BUR",NULL},
/* A4A6 */ { "YI RADICAL GGUO",NULL},
/* A4A7 */ { "YI RADICAL NYOP",NULL},
/* A4A8 */ { "YI RADICAL TU",NULL},
/* A4A9 */ { "YI RADICAL OP",NULL},
/* A4AA */ { "YI RADICAL JJUT",NULL},
/* A4AB */ { "YI RADICAL ZOT",NULL},
/* A4AC */ { "YI RADICAL PYT",NULL},
/* A4AD */ { "YI RADICAL HMO",NULL},
/* A4AE */ { "YI RADICAL YIT",NULL},
/* A4AF */ { "YI RADICAL VUR",NULL},
/* A4B0 */ { "YI RADICAL SHY",NULL},
/* A4B1 */ { "YI RADICAL VEP",NULL},
/* A4B2 */ { "YI RADICAL ZA",NULL},
/* A4B3 */ { "YI RADICAL JO",NULL},
/* A4B4 */ { "YI RADICAL NZUP",NULL},
/* A4B5 */ { "YI RADICAL JJY",NULL},
/* A4B6 */ { "YI RADICAL GOT",NULL},
/* A4B7 */ { "YI RADICAL JJIE",NULL},
/* A4B8 */ { "YI RADICAL WO",NULL},
/* A4B9 */ { "YI RADICAL DU",NULL},
/* A4BA */ { "YI RADICAL SHUR",NULL},
/* A4BB */ { "YI RADICAL LIE",NULL},
/* A4BC */ { "YI RADICAL CY",NULL},
/* A4BD */ { "YI RADICAL CUOP",NULL},
/* A4BE */ { "YI RADICAL CIP",NULL},
/* A4BF */ { "YI RADICAL HXOP",NULL},
/* A4C0 */ { "YI RADICAL SHAT",NULL},
/* A4C1 */ { "YI RADICAL ZUR",NULL},
/* A4C2 */ { "YI RADICAL SHOP",NULL},
/* A4C3 */ { "YI RADICAL CHE",NULL},
/* A4C4 */ { "YI RADICAL ZZIET",NULL},
/* A4C5 */ { "YI RADICAL NBIE",NULL},
/* A4C6 */ { "YI RADICAL KE",NULL},
/* A4C7 */ { NULL,NULL},
/* A4C8 */ { NULL,NULL},
/* A4C9 */ { NULL,NULL},
/* A4CA */ { NULL,NULL},
/* A4CB */ { NULL,NULL},
/* A4CC */ { NULL,NULL},
/* A4CD */ { NULL,NULL},
/* A4CE */ { NULL,NULL},
/* A4CF */ { NULL,NULL},
/* A4D0 */ { "LISU LETTER BA",NULL},
/* A4D1 */ { "LISU LETTER PA",NULL},
/* A4D2 */ { "LISU LETTER PHA",NULL},
/* A4D3 */ { "LISU LETTER DA",NULL},
/* A4D4 */ { "LISU LETTER TA",NULL},
/* A4D5 */ { "LISU LETTER THA",NULL},
/* A4D6 */ { "LISU LETTER GA",NULL},
/* A4D7 */ { "LISU LETTER KA",NULL},
/* A4D8 */ { "LISU LETTER KHA",NULL},
/* A4D9 */ { "LISU LETTER JA",NULL},
/* A4DA */ { "LISU LETTER CA",NULL},
/* A4DB */ { "LISU LETTER CHA",NULL},
/* A4DC */ { "LISU LETTER DZA",NULL},
/* A4DD */ { "LISU LETTER TSA",NULL},
/* A4DE */ { "LISU LETTER TSHA",NULL},
/* A4DF */ { "LISU LETTER MA",NULL},
/* A4E0 */ { "LISU LETTER NA",NULL},
/* A4E1 */ { "LISU LETTER LA",NULL},
/* A4E2 */ { "LISU LETTER SA",NULL},
/* A4E3 */ { "LISU LETTER ZHA",NULL},
/* A4E4 */ { "LISU LETTER ZA",NULL},
/* A4E5 */ { "LISU LETTER NGA",NULL},
/* A4E6 */ { "LISU LETTER HA",NULL},
/* A4E7 */ { "LISU LETTER XA",NULL},
/* A4E8 */ { "LISU LETTER HHA",NULL},
/* A4E9 */ { "LISU LETTER FA",NULL},
/* A4EA */ { "LISU LETTER WA",NULL},
/* A4EB */ { "LISU LETTER SHA",NULL},
/* A4EC */ { "LISU LETTER YA",NULL},
/* A4ED */ { "LISU LETTER GHA",NULL},
/* A4EE */ { "LISU LETTER A",NULL},
/* A4EF */ { "LISU LETTER AE",NULL},
/* A4F0 */ { "LISU LETTER E",NULL},
/* A4F1 */ { "LISU LETTER EU",NULL},
/* A4F2 */ { "LISU LETTER I",NULL},
/* A4F3 */ { "LISU LETTER O",NULL},
/* A4F4 */ { "LISU LETTER U",NULL},
/* A4F5 */ { "LISU LETTER UE",NULL},
/* A4F6 */ { "LISU LETTER UH",NULL},
/* A4F7 */ { "LISU LETTER OE",NULL},
/* A4F8 */ { "LISU LETTER TONE MYA TI",NULL},
/* A4F9 */ { "LISU LETTER TONE NA PO",NULL},
/* A4FA */ { "LISU LETTER TONE MYA CYA",NULL},
/* A4FB */ { "LISU LETTER TONE MYA BO",NULL},
/* A4FC */ { "LISU LETTER TONE MYA NA",NULL},
/* A4FD */ { "LISU LETTER TONE MYA JEU",NULL},
/* A4FE */ { "LISU PUNCTUATION COMMA",NULL},
/* A4FF */ { "LISU PUNCTUATION FULL STOP",NULL}
};

UN_DLL_LOCAL
static const struct unicode_nameannot una_00_A5[] = {
/* A500 */ { "VAI SYLLABLE EE",NULL},
/* A501 */ { "VAI SYLLABLE EEN",NULL},
/* A502 */ { "VAI SYLLABLE HEE",NULL},
/* A503 */ { "VAI SYLLABLE WEE",NULL},
/* A504 */ { "VAI SYLLABLE WEEN",NULL},
/* A505 */ { "VAI SYLLABLE PEE",NULL},
/* A506 */ { "VAI SYLLABLE BHEE",NULL},
/* A507 */ { "VAI SYLLABLE BEE",NULL},
/* A508 */ { "VAI SYLLABLE MBEE",NULL},
/* A509 */ { "VAI SYLLABLE KPEE",NULL},
/* A50A */ { "VAI SYLLABLE MGBEE",NULL},
/* A50B */ { "VAI SYLLABLE GBEE",NULL},
/* A50C */ { "VAI SYLLABLE FEE",NULL},
/* A50D */ { "VAI SYLLABLE VEE",NULL},
/* A50E */ { "VAI SYLLABLE TEE",NULL},
/* A50F */ { "VAI SYLLABLE THEE",NULL},
/* A510 */ { "VAI SYLLABLE DHEE",NULL},
/* A511 */ { "VAI SYLLABLE DHHEE",NULL},
/* A512 */ { "VAI SYLLABLE LEE",NULL},
/* A513 */ { "VAI SYLLABLE REE",NULL},
/* A514 */ { "VAI SYLLABLE DEE","	* read as logogram deeng \"child, small\" in the Book of Ndole"},
/* A515 */ { "VAI SYLLABLE NDEE",NULL},
/* A516 */ { "VAI SYLLABLE SEE",NULL},
/* A517 */ { "VAI SYLLABLE SHEE",NULL},
/* A518 */ { "VAI SYLLABLE ZEE",NULL},
/* A519 */ { "VAI SYLLABLE ZHEE",NULL},
/* A51A */ { "VAI SYLLABLE CEE",NULL},
/* A51B */ { "VAI SYLLABLE JEE",NULL},
/* A51C */ { "VAI SYLLABLE NJEE",NULL},
/* A51D */ { "VAI SYLLABLE YEE",NULL},
/* A51E */ { "VAI SYLLABLE KEE",NULL},
/* A51F */ { "VAI SYLLABLE NGGEE",NULL},
/* A520 */ { "VAI SYLLABLE GEE",NULL},
/* A521 */ { "VAI SYLLABLE MEE",NULL},
/* A522 */ { "VAI SYLLABLE NEE",NULL},
/* A523 */ { "VAI SYLLABLE NYEE",NULL},
/* A524 */ { "VAI SYLLABLE I",NULL},
/* A525 */ { "VAI SYLLABLE IN",NULL},
/* A526 */ { "VAI SYLLABLE HI",NULL},
/* A527 */ { "VAI SYLLABLE HIN",NULL},
/* A528 */ { "VAI SYLLABLE WI",NULL},
/* A529 */ { "VAI SYLLABLE WIN",NULL},
/* A52A */ { "VAI SYLLABLE PI",NULL},
/* A52B */ { "VAI SYLLABLE BHI",NULL},
/* A52C */ { "VAI SYLLABLE BI",NULL},
/* A52D */ { "VAI SYLLABLE MBI",NULL},
/* A52E */ { "VAI SYLLABLE KPI",NULL},
/* A52F */ { "VAI SYLLABLE MGBI",NULL},
/* A530 */ { "VAI SYLLABLE GBI",NULL},
/* A531 */ { "VAI SYLLABLE FI",NULL},
/* A532 */ { "VAI SYLLABLE VI",NULL},
/* A533 */ { "VAI SYLLABLE TI",NULL},
/* A534 */ { "VAI SYLLABLE THI",NULL},
/* A535 */ { "VAI SYLLABLE DHI",NULL},
/* A536 */ { "VAI SYLLABLE DHHI",NULL},
/* A537 */ { "VAI SYLLABLE LI",NULL},
/* A538 */ { "VAI SYLLABLE RI",NULL},
/* A539 */ { "VAI SYLLABLE DI",NULL},
/* A53A */ { "VAI SYLLABLE NDI",NULL},
/* A53B */ { "VAI SYLLABLE SI",NULL},
/* A53C */ { "VAI SYLLABLE SHI",NULL},
/* A53D */ { "VAI SYLLABLE ZI",NULL},
/* A53E */ { "VAI SYLLABLE ZHI",NULL},
/* A53F */ { "VAI SYLLABLE CI",NULL},
/* A540 */ { "VAI SYLLABLE JI",NULL},
/* A541 */ { "VAI SYLLABLE NJI",NULL},
/* A542 */ { "VAI SYLLABLE YI",NULL},
/* A543 */ { "VAI SYLLABLE KI",NULL},
/* A544 */ { "VAI SYLLABLE NGGI",NULL},
/* A545 */ { "VAI SYLLABLE GI",NULL},
/* A546 */ { "VAI SYLLABLE MI",NULL},
/* A547 */ { "VAI SYLLABLE NI",NULL},
/* A548 */ { "VAI SYLLABLE NYI",NULL},
/* A549 */ { "VAI SYLLABLE A",NULL},
/* A54A */ { "VAI SYLLABLE AN",NULL},
/* A54B */ { "VAI SYLLABLE NGAN",NULL},
/* A54C */ { "VAI SYLLABLE HA",NULL},
/* A54D */ { "VAI SYLLABLE HAN",NULL},
/* A54E */ { "VAI SYLLABLE WA",NULL},
/* A54F */ { "VAI SYLLABLE WAN",NULL},
/* A550 */ { "VAI SYLLABLE PA",NULL},
/* A551 */ { "VAI SYLLABLE BHA",NULL},
/* A552 */ { "VAI SYLLABLE BA",NULL},
/* A553 */ { "VAI SYLLABLE MBA",NULL},
/* A554 */ { "VAI SYLLABLE KPA",NULL},
/* A555 */ { "VAI SYLLABLE KPAN",NULL},
/* A556 */ { "VAI SYLLABLE MGBA",NULL},
/* A557 */ { "VAI SYLLABLE GBA",NULL},
/* A558 */ { "VAI SYLLABLE FA","	x (vai syllable ndole fa - A610)"},
/* A559 */ { "VAI SYLLABLE VA",NULL},
/* A55A */ { "VAI SYLLABLE TA",NULL},
/* A55B */ { "VAI SYLLABLE THA",NULL},
/* A55C */ { "VAI SYLLABLE DHA",NULL},
/* A55D */ { "VAI SYLLABLE DHHA",NULL},
/* A55E */ { "VAI SYLLABLE LA",NULL},
/* A55F */ { "VAI SYLLABLE RA",NULL},
/* A560 */ { "VAI SYLLABLE DA",NULL},
/* A561 */ { "VAI SYLLABLE NDA",NULL},
/* A562 */ { "VAI SYLLABLE SA",NULL},
/* A563 */ { "VAI SYLLABLE SHA",NULL},
/* A564 */ { "VAI SYLLABLE ZA",NULL},
/* A565 */ { "VAI SYLLABLE ZHA",NULL},
/* A566 */ { "VAI SYLLABLE CA",NULL},
/* A567 */ { "VAI SYLLABLE JA",NULL},
/* A568 */ { "VAI SYLLABLE NJA",NULL},
/* A569 */ { "VAI SYLLABLE YA",NULL},
/* A56A */ { "VAI SYLLABLE KA","	* read as logogram kai \"man\" in the Book of Ndole\n"
	"	x (vai syllable ndole ka - A611)"},
/* A56B */ { "VAI SYLLABLE KAN",NULL},
/* A56C */ { "VAI SYLLABLE NGGA",NULL},
/* A56D */ { "VAI SYLLABLE GA",NULL},
/* A56E */ { "VAI SYLLABLE MA","	x (vai syllable ndole ma - A62A)"},
/* A56F */ { "VAI SYLLABLE NA",NULL},
/* A570 */ { "VAI SYLLABLE NYA",NULL},
/* A571 */ { "VAI SYLLABLE OO",NULL},
/* A572 */ { "VAI SYLLABLE OON",NULL},
/* A573 */ { "VAI SYLLABLE HOO",NULL},
/* A574 */ { "VAI SYLLABLE WOO",NULL},
/* A575 */ { "VAI SYLLABLE WOON",NULL},
/* A576 */ { "VAI SYLLABLE POO",NULL},
/* A577 */ { "VAI SYLLABLE BHOO",NULL},
/* A578 */ { "VAI SYLLABLE BOO",NULL},
/* A579 */ { "VAI SYLLABLE MBOO",NULL},
/* A57A */ { "VAI SYLLABLE KPOO",NULL},
/* A57B */ { "VAI SYLLABLE MGBOO",NULL},
/* A57C */ { "VAI SYLLABLE GBOO",NULL},
/* A57D */ { "VAI SYLLABLE FOO",NULL},
/* A57E */ { "VAI SYLLABLE VOO",NULL},
/* A57F */ { "VAI SYLLABLE TOO",NULL},
/* A580 */ { "VAI SYLLABLE THOO",NULL},
/* A581 */ { "VAI SYLLABLE DHOO",NULL},
/* A582 */ { "VAI SYLLABLE DHHOO",NULL},
/* A583 */ { "VAI SYLLABLE LOO",NULL},
/* A584 */ { "VAI SYLLABLE ROO",NULL},
/* A585 */ { "VAI SYLLABLE DOO",NULL},
/* A586 */ { "VAI SYLLABLE NDOO",NULL},
/* A587 */ { "VAI SYLLABLE SOO","	x (vai syllable ndole soo - A612)"},
/* A588 */ { "VAI SYLLABLE SHOO",NULL},
/* A589 */ { "VAI SYLLABLE ZOO",NULL},
/* A58A */ { "VAI SYLLABLE ZHOO",NULL},
/* A58B */ { "VAI SYLLABLE COO",NULL},
/* A58C */ { "VAI SYLLABLE JOO",NULL},
/* A58D */ { "VAI SYLLABLE NJOO",NULL},
/* A58E */ { "VAI SYLLABLE YOO",NULL},
/* A58F */ { "VAI SYLLABLE KOO",NULL},
/* A590 */ { "VAI SYLLABLE NGGOO",NULL},
/* A591 */ { "VAI SYLLABLE GOO",NULL},
/* A592 */ { "VAI SYLLABLE MOO",NULL},
/* A593 */ { "VAI SYLLABLE NOO",NULL},
/* A594 */ { "VAI SYLLABLE NYOO",NULL},
/* A595 */ { "VAI SYLLABLE U",NULL},
/* A596 */ { "VAI SYLLABLE UN",NULL},
/* A597 */ { "VAI SYLLABLE HU",NULL},
/* A598 */ { "VAI SYLLABLE HUN",NULL},
/* A599 */ { "VAI SYLLABLE WU",NULL},
/* A59A */ { "VAI SYLLABLE WUN",NULL},
/* A59B */ { "VAI SYLLABLE PU",NULL},
/* A59C */ { "VAI SYLLABLE BHU",NULL},
/* A59D */ { "VAI SYLLABLE BU",NULL},
/* A59E */ { "VAI SYLLABLE MBU",NULL},
/* A59F */ { "VAI SYLLABLE KPU",NULL},
/* A5A0 */ { "VAI SYLLABLE MGBU",NULL},
/* A5A1 */ { "VAI SYLLABLE GBU",NULL},
/* A5A2 */ { "VAI SYLLABLE FU",NULL},
/* A5A3 */ { "VAI SYLLABLE VU",NULL},
/* A5A4 */ { "VAI SYLLABLE TU",NULL},
/* A5A5 */ { "VAI SYLLABLE THU",NULL},
/* A5A6 */ { "VAI SYLLABLE DHU",NULL},
/* A5A7 */ { "VAI SYLLABLE DHHU",NULL},
/* A5A8 */ { "VAI SYLLABLE LU",NULL},
/* A5A9 */ { "VAI SYLLABLE RU",NULL},
/* A5AA */ { "VAI SYLLABLE DU",NULL},
/* A5AB */ { "VAI SYLLABLE NDU",NULL},
/* A5AC */ { "VAI SYLLABLE SU",NULL},
/* A5AD */ { "VAI SYLLABLE SHU",NULL},
/* A5AE */ { "VAI SYLLABLE ZU",NULL},
/* A5AF */ { "VAI SYLLABLE ZHU",NULL},
/* A5B0 */ { "VAI SYLLABLE CU",NULL},
/* A5B1 */ { "VAI SYLLABLE JU",NULL},
/* A5B2 */ { "VAI SYLLABLE NJU",NULL},
/* A5B3 */ { "VAI SYLLABLE YU",NULL},
/* A5B4 */ { "VAI SYLLABLE KU",NULL},
/* A5B5 */ { "VAI SYLLABLE NGGU",NULL},
/* A5B6 */ { "VAI SYLLABLE GU",NULL},
/* A5B7 */ { "VAI SYLLABLE MU",NULL},
/* A5B8 */ { "VAI SYLLABLE NU",NULL},
/* A5B9 */ { "VAI SYLLABLE NYU",NULL},
/* A5BA */ { "VAI SYLLABLE O",NULL},
/* A5BB */ { "VAI SYLLABLE ON",NULL},
/* A5BC */ { "VAI SYLLABLE NGON",NULL},
/* A5BD */ { "VAI SYLLABLE HO",NULL},
/* A5BE */ { "VAI SYLLABLE HON",NULL},
/* A5BF */ { "VAI SYLLABLE WO",NULL},
/* A5C0 */ { "VAI SYLLABLE WON",NULL},
/* A5C1 */ { "VAI SYLLABLE PO",NULL},
/* A5C2 */ { "VAI SYLLABLE BHO",NULL},
/* A5C3 */ { "VAI SYLLABLE BO",NULL},
/* A5C4 */ { "VAI SYLLABLE MBO",NULL},
/* A5C5 */ { "VAI SYLLABLE KPO",NULL},
/* A5C6 */ { "VAI SYLLABLE MGBO",NULL},
/* A5C7 */ { "VAI SYLLABLE GBO",NULL},
/* A5C8 */ { "VAI SYLLABLE GBON",NULL},
/* A5C9 */ { "VAI SYLLABLE FO",NULL},
/* A5CA */ { "VAI SYLLABLE VO",NULL},
/* A5CB */ { "VAI SYLLABLE TO",NULL},
/* A5CC */ { "VAI SYLLABLE THO",NULL},
/* A5CD */ { "VAI SYLLABLE DHO",NULL},
/* A5CE */ { "VAI SYLLABLE DHHO",NULL},
/* A5CF */ { "VAI SYLLABLE LO",NULL},
/* A5D0 */ { "VAI SYLLABLE RO",NULL},
/* A5D1 */ { "VAI SYLLABLE DO","	* read as logogram lo \"in\" in the Book of Ndole\n"
	"	x (vai syllable ndole do - A62B)"},
/* A5D2 */ { "VAI SYLLABLE NDO",NULL},
/* A5D3 */ { "VAI SYLLABLE SO",NULL},
/* A5D4 */ { "VAI SYLLABLE SHO",NULL},
/* A5D5 */ { "VAI SYLLABLE ZO",NULL},
/* A5D6 */ { "VAI SYLLABLE ZHO",NULL},
/* A5D7 */ { "VAI SYLLABLE CO",NULL},
/* A5D8 */ { "VAI SYLLABLE JO",NULL},
/* A5D9 */ { "VAI SYLLABLE NJO",NULL},
/* A5DA */ { "VAI SYLLABLE YO",NULL},
/* A5DB */ { "VAI SYLLABLE KO",NULL},
/* A5DC */ { "VAI SYLLABLE NGGO",NULL},
/* A5DD */ { "VAI SYLLABLE GO",NULL},
/* A5DE */ { "VAI SYLLABLE MO",NULL},
/* A5DF */ { "VAI SYLLABLE NO",NULL},
/* A5E0 */ { "VAI SYLLABLE NYO",NULL},
/* A5E1 */ { "VAI SYLLABLE E",NULL},
/* A5E2 */ { "VAI SYLLABLE EN",NULL},
/* A5E3 */ { "VAI SYLLABLE NGEN",NULL},
/* A5E4 */ { "VAI SYLLABLE HE",NULL},
/* A5E5 */ { "VAI SYLLABLE HEN",NULL},
/* A5E6 */ { "VAI SYLLABLE WE",NULL},
/* A5E7 */ { "VAI SYLLABLE WEN",NULL},
/* A5E8 */ { "VAI SYLLABLE PE",NULL},
/* A5E9 */ { "VAI SYLLABLE BHE",NULL},
/* A5EA */ { "VAI SYLLABLE BE",NULL},
/* A5EB */ { "VAI SYLLABLE MBE",NULL},
/* A5EC */ { "VAI SYLLABLE KPE",NULL},
/* A5ED */ { "VAI SYLLABLE KPEN",NULL},
/* A5EE */ { "VAI SYLLABLE MGBE",NULL},
/* A5EF */ { "VAI SYLLABLE GBE",NULL},
/* A5F0 */ { "VAI SYLLABLE GBEN",NULL},
/* A5F1 */ { "VAI SYLLABLE FE",NULL},
/* A5F2 */ { "VAI SYLLABLE VE",NULL},
/* A5F3 */ { "VAI SYLLABLE TE",NULL},
/* A5F4 */ { "VAI SYLLABLE THE",NULL},
/* A5F5 */ { "VAI SYLLABLE DHE",NULL},
/* A5F6 */ { "VAI SYLLABLE DHHE",NULL},
/* A5F7 */ { "VAI SYLLABLE LE",NULL},
/* A5F8 */ { "VAI SYLLABLE RE",NULL},
/* A5F9 */ { "VAI SYLLABLE DE",NULL},
/* A5FA */ { "VAI SYLLABLE NDE",NULL},
/* A5FB */ { "VAI SYLLABLE SE",NULL},
/* A5FC */ { "VAI SYLLABLE SHE",NULL},
/* A5FD */ { "VAI SYLLABLE ZE",NULL},
/* A5FE */ { "VAI SYLLABLE ZHE",NULL},
/* A5FF */ { "VAI SYLLABLE CE",NULL}
};

UN_DLL_LOCAL
static const struct unicode_nameannot una_00_A6[] = {
/* A600 */ { "VAI SYLLABLE JE",NULL},
/* A601 */ { "VAI SYLLABLE NJE",NULL},
/* A602 */ { "VAI SYLLABLE YE",NULL},
/* A603 */ { "VAI SYLLABLE KE",NULL},
/* A604 */ { "VAI SYLLABLE NGGE",NULL},
/* A605 */ { "VAI SYLLABLE NGGEN",NULL},
/* A606 */ { "VAI SYLLABLE GE",NULL},
/* A607 */ { "VAI SYLLABLE GEN",NULL},
/* A608 */ { "VAI SYLLABLE ME",NULL},
/* A609 */ { "VAI SYLLABLE NE",NULL},
/* A60A */ { "VAI SYLLABLE NYE",NULL},
/* A60B */ { "VAI SYLLABLE NG",NULL},
/* A60C */ { "VAI SYLLABLE LENGTHENER",NULL},
/* A60D */ { "VAI COMMA",NULL},
/* A60E */ { "VAI FULL STOP",NULL},
/* A60F */ { "VAI QUESTION MARK",NULL},
/* A610 */ { "VAI SYLLABLE NDOLE FA","	x (vai syllable fa - A558)"},
/* A611 */ { "VAI SYLLABLE NDOLE KA","	x (vai syllable ka - A56A)"},
/* A612 */ { "VAI SYLLABLE NDOLE SOO","	x (vai syllable soo - A587)"},
/* A613 */ { "VAI SYMBOL FEENG","	* logogram for \"thing\""},
/* A614 */ { "VAI SYMBOL KEENG","	* logogram for \"foot\""},
/* A615 */ { "VAI SYMBOL TING","	* logogram for \"island\""},
/* A616 */ { "VAI SYMBOL NII","	* logogram for \"cow\"\n"
	"	* also logogram for kpe kowu \"case of gin\""},
/* A617 */ { "VAI SYMBOL BANG","	* logogram for \"finished\""},
/* A618 */ { "VAI SYMBOL FAA","	* logogram for \"die, kill\""},
/* A619 */ { "VAI SYMBOL TAA","	* logogram for \"go, carry, journey\""},
/* A61A */ { "VAI SYMBOL DANG","	* logogram for \"hear, understand\""},
/* A61B */ { "VAI SYMBOL DOONG","	* logogram for \"enter\""},
/* A61C */ { "VAI SYMBOL KUNG","	* logogram for \"head, be able\""},
/* A61D */ { "VAI SYMBOL TONG","	* logogram for \"be named\""},
/* A61E */ { "VAI SYMBOL DO-O","	* logogram for \"be small\""},
/* A61F */ { "VAI SYMBOL JONG","	* logogram for \"slave\""},
/* A620 */ { "VAI DIGIT ZERO",NULL},
/* A621 */ { "VAI DIGIT ONE",NULL},
/* A622 */ { "VAI DIGIT TWO",NULL},
/* A623 */ { "VAI DIGIT THREE",NULL},
/* A624 */ { "VAI DIGIT FOUR",NULL},
/* A625 */ { "VAI DIGIT FIVE",NULL},
/* A626 */ { "VAI DIGIT SIX",NULL},
/* A627 */ { "VAI DIGIT SEVEN",NULL},
/* A628 */ { "VAI DIGIT EIGHT",NULL},
/* A629 */ { "VAI DIGIT NINE",NULL},
/* A62A */ { "VAI SYLLABLE NDOLE MA","	x (vai syllable ma - A56E)"},
/* A62B */ { "VAI SYLLABLE NDOLE DO","	x (vai syllable do - A5D1)"},
/* A62C */ { NULL,NULL},
/* A62D */ { NULL,NULL},
/* A62E */ { NULL,NULL},
/* A62F */ { NULL,NULL},
/* A630 */ { NULL,NULL},
/* A631 */ { NULL,NULL},
/* A632 */ { NULL,NULL},
/* A633 */ { NULL,NULL},
/* A634 */ { NULL,NULL},
/* A635 */ { NULL,NULL},
/* A636 */ { NULL,NULL},
/* A637 */ { NULL,NULL},
/* A638 */ { NULL,NULL},
/* A639 */ { NULL,NULL},
/* A63A */ { NULL,NULL},
/* A63B */ { NULL,NULL},
/* A63C */ { NULL,NULL},
/* A63D */ { NULL,NULL},
/* A63E */ { NULL,NULL},
/* A63F */ { NULL,NULL},
/* A640 */ { "CYRILLIC CAPITAL LETTER ZEMLYA",NULL},
/* A641 */ { "CYRILLIC SMALL LETTER ZEMLYA","	x (cyrillic small letter ze - 0437)"},
/* A642 */ { "CYRILLIC CAPITAL LETTER DZELO",NULL},
/* A643 */ { "CYRILLIC SMALL LETTER DZELO","	x (cyrillic small letter dze - 0455)"},
/* A644 */ { "CYRILLIC CAPITAL LETTER REVERSED DZE","	x (roman sextula sign - 10193)"},
/* A645 */ { "CYRILLIC SMALL LETTER REVERSED DZE",NULL},
/* A646 */ { "CYRILLIC CAPITAL LETTER IOTA",NULL},
/* A647 */ { "CYRILLIC SMALL LETTER IOTA","	x (glagolitic small letter initial izhe - 2C3A)"},
/* A648 */ { "CYRILLIC CAPITAL LETTER DJERV",NULL},
/* A649 */ { "CYRILLIC SMALL LETTER DJERV","	x (cyrillic small letter dje - 0452)\n"
	"	x (cyrillic small letter tshe - 045B)"},
/* A64A */ { "CYRILLIC CAPITAL LETTER MONOGRAPH UK",NULL},
/* A64B */ { "CYRILLIC SMALL LETTER MONOGRAPH UK","	x (cyrillic small letter u - 0443)"},
/* A64C */ { "CYRILLIC CAPITAL LETTER BROAD OMEGA",NULL},
/* A64D */ { "CYRILLIC SMALL LETTER BROAD OMEGA","	x (cyrillic small letter omega - 0461)"},
/* A64E */ { "CYRILLIC CAPITAL LETTER NEUTRAL YER",NULL},
/* A64F */ { "CYRILLIC SMALL LETTER NEUTRAL YER","	x (cyrillic small letter hard sign - 044A)\n"
	"	x (cyrillic small letter soft sign - 044C)"},
/* A650 */ { "CYRILLIC CAPITAL LETTER YERU WITH BACK YER",NULL},
/* A651 */ { "CYRILLIC SMALL LETTER YERU WITH BACK YER","	x (cyrillic small letter yeru - 044B)"},
/* A652 */ { "CYRILLIC CAPITAL LETTER IOTIFIED YAT",NULL},
/* A653 */ { "CYRILLIC SMALL LETTER IOTIFIED YAT","	x (cyrillic small letter yat - 0463)"},
/* A654 */ { "CYRILLIC CAPITAL LETTER REVERSED YU",NULL},
/* A655 */ { "CYRILLIC SMALL LETTER REVERSED YU","	x (cyrillic small letter yu - 044E)"},
/* A656 */ { "CYRILLIC CAPITAL LETTER IOTIFIED A",NULL},
/* A657 */ { "CYRILLIC SMALL LETTER IOTIFIED A",NULL},
/* A658 */ { "CYRILLIC CAPITAL LETTER CLOSED LITTLE YUS",NULL},
/* A659 */ { "CYRILLIC SMALL LETTER CLOSED LITTLE YUS","	x (cyrillic small letter little yus - 0467)"},
/* A65A */ { "CYRILLIC CAPITAL LETTER BLENDED YUS",NULL},
/* A65B */ { "CYRILLIC SMALL LETTER BLENDED YUS","	x (cyrillic small letter little yus - 0467)\n"
	"	x (cyrillic small letter big yus - 046B)"},
/* A65C */ { "CYRILLIC CAPITAL LETTER IOTIFIED CLOSED LITTLE YUS",NULL},
/* A65D */ { "CYRILLIC SMALL LETTER IOTIFIED CLOSED LITTLE YUS","	x (cyrillic small letter iotified little yus - 0469)"},
/* A65E */ { "CYRILLIC CAPITAL LETTER YN",NULL},
/* A65F */ { "CYRILLIC SMALL LETTER YN","	* Romanian Cyrillic"},
/* A660 */ { "CYRILLIC CAPITAL LETTER REVERSED TSE",NULL},
/* A661 */ { "CYRILLIC SMALL LETTER REVERSED TSE",NULL},
/* A662 */ { "CYRILLIC CAPITAL LETTER SOFT DE",NULL},
/* A663 */ { "CYRILLIC SMALL LETTER SOFT DE",NULL},
/* A664 */ { "CYRILLIC CAPITAL LETTER SOFT EL",NULL},
/* A665 */ { "CYRILLIC SMALL LETTER SOFT EL",NULL},
/* A666 */ { "CYRILLIC CAPITAL LETTER SOFT EM",NULL},
/* A667 */ { "CYRILLIC SMALL LETTER SOFT EM",NULL},
/* A668 */ { "CYRILLIC CAPITAL LETTER MONOCULAR O",NULL},
/* A669 */ { "CYRILLIC SMALL LETTER MONOCULAR O","	* used in words based on the root for 'eye'"},
/* A66A */ { "CYRILLIC CAPITAL LETTER BINOCULAR O",NULL},
/* A66B */ { "CYRILLIC SMALL LETTER BINOCULAR O","	* used in the dual of words based on the root for 'eye'"},
/* A66C */ { "CYRILLIC CAPITAL LETTER DOUBLE MONOCULAR O",NULL},
/* A66D */ { "CYRILLIC SMALL LETTER DOUBLE MONOCULAR O","	* used in the dual of words based on the root for 'eye'"},
/* A66E */ { "CYRILLIC LETTER MULTIOCULAR O","	* used in the epithet 'many-eyed'"},
/* A66F */ { "COMBINING CYRILLIC VZMET","	* used with Cyrillic and Glagolitic letters to indicate abbreviation\n"
	"	x (combining cyrillic titlo - 0483)\n"
	"	x (combining cyrillic pokrytie - 0487)"},
/* A670 */ { "COMBINING CYRILLIC TEN MILLIONS SIGN","	x (combining cyrillic millions sign - 0489)"},
/* A671 */ { "COMBINING CYRILLIC HUNDRED MILLIONS SIGN",NULL},
/* A672 */ { "COMBINING CYRILLIC THOUSAND MILLIONS SIGN",NULL},
/* A673 */ { "SLAVONIC ASTERISK",NULL},
/* A674 */ { "COMBINING CYRILLIC LETTER UKRAINIAN IE",NULL},
/* A675 */ { "COMBINING CYRILLIC LETTER I",NULL},
/* A676 */ { "COMBINING CYRILLIC LETTER YI",NULL},
/* A677 */ { "COMBINING CYRILLIC LETTER U",NULL},
/* A678 */ { "COMBINING CYRILLIC LETTER HARD SIGN",NULL},
/* A679 */ { "COMBINING CYRILLIC LETTER YERU",NULL},
/* A67A */ { "COMBINING CYRILLIC LETTER SOFT SIGN",NULL},
/* A67B */ { "COMBINING CYRILLIC LETTER OMEGA",NULL},
/* A67C */ { "COMBINING CYRILLIC KAVYKA","	* indicates an alternative reading to part of a word\n"
	"	x (combining breve - 0306)\n"
	"	x (combining kavyka above right - 1DF6)\n"
	"	x (combining kavyka above left - 1DF7)"},
/* A67D */ { "COMBINING CYRILLIC PAYEROK","	* may indicate an omitted yer\n"
	"	x (combining vertical tilde - 033E)"},
/* A67E */ { "CYRILLIC KAVYKA","	* used to mark off word that has alternative reading\n"
	"	x (low kavyka - 2E47)"},
/* A67F */ { "CYRILLIC PAYEROK","	* may indicate an omitted yer\n"
	"	x (vertical tilde - 2E2F)"},
/* A680 */ { "CYRILLIC CAPITAL LETTER DWE",NULL},
/* A681 */ { "CYRILLIC SMALL LETTER DWE",NULL},
/* A682 */ { "CYRILLIC CAPITAL LETTER DZWE",NULL},
/* A683 */ { "CYRILLIC SMALL LETTER DZWE",NULL},
/* A684 */ { "CYRILLIC CAPITAL LETTER ZHWE",NULL},
/* A685 */ { "CYRILLIC SMALL LETTER ZHWE",NULL},
/* A686 */ { "CYRILLIC CAPITAL LETTER CCHE",NULL},
/* A687 */ { "CYRILLIC SMALL LETTER CCHE",NULL},
/* A688 */ { "CYRILLIC CAPITAL LETTER DZZE",NULL},
/* A689 */ { "CYRILLIC SMALL LETTER DZZE",NULL},
/* A68A */ { "CYRILLIC CAPITAL LETTER TE WITH MIDDLE HOOK",NULL},
/* A68B */ { "CYRILLIC SMALL LETTER TE WITH MIDDLE HOOK",NULL},
/* A68C */ { "CYRILLIC CAPITAL LETTER TWE",NULL},
/* A68D */ { "CYRILLIC SMALL LETTER TWE",NULL},
/* A68E */ { "CYRILLIC CAPITAL LETTER TSWE",NULL},
/* A68F */ { "CYRILLIC SMALL LETTER TSWE",NULL},
/* A690 */ { "CYRILLIC CAPITAL LETTER TSSE",NULL},
/* A691 */ { "CYRILLIC SMALL LETTER TSSE",NULL},
/* A692 */ { "CYRILLIC CAPITAL LETTER TCHE",NULL},
/* A693 */ { "CYRILLIC SMALL LETTER TCHE",NULL},
/* A694 */ { "CYRILLIC CAPITAL LETTER HWE",NULL},
/* A695 */ { "CYRILLIC SMALL LETTER HWE",NULL},
/* A696 */ { "CYRILLIC CAPITAL LETTER SHWE",NULL},
/* A697 */ { "CYRILLIC SMALL LETTER SHWE",NULL},
/* A698 */ { "CYRILLIC CAPITAL LETTER DOUBLE O",NULL},
/* A699 */ { "CYRILLIC SMALL LETTER DOUBLE O",NULL},
/* A69A */ { "CYRILLIC CAPITAL LETTER CROSSED O",NULL},
/* A69B */ { "CYRILLIC SMALL LETTER CROSSED O",NULL},
/* A69C */ { "MODIFIER LETTER CYRILLIC HARD SIGN","	# <super> 044A"},
/* A69D */ { "MODIFIER LETTER CYRILLIC SOFT SIGN","	# <super> 044C"},
/* A69E */ { "COMBINING CYRILLIC LETTER EF",NULL},
/* A69F */ { "COMBINING CYRILLIC LETTER IOTIFIED E",NULL},
/* A6A0 */ { "BAMUM LETTER A",NULL},
/* A6A1 */ { "BAMUM LETTER KA",NULL},
/* A6A2 */ { "BAMUM LETTER U",NULL},
/* A6A3 */ { "BAMUM LETTER KU",NULL},
/* A6A4 */ { "BAMUM LETTER EE",NULL},
/* A6A5 */ { "BAMUM LETTER REE",NULL},
/* A6A6 */ { "BAMUM LETTER TAE",NULL},
/* A6A7 */ { "BAMUM LETTER O",NULL},
/* A6A8 */ { "BAMUM LETTER NYI",NULL},
/* A6A9 */ { "BAMUM LETTER I",NULL},
/* A6AA */ { "BAMUM LETTER LA",NULL},
/* A6AB */ { "BAMUM LETTER PA",NULL},
/* A6AC */ { "BAMUM LETTER RII",NULL},
/* A6AD */ { "BAMUM LETTER RIEE",NULL},
/* A6AE */ { "BAMUM LETTER LEEEE",NULL},
/* A6AF */ { "BAMUM LETTER MEEEE",NULL},
/* A6B0 */ { "BAMUM LETTER TAA",NULL},
/* A6B1 */ { "BAMUM LETTER NDAA",NULL},
/* A6B2 */ { "BAMUM LETTER NJAEM",NULL},
/* A6B3 */ { "BAMUM LETTER M",NULL},
/* A6B4 */ { "BAMUM LETTER SUU",NULL},
/* A6B5 */ { "BAMUM LETTER MU",NULL},
/* A6B6 */ { "BAMUM LETTER SHII",NULL},
/* A6B7 */ { "BAMUM LETTER SI",NULL},
/* A6B8 */ { "BAMUM LETTER SHEUX",NULL},
/* A6B9 */ { "BAMUM LETTER SEUX",NULL},
/* A6BA */ { "BAMUM LETTER KYEE",NULL},
/* A6BB */ { "BAMUM LETTER KET",NULL},
/* A6BC */ { "BAMUM LETTER NUAE",NULL},
/* A6BD */ { "BAMUM LETTER NU",NULL},
/* A6BE */ { "BAMUM LETTER NJUAE",NULL},
/* A6BF */ { "BAMUM LETTER YOQ",NULL},
/* A6C0 */ { "BAMUM LETTER SHU",NULL},
/* A6C1 */ { "BAMUM LETTER YUQ",NULL},
/* A6C2 */ { "BAMUM LETTER YA",NULL},
/* A6C3 */ { "BAMUM LETTER NSHA",NULL},
/* A6C4 */ { "BAMUM LETTER KEUX",NULL},
/* A6C5 */ { "BAMUM LETTER PEUX",NULL},
/* A6C6 */ { "BAMUM LETTER NJEE",NULL},
/* A6C7 */ { "BAMUM LETTER NTEE",NULL},
/* A6C8 */ { "BAMUM LETTER PUE",NULL},
/* A6C9 */ { "BAMUM LETTER WUE",NULL},
/* A6CA */ { "BAMUM LETTER PEE",NULL},
/* A6CB */ { "BAMUM LETTER FEE",NULL},
/* A6CC */ { "BAMUM LETTER RU",NULL},
/* A6CD */ { "BAMUM LETTER LU",NULL},
/* A6CE */ { "BAMUM LETTER MI",NULL},
/* A6CF */ { "BAMUM LETTER NI",NULL},
/* A6D0 */ { "BAMUM LETTER REUX",NULL},
/* A6D1 */ { "BAMUM LETTER RAE",NULL},
/* A6D2 */ { "BAMUM LETTER KEN",NULL},
/* A6D3 */ { "BAMUM LETTER NGKWAEN",NULL},
/* A6D4 */ { "BAMUM LETTER NGGA",NULL},
/* A6D5 */ { "BAMUM LETTER NGA",NULL},
/* A6D6 */ { "BAMUM LETTER SHO",NULL},
/* A6D7 */ { "BAMUM LETTER PUAE",NULL},
/* A6D8 */ { "BAMUM LETTER FU",NULL},
/* A6D9 */ { "BAMUM LETTER FOM",NULL},
/* A6DA */ { "BAMUM LETTER WA",NULL},
/* A6DB */ { "BAMUM LETTER NA",NULL},
/* A6DC */ { "BAMUM LETTER LI",NULL},
/* A6DD */ { "BAMUM LETTER PI",NULL},
/* A6DE */ { "BAMUM LETTER LOQ",NULL},
/* A6DF */ { "BAMUM LETTER KO",NULL},
/* A6E0 */ { "BAMUM LETTER MBEN",NULL},
/* A6E1 */ { "BAMUM LETTER REN",NULL},
/* A6E2 */ { "BAMUM LETTER MEN",NULL},
/* A6E3 */ { "BAMUM LETTER MA",NULL},
/* A6E4 */ { "BAMUM LETTER TI",NULL},
/* A6E5 */ { "BAMUM LETTER KI",NULL},
/* A6E6 */ { "BAMUM LETTER MO","	* also used for digit one"},
/* A6E7 */ { "BAMUM LETTER MBAA","	* also used for digit two"},
/* A6E8 */ { "BAMUM LETTER TET","	* tèt\n"
	"	* also used for digit three"},
/* A6E9 */ { "BAMUM LETTER KPA","	* also used for digit four"},
/* A6EA */ { "BAMUM LETTER TEN","	* tèn\n"
	"	* also used for digit five"},
/* A6EB */ { "BAMUM LETTER NTUU","	* also used for digit six"},
/* A6EC */ { "BAMUM LETTER SAMBA","	* also used for digit seven"},
/* A6ED */ { "BAMUM LETTER FAAMAE","	* also used for digit eight"},
/* A6EE */ { "BAMUM LETTER KOVUU","	* also used for digit nine"},
/* A6EF */ { "BAMUM LETTER KOGHOM","	* also used for digit zero\n"
	"	* formerly used for number ten"},
/* A6F0 */ { "BAMUM COMBINING MARK KOQNDON",NULL},
/* A6F1 */ { "BAMUM COMBINING MARK TUKWENTIS",NULL},
/* A6F2 */ { "BAMUM NJAEMLI",NULL},
/* A6F3 */ { "BAMUM FULL STOP",NULL},
/* A6F4 */ { "BAMUM COLON",NULL},
/* A6F5 */ { "BAMUM COMMA",NULL},
/* A6F6 */ { "BAMUM SEMICOLON",NULL},
/* A6F7 */ { "BAMUM QUESTION MARK",NULL},
/* A6F8 */ { NULL,NULL},
/* A6F9 */ { NULL,NULL},
/* A6FA */ { NULL,NULL},
/* A6FB */ { NULL,NULL},
/* A6FC */ { NULL,NULL},
/* A6FD */ { NULL,NULL},
/* A6FE */ { NULL,NULL},
/* A6FF */ { NULL,NULL}
};

UN_DLL_LOCAL
static const struct unicode_nameannot una_00_A7[] = {
/* A700 */ { "MODIFIER LETTER CHINESE TONE YIN PING",NULL},
/* A701 */ { "MODIFIER LETTER CHINESE TONE YANG PING",NULL},
/* A702 */ { "MODIFIER LETTER CHINESE TONE YIN SHANG",NULL},
/* A703 */ { "MODIFIER LETTER CHINESE TONE YANG SHANG",NULL},
/* A704 */ { "MODIFIER LETTER CHINESE TONE YIN QU",NULL},
/* A705 */ { "MODIFIER LETTER CHINESE TONE YANG QU",NULL},
/* A706 */ { "MODIFIER LETTER CHINESE TONE YIN RU",NULL},
/* A707 */ { "MODIFIER LETTER CHINESE TONE YANG RU",NULL},
/* A708 */ { "MODIFIER LETTER EXTRA-HIGH DOTTED TONE BAR",NULL},
/* A709 */ { "MODIFIER LETTER HIGH DOTTED TONE BAR",NULL},
/* A70A */ { "MODIFIER LETTER MID DOTTED TONE BAR",NULL},
/* A70B */ { "MODIFIER LETTER LOW DOTTED TONE BAR",NULL},
/* A70C */ { "MODIFIER LETTER EXTRA-LOW DOTTED TONE BAR",NULL},
/* A70D */ { "MODIFIER LETTER EXTRA-HIGH DOTTED LEFT-STEM TONE BAR",NULL},
/* A70E */ { "MODIFIER LETTER HIGH DOTTED LEFT-STEM TONE BAR",NULL},
/* A70F */ { "MODIFIER LETTER MID DOTTED LEFT-STEM TONE BAR",NULL},
/* A710 */ { "MODIFIER LETTER LOW DOTTED LEFT-STEM TONE BAR",NULL},
/* A711 */ { "MODIFIER LETTER EXTRA-LOW DOTTED LEFT-STEM TONE BAR",NULL},
/* A712 */ { "MODIFIER LETTER EXTRA-HIGH LEFT-STEM TONE BAR",NULL},
/* A713 */ { "MODIFIER LETTER HIGH LEFT-STEM TONE BAR",NULL},
/* A714 */ { "MODIFIER LETTER MID LEFT-STEM TONE BAR",NULL},
/* A715 */ { "MODIFIER LETTER LOW LEFT-STEM TONE BAR",NULL},
/* A716 */ { "MODIFIER LETTER EXTRA-LOW LEFT-STEM TONE BAR",NULL},
/* A717 */ { "MODIFIER LETTER DOT VERTICAL BAR",NULL},
/* A718 */ { "MODIFIER LETTER DOT SLASH",NULL},
/* A719 */ { "MODIFIER LETTER DOT HORIZONTAL BAR",NULL},
/* A71A */ { "MODIFIER LETTER LOWER RIGHT CORNER ANGLE",NULL},
/* A71B */ { "MODIFIER LETTER RAISED UP ARROW",NULL},
/* A71C */ { "MODIFIER LETTER RAISED DOWN ARROW",NULL},
/* A71D */ { "MODIFIER LETTER RAISED EXCLAMATION MARK","	x (latin letter retroflex click - 01C3)\n"
	"	x (modifier letter retroflex click with retroflex hook - 107B9)"},
/* A71E */ { "MODIFIER LETTER RAISED INVERTED EXCLAMATION MARK",NULL},
/* A71F */ { "MODIFIER LETTER LOW INVERTED EXCLAMATION MARK",NULL},
/* A720 */ { "MODIFIER LETTER STRESS AND HIGH TONE",NULL},
/* A721 */ { "MODIFIER LETTER STRESS AND LOW TONE",NULL},
/* A722 */ { "LATIN CAPITAL LETTER EGYPTOLOGICAL ALEF",NULL},
/* A723 */ { "LATIN SMALL LETTER EGYPTOLOGICAL ALEF","	x (hebrew letter alef - 05D0)\n"
	"	x (arabic letter alef with hamza above - 0623)"},
/* A724 */ { "LATIN CAPITAL LETTER EGYPTOLOGICAL AIN",NULL},
/* A725 */ { "LATIN SMALL LETTER EGYPTOLOGICAL AIN","	x (modifier letter left half ring - 02BF)\n"
	"	x (hebrew letter ayin - 05E2)\n"
	"	x (arabic letter ain - 0639)\n"
	"	x (latin letter ain - 1D25)\n"
	"	x (modifier letter small ain - 1D5C)\n"
	"	x (latin small letter glottal i - A7BD)"},
/* A726 */ { "LATIN CAPITAL LETTER HENG","	x (cyrillic capital letter en with hook - 04C7)\n"
	"	x (latin capital letter h with descender - 2C67)"},
/* A727 */ { "LATIN SMALL LETTER HENG","	* uvular fricative, see 0281\n"
	"	* also used in Bantu linguistics for the voiced alveolar lateral fricative, see 026E\n"
	"	x (latin small letter heng with hook - 0267)\n"
	"	x (latin small letter h with palatal hook - A795)\n"
	"	x (modifier letter small heng - AB5C)"},
/* A728 */ { "LATIN CAPITAL LETTER TZ",NULL},
/* A729 */ { "LATIN SMALL LETTER TZ","	= tezh digraph\n"
	"	* palatoalveolar affricate [ts]\n"
	"	x (latin small letter ts digraph - 02A6)\n"
	"	x (latin small letter tesh digraph - 02A7)"},
/* A72A */ { "LATIN CAPITAL LETTER TRESILLO",NULL},
/* A72B */ { "LATIN SMALL LETTER TRESILLO","	* uvular ejective stop [qʼ]\n"
	"	x (turned digit three - 218B)"},
/* A72C */ { "LATIN CAPITAL LETTER CUATRILLO",NULL},
/* A72D */ { "LATIN SMALL LETTER CUATRILLO","	* velar ejective stop [kʼ]\n"
	"	x (digit four - 0034)"},
/* A72E */ { "LATIN CAPITAL LETTER CUATRILLO WITH COMMA",NULL},
/* A72F */ { "LATIN SMALL LETTER CUATRILLO WITH COMMA","	* alveolar ejective affricate [tsʼ]"},
/* A730 */ { "LATIN LETTER SMALL CAPITAL F",NULL},
/* A731 */ { "LATIN LETTER SMALL CAPITAL S",NULL},
/* A732 */ { "LATIN CAPITAL LETTER AA","	x (modifier letter small capital aa - 10780)"},
/* A733 */ { "LATIN SMALL LETTER AA",NULL},
/* A734 */ { "LATIN CAPITAL LETTER AO",NULL},
/* A735 */ { "LATIN SMALL LETTER AO",NULL},
/* A736 */ { "LATIN CAPITAL LETTER AU",NULL},
/* A737 */ { "LATIN SMALL LETTER AU",NULL},
/* A738 */ { "LATIN CAPITAL LETTER AV",NULL},
/* A739 */ { "LATIN SMALL LETTER AV",NULL},
/* A73A */ { "LATIN CAPITAL LETTER AV WITH HORIZONTAL BAR",NULL},
/* A73B */ { "LATIN SMALL LETTER AV WITH HORIZONTAL BAR",NULL},
/* A73C */ { "LATIN CAPITAL LETTER AY",NULL},
/* A73D */ { "LATIN SMALL LETTER AY",NULL},
/* A73E */ { "LATIN CAPITAL LETTER REVERSED C WITH DOT","	x (greek capital reversed dotted lunate sigma symbol - 03FF)"},
/* A73F */ { "LATIN SMALL LETTER REVERSED C WITH DOT","	x (greek small reversed dotted lunate sigma symbol - 037D)"},
/* A740 */ { "LATIN CAPITAL LETTER K WITH STROKE",NULL},
/* A741 */ { "LATIN SMALL LETTER K WITH STROKE",NULL},
/* A742 */ { "LATIN CAPITAL LETTER K WITH DIAGONAL STROKE",NULL},
/* A743 */ { "LATIN SMALL LETTER K WITH DIAGONAL STROKE",NULL},
/* A744 */ { "LATIN CAPITAL LETTER K WITH STROKE AND DIAGONAL STROKE",NULL},
/* A745 */ { "LATIN SMALL LETTER K WITH STROKE AND DIAGONAL STROKE",NULL},
/* A746 */ { "LATIN CAPITAL LETTER BROKEN L",NULL},
/* A747 */ { "LATIN SMALL LETTER BROKEN L",NULL},
/* A748 */ { "LATIN CAPITAL LETTER L WITH HIGH STROKE","	x (latin capital letter l with bar - 023D)"},
/* A749 */ { "LATIN SMALL LETTER L WITH HIGH STROKE","	x (latin small letter l with bar - 019A)"},
/* A74A */ { "LATIN CAPITAL LETTER O WITH LONG STROKE OVERLAY","	x (latin capital letter o with middle tilde - 019F)"},
/* A74B */ { "LATIN SMALL LETTER O WITH LONG STROKE OVERLAY","	x (latin small letter o with stroke - 00F8)\n"
	"	x (latin small letter barred o - 0275)"},
/* A74C */ { "LATIN CAPITAL LETTER O WITH LOOP",NULL},
/* A74D */ { "LATIN SMAL