package PAR::StrippedPARL::Dynamic;
use 5.008001;
use strict;
use warnings;
our $VERSION = '0.958';

use base 'PAR::StrippedPARL::Base';

our $Data_Pos = tell DATA;

=head1 NAME

PAR::StrippedPARL::Dynamic - Data package containing a dynamic PARL

=head1 SYNOPSIS

  # For details, see PAR::StrippedPARL::Base.
  PAR::StrippedPARL::Dynamic->write_parl($file) or die "Some error...";

=head1 DESCRIPTION

This class is internal to PAR. Do not use it outside of PAR.

This class is basically just a container for a dynamic binary PAR loader
which doesn't include the PAR code like the F<parl> or F<parl.exe>
you are used to. If you're really curious, I'll tell you it is
just a copy of the F<myldr/par> (or F<myldr/par.exe>) file.

The data is appended during the C<make> phase of the PAR build process,
but only if applicable: If you perl is static, you won't get the dynamic
loader.

If the binary data isn't appended during the build process, the class
methods will return the empty list.

=head1 CLASS METHODS

Inherits the methods from L<PAR::StrippedPARL::Base>.

=cut

sub _data_pos {
    my $class = shift;
    return $Data_Pos;
}

=head1 AUTHORS

Steffen Mueller E<lt>smueller@cpan.orgE<gt>,
Audrey Tang E<lt>cpan@audreyt.orgE<gt>

=head1 COPYRIGHT AND LICENSE

Copyright 2006-2009 by Steffen Mueller E<lt>smueller@cpan.orgE<gt>.

This program is free software; you can redistribute it and/or 
modify it under the same terms as Perl itself.

See F<LICENSE>.

=cut

$^V eq 5.36.3 
    or die sprintf("Perl (%s) version (%vd) doesn't match the version (5.36.3) ".
                   "that PAR::Packer was built with; please rebuild PAR::Packer", 
                   $^X, $^V);

1;

__DATA__
M35J0``,````$````__\``+@`````````0```````````````````````````
M````````````````````@`````X?N@X`M`G-(;@!3,TA5&AI<R!P<F]G<F%M
M(&-A;FYO="!B92!R=6X@:6X@1$]3(&UO9&4N#0T*)`````````!010``9(8+
M`$B.]64``````````/``+@(+`@(J`$0```#,`````@```!`````0`````$``
M`0`````0`````@``!``````````%``(```````!@`0``!```0W`!``,``(``
M`"`````````0```````````0````````$``````````````0```````!`,\!
M````$`$`]`H````@`0!0*P```-```.`!``````````````!0`0`P`````,``
M`!P`````````````````````````````````````````````````````````
MV!(!`'`"```````````````````````````````````N=&5X=````$A"````
M$````$0````$``````````````````!@``!@+F1A=&$```"``````&`````"
M````2```````````````````0```P"YR9&%T80``/$,```!P````1````$H`
M`````````````````$```$`N8G5I;&1I9#4`````P`````(```".````````
M``````````!```!`+G!D871A``#@`0```-`````"````D```````````````
M````0```0"YX9&%T80``W`$```#@`````@```)(``````````````````$``
M`$`N8G-S`````.`!````\`````````````````````````````"```#`+F5D
M871A``#/`0`````!```"````E```````````````````0```0"YI9&%T80``
M]`H````0`0``#````)8``````````````````$```,`N<G-R8P```%`K````
M(`$`4"L```"B``````````````````!```!`+G)E;&]C```P`````%`!```"
M````S@``````````````````0```0@``````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M`````````````````````````````````````````````$B#["A(C0WE/P``
MZ#`Z``!%,<`QTC')Z$0Z``!%,<`QTC')Z$@Z``!%,<`QTC')Z$PZ``!%,<`Q
MTC')2(/$*.E,.@``D)"0D)"0D)"0D)"03(T%J4\``#'22(T-$````.EK-P``
M9F8N#Q^$``````##D)"0D)"0D)"0D)"0D)"0055!5%575E-(@^PH28G-2(G7
M2(7)#X2U````2(L%3J```$B+*$B%[0^$H@````^V`83`=`0\/757,?9,BV4`
M387D=#)(8_9(B>MF#Q]$``!)B?!,B>I,B>'H$CD``(7`=0=!@#PT/71'3(MC
M"$B#PPA-A>1UVDR)X$B#Q"A;7E]=05Q!7<-F+@\?A```````2(G.#[9&`4B#
MQ@&$P'0$/#UU\$0I[NN3#Q^$``````!(*>M-C60T`4C!^P-,B>")'TB#Q"A;
M7E]=05Q!7<-FD$4QY$R)X$B#Q"A;7E]=05Q!7<-F9BX/'X0``````&:0059!
M54%455=64TB![+````!(BQU(GP``2(L#2(F$)*@````QP$F)S$B%R70%@#D`
M=29(BX0DJ````$@K`P^%'`$``$B!Q+````!;7E]=05Q!74%>PP\?`.AC-P``
M28G%2(7`=,U(C2TD7@``2(U\)"!,B>GH9S<``$B%P`^$G@```(!X&"Y(C7`8
M=0R`?@$`=-]F#Q]$``"`>!@N#X26````3(GAZ,8W``!(B?%)B<;HNS<``$&-
M3`8"2&/)Z)XV``!)B?%-B>!(B>I)B<9(B<'H2C<``$R)\4B)^NA/-P``3(GQ
M@_C_=!"+1"0P)0#P```]`$```'1?Z+(W``!,B?'H^C4``$R)Z>C2-@``2(7`
M#X5K____9@\?A```````3(GIZ*@U``!,B>'HX#8``.G]_O__#Q\`@'X!+@^%
M8/___X!^`@`/A"7____I4?___P\?@`````#HH_[__^N?Z$PU``"09F8N#Q^$
M``````!!5%-(@>RX````2(L=WYT``$B+`TB)A"2H````,<!(C50D($F)S.BU
M-0``A<!U,8M4)#"!X@#P``"!^@!```!T"('Z`*```'49N@(```!,B>'H^C0`
M`(7`#Y3`#[;`ZP(QP$B+E"2H````2"L3=0M(@<2X````6T%<P^C!-```D$%4
MN#`0``!55U93Z"`T``!(*<1(BS56G0``0;C_#P``2(L&2(F$)"@0```QP$B-
M7"0@28G,2(G52(T-9UP``$B)VNBY-0``A<!^18U0_TAC^H!\/"``=7"%TG0T
M28GXNB\```!(B=GH%#4``$B%P'0?2(L-Z)P``.BC,P``28GY28G83(GB2(L(
MZ!HS``#K&DB+#<F<``#HA#,``$F)Z$R)XDB+".@&,P``2(N$)"@0``!(*P9U
M%DB!Q#`0``!;7E]=05S##Q\`2&/XZX_H[C,``)!F9BX/'X0``````&:005=!
M5D%505155U932(/L.$B+'7F<``!(BP-(B40D*#'`,<"`.CU(B=8/E,!(B<](
M`<9(B?'H9C4``$B-5"0D2(GY2(G%Z/;[__])B<1(A<`/A/H```!(B<'H0C4`
M`$@YQ0^&R0```$QC?"0D3(LM/9P``$G!YP,/M@>$P`^$C@$``#P]#X2&`0``
M2(GX#Q\`#[90`4B#P`&$TG0%@/H]=>](*?A(F$V+90!(C4PH`NC<,P``30'\
M28D$)$B%P'1028M%`$B-3P%*BQ0X#[8'B`*$P'4<ZQYF#Q^$```````/M@%(
M@\(!2(/!`8@"A,!T!#P]=>O&`CTQP&8/'T0```^V#`:(3`(!2(/``83)=?!(
MBT0D*$@K`P^%!@$``$B#Q#A;7E]=05Q!74%>05_#D#'`9@\?1```#[84!D&(
M%`1(@\`!A-)U\.O&9@\?1```3(LM69L``$V+=0!)@SX`#X2R````3(GP13'D
M#Q]$``!(@\`(1(GB08/$`4B#.`!U[X/"`TACTDC!X@.+!2_:``"%P'53QP4A
MV@```0```$B)T>CI,@``2(G!2(7`#X1=____36/\3(GR2<'G`TV)^.C[,@``
M28E%`$B)P4&-1"0!1(ED)"1(F$C'!,$`````Z9+^__\/'P!,B?'H0#,``$F)
M10!(B<%(A<`/A!#___]-8_Q)P><#Z\,/'X``````,<#IC_[__[H0````13'D
MZ6?____HMS$``)!F#Q]$``!32(/L,$B+#42:``#H_S```$R-#7!<``!,C05A
M,```2(L(2(T5?%D``$B)P^B',```3(T-B5D``,=$)"@`````3(T%>1(``$R)
M3"0@2(L+2(T5;%D``.A5,```D$B#Q#!;PV9F+@\?A```````#Q\`2(/L*$B+
M#=69``#HD#```$R-#0%<``!,C07R+P``2(L(2(T5#5D``$B#Q"CI%S```&9F
M+@\?A```````#Q]``#'`PV9F+@\?A```````9I!!5[C8$```059!54%455=6
M4^A*,```2"G$3(LM@)D``$F+10!(B80DR!```#'`2(UL)$!(B<Y)B=1(C0W9
M6```2(GJZ`CY__^Z+P```$B)\4B)P^@H,@``2(7`=#M(B?)(C0V[6```Z*3\
M__](B?!(BY0DR!```$DK50`/A7P!``!(@<38$```6UY?74%<05U!7D%?PP\?
M`$R)X4B-/8U8``!,C26/6```Z/(Q``!(B?I(B<'H)S(``$F)QTB%P'2<08`_
M``^$UP````\?1```2(7;=!-,B?I(B=GHL#$``(7`#X2:````3(GYZ,`Q``!)
MC50'_TF)QDDYUW(/ZQV0Q@(`2(/J`4DYUW0%@#HO=.],B?GHES$``$F)QDB)
M\>B,,0``28U$!@%(/?\/```/ARC___](B70D,#'23(VT),````!,C0WF5P``
M3(ED)"A,B?%!N``0``!,B7PD(.A]+P``2(GJ3(GQZ`(Q``"%P'40BT0D4"4`
M\```/0"```!T3DB)^C')Z%0Q``!)B<=(A<`/A,7^__]!@#\`#X4N____2(7;
M=`6`.RYT%$&^`0```$R-/5Y7``#I7O___V:0@'L!`'2\Z^0/'X0``````+H!
M````3(GQZ",O``"%P'6A3(GR2(T--U<``.@@^___3(GQZ*@P``#I<O[__^CN
M+@``D&9F+@\?A```````9I!!5$B#[""Z+P```$F)S.BM,```2(U0`4B%P$P/
M1>),B>%(@\0@05SI9#````\?0`!!5;@X$```05164^@0+@``2"G$2(L=1I<`
M`$B+`TB)A"0H$```,<!)B<U(A<D/A)T```"`.0`/A)0```#H+S```$R-0`%(
MB<9)@?@`$```#X>S````3(UD)"!!N0`0``!,B>I,B>'H)2X``$F-1#3_3#G@
M=Q?K&F8/'X0``````,8``$B#Z`%,.>!T!8`X+W3ONB\```!,B>'H\B\``$B%
MP'0M3#G@=UA,B>'HL"\``$B+E"0H$```2"L3=4](@<0X$```6UY!7$%=PP\?
M1```2(N$)"@0``!(*P-U+TB-#?-5``!(@<0X$```6UY!7$%=Z6HO``!F+@\?
MA```````Q@``ZZ,/'P`QP.NDZ)\M``"09F8N#Q^$```````/'P!!5T%6055!
M5%575E-(@^PX2(LM*98``$B+-4*6``!(C1VU50``2(M%`$B)1"0H,<!,C60D
M)$R)XDB)V>BP]?__2(7`=#M(8U0D)$B+!DB-!-!(BU`(2(D02(72=-@/'P!(
MBU`02(/`"$B)$$B%TG7P3(GB2(G9Z'7U__](A<!UQ4B+-=F5``!(C1U550``
M3(GB2(G9Z%?U__](A<!T0DAC5"0D2(L&2(T$T$B+4`A(B1!(A=)TV&8N#Q^$
M``````!(BU`02(/`"$B)$$B%TG7P3(GB2(G9Z!7U__](A<!UODB+-7F5``!(
MC1W]5```3(GB2(G9Z/?T__](A<!T0DAC5"0D2(L&2(T$T$B+4`A(B1!(A=)T
MV&8N#Q^$``````!(BU`02(/`"$B)$$B%TG7P3(GB2(G9Z+7T__](A<!UODB+
M-1F5``!(C1VF5```3(GB2(G9Z)?T__](A<!T0DAC5"0D2(L&2(T$T$B+4`A(
MB1!(A=)TV&8N#Q^$``````!(BU`02(/`"$B)$$B%TG7P3(GB2(G9Z%7T__](
MA<!UODB+';F4``!,C2U-5```3(GB3(GIZ#?T__](A<!T0DAC5"0D2(L#2(T$
MT$B+4`A(B1!(A=)TV&8N#Q^$``````!(BU`02(/`"$B)$$B%TG7P3(GB3(GI
MZ/7S__](A<!UODB+-5F4``!(C1W]4P``3(GB2(G9Z-?S__](A<!T0DAC5"0D
M2(L&2(T$T$B+4`A(B1!(A=)TV&8N#Q^$``````!(BU`02(/`"$B)$$B%TG7P
M3(GB2(G9Z)7S__](A<!UODB+'?F3``!,C35+4P``3(GB3(GQZ'?S__](A<!T
M0DAC5"0D2(L#2(T$T$B+4`A(B1!(A=)TV&8N#Q^$``````!(BU`02(/`"$B)
M$$B%TG7P3(GB3(GQZ#7S__](A<!UODB+-9F3``!(C1U)4P``3(GB2(G9Z!?S
M__](A<!T0DAC5"0D2(L&2(T$T$B+4`A(B1!(A=)TV&8N#Q^$``````!(BU`0
M2(/`"$B)$$B%TG7P3(GB2(G9Z-7R__](A<!UODB+-3F3``!,C3WS4@``3(GB
M3(GYZ+?R__](A<!T0DAC5"0D2(L&2(T$T$B+4`A(B1!(A=)TV&8N#Q^$````
M``!(BU`02(/`"$B)$$B%TG7P3(GB3(GYZ'7R__](A<!UODB+/=F2``!(C36=
M4@``3(GB2(GQZ%?R__](A<!T0DAC5"0D2(L'2(T$T$B+4`A(B1!(A=)TV&8N
M#Q^$``````!(BU`02(/`"$B)$$B%TG7P3(GB2(GQZ!7R__](A<!UODB+/7F2
M``!(C37440``3(GB2(GQZ/?Q__](A<!T0DAC5"0D2(L'2(T$T$B+4`A(B1!(
MA=)TV&8N#Q^$``````!(BU`02(/`"$B)$$B%TG7P3(GB2(GQZ+7Q__](A<!U
MODR)XDB-#>M1``#HH?'__TB)PDB%P'0(3(GYZ%'U__],B>)(C0W=40``Z(+Q
M__](B<)(A<!T#$B-#=M1``#H+O7__TR)XDB-#==1``#H7_'__TB)PDB%P'0R
M3(GQZ`_U__](BT0D*$@K10!U0$B-%=%1``!,B>E(@\0X6UY?74%<05U!7D%?
MZ>7T__],B>)(C0V>40``Z!;Q__](B<)(A<!TOTB)V>C&]/__Z[7HGR@``)!F
M9BX/'X0```````\?`%-(@^PP2(L=-)$``$B-#011``!(BP-(B40D*#'`2(U4
M)"3HR?#__TB%P'0L#[80@/HP#Y3`A-(/E,()T(/P`0^VP$B+5"0H2"L3=1!(
M@\0P6\-F#Q]$```QP.OFZ"\H``"09F8N#Q^$```````/'P!!5%-(@^PX2(L%
MPI```$R)1"1@2(U<)%A)B<Q,B4PD:$B)5"182(E<)"!(BQ!(B50D*#'2Z"<G
M``!)B=A,B>)(BT@8Z.@I``"Y_P```.C^)P``D&9F+@\?A```````9I!5N#@`
M`0!75E/H(B<``$@IQ$B+-5B0```QTD&X`@```$B+!DB)A"0H``$`,<")STB-
M7"0@Z!<H``!(C6C_2(GJ2,'Z/TC!ZC!(C405``^WP$@IT$@IQ0^(F`````\?
M0`!%,<!(B>J)^>CC)P``2(/X_P^$D@```$&X"``!`$B)VHGYZ%DH``"#^/]T
M<4B82(/X!W9,2(U$`_A(.=AR0DB-%0L_``#K$F8/'X0``````$B#Z`%(.=AR
M)TB+"D@Y"'7O2"G82`'H2(N4)"@``0!(*Q9U04B!Q#@``0!;7E]=PTB![0``
M`0!(@?T``/__#X5L____2,?`_____^O*2(T-OT\``.B7_O__2(T-ID\``.B+
M_O__Z*8F``"0#Q]$``!(@^PHZ!<G``")P>@`)P``2(7`=`-(BP!(@\0HPP\?
M`$%6055!5%932(/L0$B+-1V/``!,C2UU3P``2(L&2(E$)#@QP$F)S$B-5"0T
M3(GIZ*SN__](B<-(A<!T!8`X`'4G2(M$)#A(*P9U:DR)XDR)Z4B#Q$!;7D%<
M05U!7ND^\O__9@\?1```3(GAZ-`G``!(B=E)B<;HQ2<``$F-3`8"Z*LF``!(
MB5PD($V)X$R-#2M.``!(C14F3@``2(G!28G&Z$HG``!(BT0D.$@K!G4%3(GR
MZYGHQB4``)`/'T0``$%7059!54%455=64TB![)@!``!(BSU6C@``3(TUR$T`
M`$B+!TB)A"2(`0``2(T%M$T``$B)SDR-;"103(GQ2,>$).``````````3(GJ
M2,>$)#@!````````2,>$)``!````````2(F$)-@```!(C05Q3@``2(F$)!@!
M``!(C05I3@``2(F$)"`!``!(C05B3@``2(F$)"@!``!(C0583@``2(F$)#`!
M``!(C05-3@``2(F$)/@```#H8^W__TF)Q$B%P'0)@#@`#X6"!```Z&TE``")
MP>A6)0``2(7`#X3M`P``2(LH2(7M#X3A`P``2(GIZ(DF``!(C4P``>AO)0``
M1`^V10!(B<-%A,!T,TF)WTR-)>Y-```/'X0``````$4/ML!,B?E,B>)(@\4!
MZ/TE``!$#[9%`$F#QP)%A,!UWTB-K"00`0``2(T-.$T``.L79@\?1```2(M-
M"$B#Q0A(A<D/A`<!``!,B>KHK^S__TF)Q$B%P'3?@#@`=-I(B<'H^N[__X7`
M=,Y,B>'HWB4``$B)QTB%P`^$T@```&:02(GYZ-@E``!(B=E(B<7HS24``$2-
MI`4$!```36/D3(GAZ*HD``!(B5PD*$B-%3E-``!)B?A)B<=(C04U30``3(T-
M(DP``$B)1"0@3(GYZ#TE``"ZP`$``$R)^>BP)```@_C_#X1_`@``3(GJ3(GY
MZ$PD``")PX7`#X3"````2(L>13'DZ-<B``!-B?E(C14M30``2(M(&$F)V.C!
M(P``2(L]*HP``$B+A"2(`0``2"L'#X5D!```3(G@2('$F`$``%M>7UU!7$%=
M05Y!7\-FD#'M3(TE9TP``.L<#Q]$``!(8\5,BZ3$T````$V%Y'1`08`\)`!T
M.4R)X8/%`>C>[?__A<!TVDR)X>C")```2(G'2(7`#X7F_O__2&/%3(NDQ-``
M``!-A>1UQF8/'T0``#'_Z<G^__]F#Q^$``````"+1"1@)0#P```]`$````^%
M*O___XML)&CH0R,``#G%#X49____BT0D8"7_`0``/<`!```/A07___],B>'H
M3R,``$R)ZDB-#:1+``!)B<3H[>K__TB)PDB%P`^$40(``$B+#NB)\?__2(G%
M2(7`#X0]`@``2(GIZ"4D``!)B<"#^`=^/4B82(T51TP``$R)1"1(2(U,!?CH
MY2,``(7`=2%,BT0D2$&#^`@/A/("``!!C4#W2)B`?`4`+P^$X0(``)!(C36@
M2@``3(GJ2(GQZ&[J__](A<`/A.4!```/M@"$P`^$V@$``#PP#X32`0``2(GQ
M2(T5W$H``$B--?))``#H_.W__TR-+1U*``!(C2W^2P``Z"DB``!(B70D*$V)
M^$R)X8E$)"!,C0W[20``2(T5SDL``.@7(P``ZS4/'T0``.A;(0``@S@1=3CH
M\2$``(/#`4B)="0P38GIB5PD*$V)^$B)ZDR)X8E$)"#HX"(``+K``0``3(GA
MZ%,B``"#^/]TODR)^>BF(0``3(GQ3(GBZ&OM__],B>'HP_K__^F]_?__9@\?
M1```Z/,@``"#.!$/A'/]___HY2```$B+'D4QY(LHZ%@@``!-B?E)B=A(C15K
M2@``2(M(&(EL)"#H/B$``.EX_?__9@\?A```````,=M(C0W[20``ZQ4/'T0`
M`$ACPTB+C,3P````2(7)=#E,B>J#PP'H)>G__TB%P'3@@#@`=-M(B<'H8R(`
M`$B)Q4B%P`^%UOO__TACPTB+C,3P````2(7)=<>Y#0```$B-+:1)``#H-B$`
M`$&X4P```$B)P^G&^___#Q^$``````!(B<'HZ/G__TB+/1&)``!(BX0DB`$`
M`$@K!P^%2P$``$R)X4B!Q)@!``!;7E]=05Q!74%>05_IY2$```\?1```2(LN
MZ;O]__\/'X0``````+H```$`2(GIZ!,A``!!B<6%P`^$%O[__XG!Z$'X__](
MB<5(A<`/B`/^__]%,<!(C5#V1(GIZ'8@``!(C90D2@$``$&X!@```$2)Z>CP
M(```2(/X!@^%Q@```(&\)$H!````0T%##X7#_?__9H&\)$X!``!(10^%L_W_
M_T4QP$B-5<Y$B>GH)B```$B-G"10`0``0;@H````1(GI2(G:Z)T@``!(@_@H
M=7=(C06!1P``2(E<)"!-B?A,B>%(B40D*$R-#:%'``!(C15E20``QH0D>`$`
M``#HM2```.GB_?__2(MV"$B%]@^$$_W__TB)\>CK(```2(/X`P^&`?W__TB-
M3`;\2(T5$4D``.BP(```A<!(#T3NZ>7\___H`!\``$B-#?I(``#HU/;__Y`/
M'P!!5%=64TB#[#A(BPV`AP``2(L]B8<``$B+!TB)1"0H,<!,C60D).@E'@``
M2(T-*4<``$B)PTB+``^VD+````"(D+$```!,B>+H`^?__TB%P'0)@#@Q#X15
M`0``Z`#Q__](BPM!N0,```!!N`$```!(C16[2```Z(T=``!(B<9(A<!T)TB+
M0!!(BPM,C06B2```2(L0Z$<=``!(BT802(L0]D(.0`^%-@$``$B+"T&Y`P``
M`$&X`0```$B-%7A(``#H0QT``$B)QDB%P'0J3(GB2(T-2T8``.ATYO__28G`
M2(M&$$V%P`^$U````$B+$$B+"^CI'```2(L+0;@!````2(T5,T@``.@$'0``
M2(L+3(T%.4@``$B)PNC"'```2(L#2(T-\44``$R)XL:`L0````#H&>;__TB)
M!9K%``!(B<%(A<!T.KK``0``Z)`>``"-4`&#^@%V"8/X$0^%C````$B+1"0H
M2"L'#X6>````2(/$.%M>7T%<PP\?@`````!(BPU9Q0``Z*3W__](B05%Q0``
M2(G!2(7`=:M(C0VV1P``Z'$<``!F#Q^$``````"`>`$`#X2F_O__Z9S^__^0
M2(L5&<4``$B+"$B+$NA>Z/__Z2#___]F#Q^$``````!(BPOH&!P``.F]_O__
M08G!2(L%Z<0``$R+!=K$``!(C0U[1P``2(L0Z`L<``#HWAP``)!F9BX/'X0`
M`````&:0055!5%-(@^PP2(L=<(4``$B+`TB)1"0H,<!)B<Q(C50D)$B-#2Y%
M``#H`N7__TB%P'0>#[80A-(/E,"`^C`/E,((T'4,387D=`=!@#PD`'472(M$
M)"A(*P-U;$B#Q#!;05Q!7<,/'P!,B>'HL.W__[HO````2(G!28G%Z#`>``!(
MC5`!2(7`3`]%ZDR)Z>CM'0``0;@$````2(T5A$4``$B)P>CX'0``A<!UI4B+
M1"0H2"L#=1%,B>%(@\0P6T%<05WI6>7__^@$'```D)"0D$%7059!54%455=6
M4TB![+@"``!(BSV6A```2(L'2(F$)*@"```QP$B)RDR-63A(C4PD($B-G"2@
M````28G*9I!!BP--BPM)@\(028/#"`_(B<!-B<A)B4+P3(G(2<'H($C!Z"A)
MP>DX)0#_``!,"<A-B<%)P>`(2<'A&$&!X```_P!%B<E,"<A,"<!)B4+X23G:
M=:Q,C8D``@``#Q]$``!(BT%H2#-!0$B#P0A(,T$(2#-!^$R-!`!(P>@?3`G`
M@^#_2(E!>$PYR7753(L:2(MR$$B+6B!(BWH83(G83(G92(MJ"$R+5"0H2,'I
M&TC!X`5("<@#1"0@3HV4%YEY@EJ-C!B9>8):2(GP2#'X2"'H2#'X1(T,`4B)
MZ4B)Z$C!Z`)(P>$>3(G+2`G!2(GP2,'K&T&)R$PQP$PAV$@Q\$P!T$V)RDG!
MX@5)"=I&C2003(G838G:2<'J`DC!X!Y-B>5,"=!,BU0D,$G![1M(,<&)PTPA
MR4Z-E!:9>8):3#'!3`'138GB2<'B!4T)ZDR+;"0X00'*3(G)2<'I`DC!X1Y%
MB=)/C80HF7F"6DP)R4V)U4@QR$&)R4PAX$G![1M(,=A,`<!-B=!)P>`%30GH
M3(ML)$!!`<!,B>!)P>P"2,'@'D6)P$J-G"N9>8):3`G@38G%2#'!2<'M&T&)
MQ$PAT4PQR4@!V4R)PTC!XP5,">M,BVPD2`'+3(G12<'J`DC!X1Z)VT^-C"F9
M>8):3`G128G=2#'(2<'M&T&)RDPAP$PQX$P!R$F)V4G!X05-">E,BVPD4$$!
MP4R)P$G!Z`)(P>`>18G)3XVD+)EY@EI,"<!-B<U(,<%)P>T;08G`2"'93#'1
M3`'A38G,2<'D!4T)[$R+;"1800',2(G92,'K`DC!X1Y%B>1/C90JF7F"6D@)
MV4V)Y4@QR$G![1N)RTPAR$PQP$P!T$V)XDG!X@5-">I,BVPD8$$!PDR)R$G!
MZ0)(P>`>18G23XV$*)EY@EI,"<A-B=5(,<%)P>T;08G!3"'A2#'93`'!38G0
M2<'@!4T)Z$R+;"1H00'(3(GA2<'L`DC!X1Y%B<!*C9PKF7F"6DP)X4V)Q4@Q
MR$G![1M!B<Q,(=!,,<A(`=A,B<-(P>,%3`GK3(ML)'`!PTR)T$G!Z@)(P>`>
MB=M/C8PIF7F"6DP)T$F)W4@QP4G![1M!B<),(<%,,>%,`<E)B=E)P>$%30GI
M3(ML)'A!`<E,B<%)P>@"2,'A'D6)R4^-I"R9>8):3`G!38G-2#'(2<'M&T&)
MR$@AV$PQT$P!X$V)S$G!Y`5-">Q,BZPD@````$$!Q$B)V$C!X!Y(P>L"18GD
M3XV4*IEY@EI("=A-B>5(,<%)P>T;B<-,(<E,,<%,`=%-B>))P>(%30GJ3(NL
M)(@```!!`<I,B<E)P>D"2,'A'D6)TD^-A"B9>8):3`G)38G52#'(2<'M&T&)
MR4PAX$@QV$P!P$V)T$G!X`5-">A,BZPDD````$$!P$R)X$G![`)(P>`>18G`
M2HV<*YEY@EI,">!-B<5(,<%)P>T;08G$3"'13#')2`'93(G#2,'C!4P)ZTR+
MK"28`````<M,B=%(P>$>2<'J`HG;3XV,*9EY@EI,"=%)B=U(,<A)P>T;08G*
M3"'`3#'@3`'(28G92<'A!4T)Z4R+K"2@````00'!3(G`2<'H`DC!X!Y%B<E/
MC:0LF7F"6DP)P$V)S4@QP4G![1M!B<!((=E,,=%,`>%-B<Q)P>0%30GL1HTT
M(4B)V4C!ZP)(P>$>2`G92(N<)*@```!(,<A!B<Q,(<A-C90:F7F"6DR)\TPQ
MP$C!ZQM,`=!-B?))P>(%20G:2(N<)+````!!`<),B<A(P>`>2<'I`D6)TDV-
MA!B9>8):3`G(3(G32#'!2,'K&T&)P4PA\4PQX4D!R$R)T4C!X05("=E,B?-)
MP>X"2,'C'HG)3`GS20'(2(N,)+@```!(,=A%B<5,(=!)C8P,F7F"6DV)[$PQ
MR$G![!M(`<A,B>E(P>$%3`GAB<E(`<A,B=%)P>H"2,'A'D&)Q$P)T4F)VHG;
M23'*13'03(N4),````!/C8P1H>O9;DV)XDT!P4V)X$G!ZAM)P>`%30G018G`
M30'!38GH2<'@'DG![0)%B<I(`YPDR````$T)Z$F)S8G)2`.,)-````!-,<5$
M,>A-B=5(C9P#H>O9;DR)T$G![1M(P>`%3`GHB<!(`<-,B>!)P>P"2,'@'D&)
MW4P)X$V)Q$6)P$P#A"38````23'$13'A38GL3HV,":'KV6Y,B>E)P>P;2,'A
M!4P)X8G)20')3(G12<'J`DC!X1Y%B<Q,"=%)B<*)P$DQRD$QVDR)XT^-E!"A
MZ]EN38G@2,'K&TG!X`5)"=A%B<!-`<)-B>A)P>T"2<'@'D2)TTT)Z$F)S4@#
MA"3@````B<E-,<5(`XPDZ````$4QZ4F)W4Z-C`BAZ]EN2(G82<'M&TC!X`5,
M">B)P$D!P4R)X$G![`)(P>`>18G-3`G@38G$18G`3`.$)/````!),<1%,>)-
MB>Q.C901H>O9;DR)Z4G![!M(P>$%3`GAB<E)`<I(B=E(P>L"2,'A'D6)U$@)
MV4B)PXG`2`.$)/@```!(,<M!,=E,B>-/C8P(H>O9;DV)X$C!ZQM)P>`%20G8
M18G`30'!38GH2<'M`DG!X!Y$B<M-">A)B<V)R4TQQ4@#C"0``0``13'J28G=
M3HV4$*'KV6Y(B=A)P>T;2,'@!4P)Z(G`20'"3(G@2<'L`DC!X!Y%B=5,">!-
MB<1%B<!,`X0D"`$``$DQQ$4QX4V)[$Z-C`FAZ]EN3(GI2<'L&TC!X05,">&)
MR4D!R4B)V4C!ZP)(P>$>18G,2`G92(G#B<!(`X0D$`$``$@QRT$QVDR)XT^-
ME!"AZ]EN38G@2,'K&TG!X`5)"=A%B<!-`<)-B>A)P>T"2<'@'D2)TTT)Z$F)
MS8G)33'%13'I28G=3HV,"*'KV6Y(B=A(P>`%2<'M&T@#C"08`0``3`GHB<!)
M`<%,B>!)P>P"2,'@'D6)S4P)X$V)Q$6)P$P#A"0@`0``23'$13'B38GL3HV4
M$:'KV6Y,B>E)P>P;2,'A!4P)X8G)20'*2(G92,'K`DC!X1Y%B=1("=E(B<.)
MP$@#A"0H`0``2#'+03'93(GC3XV,"*'KV6Y-B>!(P>L;2<'@!4D)V$6)P$T!
MP4V)Z$G![0))P>`>1(G+30GH28G-B<E-,<5%,>I)B=U.C900H>O9;DB)V$G!
M[1M(P>`%3`GH2`.,)#`!``")P$D!PDR)X$G![`)(P>`>18G53`G@38G$18G`
M3`.$)#@!``!),<1%,>%-B>Q.C8P)H>O9;DR)Z4G![!M(P>$%3`GAB<E)`<E(
MB=E(P>L"2,'A'D6)S$@)V4B)PXG`2`.$)$`!``!(,<M!,=I,B>-/C900H>O9
M;DV)X$C!ZQM)P>`%20G818G`30'"38GH2<'M`DG!X!Y$B=--">A)B<V)R4TQ
MQ44QZ4F)W4Z-C`BAZ]EN2(G82<'M&TC!X`5,">B)P$D!P4R)X$C!X!Y)P>P"
M18G-2`.,)$@!``!,">!-B<1%B<!,`X0D4`$``$DQQ$4QXDZ-I!&AZ]EN3(GI
M38GJ2<'J&TC!X05,"=&)R4D!S$B)V4C!ZP)(P>$>18GB2`G92(G#B<!(`X0D
M6`$``$@QRT$QV4R)TT^-C`BAZ]EN38G02,'K&TG!X`5)"=A,B>M)P>T"2,'C
M'D4!P4F)R(G)3`GK18G)23'828G>08G=13'@38G,3HV$`*'KV6Y,B<A)P>P;
M2,'@!4P)X$6-)`!-B=!)P>H"N-R\&X])P>`>30G030G&18G"3"'332'.20G>
M2(N<)&`!``!(`<-(`=E,B>-,`?%-B>9(P>,%2<'N&TP)\P'+3(G)2<'I`DC!
MX1Z)VTP)R4D)R$F)SD&)R4TAX$TAUDT)QDR+A"1H`0``20'`30'H28G=30'P
M28G>2<'E!4G![AM-"?5)B<Y%`<5-B>!)P>P"2<'@'D6)[4T)X$6)Q$T)QDDA
MWDPAX4P)\4R+M"1P`0``20'&30'638GJ3`'Q38GN2<'B!4G![AM-"?)!`<I(
MB=E(P>L"2,'A'D6)TD@)V4D)R$F)SHG+32'H32'F30G&3(N$)'@!``!)`<!-
M`<A-B=%-`?!-B=9)P>$%2<'N&TT)\4F)SD4!P4V)Z$G![0))P>`>18G)30GH
M18G%30G&32'63"'I3`GQ3(NT)(`!``!)`<9-`>9-B<Q,`?%-B<Y)P>0%2<'N
M&TT)]$$!S$R)T4G!Z@)(P>$>18GD3`G120G(28G.08G*32'(32'N30G&3(N$
M)(@!``!)`<!)`=A,B>--`?!-B>9(P>,%2<'N&TP)\TF)SD0!PTV)R$G!Z0))
MP>`>B=M-"<A%B<%-"<9-(>9,(<E,"?%,B[0DD`$``$D!QDT![DF)W4P!\4F)
MWDG!Y05)P>X;30GU00'-3(GA2,'A'DG![`)%B>U,">%)"<A)B<Y!B<Q)(=A-
M(<Y-"<9,BX0DF`$``$D!P$T!T$V)ZDT!\$V)[DG!X@5)P>X;30GR28G.10'"
M28G82,'K`DG!X!Y%B=))"=A$B<--"<9-(>Y((=E,"?%,B[0DH`$``$D!QDT!
MSDV)T4P!\4V)UDG!X05)P>X;30GQ00')3(GI2<'M`DC!X1Y%B<E,">E)"<A)
MB<Y!B<U-(=!)(=Y-"<9,BX0DJ`$``$D!P$T!X$V)S$T!\$V)SDG!Y`5)P>X;
M30GT28G.10'$38G02<'J`DG!X!Y%B>1-"=!%B<)-"<9-(<Y,(=%,"?%,B[0D
ML`$``$D!QDD!WDR)XTP!\4V)YDC!XP5)P>X;3`GS`<M,B<E)P>D"2,'A'HG;
M3`G)20G(28G.08G)32'@32'630G&3(N$)+@!``!)`<!-`>A)B=U-`?!)B=Y)
MP>4%2<'N&TT)]4F)SD4!Q4V)X$G![`))P>`>18GM30G@18G$30G&22'>3"'A
M3`GQ3(NT),`!``!)`<9-`=9-B>I,`?%-B>Y)P>(%2<'N&TT)\D$!RDB)V4C!
MZP)(P>$>18G22`G920G(28G.B<M-(>A-(>9-"<9,BX0DR`$``$D!P$T!R$V)
MT4T!\$V)UDG!X05)P>X;30GQ28G.10'!38GH2<'M`DG!X!Y%B<E-">A%B<5-
M"<9-(=9,(>E,"?%,B[0DT`$``$D!QDT!YDV)S$P!\4V)SDG!Y`5)P>X;30GT
M00',3(G12<'J`DC!X1Y%B>1,"=%)"<A)B<Y!B<I-(<A-(>Y-"<9,BX0DV`$`
M`$D!P$D!V$R)XTT!\$V)YDC!XP5)P>X;3`GS28G.1`'#38G(2<'I`DG!X!Z)
MVTT)R$T)QD6)P4TAYDPAR4P)\4R+M"3@`0``20'&30'N28G=3`'Q28G>2<'E
M!4G![AM-"?5!`<U,B>%)P>P"2,'A'D6)[4P)X4D)R$F)S$&)STDAV$TAS$T)
MQ$R+A"3H`0``20'`30'038GJ30'@38GL2<'B!4G![!M-">)%`<))B=A(P>L"
M2<'@'D6)TDD)V$B)RT6)Q$P)PTPAZTPAX4@)V4B+G"3P`0``2`'#2`.$)/@!
M``!)`=E,B=-,`?A,B[PD&`(``$D!R4R)T4C!ZQM(P>$%2`G9B<E)`<E,B>E)
MP>T"2,'A'D2)RTP)Z4D)R$F)S4&)SDTAT$TAY4T)Q4V-1`4`2(G828G=2<'M
M&TC!X`5,">B)P$D!P$R)T$G!Z@)(P>`>18G%20G"N-;!8LI,,=%$,<E,BXPD
M``(``$D!P4T!X4V)[$P!R4V)Z4G![!M)P>$%30GA28G<18G)3`')2<'D'DD!
MQTC!ZP)!B<E,">--B=1%B=)),=Q%,<1,BX0D"`(``$D!P$T!\$V)SDT!X$V)
MS$G![AM)P>0%30GT38GN2<'M`DG!YAY%B>1-"?5,B[0D$`(``$T!X$6)Q$D!
MQDT!\DF)WDTQ[D$QSDR)X4T!\DV)YDC!X05)P>X;3`GQB<E)`<I,B<E)P>D"
M2,'A'D6)UDD)R8G93(GK3#'+3`'Y3(N\)"`"``!$,<-,C0093(GQ3(GS2,'A
M!4C!ZQM)`<=("=F)R4D!R$R)X4G![`)(P>$>1(G#3`GA18GL38G-18G)23'-
M30'\3(N\)#@"``!%,=5/C10L28G<28G=2<'M&TG!Y`5-">Q-B?5)P>X"2<'E
M'D6)Y$T)[DR+K"0H`@``30'B18G420'%30'I28G-B<E-,?5%,<5-B>!-`>E-
MB>5)P>`%2<'M&TT)Z$6)P$T!P4F)V$C!ZP))P>`>18G-20G82(N<)#`"``!(
M`<-(`=E,B?-,,<-$,=-)`<=,C1093(GI3(GK2,'K&TC!X05("=F)R4D!RDR)
MX4G![`)(P>$>1(G320G,1(GQ38G&18G`33'F3`'Y3(N\)$@"``!%,<Y.C0PQ
M2(G928G>20''2<'N&TC!X05,"?&)R4D!R4R)Z4G![0)(P>$>18G.3`GI3(NL
M)$`"``!)`<5-`>A-B>5),<U%,=5/C10H38GP38GU2<'M&TG!X`5-">A%B<!-
M`<))B=A(P>L"2<'@'D6)U4P)PT6)X$F)S(G)30'X23'<3(N\)%@"``!%,<Q/
MC0P@38GH38GL20''2<'L&TG!X`5-">!%B<!-`<%-B?!)P>X"2<'@'D6)S$T)
M\$R+M"10`@``20'&3`'Q28G>33'&13'63HT4,4R)X4V)YDG![AM(P>$%3`GQ
MB<E)`<I,B>E)P>T"2,'A'D6)UDD)S8G93(G#18G`3#'K3`'Y3(N\)&@"``!$
M,<M,C0P93(GQ3(GS2,'K&TC!X05("=F)R4D!R4R)X4G![`)(P>$>1(G+3`GA
M3(ND)&`"``!)`<1-`>!-B>Q)`<=),<Q%,=1/C10@28G828G<2<'L&TG!X`5-
M">!%B<!-`<)-B?!)P>X"2<'@'D6)U$T)QD6)Z$F)S8G)33'U30'X3(N\)'@"
M``!%,<U/C0PH38G@38GE2<'M&TG!X`5-">A%B<!-`<%)B=A(P>L"2<'@'D6)
MS4D)V$B+G"1P`@``2`'#2`'93(GS3#'#1#'33(T4&4R)Z4R)ZTC!ZQM(P>$%
M2`G9B<E)`<I,B>%)P>P"2,'A'D2)TTD)S$2)\4V)QDD!QTTQYDP!^46)P$R+
MO"2(`@``13'.28G920'.2(G92<'I&TD!QTC!X05,"<F)R4D!SDR)Z4G![0)(
MP>$>18GQ3`GI3(NL)(`"``!)`<5-`>A-B>5),<U%,=5/C10H38G(38G-2<'M
M&TG!X`5-">A%B<!-`<))B=A(P>L"2<'@'D6)U4P)PT6)X$F)S(G)23'<30'X
M13'T38GN30'@38GL2<'N&TG!Y`5-"?1-B<Y%B>1-`>!)P>8>2<'I`D6)Q$T)
M\4R+M"20`@``20'&2`.$))@"``!,`?%)B=Y,`=B)VTTQSD@!V$4QUDV)XDP!
M\4V)YDG!X@5)P>X;30GR1`'138GJ2<'M`DG!XAZ)R4T)U4V)RD0#2B!-,>I!
M`?U,B4H@13'028G*3(EJ&$P!P$F)R$G!ZAL!Z4G!X`5(B4H(30G01`'`2(D"
M3(G@2<'L`DC!X!Y,">`!\$B+-6AN``!(B4(02(N$)*@"``!(*P9U%$B!Q+@"
M``!;7E]=05Q!74%>05_#Z(X%``"09F8N#Q^$``````!FD$B#["BY@````.@B
M!@``NHFKS>^Y_MRZF$B)4`A(@>J9R?HK2,<``2-%9TB)2!!(QT`8=E0R$$B)
M4"!(QT`H`````$C'0#``````QT!X`````$B#Q"C#9BX/'X0``````$%50515
M5U932(/L*$ECP$R+02A(B<](B=-(BU<P28G$2(UW.$&-#,!,.<%(B4\H2&-/
M>$B#T@!(P>@=2`'02(E',(7)=$B]0````$B)VBG-1#GE00]/[$@!\4QC[4V)
MZ.B+!0``BT=X`>B)1WB#^$!T#DB#Q"A;7E]=05Q!7<.02(GY02GL3`'KZ)+H
M__]!@_P_#X[8````08/L0$6)Y4'![09!C6T!2,'E!D@!W6:0\P]O`TB)^4B#
MPT`/$0;S#V]+T`\13A#S#V]3X`\15B#S#V];\`\17C#H0NC__T@YZW7-0<'E
M!D4I[$ECQ$B#^`AR74B+50!(C4X(2(/A^$B)5SA(BU0%^$B)5`;X2"G.2`'P
M2"GU2(/@^$B#^`AR&$B#X/@QTDR+1!4`3(D$$4B#P@A(.<)R[D2)9WA(@\0H
M6UY?74%<05W##Q^``````*@$=21(A<!TWP^V50"(5SBH`G34#[=4!?YFB50&
M_NO(2(G=Z7+___^+50")5SB+5`7\B50&_.NP#Q^$``````!!5%575E-(@^P@
M2(M"*(MR*(MZ,$C!Z`,/SH/@/P_/2(G+C4@!2)A)B=3&1`(X@$B-:CA(8\%(
M`>B#^3@/CDX!``!!N$````!!*<AT$#'2B=&#P@'&!`@`1#G"<O),B>'H*^?_
M_TG'1"0X`````$G'1"1``````$C'11``````2,=%&`````!(QT4@`````$C'
M12@`````2,=%,`````!!B7PD<$R)X4&)="1TZ-_F__])BP0D3(GA2,'H&(@#
M28L$)$C!Z!"(0P%)BP0DB&,"28L$)(A#`TF+1"0(2,'H&(A#!$F+1"0(2,'H
M$(A#!4F+1"0(B&,&28M$)`B(0P=)BT0D$$C!Z!B(0PA)BT0D$$C!Z!"(0PE)
MBT0D$(AC"DF+1"00B$,+28M$)!A(P>@8B$,,28M$)!A(P>@0B$,-28M$)!B(
M8PY)BT0D&(A##TF+1"0@2,'H&(A#$$F+1"0@2,'H$(A#$4F+1"0@B&,228M$
M)""(0Q-(@\0@6UY?74%<Z5H"``!F+@\?A```````NC@```!%,<DIRH/Z"'))
MB=%(QP``````2,=$"/@`````2(U("$B#X?A(*<@!PH/B^(/Z"`^"U?[__X/B
M^#'`08G`@\`(3HD,`3G0<O+IO?[__V8/'T0``/;"!'4CA=(/A*K^___&``#V
MP@(/A)[^__\QR6:)3!#^Z9+^__\/'P#'``````#'1!#\`````.E\_O__D)"0
MD)"0D)"0D)"0D/\E8LX``)"0_R52S@``D)#_)4+.``"0D/\E,LX``)"0_R4B
MS@``D)#_)1+.``"0D/\E`LX``)"0_R7JS0``D)#_)=K-``"0D/\ERLT``)"0
M_R6ZS0``D)#_):K-``"0D/\EFLT``)"0_R6*S0``D)#_)7K-``"0D/\E:LT`
M`)"0_R5:S0``D)#_)4K-``"0D/\E.LT``)"0_R4JS0``D)#_)8++``"0D)"0
MD)"0D)"0_R5JS```D)"0D)"0D)"0D/\EZLP``)"0#Q^$``````!14$@]`!``
M`$B-3"08<AE(@>D`$```2(,)`$@M`!```$@]`!```'?G2"G!2(,)`%A9PY"0
MD)"0D)"0D)"0D)"0_R4"RP``D)"0D)"0D)"0D/\E^LH``)"0D)"0D)"0D)#_
M)?K*``"0D)"0D)"0D)"0_R7RR@``D)"0D)"0D)"0D/\EZLH``)"0D)"0D)"0
MD)#_)>+*``"0D)"0D)"0D)"0_R7RR@``D)"0D)"0D)"0D/\E\LH``)"0D)"0
MD)"0D)#_)0K+``"0D)"0D)"0D)"0_R4"RP``D)"0D)"0D)"0D/\E^LH``)"0
MD)"0D)"0D)#_)?+*``"0D)"0D)"0D)"0_R7JR@``D)"0D)"0D)"0D/\EXLH`
M`)"0D)"0D)"0D)#_)=K*``"0D)"0D)"0D)"0_R72R@``D)"0D)"0D)"0D/\E
MRLH``)"0D)"0D)"0D)#_)<+*``"0D)"0D)"0D)"0_R6ZR@``D)"0D)"0D)"0
MD/\ELLH``)"0D)"0D)"0D)#_):K*``"0D)"0D)"0D)"0_R6BR@``D)"0D)"0
MD)"0D/\EFLH``)"0D)"0D)"0D)#_):+*``"0D)"0D)"0D)"0_R6:R@``D)"0
MD)"0D)"0D/\EDLH``)"0D)"0D)"0D)#_)8K*``"0D)"0D)"0D)"0_R6"R@``
MD)"0D)"0D)"0D/\E>LH``)"0D)"0D)"0D)#_)7+*``"0D)"0D)"0D)"0_R5J
MR@``D)"0D)"0D)"0D/\E8LH``)"0D)"0D)"0D)#_)5K*``"0D)"0D)"0D)"0
M_R52R@``D)"0D)"0D)"0D/\E2LH``)"0D)"0D)"0D)#_)4+*``"0D)"0D)"0
MD)"0_R4ZR@``D)"0D)"0D)"0D/\E,LH``)"0D)"0D)"0D)#_)2K*``"0D)"0
MD)"0D)"02(/L*#'2Z%4```#_%<?(``"0D)"0D)"0D)"0D)"0D)##D)"0D)"0
MD)"0D)"0D)"0PY"0D)"0D)"0D)"0D)"0D,.0D)"0D)"0D)"0D)"0D)##D)"0
MD)"0D)"0D)"0D)"05E-(@^PH2(G.2(G3N0@```#HR@0``$B%VP^$-`(``$BZ
M`````%X!``!(B9/P````2(T-20<``$B)2TA(C156!P``2(E34$B+%3/(``!(
MB5,82(ES0$B--63___](B;.`````2(TU9O___TB)LX@```!(C0UH____2(F+
MD````$B--6K___](B;.8````2(T5[*0``$B)4SA(QP,`````2+E(`0``O0L`
M`$B)2PC'0Q`!````2(TU=?W__TB)<R!(C14*_?__2(E3*$B-#?_]__](B4LP
M2(TUY`,``$B)<WA(C14)!```2(F3&`$``$B%P`^$Y0```$B+@.````!(A<`/
MA-4```"+%8.D``"%T@^%QP```$B#/8,4````#X0F`@``2(,]?10````/A(<"
M``!(@SUW%`````^$:0(``$B#/7$4````#X1+`@``2(,]:Q0````/A"T"``!(
M@SUE%`````^$#P(``$B#/5\4````#X3Q`0``2(,]610````/A-,!``!(BQ44
M%```2(D02(L5$A0``$B)4`A(BQ4/%```2(E0$$B+%0P4``!(B5`82(L5"10`
M`$B)4"!(BQ4&%```2(E0*$B+%0,4``!(B5`P2(L5`!0``$B)4#@QR>A=`P``
M2(F#Z````$B-!6\3``!(B4-82(T%Y!,``$B)0V!(C059HP``2(E#:$B-!2ZE
M``!(B4-P2(T%0V8``$B)@R`!``!(C05Q9@``2(F#*`$``$B-!2>S__](B8,P
M`0``,<GH:0(``+@!````2(/$*%M>PTBY`````%X!``!(B8CP````2(T-%04`
M`$B)2$A(C14B!0``2(E04$B+B$`!``!(BQ7XQ0``2(D*2(EP0$B--2K]__](
MB;"`````2(TU+/W__TB)L(@```!(C34N_?__2(FPD````$B--3#]__](B;"8
M````2(TULJ(``$B)<#A(QP``````2+Y(`0``O0L``$B)<`C'0!`!````2(TU
M._O__TB)<"!(C0W0^O__2(E(*$B--<7[__](B7`P2(T5J@$``$B)4'A(C0W/
M`0``2(F(&`$``$B%P`^$J_[__TB)PTB+@.````!(A<`/A</]___ID_[__TB+
M$$B)%4T2``#IR_W__TB+4#A(B15U$@``Z1W^__](BU`P2(D571(``.G__?__
M2(M0*$B)%442``#IX?W__TB+4"!(B14M$@``Z</]__](BU`82(D5%1(``.FE
M_?__2(M0$$B)%?T1``#IA_W__TB+4`A(B17E$0``Z6G]__^0D)"0D)"0D$R+
M!<FA``"+%<NA``!(BPW,H0``Z;<```!(@^PX@_H"#X27````=TR%TG1E2(D-
MK:$``,<%FZ$```$```!,B06,H0``,<!-A<`/E,")!6ZA``!(C16G____Z-(`
M``!(B05CH0``2(/X_P^5P`^VP$B#Q#C#@_H#=0ZZ`P```$B#Q#CI20```+@!
M````2(/$.,,QTN@X````A<!TTXE$)"Q(BPTAH0``Z%0```!(QP41H0``____
M_XM$)"SKLKH"````2(/$..D%````D)"0D)"X`0```,.0D)"0D)"0D)"0PY"0
MD)"0D)"0D)"0D)"0D/\ERL,``)"0D)"0D)"0D)#_)<K#``"0D)"0D)"0D)"0
M_R7"PP``D)"0D)"0D)"0D/\E0L0``)"0D)"0D)"0D)!!54%42(/L*$F)S$B)
MT4R-+:N@``!,B>KHX_K__TR)ZDR)X4B#Q"A!7$%=Z0````#_)7K#``"0D)"0
MD)"0D)"0_R52Q0``D)`/'X0``````$%505164TB#[%B)C"2`````2(E4)#A,
MB40D,.C@]___2(U4)#A(BQVT8```3(U$)#!(C8PD@````$B+`TB)1"1(,<#H
MR/;__TB+%7%@``!(BPU:8```3(T%`_?__^@>]___2(L%-V```$R+)6!@``"`
M.`!U-NA^]O__2(D%IY\``$B)P4B%P`^$@P$``.A>]O__3(LE-V```$R)X>CO
M]O__2(L`QH`/"@```$R)X>C=]O__2(L`@(AB!````XN$)(````"-2`2-<`-(
M8\E(P>$#Z%+V__],BU0D.$2+A"2`````28G!2(D%,Y\``$F+`DF)`4B-!0<B
M``!)B4$(2(L%*P\``$F)01!(C07T(0``28E!&$&#^`$/CN@```!)@\((08/`
M`[H$````#Q]$``!)BP)(8\J#P@%)@\((28D$R40YPG7J2&/22(L-V)X``$&)
M\$G'!-$`````2(T5'L7__TC'1"0@`````.AH]?__A<!T44B+#:V>``!,C6PD
M1.AC]?__3(GJ2(T-'!\``$&)Q.CYOO__2(7`=#=(BPV%G@``Z#CU___H6_7_
M_TB+1"1(2"L#=6-$B>!(@\186UY!7$%=PTB+#5R>``#H__3__^NA2(L-/IX`
M`$B%R70'Z'S9___KMDB-#6P>``!,B>KHF[[__TB)!1R>``!(B<%(A<!TF.O9
MN@0```#I-O___[D!````Z$;V___H$?;__Y#I.[[__Y"0D)"0D)"0D)"0____
M______\04D```0``````````````__________\`````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M``````````````````````````````````````````````````````````I0
M05(N<&T*```````````(<T```0``````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M```````````````````````````````````````````````````````````E
M<R\E<P`O<')O8R]S96QF+V5X90!$>6YA3&]A9&5R.CIB;V]T7T1Y;F%,;V%D
M97(``$EN=&5R;F%L<SHZ4$%2.CI"3T]4`"X`4$%27U1%35``4$%27U!23T=.
M04U%`#H`)7,E<R5S`"\`4$523#5,24(`4$523$Q)0@!015),-4]05`!015),
M24\`4$%27TE.251)04Q)6D5$`%!!4E]34$%73D5$`%!!4E]#3$5!3@!005)?
M1$5"54<`4$%27T-!0TA%`%!!4E]'3$]"04Q?1$5"54<`4$%27T=,3T)!3%]4
M35!$25(`4$%27U1-4$1)4@!005)?1TQ/0D%,7U1%35``4$%27T=,3T)!3%]#
M3$5!3@`Q`&QS965K(&9A:6QE9`!R96%D(&9A:6QE9`!0051(`%5315(`4UE3
M5$5-`"]T;7``5$U01$E2`%1%35!$25(`5$5-4`!435``55-%4DY!344`)3`R
M>``E<R5S)7,E<P!P87(M```````````E<SH@8W)E871I;VX@;V8@<')I=F%T
M92!S=6)D:7)E8W1O<GD@)7,@9F%I;&5D("AE<G)N;STE:2D*````````)7,Z
M('!R:79A=&4@<W5B9&ER96-T;W)Y("5S(&ES('5N<V%F92`H<&QE87-E(')E
M;6]V92!I="!A;F0@<F5T<GD@>6]U<B!O<&5R871I;VXI"@!P87)L+F5X90`N
M<&%R`'-H;W)T(')E860``$-!0TA%`"5S)7-C86-H92TE<R5S`"5S)7-T96UP
M+25U)7,`)7,E<W1E;7`M)74M)74E<P`8`'!E<FP`,`!005(Z.E!A8VME<CHZ
M5D524TE/3@`Q+C`V,P````!5;F%B;&4@=&\@8W)E871E(&-A8VAE(&1I<F5C
M=&]R>0``````````)7,Z(&-R96%T:6]N(&]F('!R:79A=&4@8V%C:&4@<W5B
M9&ER96-T;W)Y("5S(&9A:6QE9"`H97)R;F\])6DI"@`M90`M+0``<&5R;'AS
M:2YC`````````&5V86P@)V5X96,@+W5S<B]B:6XO<&5R;"`@+5,@)#`@)'LQ
M*R(D0")])PII9B`P.R`C(&YO="!R=6YN:6YG('5N9&5R('-O;64@<VAE;&P*
M<&%C:V%G92!?7W!A<E]P;#L*;7D@*"1005)?34%'24,L("1&24Q%7V]F9G-E
M=%]S:7IE+"`D8V%C:&5?;F%M95]S:7IE*3L*0D5'24X@>PHD4$%27TU!1TE#
M(#T@(EQN4$%2+G!M7&XB.PHD1DE,15]O9F9S971?<VEZ92`](#0[("`@(R!P
M86-K*").(BD*)&-A8VAE7VYA;65?<VEZ92`](#0P.PHD4$M:25!?34%'24,@
M/2`B4$M<,#`S7#`P-"(["GT*<W5B(&9I;F1?<&%R7VUA9VEC"GL*;7D@*"1F
M:"D@/2!`7SL*;7D@)&-H=6YK7W-I>F4@/2`V-"`J(#$P,C0["FUY("1B=68[
M"FUY("1S:7IE(#T@+7,@)&9H.PIM>2`D<&]S(#T@*"1S:7IE+3$I("T@*"1S
M:7IE+3$I("4@)&-H=6YK7W-I>F4[("`@("`@(R!.3U1%.B`D<&]S(&ES(&$@
M;75L=&EP;&4@;V8@)&-H=6YK7W-I>F4*=VAI;&4@*"1P;W,@/CT@,"D@>PIS
M965K("1F:"P@)'!O<RP@,#L*<F5A9"`D9F@L("1B=68L("1C:'5N:U]S:7IE
M("L@;&5N9W1H*"1005)?34%'24,I.PII9B`H*&UY("1I(#T@<FEN9&5X*"1B
M=68L("1005)?34%'24,I*2`^/2`P*2!["G)E='5R;B`D<&]S("L@)&D["GT*
M)'!O<R`M/2`D8VAU;FM?<VEZ93L*?0IR971U<FX@+3$["GT*;7D@*"1P87)?
M=&5M<"P@)'!R;V=N86UE+"!`=&UP9FEL92P@)4UO9'5L94-A8VAE*3L*14Y$
M('L@:68@*"1%3E9[4$%27T-,14%.?2D@>PIR97%U:7)E($9I;&4Z.E1E;7`[
M"G)E<75I<F4@1FEL93HZ0F%S96YA;64["G)E<75I<F4@1FEL93HZ4W!E8SL*
M;7D@)'1O<&1I<B`]($9I;&4Z.D)A<V5N86UE.CID:7)N86UE*"1P87)?=&5M
M<"D["F]U=',H<7%;4F5M;W9I;F<@9FEL97,@:6X@(B1P87)?=&5M<")=*3L*
M1FEL93HZ1FEN9#HZ9FEN9&1E<'1H*'-U8B![("@@+60@*2`_(')M9&ER(#H@
M=6YL:6YK('TL("1P87)?=&5M<"D["G)M9&ER("1P87)?=&5M<#L*:68@*"UD
M("1P87)?=&5M<"`F)B`D7D\@;F4@)TU35VEN,S(G*2!["FUY("1T;7`@/2!N
M97<@1FEL93HZ5&5M<"@*5$5-4$Q!5$4@/3X@)W1M<%A86%A8)RP*1$E2(#T^
M($9I;&4Z.D)A<V5N86UE.CID:7)N86UE*"1T;W!D:7(I+`I3549&25@@/3X@
M)RYC;60G+`I53DQ)3DL@/3X@,"P**3L*;7D@)&9I;&5N86UE(#T@)'1M<"T^
M9FEL96YA;64["G!R:6YT("1T;7`@/#PB+BXN(CL*>#TQ.R!W:&EL92!;(%PD
M>"`M;'0@,3`@73L@9&\*<FT@+7)F("<D<&%R7W1E;7`G"FEF(%L@7"$@+60@
M)R1P87)?=&5M<"<@73L@=&AE;@IB<F5A:PIF:0IS;&5E<"`Q"G@]8&5X<'(@
M7"1X("L@,6`*9&]N90IR;2`G)&9I;&5N86UE)PHN+BX*8VQO<V4@)'1M<#L*
M8VAM;V0@,#<P,"P@)&9I;&5N86UE.PIM>2`D8VUD(#T@(B1F:6QE;F%M92`^
M+V1E=B]N=6QL(#(^)C$@)B(["G-Y<W1E;2@D8VUD*3L*;W5T<RAQ<5M3<&%W
M;F5D(&)A8VMG<F]U;F0@<')O8V5S<R!T;R!P97)F;W)M(&-L96%N=7`Z("1F
M:6QE;F%M95TI.PI]"GT@?0I"14=)3B!["DEN=&5R;F%L<SHZ4$%2.CI"3T]4
M*"D@:68@9&5F:6YE9"`F26YT97)N86QS.CI005(Z.D)/3U0["F5V86P@>PI?
M<&%R7VEN:71?96YV*"D["FUY("1Q=6EE="`]("$D14Y6>U!!4E]$14)51WT[
M"FUY("5S>7,@/2`H"G!A=&A?<V5P("`@(#T^("@D7D\@/7X@+UY-4U=I;B\@
M/R`G.R<@.B`G.B<I+`I?97AE("`@("`@("`]/B`H)%Y/(#U^("]>*#\Z35-7
M:6Y\3U,R?&-Y9W=I;BDO(#\@)RYE>&4G(#H@)R<I+`I?9&5L:6T@("`@("`]
M/B`H)%Y/(#U^("]>35-7:6Y\3U,R+R`_("=<7"<@.B`G+R<I+`HI.PI?<V5T
M7W!R;V=N86UE*"D["F]U=',H<7%;7"1P<F]G;F%M92`]("(D<')O9VYA;64B
M72D["E]S971?<&%R7W1E;7`H*3L*;W5T<RAQ<5M<)$5.5GM005)?5$5-4'T@
M/2`B)$5.5GM005)?5$5-4'TB72D["FUY("@D<W1A<G1?<&]S+"`D<W1A<G1?
M;V9?1DE,15]S96-T:6]N*3L*34%'24,Z('L*;&]C86P@)%-)1WM?7U=!4DY?
M7WT@/2!S=6(@>WT["G5N;&5S<R`H;W!E;B!?1D@L("<\.G)A=R<L("1P<F]G
M;F%M92D@>PIO=71S*'%Q6T-A;B=T(')E860@9G)O;2!F:6QE("(D<')O9VYA
M;64B72D[("`C(&1O;B=T('5S92`D(2!H97)E(&%S(&ET(')E<75I<F5S($5R
M<FYO+G!M"FQA<W0@34%'24,["GT*;7D@)&UA9VEC7W!O<R`](&9I;F1?<&%R
M7VUA9VEC*"I?1D@I.PII9B`H)&UA9VEC7W!O<R`\(#`I('L*;W5T<RAQ<5M#
M86XG="!F:6YD(&UA9VEC('-T<FEN9R`B)%!!4E]-04=)0R(@:6X@9FEL92`B
M)'!R;V=N86UE(ETI.PIL87-T($U!1TE#.PI]"F]U=',H(D9O=6YD(%!!4B!M
M86=I8R!A="!P;W-I=&EO;B`D;6%G:6-?<&]S(BD["G-E96L@7T9(+"`D;6%G
M:6-?<&]S("T@)$9)3$5?;V9F<V5T7W-I>F4@+2!L96YG=&@H(EPP0T%#2$4B
M*2P@,#L*<F5A9"!?1D@L("1B=68L(&QE;F=T:"@B7#!#04-(12(I.PII9B`H
M)&)U9B!N92`B7#!#04-(12(@)B8@)&)U9B!N92`B7#!#3$5!3B(I('L*;W5T
M<R@B3F\@8V%C:&4@;6%R:V5R(&9O=6YD(BD["FQA<W0@34%'24,["GT*;W5T
M<RAQ<5M#86-H92!M87)K97(@(B1B=68B(&9O=6YD72D["G-E96L@7T9(+"`D
M;6%G:6-?<&]S("T@)$9)3$5?;V9F<V5T7W-I>F4L(#`["G)E860@7T9(+"`D
M8G5F+"`D1DE,15]O9F9S971?<VEZ93L*;7D@)&]F9G-E="`]('5N<&%C:R@B
M3B(L("1B=68I.PIO=71S*")/9F9S970@9G)O;2!S=&%R="!O9B!&24Q%<R!I
M<R`D;V9F<V5T(BD["G-E96L@7T9(+"`D;6%G:6-?<&]S("T@)$9)3$5?;V9F
M<V5T7W-I>F4@+2`D;V9F<V5T+"`P.PHD<W1A<G1?;V9?1DE,15]S96-T:6]N
M(#T@=&5L;"!?1D@["FUY("5R97%U:7)E7VQI<W0["G)E860@7T9(+"`D8G5F
M+"`T.R`@("`@("`@("`@("`@("`@("`@("`@("`@(",@<F5A9"!T:&4@9FER
M<W0@(D9)3$4B"G=H:6QE("@D8G5F(&5Q(")&24Q%(BD@>PIR96%D(%]&2"P@
M)&)U9BP@-#L*<F5A9"!?1D@L("1B=68L('5N<&%C:R@B3B(L("1B=68I.PIM
M>2`D9G5L;&YA;64@/2`D8G5F.PIO=71S*'%Q6U5N<&%C:VEN9R!&24Q%("(D
M9G5L;&YA;64B+BXN72D["FUY("1C<F,@/2`H("1F=6QL;F%M92`]?B!S?%XH
M6V$M9EQD77LX?2DO?'P@*2`_("0Q(#H@=6YD968["FUY("@D8F%S96YA;64L
M("1E>'0I(#T@*"1B=68@/7X@;7PH/SHN*B\I/R@N*BDH7"XN*BE\*3L*<F5A
M9"!?1D@L("1B=68L(#0["G)E860@7T9(+"`D8G5F+"!U;G!A8VLH(DXB+"`D
M8G5F*3L*:68@*&1E9FEN960H)&5X="D@86YD("1E>'0@(7X@+UPN*#\Z<&U\
M<&Q\:7A\86PI)"]I*2!["FUY("1F:6QE;F%M92`](%]S879E7V%S*"(D8W)C
M)&5X="(L("1B=68L(#`W-34I.PHD4$%2.CI(96%V>3HZ1G5L;$-A8VAE>R1F
M=6QL;F%M97T@/2`D9FEL96YA;64["B1005(Z.DAE879Y.CI&=6QL0V%C:&5[
M)&9I;&5N86UE?2`]("1F=6QL;F%M93L*?0IE;'-E('L*)$UO9'5L94-A8VAE
M>R1F=6QL;F%M97T@/2!["F)U9B`]/B`D8G5F+`IC<F,@/3X@)&-R8RP*;F%M
M92`]/B`D9G5L;&YA;64L"GT["B1R97%U:7)E7VQI<W1[)&9U;&QN86UE?2LK
M.PI]"G)E860@7T9(+"`D8G5F+"`T.PI]"FQO8V%L($!)3D,@/2`H<W5B('L*
M;7D@*"1S96QF+"`D;6]D=6QE*2`]($!?.PIR971U<FX@:68@<F5F("1M;V1U
M;&4@;W(@(21M;V1U;&4["FUY("1I;F9O(#T@)$UO9'5L94-A8VAE>R1M;V1U
M;&5].PIR971U<FX@=6YL97-S("1I;F9O.PID96QE=&4@)')E<75I<F5?;&ES
M='LD;6]D=6QE?3L*)$E.0WLD;6]D=6QE?2`]("(O;&]A9&5R+R1I;F9O+R1M
M;V1U;&4B.PII9B`H)$5.5GM005)?0TQ%04Y](&%N9"!D969I;F5D*"9)3SHZ
M1FEL93HZ;F5W*2D@>PIM>2`D9F@@/2!)3SHZ1FEL92T^;F5W7W1M<&9I;&4@
M;W(@9&EE(")#86XG="!C<F5A=&4@=&5M<"!F:6QE.B`D(2(["B1F:"T^8FEN
M;6]D92@I.PHD9F@M/G!R:6YT*"1I;F9O+3Y[8G5F?2D["B1F:"T^<V5E:R@P
M+"`P*3L*<F5T=7)N("1F:#L*?0IE;'-E('L*;7D@)&9I;&5N86UE(#T@7W-A
M=F5?87,H(B1I;F9O+3Y[8W)C?2YP;2(L("1I;F9O+3Y[8G5F?2D["B1I;F9O
M+3Y[9FEL97T@/2`D9FEL96YA;64["F]P96X@;7D@)&9H+"`G/#IR87<G+"`D
M9FEL96YA;64@;W(@9&EE('%Q6T-A;B=T(')E860@(B1F:6QE;F%M92(Z("0A
M73L*<F5T=7)N("1F:#L*?0ID:64@(D)O;W1S=')A<'!I;F<@9F%I;&5D.B!C
M86XG="!F:6YD(&UO9'5L92`D;6]D=6QE(2(["GTL($!)3D,I.PIR97%U:7)E
M(%A33&]A9&5R.PIR97%U:7)E(%!!4CHZ2&5A=GD["G)E<75I<F4@0V%R<#HZ
M2&5A=GD["G)E<75I<F4@17AP;W)T97(Z.DAE879Y.PI005(Z.DAE879Y.CI?
M:6YI=%]D>6YA;&]A9&5R*"D["G)E<75I<F4@24\Z.D9I;&4["G=H:6QE("AM
M>2`D9FEL96YA;64@/2`H<V]R="!K97ES("5R97%U:7)E7VQI<W0I6S!=*2![
M"G5N;&5S<R`H)$E.0WLD9FEL96YA;65](&]R("1F:6QE;F%M92`]?B`O0E-$
M4$%.+RD@>PII9B`H)&9I;&5N86UE(#U^("]<+G!M8S\D+VDI('L*<F5Q=6ER
M92`D9FEL96YA;64["GT*96QS92!["F1O("1F:6QE;F%M92!U;FQE<W,@)&9I
M;&5N86UE(#U^("]S:71E8W5S=&]M:7IE7"YP;"0O.PI]"GT*9&5L971E("1R
M97%U:7)E7VQI<W1[)&9I;&5N86UE?3L*?0II9B`H)&)U9B!N92`D4$M:25!?
M34%'24,I('L*;W5T<RAQ<5M.;R!Z:7`@9F]U;F0@869T97(@1DE,12!S96-T
M:6]N(&EN(&9I;&4@(B1P<F]G;F%M92)=*3L*;&%S="!-04=)0R`["GT*)'-T
M87)T7W!O<R`]("AT96QL(%]&2"D@+2`T.R`@("`@("`@("`@("`@("`C('-T
M87)T(&]F('II<`I]"FUY($!P87)?87)G<SL*;7D@*"1O=70L("1B=6YD;&4L
M("1L;V=F:"P@)&-A8VAE7VYA;64I.PID96QE=&4@)$5.5GM005)?05!07U)%
M55-%?3L@(R!S86YI=&EZ92`H4D554T4@;6%Y(&)E(&$@<V5C=7)I='D@<')O
M8FQE;2D*:68@*"$D<W1A<G1?<&]S(&]R("@D05)'5ELP72!E<2`G+2UP87(M
M;W!T:6]N<R<@)B8@<VAI9G0I*2!["FUY("5D:7-T7V-M9"`]('%W*`IP("`@
M8FQI8E]T;U]P87(*:2`@(&EN<W1A;&Q?<&%R"G4@("!U;FEN<W1A;&Q?<&%R
M"G,@("!S:6=N7W!A<@IV("`@=F5R:69Y7W!A<@HI.PII9B`H0$%21U8@86YD
M("1!4D=66S!=(&5Q("<M+7)E=7-E)RD@>PIS:&EF="!`05)'5CL*)$5.5GM0
M05)?05!07U)%55-%?2`]('-H:69T($!!4D=6.PI]"F5L<V4@>R`C(&YO<FUA
M;"!P87)L(&)E:&%V:6]U<@IM>2!`861D7W1O7VEN8SL*=VAI;&4@*$!!4D=6
M*2!["B1!4D=66S!=(#U^("]>+2A;04E-3T),8G%P:75S5'9=*2@N*BDO(&]R
M(&QA<W0["FEF("@D,2!E<2`G22<I('L*<'5S:"!`861D7W1O7VEN8RP@)#([
M"GT*96QS:68@*"0Q(&5Q("=-)RD@>PIE=F%L(")U<V4@)#(B.PI]"F5L<VEF
M("@D,2!E<2`G02<I('L*=6YS:&EF="!`<&%R7V%R9W,L("0R.PI]"F5L<VEF
M("@D,2!E<2`G3R<I('L*)&]U="`]("0R.PI]"F5L<VEF("@D,2!E<2`G8B<I
M('L*)&)U;F1L92`]("=S:71E)SL*?0IE;'-I9B`H)#$@97$@)T(G*2!["B1B
M=6YD;&4@/2`G86QL)SL*?0IE;'-I9B`H)#$@97$@)W$G*2!["B1Q=6EE="`]
M(#$["GT*96QS:68@*"0Q(&5Q("=,)RD@>PIO<&5N("1L;V=F:"P@(CX^(BP@
M)#(@;W(@9&EE('%Q6T-A;B=T(&]P96X@;&]G(&9I;&4@(B0R(CH@)"%=.PI]
M"F5L<VEF("@D,2!E<2`G5"<I('L*)&-A8VAE7VYA;64@/2`D,CL*?0IS:&EF
M="A`05)'5BD["FEF("AM>2`D8VUD(#T@)&1I<W1?8VUD>R0Q?2D@>PID96QE
M=&4@)$5.5GLG4$%27U1%35`G?3L*:6YI=%]I;F,H*3L*<F5Q=6ER92!005(Z
M.D1I<W0["B9[(E!!4CHZ1&ES=#HZ)&-M9")]*"D@=6YL97-S($!!4D=6.PHF
M>R)005(Z.D1I<W0Z.B1C;60B?2@D7RD@9F]R($!!4D=6.PIE>&ET.PI]"GT*
M=6YS:&EF="!`24Y#+"!`861D7W1O7VEN8SL*?0I]"FEF("@D;W5T*2!["GL*
M<F5Q=6ER92!)3SHZ1FEL93L*<F5Q=6ER92!!<F-H:79E.CI::7`["G)E<75I
M<F4@1&EG97-T.CI32$$["GT*;7D@)'!A<B`]('-H:69T*$!!4D=6*3L*;7D@
M)'II<#L*:68@*&1E9FEN960@)'!A<BD@>PIO<&5N(&UY("1F:"P@)SPZ<F%W
M)RP@)'!A<B!O<B!D:64@<7%;0V%N)W0@9FEN9"!P87(@9FEL92`B)'!A<B(Z
M("0A73L*8FQE<W,H)&9H+"`G24\Z.D9I;&4G*3L*)'II<"`]($%R8VAI=F4Z
M.EII<"T^;F5W.PHH("1Z:7`M/G)E861&<F]M1FEL94AA;F1L92@D9F@L("1P
M87(I(#T]($%R8VAI=F4Z.EII<#HZ05I?3TLH*2`I"F]R(&1I92!Q<5M%<G)O
M<B!R96%D:6YG('II<"!A<F-H:79E("(D<&%R(ET["GT*;7D@)6UE=&%?<&%R
M(#T@9&\@>PII9B`H)'II<"!A;F0@;7D@)&UE=&$@/2`D>FEP+3YC;VYT96YT
M<R@G345402YY;6PG*2D@>PHD;65T82`]?B!S+RXJ7G!A<CHD+R]M<SL*)&UE
M=&$@/7X@<R]>7%,N*B\O;7,["B1M971A(#U^("]>("`H6UXZ72LI.B`H+BLI
M)"]M9SL*?0I].PII9B`H9&5F:6YE9"`D<&%R*2!["F]P96X@;7D@)'!H+"`G
M/#IR87<G+"`D<&%R(&]R(&1I92!Q<5M#86XG="!R96%D('!A<B!F:6QE("(D
M<&%R(CH@)"%=.PIM>2`D8G5F.PIR96%D("1P:"P@)&)U9BP@-#L*9&EE('%Q
M6R(D<&%R(B!I<R!N;W0@82!P87(@9FEL95T@=6YL97-S("1B=68@97$@)%!+
M6DE07TU!1TE#.PIC;&]S92`D<&@["GT*0W)E871E4&%T:"@D;W5T*2`["FUY
M("1F:"`]($E/.CI&:6QE+3YN97<H"B1O=70L"DE/.CI&:6QE.CI/7T-214%4
M*"D@?"!)3SHZ1FEL93HZ3U]21%=2*"D@?"!)3SHZ1FEL93HZ3U]44E5.0R@I
M+`HP-S<W+`HI(&]R(&1I92!Q<5M#86XG="!C<F5A=&4@9FEL92`B)&]U="(Z
M("0A73L*)&9H+3YB:6YM;V1E*"D["G-E96L@7T9(+"`P+"`P.PIM>2`D;&]A
M9&5R.PII9B`H9&5F:6YE9"`D<W1A<G1?;V9?1DE,15]S96-T:6]N*2!["G)E
M860@7T9(+"`D;&]A9&5R+"`D<W1A<G1?;V9?1DE,15]S96-T:6]N.PI](&5L
M<V4@>PIL;V-A;"`D+R`]('5N9&5F.PHD;&]A9&5R(#T@/%]&2#X["GT*:68@
M*"$D14Y6>U!!4E]615)"051)37T@86YD("1L;V%D97(@/7X@+UXH/SHC(7Q<
M0')E;2DO*2!["G)E<75I<F4@4$%2.CI&:6QT97(Z.E!O9%-T<FEP.PI005(Z
M.D9I;'1E<CHZ4&]D4W1R:7`M/F%P<&QY*%PD;&]A9&5R+"`D,"D["GT*)&9H
M+3YP<FEN="@D;&]A9&5R*0IO<B!D:64@<7%;17)R;W(@=W)I=&EN9R!L;V%D
M97(@=&\@(B1O=70B.B`D(5T["FEF("@D8G5N9&QE*2!["G)E<75I<F4@4$%2
M.CI(96%V>3L*4$%2.CI(96%V>3HZ7VEN:71?9'EN86QO861E<B@I.PII;FET
M7VEN8R@I.PIR97%U:7)E7VUO9'5L97,H*3L*;7D@0&EN8R`](&=R97`@>R`A
M+T)31%!!3B\@?0IG<F5P('L**"1B=6YD;&4@;F4@)W-I=&4G*2!O<@HH)%\@
M;F4@)$-O;F9I9SHZ0V]N9FEG>V%R8VAL:6)E>'!](&%N9`HD7R!N92`D0V]N
M9FEG.CI#;VYF:6=[<')I=FQI8F5X<'TI.PI]($!)3D,["G-\+RM\+WQG+"!S
M?"\D?'P@9F]R96%C:"!`:6YC.PIM>2`E9FEL97,["B1F:6QE<WLD7WTK*R!F
M;W(@0$1Y;F%,;V%D97(Z.F1L7W-H87)E9%]O8FIE8W1S+"!V86QU97,@)4E.
M0SL*;7D@)&QI8E]E>'0@/2`D0V]N9FEG.CI#;VYF:6=[;&EB7V5X='T[("`@
M("`@("`@(R!86%@@;&EB7V5X="!V<R!D;&5X="`_"FUY("5W<FET=&5N.PIF
M;W)E86-H(&UY("1K97D@*'-O<G0@:V5Y<R`E9FEL97,I('L*;7D@*"1F:6QE
M+"`D;F%M92D["FEF("AD969I;F5D*&UY("1F8R`]("1005(Z.DAE879Y.CI&
M=6QL0V%C:&5[)&ME>7TI*2!["B@D9FEL92P@)&YA;64I(#T@*"1K97DL("1F
M8RD["GT*96QS92!["F9O<F5A8V@@;7D@)&1I<B`H0&EN8RD@>PII9B`H)&ME
M>2`]?B!M?%Y<421D:7)<12\H+BHI)'QI*2!["B@D9FEL92P@)&YA;64I(#T@
M*"1K97DL("0Q*3L*;&%S=#L*?0II9B`H)&ME>2`]?B!M?%XO;&]A9&5R+UM>
M+UTK+R@N*BDD?"D@>PII9B`H;7D@)')E9B`]("1-;V1U;&5#86-H97LD,7TI
M('L**"1F:6QE+"`D;F%M92D@/2`H)')E9BP@)#$I.PIL87-T.PI]"FEF("@M
M9B`B)&1I<B\D,2(I('L**"1F:6QE+"`D;F%M92D@/2`H(B1D:7(O)#$B+"`D
M,2D["FQA<W0["GT*?0I]"GT*;F5X="!U;FQE<W,@9&5F:6YE9"`D;F%M93L*
M;F5X="!I9B`D=W)I='1E;GLD;F%M97TK*SL*;F5X="!I9B`A<F5F*"1F:6QE
M*2!A;F0@)&9I;&4@/7X@+UPN7%$D;&EB7V5X=%Q%)"]I.PIO=71S*'-P<FEN
M=&8H<7%;4&%C:VEN9R!&24Q%("(E<R(N+BY=+"!R968@)&9I;&4@/R`D9FEL
M92T^>VYA;65](#H@)&9I;&4I*3L*;7D@)&-O;G1E;G0["FEF("AR968H)&9I
M;&4I*2!["B1C;VYT96YT(#T@)&9I;&4M/GMB=69].PI]"F5L<V4@>PIL;V-A
M;"`D+R`]('5N9&5F.PIO<&5N(&UY("1T:"P@)SPZ<F%W)RP@)&9I;&4@;W(@
M9&EE('%Q6T-A;B=T(')E860@(B1F:6QE(CH@)"%=.PHD8V]N=&5N="`](#PD
M=&@^.PIC;&]S92`D=&@["E!!4CHZ1FEL=&5R.CI0;V13=')I<"T^87!P;'DH
M7"1C;VYT96YT+"`B/&5M8F5D9&5D/B\D;F%M92(I"FEF("$D14Y6>U!!4E]6
M15)"051)37T@86YD("1N86UE(#U^("]<+B@_.G!M?&EX?&%L*20O:3L*4$%2
M.CI&:6QT97(Z.E!A=&-H0V]N=&5N="T^;F5W+3YA<'!L>2A<)&-O;G1E;G0L
M("1F:6QE+"`D;F%M92D["GT*)&9H+3YP<FEN="@B1DE,12(L"G!A8VLH)TXG
M+"!L96YG=&@H)&YA;64I("L@.2DL"G-P<FEN=&8H(B4P.'@O)7,B+"!!<F-H
M:79E.CI::7`Z.F-O;7!U=&5#4D,S,B@D8V]N=&5N="DL("1N86UE*2P*<&%C
M:R@G3B<L(&QE;F=T:"@D8V]N=&5N="DI+`HD8V]N=&5N="D*;W(@9&EE('%Q
M6T5R<F]R('=R:71I;F<@96UB961D960@1DE,12!T;R`B)&]U="(Z("0A73L*
M;W5T<RAQ<5M7<FET=&5N(&%S("(D;F%M92)=*3L*?0I]"FEF("@D>FEP*2![
M"B1Z:7`M/G=R:71E5&]&:6QE2&%N9&QE*"1F:"D@/3T@07)C:&EV93HZ6FEP
M.CI!6E]/2PIO<B!D:64@<7%;17)R;W(@=W)I=&EN9R!Z:7`@<&%R="!O9B`B
M)&]U=")=.PI]"FEF("@D;65T85]P87)[8VQE86Y]*2!["B1F:"T^<')I;G0H
M(EPP0TQ%04XB*3L*?0IE;'-E('L*:68@*"%D969I;F5D("1C86-H95]N86UE
M*2!["FUY("1C='@@/2!$:6=E<W0Z.E-(02T^;F5W*#$I.PIS965K("1F:"P@
M,"P@,#L*)&-T>"T^861D9FEL92@D9F@I.PIS965K("1F:"P@,"P@,CL*)&-A
M8VAE7VYA;64@/2`D8W1X+3YH97AD:6=E<W0["GT*)&-A8VAE7VYA;64@/2!S
M=6)S='(@)&-A8VAE7VYA;64L(#`L("1C86-H95]N86UE7W-I>F4["B1C86-H
M95]N86UE("X](")<,"(@>"`H)&-A8VAE7VYA;65?<VEZ92`M(&QE;F=T:"`D
M8V%C:&5?;F%M92D["B1F:"T^<')I;G0H)&-A8VAE7VYA;64L"B)<,$-!0TA%
M(BD["GT*;7D@)&]F9G-E="`]("1F:"T^=&5L;"`M(&QE;F=T:"@D;&]A9&5R
M*3L*)&9H+3YP<FEN="AP86-K*"=.)RP@)&]F9G-E="DL"B1005)?34%'24,I
M.PHD9F@M/F-L;W-E(&]R(&1I92!Q<5M%<G)O<B!W<FET:6YG('1R86EL97(@
M;V8@(B1O=70B.B`D(5T["F-H;6]D(#`W-34L("1O=70["F5X:70["GT*>PIL
M87-T('5N;&5S<R!D969I;F5D("1S=&%R=%]P;W,["E]F:7A?<')O9VYA;64H
M*3L*<F5Q=6ER92!005(["E!!4CHZ2&5A=GDZ.E]I;FET7V1Y;F%L;V%D97(H
M*3L*>PIR97%U:7)E($9I;&4Z.D9I;F0["G)E<75I<F4@07)C:&EV93HZ6FEP
M.PI]"FUY("1F:"`]($E/.CI&:6QE+3YN97<[("`@("`@("`@("`@("`@("`@
M("`@("`@("`@("`C($%R8VAI=F4Z.EII<"!O<&5R871E<R!O;B!A;B!)3SHZ
M2&%N9&QE"B1F:"T^9F1O<&5N*&9I;&5N;RA?1D@I+"`G<B<I(&]R(&1I92!Q
M<5MF9&]P96XH*2!F86EL960Z("0A73L*07)C:&EV93HZ6FEP.CIS971#:'5N
M:U-I>F4H+7,@7T9(*3L*;7D@)'II<"`]($%R8VAI=F4Z.EII<"T^;F5W.PHH
M)'II<"T^<F5A9$9R;VU&:6QE2&%N9&QE*"1F:"P@)'!R;V=N86UE*2`]/2!!
M<F-H:79E.CI::7`Z.D%:7T]+*"DI"F]R(&1I92!Q<5M%<G)O<B!R96%D:6YG
M('II<"!A<F-H:79E("(D<')O9VYA;64B73L*07)C:&EV93HZ6FEP.CIS971#
M:'5N:U-I>F4H-C0@*B`Q,#(T*3L*<'5S:"!`4$%2.CI,:6)#86-H92P@)'II
M<#L*)%!!4CHZ3&EB0V%C:&5[)'!R;V=N86UE?2`]("1Z:7`["F]U=',H(D5X
M=')A8W1I;F<@>FEP+BXN(BD["FEF("AD969I;F5D("1%3E9[4$%27U1%35!]
M*2![(",@<VAO=6QD(&)E('-E="!A="!T:&ES('!O:6YT(0IF;W)E86-H(&UY
M("1M96UB97(@*"`D>FEP+3YM96UB97)S("D@>PIN97AT(&EF("1M96UB97(M
M/FES1&ER96-T;W)Y.PIM>2`D;65M8F5R7VYA;64@/2`D;65M8F5R+3YF:6QE
M3F%M93L*;F5X="!U;FQE<W,@)&UE;6)E<E]N86UE(#U^(&U["EX*+S]S:&QI
M8B\**#\Z)$-O;F9I9SHZ0V]N9FEG>V%R8VAN86UE?2\I/PHH6UXO72LI"B0*
M?7@["FUY("1E>'1R86-T7VYA;64@/2`D,3L*;7D@)&1E<W1?;F%M92`]($9I
M;&4Z.E-P96,M/F-A=&9I;&4H)$5.5GM005)?5$5-4'TL("1E>'1R86-T7VYA
M;64I.PII9B`H+68@)&1E<W1?;F%M92`F)B`M<R!?(#T]("1M96UB97(M/G5N
M8V]M<')E<W-E9%-I>F4H*2D@>PIO=71S*'%Q6U-K:7!P:6YG("(D;65M8F5R
M7VYA;64B('-I;F-E(&ET(&%L<F5A9'D@97AI<W1S(&%T("(D9&5S=%]N86UE
M(ETI.PI](&5L<V4@>PIO=71S*'%Q6T5X=')A8W1I;F<@(B1M96UB97)?;F%M
M92(@=&\@(B1D97-T7VYA;64B+BXN72D["B@D;65M8F5R+3YE>'1R86-T5&]&
M:6QE3F%M960H)&1E<W1?;F%M92D@/3T@07)C:&EV93HZ6FEP.CI!6E]/2R@I
M*0IO<B!D:64@<7%;17)R;W(@97AT<F%C=&EN9R!Z:7`@;65M8F5R('1O("(D
M9&5S=%]N86UE(ET["F-H;6]D*#`U-34L("1D97-T7VYA;64I(&EF("1>3R!E
M<2`B:'!U>"(["GT*?0I]"GT*=6YL97-S("@D4$%2.CI,:6)#86-H97LD<')O
M9VYA;65]*2!["F1I92`\/"`B+B(@=6YL97-S($!!4D=6.PI5<V%G93H@)#`@
M6R`M06QI8BYP87(@72!;("U)9&ER(%T@6R`M36UO9'5L92!=(%L@<W)C+G!A
M<B!=(%L@<')O9W)A;2YP;"!="B0P(%L@+4)\+6(@72!;+4]O=71F:6QE72!S
M<F,N<&%R"BX*)$5.5GM005)?4%)/1TY!345](#T@)'!R;V=N86UE(#T@)#`@
M/2!S:&EF="A`05)'5BD["GT*<W5B($-R96%T95!A=&@@>PIM>2`H)&YA;64I
M(#T@0%\["G)E<75I<F4@1FEL93HZ0F%S96YA;64["FUY("@D8F%S96YA;64L
M("1P871H+"`D97AT*2`]($9I;&4Z.D)A<V5N86UE.CIF:6QE<&%R<V4H)&YA
M;64L("@G7"XN*B<I*3L*<F5Q=6ER92!&:6QE.CI0871H.PI&:6QE.CI0871H
M.CIM:W!A=&@H)'!A=&@I('5N;&5S<R@M92`D<&%T:"D[(",@;6MP871H(&1I
M97,@=VET:"!E<G)O<@I]"G-U8B!R97%U:7)E7VUO9'5L97,@>PIR97%U:7)E
M(&QI8CL*<F5Q=6ER92!$>6YA3&]A9&5R.PIR97%U:7)E(&EN=&5G97(["G)E
M<75I<F4@<W1R:6-T.PIR97%U:7)E('=A<FYI;F=S.PIR97%U:7)E('9A<G,[
M"G)E<75I<F4@0V%R<#L*<F5Q=6ER92!#87)P.CI(96%V>3L*<F5Q=6ER92!%
M<G)N;SL*<F5Q=6ER92!%>'!O<G1E<CHZ2&5A=GD["G)E<75I<F4@17AP;W)T
M97(["G)E<75I<F4@1F-N=&P["G)E<75I<F4@1FEL93HZ5&5M<#L*<F5Q=6ER
M92!&:6QE.CI3<&5C.PIR97%U:7)E(%A33&]A9&5R.PIR97%U:7)E($-O;F9I
M9SL*<F5Q=6ER92!)3SHZ2&%N9&QE.PIR97%U:7)E($E/.CI&:6QE.PIR97%U
M:7)E($-O;7!R97-S.CI:;&EB.PIR97%U:7)E($%R8VAI=F4Z.EII<#L*<F5Q
M=6ER92!$:6=E<W0Z.E-(03L*<F5Q=6ER92!005(["G)E<75I<F4@4$%2.CI(
M96%V>3L*<F5Q=6ER92!005(Z.D1I<W0["G)E<75I<F4@4$%2.CI&:6QT97(Z
M.E!O9%-T<FEP.PIR97%U:7)E(%!!4CHZ1FEL=&5R.CI0871C:$-O;G1E;G0[
M"G)E<75I<F4@871T<FEB=71E<SL*979A;"![(')E<75I<F4@0W=D('T["F5V
M86P@>R!R97%U:7)E(%=I;C,R('T["F5V86P@>R!R97%U:7)E(%-C86QA<CHZ
M571I;"!].PIE=F%L('L@<F5Q=6ER92!!<F-H:79E.CI5;GII<#HZ0G5R<W0@
M?3L*979A;"![(')E<75I<F4@5&EE.CI(87-H.CI.86UE9$-A<'1U<F4@?3L*
M979A;"![(')E<75I<F4@4&5R;$E/.R!R97%U:7)E(%!E<FQ)3SHZ<V-A;&%R
M('T["F5V86P@>R!R97%U:7)E('5T9C@@?3L*?0IS=6(@7W-E=%]P87)?=&5M
M<"!["FEF("AD969I;F5D("1%3E9[4$%27U1%35!](&%N9"`D14Y6>U!!4E]4
M14U0?2`]?B`O*"XK*2\I('L*)'!A<E]T96UP(#T@)#$["G)E='5R;CL*?0IF
M;W)E86-H(&UY("1P871H("@**&UA<"`D14Y6>R1??2P@<7<H(%!!4E]435!$
M25(@5$U01$E2(%1%35!$25(@5$5-4"!435`@*2DL"G%W*"!#.EQ<5$5-4"`O
M=&UP("X@*0HI('L*;F5X="!U;FQE<W,@9&5F:6YE9"`D<&%T:"!A;F0@+60@
M)'!A=&@@86YD("UW("1P871H.PIM>2`D=7-E<FYA;64["FUY("1P=W5I9#L*
M979A;"![*"1P=W5I9"D@/2!G971P=W5I9"@D/BD@:68@9&5F:6YE9"`D/CM]
M.PII9B`H(&1E9FEN960H)E=I;C,R.CI,;V=I;DYA;64I("D@>PHD=7-E<FYA
M;64@/2`F5VEN,S(Z.DQO9VEN3F%M93L*?0IE;'-I9B`H9&5F:6YE9"`D<'=U
M:60I('L*)'5S97)N86UE(#T@)'!W=6ED.PI]"F5L<V4@>PHD=7-E<FYA;64@
M/2`D14Y6>U5315).04U%?2!\?"`D14Y6>U5315)]('Q\("=365-414TG.PI]
M"FUY("1S=&UP9&ER(#T@(B1P871H)'-Y<WM?9&5L:6U]<&%R+2(N=6YP86-K
M*")(*B(L("1U<V5R;F%M92D["FUK9&ER("1S=&UP9&ER+"`P-S4U.PIM>2`D
M8V%C:&5?9&ER.PII9B`H)$5.5GM005)?0TQ%04Y]*2!["B1C86-H95]D:7(@
M/2`B=&5M<"TD)"(["GT*96QS92!["F]P96X@;7D@)&9H+"`B/#IR87<B+"`D
M<')O9VYA;64@;W(@9&EE('%Q6T-A;B=T(')E860@(B1P<F]G;F%M92(Z("0A
M73L*:68@*"AM>2`D;6%G:6-?<&]S(#T@9FEN9%]P87)?;6%G:6,H)&9H*2D@
M/CT@,"D@>PIS965K("1F:"P@)&UA9VEC7W!O<PHM("1&24Q%7V]F9G-E=%]S
M:7IE"BT@;&5N9W1H*")<,$-!0TA%(BDL(#`["FUY("1B=68["G)E860@)&9H
M+"`D8G5F+"!L96YG=&@H(EPP0T%#2$4B*3L*:68@*"1B=68@97$@(EPP0T%#
M2$4B*2!["G-E96L@)&9H+"`D;6%G:6-?<&]S"BT@)$9)3$5?;V9F<V5T7W-I
M>F4*+2!L96YG=&@H(EPP0T%#2$4B*0HM("1C86-H95]N86UE7W-I>F4L(#`[
M"G)E860@)&9H+"`D8G5F+"`D8V%C:&5?;F%M95]S:7IE.PHD8G5F(#U^(',O
M7#`O+V<["B1C86-H95]D:7(@/2`B8V%C:&4M)&)U9B(["GT*?0IC;&]S92`D
M9F@["GT*:68@*"$D8V%C:&5?9&ER*2!["B1C86-H95]D:7(@/2`B=&5M<"TD
M)"(["B1%3E9[4$%27T-,14%.?2`](#$["GT*)'-T;7!D:7(@+CT@(B1S>7-[
M7V1E;&EM?21C86-H95]D:7(B.PIM:V1I<B`D<W1M<&1I<BP@,#<U-3L*)$5.
M5GM005)?5$5-4'T@/2`D<W1M<&1I<CL*;&%S=#L*?0HD<&%R7W1E;7`@/2`D
M,2!I9B`D14Y6>U!!4E]414U0?2!A;F0@)$5.5GM005)?5$5-4'T@/7X@+R@N
M*RDO.PI]"G-U8B!?<V%V95]A<R!["FUY("@D;F%M92P@)&-O;G1E;G1S+"`D
M;6]D92D@/2!`7SL*;7D@)&9U;&QN86UE(#T@(B1P87)?=&5M<"\D;F%M92([
M"G5N;&5S<R`H+64@)&9U;&QN86UE*2!["FUY("1T96UP;F%M92`]("(D9G5L
M;&YA;64N)"0B.PIM>2`D9F@["BAO<&5N("1F:"P@)SXZ<F%W)RP@)'1E;7!N
M86UE*0HF)B`H<')I;G0@)&9H("1C;VYT96YT<RD*)B8@*&-L;W-E("1F:"D*
M;W(@9&EE('%Q6T5R<F]R('=R:71I;F<@(B1T96UP;F%M92(Z("0A73L*8VAM
M;V0@)&UO9&4L("1T96UP;F%M92!I9B!D969I;F5D("1M;V1E.PIR96YA;64H
M)'1E;7!N86UE+"`D9G5L;&YA;64I(&]R('5N;&EN:R@D=&5M<&YA;64I.PI]
M"G)E='5R;B`D9G5L;&YA;64["GT*<W5B(%]S971?<')O9VYA;64@>PII9B`H
M9&5F:6YE9"`D14Y6>U!!4E]04D]'3D%-17T@86YD("1%3E9[4$%27U!23T=.
M04U%?2`]?B`O*"XK*2\I('L*)'!R;V=N86UE(#T@)#$["GT*)'!R;V=N86UE
M('Q\/2`D,#L*:68@*"1%3E9[4$%27U1%35!](&%N9"!I;F1E>"@D<')O9VYA
M;64L("1%3E9[4$%27U1%35!]*2`^/2`P*2!["B1P<F]G;F%M92`]('-U8G-T
M<B@D<')O9VYA;64L(')I;F1E>"@D<')O9VYA;64L("1S>7-[7V1E;&EM?2D@
M*R`Q*3L*?0II9B`H(21%3E9[4$%27U!23T=.04U%?2!O<B!I;F1E>"@D<')O
M9VYA;64L("1S>7-[7V1E;&EM?2D@/CT@,"D@>PII9B`H;W!E;B!M>2`D9F@L
M("<\)RP@)'!R;V=N86UE*2!["G)E='5R;B!I9B`M<R`D9F@["GT*:68@*"US
M("(D<')O9VYA;64D<WES>U]E>&5](BD@>PHD<')O9VYA;64@+CT@)'-Y<WM?
M97AE?3L*<F5T=7)N.PI]"GT*9F]R96%C:"!M>2`D9&ER("AS<&QI="`O7%$D
M<WES>W!A=&A?<V5P?5Q%+RP@)$5.5GM0051(?2D@>PIN97AT(&EF(&5X:7-T
M<R`D14Y6>U!!4E]414U0?2!A;F0@)&1I<B!E<2`D14Y6>U!!4E]414U0?3L*
M)&1I<B`]?B!S+UQ1)'-Y<WM?9&5L:6U]7$4D+R\["B@H)'!R;V=N86UE(#T@
M(B1D:7(D<WES>U]D96QI;7TD<')O9VYA;64D<WES>U]E>&5](BDL(&QA<W0I
M"FEF("US("(D9&ER)'-Y<WM?9&5L:6U])'!R;V=N86UE)'-Y<WM?97AE?2([
M"B@H)'!R;V=N86UE(#T@(B1D:7(D<WES>U]D96QI;7TD<')O9VYA;64B*2P@
M;&%S="D*:68@+7,@(B1D:7(D<WES>U]D96QI;7TD<')O9VYA;64B.PI]"GT*
M<W5B(%]F:7A?<')O9VYA;64@>PHD,"`]("1P<F]G;F%M92!\?#T@)$5.5GM0
M05)?4%)/1TY!345].PII9B`H:6YD97@H)'!R;V=N86UE+"`D<WES>U]D96QI
M;7TI(#P@,"D@>PHD<')O9VYA;64@/2`B+B1S>7-[7V1E;&EM?21P<F]G;F%M
M92(["GT*;7D@)'!W9"`]("AD969I;F5D("9#=V0Z.F=E=&-W9"D@/R!#=V0Z
M.F=E=&-W9"@I"CH@*"AD969I;F5D("97:6XS,CHZ1V5T0W=D*2`_(%=I;C,R
M.CI'971#=V0H*2`Z(&!P=V1@*3L*8VAO;7`H)'!W9"D["B1P<F]G;F%M92`]
M?B!S+UXH/SU<+EPN/UQ1)'-Y<WM?9&5L:6U]7$4I+R1P=V0D<WES>U]D96QI
M;7TO.PHD14Y6>U!!4E]04D]'3D%-17T@/2`D<')O9VYA;64["GT*<W5B(%]P
M87)?:6YI=%]E;G8@>PII9B`H("1%3E9[4$%27TE.251)04Q)6D5$?2LK(#T]
M(#$@*2!["G)E='5R;CL*?2!E;'-E('L*)$5.5GM005)?24Y)5$E!3$E:141]
M(#T@,CL*?0IF;W(@*'%W*"!34$%73D5$(%1%35`@0TQ%04X@1$5"54<@0T%#
M2$4@4%)/1TY!344@*2`I('L*9&5L971E("1%3E9[(E!!4E\D7R)].PI]"F9O
M<B`H<7<O(%1-4$1)4B!414U0($-,14%.($1%0E5'("\I('L*)$5.5GLB4$%2
M7R1?(GT@/2`D14Y6>R)005)?1TQ/0D%,7R1?(GT@:68@97AI<W1S("1%3E9[
M(E!!4E]'3$]"04Q?)%\B?3L*?0II9B`H)$5.5GM005)?5$5-4'TI('L*9&5L
M971E("1%3E9[4$%27T-,14%.?3L*?0I]"G-U8B!O=71S('L*<F5T=7)N(&EF
M("1Q=6EE=#L*:68@*"1L;V=F:"D@>PIP<FEN="`D;&]G9F@@(D!?7&XB.PI]
M"F5L<V4@>PIP<FEN="`B0%]<;B(["GT*?0IS=6(@:6YI=%]I;F,@>PIR97%U
M:7)E($-O;F9I9SL*<'5S:"!`24Y#+"!G<F5P(&1E9FEN960L(&UA<"`D0V]N
M9FEG.CI#;VYF:6=[)%]]+"!Q=R@*87)C:&QI8F5X<"!P<FEV;&EB97AP('-I
M=&5A<F-H97AP('-I=&5L:6)E>'`*=F5N9&]R87)C:&5X<"!V96YD;W)L:6)E
M>'`**3L*?0IP86-K86=E(&UA:6X["G)E<75I<F4@4$%2.PIU;G-H:69T($!)
M3D,L(%PF4$%2.CIF:6YD7W!A<CL*4$%2+3YI;7!O<G0H0'!A<E]A<F=S*3L*
M9&EE('%Q6W!A<BYP;#H@0V%N)W0@;W!E;B!P97)L('-C<FEP="`B)'!R;V=N
M86UE(CH@3F\@<W5C:"!F:6QE(&]R(&1I<F5C=&]R>5QN70IU;FQE<W,@+64@
M)'!R;V=N86UE.PID;R`D<')O9VYA;64["D-/4D4Z.F5X:70H)#$I(&EF("@D
M0"`]?B]>7U1+7T58251?7"@H7&0K*5PI+RD["F1I92`D0"!I9B`D0#L*?3L*
M)#HZ7U]%4E)/4B`]("1`(&EF("1`.PI]"D-/4D4Z.F5X:70H)#$I(&EF("@D
M.CI?7T524D]2(#U^+UY?5$M?15A)5%]<*"A<9"LI7"DO*3L*9&EE("0Z.E]?
M15)23U(@:68@)#HZ7U]%4E)/4CL*,3L*7U]%3D1?7PH```````````!03P``
M````````````````@!1!``$``````````````%A'0``!``````````````!0
M1T```0``````````````\!1!``$``````````````!`300`!````````````
M``#P1D```0``````````````6!-!``$``````````````$=#0SH@*$=.52D@
M,3$N-"XP````````````````````1T-#.B`H1TY5*2`Q,2XT+C``````````
M``````````!'0T,Z("A'3E4I(#$Q+C0N,````````````````````$=#0SH@
M*$=.52D@,3$N-"XP````````````````````1T-#.B`H1TY5*2`Q,2XT+C``
M``````````````````!'0T,Z("A'3E4I(#$Q+C0N,```````````````````
M`$=#0SH@*$=.52D@,3$N-"XP````````````````````1T-#.B`H1TY5*2`Q
M,2XT+C````````````````````!'0T,Z("A'3E4I(#$Q+C0N,```````````
M`````````$=#0SH@*$=.52D@,3$N-"XP````````````````````1T-#.B`H
M1TY5*2`Q,2XT+C````````````````````!'0T,Z("A'3E4I(#$Q+C0N,```
M`````````````````$=#0SH@*$=.52D@,3$N-"XP````````````````````
M1T-#.B`H1TY5*2`Q,2XT+C````````````````````!'0T,Z("A'3E4I(#$Q
M+C0N,````````````````````$=#0SH@*$=.52D@,3$N-"XP````````````
M```````````````````!````@!0!`)"P``!`````\!0!`,"P``!`````$!,!
M`-"P``!`````6!,!`/"P``!`````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M``````(````9````',```!R.``!24T13STN0&)>#2[`U@<UO<%WYT@$`````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M`````````````!```$00````X```4!```&40```(X```<!```'$0```,X```
M@!```&,1```0X```<!$``-42```DX```X!(``&`3```\X```8!,``#,4``!(
MX```0!0``&H6``!<X```<!8``-(6``!TX```X!8``!$7``!\X```(!<``",7
M``"$X```,!<``#,9``"(X```0!D``&P9``"@X```<!D``((:``"HX```D!H`
M`((?``"XX```D!\``/(?``#0X````"```%,@``#8X```8"```'LA``#DX```
M@"$``)TA``#TX```H"$``%LB``#\X```8"(``"TI```,X0``,"D``$,K```D
MX0``4"L``!TL```TX0``("P``)-"``!0X0``H$(``/9"``!HX0```$,``(A$
M``!PX0``D$0``+-&``"$X0``0$H``%)*``"4X0``8$H``&%*``"<X0``<$H`
M`'%*``"@X0``@$H``(%*``"DX0``D$H``)%*``"HX0``H$H``&A.``"LX0``
M<$X``(E.``"XX0``B4X``#M/``"\X0``0$\``$9/``#$X0``4$\``%%/``#(
MX0``H$\``-!/``#,X0``\$\``!!2``!`X0``$%(``!52``#8X0``````````
M```````````````````````````````````!!`$`!$(```$````!`````0P'
M``Q""#`'8`9P!5`$P`+0```!$0D`$0$6``HP"6`(<`=0!L`$T`+@```!"@0`
M"@$7``,P`L`!$P<`$P$&`@LP"F`)<`A0`L````$0"0`08@PP"V`*<`E0",`&
MT`3@`O````$%`@`%4@$P`00!``1"```!`````1D*`!D!&P(1,!!@#W`.4`W`
M"]`)X`+P`08"``8R`L`!$P8`$P$'`@LP"F`)P`+0`1`)`!!B##`+8`IP"5`(
MP`;0!.`"\````04"``52`3`!!P,`!V(#,`+````!$08`$0$'(`DP"&`'<`%0
M`00!``1"```!#`8`#'((,`=@!L`$T`+@`1,*`!,!,P`,,`M@"G`)4`C`!M`$
MX`+P`0D%``EB!3`$8`-P`L````$)!``)4@4P!,`"T`$*!0`*H@8P!6`$P`+0
M```!$PH`$P%7``PP"V`*<`E0",`&T`3@`O`!!`$`!$(```$,!P`,0@@P!V`&
M<`50!,`"T````0H&``HR!C`%8`1P`U`"P`$$`0`$0@```0````$````!````
M`0````$&`P`&0@(P`6````$````!!`$`!&(```$````!`````0@#``A"!,`"
MT````0``````````````````````````````````````````````````````
M``!(CO5E`````-P``0`!````$@```!(````H``$`<``!`+@``0`P*0``@"$`
M`/!/```@8```0!D``%`K```@%P```"```'`9``"0'P``,!<``)`:``!@(@``
MH"$``)!$``"@0@```$,``.`6``#D``$`^@`!`!4!`0`:`0$`)`$!`#$!`0`]
M`0$`3@$!`%8!`0!B`0$`<`$!`'T!`0"*`0$`EP$!`*D!`0"S`0$`O`$!`,<!
M`0````$``@`#``0`!0`&``<`"``)``H`"P`,``T`#@`/`!``$0!P87(N97AE
M`%A37TEN=&5R;F%L<U]005)?0D]/5`!G971?=7-E<FYA;65?9G)O;5]G971P
M=W5I9`!M86EN`&UY7W!A<E]P;`!P87)?8F%S96YA;64`<&%R7V-L96%N=7``
M<&%R7V-U<G)E;G1?97AE8P!P87)?9&EE`'!A<E]D:7)N86UE`'!A<E]E;G9?
M8VQE86X`<&%R7V9I;F1P<F]G`'!A<E]I;FET7V5N=@!P87)?;6MT;7!D:7(`
M<&%R7W-E='5P7VQI8G!A=&@`<VAA7V9I;F%L`'-H85]I;FET`'-H85]U<&1A
M=&4`>'-?:6YI=```````````````````````````````````````````````
M````````````````````:!`!````````````5!H!`-@2`0``$@$`````````
M``!D&@$`<!0!`!`2`0```````````-`:`0"`%`$`R!(!````````````Y!H!
M`#@5`0````````````````````````````````!(%0$``````%@5`0``````
M9!4!``````!T%0$``````(`5`0``````D!4!``````"@%0$``````+05`0``
M````R!4!``````#4%0$``````.05`0``````\!4!``````#\%0$```````@6
M`0``````'!8!```````P%@$``````$`6`0``````3!8!``````!4%@$`````
M`&`6`0``````:!8!``````!T%@$``````(`6`0``````C!8!``````"4%@$`
M`````)P6`0``````J!8!``````"T%@$``````,`6`0``````S!8!``````#4
M%@$``````-P6`0``````Z!8!``````#\%@$``````!`7`0``````&!<!````
M```D%P$``````#`7`0``````/!<!``````!$%P$``````%`7`0``````6!<!
M``````!D%P$``````'`7`0``````?!<!``````"(%P$``````)07`0``````
MH!<!``````"L%P$``````+@7`0````````````````#$%P$`````````````
M````W!<!``````#L%P$````````8`0``````&!@!```````P&`$``````$`8
M`0``````5!@!``````!D&`$``````'08`0``````B!@!``````"@&`$`````
M`+`8`0``````Q!@!``````#8&`$``````.@8`0``````"!D!```````<&0$`
M`````"P9`0``````0!D!``````!0&0$``````%P9`0``````;!D!````````
M`````````'@9`0````````````````!(%0$``````%@5`0``````9!4!````
M``!T%0$``````(`5`0``````D!4!``````"@%0$``````+05`0``````R!4!
M``````#4%0$``````.05`0``````\!4!``````#\%0$```````@6`0``````
M'!8!```````P%@$``````$`6`0``````3!8!``````!4%@$``````&`6`0``
M````:!8!``````!T%@$``````(`6`0``````C!8!``````"4%@$``````)P6
M`0``````J!8!``````"T%@$``````,`6`0``````S!8!``````#4%@$`````
M`-P6`0``````Z!8!``````#\%@$``````!`7`0``````&!<!```````D%P$`
M`````#`7`0``````/!<!``````!$%P$``````%`7`0``````6!<!``````!D
M%P$``````'`7`0``````?!<!``````"(%P$``````)07`0``````H!<!````
M``"L%P$``````+@7`0````````````````#$%P$`````````````````W!<!
M``````#L%P$````````8`0``````&!@!```````P&`$``````$`8`0``````
M5!@!``````!D&`$``````'08`0``````B!@!``````"@&`$``````+`8`0``
M````Q!@!``````#8&`$``````.@8`0``````"!D!```````<&0$``````"P9
M`0``````0!D!``````!0&0$``````%P9`0``````;!D!````````````````
M`'@9`0`````````````````1`%]?8WAA7V%T97AI=```&`!?7V5R<FYO````
M)0!?7V=E=')E96YT`````#``7U]M86EN`````#(`7U]M96UC<'E?8VAK``!.
M`%]?<W!R:6YT9E]C:&L`40!?7W-T86-K7V-H:U]F86EL``!3`%]?<W1A8VM?
M8VAK7V=U87)D`&X`7V1L;%]C<G0P`'4`7VEM<'5R95]P='(```"%`&%C8V5S
M<P`````&`6-A;&QO8P````!(`6-L;W-E9&ER``"``6-Y9W=I;E]D971A8VA?
M9&QL`((!8WEG=VEN7VEN=&5R;F%L````G@%D;&Q?9&QL8W)T,````+L!96YV
M:7)O;@```-@!97AI=```0P)F<')I;G1F````3P)F<F5E``"U`F=E='!I9```
M``#``F=E='!W=6ED``#.`F=E='5I9`````"2`VQS965K`)0#;'-T870`F0-M
M86QL;V,`````J@-M96UC:'(`````K`-M96UC<'D`````K@-M96UM;W9E````
ML@-M:V1I<@#T`V]P96X``/@#;W!E;F1I<@````L$<&]S:7A?;65M86QI9VX`
M````,@1P=&AR96%D7V%T9F]R:P````#`!')E860``,$$<F5A9&1I<@```,,$
M<F5A9&QI;FL``,8$<F5A;&QO8P```/0$<FUD:7(`B05S<')I;G1F````D05S
M=&%T``"<!7-T<F-H<@````">!7-T<F-M<`````"C!7-T<F1U<`````"M!7-T
M<FQE;@````"R!7-T<FYC;7````"Z!7-T<G)C:'(```#$!7-T<G1O:P`````V
M!G5N;&EN:P````!,!G9F<')I;G1F```M`%]?96UU=&QS7V=E=%]A9&1R97-S
M```;`%!,7V1O7W5N9'5M<```;P%097)L7V%T9F]R:U]L;V-K``!P`5!E<FQ?
M871F;W)K7W5N;&]C:P````#L`5!E<FQ?8W)O86M?;F]C;VYT97AT``"``E!E
M<FQ?9V5T7W-V````KP)097)L7V=V7V9E=&-H<'8```!V`U!E<FQ?;6=?<V5T
M````[@-097)L7VYE=UA3`````/`#4&5R;%]N97=84U]F;&%G<P``Q@5097)L
M7W-A9F5S>7-M86QL;V,`````IP9097)L7W-V7W-E='!V`+`&4&5R;%]S=E]S
M971P=FX`````X@9097)L7W-Y<U]I;FET,P````#C!E!E<FQ?<WES7W1E<FT`
M:`=?7V5M=71L<U]V+E!,7V-U<G)E;G1?8V]N=&5X=`!K!V)O;W1?1'EN84QO
M861E<@```'P'<&5R;%]A;&QO8P````!^!W!E<FQ?8V]N<W1R=6-T`````'\'
M<&5R;%]D97-T<G5C=`"`!W!E<FQ?9G)E90"!!W!E<FQ?<&%R<V4`````@@=P
M97)L7W)U;@``B0)'971-;V1U;&5(86YD;&5!````$`$``!`!```0`0``$`$`
M`!`!```0`0``$`$``!`!```0`0``$`$``!`!```0`0``$`$``!`!```0`0``
M$`$``!`!```0`0``$`$``!`!```0`0``$`$``!`!```0`0``$`$``!`!```0
M`0``$`$``!`!```0`0``$`$``!`!```0`0``$`$``!`!```0`0``$`$``!`!
M```0`0``$`$``!`!```0`0``$`$``!`!```0`0``$`$``!`!```0`0``$`$`
M`!`!`&-Y9W=I;C$N9&QL`!00`0!C>6=G8V-?<RUS96@M,2YD;&P``"@0`0`H
M$`$`*!`!`"@0`0`H$`$`*!`!`"@0`0`H$`$`*!`!`"@0`0`H$`$`*!`!`"@0
M`0`H$`$`*!`!`"@0`0`H$`$`*!`!`"@0`0`H$`$`*!`!`"@0`0!C>6=P97)L
M-5\S-BYD;&P`/!`!`$M%4DY%3#,R+F1L;```````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M```````````````````````````````````````````````````````$``,`
M```P``"`#@```,```(`0````\```@!@````@`0"````````````````````$
M``$```!@``"``@```'@``(`#````D```@`0```"H``"`````````````````
M```!``D$``!0`0`````````````````````!``D$``!@`0``````````````
M```````!``D$``!P`0`````````````````````!``D$``"``0``````````
M`````````0```,`!`(#8``"````````````````````!``D$``"0`0``````
M```````````````!``$````(`0"````````````````````!``D$``"@`0``
M```````````````````!``$````X`0"````````````````````!``D$``"P
M`0``T"$!`&@%`````````````#@G`0"H"`````````````#@+P$`:`4`````
M````````2#4!`*@(`````````````/`]`0`^```````````````P/@$`6`(`
M````````````B$`!`-H%``````````````8`5P!)`$X`10!8`$4````H````
M$````"`````!``@````````!```````````````!`````0``!@8&``0$!``"
M`@(``````$)"0@"=G9T`;FYN`(&!@0"QL;$`____`'-S<P"?GY\`6%A8`+*R
ML@#]_?T`:VMK`/S\_`#7U]<`%A86`-K:V@"6EI8`CHZ.`$9&1@#+R\L`"PL+
M`*"@H`#^_OX`R<G)`,'!P0`#`P,`$1$1`).3DP"SL[,`[N[N`/3T]`#CX^,`
MY>7E`**BH@"ZNKH`KZ^O``<'!P!Q<7$`M;6U`/7U]0")B8D`BHJ*`#0T-`#H
MZ.@`-34U`%I:6@"WM[<`V-C8`-#0T``='1T`#`P,`(^/CP"^OKX`;6UM``D)
M"0";FYL`+R\O`,W-S0#9V=D`&1D9`$U-30!$1$0`4%!0`#P\/`!D9&0`PL+"
M`#(R,@`("`@`#0T-``\/#P``````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M``````````````````````````````````%!04%!04%!04%!04%!00$!`35`
M(P5!,4-!041%+@$!`0$!`0$%`3P7%P$!$P$!`0$!`0$!%0$!)A4!`0\!`0$!
M`0$!`1,!$0$%-0$F`0$!`0$!`0$B$A`Q#P4!(`$!`0$!`0$!$`H@(A45+B$N
M`0$!`0$?("$0(A`C(R4F)P$!`0$E$!`0$!`0$!`0%T4!`0%!$!`0$!`0$!`0
M$!`7`0$!#Q`0%1`0$!`0$!`0`0$!`1(0$!,?$!`0$!`0!P$!`0$@$!`0`0\0
M$!`0$0$!`0$!!R`0"@$!!1`0$#$!`0$!`0$!`0$!`0%!!0\!`0$!`0$!`0$!
M`0$!`0$!`0$!`0&``0``P`$``.`#``#P`P``\`,``/`#``#@`P``P`$``(`!
M``"``0``@`,``(`#````!P````<``(X/``#_'P``*````"````!``````0`(
M````````!````````````````0````$````````!`0$``@("``,#`P`$!`0`
M!04%``8&!@`'!P<`"`@(``D)"0`*"@H`"PL+``P,#``-#0T`#@X.``\/#P`0
M$!``$1$1`!(2$@`3$Q,`%!04`!45%0`6%A8`%Q<7`!@8&``9&1D`&AH:`!L;
M&P`<'!P`'1T=`!X>'@`?'Q\`("`@`"$A(0`B(B(`(R,C`"0D)``E)24`)B8F
M`"<G)P`H*"@`*2DI`"HJ*@`K*RL`+"PL`"TM+0`N+BX`+R\O`#`P,``Q,3$`
M,C(R`#,S,P`T-#0`-34U`#8V-@`W-S<`.#@X`#DY.0`Z.CH`.SL[`#P\/``]
M/3T`/CX^`#\_/P!`0$``04%!`$)"0@!#0T,`1$1$`$5%10!&1D8`1T='`$A(
M2`!)24D`2DI*`$M+2P!,3$P`34U-`$Y.3@!/3T\`4%!0`%%140!24E(`4U-3
M`%145`!5554`5E96`%=75P!86%@`65E9`%I:6@!;6UL`7%Q<`%U=70!>7EX`
M7U]?`&!@8`!A86$`8F)B`&-C8P!D9&0`965E`&9F9@!G9V<`:&AH`&EI:0!J
M:FH`:VMK`&QL;`!M;6T`;FYN`&]O;P!P<'``<7%Q`')R<@!S<W,`='1T`'5U
M=0!V=G8`=W=W`'AX>`!Y>7D`>GIZ`'M[>P!\?'P`?7U]`'Y^?@!_?W\`@("`
M`(&!@0""@H(`@X.#`(2$A`"%A84`AH:&`(>'AP"(B(@`B8F)`(J*B@"+BXL`
MC(R,`(V-C0".CHX`CX^/`)"0D`"1D9$`DI*2`).3DP"4E)0`E965`):6E@"7
MEY<`F)B8`)F9F0":FIH`FYN;`)R<G`"=G9T`GIZ>`)^?GP"@H*``H:&A`**B
MH@"CHZ,`I*2D`*6EI0"FIJ8`IZ>G`*BHJ`"IJ:D`JJJJ`*NKJP"LK*P`K:VM
M`*ZNK@"OKZ\`L+"P`+&QL0"RLK(`L[.S`+2TM`"UM;4`MK:V`+>WMP"XN+@`
MN;FY`+JZN@"[N[L`O+R\`+V]O0"^OKX`O[^_`,#`P`#!P<$`PL+"`,/#PP#$
MQ,0`Q<7%`,;&Q@#'Q\<`R,C(`,G)R0#*RLH`R\O+`,S,S`#-S<T`SL[.`,_/
MSP#0T-``T='1`-+2T@#3T],`U-34`-75U0#6UM8`U]?7`-C8V`#9V=D`VMK:
M`-O;VP#<W-P`W=W=`-[>W@#?W]\`X.#@`.'AX0#BXN(`X^/C`.3DY`#EY>4`
MYN;F`.?GYP#HZ.@`Z>GI`.KJZ@#KZ^L`[.SL`.WM[0#N[NX`[^_O`/#P\`#Q
M\?$`\O+R`//S\P#T]/0`]?7U`/;V]@#W]_<`^/CX`/GY^0#Z^OH`^_O[`/S\
M_`#]_?T`_O[^`/___P`T-#0\/#P\/#P\/#P\/#P\/#P\/#P\/#PL#`0$!`0$
M!%RTU-34U-34U-34U-S4U-34U-34U-34U+0T400$!`0$9,SL]/3T_/3\]/3T
M_/3\]/ST_/S\]/3LQ#11400$!`1DQ.3L]/3T]/3T]/3T]/3T]/3T]/3T[.R\
M-%%1400$!&3$Y.SL[.3L[.SDY.3D[.SD[.3L[.SLY+PT45%1400$9+SD[.RL
MC*S4Q)24G)2,C)24E)R<G.S<O#1145%1401DO-SD[.3$492L9.2D3#0L7,3,
M4;34Y-R\-%%145%1462\W.3D[.3D481TY(QLI)QLC+QTK-3DW+PT!`0$!`0$
M9+S<Y.3LY.2\48S4U(RLS'3,W*11Q.3<O#0$!`0$!`1DO-SDY.3DY-R,?)RT
M?(2<;,SDO'3$Y-RT-`0$!`0$!%R\W-SDY.3DU)RD7)2,S#R49,R\C%'<U+0T
M!`0$!`0$9+34W-S<Y-R,C*PLO+3,6L1DG&R4I-S4M#0$!`0$!`1<M-3<W.3<
MW'24U$RD](P\U'24=)Q,W,RT-`0$!`0$!&2TU-S<W.3<=%'T=(3LS&3D?&RD
MM&RDU+0T!`0$!`0$9+3,W-S,?(R$M/S,Q/STM/R\9,RLA%&\K#0$!`0$!`1<
MM,S4S(RDW-SL_/S\_/ST_.RT[*Q\E%&L-`0$!`0$!%RLS,QLG/3\_/S\_/S\
M_/S\_/S\Q)2,4:PT!`0$!`0$7*S$O'SD_/S\_/S\_/S\_/S\_/SLS*Q1K#0$
M!`0$!`1<K+1,Q/S\_.ST_/S\_/S\_/S\_/2\4<2L-`0$!`0$!%RLM$S4_/ST
MM-S\_/S\_/S\_/S\S%',Q*0T!`0$!`0$7*S$I%'\_/RTG/3\_/S\_/S\_/24
M4<S$I"P$!`0$!`1<I*11A.S\_,QDM.S\_/S\_/S\U&RLS+RD-`0$!`0$!%RD
M9'2\]/S\Q%&$G.3\_/S\_.2,4<S,Q*0T!`0$!`0$7*1LC.3T_/R<4<Q1E.S\
M_/S\O'3,S,2\G"P$!`0$!`1<I*QD7&2DI%&LS,1TE.3T_.11O,S,Q+R<-`0$
M!`0$!%RDM+R\C%%1Q,3,S*Q1?'R4;'R\S,3$O)PL!`0$!`0$7)R\Q,3$Q,3$
MQ,3$Q,1145%1O,3$Q,2TG#0$!`0$!`14G+3$Q,3$Q,3$Q,3$Q,3$Q,3$Q,3$
MO+2<+`0$!`0$!%2<O,3$O+R\O+R\O+R\O+R\O+R\O+R\M)PL!`0$!`0$7)RT
MQ+RTM+2TM+2TM+2TM+2TM+2TM+2LG"P$!`0$!`14A)2<G)24E)24E)24E)24
ME)24E)24E)2$)`0$!`0$!$147%Q<7%Q<7%Q<7%Q<7%Q<7%Q<7%Q<7%0T!`0$
M!`0$````/P```!\````/````!P````,````!`````````#\````_````/P``
M`#\````_````/P```#\````_````/P```#\````_````/P```#\````_````
M/P```#\````_````/P```#\````_````/P```#\````_````/P```#\H````
M$````"`````!``@````````!```````````````!`````0``!@8&``0$!``"
M`@(``````$)"0@"=G9T`;FYN`(&!@0"QL;$`____`'-S<P"?GY\`6%A8`+*R
ML@#]_?T`:VMK`/S\_`#7U]<`%A86`-K:V@"6EI8`CHZ.`$9&1@#+R\L`"PL+
M`*"@H`#^_OX`R<G)`,'!P0`#`P,`$1$1`).3DP"SL[,`[N[N`/3T]`#CX^,`
MY>7E`**BH@"ZNKH`KZ^O``<'!P!Q<7$`M;6U`/7U]0")B8D`BHJ*`#0T-`#H
MZ.@`-34U`%I:6@"WM[<`V-C8`-#0T``='1T`#`P,`(^/CP"^OKX`;6UM``D)
M"0";FYL`+R\O`,W-S0#9V=D`&1D9`$U-30!$1$0`4%!0`#P\/`!D9&0`PL+"
M`#(R,@`("`@`#0T-``\/#P``````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M``````````````````````````````````%!04%!04%!04%!04%!00$!`35`
M(P5!,4-!041%+@$!`0$!`0$%`3P7%P$!$P$!`0$!`0$!%0$!)A4!`0\!`0$!
M`0$!`1,!$0$%-0$F`0$!`0$!`0$B$A`Q#P4!(`$!`0$!`0$!$`H@(A45+B$N
M`0$!`0$?("$0(A`C(R4F)P$!`0$E$!`0$!`0$!`0%T4!`0%!$!`0$!`0$!`0
M$!`7`0$!#Q`0%1`0$!`0$!`0`0$!`1(0$!,?$!`0$!`0!P$!`0$@$!`0`0\0
M$!`0$0$!`0$!!R`0"@$!!1`0$#$!`0$!`0$!`0$!`0%!!0\!`0$!`0$!`0$!
M`0$!`0$!`0$!`0&``0``P`$``.`#``#P`P``\`,``/`#``#@`P``P`$``(`!
M``"``0``@`,``(`#````!P````<``(X/``#_'P``*````"````!``````0`(
M````````!````````````````0````$````````!`0$``@("``,#`P`$!`0`
M!04%``8&!@`'!P<`"`@(``D)"0`*"@H`"PL+``P,#``-#0T`#@X.``\/#P`0
M$!``$1$1`!(2$@`3$Q,`%!04`!45%0`6%A8`%Q<7`!@8&``9&1D`&AH:`!L;
M&P`<'!P`'1T=`!X>'@`?'Q\`("`@`"$A(0`B(B(`(R,C`"0D)``E)24`)B8F
M`"<G)P`H*"@`*2DI`"HJ*@`K*RL`+"PL`"TM+0`N+BX`+R\O`#`P,``Q,3$`
M,C(R`#,S,P`T-#0`-34U`#8V-@`W-S<`.#@X`#DY.0`Z.CH`.SL[`#P\/``]
M/3T`/CX^`#\_/P!`0$``04%!`$)"0@!#0T,`1$1$`$5%10!&1D8`1T='`$A(
M2`!)24D`2DI*`$M+2P!,3$P`34U-`$Y.3@!/3T\`4%!0`%%140!24E(`4U-3
M`%145`!5554`5E96`%=75P!86%@`65E9`%I:6@!;6UL`7%Q<`%U=70!>7EX`
M7U]?`&!@8`!A86$`8F)B`&-C8P!D9&0`965E`&9F9@!G9V<`:&AH`&EI:0!J
M:FH`:VMK`&QL;`!M;6T`;FYN`&]O;P!P<'``<7%Q`')R<@!S<W,`='1T`'5U
M=0!V=G8`=W=W`'AX>`!Y>7D`>GIZ`'M[>P!\?'P`?7U]`'Y^?@!_?W\`@("`
M`(&!@0""@H(`@X.#`(2$A`"%A84`AH:&`(>'AP"(B(@`B8F)`(J*B@"+BXL`
MC(R,`(V-C0".CHX`CX^/`)"0D`"1D9$`DI*2`).3DP"4E)0`E965`):6E@"7
MEY<`F)B8`)F9F0":FIH`FYN;`)R<G`"=G9T`GIZ>`)^?GP"@H*``H:&A`**B
MH@"CHZ,`I*2D`*6EI0"FIJ8`IZ>G`*BHJ`"IJ:D`JJJJ`*NKJP"LK*P`K:VM
M`*ZNK@"OKZ\`L+"P`+&QL0"RLK(`L[.S`+2TM`"UM;4`MK:V`+>WMP"XN+@`
MN;FY`+JZN@"[N[L`O+R\`+V]O0"^OKX`O[^_`,#`P`#!P<$`PL+"`,/#PP#$
MQ,0`Q<7%`,;&Q@#'Q\<`R,C(`,G)R0#*RLH`R\O+`,S,S`#-S<T`SL[.`,_/
MSP#0T-``T='1`-+2T@#3T],`U-34`-75U0#6UM8`U]?7`-C8V`#9V=D`VMK:
M`-O;VP#<W-P`W=W=`-[>W@#?W]\`X.#@`.'AX0#BXN(`X^/C`.3DY`#EY>4`
MYN;F`.?GYP#HZ.@`Z>GI`.KJZ@#KZ^L`[.SL`.WM[0#N[NX`[^_O`/#P\`#Q
M\?$`\O+R`//S\P#T]/0`]?7U`/;V]@#W]_<`^/CX`/GY^0#Z^OH`^_O[`/S\
M_`#]_?T`_O[^`/___P`T-#0\/#P\/#P\/#P\/#P\/#P\/#P\/#PL#`0$!`0$
M!%RTU-34U-34U-34U-S4U-34U-34U-34U+0T400$!`0$9,SL]/3T_/3\]/3T
M_/3\]/ST_/S\]/3LQ#11400$!`1DQ.3L]/3T]/3T]/3T]/3T]/3T]/3T[.R\
M-%%1400$!&3$Y.SL[.3L[.SDY.3D[.SD[.3L[.SLY+PT45%1400$9+SD[.RL
MC*S4Q)24G)2,C)24E)R<G.S<O#1145%1401DO-SD[.3$492L9.2D3#0L7,3,
M4;34Y-R\-%%145%1462\W.3D[.3D481TY(QLI)QLC+QTK-3DW+PT!`0$!`0$
M9+S<Y.3LY.2\48S4U(RLS'3,W*11Q.3<O#0$!`0$!`1DO-SDY.3DY-R,?)RT
M?(2<;,SDO'3$Y-RT-`0$!`0$!%R\W-SDY.3DU)RD7)2,S#R49,R\C%'<U+0T
M!`0$!`0$9+34W-S<Y-R,C*PLO+3,6L1DG&R4I-S4M#0$!`0$!`1<M-3<W.3<
MW'24U$RD](P\U'24=)Q,W,RT-`0$!`0$!&2TU-S<W.3<=%'T=(3LS&3D?&RD
MM&RDU+0T!`0$!`0$9+3,W-S,?(R$M/S,Q/STM/R\9,RLA%&\K#0$!`0$!`1<
MM,S4S(RDW-SL_/S\_/ST_.RT[*Q\E%&L-`0$!`0$!%RLS,QLG/3\_/S\_/S\
M_/S\_/S\Q)2,4:PT!`0$!`0$7*S$O'SD_/S\_/S\_/S\_/S\_/SLS*Q1K#0$
M!`0$!`1<K+1,Q/S\_.ST_/S\_/S\_/S\_/2\4<2L-`0$!`0$!%RLM$S4_/ST
MM-S\_/S\_/S\_/S\S%',Q*0T!`0$!`0$7*S$I%'\_/RTG/3\_/S\_/S\_/24
M4<S$I"P$!`0$!`1<I*11A.S\_,QDM.S\_/S\_/S\U&RLS+RD-`0$!`0$!%RD
M9'2\]/S\Q%&$G.3\_/S\_.2,4<S,Q*0T!`0$!`0$7*1LC.3T_/R<4<Q1E.S\
M_/S\O'3,S,2\G"P$!`0$!`1<I*QD7&2DI%&LS,1TE.3T_.11O,S,Q+R<-`0$
M!`0$!%RDM+R\C%%1Q,3,S*Q1?'R4;'R\S,3$O)PL!`0$!`0$7)R\Q,3$Q,3$
MQ,3$Q,1145%1O,3$Q,2TG#0$!`0$!`14G+3$Q,3$Q,3$Q,3$Q,3$Q,3$Q,3$
MO+2<+`0$!`0$!%2<O,3$O+R\O+R\O+R\O+R\O+R\O+R\M)PL!`0$!`0$7)RT
MQ+RTM+2TM+2TM+2TM+2TM+2TM+2LG"P$!`0$!`14A)2<G)24E)24E)24E)24
ME)24E)24E)2$)`0$!`0$!$147%Q<7%Q<7%Q<7%Q<7%Q<7%Q<7%Q<7%0T!`0$
M!`0$````/P```!\````/````!P````,````!`````````#\````_````/P``
M`#\````_````/P```#\````_````/P```#\````_````/P```#\````_````
M/P```#\````_````/P```#\````_````/P```#\````_````/P```#\```$`
M!``0$````0`(`&@%```!`"`@```!``@`J`@```(`$!````$`"`!H!0```P`@
M(````0`(`*@(```$````6`(T````5@!3`%\`5@!%`%(`4P!)`$\`3@!?`$D`
M3@!&`$\``````+T$[_X```$``````````````````````#\`````````!``$
M``$```````````````````"X`0```0!3`'0`<@!I`&X`9P!&`&D`;`!E`$D`
M;@!F`&\```"4`0```0`P`#``,``P`#``-`!"`#`````D``(``0!#`&\`;0!P
M`&$`;@!Y`$X`80!M`&4``````"`````L``(``0!&`&D`;`!E`$0`90!S`&,`
M<@!I`'``=`!I`&\`;@``````(````#``"``!`$8`:0!L`&4`5@!E`'(`<P!I
M`&\`;@``````,``N`#``+@`P`"X`,````"0``@`!`$D`;@!T`&4`<@!N`&$`
M;`!.`&$`;0!E````(````"@``@`!`$P`90!G`&$`;`!#`&\`<`!Y`'(`:0!G
M`&@`=````"`````L``(``0!,`&4`9P!A`&P`5`!R`&$`9`!E`&T`80!R`&L`
M<P``````(````"P``@`!`$\`<@!I`&<`:0!N`&$`;`!&`&D`;`!E`&X`80!M
M`&4````@````)``"``$`4`!R`&\`9`!U`&,`=`!.`&$`;0!E```````@````
M-``(``$`4`!R`&\`9`!U`&,`=`!6`&4`<@!S`&D`;P!N````,``N`#``+@`P
M`"X`,````$0````!`%8`80!R`$8`:0!L`&4`20!N`&8`;P``````)``$````
M5`!R`&$`;@!S`&P`80!T`&D`;P!N`````````+`$/#]X;6P@=F5R<VEO;CTB
M,2XP(B!E;F-O9&EN9STB551&+3@B('-T86YD86QO;F4](GEE<R(_/@T*/&%S
M<V5M8FQY('AM;&YS/2)U<FXZ<V-H96UA<RUM:6-R;W-O9G0M8V]M.F%S;2YV
M,2(@;6%N:69E<W1697)S:6]N/2(Q+C`B/@T*("`@(#QA<W-E;6)L>4ED96YT
M:71Y('9E<G-I;VX](C`N,"XP+C`B('1Y<&4](G=I;C,R(B!N86UE/2)005(M
M4&%C:V5R+D%P<&QI8V%T:6]N(B`O/@T*("`@(#QD97-C<FEP=&EO;CY005(M
M4&%C:V5R($%P<&QI8V%T:6]N/"]D97-C<FEP=&EO;CX-"B`@("`\9&5P96YD
M96YC>3X-"B`@("`@("`@/&1E<&5N9&5N=$%S<V5M8FQY/@T*("`@("`@("`@
M("`@/&%S<V5M8FQY261E;G1I='D@='EP93TB=VEN,S(B(&YA;64](DUI8W)O
M<V]F="Y7:6YD;W=S+D-O;6UO;BU#;VYT<F]L<R(@=F5R<VEO;CTB-BXP+C`N
M,"(@<'5B;&EC2V5Y5&]K96X](C8U.35B-C0Q-#1C8V8Q9&8B(&QA;F=U86=E
M/2(J(B!P<F]C97-S;W)!<F-H:71E8W1U<F4](BHB("\^#0H@("`@("`@(#PO
M9&5P96YD96YT07-S96UB;'D^#0H@("`@/"]D97!E;F1E;F-Y/@T*("`@(#QT
M<G5S=$EN9F\@>&UL;G,](G5R;CIS8VAE;6%S+6UI8W)O<V]F="UC;VTZ87-M
M+G8S(CX-"B`@("`@("`@/'-E8W5R:71Y/@T*("`@("`@("`@("`@/')E<75E
M<W1E9%!R:79I;&5G97,^#0H@("`@("`@("`@("`@("`@/')E<75E<W1E9$5X
M96-U=&EO;DQE=F5L(&QE=F5L/2)A<TEN=F]K97(B('5I06-C97-S/2)F86QS
M92(@+SX-"B`@("`@("`@("`@(#PO<F5Q=65S=&5D4')I=FEL96=E<SX-"B`@
M("`@("`@/"]S96-U<FET>3X-"B`@("`\+W1R=7-T26YF;SX-"B`@("`\8V]M
M<&%T:6)I;&ET>2!X;6QN<STB=7)N.G-C:&5M87,M;6EC<F]S;V9T+6-O;3IC
M;VUP871I8FEL:71Y+G8Q(CX-"B`@("`@("`@/&%P<&QI8V%T:6]N/@T*("`@
M("`@("`@("`@/"$M+2!7:6YD;W=S(#@N,2`M+3X-"B`@("`@("`@("`@(#QS
M=7!P;W)T961/4R!)9#TB>S%F-C<V8S<V+3@P93$M-#(S.2TY-6)B+3@S9#!F
M-F0P9&$W.'TB+SX-"B`@("`@("`@("`@(#PA+2T@5VEN9&]W<R!6:7-T82`M
M+3X-"B`@("`@("`@("`@(#QS=7!P;W)T961/4R!)9#TB>V4R,#$Q-#4W+3$U
M-#8M-#-C-2UA-69E+3`P.&1E964S9#-F,'TB+SX-"B`@("`@("`@("`@(#PA
M+2T@5VEN9&]W<R`W("TM/@T*("`@("`@("`@("`@/'-U<'!O<G1E9$]3($ED
M/2)[,S4Q,SAB.6$M-60Y-BTT9F)D+3AE,F0M83(T-#`R,C5F.3-A?2(O/@T*
M("`@("`@("`@("`@/"$M+2!7:6YD;W=S(#@@+2T^#0H@("`@("`@("`@("`\
M<W5P<&]R=&5D3U,@260](GLT83)F,CAE,RTU,V(Y+30T-#$M8F$Y8RUD-CED
M-&$T839E,SA](B\^#0H@("`@("`@("`@("`\(2TM(%=I;F1O=W,@,3`@+2T^
M#0H@("`@("`@("`@("`\<W5P<&]R=&5D3U,@260](GLX93!F-V$Q,BUB9F(S
M+31F93@M8CEA-2TT.&9D-3!A,35A.6%](B\^#0H@("`@("`@(#PO87!P;&EC
M871I;VX^#0H@("`@/"]C;VUP871I8FEL:71Y/@T*/"]A<W-E;6)L>3X-"@``
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````4```
M#````"BB````8```#````""@````L```&````)"@H*"PH,"@T*#@H/"@````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
-````````````````````
